from collections import defaultdict


def _recursive_default_dict():
    """
    Creates a recursive default dict, i.e. that allows you to create on the fly nested dict
    Example:
        rd = _recursive_default_dict()
        rd["a"]["b"]["c"] = 3
        => {"a": {"b": {"c": 3}}}
    :rtype: dict
    """
    return defaultdict(_recursive_default_dict)


def _flatten_iterate(elements):
    """
    Flatten and iterate over all element in elements.
    :type elements: tuple
    :return: an iterator
    """
    for element in elements:
        if isinstance(element, (list, tuple)):
            for e in _flatten_iterate(element):
                yield e
        else:
            yield element


def dict_with_tuple_keys_to_nested_dict(orig_dict):
    """
    Take a dict for which some 1st level keys might be tuples, and convert it to a nested dict

    Example:
        orig_dict = { "key1": "val1", "key2": "val2", ("key3", "key4"): "val3", ("key3", "key5"): "val4",
                      ("key6", "key7", "key8"): "val5"}

        dict_with_tuple_keys_to_nested_dict(orig_dict)
        => { "key1": "val1",
             "key2": "val2",
             "key3": {"key4": "val3", "key5": "val4"},
             "key6": {"key7":{ "key8": "val5"}}
          }

    :type orig_dict: collections.Mapping
    :rtype: dict
    """
    flattened_dict = _recursive_default_dict()
    for orig_key, value in orig_dict.items():
        if isinstance(orig_key, tuple):
            orig_key_list = list(_flatten_iterate(orig_key))
            if len(orig_key_list) == 1:
                flattened_dict[orig_key_list[0]] = value
            else:
                current_elem = flattened_dict
                for i in range(len(orig_key_list) - 1):
                    current_elem = current_elem[orig_key_list[i]]
                current_elem[orig_key_list[-1]] = value
        else:
            flattened_dict[orig_key] = value
    return flattened_dict
