#!/bin/bash -e
# Installing dependencies needed for dashboard and flow export feature to work in DSS.

# Don't call this directly. Use ./bin/dssadmin install-graphics-export

Usage() {
	echo >&2 "Usage: $0 [-noDeps]"
	echo >&2 "    -noDeps: do not check system dependencies"
	exit 1
}

MYDIR=$(cd "$(dirname "$0")" && pwd -P)
DKUINSTALLDIR=$(dirname "$MYDIR")

if [ -z "$DIP_HOME" -o ! -d "$DIP_HOME" ]; then
	echo >&2 "*** Error: DIP_HOME not found"
	exit 1
fi

noDeps=
while [ $# -gt 0 ]; do
	if [ "$1" = "-noDeps" ]; then
		noDeps=1
		shift
	else
		Usage
	fi
done

#
# Check system dependencies
#
if [ -z "$noDeps" ]; then
	echo "[+] Checking dependencies"
	"$DKUINSTALLDIR"/scripts/install/install-deps.sh -check \
		-without-java -without-python -with-chrome || {
		echo >&2 "
[-] Dependency check failed
[-] You can install required dependencies with:
[-]    sudo -i \"$DKUINSTALLDIR/scripts/install/install-deps.sh\" -without-java -without-python -with-chrome
[-] You can also disable this check with the -noDeps option
"
		exit 1
	}
fi

#
# Install required libraries by the export-dashboards.js and export-flow.js script
#
# Puppeteer 13.7.0 requires Node.js 10.18.1, which is the lowest version we support
# Puppeteer [14.0.0, 19.7.3) requires Node.js 14.1.0
# Puppeteer [19.7.3, 20.0.0) requires Node.js 14.14.0
# Puppeteer [20.0.0, 20.5.0) requires Node.js 16.0.0
# Puppeteer [20.5.0, 22.0.0) requires Node.js 16.3.0
# Puppeteer 22.0.0+ requires Node.js 18.0.0
#
# So here are the version, we're going to use depending on the Node.js version
# <10.18.1 => Error
# >=10.18.1 but <14.1.0 => 13.7.0
# >=14.1.0  but <14.14.0 => 19.7.2
# >=14.14.0 but <16.0.0 => 19.11.1
# >=16.0.0  but <16.3.0 => 20.4.0
# >=16.3.0 but < 18.0.0 => 21.11.0
# >=18.0.0 => 24

echo "[+] Installing additional modules for dashboard and flow export feature"
node_version=$(node -v)
# "node -v" outputs version in the format "v18.12.1"
node_version=${node_version:1} # Remove 'v' at the beginning
echo "+ Detected Node.js version ${node_version}"

IFS='.' read -r -a node_version_components <<< "${node_version}"
node_version_major=$((${node_version_components[0]}))
node_version_minor=$((${node_version_components[1]}))
node_version_patch=$((${node_version_components[2]}))
if [ $node_version_major -lt 10 ]; then
  # v9 or less Raise error
  echo "[-] Unsupported NodeJS ($node_version) version. Version 10.18.1 or more is required."
  exit 1
elif [ $node_version_major -eq 10 ]; then
  if [ $node_version_minor -lt 18 ]; then
    # v10.17 or less => Raise error
    echo "[-] Unsupported NodeJS ($node_version) version. Version 10.18.1 or more is required."
    exit 1
  elif [ $node_version_minor -eq 18 ]; then
    if [ $node_version_patch -lt 1 ]; then
      # v10.18.0 => Raise error
      echo "[-] Unsupported NodeJS ($node_version) version. Version 10.18.1 or more is required."
      exit 1
    else
      # v10.18.1+ => Go with puppeteer 13.7.0
      puppeteer_version="13.7.0"
    fi
  else
    # 10.19+ => Go with puppeteer 13.7.0
    puppeteer_version="13.7.0"
  fi
elif [ $node_version_major -lt 14 ]; then
  # v11, v12, v13 => Go with puppeteer 13.7.0
  puppeteer_version="13.7.0"
elif [ $node_version_major -eq 14 ]; then
  if [ $node_version_minor -lt 1 ]; then
    # v14.0 => Go with puppeteer 13.7.0
    puppeteer_version="13.7.0"
  elif [ $node_version_minor -lt 14 ]; then
    # v14.1-v14.13 => Go with puppeteer 19.7.2
    puppeteer_version="19.7.2"
  else
    # v14.14+ => Go with puppeteer 19.11.1
    puppeteer_version="19.11.1"
  fi
elif [ $node_version_major -lt 16 ]; then
  # v15 => Go with puppeteer 19.11.1
  puppeteer_version="19.11.1"
elif [ $node_version_major -eq 16 ]; then
  if [ $node_version_minor -lt 3 ]; then
    # v16.0-v16.2 => Go with puppeteer 20.4.0
    puppeteer_version="20.4.0"
  else
    # v16.3+ => Go with 21.11.0
    puppeteer_version="21.11.0"
  fi
elif [ $node_version_major -eq 17 ]; then
    # v17 => Go with 21.11.0
    puppeteer_version="21.11.0"
else
    # v18+ => Go with latest puppeteer
    puppeteer_version="24"
fi
echo "+ Installing Puppeteer ${puppeteer_version}"
cd "$DKUINSTALLDIR"/resources/graphics-export/
npm install puppeteer@${puppeteer_version}

# Add node to env-site.sh on macOS only
if [ "$(uname)" == "Darwin" ]; then

NODE_PATH=$(dirname "$(which node)")
cat <<EOF >>"$DIP_HOME"/bin/env-site.sh
#
# Add '${NODE_PATH}' to the PATH environment variable for graphics export where
# node was found
# You can change or remove the following line if you installed node in a
# different directory.
export PATH=\$PATH:${NODE_PATH}
EOF

fi

#
# Enable graphics export in DSS
#

echo "[+] Enabling graphics export in DSS"
"$DIP_HOME"/bin/dku __enable-graphics-export

echo "[+] Done"
