/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.fm.cloud.aws;

import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.DKUtils;
import com.dataiku.dss.shadelibawssk2.software.amazon.awssdk.services.acm.model.CertificateStatus;
import com.dataiku.dss.shadelibawssk2.software.amazon.awssdk.services.acm.model.DescribeCertificateResponse;
import com.dataiku.dss.shadelibawssk2.software.amazon.awssdk.services.acm.model.DomainValidation;
import com.dataiku.dss.shadelibawssk2.software.amazon.awssdk.services.acm.model.RequestCertificateResponse;
import com.dataiku.dss.shadelibawssk2.software.amazon.awssdk.services.acm.model.ResourceNotFoundException;
import com.dataiku.fm.cloud.DNSservice;
import com.dataiku.fm.cloud.aws.AWSClientService;
import com.dataiku.fm.cloud.aws.AWSDNSService;
import com.dataiku.fm.cloud.aws.AWSLBUtils;
import com.dataiku.fm.cloud.aws.sdk.AcmClientWrapper;
import com.dataiku.fm.model.db.LoadBalancerNodeMapping;
import com.dataiku.fm.model.db.LogicalLoadBalancer;
import com.dataiku.fm.model.db.VirtualNetwork;
import com.dataiku.fm.server.FMApp;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import org.apache.log4j.Priority;
import org.springframework.beans.factory.annotation.Autowired;

@ParametersAreNonnullByDefault
public class AWSCertificateService {
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.fm.cloud.aws.certificates");
    private final AWSClientService awsClientService;
    private final AWSDNSService awsdnsService;
    private final FMApp fmApp;
    private final long waitBetweenCalls;

    @Autowired
    public AWSCertificateService(AWSClientService awsClientService, AWSDNSService awsdnsService, FMApp fmApp) {
        this.awsClientService = awsClientService;
        this.awsdnsService = awsdnsService;
        this.fmApp = fmApp;
        this.waitBetweenCalls = 5000L;
    }

    AWSCertificateService(AWSClientService awsClientService, AWSDNSService awsdnsService, FMApp fmApp, long waitBetweenCalls) {
        this.awsClientService = awsClientService;
        this.awsdnsService = awsdnsService;
        this.fmApp = fmApp;
        this.waitBetweenCalls = waitBetweenCalls;
    }

    @Nonnull
    public String issueCertificate(LogicalLoadBalancer lb, DKUtils.SmartLogTailBuilder smartLogTail) {
        AcmClientWrapper acmClient = this.awsClientService.getACMClient(lb.getVirtualNetwork().getCloudAccount(), lb.getVirtualNetwork().getAwsRegion());
        String certificateARN = this.requestCertificate(lb, smartLogTail, acmClient);
        smartLogTail.appendLine("Certificate request created, ARN='" + certificateARN + "'. Now waiting for it to be issued", logger, Priority.INFO);
        SimplifiedCertificateDescription simplifiedCertificateDescription = this.waitUntilValidationDomainsAreAvailable(smartLogTail, acmClient, certificateARN);
        smartLogTail.appendLine("Initiating DNS record creation...or domain validation", logger, Priority.INFO);
        this.createValidationDomainsDnsRecords(lb, smartLogTail, simplifiedCertificateDescription.domainValidationList);
        this.waitForCertificateToBeIssued(smartLogTail, simplifiedCertificateDescription.certificateStatus, acmClient, certificateARN);
        return certificateARN;
    }

    private void waitForCertificateToBeIssued(DKUtils.SmartLogTailBuilder smartLogTail, CertificateStatus certificateStatus, AcmClientWrapper acmClient, String certificateARN) {
        for (int count = 0; count <= 12; ++count) {
            switch (certificateStatus) {
                case ISSUED: {
                    smartLogTail.appendLine("Certificate issued.", logger, Priority.INFO);
                    return;
                }
                case PENDING_VALIDATION: {
                    smartLogTail.appendLine("Certificate is pending validation. Waiting..." + ".".repeat(count), logger, Priority.INFO, count > 0);
                    break;
                }
                default: {
                    logger.error((Object)("Certificate is in an unexpected state: " + String.valueOf(certificateStatus)));
                    throw new IllegalStateException("Certificate is in an unexpected state: " + String.valueOf(certificateStatus));
                }
            }
            try {
                Thread.sleep(this.waitBetweenCalls);
            }
            catch (InterruptedException e) {
                logger.error((Object)"Couldn't wait for certificate request to be issued.", (Throwable)e);
                throw new IllegalStateException("Couldn't wait for certificate request to be issued.", e);
            }
            DescribeCertificateResponse describeResult = acmClient.describeCertificate(certificateARN);
            certificateStatus = describeResult.certificate().status();
        }
        smartLogTail.appendLine("Certificate is still invalid after 1 min. Check AWS certificate manager to get more details", logger, Priority.ERROR);
        throw new IllegalStateException("Certificate is still invalid after 1 min. Check AWS certificate manager to get more details");
    }

    private void createValidationDomainsDnsRecords(LogicalLoadBalancer lb, DKUtils.SmartLogTailBuilder smartLogTail, List<DomainValidation> domainValidationList) {
        for (DomainValidation domainValidation : domainValidationList) {
            String validationDomain = domainValidation.resourceRecord().name();
            String randomValue = domainValidation.resourceRecord().value();
            smartLogTail.appendLine("Initiating DNS record creation... '" + validationDomain + "' -> '" + randomValue + "'...", logger, Priority.INFO);
            this.awsdnsService.addDNSCNAMERecords(lb.getVirtualNetwork(), DNSservice.ZoneType.PUBLIC, validationDomain, randomValue);
            smartLogTail.replaceLastLine("DNS record for '" + validationDomain + "' -> '" + randomValue + "' has been successfully created.", logger, Priority.INFO);
        }
    }

    @Nonnull
    private String requestCertificate(LogicalLoadBalancer lb, DKUtils.SmartLogTailBuilder smartLogTail, AcmClientWrapper acmClient) {
        LinkedList domains = lb.getLoadBalancerNodeMapping().stream().map(LoadBalancerNodeMapping::getHostname).distinct().map(hostname -> this.awsdnsService.computeNodeMappingHostname(lb, (String)hostname)).collect(Collectors.toCollection(LinkedList::new));
        smartLogTail.appendLine("Requesting certificates for domains: '" + String.join((CharSequence)"', ", domains) + "'", logger, Priority.INFO);
        String domain = (String)domains.pop();
        RequestCertificateResponse response = acmClient.requestCertificate(domain, domains, AWSLBUtils.getCertificateTagsForResource(lb, lb.getName() + "-certificate-" + domain, this.fmApp.getFMSettings()));
        return response.certificateArn();
    }

    @Nonnull
    private SimplifiedCertificateDescription waitUntilValidationDomainsAreAvailable(DKUtils.SmartLogTailBuilder smartLogTail, AcmClientWrapper acmClient, String certificateARN) {
        for (int domainValidationOptionsRequestCount = 0; domainValidationOptionsRequestCount <= 10; ++domainValidationOptionsRequestCount) {
            try {
                Thread.sleep(this.waitBetweenCalls);
            }
            catch (InterruptedException e) {
                logger.error((Object)"Couldn't wait for certificate validation to be created.", (Throwable)e);
                throw new IllegalStateException("Couldn't wait for certificate validation to be created", e);
            }
            DescribeCertificateResponse describeResult = acmClient.describeCertificate(certificateARN);
            List domainValidationOptions = describeResult.certificate().domainValidationOptions();
            if (!domainValidationOptions.isEmpty() && domainValidationOptions.stream().noneMatch(d -> d.resourceRecord() == null)) {
                return new SimplifiedCertificateDescription(describeResult.certificate().status(), domainValidationOptions);
            }
            smartLogTail.appendLine("Certificate validation is still pending on AWS. Waiting..." + ".".repeat(domainValidationOptionsRequestCount), logger, Priority.INFO, domainValidationOptionsRequestCount > 0);
        }
        logger.error((Object)"Certificate domain validation were not created");
        throw new IllegalStateException("Certificate domain validation were not created");
    }

    public Optional<String> issueCertificate(LogicalLoadBalancer lb, DKUtils.SmartLogTailBuilder smartLogTail, @Nonnull String oldCertificateARN) {
        DescribeCertificateResponse describeResponse = this.awsClientService.getACMClient(lb.getVirtualNetwork().getCloudAccount(), lb.getVirtualNetwork().getAwsRegion()).describeCertificate(oldCertificateARN);
        HashSet<String> domainsFromCertificate = new HashSet<String>();
        domainsFromCertificate.add(describeResponse.certificate().domainName());
        domainsFromCertificate.addAll(describeResponse.certificate().subjectAlternativeNames());
        Set domainsExpected = lb.getLoadBalancerNodeMapping().stream().map(nodeMapping -> this.awsdnsService.computeNodeMappingHostname(lb, nodeMapping.getHostname())).collect(Collectors.toSet());
        if (!domainsFromCertificate.containsAll(domainsExpected)) {
            return Optional.of(this.issueCertificate(lb, smartLogTail));
        }
        describeResponse.certificate().domainValidationOptions().forEach(domainValidation -> this.addDnsRecordIfMissing(lb.getVirtualNetwork(), domainValidation.resourceRecord().name(), domainValidation.resourceRecord().value()));
        return Optional.empty();
    }

    private void addDnsRecordIfMissing(VirtualNetwork virtualNetwork, String recordName, String cname) {
        Optional<String> dnsRecord = this.awsdnsService.getDNSRecord(virtualNetwork, recordName, DNSservice.ZoneType.PUBLIC, DNSservice.RecordType.CNAME, false);
        if (dnsRecord.isEmpty()) {
            this.awsdnsService.addDNSCNAMERecords(virtualNetwork, DNSservice.ZoneType.PUBLIC, recordName, cname);
        }
    }

    public void deleteCertificate(VirtualNetwork vn, String certificateARNToDelete, @Nullable String certificateARNToKeep, DKUtils.SmartLogTailBuilder smartLogTail) {
        smartLogTail.appendLine("Deleting load balance certificate '" + certificateARNToDelete + "'", logger, Priority.INFO);
        AcmClientWrapper acmClient = this.awsClientService.getACMClient(vn.getCloudAccount(), vn.getAwsRegion());
        try {
            DescribeCertificateResponse describeCertificateResponse = this.waitCertificateToBeNotInUsed(certificateARNToDelete, smartLogTail, acmClient);
            List<DomainValidation> domainValidationOptionsToDelete = describeCertificateResponse.certificate().domainValidationOptions();
            if (certificateARNToKeep != null) {
                List<String> validationDomainsFromNewCert = acmClient.describeCertificate(certificateARNToKeep).certificate().domainValidationOptions().stream().map(d -> d.resourceRecord().name()).toList();
                domainValidationOptionsToDelete = domainValidationOptionsToDelete.stream().filter(d -> !validationDomainsFromNewCert.contains(d.resourceRecord().name())).toList();
            }
            this.removeValidationDomainsDnsRecords(vn, smartLogTail, domainValidationOptionsToDelete);
            smartLogTail.appendLine("Deleting certificate '" + certificateARNToDelete + "'...", logger, Priority.INFO);
            acmClient.deleteCertificate(certificateARNToDelete);
            smartLogTail.replaceLastLine("Certificate '" + certificateARNToDelete + "' deleted", logger, Priority.INFO);
        }
        catch (ResourceNotFoundException e) {
            smartLogTail.appendLine("Load balance certificate '" + certificateARNToDelete + "' not found. The certificate was already deleted.", logger, Priority.INFO);
        }
    }

    private void removeValidationDomainsDnsRecords(VirtualNetwork vn, DKUtils.SmartLogTailBuilder smartLogTail, List<DomainValidation> domainValidationOptionsToDelete) {
        for (DomainValidation domainValidation : domainValidationOptionsToDelete) {
            String validationDomain = domainValidation.resourceRecord().name();
            smartLogTail.appendLine("Deleting DNS record '" + validationDomain + "' corresponding to domain validation named '" + domainValidation.domainName() + "'...", logger, Priority.INFO);
            this.awsdnsService.removeDNSRecords(vn, validationDomain, DNSservice.RecordType.CNAME, false);
            smartLogTail.replaceLastLine("DNS record '" + validationDomain + "' corresponding to domain validation named '" + domainValidation.domainName() + "' deleted", logger, Priority.INFO);
        }
    }

    private DescribeCertificateResponse waitCertificateToBeNotInUsed(String certificateARN, DKUtils.SmartLogTailBuilder smartLogTail, AcmClientWrapper acmClient) {
        int deleteManagedCertificateTryCount = 10;
        while (0 < deleteManagedCertificateTryCount--) {
            try {
                Thread.sleep(this.waitBetweenCalls);
            }
            catch (InterruptedException e) {
                logger.error((Object)"Couldn't wait for certificate use checking.", (Throwable)e);
                throw new IllegalStateException("Couldn't wait for certificate use checking", e);
            }
            DescribeCertificateResponse describeResponse = acmClient.describeCertificate(certificateARN);
            if (!describeResponse.certificate().inUseBy().isEmpty()) continue;
            return describeResponse;
        }
        smartLogTail.appendLine("Could not delete certificate '" + certificateARN + "' as it is still mark as under use.", logger, Priority.ERROR);
        throw new IllegalStateException("Could not delete certificate '" + certificateARN + "' as it is still mark as under use.");
    }

    record SimplifiedCertificateDescription(CertificateStatus certificateStatus, List<DomainValidation> domainValidationList) {
    }
}

