(function() {
    'use strict';

    const app = angular.module("dataiku.fm.instances", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs", "dataiku.fm.forms"]);

    const instanceTypes = [['design', 'Design Node'], ['automation', 'Automation Node'], ['deployer', 'Deployer Node'], ['govern','Govern Node']];

    app.controller("InstancesBaseController", function($scope, $state, $stateParams, FutureProgressModal, CreateModalFromTemplate, Dialogs, FMAPI, TaggingService) {
        $scope.activeProjectTagColor = TaggingService.getDefaultColor;

        $scope.canProvision = function(instance) {
            return !(instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance)
        }
        $scope.canDeprovision = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance
        }
        $scope.canReprovision = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance
        }
        $scope.canReboot = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.cloudMachineExists && instance.physicalStatus.cloudMachineIsUp
        }
        $scope.canRestart = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance && instance.physicalStatus.cloudMachineIsUp
        }
        $scope.canStart = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.cloudMachineExists && !instance.physicalStatus.cloudMachineIsUp
                && !instance.physicalStatus.cloudInTransition
        }
        $scope.canStop = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.cloudMachineExists && instance.physicalStatus.cloudMachineIsUp
                && !instance.physicalStatus.cloudInTransition
        }
        $scope.canDelete = function(instance) {
            return !(instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance)
        }
        $scope.canReplay = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance && instance.physicalStatus.cloudMachineIsUp
        }
        $scope.canResetPassword = function(instance) {
            return instance.physicalStatus && instance.physicalStatus.hasPhysicalInstance && instance.physicalStatus.cloudMachineIsUp
        }
        $scope.isLinkedToLoadBalancer = function(instance) {
            return instance.loadBalancerId !== undefined;
        }

        $scope.isDefaultProvisionAction = function(instance) {
            return (instance.physicalStatus != null) && !$state.is('instances.instance.settings') && $scope.computeInstanceState (instance.physicalStatus) === "Not provisioned"
        }
        $scope.isDefaultStartAction = function(instance) {
            return (instance.physicalStatus != null) && !$state.is('instances.instance.settings') && $scope.computeInstanceState (instance.physicalStatus) === "Stopped"
        }
        $scope.isDefaultReprovisionAction = function(instance) {
            return (instance.physicalStatus != null) && !$state.is('instances.instance.settings') && $scope.computeInstanceState (instance.physicalStatus) === "Running"
        }

        $scope.startProvision = function(instanceId) {
            $scope.$broadcast("instanceProvision", instanceId);
            FMAPI.instances.startReprovision(instanceId).success(function(response) {
                FutureProgressModal.show($scope, response, "Provisioning instance", undefined, true, 'static', true).then(() => {
                    $scope.$broadcast("instanceStateUpdated", instanceId);
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.startReprovision = function(instanceId, instanceLabel) {
            Dialogs.confirm($scope, "Confirm reprovision", "Are you sure you want to reprovision the instance " + instanceLabel + "?").then(() => {
                $scope.startProvision(instanceId);
            });
        };

        $scope.startRestartDSS = function(instanceId, instanceLabel) {
            Dialogs.confirm($scope, "Confirm DSS restart", "Are you sure you want to restart DSS on the instance " + instanceLabel + "?").then(() => {
                FMAPI.instances.startRestartDSS(instanceId).success(function(response) {
                    FutureProgressModal.show($scope, response, "Restarting DSS", undefined, true, 'static', true).then((result) => {
                        if (result && result.infoMessages && result.infoMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Restarting DSS result", result.infoMessages);
                        }
                        $scope.$broadcast("instanceStateUpdated", instanceId);
                    });
                }).error(function(data, status, headers, config, statusText, xhrStatus) {
                        CreateModalFromTemplate("/common/dialogs/error-dialog.html", $scope, null, function(newScope) {
                            newScope.modalTitle = "Restarting DSS";
                            setErrorInScope.bind(newScope)(data, status, headers, config, statusText, xhrStatus);
                        }, true, 'static', true);
                    }
                );
            });
        };

        $scope.startDeprovision = function(instanceId, instanceLabel) {
            Dialogs.confirm($scope, "Confirm deprovision", "Are you sure you want to deprovision the instance " + instanceLabel + "?").then(() => {
                FMAPI.instances.startDeprovision(instanceId).success(function(response) {
                    FutureProgressModal.show($scope, response, "Deprovisioning instance", undefined, true, 'static', true).then((result) => {
                        if (result && result.infoMessages && result.infoMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Deprovisioning instance result", result.infoMessages);
                        }
                        $scope.$broadcast("instanceStateUpdated", instanceId);
                    });
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.startDelete = function(instanceId, instanceLabel) {
            Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the instance " + instanceLabel + "? Once deleted you will not be able to recover it!").then(() => {
                FMAPI.instances.startDelete(instanceId).success(function(response) {
                    FutureProgressModal.show($scope, response, "Deleting instance", undefined, true, 'static', true).then((result) => {
                        if (result && result.infoMessages && result.infoMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Deleting instance result", result.infoMessages);
                        }
                        $scope.$broadcast("instanceListUpdated");
                        $state.go("instances.list", {});
                    });
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.refreshLoadBalancer = function(instanceId) {
            FMAPI.instances.refreshLoadBalancer(instanceId).success(function(response) {
                FutureProgressModal.show($scope, response, "Refreshing load balancer", undefined, true, 'static', true).then(() => {
                    $scope.$broadcast("instanceListUpdated");
                    $state.go("instances.list", {});
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.startPhysicalStart = function(instanceId) {
            FMAPI.instances.startPhysicalStart(instanceId).success(function(response) {
                FutureProgressModal.show($scope, response, "Starting instance", undefined, true, 'static', true).then((result) => {
                    if (result && result.infoMessages && result.infoMessages.anyMessage) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Starting instance result", result.infoMessages);
                    }
                    $scope.$broadcast("instanceStateUpdated", instanceId);
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.startPhysicalStop = function(instanceId, instanceLabel) {
            Dialogs.confirm($scope, "Confirm stop", "Are you sure you want to stop the instance " + instanceLabel + "?").then((r) => {
                FMAPI.instances.startPhysicalStop(instanceId).success(function(response) {
                    FutureProgressModal.show($scope, response, "Stopping instance", undefined, true, 'static', true).then((result) => {
                        if (result && result.infoMessages && result.infoMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Stopping instance result", result.infoMessages);
                        }
                        $scope.$broadcast("instanceStateUpdated", instanceId);
                    });
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.startPhysicalReboot = function(instanceId, instanceLabel) {
            Dialogs.confirm($scope, "Confirm reboot", "Are you sure you want to reboot the instance " + instanceLabel + "?").then(() => {
                FMAPI.instances.startPhysicalReboot(instanceId).success(function(response) {
                    FutureProgressModal.show($scope, response, "Rebooting instance", undefined, true, 'static', true).then((result) => {
                        if (result && result.infoMessages && result.infoMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Rebooting instance result", result.infoMessages);
                        }
                        $scope.$broadcast("instanceStateUpdated", instanceId);
                    });
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.startReplaySetupActions = function(instanceId) {
            FMAPI.instances.replaySetupActions(instanceId).success(function(response) {
                FutureProgressModal.show($scope, response, "Replaying Setup Actions", undefined, true, 'static', true).then((result) => {
                    if (result && result.infoMessages && result.infoMessages.anyMessage) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Replaying setup actions result", result.infoMessages);
                    }
                    $scope.$broadcast("instanceStateUpdated", instanceId);
                });
            }).error(function(data, status, headers, config, statusText, xhrStatus) {
                    CreateModalFromTemplate("/common/dialogs/error-dialog.html", $scope, null, function(newScope) {
                        newScope.modalTitle = "Replaying Setup Actions";
                        setErrorInScope.bind(newScope)(data, status, headers, config, statusText, xhrStatus);
                    }, true, 'static', true);
                }
            );
        };

        $scope.computeInstanceStateColor = function(prefix, physicalStatus) {
            if (!physicalStatus || !physicalStatus.hasPhysicalInstance || !physicalStatus.cloudMachineExists || !physicalStatus.cloudMachineIsUp) {
                return prefix + '--notapplicable';
            }
            switch (physicalStatus.stage) {
                case "WAITING_AGENT":
                case "INITIALIZING":
                case "DSS_STARTING_UP":
                    return prefix + '--warning'; // Starting
                case "FAILED":
                case "NOT_RESPONDING":
                    return prefix + '--warning'; // In Error
                case "RUNNING":
                    return prefix + '--nominal';
                default:
                    return prefix + '--unknown';
            }
        };

        $scope.agentIsRunning = function(physicalStatus) {
            switch (physicalStatus.stage) {
                case "WAITING_AGENT":
                case "INITIALIZING":
                case "DSS_STARTING_UP":
                case "FAILED":
                case "NOT_RESPONDING":
                    return false;
                case "RUNNING":
                    return true;
                default: //unhandled case, allow it server will decide
                    return true;
            }
        }

        $scope.computeInstanceState = function(physicalStatus) {
            if (!physicalStatus || !physicalStatus.hasPhysicalInstance || !physicalStatus.cloudMachineExists) {
                return "Not provisioned";
            }
            if (!physicalStatus.cloudMachineIsUp) {
                return physicalStatus.cloudInTransition ? "Stopping" : "Stopped";
            }
            switch (physicalStatus.stage) {
                case "WAITING_AGENT":
                case "INITIALIZING":
                case "DSS_STARTING_UP":
                    return "Starting";
                case "FAILED":
                    return "In Error";
                case "RUNNING":
                    return "Running";
                case "NOT_RESPONDING":
                    return "Unresponsive";
                default:
                    return "Unknown";
            }
        };

        $scope.computeDiskUsagePercent = function(volumeSizeBytes, diskUsedBytes) {
            if (volumeSizeBytes && diskUsedBytes) {
                if (diskUsedBytes < volumeSizeBytes) {
                    return Math.min(100.0, Math.max(Math.round((100.0 * diskUsedBytes) / volumeSizeBytes), 0.0));
                } else {
                    return 100;
                }
            }
            return undefined;
        };

        $scope.computeDataDiskUsagePercent = function(instance, physicalStatus) {
            const dataVolumeSizeBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.deviceSize;
            const dataDiskUsedBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.bytesUsed;
            return $scope.computeDiskUsagePercent(dataVolumeSizeBytes, dataDiskUsedBytes);
        };

        $scope.computeOSDiskUsagePercent = function(instance, physicalStatus) {
            const osVolumeSizeBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.osDeviceSize;
            const osDiskUsedBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.osBytesUsed;
            return $scope.computeDiskUsagePercent(osVolumeSizeBytes, osDiskUsedBytes);
        };

        $scope.computeDiskSpaceAvailable = function(volumeSizeBytes, diskUsedBytes) {
            if (volumeSizeBytes && diskUsedBytes) {
                if (diskUsedBytes < volumeSizeBytes) {
                    return Math.round(100 * ((volumeSizeBytes - diskUsedBytes) / 1073741824)) / 100;
                } else {
                    return 0;
                }
            }
            return "--";
        };

        $scope.computeDataDiskSpaceAvailable = function(physicalStatus) {
            const dataVolumeSizeBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.deviceSize;
            const dataDiskUsedBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.bytesUsed;
            return $scope.computeDiskSpaceAvailable(dataVolumeSizeBytes, dataDiskUsedBytes);
        };

        $scope.computeOSDiskSpaceAvailable = function(physicalStatus) {
            const osVolumeSizeBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.osDeviceSize;
            const osDiskUsedBytes = !(physicalStatus && physicalStatus.lastHeartbeatState) ? undefined : physicalStatus.lastHeartbeatState.osBytesUsed;
            return $scope.computeDiskSpaceAvailable(osVolumeSizeBytes, osDiskUsedBytes);
        };

        $scope.computeInstanceUrl = function(physicalInstanceStatus) {
            if (physicalInstanceStatus.cloudMachineExists && physicalInstanceStatus.cloudMachineIsUp
                && (physicalInstanceStatus.stage === "DSS_STARTING_UP" || physicalInstanceStatus.stage === "RUNNING")) {
                if (physicalInstanceStatus.publicURL) {
                    return physicalInstanceStatus.publicURL;
                }
                if (physicalInstanceStatus.privateURL) {
                    return physicalInstanceStatus.privateURL;
                }
            }
            return undefined;
        };

        $scope.computeInstanceIP = function(physicalInstanceStatus) {
            if (physicalInstanceStatus.cloudMachineExists
                && physicalInstanceStatus.cloudMachineIsUp
                && (physicalInstanceStatus.stage === "DSS_STARTING_UP" || physicalInstanceStatus.stage === "RUNNING")) {
                if (physicalInstanceStatus.publicIP) {
                    return physicalInstanceStatus.publicIP;
                }
                if (physicalInstanceStatus.privateIP) {
                    return physicalInstanceStatus.privateIP;
                }
            }
            return undefined;
        };

        $scope.resetPassword = function(instanceId, username) {
            CreateModalFromTemplate("/app/instances/dialogs/reset-password-dialog.html", $scope, "ResetPasswordController", function(newScope) {
                newScope.modalTitle = `Reset ${username || 'user'} password`;
                newScope.instanceId = instanceId;
                newScope.username = username;
                newScope.parentScope = $scope;
            });
        }

    });

    app.controller("ResetPasswordController", function($scope, FMAPI, FutureWatcher) {
        $scope.validatePassword = function(form) {
            if (form.newPassword.$valid && form.confirmPassword.$valid
                || form.newPassword.$viewValue === "" && form.confirmPassword.$viewValue === "") {
                return "OK";
            }
            return form.newPassword.$error.minlength ? "TOO_SHORT" : "MISMATCH";
        };
        $scope.resetPassword = function() {
            $scope.state = "PROGRESS";
            const username = $scope.username || $scope.data.username;
            FMAPI.instances.resetUserPassword($scope.instanceId, username, $scope.data.newPassword).success(function(initialResponse) {
                FutureWatcher.watchJobId(initialResponse.jobId).success(function() {
                    $scope.state = "SUCCESS";
                    if ($scope.data.username === "admin") {
                        $scope.parentScope.$broadcast("instanceAdminPasswordReset");
                    }
                }).error(function(data, status, headers) {
                    $scope.state = "ERROR";
                    const commandExceptionPrefix = "<class 'dataikufmagent.exception.SetUserPasswordCommandException'>: ";
                    if (data.message.startsWith(commandExceptionPrefix)) {
                        data.errorType = "ExpectedException";
                        data.message = data.message.substring(commandExceptionPrefix.length);
                        data.detailedMessage = data.message;
                    }
                    setErrorInScope.bind($scope)(data, status, headers);
                });
            }).error(function(data, status, headers) {
                $scope.state = "ERROR";
                setErrorInScope.bind($scope)(data, status, headers);
            });
        };
        $scope.state = "INITIAL";
        $scope.data = {username: "", newPassword: "", confirmPassword: ""};
    });

    app.controller("InstancesListController", function($scope, $interval, FMAPI) {
        $scope.refreshInstanceState = function(instance) {
            FMAPI.instances.getPhysicalInstanceStatus(instance.id, true).success(function(physicalInstanceStatus) {
                if (physicalInstanceStatus.logicalInstanceExists) {
                    instance.label = physicalInstanceStatus.logicalInstanceLabel;
                    instance.physicalStatus = physicalInstanceStatus;
                    instance.url = $scope.computeInstanceUrl(physicalInstanceStatus);
                    instance.ip = $scope.computeInstanceIP(physicalInstanceStatus);
                    instance.agentRunning = $scope.agentIsRunning(physicalInstanceStatus);
                    instance.stateLabel = $scope.computeInstanceState(physicalInstanceStatus);
                    instance.stateBackgroundClassName = $scope.computeInstanceStateColor('instance-row__state', physicalInstanceStatus);
                } else {
                    // Instance has been deleted since then, remove it from our local list.
                    const indexOfDeletedInstance = $scope.instances.indexOf(instance);
                    if (indexOfDeletedInstance >= 0) {
                        $scope.instances.splice(indexOfDeletedInstance, 1);
                    }
                }
            }).error(setErrorInScope.bind($scope));
        };
        $scope.refreshInstanceListStates = function() {
            resetErrorInScope($scope);
            $scope.instances.forEach($scope.refreshInstanceState);
        };
        $scope.refreshInstanceList = function() {
            resetErrorInScope($scope);
            FMAPI.instances.list().success(function(data) {
                $scope.instances = data;
                $scope.refreshInstanceListStates();
            }).error(setErrorInScope.bind($scope));
        };

        $scope.$on("instanceStateUpdated", function(event, instanceId) {
            resetErrorInScope($scope);
            const instance = $scope.instances.find(instance => instance.id === instanceId);
            if (instance) {
                $scope.refreshInstanceState(instance);
            }
        });

        $scope.$on("instanceListUpdated", function() {
            $scope.refreshInstanceList();
        });

        // Initialize
        $scope.refreshInstanceList();
        const refreshInstanceListIntervalHandle = $interval($scope.refreshInstanceListStates, 15000);
        $scope.$on('$destroy', function() {
            if (refreshInstanceListIntervalHandle) {
                $interval.cancel(refreshInstanceListIntervalHandle);
            }
        });
    });

    app.service("InstanceCreationDataService", function(FMAPI, Logger) {
        const svc = {};
        svc.fetchAndSetIfNeeded = function(targetScope, targetInstanceObject, copyTargetInstance) {
            FMAPI.instances.getCreationData().success(function(data) {
                Logger.info("Set settings of instance", targetInstanceObject);
                Logger.info("with static data", data);
                targetScope.creationData = data;
                targetScope.creationData.virtualNetworksMap = {};
                data.virtualNetworks.forEach((vn) => {
                    targetScope.creationData.virtualNetworksMap[vn.id] = vn;
                });

                try {
                    if (!targetInstanceObject.cloudInstanceType) {
                        if (data.defaultCloudInstanceType) {
                            targetInstanceObject.cloudInstanceType = data.defaultCloudInstanceType;
                        } else if (data.cloudInstanceTypes.length) {
                            targetInstanceObject.cloudInstanceType = data.cloudInstanceTypes[0].id;
                        }
                    }

                    if (!targetInstanceObject.imageId) {
                        if (data.defaultImageId) {
                            targetInstanceObject.imageId = data.defaultImageId;
                        } else if (data.dssImages.length) {
                            targetInstanceObject.imageId = data.dssImages[0].id;
                        }
                    }

                    if (!targetInstanceObject.dataVolumeType) {
                        if (data.defaultDataVolumeType) {
                            targetInstanceObject.dataVolumeType = data.defaultDataVolumeType;
                        } else if (data.dataVolumeTypes.length) {
                            targetInstanceObject.dataVolumeType = data.dataVolumeTypes[0].id;
                        }
                    }

                    if (!targetInstanceObject.dataVolumeIOPS) {
                        targetInstanceObject.dataVolumeIOPS = data.defaultDataVolumeIOPS;
                    }

                    if (!targetInstanceObject.dataVolumeSizeGB) {
                        targetInstanceObject.dataVolumeSizeGB = data.defaultDataVolumeSizeGB;
                    }

                    if (!targetInstanceObject.dataVolumeSizeMaxGB) {
                        targetInstanceObject.dataVolumeSizeMaxGB = data.defaultDataVolumeSizeMaxGB;
                    }

                    if (!targetInstanceObject.virtualNetworkId && data.virtualNetworks.length) {
                        targetInstanceObject.virtualNetworkId = data.virtualNetworks[0].id;
                    }

                    if (!targetInstanceObject.volumesEncryption) {
                        targetInstanceObject.volumesEncryption = data.defaultVolumesEncryptionMode;
                    }

                    if (!targetInstanceObject.instanceSettingsTemplateId && data.instanceSettingsTemplates.length) {
                        targetInstanceObject.instanceSettingsTemplateId = data.instanceSettingsTemplates[0].id;
                    }

                    if (copyTargetInstance !== undefined) {
                        copyTargetInstance.imageId = targetInstanceObject.imageId;
                        copyTargetInstance.dataVolumeType = targetInstanceObject.dataVolumeType;
                        copyTargetInstance.dataVolumeSizeGB = targetInstanceObject.dataVolumeSizeGB;
                        copyTargetInstance.dataVolumeSizeMaxGB = targetInstanceObject.dataVolumeSizeMaxGB;
                        copyTargetInstance.dataVolumeIOPS = targetInstanceObject.dataVolumeIOPS;
                        copyTargetInstance.virtualNetworkId = targetInstanceObject.virtualNetworkId;
                        copyTargetInstance.volumesEncryption = targetInstanceObject.volumesEncryption;
                        copyTargetInstance.instanceSettingsTemplateId = targetInstanceObject.instanceSettingsTemplateId
                    }
                } catch (err) {
                    Logger.error(err);
                }
            }).error(setErrorInScope.bind(targetScope))
        };
        return svc;
    });

    app.controller("InstanceCreationController", function($scope, $state, FMAPI, InstanceCreationDataService, FeatureFlagsService, WT1, Debounce) {

        $scope.instanceTypes = instanceTypes;

        $scope.$watch('newInstance.dataVolumeSizeGB', Debounce().withDelay(500, 500).wrap(function() {
            $scope.strongWarningDataVolumeSizeGB = false;
            $scope.warningDataVolumeSizeGB = false;
            const volumeSize = $scope.newInstance.dataVolumeSizeGB;
            if (volumeSize < 75) {
                $scope.strongWarningDataVolumeSizeGB = true;
            } else if (volumeSize < 200) {
                $scope.warningDataVolumeSizeGB = true;
            }
        }));

        $scope.newInstance = {
            dssNodeType: "design",
            fmTags: [],
            cloudTags: [],
            singleNodeAsInfrastructure: true
        };
        InstanceCreationDataService.fetchAndSetIfNeeded($scope, $scope.newInstance);

        $scope.create = function(provisionUponCreation) {
            WT1.event("fm-instance-create", {
                dssNodeType : $scope.newInstance.dssNodeType
            });
    
            FMAPI.instances.create($scope.newInstance).success(function(newInstance) {
                $state.go("instances.instance.dashboard", {instanceId: newInstance.id, startReprovision: !!provisionUponCreation}, {});
            }).error(setErrorInScope.bind($scope));
        }

        $scope.encryptionsAvailable = [['NONE', 'None']];
        if ($scope.appConfig.cloud === 'AWS') {
            $scope.encryptionsAvailable = [
                ['NONE', 'None'],
                ['DEFAULT_KEY', 'Encrypt with EBS default key'],
                ['TENANT', 'Encrypt with the configured tenant CMK'],
                ['CUSTOM', 'Encrypt with a custom CMK']
            ];
        } else if ($scope.appConfig.cloud === 'AZURE') {
            $scope.encryptionsAvailable = [
                ['DEFAULT_KEY', 'Encrypt with platform managed encryption key'],
                ['CUSTOM', 'Encrypt with customer managed encryption key']
            ];
        } else if ($scope.appConfig.cloud === 'GCP') {
            $scope.encryptionsAvailable = [
                ['DEFAULT_KEY', 'Encrypt with GCE default key'],
                ['TENANT', 'Encrypt with the configured tenant key'],
                ['CUSTOM', 'Encrypt with a custom KMS key']
            ];
        }
    });

    app.controller("InstanceController", function($scope, $state, $stateParams, $interval, FMAPI, FutureWatcher, FeatureFlagsService, FutureProgressModal, InstanceCreationDataService) {
        $scope.SharedState = $scope.SharedState || {
            instanceSettingsFormIsValid : false,
            instanceSettingsFormIsDirty: false
        };

        $scope.refreshInstanceState = function() {
            resetErrorInScope($scope);
            FMAPI.instances.getPhysicalInstanceStatus($stateParams.instanceId, true).success(function(physicalInstanceStatus) {
                // Update instance details that may have been updated meanwhile
                if ($scope.instance) {
                    $scope.instance.label = physicalInstanceStatus.logicalInstanceLabel;
                    $scope.instance.externalURL = physicalInstanceStatus.externalURL;
                    $scope.instance.instanceSettingsTemplateId = physicalInstanceStatus.instanceSettingsTemplateId;
                    $scope.instance.instanceSettingsTemplateLabel = physicalInstanceStatus.instanceSettingsTemplateLabel;
                    $scope.instance.virtualNetworkLabel = physicalInstanceStatus.virtualNetworkLabel;
                    $scope.instance.dataVolumeSizeGB = physicalInstanceStatus.lastHeartbeatState ? Math.round(physicalInstanceStatus.lastHeartbeatState.deviceSize/1073741824) : physicalInstanceStatus.dataVolumeSizeGB;
                    $scope.instance.dataVolumeSizeMaxGB = physicalInstanceStatus.dataVolumeSizeMaxGB;
                    $scope.instance.osVolumeSizeGB = physicalInstanceStatus.lastHeartbeatState ? Math.round(physicalInstanceStatus.lastHeartbeatState.osDeviceSize/1073741824) : physicalInstanceStatus.rootVolumeSizeGB;
                    $scope.instance.osVolumeSizeMaxGB = physicalInstanceStatus.rootVolumeSizeMaxGB;
                }
                $scope.instanceState.url = $scope.computeInstanceUrl(physicalInstanceStatus);
                $scope.instanceState.physicalStatus = physicalInstanceStatus;
                $scope.instanceState.agentRunning = $scope.agentIsRunning(physicalInstanceStatus);
                $scope.instanceState.stateLabel = $scope.computeInstanceState(physicalInstanceStatus);
                $scope.instanceState.stateBackgroundClassName = $scope.computeInstanceStateColor('instance-row__state', physicalInstanceStatus);
                $scope.instanceState.stateBorderClassName = $scope.computeInstanceStateColor('fm-section__header', physicalInstanceStatus);
                $scope.instanceState.dataDiskUsagePercent = $scope.computeDataDiskUsagePercent($scope.instance, physicalInstanceStatus);
                $scope.instanceState.dataDiskSpaceAvailableGB = $scope.computeDataDiskSpaceAvailable(physicalInstanceStatus);
                $scope.instanceState.osDiskUsagePercent = $scope.computeOSDiskUsagePercent($scope.instance, physicalInstanceStatus);
                $scope.instanceState.osDiskSpaceAvailableGB = $scope.computeOSDiskSpaceAvailable(physicalInstanceStatus);
                $scope.instanceState.agentReady = $scope.instanceState.physicalStatus.cloudMachineIsUp
                    && ($scope.instanceState.physicalStatus.stage === "RUNNING" || $scope.instanceState.physicalStatus.stage === "FAILED");

                // Retrieve and display latest fetched logs...
                if (!$scope.agentLogs && $scope.instanceState.physicalStatus.hasPhysicalInstance) {
                    $scope.agentLogs = smartLogTailToHTML({totalLines: 1, lines: ["Loading..."], status: [1]}, true);
                    $scope.readAgentLogs();
                }
                // Fetch latest logs (if we haven't tried it yet)
                if (!$scope.fetchAgentLogsCalledOnce && $scope.instanceState.agentReady) {
                    $scope.fetchAgentLogsSilently();
                }
            }).error(setErrorInScope.bind($scope));
        };

        $scope.readAgentLogs = function() {
            FMAPI.instances.readAgentLogs($stateParams.instanceId, 100).success(function(logTail) {
                $scope.agentLogs = smartLogTailToHTML(logTail, true);
            }).error(function(err) {
                $scope.agentLogs = smartLogTailToHTML({totalLines: 1, lines: [err], status: [3]}, true);
            });
        };

        $scope.fetchAgentLogs = function() {
            if ($scope.instanceState && $scope.instanceState.physicalStatus && $scope.instanceState.physicalStatus.stage === "FAILED") {
                // If the instance is in error, just read again the logs as they are pro-actively sent by the agent when it crashes.
                $scope.readAgentLogs();
            } else {
                resetErrorInScope($scope);
                $scope.fetchAgentLogsCalledOnce = true;
                FMAPI.instances.fetchAgentLogs($stateParams.instanceId).success(function(response) {
                    FutureProgressModal.show($scope, response, "Fetching Agent Logs").then(() => {
                        $scope.readAgentLogs();
                    });
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.fetchAgentLogsSilently = function() {
            resetErrorInScope($scope);
            $scope.fetchAgentLogsCalledOnce = true;
            FMAPI.instances.fetchAgentLogs($stateParams.instanceId).success(function(initialResponse) {
                FutureWatcher.watchJobId(initialResponse.jobId).then(() => {
                    $scope.readAgentLogs();
                }).error(setErrorInScope.bind($scope));
            }).error(setErrorInScope.bind($scope));
        };

        $scope.refreshInstance = function() {
            resetErrorInScope($scope);
            FMAPI.instances.get($stateParams.instanceId).success(function(instance) {
                $scope.instance = instance;
                FMAPI.virtualNetworks.getCloudAccount(instance.virtualNetworkId).success(function(data) {
                    $scope.cloudAccount = data;
                }).error(setErrorInScope.bind($scope));
            }).error(setErrorInScope.bind($scope));
            $scope.refreshInstanceState();
        };

        $scope.onSaveButtonClicked = function() {
            $scope.$broadcast("saveButtonClicked");
        };

        $scope.$on("instanceStateUpdated", $scope.refreshInstance);
        $scope.$on("instanceUpdated", function(event, instance) {
            if (instance) {
                $scope.instance = instance;
            } else {
                $scope.refreshInstance();
            }
        });
        $scope.$on("instanceAdminPasswordReset", function() {
            $scope.instance.hasInitialAdminPassword = false;
            $scope.instance.$initialAdminPassword = undefined;
        });

        $scope.$on("dirtyChanged", function(event, dirtyValue) {
            $scope.instanceEditionState.dirty = dirtyValue;
            $scope.SharedState.instanceSettingsFormIsDirty = dirtyValue;
        });

        $scope.$on("instanceProvision", function() {
            $scope.fetchAgentLogsCalledOnce = false; // dirty the logs
        });

        Mousetrap.bind("d e b u g", function() {
            $scope.showDebugInfo = !$scope.showDebugInfo;
            $scope.$apply();
        });

        // Define scope variables
        $scope.instanceEditionState = {
            dirty: false
        };
        $scope.instanceState = {
            physicalStatus: null,
            url: null,
            stateLabel: "",
            stateBackgroundClassName: "",
            stateBorderClassName: "",
            diskSpaceAvailableGB: "--"
        };
        $scope.fetchAgentLogsCalledOnce = false;
        $scope.agentReady = false;
        $scope.showDebugInfo = false;

        // Initialize screen & launch auto-refresh
        $scope.refreshInstance();
        const refreshInstanceIntervalHandle = $interval($scope.refreshInstanceState, 15000);
        $scope.$on('$destroy', function() {
            if (refreshInstanceIntervalHandle) {
                $interval.cancel(refreshInstanceIntervalHandle);
            }
        });
        if ($stateParams.startReprovision) {
            FMAPI.instances.startReprovision($stateParams.instanceId).success(function(response) {
                FutureProgressModal.show($scope, response, "Provisioning instance").then(function() {
                    if ($state.is("instances.instance.dashboard")) { // Ensure user has not moved to another page.
                        $scope.refreshInstance();
                    }
                });
            }).error(setErrorInScope.bind($scope));
        }

        InstanceCreationDataService.fetchAndSetIfNeeded($scope, $scope);

        $scope.labelFromItemInArray = function(itemId, array, idField = 'id', labelField = 'label') {
            const item = array ? array.find(s => s[idField] === itemId) : null;
            return item ? item[labelField] : itemId;
        };
    });

    app.controller("InstanceDashboardController", function($scope, $stateParams, FMAPI) {
        $scope.getInitialAdminPassword = function() {
            FMAPI.instances.getInitialAdminPassword($stateParams.instanceId).success(function(initialAdminPassword) {
                // Use $ sign to avoid copying it and sending it back the backend.
                $scope.instance.$initialAdminPassword = initialAdminPassword;
            }).error(setErrorInScope.bind($scope));
        };
    });

    app.controller("InstanceSettingsController", function($scope, $stateParams, FMAPI, ActivityIndicator, InstanceCreationDataService) {

        $scope.instanceTypes = instanceTypes;

        FMAPI.instances.get($stateParams.instanceId).success(function(instance) {
            // eslint-disable-next-line no-undef
            $scope.editedInstance = dkuDeepCopy(instance, filterDollarKey);
            $scope.editedInstanceOriginalImageId = $scope.editedInstance.imageId; // storing original dss version to warn about downgrades
            // need to copy back in original instance so that both can be compared so to detect correctly changes
            InstanceCreationDataService.fetchAndSetIfNeeded($scope, $scope.editedInstance, $scope.instance);

        }).error(setErrorInScope.bind($scope));

        $scope.$watch('instanceSettingsForm.$valid', function(newVal) {
            $scope.SharedState.instanceSettingsFormIsValid = newVal;
        })

        $scope.$watch('instanceSettingsForm.$dirty', function(newVal) {
            $scope.SharedState.instanceSettingsFormIsDirty = newVal;
        })

        $scope.parseImageId = function(imageId) {
            const version = imageId.match(/\d+\.\d+\.\d+/g);
            if (version && version.length === 1)
                return version[0].split('.').map(i => parseInt(i));
            return null;
        }

        $scope.$watch('editedInstance.imageId', function() {
            if ($scope.editedInstanceOriginalImageId) {
                const vOld = $scope.parseImageId($scope.editedInstanceOriginalImageId);
                const vNew = $scope.parseImageId($scope.editedInstance.imageId);
                if (vOld && vNew)
                    // only compare the x.y.z versioning
                    $scope.displayDowngradeWarning = vOld[0] > vNew[0] || (vOld[0] == vNew[0] && ( vOld[1] > vNew [1] || (vOld[1] == vNew[1] && vOld[2] > vNew[2])))
                else
                    $scope.displayDowngradeWarning = false;
            }
        })

        $scope.$on("saveButtonClicked", function() {
            FMAPI.instances.save($scope.editedInstance).success(function(saveResult) {
                resetErrorInScope($scope);
                // eslint-disable-next-line no-undef
                let instance = dkuDeepCopy($scope.editedInstance, filterDollarKey);
                $scope.editedInstanceOriginalImageId = $scope.editedInstance.imageId; // storing original dss version to warn about downgrades
                $scope.displayDowngradeWarning = false;
                instance.imageLabel = $scope.creationData.dssImages.find(image => {
                    return image.id === instance.imageId;
                }).label;
                if ($scope.instanceState.physicalStatus && $scope.instanceState.physicalStatus.stage === "RUNNING") {
                    $scope.instanceState.url = $scope.editedInstance.externalURL;
                }
                $scope.$emit('instanceUpdated', instance);
                $scope.$emit('dirtyChanged', false);
                if (saveResult.needsReprovision) {
                    ActivityIndicator.warning("Instance needs to be reprovisioned for the changes to take effect.", 5000);
                } else {
                    ActivityIndicator.success("Instance settings saved");
                }
            }).error(setErrorInScope.bind($scope));
        });

        $scope.$watch("editedInstance.enableAutomatedSnapshot", function(nv) {
            if (nv && !$scope.editedInstance.automatedSnapshotPeriod) {
                $scope.editedInstance.automatedSnapshotPeriod = 3;
            }
        });

        $scope.$watch("editedInstance", function(nv) {
            if (nv) {
                // eslint-disable-next-line no-undef
                let cleanEdited = dkuDeepCopy($scope.editedInstance, filterDollarKey);
                const dirtyValue = !angular.equals(cleanEdited, $scope.instance);
                $scope.$emit('dirtyChanged', dirtyValue);
            }
        }, true);

        $scope.httpsStrategies = [
            {id: 'NONE', label: 'None (HTTP only)'},
            {id: 'SELF_SIGNED', label: 'Self-signed certificates'},
            {id: 'CUSTOM_CERTIFICATE', label: 'Enter a certificate/key for each instance'},
            {id: 'LETSENCRYPT', label: 'Generate certificates using Let\'s Encrypt'}
        ];

        if ($scope.appConfig.cloud === 'AWS') {
            $scope.keyStorageModes = [['SECRETS_MANAGER', 'Secret stored in ASM'], ['INLINE_ENCRYPTED', 'Enter key']];
        } else if ($scope.appConfig.cloud === 'AZURE') {
            $scope.keyStorageModes = [['SECRETS_MANAGER', 'Secret stored in Key Vault'], ['CERTIFICATE_MANAGER', 'Certificate stored in Key Vault'], ['INLINE_ENCRYPTED', 'Enter key']];
        } else if ($scope.appConfig.cloud === 'GCP') {
            $scope.keyStorageModes = [['SECRETS_MANAGER', 'Secret stored in Secret manager'], ['INLINE_ENCRYPTED', 'Enter key']];
        }
        $scope.keyStorageModes.push(['SELF_SIGNED', 'Temporary self-signed']);
    });

    app.controller("InstanceEventsLogController", function($scope, $stateParams, FMAPI) {
        $scope.fetch = function() {
            FMAPI.instances.getEventsLog($stateParams.instanceId).success(function(data) {
                $scope.eventsLog = data;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.fetch();
    });

    app.controller("InstanceSnapshotsController", function($scope, $state, $stateParams, FMAPI, FutureProgressModal, Dialogs, WT1) {

        $scope.selection = []

        $scope.selectedSnapshots = function() {
            return $scope.selection.length > 0;
        }

        $scope.selectAllSnapshots = function () {
            $scope.selection = [];
            for (let snap of $scope.snapshots) {
                snap.$selected = true
                $scope.selection.push(snap.id)
            }
        }

        $scope.unselectAllSnapshots = function () {
            $scope.selection = [];
            for (let snap of $scope.snapshots) {
                snap.$selected = false
            }
        }

        $scope.isSelected = function(snapshot) {
            return $scope.selection.includes(snapshot.id);
        }

        $scope.selectionChange = function(snapshot) {
           const index = $scope.selection.indexOf(snapshot.id)
           if (index >= 0) {
                snapshot.$selected = false
                $scope.selection.splice(index, 1);
           } else {
                snapshot.$selected = true
                $scope.selection.push(snapshot.id)
           }
        }

        $scope.deleteSnapshots = function(instance) {
            if ($scope.selection.length == 0) {
                Dialogs.infoMessagesDisplayOnly($scope, "No Snapshot Selected", "Please select the snapshot you want to delete.");
                return;
            }
            Dialogs.confirm($scope, "Confirm snapshots deletion", "Are you sure you want to delete " + $scope.selection.length + " snapshot(s)?").then(() => {
                WT1.event("fm-instance-snapshots-delete", {
                    nb : $scope.selection.length
                });
                FMAPI.snapshots.startDeletes(instance.id, $scope.selection).success(function(response) {
                    FutureProgressModal.show($scope, response, "Deleting snapshots", undefined, true, 'static', true).then(function(result) {
                        $scope.refreshSnapshots();
                        if (result.statusMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Snapshots deletion result", result.statusMessages);
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            });

        }

        $scope.refreshSnapshots = function() {
            FMAPI.snapshots.list($stateParams.instanceId).success(function(data) {
                if (data) {
                    // Order snapshots by creation date as it's not done in the backend.
                    data.sort((snapshot1, snapshot2) => snapshot2.creationDate - snapshot1.creationDate);
                }
                $scope.snapshots = data;
                $scope.selection = [];
            }).error(setErrorInScope.bind($scope));
        };

        $scope.getSnapshotLabel = function(snapshot) {
            if (snapshot.description) {
                return snapshot.description;
            }
            return snapshot.awsSnapshotId ? snapshot.awsSnapshotId : snapshot.azureSnapshotId;
        };

        $scope.getDisplayableSnapshotId = function(snapshot) {
            if (snapshot.awsSnapshotId) return snapshot.awsSnapshotId;
            if (snapshot.azureSnapshotId) {
                let lastSlash = snapshot.azureSnapshotId.lastIndexOf("/");
                if (lastSlash > 0) {
                    return snapshot.azureSnapshotId.substring(lastSlash + 1);
                } else {
                    return snapshot.azureSnapshotId; // there should have been a shash, so just show the true value for the user to decide
                }
            }
            return '';
        };

        $scope.createSnapshot = function() {
            Dialogs.prompt($scope, "New snapshot", "Description", "", {placeholder: "Reason to create this snapshot"}).then(function(reason) {
                FMAPI.snapshots.create($stateParams.instanceId, reason).success(function(snapshot) {
                    $scope.snapshots.unshift(snapshot);
                    $scope.refreshSnapshots();
                }).error(setErrorInScope.bind($scope));
            })
        };

        $scope.reprovisionFromSnapshot = function(instance, snapshot) {
            const snapshotLabel = $scope.getSnapshotLabel(snapshot);
            Dialogs.confirm($scope, "Confirm reprovision", "Are you sure you want to reprovision the instance " + instance.label + " from the snapshot '" + snapshotLabel + "'?").then(() => {
                FMAPI.snapshots.reprovisionInstance(instance.id, snapshot.id).success(function(response) {
                    FutureProgressModal.show($scope, response, "Reprovisioning instance").then(() => {
                        $scope.$emit("instanceStateUpdated", instance.id);
                        $state.go("instances.instance.dashboard", {instanceId: instance.id});
                    });
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.deleteSnapshot = function(instance, snapshot) {
            const snapshotLabel = $scope.getSnapshotLabel(snapshot);
            WT1.event("fm-instance-snapshot-delete", {});
            Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the snapshot '" + snapshotLabel + "'?").then(() => {
                FMAPI.snapshots.startDelete(instance.id, snapshot.id).success(function(response) {
                    FutureProgressModal.show($scope, response, "Deleting snapshot", undefined, true, 'static', true).then((result) => {
                        $scope.refreshSnapshots();
                        if (result.statusMessages.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Snapshot deletion result", result.statusMessages);
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.refreshSnapshots();
    });
}());
