(function () {
    'use strict';

    const app = angular.module("dataiku.fm.loadbalancers", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs"]);

    const managedDNSHostnamePattern = /^[^.]*$/;
    const unmanagedDNSHostnamePattern = /^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9-]*[a-zA-Z0-9])\.)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9-]*[A-Za-z0-9])$/;

    function _isLoadbalancerValid(loadBalancer) {
        return loadBalancer
            && loadBalancer.name
            && loadBalancer.name.trim().length > 0
            && loadBalancer.nodes.length > 0
            && !loadBalancer.nodes.find(mapping => !mapping.instanceId);
    }

    app.controller("LoadBalancersBaseController", function ($scope, $rootScope, $state, FutureProgressModal, Dialogs, FMAPI, TaggingService) {
        $rootScope.activeProjectTagColor = TaggingService.getDefaultColor;

        $scope.refreshLoadBalancer = function(loadBalancerId) {
            //abstract function
        }

        $scope.refreshLoadBalancerState = function(lb) {
            lb.status = {};
            $scope.refreshPhysicalLoadbalancerStatus(lb);
            $scope.refreshLogicalLoadbalancerStatus(lb);
        }

        $scope.refreshPhysicalLoadbalancerStatus = function(lb) {
            FMAPI.loadbalancers.getPhysicalLoadBalancerStatus(lb.id, true).success(function (data) {
                lb.status.physical = data;
                if (!lb.status.logical.isActionInProgress) {
                    switch (lb.status.physical.physicalStatus) {
                        case "NEED_REPROVISIONING":
                            lb.status.class = 'load-balancer-row__state--warning';
                            lb.status.fmsectionclass = 'fm-section__header--warning';
                            lb.status.label = 'Needs reprovisioning';
                            break;
                        case "NEED_UPDATING":
                            lb.status.class = 'load-balancer-row__state--warning';
                            lb.status.fmsectionclass = 'fm-section__header--warning';
                            lb.status.label = 'Needs updating';
                            break;
                    }
                }
                $scope.physicalLoadBalancerStatus = data;
            }).error(setErrorInScope.bind($scope));
        }

        $scope.refreshLogicalLoadbalancerStatus = function(lb) {
            FMAPI.loadbalancers.getLogicalLoadBalancerStatus(lb.id).success(function (data) {
                lb.status.logical = data;

                switch (lb.status.logical.state) {
                    case "PROVISIONING":
                        lb.status.class = 'load-balancer-row__state--warning';
                        lb.status.fmsectionclass = 'fm-section__header--warning';
                        lb.status.label = 'Provisioning';
                        break;
                    case "DEPROVISIONING":
                        lb.status.class = 'load-balancer-row__state--warning';
                        lb.status.fmsectionclass = 'fm-section__header--warning';
                        lb.status.label = 'Deprovisioning';
                        break;
                    case "REPROVISIONING":
                        lb.status.class = 'load-balancer-row__state--warning';
                        lb.status.fmsectionclass = 'fm-section__header--warning';
                        lb.status.label = 'Reprovisioning';
                        break;
                    case "UPDATING":
                        lb.status.class = 'load-balancer-row__state--warning';
                        lb.status.fmsectionclass = 'fm-section__header--warning';
                        lb.status.label = 'Updating';
                        break;
                    case "DELETING":
                        lb.status.class = 'load-balancer-row__state--warning';
                        lb.status.fmsectionclass = 'fm-section__header--warning';
                        lb.status.label = 'Updating';
                        break;
                    case "NOT_PROVISIONED":
                        lb.status.class = 'load-balancer-row__state--notapplicable';
                        lb.status.fmsectionclass = 'fm-section__header--notapplicable';
                        lb.status.label = 'Not provisioned';
                        break;
                    case "PROVISIONED":
                        lb.status.class = 'load-balancer-row__state--nominal';
                        lb.status.fmsectionclass = 'fm-section__header--nominal';
                        lb.status.label = 'Provisioned';
                        break;
                    default:
                        lb.status.class = 'load-balancer-row__state--unknown';
                        lb.status.fmsectionclass = 'fm-section__header--unknown';
                        lb.status.label = 'Unknown';
                        break;
                }
            });
        }

        $scope.statusMoreInfo = function (status) {
            if (status.infoMessages) {
                Dialogs.infoMessagesDisplayOnly($scope, "Status info", status.infoMessages);
            }
        }

        $scope.startDeprovision = function (loadBalancer) {
            Dialogs.confirm($scope, "Confirm deprovision", "Are you sure you want to deprovision the load balancer " + loadBalancer.name + "? Once deleted you will not be able to recover it!").then(() => {
                FMAPI.loadbalancers.startDeprovision(loadBalancer.id).success(function (response) {
                    $scope.refreshLoadBalancer(loadBalancer.id);
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.startDelete = function (loadBalancer) {
            Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the load balancer " + loadBalancer.name + "? Once deleted you will not be able to recover it!").then(() => {
                FMAPI.loadbalancers.startDelete(loadBalancer.id).success(function (response) {
                    $state.go("loadbalancers.list", {});
                    $scope.$broadcast("loadBalancerDeleted", loadBalancer.id);
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.startProvision = function (loadBalancer) {
            FMAPI.loadbalancers.startProvision(loadBalancer.id).success(function (response) {
                $scope.refreshLoadBalancer(loadBalancer.id);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.startReprovision = function (loadBalancer) {
            Dialogs.confirm($scope, "Confirm Reprovisioning", "Reprovisioning consists of deprovisioning the load balancer before provisioning it again. Are you sure you want to reprovision the load balancer " + loadBalancer.name + "?").then(() => {
                FMAPI.loadbalancers.startReprovision(loadBalancer.id).success(function (response) {
                    $scope.refreshLoadBalancer(loadBalancer.id);
                }).error(setErrorInScope.bind($scope));
            });
        };

        $scope.startUpdate = function (loadBalancer) {
            Dialogs.confirm($scope, "Confirm Updating", "Update consists of synchronising the load balancer state in FM with the physical cloud load balancer. Are you sure you want to update the load balancer " + loadBalancer.name + "?").then(() => {
                FMAPI.loadbalancers.startUpdate(loadBalancer.id).success(function (response) {
                    $scope.refreshLoadBalancer(loadBalancer.id);
                }).error(setErrorInScope.bind($scope));
            });
        };
    });

    app.controller("LoadBalancersListController", function ($scope, $controller, FutureProgressModal, Dialogs, FMAPI) {
        $controller('LoadBalancersBaseController', { $scope: $scope });

        $scope.$on("futureCompleted", function (event, jobId, result) {
            if (result) {
                $scope.refreshLoadBalancerList();
            }
        });

        $scope.$on("loadBalancerDeleted", function (event, loadBalancerId) {
            $scope.loadBalancers = $scope.loadBalancers.filter(lb => lb.id !== loadBalancerId);
        });

        $scope.refreshLoadBalancer = function (loadBalancerId) {
            resetErrorInScope($scope);
            FMAPI.loadbalancers.get(loadBalancerId).success(function (refreshedLoadBalancer) {
                $scope.refreshLoadBalancerState(refreshedLoadBalancer)
                const index = $scope.loadBalancers.findIndex(lb => lb.id === refreshedLoadBalancer.id);
                if (index !== -1) {
                    $scope.loadBalancers[index] = refreshedLoadBalancer;
                }
            }).error(setErrorInScope.bind($scope));
        };

        $scope.refreshLoadBalancerList = function () {
            resetErrorInScope($scope);
            FMAPI.loadbalancers.list().success(function (data) {
                $scope.loadBalancers = data;
                $scope.loadBalancers.forEach($scope.refreshLoadBalancerState);
            }).error(setErrorInScope.bind($scope));
        };

        $scope.$on("loadBalancerUpdated", function (event, loadbalancer) {
            $scope.refreshLoadBalancerState(loadbalancer)
            const index = $scope.loadBalancers.findIndex(lb => lb.id === loadbalancer.id);
            if (index !== -1) {
                $scope.loadBalancers[index] = loadbalancer;
            }
        });
        // Initialize
        $scope.refreshLoadBalancerList();
    });

    app.controller("LoadBalancersCreationController", function ($scope, $state, FMAPI, LoadBalancersCreationDataService, WT1) {
        $scope.newLoadBalancer = { cloudTags: [], nodes: [], certificateMode: 'NO_CERTIFICATE', publicIpMode: 'DYNAMIC_PUBLIC_IP' };
        $scope.instances = [];
        FMAPI.instances.list().success(function (data) {
            $scope.instances = data;
        }).error(setErrorInScope.bind($scope));

        $scope.isValid = () => _isLoadbalancerValid($scope.newLoadBalancer);

        $scope.getDNSStrategy = function () {
            if ($scope.creationData && $scope.newLoadBalancer.virtualNetworkId) {
                return $scope.creationData.virtualNetworksMap[$scope.newLoadBalancer.virtualNetworkId].dnsStrategy;
            }
            return "";
        }


        $scope.getPublicDNSZone = function () {
            if ($scope.creationData && $scope.newLoadBalancer.virtualNetworkId) {
                return $scope.creationData.virtualNetworksMap[$scope.newLoadBalancer.virtualNetworkId].publicDnsZone;
            }
            return "";
        }

        $scope.getPrivateDNSZone = function () {
            if ($scope.creationData && $scope.newLoadBalancer.virtualNetworkId) {
                return $scope.creationData.virtualNetworksMap[$scope.newLoadBalancer.virtualNetworkId].privateDnsZone;
            }
            return "";
        }

        $scope.getNodes = function (hostname, actualInstanceId) {
            let multiNode = hostname && $scope.newLoadBalancer.nodes.filter(mapping => mapping.hostname === hostname).length > 1;
            return $scope.instances
                .filter(function (n) {
                    return !n.loadBalancerId
                        && n.virtualNetworkId == $scope.newLoadBalancer.virtualNetworkId
                        && (n.id === actualInstanceId || !$scope.newLoadBalancer.nodes.find(existingNode => existingNode.instanceId === n.id))
                        && (!multiNode || n.dssNodeType === 'automation');
                })
                .sort((a,b) => a.label.localeCompare(b.label));
        }

        $scope.getAWSCertificateModes = function () {
            var result = [];
            result.push({
                "mode": 'NO_CERTIFICATE',
                "label": 'No certificate',
            });
            result.push({
                "mode": 'AWS_ARN',
                "label": 'Certificate ARN',
            });
            if ($scope.getPublicDNSZone()) {
                result.push({
                    "mode": 'AWS_CERTIFICATE_MANAGER',
                    "label": 'Certificate manager',
                });
            }
            return result;
        }

        $scope.getAWSPublicIPModes = function () {
            var result = [];
            result.push({
                "mode": 'DYNAMIC_PUBLIC_IP',
                "label": 'Dynamic public IP',
            });
            if ($scope.getDNSStrategy() == "NONE" || $scope.getPrivateDNSZone()) {
                result.push({
                    "mode": 'NO_PUBLIC_IP',
                    "label": 'No public IP',
                });
            }
            return result;
        }

        LoadBalancersCreationDataService.loadCreationData($scope);

        $scope.$watch('newLoadBalancer.virtualNetworkId', function (newVal) {
            $scope.newLoadBalancer.nodes = [];
            $scope.certificateModes = $scope.getAWSCertificateModes();
            $scope.publicIpModes = $scope.getAWSPublicIPModes();

            const virtualNetworkDnsStrategy = $scope.getDNSStrategy();
            if (!virtualNetworkDnsStrategy || virtualNetworkDnsStrategy == "NONE") {
                $scope.nodeMappingHostnameValidationPattern = unmanagedDNSHostnamePattern;
            } else {
                $scope.nodeMappingHostnameValidationPattern = managedDNSHostnamePattern;
            }
        })
        $scope.$watch('creationData', function (newVal) {
            if ($scope.newLoadBalancer && $scope.newLoadBalancer.virtualNetworkId) {
                $scope.certificateModes = $scope.getAWSCertificateModes();
                $scope.publicIpModes = $scope.getAWSPublicIPModes();
            }
        })
        $scope.create = function () {
            WT1.event("fm-load-balancer-create", {});

            FMAPI.loadbalancers.create($scope.newLoadBalancer).success(function (newLoadBalancer) {
                $state.go("loadbalancers.loadbalancer.dashboard", { loadBalancerId: newLoadBalancer.id }, {});
            }).error(setErrorInScope.bind($scope));
        }
    });

    app.controller("LoadBalancersController", function ($scope, $stateParams, $q, $controller, FMAPI) {
        $scope.SharedState = $scope.SharedState || { loadbalancerFormIsValid: false, loadbalancerFormIsDirty: false };
        $controller('LoadBalancersBaseController', { $scope: $scope });

        $scope.refreshAction = function () {
            $scope.refreshLoadBalancer($stateParams.loadBalancerId);
        }

        $scope.refreshLoadBalancer = function (loadBalancerId) {
            resetErrorInScope($scope);
            $q.all([
                FMAPI.loadbalancers.get(loadBalancerId),
                FMAPI.instances.list()
            ]).then(([loadBalancer, instances]) => {
                $scope.loadBalancer = loadBalancer.data;
                $scope.loadBalancer.nodes.sort(
                    function(a, b) {
                        var x = a["hostname"];
                        var y = b["hostname"];
                        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
                    }
                );
                $scope.instances = instances.data;
                $scope.$broadcast("loadBalancerUpdated", $scope.loadBalancer);
            }, setErrorInScope.bind($scope));
        };

        $scope.$on("futureCompleted", function (event, jobId, result) {
            if (result) {
                $scope.refreshLoadBalancer($stateParams.loadBalancerId);
            }
        });

        $scope.$on("loadBalancerUpdated", function (event, loadBalancer) {
            $scope.loadBalancer = loadBalancer;
            $scope.refreshLoadBalancerState($scope.loadBalancer);
        });

        $scope.onSaveButtonClicked = function () {
            $scope.$broadcast("saveButtonClicked");
        };

        // Initialize screen & launch auto-refresh
        $scope.refreshLoadBalancer($stateParams.loadBalancerId);
    });

    app.controller("LoadBalancersDashboardController", function ($scope, $q, $stateParams, FMAPI) {

        $scope.$on("loadBalancerUpdated", function (event, loadBalancer) {
            $scope.loadBalancer = loadBalancer;
        });

        if ($scope.appConfig.cloud === 'AZURE') {
            // We need the tenandID to build the link to the VM
            FMAPI.tenant.getCloudCredentials().success(function (data) {
                $scope.cloudCredentials = data;
                $scope.azureTenantId = $scope.cloudCredentials.azureTenantId;
                $scope.azureSubscription = $scope.cloudCredentials.azureSubscription;
            }).error(setErrorInScope.bind($scope));
        }

        $scope.hasTarget = function (physicalLoadBalancerStatus, node) {
            return physicalLoadBalancerStatus.awsTargetGroupLinks.hasOwnProperty(node.instanceId);
        }

        $scope.hasRule = function (physicalLoadBalancerStatus, node) {
            return physicalLoadBalancerStatus.awsRoutingRuleLinks.hasOwnProperty(node.instanceId)
        }

        $scope.getInstance = function (instanceId) {
            return $scope.instances.find(i => i.id == instanceId);
        };
    });


    app.controller("LoadBalancersSettingsController", function ($scope, $stateParams, LoadBalancersCreationDataService, FMAPI) {
        FMAPI.loadbalancers.get($stateParams.loadBalancerId).success(function (lb) {
            // eslint-disable-next-line no-undef
            $scope.editedLoadBalancer = lb;
            LoadBalancersCreationDataService.loadCreationData($scope);

            const updateHostnamePatternValidation = function () {
                const virtualNetworkDnsStrategy = $scope.getDNSStrategy();
                if (!virtualNetworkDnsStrategy || virtualNetworkDnsStrategy == "NONE") {
                    $scope.nodeMappingHostnameValidationPattern = unmanagedDNSHostnamePattern;
                } else {
                    $scope.nodeMappingHostnameValidationPattern = managedDNSHostnamePattern;
                }
            };

            $scope.$watch('editedLoadBalancer.virtualNetworkId', function (newVal) {
                if ($scope.loadBalancer && $scope.loadBalancer.virtualNetworkId == $scope.editedLoadBalancer.virtualNetworkId) {
                    // eslint-disable-next-line no-undef
                    $scope.editedLoadBalancer.nodes = dkuDeepCopy($scope.loadBalancer.nodes, filterDollarKey);
                } else {
                    $scope.editedLoadBalancer.nodes = [];
                }
                $scope.certificateModes = $scope.getAWSCertificateModes();
                $scope.publicIpModes = $scope.getAWSPublicIPModes();

                updateHostnamePatternValidation();
            })
            $scope.$watch('creationData', function (newVal) {
                if ($scope.editedLoadBalancer && $scope.editedLoadBalancer.virtualNetworkId) {
                    $scope.certificateModes = $scope.getAWSCertificateModes();
                    $scope.publicIpModes = $scope.getAWSPublicIPModes();
                }

                updateHostnamePatternValidation();
            })
        }).error(setErrorInScope.bind($scope));

        $scope.getDNSStrategy = function () {
            if ($scope.creationData && $scope.editedLoadBalancer.virtualNetworkId) {
                return $scope.creationData.virtualNetworksMap[$scope.editedLoadBalancer.virtualNetworkId].dnsStrategy;
            }
            return "";
        }

        $scope.getPublicDNSZone = function () {
            if ($scope.creationData && $scope.editedLoadBalancer.virtualNetworkId) {
                return $scope.creationData.virtualNetworksMap[$scope.editedLoadBalancer.virtualNetworkId].publicDnsZone;
            }
            return "";
        }

        $scope.getPrivateDNSZone = function () {
            if ($scope.creationData && $scope.editedLoadBalancer.virtualNetworkId) {
                return $scope.creationData.virtualNetworksMap[$scope.editedLoadBalancer.virtualNetworkId].privateDnsZone;
            }
            return "";
        }

        $scope.getNodes = function (hostname, actualInstanceId) {
            let multiNode = hostname && $scope.editedLoadBalancer.nodes.filter(mapping => mapping.hostname === hostname).length > 1;
            return $scope.instances
                .filter(function (n) {
                    return n.virtualNetworkId == $scope.editedLoadBalancer.virtualNetworkId
                        && (!n.loadBalancerId || n.loadBalancerId == $scope.editedLoadBalancer.id)
                        && (n.id === actualInstanceId || !$scope.editedLoadBalancer.nodes.find(existingNode => existingNode.instanceId === n.id))
                        && (!multiNode || n.dssNodeType === 'automation');
                })
                .sort((a,b) => a.label.localeCompare(b.label));
        }

        $scope.getAWSCertificateModes = function () {
            var result = [];
            result.push({
                "mode": 'NO_CERTIFICATE',
                "label": 'No certificate',
            });
            result.push({
                "mode": 'AWS_ARN',
                "label": 'Certificate ARN',
            });
            if ($scope.getPublicDNSZone()) {
                result.push({
                    "mode": 'AWS_CERTIFICATE_MANAGER',
                    "label": 'Certificate manager',
                });
            }
            return result;
        }

        $scope.getAWSPublicIPModes = function () {
            var result = [];
            result.push({
                "mode": 'DYNAMIC_PUBLIC_IP',
                "label": 'Dynamic public IP',
            });
            if ($scope.getDNSStrategy() == "NONE" || $scope.getPrivateDNSZone()) {
                result.push({
                    "mode": 'NO_PUBLIC_IP',
                    "label": 'No public IP',
                });
            }
            return result;
        }

        $scope.$watch('loadbalancerSettingsForm.$valid', function (newVal) {
            $scope.SharedState.loadbalancerFormIsValid = newVal && _isLoadbalancerValid($scope.editedLoadBalancer);
        })

        $scope.$watch('loadbalancerSettingsForm.$dirty', function (newVal) {
            $scope.SharedState.loadbalancerFormIsDirty = newVal;
        })

        $scope.$watch("editedLoadBalancer", function (nv) {
            $scope.SharedState.loadbalancerFormIsValid = _isLoadbalancerValid($scope.editedLoadBalancer);
            if ($scope.loadBalancer && $scope.editedLoadBalancer && nv) {
                // eslint-disable-next-line no-undef
                let cleanEdited = dkuDeepCopy($scope.editedLoadBalancer, filterDollarKey);
                // eslint-disable-next-line no-undef
                let originalLoadBalancer = dkuDeepCopy($scope.loadBalancer, filterDollarKey);
                delete originalLoadBalancer.status;
                $scope.SharedState.loadbalancerFormIsDirty = !angular.equals(cleanEdited, originalLoadBalancer);
            }
        }, true);

        $scope.$on("saveButtonClicked", function () {
            FMAPI.loadbalancers.save($scope.editedLoadBalancer).success(function (saveResult) {
                resetErrorInScope($scope);
                // eslint-disable-next-line no-undef
                $scope.loadBalancer = dkuDeepCopy($scope.editedLoadBalancer, filterDollarKey);
                $scope.$emit('loadBalancerUpdated', $scope.loadBalancer);
                $scope.SharedState.loadbalancerFormIsDirty = false;
            }).error(setErrorInScope.bind($scope));
        });
    });

    app.service("LoadBalancersCreationDataService", function (FMAPI, Logger) {
        const svc = {};
        svc.loadCreationData = function (targetScope) {
            FMAPI.loadbalancers.getCreationData().success(function (data) {
                Logger.info("with static data", data);
                targetScope.creationData = data;
                targetScope.creationData.virtualNetworksMap = {};
                data.virtualNetworks.forEach((vn) => {
                    targetScope.creationData.virtualNetworksMap[vn.id] = vn;
                });
            }).error(setErrorInScope.bind(targetScope))
        };
        return svc;
    });

}());
