(function() {
'use strict';

const app = angular.module('dataiku.fm.api', []);

const services = angular.module("dataiku.services");

services.run(function initMonkeyPatchPromiseWithSuccessError($http) {
    services.addSuccessErrorToPromise = function(promise) {
        promise.success = function(callback) {
            promise.then((resp) => {
                return callback(resp.data, resp.status, resp.headers, resp.config, resp.statusText, resp.xhrStatus)
            });
            return promise;
        };

        promise.error = function(callback) {
            promise.then(null, resp => callback(resp.data, resp.status, resp.headers, resp.config, resp.statusText, resp.xhrStatus));
            return promise;
        };
        return promise;
    };

    function monkeyPatchHttpGetForSuccessError() {
        const get = $http.get;
        $http.get = function() {
            let promise = get.apply(this, arguments);
            const decoratedPromise = services.addSuccessErrorToPromise(promise);
            return decoratedPromise;
        }
    }
    monkeyPatchHttpGetForSuccessError();
});

app.factory("FMAPI", ["APIXHRService", '$q', '$rootScope', 'Logger', '$$cookieReader', function(APIXHRService) {
    var API_PATH = '/api/ui/';

return {
core: {
    getConfiguration: function(){
        return APIXHRService("GET", API_PATH + "get-configuration");
    }
},
registration: {
    installLicense: function(license){
        return APIXHRService("POST", API_PATH + "install-license", {
            license: license
        });
    }
},
auth: {
    login: function(tenantId, login, password) {
        return APIXHRService("POST", API_PATH + "login", {
            tenantId: tenantId,
            login: login,
            password: password
        });
    },
    logout: function(){
        return APIXHRService("POST", API_PATH + "logout")
    },
    getSAMLRedirectURL : function(tenantId){
        return APIXHRService("POST", API_PATH + "get-saml-redirect-url", {tenantId: tenantId});
    },
    getOpenIDRedirectURL : function(){
        return APIXHRService("GET", API_PATH + "get-openid-redirect-url", {dssUrl: urlWithProtocolAndHost()});
    },
    exchangeAuthorizationCode : function(code, state, tenantId) {
        return APIXHRService("POST", API_PATH + "openid-callback", {code: code, state: state, tenantId: tenantId, dssUrl: urlWithProtocolAndHost()});
    }
},

tenant: {
    getCloudCredentials: function() {
        return APIXHRService("GET", API_PATH + "tenant/get-cloud-credentials", {})
    },
    setCloudCredentials: function(cloudCredentials) {
        return APIXHRService("POST", API_PATH + "tenant/set-cloud-credentials", {
            cloudCredentials: JSON.stringify(cloudCredentials)
        })
    },
    uploadCertificate: function(cloudCredentials) {
        return APIXHRService("POST", API_PATH + "tenant/set-certificate", {
            cloudCredentials: JSON.stringify(cloudCredentials)
        })
    },
    getCloudTags: function() {
        return APIXHRService("GET", API_PATH + "tenant/get-cloud-tags", {})
    },
    setCloudTags: function(cloudTags) {
        return APIXHRService("POST", API_PATH + "tenant/set-cloud-tags", {
            cloudTags: angular.toJson(cloudTags)
        })
    },
    refreshLicense: function(){
        return APIXHRService("POST", API_PATH + "tenant/refresh-license");
    },

    getGlobalLicensingSummary: function() {
        return APIXHRService("GET", API_PATH + "tenant/get-licensing-summary");
    },
    getGlobalLicensingStatus: function() {
        return APIXHRService("GET", API_PATH + "tenant/get-licensing-status");
    },

    sublicenses: {
        list: function() {
            return APIXHRService("GET", API_PATH + "tenant/sublicenses/list");
        },
        get: function(sublicenseId) {
            return APIXHRService("GET", API_PATH + "tenant/sublicenses/get", {sublicenseId});
        },
        create: function(sublicense) {
            return APIXHRService("POST", API_PATH + "tenant/sublicenses/create", {
                sublicense: JSON.stringify(sublicense)
            });
        },
        save: function(sublicense) {
             return APIXHRService("POST", API_PATH + "tenant/sublicenses/save", {
                sublicense: JSON.stringify(sublicense)
            });
        },
        delete: function(sublicenseId) {
             return APIXHRService("DELETE", API_PATH + "tenant/sublicenses/delete?sublicenseId=" + sublicenseId);
        }
    },

    getProxySettings: function() {
        return APIXHRService("GET", API_PATH + "tenant/get-proxy-settings", {});
    },
    setProxySettings: function(proxySettings) {
        return APIXHRService("PUT", API_PATH + "tenant/set-proxy-settings", {
            proxySettings: angular.toJson(proxySettings)
        });
    }
},

accounts: {
        list: function() {
            return APIXHRService("GET", API_PATH + "accounts/list");
        },
        create: function(account) {
            return APIXHRService("POST", API_PATH + "accounts/create", {
                account: JSON.stringify(account)
            });
        },
        get: function(accountId) {
            return APIXHRService("GET", API_PATH + "accounts/get", {
                accountId: accountId
            });
        },
        save: function(account) {
            return APIXHRService("POST", API_PATH + "accounts/save", {
                account: JSON.stringify(account)
            });
        },
        delete: function(accountId) {
            return APIXHRService("POST", API_PATH + "accounts/delete", {
                accountId: accountId
            });
        },
        getCreationData: function() {
            return APIXHRService("GET", API_PATH + "accounts/get-creation-data");
        },
    },

virtualNetworks: {
    list: function() {
        return APIXHRService("GET", API_PATH + "virtual-networks/list");
    },
    startCreate: function(virtualNetwork) {
        return APIXHRService("POST", API_PATH + "virtual-networks/start-create", {
            virtualNetwork: JSON.stringify(virtualNetwork)
        });
    },
    get: function(virtualNetworkId) {
        return APIXHRService("GET", API_PATH + "virtual-networks/get", {
            virtualNetworkId: virtualNetworkId
        });
    },
    getCloudAccount: function(virtualNetworkId) {
        return APIXHRService("GET", API_PATH + "virtual-networks/cloud-account/get", {
            virtualNetworkId: virtualNetworkId
        });
    },

    save: function(virtualNetwork) {
        return APIXHRService("POST", API_PATH + "virtual-networks/save", {
            virtualNetwork: JSON.stringify(virtualNetwork)
        });
    },
    startDelete: function(virtualNetworkId, options) {
        return APIXHRService("POST", API_PATH + "virtual-networks/start-delete", {
            virtualNetworkId: virtualNetworkId,
            deleteAutogeneratedSecurityGroups: options.deleteAutogeneratedSecurityGroups,
            deleteAutogeneratedPeering: options.deleteAutogeneratedPeering
        });
    },
    getDefaultValues: function() {
        return APIXHRService("GET", API_PATH + "virtual-networks/get-default-values", {});
    },
},

instances: {
    list: function() {
        return APIXHRService("GET", API_PATH + "instances/list");
    },
    create: function(instance) {
        return APIXHRService("POST", API_PATH + "instances/create", {
            instance: JSON.stringify(instance)
        });
    },
    getCreationData: function() {
        return APIXHRService("GET", API_PATH + "instances/get-creation-data");
    },
    startReprovision: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-reprovision", {
            instanceId: instanceId
        });
    },
    startRestartDSS: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-restart-dss", {
            instanceId: instanceId
        });
    },
    startDeprovision: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-deprovision", {
            instanceId: instanceId
        });
    },
    startDelete: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-delete", {
            instanceId: instanceId
        });
    },
    refreshLoadBalancer: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/refresh-load-balancer", {
            instanceId: instanceId
        });
    },
    startPhysicalStart: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-physical-start", {
            instanceId: instanceId
        });
    },
    startPhysicalStop: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-physical-stop", {
            instanceId: instanceId
        });
    },
    startPhysicalReboot: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/start-physical-reboot", {
            instanceId: instanceId
        });
    },
    fetchAgentLogs: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/fetch-agent-logs", {
            instanceId: instanceId
        });
    },
    readAgentLogs: function(instanceId, nbLines) {
        return APIXHRService("GET", API_PATH + "instances/read-agent-logs", {
            instanceId: instanceId,
            nbLines: nbLines
        });
    },
    get: function(instanceId) {
        return APIXHRService("GET", API_PATH + "instances/get", {
            instanceId: instanceId
        });
    },
    getInitialAdminPassword: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/get-initial-admin-password", {
            instanceId: instanceId
        });
    },
    resetUserPassword: function(instanceId, username, password) {
        return APIXHRService("POST", API_PATH + "instances/reset-user-password", {
            instanceId: instanceId,
            username: username,
            password: password
        });
    },
    replaySetupActions: function(instanceId) {
        return APIXHRService("POST", API_PATH + "instances/replay-setup-actions", {
            instanceId: instanceId
        });
    },
    getPhysicalInstanceStatus: function(instanceId, noSpinner) {
        return APIXHRService("GET", API_PATH + "instances/get-physical-instance-status", {
            instanceId: instanceId
        }, noSpinner ? 'nospinner' : undefined);
    },
    save: function(instance) {
        return APIXHRService("POST", API_PATH + "instances/save", {
            instance: JSON.stringify(instance)
        });
    },
    getEventsLog: function(instanceId) {
        return APIXHRService("GET", API_PATH + "instances/get-events-log", {
            instanceId: instanceId
        });
    },
    disableUser: function(username, instanceId=[]) {
         return APIXHRService("GET", API_PATH + "instances/disable-user", {username, instanceId});
    }
},

loadbalancers: {
    list: function() {
        return APIXHRService("GET", API_PATH + "load-balancers/list");
    },

    create: function(loadBalancer) {
        return APIXHRService("POST", API_PATH + "load-balancers/create", {
            loadBalancer: JSON.stringify(loadBalancer)
        });
    },
    getCreationData: function() {
        return APIXHRService("GET", API_PATH + "load-balancers/get-creation-data");
    },
    get: function(loadBalancerId) {
        return APIXHRService("GET", API_PATH + "load-balancers/" + loadBalancerId);
    },
    getPhysicalLoadBalancerStatus: function(loadBalancerId, noSpinner) {
        return APIXHRService("GET", API_PATH + "load-balancers/" + loadBalancerId + "/physical/status", {}, noSpinner ? 'nospinner' : undefined);
    },
    getLogicalLoadBalancerStatus: function(loadBalancerId, noSpinner) {
        return APIXHRService("GET", API_PATH + "load-balancers/" + loadBalancerId + "/status", {
            loadBalancerId: loadBalancerId
        }, noSpinner ? 'nospinner' : undefined);
    },
    startDelete: function(loadBalancerId) {
        return APIXHRService("DELETE", API_PATH + "load-balancers/" + loadBalancerId);
    },
    startProvision: function(loadBalancerId) {
        return APIXHRService("POST", API_PATH + "load-balancers/" + loadBalancerId + "/start-provision");
    },
    startReprovision: function(loadBalancerId) {
        return APIXHRService("POST", API_PATH + "load-balancers/" + loadBalancerId + "/start-reprovision");
    },
    startDeprovision: function(loadBalancerId) {
        return APIXHRService("POST", API_PATH + "load-balancers/" + loadBalancerId + "/start-deprovision");
    },
    startUpdate: function(loadBalancerId) {
        return APIXHRService("POST", API_PATH + "load-balancers/" + loadBalancerId + "/start-update");
    },
    save: function(loadbalancer) {
        return APIXHRService("PUT", API_PATH + "load-balancers/", {
            loadBalancer: JSON.stringify(loadbalancer)
        });
    },
},

instancesettingstemplates: {
    list: function() {
        return APIXHRService("GET", API_PATH + "instance-settings-templates/list");
    },
    create: function(instancesettingstemplate) {
        return APIXHRService("POST", API_PATH + "instance-settings-templates/create", {
            instancesettingstemplate: JSON.stringify(instancesettingstemplate)
        });
    },
    get: function(instanceSettingsTemplateId) {
        return APIXHRService("GET", API_PATH + "instance-settings-templates/get", {
            instancesettingstemplateId: instanceSettingsTemplateId
        });
    },
    save: function(instancesettingstemplate) {
        return APIXHRService("POST", API_PATH + "instance-settings-templates/save", {
            instancesettingstemplate: JSON.stringify(instancesettingstemplate)
        });
    },
    delete: function(instanceSettingsTemplateId) {
        return APIXHRService("POST", API_PATH + "instance-settings-templates/delete", {
            instancesettingstemplateId: instanceSettingsTemplateId
        });
    }
},

snapshots: {
    list: function(instanceId) {
        return APIXHRService("GET", API_PATH + "snapshots/list", {
            instanceId: instanceId
        });
    },
    create: function(instanceId, reasonForSnapshot) {
        return APIXHRService("POST", API_PATH + "snapshots/create", {
            instanceId: instanceId,
            reasonForSnapshot: reasonForSnapshot
        })
    },
    reprovisionInstance: function(instanceId, snapshotId) {
        return APIXHRService("GET", API_PATH + "snapshots/reprovision-instance", {
            instanceId: instanceId,
            snapshotId: snapshotId
        });
    },
    startDelete: function(instanceId, snapshotId) {
        return APIXHRService("POST", API_PATH + "snapshots/start-delete", {
            instanceId: instanceId,
            snapshotId: snapshotId
        });
    },
    startDeletes: function(instanceId, snapshots) {
        return APIXHRService("POST", API_PATH + "snapshots/start-deletes", {
            instanceId: instanceId,
            snapshots: JSON.stringify(snapshots)
        });
    }
},

blueprints: {
    getNetworks: function() {
        return APIXHRService("GET", API_PATH + "blueprints/get-networks");
    },
    deploy: function(blueprintRequest) {
        return APIXHRService("POST", API_PATH + "blueprints/deploy", {
            "blueprintRequest": JSON.stringify(blueprintRequest)
        })
    }
},

users: {
    list: function() {
        return APIXHRService("GET", API_PATH + "users/list");
    },
    create: function(user) {
        return APIXHRService("POST", API_PATH + "users/create", {
            user: JSON.stringify(user)
        });
    },
    get: function(emailAddress) {
        return APIXHRService("GET", API_PATH + "users/get", {
            emailAddress: emailAddress
        });
    },
    save: function(emailAddress, user) {
        return APIXHRService("POST", API_PATH + "users/save", {
            emailAddress: emailAddress,
            user: JSON.stringify(user)
        });
    },
    delete: function(emailAddress) {
        return APIXHRService("POST", API_PATH + "users/delete", {
            emailAddress: emailAddress
        });
    }
},

maintenance: {
    getDiagnoses: function() {
        return APIXHRService("GET", API_PATH + "maintenance/diagnoses");
    },
    startDiagnosis: function() {
        return APIXHRService("POST", API_PATH + "maintenance/diagnoses");
    },
    deleteDiagnosis: function(diagnosis) {
        return APIXHRService("DELETE", API_PATH + "maintenance/diagnoses/" + diagnosis.name);
    },
},

futures: {
    getUpdate: function(futureId) {
        return APIXHRService("GET", API_PATH + "futures/get-update", {
            futureId: futureId
        }, "nospinner");
    },
    peekUpdate: function(futureId) {
        return APIXHRService("GET", API_PATH + "futures/peek-update", {
            futureId: futureId
        }, "nospinner");
    },
    abort: function(futureId) {
        return APIXHRService("POST", API_PATH + "futures/abort", {
            futureId: futureId
        });
    }
},

security: {
    getSettings: function() {
        return APIXHRService("GET", API_PATH + "security/settings");
    },
    saveSettings: function(securitySettings) {
        return APIXHRService("PUT", API_PATH + "security/settings", { securitySettings: JSON.stringify(securitySettings) });
    },
    testLdapSettings: function(ldapSettings) {
        return APIXHRService("POST", API_PATH + "security/test-ldap-settings", { data: JSON.stringify(ldapSettings) });
    },
    testAzureADSettings: function(data) {
        return APIXHRService("POST", API_PATH + "security/test-azure-ad-settings", { data: JSON.stringify(data) });
    },
    testLdapGetUserDetails: function(data) {
        return APIXHRService("POST", API_PATH + "security/test-ldap-get-user-details", {data: JSON.stringify(data)});
    },
},

sso: {
    fetchOpenIDConfig: function(wellKnownURL) {
        return APIXHRService("GET", API_PATH + "sso/fetch-openid-config", { wellKnownURL: wellKnownURL });
    },
}

};
}]);
})();
