(function(){
'use strict';

var app = angular.module('dataiku.widgets.futures', ['dataiku.services']);

/**
 * A simple helper to watch a future state.
 *
 *  - Does not auto-abort the future when going out of scope !
 *
 * call with FutureWatcher.watchJobId(jobId)
 */
app.service("FutureWatcher", function($q, FMAPI){

    function enrichPromise(deferred) {
        deferred.promise.success = function(fn) {
            deferred.promise.then(function(data) {
                fn(data.data, data.status, data.headers);
            });
            return deferred.promise;
        };

        deferred.promise.error = function(fn) {
            deferred.promise.then(null, function(data) {
                fn(data.data, data.status, data.headers);
            });
            return deferred.promise;
        };

        deferred.promise.update = function(fn) {
            deferred.promise.then(null, null,function(data) {
                fn(data.data, data.status, data.headers);
            });
            return deferred.promise;
        };
    }

    var FutureWatcher = {
        watchJobId : function(jobId) {
            var hasWaitedFor = 0;
            var delayBetweenCalls = 500;
            var deferred = $q.defer();
            enrichPromise(deferred);

            function refresh(){
                FMAPI.futures.getUpdate(jobId).success(function(data,status,headers) {
                    hasWaitedFor += delayBetweenCalls;
                    var kwargs = {data:data, status:status, headers:headers};
                    if (data.hasResult || data.unknown) {
                        deferred.resolve(kwargs);
                    } else {
                        if (hasWaitedFor > 300000) { // 5min
                            delayBetweenCalls = 10000;
                        } else if (hasWaitedFor > 120000) { // 2min
                            delayBetweenCalls = 3000;
                        } else if (hasWaitedFor > 30000) { // 30s
                            delayBetweenCalls = 1000;
                        }
                        deferred.notify(kwargs);
                        window.setTimeout(refresh, delayBetweenCalls);
                    }
                }).error(function(data, status, headers){
                    var kwargs = {data:data, status:status, headers:headers};
                    deferred.reject(kwargs);
                });
            }

            refresh();

            return deferred.promise;
        },
        watchPeekJobId : function(jobId) {
            var hasWaitedFor = 0;
            var delayBetweenCalls = 500;
            var deferred = $q.defer();
            enrichPromise(deferred);

            function refresh(){
                FMAPI.futures.peekUpdate(jobId).success(function(data,status,headers) {
                    hasWaitedFor += delayBetweenCalls;
                    var kwargs = {data:data, status:status, headers:headers};
                    if (data.hasResult || data.unknown) {
                        deferred.resolve(kwargs);
                    } else {
                        if (hasWaitedFor > 300000) { // 5min
                            delayBetweenCalls = 10000;
                        } else if (hasWaitedFor > 120000) { // 2min
                            delayBetweenCalls = 3000;
                        } else if (hasWaitedFor > 30000) { // 30s
                            delayBetweenCalls = 1000;
                        }
                        deferred.notify(kwargs);
                        window.setTimeout(refresh, delayBetweenCalls);
                    }
                }).error(function(data, status, headers){
                    var kwargs = {data:data, status:status, headers:headers};
                    deferred.reject(kwargs);
                });
            }
            refresh();
            return deferred.promise;
        }
    }
    return FutureWatcher;
});

app.service("FutureProgressModal", function(FutureWatcher, $q, CreateModalFromTemplate, ProgressStackMessageBuilder, FMAPI, $timeout, Logger){
    var FutureProgressModal = {
        // Returns a promise that resolves when the future is done but never rejects
        show : function(parentScope, initialResponse, modalTitle, afterCompileCallback, noFocus, backdrop, keyboard) {

            var deferred = $q.defer();
            if ( initialResponse.hasResult ) {
                // Keep the last log if any
                if (angular.isObject(initialResponse.result) && initialResponse.log) {
                    initialResponse.result.futureLog = initialResponse.log;
                }
                deferred.resolve(initialResponse.result);
            } else {
                CreateModalFromTemplate("/templates/widgets/future-progress-modal.html", parentScope, null, function(newScope){
                    if (afterCompileCallback) {
                        afterCompileCallback(newScope);
                    }
                    newScope.futureResponse = initialResponse;
                    newScope.modalTitle = modalTitle;
                    newScope.percentage = 0;                    
                    newScope.abort = function () {
                        FMAPI.futures.abort(initialResponse.jobId).error(setErrorInScope.bind(newScope));
                    };
                    
                    FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(function(data) {
                        newScope.finalResponse = data;
                        newScope.futureResponse = null;
                        newScope.dismiss();
                        if (angular.isObject(data.result) && data.log) {
                            data.result.futureLog = data.log;
                        }
                        deferred.resolve(data.result);
                    }).update(function(data){
                        newScope.percentage =  ProgressStackMessageBuilder.getPercentage(data.progress);
                        newScope.futureResponse = data;
                        newScope.stateLabels = ProgressStackMessageBuilder.build(newScope.futureResponse.progress, true);
                    }).error(function(data, status, headers) {
                        Logger.info(data)
                        // Remove the future response to remove progress bar / state
                        newScope.futureResponse = null;
                        // Keep the failure log
                        newScope.failureLog = data.logTail;
                        deferred.reject({data: data, status: status, headers: headers});
                        setErrorInScope.bind(newScope)(data, status, headers);
                    });
                }, noFocus, backdrop, keyboard);
            }

            return deferred.promise;
        },

        /* Shows the modal only if the job is still running.
         * Does not take the job from the future service
         * The modal disappears automatically if the job succeeds.
         * Returns a promise that resolves when the future succeeds
         */
        showPeekOnlyIfRunning : function(parentScope, jobId, modalTitle){
            var newScope = parentScope.$new();
            var shown = false;
            var theModalScope = null;

            var deferred = $q.defer();

            function showIfNeeded(){
                if (!shown) {
                    shown = true;
                    CreateModalFromTemplate("/templates/widgets/future-progress-modal.html", newScope, null, function(modalScope){
                        theModalScope = modalScope;
                        modalScope.modalTitle = modalTitle;

                        newScope.abort = function() {
                            FMAPI.futures.abort(jobId)
                                .success(modalScope.dismiss)
                                .error(setErrorInScope.bind(modalScope));
                        };
                    });
                }
            }

            FutureWatcher.watchPeekJobId(jobId)
            .success(function() {
                newScope.futureResponse = null;
                if (theModalScope) theModalScope.dismiss();
                deferred.resolve();
            }).update(function(data){
                if (data.alive == false) {
                    if (theModalScope) theModalScope.dismiss();
                    deferred.resolve();
                } else {
                    showIfNeeded();
                    newScope.percentage =  ProgressStackMessageBuilder.getPercentage(data.progress);
                    newScope.futureResponse = data;
                    newScope.stateLabels = ProgressStackMessageBuilder.build(newScope.futureResponse.progress, true);
                }
            }).error(function(data, status, headers) {
                newScope.futureResponse = null;
                setErrorInScope.bind(newScope)(data, status, headers);
                deferred.reject({data: data, status: status, headers: headers});
            });

            return deferred.promise;
        },
        
        reopenableModal : function(parentScope, initialResponse, modalTitle) {
            var handle = {shown:false};
            var deferred = $q.defer();
            if ( initialResponse.hasResult ) {
                deferred.resolve(initialResponse.result);
            } else {
                var hooks = {};
                handle.open = function() {
                    if (handle.shown) return;
                    handle.shown = true;
                    CreateModalFromTemplate("/templates/widgets/future-progress-modal.html", parentScope, null, function(newScope){
                        if (hooks.isDone) {
                            // too slow to open
                            $timeout(function() {newScope.dismiss();}); // because dismiss() isn't even in the scope at this point
                            return;
                        }
                        newScope.futureResponse = initialResponse;
                        newScope.modalTitle = modalTitle;
                        newScope.percentage = 0;
                        
                        newScope.abort = function() {
                            FMAPI.futures.abort(initialResponse.jobId).error(setErrorInScope.bind(newScope));
                        }
                        // react to changes in the future state
                        hooks.update = function(data) {
                            newScope.percentage =  ProgressStackMessageBuilder.getPercentage(data.progress);
                            newScope.futureResponse = data;
                            newScope.stateLabels = ProgressStackMessageBuilder.build(newScope.futureResponse.progress, true);
                        };
                        hooks.success = function(data) {
                            newScope.finalResponse = data;
                            newScope.futureResponse = null;
                            newScope.dismiss();
                        };
                        hooks.error = function(data, status, headers) {
                            newScope.futureResponse = null;
                            setErrorInScope.bind(newScope)(data, status, headers);
                        };
                        newScope.$on("$destroy", function() {
                            // stop listening on the changes to the future
                            delete hooks.success;
                            delete hooks.update;
                            delete hooks.error;
                            handle.shown = false;
                        });
                    });
                };
                FutureWatcher.watchJobId(initialResponse.jobId)
                .success(function(data) {
                    hooks.isDone = true;
                    if (hooks.success) {
                        hooks.success(data);
                    }
                    deferred.resolve(data.result);
                }).update(function(data){
                    if (hooks.update) {
                        hooks.update(data);
                    }
                }).error(function(data, status, headers) {
                    hooks.isDone = true;
                    if (hooks.error) {
                        hooks.error(data, status, headers);
                    }
                    deferred.reject({data: data, status: status, headers: headers});
                });
            }
            handle.promise = deferred.promise; 
            return handle;
        }
    }
    return FutureProgressModal;
});


app.component("jobProgress", {
    templateUrl: 'common/edited/widgets/future-progress.html',
    bindings: {
        job: '<',
        short: '<',
    },

    controller: function($scope, $q, FMAPI, FutureWatcher, ProgressStackMessageBuilder, Logger) {
        const $ctrl = this;
        
        $ctrl.$onInit = function() {
            $scope.short = $ctrl.short;
            $scope.job = $ctrl.job;
            var deferred = $q.defer();
            $scope.futureResponse = $ctrl.job;
            $scope.progressBarClass = "progress"

            if ($ctrl.job) {
                $scope.error = $scope.futureResponse.exception || ($scope.futureResponse.hasResult && !$scope.futureResponse.result.success);

                if ($ctrl.job.alive || $scope.error) {

                    $scope.showJob = true;
                    FMAPI.futures.getUpdate($ctrl.job.jobId).success(function(data, status, headers) {
                        $scope.futureResponse = data;
                        $scope.percentage = 0;
                        var type = data.type
                        var action = data.action
                        $scope.error = false;

                        FutureWatcher.watchJobId(data.jobId)
                            .success(function(data) {
                                $scope.futureResponse = data;
                                deferred.resolve(data.result);
                                $scope.percentage = 100;
                                $scope.error = $scope.futureResponse.exception || ($scope.futureResponse.hasResult && !$scope.futureResponse.result.success);
                                if (!$scope.error) {
                                    $scope.progressBarClass = "progress-success"
                                    $scope.showJob = false;
                                } else {
                                    $scope.progressBarClass = "progress-danger"
                                }
                                $scope.$emit('futureCompleted', $ctrl.job.jobId, !$scope.error);
                            }).update(function(data){
                                $scope.error = false;
                                $scope.percentage =  ProgressStackMessageBuilder.getPercentage(data.progress);
                                $scope.futureResponse = data;
                                $scope.futureResponse.type = type;
                                $scope.futureResponse.action = action;
                                $scope.progressBarClass = "progress-striped"

                                $scope.stateLabels = ProgressStackMessageBuilder.build($scope.futureResponse.progress, false);
                            }).error(function(data, status, headers) {
                                Logger.info(data)
                                $scope.percentage = 100;
                                $scope.error = true;
                                $scope.progressBarClass = "progress-danger"
                                // Keep the failure log
                                $scope.failureLog = data.logTail;
                                deferred.reject({data: data, status: status, headers: headers});
                                setErrorInScope.bind($scope)(data, status, headers);
                                $scope.$emit('futureCompleted', $ctrl.job.jobId, false);
                            });
                    }).error(function(data, status, headers){
                        $scope.error = true;
                        var kwargs = {data:data, status:status, headers:headers};
                        deferred.reject(kwargs);
                        $scope.percentage = 100;
                        $scope.progressBarClass = "progress-danger"
                        setErrorInScope.bind($scope)(data, status, headers);
                    }); 
                } else {
                    $scope.percentage = 100;
                    $scope.error = $scope.futureResponse.exception || ($scope.futureResponse.hasResult && !$scope.futureResponse.result.success);
                    if (!$scope.error) {
                        $scope.progressBarClass = "progress-success"
                        $scope.showJob = false;
                    } else {
                        $scope.showJob = true;
                        setErrorInScope.bind($scope)($scope.futureResponse.exception, 500, function(key) {
                            if (key === 'Content-Type') {
                                return 'application/json';
                            }
                            return '';
                        });
                        $scope.progressBarClass = "progress-danger"
                    }
                }
            } else {
                $scope.showJob = false;
            }
            
        }

        $scope.alertClasses = {
            SUCCESS: 'alert-success',
            FATAL: 'alert-danger',
            ERROR: 'alert-danger',
            WARNING: 'alert-warning',
            INFO: 'alert-info'
        }
        $scope.getTitle = function() {
            switch ($scope.futureResponse.type) {
                case 'LOAD_BALANCER':
                    switch ($scope.futureResponse.action) {
                        case 'DELETE':
                            return 'Deleting Load Balancer';
                        case 'DEPROVISION':
                            return 'Deprovisioning Load Balancer';
                        case 'PROVISION':
                            return 'Provisioning Load Balancer';
                        case 'REPROVISION':
                            return 'Reprovisioning Load Balancer';
                        case 'UPDATE':
                            return 'Updating Load Balancer';
                        default:
                            return 'Unknown Action for Load Balancer';
                    }
                default:
                    return 'Unknown Source Type';
            }
        };
    }
});

})();
