(function() {
'use strict';

const app = angular.module('dataiku.admin.codeenvs.common', ['dataiku.logs']);

app.directive('codeEnvSecurityPermissions', function(DataikuAPI, $rootScope, PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-envs/common/security-permissions.html',
        replace : true,
        scope : {
                codeEnv  : '='
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.ui = {};

            function makeNewPerm(){
                $scope.newPerm = {
                    update: true,
                    delete: true,
                    use: true
                }
            }
            makeNewPerm();

            const fixupPermissions = function() {
                if (!$scope.codeEnv) return;
                /* Handle implied permissions */
                $scope.codeEnv.permissions.forEach(function(p) {
                    p.$updateDisabled = false;
                    p.$manageUsersDisabled = false;
                    p.$useDisabled = false;
                    if ($scope.codeEnv.usableByAll) {
                        p.$useDisabled = true;
                    }
                    if (p.update) {
                        p.$useDisabled = true;
                    }
                    if (p.manageUsers) {
                        p.$useDisabled = true;
                        p.$updateDisabled = true;
                    }
                });
            };
            DataikuAPI.security.listGroups(false).success(function(allGroups) {
                if (allGroups) {
                    allGroups.sort();
                }
                $scope.allGroups = allGroups;
                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = data.map(user => '@' + user.login);
                }).error(setErrorInScope.bind($scope));
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("codeEnv.owner", function() {
                $scope.ui.ownerLogin = $scope.codeEnv.owner;
            });
            $scope.addPermission = function() {
                $scope.codeEnv.permissions.push($scope.newPerm);
                makeNewPerm();
            };

            $scope.$watch("codeEnv.usableByAll", function(nv, ov) {
                fixupPermissions();
            })
            $scope.$watch("codeEnv.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
                fixupPermissions();
            }, true)
            $scope.$watch("codeEnv.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
                fixupPermissions();
            }, false)
            $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.codeEnv, $scope.allGroups);
            fixupPermissions();

            // Ownership mgmt
            $scope.$watch("ui.ownerLogin", function() {
                PermissionsService.transferOwnership($scope, ($scope.codeEnv || {}).desc, "code env");
            });
        }
    };
});

app.directive('codeEnvContainers', function (DataikuAPI, $rootScope) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/code-envs/common/code-env-containers.html',
        replace: true,
        scope: {
            codeEnv: '=',
            envLang: '=',
            deploymentMode: '='
        },
        controller: function ($scope) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;

            DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings().then(({data}) => {
                $scope.enableCodeEnvResources = data.enableCodeEnvResources;
            });

            let _mode = "NONE";
            if ($scope.codeEnv.allContainerConfs) {
                _mode = "ALL"
            } else if (!$scope.codeEnv.allContainerConfs && $scope.codeEnv.containerConfs.length !== 0) {
                _mode = "ALLOWED";
            }
            let _sparkMode = "NONE";
            if ($scope.codeEnv.allSparkKubernetesConfs) {
                _sparkMode = "ALL";
            } else if (!$scope.codeEnv.allSparkKubernetesConfs && $scope.codeEnv.sparkKubernetesConfs.length !== 0) {
                _sparkMode = "ALLOWED";
            }
            let _codeStudioMode = $scope.codeEnv.rebuildDependentCodeStudioTemplates;

            $scope.containerSelection = function (newMode) {
                if (!arguments.length) {
                    return _mode;
                }

                _mode = newMode;

                switch (newMode) {
                    case "NONE":
                        $scope.codeEnv.allContainerConfs = false;
                        $scope.codeEnv.containerConfs = [];
                        break;
                    case "ALLOWED":
                        $scope.codeEnv.allContainerConfs = false;
                        break;
                    case "ALL":
                        $scope.codeEnv.allContainerConfs = true;
                        break;
                }
            };
            $scope.sparkKubernetesSelection = function (newMode) {
                if (!arguments.length) {
                    return _sparkMode;
                }

                _sparkMode = newMode;

                switch (newMode) {
                    case "NONE":
                        $scope.codeEnv.allSparkKubernetesConfs = false;
                        $scope.codeEnv.sparkKubernetesConfs = [];
                        break;
                    case "ALLOWED":
                        $scope.codeEnv.allSparkKubernetesConfs = false;
                        break;
                    case "ALL":
                        $scope.codeEnv.allSparkKubernetesConfs = true;
                        break;
                }
            };

            $scope.codeStudioRebuildTemplateSelection = function (newMode) {
                if (!arguments.length) {
                    return _codeStudioMode;
                }

                _codeStudioMode = newMode;
                $scope.codeEnv.rebuildDependentCodeStudioTemplates = _codeStudioMode;
            };

            $scope.removeOutdatedContainerConfs = function() {
                $scope.codeEnv.containerConfs = $scope.codeEnv.containerConfs.filter(o => $scope.outdatedContainerConfs.indexOf(o) === -1);
            };

            $scope.removeOutdatedSparkKubernetesConfs = function() {
                $scope.codeEnv.sparkKubernetesConfs = $scope.codeEnv.sparkKubernetesConfs.filter(o => $scope.outdatedSparkKubernetesConfs.indexOf(o) === -1);
            };

            $scope.containsMismatchedPythonHooks = function() {
                if ($scope.codeEnv && $scope.codeEnv.desc && $scope.codeEnv.desc.predefinedContainerHooks && $scope.codeEnv.desc.pythonInterpreter) {
                    return $scope.codeEnv.desc.predefinedContainerHooks.some(hook =>
                        hook.type &&
                        hook.type !== $scope.codeEnv.desc.pythonInterpreter + "_SUPPORT" &&
                        /^PYTHON\d{2,3}_SUPPORT$/.test(hook.type));
                }

                return false;
            };

            DataikuAPI.containers.listNames(null, "USER_CODE")
                .success(data => {
                    $scope.containerNames = data;
                    $scope.outdatedContainerConfs = $scope.codeEnv.containerConfs.filter(o => $scope.containerNames.indexOf(o) === -1)

                    $scope.$watch("containerNames && codeEnv.containerConfs", function(nv, ov) {
                        $scope.outdatedContainerConfs = $scope.codeEnv.containerConfs.filter(o => $scope.containerNames.indexOf(o) === -1)
                    });
                })
                .error(setErrorInScope.bind($scope));
            DataikuAPI.containers.listSparkNames()
                .success(data => {
                    $scope.sparkKubernetesNames = data;
                    $scope.outdatedSparkKubernetesConfs = $scope.codeEnv.sparkKubernetesConfs.filter(o => $scope.sparkKubernetesNames.indexOf(o) === -1)

                    $scope.$watch("sparkKubernetesNames && codeEnv.sparkKubernetesConfs", function(nv, ov) {
                        $scope.outdatedSparkKubernetesConfs = $scope.codeEnv.sparkKubernetesConfs.filter(o => $scope.sparkKubernetesNames.indexOf(o) === -1)
                    });
                })
                .error(setErrorInScope.bind($scope));

            $scope.cacheLocations = [
                ['BEGINNING', 'Beginning'],
                ['AFTER_START_DOCKERFILE', 'After initial hooks'],
                ['AFTER_PACKAGES', 'After installation of packages'],
                ['AFTER_AFTER_PACKAGES_DOCKERFILE', 'After post-installation hooks'],
                ['END', 'At end'],
                ['NONE', 'None']
            ];

            $scope.runtimeAdditionTypes = [
                ['SYSTEM_LEVEL_CUDA_122_CUDNN_897', 'GPU support for Visual Machine Learning'],
                ['CUDA_SUPPORT_FOR_TORCH2_WITH_PYPI_NVIDIA_PACKAGES', 'GPU support for Torch 2'],
                ['BASIC_GPU_ENABLING', 'GPU support for Torch 1 (with +cuXXX variant)'],
                ['PYTHON36_SUPPORT', 'Install Python 3.6'],
                ['PYTHON37_SUPPORT', 'Install Python 3.7'],
                ['PYTHON38_SUPPORT', 'Install Python 3.8'],
                ['SYSTEM_LEVEL_CUDA_112_CUDNN_811', '[Deprecated] GPU support with CUDA 11.2'],
            ];

            $scope.runtimeAdditionTypesDescriptions = [
                'Visual Machine Learning, Visual Deep Learning and Visual Time Series Forecasting',
                'Prompt studios, LLM recipes & Knowledge banks (for local Hugging Face models)',
                'Object detection, image classification',
                'Add support for Python 3.6',
                'Add support for Python 3.7',
                'Add support for Python 3.8',
                '',
            ];
        }
    };
});

app.directive('codeEnvResources', ['DataikuAPI', 'Dialogs', 'CreateModalFromTemplate', function (DataikuAPI, Dialogs, CreateModalFromTemplate) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/code-envs/common/code-env-resources.html',
        replace: true,
        scope: {
            codeEnv: '=',
            envName: '=',
            editorOptions: '=',
            nodeType: '@',
            canUpdateCodeEnv: '=',
            deploymentMode: '='
        },
        controller: function($scope) {
            $scope.showResourcesSize = false;
            $scope.resourcesCanSelectFn = () => true;
            $scope.resourcesBrowserPath = "/";

            $scope.registerExecuteResourcesBrowseFn = function(fnToRegister) {
                $scope.executeResourcesBrowse = fnToRegister;
            }

            $scope.getResourcesEnvVars = function() {
                switch ($scope.nodeType) {
                    case "DESIGN":
                        DataikuAPI.admin.codeenvs.design.getResourcesEnvVars("PYTHON", $scope.envName).then((response) => {
                            $scope.resourcesEnvironmentVariables = response.data.variables;
                        });
                        break;
                    case "AUTOMATION":
                        DataikuAPI.admin.codeenvs.automation.getResourcesEnvVars("PYTHON", $scope.envName, $scope.codeEnv.versionId).then((response) => {
                            $scope.resourcesEnvironmentVariables = response.data.variables;
                        });
                        break;
                }
            }

            $scope.$on('refreshCodeEnvResources', function (event, opt) {
                if(opt.envName === $scope.envName && (!opt.versionId || opt.versionId === $scope.codeEnv.versionId)){
                    $scope.getResourcesEnvVars();
                    $scope.executeResourcesBrowse($scope.resourcesBrowserPath);
                }
            });

            $scope.resourcesBrowseFn = function (relativePath) {
                switch ($scope.nodeType) {
                    case "DESIGN":
                        return DataikuAPI.admin.codeenvs.design.browseResources("PYTHON", $scope.envName, relativePath, $scope.showResourcesSize);
                    case "AUTOMATION":
                        return DataikuAPI.admin.codeenvs.automation.browseResources("PYTHON", $scope.envName, $scope.codeEnv.versionId, relativePath, $scope.showResourcesSize);
                }
            }

            $scope.computeSize = function () {
                switch ($scope.nodeType) {
                    case "DESIGN":
                        DataikuAPI.admin.codeenvs.design.browseResources("PYTHON", $scope.envName, "/", true).then((response) => {
                            $scope.resourcesSize = response.data.size;
                            $scope.showResourcesSize = true;
                            $scope.executeResourcesBrowse($scope.resourcesBrowserPath); // to refresh browser
                        });
                        break;
                    case "AUTOMATION":
                        DataikuAPI.admin.codeenvs.automation.browseResources("PYTHON", $scope.envName, $scope.codeEnv.versionId, "/", true).then((response) => {
                            $scope.resourcesSize = response.data.size;
                            $scope.showResourcesSize = true;
                            $scope.executeResourcesBrowse($scope.resourcesBrowserPath); // to refresh browser
                        });
                        break;
                }
            }

            $scope.clearResources = function () {
                Dialogs.confirm($scope, 'Clear resources', 'Are you sure you want to clear the resources?').then(function() {
                    switch ($scope.nodeType) {
                        case "DESIGN":
                            DataikuAPI.admin.codeenvs.design.clearResources("PYTHON", $scope.envName).then(() => {
                                $scope.showResourcesSize = false;
                                $scope.executeResourcesBrowse("/"); // to refresh browser
                            });
                            break;
                        case "AUTOMATION":
                            DataikuAPI.admin.codeenvs.automation.clearResources("PYTHON", $scope.envName, $scope.codeEnv.versionId).then(() => {
                                $scope.showResourcesSize = false;
                                $scope.executeResourcesBrowse("/"); // to refresh browser
                            });
                            break;
                    }
                });
            }

            $scope.openUploadResourcesModal = function(){
                CreateModalFromTemplate("/templates/admin/code-envs/common/upload-resources-modal.html", $scope, "AdminCodeEnvsUploadResourcesController")
            }

            $scope.canBeUpdated = function() {
                return $scope.canUpdateCodeEnv && ['DESIGN_MANAGED', 'PLUGIN_MANAGED', 'AUTOMATION_SINGLE', 'AUTOMATION_VERSIONED'].includes($scope.deploymentMode);
            };

            DataikuAPI.codeenvs.getUserAccessibleCodeEnvSettings().then(({data}) => {
                $scope.enableCodeEnvResources = data.enableCodeEnvResources;
            });

            $scope.getResourcesEnvVars();
        }
    };
}]);

app.controller("AdminCodeEnvsUploadResourcesController", ['$scope', 'Assert', 'DataikuAPI', function($scope, Assert, DataikuAPI) {
    $scope.resourcesToUpload = false;
    $scope.overwrite = false;

    $scope.upload = function() {
        Assert.trueish($scope.resourcesToUpload, "No resource to upload");

        const parentScope = $scope.$parent.$parent;
        switch (parentScope.nodeType) {
            case "DESIGN":
                DataikuAPI.admin.codeenvs.design.uploadResources("PYTHON", parentScope.envName, parentScope.resourcesBrowserPath, $scope.resourcesToUpload, $scope.overwrite).then(() => {
                    $scope.dismiss();
                }).catch(setErrorInScope.bind($scope)).finally(() => {
                    parentScope.executeResourcesBrowse(parentScope.resourcesBrowserPath); // to refresh browser
                });
                break;
            case "AUTOMATION":
                DataikuAPI.admin.codeenvs.automation.uploadResources("PYTHON", parentScope.envName, parentScope.codeEnv.versionId, parentScope.resourcesBrowserPath, $scope.resourcesToUpload, $scope.overwrite).then(() => {
                    $scope.dismiss();
                }).catch(setErrorInScope.bind($scope)).finally(() => {
                    parentScope.executeResourcesBrowse(parentScope.resourcesBrowserPath); // to refresh browser
                });
                break;
        }
    }
}]);


app.component('dssInternalCodeEnv', {
    templateUrl: "/templates/admin/code-envs/common/dss-internal-code-env.html",
    bindings: {
        internalCodeEnvType: '@',
        codeEnvDescription: '@'
    },
    controller: function($scope, DataikuAPI, FutureProgressModal, Dialogs, $state, $filter, $rootScope, CodeEnvService) {
        const $ctrl = this;

        $ctrl.appConfig = $rootScope.appConfig;
        $ctrl.mayCreateCodeEnvs = $rootScope.mayCreateCodeEnvs;
        $ctrl.supportedPythonInterpreters = [];
        $ctrl.codeEnvConfig = {
            pythonInterpreter: null
        };

        $ctrl.$onInit =  () => {
            $ctrl.checkAndShowCodeEnv();
            $ctrl.fetchSupportedPythonInterpreters();
            $ctrl.fetchDefaultAvailableInterpreter();
        };

        $ctrl.fetchSupportedPythonInterpreters = function() {
            const enrichInterpretersDescription = pythonInterpreters => DataikuAPI.codeenvs.getSupportedInterpreters($ctrl.internalCodeEnvType)
            .then(function({data}) {
                $ctrl.supportedPythonInterpreters = pythonInterpreters.filter(interpreter => data.includes(interpreter[0]))
            })
            .catch(setErrorInScope.bind($scope))
            CodeEnvService.getPythonInterpreters().then(enrichInterpretersDescription);
        }

        $ctrl.fetchDefaultAvailableInterpreter = function() {
            DataikuAPI.codeenvs.getDSSInternalDefaultAvailableInterpreter($ctrl.internalCodeEnvType)
            .then(function({data}) {
                $ctrl.codeEnvConfig.pythonInterpreter = data;
            })
            .catch(setErrorInScope.bind($scope))
        }

        $ctrl.checkAndShowCodeEnv = function () {
            DataikuAPI.codeenvs.checkDSSInternalCodeEnv($ctrl.internalCodeEnvType)
            .then(function({data}) {
                $ctrl.codeEnvExists = Object.keys(data).length > 0;
                if ($ctrl.codeEnvExists) {
                    $ctrl.codeEnv = data.value;
                    if ($ctrl.appConfig.isAutomation) {
                        $ctrl.codeEnvHref = $state.href("admin.codeenvs-automation.python-edit", { envName: $ctrl.codeEnv.envName });
                    } else {
                        $ctrl.codeEnvHref = $state.href("admin.codeenvs-design.python-edit", { envName: $ctrl.codeEnv.envName });
                    }

                    if ($ctrl.internalCodeEnvType === "PII_DETECTION_CODE_ENV") {
                        DataikuAPI.codeenvs.checkInternalCodeEnvUsedForPII().then(function({data}) {
                            $ctrl.showPiiDetectionCodeEnvWarning = !data;
                        }).catch(setErrorInScope.bind($scope));
                    }
                    if ($ctrl.internalCodeEnvType === "RAG_CODE_ENV") {
                        DataikuAPI.codeenvs.checkInternalCodeEnvUsedForRAG().then(function({data}) {
                            $ctrl.showRagCodeEnvWarning = !data;
                        }).catch(setErrorInScope.bind($scope));
                    }
                    if ($ctrl.internalCodeEnvType === "HUGGINGFACE_LOCAL_CODE_ENV") {
                        DataikuAPI.codeenvs.checkInternalCodeEnvUsedForHF().then(function({data}) {
                            $ctrl.hfLocalConnectionsWithWrongCodeEnv = data
                            .map(name => ({"name": name, "href": $state.href("admin.connections.edit", { connectionName: name })}));
                        }).catch(setErrorInScope.bind($scope));
                    }

                    $ctrl.outdated = $ctrl.codeEnv.outdatedItems.length > 0;
                    $ctrl.showOutdatedRequirementsWarning = $ctrl.codeEnv.outdatedItems.includes("REQUIREMENTS");
                    $ctrl.showOutdatedContainerImagesWarning = $ctrl.codeEnv.outdatedItems.includes("CONTAINER_IMAGES");
                }
            })
            .catch(setErrorInScope.bind($scope))
        };

        $ctrl.createCodeEnv = function () {
            DataikuAPI.codeenvs.createForDSSInternal($ctrl.internalCodeEnvType, $ctrl.codeEnvConfig.pythonInterpreter)
            .then(function(resp) {
                FutureProgressModal.show($scope, resp.data, $filter('capitalize')($ctrl.codeEnvDescription + " code environment"), undefined, 'static', false).then(function(result) {
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Creation result", result.messages, result.futureLog, undefined, 'static', false);
                    }
                    $ctrl.checkAndShowCodeEnv();
                });
            })
            .catch(setErrorInScope.bind($scope));
        };

        $ctrl.updateCodeEnv = function () {
            DataikuAPI.codeenvs.updateForDSSInternal($ctrl.internalCodeEnvType)
            .then(function(resp) {
                FutureProgressModal.show($scope, resp.data, $filter('capitalize')($ctrl.codeEnvDescription + " code environment"), undefined, 'static', false).then(function(result) {
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Update result", result.messages, result.futureLog, undefined, 'static', false);
                    }
                    $ctrl.checkAndShowCodeEnv();
                });
            })
            .catch(setErrorInScope.bind($scope));
        };
    }
});


app.component('codeEnvUsage', {
    templateUrl: '/templates/admin/code-envs/common/code-env-usage.html',
    bindings: {
        envLang: '@',
        envName: '@',
    },
    controller: function($scope, DataikuAPI, StateUtils, FullModelLikeIdUtils, Logger, $filter) {
        const $ctrl = this;
        
        //keys based on EnvUsage enum from CodeEnvModel.java
        const getUsageLink = {
            PROJECT: (_objectId, projectKey) => StateUtils.href.project(projectKey, {page: 'settings', selectedTab: 'code-envs'}),
            RECIPE: StateUtils.href.recipe,
            NOTEBOOK: StateUtils.href.jupyterNotebook,
            PLUGIN: StateUtils.href.plugin,
            SCENARIO: (objectId, projectKey) => StateUtils.href.scenario(objectId, projectKey, {tab: 'settings'}),
            SCENARIO_STEP: (objectId, projectKey) => StateUtils.href.scenario(objectId, projectKey, {tab: 'steps'}),
            SCENARIO_TRIGGER: (objectId, projectKey) => StateUtils.href.scenario(objectId, projectKey, {tab: 'settings'}),
            DATASET_METRIC: (objectId, projectKey) => StateUtils.href.dataset(objectId, projectKey, {tab: 'settings'}),
            DATASET_CHECK: (objectId, projectKey) => StateUtils.href.dataset(objectId, projectKey, {tab: 'settings'}),
            DATASET: (objectId, projectKey) => StateUtils.href.dataset(objectId, projectKey, {tab: 'settings'}),
            WEBAPP: (objectId, projectKey) => StateUtils.href.webapp(objectId, projectKey, {tab: 'edit'}),
            REPORT: (objectId, projectKey) => StateUtils.href.report(objectId, projectKey, {tab: 'edit'}),
            RETRIEVABLE_KNOWLEDGE: (objectId, projectKey) => StateUtils.href.retrievableKnowledge(objectId, projectKey, {tab: 'settings'}),
            API_SERVICE_ENDPOINT: StateUtils.href.lambdaService,
            SAVED_MODEL: (objectId, projectKey) => {
                try {
                    const obj = FullModelLikeIdUtils.parse(objectId);
                    if (obj.savedModelId) {
                        return StateUtils.href.savedModel(obj.savedModelId, projectKey);
                    }
                } catch(e) {
                    // do not generate link
                }
                Logger.error("code env usage - unsupported saved model id", objectId, projectKey);
                return null;
            },
            MODEL: (objectId, projectKey) => {
                try {
                    const obj = FullModelLikeIdUtils.parse(objectId);
                    if (obj.analysisId) {
                        return StateUtils.href.analysis(obj.analysisId, projectKey, {tab: 'ml.list'});
                    }
                } catch(e) {
                    if (objectId) {
                        return StateUtils.href.analysis(objectId, projectKey, {tab: 'ml.list'});
                    }
                }
                Logger.error("code env usage - unsupported model id", objectId, projectKey);
                return null;
            },
            CODE_STUDIO_TEMPLATE: StateUtils.href.codeStudioTemplate,
        };
        
        $ctrl.uiState = {
            filters: {
                query: '',
                resetable: false,
                types: [],
                projects: []
            },
            options: {
                types: [],
                projects: []
            },
            filteredUsages: [],
            filteredProjectsCount: 0
        };

        $ctrl.filterUsages = () => {
            if ($ctrl.accessibleUsages) {
                const qry = $ctrl.uiState.filters.query.trim().toLocaleLowerCase();
                const nbSelectedTypes = $ctrl.uiState.filters.types.length;
                const nbSelectedProjects = $ctrl.uiState.filters.projects.length;
                $ctrl.uiState.filteredUsages = $ctrl.accessibleUsages
                    //query filter
                    .filter(usage => !qry
                                  || usage.envUsage.toLocaleLowerCase().includes(qry)
                                  || usage.projectLabel.toLocaleLowerCase().includes(qry)
                                  || (usage.objectId && usage.objectId.toLocaleLowerCase().includes(qry)))
                    //type filter
                    .filter(usage => !nbSelectedTypes || $ctrl.uiState.filters.types.indexOf(usage.envUsage) >= 0)
                    //project filter
                    .filter(usage => !nbSelectedProjects || $ctrl.uiState.filters.projects.indexOf(usage.projectLabel) >= 0);
                $ctrl.uiState.filters.resetable = nbSelectedTypes > 0 || nbSelectedProjects > 0 || qry.length;
                $ctrl.uiState.filteredProjectsCount = $ctrl.uiState.filteredUsages.reduce(
                    (projects, usage) => {
                        projects.add(usage.projectKey)
                        return projects;
                    },
                    new Set()
                ).size;
            }
        };
        
        $ctrl.resetFilters = function(){
            $ctrl.uiState.filters.query = '';
            $ctrl.uiState.filters.types = [];
            $ctrl.uiState.filters.projects = [];
            $ctrl.filterUsages();
        };

        function makeOptions(items, field){
            //dedup item on fields, and generate a label with $itemCount
            return items
                .reduce((dedup, item) => {
                    const dedupedItem = dedup.find(dedupItem => dedupItem.id === item[field]);
                    if (dedupedItem) {
                        dedupedItem.count++;
                    } else {
                        dedup.push({
                            id: item[field],
                            count: 1
                        });
                    }
                    return dedup;
                }, [])
                .map(dedup => {
                    dedup.label = `${dedup.id} (${dedup.count})`;
                    return dedup
                });
        }

        $ctrl.getUsage = function () {
            DataikuAPI.admin.codeenvs.design
                .listUsages($ctrl.envLang, $ctrl.envName, false)
                .success(function(usagesList){
                    $ctrl.inaccessibleObjectsCount = usagesList.filter(usage => !usage.accessible).length;
                    $ctrl.accessibleUsages = usagesList
                        .filter(usage => usage.accessible)
                        .map(usage => {
                            usage.projectLabel = usage.projectKey === '__DKU_ANY_PROJECT__' ? 'All' : usage.projectKey;
                            usage.href = getUsageLink[usage.envUsage](usage.objectId, usage.projectKey);
                            if (usage.envUsage === 'PROJECT') {
                                usage.linkLabel = 'Go to project settings';
                            } else if (usage.envUsage === 'SAVED_MODEL') {
                                usage.linkLabel = 'Go to Saved models versions';
                            } else if (usage.envUsage === 'MODEL') {
                                usage.linkLabel = 'Go to Analysis page';
                            } else if (usage.href) {
                                usage.linkLabel = 'Go to ' + $filter('typeToName')(usage.envUsage);
                            }
                            return usage;
                        });
                    $ctrl.uiState.options.types = makeOptions($ctrl.accessibleUsages, 'envUsage');
                    $ctrl.uiState.options.projects = makeOptions($ctrl.accessibleUsages, 'projectLabel');
                    $ctrl.filterUsages();
                })
                .error(setErrorInScope.bind($scope));
        };
    }
});

}());
