(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('groupTile', {
        bindings: {
            tile: '<',
            selected: '<', // boolean
            insight: '<',
            editable: '<',
            minHeight: '<', // number
            backgroundColor: '<', // string
            gridColor: '<', // string
            currentGroupTileIdInEditMode: '<', // string | null
            hook: '<',
            activeFilters: '<',
            filters: '<',
            page: '<',
            showGrid: '<', // boolean
            tileSpacing: '<', // number
            preselectedTile: '<', // DashboardTile
            canExportDatasets: '<', // boolean
            insightsMap: '<', // { [insightId: string]: Insight }
            accessMap: '<', // { [insightId: string]: string }
            dashboardTheme: '<', // DSSVisualizationTheme
            filtersChange: '&', // ({ $filters }) => void
            filtersParamsChange: '&', // ({ $filtersParams }) => void,
            isDeactivatedChange: '&', // ({ $isDeactivated }) => void
            preselectedTileChange: '&', // ({ $preselectedTile }) => void,
            raiseError: '&', // ({ $errorData }) => void,
            isTileLockedChange: '&' // () => void,
        },
        templateUrl: '/static/dataiku/js/dashboards/components/group-tile/group-tile.component.html',
        controller: function(TileLoadingState, Logger, $timeout, TileLoadingBehavior, $scope, TileUtils, $element) {
            const ctrl = this;

            ctrl.display = false;

            const overflowCSSClass = 'dashboard-tile__group-tile-container--overflowing-inner-tiles';
            let successRoutine = null;
            let reloadGrid = null;
            let resizeGrid = null;
            let hasOverflowingInnerTiles = false;
            let scrollingContainer = null;

            ctrl.$onInit = function() {
                if (ctrl.tile.autoLoad) {
                    ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.WAITING;
                }
                ctrl.hook.loadPromises[ctrl.tile.$tileId] = ctrl.load;
                ctrl.hook.reloadPromises[ctrl.tile.$tileId] = ctrl.reload;
                scrollingContainer = getScrollingContainer();
                /*
                 * We eliminate the visible padding around the grid by applying negative top and left margins to the child tiles.
                 * To ensure the grid fully occupies the container's space, we extend its width (width: calc(var(--tile-spacing) + 100%)).
                 * This setup causes the content to overflow horizontally, but the horizontal overflow is hidden.
                 *
                 * We attach a scroll event handler to prevent any unexpected horizontal scrolling while the user drags and drops the tiles.
                 */
                scrollingContainer.on('scroll', removeLeftScroll);
            };

            ctrl.$onChanges = function(changes) {
                if (changes.currentGroupTileIdInEditMode || changes.editable) {
                    ctrl.isGroupTileInnerGridEditable = ctrl.editable && ctrl.tile.$tileId === ctrl.currentGroupTileIdInEditMode;
                }
            };

            ctrl.$onDestroy = function() {
                if (scrollingContainer != null) {
                    scrollingContainer.off('scroll', removeLeftScroll);
                    scrollingContainer = null;
                }
            };

            ctrl.handleTilesChange = function(tiles) {
                handleChildTileVerticallyOverflowingGroupTile(tiles);
                $timeout(() => {
                    ctrl.tile.grid.tiles = [...tiles];
                });
            };

            ctrl.handleGridLoadingStateChange = function(isLoading) {
                $timeout(() => {
                    ctrl.isLoading = isLoading;
                    if (successRoutine != null) {
                        ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.COMPLETE;
                        successRoutine();
                        successRoutine = null;
                    }
                });
            };

            ctrl.handleGridInit = function(api) {
                reloadGrid = api.reload;
                resizeGrid = api.resizeGrid;
                handleChildTileVerticallyOverflowingGroupTile(ctrl.tile.grid.tiles);
                $scope.$watch('$ctrl.tile.box.height', () => {
                    handleChildTileVerticallyOverflowingGroupTile(ctrl.tile.grid.tiles);
                });
            };

            ctrl.handleToggleGroupTileScrollBarDisplay = function(forceShow, scrollTopOnHide) {
                // If the group tile has overflowing inner tiles, the scroll bar is always shown
                if (hasOverflowingInnerTiles) {
                    return;
                }
                if (forceShow) {
                    scrollingContainer.addClass(overflowCSSClass);
                } else {
                    if (scrollTopOnHide) {
                        scrollingContainer.scrollTop(0);
                    }
                    scrollingContainer.removeClass(overflowCSSClass);
                }
            };

            ctrl.load = function(resolve) {
                Logger.info('Start loading group tile', ctrl.tile.$tileId);
                ctrl.hook.loadStates[ctrl.tile.$tileId] = TileLoadingState.WAITING;
                ctrl.isLoading = true;
                ctrl.display = true;
                successRoutine = resolve;
                return TileLoadingBehavior.DELAYED_COMPLETE;
            };

            ctrl.reload = function(resolve) {
                if (reloadGrid == null) {
                    return;
                }
                Logger.info('Start re-loading group tile', ctrl.tile.$tileId);
                ctrl.isLoading = true;
                reloadGrid();
                successRoutine = resolve;
                return TileLoadingBehavior.DELAYED_COMPLETE;
            };

            $scope.$on('dashboardResizeGroupTile', (_, { tile }) => {
                if (tile != null && tile.$tileId === ctrl.tile.$tileId && resizeGrid != null) {
                    resizeGrid();
                }
            });

            function getScrollingContainer() {
                return $element.closest('.dashboard-tile__group-tile-container');
            }

            function removeLeftScroll() {
                if (scrollingContainer != null && scrollingContainer.scrollLeft() > 0) {
                    scrollingContainer.scrollLeft(0);
                }
            }

            function handleChildTileVerticallyOverflowingGroupTile(tiles) {
                let nextHasOverflowingInnerTiles = false;
                tiles.forEach(tile => {
                    if (TileUtils.isChildTileVerticallyOverflowingGroupTile(tile, ctrl.tile.box.height)) {
                        nextHasOverflowingInnerTiles = true;
                    }
                });
                if (hasOverflowingInnerTiles !== nextHasOverflowingInnerTiles) {
                    hasOverflowingInnerTiles = nextHasOverflowingInnerTiles;
                    if (hasOverflowingInnerTiles) {
                        scrollingContainer.addClass(overflowCSSClass);
                    } else {
                        scrollingContainer.scrollTop(0).removeClass(overflowCSSClass);
                    }
                    if (resizeGrid != null) {
                        resizeGrid();
                    }
                }
            }
        }
    });
})();
