(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/edit.js
     */
    app.directive('dashboardTileParams', function($stateParams, DashboardPageUtils, DashboardUtils, $timeout, TileUtils, ColorUtils, DSSVisualizationThemeUtils, DefaultDSSVisualizationTheme) {
        return {
            restrict: 'EA',
            templateUrl: '/templates/dashboards/tile-params.html',
            scope: {
                tile: '<', // Tile
                insight: '<', // Insight
                dashboard: '<', // Dashboard
                canModerateDashboards: '<', // boolean
                toggleDashboardSettings: '&', // () => void
                theme: '<'
            },
            link: function($scope, $element) {
                $scope.$stateParams = $stateParams;
                $scope.getDefaultTileTitle = getDefaultTileTitle;
                $scope.canEditInsight = function(insight) {
                    return DashboardUtils.canEditInsight(insight, $scope.canModerateDashboards);
                };
                $scope.DashboardUtils = DashboardUtils;
                $scope.TileUtils = TileUtils;
                $scope.themeColors = ColorUtils.getThemeColorsWithBlackWhite($scope.theme);
                $scope.colors = ColorUtils.generateThemePaletteColors(DSSVisualizationThemeUtils.getThemeOrDefault($scope.theme).colors, $scope.themeColors.length > 0);
                $scope.titleDefaultFormatting = getTitleDefaultFormatting();
                $scope.originDashboard = {
                    id: $stateParams.dashboardId,
                    name: $stateParams.dashboardName,
                    pageId: $stateParams.pageId,
                    edit: true
                };

                const DEFAULT_BACKGROUND_COLOR = '#ffffff';

                $scope.handleTileTitleChange = function() {
                    if ($scope.tile.titleOptions.title === '' || $scope.tile.titleOptions.title === getDefaultTileTitle($scope.tile, $scope.insight)) {
                        $scope.updateTileTitleOptions({
                            title: undefined
                        });
                    } else {
                        $scope.updateTileTitleOptions({
                            title: $scope.tile.titleOptions.title
                        });
                    }
                };

                $scope.fillIfEmpty = function() {
                    if (!$scope.tile.titleOptions.title) {
                        $scope.updateTileTitleOptions({
                            title: getDefaultTileTitle($scope.tile, $scope.insight)
                        });
                    }
                };

                $scope.$watch('tile.insightType', function() {
                    $timeout(function() {
                        $element.find('select.display-mode-select').selectpicker('refresh');
                    });
                });

                $scope.$watch('theme', function(nv, ov) {
                    if (nv && (!ov || !_.isEqual(nv.colors, ov.colors))) {
                        $scope.themeColors = ColorUtils.getThemeColorsWithBlackWhite($scope.theme);
                        $scope.colors = ColorUtils.generateThemePaletteColors(nv.colors, $scope.themeColors.length > 0);
                    };
                    if (nv && (!ov || !_.isEqual(nv.tileTitleFormatting, ov.tileTitleFormatting))) {
                        this.titleDefaultFormatting = getTitleDefaultFormatting();
                    };
                });

                $scope.openUploadPictureDialog = TileUtils.openUploadPictureDialog;

                $scope.setTileBorderOptions = function(borderOptions) {
                    $scope.assignToTile('borderOptions', { ...$scope.tile.borderOptions, ...borderOptions })
                        .then(() => resizeTile($scope.tile.$tileId));
                };

                $scope.setTileTextAlignment = function(newAlignment) {
                    $scope.assignToTile('tileParams.textAlign', newAlignment);
                };

                $scope.setTileVerticalTextAlignment = function(newAlignment) {
                    $scope.assignToTile('tileParams.verticalAlign', newAlignment);
                };

                $scope.updateTileTitleDisplayMode = function(showTitle) {
                    $scope.updateTileTitleOptions({ showTitle })
                        .then(() => resizeTile($scope.tile.$tileId));
                };

                $scope.resetBorder = () => {
                    const baseTheme = getBaseTheme();
                    $scope.tile.borderOptions = baseTheme.tileFormatting.borderOptions;
                    $timeout(() => $scope.$apply());
                };

                $scope.shouldShowResetBorder = function() {
                    const baseTheme = getBaseTheme();
                    return (
                        $scope.tile.borderOptions.color !== baseTheme.tileFormatting.borderOptions.color
                        || $scope.tile.borderOptions.radius !== baseTheme.tileFormatting.borderOptions.radius
                        || $scope.tile.borderOptions.size !== baseTheme.tileFormatting.borderOptions.size
                    );
                };

                $scope.resetBackgroundColor = () => {
                    $scope.assignToTile('backgroundColor', $scope.theme ? $scope.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
                };

                $scope.shouldShowResetBackgroundColorButton = () => {
                    return $scope.tile && $scope.tile.backgroundColor !== ($scope.theme ? $scope.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
                };

                $scope.resetBackgroundOpacity = () => {
                    $scope.assignToTile('backgroundOpacity', getBaseTheme().tileFormatting.backgroundOpacity);
                };

                $scope.shouldShowResetBackgroundOpacity = () => {
                    return $scope.tile && $scope.tile.backgroundOpacity !== getBaseTheme().tileFormatting.backgroundOpacity;
                };

                $scope.updateTileTitleOptions = function(partialTitleOptions) {
                    let partialUpdate = partialTitleOptions;
                    if (!$scope.tile.titleOptions.hasBackground && !partialTitleOptions.hasBackground) {
                        /*
                         * In case hasBackground is not defined or false, we remove from the update
                         * to avoid setting the tile in dirty as titleOptions for tile doesn't have
                         * a hasBackground field.
                         */
                        partialUpdate = _.omit(partialTitleOptions, 'hasBackground');
                    }
                    return $scope.assignToTile('titleOptions', { ...$scope.tile.titleOptions, ...partialUpdate });
                };

                // Easiest way to apply change from Angular component. Inspired by assignToChartDef
                $scope.assignToTile = function(propertyPath, value) {
                    _.set($scope.tile, propertyPath, value);
                    return $timeout(() => $scope.$apply());
                };

                function getDefaultTileTitle(tile, insight) {
                    switch (tile.tileType) {
                        case 'INSIGHT':
                            return (insight || {}).name;
                        case 'TEXT':
                            return 'Text tile';
                        case 'IMAGE':
                            return 'Image tile';
                    }
                }

                function resizeTile(tileId) {
                    const tileElement = DashboardPageUtils.getTileElementFromTileId($(document.querySelector('.dashboard-export-page-wrapper')), tileId);
                    if (tileElement == null || tileElement.length === 0) {
                        return;
                    }
                    const tileWrapper = DashboardPageUtils.getTileWrapperFromElement(tileElement);
                    if (tileWrapper == null || tileWrapper.length === 0) {
                        return;
                    }
                    angular.element(tileWrapper).scope().$broadcast('resize');
                }

                function getBaseTheme() {
                    return ($scope.theme || DefaultDSSVisualizationTheme);
                }

                function getTitleDefaultFormatting() {
                    const baseTheme = getBaseTheme();
                    return { fontColor: baseTheme.tileTitleFormatting.fontColor, fontSize: baseTheme.tileTitleFormatting.fontSize };
                }
            }
        };
    });
})();
