(function () {
    'use strict';

    var app = angular.module('dataiku.controllers');

    app.service("DatasetTypesService", function() {
        function isTemporalType(t) {
            return ['date', 'dateonly', 'datetimenotz'].indexOf(t) >= 0;
        }

        function isDateOnly(t) {
            return ['dateonly'].indexOf(t) >= 0;
        }

        return {
            isTemporalType:isTemporalType,
            isDateOnly:isDateOnly
        };
    });

    app.controller("DatasetsCommon", function ($scope, $rootScope, $stateParams, DataikuAPI, DatasetsService, CreateModalFromTemplate, WT1) {

        $scope.compareColumnValuesView = {
            isCompareViewVisible: false,
            lastSelectedRow: null,
            lastSelectedColId: null,
            height: 0,
        }

        $rootScope.$on('triggerCompareColumnValues', function(event, rowPromise, colId) {
            $rootScope.$broadcast('compareColumnValuesViewResize', 'open');
            rowPromise.then(row => {
                $scope.compareColumnValuesView.lastSelectedRow = row;
                $scope.$apply();
            })
            $scope.compareColumnValuesView.lastSelectedColId = colId;
            // if the compare view is already open we don't trigger the event
            if(!$scope.compareColumnValuesView.isCompareViewVisible) {
                WT1.event("compare-column-values-open", {});
            }
            $scope.compareColumnValuesView.isCompareViewVisible = true;
        });

        $rootScope.$on('$stateChangeStart', () => {
            if($scope.compareColumnValuesView.isCompareViewVisible) {
                $scope.closeCompareColumnValuesView();
            }
        });

        $scope.closeCompareColumnValuesView = function(){
            $rootScope.$broadcast('compareColumnValuesViewResize', 'close');
            $scope.compareColumnValuesView.isCompareViewVisible = false;
            $scope.compareColumnValuesView.lastSelectedColId = null;
            $scope.compareColumnValuesView.lastSelectedCell = null;
            $scope.compareColumnValuesView.height = 0;
            WT1.event("compare-column-values-close", {});
        };

        $rootScope.$on("compareColumnValuesViewResized", (event, value) => {
            // apply to make sure that the height is applied to the template before the following broadcast
            $scope.$apply(() => {
                $scope.compareColumnValuesView.height = value;
            })
            $rootScope.$broadcast('compareColumnValuesViewResize');

        });

        $scope.createAndPin = function(datasetName) {
            const insight = {
                projectKey: $stateParams.projectKey,
                type: 'dataset_table',
                params: { datasetSmartName: datasetName },
                name: datasetName
            };
            DataikuAPI.explores.getScript($stateParams.projectKey, datasetName).success(function(shaker) {
                insight.params.shakerScript = shaker;
                insight.params.shakerScript.origin = "DATASET_EXPLORE";
                CreateModalFromTemplate("/templates/dashboards/insights/create-and-pin-insight-modal.html", $scope, "CreateAndPinInsightModalController", function(newScope) {
                    newScope.init(insight);
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.clearDataset = function(datasetName) {
            DatasetsService.clear($scope, $stateParams.projectKey, datasetName).then(function() {
                $scope.$broadcast('refresh-table');
            });
        };
    });


    app.controller('DatasetsListController', function($scope, $controller, $stateParams, $state, $q, translate,
                          DatasetsService, DataikuAPI, CreateModalFromTemplate, TopNav, ComputablesService, Fn, AddDatasetWorkflowService) {

        $controller("DatasetsCommon", {$scope: $scope});
        $controller('_TaggableObjectsListPageCommon', {$scope: $scope});
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_NONE, null);
        TopNav.setNoItem();

        const METRICS = ['status.size.totalValue', 'status.records.totalValue']
        $scope.showClearData = true;
        $scope.hasMetrics = false;
        $scope.selection = $.extend({
            filterQuery: {
                userQuery: '',
                tags: [],
                interest: {
                    starred: '',
                },
            },
            filterParams: {
                userQueryTargets: ["name","type","tags"],
                propertyRules: {"tag": "tags"},
                exactMatch: ["tags"]
            },
            orderQuery: "-lastModifiedOn",
            orderReversed: false,
        }, $scope.selection || {});

        $scope.sortBy = [
            { value: 'name', label: translate('PROJECT.DATASETS_LIST.SORT.NAME', 'Name') },
            { value: 'type', label: translate('PROJECT.DATASETS_LIST.SORT.TYPE', 'Type') },
            { value: 'status.size.totalValue', label: translate('PROJECT.DATASETS_LIST.SORT.SIZE', 'Size') },
            { value: 'status.records.totalValue', label: translate('PROJECT.DATASETS_LIST.SORT.RECORDS', 'Records') },
            { value : '-lastModifiedOn', label : translate('PROJECT.DATASETS_LIST.SORT.LAST_MODIFIED', 'Last modified') }
        ];

        $scope.sortCookieKey = 'datasets';
        $scope.maxItems = 20;

        $scope.goToItem = function(item) {
            $scope.$state.go('projects.project.datasets.dataset.explore', {datasetName: item.name, projectKey: $stateParams.projectKey});
        };
        
        $scope.getMetrics = function() {
            if ($scope.selection && METRICS.includes($scope.selection.orderQuery)) {
                return true;
            }
            return false;
        };

        $scope.changeQueryFilterSelection = function() {
            if ($scope.getMetrics() && !$scope.hasMetrics) {
                $scope.list();
            }
        };

        $scope.list = function() {
            const withMetrics = $scope.getMetrics();
            return DataikuAPI.datasets.listHeads($stateParams.projectKey, {}, $scope.getMetrics()).success(function (data) {
                $scope.datasets = data;
                $scope.filteredOut = data.filteredOut;
                $scope.listItems = data.items;
                $scope.restoreOriginalSelection();
                $scope.hasMetrics = withMetrics;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.openAddDatasetModal = function() {
            AddDatasetWorkflowService.openAddDatasetModal($scope);
        }

        $scope.$on('projectTagsUpdated', function (e, args) {
             if (args.refreshFlowFilters) $scope.list();
        });
        $scope.list();
    });


    app.controller("NewManagedDatasetController", function ($scope, $state, $stateParams, DataikuAPI, WT1, SqlConnectionNamespaceService) {
        WT1.event("dataset-new-managed-box");
        addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

        $scope.newDataset ={
            name : null,
            settings : {
                specificSettings : {},
                zone: $scope.getRelevantZoneId($stateParams.zoneId)
            }
        };

        DataikuAPI.datasets.getManagedDatasetOptionsNoContext($stateParams.projectKey).success(function(data) {
            $scope.managedDatasetOptions = data;
            if (!$scope.newDataset.settings.connection && data.connections.length) {
                $scope.newDataset.settings.connection = data.connections[0];
            }
            $scope.partitioningOptions = [
                {"id" : "NP", "label" : "Not partitioned"},
            ].concat(data.projectPartitionings)

            $scope.newDataset.settings.partitioningOptionId = "NP";
        });

        $scope.$watch("newDataset.settings.connection", function(nv, ov) {
            if (nv && nv.formats && nv.formats.length) {
                $scope.newDataset.settings.specificSettings.formatOptionId = nv.formats[0].id;
            }
            delete $scope.newDataset.settings.typeOptionId;
            if (nv && nv.fsProviderTypes && nv.fsProviderTypes.length >= 1) {
                $scope.newDataset.settings.typeOptionId = nv.fsProviderTypes[0];
            }

            if($scope.newDataset.settings.connection) {
                SqlConnectionNamespaceService.setTooltips($scope, $scope.newDataset.settings.connection.connectionType);
            }

            SqlConnectionNamespaceService.resetState($scope, $scope.newDataset.settings.specificSettings);
        }, true);

        $scope.create = function () {
            resetErrorInScope($scope);
            $scope.newDataset.settings.connectionId = $scope.newDataset.settings.connection.id;
            DataikuAPI.datasets.newManagedDataset($stateParams.projectKey, $scope.newDataset.name, $scope.newDataset.settings).success(function (data) {
                $state.go('projects.project.datasets.dataset.settings', { datasetName: $scope.newDataset.name });
                $scope.dismiss();
            }).error(setErrorInScope.bind($scope));
        };

        $scope.fetchCatalogs = function(connectionName, origin, connectionType) {
            SqlConnectionNamespaceService.listSqlCatalogs(connectionName, $scope, origin, connectionType);
        };

        $scope.fetchSchemas = function(connectionName, origin, connectionType) {
            const catalog = $scope.newDataset.settings.specificSettings.overrideSQLCatalog ||
                ($scope.newDataset.settings.connection ? $scope.newDataset.settings.connection.unoverridenSQLCatalog : '');
            SqlConnectionNamespaceService.listSqlSchemas(connectionName, $scope, catalog, origin, connectionType);
        };
    });


    app.controller("DatasetCommonController", function ($controller, $scope, $stateParams, $rootScope, DataikuAPI, TopNav, $state, GlobalProjectActions, CreateModalFromTemplate, DatasetCustomFieldsService, AnyLoc, FlowBuildService, DataQualityComputationTrackingService) {
        TopNav.setItem(TopNav.ITEM_DATASET, $stateParams.datasetName);
        $scope.datasetHooks = {};

        /* Check if this dataset has preview custom fields */
        function getCurrentDatasetFullInfo() {
            return DataikuAPI.datasets.getFullInfo($stateParams.projectKey, $stateParams.projectKey, $stateParams.datasetName).success(function(data) {
                $scope.datasetFullInfo = data;

                if ($scope.datasetFullInfo.type == "Inline") {
                    $scope.editableDataset = true;
                }

                TopNav.setItem(TopNav.ITEM_DATASET, $stateParams.datasetName, {
                    datasetType : data.type,
                    name : $stateParams.datasetName,
                    creatingRecipe: data.creatingRecipe,
                    creatingContinuousRecipe: data.creatingContinuousRecipe,
                    usedByRecipes: data.recipes,
                    customFields: data.dataset.customFields,
                    customFieldsPreview: DatasetCustomFieldsService.buildCustomFieldsPreviews(data.dataset.customFields)
                });

            }).error(setErrorInScope.bind($scope));
        }
        getCurrentDatasetFullInfo();

        $rootScope.$on('customFieldsSaved', function(event, item, customFields) {
            if (TopNav.sameItem(item, TopNav.getItem())) {
                let newItem = TopNav.getItem();
                newItem.data.customFields = customFields;
                newItem.data.customFieldsPreview = DatasetCustomFieldsService.buildCustomFieldsPreviews(customFields);
            }
        });

        $scope.newAnalysis = function() {
            GlobalProjectActions.smartNewAnalysis($scope, $stateParams.datasetName);
        };

        $scope.buildOpenDataset = function() {
            const loc = AnyLoc.makeLoc($stateParams.projectKey, $stateParams.datasetName);
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "DATASET", loc, { redirectToJobPage: true });
        };

        Mousetrap.bind("g e", function() {
            $state.go("projects.project.datasets.dataset.explore", {
                projectKey : $stateParams.projectKey,
                datasetName : $stateParams.datasetName
            });
        });
        Mousetrap.bind("g v", function() {
            $state.go("projects.project.datasets.dataset.visualize", {
                projectKey : $stateParams.projectKey,
                datasetName : $stateParams.datasetName
            });
        });

        // watch the DQ computations to update the current status
        const dqComputationSubscription = DataQualityComputationTrackingService.observeObjectChanges($stateParams.projectKey, $stateParams.datasetName, null).subscribe(() => {
            DataikuAPI.dataQuality.getDatasetCurrentDailyStatus(
                $stateParams.projectKey, $stateParams.projectKey, $stateParams.datasetName
            ).success(
                (res) => $scope.datasetFullInfo.dataQualityStatus = res
            ).error(() => {}); // this is not directly related to a user action, there is no point showing them an error
        });

        $scope.$on("$destroy", function() {
            Mousetrap.unbind("g e");
            Mousetrap.unbind("g v");
            dqComputationSubscription.unsubscribe();
        });
    });

    app.controller("DatasetNewController", function ($scope, $stateParams, DataikuAPI) {
    });


    app.controller("DatasetSettingsController", function ($scope, $rootScope, $state, $stateParams, $q, $timeout, $controller,
                                                          $location, Assert, TopNav, CachedAPICalls, Dialogs, DataikuAPI, WT1, ActivityIndicator, DatasetUtils, DKUtils, LoggerProvider) {
        var Logger = LoggerProvider.getLogger('datasets.settings');

        $scope.uiState = {
            activeTab: $location.hash() || 'connection'
        };

        $scope.$watch('uiState.activeTab', function(nv, ov) {
            if (nv && nv !== ov) {
                $location.hash(nv);
            }
        });

        $scope.anyPipelineTypeEnabled = function() {
            return $rootScope.projectSummary.sparkPipelinesEnabled || $rootScope.projectSummary.sqlPipelinesEnabled;
        };

        if ($stateParams.datasetName) {
            TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_DATASET, "settings");
        } else {
            TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_NEW_DATASET, "settings");
            TopNav.setItem(TopNav.ITEM_DATASET, "New dataset", { name: "New dataset", dummy: true,
                newDatasetType : $stateParams.type});
        }

        // Dataset Type specific controllers may set conditions
        // values for the form to be valid.
        // Conditions take a string as key.
        // The form is valid when all the predicates evaluate to true.
        $scope.saveHooks = [];
        $scope.canBeSaved = function() {
            for(var i = 0 ; i < $scope.saveHooks.length ; i++) {
                if(!$scope.saveHooks[i]()) {
                    return false;
                }
            }
            return true;
        };

        $scope.table = null;
        $scope.charsets = null;
        $scope.types = null;
        $scope.formats = null;
        $scope.loading = true;
        $scope.detectionResults = null;
        $scope.dataset_name = '';
        $scope.redetectFormatOnCoreParamsChanged = false;
        $scope.datasetShortStatus = null;

        /* ************************* Core loading and saving
         */
        var promises = [];
        promises.push(CachedAPICalls.datasetTypes);
        promises.push(CachedAPICalls.datasetCommonCharsets);
        promises.push(CachedAPICalls.datasetFormatTypes);
        promises.push(null);

        if ($stateParams.datasetName) {
            $scope.datasetName = $stateParams.datasetName;
            document.dispatchEvent(new CustomEvent("PushAllDatasetPromises"));
            promises.push(DataikuAPI.datasets.get($stateParams.projectKey, $stateParams.datasetName, $stateParams.projectKey));
            $scope.isNewDataset = false;
        } else {
            $scope.dataset = {
                'name': null,
                'projectKey': $stateParams.projectKey,
                'params': {
                    'variablesExpansionLoopConfig': {},
                },
                'partitioning': {dimensions:[]},
                'tags':[],
                'schema': {}
            };
            $scope.isNewDataset = true;
            Assert.trueish($stateParams.type, 'no type in stateParams');
            $scope.dataset.type = $stateParams.type;
            // FIXME
            $scope.datasetShortStatus = {};
        }

        $q.all(promises).then(function (values) {
            const initialHash = $location.hash();
            document.dispatchEvent(new CustomEvent("GotAllDatasetPromises"));
            Logger.info("All promises done, loading UI", values[4]);
            $scope.types = values[0];
            $scope.charsets = values[1];
            $scope.formats = values[2];

            $scope.formatsAsMap = angular.copy($scope.formats);
            for (var fmtIdx in $scope.formatsAsMap) {
                var fmt = $scope.formatsAsMap[fmtIdx];
                var o = {};
                for (var paramIdx in fmt.params) {
                    o[fmt.params[paramIdx].name] = fmt.params[paramIdx];
                }
                $scope.formatsAsMap[fmtIdx] = o;
            }
            if ($stateParams.datasetName) {
                $scope.dataset = values[4].data;
                $scope.dataset_name = $scope.dataset.name;
                $scope.origDataset = angular.copy($scope.dataset);
            } else {
                $scope.origDataset = angular.copy($scope.dataset);
                $scope.firstPreviewDone = true;
                /* No dataset yet -> Always redetect on core params changed */
                $scope.redetectFormatOnCoreParamsChanged = true;
            }

            $scope.datasetKindForConsistency = DatasetUtils.getKindForConsistency($scope.dataset);

            var datasetType = $scope.types[$scope.dataset.type];
            var ctrl = null;
            /* Load the correct additional functions */

            if (datasetType.customDataset) {
                $scope.interactionType = "custom-dataset";
                ctrl = "CustomDatasetController";
            } else if ($scope.dataset.type == "Inline" && $stateParams.datasetName) {
                $scope.interactionType = "editable";
                ctrl = "EditableDatasetController";
            } else if ($scope.dataset.type == "Inline") {
                $scope.interactionType = "managed_fslike";
                ctrl = "EditableDatasetController";
            } else if ($stateParams.datasetName && $scope.types[$scope.dataset.type].sql && $scope.dataset.managed) {
                $scope.interactionType = 'managed_sql';
                ctrl = "ManagedSQLDatasetController";
            } else if ($scope.dataset.type == "UploadedFiles"  && !$stateParams.datasetName) {
                $scope.interactionType = "upload_wizard";
                ctrl = "UploadWizardController";
            } else if ($scope.dataset.type == "MongoDB") {
                $scope.interactionType = "mongodb";
                ctrl = "MongoDBDatasetController";
            } else if ($scope.dataset.type == "SharePointOnlineList") {
                $scope.interactionType = "sharepointonlinelist";
                ctrl = "SharePointOnlineListDatasetController";
            } else if ($stateParams.datasetName && $scope.dataset.type == "Cassandra" && $scope.dataset.managed) {
                $scope.interactionType = "managed_cassandra";
                ctrl = "ManagedCassandraDatasetController";
            } else if($scope.dataset.type == "Twitter") {
                $scope.interactionType = 'twitter_stream';
                ctrl = "TwitterStreamDatasetController";
            } else if($scope.dataset.type == "ElasticSearch") {
                $scope.interactionType = 'elasticsearch';
                ctrl = "ElasticSearchDatasetController";
            } else if($scope.dataset.type == "DynamoDB") {
                $scope.interactionType = 'dynamodb';
                ctrl = "DynamoDBDatasetController";
            } else if ($scope.dataset.type == "JobsDB") {
                $scope.interactionType = 'jobsdb';
                ctrl = "MetricsDatasetController";
            } else if ($scope.dataset.type == "StatsDB") {
                $scope.interactionType = 'statsdb';
                ctrl = "StatsDBDatasetController";
            } else if ($scope.dataset.type == "ExperimentsDB") { // here, otherwise interactionType ends up being managed_fslike
                $scope.interactionType = 'experimentsdb';
                ctrl = "ExperimentsDBDatasetController";
            } else if ($scope.dataset.type == "Labels") {
                $scope.interactionType = 'labels';
                ctrl = "LabelsDatasetController";
            } else if ($stateParams.datasetName && $scope.dataset.managed) {
                $scope.interactionType = 'managed_fslike';
                ctrl = "ManagedFSLikeDatasetController";
            } else if ($scope.dataset.type == "hiveserver2") { // almost sql, but special handling of connections
                $scope.interactionType = 'external_hive';
                ctrl = "ExternalHiveDatasetController";
            } else if ($scope.types[$scope.dataset.type].sql) {
                $scope.interactionType = 'external_sql';
                ctrl = "ExternalSQLDatasetController";
            } else if ($scope.dataset.type == "Cassandra") {
                $scope.interactionType = "external_cassandra";
                ctrl = "ExternalCassandraDatasetController";
            } else {
                $scope.interactionType = 'external_other';
                ctrl =  "ExternalStreamOrientedDatasetController";
            }

            $controller(ctrl, {$scope:$scope});
            // As some controllers might update the scope's dataset on initialization (notably the EditableDatasetController), we need a refresh
            $scope.origDataset = angular.copy($scope.dataset);

            $scope.onLoadComplete();
            $scope.loading = false;
            $scope.loadDone = true;

            if (initialHash && $scope.uiState.activeTab !== initialHash) {
                $scope.uiState.activeTab = initialHash;
            }

            getDigestTime($scope, function(time) {
                WT1.event("page-dataset-loaded", {digestTime : time, isNew : !$scope.dataset.name});
            });
        }, function (errors) {
            setErrorInScope.bind($scope)(errors.data, errors.status, errors.headers);
        });

        //TODO @flow factorize
        $scope.resynchronizeMetastore = function() {
            Dialogs.confirmPositive($scope,
                'Hive metastore resynchronization',
                'Are you sure you want to resynchronize this dataset to the Hive metastore?')
                .then(function() {
                    ActivityIndicator.waiting('Synchronizing to Hive metastore...');
                    const datasetRef = {
                        type: 'DATASET',
                        projectKey: $stateParams.projectKey,
                        id: $scope.dataset.name
                    };
                    DataikuAPI.datasets.synchronizeOneHiveMetastore(datasetRef, $scope.dataset.params).success(function(data) {
                        if (data.anyMessage && (data.warning || data.error)) {
                            ActivityIndicator.hide();
                            Dialogs.infoMessagesDisplayOnly($scope, "Metastore synchronization", data);
                        } else {
                            // nothing to show
                            ActivityIndicator.success('Hive metastore successfully synchronized');
                        }
                    }).error(function(data, status, headers) {
                        ActivityIndicator.error("Failed to synchronize Hive metastore");
                        setErrorInScope.call($scope,data,status,headers);
                    });
                });
        };

        //TODO @flow factorize
        $scope.resynchronizeDataset = function() {
            ActivityIndicator.waiting('Synchronizing from Hive metastore...');
            WT1.event("update-from-hive");
            DataikuAPI.datasets.updateFromHive($stateParams.projectKey, $scope.dataset.name).success(function(data,status,headers){
                ActivityIndicator.success('Dataset successfully synchronized');
                DataikuAPI.datasets.get($stateParams.projectKey, $stateParams.datasetName, $stateParams.projectKey).success(function(data) {
                    $scope.dataset = data;
                    $scope.dataset_name = $scope.dataset.name;
                    $scope.origDataset = angular.copy($scope.dataset);
                }).error(setErrorInScope.bind($scope));
            }).error(function(data, status, headers) {
                ActivityIndicator.error("Failed to synchronize Hive metastore");
                setErrorInScope.call($scope,data,status,headers);
            });
        };

        $scope.$on('datasetSchemaChanged', (event, data) => {
            $scope.dataset.schema = data.schema;
            $scope.origDataset.schema = angular.copy(data.schema);
        });

        $scope.saveDataset = function() {
            if (!$stateParams.datasetName) {
                /* Creation */
                $scope.dataset.name = $scope.uiState.new_dataset_name;
                $scope.origDataset = null;
                return DataikuAPI.datasets.create($stateParams.projectKey, $scope.dataset, $stateParams.zoneId).success(function() {
                    WT1.event("create-dataset", {
                        connectionType: $scope.dataset ? $scope.dataset.type : "unknown",
                        partitioningFrom: $scope.dataset ? $scope.dataset.partitioning : "unknown"
                    });
                    $rootScope.$broadcast(dkuEvents.datasetChanged);
                    if ($scope.dataset && $scope.dataset.type && $scope.dataset.type === "Inline"
                        && $scope.dataset.params && $scope.dataset.params.importSourceType && $scope.dataset.params.importSourceType  === "NONE") {
                        // If we are creating an empty editable dataset, directly goes to the edit page,
                        // because everyone (ie: me, I and myself) always wants want to do that anyway
                        $state.transitionTo("projects.project.datasets.dataset.edit", {projectKey : $stateParams.projectKey,
                            datasetName : $scope.uiState.new_dataset_name});
                    } else {
                        $state.transitionTo("projects.project.datasets.dataset.explore", {projectKey : $stateParams.projectKey,
                            datasetName : $scope.uiState.new_dataset_name});
                    }
                }).error(setErrorInScope.bind($scope));

            } else {
                Assert.trueish($scope.dataset.name, 'dataset has no name');

                var saveAfterConflictCheck = function() {
                    return DataikuAPI.datasets.saveWithRecipesFixup($stateParams.projectKey, $scope.dataset, {}, false)
                        .error(setErrorInScope.bind($scope))
                        .then(function(saveResp) {
                            if (saveResp.data.result) {
                                // So the RHP knows the change without having to refresh the page
                                $rootScope.$broadcast('datasetSchemaChanged', {schema: saveResp.data.result.schema});
                                
                                return saveResp.data.result.versionTag;
                            } else {
                                return Dialogs.confirmInfoMessages($scope,
                                    "Confirm dataset saving", saveResp.data.messages).then(function(){
                                    return DataikuAPI.datasets.saveWithRecipesFixup($stateParams.projectKey, $scope.dataset, {}, true)
                                        .error(setErrorInScope.bind($scope))
                                        .then(function(forcedSaveResp) {
                                            Assert.trueish(forcedSaveResp.data.result, 'response has no results');
                                            return forcedSaveResp.data.result.versionTag;
                                        });
                                });
                            }
                        }).then(function(newVersionTag) {
                            $rootScope.$broadcast(dkuEvents.datasetChanged);
                            // Reset the modification detector
                            $scope.origDataset = angular.copy($scope.dataset);
                            $scope.dataset.versionTag = newVersionTag;
                            $scope.origDataset.versionTag = newVersionTag;
                        });
                };

                return DataikuAPI.datasets.checkSaveConflict($stateParams.projectKey,$scope.dataset)
                    .error(setErrorInScope.bind($scope))
                    .then(function({data: conflictResult}) {
                        if(!conflictResult.canBeSaved) {
                            return Dialogs.openConflictDialog($scope,conflictResult).then(
                                function(resolutionMethod) {
                                    if(resolutionMethod == 'erase') {
                                        return saveAfterConflictCheck();
                                    } else if(resolutionMethod == 'ignore') {
                                        return DKUtils.reloadState();
                                    }
                                }
                            );
                        } else {
                            return saveAfterConflictCheck();
                        }
                    });
            }
        };

        $scope.goToPreview = function() {
            $scope.uiState.activeTab = "preview";
            $scope.$broadcast('tabSelect', 'preview');
        };

        $scope.datasetIsDirty = function () {
            if (!$scope.dataset)
                return false;

            function cleanDimensionPatterns(input) {
                if (input == null) return null;
                const datasetToCheck = angular.copy(input);
                if (datasetToCheck && datasetToCheck.partitioning && datasetToCheck.partitioning.dimensions) {
                    for (var i = 0; i < datasetToCheck.partitioning.dimensions.length; i++) {
                        delete datasetToCheck.partitioning.dimensions[i].patterns;
                    }
                }
                return datasetToCheck;
            }

            function cleanSavedFiles(input) {
                if (input == null) return null;
                const datasetToCheck = angular.copy(input);
                delete datasetToCheck.savedFiles;
                return datasetToCheck;
            }

            function cleanFormatToggleInfo(input) {
                // Clean toggle-state type info in some types of params (these are used to indicate changes between formats the user is trying)
                // As the old values like sheetsToColumnOld are set when handling the response they would create a false dirty state 
                if (input == null) return null;
                const datasetToCheck = angular.copy(input);
                if (datasetToCheck.formatType === 'excel' && datasetToCheck.formatParams) {
                    delete datasetToCheck.formatParams.sheetsToColumnOld;
                }
                return datasetToCheck;
            }

            const datasetToCheck = cleanFormatToggleInfo(cleanSavedFiles(cleanDimensionPatterns($scope.dataset)));
            const origDataset = cleanFormatToggleInfo(cleanSavedFiles(cleanDimensionPatterns($scope.origDataset)));

            return !angular.equals(datasetToCheck, origDataset) || $scope.dataset && $scope.dataset.name != $scope.dataset_name;
        };

        checkChangesBeforeLeaving($scope, function(){
            if ($scope.uiState.bypassDirtinessCheck) {
                return false;
            }

            var danger = false;
            /* Not yet loaded */
            if (!$scope.dataset) return;
            if ($scope.dataset.name) {
                /* Existing dataset */
                danger = $scope.origDataset && $scope.datasetIsDirty();
            } else {
                /* New dataset, don't do bogus renaming check */
                danger = $scope.origDataset && !angular.equals($scope.dataset, $scope.origDataset);
            }
            return danger;
        });

        $scope.smartWatch = function (expr, apply) {
            var stop;
            $scope.$watch(expr, function (nv, ov) {
                if (stop) {
                    $timeout.cancel(stop);
                }
                stop = $timeout(apply, 1000);
            }, true);
        };

        // this function need to be in hooks to be usable from the right panel
        if($scope.datasetHooks) {
            $scope.datasetHooks.askForSaveBeforeRenaming = () => {
                if ($scope.datasetIsDirty()) {
                    return Dialogs.confirm($scope, "Save dataset settings", `Renaming requires dataset settings to be saved beforehand`, {
                        btnConfirm: 'Save',
                        positive: true,
                    }).then($scope.saveDataset);
                } else {
                    return $q.resolve();
                }
            }
        }

        $scope.supportsWriteSQLComment = function(type) {
            const supportedDB = ["Snowflake", "Databricks", "PostgreSQL", "Oracle", "Redshift", "MySQL", "BigQuery"]
            return supportedDB.includes(type)
        }

        $scope.supportsWriteSQLCommentInCreateTableStatement = function(type) {
            return ["Snowflake", "Databricks", "MySQL", "BigQuery"].includes(type);
        }

        Logger.info("Done loading dataset controller");
    });


    app.controller("DatasetHistoryController", function ($scope, TopNav) {
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_DATASET, "history");
    });
    
    app.controller("DatasetStatisticsController", function ($scope, TopNav, $state) {
        $scope.$state = $state;
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_DATASET, "statistics");
    });

    app.controller("DatasetDataQualityController", function ($scope, TopNav, $state, $stateParams, GraphZoomTrackerService) {
        $scope.$state = $state;
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_DATASET, "data-quality");

        // tracks the last view visited in order to be able to 'go back' from the edit screen
        $scope.lastViewStateVisited = 'current-status';
        $scope.$on('$stateChangeStart', function (event, toState, toParams, fromState) {
            const route = fromState.name;
            if(route.endsWith('data-quality.view.current-status')) {
                $scope.lastViewStateVisited = 'current-status';
            } else if(route.endsWith('data-quality.view.timeline')) {
                $scope.lastViewStateVisited = 'timeline';
            } else if(route.endsWith('data-quality.view.history')) {
                $scope.lastViewStateVisited = 'history';
            }
        });

        if (!$stateParams.fromFlow) {
            // Do not change the focus item zoom if coming from flow
            if ($stateParams.datasetFullName) {
                GraphZoomTrackerService.setFocusItemByFullName("dataset", $stateParams.datasetFullName);
            } else {
                GraphZoomTrackerService.setFocusItemByName("dataset", $stateParams.datasetName);
            }
        }
    });

    app.controller("DatasetMetricsController", function ($scope, TopNav, $state, $rootScope, $location) {
        $scope.$state = $state;
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_DATASET, "metrics");

        if ($scope.metricsCallbacks.isPartitioned()) {
            $scope.views = {
                    values: [{name:'Partitions table', id:'Table'}, {name:'Partitions histories', id:'Partitions'}, {name:'Last value', id:'Last value'}, {name:'History', id:'History'}],
                    selected: 'Table'
                };
        } else {
            $scope.views = {
                    values: [{name:'Last value', id:'Last value'}, {name:'History', id:'History'}],
                    selected: 'Last value'
                };
        }
        if ($scope.metricsCallbacks.hasColumnsView()) {
            $scope.views.values.push({name:'Columns', id:'Columns'});
        }

        $scope.displayedMetrics = {metrics : [], $loaded : false};

        // function is not there when the page is loaded the first time, but is there when tabs change
        if ( $scope.refreshAllComputedMetrics ) $scope.refreshAllComputedMetrics();
    });

    app.controller("ExperimentTrackingController", function ($scope, TopNav, $state) {
        $scope.$state = $state;
        TopNav.setLocation(TopNav.TOP_EXPERIMENT_TRACKING, TopNav.LEFT_EXPERIMENT_TRACKING, TopNav.TABS_NONE, null);
        TopNav.setNoItem();
    });

    app.controller("DatasetLabController", function ($scope, $controller, $stateParams, $state,
        $filter, DataikuAPI, DatasetUtils, DatasetLabUtils, GlobalProjectActions, $rootScope, StateUtils,
        CreateModalFromTemplate, Logger, DEEPHUB_PREDICTION_TYPE_CODE_ENV_TYPE, CachedAPICalls) {
        $controller('NotebooksCommons', { $scope: $scope });

        $scope.useCurrentAnalysisForNewMlTask = false;

        function fetchBackends(callback) {
            DataikuAPI.analysis.mlcommon.listBackends($stateParams.projectKey, $scope.datasetSmartName, '')
                .success(data => {
                    $scope.backends = data;
                    callback();
                }).error(setErrorInScope.bind($scope));
        }

        function generateAnalysisName(policy, taskData) {
            return DatasetLabUtils.generateAnalysisName(policy, taskData, $scope.datasetSmartName);
        }

        $scope.selectPolicy = function(taskData, policy) {
            DatasetLabUtils.updateTaskWithNewPolicy(taskData, policy, $scope.datasetSmartName);
        };

        function prepareGuessPolicies(taskData) {
            DatasetLabUtils.prepareGuessPolicies(taskData, $scope.backends);
        };

        function displayNewPredictionModal(policyGroup, resetColumn = true, predictionType = null) {
            // Force user to re-set the target variable each time the modal is opened.
            // And pick the default policy depending on the prediction guess policy group.
            if (resetColumn) {
                $scope.predictionTaskData.managedFolderSmartId = null;
                $scope.predictionTaskData.targetVariable = null;
                $scope.predictionTaskData.timeVariable = null;
                $scope.predictionTaskData.treatmentVariable = null;
            }
            $scope.predictionTaskData.timeseriesIdentifiers = [];

            if (policyGroup.startsWith('deephub-')) {
                $scope.deepHubCodeEnvsAvailable = {};
                function checkDeepHubCodeEnvAvailable(predictionType) {
                    DataikuAPI.codeenvs.checkDSSInternalCodeEnv(DEEPHUB_PREDICTION_TYPE_CODE_ENV_TYPE[predictionType])
                        .then(function ({data}) {
                            $scope.deepHubCodeEnvsAvailable[predictionType] = Object.keys(data).length > 0;
                        }).catch(setErrorInScope.bind($scope));
                };
                checkDeepHubCodeEnvAvailable(predictionType);
                const policy = $scope.predictionTaskData.guessPolicies[policyGroup].find(e => e.id === predictionType);
                DatasetLabUtils.updateTaskWithNewPolicy($scope.predictionTaskData, policy, $scope.datasetSmartName);
            } else {
                DatasetLabUtils.updateTaskWithDefaultPolicy($scope.predictionTaskData, policyGroup, $scope.datasetSmartName);
            }

            const template = getPredictionCreationModalTemplate(policyGroup);
            CreateModalFromTemplate(template, $scope, "NewPMLTaskModalController", (modalScope) => {
                modalScope.predictionType = predictionType;
                modalScope.policyGroup = policyGroup;
            });
        }

        function getPredictionCreationModalTemplate(policyGroup) {
            let doctorType;
            switch (policyGroup) {
            case 'deephub-computer-vision':
                doctorType = 'deephub';
                break;
            case 'timeseries-forecasting':
                doctorType = 'timeseries-forecasting';
                break;
            case 'causal-prediction':
                doctorType = 'causal-prediction';
                break;
            case 'auto':
            case 'expert':
                doctorType = 'prediction';
                break;
            default:
                Logger.error("Unknown guess policy group: " + policyGroup);
                return "";
            }
            return `/templates/datasets/create-${ doctorType }-modal.html`;
        }

        function prepareNewPredictionModal(column, mode, predictionType, managedFolderSmartId = null) {
            const loc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.datasetSmartName);
            const resetVariable = !column;
            if (column) {
                if (predictionType === "TIMESERIES_FORECAST") {
                    $scope.predictionTaskData.timeVariable = column;
                } else {
                    $scope.predictionTaskData.targetVariable = column;
                    if(mode === "deephub-computer-vision") {
                        $scope.predictionTaskData.managedFolderSmartId = managedFolderSmartId;
                    }
                }
            }

            // Retrieve dataset columns
            DataikuAPI.datasets.get(loc.projectKey, loc.name, $stateParams.projectKey).success(data => {
                $scope.possibleColumns = data.schema.columns.map(it => it.name); 

                // Retrieve the prediction guess policies only if not already done.
                if ($scope.predictionTaskData.guessPolicies) {
                    displayNewPredictionModal(mode, resetVariable, predictionType);
                } else {
                    CachedAPICalls.pmlGuessPolicies.then(pmlGuessPolicies => {
                        $scope.predictionTaskData.guessPolicies = pmlGuessPolicies;
                        prepareGuessPolicies($scope.predictionTaskData);

                        displayNewPredictionModal(mode, resetVariable, predictionType);
                    }).catch(setErrorInScope.bind($scope));
                }
            });
        }

        $scope.newPrediction = function(column, datasetSmartName, mode = 'auto', predictionType = null, managedFolderSmartId = null) {
            $scope.datasetSmartName = datasetSmartName;
            // Retrieve the available backends only if not already done.
            if ($scope.backends) {
                prepareNewPredictionModal(column, mode, predictionType, managedFolderSmartId);
            } else {
                fetchBackends(prepareNewPredictionModal.bind(this, column, mode, predictionType, managedFolderSmartId));
            }
        };

        $scope.newDeepLearningPrediction = function(datasetSmartName) {
            $scope.newPrediction(undefined, datasetSmartName, 'expert');
        }

        $scope.newDeepHubPrediction = function(datasetSmartName, predictionType) {
            $scope.newPrediction(undefined, datasetSmartName, "deephub-computer-vision", predictionType);
        }

        $scope.newTimeseriesForecastPrediction = function(datasetSmartName) {
            $scope.newPrediction(undefined, datasetSmartName, "timeseries-forecasting");
        }

        $scope.newCausalPrediction = function(datasetSmartName) {
            $scope.newPrediction(undefined, datasetSmartName, "causal-prediction");
        };

        function displayNewClusteringModal(mode) { 
            DatasetLabUtils.updateTaskWithDefaultPolicy($scope.clusteringTaskData, mode, $scope.datasetSmartName);

            CreateModalFromTemplate('/templates/datasets/create-clustering-modal.html', $scope, null, modalScope => {
                if (mode === "auto" || mode === "expert") {
                    modalScope.policyGroups = [
                        {
                            name: "auto",
                            displayName: "AutoML",
                            desc: "Let Dataiku create your models.",
                            img: "automated"
                        },
                        {
                            name: "expert",
                            displayName: "Expert",
                            desc: "Have full control over the creation of your models.",
                            img: "expert-spaceship"
                        }
                    ]
                }
                modalScope.taskData = $scope.clusteringTaskData;
                modalScope.canCreateMlTaskTemplate = function(taskData) {
                    return !!taskData.analysisName && !!taskData.backendType && !!taskData.selectedPolicy;
                };
            });
        }

        function prepareNewClusteringModal(mode = 'auto') {
            // Retrieve the clustering guess policies only if not already done.
            if ($scope.clusteringTaskData.guessPolicies) {
                displayNewClusteringModal(mode);
            } else {
                CachedAPICalls.cmlGuessPolicies.then(cmlGuessPolicies => {
                    $scope.clusteringTaskData.guessPolicies = cmlGuessPolicies;
                    prepareGuessPolicies($scope.clusteringTaskData);
                    $scope.clusteringTaskData.analysisName = generateAnalysisName($scope.clusteringTaskData.selectedPolicy);
                    displayNewClusteringModal(mode);
                }).catch(setErrorInScope.bind($scope));
            }
        }

        $scope.newClustering = function(datasetSmartName, mode) {
            $scope.datasetSmartName = datasetSmartName;

            // Retrieve the available backends only if not already done.
            if ($scope.backends) {
                prepareNewClusteringModal();
            } else {
                fetchBackends(prepareNewClusteringModal.bind(this, mode));
            }
        };

        $scope.newAnalysis = function(datasetSmartName) {
            CreateModalFromTemplate('/templates/analysis/new-analysis-modal.html', $scope, 'NewAnalysisModalController', (modalScope) => {
                modalScope.newAnalysis.datasetSmartName = datasetSmartName;
            });
        }
        
        $scope.predictionTaskData = {};
        $scope.clusteringTaskData = {};

        $scope.onBackendChange = DatasetLabUtils.updateTaskWithNewBackendType;

        $scope.createClusteringTemplate = function() {
            DataikuAPI.analysis.createClusteringTemplate(
                $stateParams.projectKey,
                $scope.datasetSmartName,
                $scope.clusteringTaskData.analysisName,
                $scope.clusteringTaskData.backendType,
                '',
                $scope.clusteringTaskData.selectedPolicy.id
            ).success(data => {
                $scope.dismiss && $scope.dismiss();
                $rootScope.mlTaskJustCreated = true;
                if ($scope.clusteringTaskData.selectedPolicy.id === 'ALGORITHMS' || $scope.clusteringTaskData.selectedPolicy.id === 'CUSTOM') {
                    $state.go('projects.project.analyses.analysis.ml.clustmltask.list.design.clustering-algorithms', { analysisId: data.analysisId, mlTaskId: data.mlTaskId});
                } else {
                    $state.go('projects.project.analyses.analysis.ml.clustmltask.list.results', { analysisId: data.analysisId, mlTaskId: data.mlTaskId });
                }
            }).error(setErrorInScope.bind($scope));
        };

        $scope.getNotebookHref = function (notebook) {
            if (notebook.type == "JUPYTER") {
                return StateUtils.href.jupyterNotebook(notebook.id, notebook.projectKey);
            } else if (notebook.type == "SQL") {
                return StateUtils.href.sqlNotebook(notebook.id, notebook.projectKey);
            }
        }
        
        // whenever the dataset is ready to be inspected, load the notebooks & analyses already on it
        $scope.$watch("datasetSmartName", nv => {
            if (!nv) return;
            $scope.usability = {};
            var parts = nv.match(/([^.]+)\.(.+)/) || [nv, $stateParams.projectKey, nv]; // [smart, project, dataset]
            DataikuAPI.datasets.getFullInfo($stateParams.projectKey, parts[1], parts[2])
                .success(function(data) {
                    var hasSql = false;
                    ["sql", "hive", "impala", "pig", "sql99"].forEach(function(thing) {
                        $scope.usability[thing] = GlobalProjectActions.specialThingMaybePossibleFromDataset(data.dataset, thing);
                        hasSql = hasSql || $scope.usability[thing].ok;
                    });
                    $scope.usability.spark = { ok: true };
                    if (!$rootScope.appConfig.sparkEnabled) {
                        if (!$rootScope.appConfig.communityEdition) {
                            $scope.usability.spark.click = $scope.showCERestrictionModal.bind(null, 'Spark');
                        } else if (!$rootScope.addLicInfo.sparkLicensed) {
                            $scope.usability.spark.ok = false;
                            $scope.usability.spark.reason = "Spark is not licensed";
                        } else {
                            $scope.usability.spark.ok = false;
                            $scope.usability.spark.reason = "Spark is not configured";
                        }
                    }
                }).error(setErrorInScope.bind($scope));

            $scope.newAnalysisName = "Analyze " + $scope.datasetSmartName;
            
            DataikuAPI.analysis.listOnDataset($stateParams.projectKey, $scope.datasetSmartName, true)
                .success(data =>  {
                    $scope.analysesWithoutMLTasks = data.filter(analysis => analysis.nbMLTasks === 0)
                    $scope.analysesWithMLTasks = data.filter(analysis => analysis.nbMLTasks !== 0)

                    data.forEach(function (analysis) {
                        analysis.modelCount = analysis.mlTasks.reduce((sum, task) => sum + task.modelCount, 0);
                        analysis.icon = analysis.nbMLTasks === 1 ? $filter('analysisTypeToIcon')(analysis.mlTasks[0].taskType, analysis.mlTasks[0].backendType, analysis.mlTasks[0].predictionType, 16) : 'dku-icon-ml-analysis-16';
                    });
                }).error(setErrorInScope.bind($scope));

            DataikuAPI.datasets.listNotebooks($stateParams.projectKey, nv)
                .success(data => {
                    $scope.notebooks = data;
                    data.forEach(function(nb) {
                        const lowerCaseLanguage = typeof nb.language === 'string' ? nb.language.toLowerCase() : '';
                        switch (nb.type) {
                            case 'JUPYTER':
                                if (lowerCaseLanguage.startsWith('python')) {
                                    nb.icon = 'python';
                                } else if (lowerCaseLanguage === 'ir' || lowerCaseLanguage === 'r'){
                                    nb.icon = 'r';
                                } else if (lowerCaseLanguage === 'julia') {
                                    nb.icon = 'julia';
                                } else if (['scala', 'toree'].includes(lowerCaseLanguage)) {
                                    nb.icon = 'spark_scala';
                                }
                                break;
                            case 'SQL': // @virtual(hive-hproxy), @virtual(hive-jdbc) => hive | @virtual(impala-jdbc) => impala | sql
                                nb.icon = (nb.connection.match(/^@virtual\((hive|impala)-\w+\)/) || [_,'sql'])[1];
                                break;
                            default:
                                nb.icon = 'nav_notebook';
                        }
                    });
                }).error(setErrorInScope.bind($scope));
        });

        $scope.newNotebook = () => {
            CreateModalFromTemplate("/templates/notebooks/new-notebook-modal.html", $scope, 'NewNotebookModalController');
        };

        $scope.newNotebookFromTemplate = () => {
            CreateModalFromTemplate("/templates/notebooks/new-notebook-from-template-modal.html", $scope, 'NewNotebookFromTemplateModalController');
        };

        $scope.newNotebookFromFile = () => {
            CreateModalFromTemplate("/templates/notebooks/new-notebook-from-file-modal.html", $scope, 'NewNotebookFromFileModalController');
        };
    });

    app.controller("NewPMLTaskModalController", function($scope, $state, $stateParams, $rootScope, $timeout, DataikuAPI, DatasetLabUtils) {
        $scope.canCreateMlTaskTemplate = function(taskData) {
            return !!taskData.analysisName && !!taskData.backendType && !!taskData.selectedPolicy;
        };

        $scope.updateAnalysisName = function() {
            $scope.predictionTaskData.analysisName = DatasetLabUtils.generateAnalysisName($scope.predictionTaskData.selectedPolicy, $scope.predictionTaskData, $scope.datasetSmartName);
        };

        $scope.createPredictionTemplate = function(predictionType) {
            const taskData = $scope.predictionTaskData;
            const policyId = $scope.predictionTaskData.selectedPolicy.id;

            if ($scope.useCurrentAnalysisForNewMlTask) {
                // Override default template creation functions in order to reuse the current Analysis instead of creating new one.
                DataikuAPI.analysis.pml.createAndGuess(
                    $scope.analysisCoreParams.projectKey,
                    $scope.analysisCoreParams.id,
                    taskData.targetVariable,
                    taskData.backendType,
                    taskData.backendName,
                    policyId,
                    predictionType,
                    taskData.managedFolderSmartId,
                    taskData.timeVariable,
                    taskData.timeseriesIdentifiers,
                    taskData.treatmentVariable,
                ).success(data => finishMlTaskCreation(policyId, { mlTaskId: data.id })).error(setErrorInScope.bind($scope));
            } else {
                DataikuAPI.analysis.createPredictionTemplate(
                    $stateParams.projectKey,
                    $scope.datasetSmartName,
                    taskData.analysisName,
                    taskData.backendType,
                    '',
                    taskData.targetVariable,
                    policyId,
                    taskData.managedFolderSmartId,
                    predictionType,
                    taskData.timeVariable,
                    taskData.timeseriesIdentifiers,
                    taskData.treatmentVariable
                ).success(data => finishMlTaskCreation(policyId, { analysisId: data.analysisId, mlTaskId: data.mlTaskId })).error(setErrorInScope.bind($scope));
            }
        };

        function finishMlTaskCreation(policyId, params) {
            $rootScope.mlTaskJustCreated = true;

            if (policyId === 'DEEP') {
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.design.classical-keras-build', params);
            } else if (policyId === 'ALGORITHMS' || policyId === 'CUSTOM') {
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.design.classical-algorithms', params);
            } else if (policyId.startsWith("DEEP_HUB")) {
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.design.deephub-learning', params);
            } else if (policyId.startsWith("TIMESERIES")) {
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.design.timeseries-learning', params);
            } else if (policyId.startsWith("CAUSAL")) {
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.design.causal-learning', params);
            } else {
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.results.sessions', params);
            }
        }

        $scope.forbiddenPredictionCreationReason = function(taskData) {
            const policyId = taskData.selectedPolicy.id;
            if (!taskData.targetVariable) {
                const targetRoleName = policyId === "CAUSAL_PREDICTION" ? "outcome" : "target";
                return `The ${targetRoleName} is missing`;
            }
            if (taskData.backendType === "DEEP_HUB" && !taskData.managedFolderSmartId) {
                return "The image folder is missing";
            }

            if (policyId.startsWith("TIMESERIES")) {
                if (!taskData.timeVariable) {
                    return "The time variable is missing";
                }
                if (taskData.timeVariable === taskData.targetVariable) {
                    return "The target and the time variables cannot be the same feature";
                }
                if ((taskData.timeseriesIdentifiers || []).some(id => !$scope.possibleColumns.includes(id))) {
                    return "Some of the given time series identifiers are not proper column names";
                }
                if ((taskData.timeseriesIdentifiers || []).includes(taskData.timeVariable)) {
                    return "The time variable cannot be a time series identifier";
                }
                if ((taskData.timeseriesIdentifiers || []).includes(taskData.targetVariable)) {
                    return "The target variable cannot be a time series identifier";
                }
            }

            if (policyId === "CAUSAL_PREDICTION") {
                if (!taskData.treatmentVariable) {
                    return "The treatment variable is missing";
                }
                if (taskData.treatmentVariable === taskData.targetVariable) {
                    return "The outcome and the treatment variables cannot be the same feature";
                }
            }
        }

        function initPolicyGroups() {
            if ($scope.policyGroup === "auto" || $scope.policyGroup === "expert") {
                $scope.policyGroups = [
                    {
                        name: "auto",
                        displayName: "AutoML",
                        desc: "Let Dataiku create your models.",
                        img: "automated"
                    },
                    {
                        name: "expert",
                        displayName: "Expert",
                        desc: "Have full control over the creation of your models.",
                        img: "expert-spaceship"
                    }
                ]
            }
            if ($scope.policyGroup === "timeseries-forecasting") {
                $scope.policyGroups = [
                    { name: "timeseries-forecasting" }
                ];

                $scope.isSpecialRole = function(column) {
                    return [$scope.predictionTaskData.targetVariable, $scope.predictionTaskData.timeVariable].includes(column);
                };
            }
            if ($scope.policyGroup === "causal-prediction") {
                $scope.policyGroups = [
                    { name: "causal-prediction" }
                ];
                $scope.isSpecialRole = function(column) {
                    return [$scope.predictionTaskData.targetVariable, $scope.predictionTaskData.treatmentVariable].includes(column);
                };
            }
        }

        const deregister = $scope.$watch("policyGroup", function(nv) {
            if (!nv) return;
            initPolicyGroups();
            deregister();
        });

        if (!$scope.predictionTaskData.targetVariable && $scope.appConfig.userProfile.mayVisualML) {
            // Trigger click on target input to display the dropdown menu.
            // Wait a bit till the modal height is computed so that the dropdown could be properly displayed.
            const delayedQuerySelector = $timeout(() => {
                if ($scope.policyGroup.startsWith('deephub-') && $scope.deepHubCodeEnvsAvailable && !$scope.deepHubCodeEnvsAvailable[$scope.predictionType]) {
                    return;
                }
                if ($scope.policyGroup.startsWith('causal-')) {
                    document.querySelector('#treatment-selector + .bootstrap-select').click();
                } else {
                    document.querySelector('#target-selector + .bootstrap-select').click();
                }
            }, 400);

            $scope.$on("$destroy", function() {
                $timeout.cancel(delayedQuerySelector);
            });
        }

        $scope.internalCodeEnvsHRef = function() {
            if ($scope.appConfig.isAutomation) {
                return $state.href("admin.codeenvs-automation.internal");
            } else {
                return $state.href("admin.codeenvs-design.internal");
            }
        }
    });

    app.service("DatasetLabUtils", function() {
        return {
            prepareGuessPolicies,
            updateTaskWithNewBackendType,
            generateAnalysisName,
            updateTaskWithNewPolicy,
            updateTaskWithDefaultPolicy,
        };

        //////////////////

        /**
         * For a given guess policy, enrich its supported backends with details if they are not currently available
         * @param {Object} policy
         * @param {Array} backends
        **/
        function updateGuessPolicyBackends(policy, backends) {
            const policyBackends = [];
            const policyBackendsDescriptions = [];

            backends.forEach(backend => {
                policy.supported_backends.forEach(function (supportedBackend) {
                    if (supportedBackend === backend.type) {
                        let title = backend.displayName;
                        let description = backend.description;
                        if (backend.statusMessage) {
                            title += ' (Unavailable)';
                            description += '<br/><strong>' + backend.statusMessage + '</strong>';
                        }
                        if (supportedBackend === "MLLIB") {
                            description += "<span class='text-warning faic mleft8'><i class='dku-icon-warning-fill-16 mright4'></i>MLLib is deprecated and will soon be removed.</span>"
                        }
                        backend.title = title;
                        policyBackends.push(backend);
                        policyBackendsDescriptions.push(description);
                    }
                });
            });

            policy.backends = policyBackends;
            policy.backendDomDescriptions = policyBackendsDescriptions;

            if (!policy.selectedBackend) {
                policy.selectedBackend = { displayName: angular.copy(policy.backends[0]).displayName };
            }
        }

        /**
         * Apply `updateGuessPolicyBackends` (see above) for each guess policy in each policy group
         * @param {Object} taskData
         * @param {Array} backends
        **/
        function prepareGuessPolicies(taskData, backends) {
            angular.forEach(taskData.guessPolicies, function(guessPolicyGroup) {
                guessPolicyGroup.forEach(policy => updateGuessPolicyBackends(policy, backends))
            });
        }

        /**
         * When the user selected a new backend in the ML task creation modal, look for the backend of the given policy that match the newly selected backend,
         * and if it exists, update the backend type of `taskData` with the type of the newly selected backend
         * @param {Object} taskData
         * @param {Object} policy
        **/
        function updateTaskWithNewBackendType(taskData, policy = taskData.selectedPolicy) {
            const foundBackend = policy.backends.find(backend => (backend.displayName === policy.selectedBackend.displayName));
            if (foundBackend) {
                taskData.backendType = foundBackend.type;
            }
        };

        /**
         * Generate the new analysis name, depending on the params given by the user
         * @param {Object} policy
         * @param {Object} taskData
         * @param {String} datasetSmartName
        **/
        function generateAnalysisName(policy, taskData, datasetSmartName) {
            if (!policy) {
                return "Analyze " + datasetSmartName;
            }

            let analysisName = policy.analysis_name.replace("{dataset}", datasetSmartName);

            if (taskData && taskData.targetVariable) {
                analysisName = analysisName.replace("{target}", taskData.targetVariable);
            }
            if (taskData && taskData.treatmentVariable) {
                analysisName = analysisName.replace("{treatment}", taskData.treatmentVariable);
            }
            return analysisName;
        };

        /**
         * When the user selects a new policy, update related fields (name of the analysis, selected backend)
         * @param {Object} taskData
         * @param {Object} policy
         * @param {String} datasetSmartName
        **/
        function updateTaskWithNewPolicy(taskData, policy, datasetSmartName) {
            taskData.selectedPolicy = policy;
            taskData.analysisName = generateAnalysisName(policy, taskData, datasetSmartName);
            updateTaskWithNewBackendType(taskData);
        };

        /**
         * Select the default policy (aka the first one) for a specified guess policy mode, and then update related fields
         * @param {Object} taskData
         * @param {String} mode
         * @param {String} datasetSmartName
        **/
        function updateTaskWithDefaultPolicy(taskData, mode, datasetSmartName) {
            updateTaskWithNewPolicy(taskData, taskData.guessPolicies[mode][0], datasetSmartName);
        }
    });

    app.controller("ProjectMassTableToDatasetController", function ($scope, Assert, $stateParams, $state, DataikuAPI, Dialogs, FutureProgressModal, $rootScope, WT1) {
        $scope.projectKey = $stateParams.projectKey;
        $scope.massImportData = null;
        $scope.uiState = {schemas:[], sourceSchema:null};

        $scope.acceptMassImport = function (zoneId) {
            $scope.candidates.hiveImportCandidates.forEach(candidate => {
                if (candidate.selectedConnectionId) {
                    candidate.selectedConnection = candidate.possibleConnections.find(pc => pc.id == candidate.selectedConnectionId)
                }
            });

            DataikuAPI.connections.massImportTableCandidates($stateParams.projectKey, $scope.candidates.sqlImportCandidates, $scope.candidates.hiveImportCandidates, $scope.candidates.elasticSearchImportCandidates, zoneId).success(function (data) {
                FutureProgressModal.show($scope, data, "Importing tables").then(function (importResult) {
                    if (importResult) {
                        var allCandidates = $scope.candidates.sqlImportCandidates
                            .concat($scope.candidates.hiveImportCandidates)
                            .concat($scope.candidates.elasticSearchImportCandidates);
                        
                        if(importResult.anyMessage) {
                            sendWT1Event({
                                importCount: importResult.messages.filter(m => m.severity === 'SUCCESS').length,
                                errorCount: importResult.messages.filter(m => m.severity === 'ERROR').length
                            });
                        }
                        
                        /* if only one item and success, go directly to it */
                        if (importResult.anyMessage && importResult.success && allCandidates.length == 1) {
                            $state.go("projects.project.datasets.dataset.explore", {
                                datasetName: allCandidates[0].datasetName
                            });
                        } else if (importResult.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Import report", importResult).then(() => {
                                $state.go("projects.project.datasets.list", {
                                    projectKey: $scope.projectKey
                                });
                            });
                        }
                    }
                });
            }).error(setErrorInScope.bind($scope));
        };

        $scope.getDatasetCountToImport = function() {
            if (!$scope.tableImportCandidates) {
                return 0;
            }
            if (!$scope.isElasticSearch) {
                return $scope.tableImportCandidates.length;
            }
            // For ElasticSearch import, if several candidates have the same dataset name, import them as an index pattern in a single dataset
            const datasets = new Set($scope.tableImportCandidates.map(c => c.datasetName));
            return datasets.size;
        }

        $scope.acceptMassImportAlation = function(tables) {
            DataikuAPI.connections.massImportSQL($stateParams.projectKey, $scope.uiState.connectionName, {'tables': tables}).success(function(data) {
                FutureProgressModal.show($scope, data, "Importing tables").then(function(importResult) {
                    if (importResult) {
                        if ($stateParams.fromExternal === "alation" && tables.length === 1 && importResult.maxSeverity === "INFO") {
                            // Fastpath when importing from Alation: skip the success message
                            $state.go("projects.project.datasets.dataset.explore", {"datasetName" : tables[0].datasetName});
                        } else if (importResult.anyMessage) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Import report", importResult).then(function() {$scope.massImportData = null;});
                        } else {
                            $scope.massImportData = null;
                        }
                    }
                });
            }).error(setErrorInScope.bind($scope));
        };

        let importData = JSON.parse($stateParams.importData);
        $scope.isElasticSearch = importData !== null && importData.workflowType == "ELASTIC_SEARCH";

        $scope.removeOneCandidate = function(candidate) {
            $scope.tableImportCandidates.splice($scope.tableImportCandidates.indexOf(candidate), 1);

            const candidatesSources = [$scope.candidates.sqlImportCandidates, $scope.candidates.hiveImportCandidates, $scope.candidates.elasticSearchImportCandidates];
            for (const candidates of candidatesSources) {
                const index = candidates.indexOf(candidate);
                candidates.splice(index, 1);
            }
        }


        function refreshList() {
            function cb(data){
                $scope.candidates = data;
                $scope.tableImportCandidates = [];
                data.sqlImportCandidates.forEach(e => {
                    e.connectionType = 'SQL';
                    $scope.tableImportCandidates.push(e);
                });
                data.hiveImportCandidates.forEach(e => {
                    e.connectionType = 'HIVE';
                    if (e.selectedConnection) {
                        e.selectedConnectionId = e.selectedConnection.id;
                    }
                    $scope.tableImportCandidates.push(e);
                });
                data.elasticSearchImportCandidates.forEach(e => {
                    e.connectionType = 'ELASTIC_SEARCH';
                    $scope.tableImportCandidates.push(e);
                });
            }

            if (importData.workflowType == "KEYS") {
                DataikuAPI.connections.getTableImportCandidatesFromKeys(importData.tableKeys, importData.remarks, $scope.projectKey).success(function(data){
                    FutureProgressModal.show($scope, data, "Get import data").then(cb)
                }).error(setErrorInScope.bind($scope));
            } else if (importData.workflowType == "SQL" || importData.workflowType == "HIVE" || importData.workflowType == "ELASTIC_SEARCH") {
                DataikuAPI.connections.getTableImportCandidatesFromExplorer(importData.workflowType, importData.selectedTables, $scope.projectKey).success(function(data){
                    FutureProgressModal.show($scope, data, "Get import data").then(cb)
                }).error(setErrorInScope.bind($scope));
            } else if (importData.workflowType == "ALATION_MCC") {
                DataikuAPI.connections.getTableImportCandidatesFromAlationMCC($stateParams.projectKey, importData.alationSelection)
                    .success(cb)
                    .error(setErrorInScope.bind($scope));
            }
        }

        addDatasetUniquenessCheck($scope, DataikuAPI, $scope.projectKey);

        if ($stateParams.importData) {

            if ($stateParams.fromExternal === "alation") {
                $scope.uiState.fromExternal = "alation";
                Assert.inScope($rootScope, 'alationCatalogSelection');
                DataikuAPI.connections.listMassImportSQLFromAlation($stateParams.projectKey, $rootScope.alationCatalogSelection).success(function(data){
                    $scope.massImportData = {"tables": data};
                    $scope.uiState.connectionName = data.connectionName;
                    data.tables.forEach(function(x) { x.checked = true; });
                }).error(setErrorInScope.bind($scope));
            } else {
                refreshList();
            }

        }

        function sendWT1Event(resultData) {
            WT1.event('import-external-tables', {
                ...importData.wt1Context,
                ...resultData,
                targetProjecth: md5($scope.projectKey)
            });
        }
    });

    app.controller("AlationOpenController", function ($scope, $stateParams, $state, DataikuAPI, GlobalProjectActions, Fn, TopNav, ActivityIndicator, Dialogs, MonoFuture, FutureProgressModal, $rootScope) {
        TopNav.setNoItem();

        $scope.importNewDataset = {}

        $scope.import = function(){
            $state.go("projects.project.tablesimport", {
                projectKey : $scope.importNewDataset.targetProjectKey,
                importData : JSON.stringify({
                    workflowType : "ALATION_MCC",
                    alationSelection: $scope.alationOpen.catalogSelection
                })
            })
        }

        DataikuAPI.connections.getAlationOpenInfo($stateParams.alationOpenId).success(function(data){
            $scope.alationOpen = data;
            if (data.connectionName) $scope.uiState.connectionName = data.connectionName;
            //data.tables.forEach(function(x) { x.checked = true; });
        }).error(setErrorInScope.bind($scope));
    });

    app.controller("ProjectsProjectMassImportController", function ($scope, CreateModalFromTemplate, $stateParams, $state, DataikuAPI, GlobalProjectActions, Fn, TopNav, ActivityIndicator, Dialogs, MonoFuture, FutureProgressModal) {
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_NONE, null);
        TopNav.setNoItem();
    });

    app.service("ConnectionExplorationService", function(CreateModalFromTemplate, MonoFuture, DataikuAPI, $state, $stateParams, $timeout, Logger, translate) {
        const svc = {}
        const ANY = "_any_";
        const anyLabel = translate("CONNECTION_EXPLORER.GLOBAL.FILTER.NO_RESTRICTIONS", "No restrictions");

        /* HANA organizes tables and calculation views into virtual packages embedded in table names,
         * in the form of package.name/table:name:in:the:package
         * or /package.name:sometimes:with:colons/table:name:in:the:package.
         * Since it's messy for users, allow quick filtering on these virtual packages */
        svc.buildHANAPackagesList = function(massImportData) {
            massImportData.hanaPackages = null;
            if (massImportData.connectionType == "SAPHANA") {
                const packagesSet = new Set(massImportData.tables.map(e => {
                    let firstSlash = e.table.indexOf("/");
                    if (firstSlash == 0) {
                        firstSlash = e.table.indexOf("/", 1);
                    }
                    if (firstSlash > 0) {
                        return e.table.substring(0, firstSlash)
                    } else {
                        return null;
                    }
                }));
                packagesSet.delete(null);
                if (packagesSet.size > 0) {
                    massImportData.hanaPackages =  Array.from(packagesSet);
                }
            }
        }

        svc.previewInModal = function(parentScope, connectionType, table) {
            Logger.info("Previewing table", table)
            CreateModalFromTemplate("/templates/datasets/modals/catalog-explorer-preview-modal.html", parentScope, null, function(newScope) {
                let previewMonoFuture = null;
                if (connectionType === "ElasticSearch") {
                    previewMonoFuture = MonoFuture(newScope).wrap(DataikuAPI.datasets.elasticsearch.preview);
                    newScope.previewName = table.name;
                    newScope.previewDataset = {
                        type: connectionType,
                        params: {
                            index: table.name,
                            connection: table.connectionName,
                        },
                        partitioning: {dimensions:[]},
                        tags:[],
                        schema: {}
                    };
                } else {
                    previewMonoFuture = MonoFuture(newScope).wrap(DataikuAPI.datasets.externalSQL.preview);
                    newScope.previewName = table.table;
                    newScope.previewDataset = {
                        type: connectionType,
                        params: {
                            catalog: table.catalog,
                            schema: table.schema,
                            table: table.table,
                            connection: table.connectionName,
                            mode: "table"
                        },
                        partitioning: {dimensions:[]},
                        tags:[],
                        schema: {}
                    };
                }
                newScope.status = 'NOT_ASKED';
                newScope.previewResult = null;
                previewMonoFuture(newScope.previewDataset).success(function (response) {
                    newScope.previewResult = response.result;
                    if ((!newScope.previewResult.connectionOK && (newScope.previewResult.connectionError || newScope.previewResult.connectionErrorMsg))
                        || (!newScope.previewResult.queryOK && newScope.previewResult.queryError)) {
                        newScope.status = 'FAILURE';
                        setErrorDetailsInScope.bind(newScope)(newScope.previewResult.connectionError || newScope.previewResult.connectionErrorMsg || newScope.previewResult.queryError);
                    } else {
                        newScope.status = 'SUCCESS';
                        if ((newScope.previewResult.queryOK && newScope.previewResult.querySchema) || (connectionType === "ElasticSearch" && newScope.previewResult.preview)) {
                            newScope.previewDataset.schema = newScope.previewResult.schemaDetection.newSchema;
                        }
                    }
                }).update(function() {
                    newScope.status = 'PENDING';
                }).error(function (a, b, c) {
                    newScope.status = 'FAILURE';
                    setErrorInScope.bind(newScope)(a,b,c);
                });
            });
        }

        svc.handleListRefreshInScope = function($scope, connectionName, explorerSettings, updateState) {
            Logger.info("Refreshing tables list, connectionName=", connectionName,
                        "catalog=", explorerSettings.catalog,
                        "schema=", explorerSettings.schema,
                        "isHDFS=", explorerSettings.isHDFS,
                        "isElasticSearch=", explorerSettings.isElasticSearch);

            let monofuture;

            if (explorerSettings.isHdfs || explorerSettings.isElasticSearch) {
                const endPoint = explorerSettings.isElasticSearch ? DataikuAPI.connections.listElasticSearchMassImportIndices : DataikuAPI.connections.listHiveMassImportTables;
                monofuture = MonoFuture($scope).wrap(endPoint)(connectionName, $stateParams.projectKey).success(function (data) {
                    $state.go('.', {connectionName: connectionName, schemaName: null, catalogName: null}, {notify: false});
                    $scope.listFuture = null;
                    $scope.showInputScreen = false;

                    if (explorerSettings.isElasticSearch) {
                        data.result.tables.forEach( e => e.aliasesOrIndicesCount = (e.aliasesOrIndices && e.aliasesOrIndices.length) || 0)
                    }
                    $timeout(() => {
                        $scope.massImportData = {"tables": data.result.tables, connectionType: explorerSettings.isHdfs ? "HDFS" : "ElasticSearch"};
                        $scope.hasSystemTables = false;
                    });
                });
            } else {
                const catalog = explorerSettings.catalog === ANY ? null : explorerSettings.catalog;
                const schema = explorerSettings.schema === ANY ? null : explorerSettings.schema;
                monofuture = MonoFuture($scope).wrap(DataikuAPI.connections.listSQLMassImportTables)(connectionName, catalog, schema, $stateParams.projectKey).success(function (data) {
                    if (updateState) {
                        $state.go('.', {
                            connectionName: $scope.connection.name,
                            catalogName: $scope.catalog,
                            schemaName: $scope.schema
                        }, {notify: false});
                    }

                    $scope.listFuture = null;
                    $scope.showInputScreen = false;

                    $timeout(() => {
                        $scope.massImportData = data.result;
                        $scope.isCatalogPresent = $scope.massImportData.tables.find(e => e && e.catalog);
                        $scope.isSchemaPresent = $scope.massImportData.tables.find(e => e && e.schema);
                        $scope.hasSystemTables = $scope.massImportData.tables.find(e => e && e.systemTable);
                        svc.buildHANAPackagesList($scope.massImportData);
                    });
                });
            }
            monofuture.update(function (data) {
                $scope.listFuture = data;
            }).error(function (data, status, headers) {
                $scope.listFuture = null;
                setErrorInScope.bind($scope)(data, status, headers);
            });
        }

        svc.fetchSchemas = function(catalogAware, connectionName, projectKey, targetStruct, reportErrorFn) {
            if (catalogAware) {
                DataikuAPI.connections.listSQLMassImportSchemasWithCatalogs(connectionName, projectKey).success(function(data) {
                    targetStruct.connectionOfSchemas = connectionName;
                    targetStruct.fetchedSchemas = data;
                    const uniqueCatalogs = [...new Set(data.map(schema => schema.catalog))];
                    const uniqueSchemas = [...new Set(data.map(schema => schema.schema))];
                    targetStruct.catalogs = [{ label: anyLabel, catalog: ANY }].concat(uniqueCatalogs.map(c => ({
                        label: c,
                        catalog: c
                    })));
                    targetStruct.schemas = [{ label: anyLabel, schema: ANY }].concat(uniqueSchemas.map(s => ({
                        label: s,
                        schema: s
                    })));
                    if (targetStruct.catalog && targetStruct.catalog !== ANY && !uniqueCatalogs.contains(targetStruct.catalog)) {
                        targetStruct.catalog = null;
                    }
                    if (targetStruct.schema && targetStruct.schema !== ANY && !uniqueSchemas.contains(targetStruct.schema)) {
                        targetStruct.schema = null;
                    }
                }).error(reportErrorFn);
            } else {
                DataikuAPI.connections.listSQLMassImportSchemas(connectionName, projectKey).success(function(data) {
                    targetStruct.connectionOfSchemas = connectionName;
                    targetStruct.catalogs = [{ label: anyLabel, catalog: ANY }];
                    targetStruct.schemas = [{ label: anyLabel, schema: ANY }].concat(data.map(s => ({
                        label: s,
                        schema: s
                    })));
                    targetStruct.catalog = ANY;
                    if (targetStruct.schema && targetStruct.schema !== ANY && !data.contains(targetStruct.schema)) {
                        targetStruct.schema = null;
                    }
                }).error(reportErrorFn);
            }
        }

        return svc; 
    })

    app.service("SqlConnectionNamespaceService", function(MonoFuture, DataikuAPI, $stateParams, translate, WT1) {
        const svc = {}

        svc.setTooltips = function(targetScope, connectionType) {
            targetScope.listSqlCatalogsTooltip = (connectionType === 'BigQuery') ?
                translate("GLOBAL.REFRESH_CATALOGS.TOOLTIP.BIGQUERY", "List BigQuery projects")
                : (connectionType === 'Snowflake') ? translate("GLOBAL.REFRESH_CATALOGS.TOOLTIP.SNOWFLAKE", "List databases")
                : (connectionType === 'SQLServer') ? translate("GLOBAL.REFRESH_CATALOGS.TOOLTIP.SQLSERVER", "List databases")
                : translate("GLOBAL.REFRESH_CATALOGS.TOOLTIP", "List catalogs");

            targetScope.listSqlSchemasTooltip = (connectionType === 'BigQuery') ?
                translate("GLOBAL.REFRESH_SCHEMAS.TOOLTIP.BIG_QUERY", "List BigQuery datasets")
                : translate("GLOBAL.REFRESH_SCHEMAS.TOOLTIP", "List schemas");
        }

        svc.resetState = function(targetScope, targetDatasetSettings) {
            targetScope.availableCatalogs = null;
            targetScope.availableSchemasMap = {};
            if (targetDatasetSettings) {
                targetDatasetSettings.overrideSQLSchema = null;
                targetDatasetSettings.overrideSQLCatalog = null;
            }
            svc.abortListSqlCatalogs(targetScope);
            svc.abortListSqlSchemas(targetScope);
        }

        svc.listSqlCatalogs = function(connectionName, targetScope, origin, connectionType, inputElementId) {
            targetScope.refreshCatalogOngoing = true;
            const monoFuturelistSqlCatalogs = MonoFuture(targetScope);
            targetScope.monoFuturelistSqlCatalogs = monoFuturelistSqlCatalogs;
            monoFuturelistSqlCatalogs.wrap(DataikuAPI.connections.listSQLCatalogs)(connectionName, $stateParams.projectKey)
                .noSpinner()
                .success(function(data) {
                    targetScope.monoFuturelistSqlCatalogs = undefined;
                    targetScope.availableCatalogs = data.result;
                    targetScope.refreshCatalogOngoing = false;
                    // the input element is not rendered yet at that point, a setTimeout is needed
                    setTimeout(function() {
                        const inputEltName = inputElementId ? inputElementId : 'catalog_input';
                        const inputElt = document.getElementById(inputEltName);
                        if (inputElt) {
                            inputElt.focus();
                        }
                    }, 50); // we need a very small window otherwise the list of available catalogs is not populated

                    WT1.event("refresh-catalogs",{
                        origin : origin,
                        connectionType : connectionType,
                        status: 'OK'
                    });

                }).error(function(error, status, headers) {
                    targetScope.refreshCatalogOngoing = false;
                    setErrorInScope.bind(targetScope)(error, status, headers);
                    targetScope.monoFuturelistSqlCatalogs = undefined;

                    WT1.event("refresh-catalogs",{
                        origin : origin,
                        connectionType : connectionType,
                        status: 'KO'
                    });
                });
        }

        svc.listSqlSchemas = function(connectionName, targetScope, inputCatalog, origin, connectionType, inputElementId) {
            targetScope.refreshSchemaOngoing = true;
            const monoFuturelistSqlSchemas = MonoFuture(targetScope);
            targetScope.monoFuturelistSqlSchemas = monoFuturelistSqlSchemas;
            monoFuturelistSqlSchemas.wrap(DataikuAPI.connections.listSQLSchemas)(connectionName, $stateParams.projectKey, inputCatalog)
                .noSpinner()
                .success(function(data) {
                    targetScope.monoFuturelistSqlSchemas = undefined;
                    targetScope.availableSchemasMap[inputCatalog || ''] = data.result;
                    targetScope.refreshSchemaOngoing = false;
                    // the input element is not rendered yet at that point, a setTimeout is needed
                    setTimeout(function() {
                        const inputEltName = inputElementId ? inputElementId : 'schema_input';
                        const inputElt = document.getElementById(inputEltName);
                        if (inputElt) {
                            inputElt.focus();
                        }
                    }, 50); // we need a very small window otherwise the list of available schemas is not populated

                    WT1.event("refresh-schemas",{
                        origin : origin,
                        connectionType : connectionType,
                        status: 'OK'
                    });

                }).error(function(error, status, headers) {
                    targetScope.refreshSchemaOngoing = false;
                    setErrorInScope.bind(targetScope)(error, status, headers);
                    targetScope.monoFuturelistSqlSchemas = undefined;

                    WT1.event("refresh-schemas",{
                        origin : origin,
                        connectionType : connectionType,
                        status: 'KO'
                    });
                });
        }

        svc.abortListSqlCatalogs = function(targetScope) {
            if (targetScope.refreshCatalogOngoing && targetScope.monoFuturelistSqlCatalogs) {
                targetScope.monoFuturelistSqlCatalogs.abort();
                targetScope.refreshCatalogOngoing = false;
                targetScope.monoFuturelistSqlCatalogs = undefined;
            }
        };

        svc.abortListSqlSchemas = function(targetScope) {
            if (targetScope.refreshSchemaOngoing && targetScope.monoFuturelistSqlSchemas) {
                targetScope.monoFuturelistSqlSchemas.abort();
                targetScope.refreshSchemaOngoing = false;
                targetScope.monoFuturelistSqlSchemas = undefined;
            }
        };

        return svc;
    })

    app.controller("ConnectionsExplorerController", function ($scope, $filter, CreateModalFromTemplate, ConnectionExplorationService, $stateParams, $state, DataikuAPI, GlobalProjectActions, Fn, TopNav, ActivityIndicator, Dialogs, MonoFuture, FutureProgressModal, $timeout, translate) {
        const ANY = '_any_';
        const anyLabel = translate("CONNECTION_EXPLORER.GLOBAL.FILTER.NO_RESTRICTIONS", "No restrictions");

        if ($stateParams.projectKey) {
            TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_NONE, null);
            TopNav.setNoItem();
            $scope.massImportTargetProjectKey = $stateParams.projectKey;
        }
        $scope.catalog = $stateParams.catalogName ? $stateParams.catalogName : ANY;
        $scope.schema = $stateParams.schemaName ? $stateParams.schemaName : ANY;
        $scope.schemas = [{ label: anyLabel, schema: ANY }];
        $scope.catalogs = [{ label: anyLabel, catalog: ANY }];
        $scope.massImportData = null;
        $scope.uiState = {schemas: [], sourceSchema: null, importConnectionName: null};
        $scope.selection = {orderReversed: false, filterQuery: {systemTable: 'false'} };
        $scope.isCatalogPresent = false;
        $scope.hasSystemTables = false;
        $scope.connection = null;

        const connectionNameFormatter = $filter('connectionNameFormatter');
        $scope.getConnectionName = (connection) => connectionNameFormatter(connection.displayName);

        $scope.importTables = function (zoneId) {
            if ($stateParams.projectKey) {
                $state.go('projects.project.tablesimport', {
                    projectKey: $stateParams.projectKey,
                    importData: JSON.stringify($scope.getImportData()),
                    zoneId
                });
            } else {
                CreateModalFromTemplate("/templates/datasets/tables-import-project-selection-modal.html", $scope,
                    "TablesImportProjectSelectionModalController");
            }
        };
        $scope.sortBy = function (columnName) {
            if ($scope.selection.orderQuery == columnName) {
                $scope.selection.orderReversed = !$scope.selection.orderReversed;
            } else {
                $scope.selection.orderQuery = columnName;
            }
        };
        $scope.isSortedBy = function (columnName, reversed) {
            if (!$scope.selection || !$scope.selection.orderQuery) return false;
            return $scope.selection.orderQuery == columnName && $scope.selection.orderReversed === reversed;
        };
        $scope.getImportData = function () {
            let workflowType;
            if ($scope.isHdfs) {
                workflowType = "HIVE";
            } else if ($scope.isElasticSearch) {
                workflowType = "ELASTIC_SEARCH"
            } else {
                workflowType = "SQL";
            }
            return {
                wt1Context: {
                    from: 'connection-explorer'
                },
                workflowType,
                selectedTables : $scope.selection.selectedObjects
            };
        };

        $scope.isCatalogAware = function() {
            if (!$scope.connection) {
                return false;
            }
            if (['Databricks', 'BigQuery', 'Snowflake', 'SQLServer', 'Trino'].includes($scope.connection.type)) {
                return true;
            }
            if ($scope.connection.type == "JDBC") {
                return $scope.connection.jdbcConnectionSupportsBrowsingCatalogs;
            }
            return false;
        };

        function assignConnectionRelatedVariables() {
            $scope.isElasticSearch = $scope.connection.type === 'ElasticSearch';
            $scope.isElasticSearchV7 = $scope.isElasticSearch && $scope.connection.dialect === 'ES_7';
            $scope.isHdfs = $scope.connection.type === 'HDFS';
            if ($scope.isHdfs) {
                $scope.selection.orderQuery = "name";
            } else {
                $scope.selection.orderQuery = "key.name";
            }
            if ($scope.isElasticSearch || $scope.isHdfs) {
                delete $scope.selection.filterQuery.systemTable; // no systemTable for ES and Hive, so remove it from the filterQuery
            }
            if ($scope.connection.defaultCatalogAndSchemaInputMode == "DROPDOWN") {
                $scope.switchToDropdownMode();
            } else {
                $scope.switchToFreeTextMode();
            }

            // Should we only do it if one is not already selected? Let's do it always for the time being
            if ($scope.connection.defaultCatalog) {
                $scope.catalog = $scope.connection.defaultCatalog;
            }
            if ($scope.connection.defaultSchema) {
                $scope.schema = $scope.connection.defaultSchema;
            }
        }

        $scope.refreshList = function () {
            ConnectionExplorationService.handleListRefreshInScope($scope, $scope.connection.name, $scope, true);
        };

        $scope.indeterminateSelectionState = function() {
            if ($scope.massImportData) {
                var tables = $filter('filter')($scope.massImportData.tables, $scope.uiState.query);
                var selectedCount = tables.filter(function(table) {
                    return table.checked;
                }).length;
                var unselectedCount = tables.filter(function(table) {
                    return !table.checked;
                }).length;
                return selectedCount != tables.length && unselectedCount != tables.length;
            } else {
                return false;
            }
        };

        $scope.showPreview = function(table) {
            ConnectionExplorationService.previewInModal($scope, $scope.massImportData.connectionType, table);
        }

        DataikuAPI.connections.listMassImportSources($stateParams.projectKey).success(function(data) {
            $scope.massImportSourcesResult = data;
            $scope.connections = data.sources;
            if ($stateParams.connectionName) {
                $scope.connection = $scope.connections.find(c => {
                    return c.name === $stateParams.connectionName
                });
                assignConnectionRelatedVariables();
                if (($scope.connection.type === "ElasticSearch" && $scope.connection.dialect === "ES_7")|| $scope.connection.type === 'HDFS' || $stateParams.schemaName) {
                    $scope.refreshList();
                } else {
                    $scope.showInputScreen = true;
                }
            } else {
                $scope.showInputScreen = true;
            }
        }).error(setErrorInScope.bind($scope));

        $scope.$watch('connection', (connection, oldVal) => {
            if (connection) {
                $scope.massImportData = { "tables": [] };
                $scope.hasSystemTables = false;
                assignConnectionRelatedVariables();

                if (!$scope.isHdfs && $scope.connection.name != $scope.connectionOfSchemas) {
                    $scope.connectionOfSchemas = null;
                    $scope.fetchedSchemas = null;
                    $scope.schemas = [{ label: anyLabel, schema: ANY }];
                    $scope.catalogs = [{ label: anyLabel, catalog: ANY }];
                    if (oldVal) {
                        // We are changing the connection
                        $scope.catalog = ANY;
                        $scope.schema = ANY;
                    } else {
                        // We are loading the connection for the first time
                        if ($scope.schema !== ANY) {
                            $scope.schemas = $scope.schemas.concat({ label: $scope.schema, schema: $scope.schema });
                        }
                        if ($scope.catalog !== ANY) {
                            $scope.catalogs = $scope.catalogs.concat({ label: $scope.catalog, catalog: $scope.catalog });
                        }
                    }
                }
            }
        });

        $scope.switchToFreeTextMode = function() {
            $scope.currentCatalogAndSchemaSelectionMode = "FREE_TEXT";
            if ($scope.catalog === ANY) {
                $scope.catalog = "";
            }
            if ($scope.schema === ANY) {
                $scope.schema = "";
            }
        }

        $scope.switchToDropdownMode= function() {
            $scope.currentCatalogAndSchemaSelectionMode = "DROPDOWN";
            if ($scope.catalog === "") {
                $scope.catalog = ANY;
            }
            if ($scope.schema === "") {
                $scope.schema = ANY;
            }
        }

        $scope.$watch('catalog', (newVal, oldVal) => {
            // Update the available schemas depending on the selected catalog
            if (newVal !== oldVal && $scope.fetchedSchemas) {
                let availableSchemas;
                if (newVal && newVal !== ANY) {
                    availableSchemas = $scope.fetchedSchemas.filter(schema => schema.catalog === newVal);
                } else {
                    availableSchemas = $scope.fetchedSchemas;
                }
                const uniqueSchemas = [...new Set(availableSchemas.map(schema => schema.schema))];
                $scope.schemas = [{ label: anyLabel, schema: ANY }].concat(uniqueSchemas.map(s => ({
                    label: s,
                    schema: s
                })));

                if ($scope.schema !== ANY && !uniqueSchemas.includes($scope.schema)) {
                    $scope.schema = ANY;
                }
            }
        });

        $scope.fetchSchemas = function() {
            ConnectionExplorationService.fetchSchemas($scope.isCatalogAware(), $scope.connection.name, $stateParams.projectKey, $scope, setErrorInScope.bind($scope));
        };
    });

    app.controller("DatasetSearchController", function ($scope, TopNav, $state) {
        $scope.$state = $state;
        TopNav.setLocation(TopNav.TOP_FLOW, "datasets", TopNav.TABS_DATASET, "search");
    });

})();
