(function() {
'use strict';

const app = angular.module('dataiku.apideployer');

app.service('APIDeployerAsyncHeavyStatusLoader', function(DataikuAPI, Logger, DeployerUtils) {
    return {
        newLoader: function(heavyStatusPerDeployment) {
            const loader = {};
            const oneByOneDeploymentIdList = [];
            const perInfraList = [];

            let canLoadStatus = true;
            loader.stopLoading = function() {
                canLoadStatus = false;
            };

            let loadingOneByOne = true;
            let loadingPerInfra = true;

            loader.isLoading = () => loadingOneByOne || loadingPerInfra;

            loader.stillRefreshing = () => loader.isLoading();

            loader.loadFromInfraLightStatus = function(infraLightStatus) {
                if (infraLightStatus.infraBasicInfo.canListHeavyStatus) {
                    perInfraList.push(infraLightStatus);
                } else {
                    oneByOneDeploymentIdList.push(...infraLightStatus.deployments.map(_ => _.id));
                }

                this._loadAll();
            };

            loader.loadFromDeploymentLightStatusList = function(deploymentLightStatusList) {
                const perInfraDeployments = new Map();

                deploymentLightStatusList.forEach((deploymentLightStatus) => {
                    if (deploymentLightStatus.infraBasicInfo.canListHeavyStatus) {
                        const infraId = deploymentLightStatus.infraBasicInfo.id;
                        if (perInfraDeployments.has(infraId)) {
                            perInfraDeployments.get(infraId).deployments.push(deploymentLightStatus.deploymentBasicInfo);
                        } else {
                            perInfraDeployments.set(infraId, {
                                infraBasicInfo: deploymentLightStatus.infraBasicInfo,
                                deployments: [deploymentLightStatus.deploymentBasicInfo]
                            });
                        }
                    } else {
                        oneByOneDeploymentIdList.push(deploymentLightStatus.deploymentBasicInfo.id);
                    }
                });

                if (perInfraDeployments.size) {
                    perInfraList.push(...perInfraDeployments.values());
                }

                this._loadAll();
            };

            loader.loadFromServiceStatus = function(serviceStatus) {
                const perInfraDeployments = new Map();

                serviceStatus.infras.forEach((infraBasicInfo) => {
                    if (infraBasicInfo.canListHeavyStatus) {
                        perInfraDeployments.set(infraBasicInfo.id, {infraBasicInfo, deployments: []});
                    }
                });

                serviceStatus.deployments.forEach((deploymentBasicInfo) => {
                    if (perInfraDeployments.has(deploymentBasicInfo.infraId)) {
                        perInfraDeployments.get(deploymentBasicInfo.infraId).deployments.push(deploymentBasicInfo);
                    } else {
                        oneByOneDeploymentIdList.push(deploymentBasicInfo.id);
                    }
                });

                if (perInfraDeployments.size) {
                    perInfraList.push(...perInfraDeployments.values());
                }

                this._loadAll();
            };

            loader._loadAll = function () {
                this._loadPerInfra();
                this._loadOneByOne();
            }

            const nbSimultaneousInfraLoading = 2;
            let runningInfraQueries = 0;
            loader._loadPerInfra = function() {
                if (!perInfraList.length && !runningInfraQueries || !canLoadStatus) {
                    loadingPerInfra = false;
                    return;
                }

                const infraStatusToLoad = perInfraList.splice(0, nbSimultaneousInfraLoading - runningInfraQueries);
                runningInfraQueries += infraStatusToLoad.length;

                for (let infraStatus of infraStatusToLoad) {
                    const infraId = infraStatus.infraBasicInfo.id;
                    const deploymentIds = new Set(infraStatus.deployments.map(_ => _.id));

                    Logger.info("Sending heavy status list request for deployments of infra " + infraId);
                    DataikuAPI.apideployer.deployments.listHeavyStatus(infraId, false)
                        .success(function(heavyStatusList) {
                            Logger.info("Got heavy status list for infra " + infraId);
                            heavyStatusList.forEach((heavyStatus) => {
                                heavyStatusPerDeployment[heavyStatus.deploymentId] = heavyStatus;
                            });
                            runningInfraQueries -= 1;
                            loader._loadPerInfra();
                        }).error(function(a,b,c) {
                            Logger.warn("Failed to load heavy status list for infra " + infraId);
                            deploymentIds.forEach((deploymentId) => {
                                heavyStatusPerDeployment[deploymentId] = {
                                    health: "LOADING_FAILED",
                                    healthMessages: DeployerUtils.getFailedHeavyStatusLoadMessage(getErrorDetails(a,b,c))
                                };
                            });
                            runningInfraQueries -= 1;
                            loader._loadPerInfra();
                        });
                }



            };

            const nbSimultaneousLoading = 2;
            let running = 0;
            loader._loadOneByOne = function() {
                const noSpinner = true;

                if (!oneByOneDeploymentIdList.length && !running || !canLoadStatus) {
                    loadingOneByOne = false;
                    return;
                }
                const idListForStatusLoad = oneByOneDeploymentIdList.splice(0, nbSimultaneousLoading - running);
                running += idListForStatusLoad.length;
                for (let deploymentId of idListForStatusLoad) {
                    Logger.info("Sending heavy status request for " + deploymentId);
                    DataikuAPI.apideployer.deployments.getHeavyStatus(deploymentId, false, noSpinner).success(function(heavyStatus) {
                        Logger.info("Got heavy status for " + deploymentId + "; health: " + heavyStatus.health);
                        heavyStatusPerDeployment[deploymentId] = heavyStatus;
                        running -= 1;
                        loader._loadOneByOne();
                    }).error(function(a,b,c) {
                        Logger.warn("Failed to load heavy status for " + deploymentId);
                        heavyStatusPerDeployment[deploymentId] = {
                            health: "LOADING_FAILED",
                            healthMessages: DeployerUtils.getFailedHeavyStatusLoadMessage(getErrorDetails(a,b,c))
                        };
                        running -= 1;
                        loader._loadOneByOne();
                    });
                }
            };
            return loader;
        }
    }
});

app.component("apiDeployerMonitoringModelStatus", {
    bindings: {
        monitoring: '<',
        lightStatus: '<',
        overallModelStatusIcon: '<',
        overallModelStatusLabel: '<'
    },
    templateUrl: '/templates/api-deployer/api-deployer-monitoring-model-status.html',
    controller: function($scope, UnifiedMonitoringService) {
        const $ctrl = this;
        $ctrl.uiState = {};
        $ctrl.getStatusIcon = function (status) {
            return UnifiedMonitoringService.getStatusIcon(status);
        }

        $ctrl.$onChanges = function() {
            if ($ctrl.monitoring) {
                $ctrl.uiState.modelMonitoring = []

                if ($ctrl.monitoring.modelStatus) {
                    const smvElements = {
                        smvSmartId: $ctrl.monitoring.modelStatus.smvSmartId,
                        savedModelId:  $ctrl.monitoring.modelStatus.savedModelId,
                        savedModelName: $ctrl.monitoring.modelStatus.savedModelName
                    };
                    $ctrl.monitoring.modelStatus.mesMonitoringStatuses.forEach((mes, mesIdx) => {
                        const mesChecks = UnifiedMonitoringService.mesMonitoringStatusToChecks(mes);
                        const mesElements = {
                            mesName: mes.mesName,
                            mesId: mes.id,
                            mesRowSpan: mesChecks.length
                        };
                        mesChecks.forEach((check, checkIdx) => {
                            let resElem = {...check};
                            if (mesIdx === 0 && checkIdx === 0) {
                                resElem = {...resElem, ...smvElements};
                            }
                            if (checkIdx === 0) {
                                resElem = {...resElem, ...mesElements};
                            }
                            $ctrl.uiState.modelMonitoring.push(resElem);
                        });
                    });
                }
            }
        }

        $ctrl.hasModelStatus = function() {
            return $ctrl.uiState.modelMonitoring && $ctrl.uiState.modelMonitoring.length > 0;
        }

        $ctrl.buildSmvUrl = function(monitoring) {
            if ($ctrl.lightStatus && $ctrl.lightStatus.packages.length > 0) {
                const currentPackage = $ctrl.lightStatus.packages[$ctrl.lightStatus.packages.length - 1];
                if (currentPackage.designNodeInfo) {
                return currentPackage.designNodeInfo.url + '/projects/' + currentPackage.designNodeInfo.projectKey + '/' + 'savedmodels/' + monitoring.savedModelId + '/p/' + monitoring.smvSmartId + '/tabular-summary';
                }
            }
        }

        $ctrl.buildMesUrl = function (mesId) {
            if ($ctrl.lightStatus && $ctrl.lightStatus.packages.length > 0) {
                const currentPackage = $ctrl.lightStatus.packages[$ctrl.lightStatus.packages.length - 1];
                if (currentPackage.designNodeInfo) {
                    return currentPackage.designNodeInfo.url + '/projects/' + currentPackage.designNodeInfo.projectKey + '/' + 'modelevaluationstores/' + mesId + '/evaluations/';
                }
            }
        }
    }
});

app.component("apiDeployerMonitoringGovernStatus", {
    bindings: {
        monitoring: '<',
        governStatusIcon: '<',
        governStatusLabel: '<'
    },
    templateUrl: '/templates/api-deployer/api-deployer-monitoring-govern-status.html',
    controller: function($rootScope, UnifiedMonitoringService) {
        const $ctrl = this;
        $ctrl.uiState = {};

        $ctrl.$onChanges = function() {
            if ($ctrl.monitoring && $ctrl.monitoring.governanceStatus) {
                $ctrl.uiState.governProjectUrl = `${$rootScope.appConfig.governURL}/artifact/${$ctrl.monitoring.governanceStatus.governArtifactId}`;
                $ctrl.uiState.governStatusIcon = UnifiedMonitoringService.getStatusIcon($ctrl.monitoring.umGovernanceStatus);
                $ctrl.uiState.governStatusLabel = UnifiedMonitoringService.getGovernStatusLabel($ctrl.monitoring.governanceStatus.validationStatus);
                $ctrl.uiState.governPolicyLabel = UnifiedMonitoringService.getGovernPolicyLabel($ctrl.monitoring.governCheckPolicy);
            }
            $ctrl.uiState.hasGovernStatus = $ctrl.monitoring.umGovernanceStatus !== "NO_STATUS";
        }
    }
});

app.component("deployerCopiableElement", {
    transclude: true,
    bindings: {
        text: '<',
        tooltipDirection: '@'
    },
    template: `
    <div class='deployer-copiable-element df' ng-click="$ctrl.copyToClipboard()">
        <div ng-if="!$ctrl.transcludePresent" show-tooltip-on-text-overflow text-tooltip="$ctrl.text" tooltip-direction="$ctrl.tooltipDirection"></div>
        <div ng-if="$ctrl.transcludePresent">
            <ng-transclude></ng-transclude>
        </div>
        &nbsp;
        <i class="dku-icon-copy-step-16"></i>
    </div>`,
    controller: function($transclude, ClipboardUtils) {
        const $ctrl = this;
        $ctrl.transcludePresent = false;

        $ctrl.$onInit = function() {
            $ctrl.tooltipDirection = $ctrl.tooltipDirection || 'tooltip-bottom';
            $transclude((content) => $ctrl.transcludePresent = content.length > 0);
        }
        $ctrl.copyToClipboard = function() {
            ClipboardUtils.copyToClipboard($ctrl.text, 'Copied to clipboard');
        }
    }
});

app.component("deployerExternalLinkElement", {
    bindings: {
        tooltipDirection: '@',
        text: '<',
        link: '<'
    },
    template: `
    <a class="df" href="{{$ctrl.link}}" target="_blank">
        <div show-tooltip-on-text-overflow text-tooltip="$ctrl.text" tooltip-direction="$ctrl.tooltipDirection">
            {{$ctrl.text}}
        </div>
        &nbsp;
        <i class="dku-icon-arrow-external-link-16"></i>
    </a>`,
    controller: function() {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $ctrl.tooltipDirection = $ctrl.tooltipDirection || 'tooltip-bottom';
        }
    }
});

app.component("displayCodeModal", {
    bindings: {
        modalControl: '<',
        title: '<',
        text: '<'
    },
    template: `
    <dku-modal-header-with-totem modal-title="{{$ctrl.title}}" modal-totem="dku-icon-code-24">
    </dku-modal-header-with-totem>
    <form class="dkuform-modal-horizontal dkuform-modal-wrapper" name="displayText">
        <div class="modal-body oh">
            <pre class="scroll">{{$ctrl.text}}</pre>
        </div>
        <div class="modal-footer modal-footer-std-buttons">
            <button type="button" class="btn btn--secondary" ng-click="$ctrl.copyToClipboard()">Copy</button>
            <button type="button" class="btn btn--primary" ng-click="$ctrl.modalControl.dismiss()">Close</button>
        </div>
    </form>`,
    controller: function(ClipboardUtils) {
        const $ctrl = this;
        $ctrl.copyToClipboard = function() {
            ClipboardUtils.copyToClipboard($ctrl.text);
            $ctrl.modalControl.dismiss();
        }
    }
});

app.component("displayOpenAPIModal", {
    bindings: {
        modalControl: '<',
        title: '<',
        enabledInfo: '<',
        text: '<'
    },
    template: `
    <dku-modal-header-with-totem modal-title="{{$ctrl.title}}" modal-totem="dku-icon-code-24">
    </dku-modal-header-with-totem>
    <form class="dkuform-modal-horizontal dkuform-modal-wrapper" name="displayText">
        <div class="modal-body oh">
            <div ng-if="$ctrl.enabledInfo">
                <div class="alert alert-info mbot16">
                    <strong><i class="icon-info-sign"></i></strong> OpenAPI documentation file is not served on a dedicated URI for kubernetes clusters of API nodes. You can access it from the Deployer Python API, <doclink-wrapper page="apinode/api-documentation#publishing-openapi-documentation">see Documentation</doclink-wrapper> for more details.
                </div>
            </div>
            <pre class="scroll">{{$ctrl.text}}</pre>
        </div>
        <div class="modal-footer modal-footer-std-buttons">
            <button type="button" class="btn btn--secondary" ng-click="$ctrl.copyToClipboard()">Copy</button>
            <button type="button" class="btn btn--primary" ng-click="$ctrl.modalControl.dismiss()">Close</button>
        </div>
    </form>`,
    controller: function(ClipboardUtils) {
        const $ctrl = this;
        $ctrl.copyToClipboard = function() {
            ClipboardUtils.copyToClipboard($ctrl.text);
            $ctrl.modalControl.dismiss();
        }
    }
});

app.directive('apiDeploymentCard', function($state, DataikuAPI, TaggingService, DeployerUtils, APIDeployerDeploymentUtils,
    DeployerDeploymentTileService, ClipboardUtils, CreateModalFromComponent, displayOpenAPIModalDirective, $filter) {
    return {
        scope: {
            lightStatus: '=',
            heavyStatus: '=',
            showMonitoring: '=',
            onRightClick: '&',
            customRightClick: '@',
        },
        templateUrl: '/templates/api-deployer/deployment-card.html',
        replace: true,
        link: function(scope, elem, attrs) {
            scope.dashboardTile = attrs.hasOwnProperty("deploymentDashboardTile");
            scope.TaggingService = TaggingService;
            scope.participatingVersions = DeployerUtils.getParticipatingVersions(scope.lightStatus.deploymentBasicInfo);
            const originInfoPerPackage = APIDeployerDeploymentUtils.getOriginInfoPerPackage(scope.lightStatus);
            const relatedInfoPerProject = {};
            scope.relatedInfoPerProject = [];
            scope.infraSupportOpenAPIURI = APIDeployerDeploymentUtils.infraSupportOpenAPIURI(scope.lightStatus.infraBasicInfo.type);
            scope.openAPIUrls = APIDeployerDeploymentUtils.computeSimpleOpenAPIURLs(scope.lightStatus);

            if (!scope.dashboardTile) {
                DataikuAPI.apideployer.deployments.getDeploymentOpenAPI(scope.lightStatus.deploymentBasicInfo.id).success(function(data) {
                    scope.openAPIDocContent = data;
                }).error(setErrorInScope.bind(scope));
            }

            scope.participatingVersions.forEach(function(version) {
                const originInfo = originInfoPerPackage[version] || {};
                if(originInfo.projectKey &&
                    (originInfo.projectKey in relatedInfoPerProject) &&
                    (originInfo.url === relatedInfoPerProject[originInfo.projectKey].url)) {
                    return;
                } else if(originInfo.projectKey) {
                    relatedInfoPerProject[originInfo.projectKey] = originInfo;
                    scope.relatedInfoPerProject.push(originInfo);
                }
            });

            let loadingSparkline;
            function loadSparkline() {
                if ((scope.deploymentStatus.currentState == "HEALTHY" || scope.deploymentStatus.currentState == "WARNING") &&
                    !loadingSparkline && !scope.lightStatus.failedToLoadSparkline && !scope.heavyStatus.sparkline &&
                    scope.lightStatus.infraBasicInfo.carbonAPIEnabled) {
                    loadingSparkline = true;
                    DataikuAPI.apideployer.deployments.getChartData(scope.lightStatus.deploymentBasicInfo.id, null, "OVERALL_QPS_COMBINED", "SIX_HOURS")
                        .success(function(spkData) {
                            let datapoints;
                            if (spkData[0] && spkData[0].datapoints) {
                                datapoints = spkData[0].datapoints;
                            } else {
                                datapoints = APIDeployerDeploymentUtils.buildZeroDatapoints("SIX_HOURS");
                            }
                            loadingSparkline = false;
                            scope.heavyStatus.sparkline = datapoints.map(x => x[0]);
                    }).error(function(a,b,c) {
                        loadingSparkline = false;
                        scope.lightStatus.failedToLoadSparkline = getErrorDetails(a,b,c);
                    });
                }
            }

            scope.redirect = function() {
                if (attrs.hasOwnProperty("redirectToDeploymentPage")) {
                    $state.go('apideployer.deployments.deployment.status', {deploymentId: scope.lightStatus.deploymentBasicInfo.id});
                }
            }

            scope.showOpenAPIJson = function() {
                let text;
                try {
                    text = $filter('prettyjson')(scope.openAPIDocContent);
                } catch (e) {
                    text = scope.openAPIDocContent;
                }

                CreateModalFromComponent(displayOpenAPIModalDirective, {
                    title: "OpenAPI Documentation",
                    enabledInfo: !scope.infraSupportOpenAPIURI,
                    text: text
                }, ['modal-wide']);
            }

            scope.copyInputToClipboard = function() {
                ClipboardUtils.copyToClipboard(scope.openAPIUrls);
            }

            scope.handleRightClick = function(event) {
                scope.onRightClick({$event: event});
            }

            scope.shouldHandleRightClick = function() {
                return scope.customRightClick ? "true" : "false";
            }

            scope.$watch('heavyStatus', function() {
                scope.deploymentStatus = DeployerDeploymentTileService.getDeploymentHealth(scope.heavyStatus);
                if (scope.lightStatus && scope.heavyStatus) {
                    loadSparkline();
                }
            });

            scope.$watch('lightStatus', function() {
                if (scope.lightStatus && scope.heavyStatus) {
                    loadSparkline();
                }
            }, true);
        }
    };
});

app.component('apiInlineDeploymentCard', {
    bindings: {
        projectName: '=',
        lightStatus: '<',
        heavyStatus: '<',
        lastDeploymentAction: '<',
        showLastAction: '<',
        refresh: '<'
    },
    templateUrl: '/templates/api-deployer/inline-deployment-card.html',
    controller: function(DeployerUtils, APIDeployerDeploymentUtils, DeployerDeploymentTileService,
        CreateModalFromComponent, displayOpenAPIModalDirective, $filter, DataikuAPI, $state, WT1) {

        const $ctrl = this;
        $ctrl.uiState = {
            openDetails: false
        };

        $ctrl.showOpenAPIJson = function() {
            let text;
            try {
                text = $filter('prettyjson')($ctrl.openAPIDocContent);
            } catch (e) {
                text = $ctrl.openAPIDocContent;
            }

            CreateModalFromComponent(displayOpenAPIModalDirective, {
                title: "OpenAPI Documentation",
                enabledInfo: !$ctrl.infraSupportOpenAPIURI,
                text: text
            }, ['modal-wide']);
        }

        $ctrl.$onChanges = function(changes) {
            if (changes.heavyStatus && $ctrl.heavyStatus) {
                $ctrl.deploymentStatus = DeployerDeploymentTileService.getDeploymentHealth($ctrl.heavyStatus);
            }
            if (changes.lightStatus && $ctrl.lightStatus) {
                DataikuAPI.apideployer.deployments.getDeploymentOpenAPI($ctrl.lightStatus.deploymentBasicInfo.id).success(function(data) {
                    $ctrl.openAPIDocContent = data;
                });
                $ctrl.uiState.infraHref = $state.href('apideployer.infras.infra.status', { infraId: $ctrl.lightStatus.infraBasicInfo.id });
                $ctrl.participatingVersions = DeployerUtils.getParticipatingVersions($ctrl.lightStatus.deploymentBasicInfo);
                const originInfoPerPackage = APIDeployerDeploymentUtils.getOriginInfoPerPackage($ctrl.lightStatus);
                $ctrl.relatedInfoPerProject = [];
                $ctrl.infraSupportOpenAPIURI = APIDeployerDeploymentUtils.infraSupportOpenAPIURI($ctrl.lightStatus.infraBasicInfo.type);
                $ctrl.openAPIUrls = APIDeployerDeploymentUtils.computeSimpleOpenAPIURLs($ctrl.lightStatus);

                $ctrl.participatingVersions.forEach(function(version) {
                    const originInfo = originInfoPerPackage[version] || {};
                    if (originInfo.projectKey &&
                        (originInfo.projectKey in $ctrl.relatedInfoPerProject) &&
                        (originInfo.url === $ctrl.relatedInfoPerProject[originInfo.projectKey].url)) {
                        return;
                    } else if (originInfo.projectKey) {
                        $ctrl.relatedInfoPerProject[originInfo.projectKey] = originInfo;
                        $ctrl.relatedInfoPerProject.push(originInfo);
                    }
                });
            }
        }

        $ctrl.toggleDetails = function() {
            $ctrl.uiState.openDetails = !$ctrl.uiState.openDetails;
            WT1.event('api-deployer-deployment-details-toggle', { deploymentType: $ctrl.lightStatus.infraBasicInfo.type, state: $ctrl.uiState.openDetails? 'opened': 'closed' });
        }

        $ctrl.lastUpdateCompleted = function() {
            return $ctrl.lastDeploymentAction?.type === 'UPDATE' && !$ctrl.lastDeploymentAction.inProgress;
        }

        $ctrl.lastActionStatus = function() {
            if ($ctrl.lastDeploymentAction && !$ctrl.lastDeploymentAction.inProgress) {
                switch ($ctrl.lastDeploymentAction.status) {
                    case 'SUCCESS':
                    case 'INFO':
                        return 'Succeeded';
                    case 'WARNING':
                        return 'Succeeded with warning';
                    case 'ERROR':
                        return 'Failed';
                }
            }
            return '';
        }
    }
});

app.directive('apiDeploymentsListWidget', function($state, TaggingService) {
    return {
        scope: {
            deployments: '=apiDeploymentsListWidget',
            statusPage: '=',
            healthMap: '='
        },
        templateUrl: '/templates/api-deployer/deployment-list.html',
        replace: true,
        link: function(scope) {
            scope.TaggingService = TaggingService;

            scope.redirect = function(deployment) {
                $state.go('apideployer.deployments.deployment.status', {deploymentId: deployment.id});
            };
        }
    };
});


app.controller('APIDeployerPackagesPanelController', function($scope, $controller, APIDeployerServicesService, DeployerUtils, TaggingService) {
    $controller("_DeployerPackagesPanelController", {$scope});

    $scope.TaggingService = TaggingService;

    $scope.getPackageDeployments = function(deployments, versionId) {
        return deployments.filter(d => DeployerUtils.getParticipatingVersions(d).includes(versionId));
    }

    $scope.deployVersion = function(serviceStatus, versionId) {
        APIDeployerServicesService
            .deployVersion(serviceStatus, versionId, DeployerUtils.DEPLOY_SOURCE.PACKAGE_PANEL)
            .catch(setErrorInScope.bind($scope.$parent));
    };

    $scope.$watch("serviceStatusList", function(nv) {
        if (!nv) return;

        $scope.uiState.fullServiceList = $scope.computeFullList(nv);
    });
});

app.controller('APIDeployerDeploymentCopyModalController', function($scope, DataikuAPI, Assert, APIDeployerDeploymentService, DeploymentUtils, APIDeployerDeploymentUtils, APIDeploymentInfraHelper) {
    $scope.newDepl = $scope.newDepl || {};
    $scope.isEndpointCompatibleWithDeployAnywhere = APIDeployerDeploymentUtils.isEndpointCompatibleWithDeployAnywhere;
    $scope.isEndpointCompatibleWithDeployAnywhereDatabricks = APIDeployerDeploymentUtils.isEndpointCompatibleWithDeployAnywhereDatabricks;

    if ($scope.oldDepl.endpointId) {
        $scope.newDepl.endpointId = $scope.oldDepl.endpointId;
    }

    function fetchStagesThenSortInfras(infraBasicInfoList) {
        DataikuAPI.apideployer.listStages()
        .success(stages => {
            $scope.infraBasicInfoList = APIDeployerDeploymentUtils.sortedInfrasByStages(infraBasicInfoList, stages);
        })
        .error(setErrorInScope.bind($scope));
    }

    DataikuAPI.apideployer.infras.listBasicInfo()
        .success(infraBasicInfoList => {
            $scope.infraBasicInfoList = infraBasicInfoList.infras;
            if ($scope.infraBasicInfoList.length == 1) {
                $scope.newDepl.infraId = $scope.infraBasicInfoList[0].id;
            }
            fetchStagesThenSortInfras(infraBasicInfoList.infras);
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.apideployer.publishedAPIServices.listLightStatus()
        .success(function(serviceStatusList) {
            $scope.serviceStatusList = serviceStatusList;
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.apideployer.deployments.listBasicInfo()
        .success(deploymentBasicInfoList => {$scope.deploymentIdList = deploymentBasicInfoList.deployments.map(depl => depl.id)})
        .error(setErrorInScope.bind($scope));

    function autoSetDeploymentId() {
        if (!$scope.newDepl.publishedServiceId || !$scope.newDepl.infraId) {
            return;
        }
        let newDeplId = DeploymentUtils.sanitizeForK8S($scope.newDepl.publishedServiceId + '-on-' + $scope.newDepl.infraId);
        let counter = 0;
        while (($scope.deploymentIdList || []).indexOf(newDeplId + (counter ? ('-' + counter) : '')) >= 0) {
            counter++;
        }
        $scope.newDepl.id = newDeplId + (counter ? ('-' + counter) : '');

    }
    $scope.$watch('newDepl.infraId', autoSetDeploymentId);

    function setupEndpoints() {
        if (!$scope.newDepl.publishedServiceId || !$scope.serviceStatusList || !$scope.oldDepl.generationsMapping || $scope.oldDepl.generationsMapping.mode != "SINGLE_GENERATION") {
            return;
        }

        const selectedInfra = $scope.infraBasicInfoList.find(e => e.id === $scope.newDepl.infraId);
        $scope.selectedInfraType = typeof selectedInfra === 'undefined' ? null : selectedInfra.type;

        $scope.endpoints = [];
        for (let status of $scope.serviceStatusList) {
            if (status.serviceBasicInfo.id == $scope.newDepl.publishedServiceId) {
                for (let pkg of status.packages) {
                    if (pkg.id === $scope.oldDepl.generationsMapping.generation) {
                        $scope.endpoints = pkg.endpoints;
                        break;
                    }
                }
                break;
            }
        }

        if (APIDeploymentInfraHelper.isFullyManagedInfra($scope.selectedInfraType)) {
            if ($scope.selectedInfraType === 'DATABRICKS') {
                $scope.compatibleEndpoints = $scope.endpoints.filter(e => $scope.isEndpointCompatibleWithDeployAnywhereDatabricks(e.type));
            } else {
                $scope.compatibleEndpoints = $scope.endpoints.filter(e => $scope.isEndpointCompatibleWithDeployAnywhere(e.type));
            }

            if ($scope.compatibleEndpoints.length) {
                if (!$scope.compatibleEndpoints.map(e => e.id).includes($scope.newDepl.endpointId)) {
                    $scope.newDepl.endpointId = $scope.compatibleEndpoints[0].id;
                }
            } else {
                $scope.newDepl.endpointId = null;
            }
        }
    }
    $scope.$watch('serviceStatusList', setupEndpoints);
    $scope.$watch('newDepl.infraId', setupEndpoints);

    $scope.ok = function() {
        return APIDeployerDeploymentService.openGovernanceStatusDeploymentId($scope.oldDepl.id, $scope.newDepl.infraId, undefined).then(function() {
            return DataikuAPI.apideployer.deployments.copy($scope.oldDepl.id, $scope.newDepl.id, $scope.newDepl.infraId, $scope.newDepl.endpointId)
                .success($scope.resolveModal)
                .error(setErrorInScope.bind($scope));
        });
    };
});

app.controller('APIDeployerDeploymentCreationModalController', function($scope, DataikuAPI, APIDeployerDeploymentService, Assert, DeploymentUtils, APIDeployerDeploymentUtils, APIDeploymentInfraHelper) {
    $scope.newDepl = $scope.newDepl || {};
    $scope.isEndpointCompatibleWithDeployAnywhere = APIDeployerDeploymentUtils.isEndpointCompatibleWithDeployAnywhere;
    $scope.isEndpointCompatibleWithDeployAnywhereDatabricks = APIDeployerDeploymentUtils.isEndpointCompatibleWithDeployAnywhereDatabricks;

    function fetchStagesThenSortInfras(infraBasicInfoList) {
        DataikuAPI.apideployer.listStages()
        .success(stages => {
            $scope.infraBasicInfoList = APIDeployerDeploymentUtils.sortedInfrasByStages(infraBasicInfoList, stages);
        })
        .error(setErrorInScope.bind($scope));
    }

    DataikuAPI.apideployer.infras.listBasicInfo()
        .success(infraBasicInfoList => {
            $scope.infraBasicInfoList = infraBasicInfoList.infras;
            if ($scope.infraBasicInfoList.length == 1) {
                $scope.newDepl.infraId = $scope.infraBasicInfoList[0].id;
            }
            fetchStagesThenSortInfras(infraBasicInfoList.infras);
        })
        .error(setErrorInScope.bind($scope));

    // TODO: delete this one. listLightStatus() should be sufficient. But we need to fix sc-157718 before
    DataikuAPI.apideployer.publishedAPIServices.listBasicInfo()
        .success(serviceBasicInfoList => {$scope.serviceBasicInfoList = serviceBasicInfoList.services})
        .error(setErrorInScope.bind($scope));

    DataikuAPI.apideployer.publishedAPIServices.listLightStatus()
        .success(function(serviceStatusList) {
            $scope.serviceStatusList = serviceStatusList;
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.apideployer.deployments.listBasicInfo()
        .success(deploymentBasicInfoList => {$scope.deploymentIdList = deploymentBasicInfoList.deployments.map(depl => depl.id)})
        .error(setErrorInScope.bind($scope));

    function setupVersionsIds() {
        if (!$scope.newDepl.publishedServiceId || !$scope.serviceBasicInfoList) {
            return;
        }
        $scope.versionsIds = [];
        for (let sbi of $scope.serviceBasicInfoList) {
            if (sbi.id == $scope.newDepl.publishedServiceId) {
                $scope.versionsIds = sbi.versionsIds;
                break;
            }
        }
        if (!$scope.versionsIds.includes($scope.newDepl.versionId)) {
            if ($scope.versionsIds.length) {
                $scope.newDepl.versionId = $scope.versionsIds[0];
            } else {
                delete $scope.newDepl.versionId;
            }
        }
    }
    $scope.$watch('newDepl.publishedServiceId', setupVersionsIds);
    $scope.$watch('serviceBasicInfoList', setupVersionsIds);

    function autoSetDeploymentId() {
        if (!$scope.newDepl.publishedServiceId || !$scope.newDepl.infraId) {
            return;
        }
        let newDeplId = DeploymentUtils.sanitizeForK8S($scope.newDepl.publishedServiceId + '-on-' + $scope.newDepl.infraId);
        let counter = 0;
        while (($scope.deploymentIdList || []).indexOf(newDeplId + (counter ? ('-' + counter) : '')) >= 0) {
            counter++;
        }
        $scope.newDepl.id = newDeplId + (counter ? ('-' + counter) : '');

    }
    $scope.$watch('newDepl.publishedServiceId', autoSetDeploymentId);
    $scope.$watch('newDepl.infraId', autoSetDeploymentId);

    function setupEndpoints() {
        if (!$scope.newDepl.publishedServiceId || !$scope.newDepl.versionId || !$scope.serviceStatusList) {
            return;
        }

        const selectedInfra = $scope.infraBasicInfoList.find(e => e.id === $scope.newDepl.infraId);
        $scope.selectedInfraType = typeof selectedInfra === 'undefined' ? null : selectedInfra.type;

        $scope.endpoints = [];
        for (let status of $scope.serviceStatusList) {
            if (status.serviceBasicInfo.id == $scope.newDepl.publishedServiceId) {
                for (let pkg of status.packages) {
                    if (pkg.id === $scope.newDepl.versionId) {
                        $scope.endpoints = pkg.endpoints;
                        break;
                    }
                }
                break;
            }
        }

        if (APIDeploymentInfraHelper.isFullyManagedInfra($scope.selectedInfraType)) {
            if ($scope.selectedInfraType === 'DATABRICKS') {
                $scope.compatibleEndpoints = $scope.endpoints.filter(e => $scope.isEndpointCompatibleWithDeployAnywhereDatabricks(e.type));
            } else {
                $scope.compatibleEndpoints = $scope.endpoints.filter(e => $scope.isEndpointCompatibleWithDeployAnywhere(e.type));
            }

            if ($scope.compatibleEndpoints.length) {
                if (!$scope.compatibleEndpoints.map(e => e.id).includes($scope.newDepl.endpointId)) {
                    $scope.newDepl.endpointId = $scope.compatibleEndpoints[0].id;
                }
            } else {
                $scope.newDepl.endpointId = null;
            }
        }
    }
    $scope.$watch('newDepl.publishedServiceId', setupEndpoints);
    $scope.$watch('newDepl.versionId', setupEndpoints);
    $scope.$watch('newDepl.infraId', setupEndpoints);
    $scope.$watch('serviceStatusList', setupEndpoints);

    $scope.ok = function() {
        return APIDeployerDeploymentService.openGovernanceStatusNewDeployment($scope.newDepl.publishedServiceId, $scope.newDepl.infraId, $scope.newDepl.versionId).then(function() {
            return DataikuAPI.apideployer.deployments.create($scope.newDepl.id, $scope.newDepl.publishedServiceId, $scope.newDepl.infraId, $scope.newDepl.versionId, $scope.newDepl.endpointId)
                .success($scope.resolveModal)
                .error(setErrorInScope.bind($scope));
        });
    };
});

app.controller('APIDeployerDeploymentController',
    function($controller, $scope, $state, WT1, DataikuAPI, Assert, CreateModalFromComponent, apiDeploymentModalDirective, CreateModalFromTemplate, FutureProgressModal,
             APIDeploymentInfraHelper, Ng2EndpointMetricsRefreshService, StaticApiDeploymentSyncHelper, APIDeployerDeploymentService, ActivityIndicator, STATIC_REFRESH_MODES) {
    $controller('_DeployerDeploymentController', {$scope});

    $scope.unsavedTestQueries =  {} // map endpointId -> json query

    $scope.deleteWarning = null;

    $scope.uiState = $scope.uiState || {};

    $scope.$watch("savedDeploymentSettings", () => {
        if ($scope.savedDeploymentSettings && $scope.savedDeploymentSettings.enabled && ['STATIC', 'K8S'].includes($scope.savedDeploymentSettings.type)) {
            const nodeTypes = APIDeploymentInfraHelper.isStaticInfra($scope.savedDeploymentSettings.type) ? "API node(s)" : "Kubernetes node(s)";
            $scope.deleteWarning = `The deployed API service will not be automatically deleted from the ${nodeTypes}. To delete it, please ensure the deployment has been disabled and updated before deleting the deployment.`;
        } else {
            $scope.deleteWarning = null;
        }
    });

    $scope.triggerMetricsRefresh = function() {
        Ng2EndpointMetricsRefreshService.triggerRefresh();
    }

    $scope.deployAPIDeployment = function(forceRebuild, skipGovernance) {
        $scope.uiState.askMode = false;
        $scope.uiState.refreshMode = STATIC_REFRESH_MODES.FULL;
        $scope.uiState.rebuildCodeEnv = forceRebuild;
        $scope.uiState.skipGovernanceCheck = skipGovernance;
        $scope.deployOrUpdate()
    }

    $scope.updateAPIDeployment = function(forceRebuild, skipGovernance, isSaveAndUpdate = false) {
        $scope.uiState.askMode = true;
        $scope.uiState.refreshMode = STATIC_REFRESH_MODES.LIGHT;
        $scope.uiState.rebuildCodeEnv = forceRebuild;
        $scope.uiState.skipGovernanceCheck = skipGovernance;
        $scope.deployOrUpdate(isSaveAndUpdate)
    }

    $scope.deployStatic = function(withPrepare = true) {
        $scope.uiState.nextStep = "DEPLOY";
        DataikuAPI.apideployer.deployments.executeSyncStatic($scope.lightStatus.deploymentBasicInfo.id, $scope.uiState.refreshMode === "FULL", withPrepare).success(function(data) {
            $scope.openProgressModal(data.jobId);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.onUpdateFinished = function() {
        $scope.getLightStatus();
        $scope.lightStatus.neverEverDeployed = false;
    }

    $scope.openProgressModal = function(jobId) {
        CreateModalFromComponent(apiDeploymentModalDirective, {
            deploymentId: $scope.lightStatus.deploymentBasicInfo.id,
            jobId: jobId,
            insufficientPermissionsMessage: $scope.insufficientPermissionsMessage(),
            lastDeploymentAction: $scope.lastDeploymentAction,
            deploymentType: $scope.savedDeploymentSettings.type,
            peekingUpdateStarted: $scope.onPeekingUpdateStarted,
            peekingUpdateEnded: $scope.fetchLastDeploymentActionUntilCanceled,
            deploy: APIDeploymentInfraHelper.isStaticInfra($scope.savedDeploymentSettings.type) ? $scope.deployStatic : $scope.deployOrUpdate
        }, ['modal-w800', 'future-progress-modal'])
    }

    $scope.deployOrUpdate = function(isSaveAndUpdate = false) {
        Assert.trueish($scope.lightStatus);
        const actionFn = () => {
            if (APIDeploymentInfraHelper.isStaticInfra($scope.savedDeploymentSettings.type)) {
                if ($scope.uiState.askMode) {
                    StaticApiDeploymentSyncHelper.askMode($scope)
                } else {
                    $scope.deployStatic();
                }
            } else {
                DataikuAPI.apideployer.deployments.executeSyncDockerBased($scope.lightStatus.deploymentBasicInfo.id, $scope.uiState.rebuildCodeEnv).success(function(data) {
                    $scope.openProgressModal(data.jobId);
                }).error(setErrorInScope.bind($scope));
            }
        }

        if ($scope.uiState.skipGovernanceCheck) {
            actionFn();
        } else {
            APIDeployerDeploymentService.openGovernanceStatusDeploymentId($scope.lightStatus.deploymentBasicInfo.id).then(actionFn);
        }

        if (!!$scope.lightStatus && $scope.lightStatus.neverEverDeployed) {
            WT1.event('api-deployer-deployment-deploy', { deploymentType: $scope.lightStatus.infraBasicInfo.type });
        } else if (isSaveAndUpdate) {
            WT1.event('api-deployer-deployment-save-and-update', { deploymentType: $scope.lightStatus.infraBasicInfo.type, isDeploymentDisabled: !$scope.deploymentSettings.enabled });
        } else {
            WT1.event('api-deployer-deployment-update', { deploymentType: $scope.lightStatus.infraBasicInfo.type });
        }
    };

    $scope.diagnoseDeployment = function() {
        // sanity check (like copyDeployment() )
        if (!$scope.lightStatus
            || !$scope.lightStatus.deploymentBasicInfo || !$scope.lightStatus.deploymentBasicInfo.id
            || !$scope.lightStatus.infraBasicInfo || $scope.lightStatus.infraBasicInfo.type == "STATIC") {
            return;
        }
        DataikuAPI.apideployer.deployments.startDiagnosis($scope.lightStatus.deploymentBasicInfo.id).success(function(data) {
            FutureProgressModal.show($scope, data, "Generating diagnostic").then(function(result) {
                downloadURL(DataikuAPI.apideployer.deployments.getDiagnosisURL($scope.lightStatus.deploymentBasicInfo.id, result));
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.getFullCheckReport = function() {
        if (!$scope.lightStatus || !$scope.heavyStatus || !$scope.lightStatus.deploymentBasicInfo || !$scope.lightStatus.deploymentBasicInfo.id
            || !['SAGEMAKER', 'SNOWPARK', 'AZURE_ML', 'DATABRICKS', 'VERTEX_AI'].includes($scope.lightStatus.infraBasicInfo.type)) {
            return;
        }
        $scope.uiState.loadingFullCheckReport = true;
        DataikuAPI.apideployer.deployments.getFullCheckReport($scope.lightStatus.deploymentBasicInfo.id)
            .success(function(statusReport) {
                $scope.uiState.loadingFullCheckReport = false;
                $scope.fullCheckReport = statusReport;
                ActivityIndicator.success("Deployment full check report retrieved successfully.");
            })
            .error(setErrorInScope.bind($scope))
            .error(function() {
                $scope.uiState.loadingFullCheckReport = false;
            });
    };

    $scope.copyDeployment = function() {
        if (!$scope.lightStatus || !$scope.lightStatus.deploymentBasicInfo || !$scope.lightStatus.deploymentBasicInfo.id ||
                !$scope.lightStatus.serviceBasicInfo || !$scope.lightStatus.serviceBasicInfo.id) {
            return;
        }
        return CreateModalFromTemplate("/templates/api-deployer/copy-deployment-modal.html",
            angular.extend($scope, {
                oldDepl: {
                    id: $scope.lightStatus.deploymentBasicInfo.id,
                    infraType: $scope.lightStatus.infraBasicInfo.type,
                    endpointId: $scope.lightStatus.deploymentBasicInfo.endpointId,
                    generationsMapping: $scope.lightStatus.deploymentBasicInfo.generationsMapping,
                },
                newDepl: {publishedServiceId: $scope.lightStatus.serviceBasicInfo.id}
            })
        ).then(function(newDeployment) {
            $state.go('apideployer.deployments.deployment.status', {deploymentId: newDeployment.id});
            WT1.event('api-deployer-deployment-copy', { deploymentType: newDeployment.type });
        });
    };

    $scope.saveAndUpdateApiDeployment = function(forceRebuildCodeEnv) {
        APIDeployerDeploymentService.openGovernanceStatusDeployment($scope.deploymentSettings).then(function() {
            $scope.saveDeployment().then(function() { $scope.updateAPIDeployment(forceRebuildCodeEnv, true, true); });
       });
    };

    $scope.saveApiDeployment = function() {
        WT1.event('api-deployer-deployment-save', { deploymentType: $scope.deploymentSettings.type, isDeploymentDisabled: !$scope.deploymentSettings.enabled });
        APIDeployerDeploymentService.openGovernanceStatusDeployment($scope.deploymentSettings).then(function() {
            $scope.saveDeployment();
       });
    };

    const allowedTransitions = [
        'apideployer.deployments.deployment.status',
        'apideployer.deployments.deployment.history',
        'apideployer.deployments.deployment.settings'
    ];
    checkChangesBeforeLeaving($scope, $scope.deploymentIsDirty, null, allowedTransitions);
    if (!$scope.deploymentSettings) {
        $scope.getDeploymentSettings();
    }
});

app.controller('APIDeployerDeploymentStatusController', function($scope, $state, $filter, TopNav, DataikuAPI,  DeployerUtils,
    DeployerDeploymentTileService, DataikuCloudService, UnifiedMonitoringService, CreateModalFromComponent, displayCodeModalDirective) {

    $scope.refresh = function() {
        try {
            $scope.refreshLightAndHeavy()
        } finally {
            $scope.triggerMetricsRefresh()
        }
    }

    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_API_DEPLOYER, 'deployments', null, 'status');

    $scope.isCloud = DataikuCloudService.isDataikuCloud();
    $scope.chartURL = DataikuAPI.apideployer.deployments.chartURL;

    $scope.uiState = {};

    $scope.setCurrentEndpoint = function(endpoint) {
        $scope.currentEndpoint = endpoint;
    };

    $scope.setCurrentEndpointId = function(endpointId) {
        if ($scope.heavyStatus && $scope.heavyStatus.endpoints) {
            const endpoint = $scope.heavyStatus.endpoints.find(e => e.id === endpointId);
            $scope.setCurrentEndpoint(endpoint);
        }
    };

    $scope.goToSettings = function() {
        $state.go('apideployer.infras.infra.settings', {infraId: $scope.lightStatus.deploymentBasicInfo.infraId});
    }

    $scope.getModelOverallStatusIcon = function(monitoring) {
        const status = monitoring ? monitoring.umModelStatus : "NO_STATUS";
        return UnifiedMonitoringService.getStatusIcon(status);
    }

    $scope.getModelOverallStatusLabel = function(monitoring) {
        const status = monitoring ? monitoring.umModelStatus : "NO_STATUS";
        return UnifiedMonitoringService.getUMStatusDisplayLabels(status);
    }

    $scope.getGovernStatusIcon = function(monitoring) {
        const status = monitoring ? monitoring.umGovernanceStatus : "NO_STATUS";
        return UnifiedMonitoringService.getStatusIcon(status);
    }

    $scope.getGovernStatusLabel = function(monitoring) {
        if (monitoring && monitoring.governanceStatus && monitoring.governanceStatus.validationStatus) {
            return UnifiedMonitoringService.getGovernStatusLabel(monitoring.governanceStatus.validationStatus);
        }
        return "No governance status";
    }

    $scope.isEntryEndpoint = function(endpointId) {
        return !!$scope.deploymentSettings && !!$scope.deploymentSettings.endpointId && endpointId === $scope.deploymentSettings.endpointId
    }

    $scope.isSampleCodeCompatibleEndpoint = function(endpointId) {
        return !(['SAGEMAKER', 'AZURE_ML', 'VERTEX_AI', 'DATABRICKS'].includes($scope.lightStatus.infraBasicInfo.type) && !$scope.isEntryEndpoint(endpointId))
    }

    $scope.isQueriesCompatibleEndpoint = function(endpointId) {
        return !(['SAGEMAKER', 'AZURE_ML', 'VERTEX_AI', 'DATABRICKS', 'SNOWPARK'].includes($scope.lightStatus.infraBasicInfo.type) && !$scope.isEntryEndpoint(endpointId))
    }

    $scope.isSampleCodeFullySupportedEndpoint = function(endpointId, language) {
        return !(['SNOWPARK'].includes($scope.lightStatus.infraBasicInfo.type) && !$scope.isEntryEndpoint(endpointId) && language !== "PYTHON")
    }

    $scope.showEndpointCode = function(endpoint) {
        CreateModalFromComponent(displayCodeModalDirective, {
            title: endpoint.id + " code details",
            text: endpoint.code
        }, ['modal-wide']);
    }

    $scope.$watch("lightStatus", function(nv) {
        if (!nv || nv.neverEverDeployed) {
            return;
        }

        const entryEndpointId = nv.deploymentBasicInfo.endpointId; // Used in Deploy Anywhere, undefined otherwise
        $scope.getHeavyStatus().then(function(data) {
            $scope.heavyStatus = data.data;
            $scope.heavyStatus.endpoints = $scope.heavyStatus.endpoints.map(e => {return {... e, typename: $filter('endpointTypeToName')(e.type)}});
            $scope.deploymentStatus = DeployerDeploymentTileService.getDeploymentHealth($scope.heavyStatus);
            if (!$scope.heavyStatus || !$scope.heavyStatus.endpoints || !$scope.heavyStatus.endpoints.length) {
                return;
            }
            $scope.setCurrentEndpointId(entryEndpointId ? entryEndpointId : $scope.heavyStatus.endpoints[0].id);
            $scope.uiState.showHealthMessages = !['HEALTHY', 'DISABLED'].includes($scope.heavyStatus.health);

            $scope.uiState.expandMessages = $scope.heavyStatus.health === 'UNKNOWN' && $scope.lightStatus.infraBasicInfo.type === 'SAGEMAKER';

            DataikuAPI.unifiedMonitoring.deployer.apiEndpoints.get(nv.deploymentBasicInfo.id, $scope.currentEndpoint.id).success((monitoring) => {
                if (monitoring) {
                    if (monitoring.disabledInfra) {
                        $scope.monitoring = undefined;
                        if ($scope.uiState.activeTab === 'health') {
                            $scope.uiState.activeTab = 'summary';
                        }
                        $scope.monitoringDisabled = true;
                    }
                    else {
                        $scope.monitoring = monitoring;
                        $scope.monitoringDisabled = false;
                    }
                }
            }).error(setErrorInScope.bind($scope));
        }, function(err) {
            $scope.heavyStatus = {
                health: "LOADING_FAILED",
                healthMessages: DeployerUtils.getFailedHeavyStatusLoadMessage(getErrorDetails(err.data, err.status, err.headers))
            };
        });
    });

    $scope.$watch("fullCheckReport", function(nv) {
        if(!nv) return;

        $scope.uiState.showHealthMessages = !['HEALTHY', 'DISABLED'].includes(nv.health);

        // Trick to avoid retrieving heavy status again to update the status icon
        let heavyStatusCopy = {...$scope.heavyStatus};
        heavyStatusCopy.health = nv.health;
        heavyStatusCopy.healthMessages = nv.statusMessages;
        $scope.heavyStatus = heavyStatusCopy;
    });
});

app.controller('APIDeployerDeploymentUpdatesController', function($scope, $state, TopNav, DataikuAPI) {
    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_API_DEPLOYER, 'deployments', null, 'last-updates');

    $scope.listDeploymentUpdateHeads = function() {
        return DataikuAPI.apideployer.deployments.listDeploymentUpdateHeads($state.params.deploymentId);
    };

    $scope.getDeploymentUpdate = function(startTimestamp) {
        return DataikuAPI.apideployer.deployments.getDeploymentUpdate($state.params.deploymentId, startTimestamp);
    };

    $scope.getDeploymentUpdateSettingsDiff = function(startTimestamp) {
        return DataikuAPI.apideployer.deployments.getDeploymentUpdateSettingsDiff($state.params.deploymentId, startTimestamp);
    };
});


app.component('apiDeploymentUpdateProgress', {
    bindings: {
        deploymentUpdate: '<',
    },
    template: `
        <deployment-hooks-states hook-execution-status="$ctrl.deployment.deploymentHookExecutionStatus" max-width-px="700"></deployment-hooks-states>

        <div class="dkuform-modal-wrapper" ng-if="$ctrl.staticDeploymentPrepareFailed()">
            <div class="alert alert-error">
                <h4><strong>Upload of some of the version packages to the API nodes failed.</strong></h4>
                <p>
                    It will not be possible to activate the deployment successfully on all API nodes<br />
                    It may still be possible to activate the deployment on some of the API nodes
                </p>
                <p style="margin-bottom:0">
                    Upload <strong>succeeded on {{$ctrl.deployment.prepareResult.nbOKNodes}}</strong> nodes and <strong>failed on {{$ctrl.deployment.prepareResult.nbNOKNodes}}</strong> nodes.
                </p>
            </div>
        </div>

        <div class="dkuform-modal-wrapper" ng-if="$ctrl.staticDeploymentSyncFailed()">
            <div class="alert alert-error" ng-if="($ctrl.deployment.deployResult.nbOKNodes + $ctrl.deployment.deployResult.nbNOKNodes) > 0">
                <h4>Activation of the new version(s) failed on some of the API nodes</h4>
                <p>
                    The service may be in a partially-inconsistent state (differing versions).
                </p>
                <p style="margin-bottom: 0">
                    Activation <strong>succeeded on {{$ctrl.deployment.deployResult.nbOKNodes}}</strong> nodes and <strong>failed on {{$ctrl.deployment.deployResult.nbNOKNodes}}</strong> nodes.
                </p>
            </div>
        </div>

        <div info-messages-raw-list="$ctrl.deployment" />
    `,
    controller: function(ApiDeploymentProgress) {
        const $ctrl = this;

        $ctrl.deployment = ApiDeploymentProgress.initDeployment();

        $ctrl.staticDeploymentPrepareFailed = function() {
            return ApiDeploymentProgress.staticDeploymentPrepareFailed($ctrl.deployment, $ctrl.deploymentUpdate?.deployment.type);
        }

        $ctrl.staticDeploymentSyncFailed = function() {
            return ApiDeploymentProgress.staticDeploymentSyncFailed($ctrl.deployment, $ctrl.deploymentUpdate?.deployment.type);
        }

        $ctrl.$onChanges = function(changes) {
            const report = changes?.deploymentUpdate.currentValue?.report;
            if (report) {
                report.deploymentHookExecutionStatus.hasHooks = true; // TODO This is a by-pass. It will require refactoring of hooks to share responsibility for displaying deployment messages
                ApiDeploymentProgress.updateDeploymentStatus(report, $ctrl.deployment);
            }
        }
    }
});

app.controller('APIDeployerDeploymentLogsController', function($scope, $state, TopNav, DataikuAPI, Logs) {
    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_API_DEPLOYER, 'deployments', null, 'logs');

    $scope.getLog = DataikuAPI.apideployer.deployments.getLog;
    $scope.deploymentId = $state.params.deploymentId;
    $scope.downloadURL = function(deploymentId, logName) {
        return "/dip/api/api-deployer/deployments/stream-log?deploymentId=" + encodeURIComponent(deploymentId) + "&logName=" + encodeURIComponent(logName);
    }
    $scope.downloadAllURL = function(deploymentId) {
        return "/dip/api/api-deployer/deployments/all-logs-zip?deploymentId=" + encodeURIComponent(deploymentId);
    }

    function refreshLogList() {
        DataikuAPI.apideployer.deployments.listLogs($scope.deploymentId).success(function(data) {
            $scope.logs = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.$watch("lightStatus", function(nv, ov) {
        refreshLogList();
    });
    refreshLogList();
});

app.controller('APIDeployerDeploymentStatusEndpointStatusController', function(
    $scope, $controller, $state, TopNav, DataikuAPI, FutureProgressModal, Assert, CodeMirrorSettingService,
    DeploymentStatusEndpointSampleCodeGenerator, DeployerUtils, APIDeployerDeploymentUtils, RemoteResourcesLinksUtils
) {
    Assert.trueish($scope.lightStatus);
    Assert.trueish($scope.lightStatus.infraBasicInfo);
    Assert.trueish($scope.heavyStatus);

    $scope.codeMirrorSettingService = CodeMirrorSettingService;
    $scope.remoteResourcesLinksUtils = RemoteResourcesLinksUtils;

    // To avoid confusion between nvd3.js and the good old "nv" from AngularJS's watch.
    // Renaming global variable nv as nvd3.
    // eslint-disable-next-line no-undef
    const nvd3 = nv;

    $scope.epUIState= {
        activeTab: "summary",
        sampleCodeLang: "CURL",
        chartsTimeRange: "SIX_HOURS"
    };
    $scope.chartsTimeRanges = [
        {
            id: 'ONE_HOUR',
            label: '1 hour'
        },
        {
            id: 'SIX_HOURS',
            label: '6 hours'
        },
        {
            id: 'ONE_DAY',
            label: '1 day'
        }
    ];
    $scope.sampleCodeOptions = [
        ["CURL", "Shell (cURL)"],
        ["PYTHON", "Python"],
        ["R", "R"],
        ["JAVA", "Java"]
        //["CSHARP", "C#"],
        //["PHP", "PHP"]
    ];

    function onCodeLangChanged() {
        if ($scope.epUIState.sampleCodeLang) {
            if ($scope.lightStatus.infraBasicInfo.type === 'AZURE_ML') {
                const deploymentInfo = $scope.lightStatus.deploymentBasicInfo.deploymentInfo;
                $scope.sampleCode = DeploymentStatusEndpointSampleCodeGenerator.generateAzureMLSampleCode($scope.epUIState.sampleCodeLang, $scope.endpoint, deploymentInfo.azSubscription, deploymentInfo.azResourceGroup, deploymentInfo.azWorkspace, deploymentInfo.azOnlineEndpointName);
            } else if ($scope.lightStatus.infraBasicInfo.type === 'DATABRICKS') {
                const deploymentInfo = $scope.lightStatus.deploymentBasicInfo.deploymentInfo;
                $scope.sampleCode = DeploymentStatusEndpointSampleCodeGenerator.generateDatabricksSampleCode(
                    $scope.epUIState.sampleCodeLang, $scope.endpoint, deploymentInfo.dbxHostBaseUrl, deploymentInfo.dbxEndpointName
                );
            } else if ($scope.lightStatus.infraBasicInfo.type === 'SAGEMAKER') {
                const deploymentInfo = $scope.lightStatus.deploymentBasicInfo.deploymentInfo;
                $scope.sampleCode = DeploymentStatusEndpointSampleCodeGenerator.generateSageMakerSampleCode($scope.epUIState.sampleCodeLang, $scope.endpoint, deploymentInfo.awsRegion, deploymentInfo.sageMakerEndpointName);
            } else if ($scope.lightStatus.infraBasicInfo.type === 'SNOWPARK') {
                const deploymentInfo = $scope.lightStatus.deploymentBasicInfo.deploymentInfo;
                const serviceURL = $scope.serviceURLs[0];
                $scope.sampleCode = DeploymentStatusEndpointSampleCodeGenerator.generateSnowparkSampleCode($scope.epUIState.sampleCodeLang, $scope.endpoint, deploymentInfo.snowparkUdfName, serviceURL);
            } else if ($scope.lightStatus.infraBasicInfo.type === 'VERTEX_AI') {
                const deploymentInfo = $scope.lightStatus.deploymentBasicInfo.deploymentInfo;
                $scope.sampleCode = DeploymentStatusEndpointSampleCodeGenerator.generateVertexAISampleCode($scope.epUIState.sampleCodeLang, $scope.endpoint, deploymentInfo.gcpRegion, deploymentInfo.gcpProjectId, deploymentInfo.vertexAIEndpointId);
            } else if ($scope.serviceURLs && $scope.serviceURLs.length > 0) {
                $scope.sampleCode = DeploymentStatusEndpointSampleCodeGenerator.generateSampleCode($scope.lightStatus, $scope.serviceURLs, $scope.endpoint, $scope.epUIState.sampleCodeLang, $scope.heavyStatus);
            } else {
                $scope.sampleCode = '';
            }
        } else {
            $scope.sampleCode = '';
        }
    }

    $scope.$watch("epUIState.sampleCodeLang", onCodeLangChanged);
    $scope.$watch("currentEndpoint", function(nv) {
        if (nv) {
            $scope.endpoint = $scope.currentEndpoint;
            $scope.genericEndpointType = APIDeployerDeploymentUtils.getGenericEndpointType($scope.currentEndpoint);

            $scope.participatingVersions = DeployerUtils.getParticipatingVersions($scope.lightStatus.deploymentBasicInfo);;
            $scope.SMVersionPerPackage = APIDeployerDeploymentUtils.getSMVersionPerPackageForEndpoint($scope.lightStatus, $scope.currentEndpoint.id, $scope.participatingVersions);

            $scope.serviceURLs = APIDeployerDeploymentUtils.computeEndpointURLs($scope.lightStatus, $scope.heavyStatus, $scope.endpoint);

            // code and queries tabs are disabled for some endpoints of DA infras, we force nav to the summary
            if (
                ['SAGEMAKER', 'AZURE_ML', 'VERTEX_AI','DATABRICKS'].includes($scope.lightStatus.infraBasicInfo.type) &&
                $scope.currentEndpoint.id !== $scope.deploymentSettings.endpointId &&
                ['test','code'].includes($scope.uiState.activeTab)
            ) {
                $scope.uiState.activeTab = 'summary';
            }

            DataikuAPI.unifiedMonitoring.deployer.apiEndpoints.get($scope.lightStatus.deploymentBasicInfo.id, $scope.currentEndpoint.id).success((monitoring) => {
                if (monitoring) {
                    if (monitoring.disabledInfra) {
                        $scope.monitoring = undefined;
                        if ($scope.uiState.activeTab === 'health') {
                            $scope.uiState.activeTab = 'summary';
                        }
                        $scope.monitoringDisabled = true;
                    }
                    else {
                        $scope.monitoring = monitoring;
                        $scope.monitoringDisabled = false;
                    }
                }
            }).error(setErrorInScope.bind($scope));;

            onCodeLangChanged();
            loadCharts($scope.endpoint);

            if (['SAGEMAKER', 'VERTEX_AI', 'AZURE_ML', 'DATABRICKS'].includes($scope.lightStatus.infraBasicInfo.type)) {
                $scope.epUIState.sampleCodeLang = "PYTHON"
                $scope.sampleCodeOptions = [
                    ["PYTHON", "Python"],
                ];
            } else if ($scope.lightStatus.infraBasicInfo.type === "SNOWPARK") {
                $scope.epUIState.sampleCodeLang = "SQL"
                $scope.sampleCodeOptions = [
                    ["SQL", "SnowSQL"],
                    ["PYTHON", "Python"],
                ];
            } else if ($scope.endpoint.type == "STD_FORECAST") {
                $scope.sampleCodeOptions = [
                    ["CURL", "Shell (cURL)"],
                    ["PYTHON", "Python"],
                    ["R", "R"],
                    ["JAVA", "Java"]
                ];
            } else {
                $scope.sampleCodeOptions = [
                    ["CURL", "Shell (cURL)"],
                    ["PYTHON", "Python"],
                    ["R", "R"],
                    ["JAVA", "Java"]
                ];
            }
        }
    });

    $scope.$watch("epUIState.chartsTimeRange", function(nv) {
        if (nv && $scope.currentEndpoint) {
            loadCharts($scope.currentEndpoint);
        }
    });

    $scope.$watch("epUIState.activeTab", function(nv, ov) {
        if (nv && ov && nv =="summary") {
            loadCharts($scope.currentEndpoint);
        }
    });

    function cleanupChart(svgId) {
        d3.selectAll('svg#' + svgId + ' > *').remove();
        d3.selectAll('.nvtooltip').remove();
    }

    function displayQPSChart(datapoints, svgId) {
        const data = [{
            key: "QPS",
            values: datapoints,
            color: "rgb(42, 177, 172)"
        },{
            values: datapoints,
            area: true,
            color: "rgba(42, 177, 172, 0.5)"
        }];
        const maxVal = d3.max(datapoints.map(dp => dp.y)) || 10;
        nvd3.addGraph(function() {
            const chart = nvd3.models.lineChart().options({
                useInteractiveGuideline:true
            });
            chart.useVoronoi(false);
            chart.showLegend(false);
            chart.margin({top: 5, right: 20, bottom: 20, left: 40});
            chart.xAxis.tickFormat(d => d3.time.format('%H:%M')(new Date(d*1000)));
            chart.yAxis.tickFormat(x => d3.format(',.1f')(x));
            chart.yDomain([0,maxVal])

            d3.select('svg#' + svgId)
                .datum(data)
                .transition().duration(500)
                .call(chart);

            nvd3.utils.windowResize(chart.update);

            return chart;
        });
    }
    function displayTimeChart(series, svgId) {
        const data = series.map(function(serie, i) {
            return {
                key: serie.target.replace(".totalProcessing.p95", ""),
                max: d3.max(serie.datapoints.map(dp => dp[0])),
                values: serie.datapoints.map(dp => ({x: dp[1], y: dp[0]}))
            }
        });
        const maxVal = d3.max(data.map(serie => serie.max)) || 10;
        nvd3.addGraph(function() {
            const chart = nvd3.models.lineChart().options({
                useInteractiveGuideline:true
            });
            chart.showLegend(false);
            chart.useVoronoi(false);
            chart.margin({top: 5, right: 20, bottom: 20, left: 40});
            chart.xAxis.tickFormat(d => d3.time.format('%H:%M')(new Date(d*1000)));
            chart.yAxis.tickFormat(x => d3.format(',.0f')(x) + " ms");
            chart.yDomain([0,maxVal])

            d3.select('svg#' + svgId)
                .datum(data)
                .transition().duration(500)
                .call(chart);

            nvd3.utils.windowResize(chart.update);

            return chart;
        });
    }

    function loadCharts(endpoint) {
        if ($scope.lightStatus && $scope.lightStatus.deploymentBasicInfo.enabled && $scope.lightStatus.infraBasicInfo.carbonAPIEnabled) {
            // cleanup and reload the queries chart
            delete endpoint.succeededToLoadQueries;
            delete endpoint.failedToLoadQueries;
            cleanupChart('deployment-graph-queries');
            DataikuAPI.apideployer.deployments.getChartData($scope.lightStatus.deploymentBasicInfo.id, endpoint.id, "ENDPOINT_QPS_COMBINED", $scope.epUIState.chartsTimeRange || "SIX_HOURS").success(function(spkData) {
                endpoint.succeededToLoadQueries = true;
                let datapoints;
                if (spkData[0] && spkData[0].datapoints) {
                    datapoints = spkData[0] && spkData[0].datapoints;
                } else {
                    // if no data, fill datapoints with dummy values in order to display the graph anyways
                    datapoints = APIDeployerDeploymentUtils.buildZeroDatapoints($scope.epUIState.chartsTimeRange);
                }
                displayQPSChart(datapoints.map(dp => ({x: dp[1], y: dp[0]})), 'deployment-graph-queries');
            }).error(function(a,b,c) {
                endpoint.failedToLoadQueries = getErrorDetails(a,b,c);
            });
            // cleanup and reload the processingtime chart
            delete endpoint.succeededToLoadTiming;
            delete endpoint.failedToLoadTiming;
            cleanupChart('deployment-graph-processingtime');
            DataikuAPI.apideployer.deployments.getChartData($scope.lightStatus.deploymentBasicInfo.id, endpoint.id, "ENDPOINT_TIMING_SPLIT", $scope.epUIState.chartsTimeRange || "SIX_HOURS").success(function(spkData) {
                endpoint.succeededToLoadTiming = true;
                if (!spkData.length) {
                    spkData.push({
                        target: '',
                        datapoints: APIDeployerDeploymentUtils.buildZeroDatapoints($scope.epUIState.chartsTimeRange)
                    });
                }
                displayTimeChart(spkData, 'deployment-graph-processingtime');
            }).error(function(a,b,c) {
                endpoint.failedToLoadTiming = getErrorDetails(a,b,c);
            });
        }
    }

    $scope.disabledSageMakerEndpointLinkMessage = function() {
        if ($scope.lightStatus.deploymentBasicInfo.deploymentInfo.sageMakerEndpointName && !$scope.lightStatus.deploymentBasicInfo.enabled) {
            return "Cloud endpoint does not currently exist as the deployment is disabled";
        } else {
            return null;
        }
    }

    $scope.disabledAZOnlineEndpointLinkMessage = function() {
        if ($scope.lightStatus.deploymentBasicInfo.deploymentInfo.azOnlineEndpointName && !$scope.lightStatus.deploymentBasicInfo.enabled) {
            return "Online endpoint does not currently exist as the deployment is disabled";
        } else {
            return null;
        }
    }

    $scope.$on("$destroy", function() {
        // ensure chart tooltips get removed when navigating elsewhere
        d3.selectAll('.nvtooltip').remove();
    });
});

app.controller('APIDeployerDeploymentHistoryController', function($scope, TopNav) {
    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_API_DEPLOYER, 'deployments', null, 'history');

    if (!$scope.deploymentSettings) {
        $scope.getDeploymentSettings();
    }
});


app.controller('APIDeployerDeploymentSettingsController', function($scope, $q, TaggingService, TopNav, DataikuAPI,
    GENERATION_MAPPING_STRATEGIES, GENERATION_MAPPING_MODES, APIDeployerDeploymentUtils, APIDeploymentInfraHelper, ActivityIndicator, FutureWatcher, $controller) {
    $controller('_DeployerPermissionsController', {$scope: $scope});

    TopNav.setNoItem();
    TopNav.setLocation(TopNav.TOP_API_DEPLOYER, 'deployments', null, 'settings');

    $scope.infraSupportOpenAPIURI = false;
    $scope.isEndpointCompatibleWithDeployAnywhere = APIDeployerDeploymentUtils.isEndpointCompatibleWithDeployAnywhere;
    $scope.isEndpointCompatibleWithDeployAnywhereDatabricks = APIDeployerDeploymentUtils.isEndpointCompatibleWithDeployAnywhereDatabricks;
    $scope.generationsMappingModes = GENERATION_MAPPING_MODES;
    $scope.generationsMappingStrategies = GENERATION_MAPPING_STRATEGIES;
    $scope.endpointsPerPackage = {};
    $scope.customEndpointNamePattern = '';
    $scope.customEndpointNameMinLength = 0;
    $scope.customEndpointNameMaxLength = 63;

    $scope.uiState = {
        settingsPane: 'general',
        showInstanceTypeList: false
    };

    $scope.inlineContainerConfig = {
        name: "inline",
        type: "KUBERNETES",
        baseImageType: "EXEC",
        properties: [],
    };

    $scope.authorisationMethodLabel = new Map();
    $scope.authorisationMethodLabel.set('PUBLIC', "Public");
    $scope.authorisationMethodLabel.set('API_KEYS', "API keys");
    $scope.authorisationMethodLabel.set('OAUTH2', "JWT/OAuth2");

    $scope.requestAzureMLInstanceTypes = function(infraId, deploymentInfo) {
        const deploymentInfoOrEmpty = deploymentInfo || {};
        $scope.deployerAPIBase.infras.getAzureMLInstanceTypes(infraId, undefined, deploymentInfoOrEmpty.azWorkspace, deploymentInfoOrEmpty.azResourceGroup, deploymentInfoOrEmpty.azSubscription).success(function (data) {
            $scope.instanceTypes = data;
            $scope.uiState.showInstanceTypeList = true;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.requestDatabricksWorkloadTypes = function() {
        $scope.deployerAPIBase.infras.getDatabricksWorkloadTypes().success(function (data) {
            $scope.workloadTypes = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.requestDatabricksWorkloadSizeTypes = function() {
        $scope.deployerAPIBase.infras.getDatabricksWorkloadSizeTypes().success(function (data) {
            $scope.workloadSizeTypes = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.requestDatabricksExperimentNames = function() {
        if ($scope.uiState.isExperimentLoadingInProgress) {
            return;
        }
        DataikuAPI.externalinfras.infos.listDatabricksExperiments($scope.connectionName)
            .success(function (futureResp) {
                $scope.uiState.isExperimentLoadingInProgress = true;
                FutureWatcher.watchJobId(futureResp.jobId)
                    .success(function (resp) {
                        $scope.experimentNames = resp.result
                            .map(e => e.name)
                            .sort((a, b) => a.localeCompare(b));
                    })
                    .error(setErrorInScope.bind($scope))
                    .finally(() => {
                        $scope.uiState.isExperimentLoadingInProgress = false;
                    });
            })
            .error(setErrorInScope.bind($scope));
    }

    $scope.requestDatabricksModelNames = function() {
        if ($scope.uiState.isModelLoadingInProgress) {
            return;
        }

        DataikuAPI.externalinfras.infos.listDatabricksRegisteredModels($scope.connectionName, $scope.usesUnityCatalog)
            .success(function (futureResp) {
                $scope.uiState.isModelLoadingInProgress = true;
                FutureWatcher.watchJobId(futureResp.jobId)
                    .success(function (resp) {
                        $scope.modelNames = resp.result
                            .map(m => m.name)
                            .sort((a, b) => a.localeCompare(b));
                    })
                    .error(setErrorInScope.bind($scope))
                    .finally(() => {
                        $scope.uiState.isModelLoadingInProgress = false;
                    });
            })
            .error(setErrorInScope.bind($scope));
    }

    $scope.requestVertexAIInstanceTypes = function(infraId, deploymentInfo) {
        const deploymentInfoOrEmpty = deploymentInfo || {};
        $scope.deployerAPIBase.infras.getVertexAIInstanceTypes(infraId, undefined, deploymentInfoOrEmpty.gcpProjectId, deploymentInfoOrEmpty.gcpRegion).success(function (data) {
            $scope.instanceTypes = data;
            $scope.uiState.showInstanceTypeList = true;
        }).error(setErrorInScope.bind($scope));
    }

    const requestVertexAIAcceleratorTypes = function() {
        $scope.deployerAPIBase.infras.getVertexAIAcceleratorTypes().success(function (data) {
            $scope.acceleratorTypes = data.map(d => {
                return { value: d, name: d === "ACCELERATOR_TYPE_UNSPECIFIED" ? "None" : d };
            });
        }).error(setErrorInScope.bind($scope));
    }
    requestVertexAIAcceleratorTypes();

    $scope.auditLogStorageUrl = null;
    $scope.fetchAuditLogStorageUrl = function() {
        $scope.deployerAPIBase.deployments.getAuditLogStorageUrl($scope.lightStatus.infraBasicInfo.id)
            .success(function(storageUrl) {
                $scope.auditLogStorageUrl = storageUrl;
            }).error(setErrorInScope.bind($scope));
    }

    $scope.$watchGroup(
        [
            "deploymentSettings.overrideSettings.unityCatalogSettings",
            "deploymentSettings.usesUnityCatalog",
            "lightStatus.infraBasicInfo.usesUnityCatalog"
        ],
        function([overrides, uses]) {
            if (!$scope.deploymentSettings || !$scope.lightStatus || !$scope.lightStatus.infraBasicInfo) return;
            if (overrides) {
                $scope.usesUnityCatalog = uses;
            } else {
                $scope.usesUnityCatalog = $scope.lightStatus.infraBasicInfo.usesUnityCatalog
            }
        }
    );

    const canAccessAuditing = function(infraType) {
        if (infraType === 'DATABRICKS') {
            return false;
        }

        if (!$scope.isCloud) {
            return true;
        }

        return (infraType !== 'STATIC') || $scope.onlyForDataikerAdminWhenInCloud;
    };

    $scope.$watch("lightStatus", function() {
        if (!$scope.lightStatus) return;
        const infraBasicInfo = $scope.lightStatus.infraBasicInfo;
        const infraType = infraBasicInfo.type;
        $scope.connectionName = infraBasicInfo.connectionName;
        $scope.infraSupportOpenAPIURI = APIDeployerDeploymentUtils.infraSupportOpenAPIURI(infraType);
        $scope.uiState.showAuditingTab = canAccessAuditing(infraType);
        $scope.fetchAuditLogStorageUrl();
        DataikuAPI.apideployer.infras.getLightStatus(infraBasicInfo.id)
            .success(function(infraStatus) {
                if (APIDeploymentInfraHelper.isK8SInfra(infraType) && infraStatus.canDeploy) {
                    $scope.inlineContainerConfig.kubernetesNamespace = infraStatus.infraBasicInfo.k8sNamespace;
                    $scope.inlineContainerConfig.kubeCtlContext = infraStatus.infraBasicInfo.k8sContext;
                    $scope.inlineContainerConfig.kubeConfigPath = infraStatus.infraBasicInfo.k8sConfigPath;
                    $scope.inlineContainerConfig.properties = infraStatus.infraBasicInfo.k8sProperties;
                    $scope.inlineContainerConfig.baseImage = infraStatus.infraBasicInfo.baseImageTag;
                    $scope.inlineContainerConfig.repositoryURL = infraStatus.infraBasicInfo.registryHost;
                    $scope.inlineContainerConfig.prePushMode = infraStatus.infraBasicInfo.prePushMode;
                    $scope.inlineContainerConfig.prePushScript = infraStatus.infraBasicInfo.prePushScript;
                }

                if (infraType === 'DATABRICKS') {
                    $scope.experimentLocation = infraBasicInfo.defaultExperimentLocation || '';
                    $scope.prependExperimentLocation = function () {
                        const experimentName = $scope.deploymentSettings.customExperimentName || '';
                        if (!experimentName.startsWith($scope.experimentLocation)) {
                            const experimentFolderName = experimentName.split('/').pop();
                            $scope.deploymentSettings.customExperimentName = `${infraBasicInfo.defaultExperimentLocation}/${experimentFolderName}`.replace(/\/{1,}/g, '/');
                        }
                    }

                    $scope.catalogName = infraBasicInfo.defaultCatalogName || '';
                    $scope.schemaName = infraBasicInfo.defaultSchemaName || '';
                    const prefix = `${$scope.catalogName}.${$scope.schemaName}`
                    $scope.prependCatalogAndSchema = function () {
                        const modelName = $scope.deploymentSettings.customModelName || '';
                        if (!modelName.startsWith(prefix)) {
                            const modelBaseName = modelName.split('.').pop();
                            $scope.deploymentSettings.customModelName = `${prefix}.${modelBaseName}`.replace(/\.{1,}/g, '.');
                        }
                    }

                    $scope.requestDatabricksWorkloadTypes();
                    $scope.requestDatabricksWorkloadSizeTypes();
                }

                if (infraType === 'SAGEMAKER' && !$scope.instanceTypes) {
                    $scope.deployerAPIBase.infras.getSageMakerInstanceTypes().success(function (data) {
                        $scope.instanceTypes = data;
                    }).error(setErrorInScope.bind($scope));
                }

                if (infraType === 'SAGEMAKER') {
                    $scope.customEndpointNamePattern = new RegExp('^[a-zA-Z0-9]([-a-zA-Z0-9]*[a-zA-Z0-9])?$');
                } else if (infraType === 'AZURE_ML') {
                    $scope.customEndpointNamePattern = new RegExp('^[a-zA-Z](?!.*--)([-a-zA-Z0-9]*[a-zA-Z0-9])?$');
                    $scope.customEndpointNameMinLength = 3;
                    $scope.customEndpointNameMaxLength = 32;
                }

                $scope.endpointsPerPackage = APIDeployerDeploymentUtils.getEndpointsPerPackage($scope.lightStatus);

                if (APIDeploymentInfraHelper.isFullyManagedInfra(infraType)) {
                    $scope.endpoints = $scope.endpointsPerPackage[$scope.deploymentSettings.generationsMapping.generation];
                    if (infraType === 'DATABRICKS') {
                        $scope.compatibleEndpoints = $scope.endpoints.filter(e => $scope.isEndpointCompatibleWithDeployAnywhereDatabricks(e.type));
                    } else {
                        $scope.compatibleEndpoints = $scope.endpoints.filter(e => $scope.isEndpointCompatibleWithDeployAnywhere(e.type));
                    }

                    $scope.isCurrentEndpointIncompatible = false;
                    if (!!($scope.deploymentSettings.endpointId) && !$scope.compatibleEndpoints.find(e => e.id === $scope.deploymentSettings.endpointId)) {
                        $scope.isCurrentEndpointIncompatible = true;
                    }
                }
            })
            .error(setErrorInScope.bind($scope));
    });

    $scope.getAllTags = function () {
        var deferred = $q.defer();
        if (!$scope.hasOwnProperty("allProjectLevelTags")) {
            $scope.allProjectLevelTags = [];
            DataikuAPI.apideployer.deployments.listTags()
                .success(function(data) {
                    $scope.allProjectLevelTags = TaggingService.fillTagsMapFromArray(data);
                    deferred.resolve($scope.allProjectLevelTags);
                })
                .error(() => {
                    setErrorInScope.bind($scope);
                    deferred.resolve($scope.allProjectLevelTags);
                });
        }
        else {
            deferred.resolve($scope.allProjectLevelTags);
        }
        return getRewrappedPromise(deferred);
    };

    $scope.startEditTags  = function() {
        $scope.uiState.newTags = angular.copy($scope.deploymentSettings.tags);
        $scope.uiState.isEditingTags = true;
    };

    $scope.cancelEditTags  = function() {
        $scope.uiState.newTags = null;
        $scope.uiState.isEditingTags = false;
    };

    $scope.validateEditTags  = function() {
        if ($scope.uiState.isEditingTags) {
            $scope.deploymentSettings.tags = angular.copy($scope.uiState.newTags);
            $scope.uiState.isEditingTags = false;
        }
    };

    if (!$scope.deploymentSettings) {
        $scope.getDeploymentSettings();
    }

    $scope.autoFillEventServer = function() {
        $scope.deployerAPIBase.infras.getDkuEventServerUrlAndKeys($scope.lightStatus.infraBasicInfo.type === 'K8S').success(function (urlAndKeys) {
            if (urlAndKeys && urlAndKeys.eventServerUrl) {
                $scope.deploymentSettings.apiNodeLogging.eventServerURL = urlAndKeys.eventServerExtUrl ? urlAndKeys.eventServerExtUrl : "";
                $scope.deploymentSettings.apiNodeLogging.eventServerAuthKey = (urlAndKeys.authKeys && urlAndKeys.authKeys.length)? urlAndKeys.authKeys[0].key : "";
            } else {
                ActivityIndicator.warning("No local event server configured");
            }
        }).error(setErrorInScope.bind($scope));
    };

    // don't initialize until obj is available or else timing issues can occur
    const deregister = $scope.$watch("deploymentSettings", function(nv, ov) {
        if (!nv) return;

        $scope.initPermissions($scope.deploymentSettings.auth, {
            queryThroughDeployer: true,
        }, false);

        deregister();
    }, false);

});


app.controller('APIDeployerTestQueriesController', function($scope, DataikuAPI, Assert, $stateParams, APIDeployerDeploymentUtils) {
    Assert.inScope($scope, 'endpoint');

    $scope.getHeight =  function(q) {
        return Object.keys(q.features).length + 4;
    };

    $scope.isClustering = function () {
        return $scope.endpoint && $scope.endpoint.type === "STD_CLUSTERING";
    }

    $scope.$watch("endpoint", function(nv) {
        if (nv) {
            $scope.genericEndpointType = APIDeployerDeploymentUtils.getGenericEndpointType(nv);
            delete $scope.testQueriesResult;
        }
        if ($scope.endpoint.testQueries.length > 0) {
            $scope.uiState.showUnsavedTestQuery = false;
            $scope.uiState.testQueryIndex = 0;
            $scope.endpoint.testQueries.map((tq, i) => tq.queryIndex = i);
        } else {
            $scope.uiState.showUnsavedTestQuery = true;
        }
    });

    $scope.uiState = {
        requestType: "EMPTY",
        queriesBatchSize: 1,
        inputDatasetSmartName: null
    };
    if (!$scope.endpoint.testQueries) {
        $scope.endpoint.testQueries = []
    }

    $scope.showTestQuery = function(index) {
        $scope.uiState.showUnsavedTestQuery = false;
        $scope.uiState.testQueryIndex = index;
    };

    $scope.showUnsavedTestQuery = function() {
        $scope.uiState.showUnsavedTestQuery = true;
        delete $scope.uiState.testQueryIndex;
    };

    $scope.getCollectedColumnMappings = function() {
        const mappings = {};
        $scope.endpoint.lookups.forEach(function(lookup) {
            angular.forEach(lookup.columnsMapping, function(v,k) {
                mappings[k] = v;
            });
        });
        return mappings;
    };

    $scope.runTestQueries = function() {
        const deploymentId = $scope.lightStatus.deploymentBasicInfo.id;
        const endpointId = $scope.endpoint.id;
        let unsavedTestQueries = {}
        $.each($scope.unsavedTestQueries, function(k, v) {
            if (v) {
                unsavedTestQueries[k] = { q: $scope.unsavedTestQueries[k] }
            }
        })
        DataikuAPI.apideployer.deployments.runTestQuery(deploymentId, endpointId, $scope.endpoint.testQueries, unsavedTestQueries)
            .success(function(testQueriesResult) {
                $scope.testQueriesResult = testQueriesResult;
                for (let curResult of $scope.testQueriesResult.responses) {
                    const contentTypeHeader = curResult.headers.find(h => h.first.toLowerCase() === "content-type");
                    if (contentTypeHeader && contentTypeHeader.second.startsWith("image/")) {
                        curResult.imageType = contentTypeHeader.second;
                        curResult.url = `data:${curResult.imageType};base64,${curResult.response}`;
                    }
                }
            })
            .error(setErrorInScope.bind($scope));
    };

    $scope.sendPlayTestQueriesMsg = function() {
        $scope.runTestQueries();
    };
});

app.controller('APIDeployerDeploymentsDashboardController', function($scope, $controller, $state, $filter, WT1, APIDeployerAsyncHeavyStatusLoader, DeployerUtils, APIDeployerDeploymentService) {
    $controller('_DeployerDeploymentDashboardController', {$scope});

    if ($scope.isFeatureLocked) return;

    $scope.uiState.query.healthStatusMap = [
        'HEALTHY',
        'WARNING',
        'UNHEALTHY',
        'ERROR',
        'UNKNOWN',
        'DISABLED',
        'LOADING_FAILED'
    ].map(hs => ({
        id: hs,
        $selected: false
    }));
    $scope.orderByExpression = ['serviceBasicInfo.id', (deployment) => $filter('deploymentToGenerationList')(deployment.deploymentBasicInfo), '-deploymentBasicInfo.enabled', 'deploymentBasicInfo.id'];

    $scope.stillRefreshing = function() {
        return !$scope.globalLightStatusLoaded || !loader || loader.stillRefreshing();
    };

    $scope.hasShownDeployments = function(deployments) {
        return deployments.length;
    };

    $scope.canCreateDeployments = function() {
        return true; //TODO @mad
    };

    $scope.showRightClickMenu = function($event, cell) {
        const url = $state.href('apideployer.deployments.deployment.status', {deploymentId: cell.deploymentBasicInfo.id});
        $scope.showOpenInANewTabMenu($event, url);
    }

    $scope.startCreateDeployment = function() {
        APIDeployerDeploymentService.startCreateDeployment().then(function(newDeployment) {
            $state.go('apideployer.deployments.deployment.status', {deploymentId: newDeployment.id});
            WT1.event('api-deployer-deployment-create', {deploymentType: newDeployment.type });
        });
    };

    function filterOnSearchBarQuery(lightStatus) {
        if (!$scope.uiState.query.q) return true;
        const query = $scope.uiState.query.q.toLowerCase();
        return lightStatus.deploymentBasicInfo.publishedServiceId.toLowerCase().includes(query)
            || lightStatus.serviceBasicInfo.name.toLowerCase().includes(query)
            || DeployerUtils.getParticipatingVersions(lightStatus.deploymentBasicInfo).join(', ').toLowerCase().includes(query)
            || lightStatus.deploymentBasicInfo.id.toLowerCase().includes(query)
            || lightStatus.deploymentBasicInfo.infraId.toLowerCase().includes(query)
            || lightStatus.deploymentBasicInfo.type.toLowerCase().includes(query);
    }

    $scope.deploymentIsInUI = function(lightStatus) {
        const selectedServices = $scope.uiState.query.services.filter(service => service.$selected);
        const selectedStatuses = $scope.uiState.query.healthStatusMap.filter(hs => hs.$selected);
        const deploymentHealthStatus = $scope.heavyStatusPerDeploymentId[lightStatus.deploymentBasicInfo.id];

        return filterOnSearchBarQuery(lightStatus) &&
            (!selectedServices.length || selectedServices.find(service => service.serviceBasicInfo.id === lightStatus.deploymentBasicInfo.publishedServiceId)) &&
            (!$scope.uiState.query.tags.length || $scope.uiState.query.tags.find(tag => lightStatus.deploymentBasicInfo.tags.find(deplTag => deplTag === tag))) &&
            (!selectedStatuses.length || selectedStatuses.find(hs => deploymentHealthStatus && deploymentHealthStatus.health === hs.id || hs.id === 'DISABLED' && !lightStatus.deploymentBasicInfo.enabled));
    };

    $scope.getFilteredDeploymentCountText = function(deployments) {
        const counts = getDeploymentsCounts(deployments.filter(deployment => $scope.deploymentIsInUI(deployment)));

        return (counts.disabled > 0 ? counts.enabled + '/' : '') + counts.total;
    };

    $scope.$watch('serviceStatusList', function(nv) {
        if (!nv) return;

        $scope.uiState.query.services = angular.copy($scope.serviceStatusList);
    });

    function getDeploymentsCounts(deploymentsStatus) {
        const counts = {
            total: deploymentsStatus.length,
            enabled: deploymentsStatus.filter(ls => ls.deploymentBasicInfo.enabled).length,
        }
        counts.disabled = counts.total - counts.enabled;
        return counts;
    }

    let loader;
    $scope.$watch("deploymentsStatusList", function(nv) {
        if (!nv) return;
        $scope.hasCarbonAPIEnabled = $scope.deploymentsStatusList.some(deployment => deployment.infraBasicInfo.carbonAPIEnabled);
        $scope.deploymentsCounts = getDeploymentsCounts(nv);
        loader = APIDeployerAsyncHeavyStatusLoader.newLoader($scope.heavyStatusPerDeploymentId);
        loader.loadFromDeploymentLightStatusList(nv);
    });

    $scope.$on("$destroy", function() {
        loader.stopLoading();
    });
});


})();
