(function() {
    'use strict';

    const app = angular.module('dataiku.deployer');

    app.factory("deploymentHealthIcon", function () {
        const deploymentHealthIcons = {
            HEALTHY: 'dku-icon-checkmark-circle-outline-',
            WARNING: 'dku-icon-warning-fill-',
            OUT_OF_SYNC: 'dku-icon-arrow-circular-strike-',
            UNHEALTHY: 'dku-icon-dismiss-circle-fill-',
            ERROR: 'dku-icon-dismiss-circle-fill-',
            UNKNOWN: 'dku-icon-empty-',
            LOADING: 'icon-spin dku-icon-arrow-circular-',
            DISABLED: 'dku-icon-line-',
            LOADING_FAILED: 'dku-icon-dismiss-circle-fill-',
        };
        return {
            build: (health, classes, iconSize) => {
                if (!health) {
                    return classes + deploymentHealthIcons.LOADING + iconSize;
                }
                if (deploymentHealthIcons[health]) {
                    return classes + deploymentHealthIcons[health] + iconSize;
                }
                return "";
            }
        }
    });

    app.filter('deploymentHealthToIcon', function(deploymentHealthIcon) {
        return function(health, iconSize) {
            return deploymentHealthIcon.build(health, "deployer-deployment-status-icon ", iconSize || "20");
        };
    });

    app.filter('deploymentHealthToRoundedIcon', function(deploymentHealthIcon) {
        return function(health) {
            return deploymentHealthIcon.build(health, "deployer-deployment-status-icon-round ", "24");
        };
    });

    app.filter('healthStatusToFriendly', function() {
        return function(healthStatus) {
            if (!healthStatus) return;
            return healthStatus.charAt(0).toUpperCase() + healthStatus.substr(1).toLowerCase().replaceAll('_', ' ');
        };
    });

    app.filter('deploymentUpdateStateToIconClasses', function() {
        return (state) => {
            const classes = {
                'RUNNING': 'dku-icon-arrow-circular-16 icon-spin',
                'SUCCESS': 'dku-icon-checkmark-circle-outline-16 success',
                'INFO': 'dku-icon-checkmark-circle-fill-16 success',
                'WARNING': 'dku-icon-warning-fill-16 warning',
                'ERROR': 'dku-icon-dismiss-circle-fill-16 failed',
                'ABORTED': 'dku-icon-cancel-16 aborted'
            }[state];

            const defaultClasses = 'dku-icon-line-16 aborted';

            return classes || defaultClasses;
        }
    });

    app.service('DeployerDeploymentTileService', function($filter) {
        const deploymentHeavyStatusToHealthMessage = function(heavyStatus) {
            const displayedHealth = $filter('healthStatusToFriendly')(heavyStatus.health);
            if (heavyStatus.health === "HEALTHY") {
            // there still might be info messages in this case (appearing in the status page)
            // we do not want to startle the user by having them appear in the tooltip however
                return displayedHealth;
            }

            let severityMessage = heavyStatus.healthMessages.messages.find(msg => msg.severity == heavyStatus.healthMessages.maxSeverity);
            let message = displayedHealth;

            if (severityMessage) {
                message += ' - ' + severityMessage.message;
            }

            return message
        }
        return {
            getDeploymentHealth: function(heavyStatus) {
                var currentState = "LOADING";
                var message = "";

                if (heavyStatus) {
                    currentState = heavyStatus.health;
                    message =  deploymentHeavyStatusToHealthMessage(heavyStatus);
                }

                return {
                    currentState: currentState,
                    message: message
                };
            },
            getDeploymentHealthMap: function(deployments, heavyStatusMap) {
                if (!deployments || !heavyStatusMap) return;

                const healthMap = {};

                deployments.forEach(deployment => {
                    healthMap[deployment.id] = this.getDeploymentHealth(heavyStatusMap[deployment.id]);
                });

                return healthMap;
            }
        }
    });

    app.controller('_DeployerDeploymentDashboardController', function($scope, $controller, $filter, TopNav, TaggingService, openDkuPopin) {
        $controller('_DeployerBaseController', {$scope});
        const navLocation = `TOP_${$scope.deployerType.toUpperCase()}_DEPLOYER`;
        TopNav.setNoItem();
        TopNav.setLocation(TopNav[navLocation], 'deployments');

        if ($scope.isFeatureLocked) return;

        $scope.uiState = $scope.uiState || {};
        $scope.uiState.query = {q: '', tags: []};
        $scope.uiState.query[`${$scope.publishedItemType}s`] = [];

        $scope.orderByExpression = [];

        $scope.rowHeaderHeight = 45;

        $scope.deployerAPIBase.deployments.listTags()
            .success(tags => { TaggingService.setProjectTags(TaggingService.fillTagsMapFromArray(tags)); })
            .error(setErrorInScope.bind($scope));

        $scope.deployerHasPublishedPackages = function(itemList) {
            return (itemList || []).some(item => !!item.packages.length);
        };

        $scope.displayedStages = angular.copy($scope.stages);

        const getDeploymentsPerStage = function(lightStatusList) {
            const deploymentsPerStage = $scope.displayedStages.reduce((obj, stage) => ({ ...obj, [stage.id]: [] }), {});
            lightStatusList.forEach(function(lightStatus) {
                let stageId = lightStatus.infraBasicInfo.stage;
                if (!$scope.displayedStages.find(stage => stage.id === stageId)) {
                    stageId = "__OTHERS__";
                }
                if (!(stageId in deploymentsPerStage)) {
                    deploymentsPerStage[stageId] = [];
                    $scope.displayedStages.push({id: stageId});
                }
                deploymentsPerStage[stageId].push(lightStatus);
            });
            return deploymentsPerStage;
        };

        $scope.refreshForFilter = function(filteredDeploymentStatusList) {
            $scope.deploymentsPerStage = getDeploymentsPerStage(filteredDeploymentStatusList);
            $scope.deploymentsByRow = [];
            $scope.deploymentHeaders = $scope.deploymentHeaders || Object.keys($scope.deploymentsPerStage);
            $scope.columnWidths =  $scope.columnWidths || $scope.deploymentHeaders.map(_ => 330);
            const rowCount = Math.max.apply(null, Object.values($scope.deploymentsPerStage).map(d => d.length));

            if (rowCount > 0) {
                for (let row = 0; row < rowCount; row++) {
                    const rowData = [];
                    for (let stage in $scope.deploymentsPerStage) {
                        rowData.push(($scope.deploymentsPerStage[stage] || [])[row]);
                    }
                    $scope.deploymentsByRow.push(rowData);
                }
            }
        };

        function filterDeployments() {
            const filteredDeploymentStatusList = $filter('orderBy')(
                $scope.deploymentsStatusList.filter(lightStatus => $scope.deploymentIsInUI(lightStatus)), $scope.orderByExpression
            );
            $scope.refreshForFilter(filteredDeploymentStatusList);
        }

        $scope.selectAndFilterDeployments = function(item) {
            item.$selected = !item.$selected;
            filterDeployments();
        };

        $scope.$watch('uiState.query', (nv, ov) => {
            if (nv !== ov) {
                filterDeployments();
            }
        }, true);

        $scope.$on('tagSelectedInList', function(e, tag) {
            let index = $scope.uiState.query.tags.indexOf(tag);
            if (index >= 0) {
                $scope.uiState.query.tags.splice(index, 1);
            } else {
                $scope.uiState.query.tags.push(tag);
            }
            e.stopPropagation();
        });

        $scope.refreshAllDeployments = function() {
            $scope.heavyStatusPerDeploymentId = {};
            $scope.globalLightStatusLoaded = false;
            $scope.deployerAPIBase.globalLightStatus()
                .success(gls => {
                    $scope.globalLightStatusLoaded = true;
                    $scope.deploymentsStatusList = gls.deploymentsStatus;
                    $scope.infraStatusList = gls.infrasStatus;
                    $scope[`${$scope.publishedItemType}StatusList`] = gls[`${$scope.publishedItemType}sStatus`];
                    $scope.refreshForFilter($scope.deploymentsStatusList);
                })
                .error(setErrorInScope.bind($scope));
        }

        $scope.refreshAllDeployments();

        $scope.deployerHasPublishedPackages = function(itemList) {
            return (itemList || []).some(item => !!item.packages.length);
        };

        $scope.countSelectedItems = function(itemList) {
            return (itemList || []).filter(item => item.$selected).length;
        };

        $scope.clearSelectedItems = function(itemList) {
            return (itemList || []).forEach(item => item.$selected = false);
        };

        $scope.clearAndRefreshFilter = function(itemList) {
            $scope.clearSelectedItems(itemList);
            filterDeployments();
        }

        $scope.getSelectedHealthStatuses = function() {
            return $scope.countSelectedItems($scope.uiState.query.healthStatusMap) ? $scope.uiState.query.healthStatusMap.filter(healthStatus => healthStatus.$selected).map(healthStatus => $filter('healthStatusToFriendly')(healthStatus.id)).join(', ') : 'All states';
        }

        $scope.showOpenInANewTabMenu = function($event, url) {
            const template = `<ul class="dropdown-menu">
                <li><a ng-click="openInNewTab('${url}')">Open in a new tab...</a></li>
            </ul>`;
            const isElsewhere = (_, e) => $(e.target).parents('.dropdown-menu').length == 0;
            $scope.lockForPopup();
            const dkuPopinOptions = {
                template: template,
                isElsewhere: isElsewhere,
                popinPosition: 'CLICK',
                onDismiss: $scope.unlockAfterPopup
            };
            $scope.popupDismiss = openDkuPopin($scope, $event, dkuPopinOptions);
        };

        $scope.openInNewTab = (url) => {
            if ($scope.isPopupActive) {
                if ($scope.popupDismiss) {
                    $scope.popupDismiss();
                    $scope.popupDismiss = undefined;
                }
            }
            window.open(url, '_blank');
        }

        $scope.lockForPopup = () => $scope.isPopupActive = true;

        $scope.unlockAfterPopup = () => $scope.isPopupActive = false;
    });

    app.controller('_DeployerPackagesPanelController', function($scope, $stateParams) {
        $scope.DEPLOYED_STATE = {
            'ALL': $scope.deployerType === 'api' ? 'All versions' : 'All bundles',
            'DEPLOYED': 'Deployed',
            'NOT_DEPLOYED': 'Not deployed'
        }

        $scope.uiState = {
            query: {
                search: '',
                deploy: $scope.DEPLOYED_STATE.ALL
            },
            noSelectedItem: true
        };

        function filterPublishedItemList() {
            $scope.uiState.noSelectedItem = itemList.every(item => !item.$selected);

            itemList.forEach(item => {
                let someVersionsAreShown = false;
                item.packages.forEach(pkg => {
                    // show deployed or not
                    pkg.$show = $scope.uiState.query.deploy === $scope.DEPLOYED_STATE.ALL
                        || ($scope.uiState.query.deploy === $scope.DEPLOYED_STATE.DEPLOYED && pkg.$deployed)
                        || ($scope.uiState.query.deploy === $scope.DEPLOYED_STATE.NOT_DEPLOYED && !pkg.$deployed);

                    // filtering by query: if query doesn't match item key/name, filter on packages
                    const query = $scope.uiState.query.search.toLowerCase();
                    if (!item[`${$scope.publishedItemType}BasicInfo`].id.toLowerCase().includes(query) &&
                        !item[`${$scope.publishedItemType}BasicInfo`].name.toLowerCase().includes(query)) {
                        // query matches package id or the id of one of the package deployments
                        pkg.$show = pkg.$show && ( pkg.id.toLowerCase().includes(query)
                            || pkg.$deployments.some(d => d.toLowerCase().includes(query)) );

                    }
                    someVersionsAreShown = pkg.$show || someVersionsAreShown;
                });
                // hide item if it is filtered out or all its packages are filtered out
                item.$show = ($scope.uiState.noSelectedItem || item.$selected) && someVersionsAreShown;
            });
        }

        let itemList;
        $scope.computeFullList = function(statusList) {
            itemList = statusList.filter(item => {
                    // remove any published items with no packages
                    return item.packages.length;
                }).map(item => {
                    item.$show = true; // whether the published item should be shown in list

                    item.packages.forEach(pkg => {
                        pkg.$show = true;
                        let deploymentsInfo = $scope.getPackageDeployments(item.deployments, pkg.id)
                        pkg.$deployments = deploymentsInfo.map(d => d.id);
                        pkg.$deployed = deploymentsInfo.some(d => {
                            if (d.deploymentTag) {
                                return d.deploymentTag.lastModifiedOn > 0
                            } else { //handle old deployment without deploymentTag
                                return !!d.id
                            }
                        });
                    });
                    item.$latestPackage = item.packages.slice(-1).pop();
                    return item;
                });
            return itemList;
        }

        $scope.select = function(item) {
            item.$selected = !item.$selected;
            filterPublishedItemList();
        };

        $scope.clearItemFilter = function() {
            $scope.clearSelectedItems(itemList);
            filterPublishedItemList();
        };

        $scope.setDeployFilter = function(deploy) {
            $scope.uiState.query.deploy = deploy;
            filterPublishedItemList();
        };

        $scope.clearFilters = function() {
            $scope.uiState.query.search = '';
            $scope.uiState.query.deploy = $scope.DEPLOYED_STATE.ALL;
            $scope.clearItemFilter();
        };

        $scope.getFilteredPackageCount = function(item) {
            return item.packages.filter(pkg => pkg.$show).length;
        };

        $scope.deployerHasMatchingPublishedPackages = function() {
            return (itemList || []).some(item => item.$show && !!$scope.getFilteredPackageCount(item));
        };

        $scope.isNewPackage = function(pck) {
            return !pck.$deployed && pck.publishedOn >= (new Date()).getTime() - 24 * 3600 * 1000;
        };

        $scope.hasActiveFilters = function() {
            return $scope.uiState.query.search
                || $scope.uiState.query.deploy !== $scope.DEPLOYED_STATE.ALL
                || (itemList || []).some(item => item.$selected);
        };

        $scope.$watch("uiState.query.search", function(nv, ov) {
            if (nv === ov) return;
            filterPublishedItemList();
        });
    });

    app.controller('_DeployerDeploymentController', function($controller, $scope, $state, Dialogs, Assert, $rootScope, TaggingService, $filter, $q, $interval,
                                                             ActivityIndicator, WT1, SharedLastDeploymentAction) {
        $controller("_DeployerBaseController", {$scope: $scope});
        $scope.savedDeploymentSettings; // for dirtinessDetection

        $rootScope.activeProjectTagColor = TaggingService.getTagColor;

        $scope.getLightStatus = function() {
            return $scope.deployerAPIBase.deployments.getLightStatus($state.params.deploymentId).success(lightStatus => {
                $scope.lightStatus = lightStatus;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.getHeavyStatus = function(noSpinner = false) {
            Assert.trueish($scope.lightStatus);
            return $scope.deployerAPIBase.deployments.getHeavyStatus($scope.lightStatus.deploymentBasicInfo.id, true, noSpinner).success(function(data) {
                $scope.heavyStatus = data;
            });
        };

        $scope.refreshLightAndHeavy = function(){
            $scope.getLightStatus(); // Heavy is refreshed by watch
            $scope.refreshLastDeploymentAction();
        };

        $scope.getDeploymentSettings = function() {
            return $scope.deployerAPIBase.deployments.getSettings($state.params.deploymentId).success(function(data) {
                $scope.savedDeploymentSettings = angular.copy(data);
                $scope.deploymentSettings = data;
                if ($scope.deploymentSettings === "VERTEX_AI" && !$scope.deploymentSettings.machineConfig.acceleratorType) {
                    // It will make $scope.acceleratorTypes work correctly
                    $scope.deploymentSettings.machineConfig.acceleratorType = "ACCELERATOR_TYPE_UNSPECIFIED";
                }
            }).error(setErrorInScope.bind($scope));
        };

        $scope.deploymentIsDirty = function() {
            return $scope.deploymentSettings && !angular.equals($scope.savedDeploymentSettings, $scope.deploymentSettings);
        };

        $scope.isDeploymentSettingsInvalid = function() {
            return $scope.deploymentSettingsForm.$invalid;
        }

        $scope.setTagsValidity = function(isValid) {
            if (!$scope.deploymentSettingsForm) return;

            $scope.deploymentSettingsForm.$setValidity('tagsEditableList', isValid);
        };

        $scope.setEnvVarsValidity = function(isValid) {
            if (!$scope.deploymentSettingsForm) return;

            $scope.deploymentSettingsForm.$setValidity('envVarsEditableList', isValid);
        };

        $scope.setLabelsValidity = function(isValid) {
            if (!$scope.deploymentSettingsForm) return;

            $scope.deploymentSettingsForm.$setValidity('labelsEditableList', isValid);
        };

        $scope.setAnnotationsValidity = function(isValid) {
            if (!$scope.deploymentSettingsForm) return;

            $scope.deploymentSettingsForm.$setValidity('annotationsEditableList', isValid);
        };

        $scope.saveDeployment = function() {
            Assert.trueish($scope.deploymentSettings);
            if (!$scope.deploymentIsDirty() || $scope.isDeploymentSettingsInvalid()) return;
            return $scope.deployerAPIBase.deployments.save($scope.deploymentSettings).success(function(data) {
                $scope.getLightStatus();
                $scope.savedDeploymentSettings = angular.copy(data);
                $scope.deploymentSettings = data;
            }).error(setErrorInScope.bind($scope));
        };

        $scope.deleteDeployment = function() {
            if (!$scope.lightStatus) {
                return;
            }
            Dialogs.confirmAlert($scope, 'Delete deployment', 'Are you sure you want to delete this deployment?',
                                 $scope.deleteWarning, 'WARNING').then(function() {
                WT1.event(`${$scope.deployerType}-deployer-deployment-delete`, { deploymentType: $scope.lightStatus.deploymentBasicInfo.type });
                $scope.deployerAPIBase.deployments.delete($scope.lightStatus.deploymentBasicInfo.id)
                    .success(_ => $state.go(`${$scope.deployerType}deployer.deployments.dashboard`))
                    .error(function(data, status, headers, config, statusText, xhrStatus) {
                        // Assume a coded exception comes from the pre-delete actions
                        if(data && data.code) {
                            Dialogs.confirmAlert($scope, data.title, 'Do you want to delete the deployment anyway?', data.detailedMessage, "ERROR").then(function() {
                                $scope.deployerAPIBase.deployments.delete($scope.lightStatus.deploymentBasicInfo.id, true)
                                .success(_ => $state.go(`${$scope.deployerType}deployer.deployments.dashboard`))
                                .error(setErrorInScope.bind($scope));
                            });
                        } else {
                            setErrorInScope(data, status, headers, config, statusText, xhrStatus)
                        }
                    });
            });
        };

        $scope.lastDeploymentAction = null;
        $scope.cancelFetchLastDeploymentAction = null;

        function lastDeploymentActionChanged(deploymentAction) {
            if (deploymentAction === null) return $scope.lastDeploymentAction !== null;
            else {
                if ($scope.lastDeploymentAction === null) return true;
                return $scope.lastDeploymentAction.type !== deploymentAction.type || $scope.lastDeploymentAction.inProgress !== deploymentAction.inProgress
            }
        }

        function deploymentUpdateFinished(deploymentAction) {
            return $scope.lastDeploymentAction?.inProgress &&
                $scope.lastDeploymentAction.type === 'UPDATE' &&
                deploymentAction?.type === 'UPDATE' &&
                !deploymentAction.inProgress;
        }

        $scope.fetchLastDeploymentActionUntilCanceled = function() {
            $scope.deployerAPIBase.deployments.getLastDeploymentAction($state.params.deploymentId).success(lastDeploymentAction => {
                SharedLastDeploymentAction.setData(lastDeploymentAction);
                if (lastDeploymentActionChanged(lastDeploymentAction)) {
                    if (deploymentUpdateFinished(lastDeploymentAction)) {
                        if (lastDeploymentAction.status === 'SUCCESS' ||
                            lastDeploymentAction.status === 'WARNING' ||
                            lastDeploymentAction.status === 'INFO') {
                            let message = 'Deployment updated successfully' +
                                (lastDeploymentAction.status === 'WARNING' ? ' (with warnings)' : '');
                            ActivityIndicator.success(message);
                        } else {
                            ActivityIndicator.error('Deployment update failed');
                        }
                        $scope.onUpdateFinished();
                    }
                    // the last deployment action changed, we may have to change the pace at which we poll last deployment action
                    $scope.stopFetchLastDeploymentAction();
                }
                $scope.lastDeploymentAction = lastDeploymentAction;
                if ($scope.cancelFetchLastDeploymentAction == null) {
                    $scope.fetchLastDeploymentActionRepeat();
                }
            }).error(function(a, b, c) {
                $scope.stopFetchLastDeploymentAction();
                setErrorInScope.bind($scope)(a, b, c);
            });
        }

        $scope.fetchLastDeploymentActionRepeat = function() {
            let interval;
            if ($scope.lastDeploymentAction === null || !$scope.lastDeploymentAction.inProgress) {
                // nothing is ongoing, polling last actions on this deployment at a slower pace
                interval = 10;
            } else {
                interval = 1;
            }
            $scope.cancelFetchLastDeploymentAction = $interval(function() {
                $scope.fetchLastDeploymentActionUntilCanceled();
            }, interval*1000);
        }

        $scope.stopFetchLastDeploymentAction = function() {
            if ($scope.cancelFetchLastDeploymentAction) {
                $interval.cancel($scope.cancelFetchLastDeploymentAction);
            }
            $scope.cancelFetchLastDeploymentAction = null;
        }

        $scope.refreshLastDeploymentAction = function() {
            $scope.stopFetchLastDeploymentAction();
            $scope.fetchLastDeploymentActionUntilCanceled();
        }

        $scope.$on('$destroy', function () {
            $scope.stopFetchLastDeploymentAction();
        });

        $scope.onPeekingUpdateStarted = function() {
            $scope.stopFetchLastDeploymentAction();
            $scope.deployerAPIBase.deployments.getLastDeploymentAction($state.params.deploymentId).success(lastDeploymentAction => {
                SharedLastDeploymentAction.setData(lastDeploymentAction);
                $scope.lastDeploymentAction = lastDeploymentAction;

                // An edge case of an "instant" deployment when its update is already finished by the time we start peeking into it
                if (lastDeploymentAction && !lastDeploymentAction.inProgress) {
                    $scope.onUpdateFinished();
                }
            }).error(setErrorInScope.bind($scope));
        }

        $scope.otherActionInProgressMessage = function() {
            if ($scope.lastDeploymentAction?.inProgress) {
                return $filter('capitalize')($scope.lastDeploymentAction.type.toLowerCase()) + " in progress";
            }
            else {
                return null;
            }
        }

        $scope.showProgress = function() {
            if ($scope.lastDeploymentAction) {
                $scope.openProgressModal();
            }
        }

        $scope.insufficientPermissionsMessage = function() {
            return $scope.lightStatus && $scope.lightStatus.hasFullDeployPermissions ? "" : "You don't have enough permissions on this deployment";
        }

        /* Main */
        $scope.getLightStatus();
        $scope.fetchLastDeploymentActionUntilCanceled();
    });

    app.component('deploymentAction', {
        bindings: {
            action:'<',
            show: '&'
        },
        template: `
        <div class="horizontal-flex mx-center-children-vertically">
            <i class="dku-icon-arrow-circular-16 icon-spin mright16"></i>
            <div>
                {{ $ctrl.action.type | lowercase | capitalize }} in progress since <em>{{ $ctrl.action.startTimestamp | date:'yyyy-MM-dd HH:mm:ss' }}</em>
                , by <em>{{ $ctrl.action.requester.displayName || $ctrl.action.requester.login }}</em>. Thank you for your patience while the content is loading...
            </div>
            <div class="mleftauto mright8">
                <a ng-click="$ctrl.show()">
                    Show progress
                </a>
            </div>
        </div>
        `,
        controller: function() {
            const $ctrl = this;
        }
    });

    app.component('deploymentUpdates', {
        bindings: {
            listUpdateHeads: '<',
            getUpdate: '&',
            getSettingsDiff: '&',
            deployerType: '<'
        },
        templateUrl: '/templates/deployer/deployment-updates.html',
        controller: function($scope, CreateModalFromTemplate, DiffFormatter, $filter, SharedLastDeploymentAction, StateUtils) {

            const $ctrl = this;

            $ctrl.$onInit = function() {
                $ctrl.isProjectDeployer = $ctrl.deployerType === "project";
                $ctrl.emptyStateImage = `static/dataiku/images/deployer/empty-state-last-updates-${$ctrl.deployerType}-deployer.svg`;
                $ctrl.refreshUpdatesList();
            };

            $ctrl.refreshUpdatesList = function(reloadSelectedUpdate = false) {
                $ctrl.listUpdateHeads()
                .success(function(data) {
                    $ctrl.updateHeads = data || [];
                    if ($ctrl.updateHeads.length === 0) return;
                    if (!$ctrl.selectedUpdate) {
                        $ctrl.selectUpdate(data[0])
                    } else if (reloadSelectedUpdate) {
                        const refreshedUpdate = data.find(d => d.startTimestamp === $ctrl.selectedUpdate.startTimestamp);
                        if (refreshedUpdate?.startTimestamp) loadUpdate(refreshedUpdate);
                    }
                })
                .error(setErrorInScope.bind($scope));
            };

            $ctrl.selectUpdate = function(update) {
                if ($ctrl.selectedUpdate?.startTimestamp === update.startTimestamp) {
                    return;
                }

                loadUpdate(update);
                loadSettingsDiff(update);
            };

            function setDeploymentStatus(heavyStatus) {
                if (!heavyStatus) {
                    $ctrl.deploymentStatusIcon = "dku-icon-empty-16";
                    $ctrl.deploymentStatusTooltipText = "Unknown";
                    return;
                }
                $ctrl.deploymentStatusIcon = $filter('deploymentHealthToIcon')(heavyStatus.health, 16);


                $ctrl.deploymentStatusTooltipText = $filter('healthStatusToFriendly')(heavyStatus.health);
                const description = heavyStatus.healthMessages.messages.find(msg => msg.severity === heavyStatus.healthMessages.maxSeverity);

                if (description) {
                    $ctrl.deploymentStatusTooltipText += ` - ${description.message}`;
                }
            }

            function setPublishedItemValues(update) {
                const { publishedItemIds: ids, deployment } = update;

                $ctrl.publishedItemValues = {
                    text: ids.join(", "),
                    url: $ctrl.isProjectDeployer
                        ? StateUtils.href.projectDeployer.bundle(deployment.publishedProjectKey, ids[0])
                        : StateUtils.href.apiDeployer.serviceVersions(deployment.publishedServiceId, ids)
                };
            }

            function loadUpdate(update) {
                $ctrl.getUpdate({ startTimestamp: update.startTimestamp })
                    .success(function (data) {
                        if (!data) return;
                        $ctrl.selectedUpdate = data;
                        $ctrl.selectedUpdate.status = update.status; // get the status from the selected update head
                        setDeploymentStatus(data.heavyStatus);
                        setDeployedAs();
                        setPublishedItemValues(data);
                    })
                    .error(setErrorInScope.bind($scope));
            }

            function loadSettingsDiff(update) {
                $ctrl.getSettingsDiff({ startTimestamp: update.startTimestamp })
                    .success(function(diff) {
                        const allChanges = Array.isArray(diff.items) ? diff.items : [];
                        const changesToDisplay = allChanges.filter(c =>
                            c?.fileChange?.addedLines > 0 || c?.fileChange?.removedLines > 0
                        );

                        // The backend returns an empty array in case there is no previous update
                        const hasPreviousUpdate = allChanges.length > 0;
                        const hasVisibleChanges = changesToDisplay.length > 0;
                        const displaysInfraWarning = shouldDisplayInfraWarning(allChanges, $ctrl.selectedUpdate, diff.hasInfraChange);
                        const hasDataToDisplay = hasVisibleChanges || displaysInfraWarning;

                        $ctrl.changesToDisplay = changesToDisplay;
                        $ctrl.displaysInfraWarning = displaysInfraWarning;

                        $ctrl.isViewChangesBtnEnabled = displaysInfraWarning || (hasPreviousUpdate && hasDataToDisplay);
                        $ctrl.changesBtnTooltipText = getViewChangesTooltip(hasPreviousUpdate, hasDataToDisplay);
                    })
                    .error(setErrorInScope.bind($scope));
            }

            function getViewChangesTooltip(hasPreviousUpdate, hasDataToDisplay) {
                if (!hasPreviousUpdate) return "No prior update available to compare changes";
                if (!hasDataToDisplay) return "No known changes for this update";
                return null;
            }

            function shouldDisplayInfraWarning(changes, selectedUpdate, isInfraSettingsChanged) {
                const isUpdateNotSuccessful = selectedUpdate?.status !== "SUCCESS";
                const isInfraSettingsDiffMissing = !changes.some(update =>
                    update.id?.toLowerCase?.() === "infrastructure"
                );
                return isUpdateNotSuccessful && isInfraSettingsDiffMissing && isInfraSettingsChanged;
            }

            $ctrl.openUpdateChangesModal = function() {
                const templateUrl = "/templates/deployer/update-changes-modal.html";
                CreateModalFromTemplate(templateUrl, $scope, null, function(modalScope) {
                    modalScope.toggle = update => update.shown = !update.shown;
                    modalScope.displaysInfraWarning = $ctrl.displaysInfraWarning;

                    modalScope.displayedChanges = $ctrl.changesToDisplay.map(update => ({
                        ...update,
                        shown: true,
                        rendered: DiffFormatter.formatChange(update.fileChange)
                    }));
                }, true);
            };

            function setDeployedAs() {
                const { infra, deployment } = $ctrl.selectedUpdate || {};
                $ctrl.deployedAsItems = [];
                if (!$ctrl.isProjectDeployer || !infra.setDeployAsUser ) return;
                // Check for perProjectDeployAs before Default DeployAs settings
                const perProjectUser = infra.perProjectDeployAsUser?.find(item => item.key === deployment?.deployedProjectKey)?.value;
                const automationUser = perProjectUser || infra.deployAsUser;
                if (!automationUser) return;
                if (infra.automationNodes?.length > 0) {
                    infra.automationNodes.map(node => {
                        $ctrl.deployedAsItems.push({
                            name: `${automationUser} (${node.automationNodeId})`,
                            URL: `${node.automationNodeUrl}/profile/${automationUser}/`
                        })
                    })
                } else {
                    $ctrl.deployedAsItems.push({
                        name: `${automationUser}`,
                        URL: `${infra.automationNodeUrl}/profile/${automationUser}/`
                    })
                }
            }

            $scope.$watch(function() { return SharedLastDeploymentAction.data; }, function(nv, ov) {
                if (nv && nv !== ov) {
                    $ctrl.refreshUpdatesList(true);
                }
            }, true);
        }
    });

    app.factory('SharedLastDeploymentAction' , function() {
        var service = {
            data: null,
            setData: function(data) {
                service.data = angular.copy(data);
            }
        };
        return service;
    });

})();
