(function() {
    'use strict';

    const app = angular.module('dataiku.deployer');

    app.constant('GOVERN_CHECK_POLICIES', [
        { id: 'PREVENT', label: 'Prevent the deployment of unapproved packages' },
        { id: 'WARN', label: 'Warn and ask for a confirmation before deploying unapproved packages' },
        { id: 'NO_CHECK', label: 'Always deploy without checking', default: true },
    ])

    app.filter('infoMessageSeverityToIcon', function() {
        const dict = {
            SUCCESS: 'dku-icon-checkmark-circle-outline-20 text-success',
            INFO: 'dku-icon-checkmark-circle-outline-20 text-success',
            WARNING: 'dku-icon-warning-fill-20 text-warning',
            ERROR: 'dku-icon-dismiss-circle-fill-20 text-error'
        };
        return function(severity) {
            if (!severity) {
                return dict.SUCCESS;
            }
            return dict[severity] || '';
        };
    });

    app.controller('_DeployerInfrasListController', function($scope, $controller, $state, TopNav, CreateModalFromTemplate, WT1, $rootScope, GOVERN_CHECK_POLICIES) {
        $scope.GOVERN_CHECK_POLICIES = GOVERN_CHECK_POLICIES;

        $controller('_DeployerBaseController', {$scope});

        const navLocation = `TOP_${$scope.deployerType.toUpperCase()}_DEPLOYER`;
        TopNav.setNoItem();
        TopNav.setLocation(TopNav[navLocation], 'infras');

        if ($scope.isFeatureLocked) return;

        $scope.uiState = {};

        $scope.canCreateInfras = function() {
            return $rootScope.appConfig.admin;
        };

        $scope.startCreateInfra = function() {
            CreateModalFromTemplate(`/templates/${$scope.deployerType}-deployer/new-infra-modal.html`, $scope).then(function(newInfra) {
                $state.go(`${$scope.deployerType}deployer.infras.infra.settings`, {infraId: newInfra.id});
                WT1.event(`${$scope.deployerType}-deployer-infra-setup`, {infraType: newInfra.type});
            });
        };

        $scope.refreshInfraStatusList = function() {
            $scope.deployerAPIBase.infras.listLightStatus()
                .success(function(infraStatusList) {
                    $scope.infraStatusList = infraStatusList;
                }).error(setErrorInScope.bind($scope));
        };

        $scope.refreshInfraStatusList();
    });

    app.controller('_DeployerInfraController', function($scope, $state, $controller, Dialogs, ActivityIndicator, WT1, CreateModalFromComponent, multiNodeInfraApiKeyCreationModalDirective, DataikuAPI, CreateModalFromTemplate) {
        $controller('_DeployerBaseController', {$scope: $scope});
        $scope.refreshInfraStatus = function() {
            $scope.deployerAPIBase.infras.getLightStatus($state.params.infraId)
            .success(infraStatus => {
                $scope.infraStatus = infraStatus;
                $scope.isMultiAutomationNodeInfra = infraStatus && infraStatus.infraBasicInfo.type === 'MULTI_AUTOMATION_NODE';
            }).error(setErrorInScope.bind($scope));
        }

        $scope.deleteInfra = function() {
            if (!$scope.infraStatus) {
                return;
            }
            if ($scope.infraStatus.deployments.length) {
                Dialogs.error($scope, 'Delete infra', 'You cannot delete this infra because it still has deployments!');
                return;
            }
            Dialogs.confirm($scope, 'Delete infra','Are you sure you want to delete this infra?').then(function() {
                WT1.event(`${$scope.deployerType}-deployer-infra-delete`, {infraType: $scope.infraStatus.infraBasicInfo.type});
                $scope.deployerAPIBase.infras.delete($scope.infraStatus.infraBasicInfo.id)
                    .success(() => {
                        ActivityIndicator.success(`Infra ${$scope.infraStatus.infraBasicInfo.id} successfully deleted.`)
                        $state.go($scope.deployerType + 'deployer.infras.list');
                    })
                    .error(setErrorInScope.bind($scope));
            });
        };

        $scope.generatePersonalAPIKey = function() {
            CreateModalFromComponent(multiNodeInfraApiKeyCreationModalDirective, {
                infraId: $scope.infraStatus.infraBasicInfo.id,
                isAdmin: $scope.infraStatus.isAdmin,
            });
        }

        $scope.runConsistencyChecks = () => {
            WT1.event(`multi-node-project-infra-run-consistency-checks`, {});
            DataikuAPI.projectdeployer.infras.runConsistencyChecks($scope.infraStatus.infraBasicInfo.id)
                .then(result => {
                    CreateModalFromTemplate("/templates/project-deployer/consistency-check-report-modal.html", $scope, null, function(modalScope) {
                        modalScope.report = result.data;
                        modalScope.automationNodes = (result.data.automationNodeIds || []).join(", ");
                        let nbOkConnections = 0;
                        let nbKoConnections = 0;
                        Object.values(modalScope.report.connectionsChecks).forEach(connection => {
                            if (connection.messages.length === 0) {
                                connection.$reduced = true;
                                nbOkConnections++;
                            } else {
                                nbKoConnections++;
                            }
                        })
                        modalScope.nbKoConnections = nbKoConnections;
                        modalScope.nbOkConnections = nbOkConnections;
                    });
                })
                .catch(setErrorInScope.bind($scope));
        }

        $scope.refreshInfraStatus();
    });

    app.controller('_DeployerInfraStatusController', function($scope, TopNav, DataikuCloudService) {
        const navLocation = `TOP_${$scope.deployerType.toUpperCase()}_DEPLOYER`;
        $scope.isCloud = DataikuCloudService.isDataikuCloud();

        TopNav.setNoItem();
        TopNav.setLocation(TopNav[navLocation], 'infras', null, 'status');
    });

    app.controller('_DeployerInfraSetupModalController', function($scope, DeployerUtils, GOVERN_CHECK_POLICIES, DataikuCloudService) {
        $scope.isCloud = DataikuCloudService.isDataikuCloud();

        $scope.newInfra = {
            stage: (($scope.stages || [])[0] || {}).id,
            governCheckPolicy: GOVERN_CHECK_POLICIES.find(gcp => gcp.default).id
        };

        $scope.hasUrlSuffix = DeployerUtils.hasUrlSuffix;

        $scope.ok = function() {
            $scope.deployerAPIBase.infras.create($scope.newInfra)
                .success($scope.resolveModal)
                .error(setErrorInScope.bind($scope));
        };
    });

    app.controller('_DeployerInfraHistoryController', function($scope, TopNav) {
        const navLocation = `TOP_${$scope.deployerType.toUpperCase()}_DEPLOYER`;
        TopNav.setNoItem();
        TopNav.setLocation(TopNav[navLocation], 'infras', null, 'history');
    });

    app.controller('_DeployerInfraSettingsController', function($scope, $controller, $state, TopNav, ActivityIndicator,
        GOVERN_CHECK_POLICIES, DeployerUtils, WT1) {
        $scope.GOVERN_CHECK_POLICIES = GOVERN_CHECK_POLICIES;

        let projectStandardsSeverities = [
            { id: 1, label: 'Lowest' },
            { id: 2, label: 'Low'},
            { id: 3, label: 'Medium'},
            { id: 4, label: 'High'},
            { id: 5, label: 'Critical'},
        ];

        $scope.maxSeverityOptions = [{ id: 0, label: 'No issues', default: true }, ...projectStandardsSeverities];
        $scope.checkErrorHandlingOptions = [{ id: 0, label: 'Ignore check errors', default: true }, ...projectStandardsSeverities];

        const navLocation = `TOP_${$scope.deployerType.toUpperCase()}_DEPLOYER`;
        TopNav.setNoItem();
        TopNav.setLocation(TopNav[navLocation], 'infras', null, 'settings');

        $scope.uiState = {
            settingsPane: 'general',
            hooks: {
                selected: undefined,
                errorsAndWarnings: undefined,
                existingHookNames: []
            }
        };

        $scope.hasUrlSuffix = DeployerUtils.hasUrlSuffix;

        // Hackish : to check for the existence of a nodes directory, we test to see if the deployer is registered in the nodes directory.
        // only use this in a context where you already know that the deployer is active.
        $scope.deployerAndNodesDirectoryEnabled = $scope.appConfig.nodesDirectoryManagedDeployerServer || $scope.appConfig.nodesDirectoryManagedDeployerClient;

        $scope.invalidTabs = new Set();
        // When we change tab, we update the fields of infraSettingsForm. So the status of $scope.infraSettingsForm.$invalid is updated (since the current form changes).
        // But we don't want to forget a previous invalid tab, so we add it in invalidTabs.
        $scope.$watch("uiState.settingsPane", function(nv, ov) {
            if (nv === ov) return;
            if ($scope.infraSettingsForm.$invalid) {
                $scope.invalidTabs.add(ov); // we store the id of the previous invalid tab
            }
            $scope.invalidTabs.delete(nv); // the new tab validity will be managed by $scope.infraSettingsForm.$invalid

            // Those fields validity are managed by hand and not auto cleaned when the field disappears during a tab switch.
            // So we reset their validity manually.
            $scope.setTagsValidity(true);
            $scope.setEnvVarsValidity(true);
            $scope.setLabelsValidity(true);
            $scope.setAnnotationsValidity(true);
        });

        $scope.referencedConnections = [];
        $scope.$watch("infra.remappedConnections", function(nv, ov) {
            if (nv === ov) return;
            $scope.referencedConnections = Object.entries(nv || {})
                                                 .filter(([name, connec]) => isConnectionFsLike(connec))
                                                 .map(([name, connec]) => name);
        });

        // This check is the same that the one in ConnectionPathTargetProcessor#createFSProvider().
        // So we don't use $scope.isFsProviderizable() from connections.js
        function isConnectionFsLike(connection) {
            switch (connection.type) {
                case "Filesystem":
                case "Azure":
                case "EC2":
                case "GCS":
                    return true;
                default:
                    return false;
            }
        }

        // An object storage connection needs a bucket or a container to work
        function isConnectionObjectStorage(connection) {
            switch (connection.type) {
                case "Azure":
                case "EC2":
                case "GCS":
                    return true;
                default:
                    return false;
            }
        }

        let savedInfra; // for dirtyness detection
        function refreshInfra() {
            $scope.deployerAPIBase.infras.getSettings($state.params.infraId)
                .success(infra => {
                    $scope.infra = infra;

                    if (!$scope.infra.authConnection) {
                        // It will make $scope.connectionsWithEnvironment work correctly
                        $scope.infra.authConnection = null;
                    }
                    if ($scope.infra.type === "VERTEX_AI" && !$scope.infra.machineConfig.acceleratorType) {
                        // It will make $scope.acceleratorTypes work correctly
                        $scope.infra.machineConfig.acceleratorType = "ACCELERATOR_TYPE_UNSPECIFIED";
                    }

                    $scope.updateExistingHookNames();
                    savedInfra = angular.copy(infra);
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.infraIsDirty = function() {
            return !angular.equals(savedInfra, $scope.infra);
        };

        $scope.isInfraSettingsFormInvalid = function() {
            return $scope.infraSettingsForm.$invalid || $scope.invalidTabs.size;
        }

        $scope.setTagsValidity = function(isValid) {
            if (!$scope.infraSettingsForm) return;

            $scope.infraSettingsForm.$setValidity('tagsEditableList', isValid);
        };

        $scope.setEnvVarsValidity = function(isValid) {
            if (!$scope.infraSettingsForm) return;

            $scope.infraSettingsForm.$setValidity('envVarsEditableList', isValid);
        };

        $scope.getUrlSuffixWarning = function(value) {
            if ($scope.hasUrlSuffix(value)) {
                return "URL should be http[s]://host[:port]. A URL suffix is unexpected and will likely not work.";
            }
            return null;
        }

        $scope.setLabelsValidity = function(isValid) {
            if (!$scope.infraSettingsForm) return;

            $scope.infraSettingsForm.$setValidity('labelsEditableList', isValid);
        };

        $scope.setAnnotationsValidity = function(isValid) {
            if (!$scope.infraSettingsForm) return;

            $scope.infraSettingsForm.$setValidity('annotationsEditableList', isValid);
        };

        function getConnectionDetails(connectionName) {
            if (!($scope.infra.remappedConnections)) {
                return null;
            }
            return $scope.infra.remappedConnections[connectionName];
        }

        $scope.isAzureConnectionOAuth = function(connectionName) {
            const connection = getConnectionDetails(connectionName);
            return !!(connection)
            && connection.type == "Azure"
            && connection.params.authType == "OAUTH2_APP";
        }

        function doesConnectionHaveABucket(connection) {
            if (!connection || !(connection.params)) {
                return false;
            }
            return !!(connection.params.chbucket) || !!(connection.params.chcontainer);
        }

        $scope.doesFsLikeSettingsOverrideConnectionBucket = function(connectionName) {
            const fsLikeSettings = $scope.infra.defaultApiNodeLogging.fsLikeSettings;
            const connection = getConnectionDetails(connectionName);
            return doesConnectionHaveABucket(connection) && !!(fsLikeSettings.bucket);
        }

        $scope.doWeHaveAConnectionButNoBucketDefinedHereNorInFsLikeSettings = function(connectionName) {
            const connection = getConnectionDetails(connectionName);
            if (!connection || !isConnectionObjectStorage(connection)) {
                return false;
            }
            const fsLikeSettings = $scope.infra.defaultApiNodeLogging.fsLikeSettings;
            return !doesConnectionHaveABucket(connection) && !(fsLikeSettings.bucket);
        }

        $scope.auditLogStorageUrl = null;
        $scope.fetchAuditLogStorageUrl = function() {
            const fsLikeSettings = $scope.infra.defaultApiNodeLogging.fsLikeSettings;
            $scope.deployerAPIBase.infras.getAuditLogStorageUrl($state.params.infraId, fsLikeSettings.connectionName, fsLikeSettings.pathWithinConnection, fsLikeSettings.bucket)
                .success(function(storageUrl) {
                    $scope.auditLogStorageUrl = storageUrl;
                }).error(setErrorInScope.bind($scope));
        }

        $scope.$watch("infra.defaultApiNodeLogging.fsLikeSettings", function(nv, ov) {
            if (nv === ov) return;
            $scope.fetchAuditLogStorageUrl();
        }, true);

        $scope.saveInfra = function(createOrUpdateHook) {
            if (!$scope.infra) return;
            // Hitting ctrl-s or meta-s while creating/editing a hook will also execute `saveInfra` as defined by the `global-keydown` directive in `infra-settings.html`,
            // but without the updates on the hook persisted to `$scope.infra`. So we stop the execution here in that case.
            if ($scope.uiState.hooks.selected && !createOrUpdateHook) return;

            WT1.event(`${$scope.deployerType}-deployer-infra-save`, {infraType: $scope.infra.type});

            $scope.deployerAPIBase.infras.save($scope.infra)
                .success(function(result) {
                    if ($scope.isInfraSettingsFormInvalid()) {
                        ActivityIndicator.warning("Saved with some invalid fields");
                    }
                    if (result && result.maxSeverity && result.maxSeverity !== "SUCCESS") {
                        $scope.uiState.hooks.errorsAndWarnings = result;
                    } else {
                        $scope.uiState.hooks.errorsAndWarnings = undefined;
                    }
                    refreshInfra();
                    $scope.refreshInfraStatus();
                    if (createOrUpdateHook) {
                        $scope.updateExistingHookNames();
                        $scope.uiState.hooks.selected = undefined;
                    }
                }).error(setErrorInScope.bind($scope));
        };

        $scope.updateExistingHookNames = function() {
            $scope.uiState.hooks.existingNames = $scope.infra.deploymentHookSettings ?
                $scope.infra.deploymentHookSettings.preDeploymentHooks.map(h => h.name).concat($scope.infra.deploymentHookSettings.postDeploymentHooks.map(h => h.name))
                : [];
        }

        $scope.resetHookErrorsAndWarnings = function($event) {
            $scope.uiState.hooks.errorsAndWarnings = undefined;
        }

        $scope.updateHooksSettings = function($event) {
            $scope.infra.deploymentHookSettings.hookCodeEnvName = $event.hookCodeEnvName;
            $scope.infra.deploymentHookSettings.runHooksAsUser = $event.runHooksAsUser;
            $scope.infra.deploymentHookSettings.maxHooksKernels = $event.maxHooksKernels;
        }

        $scope.addOrUpdateHook = function($event) {
            $scope.uiState.hooks.isPreDeployment = $event.isPreDeploymentHook;
            $scope.uiState.hooks.isNew = $event.isNew;
            $scope.uiState.hooks.selected = $event.hook;
        }

        $scope.cancelAddOrUpdateHook = function($event) {
            $scope.uiState.hooks.selected = undefined;
        }

        $scope.confirmAddOrUpdateHook = function($event) {
            if (!$scope.uiState.hooks.isNew && $scope.uiState.hooks.isPreDeployment !== $event.isPreDeploymentHook) {
                $scope.deleteHook({
                    isPreDeploymentHook: $scope.uiState.hooks.isPreDeployment,
                    hook: $scope.uiState.hooks.selected,
                });
                const hooks = $event.isPreDeploymentHook ? $scope.infra.deploymentHookSettings.preDeploymentHooks : $scope.infra.deploymentHookSettings.postDeploymentHooks;
                hooks.push($scope.uiState.hooks.selected);
            } else if ($scope.uiState.hooks.isNew) {
                const hooks = $event.isPreDeploymentHook ? $scope.infra.deploymentHookSettings.preDeploymentHooks : $scope.infra.deploymentHookSettings.postDeploymentHooks;
                hooks.push($scope.uiState.hooks.selected);
            }
            $scope.uiState.hooks.selected.enabled = $event.hook.enabled;
            $scope.uiState.hooks.selected.name = $event.hook.name;
            $scope.uiState.hooks.selected.description = $event.hook.description;
            $scope.uiState.hooks.selected.code = $event.hook.code;
            $scope.$broadcast("tabSelect", ($event.isPreDeploymentHook ? "pre" : "post") + "-deployment-hooks")
            if ($event.save && $scope.infraIsDirty()) {
                $scope.saveInfra(true);
            } else {
                $scope.updateExistingHookNames();
                $scope.uiState.hooks.selected = undefined;
            }
        }

        $scope.deleteHook = function($event) {
            const hooks = $event.isPreDeploymentHook ? $scope.infra.deploymentHookSettings.preDeploymentHooks : $scope.infra.deploymentHookSettings.postDeploymentHooks;
            const index = hooks.indexOf($event.hook);
            if (index > -1) {
                hooks.splice(index, 1);
            }
            $scope.updateExistingHookNames();
        }

        /********* Permissions *********/
        $controller('_DeployerPermissionsController', {$scope: $scope});

        // don't initialize until obj is available or else timing issues can occur
        const deregister = $scope.$watch("infra", function(nv, ov) {
            if (!nv) return;

            $scope.initPermissions($scope.infra, {
                deploy: true,
                admin: false,
                read: false
            }, false);

            deregister();
        }, false);

        $scope.$watch("infra.permissions", function(nv, ov) {
            if (!nv) return;
            $scope.onPermissionChange($scope.infra);
        }, true);

        $scope.$watch("infra.permissions", function(nv, ov) {
            if (!nv) return;
            $scope.onPermissionChange($scope.infra);
        }, false);

        refreshInfra();
        checkChangesBeforeLeaving($scope, $scope.infraIsDirty);
    });

    app.filter('governCheckPolicyIdToLabel', function(GOVERN_CHECK_POLICIES) {
        return function(governCheckPolicyId) {
            if (!governCheckPolicyId) {
                return;
            }
            const governCheckPolicy = GOVERN_CHECK_POLICIES.find(policy => policy.id === governCheckPolicyId);
            return governCheckPolicy ? governCheckPolicy.label : governCheckPolicyId;
        };
    });

    app.component("deploymentHooks", {
        bindings: {
            deploymentHookSettings: '<',
            infraType: '<',
            updateHooksSettings: '&',
            onAddOrUpdateHook: '&',
            onDeleteHook: '&',
        },
        template: `
<div class="flex w100">
    <div class="vertical-flex h100">
        <div>
            <div class="controls pull-right">
                <button data-qa-add-hook-btn class="btn btn--primary" ng-click="$ctrl.addHook()">Add new hook</button>
            </div>
            <span>Hooks are customized python code that executes automatically before or after every deployment made on this infrastructure.</span>

        </div>
        <div ng-if="$ctrl.infraType === 'MULTI_AUTOMATION_NODE'">
            <span class="text-warning">Each hook will be executed only once per deployment</span>
        </div>
        <div class="control-group mtop16">
            <label for="codeEnvName" class="control-label">Code-env</label>
            <div class="controls">
                <select dku-bs-select ng-model="$ctrl.deploymentHookSettings.hookCodeEnvName" ng-options="env.envName as env.envDesc for env in $ctrl.envNamesWithDescs" watch-model="$ctrl.envNamesWithDescs" ng-change="$ctrl.onSettingsChange()" data-live-search="true" id="codeEnvName"></select>
                <span class="help-inline">All hooks will be executed using this code environment</span>
            </div>
        </div>
        <div class="control-group mtop16">
            <label for="runAsUser" class="control-label">Run as user</label>
            <div class="controls">
                <ng2-typeahead [(value)]="$ctrl.deploymentHookSettings.runHooksAsUser" [suggestions]="$ctrl.usersForHookExecution" no-match-tooltip="No matching login" (value-change)="$ctrl.onSettingsChange()"
                               autocomplete-panel-width="600" placeholder="Deployer node login" style="display: inline-block"
                               disabled-if-message="!$ctrl.isAdmin ? 'Only an instance administrator can change the running user of hooks.' : null" ng-class="{'disabled-block': !$ctrl.isAdmin}">
                </ng2-typeahead>
                <span class="help-inline">All hooks will be executed impersonating this user. Leave empty to run hooks as the user requesting the deployment.</span>
            </div>
        </div>
        <div class="control-group mtop16">
            <label for="maxHooksKernels" class="control-label">Concurrent running hooks</label>
            <div class="controls">
                <input type="number" required min="1" ng-model="$ctrl.deploymentHookSettings.maxHooksKernels" ng-change="$ctrl.onSettingsChange()" class="editable-list__override-input-style" id="maxHooksKernels"/>
                <span class="help-inline">Limits the number of deployment hooks kernels running at the same time</span>
            </div>
        </div>
        <div class="alert alert-warning mtop16 mbot8" ng-if="$ctrl.deploymentHookSettings.maxHooksKernels > $ctrl.globalMaxKernels">
            <i class="icon-info-sign"></i> The global maximum number of hooks is {{$ctrl.globalMaxKernels}}, but you have set a higher value. The effective maximum concurrent kernels for this infrastructure will be capped by this global limit of {{globalMaxKernels}}.
        </div>
        <div class="mtop24 padding-none deployment-hooks-table">
            <tabs>
                <pane title="PRE-DEPLOYMENT HOOKS">
                    <deployment-hooks-list
                            hooks="$ctrl.deploymentHookSettings.preDeploymentHooks"
                            is-pre-deployment-hook="true"
                            show-hook-details="$ctrl.selectHook($event)"
                            delete-hook="$ctrl.deleteHook($event)"
                    ></deployment-hooks-list>
                </pane>
                <pane title="POST-DEPLOYMENT HOOKS">
                    <deployment-hooks-list
                            hooks="$ctrl.deploymentHookSettings.postDeploymentHooks"
                            is-pre-deployment-hook="false"
                            show-hook-details="$ctrl.selectHook($event)"
                            delete-hook="$ctrl.deleteHook($event)"
                    ></deployment-hooks-list>
                </pane>
            </tabs>
        </div>
    </div>
</div>
`,
        controller: function deploymentHooksController($rootScope, $scope, DataikuAPI, WT1, DataikuCloudService) {
            const $ctrl = this;

            this.$onInit = function() {
                WT1.event("deployer-hooks-open", {infraType: $ctrl.infraType});
                $ctrl.isPreDeploymentHook = true;
                if ($ctrl.infraType === "AUTOMATION_NODE") {
                    $ctrl.infraClientParameter = "automation_client, deploying_user, deployed_project_key, ";
                    $ctrl.infraClientDoc = ":param automation_client: an api client to connect to the automation node with the credentials defined in the infrastructure settings\n" +
                        "    :type automation_client: A :class:`dataikuapi.dssclient.DSSClient`\n" +
                        "    :param str deploying_user: identifier of the DSS user executing the deployment\n" +
                        "    :param str deployed_project_key: key on the automation node of the deployed project\n";
                } else if ($ctrl.infraType === "MULTI_AUTOMATION_NODE") {
                    $ctrl.infraClientParameter = "automation_clients, deploying_user, deployed_project_key, ";
                    $ctrl.infraClientDoc = ":param automation_clients: a dict of api clients to connect to the automation node with the credentials defined in the infrastructure settings, with node ids as keys\n" +
                        "    :type automation_clients: dict[str, :class:`dataikuapi.dssclient.DSSClient]`\n" +
                        "    :param str deploying_user: identifier of the DSS user executing the deployment\n" +
                        "    :param str deployed_project_key: key on the automation node of the deployed project\n";
                } else {
                    $ctrl.infraClientParameter = "";
                    $ctrl.infraClientDoc = "";
                }

                $ctrl.isAdmin = $rootScope.appConfig.admin;
                const builtInEnv = {
                    envName: undefined,
                    envDesc: "Use DSS builtin env"
                };
                $ctrl.envNamesWithDescs = [builtInEnv];
                $ctrl.usersForHookExecution = [];
                $ctrl.globalMaxKernels = $rootScope.appConfig.globalMaxDeploymentHooksKernels;
                DataikuAPI.codeenvs
                    .listNames("PYTHON")
                    .then(function(result) {
                        const newEnvs = result.data.map(function(env) {
                            return {
                                envName: env,
                                envDesc: env
                            };
                        });
                        $ctrl.envNamesWithDescs = [builtInEnv].concat(newEnvs);
                    })
                    .catch(setErrorInScope.bind($ctrl));
                DataikuAPI.security
                    .listUsers()
                    .then(function(result) {
                        let users = result.data;
                        if (DataikuCloudService.isDataikuCloud()) {
                            // Based on the current filters done in shared.js for runAs for scenarios
                            users = users.filter(user => !user.login.includes("saas+admin@dataiku.com"))    
                        }
                        $ctrl.usersForHookExecution = users.map(u => u.login);
                    })
                    .catch(setErrorInScope.bind($ctrl));
            }

            $scope.$on("paneSelected", function (_, pane) {
                $ctrl.isPreDeploymentHook = pane.slug === "pre-deployment-hooks";
            });

            $ctrl.onSettingsChange = function() {
                $ctrl.updateHooksSettings({
                    $event: {
                        hookCodeEnvName: $ctrl.deploymentHookSettings.hookCodeEnvName,
                        runHooksAsUser: $ctrl.deploymentHookSettings.runHooksAsUser,
                        maxHooksKernels: $ctrl.deploymentHookSettings.maxHooksKernels,
                    }
                });
            }

            $ctrl.addHook = function() {
                WT1.event("deployer-hooks-create");
                $ctrl.onAddOrUpdateHook({
                    $event: {
                        isPreDeploymentHook: $ctrl.isPreDeploymentHook,
                        isNew: true,
                        hook: {
                            name: undefined,
                            description: "",
                            type: "inline",
                            enabled: true,
                            code: `def execute(requesting_user, deployment_id, deployment_report, deployer_client, ${$ctrl.infraClientParameter}**kwargs):
    """
    Custom hook function.

    :param str requesting_user: identifier of the DSS user requesting the deployment
    :param str deployment_id: id of the deployment in progress
    :param deployment_report: status of the deployment and messages related if any.
                              In case of a pre deployment hook, this parameter will be None
                              In case of a post deployment hook, it will be a dictionary with:
                                - "status" for the deployment status, can be "SUCCESS", "WARNING" or "ERROR"
                                - "messages" for the list of messages related to the deployment, as strings
    :type deployment_report: dict or None
    :param deployer_client: an api client to connect to the deployer node with the credentials of the user running hooks
    :type deployer_client: A :class:\`dataikuapi.dssclient.DSSClient\`
    ${$ctrl.infraClientDoc}
    :returns: The execution status of the hook and a message as string.
        Use \`HookResult.success(message)\`, \`HookResult.warning(message)\` or \`HookResult.error(message)\`
    """
    return HookResult.success("Ok")
    `
                        }
                    }
                });
            }

            $ctrl.selectHook = function($event) {
                WT1.event("deployer-hooks-update");
                $ctrl.onAddOrUpdateHook({
                    $event: {
                        isPreDeploymentHook: $event.isPreDeploymentHook,
                        isNew: false,
                        hook: $event.hook
                    }
                })
            }

            $ctrl.deleteHook = function($event) {
                WT1.event("deployer-hooks-delete");
                $ctrl.onDeleteHook({
                    $event: $event
                });
            }
        }
    });

    app.component("deploymentHooksList", {
        bindings: {
            hooks: '<',
            isPreDeploymentHook: '<',
            showHookDetails: '&',
            deleteHook: '&',
        },
        template: `
<div class="mtop30">
    <table ng-if="$ctrl.hooks.length > 0" class="table table-bordered table-fixed table-hover" aria-label="Deployment hooks list">
        <thead>
        <tr>
            <th class="hook-row-drag">&nbsp;</th>
            <th class="hook-row-name">Name</th>
            <th class="hook-row-description">Description</th>
            <th class="hook-row-status">Active</th>
            <th class="hook-row-delete">&nbsp;</th>
        </tr>
        </thead>
        <tbody ui-sortable="$ctrl.sortableOptions" ng-model="$ctrl.hooks">
        <tr ng-repeat="hook in $ctrl.hooks">
            <td class="hook-row-drag">
                <i class="sort-handle editable-list__drag-icon dku-icon-dots-multiple-16"/>
            </td>
            <td class="hook-row-name mx-textellipsis">
                    <span>
                        <a ng-click="$ctrl.showDetails(hook)">
                            {{hook.name}}
                        </a>
                    </span>
            </td>
            <td class="hook-row-description mx-textellipsis">
                <span>{{hook.description}}</span>
            </td>
            <td class="hook-row-status">
                <label class="dku-toggle dku-toggle--no-margin">
                    <input type="checkbox" ng-model="hook.enabled"/>
                    <span/>
                </label>
            </td>
            <td class="hook-row-delete">
                <div class="tac">
                    <span ng-click="$ctrl.delete(hook)"><i class="dku-icon-trash-16"></i></span>
                </div>
            </td>
        </tr>
        </tbody>
    </table>
    <div ng-if="$ctrl.hooks.length === 0" class="info-message-inline-display">No {{$ctrl.isPreDeploymentHook ? "pre" : "post"}} deployment hook defined</div>
</div>`,
        controller: function deploymentHooksListController($scope, Dialogs) {
            const $ctrl = this;

            $ctrl.sortableOptions = {
                axis: 'y',
                cursor: 'move',
                cancel: '',
                handle: '.sort-handle',
            };

            $ctrl.delete = function(hook) {
                Dialogs.confirm($scope, "Confirm deletion", "Are you sure you want to delete the hook " + hook.name + "?").then(function() {
                    $ctrl.deleteHook({
                        $event: {
                            isPreDeploymentHook: $ctrl.isPreDeploymentHook,
                            hook: hook,
                        }
                    });
                }, function() {
                    // cancel case, do nothing
                });
            }

            $ctrl.showDetails = function(hook) {
                $ctrl.showHookDetails({
                    $event: {
                        isPreDeploymentHook: $ctrl.isPreDeploymentHook,
                        hook: hook,
                    }
                });
            }
        }
    });

    app.component("deploymentHooksDetails", {
        bindings: {
            isPreDeploymentHook: '<',
            hook: '<',
            isNewHook: '<',
            existingHookNames: '<',
            hookCodeEnvName: '<',
            hookType: '<',
            backToList: '&',
            addOrUpdateHook: '&',
        },
        template: `
<div class="small-lr-padding page-top-padding padbot16 h100">
    <form name="$ctrl.hookDetailsForm" global-keydown="{'ctrl-s meta-s':'!$ctrl.disableAddOrUpdateButton() && $ctrl.addOrUpdate(true)'}" class="dkuform-horizontal-tight-pdeps generic-white-box h100 vertical-flex">
        <div class="noflex">
            <div class="btn-group pull-right">
                <button class="btn btn--secondary mright4" ng-click="$ctrl.cancel()">Cancel</button>
                <span ng-if="!$ctrl.disableAddOrUpdateButton()">
                    <button data-qa-create-btn class="btn btn--primary" ng-click="$ctrl.addOrUpdate(true)">{{$ctrl.addOrUpdateLabel}} and save</button>
                    <button class="btn btn--primary btn--icon dropdown-toggle manual-caret mleft0" data-toggle="dropdown">
                        <i class="icon-caret-down" />
                    </button>
                    <ul class="dropdown-menu pull-right">
                        <li>
                            <a ng-click="$ctrl.addOrUpdate(false)">
                                Only {{$ctrl.addOrUpdateLabel}}, don't save infrastructure settings
                            </a>
                        </li>
                    </ul>
                </span>
                <button ng-if="$ctrl.disableAddOrUpdateButton()" disabled class="btn btn--primary">{{$ctrl.addOrUpdateLabel}} and save</button>
            </div>
            <div class="control-group">
                <label for="hookId" class="control-label">Name</label>
                <div class="controls">
                    <input data-qa-hook-name-input type="text" ng-model="$ctrl.name" class="editable-list__override-input-style" id="hookId" ng-required="true" auto-focus="true"/>
                </div>
                <div class="alert alert-warning mtop8 mbot0" ng-if="$ctrl.isNameDuplicate()">
                    <div>
                        <i class="icon-dku-warning"></i> Another hook with the same name already exists.
                    </div>
                </div>
            </div>
        </div>
        <div class="control-group width-fit-content">
            <label class="control-label dku-toggle dku-toggle--no-margin">Set as</label>
            <div class="controls horizontal-flex">
                <label for="preDeployment">
                    <input type="radio" name="preOrPost" ng-model="$ctrl.isPreDeployment" ng-value="true" id="preDeployment"/>
                    &nbsp;Pre-deployment
                </label>
                <label for="postDeployment" class="mleft32">
                    <input type="radio" name="preOrPost" ng-model="$ctrl.isPreDeployment" ng-value="false" id="postDeployment"/>
                    &nbsp;Post-deployment
                </label>
            </div>
        </div>
        <div class="control-group">
            <label for="hookStatus" class="control-label dku-toggle dku-toggle--no-margin">Active</label>
            <div class="controls">
                <label class="dku-toggle dku-toggle--no-margin">
                    <input type="checkbox" ng-model="$ctrl.enabled" id="hookStatus"/>
                    <span/>
                </label>
            </div>
        </div>
        <div class="control-group">
            <label for="hookDescription" class="control-label">Description</label>
            <div class="controls">
                <textarea ng-model="$ctrl.description" class="w50" rows="3" id="hookDescription"/>
                <span class="help-inline">Describe what this hook is about</span>
            </div>
        </div>
        <div code-snippet-editor
             code="$ctrl.code"
             sample-type="'python'"
             categories="$ctrl.codeCategories"
             save-category="$ctrl.codeSaveCategory"
             editor-options="$ctrl.editorOptions"
             displayed
             class="mtop16">
        </div>
        <div class="mbot16 dku-border-left dku-border-bottom dku-border-right pad8">
            <div class="horizontal-flex aic">
                <div class="btn-group">
                    <button class="btn btn-default"
                        title="Only checks that the code compiles. Does not run it."
                        ng-click="$ctrl.validate()">
                        Validate
                    </button>
                </div>
                <div class="mleft8">
                    <div ng-show="$ctrl.validationDone && $ctrl.validationError" class="text-error">
                        <i class="icon-warning-sign"/>&nbsp;{{$ctrl.validationError.message}}
                        <a ng-click="$ctrl.gotoLine($ctrl.codeMirror, $ctrl.validationError.line, $ctrl.validationError.column)">{{$ctrl.errorPositionText}}</a>
                    </div>
                    <div class="text-success" ng-show="$ctrl.validationDone && !$ctrl.validationError">
                        <i class="icon-ok"/>&nbsp;Validation successful
                        <em class="smallgrey"> Hit Shift-Enter to run.</em>
                    </div>
                    <em class="smallgrey" ng-show="!$ctrl.validationDone">Hint: Hit Ctrl-Enter to validate, Shift-Enter to run</em>
                    </div>
            </div>
        </div>

    </form>
</div>`,
        controller: function deploymentHooksListController($scope, Dialogs, WT1, DataikuAPI, CodeMirrorSettingService) {
            const $ctrl = this;

            this.$onChanges = function(changes) {
                if (changes) {
                    $ctrl.isPreDeployment = changes.isPreDeploymentHook.currentValue;
                    $ctrl.enabled = changes.hook.currentValue.enabled;
                    $ctrl.name = changes.hook.currentValue.name;
                    $ctrl.description = changes.hook.currentValue.description;
                    $ctrl.code = changes.hook.currentValue.code;
                    $ctrl.initialHook = {
                        isPreDeploymentHook: $ctrl.isPreDeployment,
                        enabled: $ctrl.enabled,
                        name: $ctrl.name,
                        description: $ctrl.description,
                        code: $ctrl.code,
                    };
                    $ctrl.addOrUpdateLabel = $ctrl.isNewHook ? "create" : "update";
                    $ctrl.codeCategories = [`py-${$ctrl.hookType}-deployer-hook`, `user-py-${$ctrl.hookType}-deployer-hook`];
                    $ctrl.codeSaveCategory = `user-py-${$ctrl.hookType}-deployer-hook`;
                    $ctrl.editorOptions = $ctrl.getEditorOptions();
                }
            }

            $ctrl.getEditorOptions = function() {
                const options = CodeMirrorSettingService.get("text/x-python", {
                    onLoad: function(cm) {
                        $ctrl.codeMirror = cm;
                    }
                });
                options.extraKeys['Meta-Enter'] = function(cm) {$ctrl.validate();};
                options.extraKeys['Ctrl-Enter'] = function(cm) {$ctrl.validate();};
                return options;
            }

            $ctrl.cancel = function() {
                WT1.event("deployer-hooks-edit-cancel");
                if ($ctrl.noChange()) {
                    $ctrl.backToList({
                        $event: {}
                    });
                } else {
                    Dialogs.confirm($scope, "Confirm cancellation", "Are you sure you want to drop your changes?").then(function() {
                        $ctrl.backToList({
                            $event: {}
                        });
                    }, function() {
                        // cancel case, do nothing
                    });
                }
            }

            $ctrl.addOrUpdate = function(save) {
                WT1.event("deployer-hooks-add-or-update", {saveInfra: save, preHook: $ctrl.isPreDeployment});
                $ctrl.addOrUpdateHook({
                    $event: {
                        isPreDeploymentHook: $ctrl.isPreDeployment,
                        save: save,
                        hook: {
                            enabled: $ctrl.enabled,
                            name: $ctrl.name,
                            description: $ctrl.description,
                            code: $ctrl.code
                        }
                    }
                })
            }

            $ctrl.disableAddOrUpdateButton = function() {
                return $ctrl.noChange() || $ctrl.isInvalid();
            }

            $ctrl.noChange = function() {
                const currentHook = {
                    isPreDeploymentHook: $ctrl.isPreDeployment,
                    enabled: $ctrl.enabled,
                    name: $ctrl.name,
                    description: $ctrl.description,
                    code: $ctrl.code,
                };
                return angular.equals(currentHook, $ctrl.initialHook);
            }

            $ctrl.isInvalid = function() {
                return $ctrl.hookDetailsForm.$invalid;
            }

            $ctrl.isNameDuplicate = function() {
                return $ctrl.name && $ctrl.name !== $ctrl.initialHook.name && $ctrl.existingHookNames.includes($ctrl.name);
            }

            $ctrl.validate = function() {
                DataikuAPI.deployer.hooks.checkCodeCompiles($ctrl.code, $ctrl.hookCodeEnvName)
                    .success(function(data) {
                        $ctrl.validationDone = true;
                        $ctrl.validationError = data.messages.length > 0 ? data.messages[0] : undefined;
                        $ctrl.errorPositionText = $ctrl.validationError ? `(at line ${$ctrl.validationError.line}, character ${$ctrl.validationError.column})` : undefined;
                    }).error(setErrorInScope.bind($scope));
            }

            $ctrl.gotoLine = function(cm, line, column) {
                if (cm && line > 0) {
                    const pos = { ch: column, line: line - 1 };
                    cm.scrollIntoView(pos);
                    cm.setCursor(pos);
                    cm.focus();
                }
            }
        }
    });

    app.component("deploymentHooksIssues", {
        bindings: {
            messages: '<',
            reset: '&',
        },
        template: `
<div class="generic-shadow-box mbot20 noflex">
    <div class="alert {{$ctrl.messages.maxSeverity|severityAlertClass}} df jcsb">
        <div class="df">
            <i class="{{$ctrl.messages.maxSeverity|deploymentHooksIssuesSeverityIcon}}"/>
            <h4 class="padleft16">Issues have been detected in some deployment hook settings.</h4>
        </div>
        <div>
            <a ng-click="$ctrl.resetHookErrors()">
                <i class="dku-icon-dismiss-16"></i>
            </a>
        </div>
    </div>
    <div info-messages-raw-list="$ctrl.messages"></div>
</div>`,
        controller: function deploymentHooksIssuesController() {
            const $ctrl = this;

            $ctrl.resetHookErrors = function() {
                $ctrl.reset({
                    $event: {}
                });
            }
        }
    });

    app.filter("deploymentHooksIssuesSeverityIcon", function() {
        const dict = {
            SUCCESS: 'dku-icon-info-circle-outline-16',
            INFO: 'dku-icon-info-circle-outline-16',
            WARNING: 'dku-icon-warning-fill-16',
            ERROR: 'dku-icon-dismiss-circle-fill-16'
        };
        return function(severity) {
            if (!severity) {
                return dict.SUCCESS;
            }
            return dict[severity] || '';
        };
    });

    app.component("deploymentHooksStates", {
        bindings: {
            hookExecutionStatus: '<',
            maxWidthPx: '<',
        },
        template: `
<div data-qa-deployment-hooks-status-container class="mtop8">
    <div ng-if="$ctrl.hookExecutionStatus.preHooks > 0">
        Executing {{$ctrl.hookExecutionStatus.preHooks}} {{'pre deployment hook' | plurify : $ctrl.hookExecutionStatus.preHooks}}
        <div ng-repeat="hookResult in $ctrl.hookExecutionStatus.preHookResults" class="horizontal-flex">
            <i class="{{hookResult.status | deploymentHookStatusToIcon}}"></i>
            <div class="vertical-flex padleft4">
                <span class="ellipsed">{{$ctrl.deploymentHookText(hookResult)}}</span>
                <show-ellipsed-text-and-copy ng-if="hookResult.message" class="height-fit-content horizontal-flex"
                    text="hookResult.message"
                    max-width-px="$ctrl.maxWidthPx"></show-ellipsed-text-and-copy>
            </div>
        </div>
    </div>
    <div ng-if="$ctrl.hookExecutionStatus.preHooks == 0">No pre deployment hook defined or enabled for this infrastructure</div>
    <div ng-if="$ctrl.hookExecutionStatus.deploymentStatus !== 'NOT_STARTED' && $ctrl.hookExecutionStatus.hasHooks" ng-class="{
            'text-warning': $ctrl.hookExecutionStatus.deploymentStatus === 'SKIPPED',
            'text-error': $ctrl.hookExecutionStatus.deploymentStatus === 'FAILED'
        }" class="mtop4 mbot4">{{$ctrl.hookExecutionStatus.deploymentStatusMessage}}</div>
    <div ng-if="$ctrl.hookExecutionStatus.postHooks > 0">
        Executing {{$ctrl.hookExecutionStatus.postHooks}} {{'post deployment hook' | plurify : $ctrl.hookExecutionStatus.postHooks}}
        <div ng-repeat="hookResult in $ctrl.hookExecutionStatus.postHookResults" class="horizontal-flex">
            <i class="{{hookResult.status | deploymentHookStatusToIcon}}"></i>
            <div class="vertical-flex padleft4">
                <span class="ellipsed">{{$ctrl.deploymentHookText(hookResult)}}</span>
                <show-ellipsed-text-and-copy ng-if="hookResult.message" class="height-fit-content horizontal-flex"
                    text="hookResult.message"
                    max-width-px="$ctrl.maxWidthPx"></show-ellipsed-text-and-copy>
            </div>
        </div>
    </div>
    <div ng-if="$ctrl.hookExecutionStatus.postHooks == 0">No post deployment hook defined or enabled for this infrastructure</div>
</div>`,
        controller: function deploymentHooksStatesController() {
            const $ctrl = this;

            $ctrl.deploymentHookText = function(hookResult) {
                const withMessage = hookResult.message ? ", with message:" : ""
                switch (hookResult.status) {
                    case "SUCCESS": return `Hook "${hookResult.name}" ended successfully${withMessage}`;
                    case "WARNING": return `Hook "${hookResult.name}" returned a warning${withMessage}`;
                    case "ERROR": return `Hook "${hookResult.name}" returned an error${withMessage}`;
                    case "NOT_FINISHED": return `Executing hook "${hookResult.name}"...`;
                    case "INTERRUPTED": return `Hook "${hookResult.name}" was interrupted`;
                    case "TIMED_OUT": return `Hook "${hookResult.name}" timed out`;
                    default: return `Unknown status "${hookResult.status}" for hook "${hookResult.name}"`;
                }
            }
        }
    });

    app.filter('deploymentHookStatusToIcon', function() {
        const dict = {
            SUCCESS: 'dku-icon-checkmark-circle-fill-12 text-success dib',
            WARNING: 'dku-icon-warning-fill-12 text-warning',
            ERROR: 'dku-icon-error-circle-outline-12 text-error',
            NOT_FINISHED: 'dku-icon-arrow-clockwise-dashes-12 icon-spin height-fit-content mtop4',
            INTERRUPTED: 'dku-icon-cancel-12 text-error',
            TIMED_OUT: 'dku-icon-clock-12 text-error',
        };
        return function(status) {
            if (!status) {
                return '';
            }
            return dict[status] || '';
        };
    });
})();
