(function() {
    'use strict';

    const app = angular.module('dataiku.projectdeployer');

    app.controller('ProjectDeployerInfrasListController', function($scope, $controller, DataikuAPI) {
        $controller('_DeployerInfrasListController', {$scope: $scope});

        if ($scope.isFeatureLocked) return;

        $scope.uiState = $scope.uiState || {};

        $scope.filterInfra = function(infraStatus) {
            if (!$scope.uiState.query) return true;
            const query = $scope.uiState.query.toLowerCase();
            return infraStatus.infraBasicInfo.id.toLowerCase().includes(query)
                || infraStatus.infraBasicInfo.stage.toLowerCase().includes(query)
                || (infraStatus.infraBasicInfo.automationNodeExternalUrl && infraStatus.infraBasicInfo.automationNodeExternalUrl.toLowerCase().includes(query))
                || (infraStatus.infraBasicInfo.automationNodes && infraStatus.infraBasicInfo.automationNodes.find(automationNode => automationNode.automationNodeExternalUrl.toLowerCase().includes(query)));
        }

        $scope.$watch("infraStatusList", function(nv, ov) {
            if (nv) {
                nv.forEach(function(infraStatus) {
                    const infraId = infraStatus.infraBasicInfo.id;
                    DataikuAPI.projectdeployer.infras.checkStatus(infraId)
                        .success(function(healthStatus) {
                            infraStatus.infraHealthError = healthStatus.messages.find(function(msg) {
                                return msg.severity === healthStatus.maxSeverity;
                            }) || {};
                        })
                        .error(setErrorInScope.bind($scope));
                });
            }
        });
    });

    app.controller('ProjectDeployerInfraController', function($scope, $controller) {
        $controller('_DeployerInfraController', {$scope: $scope});
    });

    app.controller('ProjectDeployerInfraSetupModalController', function($rootScope, $scope, $controller) {
        $controller('_DeployerInfraSetupModalController', {$scope: $scope});
        $scope.multiNodeInfrasAllowed = $rootScope.appConfig.allowMultiNodeProjectDeployerInfrastructures;
        $scope.newInfra.type = 'AUTOMATION_NODE';
    });

    app.controller('ProjectDeployerInfraStatusController', function($scope, $controller, DataikuAPI, ProjectDeployerAsyncHeavyStatusLoader, DeployerDeploymentTileService) {
        $controller('_DeployerInfraStatusController', {$scope: $scope});

        $scope.$watch("infraStatus", function(nv) {
            if (nv) {
                DataikuAPI.projectdeployer.infras.checkStatus($scope.infraStatus.infraBasicInfo.id)
                    .success(function(healthStatus) {
                        if (healthStatus.anyMessage) {
                            $scope.infraHealthErrorMessage = healthStatus.messages.find(function(msg) {
                                return msg.severity === healthStatus.maxSeverity;
                            }).message;
                        }
                    })
                    .error(setErrorInScope.bind($scope));

                DataikuAPI.projectdeployer.publishedProjects.listBasicInfo()
                .success(function(projectBasicInfoList) {
                    const projectBasicInfoMap = projectBasicInfoList.projects.reduce((obj, basicInfo) => ({ ...obj, [basicInfo.id]: basicInfo }), {});
                    $scope.pseudoLightStatusList = $scope.infraStatus.deployments.map(function(deployment) {
                        return {
                            deploymentBasicInfo: deployment,
                            projectBasicInfo: projectBasicInfoMap[deployment.publishedProjectKey]
                        };
                    });
                });

                const infraStatusList = [{
                    infraBasicInfo: $scope.infraStatus.infraBasicInfo,
                    deployments: $scope.infraStatus.deployments
                }];
                const heavyStatusByDeploymentId = {};
                let loader = ProjectDeployerAsyncHeavyStatusLoader.newLoader(infraStatusList, heavyStatusByDeploymentId);
                loader.loadHeavyStatus();

                const deregister = $scope.$watch(function(){
                    return loader.stillRefreshing();
                }, function(nv, ov) {
                    if (nv || ov === nv) return;
                    $scope.healthMap = DeployerDeploymentTileService.getDeploymentHealthMap($scope.infraStatus.deployments, heavyStatusByDeploymentId);
                    deregister();
                });

                $scope.$on('$destroy', function() {
                    loader && loader.stopLoading();
                });
            }
        });
    });

    app.controller('ProjectDeployerInfraHistoryController', function($scope, $controller) {
        $controller('_DeployerInfraHistoryController', {$scope: $scope});
    });

    app.controller('ProjectDeployerInfraSettingsController', function($scope, $controller, $state, DataikuAPI, Logger) {
        $controller('_DeployerInfraSettingsController', {$scope: $scope});
        $scope.infraUsersLogin = [];
        $scope.automationProjectKeys = [];
        $scope.propagationOptions= [
            { code: "NONE", label:"None" },
            { code: "READ_ONLY", label:"Limited" },
            { code: "ALL", label:"Full" },
        ]
        $scope.$watch("infra", (nv) => {
            if (nv) {
                DataikuAPI.projectdeployer.infras.listUserLogins(nv.id).success(function(data) {
                    let users = data;
                    if ($scope.isCloud) {
                        // Based on the current filters done in shared.js for runAs for scenarios
                        users = users.filter(user => !user.includes("saas+admin@dataiku.com"))
                    }
                    $scope.infraUsersLogin = users;
                }).error(function (error) {
                    // If an error prevents from retrieving the list of users, just log it as no need to alarm the user.
                    // This can be normal when the automation node URL or API Key are incorrect or the automation node is down.
                    Logger.error(error.message);
                });
                fetchAutomationNodesFromNodesDirectory();
                fetchOtherAutomationNodeUrls();
            } else {
                $scope.infraUsersLogin = [];
            }
        });
        $scope.$watch("infraStatus", (nv) => {
            if (nv) {
                $scope.automationProjectKeys = nv.deployments.map(depl => depl.deployedProjectKey?depl.deployedProjectKey:depl.publishedProjectKey);
            } else {
                $scope.automationProjectKeys = [];
            }
            $scope.updateUnsetDeployAsAutomationProjectKeys();
            $scope.updateUnsetRunAsAutomationProjectKeys();
        });

        $scope.updateUnsetDeployAsAutomationProjectKeys = function() {
            let newArray;
            if ($scope.infra && $scope.infra.perProjectDeployAsUser) {
                newArray = _.difference($scope.automationProjectKeys, $scope.infra.perProjectDeployAsUser.map(pp => pp.key));
            } else {
                newArray = $scope.automationProjectKeys;
            }
            // we must change the reference iff array content changes, or we may enter in an infinite notification / update loop
            if (!_.isEqual(newArray, $scope.unsetDeployAsAutomationProjectKeys)) {
                $scope.unsetDeployAsAutomationProjectKeys = newArray;
            }
        }

        $scope.updateUnsetRunAsAutomationProjectKeys = function() {
            let newArray;
            if ($scope.infra && $scope.infra.perProjectRunAsUser) {
                newArray = _.difference($scope.automationProjectKeys, $scope.infra.perProjectRunAsUser.map(pp => pp.key));
            } else {
                newArray = $scope.automationProjectKeys;
            }
            // we must change the reference iff array content changes, or we may enter in an infinite notification / update loop
            if (!_.isEqual(newArray, $scope.unsetRunAsAutomationProjectKeys)) {
                $scope.unsetRunAsAutomationProjectKeys = newArray;
            }
        }

        $scope.getUnsetDeployAsAutomationProjectKeys = function() {
            return $scope.unsetDeployAsAutomationProjectKeys;
        }

        $scope.getUnsetRunAsAutomationProjectKeys = function() {
            return $scope.unsetRunAsAutomationProjectKeys;
        }

        function isMultiInfraAndNodesDirectoryEnabled() {
            return $scope.infra.type === 'MULTI_AUTOMATION_NODE' && $scope.deployerAndNodesDirectoryEnabled;
        }

        function fetchAutomationNodesFromNodesDirectory() {
            if (isMultiInfraAndNodesDirectoryEnabled()) {
                DataikuAPI.projectdeployer.infras.getAutomationNodesFromNodesDirectory($state.params.infraId)
                    .then(result => {
                        $scope.uiState.automationNodesFromNodesDirectory = result.data || [];
                        $scope.refreshNodesFromNodesDirectoryNotInInfra();
                    })
                    .catch(setErrorInScope.bind($scope));
            }
        }

        $scope.addNodeFromNodesDirectory = function() {
            const node = $scope.uiState.selectedNodeToAdd;
            if (node) {
                const automationNodes = [...$scope.infra.automationNodes];
                automationNodes.push({...node});
                $scope.infra.automationNodes = automationNodes;
                $scope.refreshNodesFromNodesDirectoryNotInInfra();
            }
        }

        $scope.refreshNodesFromNodesDirectoryNotInInfra = function() {
            if (isMultiInfraAndNodesDirectoryEnabled()) {
                const nodeIdsInInfra = $scope.infra.automationNodes.filter(a => a.nodeId).map(a => a.nodeId);
                $scope.uiState.automationNodesFromNodesDirectoryNotInInfra = $scope.uiState.automationNodesFromNodesDirectory.filter(n => !nodeIdsInInfra.includes(n.nodeId));
            }
        }

        function fetchConnections() {
            DataikuAPI.projectdeployer.infras.listConnectionsNames($state.params.infraId, null).success(connections => {
                $scope.uiState.availableConnections = connections;
            }); // no .error() to avoid red banner if automation node is down
            DataikuAPI.connections.getNames("all").success(connections => {
                $scope.uiState.connectionNames = connections;
            }).error(setErrorInScope.bind($scope));
        };

        function fetchContainerExecConfigNames() {
            DataikuAPI.projectdeployer.infras.listContainerExecNames($state.params.infraId, null).success(names => {
                $scope.uiState.availableContainerExecNames = names;
            }); // no .error() to avoid red banner if automation node is down
            DataikuAPI.containers.listNames(null, null).success(containers => {
                $scope.uiState.localContainerExecNames = containers;
            }).error(setErrorInScope.bind($scope));
        };

        fetchConnections();
        fetchContainerExecConfigNames();


        function fetchOtherAutomationNodeUrls() {
            DataikuAPI.projectdeployer.infras.listOtherAutomationNodeUrls($state.params.infraId)
                .then(result => {
                    $scope.uiState.automationNodeUrlToInfraIds = result.data;
                })
        }

        $scope.getInfoMessageAboutOtherAutomationNodeUrls = function() {
            if (!$scope.uiState.automationNodeUrlToInfraIds) {
                return "";
            }

            if ($scope.infra.type === 'MULTI_AUTOMATION_NODE') {
                if (!$scope.infra.automationNodes) {
                    return "";
                }
                const infoMessages = [];

                $scope.infra.automationNodes.forEach(automationNode => {
                    if (automationNode.automationNodeUrl in $scope.uiState.automationNodeUrlToInfraIds) {
                        const plural = $scope.uiState.automationNodeUrlToInfraIds[automationNode.automationNodeUrl].length > 1;
                        infoMessages.push(`The automation node URL ${automationNode.automationNodeUrl} is also used on infrastructure${plural? 's': ''}:
                         ${$scope.uiState.automationNodeUrlToInfraIds[automationNode.automationNodeUrl].join(', ')}`);
                    }
                })
                return infoMessages.join("<br>");
            }
            else if ($scope.infra.type === 'AUTOMATION_NODE') {
                if (!$scope.infra.automationNodeUrl || ! ($scope.infra.automationNodeUrl in $scope.uiState.automationNodeUrlToInfraIds)) {
                    return "";
                }
                const plural = $scope.uiState.automationNodeUrlToInfraIds[$scope.infra.automationNodeUrl].length > 1;
                return `The automation node URL ${$scope.infra.automationNodeUrl} is also used on infrastructure${plural ? 's' : ''}:
                 ${$scope.uiState.automationNodeUrlToInfraIds[$scope.infra.automationNodeUrl].join(', ')}`
            }
        }
    });

    app.filter('webappTypeToIconDeployer', function(TypeMappingService, TYPE_MAPPING) {
        return (type, size) => TypeMappingService.iconProcessor(type,  TYPE_MAPPING.WEBAPPS_TYPES, undefined, () => 'icon-picture', size);
    });

    app.filter('webappTypeToColorDeployer', function(TYPE_MAPPING) {
        return (type) => TYPE_MAPPING.WEBAPPS_TYPES[type.toLowerCase()] ? 'notebook' : 'flow';
    });

    app.filter('webappTypeToTextDeployer', function() {
        return (type) => {
            const text = {
                'WEB_APP': 'Webapp',
                'BOKEH': 'Bokeh',
                'DASH': 'Dash',
                'SHINY': 'Shiny',
                'STANDARD': 'Standard code webapp',
                'CODE_STUDIO_AS_WEBAPP': 'Code studio as webapp'
            }[type];

            const visualWebAppTypeText = 'Visual Webapp';

            return text || visualWebAppTypeText;
        }
    });

    app.filter('webappStatusToIconDeployer', function() {
        return (status, size) => {
            const icon = {
                'STOPPED': 'dku-icon-dismiss',
                'BACKEND_READY': 'dku-icon-checkmark',
                'BACKEND_NOT_READY': 'dku-icon-dismiss',
                'BACKEND_NOT_ENABLED': 'dku-icon-line'
            }[status];

            const questionIcon = 'dku-icon-question';

            return `${icon || questionIcon}-${size}`;
        };
    });

    app.filter('webappStatusToTextDeployer', function() {
        return (status) => ({
            'STOPPED': 'Stopped',
            'BACKEND_READY': 'Running',
            'BACKEND_NOT_READY': 'Engine start is in progress',
            'BACKEND_NOT_ENABLED': 'Not enabled'
        })[status] || '';
    });
})();
