(function(){
    'use strict';

    var app = angular.module('dataiku.directives.widgets', ['dataiku.filters', 'dataiku.services', 'ui.keypress', 'dataiku.common.lists']);

    /* "Generic" widgets */

    app.directive("plusIcon", function(){
        return {
            restrict : 'A',
            replace:true,
            template : '<span class="dku-plus-icon-16">+</span>'
        }
    });
    app.directive("timesIcon", function(){
        return {
            restrict : 'A',
            replace:true,
            template : '<span style="font-size:1.3em; vertical-align: top">&times</span>'
        }
    });

    const addStarComponentBehaviour = ($ctrl, InterestWording) => {
        const toggle = (nextStatus) => {
            $ctrl.onToggle({ nextStatus });
        };

        $ctrl.isStarring = () => $ctrl.status;
        $ctrl.toggleStar = () => toggle(true);
        $ctrl.toggleUnstar = () => toggle(false);

        const { labels, tooltips } = InterestWording;
        $ctrl.labels = { ...labels };
        $ctrl.tooltips = { ...tooltips };
    };

    app.component('starInterest', {
        templateUrl: '/templates/widgets/star-interest.html',
        bindings: {
            status: '<',
            onToggle: '&',
            tooltipPosition: '@?',
        },
        controller: function(InterestWording) {
            const $ctrl = this;
            addStarComponentBehaviour($ctrl, InterestWording);
        },
    });

    app.component('starButton', {
        templateUrl: '/templates/widgets/star-button.html',
        bindings: {
            status: '<',
            onToggle: '&',
            nbStarred: '<',
            onShowUsersWithStar: '&',
            disabled: '<?',
        },
        controller: function(InterestWording) {
            const $ctrl = this;
            addStarComponentBehaviour($ctrl, InterestWording);

            $ctrl.isDisabled = () => !!($ctrl.disabled);
        },
    });

    const addWatchComponentBehaviour = ($ctrl, InterestWording, WatchInterestState) => {
        const toggle = (nextStatus) => {
            $ctrl.onToggle({ nextStatus });
        };

        const { values: { YES, ENO }, isWatching } = WatchInterestState;
        $ctrl.isWatching = () => isWatching($ctrl.status);
        $ctrl.toggleWatch = () => toggle(YES);
        $ctrl.toggleUnwatch = () => toggle(ENO);

        const { labels, tooltips } = InterestWording;
        $ctrl.labels = { ...labels };
        $ctrl.tooltips = { ...tooltips };
    };

    app.component('watchInterest', {
        templateUrl: '/templates/widgets/watch-interest.html',
        bindings: {
            status: '<',
            onToggle: '&',
            tooltipPosition: '@?',
        },
        controller: function(InterestWording, WatchInterestState) {
            const $ctrl = this;
            addWatchComponentBehaviour($ctrl, InterestWording, WatchInterestState);
        },
    });

    app.component('watchButton', {
        templateUrl: '/templates/widgets/watch-button.html',
        bindings: {
            status: '<',
            onToggle: '&',
            nbWatching: '<',
            onShowWatchingUsers: '&',
        },
        controller: function(InterestWording, WatchInterestState) {
            const $ctrl = this;
            addWatchComponentBehaviour($ctrl, InterestWording, WatchInterestState);
        },
    });

    app.component('multiRegionBar', {
        bindings : {
            regions: '<' // regions should be a list with objects like {size: [int] length of the region, color: [str] color code of the region}
        },
        templateUrl: "/templates/directives/multi-region-bar.html",
        controller: function multiRegionBarController() {
            const $ctrl = this;
            let updateTotal = function () {
                $ctrl.total = $ctrl.regions.reduce((acc, region) => acc + region.size, 0);
            }

            $ctrl.onInit = function() {
                updateTotal();
            }
            $ctrl.$onChanges = function () {
                updateTotal();
            }
        }
    })

    app.directive("apiErrorAlert", function ($rootScope, $injector, ActivityIndicator, CreateModalFromTemplate, OpalsService, ClipboardUtils) {
        // FM doesn't have DataikuAPI or RequestCenterService
        let DataikuAPI = $injector.has('DataikuAPI') ? $injector.get('DataikuAPI') : null;
        let RequestCenterService = $injector.has('RequestCenterService') ? $injector.get('RequestCenterService') : null;

        return {
            restrict : 'A',
            scope: {
                apiErrorAlert : '=',
                closable : '=',
                errorFoldable : '@',
                canBeUnexpected : '=?'
            },
            link : function($scope) {
                if ($scope.canBeUnexpected === undefined) {
                    $scope.canBeUnexpected = true;
                }
                $scope.options = {
                    canBeUnexpected : $scope.canBeUnexpected,
                    closable : $scope.closable,
                    errorFoldable : $scope.errorFoldable
                }
                $scope.open = true;
                $scope.reset = function() {
                    if ($scope.apiErrorAlert) {
                        $scope.apiErrorAlert.httpCode  = null;
                        $scope.apiErrorAlert.errorType = null;
                    }
                }

                $scope.openHelpCenterDoc = function(event, href) {
                    // if the help center is not activated, just let the event open a new tab by default
                    // also check if there is no open modal as it prevent the use of opals
                    OpalsService.isEnabled().then(function(isOpalsEnabled) {
                        if (isOpalsEnabled && $('.modal-container').length === 0) {
                            OpalsService.navigateToAndShowDrawer(OpalsService.PAGES.EMBEDDED_BROWSER, { href });
                        } else {
                            window.open(href, '_blank');
                        }
                    });
                    event.preventDefault();
                }
                $scope.isUnauthorizedProfileError = $scope.apiErrorAlert && $scope.apiErrorAlert.code && $scope.apiErrorAlert.code==="ERR_USER_ACTION_FORBIDDEN_BY_PROFILE";
                $scope.isCredentialError = $scope.apiErrorAlert && $scope.apiErrorAlert.code && ($scope.apiErrorAlert.code==="ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL" ||$scope.apiErrorAlert.code==="ERR_CONNECTION_NO_CREDENTIALS");

                $scope.canCopyErrorToClipboard = function() {
                    return $scope.apiErrorAlert
                        // any exception that uses smart-log-tail should show the copy error button
                        && ['com.dataiku.dip.io.CustomPythonKernelException',
                            'com.dataiku.dip.exceptions.ExternalProcessFailedException',
                            'com.dataiku.dip.io.SocketBlockLinkIOException',
                            'com.dataiku.dip.io.SocketBlockLinkKernelException'].includes($scope.apiErrorAlert.errorType)
                        && $scope.apiErrorAlert.logTail
                        && $scope.apiErrorAlert.logTail.lines;
                }
                $scope.copyErrorToClipboard = function() {
                    ClipboardUtils.copyToClipboard($scope.apiErrorAlert.logTail.lines.join('\n'));
                }
            },
            templateUrl: '/templates/api-error-alert.html',
            controller: function($scope) {
                this.$onInit = function() {
                    if ($scope.apiErrorAlert && $scope.apiErrorAlert.code && $scope.apiErrorAlert.code === "ERR_USER_ACTION_FORBIDDEN_BY_PROFILE") {
                        $rootScope.sendWT1RequestUpgradeProfileShow();
                    }
                }
            }
        }
    });

    app.directive("sidekickAlert", function() {
        return {
            restrict : "E",
            transclude: true,
            templateUrl : "/templates/sidekick-alert.html"
        }
    });

    app.filter("detailedMessageOrMessage", function(){
        return function(input) {
            if (!input) return "";
            return input.detailedMessage || input.message;
        }
    });

    /**
        /!\ Note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.
    **/
    app.factory("disableElement", function(getBootstrapTooltipPlacement) {
        return function(element, disabled, message, position) {
            if (disabled === true) {
                element.addClass("disabled");
                element.prop("disabled", "disabled");
                element.css("position", "relative");
                element.css("pointer-events", "auto");
                var div = $('<div>').addClass("fh disabled-if-overlay").attr("title", message).appendTo(element);
                div.on('click', function () { return false; });
                if (message && message.length) {
                    div.tooltip({container: "body", placement: getBootstrapTooltipPlacement(position)})
                        .on('hide.bs.modal', function(e) {
                            // Avoid wrong behavior with stacked modals: see sc-86865
                            e.stopPropagation();
                        });
                }
            } else if (disabled === false) {
                element.removeClass("disabled");
                element.css("pointer-events", null);
                element.prop("disabled", null);
                element.find('.disabled-if-overlay').tooltip('destroy').remove();
            }
        }
    });

    /*
        /!\ Do not use on a HTML element that also uses a `title` attribute, as two overlapping tooltips would appear.
        Instead you should handle the tooltip content in all cases (disabled + your other cases) in the `title` attribute. Do not forget to escape the ' in the
        disabled if RO message: "You don\'t have write permissions for this project".
        Also note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.
    */
    app.directive("disabledIfRo", function(disableElement, translate){
        return {
            restrict : 'A',
            link : function(scope, element) {
                scope.$watch("!canWriteProject()", function(nv) {
                    if (nv === undefined) return;
                    return disableElement(element, nv, translate("PROJECT.PERMISSIONS.WRITE_ERROR", "You don't have write permissions for this project"));
                });
            }
        }
    });

    app.directive("disabledIfProjectFolderRo", function($rootScope, disableElement, translate) {
        return {
            restrict : 'A',
            link : function(scope, element) {
                scope.$watch(function() {
                    return !$rootScope.isDSSAdmin() && !$rootScope.canWriteInProjectFolder();
                }, function(nv) {
                    if (nv === undefined) return;
                    return disableElement(element, nv, translate("PROJECT.FOLDER.PERMISSIONS.WRITE_ERROR", "You don't have write contents permissions on this folder"));
                });
            }
        }
    });

    /*
        /!\ If you use the disabledMessage attribute, do not use on a HTML element that also uses a `title` attribute, as two overlapping tooltips would appear.
        Instead you should handle the tooltip content in all cases (disabled + your other cases) in the `title` attribute.
        Also note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.
    */
    app.directive("disabledIf", function(disableElement){
        return {
            restrict : 'A',
            link: function(scope, element, attrs) {
                scope.$watch(attrs.disabledIf, function(nv) {
                    return disableElement(element, nv, attrs.disabledMessage, attrs.disabledPosition);
                });
            }
        }
    });

    /*
        /!\ Do not use on a HTML element that also uses a 'title' attribute, as two overlapping tooltips would appear.
        Instead you should handle the tooltip content in all cases (disabled + your other cases) in the `title` attribute.
        Also note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.

        Sets element as disabled if disabledIfMessage is a non-empty string
        and displays that string as the tooltip
    */
    app.directive("disabledIfMessage", function(disableElement){
        return {
            restrict : 'A',
            scope : {
                disabledIfMessage: '='
            },
            link: function(scope, element, attrs) {
                scope.$watch('disabledIfMessage', function(nv) {
                    return disableElement(element, !!nv, scope.disabledIfMessage, attrs.disabledPosition);
                });
            }
        }
    });

    app.directive("disabledBlockIfRo", function(){
        return {
            restrict : 'A',
            link : function(scope, element) {
                scope.$watch("canWriteProject()", function(nv) {
                    if (nv === false) {
                        element.addClass("disabled-block");
                    } else if( nv === true) {
                        element.removeClass("disabled-block");
                    }
                });
            }
        }
    });

    /* similar to ng-show but uses CSS visibility rather than display property (no movement in the page) */
    app.directive('visibleIf', function() {
        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                var toggle = function (show){
                    $(element).css('visibility', show ? 'visible': 'hidden');
                };
                toggle(scope.$eval(attrs.visibleIf));
                scope.$watch(attrs.visibleIf, toggle);
            }
        };
    });

    /* Directive that can be used to forward API errors to an blockApiError in scope */
    app.directive("apiErrorContext", function() {
        return {
            controller: function ($scope) {
                this.setError = setErrorInScope.bind($scope);
            }
        }
    });

    /* API error that displays as an alert block */
    app.directive('blockApiError', function() {
        return {
            templateUrl: '/templates/block-api-error.html',
            replace: false,
            restrict: 'ECA',
            link: function(scope) {
              // can be used by children to report their error.
              scope.setError = setErrorInScope.bind(scope);
            }
        };
    });

    /* API warning InfoMessages that displays as an alert block */
    app.component('blockApiWarning', {
        templateUrl: '/templates/block-api-warning.html',
        bindings: {
            warningMessages: '='
        },
        controller: function() {
            this.resetWarnings = () => {
                if (this.warningMessages)
                    this.warningMessages.messages = null;
            };
        }
    });

    app.directive("tlUser", function($rootScope){
        return {
            template : `
                <span class="who" ng-if="item.user" toggle="tooltip" title="{{item.details.userDisplayName || item.user}} (@{{item.user}})">
                    <span ng-if="item.user === 'no:auth'" ng-bind="item.user | niceLogin"></span>
                    <a ng-if="item.user !== 'no:auth'" href="/profile/{{item.user}}/">
                        {{item.user == rootScope.appConfig.user.login ? "You" : (item.details.userDisplayName ? item.details.userDisplayName : item.user)}}
                    </a>
                </span>`,
            scope : false,
            link : function($scope) {
                $scope.rootScope= $rootScope;
            }
        }
    });

    app.directive('metadataObjectModal', () => {
        return {
            scope : true,
            link : function($scope) {
                if ($scope.metadataObjectParent.customMeta && $scope.metadataObjectParent.customMeta.kv) {
                    $scope.localObject = angular.copy($scope.metadataObjectParent.customMeta.kv);
                } else {
                    $scope.localObject = { };
                }
                $scope.save = function() {
                    $scope.metadataObjectParent.customMeta = { 'kv' : angular.copy($scope.localObject) };
                    $scope.$emit("metadataObjectUpdated");
                    $scope.dismiss();
                };
            }
        }
    });

     app.directive('metadataObjectLink', function(CreateModalFromTemplate){
        return {
            restrict : 'AE',
            scope : {
                metadataObjectParent : '='
            },
            template: '<div class="metadata-object-link"><pre class="small-pre">{{metadataObjectParent.customMeta.kv | json}}</pre><button title="Object metadata" class="btn btn--secondary" ng-click="openModal()"><i class="icon-superscript" />&nbsp;Edit</button></div>',
            link : ($scope) => {
                $scope.openModal = function(){
                    CreateModalFromTemplate("/templates/widgets/metadata-object-modal.html", $scope,
                        null, null);
                }
            }
        }
    });

     app.directive('overrideTableModal', function($stateParams, DataikuAPI){
        return {
            scope : true,
            // compile/pre to execute before listForm's link (for transcope)
            compile : function(){ return { pre: ($scope) => {
                $scope.simplifiedObjectToOverride = {};
                $scope.$watch("objectToOverride", function(nv){
                    if (nv) {
                        $scope.simplifiedObjectToOverride = angular.copy($scope.objectToOverride);
                        $.each($scope.simplifiedObjectToOverride, function(k) {
                            if (k.indexOf("$") == 0) {
                                delete $scope.simplifiedObjectToOverride[k];
                            }
                        });
                        delete $scope.simplifiedObjectToOverride["overrideTable"];
                        delete $scope.simplifiedObjectToOverride["change"];
                        delete $scope.simplifiedObjectToOverride["versionTag"];
                    }
                }, true);
                if ($scope.overrideTableParent.overrideTable) {
                    $scope.localTable = angular.copy($scope.overrideTableParent.overrideTable);
                } else {
                    $scope.localTable = { "overrides" : []};
                }
                $scope.save = function() {
                    if ($scope.localTable.overrides.length > 0) {
                        $scope.overrideTableParent.overrideTable = angular.copy($scope.localTable)
                    } else {
                        $scope.overrideTableParent.overrideTable = null;
                    }
                    $scope.$emit("overrideTableUpdated");
                    $scope.dismiss();

                };
                $scope.getValue = (function(override) {
                    DataikuAPI.variables.expandExpr($stateParams.projectKey, override.expr).success(function(data){
                        override.$$computedValue = data.id;
                    }).error(setErrorInScope.bind(this));
                }).bind($scope); // bind on parent scope
            } }; }
        }
    });

    app.directive('overrideTableBtnLink', function(CreateModalFromTemplate){
        return {
            scope : {
                overrideTableParent : '=',
                objectToOverride : '='
            },
            template: '<div class="override-table-link"><pre class="small-pre">{{overrideDesc}}</pre><button title="Override variables" class="btn btn--secondary" ng-click="openModal()"><i class="icon-superscript" />&nbsp;Edit</button></div>',
            link : ($scope) => {
                $scope.overrideDesc = '';
                $scope.$watch('overrideTableParent.overrideTable', function(nv) {
                    if ( nv == null) return;
                    var desc = '';
                    if ( $scope.overrideTableParent.overrideTable.overrides != null ) {
                        $scope.overrideTableParent.overrideTable.overrides.forEach(function(override) {
                            desc = desc + override.path + " ";
                        });
                    }
                    $scope.overrideDesc = desc;
                }, true);
                $scope.openModal = function(){
                    CreateModalFromTemplate("/templates/widgets/override-table-modal.html", $scope,
                        null, null);
                }
            }
        }
    });
     app.directive('overrideTableLink', function(CreateModalFromTemplate){
        return {
            scope : {
                overrideTableParent : '=',
                objectToOverride : '='
            },
            template: '<a title="Override variables" ng-class="{\'override-table-link\': true, \'overriden\': overrideTableParent.overrideTable.overrides.length}" ng-click="openModal()"><i class="dku-icon-text-superscript-16" /></a>',
            link : ($scope) => {
                $scope.openModal = function(){
                    CreateModalFromTemplate("/templates/widgets/override-table-modal.html", $scope,
                        null, null);
                }
            }
        }
    });
    app.directive('dkuIndeterminate', function() {
        return {
            restrict: 'A',
            link: function(scope, element, attributes) {
                scope.$watch(attributes.dkuIndeterminate, function(value) {
                    element.prop('indeterminate', !!value);
                });
            }
        };
    });

    app.directive('validFile',function(){
        return {
            require:'ngModel',
            link:function(scope, el, attrs, ngModel) {
                el.bind('change', function() {
                    var val = 'multiple' in attrs ? this.files : this.files[0];
                    scope.$apply(function() {
                        ngModel.$setViewValue(val);
                        ngModel.$render();
                    });
                });
            }
        };
    });


    app.directive('sparkline', function() {
        return {
            scope: {
                sparkline: '='
            },
            link: function(scope, element) {
                const data = scope.sparkline;
                const rect = element[0].parentElement.getBoundingClientRect();
                const x = d3.scale.linear().domain([0, data.length-1]).range([0, rect.width]);
                const y = d3.scale.linear().domain([0, d3.max(data) || 0]).range([rect.height, 4]);
                const line = d3.svg.line().x((d, i) => x(i)).y(d => y(d || 0));

                d3.select(element[0]).html("")
                    .append("svg:svg")
                    .append("svg:path")
                    .attr("d", line(data))
                    .attr("stroke-width", "2px")
                    .attr("stroke", "#add8e6")
                    .attr("fill", "#add8e6")
                    .attr("fill-opacity", .3);
            }
        }
    });

    app.directive('weekDaysPicker', function(translate) {
        return {
            scope : {
                selection:'=ngModel',
                onChange:'&?'
            },
            template: `<div class="weekdays-picker">
                <span ng-repeat="day in days" ng-click="toggle(day.value)" ng-class="[{selected: hasSelected(day.value)}]" >{{day.label[0]}}</span>
            </div>`,
            link: function($scope) {
                //  global WEEKDAYS we use an object because the technical name should stays the same
                $scope.days = [...WEEKDAYS].map(weekday => {
                return {
                    "value": weekday,
                    "label": translate('GLOBAL.WEEKDAYS.' + weekday.toUpperCase(), weekday)
                }});

                $scope.hasSelected = (day) => {
                    return $scope.selection.includes(day);
                };

                $scope.toggle = (day) => {
                    if($scope.selection.includes(day)) {
                        $scope.selection = $scope.selection.filter(s => s !== day);
                    } else {
                        $scope.selection = [...$scope.selection, day];

                    }
                    if($scope.onChange) {
                        $scope.onChange($scope.selection);
                    }
                };

                $scope.$watch('selection', () => {
                    if(!Array.isArray($scope.selection)) {
                        $scope.selection = [];
                    }
                });
            }
        }
    });

    app.directive('inlineDateRangePicker', ['$timeout', () => { // inline-date-range-picker
        return {
            scope : {
                from:'=',
                to:'=',
                tz:'=',
                onChange:'&?',
            },
            template: `
                <div class="inline-date-range-picker" ng-class="{ 'inline-date-range-picker--big-inputs': hasBiggerInputs }">
                    <fieldset>
                        <div class="fieldLabel" translate="WIDGET.DATE_RANGE_PICKER.FROM">From</div>
                        <div class="inline-date-range-picker__row">
                            <input class="inline-date-range-picker__date" ng-model="from" ng-keypress="onFromKeypress($event)" ng-blur="onFromChange()" type="date" name="dateFrom" />
                            <input class="inline-date-range-picker__time" ng-model="from" ng-model-options="{ timeSecondsFormat: 'ss' }" ng-keypress="onFromKeypress($event)" ng-blur="onFromChange()" type="time" name="timeFrom" step="1" />
                        </div>
                    </fieldset>
                    <fieldset>
                        <div class="fieldLabel" translate="WIDGET.DATE_RANGE_PICKER.TO">To</div>
                        <div class="inline-date-range-picker__row">
                            <input class="inline-date-range-picker__date" ng-model="to" ng-keypress="onToKeypress($event)" ng-blur="onToChange()" type="date" name="dateTo" />
                            <input class="inline-date-range-picker__time" ng-model="to" ng-model-options="{ timeSecondsFormat: 'ss' }" ng-keypress="onToKeypress($event)" ng-blur="onToChange()" type="time" name="timeTo" step="1" />
                        </div>
                    </fieldset>
                    <fieldset>
                        <div class="fieldLabel" translate="WIDGET.DATE_RANGE_PICKER.TIMEZONE">Timezone</div>
                        <basic-select class="inline-date-range-picker__timezone" items="timezone_ids" ng-model="tz"></basic-select>
                    </fieldset>
                    <div class="ff-date-range-filter-hint" ng-show="to && from && to < from">
                        <i class="icon-warning-sign"></i>&nbsp;<span translate="WIDGET.DATE_RANGE_PICKER.WARNING.START_DATE_AFTER_END_DATE">Start date is later than end date. The result will always be empty.</span>
                    </div>
                </div>
                `,
            link : function($scope) {
                $scope.timezone_ids = [ "UTC",
                    "Africa/Abidjan", "Africa/Addis_Ababa", "Africa/Algiers", "Africa/Bamako", "Africa/Bangui", "Africa/Brazzaville", "Africa/Cairo",
                    "Africa/Casablanca", "Africa/Conakry", "Africa/Dakar", "Africa/Djibouti", "Africa/Harare", "Africa/Johannesburg", "Africa/Kigali",
                    "Africa/Kinshasa", "Africa/Lagos", "Africa/Libreville", "Africa/Mogadishu", "Africa/Nairobi", "Africa/Ndjamena", "Africa/Niamey",
                    "Africa/Nouakchott", "Africa/Ouagadougou", "Africa/Tripoli", "Africa/Tunis",
                    "America/Adak", "America/Anchorage", "America/Argentina/Buenos_Aires", "America/Aruba", "America/Bogota", "America/Cancun", "America/Caracas",
                    "America/Cayenne", "America/Cayman", "America/Chicago", "America/Costa_Rica", "America/Dawson_Creek", "America/Denver", "America/Detroit",
                    "America/El_Salvador", "America/Goose_Bay", "America/Grenada", "America/Guadeloupe", "America/Guatemala", "America/Guyana", "America/Halifax",
                    "America/Havana", "America/Indianapolis", "America/Jamaica", "America/Juneau", "America/La_Paz", "America/Lima", "America/Los_Angeles",
                    "America/Martinique", "America/Mexico_City", "America/Monterrey", "America/Montevideo", "America/Montserrat", "America/Nassau", "America/New_York",
                    "America/Noronha", "America/Panama", "America/Puerto_Rico", "America/Santiago", "America/Sao_Paulo", "America/St_Johns", "America/Tijuana",
                    "America/Toronto", "America/Vancouver", "America/Winnipeg",
                    "Arctic/Longyearbyen",
                    "Asia/Baghdad", "Asia/Bahrain", "Asia/Baku", "Asia/Bangkok", "Asia/Beirut", "Asia/Brunei", "Asia/Calcutta",
                    "Asia/Damascus", "Asia/Dhaka", "Asia/Dubai", "Asia/Hebron", "Asia/Ho_Chi_Minh", "Asia/Hong_Kong", "Asia/Irkutsk",
                    "Asia/Jakarta", "Asia/Jerusalem", "Asia/Kabul", "Asia/Karachi", "Asia/Kathmandu", "Asia/Kuala_Lumpur", "Asia/Kuwait",
                    "Asia/Macao", "Asia/Macau", "Asia/Manila", "Asia/Phnom_Penh", "Asia/Qatar", "Asia/Riyadh", "Asia/Saigon",
                    "Asia/Seoul", "Asia/Shanghai", "Asia/Singapore", "Asia/Taipei", "Asia/Tehran", "Asia/Tel_Aviv", "Asia/Tokyo",
                    "Atlantic/Azores", "Atlantic/Bermuda", "Atlantic/Canary", "Atlantic/Cape_Verde", "Atlantic/Madeira", "Atlantic/Reykjavik",
                    "Australia/ACT", "Australia/Adelaide", "Australia/Brisbane", "Australia/Canberra", "Australia/Darwin", "Australia/Eucla", "Australia/Lord_Howe",
                    "Australia/Melbourne", "Australia/NSW", "Australia/North", "Australia/Perth", "Australia/Queensland", "Australia/South", "Australia/Sydney",
                    "Australia/Tasmania", "Australia/Victoria", "Australia/West",
                    "Brazil/East", "Brazil/West",
                    "Canada/Atlantic", "Canada/Central", "Canada/East-Saskatchewan", "Canada/Eastern", "Canada/Mountain", "Canada/Newfoundland", "Canada/Pacific",
                    "Canada/Saskatchewan", "Canada/Yukon",
                    "Europe/Amsterdam", "Europe/Athens", "Europe/Belfast", "Europe/Berlin", "Europe/Bratislava", "Europe/Brussels", "Europe/Bucharest",
                    "Europe/Budapest", "Europe/Busingen", "Europe/Copenhagen", "Europe/Dublin", "Europe/Helsinki", "Europe/Istanbul", "Europe/Kiev",
                    "Europe/Lisbon", "Europe/Ljubljana", "Europe/London", "Europe/Luxembourg", "Europe/Madrid", "Europe/Malta", "Europe/Minsk",
                    "Europe/Monaco", "Europe/Moscow", "Europe/Nicosia", "Europe/Oslo", "Europe/Paris", "Europe/Prague", "Europe/Riga",
                    "Europe/Rome", "Europe/Sarajevo", "Europe/Sofia", "Europe/Stockholm", "Europe/Tallinn", "Europe/Uzhgorod", "Europe/Vienna",
                    "Europe/Vilnius", "Europe/Warsaw", "Europe/Zagreb", "Europe/Zurich",
                    "Indian/Cocos", "Indian/Maldives", "Indian/Mauritius", "Indian/Mayotte", "Indian/Reunion",
                    "Pacific/Apia", "Pacific/Auckland", "Pacific/Chatham", "Pacific/Enderbury", "Pacific/Gambier", "Pacific/Guam", "Pacific/Honolulu",
                    "Pacific/Kiritimati", "Pacific/Marquesas", "Pacific/Niue", "Pacific/Noumea", "Pacific/Pitcairn", "Pacific/Tahiti", "Pacific/Wallis"
                ];

                $scope.hasBiggerInputs = navigator.userAgent.includes("Gecko/");

                $scope.$watch("to", (nv, ov) => {
                    if (ov && !nv) {
                        // Trigger an onChange event when user click on the "clear" button
                        // (side-effect: also trigger an event when user enters an invalid date)
                        $scope.onChange && $scope.onChange();
                    }
                });

                $scope.$watch("from", (nv, ov) => {
                    if (ov && !nv) {
                        // Trigger an onChange event when user click on the "clear" button
                        // (side-effect: also trigger an event when user enters an invalid date)
                        $scope.onChange && $scope.onChange();
                    }
                });

                $scope.$watch("tz", (nv, ov) => {
                    if (nv !== ov) {
                        $scope.onChange && $scope.onChange();
                    }
                });

                $scope.onFromKeypress = ({key}) => {
                    if (key === 'Enter') {
                        $scope.onFromChange();
                    }
                };

                $scope.onFromChange = () => {
                    if ($scope.from && $scope.to) {
                        if ($scope.from.getTime() > $scope.to.getTime()) {
                            $scope.to = new Date($scope.from.getTime());
                        }
                    }
                    $scope.onChange && $scope.onChange();
                };

                $scope.onToKeypress = ({key}) => {
                    if (key === 'Enter') {
                        $scope.onToChange();
                    }
                };

                $scope.onToChange = () => {
                    if ($scope.from && $scope.to) {
                        if ($scope.from.getTime() > $scope.to.getTime()) {
                            $scope.from = new Date($scope.to.getTime());
                        }
                    }
                    $scope.onChange && $scope.onChange();
                };
            }
        }
    }]);

    app.directive('executionPlan', function() {
        return {
            restrict: "AE",
            scope: {
                executionPlan: '=ngModel'
            },
            templateUrl: '/templates/widgets/execution-plan.html',
            link: () => {
                //nothing to do for now...
            }
        };
    });

    // SO : http://stackoverflow.com/questions/18368485/angular-js-resizable-div-directive
    app.directive('resizer', function($document,Throttle,$rootScope) {
        return function($scope, $element, $attrs) {

            $element.addClass('content-resizer');

            $element.on('mousedown', function(event) {
                $element.parent().addClass("resizing");
                event.preventDefault();
                $document.on('mousemove', mousemove);
                $document.on('mouseup', mouseup);
            });

            function mousemove(event) {

                if ($attrs.resizer == 'vertical') {
                    // Handle vertical resizer
                    let x = event.pageX;

                    if ($attrs.resizerContainer) {
                        const containerWidth = $($attrs.resizerContainer).width();
                        const maxWidth = containerWidth - parseInt($attrs.resizerWidth);
                        x = Math.min(maxWidth, x);
                    }
                    x = Math.max(0, x);

                    $element.css({
                        left: x + 'px'
                    });

                    $($attrs.resizerLeft).css({
                        width: x + 'px'
                    });
                    $($attrs.resizerRight).css({
                        left: (x + parseInt($attrs.resizerWidth)) + 'px'
                    });

                } else {
                    // Handle horizontal resizer
                    let y = window.innerHeight - event.pageY;
                    if ($attrs.resizerContainer) {
                        const containerHeight = $($attrs.resizerContainer).height();
                        const maxHeight = containerHeight - parseInt($attrs.resizerHeight);
                        y = Math.min(maxHeight, y);
                    }
                    y = Math.max(0, y);

                    $element.css({
                        bottom: y + 'px'
                    });

                    $($attrs.resizerTop).css({
                        bottom: (y + parseInt($attrs.resizerHeight)) + 'px'
                    });
                    $($attrs.resizerBottom).css({
                        height: y + 'px'
                    });
                }
            }

            function mouseup() {
                $document.unbind('mousemove', mousemove);
                $document.unbind('mouseup', mouseup);
                $element.parent().removeClass("resizing");
                $rootScope.$broadcast('reflow');
            }
        };
    });

    app.directive('fatTable', function($compile,$rootScope,Debounce,$http,$templateCache) {

         return {
            restrict : 'A',
            scope : {
                rows:'=',
                as : '@',
                rowIndexAs:'@',
                headers:'=',
                columnWidths : '=',
                headerTemplate:'@',
                cellTemplate:'@',
                printNewLinesAsSymbols:'@',
                rowHeight:'=',
                headerHeight : '=',
                digestChildOnly:'=?'
            },
            link : function(scope, element, attrs) {
                $http.get(scope.cellTemplate, {cache: $templateCache}).then(function(resp) {
                    let cellTemplateHTML = resp.data;
                    $http.get(scope.headerTemplate, {cache: $templateCache}).then(function(resp) {
                        let headerTemplateHTML = resp.data;
                        // We don't use Debounce here because it always triggers a full digest cycle!
                        var digestTimeout = undefined;
                        function debouncedDigestCycle() {
                            if(digestTimeout === undefined) {
                                digestTimeout = setTimeout(function() {
                                    digestTimeout = undefined;
                                    if(scope.digestChildOnly) {
                                        var elmScope = element.scope();
                                        if(elmScope) {
                                            // Partial digestion
                                            elmScope.$digest();
                                        }
                                    } else {
                                        // Full digestion
                                        $rootScope.$digest();
                                    }
                                },10);
                            }
                        }

                        function cleanDOM(div) {
                            // Destroy the cell's __fat_scope__
                            var fs = div.__fat_scope__;
                            if(fs) {
                                fs.$destroy();
                            }
                            div.__fat_scope__ = undefined;
                            // Make sure there is no refs to the scope in JQuery's cache
                            $(div).data('$scope',null);
                        }

                        function buildModel() {
                           var tableData = new fattable.SyncTableModel();
                           tableData.getCellSync = function(i,j) {
                               var arr = scope.rows;
                               if(!arr || !arr.length || i<0 || i>=arr.length) {
                                   return {i:i,j:j,v:undefined,t:'c'};
                               } else {
                                   var row = arr[i];
                                   if(!row || !row.length || j < 0 || j >=row.length) {
                                        return {i:i,j:j,v:undefined,t:'c'};
                                   }
                                   return {i:i,j:j,v:row[j],t:'c'};
                               }
                           };
                           tableData.getHeaderSync = function(i) {
                               var arr = scope.headers;
                               if(!arr || !arr.length || i<0 || i>=arr.length) {
                                   return {i:i,v:undefined,t:'h'};
                               } else {
                                  return {i:i,v:arr[i],t:'h'};
                               }
                           };
                           return tableData;
                        }

                        var livingCells = [];

                        function buildPainter() {
                           var painter = new fattable.Painter();

                           var prepareElement = function(template) {
                              return function(cellDiv, data) {
                                  if(!cellDiv.__fat_scope__) {
                                      var elementScope = element.scope();
                                      if(elementScope) {
                                          cellDiv.__fat_scope__ = elementScope.$new();
                                          $(cellDiv).append($compile(template)(cellDiv.__fat_scope__));
                                      }
                                  }
                                  if(cellDiv.__fat_scope__) {
                                      let v = data.v;
                                      if (scope.printNewLinesAsSymbols && (typeof v === 'string' || v instanceof String)) {
                                          v = v.replace(/(\r\n|\n)/g, "¶");
                                      }
                                      cellDiv.__fat_scope__[attrs.as] = v;
                                      if(attrs.rowIndexAs && data.t == 'c') {
                                            cellDiv.__fat_scope__[attrs.rowIndexAs] = data.i;
                                      }
                                      debouncedDigestCycle();
                                  }
                               };
                           };

                           painter.fillCell = prepareElement(cellTemplateHTML);
                           painter.fillHeader = prepareElement(headerTemplateHTML);

                           painter.fillCellPending = (cellDiv) => {
                               cellDiv.textContent = "";
                               cellDiv.className = "pending";
                           };

                           painter.fillHeaderPending = (cellDiv) => {
                              cellDiv.textContent = "";
                              cellDiv.className = "pending";
                           };

                           painter.setupCell = function(div) {
                              livingCells.push(div);
                           };

                           painter.setupHeader = painter.setupCell;

                           painter.cleanUpCell = function(div) {
                               livingCells = livingCells.filter(function(x) {
                                   return x!=div;
                               });
                               cleanDOM(div);
                           };

                           painter.cleanUpHeader = painter.cleanUpCell;

                           return painter;
                        }
                        var oldTable;

                        function redraw() {
                            if(oldTable) {
                                oldTable.cleanUp();
                                // bug in fattable : cleanUp() at line 702 is not checking the variable holding the scroll proxy, so
                                // the scroll elements still try to call onScroll (until the next DOM rebuild where they're removed)
                                if (oldTable.scroll != null) {
                                    oldTable.scroll.onScroll = function() {}; // NOSONAR: noop
                                }
                            }
                            const table = fattable({
                                "container": element[0],
                                "model": buildModel(),
                                "nbRows": scope.rows? scope.rows.length:0,
                                "rowHeight": scope.rowHeight,
                                "headerHeight": scope.headerHeight,
                                "painter": buildPainter(),
                                "columnWidths": scope.columnWidths
                            });
                            if(oldTable && oldTable.scroll) {
                                var y = oldTable.scroll.scrollTop;
                                var x = oldTable.scroll.scrollLeft;
                                table.scroll.setScrollXY(x,y);
                            }
                            oldTable = table;
                        }

                       var debouncedRedraw = Debounce().withDelay(50,200).wrap(redraw);
                       scope.$watch('rows', debouncedRedraw, false);
                       scope.$watch('headers', debouncedRedraw, false);
                       $(window).on('resize', debouncedRedraw);

                       element.scope().$on("reflow", debouncedRedraw);

                       scope.$on("$destroy", function () {
                           if(oldTable) {
                                oldTable.cleanUp();
                                oldTable=null;
                           }
                           for(var i = 0 ; i < livingCells.length ; i++) {
                                cleanDOM(livingCells[i]);
                           }
                           livingCells = [];
                           $(window).off("resize", debouncedRedraw);
                       });
                    });
               });
            }
         };
    });

    app.directive('registerModelForForm', function () {
        return {
            scope: {form: '=registerModelForForm'},
            require: 'ngModel',
            controller: function ($element,$scope) {
                var ngModel = $element.controller('ngModel');
                $scope.form.$addControl(ngModel);
            }
        };
    });

    app.directive('fatRepeat', function($compile, $rootScope, $timeout, Debounce, FatTouchableService, FatDraggableService) {

        return {
            transclude:true,
            scope:{
                fatRepeat:'=', // Array
                fatDraggable:'=?', // If items should be draggable
                fatDraggableOnDrop:'=?', // Callback called when drag ends
                as:'@', // Name of each item
                rowHeight:'=', // Height of each row
                colWidth: '=?', // width of each column
                digestChildOnly:'=?', // If true, doesn't trigger a full digest cycle each time a cell updated, but call
                // $digest() on child scope only. It's generally MUCH faster, but you need to make sure
                // that your watches have no side effects on parent scopes.
                initScope: '=?', // item scope init function
                tableModel: '=?', // Custom fattable Model
                inForm:'=?',
                layoutMode: '@?', //one of row, mosaic (or potentially left blank for list mode)
                listPadding: '=?', // padding to be introduced before first and after last item
                fTrackTable: '&', // to allow containers to control the scroll or other aspect of the underlying table
                disableScrollTo: '@', // Disable scrollToLine event
                enableAsync: '=?',
                nbRows: '=?',
                chunkSize: '=?',
                getRowChunk: '=?',
                pageFromData: '=?',
                allowHorizontalScroll: '=?', // compute dynamically the row width to allow horizontal scrolling. Only valid for list mode.
                enableDragMove: '<?', // whether the underlying fattable allows middle-click dragging to scroll (default to true)
                dedicatedWidthForVScrollbar: "<?", // whether we always allocate a width for the verticall scrollbar (defaults to true)
            },
            restrict:'A',
            compile: function(_element,_attrs,transclude) {

                return function(scope, element, attrs) {
                    const HORIZ_SCROLL_H = 1;
                    const VERT_SCROLL_W = 8;


                    element.addClass(scope.layoutMode ? 'fat-row' : 'fat-repeat');
                    if (scope.layoutMode=="row") {
                        $(element).css('height', (parseInt(scope.rowHeight,10) + HORIZ_SCROLL_H).toString() + 'px');
                    }

                    // We don't use Debounce here because it always triggers a full digest cycle!
                    var digestTimeout = undefined;
                    function debouncedDigestCycle() {
                        if(digestTimeout === undefined) {
                            digestTimeout = setTimeout(function() {
                                digestTimeout = undefined;
                                if(scope.digestChildOnly) {
                                    var elmScope = element.scope();
                                    if(elmScope) {
                                        // Partial digestion
                                        elmScope.$digest();
                                    }
                                } else {
                                    // Full digestion
                                    $rootScope.$digest();
                                }
                            },10);
                        }
                    }

                    function cleanDOM(div) {
                        // Destroy the cell's __fat_scope__
                        var fs = div.__fat_scope__;
                        if(fs) {
                            fs.$destroy();
                        }
                        div.__fat_scope__ = undefined;
                        // Make sure there is no refs to the scope in JQuery's cache
                        $(div).data('$scope',null);
                    }

                    function buildModel() {
                        if (scope.tableModel) {
                            return scope.tableModel.call(scope);
                        }
                        if (scope.enableAsync) {
                            const asyncTableData = new fattable.PagedAsyncTableModel();
                            asyncTableData.fetchCellPage = (pageName, cb) => {
                                var promise = scope.getRowChunk(pageName);
                                promise.then(response => {
                                    cb(scope.pageFromData(pageName, response.data));
                                });
                            };
                            asyncTableData.cellPageName = (row, col) => {
                                return Math.trunc(row / Math.max(1, scope.chunkSize));
                            };
                            if (scope.fatRepeat && scope.fatRepeat.length > 0) {
                                var initialPageName = asyncTableData.cellPageName(0, 0);
                                const initialPage = scope.pageFromData(initialPageName, {items: [...scope.fatRepeat]});
                                asyncTableData.pageCache.set(initialPageName, initialPage);
                            }
                            return asyncTableData;
                        }
                        const tableData = new fattable.SyncTableModel();

                        tableData.getCellSync = function(i,j) {
                            const arr = scope.fatRepeat;
                            const idx = i*scope.numColumns + j;
                            if(!arr || idx<0 || idx>=arr.length) {
                                return undefined;
                            }
                            return arr[idx];
                        };
                        return tableData;
                    }

                    var livingCells = [];

                    // Scroll bar handling. Observes the changes inside the table and redraw with a forced width if needed
                    const vScrollBarWidth = 11;
                    const getContentBaseWidth = () => element.width() - (scope.dedicatedWidthForVScrollbar !== false ? vScrollBarWidth : 0); // the size available for the content when no scroll is possible or needed
                    let forcedMinimumColumWidth = 0;
                    let resetMinimumColumnWidth = true;
                    let mutationObserver;
                    if(scope.allowHorizontalScroll) {
                        element.addClass('fat-repeat--with-horizontal-scroll')
                        mutationObserver = new MutationObserver(() => {
                                mutationObserver.takeRecords();

                                const requiredWidth = livingCells.reduce(
                                    (acc, cell)=> Math.max(cell.scrollWidth, acc),
                                    0,
                                );

                                if(requiredWidth != forcedMinimumColumWidth && requiredWidth > getContentBaseWidth()) {
                                    forcedMinimumColumWidth = requiredWidth;
                                    resetMinimumColumnWidth = false;
                                    debouncedRedraw();
                                }
                            },
                        );
                        mutationObserver.observe(element[0], {childList: true, subtree: true});
                    }

                    function buildPainter() {
                        var painter = new fattable.Painter();
                        painter.fillCell = function(cellDiv, data) {
                            cellDiv.className = '';
                            if(!cellDiv.__fat_scope__) {
                                var elementScope = element.scope();
                                if(elementScope) {
                                    cellDiv.__fat_scope__ = elementScope.$new();
                                    transclude(cellDiv.__fat_scope__,function(clone) {
                                        $(cellDiv).append(clone);
                                    });
                                }
                            }
                            if(cellDiv.__fat_scope__) {
                                cellDiv.__fat_scope__[attrs.as] = data;
                                debouncedDigestCycle();
                                if (scope.initScope) {
                                    scope.initScope(cellDiv.__fat_scope__);
                                }
                            }
                        };
                        painter.fillCellPending = (cellDiv) => {
                            cellDiv.className = "fat-repeat-pending-row";
                        };
                        painter.setupCell = function(div) {
                            livingCells.push(div);
                        };
                        painter.cleanUpCell = function(div) {
                            livingCells = livingCells.filter(function(x) {
                                return x!=div;
                            });
                            cleanDOM(div);
                        };
                        return painter;
                    }

                    var oldTable;

                    function redraw() {
                        if (oldTable) {
                            oldTable.cleanUp();
                        }
                        let fatRepeatLength = scope.fatRepeat? scope.fatRepeat.length:0;
                        if (scope.layoutMode=="row") {
                            // row mode
                            scope.numColumns = fatRepeatLength;
                            scope.numRows = 1;
                        }
                        else if (scope.layoutMode=="mosaic") {
                            scope.numColumns = Math.floor((element.innerWidth() - VERT_SCROLL_W)/ scope.colWidth);
                            scope.numRows = Math.ceil(fatRepeatLength / scope.numColumns);
                        }
                        else if (scope.enableAsync) {
                            scope.numRows = scope.nbRows;
                            scope.numColumns = 1;
                        } else {
                            scope.numRows = fatRepeatLength;
                            scope.numColumns = 1;
                        }

                        if (scope.listPadding && scope.layoutMode != "row") { // pad the end via a whole row
                            scope.numRows++
                        }

                        let columnWidths = [getContentBaseWidth()];

                        if (['mosaic','row'].includes(scope.layoutMode)) {
                            columnWidths = Array.from({length: scope.numColumns}, () => scope.colWidth);
                            if (columnWidths.length>0  && scope.layoutMode=="row" && scope.listPadding) {
                                if (typeof scope.listPadding === 'string') scope.listPadding = parseInt(scope.listPadding, 10);
                                columnWidths[columnWidths.length-1] += 2 * scope.listPadding;
                            }
                        } else if(scope.allowHorizontalScroll) { // list mode with horizontal scroll bar
                            if(resetMinimumColumnWidth) {
                                forcedMinimumColumWidth = 0;
                            }
                            columnWidths = [Math.max(forcedMinimumColumWidth, columnWidths[0])];
                            resetMinimumColumnWidth = true;
                        }

                        var table = fattable({
                            "container": element[0],
                            "model": buildModel(),
                            "nbRows": scope.numRows,
                            "rowHeight": scope.rowHeight,
                            "headerHeight": 0,
                            "painter": buildPainter(),
                            "columnWidths": columnWidths,
                            "enableDragMove": scope.enableDragMove !== false // defaults to true if not defined
                        });

                        if (attrs.fatDraggable !== undefined && typeof scope.fatDraggableOnDrop === 'function' && !scope.$$destroyed) {
                            FatDraggableService.setDraggable({
                                element: table.container,
                                onDrop: scope.fatDraggableOnDrop,
                                axis: 'y',
                                scrollBar: table.scroll,
                                classNamesToIgnore: ['icon-sort-by-attributes', 'sort-indication', 'pull-right']
                            })

                            // We have to set the fat-draggable__item class to an inner child because fattable re-use
                            // the same cell div for different items
                            for (let cellKey in table.cells) {
                                if (!table.cells.hasOwnProperty(cellKey)) continue;
                                let cellDiv = table.cells[cellKey];
                                let cellDivColumnHeader = cellDiv && cellDiv.children && cellDiv.children[0];
                                cellDivColumnHeader && cellDivColumnHeader.classList.add('fat-draggable__item');
                            }
                        }

                        if (isTouchDevice()) {
                            if (oldTable && typeof(scope.unsetTouchable) === "function") {
                                scope.unsetTouchable();
                            }
                            scope.unsetTouchable = FatTouchableService.setTouchable(scope, element, table);
                        }

                        if (oldTable) {
                            var y = oldTable.scroll.scrollTop;
                            var x = oldTable.scroll.scrollLeft;
                            table.scroll.setScrollXY(x,y);
                        }

                        oldTable = table;

                        if (scope.fTrackTable) scope.fTrackTable({table:oldTable});

                        if (scope.layoutMode=="row" && scope.listPadding) {
                            $(element).find('.fattable-viewport').css('padding-left', scope.listPadding);
                        }

                        if (scope.inForm) {
                            _element.find('[ng-model]').each((idx, el) => {
                                scope.inForm.$addControl(angular.element(el).controller('ngModel'));
                            });
                        }
                    }

                    scope.$on('moveScroll', (event, x, y) => {
                        oldTable.scroll.setScrollXY(oldTable.scroll.scrollLeft + x, oldTable.scroll.scrollTop + y);
                    });

                    scope.$watchCollection('fatRepeat', redraw);

                    scope.$on('redrawFatTable', redraw);
                    scope.$on('repaintFatTable', function () { debouncedRedraw(); }); //works better wrapped in a fnc!

                    if (scope.disableScrollTo === undefined) {
                        scope.$on('scrollToLine', function(e, lineNum) {
                            if (oldTable) {
                                let nbRowsVisible = oldTable.h / oldTable.rowHeight; // we need the float value
                                let firstVisibleRow = oldTable.scroll.scrollTop / oldTable.rowHeight; // we need the float value
                                let x = oldTable.scroll.scrollLeft;
                                if (lineNum == -1) {
                                    let y = oldTable.nbRows * oldTable.rowHeight;
                                    oldTable.scroll.setScrollXY(x, y);
                                } else if (lineNum <= firstVisibleRow) {
                                    let y = Math.max(lineNum, 0) * oldTable.rowHeight;
                                    oldTable.scroll.setScrollXY(x,y);
                                } else if (lineNum >= firstVisibleRow + nbRowsVisible - 1) {
                                    let y = (Math.min(lineNum, oldTable.nbRows) + 1) * oldTable.rowHeight - oldTable.h;
                                    oldTable.scroll.setScrollXY(x,y);
                                }
                            }
                        });
                    }

                    var debouncedRedraw = Debounce().withDelay(50,200).wrap(redraw);
                    $(window).on('resize', debouncedRedraw);

                    element.scope().$on("reflow", debouncedRedraw);

                    scope.$on("$destroy", function () {
                        if(oldTable) {
                            oldTable.cleanUp();
                            oldTable=null;
                        }
                        for(var i = 0 ; i < livingCells.length ; i++) {
                            cleanDOM(livingCells[i]);
                        }
                        livingCells = [];
                        $(window).off("resize", debouncedRedraw);
                        if(mutationObserver) {
                            mutationObserver.disconnect();
                        }
                    });


                };
            }
        };
    });

    app.directive('spinner', function() {
        return {
            template: '<div id="qa_spinner" class="spinnerContainer"></div>',
            replace: true,
            restrict: 'E',
            link: function(_scope, element, attrs) {
                let opts = {
                  lines: 6, // The number of lines to draw
                  length: 0, // The length of each line
                  width: 10, // The line thickness
                  radius: 10, // The radius of the inner circle
                  corners: 1, // Corner roundness (0..1)
                  rotate: 0, // The rotation offset
                  color: '#fff', // #rgb or #rrggbb
                  speed: attrs.speed || 1, // Rounds per second
                  trail: 60, // Afterglow percentage
                  shadow: false, // Whether to render a shadow
                  hwaccel: false, // Whether to use hardware acceleration
                  className: 'spinner', // The CSS class to assign to the spinner
                  zIndex: 2e9, // The z-index (defaults to 2000000000)
                  top: 'auto', // Top position relative to parent in px
                  left: 'auto' // Left position relative to parent in px
                };
                new Spinner(opts).spin(element[0]);
          }
       };
    });

    app.directive('formTemplate', function(){
        return {
            templateUrl: '/templates/form-template.html',
            replace: true,
            restrict: 'E',
            link: function(scope, element, attrs){
                function initializeDefaultValues() {
                    if (scope.formDefinition) {
                        for (const formDefinitionElement of scope.formDefinition) {
                            if (!formDefinitionElement.params) {
                                continue;
                            }
                            if (!scope.model.hasOwnProperty(formDefinitionElement.name)) {
                                scope.model[formDefinitionElement.name] = {};
                            }
                            for (const param of formDefinitionElement.params) {
                                if (!scope.model[formDefinitionElement.name].hasOwnProperty(param.name) && param.defaultValue) {
                                    scope.model[formDefinitionElement.name][param.name] = param.defaultValue;
                                }
                            }
                        }
                    }
                }

                if (attrs.monitor) {
                    scope.model = {};
                    scope.$watch(attrs.monitor, () => {
                        const model = scope.$eval(attrs.model);
                        if (model && Object.keys(model).length > 0) {
                            scope.model = model;
                        }
                        scope.formDefinition = scope.$eval(attrs.formDefinition);

                        initializeDefaultValues();
                    });
                } else {
                    scope.model = scope.$eval(attrs.model);
                    scope.formDefinition = scope.$eval(attrs.formDefinition);
                }
            }
        };
    });

    app.directive('formTemplateElement', function(){
        return {
            templateUrl: '/templates/form-template-element.html',
            replace: true,
            restrict: 'EA',
            scope: {
                model: '=',
                field: '=',
                onCoreParamsChanged: '&',
                disabled: '=',
                label: '@',
                validator: '<?',
                errorMessage: '@?'
            },
            link: () => {
                // noop
            },
            controller: function($scope) {
                $scope.label = $scope.label || $scope.field.label || $scope.field.name;
                $scope.getInputType = function(value) {
                    if ($scope.field.maskIf && typeof $scope.field.maskIf === 'string') {
                        $scope.field.maskIf = new Function('value',
                            'return (' + $scope.field.maskIf + ')(value)');
                    }
                    return $scope.field.maskIf && $scope.field.maskIf(value) ? "password" : "text";
                };
                // If not supplied, the validator should always return true
                if (!$scope.validator) {
                    $scope.validator = () => true;
                }
            }
        };
    });

    app.directive('forceInteger', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, attrs, ngModel) {
                function fromUser(text) {
                    return text;
                }
                function toUser(text, ) {
                    if(attrs.forceInteger == 'false'){
                        return text;
                    }
                    return parseInt(text || 0, 10);
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);

            }
        };
    });

    app.directive('forceIntegerAllowEmpty', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, attrs, ngModel) {
                function fromUser(text) {
                    if (text === '') {
                        return null; // Allow empty input
                    }
                    return text;
                }

                function toUser(text) {
                    if (text === '' || text == null) {
                        return ''; // Keep the empty input
                    }
                    return parseInt(text, 10);
                }

                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });

    app.directive('convertSpecialChar', function() {
         return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, _attr, ngModel) {
                function fromUser(text) {
                    /* global convertSpecialChars */
                    const result = convertSpecialChars(text);
                    ngModel.$setValidity('singleChar', 1 === result.length);
                    return result;
                }
                function toUser(text) {
                    // For the moment we do not convert the unicode character into its ASCII representation as it can
                    // be displayed as-is in text inputs.
                    return text == null ? null : text.replace('\t', '\\t');
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });

    app.directive('convertPercentage', function() {
         return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, _attr, ngModel) {
                // calling round to avoid long decimal tail after floating point math operations e.g. 0.072*100=7.199999999999999
                function round(n){
                    return Math.round(n * 10 ** 12) / 10 ** 12;
                }

                function fromUser(text) {
                    return text == null ? null : round(parseFloat(text) / 100);
                }
                function toUser(text) {
                    return text == null ? null : round(parseFloat(text) * 100);
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });
    app.directive('forceDouble', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, _attr, ngModel) {
                function fromUser(text) {
                    return text;
                }

                function toUser(text) {
                    // Don't silently replace empty by zero !
                    if(text!=null && text!=undefined && text!=='') {
                        return parseFloat(text);
                    } else return '';
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });

    // Droparea is a component that handle file drops on it and callbacks the method defined on its drop parameter
    app.directive("droparea", function($filter, $timeout){
        return {
            restrict: 'E',
            template: '<div class="droparea h100"  ng-class="{candrop: candrop}">'+
                '<form class="upload h100" ></form>'+
                '<div ng-transclude class="nested-template-container h100" ></div>' +
            '</div>',
            replace: true,
            transclude: true,
            scope: {
                drop: '&',
                validate: '&',
                //paramaters used by droparea directive to expose to its parent a candrop flag
                isDroppable: '=?',
                customId: '@?',
            },
            link: function(scope, element, attrs){
                scope.multiple = 'multiple' in attrs;
                scope.noUploadOnClick = 'noUploadOnClick' in attrs;
                scope.candrop = false;
                scope.$watch('candrop', () => {
                    scope.isDroppable = scope.candrop;
                })

                // input fallback
                function addFileManually() {
                    var evt = document.createEvent("MouseEvents");
                    evt.initEvent('click', true, true );
                    input[0].dispatchEvent(evt);
                }

                if (!scope.noUploadOnClick)  {
                    element.click(function(e){
                        e.stopPropagation();
                        addFileManually();
                    });
                }

                element.on('click', 'form.upload input', function(e){
                    e.stopPropagation();
                }).on('change', 'form.upload input', function() {
                    const files = this.files;
                    scope.$apply(function() {
                        scope.drop({'files': files});
                    });
                    createInput();
                });

                var input;
                const id = scope.customId || 'qa_upload_dataset_input-files';
                function createInput(){
                    element.find('form.upload').find('input').remove();
                    input = $(`<input type="file" name="file" id="${id}" multiple />`);
                    element.find('form.upload').append(input);
                }
                createInput();

                // drop file
                function applyDragEnterLeave(e) {
                    e.stopPropagation();
                    e.preventDefault();
                    scope.candrop = false;
                    scope.$apply();
                }
                function cancelEnterLeaveTimeout() {
                    if (scope.enterLeaveTimeout) {
                        $timeout.cancel(scope.enterLeaveTimeout);
                    }
                }
                function dragEnterLeave(e) {
                    cancelEnterLeaveTimeout();
                    //debouncing applyDragEnterLeave to prevent flickering when hovering element's children
                    scope.enterLeaveTimeout = $timeout(function() {
                        applyDragEnterLeave(e);
                    }, 100);
                }
                element.bind("dragenter", dragEnterLeave);
                element.bind("dragleave", dragEnterLeave);
                element.bind("dragover", function(e) {
                    cancelEnterLeaveTimeout();
                    e.stopPropagation();
                    e.preventDefault();
                    scope.$apply(function(){
                        var evt = e.originalEvent;
                        if (evt.dataTransfer &&
                            evt.dataTransfer.types &&
                            (
                                (evt.dataTransfer.types.indexOf && evt.dataTransfer.types.indexOf('Files') >= 0) ||
                                (evt.dataTransfer.types.contains && evt.dataTransfer.types.contains('Files'))
                            )
                        ) {
                            // feedback
                            scope.candrop = true;
                            var af = evt.dataTransfer.effectAllowed;
                            evt.dataTransfer.dropEffect = ('move' == af || 'linkMove' == af) ? 'move' : 'copy';
                        }
                    });
                });
                element.bind("drop", function(e) {
                    e.stopPropagation();
                    e.preventDefault();
                    scope.$apply(function(){
                        var evt = e.originalEvent;
                        if (evt.dataTransfer &&
                            evt.dataTransfer.types &&
                            (
                                (evt.dataTransfer.types.indexOf && evt.dataTransfer.types.indexOf('Files') >= 0) ||
                                (evt.dataTransfer.types.contains && evt.dataTransfer.types.contains('Files'))
                            ) &&
                            (scope.multiple || evt.dataTransfer.files.length == 1)
                        ){
                            scope.drop({'files': evt.dataTransfer.files});
                            scope.candrop = false;
                        }
                    });
                });
            }
        };
    });

    app.directive('commaSeparatedView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    if (value == null || value.length == 0) return [];
                    return value.split(",");
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return "";
                    return value.join(",");
                });
            }
        };
    });
    app.directive('jsonArrayView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    ngModel.$setValidity('json', true);
                    if (value == null || value.length == 0) return [];
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                         ngModel.$setValidity('json', false);
                         return null;
                    }
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return null;
                    return JSON.stringify(value);
                });
            }
        };
    });
    app.directive('jsonArrayPrettyView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    ngModel.$setValidity('json', true);
                    if (value == null) return [];
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                         ngModel.$setValidity('json', false);
                         return null;
                    }
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return null;
                    return JSON.stringify(value, undefined, 3);
                });
            }
        };
    });

    app.directive('jsonObjectPrettyView', function (Logger){
            return {
                require: 'ngModel',
                link: function(scope, elem, attrs, ngModel) {
                    var el = elem[0];
                    //For DOM -> model transformations
                    ngModel.$parsers.push(function(value) {
                        ngModel.dkuJSONError = null;
                        ngModel.$setValidity('json', true);
                        if (value == null || value.length == 0) return null;
                        try {
                            return JSON.parse(value);
                        } catch (e) {
                            ngModel.$setValidity('json', false);
                            Logger.info("Error while parsing JSON: ", value, e);
                            ngModel.dkuJSONError = e.toString();
                            if ('keepOldIfInvalid' in attrs) {
                                return ngModel.$modelValue;
                            } else {
                                return null;
                            }
                        }
                    });

                    //For model -> DOM transformation
                    ngModel.$formatters.push(function(value) {
                        if (value == undefined) return null;
                        var prevSelStart = el.selectionStart;
                        var prevSelEnd = el.selectionEnd;
                        var prevScroll = el.scrollTop;
                        if(el == document.activeElement) {
                            setTimeout(function() {
                                el.setSelectionRange(prevSelStart,prevSelEnd);
                                el.scrollTop = prevScroll;
                            },0);
                        }
                        return JSON.stringify(value,undefined,3);
                    });

                    if (attrs.deepUpdate) {
                        scope.$watch(attrs.ngModel, () => {
                            try {
                                var formatters = ngModel.$formatters, idx = formatters.length;
                                var viewValue = ngModel.$modelValue;
                                while (idx--) {
                                    viewValue = formatters[idx](viewValue);
                                }
                                if (viewValue != null) {
                                    ngModel.$viewValue = viewValue;
                                    ngModel.$render();
                                }
                            } catch (e) {
                                Logger.info("JSON is invalid, not rendering ...")
                            }
                        }, true);
                    }
                }
            };
    });

    app.directive('jsonObjectView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    ngModel.$setValidity('json', true);
                    if (value == null || value.length == 0) return null;
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                         ngModel.$setValidity('json', false);
                         return null;
                    }
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return null;
                    return JSON.stringify(value);
                });
            }
        };
    });

    app.directive('commaSeparatedIntegerView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    if (value == null) return [];
                    var ret = value.split(",").map(function(x) { return parseInt(x, 10); }).filter(function(x) { return !isNaN(x);});
                    return ret;
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return "";
                    return value.join(",");
                });
            }
        };
    });

    app.directive('commaSeparatedFloatView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    if (value == null) return [];
                    var ret = value.split(",").map(function(x) { return parseFloat(x, 10); }).filter(function(x) { return !isNaN(x);});
                    return ret;
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return "";
                    return value.join(",");
                });
            }
        };
    });


    app.directive('customValidation', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                function apply_validation(value) {
                    ngModel.$setValidity('customValidation', true);
                    let cv = scope.$eval(attrs.customValidation);
                    var valid = cv && cv(value);
                    ngModel.$setValidity('customValidation', valid);
                    return value;
                }

                //For DOM -> model validation
                ngModel.$parsers.push(apply_validation);

                //For model -> DOM validation
                ngModel.$formatters.push(function(value) {
                    apply_validation();
                    return value;
                });
            }
        };
    });

    app.directive('fixedPanes', function($timeout,$rootScope){
        return {
            restrict: 'A',
            link: function(scope, element, attrs){
                scope.showLeftPane = scope.$eval(attrs.showLeftPane) || false;
                scope.showRightPane = scope.$eval(attrs.showRightPane)  || false;

                scope.setShowLeftPane = function(showLeftPane) {
                    if (scope.showLeftPane != showLeftPane) {
                        scope.showLeftPane = showLeftPane;
                        $timeout(function(){
                            scope.$broadcast('resizePane');
                            $rootScope.$broadcast('reflow');
                        }, 250);

                    }
                }
                scope.openLeftPane = function() {
                    scope.setShowLeftPane(true);
                }
                scope.closeLeftPane = function() {
                    scope.setShowLeftPane(false);
                }
                scope.toggleLeftPane = function(){
                    scope.setShowLeftPane(!scope.showLeftPane);
                };


                scope.setShowRightPane = function(showRightPane) {
                    if (scope.showRightPane != showRightPane) {
                        scope.showRightPane = showRightPane;
                        $timeout(function(){
                            scope.$broadcast('resizePane');
                            $rootScope.$broadcast('reflow');
                        }, 250);
                    }
                }
                scope.openRightPane = function() {
                    scope.setShowRightPane(true);
                }
                scope.closeRightPane = function() {
                    scope.setShowRightPane(false);
                }
                scope.toggleRightPane = function(){
                    scope.setShowRightPane(!scope.showRightPane);
                };
            }
        };
    });

    app.directive('watchScroll', function() {
       return {
           restrict : 'A',
            link: function(_scope, element){
                $(element).addClass("watch-scroll");
                function scrollStart() {
                    $(element).addClass("scrolling");
                }
                function scrollEnd() {
                    $(element).removeClass("scrolling");
                }
                var scrolling = false;
                $(element).scroll(function() {
                    if (!scrolling) scrollStart();
                    clearTimeout($.data(this, 'scrollTimer'));
                    $.data(this, 'scrollTimer', setTimeout(function() {
                        scrollEnd();
                    }, 200));
                });
            }
        };
    });

    app.directive('tabs', function($filter, $location, $compile, $timeout, Logger) {
        return {
            restrict: 'E',
            scope: true,
            // replace: true,
            myTemplateNonScrollable: '<div class="tabbable">' +
                '<ul class="nav nav-tabs">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }}" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab">' +
                        '<span class="title"><i ng-show="pane.icon" ng-class="getIconClass(pane.icon)"></i><span fw500-width> {{ pane.title }}</span></span>'+
                        '<br ng-if="pane.subtitle"/><span class="subtitle" ng-if="pane.subtitle" ng-bind-html="pane.subtitle"></span>'+
                        '</a>' +
                    '</li>' +
                '</ul>' +
                '<div class="tab-content" ></div>' +
            '</div>',
            myTemplateScrollable: '<div class="tabbable">' +
                '<div class="scroller scroller-left"><i class="icon-chevron-left"></i></div>'+
                '<div class="scroller scroller-right"><i class="icon-chevron-right"></i></div>'+
                '<div class="tabs-scroll-wrapper">'+
                '<ul class="nav nav-tabs tabs-scroll-zone">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }}" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab"><i ng-show="pane.icon" class="icon-{{ pane.icon }}"></i> {{ pane.title }}</a>' +
                    '</li>' +
                '</ul></div>' +
                '<div class="tab-content" ></div>' +
            '</div>',
            myTemplateNewStyleScrollable: '<div class="tabbable">' +
                '<div class="scroller scroller-left"><i class="icon-chevron-left"></i></div>'+
                '<div class="scroller scroller-right"><i class="icon-chevron-right"></i></div>'+
                '<div class="tabs-scroll-wrapper">'+
                '<ul class="column-header-tabs tabs-scroll-zone">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }} tab" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<span class="title" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab"><i ng-show="pane.icon" class="icon-{{ pane.icon }}"></i> {{ pane.title }}</span>' +
                    '</li>' +
                '</ul></div>' +
                '<div class="tab-content" ></div>' +
            '</div>',
             myTemplateNewStyle: '<div class="">' +
                '<ul class="column-header-tabs" style="margin: 0">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }} tab" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<span class="title" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab"><i ng-show="pane.icon" class="icon-{{ pane.icon }}"></i> {{ pane.title }}</span>' +
                    '</li>' +
                '</ul>' +
                '<div class="tab-content" ></div>' +
            '</div>',
            myTemplateNonScrollableWithNewFeature: '<div class="tabbable">' +
                '<ul class="nav nav-tabs">' +
                    '<li ng-repeat="pane in panes|filter:{visible: true}" class="{{ pane.position }}" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                    	'<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab" ng-if="pane.newInVersionFeature !== undefined">' +
                            '<ng2-new-in-version-popover ' +
                                'class="tabs-new-in-version-popover" ' +
                                'target-version="{{pane.newInVersionTargetVersion}}" ' +
                                'popover-id="{{pane.newInVersionPopoverId}}" ' +
                                'feature-name="{{pane.newInVersionFeature}}" ' +
                                'display-mode="{{pane.newInVersionDisplayMode}}">' +
                                '<div popoverContent>' +
                                    '<p>{{pane.newInVersionContent}}</p>' +
                                '</div>' +
                                '<div popoverTarget>' +
                                    '<span class="title"><i ng-show="pane.icon" ng-class="getIconClass(pane.icon)"></i><span fw500-width> {{ pane.title }}</span></span>'+
                                    '<br ng-if="pane.subtitle"/><span class="subtitle" ng-if="pane.subtitle" ng-bind-html="pane.subtitle"></span>'+
                                '</div>' +
                            '</ng2-new-in-version-popover>' +
                        '</a>' +

                        '<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab" ng-if="pane.newInVersionFeature === undefined">' +
                        '<span class="title"><i ng-show="pane.icon" ng-class="getIconClass(pane.icon)"></i><span fw500-width> {{ pane.title }}</span></span>'+
                        '<br ng-if="pane.subtitle"/><span class="subtitle" ng-if="pane.subtitle" ng-bind-html="pane.subtitle"></span>'+
                        '</a>' +
                    '</li>' +
                '</ul>' +
                '</ul>' +
                '<div class="tab-content" ></div>' +
            '</div>',
                
            compile: function(tElement, attrs){
                // get the panes of the tabs
                var originalContent = $('<div></div>').html(tElement.contents()).contents();
                var el;
                if (attrs.hasOwnProperty("newFeature")) {
                    el = $(this.myTemplateNonScrollableWithNewFeature);
                } else if (attrs.newStyle && attrs.scrollable) {
                    el = $(this.myTemplateNewStyleScrollable);
                } else if (attrs.newStyle) {
                    el = $(this.myTemplateNewStyle);
                } else if (attrs.scrollable) {
                    el = $(this.myTemplateScrollable);
                } else  {
                    el = $(this.myTemplateNonScrollable);
                }
                if (tElement.hasClass('vertical-flex')) {
                    el.addClass('vertical-flex h100');
                    el.children(':not(.tab-content)').addClass('noflex');
                    el.children('.tab-content').addClass('flex');
                }
                tElement.replaceWith(el);
                el.find('.tab-content').append(originalContent);
                return this.link;
            },
            controller: function($scope, $element, $rootScope) {
                var panes = $scope.panes = [];
                $scope.select = function(pane, force) {
                    $scope.$emit("opalsCurrentTab", pane.slug);
                    if (!pane.selected){
                        angular.forEach(panes, function(p) {
                            p.selected = false;
                        });
                        pane.selected = true;
                        $timeout(function() {
                            pane.displayed = true;
                        });
                        if ($scope.onSelect) {
                            $scope.onSelect(pane);
                        }
                        $scope.$emit('paneSelected', pane);
                        $scope.$broadcast('paneSelected', pane);
                        $rootScope.$broadcast("reflow");

                        if (!force){
                            // TEMPORARY TEMPORARY :
                            // Disable updating of the location hash, because
                            //  it breaks ui-router's interaction with browser history.
                            //   - If you are on state A with no hash, transitionTo(B), then back takes you back to A
                            //   - If you are on state A with no hash, then on state B with hash, transitionTo(C)
                            //      - will propagate the hash so you will actually be on C#hash, which we don't want
                            //      - back button will take you back to A, the B#hash state has disappeared from browser
                            //        history
                            $location.hash(pane.slug, true).replace(true);
                        }
                    }
                };

                this.select = $scope.select;
                this.addPane = function(pane) {
                    panes.push(pane);
                    if (panes.length == 1) {
                        $scope.select(pane, true);
                    }
                };

                $scope.$on('tabSelect', function(e, slug){
                    var pane = $filter('filter')(panes, {'slug':slug});
                    if(pane && pane.length){
                       $scope.select(pane[0]);
                    } else {
                        Logger.warn("Failed to select a pane for slug ", slug, " amongst", panes, " filtered", pane);
                    }
                });
                this.verticalFlex = $element.hasClass('vertical-flex');

                $scope.getIconClass = function(icon) {
                    if (icon && icon.startsWith('dku-icon')) {
                        return icon;
                    } else {
                        return 'icon-' + icon;
                    }
                };
            },
            link : ($scope, element, attrs) => {
                $scope.noHashUpdate = "noHashUpdate" in attrs;
                if (attrs.scrollable) {

                    var totalWidth = function(){
                        var itemsWidth = 0;
                        $('.tabs-scroll-zone li', element).each(function(){
                            var itemWidth = $(this).outerWidth();
                            itemsWidth+=itemWidth;
                        });
                        return itemsWidth;
                    };
                    var scrollBarWidths = 46; /* 2x23 */

                    var hiddenWidth = function(){
                        return (
                            ($('.tabs-scroll-wrapper', element).outerWidth()) -
                            totalWidth()-getLeftPosi())
                            -
                            scrollBarWidths;
                    };
                    var getLeftPosi = function(){
                        return $('.tabs-scroll-zone').position().left;
                    };

                    $('.scroller-right', element).click(function() {
                        $('.scroller-left', element).show();
                        $('.scroller-right', element).hide();
                        $('.tabs-scroll-zone', element).animate({left:"+="+hiddenWidth()+"px"});
                    });

                    $(".scroller-left", element).click(function() {
                        $('.scroller-right', element).show();
                        $('.scroller-left', element).hide();
                        $('.tabs-scroll-zone', element).animate({left:"-="+getLeftPosi()+"px"});
                    });
                    $timeout(function(){
                        if (($('.tabs-scroll-wrapper', element).outerWidth()) < totalWidth()) {
                            $('.scroller-right').show();
                        }
                    }, 0);
                }
                if (attrs.paneHeaderStyle) {
                    $scope.paneHeaderStyle = attrs.paneHeaderStyle;
                } else {
                    $scope.paneHeaderStyle = '';
                }
            }
        };
    });

    app.directive('pane', function($filter, $location, $timeout, $compile, $rootScope, $stateParams) {
        var paneTemplate = $compile('<div class="tab-pane" ng-class="{active: struct.selected}"></div>');
        return {
            require: '^tabs',
            restrict: 'E',
            terminal: true,
            scope: true,
            compile: function(tElement){
                // get the content of the pane
                var transcludeFunction = $compile(tElement.contents());

                return function(scope, element, attrs, tabsCtrl){

                    // replace the pane by the paneTemplate
                    paneTemplate(scope, function(clone){
                        element.replaceWith(clone);
                        element = clone;
                    });

                    // append the content of the pane
                    transcludeFunction(scope, function(clone){
                        element.append(clone);
                    });

                    scope.struct = {
                        title: attrs.title,
                        tabClass: attrs.tabClass,
                        subtitle: scope.$eval(attrs.subtitle),
                        slug: attrs.slug || $filter('slugify')(attrs.title),
                        icon: attrs.icon,
                        visible: angular.isUndefined(attrs.visiblePane)?true:scope.$eval(attrs.visiblePane),
                        position: attrs.position,
                        newInVersionFeature: attrs.newInVersionFeature,
                        newInVersionTargetVersion: attrs.newInVersionTargetVersion,
                        newInVersionPopoverId: attrs.newInVersionPopoverId,
                        newInVersionDisplayMode: attrs.newInVersionDisplayMode,
                        newInVersionContent: attrs.newInVersionContent,
                    };

                    element.addClass("tab-" + scope.struct.slug);
                    if (scope.struct.tabClass) {
                        element.addClass(scope.struct.tabClass);
                    }
                    if (tabsCtrl.verticalFlex) { element.addClass('fh'); }

                    attrs.$observe('title', function(val){
                        // If the title attribute is modified
                        scope.struct.title = val;
                    });
                    scope.$watch(attrs.subtitle, (nv) => {
                        // If the title attribute is modified
                        scope.struct.subtitle = nv;
                    });

                    // Removing the title from the element itself, to prevent a tooltip when hovering anywhere over
                    // the content.
                    element.removeAttr('title');
                    // having a pb when combined with ng-repeat
                    $timeout(function(){element.removeAttr('title');}, 10);

                    // register itself
                    tabsCtrl.addPane(scope.struct);
                    if (($location.hash() === scope.struct.slug || $stateParams.tabSelect === scope.struct.slug) && scope.struct.visible){
                        tabsCtrl.select(scope.struct);
                    }

                    attrs.$observe('visiblePane', function(value){
                        scope.struct.visible = angular.isUndefined(value)?true:value=="true";
                    });

                    scope.$watch('struct.selected', (nv) => {
                        if (nv){
                            if (attrs.noResizeHack == null) {
                                // ugly hack, this will trigger a window resize event, thus refreshing flowchart layout
                                // and CodeMirror-alikes
                                window.dispatchEvent(new Event('resize'));
                            }
                            $rootScope.$broadcast("reflow");
                        }
                    });
                };
            },
        };
    });

    app.directive("detectIframeClicks", function() {
        return {
            scope: false,
            restrict: "A",
            link: function(_scope, element){
                var overIFrame = false;
                element.mouseenter(function() {
                    overIFrame = true;
                });
                element.mouseleave(function() {
                    overIFrame = false;
                });
                $(window).blur(function() {
                    if (overIFrame) {
                        $(document).trigger("click");
                    }
                });
            }
        }
    })

    app.directive('sortTable', function($rootScope){
        return {
            scope: true,
            controller: function($scope, $attrs) {
                this.setSort = function(col) {
                    if($scope.sortColumn){
                        $scope.cols[$scope.sortColumn].removeClass('sort-descending').removeClass('sort-ascending');
                    }
                    if ($scope.sortColumn === col) {
                        $scope.sortDescending = !$scope.sortDescending;
                    } else {
                        $scope.sortColumn = col;
                        $scope.sortDescending = false;
                    }
                    this.refresh();
                };

                this.refresh = function(){
                    if($scope.cols[$scope.sortColumn]){
                        if ($scope.sortDescending) {
                            $scope.cols[$scope.sortColumn].addClass("sort-descending");
                        } else {
                            $scope.cols[$scope.sortColumn].addClass("sort-ascending");
                        }
                        if(! $rootScope.$$phase) $scope.$apply();
                    }
                };

                $scope.sortColumn = $attrs.sortColumn;
                $scope.sortDescending = $scope.$eval($attrs.sortDescending) || false;
                if($attrs.sortTable){
                    if ($attrs.sortTable[0] == "-") {
                        $scope.sortDescending = true;
                        $scope.sortColumn = $attrs.sortTable.substring(1);
                    } else {
                        $scope.sortColumn = $attrs.sortTable;
                    }
                }

                $scope.cols = {};
                this.addCol = function(col, element){
                    $scope.cols[col] = element;
                    element.addClass("sortable");
                    if(angular.isUndefined($scope.sortColumn)){
                        this.setSort(col);
                    } else {
                        this.refresh();
                    }
                };
            }
        };
    });

    /* This is an alternative version of sortTable, featuring a two way binding of sortColumn and sortDescending using
        standard angular features rather than playing with attributes.

        Using it, one can easily pass initial sortColumn and sortDescending as variables, and get back all modifications
        performed by the user.

        Note: when this version is used, one should not use sortColumn and sortDescending as parameters of the orderBy angular pipe.
        Variables passed as parameters should be used.

        Example:

            <table sort-table-dyn
                sort-column="myScope.mySortColumn" sort-descending="myScope.mySortDescending">
                    ...
                <tbody>
                    <tr ng-repeat="item in selection.filteredObjects | orderBy:myScope.mySortColumn:myScope.mySortDescending">

        It is similar to sortTable. However, adding double binding to the former implementation seemed would lead to a code
        a lot less understandable, especially if we want to avoid a refactoring of all current sort-table usages.

        Note: when using a complex sort expression such as a getter function or an array of sort expression, ensure you add the `use-complex-sort-expression='true'` to your component.
    */
    app.directive('sortTableDyn', function($rootScope){
        return {
            scope:{
                sortColumn: '=',
                sortDescending: '=',
                useComplexSortExpression: '@'
            },
            controller: function($scope) {
                this.setSort = function(col) {
                    if ($scope.useComplexSortExpression) {
                        if ($scope.sortColumn === col) {
                            $scope.sortDescending = !$scope.sortDescending;
                        } else {
                            $scope.sortColumn = col;
                            $scope.sortDescending = false;
                        }
                        $scope.currentSortColumnName = $scope.sortColumn;
                        $scope.currentSortDescending = $scope.sortDescending;
                    } else {
                        // Initialize the local sort direction with the one from the component.
                        if ($scope.currentSortDescending === undefined) {
                            $scope.currentSortDescending = $scope.sortDescending;
                        }

                        // If we are trying to sort on the same column that we are already sorted with,
                        // inverse the order using `-column` notation instead of reversing the entire order
                        if ($scope.currentSortColumnName === col) {
                            $scope.currentSortDescending = !$scope.currentSortDescending;
                            if ($scope.currentSortDescending) {
                                $scope.sortColumn = `-${col}`;
                            } else {
                                $scope.sortColumn = col;
                            }
                        } else {
                            $scope.sortColumn = col;
                            $scope.currentSortDescending = false;
                        }

                        $scope.currentSortColumnName = col;
                        $scope.sortDescending = false;
                    }
                    this.refresh();
                };

                this.refresh = function(){
                    Object.values($scope.cols).forEach(e => e.removeClass('sort-descending').removeClass('sort-ascending'));
                    if($scope.cols[$scope.currentSortColumnName]){
                        if ($scope.currentSortDescending) {
                            $scope.cols[$scope.currentSortColumnName].addClass("sort-descending");
                        } else {
                            $scope.cols[$scope.currentSortColumnName].addClass("sort-ascending");
                        }
                        if(! $rootScope.$$phase) $scope.$apply();
                    }
                };


                $scope.cols = {};
                this.addCol = function(col, element){
                    $scope.cols[col] = element;
                    element.addClass("sortable");
                    this.refresh();
                };
            }
        };
    });

    app.directive('sortCol', function(){
        return {
            scope: true,
            require: '^sortTable',
            link: function(scope, element, attrs, sortTableCtrl){
                sortTableCtrl.addCol(attrs.sortCol, element);
                element.on('click', function(){
                    sortTableCtrl.setSort(attrs.sortCol);
                });
            }
        };
    });

    app.directive('sortColDyn', function(){
        return {
            scope: true,
            require: '^sortTableDyn',
            link: function(scope, element, attrs, sortTableCtrl){
                sortTableCtrl.addCol(attrs.sortColDyn, element);
                element.on('click', function(){
                    sortTableCtrl.setSort(attrs.sortColDyn);
                });
            }
        };
    });

    app.directive('daterangepicker', function($rootScope){
        return {
            restrict: 'A',
            template : ' <div class="input-append" style="margin-bottom:0px;"><input type="text" style="opacity:0;'
                 +'position:absolute;top:-3000px;left:-3000px"/>'
                 +'<input type="text" class="theInput" ng-disabled="!!disabled"/>',
            scope: {
                startDate: '=',
                endDate: '=',
                opensDirection:'@',
                opens: '@',
                format: '@',
                timePickerIncrement : '@',
                presetsToEndOfDay : '=',
                fieldWidth : '@',
                singleDatePicker: '=',
                onChange: "=?",
                disabled: '<?'
            },
            replace : true,
            link: function(scope, element, attrs){

                var input = element.find('.theInput');
                var picker = undefined;

                if (scope.fieldWidth) {
                    input.width(scope.fieldWidth);
                }

                element.find('input').keydown(function(e){
                    if(e.keyCode==13 && picker) {
                        picker.hide();
                    }
                });

                // Create the date picker if not already done, and only if the format is set
                function init() {
                    if(!picker && scope.format) {
                        var endOfDayDelta = scope.presetsToEndOfDay ? 1 : 0;
                        picker = input.daterangepicker({
                              format:scope.format,
                              timePickerIncrement: scope.timePickerIncrement ? parseInt(scope.timePickerIncrement) : 60,
                              timePicker: scope.format.indexOf('HH')!=-1,
                              opens: attrs.opens || 'right',
                              timePicker12Hour:false,
                              autoApply:true,
                              separator : ' / ',
                              singleDatePicker: !!scope.singleDatePicker,
                              ranges: {
                                 'Today': [moment(), moment().add('days', endOfDayDelta)],
                                 'Yesterday': [moment().subtract('days', 1), moment().subtract('days', 1 - endOfDayDelta)],
                                 'Last 7 Days': [moment().subtract('days', 6), moment().add('days', endOfDayDelta)],
                                 'Last 30 Days': [moment().subtract('days', 29), moment().add('days', endOfDayDelta)],
                                 'This Month': [moment().startOf('month'), moment().endOf('month')],
                                 'Last Month': [moment().subtract('month', 1).startOf('month'), moment().subtract('month', 1).endOf('month')]
                               },
                               locale: {firstDay: 1},
                               opensDirection: attrs.opensDirection || 'down'
                        },changeDate).data('daterangepicker');

                        picker.element.on('hide.daterangepicker', function (ev, picker) {
                            if (picker.element.val().length === 0) {
                                scope.startDate = null;
                                scope.endDate = null;
                                if(!$rootScope.$$phase) {
                                    $rootScope.$digest();
                                }
                            }
                        });

                        // open upwards if no room below and vice versa
                        if (attrs.opensDirection === 'auto') {
                            picker.element.on('show.daterangepicker', function(ev, picker) {
                                const currentDirection = picker.opensDirection;
                                let newDirection = 'down'

                                if (picker.element.offset().top + picker.element.outerHeight() + picker.container.outerHeight() > $(window).height()) {
                                    newDirection = 'up';
                                }

                                // need to close and reopen for change to take effect
                                if (currentDirection !== newDirection) {
                                    picker.opensDirection = newDirection;
                                    picker.hide();
                                    picker.show();
                                }
                            });
                        }
                    }
                    if(picker)
                        return true;

                    return false;
                }

                var insideWatch = false;
                var insideUserCallback = false;

                // Update the scope from the date picker state
                function changeDate() {

                    if(!init() || insideWatch) return;
                    try {
                        insideUserCallback = true;
                        picker.updateFromControl();
                        if(!scope.format) return;
                        scope.startDate = picker.startDate.format(scope.format);
                        scope.endDate = picker.endDate.format(scope.format);
                        if(!$rootScope.$$phase) {
                            $rootScope.$digest();
                        }
                        if (scope.onChange) {
                            scope.onChange();
                        }
                    } finally {
                        insideUserCallback = false;
                    }
                }

                // Update date picker state from scope
                scope.$watch('[startDate,endDate]', (nv) => {
                    if(!init() || insideUserCallback) return;
                    try {
                        insideWatch = true;
                        if (!nv[0] && !nv[1]){
                            picker.element.val("");
                            return;
                        }
                        if(scope.startDate) {
                            picker.setStartDate(moment(scope.startDate,scope.format));
                        }
                        if(scope.endDate) {
                            picker.setEndDate(moment(scope.endDate,scope.format));
                        }
                    } finally {
                        insideWatch = false;
                    }
                },true);

                scope.$watch('format',function(nv, ov) {
                    if(!picker) {
                        init();
                    } else if (nv != ov) {
                        picker.format = nv;
                        picker.timePicker = nv.indexOf('HH')!=-1;
                        picker.timePickerIncrement= scope.timePickerIncrement ? parseInt(scope.timePickerIncrement) : 60;
                        picker.updateInputText();
                        changeDate();
                        picker.updateCalendars();
                    }

                });

                scope.$on('$destroy',function() {
                    if(picker) {
                        picker.remove();
                        picker = undefined;
                    }
                })
            }
        };
    });


    /*
     *   Very similar to dkuHelpPopover, but the template can be inlined
     *   Usage :
     *
     *   <button class="btn btn-small" dku-inline-popover>
     *        <label>
     *            <span class="icon-question">&nbsp;</span>
     *                      Button text
     *        </label>
     *        <content title="Help me">
     *            <h2>Introduction</h2>
     *            <p>Blablabla</p>
     *        </content>
     *   </button>
     *
     */
    app.directive('dkuInlinePopover',function($timeout, $interpolate, $compile) {
        return {
            restrict : 'A',
            scope : true,
            transclude:true,
            template:'',

            compile:function(_element,_attrs,transclude) {
                return function(scope,element,attrs) {
                    let destroyPopover;
                    transclude(scope.$new(), function(clone) {

                        var contentFilter = clone.filter("content");
                        var buttonText = clone.filter('label').contents();
                        var popoverContent = contentFilter.contents();
                        var title = contentFilter.attr("title") ? $interpolate(contentFilter.attr("title"))(scope) : null;
                        let popoverRef = null;

                        // I'VE NO FUCKING IDEA OF WHY IT WORKS
                        // timeout is the universal fix :D
                        $timeout(function() {element.append(buttonText);});
                        var shown = false;
                        var options = {
                                html: true,
                                content: popoverContent,
                                placement: (attrs.placement in scope) ? scope[attrs.placement]  : (attrs.placement || 'right'),
                                container: attrs.container?attrs.container:undefined,
                                title: title
                        };

                        if (attrs.on === "hover") {
                            options.animation = false;
                            element.popover(options).on("mouseenter", function () {
                                $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                $(this).popover("show");
                                if (attrs.popoverClass) {
                                    $timeout(() => {
                                        const popover = element.data('popover').$tip;
                                        popoverRef = popover;
                                        popover.addClass(attrs.popoverClass);
                                    });
                                }
                            }).on("mouseleave", function () {
                                $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                popoverRef = null;
                                $(this).popover("hide");
                            });
                        } else {  // click
                            element.popover(options);
                            function show() {
                                shown = true;
                                //capturing jquery destroy popover function for the current element, otherwise the method is not available anymore on $destroy of the directive (certainly because the elements got already removed from the DOM at that point)
                                destroyPopover = (element.data('popover')['destroy']).bind(element.data('popover'));
                                window.setTimeout(function() {
                                    $("html").click(hide);
                                    const popover = element.data('popover').$tip;
                                    if (attrs.clickable) popover.click(stopPropagation);
                                    if (attrs.popoverClass) {
                                        popover.addClass(attrs.popoverClass);

                                        $compile(popoverContent)(scope);
                                        // re-show to get appropriate sizing
                                        element.popover('show');
                                    }
                                }, 0);
                            }

                            function stopPropagation($event) {
                                $event.stopPropagation();
                            }

                            function hide() {
                                shown = false;
                                element.popover('hide');
                                $("html").unbind("click", hide);
                                const popover = element.data('popover').$tip;
                                popover.unbind('click', stopPropagation);
                                popover.hide();
                            }

                            element.click(function() {
                                if(shown) {
                                    hide();
                                    $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                } else {
                                    show();
                                    $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                }
                            });
                        }

                        $(document).on("keydown.dkuInlinePopover", function(e) {
                            if(e.key === "Escape"){
                                destroy();
                            }
                        });

                        scope.uiRefreshToggle = false;

                        scope.$on("$destroy", function() {
                            destroy();
                        });

                        function destroy(){
                            if (typeof destroyPopover === "function") {
                                destroyPopover();
                            }
                            element.popover('destroy');
                            if (popoverRef) {
                                popoverRef.remove();
                                popoverRef = null;
                            }
                            $(document).off("keydown.dkuInlinePopover");
                        }
                    });
                };
            }
        };
    });

    /**
     * Very similar to dkuHelpPopover, but the template is a Markdown string
     *   Usage :
     *
     *   <button class="btn btn-small" dku-md-popover="# Yeah\n* Markdown" title="popover title">
     *        <label>
     *            <span class="icon-question">&nbsp;</span>
     *                      Button text
     *        </label>
     *        <content title="Help me">
     *            <h2>Introduction</h2>
     *            <p>Blablabla</p>
     *        </content>
     *   </button>
     *
     */
    app.directive('dkuMdPopover',function(MarkedSettingService, translate) {
        return {
            restrict : 'A',
            link : function($scope, element, attrs) {
                var shown = false;
                let destroyPopover;

                const getTitle = () => attrs.dkuMdTitle || translate("GLOBAL.HELP", "Help")

                var hide = function() {
                    $("html").unbind("click", hide);
                    element.popover('hide');
                    shown=false;
                };

                var show = function() {
                    shown = true;
                    //capturing jquery destroy popover function for the current element, otherwise the method is not available anymore on $destroy of the directive (certainly because the elements got already removed from the DOM at that point)
                    destroyPopover = (element.data('popover')['destroy']).bind(element.data('popover'));

                    marked.setOptions(MarkedSettingService.get($scope, attrs));
                    var contentElt = marked(attrs.dkuMdPopover);
                    var ret = $("<div class=\"" + (attrs.popoverClazz||"") + "\"></div>");
                    ret.html(contentElt);
                    element.popover('show');
                    var popover = element.data('popover');
                    $(popover.$tip)
                        .find('.popover-content')
                        .empty().append(ret)
                        .off("click.dku-pop-over")
                        .on("click.dku-pop-over", function(e) {
                            e.stopPropagation();
                        });
                    element.popover('show');

                    // In case the title has changed since the popup was initially built, update it.
                    let currentTitle = getTitle();
                    if (currentTitle !== options.title) {
                        $(popover.$tip).find('.popover-title').html(currentTitle);
                    }

                    window.setTimeout(function() { $("html").click(hide); }, 0);
                };
                var placement = element.data("placement") || "right";
                var options = {
                    html: true,
                    content: "",
                    placement: placement,
                    title: getTitle()
                };
                var container = element.data("container") || "body";
                if (container) {
                    options.container = container;
                }

                element.popover(options);
                element.click(function() {
                    if(shown) {
                        hide();
                    } else {
                        show();
                    }
                });

                $scope.$on('$destroy', function() {
                    if (typeof destroyPopover === "function") {
                        destroyPopover();
                    }
                });
            }
        };
    });

    /*
     * Usage :
     * bl br tl tr
     *
     *   <button class="btn btn-small" dku-inline-popup position="bl">
     *        <label>
     *            <span class="icon-question">&nbsp;</span>
     *                      Button text
     *        </label>
     *        <content title="Help me">
     *            <h2>Introduction</h2>
     *            <p>Blablabla</p>
     *        </content>
     *   </button>
     *
     * The popup content is created lazily. It is then only hidden. It is removed when
     * parent is destroyed
     */
    app.directive('dkuInlinePopup',function($timeout) {
        return {
            restrict : 'A',
            scope : true,
            transclude:true,
            template:'',
            compile:function(_element,_attrs,transclude) {
                return function(scope, element) {
                    transclude(scope.$new(), function(clone) {
                        var state = { popupElement : null};
                        var shown = false;
                        var buttonText = clone.filter('label').contents();
                        var popupContent = clone.filter('content').contents();
                        var addClazz = clone.filter('content').attr('class').split(/\s+/);


                        $timeout(function() {element.append(buttonText);});

                        var hide = function() {
                            if (state.popupElement) {
                                state.popupElement.hide();
                            }
                            $("html").unbind("click", hide);
                            shown=false;
                        };
                        var show = function() {
                            shown = true;
                            if (state.popupElement ==null) {
                                state.popupElement = $("<div class='dku-inline-popup' />");
                                $.each(addClazz, function(idx, val) { state.popupElement.addClass(val)});
                                state.popupElement.append(popupContent);
                                $("body").append(state.popupElement);
                            }
                            state.popupElement.css("overflow", "scroll");
                            state.popupElement.css("position", "absolute");
                            state.popupElement.css("left", element.offset().left);
                            var windowh = window.innerHeight || document.documentElement.clientHeight || document.body.clientHeight;
                            var etop = element.offset().top + element.outerHeight();
                            state.popupElement.css("top", etop);
                            state.popupElement.css("max-height", windowh - etop);

                            state.popupElement.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                                e.stopPropagation();
                            });
                            window.setTimeout(function() { $("html").click(hide) }, 0);

                            state.popupElement.show()
                        };
                        scope.hide = function(){
                            hide();
                        }

                        element.click(function(){
                            if (shown) hide();
                            else show();
                        })
                        scope.$on("$destroy", function() {
                            hide();
                            if (state.popupElement) {
                                state.popupElement.remove();
                                state.popupElement = null;
                            }
                        });
                    });
                };
            }
        };
    });


    app.directive('editableSlider', function($parse, $compile) {
        return {
            restrict : 'A',
            link : function($scope, element, attrs) {
                var tpl = $compile('<div class="ngrs-value-runner">'
                    +'    <div class="ngrs-value ngrs-value-min" style="max-width: 70px;">'
                    +'      <input ng-show="sliderEditManual1" select-all auto-focus="{{sliderEditManual1}}" step="0.1" ng-blur="sliderEditManual1=false;'+attrs.onHandleUp+'()" ng-type="number" next-on-enter blur-model="'+attrs.modelMin+'" ng-disabled="disableEditable" id="qa_editable_slider_min_input" />'
                    +'      <div ng-hide="sliderEditManual1"><a ng-click="sliderEditManual1=true" style="color: inherit; overflow: hidden; text-overflow: ellipsis;" title="{{'+attrs.modelMin+'}}" id="qa_editable_slider_min_pointer">{{'+attrs.modelMin+'}}</a></div>'
                    +' </div>'
                    +' <div class="ngrs-value ngrs-value-max" style="max-width: 70px;">'
                    +'    <input ng-show="sliderEditManual2" select-all auto-focus="{{sliderEditManual2}}" step="0.1" ng-blur="sliderEditManual2=false;'+attrs.onHandleUp+'()" ng-type="number" next-on-enter blur-model="'+attrs.modelMax+'" ng-disabled="disableEditable" id="qa_editable_slider_max_input"/>'
                    +'    <div ng-hide="sliderEditManual2"><a ng-click="sliderEditManual2=true" style="color: inherit; overflow: hidden; text-overflow: ellipsis;" title="{{'+attrs.modelMax+'}}" id="qa_editable_slider_max_pointer">{{'+attrs.modelMax+'}}</a></div>'
                    +'   </div>'
                    +'</div>');

                $scope.sliderEditManual1 = false;
                $scope.sliderEditManual2 = false;

                $scope.disableEditable = false;
                $scope.disableEditableAsString = attrs.disableEditableAsString;
                if ($scope.disableEditableAsString) {
                    $scope.disableEditable = $scope.$eval($scope.disableEditableAsString);
                    $scope.$watch($scope.disableEditableAsString, (nv) => {
                        $scope.disableEditable = nv;
                    });
                }

                attrs.$set('showValues','false');
                element.append(tpl($scope));
            }
        }
    });

    app.directive('dkuHelpPopover', function($parse, $compile, $http, $timeout, $q, $templateCache) {
        return {
            restrict : 'A',
            scope:true,
            controller: function($scope){
                this.dismissPopover = function() {
                    $scope.dismissPopover();
                }
                this.togglePopover = function() {
                    $scope.togglePopover();
                }
            },
            link : function($scope, element, attrs) {

                // By default, a bootstrap popover is not displayed if it has no title and no content
                // We need to create such popover in some cases!
                function execInPatchedEnv(fn) {
                    var old = $.fn.popover.Constructor.prototype.hasContent;
                    $.fn.popover.Constructor.prototype.hasContent = function() {
                        return true;
                    };
                    try {
                        fn();
                    } finally {
                        $.fn.popover.Constructor.prototype.hasContent = old;
                    }
                }

                var shown = false;
                var getter = $parse(attrs.dkuHelpPopover);
                var templateUrl = getter($scope);

                var hide = function() {
                    $("html").unbind("click", blur);
                    element.popover('hide');
                    shown=false;
                };

                //selector for elements that do not create blur/hide event
                //(all children will block hide event)
                var noBlur = [];
                if (attrs.noBlur) {
                    noBlur = $scope.$eval(attrs.noBlur);
                }
                var blur = function(evt) {
                    var ignore = false;
                    $.each(noBlur, function(_, selector) {
                        if($(evt.target).is(selector) || $(evt.target).parents(selector).length) {
                            ignore = true;
                        }
                    });
                    ignore || hide();
                }

                $scope.dismissPopover = function(){
                    hide();
                }

                var theTip;

                var show = function() {

                    shown = true;
                    $http.get(templateUrl, {cache: $templateCache}).then(function(response) {

                        var tmplData = $('<div/>').html(response.data).contents();
                        var tmpl = $compile(tmplData);
                        var popover = element.data('popover');
                        var html = tmpl($scope);
                        execInPatchedEnv(function() {
                            element.popover('show');
                            $(popover.$tip)
                                .find('.popover-content')
                                .html(html)
                                .off("click.dku-pop-over")
                                .off("click.dku-pop-over")
                                .off("click",".dropdown-menu")
                                .on("click.dku-pop-over", function(e) {
                                    e.stopPropagation();
                            });
                            theTip = popover.$tip;
                            element.popover('show');

                            if (attrs.noArrow) {
                                $(popover.$tip).find(".arrow").remove();
                                $(popover.$tip).css("top", $(popover.$tip).offset().top-20);
                            }
                            if (attrs.contentClazz) {
                                $(popover.$tip).find('.popover-content').addClass(attrs.contentClazz);
                            }
                            if(attrs.titleClazz) {
                                $(popover.$tip).find('.popover-title').addClass(attrs.titleClazz);
                            }
                            if (attrs.forceTopPositive) {
                                var currentTop = $(popover.$tip).css('top');
                                if (currentTop.charAt(0) === '-') {
                                    $(popover.$tip).css('top', '0px');
                                    // shift arrow accordingly
                                    $(popover.$tip).find('.arrow').css('transform', 'translateY(' + currentTop + ')');
                                }
                            }
                        });
                        window.setTimeout(function() { $("html").click(blur); }, 0);

                    });
                };

                var toggle = function() {
                    if (shown) {
                        hide();
                    }
                    else {
                        show();
                    }
                };

                $scope.togglePopover = function(){
                    return toggle();
                }

                $scope.showPopover = function(){
                    return show();
                }

                $scope.hidePopover = function(){
                    return hide();
                }

                var placement = element.data("placement") || "right";

                var options = {
                    html: true,
                    content: "",
                    placement: placement,
                    title: element.data("title")
                };
                var container = element.data("container");
                if (container) {
                    options.container = container;
                }
                element.popover(options);
                element.click(toggle);
                element[0].showPopover = function() {
                    if (!shown) {
                        show();
                    }
                };

                $scope.$on('$destroy', function() {
                       if(theTip && theTip.remove) {
                            theTip.remove();
                       }
                });

                element[0].hidePopover = function() {
                    if (shown) {
                        hide();
                    }
                };


            }
        };
    });

    app.directive('bzGauge', function() {
        return {
            restrict: 'E',
            template: "<div class='bz-gauge'><div class='mercury' style='width: {{ gaugeWidth }}%; background-color: {{ color }};'></div>",
            replace: true,
            scope: {
                color: "=color",
                val: "=val",
                total: "=total"
            },
            link: function(scope) {
                scope.gaugeWidth = (100.* scope.val / scope.total) | 0;
            }
        }
    })

    var debouncer = function(f, delay) {
      var delayer = null;
      return function() {
        if (delayer === null) {
          f();
          delayer = setTimeout(function() {
            delayer = null;
          }, delay)
        }
      }
    }

    app.directive('editableLabel', function() {
      return {
            restrict: 'E',
            template: "<div class='editable-label'><label>{{ val }}</label><input></input></div>",
            replace: true,
            scope: {
                val: '=ngModel',
            },
            link: function(scope, element) {
                var $label = element.find("label");
                var $input = element.find("input");

                var isEdit = false;

                var enterEdition = function() {
                  isEdit = true;
                  element.addClass("edit");
                  $input.focus();
                  $input.val(scope.val);
                }

                scope.validate = function(val) {
                  var trimmed = val.trim();
                  return (trimmed.length > 0);
                }

                var quitEdition = function() {
                  if (!isEdit) {
                    return;
                  }
                  var candidate = $input.val();
                  if (scope.validate(candidate)) {
                    scope.val = candidate.trim();
                    scope.$apply();
                    element.removeClass("edit");
                  }
                  isEdit = false;
                }

                var toggleEdition = function() {
                  if (isEdit) {
                    quitEdition();
                  }
                  else {
                    enterEdition();
                  }
                }

                toggleEdition = debouncer(toggleEdition, 400);
                $input.blur(toggleEdition);
                $input.change(toggleEdition);
                $label.click(toggleEdition);

            }
        }
    });


    app.directive('editableText', function() {
      return {
            restrict: 'E',
            template: "<div class='editable-text'><div>{{ message() }}</div><textarea></textarea></div>",
            replace: true,
            scope: {
                emptyMessage: '@emptyMessage',
                val: '=ngModel',
            },
            link: function(scope, element) {
                var $input = element.find("textarea");

                var isEdit = false;

                if (scope.val === undefined) {
                  scope.val = "";
                }
                scope.message = function() {
                  if (scope.val && (scope.val.trim().length > 0)) {
                    return scope.val;
                  }
                  else {
                    return scope.emptyMessage;
                  }
                }

                var enterEdition = function() {
                  isEdit = true;
                  element.addClass("edit");
                  $input.focus();
                  $input.val(scope.val);
                }

                scope.validate = function(val) {
                  var trimmed = val.trim();
                  return (trimmed.length > 0);
                }

                var quitEdition = function() {
                  if (!isEdit) {
                    return;
                  }
                  var candidate = $input.val();
                  if (scope.validate(candidate)) {
                    scope.val = candidate.trim();
                    scope.$apply();
                    element.removeClass("edit");
                  }
                  isEdit = false;
                }

                var toggleEdition = function() {
                  if (isEdit) {
                    quitEdition();
                  }
                  else {
                    enterEdition();
                  }
                }

                toggleEdition = debouncer(toggleEdition, 100);

                $input.blur(toggleEdition);
                $input.change(toggleEdition);
                element.click(toggleEdition);

            }
        }
    });

    app.directive('multiSelect', function(){
        return {
            require: 'ngModel',
            scope: true,
            link: function(scope, _element, _attrs, ngModel){
                scope.selectedItems = [];
                scope.allToggled = false;
                scope.someToggled = false;

                scope.toggleItem = function(item){
                    if (scope.selectedItems.indexOf(item) < 0) {
                        scope.selectedItems.push(item);
                    } else {
                        scope.selectedItems.splice(scope.selectedItems.indexOf(item), 1);
                    }
                    scope.allToggled = scope.selectedItems.length === ngModel.$viewValue.length;
                    scope.someToggled = scope.selectedItems.length > 0 && !scope.allToggled;

                    item.selected = !item.selected;
                };
                scope.toggleAll = function(){
                    var selected;
                    if(!scope.allToggled){
                        scope.selectedItems = angular.copy(ngModel.$viewValue);
                        selected = true;
                    } else {
                        scope.selectedItems = [];
                        selected = false;
                    }
                    scope.allToggled = scope.selectedItems.length === ngModel.$viewValue.length;
                    scope.someToggled = scope.selectedItems.length > 0 && !scope.allToggled;

                    angular.forEach(ngModel.$viewValue, function(item){
                        item.selected = selected;
                    });
                };
                scope.$on('clearMultiSelect', function(){
                    scope.selectedItems = [];
                    angular.forEach(ngModel.$viewValue, function(item){
                        item.selected = false;
                    });
                    scope.allToggled = false;
                    scope.someToggled = false;
                });
            }
        };
    });


    app.directive('modal', function($window){
        // This directive ensure the proper height of the modals
        return {
            restrict: 'C',
            link: function(scope, element, attrs){
                if (attrs.autoSize == "false") return;

                var content = element.find('.modal-body');

                // body height
                let oldMinHeight = content.css('minHeight');
                content.css('height', 0); //to get the padding height
                content.css('minHeight', 0);
                var paddingHeight = content.innerHeight();
                content.css('height', '');
                content.css('minHeight', oldMinHeight);


                function findOverflown(node){
                    if (node === undefined)
                        return [];
                    var overflownNodes = [];
                    for (var i = node.childNodes.length - 1; i >= 0; i--) {
                        var child = node.childNodes[i];
                        if (child.offsetHeight > 0){
                            var overflowCss = $(child).css('overflow');
                            var overflowYCss = $(child).css('overflow-y');
                            var scrollValues = ['auto', 'scroll'];
                            /* global contains */
                            if (contains(scrollValues, overflowCss) || contains(scrollValues, overflowYCss)) {
                                // Special code mirror cases
                                if(!$(child).hasClass('CodeMirror-hscrollbar') && !$(child).hasClass('CodeMirror-vscrollbar')
                                    && !$(child).hasClass('CodeMirror-scroll')) {
                                    overflownNodes.push(child);
                                }
                            } else {
                                overflownNodes = overflownNodes.concat(findOverflown(child));
                            }
                        }
                    }
                    return overflownNodes;
                }

                var sizeModal = function(){
                    // sometimes the modal-body is not instantiated at load, get it (again) here. If the
                    // modal-body was not there at modal creation time, then padding might be crappy
                    var content = element.find('.modal-body');
                    if (content.hasClass('modal-no-sizing')) return;

                    var oldMinHeight = content.css('minHeight');
                    content.css('height', '');
                    content.css('minHeight', '');
                    // find overflown elements
                    // We maximize the height of overflown content so they'll be the ones with the scrollbar
                    // var overflown = content.find('*').filter(function () {
//                         // select only the overflown content visible (positive height)
//                         return ['auto', 'scroll'].indexOf($(this).css('overflow')) >= 0 && this.offsetHeight > 0;
//                     });
//
                    var overflown = $(findOverflown(content[0]));

                    // remember current scroll position
                    var scrolls = overflown.map(function(){return this.scrollTop;});

                    overflown.css({'maxHeight': 'none', height: 0});
                    // height of non overflown content
                    var nonOverflownHeight = content.innerHeight();

                    overflown.height('');
                    var newHeight;
                    if (element.innerHeight() > $($window).height()) {
                        newHeight = $($window).height() - element.find('.modal-header').innerHeight() - element.find('.modal-footer').innerHeight() - paddingHeight - 10*2;
                    } else {
                        newHeight = content.innerHeight() - paddingHeight;
                    }
                    // preventing borders to be blurry : since modals are going to be centered on the screen, if the
                    // window height ends up being odd, then everything is going to be 1/2 pixel misaligned, and
                    // borders become all blurry messes... So we fix the modal-body height to be even. The rest of
                    // the window height is header+footer, so you have to set their size(s) to obtain a final even height.
                    if ( element.innerHeight() % 2 == 1 ) {
                        var maxHeight = parseInt(content.css('maxHeight'));
                        if ( newHeight + 1 > maxHeight) {
                            content.css('height', newHeight - 1);
                        } else {
                            content.css('height', newHeight + 1);
                        }
                    }

                    if(overflown.length){
                        // dispatch the remaining height between overflown content
                        var heightPerOverflown = (content.innerHeight() - nonOverflownHeight) / overflown.length;
                        if (heightPerOverflown > 0) {
                            overflown.height(heightPerOverflown);
                        }
                        // is the focused element within the modal ?
                        if(! $(document.activeElement).parents(element).length){
                            // focus overflow to allow scroll
                            overflown.attr('tabindex', 0).focus();
                        }
                        overflown.each(function(i){
                            // preserve former scroll
                            $(this).scrollTop(scrolls[i]);
                        });
                    }
                    content.css('minHeight', oldMinHeight);
                };

                // resize when window change
                $($window).on('resize.modal', sizeModal);
                // resize when something change (tab change, form expand...)
                scope.$watch(sizeModal, null);
                // init resize
                sizeModal();


                scope.$on('$destroy', function(){
                    $($window).off('resize.modal');
                });

                // focus first input
                element.find('input').first().focus();
            }
        };
    });

    app.directive('bsTypeahead', () => {
        return {
            priority: 100,
            link: function(scope,element, attr){
                var typeahead = element.data('typeahead');

                // override show function
                typeahead.show = function () {
                    var pos = $.extend({}, this.$element.offset(), {
                        height: this.$element[0].offsetHeight
                    });

                    $(document.body).append(this.$menu);
                    this.$menu.addClass(attr.class);
                    this.$menu.css({
                        position: 'absolute',
                        top: pos.top + pos.height,
                        left: pos.left,
                        'z-index': 5000
                    }).show();

                    this.shown = true;
                    return this;
                };
            }
        };
    });

    /**
     * select displaying columns of a provided schema together with their types
     * If the `$$groupKey` is present on the columns items, then the columns will be displayed under groups given the key,
     * Otherwise they will be just displayed in the same order as they appear in the columns array
     *
     * @param {Array} columns - Array containing the columns with their `name`, `type` and optional `$$groupKey` fields
     * @param {Object} selectedColumn - NgModel representing the selected column
     * @param {boolean} disableTypes - If true, no types will be displayed by the names
     * @param {boolean} immediateRefresh - If true, the column list will be updated as soon as there is any change on the schema
     *
     */
    app.directive('columnSelect', ($filter) => {
        return {
            restrict:'E',
            scope: {
                selectedColumn: '=ngModel',
                columns: '=',
                disableTypes: '=',
                immediateRefresh: '='
            },
            template: '<select '+
                    ' dku-bs-select="{\'liveSearch\' : true}" immediate-refresh="{{immediateRefresh? immediateRefresh: false}}"  '+
                    ' ng-model="uiState.selectedColumn"'+
                    ' class="qa_recipe_split-select-column"'+
                    ' ng-options="column.name as column.name group by column.$$groupKey for column in columns"'+
                    ' options-annotations="filteredTypes" '+
                    ' />',
            link: {
                pre: function(scope) {
                    //compute types in prelink function so that it is available to optionsAnnotations
                    function computeTypes() {
                        let typeToName = $filter("columnTypeToName");
                        scope.filteredTypes = scope.disableTypes ? [] : scope.columns.map(function(column){ return typeToName(column.type)});
                    }
                    computeTypes();
                    scope.uiState = {selectedColumn: scope.selectedColumn};
                    scope.$watch("columns", computeTypes, true);
                    scope.$watch("uiState.selectedColumn", function(){
                        scope.selectedColumn = scope.uiState.selectedColumn;
                    });
                    scope.$watch("selectedColumn", function(){
                        scope.uiState.selectedColumn = scope.selectedColumn;
                    });
                }
            }
        };
    });

    //select diplaying columns of a provided schema together with their types with provided filter
    app.directive('columnSelectWithFilter', () => {
        return {
            restrict:'E',
            scope: {
                selectedColumn: '=ngModel',
                columns: '=',
                disableTypes: '=',
                filterFn: '='
            },
            template: '<select '+
                    ' dku-bs-select="{\'liveSearch\' : true}" '+
                    ' ng-model="uiState.selectedColumn"'+
                    ' class="qa_recipe_split-select-column"'+
                    ' ng-options="column.name as column.name for column in columns | filter: filterFn"'+
                    ' options-annotations="types" '+
                    ' />',
            link: {
                pre: function(scope) {
                    //compute types in prelink function so that it is available to optionsAnnotations
                    function computeTypes() {
                        scope.types = scope.disableTypes ? [] : scope.columns.filter(scope.filterFn).map(function(column){ return column.type});
                    }
                    computeTypes();
                    scope.uiState = {selectedColumn: scope.selectedColumn};
                    scope.$watch("columns", computeTypes, true);
                    scope.$watch("uiState.selectedColumn", function(){
                        scope.selectedColumn = scope.uiState.selectedColumn;
                    });
                    scope.$watch("selectedColumn", function(){
                        scope.uiState.selectedColumn = scope.selectedColumn;
                    });
                }
            }
        };
    });

    app.factory('ColumnGeneratorService', function() {
        function getIconFromType(type) {
            switch (type) {
                case "NUMERIC":
                    return "#";
                case "CATEGORY":
                    return "<span class='icon icon-font'></span>"
                case "TEXT":
                    return "<span class='icon-italic'></span>"
                case "VECTOR":
                    return "<span style='font-size: 14px'>[ ]</span>"
                default:
                    return "";
            }
        }
        function getHTMLContent(name, perFeature) {
            const itemElement = $('<div class="ml-col-select__item">');
            const iconWrapper = $('<div class="ml-col-select__icon-wrapper">');
            iconWrapper.append(getIconFromType(perFeature[name].type));
            itemElement.prop('title', name);
            itemElement.append(iconWrapper);
            itemElement.append(document.createTextNode(' ' + name));
            return itemElement.prop('outerHTML');
        }
        return {
            getPossibleColumns: function(columns, authorizedTypes, authorizedRoles, perFeature) {
                return columns.filter(x => {
                    const isTypeAuthorized = !authorizedTypes || authorizedTypes.includes(perFeature[x].type);
                    const isRoleAuthorized = !authorizedRoles || authorizedRoles.includes(perFeature[x].role);
                    return isTypeAuthorized && isRoleAuthorized;
                }).sort().map(v => {
                    return {
                        name: v,
                        html: getHTMLContent(v, perFeature)
                    }
                });
            }
        }
    })

    app.directive('mlColumnSelectWithType', function(ColumnGeneratorService) {
        return {
            restrict: 'E',
            scope: {
                perFeature: "=",
                selectedColumn: "=ngModel",
                // authorizedTypes is optional. If not specified, all types are authorized. Otherwise,
                // must be an array of authorized types (e.g ["CATEGORY", "NUMERIC"])
                authorizedTypes: "=",
                // authorizedRoles is optional. If not specified, all roles are authorized. Otherwise,
                // must be an array of authorized roles (e.g ["INPUT", "REJECT"])
                authorizedRoles: "=",
                // alreadyComputedColumns is optional. If specified, must be a Set.
                alreadyComputedColumns: "=",
            },
            template: '<select ng-model="selectedColumn"' +
                      '        dku-bs-select="{\'liveSearch\':true}"' +
                      '        options-annotations="columnsAnnotations">' +
                      '    <option ng-repeat="c in columns"' +
                      '            value="{{c.name}}"' +
                      '            data-content="{{c.html}}">' +
                      '            {{c.name}}' +
                      '    </option>' +
                      '</select>',
            link: ($scope) => {
                function setUpColumns() {
                    $scope.columns = ColumnGeneratorService.getPossibleColumns(Object.keys($scope.perFeature), $scope.authorizedTypes, $scope.authorizedRoles, $scope.perFeature).map(col => {
                        col.isComputed = ($scope.alreadyComputedColumns && $scope.alreadyComputedColumns.has(col.name));
                        return col
                    })

                    $scope.columnsAnnotations = $scope.columns.map(v => v.isComputed ? 'already computed' : '');
                }

                $scope.$watch("perFeature", function(nv) {
                    if (nv !== undefined) {
                        setUpColumns();
                    }
                });

                $scope.$watch("alreadyComputedColumns", function(nv) {
                    if (nv !== undefined) {
                        setUpColumns();
                    }
                });

            }
        }
    });


    app.directive('mappingEditor',function(Debounce, $timeout) {
        return {
            restrict:'E',
            scope: {
                mapping: '=ngModel',
                onChange: '&',
                noChangeOnAdd: '<',
                addLabel: '@',
                validate: '=?',
                colors: '=?',
                withColor: '=?',
                keepInvalid: '=?',
                required: '<',
                requiredKey: '<?',
                requiredValue: '<?',
                uniqueKeys: '<?',
                typeAhead: '=',
                keyPlaceholder: '@',
                valuePlaceholder: '@'
            },
            templateUrl : '/templates/shaker/mappingeditor.html',
            compile: () => ({
                pre: function (scope, element, attrs) {
                    const textarea = element.find('textarea');
                    textarea.on('keydown', function (e) {
                        let keyCode = e.keyCode || e.which;
                        //tab key
                        if (keyCode === 9) {
                            e.preventDefault();
                            if (!scope.$$phase) scope.$apply(function () {
                                let tabPosition = textarea[0].selectionStart;
                                scope.bulkMapping = scope.bulkMapping.slice(0, tabPosition) + '\t' + scope.bulkMapping.slice(tabPosition);
                                $timeout(function () {
                                    textarea[0].selectionEnd = tabPosition + 1;
                                });
                            });
                        }
                    });
                    scope.changeMode = function () {
                        if (!scope.showBulk) {
                            scope.bulkMapping = scope.mapping.map(m => (m.from === undefined ? '' : m.from) + '\t' + (m.to === undefined ? '' : m.to)).join('\n');
                        }
                        scope.showBulk = !scope.showBulk;
                    };

                    scope.$watch('bulkMapping', Debounce().withDelay(400, 400).wrap((nv) => {
                            if (!angular.isUndefined(nv)) {
                                if (!nv.length) {
                                    scope.mapping = [];
                                } else {
                                    scope.mapping = nv.split('\n').map((l, index) => {
                                        const color = scope.mapping && scope.mapping[index] && scope.mapping[index].color ? scope.mapping[index].color : 'grey';
                                        //regexp to split into no more than 2 parts (everything to the right of a tab is one piece)
                                        const parts = l.split(/\t(.*)/);
                                        return {from: parts[0], to: parts[1], color: color};
                                    });
                                }
                            }
                        })
                    );
                    if (angular.isUndefined(scope.mapping)) {
                        scope.mapping = [];
                    }
                    if (!scope.addLabel) scope.addLabel = 'Add another';



                    if ('preAdd' in attrs) {
                        scope.preAdd = scope.$parent.$eval(attrs.preAdd);
                    } else {
                        scope.preAdd = Object.keys(scope.mapping).length === 0;
                    }
                    if (scope.onChange) {
                        scope.callback = scope.onChange.bind(scope, {mapping: scope.mapping});
                    }
                }
            })

        };
    });


    /**
     * Simple form form a sampling edition with no filters.
     * Supports changing dataset on the fly
     * Does not support auto refresh mechanism.
     */
    app.directive("samplingForm", function(DataikuAPI, $stateParams, DatasetInfoCache, DatasetUtils, SamplingData, translate) {
        return {
            scope : {
                selection : '=',
                datasetSmartName : '<',
                backendType : '=',
                label : '=',
                disabled: '=',
                helpMessage: '@',
                showPartitionsSelector: '=',
            },
            templateUrl : '/templates/widgets/sampling-form.html',
            link : function($scope) {
                $scope.translate = translate;
                $scope.streamSamplingMethods = SamplingData.streamSamplingMethods;
                $scope.streamSamplingMethodsDesc = SamplingData.streamSamplingMethodsDesc;

                $scope.getPartitionsList = function () {
                    return DataikuAPI.datasets.listPartitions($scope.dataset).error(setErrorInScope.bind($scope))
                        .then(function (ret) {
                            return ret.data;
                        });
                };

                $scope.$watch("datasetSmartName",(nv) => {
                    if (nv) {
                        var loc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.datasetSmartName);
                        var promise = DatasetInfoCache.getSimple(loc.projectKey, loc.name)
                        promise.then(function(data){
                            $scope.dataset = data;
                            if ($scope.dataset.partitioning.dimensions.length == 0){
                                $scope.selection.partitionSelectionMethod = "ALL";
                            }
                            $scope.$broadcast("datasetChange")
                        });
                    }
                });
            }
        }
});



    app.component("changePartitionedEnabledModal", {
        bindings: {
            modalControl: '<',
            settingsLost: '<'
        },
        templateUrl: "/templates/analysis/prediction/change-partitioned-enabled-modal.html",
        controller: function changePartitionedEnabledModalController() {
            const $ctrl = this;
            $ctrl.resolve = function(){
                $ctrl.modalControl.resolve();
            }
            $ctrl.dismiss = function() {
                $ctrl.modalControl.dismiss();
            }
        }
    });

    app.component("uncertaintyForm", {
        bindings: {
            uncertainty : '='
        },
        templateUrl: '/templates/widgets/uncertainty-form.html'
    })

    /**
     * Simple form for sampling edition with partitions and no filters.
     * Supports changing dataset on the fly
     * Does not support auto refresh mechanism.
     */
    app.directive("partitionedModelForm", function(DataikuAPI, $stateParams, DatasetInfoCache, DatasetUtils, CreateModalFromComponent, changePartitionedEnabledModalDirective) {
        return {
            scope : {
                splitPolicy: '=',
                datasetSmartName : '=',
                mlTaskDesign : '='
            },
            templateUrl : '/templates/widgets/partitioned-model-form.html',
            link : function($scope) {
                $scope.getPartitionsList = function () {
                    return DataikuAPI.datasets.listPartitions($scope.dataset)
                        .error(setErrorInScope.bind($scope))
                        .then(resp => resp.data);
                };

                $scope.partitionedModel = $scope.mlTaskDesign.partitionedModel
                $scope.uiState = {
                    partitionedModelEnabled: $scope.partitionedModel.enabled
                }
                $scope.settingsLost = function () {
                    let lost = []
                    if ($scope.mlTaskDesign && $scope.mlTaskDesign.overridesParams && $scope.mlTaskDesign.overridesParams.overrides.length !== 0) lost.push("Overrides");
                    if ($scope.mlTaskDesign && $scope.mlTaskDesign.uncertainty && $scope.mlTaskDesign.uncertainty.predictionIntervalsEnabled) lost.push("Uncertainty");
                    return lost
                }
                $scope.onChangePartitionEnabled = function(){
                    if ($scope.uiState.partitionedModelEnabled && $scope.settingsLost().length !== 0){
                        CreateModalFromComponent(changePartitionedEnabledModalDirective, {settingsLost: $scope.settingsLost()})
                            .then(
                                () => {
                                    $scope.mlTaskDesign.overridesParams.overrides = []
                                    if ($scope.mlTaskDesign.uncertainty) {
                                        $scope.mlTaskDesign.uncertainty.predictionIntervalsEnabled = false
                                    }
                                    $scope.partitionedModel.enabled = $scope.uiState.partitionedModelEnabled;
                                },
                                () => {
                                    // When modal is dismissed we reset the ui state to the previous value
                                    $scope.uiState.partitionedModelEnabled = $scope.partitionedModel.enabled;
                                });
                    }
                    else{
                        $scope.partitionedModel.enabled = $scope.uiState.partitionedModelEnabled;
                    }
                };

                $scope.partitioningDisabledReason = function () {
                    if (!$scope.dataset) {
                        return "Loading…";
                    } else if ($scope.dataset.partitioning.dimensions.length == 0) {
                        return "input dataset is not partitioned";
                    } else if ($scope.splitPolicy != 'SPLIT_MAIN_DATASET') {
                        return "train/test split policy is not compatible";
                    }
                    return; // not disabled
                };

                $scope.dimensionsList = function() {
                    return $scope.dataset.partitioning.dimensions
                        .map(dim => `<b>${sanitize(dim.name)}</b>`)
                        .join(', ');
                };

                $scope.$watch("datasetSmartName", function(nv) {
                    if (nv) {
                        const loc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.datasetSmartName);
                        DatasetInfoCache.getSimple(loc.projectKey, loc.name).then(function(data){
                            $scope.dataset = data;
                            if ($scope.dataset.partitioning.dimensions.length === 0){
                                $scope.partitionedModel.ssdSelection.partitionSelectionMethod = "ALL";
                            }
                            $scope.$broadcast("datasetChange")
                        });
                    }
                });
            }
        }
    });


    /**
     * Simple form for inserting ordering rules for export/sampling
     * which is a list of columns and order (asc or desc)
     */
    app.directive("orderingRulesForm", function() {
        return {
            scope: {
                rules: '='
            },
            templateUrl: '/templates/widgets/ordering-rules-form.html'
        };
    });

    app.directive("ngScopeElement", function () {
        var directiveDefinitionObject = {
            restrict: "A",
            compile: function compile() {
                return {
                    pre: function preLink(scope, iElement, iAttrs) {
                        scope[iAttrs.ngScopeElement] = iElement;
                    }
                };
            }
        };
        return directiveDefinitionObject;
    });


    app.directive('customElementPopup', function($timeout, $compile, $rootScope) {
        // Attrs:
        //   - cep-position = align-left-bottom, align-right-bottom, smart
        //   - cep-width = fit-main (adapt size of popover to size of mainzone)
    return {
        restrict: 'A',
        scope: true, // no isolated scope, we want our user to call us
        compile: function(element, attrs) {
            var closeOthers = attrs.closeOthers !== "false",  // opt-out
                closeOnClick = attrs.closeOnClick === "true", // opt-in
                allowModals = attrs.allowModals === "true", // don't close after modal is opened/closed
                popoverTemplate = element.find('.popover').detach(),
                dismissDeregister = null
            return function($scope, element, attrs) {
                var identifier = {}, popover = null,
                    position = attrs.cepPosition || "align-left-bottom",
                    hidePopoverButton = attrs.hidePopoverButton === "true",
                    startMainZone = $(".mainzone", element),
                    popoverShown = false,
                    onHideCallback = attrs.onHideCallback,
                    onShowCallback = attrs.onShowCallback;

                function isChildOfPopup(target) {
                    return $(target).closest(popover).length > 0;
                }

                function hideIfNoIdentifierOrNotMe(event, evtIdentifier) {
                    const isButtonClicked = $(event.target).closest(element).length > 0;
                    if (isButtonClicked || allowModals && $(event.target).closest('.modal-container, .modal-backdrop').length > 0) {
                        return;
                    }
                    const evtIdChange = (!evtIdentifier || identifier !== evtIdentifier);
                    if (evtIdChange && !isChildOfPopup(event.target) && !(evtIdentifier &&  isChildOfPopup(evtIdentifier))){
                        hide();
                    }
                }
                function hide() {
                    if (popover) {
                        popover.hide().detach();
                    }
                    $timeout(() => {popoverShown = false;});
                    $("html").unbind("click", hideIfNoIdentifierOrNotMe);
                    (startMainZone.length ? startMainZone : $(".mainzone", element)).removeClass('popover-shown');
                    if (onHideCallback && $scope.$eval(onHideCallback) instanceof Function) {
                        $scope.$eval(onHideCallback)();
                    }
                }
                function addPositionOffset(direction, value) {
                    try {
                        var res = value + parseInt(attrs['cepOffset' + direction]);
                        return isNaN(res) ? value : res;
                    } catch(e) {
                        return value;
                    }
                }
                function show() {
                    // clear other sub-popovers
                    $rootScope.$broadcast('dismissSubPopovers');
                    var mainZone = startMainZone.length ? startMainZone : $(".mainzone", element),
                        mzOff = mainZone.offset();
                    popoverShown = true;
                    if (popover === null) {
                        popover = $compile(popoverTemplate.get(0).cloneNode(true))($scope);
                        // here the template is compiled but not resolved
                        // => popover.innerWidth() etc. are incorrect until the next $digest
                    }
                    popover.css('visibility', 'hidden').appendTo("body");

                    function computeSmartPosition(offset) {
                        if (mzOff.left * 2 < window.innerWidth) {
                            offset.left = mzOff.left;
                        } else {
                            offset.right = window.innerWidth - mzOff.left - mainZone.innerWidth();
                        }
                        if (mzOff.top * 2 < window.innerHeight) {
                            offset.top = mzOff.top + mainZone.height();
                        } else {
                            offset.bottom = window.innerHeight - mzOff.top;
                        }
                        popover.css(offset);
                    }

                    window.setTimeout(function() {  // see above
                        let offset;
                        switch (position) {
                        case 'align-left-bottom':
                            offset = {
                                left: addPositionOffset('Left',mzOff.left),
                                top: addPositionOffset('Top',mzOff.top + mainZone.innerHeight())
                            };
                            if (mzOff.top + mainZone.innerHeight() + popover[0].scrollHeight > window.innerHeight) {
                                offset.height = Math.max(window.innerHeight - mzOff.top - mainZone.innerHeight() - 4, 50);
                                offset.overflow = "auto";
                            } else {
                                offset.height = '';
                                offset.overflow = '';
                            }
                            popover.css(offset);
                            break;
                        case 'align-right-bottom':
                            popover.css({
                                top: addPositionOffset('Top',mzOff.top + mainZone.innerHeight()),
                                left: addPositionOffset('Left', mzOff.left + mainZone.innerWidth() - popover.innerWidth())
                            });
                            break;
                        case 'smart':
                            offset = { left: 'auto', right: 'auto', top: 'auto', bottom: 'auto' };
                            computeSmartPosition(offset);
                            break;
                        case 'auto':
                            offset = { left: 'auto', right: 'auto', top: 'auto', bottom: 'auto' };
                            if (window.innerHeight - (mzOff.top+mainZone.innerHeight() + popover.innerHeight()) > 0) {
                                offset.top = mzOff.top + mainZone.innerHeight();
                                offset.left = mzOff.left + mainZone.innerWidth() - popover.innerWidth();
                                offset.right = mzOff.right + mainZone.innerWidth();
                                popover.css(offset);
                             }
                            else {
                                computeSmartPosition(offset);
                             }
                             break;
                        case 'align-left-top':
                            if (hidePopoverButton) {
                                popover.css({ left: mzOff.left, top: mzOff.top, bottom: 'auto' });
                            } else {
                                popover.css({ left: mzOff.left, bottom: window.innerHeight - mzOff.top, top: 'auto' });
                            }
                            break;
                        case 'align-right':
                            popover.css({ left: mzOff.left+mainZone.outerWidth(), top: mzOff.top, bottom: 'auto' });
                            break;
                        case 'align-right-top':
                            popover.css({ left: mzOff.left + mainZone.innerWidth() - popover.innerWidth(),
                                bottom: window.innerHeight - mzOff.top, top: 'auto' });
                            break;
                        }
                        if (attrs.cepWidth === 'fit-main') {
                            popover.css("width", mainZone.innerWidth());
                        }
                        popover.css('visibility', 'visible');
                    }, 100);

                    popover.show();
                    mainZone.addClass('popover-shown');
                    popover.add(".mainzone", element).off("click.dku-pop-over");
                    popover.on("click.dku-pop-over", function() {
                            if (closeOthers) {
                                $("html").triggerHandler('click.cePopup', identifier);
                            }
                        });
                    $(".mainzone", element).on("click.dku-pop-over", function() {
                        if (closeOthers) {
                            $("html").triggerHandler('click.cePopup', identifier);
                        }
                    });
                    if (closeOnClick) popover.on("click.dku-pop-over", hide);
                    window.setTimeout(function() {
                        $("html").on('click.cePopup', function (event, evtIdentifier) {
                            hideIfNoIdentifierOrNotMe(event, evtIdentifier);
                        });
                    }, 0);

                    if (dismissDeregister) {
                        dismissDeregister();
                    }
                    dismissDeregister = $rootScope.$on("dismissPopovers", function(){
                        if (!allowModals) {
                            hide();
                            if (dismissDeregister) {
                                dismissDeregister();
                                dismissDeregister = null;
                            }
                        }
                    });

                    if (onShowCallback && $scope.$eval(onShowCallback) instanceof Function) {
                        $scope.$eval(onShowCallback)();
                    }
                }

                $scope.showPopover = show;
                $scope.hidePopover = hide;
                $scope.popoverShown = function() { return popoverShown; };
                $scope.togglePopover = function(event) {
                    if (popoverShown) {
                        hide();
                    }
                    else {
                        if (event) $("html").triggerHandler('click.cePopup', identifier);
                        show();
                    }
                };
            };
        } };
    });

    app.directive("sidebarTabL1Link", ($compile) => {
        return {
            replace: true,
            priority: 100,
            scope : {
                tabName : "@",
                label : "@",
                disabled: "@",
                tooltipText: "@"
            },
            link : function(scope, element, attrs) {
                scope.$watch(() => attrs.disabled, () => {
                    let template;
                    if (scope.$eval(attrs.disabled)) {
                        template = '<li class="l1" style="color: #666666; opacity: 0.5;">' +
                            '       <a data-toggle="tooltip" data-placement="right" title="{{tooltipText}}" data-container="body">' +
                            '           {{ label }}' +
                            '       </a>'+
                            '       </li>';
                    } else {
                        template = '<li class="l1" tab-active="{{tabName}}" full-click><a main-click tab-set="{{tabName}}">{{label}}</a></li>';
                    }
                    element.html(template);
                    $compile(element.contents())(scope);
                });
            }
        }
    });

    app.directive("sidebarTabL1Href", () => {
        return {
            template: '<li class="l1" tab-active="{{tabName}}" full-click><a main-click ui-sref=".{{tabName}}" data-qa-analysis-page__left-tab="{{tabName}}">{{label}}</a></li>',
            replace: true,
            priority: 100,
            scope : {
                tabName : "@",
                label : "@"
            }
        }
    });

    app.directive("sidebarTabL2Link", function($compile){
        return {
            replace: true,

            priority: 100,
            scope : {
                tabName : "@",
                linkName : "@",
                label : "@",
                disableLink: "@",
                disableMessage: "@",
                useHref: "@",
                hrefParams: "=",
                sidekickPulsar: "=",
                alternativeTabs: "="
            },
            link : function(scope, element, attrs) {
                scope.$watch(() => attrs.disableLink, () => {
                    let template;
                    if (scope.$eval(attrs.disableLink)) {
                        template = `<li toggle="tooltip" container="body" title="{{disableMessage}}" style="opacity: 0.5">
                        <div class="l2">{{label}}</div>
                    </li>`;
                    } else if (attrs.useHref){
                        template = '<li class="l2" tab-active="{{tabName}}" alternative-tabs="{{alternativeTabs}}" full-click><div class="padded"><a main-click ui-sref=".{{linkName ? linkName : tabName}}(hrefParams)">{{label}}<span class="mleft8 sidekick-pulsar" ng-if="sidekickPulsar"></span></a></div></li>';
                    } else {
                        template = '<li class="l2" tab-active="{{tabName}}" alternative-tabs="{{alternativeTabs}}" full-click><div class="padded"><a main-click tab-set="{{tabName}}">{{label}}<span class="mleft8 sidekick-pulsar" ng-if="sidekickPulsar"></span></a></div></li>';
                    }
                    element.html(template);
                    $compile(element.contents())(scope);
                });
            }
        }
    });

    app.directive("sidebarRoutingTabL2Link", function($state, $timeout, $compile){
        return {
            replace: true,
            priority: 100,
            scope : {
                tabName : "@",
                tabModel: "=",
                label : "@",
                sidekickPulsar: "="
            },
            link: function(scope, element, attrs) {
                scope.disabledMessage = attrs.disabledMessage;
                scope.onDisableUpdate = function(nv) {
                    element.empty();
                    let tpl;
                    if (nv) {
                        tpl = $compile(`
                        <div>
                            <li toggle="tooltip" container="body" title="{{disabledMessage}}" style="opacity: 0.5">
                                <div class="l2">{{label}}</div>
                            </li>
                        </div>
                        `);
                    } else {
                        tpl = $compile(`
                        <div>
                            <li class="l2" ng-class="{'active': tabName == tabModel}">
                                <div class="padded"><a ng-click="goTab()">{{label}}<span class="mleft8 sidekick-pulsar" ng-if="sidekickPulsar"></a></span></a></div>
                            </li>
                        </div>
                        `);
                    }
                    element.append(tpl(scope));
                }
                attrs.$observe('disableLink', function(nv) {
                    scope.onDisableUpdate(nv);
                });
                scope.onDisableUpdate(attrs.disableLink);
                scope.goTab = function() {
                    scope.tabModel = scope.tabName;
                    const stateNameParts = $state.current.name.split(".");
                    stateNameParts[stateNameParts.length-1] = scope.tabName;
                    $timeout(function() {
                        $state.go(stateNameParts.join('.'));
                    });
                }
            }
        }
    });

    app.directive("topLevelTabState", function($rootScope){
        return {
            template: '<a class="tab" ng-class="{\'enabled\' : topNav.tab == tabName}" ui-sref="{{sref}}">{{label}}</a>',
            replace: true,
            scope : {
                tabName : "@",
                sref : "@",
                label: '@'
            },
            link : function($scope) {
                $scope.topNav = $rootScope.topNav;
            }
        }
    });


    app.service('ModalSeeThroughService', () => {
        const getHeaderHideClass = (seeThrough) => seeThrough ? 'see-through' : 'non-see-through';
        const svc = {
            isSeeThrough: false,
            toggleSeeThrough() {
                svc.setSeeThrough(!svc.isSeeThrough);
            },
            setSeeThrough(val) {
                svc.isSeeThrough = val;
                const divs = $('div.modal-container, div.modal-backdrop, div.popover');
                const toAdd = getHeaderHideClass(svc.isSeeThrough);
                const toRemove = getHeaderHideClass(!svc.isSeeThrough);
                divs.addClass(toAdd).removeClass(toRemove);
            }
        };
        return svc;
    })


    const dkuModalHeader = ($timeout, ModalSeeThroughService, translate) => {

        return {
            template : `
            <div class="vertical-flex ais gap-2x">
                <div class="modal-header {{modalClass}}">
                    <i ng-if="modalTotem || hasSvgTotem()"
                        class="modal-header__totem {{modalTotem}}"
                        ng-transclude="totemSvg"
                    ></i>
                    <h4 class="modal-header__title" ng-transclude="title">
                        <div class="mx-textellipsis" show-tooltip-on-content-overflow toggle="tooltip-bottom" title="{{modalTitle}}" container="body">
                            {{modalTitle}}
                        </div>
                    </h4>
                    <div class="modal-header__menu-wrapper" ng-if="hasMenu()" ng-transclude="menu"></div>
                    <div class="modal-header__button-wrapper">
                        <button id="modal-btn-see-through"
                            type="button"
                            class="btn btn--text btn--dku-icon see-through no-modal-autofocus"
                            ng-click="toggleSeeThrough()"
                            toggle="tooltip-bottom" title="{{ModalSeeThroughService.isSeeThrough ? '${translate('MODALS.BACK_TO_DIALOG', 'Back to dialog')}' : '${translate('MODALS.SEE_BEHIND', 'See behind')}'}}" container="body"
                        >
                            <i class="dku-modal-see-through__icon--not-hovered {{ModalSeeThroughService.isSeeThrough ? 'dku-icon-eye-off-20' : 'dku-icon-eye-20'}}"></i>
                            <i class="dku-modal-see-through__icon--hovered {{ModalSeeThroughService.isSeeThrough ? 'dku-icon-eye-20' : 'dku-icon-eye-off-20'}}"></i>
                        </button>
                        <button id="modal-btn-close" type="button"
                            class="btn btn--text btn--dku-icon no-modal-autofocus"
                            data-dismiss="{{modalClose && modalClose() ? '' : 'modal'}}" ng-click="close()"
                            toggle="tooltip-bottom" title="${ translate('MODALS.CLOSE_WINDOW', 'Close window') }"
                        >
                            <i class="dku-icon-dismiss-20"></i>
                        </button>
                    </div>
                </div>
                <ul ng-if="hasTabs()" class="modal-tabs" ng-transclude="tabs"></ul>
            </div>
            `,
            scope : { modalTitle : "@", modalTotem : "@?", modalClass: "@?", modalClose: "&?" },
            replace : true,
            transclude: {
                'title': '?dkuModalTitle',
                'totemSvg': '?dkuModalTotem',
                'tabs': '?dkuModalTabs',
                'menu': '?dkuModalMenu'
            },
            link: function (scope, $element, $attrs, $thisCtrl, $transclude) {
                // edge case - when a new modal opens, we make if visible regardless of the current see-through status
                ModalSeeThroughService.setSeeThrough(false);

                const $seeThroughButton = $element.find('#modal-btn-see-through');
                scope.ModalSeeThroughService = ModalSeeThroughService;
                scope.close = function () {
                    if (scope.modalClose && scope.modalClose() instanceof Function) return scope.modalClose()();
                    return false;
                }
                scope.hasTabs = function () {return $transclude.isSlotFilled('tabs');}
                scope.hasMenu = function () {return $transclude.isSlotFilled('menu');}
                scope.hasSvgTotem = function () {return $transclude.isSlotFilled('totemSvg');}

                scope.toggleSeeThrough = function () {
                    // force-reload the tooltip for changed text
                    $seeThroughButton.tooltip('hide');
                    $timeout(() => $seeThroughButton.tooltip('show'));
                    ModalSeeThroughService.toggleSeeThrough();
                }

                scope.$on('$destroy', () => {
                    // tooltip is attached to body (only way to prevent transparency to affect it), so we need to manually destroy it in case user close modal with mouse still on button
                    // since the modal DOM is removed before the scope is destroyed, using $element.tooltip('destroy') doesn't work.
                    // this method is a bit brutal but in practice any tooltip displayed at this moment should come from inside the modal, so it's very unlikely it has a negative side-effect
                    $('.tooltip').remove();
                    // always restore visibility on modal close.
                    ModalSeeThroughService.setSeeThrough(false);
                });
            }
        }
    }

    // deprectated. use dkuModalHeader with optional modalTotem attribute.
    app.directive("dkuModalHeaderWithTotem", dkuModalHeader); // dku-modal-header-with-totem

    app.directive("dkuModalHeader", dkuModalHeader); // dku-modal-header


    app.directive('dkuEnter', function() {
        return function(scope, element, attrs) {
            element.bind("keydown keypress", function(event) {
                if(event.which === 13) {
                        scope.$apply(function(){
                                scope.$eval(attrs.dkuEnter, {$event: event});
                        });
                        event.preventDefault();
                }
            });
        };
    });


    app.directive('cancelOnEscape', function() {
        return function(_scope, element) {
            var val = element.val();

            element.bind("focus", () => {
                val = element.val();
            });

            element.bind("keydown keypress", function(event) {
                if(event.which === 27) {
                    element.val(val);
                    element.blur();
                    event.preventDefault();
                }
            });
        };
    });

    app.directive('svgTitles', function($sanitize) {
        function go(tooltip, stack, evt) {
            if (stack.length) {
                if (evt) {
                    var pos = {};
                    if (evt.clientX * 2 > window.innerWidth ) {
                        pos.right = (window.innerWidth - evt.clientX) + 'px';
                        pos.left  = 'auto';
                    } else {
                        pos.left  = evt.clientX + 'px';
                        pos.right = 'auto';
                    }
                    if (evt.clientY * 2 > window.innerHeight) {
                        pos.bottom = (window.innerHeight - evt.clientY) + 'px';
                        pos.top    = 'auto';
                    } else {
                        pos.top    = evt.clientY + 'px';
                        pos.bottom = 'auto';
                    }
                    tooltip.style(pos);
                }
                tooltip.html($sanitize(stack[stack.length - 1].getAttribute('data-title')));
                tooltip.style({display: 'block'});
            } else {
                tooltip.style({display: 'none', left: '0', top: '0'});
            }
        }
        return { restrict: 'A', scope: false, controller: function($element) {
            var _elt = $element.get(0),
                elt = _elt.tagName.toLowerCase() === 'svg' ? _elt : _elt.querySelector('svg'),
                svg = d3.select(elt),
                stack = [],
                tooltip = d3.select(document.body.insertBefore(document.createElement('div'), null))
                            .attr('class', 'svg-title-tooltip');
            return {
                update: function() {
                    svg.selectAll('[data-title]')
                    .on('mouseover.svgTitle', function mouseover() {
                        var i = stack.indexOf(this);
                        if (stack.length === 0 || i + 1 !== stack.length) {
                            if (i !== -1) { stack.splice(i, 1); }
                            stack.push(this);
                        }
                        go(tooltip, stack, d3.event);
                    }).on('mouseout.svgTitle', function mouseout() {
                        var i = stack.indexOf(this);
                        if (i !== -1) { stack.splice(i, 1); }
                        go(tooltip, stack, d3.event);
                    });
                },
                delete: function() {
                    tooltip.remove();
                }
            };
        }, link: function(scope, element, attrs, ctrl) {
            ctrl.update();
            scope.$on('$destroy', ctrl.delete);
        } };
    });

    app.directive('dkuBetterTooltip', ($sanitize) => {
        // Attrs:
        //   - dbt-placement = "top" / "bottom"
        //   - dbt-title
        var ret = {
            restrict : 'A',
        };
        ret.link = function(_scope, element, attrs) {
            var tooltip = null;
            function show(){
                tooltip = $("<div />");
                tooltip.html($sanitize(attrs.dbtTitle));
                tooltip.addClass("dbt-tooltip");
                tooltip.css("pointer-events", "none");
                if (attrs.dbtClazz) {
                    tooltip.addClass(attrs.dbtClazz);
                }
                $("body").append(tooltip); //so we have access to dimensions

                var posLeft = 0;
                var posTop = 0;
                var left = $(element).offset().left;
                var top = $(element).offset().top;
                var placement = attrs.dbtPlacement;
                var rect = $(element).get(0).getBoundingClientRect();
                if (placement == "top") {
                    posLeft = left + rect.width / 2 - tooltip.width()/2;
                    posTop = top - tooltip.height() - 10;
                } else if(placement == "top-right"){
                    posLeft = left + rect.width;
                    posTop = top - tooltip.height() - 10;
                } else if(placement == "top-left"){
                    posLeft = left - tooltip.width() - 10;
                    posTop = top - tooltip.height() - 10;
                } else if(placement == "bottom-left"){
                    posLeft = left - tooltip.width() - 10;
                    posTop = top + rect.height;
                } else if(placement == "bottom-right"){
                    posLeft = left + rect.width;
                    posTop = top + rect.height;
                } else if(placement == "bottom"){
                    posLeft = left + rect.width / 2 - tooltip.width()/2;
                    posTop = top + rect.height;
                } else if(placement == "left"){
                    posLeft = left - tooltip.width() - 10;
                    posTop = top + rect.height / 2 - tooltip.height()/2;
                } else if(placement == "right"){
                    posLeft = left + rect.width;
                    posTop = top + rect.height / 2 - tooltip.height()/2;
                }
                tooltip.css("left", posLeft);
                tooltip.css("top", posTop);
                $("body").append(tooltip);
            }
            function hide(){
                tooltip.remove();
            }
            element.on("mouseover.dbt", show);
            element.on("mouseout.dbt", hide);
        };
        return ret;
    });

    app.directive('svgTooltip', ($sanitize) => {

        return {
            scope: false,
            restrict: 'A',
            link: function($scope, element, attrs) {

                var $container = $(attrs.container || 'body').filter(':visible');
                var $tooltip = $('<div class="svg-tooltip ' + (attrs.tooltipClass || '') + '"></div>').appendTo($container);

                $scope.setTooltipContent = function(content) {
                    $tooltip.html($sanitize(content));
                };

                $scope.hideTooltip = function() {
                    $tooltip.css("opacity", 0);
                };

                $scope.showTooltip = function(x, y) {
                    let containerOffset = $container.offset();
                    let elOffset = $(element).offset();
                    $tooltip.css("top", (y + elOffset.top - containerOffset.top + 5) + "px");
                    $tooltip.css("left", (x + elOffset.left - containerOffset.left +  5) + "px");
                    $tooltip.css("opacity", 1);
                };

                $scope.$on("$destroy", function() {
                   $tooltip.remove();
                });
            }
        };
    });


    /**
     * In the new dataset page, each dataset name is a link-looking button to create the dataset.
     * the exact look depends on the dataset type status (license, available connection...)
     */
    app.component('newDatasetButton', {
        bindings : {
            type : "<",
        },
        template: `<button
            ng-class="{'dataset-notlicensed-ce' : $ctrl.type.status == 'NOT_LICENSED_CE'}"
            ng-disabled="$ctrl.type.status == 'NOT_LICENSED_EE' || $ctrl.type.status == 'NO_CONNECTION'"
            title="{{$ctrl.type.tooltip}}"
        >
            {{$ctrl.type.label}}
        </button>`,
    });

    /**
     * This directive watches the width of the new dataset page, and automatically updates the width of a centered container so that the tiles are always centered and aligned.
     * I used this method because there is no way through css alone to guarantee that the tiles will flow correctly, that the plugin block will be aligned with the main block, and that the top-right links are aligned with the tiles.
     */
    app.directive('newDatasetPageAlignmentManager', function() { // new-dataset-page-alignment-manager
        return {
            restrict: 'A',
            link: function($scope, $element) {
                const tileWidth = 301; // the tile is 300px, but when zoom level is < 100%, border grows slightly

                const resizeObserver = new ResizeObserver((el) => {
                    const elementWidth = el[0].contentRect.width;
                    $element.find('.new-dataset-page__centered-container').css('width', tileWidth * Math.floor(elementWidth / tileWidth));
                });

                resizeObserver.observe($element[0]);
                $scope.$on('$destroy', () => {
                    resizeObserver.disconnect();
                })
            }
        }
    })

    app.filter("singleChecklistState", function(){
        return function(input) {
            var total = 0, done = 0;
            input.items.forEach(function(x) {
                total++;
                if (x.done) done++;
            });
            if (total == 0) return "";
            return "<span>(" + done + "/" + total + " done)</span>";
        }
    });
    /** Emits "checklistEdited" on any change */
    app.directive('objectChecklist', ($rootScope) => {
        return {
            restrict : 'A',
            scope : {
                "checklist" : "=",
                "itemsOnly" : "=",
                "readOnly" : "="
            },
            templateUrl: "/templates/widgets/checklist.html",
            link : ($scope, element) => {
                $scope.state = {
                    addingItem : false,
                    editingItem : null, // The item being edited
                    editingItemText : null // The new text of the item being edited
                };

                $scope.onItemStateChange = function(){
                    $scope.$emit("checklistEdited", "item-state-change");
                };

                $scope.enterEditItem = function(item, $event){
                    if ($event.target.tagName.toLowerCase() == "a") {
                            return;
                    }
                    // Cancel the other one first
                    if ($scope.state.editingItem) {
                        $scope.cancelEditItem();
                    }

                    item.editingText = true;
                    $scope.state.editingItem = item;
                    $scope.state.editingItemText = item.text;

                    window.setTimeout(function() {
                        $(".checklist-items .edit-zone", element).on("click.checklistEditItem", function(e) {
                            e.stopPropagation();
                        });
                        $("html").on("click.checklistEditItem", function(event){
                            if ($(event.target).hasClass('modal-backdrop') || $(event.target.parentNode).hasClass('modal-header') || $(event.target.parentNode).hasClass('modal-footer')) {
                                return;
                            }
                            $scope.$apply(function(){$scope.cancelEditItem()});
                        })
                    }, 0);
                };

                $scope.validateEditItem = function(){
                    if ($scope.state.editingItemText.trim().length == 0) return;

                    $scope.state.editingItem.text = $scope.state.editingItemText;
                    $scope.cancelEditItem();
                    $scope.$emit("checklistEdited", "validate-edit");
                };

                $scope.cancelEditItem = function(){
                    if ($('.codemirror-editor-modal').is(':visible')) {
                        return;
                    }
                    $scope.state.editingItem.editingText = false;
                    $scope.state.editingItem = null;
                    $scope.state.editingItemText = null;
                    $(".checklist-items .edit-zone", element).off("click.checklistEditItem");
                    $("html").off("click.checklistEditItem");
                };

                $scope.deleteItem = function(item) {
                    $scope.checklist.items.splice($scope.checklist.items.indexOf(item), 1);
                    $scope.$emit("checklistEdited", "delete");
                };

                $scope.enterAddItem = function() {
                    $scope.state.addingItem = true;
                    $scope.state.newItemText = "";
                    window.setTimeout(function() {
                        $(".new-item-zone", element).on("click.checklistAddNewItem", function(e) {
                                e.stopPropagation();
                        });

                        $("html").on("click.checklistAddNewItem", function(event){
                            if ($(event.target).hasClass('modal-backdrop') || $(event.target.parentNode).hasClass('modal-header') || $(event.target.parentNode).hasClass('modal-footer')) {
                                return;
                            }
                            $scope.$apply(function(){$scope.leaveAddItem()});
                        })
                    }, 0);
                };

                $scope.leaveAddItem = function(){
                    if ($('.codemirror-editor-modal').is(':visible')) {
                        return;
                    }
                    $scope.state.addingItem = false;
                    $(".new-item-zone", element).off("click.checklistAddNewItem");
                    $("html").off("click.checklistAddNewItem");
                };

                $scope.addNewItem = function() {
                    if ($scope.state.newItemText.length == 0) return;

                    $scope.checklist.items.push({
                        text : $scope.state.newItemText,
                        createdOn : new Date().getTime(),
                        createdBy : $rootScope.appConfig.login
                    });
                    $scope.$emit("checklistEdited", "add");
                    $scope.state.newItemText = "";
                };

                $scope.$watch("checklist", function(nv){
                    if (nv && nv.$newlyCreated) {
                        $scope.enterAddItem();
                        nv.$newlyCreated = false;
                    }
                });

                $scope.$on("$destroy", function(){
                    // noop
                })

            }
        }
    });


app.directive("sparkOverrideConfig", function($rootScope){
    return { // task = holder of sparkPreparedDFStorageLevel, MLTask or MLLib recipe desc
        scope : { config: '=', task: '=', taskType: '@' },
        templateUrl : '/templates/widgets/spark-override-config.html',
        link : function($scope) {
            $scope.rootAppConfig = $rootScope.appConfig;
            /* Initialize with first Spark conf */
            $scope.$watch("config", function(nv) {
                if (nv && nv.inheritConf == null) {
                    if ($rootScope.appConfig.sparkExecutionConfigs.length) {
                        nv.inheritConf = $rootScope.appConfig.sparkExecutionConfigs[0];
                    }
                }
            });
        }
    }
})

// (!) DEPRECATED: prefer using dss-slider (an Angular component) or its downgraded version.
app.directive("dkuSlider", function($window, $timeout){
    return {
        scope : {
            min : '=',
            max : '=',
            value : '=',
            nbDecimalPlaces : '=?',
            onChange: '&?'
        },
        link: function($scope, elem) {
            $scope.sliding = false;

            $scope.startSliding = function($event) {
                $scope.computeNewCursorPosition($event);
                $scope.sliding = true;
                $($window).on('mouseup', $scope.stopSliding);
                $($window).on('mousemove', $scope.slideCursor);
            }

            $scope.slideCursor = function($event) {
                $event.preventDefault(); //useful to avoid selecting content on vertical mouse move while sliding
                if ($scope.sliding) {
                    $scope.computeNewCursorPosition($event);
                }
            }

            $scope.stopSliding = function() {
                $scope.sliding = false;
                $($window).off('mouseup', $scope.stopSliding);
                $($window).off('mousemove', $scope.slideCursor);
                $scope.value = $scope.cursorValue;
                if ($scope.onChange) {
                    $scope.onChange();
                }
                $timeout(function(){$scope.value = $scope.cursorValue;});
            }


            $scope.computeNewCursorPosition = function($event) {
                var sliderWidth = $(elem).width() - $(elem).find('.cursor').width() - 1;
                var sliderXPosition = $(elem).offset().left;
                var xPosition = $event.pageX - sliderXPosition;
                if (xPosition < 0) {
                    xPosition = 0;
                }
                if (xPosition > sliderWidth) {
                    xPosition = sliderWidth;
                }
                $(elem).find('.cursor').css('left',xPosition + 'px');
                $scope.computeNewScopeValue(xPosition, sliderWidth);
                $scope.fixExtremityVisibility();
            }

            $scope.roundAccordingToNbDecimalPlaces = function(num) {
                if (typeof($scope.nbDecimalPlaces)==='undefined') {
                    $scope.nbDecimalPlaces = 0;
                }
                return Number(Math.round(num + "e+"+$scope.nbDecimalPlaces) + "e-"+$scope.nbDecimalPlaces);
            }

            $scope.computeNewScopeValue = function(xPosition, sliderWidth) {
                var range = $scope.max - $scope.min;
                $scope.cursorValue = $scope.roundAccordingToNbDecimalPlaces(range*xPosition/sliderWidth + $scope.min);
                $timeout(() => $scope.$apply());
            }

            $scope.initCursorPosition = function() {
                if (_.isNil($scope.value)) {
                    $scope.value = $scope.roundAccordingToNbDecimalPlaces(($scope.max - $scope.min)/2 + $scope.min);
                }
                $scope.cursorValue = $scope.value;
                var range = $scope.max - $scope.min;
                var xPosition = ($scope.value - $scope.min)*100/range;
                $(elem).find(".cursor").css("left", "auto");
                $(elem).find(".cursor").css("right", "auto");
                if (xPosition<50) {
                    $(elem).find('.cursor').css('left',xPosition + '%');
                } else {
                    $(elem).find('.cursor').css('right',100-xPosition + '%');
                }
                const maxLabelOpacity = xPosition > 90 ? 0 : 1;
                const minLabelOpacity = xPosition < 10 ? 0 : 1;
                $(elem).find('.range-max').css('opacity', maxLabelOpacity);
                $(elem).find('.range-min').css('opacity', minLabelOpacity);

                //  Mandatory to update cursor value when component is visible at two different places
                $timeout(() => $scope.$apply());
            }

            $timeout(function() {
                $scope.initCursorPosition();
            });

            $scope.$watch("value", function(nv) {
                if (_.isNil(nv)) return;
                $scope.initCursorPosition();
            });

            $scope.fixExtremityVisibility = function() {
                var cursorValueLeft = $(elem).find('.cursor-value').offset().left;
                var cursorValueRight = $(elem).find('.cursor-value').offset().left + $(elem).find('.cursor-value').width();
                var rangeMinRight = $(elem).find('.range-min').offset().left + $(elem).find('.range-min').width();
                var rangeMaxLeft = $(elem).find('.range-max').offset().left;
                var confortableGap = 5;
                if (rangeMinRight + confortableGap > cursorValueLeft) {
                    $(elem).find('.range-min').stop(true, false).fadeTo(40, 0);
                } else {
                    $(elem).find('.range-min').stop(true, false).fadeTo(40, 1);
                }
                if (rangeMaxLeft - confortableGap < cursorValueRight) {
                    $(elem).find('.range-max').stop(true, false).fadeTo(40, 0);
                } else {
                    $(elem).find('.range-max').stop(true, false).fadeTo(40, 1);
                }
            }

        },
        templateUrl : '/templates/widgets/dku-slider.html'
    }
});


app.directive("dkuArrowSlider",['$timeout', function($timeout) {
    return {
        restrict: "A",

        link: function(scope, elem, attrs) {

          /*
           * Inner variables
           */

            let frameSelector = attrs.frameSelector;
            let sliderSelector = attrs.sliderSelector;
            let slidingElementsSelector = sliderSelector + ' > *:visible';

            let minOffsetLeft = 0;
            let maxOffsetRight = 0;

            let modelListName = attrs.modelListName;

            scope.canSlideRightFlag = false;
            scope.canSlideLeftFlag = false;

          /*
           * Watchers / init
           */

            scope.$watch(modelListName, function(nv, ov) {
                let nvLength = nv ? nv.length : 0;
                let ovLength = ov ? ov.length : 0;
                //$timeout is needed to make sure slideToEnd is called after the currently processing $digest is done,
                //ie once the ng-repeat refresh is done and the new chip has been added
                $timeout(function() {
                    scope.computeNeedSlider();
                    if (scope.needSlider()) {
                        scope.initArrowSlider();
                    }
                    //$timeout to make sure arrow slider initialization is done (otherwise positioning computations may be off)
                    $timeout(function() {
                        if (nvLength < ovLength) {
                            if (!scope.needSlider()) {
                                slideToBegining();
                                removeArrowSliderStyle();
                            } else if (!isLastChipBeyondSliderBottom()) {
                                slideToEnd();
                            }
                        } else if (nvLength > ovLength) {
                            if (scope.needSlider()) {
                                slideToEnd();
                            } else {
                                scope.$broadcast('DKU_ARROW_SLIDER:animation_over');
                            }
                        }
                    }, 0);
                }, 0, false);
            }, true);

            scope.onResize = function() {
                scope.computeNeedSlider();
                if (!scope.needSlider()) {
                    slideToBegining();
                } else {
                    initOffsetsExtremas();
                    if (!isLastChipBeyondSliderBottom()) {
                        slideToEnd();
                    }
                    setCanSlideTags(scope.canSlideLeft(), scope.canSlideRight());
                }
                $timeout(function() {
                    scope.$apply();
                });
            }

            let loop;
            function resizeHandler() {
                clearTimeout(loop);
                loop = setTimeout(scope.onResize, 30);   //so that resize callback is called only once resize is done for good
            }

            $(window).on("resize.dkuArrowSlider", resizeHandler);
            scope.$on("$destroy", function(){
                $(window).off("resize.dkuArrowSlider", resizeHandler);
            });

            scope.$on("slideToId", function(event, frameSelectorAttr, sliderSelectorAttr, targetId) {
                if (frameSelector == frameSelectorAttr && sliderSelector == sliderSelectorAttr) {
                    slideToId(targetId);
                }
            });

            scope.initArrowSlider = function () {
                if (scope.needSlider()) {
                    $(frameSelector).css('position', 'relative');
                    $(sliderSelector).css('position', 'absolute');
                    $(sliderSelector).css('will-change', 'left');
                    $(sliderSelector).css('transition', 'left 150ms ease-out, right 150ms ease-out');
                    if (isNaN($(sliderSelector).css('left').replace('px', ''))) {
                        $(sliderSelector).css('left', '0px');
                        setCanSlideTags(false, true);
                    }
                    initOffsetsExtremas();
                    $timeout(function() {
                        scope.$broadcast("DKU_ARROW_SLIDER:arrow_slider_initialized");
                    });
                }
            };

           function removeArrowSliderStyle() {
                $(frameSelector).removeAttr("style");
                $(sliderSelector).removeAttr("style");
            }

            function initOffsetsExtremas() {
                minOffsetLeft = $(frameSelector).offset().left;
                maxOffsetRight = minOffsetLeft + $(frameSelector).width();
            }

            /*
            * Animation functions
            */

            scope.slideLeft = function() {
                let lastHiddenElement;
                let slidingElement = $(slidingElementsSelector);
                if (!slidingElement) return;
                for (let i = 0; i < slidingElement.length; i++) {
                    let elem = slidingElement[i];
                    if (!isElementVisible(elem)) {
                        lastHiddenElement = elem;
                    } else {
                        //if the element we wanna display is not the last one, make sure user can see there is more to come
                        if (i-1 > 0) {
                            let newSlidingElementLeft = getLeftAsNumber(sliderSelector) + getHiddenWidth(lastHiddenElement) + 20;
                            animatedSlide(newSlidingElementLeft);
                            setCanSlideTags(true, true);
                        } else {
                            slideToBegining();
                        }
                        break;
                    }
                }
            };

            scope.slideRight = function() {
                let lastHiddenElement;
                let slidingElement = $(slidingElementsSelector);
                if (!slidingElement) return;
                for (let i = slidingElement.length - 1; i >= 0; i--) {
                    let elem = slidingElement[i];
                    if (!isElementVisible(elem)) {
                        lastHiddenElement = elem;
                    } else {
                        //if the element we wanna display is not the last one, make sure user can see there is more to come
                        if (i + 1 < slidingElement.length - 1) {
                            let newSlidingElementLeft = getLeftAsNumber(sliderSelector) - getHiddenWidth(lastHiddenElement) - 20;
                            animatedSlide(newSlidingElementLeft);
                            setCanSlideTags(true, true);
                        } else {
                            slideToEnd();
                        }
                        break;
                    }
                }
            };

            function slideToBegining() {
                animatedSlide(0);
                setCanSlideTags(false, true);
            }

            function slideToEnd() {
                let newSlidingElementLeft = - (getTrueSliderWidth() - $(frameSelector).width()) -1 ;
                animatedSlide(newSlidingElementLeft);
                setCanSlideTags(true, false);
            }

            function slideToId(id) {
                let targetElementSelector = sliderSelector + ' > [id="'+ id +'"]:visible';
                if ($(targetElementSelector).length > 0 && !isElementVisible(targetElementSelector)) {
                    let targetElementOffsetLeft = $(targetElementSelector).offset().left;
                    let sliderOffsetLeft = $(sliderSelector).offset().left;

                    let sliderWidth = getTrueSliderWidth();
                    let frameWidth = $(frameSelector).width();

                    let targetElementPosition = targetElementOffsetLeft - sliderOffsetLeft - 20;
                    let widthAfterTargetElement =  sliderWidth - targetElementPosition;

                    if ($(targetElementSelector).is(':first-child')) {
                        slideToBegining();
                    } else if (widthAfterTargetElement > frameWidth) {
                        animatedSlide(-targetElementPosition, true);
                    } else {
                        slideToEnd();
                    }
                } else {
                    scope.$broadcast('DKU_ARROW_SLIDER:animation_over');
                }
            }

            function animatedSlide(newPosition, checkCanSlide) {
                $(sliderSelector).on('transitionend', function() {
                    scope.$broadcast('DKU_ARROW_SLIDER:animation_over');
                    if (checkCanSlide) {
                        setCanSlideTags(scope.canSlideLeft(), scope.canSlideRight());
                    }
                    $(sliderSelector).off('transitionend');
                });
                $(sliderSelector).css('left', newPosition + 'px');
            }

          /*
           * Double Click Handler
           */

            function dblClickHandler(counter, timer, clickFunc, dblClickFunc) {
                return function() {
                    if (counter <= 2) {
                        counter++;
                    }
                    if (counter == 1) {
                        clickFunc();
                        timer = $timeout(function(){
                            counter = 0;
                        }, 150);
                    }
                    if (counter == 2) {
                        dblClickFunc();
                        $timeout.cancel(timer);
                        counter = 0;
                    }
                };
            }

            let leftClickCounter = 0;
            let leftClickTimer;
            scope.slideLeftClickHandler = dblClickHandler(leftClickCounter, leftClickTimer, scope.slideLeft, slideToBegining);

            let rightClickCounter = 0;
            let rightClickTimer;
            scope.slideRightClickHandler = dblClickHandler(rightClickCounter, rightClickTimer, scope.slideRight, slideToEnd);

            /*
             * Checking if sliding is needed / possible functions
             */

            function neededDomElementsExist () {
                return $(sliderSelector).length > 0 && $(slidingElementsSelector).length > 0;
            }

            let isNeedSlider = false;
            scope.computeNeedSlider = function() {
                isNeedSlider =  neededDomElementsExist() && getTrueSliderWidth() > $(frameSelector).width();
            };

            scope.needSlider = function() {
                return isNeedSlider;
            };

            scope.canSlideRight = function() {
                return scope.needSlider() && !isElementVisible($(slidingElementsSelector)[$(slidingElementsSelector).length - 1]);
            };

            scope.canSlideLeft = function() {
                return scope.needSlider() && !isElementVisible($(slidingElementsSelector)[0]);
            };

            function setCanSlideTags(canSlideLeftFlag, canSlideRightFlag) {
                scope.canSlideLeftFlag = canSlideLeftFlag;
                scope.canSlideRightFlag = canSlideRightFlag;
            }

            /*
             * Private visual computing helpers
             */

            function isElementVisible(elem) {
                elem = $(elem);
                if (elem.length > 0) {
                    var elemOffsetLeft = elem.offset().left;
                    var elemOffsetRight = elemOffsetLeft + elem.innerWidth();
                    return !scope.needSlider() || (minOffsetLeft <= elemOffsetLeft && elemOffsetRight <= maxOffsetRight);
                }
            }

            function isLastChipBeyondSliderBottom() {
                var lastEl = $(slidingElementsSelector)[$(slidingElementsSelector).length - 1];
                var lastElRightOffset = $(lastEl).offset().left + $(lastEl).outerWidth();
                return lastElRightOffset >= maxOffsetRight;
            }

            function getTrueSliderWidth() {
                var maxIndex = $(slidingElementsSelector).length - 1;
                return $($(slidingElementsSelector)[maxIndex]).offset().left + $($(slidingElementsSelector)[maxIndex]).outerWidth() - $($(slidingElementsSelector)[0]).offset().left;
            }

            function getHiddenWidth(elem) {
                var elemOffsetLeft = $(elem).offset().left;
                var elemOffsetRight = elemOffsetLeft + $(elem).outerWidth();

                if (elemOffsetLeft < minOffsetLeft) {
                   return  minOffsetLeft - elemOffsetLeft;
                } else if (maxOffsetRight < elemOffsetRight) {
                    return elemOffsetRight - maxOffsetRight;
                }
                return 0;
            }

            function getLeftAsNumber(elem) {
                var left = $(elem).css('left');
                left = left.replace('px', '');
                if (!isNaN(left)) {
                    return parseInt(left);
                }
                return 0;
            }

      /*
           * Initialisation
           */
            $timeout(function() {
                scope.computeNeedSlider();
                if (scope.needSlider()) {
                    scope.initArrowSlider();
                }
            }, 0);

        }
    };
}]);


app.directive("uiCheckbox", function() {
    return {
        scope: {},
        require: "ngModel",
        restrict: "A",
        replace: "true",
        template: "<button type=\"button\"  ng-class=\"{'chkbox-btn-normal' : true, 'btn' : true, 'checked': checked===true}\">" +
            "<i ng-class=\"{'dku-icon-checkmark-12': checked===true}\"></span>" +
            "</button>",
        link: function(scope, elem, attrs, modelCtrl) {
            scope.size = "default";
            // Default Button Styling
            scope.stylebtn = {};
            // Default Checkmark Styling
            scope.styleicon = {"width": "10px", "left": "-1px"};
            // If size is undefined, Checkbox has normal size (Bootstrap 'xs')
            if(attrs.large !== undefined) {
                scope.size = "large";
                scope.stylebtn = {"padding-top": "2px", "padding-bottom": "2px", "height": "30px"};
                scope.styleicon = {"width": "8px", "left": "-5px", "font-size": "17px"};
            }
            if(attrs.larger !== undefined) {
                scope.size = "larger";
                scope.stylebtn = {"padding-top": "2px", "padding-bottom": "2px", "height": "34px"};
                scope.styleicon = {"width": "8px", "left": "-8px", "font-size": "22px"};
            }
            if(attrs.largest !== undefined) {
                scope.size = "largest";
                scope.stylebtn = {"padding-top": "2px", "padding-bottom": "2px", "height": "45px"};
                scope.styleicon = {"width": "11px", "left": "-11px", "font-size": "30px"};
            }

            var trueValue = true;
            var falseValue = false;

            // If defined set true value
            if(attrs.ngTrueValue !== undefined) {
                trueValue = attrs.ngTrueValue;
            }
            // If defined set false value
            if(attrs.ngFalseValue !== undefined) {
                falseValue = attrs.ngFalseValue;
            }

            // Check if name attribute is set and if so add it to the DOM element
            if(scope.name !== undefined) {
                elem.name = scope.name;
            }

            // Update element when model changes
            scope.$watch(function() {
                if(modelCtrl.$modelValue === trueValue || modelCtrl.$modelValue === true) {
                    modelCtrl.$setViewValue(trueValue);
                } else {
                    modelCtrl.$setViewValue(falseValue);
                }
                return modelCtrl.$modelValue;
            }, () => {
                scope.checked = modelCtrl.$modelValue === trueValue;
            }, true);

            // On click swap value and trigger onChange function
            elem.bind("click", function() {
                scope.$apply(function() {
                    if(modelCtrl.$modelValue === falseValue) {
                        modelCtrl.$setViewValue(trueValue);
                    } else {
                        modelCtrl.$setViewValue(falseValue);
                    }
                });
            });
        }
    };
});

const EDITABLE_LIST_ITEM_PREFIX = 'it'; // this is the property name used for items in the editableList* directives

/**
 * This directive is a fork of list-form that implements the new editable lists specifications.
 *
 * @param {Array}       ngModel                         - The list to bind to display.
 * @param {string}      [label]                         - Text to display when referring to this component.
 * @param {boolean}     [sortable=false]                - True to make the list sortable. Allows to rearrange list order by drag-and-dropping.
 * @param {Function}    [onAdd]                         - The function called when adding an item.
 * @param {Function}    [addPosition]                   - An optional function to provide the position for new item insertion.
 * @param {Function}    [onRemove]                      - The function called when removing an item.
 * @param {Function}    [onChange]                      - Callback called on change.
 * @param {boolean}     [noChangeOnAdd=false]           - True to prevent the callback onChange to be called when an item is added.
 * @param {boolean}     [required=false]                - Can the items of the list be empty. Used with the 'editable-list-input' component.
 * @param {Object}      [template]                      - Template of the items in the list. Used when the list items are objects.
 * @param {Function}    [prepare]                       - The function called on list update, to set items default value.
 * @param {Function}    [transcope]                     - Functions/objects to pass to the editableList scope.
 * @param {Array}       [suggests]                      - List of possible values of an item of the list. Can be displayed in a dropdown under a text input for instance.
 * @param {boolean}     [hasDivider=true]               - False to hide the divider line between items.
 * @param {string}      addLabel                        - Text to display in the add button; Optional if disableAdd is true.
 * @param {boolean}     [disableAdd=false]              - True to hide the Add button.
 * @param {boolean}     [disableRemove=false]           - True to hide the Remove buttons.
 * @param {boolean}     [enableEdit=false]              - True to show the Edit buttons.
 * @param {(item: list-item) => Promise<falsy | list-item>} [editItem]
 *                                                      - The callback called when editing an item. Parameter is the edited item, must return a promise, if resolved value is falsy, edit is canceled, otherwise must return the modified item.
 * @param {boolean}     [disableCreateOnEnter=false]    - True to prevent creating a new item when pressing Enter in the last focused item of the list. Focus the first item of the list instead.
 * @param {boolean}     [skipToNextFocusable=false]     - True to focus the next focusable item on Enter if the immediate next item can't be focused (e.g. deleted item).
 * @param {boolean}     [fullWidthList=false]           - True to make the list fill the full width available in the container.

 */
app.directive('editableList', function($timeout) { return {
    restrict: 'E',
    transclude: true, replace: true,
    templateUrl: '/templates/widgets/editable-list.html',
    require: '?ngModel',
    scope: {
        label: '@',
        ngModel: '<',
        sortable: '=',
        onAdd: '<',
        addPosition: '<?',
        onRemove: '<',
        onChange: '=',
        noChangeOnAdd: '<',
        required: '<',
        template: '=',
        prepare: '=',
        transcope: '=',
        suggests: '=',
        hasDivider: '<',
        addLabel: '@',
        disableAdd: '<',
        disableRemove: '<',
        enableEdit: '<',
        editItem: '<',
        disableCreateOnEnter: '<',
        skipToNextFocusable: '<',
        fullWidthList: '<',
        keyPlaceholder: '@',
        valuePlaceholder: '@'
    },
    compile: function(elt, attrs, transclude) {
        const ITEMS_CLASSNAME = 'editable-list__items';
        const ITEM_CLASSNAME = 'editable-list__item';
        const DIVIDER_CLASSNAME = 'editable-list__item--divider';
        const ICON_CONTAINER_CLASSNAME = 'editable-list__icon';
        const DRAG_ICON_CLASSNAME = 'editable-list__drag-icon';
        const DRAG_ICON_QA_SELECTOR = 'data-qa-editable-list-drag';
        const DELETE_BUTTON_CLASSNAME = 'editable-list__delete';
        const EDIT_BUTTON_CLASSNAME = 'editable-list__edit';
        const DELETE_BUTTON_QA_SELECTOR = 'data-qa-editable-list-delete';
        const EDIT_BUTTON_QA_SELECTOR = 'data-qa-editable-list-edit';
        const ITEM_TEMPLATE_CLASSNAME = 'editable-list__template';
        const EDITING_CLASSNAME = 'editable-list__template--editing';

        var itemsExpr = attrs.ngModel,
            klass = attrs['class'],
            focusableInputs = ['input:not([type=checkbox])', 'textarea', 'select'];

        return function(scope, elt){
            var lis = []; // the LIs

            if (klass) { // report CSS classes
                elt.className += ' ' + klass;
            }

            var insertTranscope = function(into) {
                if (typeof scope.transcope === 'object') {
                    for (var k in scope.transcope) {
                        into[k] = scope.transcope[k];
                    }
                }
            };

            insertTranscope(scope);
            scope.ngModel = [];

            scope.$parent.$watch(itemsExpr, function(v) {
                scope.ngModel = v || [];
            });

            // default hasDivider to true
            if (!angular.isDefined(scope.hasDivider)) {
                scope.hasDivider = true;
            }

            // Utilities
            function parentOf(child, className) {
                while (child && !child.classList.contains(className)) {
                    child = child.parentElement;
                }
                return angular.element(child);
            }

            function templateOf(child) {
                return parentOf(child, ITEM_TEMPLATE_CLASSNAME);
            }

            function liOf(child) {
                return parentOf(child, ITEM_CLASSNAME);
            }

            function indexOf(li) {
                for (var i = 0; i < lis.length; i++) {
                    if (lis[i].element[0] === li) return i;
                }
                // cond always true, prevent error w/ CodePen loop
                if (i || !lis.length) return -1;
            }

            function prepare(it) {
                if (scope.prepare) {
                    scope.prepare(it);
                }
            }

            function template() {
                switch(typeof scope.template) {
                    case 'function': return scope.template();
                    case 'object': return angular.extend({}, scope.template);
                    case 'string' : return scope.template;
                    default: return {};
                }
            }

            function regularEnter(evt) {  // press button, return in textarea...
                return evt.target.tagName.toLowerCase() !== 'input'
                        || evt.target.type === 'button';
            }

            // Edit entry and update list
            scope.edit = function(i) {
                if (scope.enableEdit && angular.isFunction(scope.editItem)) {
                    const itemToEdit = scope.ngModel[i];
                    scope.editItem(itemToEdit).then((newItem) => {
                        if (!newItem) return;
                        scope.ngModel[i] = newItem;
                        update(scope.ngModel, false);
                        scope.$parent.$apply();
                    });
                }
            };

            // Remove & update DOM
            scope.remove = function(i) {
                const removedElt = scope.ngModel.splice(i, 1)[0];
                update(scope.ngModel, false);
                if (!scope.disableRemove) {
                    scope.$parent.$apply();
                }
                scope.onRemove && scope.onRemove(removedElt);
            };

            var changing = false;

            function updateSuggests() {
                for (var i = lis.length; i-- > 0;) {
                    lis[i].scope.suggests = scope.suggests;
                }
            }

            function update(items, preventOnChange) {
                var change = !changing && scope.onChange && !preventOnChange;
                
                changing = true;

                for (var i = lis.length; i-- > 0;) {
                    lis[i].element.remove();
                    lis[i].scope.$destroy();
                    lis.splice(i, 1);
                }

                for (i = items.length - 1; i >= 0; i--) {
                    var childScope = scope.$new(),
                        childLi = angular.element('<li class="' + ITEM_CLASSNAME + (scope.hasDivider ? ' ' + DIVIDER_CLASSNAME : '') + '"></li>'),
                        childDrag = angular.element('<div class="' + ICON_CONTAINER_CLASSNAME + '"><i ' + DRAG_ICON_QA_SELECTOR + ' class="' + DRAG_ICON_CLASSNAME +' dku-icon-dots-multiple-16"></i></div>'),
                        childEdit = angular.element('<button type="button" ' + EDIT_BUTTON_QA_SELECTOR + ' class="btn btn--secondary btn--text btn--dku-icon btn--icon ' + EDIT_BUTTON_CLASSNAME + ' " tabindex="-1"> <i class="dku-icon-edit-16"></i></button>'),
                        childDelete = angular.element('<button type="button" ' + DELETE_BUTTON_QA_SELECTOR + ' class="btn btn--text btn--danger btn--dku-icon btn--icon ' + DELETE_BUTTON_CLASSNAME + ' " tabindex="-1"> <i class="dku-icon-trash-16"></i></button>'),
                        childTemplate = angular.element('<div class="' + ITEM_TEMPLATE_CLASSNAME + '"></div>');

                    childScope[EDITABLE_LIST_ITEM_PREFIX] = items[i];
                    childScope.suggests = scope.suggests;
                    prepare(childScope[EDITABLE_LIST_ITEM_PREFIX]);
                    childScope.$index = i;
                    childDelete.click(scope.remove.bind(this, i));
                    childEdit.click(scope.edit.bind(this, i));

                    scope.sortable && childLi.append(childDrag);
                    childLi.append(childTemplate);
                    scope.enableEdit && childLi.append(childEdit);
                    !scope.disableRemove && childLi.append(childDelete);

                    transclude(childScope, function(clone) {
                        childTemplate.prepend(clone);
                    });

                    lis.unshift({ element: childLi, scope: childScope });
                    itemsContainerEl.prepend(childLi);
                }

                const children = itemsContainerEl.children();
                const lastChild = children[children.length - 1];
                itemsContainerEl[0].scrollTop = lastChild && lastChild.offsetTop || 0;

                if (change) {
                    scope.onChange(scope.ngModel);
                }

                changing = false;
            }

            if (scope.onChange) {
                // Use a jQuery event handler, not a DOM one, because we want
                // the .trigger("change") in the bs-typeahead to trigger this
                $(elt[0]).on('change', function(evt) {
                    function doIt(){
                        changing = true;
                        scope.onChange(scope.ngModel);
                        changing = false;
                    }

                    if (!changing) {
                        /* This is the same hack that we did to fix #1222.
                         * When you have a bs-typeahead, you have an non-empty field, then
                         * you chnage data to get a suggestion. Clicking on the suggestion
                         * will exit the input field, triggering a change event.
                         * However, the change event triggers before the click has its own actions:
                         * which is, changing the value of the input and triggering another
                         * "change" and "input" event.
                         * By delaying the taking into account of this, we leave time to the browser
                         * to process the click and to have it repercuted to the Angular model
                         */
                        var uglyBSHack = $(evt.target).attr("bs-typeahead") != null;
                        if (uglyBSHack) {
                            window.setTimeout(doIt, 150);
                        } else {
                            doIt();
                        }
                    }
                });
            }

            scope.$watch('ngModel', function (newValue) {
                update(newValue, false);
            });
            if (scope.suggests) { scope.$watch('suggests', updateSuggests, true); }

            // Editing row, focus & blur
            var editing = null;

            function edit(li) {
                if (editing === li) return;
                if (editing) editing.removeClass(EDITING_CLASSNAME);
                editing = li;
                if (editing) {
                    editing.addClass(EDITING_CLASSNAME);
                }
            }

            elt[0].addEventListener('focus', function(evt) {
                if (focusableInputs.indexOf(evt.target.tagName.toLowerCase()) >= 0) {
                    edit(templateOf(evt.target));
                    evt.target.select();
                }
            }, true);

            elt[0].addEventListener('blur', function(evt) {
                if (focusableInputs.indexOf(evt.target.tagName.toLowerCase()) >= 0) {
                    edit(null);
                    window.getSelection().removeAllRanges();
                }
            }, true);

            function skipToNextFocusable(next) {
                let nextElement = lis[next].element[0];
                let focusable = nextElement.querySelector(focusableInputs.join(', '));
                while (next > -1 && !focusable) {
                    next = indexOf(nextElement.nextSibling);
                    if (next < 0) {
                        break;
                    }
                    nextElement = lis[next].element[0];
                    focusable = nextElement.querySelector(focusableInputs.join(', '));
                }
                return next;
            }

            elt.on('keydown', function(evt) {
                var next = null;
                switch (evt.keyCode) {
                    case 27:
                        evt.target.blur();
                        return true;
                    case 13:
                         if (regularEnter(evt)) return true;
                         evt.target.blur();
                         next = indexOf(templateOf(evt.target)[0].parentElement.nextSibling);
                         break;
                    default:
                        return true;
                }
                if (scope.skipToNextFocusable && next > -1) {
                    next = skipToNextFocusable(next);
                }
                next = scope.disableCreateOnEnter && next != null && next < 0 ? 0 : next;
                if (next > -1) {
                    const nextElement = lis[next].element[0];
                    const focusable = nextElement.querySelector(focusableInputs.join(', '));
                    if (focusable) focusable.focus();
                } else {
                    scope.add();
                }

                evt.preventDefault();
                evt.stopPropagation();
                return false;
            });

            var itemToAdd = template();

            prepare(itemToAdd);

            var deregWatchPrepare = scope.$watch('prepare', function() {
                if (scope.prepare) {
                    prepare(itemToAdd);
                    deregWatchPrepare();
                }
            });

            const itemsContainerEl = elt.find('.' + ITEMS_CLASSNAME);

            scope.add = function() {
                itemToAdd = template();
                prepare(itemToAdd);

                let addAtIndex = scope.addPosition ? scope.addPosition() : lis.length;
                if (addAtIndex < 0 || addAtIndex > lis.length) {
                    //to recover from errors in addPosition() function
                    addAtIndex = lis.length;
                }

                scope.ngModel.splice(addAtIndex, 0, itemToAdd);
                scope.hasAddedItem = true;
                update(scope.ngModel, scope.noChangeOnAdd);
                let addedElement = lis[addAtIndex].element[0];

                $timeout(function() {
                    const focusable = addedElement.querySelector(focusableInputs.join(', '));
                    if (focusable) focusable.focus();
                });
                scope.onAdd && scope.onAdd();
            }

            // Drag / drop
            if (!scope.sortable) return;

            elt.addClass('editablie-list--sortable');

            var dragging = null, draggingIndex = null, draggingOpacityTimeout = null;

            // Only allow dragging on handles
            elt.on('mouseover', function(evt) {
                if (evt.target.classList.contains(DRAG_ICON_CLASSNAME)) {
                    liOf(evt.target).prop('draggable', true);
                }
            });

            elt.on('mouseout', function(evt) {
                if (evt.target.classList.contains(DRAG_ICON_CLASSNAME) && !dragging) {
                    liOf(evt.target).prop('draggable', false);
                }
            });

            // Actual drag/drop code
            elt.on('dragstart', function(evt) {
                (evt.originalEvent || evt).dataTransfer.setData('text/plain', null);
                (evt.originalEvent || evt).dataTransfer.effectAllowed = 'move';
                dragging = liOf(evt.target)[0];
                draggingIndex = indexOf(dragging);
                itemsContainerEl.addClass('dragging');
                evt.target.classList.add('dragging');
                draggingOpacityTimeout = window.setTimeout(function() {
                    dragging.style.opacity = 0;
                }, 200); // later to let time for snapshot
            });

            elt.on('dragenter', function(evt) {
                if (!dragging || evt.target === elt[0]) return;
                var li = liOf(evt.target)[0];
                if (!li || li === dragging) return;
                li.classList.add(draggingIndex < indexOf(li) ? 'drag-below' : 'drag-above');
            });

            elt.on('dragleave', function(evt) {
                evt.target.classList.remove('drag-above', 'drag-below');
            });

            elt.on('dragover', function(evt){
                if (!dragging || evt.target === elt[0]) return;
                var li = liOf(evt.target)[0];
                if (!li || li === dragging) return;
                evt.preventDefault();
                (evt.originalEvent || evt).dataTransfer.dropEffect = 'move';
            });

            elt.on('drop', function(evt) {
                if (!dragging) return;
                evt.preventDefault();
                const dropIndex = indexOf(evt.target), dragIndex = draggingIndex;
                const itemsContainer = elt.find('.' + ITEMS_CLASSNAME)[0];
                if (dropIndex > draggingIndex) { // insert after
                    itemsContainer.insertBefore(dragging, evt.target.nextSibling);
                } else { // insert before
                    itemsContainer.insertBefore(dragging, evt.target);
                }
                dragEnd();
                scope.$apply(function() {
                    scope.ngModel.splice(dropIndex, 0, scope.ngModel.splice(dragIndex, 1)[0]);
                    update(scope.ngModel, false);
                });
            });

            elt.on('dragend', dragEnd);

            function dragEnd() {
                dragging.style.opacity = 1;
                itemsContainerEl.removeClass('dragging');
                dragging.classList.remove('dragging');
                if (draggingOpacityTimeout != null) {
                    window.clearTimeout(draggingOpacityTimeout);
                }
                dragging = null;
                draggingIndex = null;
                draggingOpacityTimeout = null;
                elt.find('.drag-above').removeClass('drag-above');
                elt.find('.drag-below').removeClass('drag-below');
            }
        };
    }
}; });


app.directive('checkCategoryNameUnique', function() {
    return {
        require: 'ngModel',
        scope: false,
        compile: function() {
            return function(scope, _elt, _attrs, ngModel) {
                const index = scope.$index;

                function checkUnique(nv) {
                    const isUnique = !scope.generalSettings.globalTagsCategories.find((it, idx) => it.name == nv[index].name && idx != index);
                    ngModel.$setValidity('unique', isUnique);
                    return nv;
                }

                scope.$watch('generalSettings.globalTagsCategories', checkUnique, true);
            }
        }
    }

});

app.directive('editableListInput', function ($parse, $timeout) {
    return {
        replace: true,
        require: '?ngModel',
        restrict: 'E',
        scope: {
            ngModel: '=',
            type: '@',
            onChange: '&',
            onKeyUpCallback: '&',
            placeholder: '@',
            required: '<',
            bsTypeahead: '=',
            classes: '@',
            unique: '<',
            trimParam: '<',
            checkWarning: '=', // needs to be a function that evaluates fast, it's not debounced
            disableFormValidation: '<', // if true, this item  being invalid doesn't make the parent form invalid
            pattern: '@',
            fullWidthInput: '<'
        },
        templateUrl: '/templates/widgets/editable-list-input.html',
        compile: function() {

            return function(scope, elt, attrs, ngModel) {
                const propertyToCompare = attrs.ngModel.split(EDITABLE_LIST_ITEM_PREFIX + '.')[1];
                var setItemValidity;
                var $elt = $(elt);

                function updateModel(evt) {
                    const localScope = angular.element(evt.target).scope();
                    if (localScope) {
                        $parse(attrs.ngModel).assign(scope.$parent, localScope.ngModel);
                    }
                    if (setItemValidity) {
                        setItemValidity();
                    }
                }

                scope.onKeyUp = function(evt) {
                    updateModel(evt);
                    setTimeout(() => $elt.trigger("keyup"));
                    if (scope.onKeyUpCallback) {
                        scope.onKeyUpCallback();
                    }
                };

                var editableListScope = scope.$parent.$parent;
                scope.parentListItems = editableListScope ? editableListScope.ngModel : [];
                const index = scope.$parent.$index;
                if (editableListScope.editableListForm) {
                    if (scope.disableFormValidation) {
                        editableListScope.editableListForm.$removeControl(ngModel.$$parentForm);
                    } else {
                        editableListScope.editableListForm.$addControl(ngModel.$$parentForm);
                    }
                }

                function checkUnique() {
                    if (!scope.ngModel) {
                        return;
                    }
                    const isUnique = !scope.parentListItems.find((it, idx) => resolveValue(it, propertyToCompare) === scope.ngModel && idx != index);
                    ngModel.$setValidity('unique', isUnique);
                }

                if (scope.unique || scope.required) {
                    setItemValidity = () => {
                        $timeout(() => scope.$parent[EDITABLE_LIST_ITEM_PREFIX].$invalid = ngModel.$$parentForm.$invalid);
                    }
                    setItemValidity();
                }

                //Since we delete and recreate the editableList items on update (add/remove) we loose information on which one have been touched/modified.
                //For validation we need this information to keep displaying the error message when recreated so we store it in the item
                //if parentScope contains different editableListInput we need to differentiate $touched
                scope.touchedId = `$touched.${attrs.ngModel}`;

                scope.onBlur = function() {
                    scope.$parent[EDITABLE_LIST_ITEM_PREFIX][scope.touchedId] = true;
                }

                // if an already saved input is empty/invalid we want to display the error message even if not touched
                if (!editableListScope.hasAddedItem) {
                    scope.onBlur();
                }

                if (scope.unique) {
                    scope.$watch('parentListItems', checkUnique, true);
                }

                let changing = false;
                $(elt[0]).on('change', function(evt) {
                    function doIt(){
                        changing = true;
                        updateModel(evt);
                        $elt.trigger("change");
                        changing = false;
                    }

                    if (!changing) {
                        /* This is the same hack that we did to fix #1222.
                         * When you have a bs-typeahead, you have an non-empty field, then
                         * you chnage data to get a suggestion. Clicking on the suggestion
                         * will exit the input field, triggering a change event.
                         * However, the change event triggers before the click has its own actions:
                         * which is, changing the value of the input and triggering another
                         * "change" and "input" event.
                         * By delaying the taking into account of this, we leave time to the browser
                         * to process the click and to have it repercuted to the Angular model
                         */
                        var uglyBSHack = $(evt.target).attr("bs-typeahead") != null;
                        if (uglyBSHack) {
                            window.setTimeout(doIt, 200);
                        } else {
                            doIt();
                        }
                    }
                });
            }
        }
    }
});

app.directive("timeZoneList", function(CachedAPICalls) {
    return {
        restrict : 'A',
        link : function(scope) {
            CachedAPICalls.timezonesList.then(function(timezonesList) {
                scope.timezone_ids = timezonesList.ids;
            }).catch(setErrorInScope.bind(this));
        }
    }
});

// Performance-oriented one-way binding
// Raw (unescaped) HTML only, no expression, must be updated explicily
// Must be bound in a map, e.g. {a: "Label", b: "<strong>error</strong>"}
app.directive('fastBind', function() {
    return {
        scope: false,
        priority: -1,
        link: function(scope, element, attrs) {
            var elts = [], keys = [], root = element[0];
            element.find('[fast-bound]').each(function(i){
                elts[i] = this;
                keys[i] = this.getAttribute('fast-bound');
            });
            scope[attrs.fastBind] = function(map) {
                if (!map) {
                    element[0].style.visibility = 'hidden';
                } else {
                    for (let i = elts.length - 1; i>=0; i--) {
                        elts[i].innerHTML = map[keys[i]];
                    }
                    root.style.visibility = 'visible';
                }
            };
        }
    };
});

app.directive("smartLogTail", function(){
    return {
        restrict : 'A',
        replace: true,
        scope : {
            smartLogTail : '=',
            emptyPlaceholder: '@',
        },
        template : '<pre class="smart-log-tail-content">'+
                '<span ng-if="smartLogTail.lines.length == 0" style="font-style: italic">{{ emptyPlaceholder }}</span>'+
                '<span ng-repeat="line in smartLogTail.lines track by $index" '+
                    'ng-class="{\'text-error\':  smartLogTail.status[$index] == TAIL_STATUS.ERROR,'+
                           '\'text-warning\': smartLogTail.status[$index] == TAIL_STATUS.WARNING,'+
                           '\'text-success\': smartLogTail.status[$index] == TAIL_STATUS.SUCCESS, }">'+
                    '{{line}}'+
                '</span>'+
                '</pre>',
        link : function(scope){
            scope.TAIL_STATUS = {
                DEBUG: 0,
                INFO: 1,
                WARNING: 2,
                ERROR: 3,
                SUCCESS: 4
            };
        }
    }
});

app.directive("automationEditOverlay", function(){
    return {
        replace:true,
        template : '<div ng-cloak ng-if="appConfig.isAutomation" class="automation-edit-overlay"><div class="text"><div class="line1">Automation node</div><div class="line2">Edits will be lost at next bundle import</div></div></div>',
    }
});

app.directive("infoMessagesList", function(translate){
    return {
        restrict : 'A',
        scope : {
            infoMessagesList : '='
        },
        template : '<ul class="info-messages-list"><li ng-repeat="message in infoMessagesList">'+
                    '<div ng-class="\'message-\' + (message.severity.toLowerCase())">'+
                        '<div ng-if="message.title && message.details">'+
                            '<h4 >{{message.title}}</h4>'+
                            '<span>{{message.details}}</span>'+
                            '<span ng-show="message.line">' + translate('RECIPE.STEP.INFO.AT_LINE', ' (at line {{line}})', { line: '{{message.line}}' }) +
                        '</div>'+
                        '<div ng-if="message.title && !message.details">'+
                            '<span>{{message.title}}</span>'+
                            '<span ng-show="message.line">' + translate('RECIPE.STEP.INFO.AT_LINE', ' (at line {{line}})', { line: '{{message.line}}' }) +
                        '</div>'+

                    '</div>'+
                '</li></ul>',
    }
});

app.directive('masterBreadcrumb', function($rootScope) {
    return {
        templateUrl: '/templates/master-breadcrumb.html',
        scope: true,
        link: function($scope) {
            $scope.breadcrumbData = $rootScope.masterBreadcrumbData;
        }
    }
});

app.service("InfoMessagesModal", function($q, CreateModalFromTemplate){
    var InfoMessagesModal = {
        /* Shows only if there is a message */
        showIfNeeded : function(parentScope, messages, modalTitle, subHeader = null) {

            if (messages.messages.length > 0) {
                return CreateModalFromTemplate("/templates/widgets/info-messages-display.html", parentScope, null, function(newScope){
                    newScope.modalTitle = modalTitle;
                    newScope.messages = messages;
                    newScope.subHeader = subHeader;
                });
            }
            return $q.resolve();
        }
    }
    return InfoMessagesModal;
});

app.directive("refreshCodemirrorOn", function($timeout){
    return {
        link : function($scope, element, attrs) {
            $scope.$watch(attrs.refreshCodemirrorOn, () => {
                $timeout(function(){
                    element.find(".CodeMirror").each(function(_, e) {
                        if (e.CodeMirror) e.CodeMirror.refresh();
                    });
                }, 0);
            });
        }
    }
})

app.filter("infoMessageAlertClass", function(){
    return function(input){
        var dict = {
            'ERROR': 'alert-danger',
            'WARNING': 'alert-warning',
            'INFO': 'alert-info',
            'SUCCESS': 'alert-success',
        };
        return dict[input.severity];
    }
})

app.filter("severityAlertClass", function(){
    return function(input){
        var dict = {
            'ERROR': 'alert-danger',
            'WARNING': 'alert-warning',
            'INFO': 'alert-info'
        };
        return input != null ? dict[input] : 'alert-info';
    }
})

app.directive("infoMessagesRawListWithAlert", function(){
    return {
        templateUrl : '/templates/widgets/info-messages-raw-list-with-alert.html',
        scope : {
            data : '=infoMessagesRawListWithAlert',
            showReduceAction: '='
        }
    }
})

app.directive("deployerInfoMessagesRawListWithAlert", function(){
    return {
        templateUrl : '/templates/widgets/deployer-info-messages-raw-list-with-alert.html',
        scope : {
            data : '=deployerInfoMessagesRawListWithAlert',
            showReduceAction: '=',
            reducedAtStartup: '='
        },
        link: function($scope) {
            $scope.$watchCollection("data.messages", () => {
                const messages = $scope.data.messages.slice().sort((m1, m2) => {
                    if (m1.severity === 'ERROR') return -1;
                    if (m2.severity === 'ERROR') return 1;
                    if (m1.severity === 'WARNING') return -1;
                    if (m2.severity === 'WARNING') return 1;
                    if (m1.severity === 'SUCCESS') return -1;
                    if (m2.severity === 'SUCCESS') return 1;
                    return 0;
                });

                $scope.data.$reduced = $scope.reducedAtStartup;

                $scope.getAlertSeverityClass = function() {
                    switch($scope.data.maxSeverity) {
                        case 'ERROR':
                            return 'alert-danger';
                        case 'WARNING':
                            return 'alert-warning';
                        case 'SUCCESS':
                            return 'alert-success';
                        case 'INFO':
                            return 'alert-info';
                    }
                }

                const nbError = messages.filter(m => m.severity === 'ERROR').length;
                const nbWarning = messages.filter(m => m.severity === 'WARNING').length;

                if (nbError && nbWarning) {
                    $scope.header = `${nbError} error${nbError > 1 ? 's' : ''} and ${nbWarning} warning${nbWarning > 1 ? 's' : ''} were encountered`;
                }
                else if (nbError) {
                    $scope.header = `${nbError} error${nbError > 1 ? 's were' : ' was'} encountered`;
                }
                else if (nbWarning) {
                    $scope.header = `${nbWarning} warning${nbWarning > 1 ? 's were' : ' was'} encountered`;
                }
                else if ($scope.data.messages.filter(m => m.severity === 'SUCCESS').length) {
                    $scope.header = "Success";
                }
                else {
                    $scope.header = "Information";
                }
            });
        }
    }
})

app.directive("infoMessagesRawList", function(){
    return {
        templateUrl : '/templates/widgets/info-messages-raw-list.html',
        scope : {
            data : '=infoMessagesRawList'
        }
    }
})

app.directive("featureLocked", function(){
    return {
        templateUrl : '/templates/widgets/feature-locked.html',
        restrict : 'EA',
        scope : {
            featureName : '='
        }
    }
});

app.directive("qrCode", function(){
    return {
        scope : {
            qrCode : '='
        },
        template : "<div class='qr'></div>",
        link : function($scope, element) {
            $scope.$watch("qrCode", function(nv) {
                if (nv) {
                    /* global QRCode */
                    new QRCode(element.find(".qr")[0], {
                        text : $scope.qrCode,
                        width: 128,
                        height: 128,
                        colorDark : "#000000",
                        colorLight : "#ffffff",
                        correctLevel : QRCode.CorrectLevel.H
                    });
                }
            });
        }
    }
});

app.directive("dkuFoldable", function(){
    return {
        scope : true,
        controller : ['$scope', '$attrs', function($scope, $attrs) {
            $scope.foldableOpen = $scope.$eval($attrs.open);
        }],
        link : function($scope, _element, attrs){
            $scope.foldableToggle = function(){
                $scope.foldableOpen = !$scope.foldableOpen;
            };
            function setChevronClazz(){
                $scope.foldableChevronClazz = $scope.foldableOpen ? "dku-icon-chevron-up-16" : "dku-icon-chevron-down-16";
            }
            $scope.$watch("foldableOpen", setChevronClazz);
            setChevronClazz();
            $scope.$watch(attrs.open, function(nv, ov){
                if (nv != ov) $scope.foldableToggle();
            });
        }
    }
});

app.directive("dkuFoldableRightPanel", function(LocalStorage, STANDARDIZED_SIDE_PANEL_KEY){
    return {
        scope : true,
        require : "dkuFoldable",
        link : function($scope, _element, attrs) {
            let objectType = $scope.objectType !== undefined ? $scope.objectType : "defaultObjectType";
            // Strip the LOCAL_ and FOREIGN_ prefix to have the same expanded/collapsed section in the Flow and in the actual object item view (ex: Dataset).
            if (objectType.startsWith("LOCAL_")) {
                objectType = objectType.substring(6);
            } else if (objectType.startsWith("FOREIGN_")) {
                objectType = objectType.substring(8);
            }
            const key = STANDARDIZED_SIDE_PANEL_KEY + '.' + objectType + '.' + attrs.name;
            let localValue = LocalStorage.get(key);
            if(localValue !== undefined){
                $scope.foldableOpen =  localValue;
            }
            $scope.$watch("foldableOpen", function(nv, ov){
                if (nv != ov) {
                    LocalStorage.set(key, nv);
                }
            });
        }
    }
});

/*
 * Add on for dkuFoldable
 */
app.directive("openOnDragEnter", function($timeout) {
    return {
        restrict: 'A',
        link : function($scope, element) {
            var previousState = null;
            var nesting = 0;

            element.on('dragenter', () => {
                if (nesting++ === 0) {
                    previousState = $scope.foldableOpen;
                    $scope.$apply(function() {
                        $scope.foldableOpen = true;
                    });
                }
            });
            element.on('dragleave', () => {
                if (--nesting === 0) {
                    $scope.$apply(function() {
                        $scope.foldableOpen = previousState;
                        previousState = null;
                    });
                }
            });
            $(document).on('dragend', () => {
                $timeout(function() { nesting = 0; });
            });
        }
    }
});

app.directive("rightColumnDescriptionTags", function(){
    return {
        scope : {
            object : '='
        },
        templateUrl : '/templates/widgets/right-column-description-tags.html'
    }
});

/*
 * Verry usefull to repeat template passed through transclude in a ng-repeat directive.
 * In the HTML template 'inject' directive call should be at the same level as 'ng-repeat' directive call
 */

app.directive('dkuInject', function(){
  return {
    link: function($scope, $element, $attrs, controller, $transclude) {
      if (!$transclude) {
        /* global minErr, startingTag */
        throw minErr('ngTransclude')('orphan',
         'Illegal use of ngTransclude directive in the template! ' +
         'No parent directive that requires a transclusion found. ' +
         'Element: {0}',
         startingTag($element));
      }
      var innerScope = $scope.$new();
      $transclude(innerScope, function(clone) {
        $element.empty();
        $element.append(clone);
        $element.on('$destroy', function() {
          innerScope.$destroy();
        });
      });
    }
  };
});

/*
 * Custom carousel : can take any html template and use it to populate the slides (makes use the dkuInject directive to combine transclude and ngRepeat)
 */

app.directive('dkuCarousel', function($timeout){
    return {
        transclude: true,
        templateUrl: '/templates/projects/dku-carousel.html',
        restrict: 'A',
        scope: {
            entries : '=dkuCarousel',
            initialIndex : '=?'
        },
        link: function($scope, element){

            $scope.element = element;

            $scope.index = 0;
            if ($scope.initialIndex) {
                $scope.index = $scope.initialIndex;
            }

            $scope.slideLeft = function() {
                if (!$scope.entries) return;
                var maxIndex = $scope.entries.length - 1;
                var newIndex = $scope.index - 1;
                if (newIndex < 0) {
                    newIndex = maxIndex;
                }
                slide(newIndex, -1);
            };

            $scope.slideRight = function() {
                if (!$scope.entries) return;
                var maxIndex = $scope.entries.length - 1;
                var newIndex = $scope.index + 1;
                if (newIndex > maxIndex) {
                    newIndex = 0;
                }
                slide(newIndex, 1);
            };

            var slide = function (newIndex, direction) {
                var slider = $(element).find('.slider');

                // In order to give the illusion the carousel is wrapping
                var firstSlideClone = $(element).find('.slide:first-child').clone().addClass('clone');
                var lastSlideClone = $(element).find('.slide:last-child').clone().addClass('clone');
                $(slider).prepend(lastSlideClone);
                $(slider).append(firstSlideClone);

                var slides = $(element).find('.slide');
                var domNbSlides = $scope.entries.length + 2;        //since we've juste added to new slides
                var domIndex = $scope.index + 1;    //since we've just preprended a new slide
                var newDomIndex = domIndex + 1 * direction;

                var leftPosition = -1 * domIndex * 100 / domNbSlides;
                var newLeftPosition = -1 * newDomIndex * 100 / domNbSlides;
                var sliderWidth = domNbSlides * 100;
                $(slider).addClass('animating');
                $(slider).css('width', sliderWidth + '%');
                $(slider).css('transform', 'translate(' + leftPosition + '%, 0)');
                $(slides).css('width', 100/domNbSlides + '%');

                $timeout(function() {
                    $(slider).addClass('transition');
                    $(slider).css('transform', 'translate(' + newLeftPosition + '%, 0)');
                }, 0);

                $timeout(function() {
                    $(slider).removeClass('transition');
                    $scope.index = newIndex;
                    $(slider).removeAttr('style');
                    $(slider).removeClass('animating');
                    $(slides).removeAttr('style');
                    firstSlideClone.remove();
                    lastSlideClone.remove();
                }, 200);
            }
        }
    };
});

app.directive('displayAmount', function(){
    return {
        template: '<span>{{amount}} {{unit}}<span ng-if="amount > 1">s</span></span>',
        restrict: 'AE',
        scope: {
            unit: '=',
            amount: '='
        },
        link: function(){
            // noop
        }
    };
});


app.directive('editableText', function($timeout){
    return {
        template: '<div class="dku-editable-text">' +
        '<div ng-show="!editing" ng-click="edit()" class="horizontal-flex">' +
        '<div class="flex mx-textellipsis">{{model || placeholder}}</div><div class="noflex"><i class="icon-pencil" /></div>' +
        '</div>' +
        '<input type="text" ng-model="model" placeholder="{{placeholder}}" ng-blur="editing = false" ng-show="editing" blur-on-enter />' +
        '</div>',
        restrict: 'A',
        scope: {
            model: '=editableText',
            placeholder: '='
        },

        link: function($scope, element){
            var input = $(element.find('input'));
            $scope.edit = function() {
                $scope.editing = true;
                $timeout(function() {
                    input.focus();
                });
            };
        }
    };
});

app.directive('ngRightClick', function($parse) {
    return function (scope, element, attrs) {
        var fn = $parse(attrs.ngRightClick);
        element.bind('contextmenu', function (event) {
            scope.$apply(function () {
                if (attrs.ngRightClickPreventDefault !== 'false') {
                    event.preventDefault();
                }
                fn(scope, {$event: event});
            });
        });
    };
});

// intercept clicks to allow preventing it based on a condition
// works for any mouse button
app.directive('checkClickable', function() {
    return {
        scope: { allowClick: '<checkClickable' },
        link: function(scope, element) {
            const maybePreventClick = (event) => {
                const allow = typeof scope.allowClick === 'function' ? scope.allowClick() : scope.allowClick;
                if(!allow) {
                    event.preventDefault();
                }
            };
            element.on('click', maybePreventClick);
            element.on('auxclick', maybePreventClick);
        }
    }
});

//TODO: use this factory in folder_edit.js instead of openMenu() when the two branchs will be merged together
app.factory("openDkuPopin", function($timeout, $compile) {
    //Args in options: template, isElsewhere, callback, popinPosition = "SMART", onDismiss, doNotCompile, arrow
    return function($scope, $event, options) {
        var opts = angular.extend({popinPosition: 'SMART'}, options);
        var newDOMElt = $(opts.template);

        var newScope = $scope.$new();
        if (!opts.doNotCompile) {
            $compile(newDOMElt)(newScope);
        }

        // By default an arrow is displayed if the popin position strategy is SMART
        if (!angular.isDefined(opts.arrow)) {
            opts.arrow = opts.popinPosition == 'SMART';
        }

        newScope.dismiss = function(skipOnDismiss){
            newScope.$destroy();
            if (typeof(opts.onDismiss) === "function" && !skipOnDismiss) {
                opts.onDismiss(newScope);
            }
        };

        newScope.$on("$destroy", () => $timeout(() => {
            newDOMElt.remove();
            $('body').off('click', hideOnClickElsewhere);
            $('body').off('contextmenu', hideOnClickElsewhere);
        }))

        var hideOnClickElsewhere = function(e) {
            if(typeof(opts.isElsewhere)==="function" && opts.isElsewhere(newDOMElt, e)) {
                newScope.dismiss();
            }
        };

        $timeout(function(){
            newScope.$apply(function() {
                $("body").append(newDOMElt);
                switch (opts.popinPosition) {
                    case "SMART":
                        smartPositionning();
                        break;
                    case "CLICK":
                        clickPositionning();
                        break;
                    default:
                        break;
                }
                newDOMElt.show();
                $('body').on('click', hideOnClickElsewhere);
                $('body').on('contextmenu', hideOnClickElsewhere);
                if (typeof(opts.callback) === "function") {
                    opts.callback(newScope);
                }
            });
        });


        // Positions the popin so that it is always aligned with one side of the element
        function smartPositionning() {
            var element_X = $($event.target).offset().left;
            var element_Y = $($event.target).offset().top;
            var element_W = $($event.target).outerWidth(true);
            var element_H = $($event.target).outerHeight(true);
            var popin_W = $(newDOMElt).outerWidth(true);
            var popin_H = $(newDOMElt).outerHeight(true);
            var window_W = window.innerWidth;
            var window_H = window.innerHeight;

            var popin_on_bottom = (element_Y + element_H + popin_H < window_H);
            var popin_aligned_on_right = (element_X + popin_W > window_W);

            var popin_X = (popin_aligned_on_right) ? (element_X + element_W - popin_W) : (element_X);
            var popin_Y = (popin_on_bottom) ? (element_Y + element_H) : (element_Y - popin_H);

            newDOMElt.css("top", popin_Y);
            newDOMElt.css("left", popin_X);

            // Add an arrow linking the popin to the element which triggered it
            if (opts.arrow) {
                var cssClass = "";
                cssClass += (popin_on_bottom) ? 'bottom-' : 'top-';
                cssClass += (popin_aligned_on_right) ? 'left' : 'right';
                newDOMElt.addClass(cssClass);
                newDOMElt.addClass(popin_on_bottom ? 'bottom' : 'top');
            }
        }

        // Positions the popin so that its content is displayed at the location of the mouse click
        function clickPositionning() {
            var mouse_X = $event.clientX;
            var mouse_Y = $event.clientY;
            var popin_W = $(newDOMElt).outerWidth(true);
            var popin_H = $(newDOMElt).outerHeight(true);
            var window_W = window.innerWidth;
            var window_H = window.innerHeight;

            var popin_on_bottom = (mouse_Y + popin_H < window_H);
            var popin_on_right = (mouse_X + popin_W < window_W);

            var popin_X = (popin_on_right) ? (mouse_X) : (mouse_X - popin_W);
            var popin_Y = (popin_on_bottom) ? (mouse_Y) : (mouse_Y - popin_H);

            newDOMElt.css("top", popin_Y);
            newDOMElt.css("left", popin_X);

            // Add an arrow linking the popin to the element which triggered it
            if (opts.arrow) {
                var cssClass = "";
                cssClass += (popin_on_bottom) ? 'bottom-' : 'top-';
                cssClass += (popin_on_right) ? 'right' : 'left';
                newDOMElt.addClass(cssClass);
            }
        }

        //returning function to remove popin
        return newScope.dismiss;
    }
});

app.factory('TreeViewSortableService', function() {
    let currentNode;
    return {
        setCurrent: (node) => {currentNode = node;},
        getCurrent: () => currentNode
    };
});

app.directive('treeView', function($timeout, openDkuPopin, TreeViewSortableService) {
    return {
        templateUrl: '/templates/widgets/tree-view-node.html',
        restrict: 'AE',
        scope: {
            nodes: '=treeView',
            rootNodes: '=?',
            depth: '<',
            nodeName: '<',
            onClick: '<',
            onArrowClick: '<',
            iconClass: '<',
            iconTitle: '<',
            rightIconClass: '<',
            rightIconTitle: '<',
            nodeClass: '<',
            showDragHandles: '<',
            scrollToNodeFn: '=?',
            getTaxonomyMassExpandCollapseStateFn: '=?',
            expandAllFn: '=?',
            collapseAllFn: '=?',
            setReduceFn: '=?',
            setUnfoldedNodeIdsFn: '<?',
            getUnfoldedNodeIdsFn: '<?',
            getNodeIdsHavingChildrenFn: '<?',
            getRightClickMenuTemplate: '<?',
            contextMenuFns: '=?',
            showContextMenu: '=?'
        },
        link: function($scope, $el) {
            if (!$scope.nodes) {
                throw new Error("No nodes provided to tree view");
            }


            $scope.depth = $scope.depth || 0;
            $scope.MARGIN_PER_DEPTH = 15;
            $scope.uiState = {};

            $scope.activateSortable = function() {
                const parentNode = ($scope.$parent && $scope.$parent.node || $scope.uiState);
                (TreeViewSortableService.getCurrent() || {}).$sortableEnabled = false;
                parentNode.$sortableEnabled = true;
                $scope.nodes.forEach(n => n.$reduced = true);
                TreeViewSortableService.setCurrent(parentNode);
            };

            // disabled dragstart event in order to disable native browser drag'n'drop and allow mouse up event when drag'n'dropping
            $timeout(() => $el.find('> ul > li > div.tree-view-node > div > span.handle-row').each((idx, el) => { el.ondragstart = () => false }));
            $scope.onSortableStop = function() {
                for (let i = 0; i < $scope.nodes.length; i++) {
                    delete $scope.nodes[i].$tempReduced;
                }
            };
            $scope.onNodeMouseDown = function(node) {
                node.$tempReduced = true;
            };
            $scope.treeViewSortableOptions = {
                axis:'y',
                cursor: 'move',
                cancel: '',
                handle: '.tree-view-drag-handle',
                update: $scope.onSortableStop
            };

            $scope.onNodeClick = function(node, evt) {
                if (!$(evt.originalEvent.explicitOriginalTarget).is('i.icon-reorder')) {
                    $scope.onClick(node);
                }
            };

            $scope.openContextMenu = function (node, $event) {
                if (!$scope.showContextMenu) {
                    return;
                }

                node.$rightClicked = true;

                let template = `<ul class="dropdown-menu" ng-click="popupDismiss()">`;
                switch ($scope.getNodeMassExpandCollapseStateFn(node.id)) { //NOSONAR
                    case "EXPAND_ALL":
                        template += `
                            <li class="dku-border-bottom">
                                <a href="#" ng-click="expandChildren('`+node.id+`')">
                                    <i class="dku-icon-chevron-double-down-16 icon-fixed-width" /> Expand all
                                </a>
                            </li>`;
                        break;
                    case "COLLAPSE_ALL":
                        template += `
                            <li class="dku-border-bottom">
                                <a href="#" ng-click="collapseChildren('`+node.id+`');">
                                    <i class="dku-icon-chevron-double-right-16 icon-fixed-width" /> Collapse children
                                </a>
                            </li>`;
                        break;
                    default:
                        break;
                }
                template += $scope.getRightClickMenuTemplate(node);

                template += `</ul>`;

                let isElsewhere = function (elt, e) {
                    let result = $(e.target).parents('.dropdown-menu').length == 0;
                    if (result) {
                        delete node.$rightClicked;
                    }
                    return result;
                };

                $scope.popupDismiss = openDkuPopin($scope, $event, {template:template, isElsewhere:isElsewhere, popinPosition:'CLICK'});
            };

            // Return an array containing all the parent nodes and the searched node itself
            function getAncestorsOfNodeId(nodeId, nodes = $scope.rootNodes, ancestors=[]) {
                let n = nodes.find(n => n.id == nodeId);

                if (angular.isDefined(n)) {
                    return ancestors.concat(n);
                } else {
                    for (let i=0; i<nodes.length; i++) {
                        let node = nodes[i];
                        if (node.children && node.children.length > 0) {
                            let r = getAncestorsOfNodeId(nodeId, node.children, ancestors.concat(node));
                            if (r) {
                                return r;
                            }
                        }
                    }
                    return null;
                }
            }

            // Return an array containing all the children nodes and the node itself
            function getDescendantsOfNode(node, descendants=[node]) {
                if (node.children.length > 0) {
                    for (let i=0; i<node.children.length; i++) {
                        let child = node.children[i];
                        let r = getDescendantsOfNode(child, [child]);
                        descendants = descendants.concat(r);
                    }
                }
                return descendants;
            }

            function getNodeFromNodeId(nodeId, nodes = $scope.rootNodes) {
                let n = nodes.find(n => n.id == nodeId);

                if (angular.isDefined(n)) {
                    return n;
                } else {
                    for (let i=0; i<nodes.length; i++) {
                        let node = nodes[i];
                        if (node.children && node.children.length > 0) {
                            let r = getNodeFromNodeId(nodeId, node.children);
                            if (r) {
                                return r;
                            }
                        }
                    }
                    return null;
                }
            }


            /*** Folding & Unfolding functions ***/
            $scope.getTaxonomyMassExpandCollapseStateFn = function() {
                let IDsUnfolded = $scope.getUnfoldedNodeIdsFn();
                let IDsHavingChildren = $scope.getNodeIdsHavingChildrenFn();

                if (IDsHavingChildren.length == 0) {
                    return "";
                } else if (IDsHavingChildren.length > IDsUnfolded.length) {
                    return "EXPAND_ALL";
                } else {
                    return "COLLAPSE_ALL";
                }
            };

            $scope.getNodeMassExpandCollapseStateFn = function(nodeId) {
                let node = getNodeFromNodeId(nodeId);
                if (angular.isDefined(node)) {
                    let descendants = getDescendantsOfNode(node).filter(n => n.id != nodeId);
                    let hasDescendants = descendants.length > 0;
                    let hasCollapsedChildren = !!descendants.find(n => n.$reduced == true && angular.isDefined(n.children) && n.children.length > 0);
                    let hasGrandChildren = !!descendants.find(n => angular.isDefined(n.children) && n.children.length > 0);
                    if (!hasDescendants) {
                        return "";
                    } else if (node.$reduced || hasCollapsedChildren) {
                        return "EXPAND_ALL";
                    } else if (hasGrandChildren) {
                        return "COLLAPSE_ALL";
                    } else {
                        return "";
                    }
                }
            }

            $scope.expandAllFn = function() {
                setReduceMultiple($scope.nodes, false);
            };

            $scope.collapseAllFn = function() {
                setReduceMultiple($scope.nodes, true);
            };

            $scope.expandChildren = function(nodeId) {
                let node = getNodeFromNodeId(nodeId);
                setReduceMultiple([node], false);
            };

            $scope.collapseChildren = function(nodeId) {
                let node = getNodeFromNodeId(nodeId);
                if (angular.isDefined(node.children)) {
                    setReduceMultiple(node.children, true);
                }
            };

            function expandNodes(nodes) {
                setReduceMultiple(nodes, false, false);
            }

            function setReduceMultiple(nodes, fold, recursive = true) {
                if (!angular.isDefined(nodes) || !nodes.length || nodes.length == 0) {
                    return;
                }

                angular.forEach(nodes, function(node) {
                    $scope.setReduceFn(node, fold);

                    if (recursive) {
                        setReduceMultiple(node.children, fold, recursive);
                    }
                });
            }

            $scope.setReduceFn = function(node, reduce) {
                // No need to set the value if it hasn't changed
                if (!angular.isDefined(node) || node.$reduced == reduce) {
                    return;
                }

                if (reduce) {
                    node.$reduced = true;
                } else {
                    delete node.$reduced;
                }

                // Keeping parent aware of this change if he gave us some way to.
                if (typeof($scope.getUnfoldedNodeIdsFn)==="function" && typeof($scope.setUnfoldedNodeIdsFn)==="function") {
                    let unfoldedNodeIds = $scope.getUnfoldedNodeIdsFn();
                    let index = unfoldedNodeIds.indexOf(node.id);

                    if (reduce) {
                        if (index > -1) {
                            unfoldedNodeIds.splice(index, 1);
                        }
                    } else {
                        if (index == -1 && node.children.length > 0) {
                            unfoldedNodeIds.push(node.id);
                        }
                    }
                    $scope.setUnfoldedNodeIdsFn(unfoldedNodeIds);
                }
            }


            /*** Scrolling functions ***/
            $scope.scrollToNodeFn = function(nodeId, duration) {
                let ancestors = getAncestorsOfNodeId(nodeId);

                if (ancestors) {
                    let node = ancestors.pop();
                    expandNodes(ancestors);
                    $timeout(() => $scope.triggerScroll(node, duration));
                }
            };

            $scope.shouldScrollToNode = function(node) {
                return node.$scrollToMe;
            };

            $scope.scrollDuration = function(node) {
                return node.$scrollDuration;
            };

            $scope.triggerScroll = function(node, duration) {
                node.$scrollToMe = true;
                node.$scrollDuration = duration;
            };

            $scope.onScrollTriggered = function(node) {
                node.$scrollToMe = false;
                delete node.$scrollDuration;
            };
        }
    };
});


app.directive("dkuHtmlTooltip", function($timeout, openDkuPopin) {
    return {
        template : '<div ng-mouseenter="displayTooltip($event)" ng-mouseleave="removeTooltip()" class="dku-html-tooltip-activation-zone {{triggerClass}}"><div ng-transclude="trigger"></div><div style="display: none;" ng-transclude="content"></div></div>',
        restrict: 'A',
        scope: {
            fromModal: '=?',
            tooltipClass: '@',
            triggerClass: '@',
            position: '@?', // if not present, uses openDkuPopin, otherwises sets the position manually
        },
        transclude: {
            'trigger': 'tooltipTrigger',
            'content': 'tooltipContent'
        },
        link: function($scope, elmnt, attrs) {

            var tooltipDisplayed = false;

            /*
             * _removeTooltip will be returned by openDkuPopin service when popin will actually be displayed.
             * This is b/c openDkuPopin create a new $scope and a DOM element when displaying a popin,
             * and returns a method to destroy this specific scope and remove new DOM element from DOM.
             */
            var _removeTooltip = function() {
                // noop
            };

            $scope.removeTooltip = function() {
                _removeTooltip();
                _removeTooltip = function() {
                    // noop
                };
                tooltipDisplayed = false;
            };

            // Remove the tooltip when the current directive is destroy while it was opened
            $scope.$on('$destroy', ()=> _removeTooltip());

            $scope.displayTooltip = function($event) {
                if (!tooltipDisplayed && (!attrs.dkuHtmlTooltipShow || $scope.$parent.$eval(attrs.dkuHtmlTooltipShow))) {
                    var cssClass = $scope.fromModal ? "dku-html-tooltip from-modal" : "dku-html-tooltip";
                    var content = $(elmnt).find('tooltip-content')[0].innerHTML;
                    var template = `<div class="${cssClass} ${$scope.tooltipClass || ''}">${content}</div>`;
                    var isElsewhere = function (tooltipElement, event) {
                        return $(event.target).parents('.dku-html-tooltip').length == 0 || $(event.target).parents('.dku-html-tooltip')[0] != tooltipElement;
                    };
                    if (
                        ['top', 'bottom', 'left', 'right', 'top-right', 'top-left', 'bottom-right', 'bottom-left'].includes(
                            $scope.position
                        )
                    ) {
                        _removeTooltip = openManuallyPositionedTooltip(template, elmnt, $scope.position);
                    } else {
                        var dkuPopinOptions = {
                            template: template,
                            isElsewhere: isElsewhere,
                            doNotCompile: true
                        };
                        _removeTooltip = openDkuPopin($scope, $event, dkuPopinOptions);
                    }
                    tooltipDisplayed = true;
                }
            };

            /*
             * Opens a tooltip for the element, using the template, at a manually set position
             * (top, bottom, right, left, top-right, top-left, bottom-right, bottom-left)
             * Returns a function that removes the tooltip
             */
            function openManuallyPositionedTooltip(template, element, position) {
                let tooltip = $(template);
                $('body').append(tooltip); //so we have access to dimensions

                let posLeft = 0;
                let posTop = 0;
                // left/top offset of the element
                const left = $(element).offset().left;
                const top = $(element).offset().top;
                // (Outer) width/height of the element
                const outerWidth = $(element).outerWidth();
                const outerHeight = $(element).outerHeight();
                // (Outer) width/heigth of the tooltip
                const tooltipOuterWidth = tooltip.outerWidth();
                const tooltipOuterHeight = tooltip.outerHeight();
                // Margin (= 2 * size of the arrow)
                const margin = 10;

                // We set the tooltip left/top offset according to desired position
                if (position == 'top') {
                    posLeft = left + outerWidth / 2 - tooltipOuterWidth / 2;
                    posTop = top - tooltipOuterHeight - 3 * margin / 2;
                } else if (position == 'top-right') {
                    posLeft = left + outerWidth;
                    posTop = top - tooltipOuterHeight - 3 * margin / 2;
                } else if (position == 'top-left') {
                    posLeft = left - tooltipOuterWidth + margin;
                    posTop = top - tooltipOuterHeight - 3 * margin / 2;
                } else if (position == 'bottom-left') {
                    posLeft = left - tooltipOuterWidth + margin;
                    posTop = top + outerHeight + margin / 2;
                } else if (position == 'bottom-right') {
                    posLeft = left + outerWidth;
                    posTop = top + outerHeight + margin / 2;
                } else if (position == 'bottom') {
                    posLeft = left + outerWidth / 2 - tooltipOuterWidth / 2;
                    posTop = top + outerHeight + margin / 2;
                } else if (position == 'left') {
                    posLeft = left - tooltipOuterWidth - margin / 2;
                    posTop = top - tooltipOuterHeight / 2;
                } else if (position == 'right') {
                    posLeft = left + outerWidth + 3 * margin / 2;
                    posTop = top - tooltipOuterHeight / 2;
                }
                tooltip.css('left', posLeft);
                tooltip.css('top', posTop);
                tooltip.addClass(position); // Adds the arrow at the right position

                $('body').append(tooltip);

                function dismiss() {
                    tooltip.remove();
                }

                return dismiss;
            }
        }
    }
});


app.directive("summaryOfError", function(Dialogs){
    return {
        // No isolated scope because we need to attach modals to the parent
        restrict: 'A',
        scope: true,
        templateUrl: "/templates/errors/summary-of-error.html",
        link: function($scope, _element, attrs) {


            $scope.$watch(attrs.summaryOfError, (nv) => {
                $scope.error = nv;
                $scope.isCredentialError = $scope.error && $scope.error.code && ($scope.error.code==="ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL" || $scope.error.code==="ERR_CONNECTION_NO_CREDENTIALS");
            });


            $scope.openMoreInfo = function(){
                 Dialogs.displaySerializedError($scope, $scope.error);
            }
        }
    }
});

    app.directive("errorFixability", function($rootScope, $state){
    return {
        restrict: 'A',
        templateUrl: "/templates/errors/error-fixability.html",
        scope: {
            error : "="
        },
        link: function($scope) {
            $scope.$state = $state;
            $scope.wl = $rootScope.wl;
        }
    }
});

app.directive('barMetrics', function () {
    return {
        scope: {data: '=barMetrics', height:'='},
        template: '<svg class="gpu-bar-metrics"></svg>',
        link: function ($scope, el) {
            const HEIGHT = $scope.height || 10;

            let svg = d3.select(el[0]).select('svg');
            let defs = svg.append('defs');

            let mainGradient = defs.append('linearGradient')
                .attr('id', 'mainGradient');

            mainGradient.append('stop')
                .attr('class', 'gpu-bar-metrics__stop-left')
                .attr('offset', '0');
            mainGradient.append('stop')
                .attr('class', 'gpu-bar-metrics__stop-right')
                .attr('offset', '1');

            let container = d3.select(el[0]).select('svg').append('g').attr('class', 'container').attr('transform', `translate(0, ${Math.max(HEIGHT/2,10)})`);
            $scope.$watchCollection('data', function (data) {
                if (data) {
                    const PREFIX_WIDTH = el[0].clientWidth / 4;
                    const POSTFIX_WIDTH = el[0].clientWidth / 5;
                    const MAX_BAR_WIDTH = el[0].clientWidth-PREFIX_WIDTH-POSTFIX_WIDTH;
                    const scales = data.map(d => d3.scale.linear().domain([d.min, d.max]).range([0, MAX_BAR_WIDTH]));
                    let g = container.selectAll('g.metric').data($scope.data);
                    let SPACE_BETWEEN = 8;
                    let gEnter = g.enter().append('g').attr('class', 'metric').attr('transform', (d, i) => `translate(0, ${i * (HEIGHT + SPACE_BETWEEN )})`);
                    gEnter.append('rect').attr('x',PREFIX_WIDTH).attr('y',-HEIGHT/2).attr('width', MAX_BAR_WIDTH).attr('height', HEIGHT).classed('gpu-bar-metrics__filled', true);
                    gEnter.append('rect').attr('x',PREFIX_WIDTH).attr('y',-HEIGHT/2).attr('class', 'val-inverted').attr('width', MAX_BAR_WIDTH).attr('height', HEIGHT).attr('fill', '#ececec');
                    gEnter.append('text').attr("alignment-baseline","middle").attr('text-anchor','start').attr('y', 0).attr('height', 50).text(d => d.label).attr('x', 0).attr("font-size", "10px").attr("fill", "grey");

                    gEnter.append('text').attr("alignment-baseline","middle").attr('text-anchor', 'end').attr('class', 'percentage').attr('y', 0).attr('x', el[0].clientWidth).attr("font-size", "10px").attr("fill", "grey");
                    g.select('text.percentage').text(d => `${Math.round(d.value * 100 / d.max)} %`);
                    g.select('rect.val-inverted').transition().attr('width', (d, i) => MAX_BAR_WIDTH - scales[i](d.value)).attr('x', (d, i) => PREFIX_WIDTH + scales[i](d.value));
                }
            });
        }
    }
});

app.directive('gitCheckoutSelect', function (SpinnerService, DataikuAPI) {
    return {
        scope: {
            gitCheckout: '=ngModel',
            gitRepository: '=repository',
            gitLogin: '=login',
            gitPassword: '=password'
        },
        templateUrl: '/templates/widgets/git-checkout-select.html',
        link: function ($scope) {
            $scope.gitCustomCheckout = true;
            $scope.gitLoadedRefsRepo = ''; // The repository where the currently loaded references are from

            $scope.$watch('gitRepository', function(nv) {
                if (nv && nv !== '') {
                    // When you change the input repository, you shouldn't have the old references list
                    $scope.gitCustomCheckout = $scope.gitRepository !== $scope.gitLoadedRefsRepo;
                }
            });

            $scope.listRemoteRefs = function () {
                // If the parent scope can show the error (`block-api-error` directive), it'll be better presented that way
                // Otherwise, we'll display it in that directive
                const errorScope = angular.isFunction($scope.$parent.setError) ? $scope.$parent : $scope;

                if ($scope.gitRepository && $scope.gitRepository !== '') {
                    resetErrorInScope(errorScope);

                    SpinnerService.lockOnPromise(DataikuAPI.git.listRemoteRefs($scope.gitRepository, $scope.gitLogin, $scope.gitPassword)
                        .then(function(response) {
                            $scope.gitCheckoutRefs = response.data;
                            $scope.gitCustomCheckout = false;
                            $scope.gitLoadedRefsRepo = $scope.gitRepository
                        }, function(err) {
                            $scope.gitCheckoutRefs = [];
                            $scope.gitCustomCheckout = true;
                            $scope.gitLoadedRefsRepo = '';
                            errorScope.setError(err);
                        })
                    )
                }
            };
        }
    }
});

app.directive('gitRemoteUrl', function (SpinnerService, DataikuAPI) {
    return {
        scope: {
            gitRef: '=ngModel'
        },
        templateUrl: '/templates/widgets/git-remote-url.html',
        link: function ($scope) {
            $scope.remoteUrlPaste = (event) => {
                if (!$scope.gitRef.remote || $scope.gitRef.remote.trim() === '') {
                    const pastedUrl = (event.originalEvent.clipboardData.getData("text") || '').trim();
                    if (pastedUrl) {
                        const matches = /^(https?:\/\/)([^:@]+):?([^@]+)?@(.+)$/i.exec(pastedUrl);
                        if (matches && matches.length === 5) {
                            const [,scheme,login,password,rest] = matches;
                            $scope.gitRef.remote = scheme + rest;
                            $scope.gitRef.remoteLogin = login;
                            $scope.gitRef.remotePassword = password;
                            event.preventDefault();
                            event.stopPropagation();
                            return false;
                        }
                    }
                }
                return true;
            };

            $scope.isHTTPSRemote = () => $scope.gitRef && /^\s*https?:/i.test($scope.gitRef.remote || '');
        }
    }
});

/**
 * Monitoring directive that displays information about currently available gpus, and can allow selection of gpus
 *
 * @param {Array<string>>} metrics - the metrics to display from nvidia-smi e.g. ['GPU','Memory']
 * @param {Array} selected - the selected gpus (works a bit like ng-model=listOfSelectedGPUS)
 * @param {boolean} selectable - whether the template will be selectable via mouseclick
 * @param {boolean} singleGpuOnly - whether selection will be forced to 1 gpu
*/
app.directive('gpuOnlineStats', function ($interval, Notification,Logger) {
    return {
        scope: {metrics:"=", selected:"=", onSelectedChange:"&?", selectable:"=", singleGpuOnly: "=?"},
        transclude: true,
        restrict: 'A',
        template: `<div ng-if="gpuCount" class="gpu-online-stats-wrapper">
                        <div ng-show="gpuStats && gpuResponse.status === 'OK'" ng-repeat="i in range(gpuCount)" class="gpu-online-stats__gpu-block" ng-click="selectable && selected && clickOnGpu(i)" ng-class="{selected:isGPUselected(i),'gpu-online-stats__gpu-block--selectable':selectable }">
                            <i class="icon icon-dku-gpu-card"></i>
                            <div>
                                <div class="gpu-online-stats__title">
                                    <span show-tooltip-on-text-overflow text-tooltip="gpuResponse.stats[i].name +' ['+ gpuResponse.stats[i].index + ']'" tooltip-direction="'tooltip'" ></span>
                                    <span class="gpu-online-stats__secondary-title">{{gpuResponse.stats[i].memoryTotal}} MB</span>
                                </div>
                                <div bar-metrics="gpuStats[i]" class="gpu-online-stats__gpu-graph" height="6"></div>
                            </div>
                        </div>
                    </div>
                    <div class="alert alert-error" ng-show="gpuResponse.status === 'ERROR'" style="text-align: center">
                        <span>{{gpuResponse.error}}</span>
                    </div>`,
        link: function (scope) {
            if (scope.selected && !(scope.selected instanceof Array)) {
                Logger.error(`gpuOnlineStats directive accepts a Array as a <selected> parameter, but ${scope.selected.constructor.name} was given`)
                scope.selected = null;
            }
            Notification.publishToBackend('gpu-monitoring-start');

            let KEEP_ALIVE_INTERVAL_MS = 2*1000;
            let cancelKeepAlive = $interval(function () {
                Notification.publishToBackend('timeoutable-task-keepalive', {
                    taskId: "gpu-monitoring"
                });
            }, KEEP_ALIVE_INTERVAL_MS);
            scope.range = function (n) {
                return Array.range(n);
            };

            scope.clickOnGpu = i => {
                if (!scope.selected.includes(i)){
                    if (scope.singleGpuOnly) { // first need to remove all the others
                        scope.selected.length = 0;
                    }
                    scope.selected.push(i);

                } else {
                    scope.selected.splice(scope.selected.indexOf(i),1);
                }

                if (angular.isFunction(scope.onSelectedChange)) {
                    scope.onSelectedChange();
                }
            };
            scope.isGPUselected = i => scope.selected && scope.selected.includes(i);

            scope.$on('$destroy', function () {
                $interval.cancel(cancelKeepAlive);
            });
            Notification.registerEvent("gpu-stats-response", function (evt, message) {
                scope.gpuResponse = message.response;
                if (scope.gpuResponse.status === 'OK') {

                    scope.gpuStats = message.response.stats.map(g => [{
                        label: 'Memory',
                        value: g.memoryUsed,
                        min: 0,
                        max: g.memoryTotal
                    }, {
                        label: 'GPU',
                        value: g.utilizationGpu,
                        min: 0,
                        max: 100
                    }].filter(d => d.value !== undefined && (!scope.metrics || scope.metrics.includes(d.label))));
                    scope.gpuCount = scope.gpuStats.length;
                }
            });
        }
    }
});
app.component('gpuSelector', {
    templateUrl: '/templates/widgets/gpu-selector.html',
    bindings: {
        gpuConfig: '<',
        gpuCapabilities: '<',
        onSettingsChange: '&?',
        selectedEnv: '<',
        name: '<',
        inContainer: '<',
        forceSingleGpu: '<',
        envMode: '<?',
        defaultEnvName: '<?',
        isScoreEval: '<?',
        llmMeshInUse: '<?'
    },
    controller: function($rootScope, $stateParams, $attrs, Notification, DataikuAPI, $filter, GpuUsageService, GPU_SUPPORTING_CAPABILITY, Debounce) {
        const $ctrl = this;
        $ctrl.numGpus = undefined;
        $ctrl.gpuStatus = undefined;
        $ctrl.possibleUsageWarning = "";
        $ctrl.currentEnvName = "";
        $ctrl.changeSettings = () => {}; // no-op unless configured in onInit
        $ctrl.onNumGpusChange = onNumGpusChange;
        $ctrl.GPU_SUPPORTING_CAPABILITY = GPU_SUPPORTING_CAPABILITY;
        $ctrl.allowModeChange = true;

        let lastGpuStats = [];
        let gpuCount;

        $ctrl.$onInit = () => {
            $ctrl.allowModeChange = GpuUsageService.allowChangingMode(!!$ctrl.isScoreEval, $ctrl.gpuCapabilities)

            getGpuUsability();
            $ctrl.currentEnvName = GpuUsageService.getCurrentEnvName($ctrl.selectedEnv, $ctrl.envMode, $ctrl.defaultEnvName);

            const isKeras = $ctrl.gpuCapabilities.includes(GPU_SUPPORTING_CAPABILITY.KERAS);
            if (isKeras && !$ctrl.gpuConfig.params.perGPUMemoryFraction) {
                $ctrl.gpuConfig.params.perGPUMemoryFraction = 0.7;
            }
            registerGpuStatsResponseEvent();

            if (angular.isFunction($ctrl.onSettingsChange)) {
                $ctrl.changeSettings = Debounce().withDelay(200, 500).wrap(() => $ctrl.onSettingsChange());
            }

            // needed to ensure all gpu params are in sync
            disableGpuBasedOnEnvSupport();
            updateForContainerizedExecution();
        };

        $ctrl.$onChanges = (changes) => {
            if ($ctrl.gpuConfig) {
                if (changes.inContainer) {
                    updateForContainerizedExecution();
                }

                if (changes.selectedEnv || changes.envMode || changes.defaultEnvName) {
                    $ctrl.currentEnvName = GpuUsageService.getCurrentEnvName($ctrl.selectedEnv, $ctrl.envMode, $ctrl.defaultEnvName);
                    disableGpuBasedOnEnvSupport();
                }

                if (changes.gpuCapabilities) {
                    getGpuUsability();
                    disableGpuBasedOnEnvSupport();
                }
            }
        };

        /**
         * Checks how many gpus can be used given the current enabled capabilities
         */
        function getGpuUsability() {
            const gpuUsability = GpuUsageService.checkGpuUsability($ctrl.gpuCapabilities, $ctrl.forceSingleGpu);
            $ctrl.singleGpuOnly = gpuUsability.singleGpuOnly;
            $ctrl.possibleUsageWarning = gpuUsability.possibleUsageWarning;
        }

        /**
         * Checks the selected code env, and disables the gpu exec option if it is known to not be supported by the env
         */
        function disableGpuBasedOnEnvSupport() {
            if ($ctrl.isScoreEval) {
                // We don't verify whether the environment supports GPU when in either the scoring or eval recipes.
                // Given that the model has already been created, we don't need to bother automatically disabling settings, based on what the
                // environment 'might' support
                $ctrl.envSupportsGpu = true;
            }
            else {
                GpuUsageService.getEnvSupportsGpu($rootScope.appConfig.isAutomation, $ctrl.gpuCapabilities, $ctrl.currentEnvName).then(result => {
                    $ctrl.envSupportsGpu = result;
                    if (!$ctrl.envSupportsGpu) {
                        $ctrl.gpuConfig.params.useGpu = false;
                    }
                });
            }
        }

        /**
         * Checks whether task is set to run in container, and sets the number of gpus based on the gpulist
         */
        function updateForContainerizedExecution() {
            if ($ctrl.inContainer) {
                if ($ctrl.gpuConfig.params.gpuList && $ctrl.gpuConfig.params.gpuList.length > 0) {
                    $ctrl.numGpus = $ctrl.gpuConfig.params.gpuList.length;
                } else {
                    // normally don't hit this, as new tasks set gpu0 in gpuList
                    // here in case someone changes from non-container to container
                    $ctrl.numGpus = 1;
                    onNumGpusChange(); // does paramUpdate call
                }
            }
        }

        /**
         * Event listener for the 'number of desired gpus' input when running in container
         * Creates a gpulist based on the selected number of gpus
         */
        function onNumGpusChange() {
            if ($ctrl.numGpus && $ctrl.numGpus > 0) {
                $ctrl.gpuConfig.params.gpuList = Array.range($ctrl.numGpus);
                $ctrl.onSettingsChange()
            }
        }

        function registerGpuStatsResponseEvent() {
            Notification.registerEvent("gpu-stats-response", function (evt, message) {
                $ctrl.gpuStatus = message.response.status;
                if (!gpuCount) {
                    gpuCount = message.response.stats.length;
                }

                if (gpuCount > 0 && gpuCount <  $ctrl.gpuConfig.params.gpuList.length) {
                    // available gpus appear to changed, resetting selection
                    $ctrl.gpuConfig.params.gpuList = Array.range(gpuCount);
                }
                lastGpuStats = message.response.stats;
            });
        }
    }
});

    /*
        See NPSSurveyState for NPS survey timing logic
    */
    app.component('npsSurvey', {
        templateUrl: '/templates/widgets/nps-survey.html',
        bindings: {
            appConfig: '='
        },
        controller: ['$rootScope', '$scope', '$timeout', '$filter', 'DataikuAPI', 'WT1', 'Logger',
            function npsSurveyCtrl($rootScope, $scope, $timeout, $filter, DataikuAPI, WT1, Logger) {
                const ctrl = this;
                ctrl.Actions = {
                    NEW: 1,
                    SUBMIT: 2,
                    POSTPONE: 3,
                    OPTOUT: 4
                };
                ctrl.showSurvey = false;
                ctrl.active = false;
                ctrl.finished = false;
                ctrl.scores = Array.from({length: 11}, (_, i) => (i));
                ctrl.response = '';
                ctrl.futureSurveyParticipation = false;

                ctrl.display = function(){
                    ctrl.email = this.appConfig.user && this.appConfig.user.email ? this.appConfig.user.email : '';
                    ctrl.showSurvey = true;
                    $timeout(function() { ctrl.active = true; });
                }

                // we do not have a service where to trigger the display of the service
                // instead we expose the display method onto the rootScope
                $rootScope.showNpsSurvey = ctrl.display;

                ctrl.hide = function hide(){
                    $timeout(() => {
                        ctrl.active = false;
                        // to avoid a flicker showing the form again when done submitting
                        $timeout(() => {
                            ctrl.showSurvey = false;
                            ctrl.finished = false;
                            ctrl.response = '';
                            ctrl.futureSurveyParticipation = false;
                            ctrl.email = this.appConfig.user && this.appConfig.user.email ? this.appConfig.user.email : '';
                            ctrl.surveyScore = undefined;
                        }, 0);
                    }, 1000);
                }

                ctrl.$onInit = function() {
                    if (ctrl.appConfig) {
                        Logger.debug("appConfig loaded, initializing.");
                        if (ctrl.appConfig.npsSurveyActive) {
                            ctrl.display();
                        }
                    } else {
                        Logger.info("appConfig not loaded yet, watching it to wait for its initialization");
                        const deregisterAppConfigListener = $scope.$watch("$ctrl.appConfig", () => {
                            if (ctrl.appConfig) {
                                Logger.debug("appConfig loaded, initializing.");
                                if (ctrl.appConfig.npsSurveyActive) {
                                    ctrl.display();
                                }
                                deregisterAppConfigListener();
                            }
                        });
                    }
                };

                ctrl.finish = function(action) {
                    // if user clicks on something after submitting
                    if (ctrl.finished) {
                        return;
                    }

                    DataikuAPI.profile.setNPSSettings(action).success((data) => {
                        let eventParams = {
                            action: action,
                            npsState: data && data.state ? data.state : ''
                        };

                        if (action === 'SUBMIT') {
                            WT1.event('nps-survey', angular.extend(eventParams, {
                                score: ctrl.surveyScore,
                                response: $filter('escapeHtml')(ctrl.response || ''),
                                email: $filter('escapeHtml')(ctrl.email || ''),
                                futureSurveyParticipation: ctrl.futureSurveyParticipation || false,
                            }));
                            ctrl.finished = true;
                            ctrl.hide();
                        } else {
                            ctrl.hide();

                            WT1.event('nps-survey-decline', eventParams);
                        }
                    }).error(setErrorInScope.bind($scope));
                };

                ctrl.selectScore = function(score) {
                    ctrl.surveyScore = score;
                };
            }
        ]
    });


/*
    Tiny directive to handle the display of a sort icon in a table.

    Fields:
      * isSortCol: whether the current col is used for sorting (and the icon should be displayed)
      * ascending: whether the sort is ascending
      * iconOnRight: whether the icon is put on the right of the column name

    Besides, if you want to display a grayer version of the icon when hovering on the column name,
    your must add the .contains-sort-by-column-icon in the container of the column

    Example:
        <div class="my-container contains-sort-by-column-icon">
            <span>My title column</span>
            <sort-by-column-icon isSortCol="isSortCol" ascending="selection.orderReverse" icon-on-right="true"></sort-by-column-icon>
        </div>
*/
app.directive("sortByColumnIcon", function() {
    return {
        scope: {
            isSortCol: "=",
            ascending: "=",
            iconOnRight: "@"
        },
        template: `<div class="sort-by-column-icon__wrapper" ng-class="iconOnright ? 'icon-on-right' : 'icon-on-left'">
                      <i ng-if="!isSortCol" class="icon-sort-by-attributes-alt sort-by-column-icon--display-on-hover"></i>
                      <i ng-if="isSortCol && ascending" class="icon-sort-by-attributes"></i>
                      <i ng-if="isSortCol && !ascending" class="icon-sort-by-attributes-alt"></i>
                   </div>`
    }
});

app.filter('formatConfusionMatrixValue', function() {
    return function(value, isWeighted) {
        if (typeof value !== 'number') {
            return value; // for when it's percentage
        }
        return isWeighted ? value.toFixed(2) : value.toFixed(0);
    };
});

/*
    Binary Classification Confusion Matrix widget
*/
app.directive("bcConfusionMatrix", function() {
    return {
        templateUrl:"/templates/widgets/bc_confusion_matrix.html",
        scope: {
            modelClasses: "=",
            data: "=",
            displayMode: "=",
            metricsWeighted: "="
        },
        controller: function($scope) {
            // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
            $scope.puppeteerHook_elementContentLoaded = true;
        }
    }
});

/*
    Multi Classification Confusion Matrix widget
*/
app.directive("mcConfusionMatrix", function() {
    return {
        templateUrl:"/templates/ml/prediction-model/mc_confusion.html",
        scope: {
            modelData: "=",
            displayMode: "=",
            metricsWeighted: "="
        },
        controller: 'MultiClassConfusionMatrixController'
    }
});

/**
 * Generic component that displays a d3 brush.
 */
app.directive('rangeBrush', function(Fn) {
    return {
        restrict : 'A',
        templateUrl : '/templates/widgets/range-brush.html',
        scope : {
            range : '=',
            selectedRange : '=',
            onChange : '&',
            onDrillDown : '&',
            snapRanges : '=',
            onInit: '&',
            brushWidth: '@',
            enablePadding: '=?'
        },
        replace : true,
        link : function($scope, element) {
            $scope.enablePadding = angular.isDefined($scope.enablePadding) ? $scope.enablePadding : true;

            const padding = $scope.enablePadding ? 10 : 4; // 4 to be able to display the handles
            const brushHeight = 60;
            const dateLineHeight = 25;
            const triggerHeight = 18;
            const triggerWidth = 0.8 * triggerHeight;
            const handleHeight = 20;
            const handleWidth = 8;
            const separatorHeight = 3;
            const separatorOffset = -2;

            // the svg needs to update when the width changes (different # of ticks, for ex)
            var eventName = 'resize.brush.' + $scope.$id;
            $(window).on(eventName, function() { if ( $scope.range != null ) $scope.refreshRange();});
            $scope.$on('$destroy', function(){$(window).off(eventName)});
            // also add a watch on the width for the cases where the size changes as a result of
            // stuff being shown/hidden
            $scope.$watch(
                function () {return element.width();},
                function () { if ( $scope.range != null ) $scope.refreshRange(); }
            );

            // get the brush : the root of the template
            var brushSvg = d3.select(element[0]);
            // resize
            brushSvg.attr("height", brushHeight);

            // add stuff in the svg (layer in this order: to get the display and click-sensitivity right)
            var xAxisG = brushSvg.append("g").attr("class", "x axis").attr("transform", "translate(0, " + dateLineHeight + ")");

            var brushG = brushSvg.append("g").attr("class", "x brush"); // brush (catches mouse events to drag handles and brush extend)
            var triggersG = brushSvg.append("g").attr("class", "x triggers").attr("transform", "translate(0, " + dateLineHeight + ")"); // layer with the triggers, clickable
            var brushInversionG = brushSvg.append("g").attr("class", "x brush-inversion"); // the inverse of the brush (click-through)
            var brushHandlesG = brushSvg.append("g").attr("class", "x brush-handles"); // the brush handles (click-through)
            var brushContentG = brushSvg.append("g").attr("class", "x brush-content"); // where to append stuff (like chart preview)

            const brushContentWidth = $scope.brushWidth - (2 * padding);

            if ($scope.brushWidth) {
                brushSvg.style("width", $scope.brushWidth + 'px');
                brushContentG.attr("transform", "translate(" + padding + ", 0)");
                brushContentG.attr("width", brushContentWidth);
            }

            brushSvg.on('dblclick', () => {
                var insideExtent = false;
                if ( xScale != null ) {
                    var pos = d3.mouse(element[0]);
                    var xPos = xScale.invert(pos[0]).getTime();
                    insideExtent = $scope.selectedRange.from < xPos && $scope.selectedRange.to > xPos;
                }
                if (insideExtent && $scope.onDrillDown() != null) {
                    // why can't I pass the first () in the html? dunno...
                    $scope.onDrillDown()($scope.selectedRange.from, $scope.selectedRange.to);
                } else {
                    $scope.selectedRange.from = $scope.range.from;
                    $scope.selectedRange.to = $scope.range.to;
                    $scope.refreshRange();
                    $scope.onChange();
                }
            });

            var xScale = null;

            // update the total range, and then the graph
            $scope.refreshRange = function() {
                if ( $scope.range == null || $scope.selectedRange == null ) {
                    return;
                }
                var width = $(element).innerWidth();
                // the full range we are selecting in
                var axisRange = [new Date($scope.range.from), new Date($scope.range.to)];
                // the selected range
                var extentRange = [new Date($scope.selectedRange.from), new Date($scope.selectedRange.to)];
                // make the scale
                xScale = d3.time.scale().domain(axisRange).range([padding, width - padding]);
                // prepare the brush callback
                var brushed = function() {
                    var extent = brush.extent();
                    // If we are simply clicking on the brush (one point interval), go back to previous range.
                    if (extent[1] - extent[0] === 0) {
                        extent[0] = new Date($scope.selectedRange.from);
                        extent[1] = new Date($scope.selectedRange.to);
                    }
                    if (d3.event.mode === "move") {
                        if ( $scope.rounding == 'day') {
                            var startDay = d3.time.day.round(extent[0]);
                            var daySpan = Math.round((extent[1] - extent[0]) / (24 * 3600 * 1000));
                            var endDay = d3.time.day.offset(startDay, daySpan);
                            extent = [startDay, endDay];
                        }
                    } else {
                        if ( $scope.rounding == 'day') {
                            extent = extent.map(d3.time.day.round);
                            if (extent[0] >= extent[1] ) {
                                extent[0] = d3.time.day.floor(extent[0]);
                                extent[1] = d3.time.day.ceil(extent[1]);
                            }
                        }
                    }
                    d3.select(this).call(brush.extent(extent));
                    var xS = xScale(extent[0]);
                    var xE = xScale(extent[1]);
                    brushInversionG.selectAll('.s').attr("x", 0).attr("width", xS);
                    brushInversionG.selectAll('.e').attr("x", xE).attr("width", width - xE);
                    brushHandlesG.selectAll(".s").attr("transform", "translate(" + xS + ", 0)");
                    brushHandlesG.selectAll(".e").attr("transform", "translate(" + xE + ", 0)");

                    $scope.selectedRange.from = extent[0].getTime();
                    $scope.selectedRange.to = extent[1].getTime();
                    $scope.onChange();
                };

                // make the brush
                var brush = d3.svg.brush().x(xScale).on("brush", brushed).extent(extentRange);
                // make the axis from the scale
                var xAxis = d3.svg.axis().scale(xScale).tickFormat(Fn.getCustomTimeFormat()).orient("top").tickSize(-(brushHeight - dateLineHeight));
                // and create the svg objects
                var a = xAxisG.call(xAxis);
                var b = brushG.call(brush);
                triggersG.selectAll("*").remove();
                var t = triggersG.selectAll(".trigger").data($scope.snapRanges);

                var xS = xScale(extentRange[0]);
                var xE = xScale(extentRange[1]);

                // draw the triggers
                var triggerPadding = (brushHeight - dateLineHeight - triggerHeight) / 2;
                t.enter().append("path")
                    .classed("trigger", true)
                    .classed("success", function(d) {return d.outcome == 'SUCCESS';})
                    .classed("failed", function(d) {return d.outcome == 'FAILED';})
                    .classed("aborted", function(d) {return d.outcome == 'ABORTED';})
                    .classed("warning", function(d) {return d.outcome == 'WARNING';})
                    .attr("d", function(d) { return "M" + xScale(new Date(d.start)) + "," + triggerPadding + " l"+triggerWidth+","+(triggerHeight/2)+" l-"+triggerWidth+","+(triggerHeight/2)+" z"; })
                    .on("click", function(d){
                        $scope.selectedRange.from = d.start; $scope.selectedRange.to = d.end; $scope.refreshRange(); $scope.onChange();
                    });

                // remove the axis line
                a.selectAll(".domain").remove();

                // style the brush
                b.selectAll("rect").attr("y", 0).attr("height", brushHeight);
                // create the handles the handles
                brushHandlesG.selectAll(".resize").remove();
                brushHandlesG.append("g").classed("resize", true).classed("s", true).attr("transform", "translate(" + xS + ", 0)");
                brushHandlesG.append("g").classed("resize", true).classed("e", true).attr("transform", "translate(" + xE + ", 0)");
                var bh = brushHandlesG.selectAll(".resize");
                bh.append("rect").classed("separator", true).attr("y", 0).attr("height", brushHeight).attr("x", separatorOffset).attr("width", separatorHeight);
                bh.append("rect").classed("handle", true).attr("y", (brushHeight - handleHeight) / 2).attr("height", handleHeight).attr("x", -(handleWidth/2)).attr("width", handleWidth);
                // add the invert of the brush for the overlay outside of the brush
                brushInversionG.selectAll("rect").remove();
                brushInversionG.append("rect").attr("x", 0).attr("width", xS).attr("y", 0).attr("height", brushHeight).classed("s", true);
                brushInversionG.append("rect").attr("x", xE).attr("width", width - xE).attr("y", 0).attr("height", brushHeight).classed("e", true);
            };

            // add event handler to adjust the brush when the selection changes
            $scope.$watch('range', (nv) => {
                if ( nv == null ) return;
                $scope.refreshRange();
            }, true);
            $scope.$watch('snapRanges', (nv) => {
                if ( nv == null ) return;
                $scope.refreshRange();
            }, true);
            $scope.$watch('selectedRange', (nv) => {
                if ( nv == null ) return;
                $scope.refreshRange();
            }, true);
            $scope.$watch('brushWidth', () => {
                brushSvg.style("width", $scope.brushWidth + 'px');
            }, true);

            $scope.onInit && typeof $scope.onInit === 'function' && $scope.onInit({ brushContentG: brushContentG, brushContentHeight: brushHeight, brushContentWidth: brushContentWidth });
        }
    };
});
app.directive('datasetCreatorSelector', function ($parse) {
    return {
        templateUrl: '/templates/widgets/dataset-creator-selector.html',
        require:'^ngModel',
        scope: {
            ngModel: '=',
            managedDatasetOptions: '=',
            newDataset: '=',
            canCreate: '=',
            canSelectForeign: '=',
            markCreatedAsBuilt: '=',
            qa: '@'
        },
        controller: ['$scope', 'DataikuAPI', '$stateParams', 'DatasetUtils', 'SqlConnectionNamespaceService', function ($scope, DataikuAPI, $stateParams, DatasetUtils, SqlConnectionNamespaceService) {
            addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);
            $scope.partitioningOptions = [{id: "NP", label: "Not partitioned"}];
            $scope.io = {"newOutputTypeRadio": "create"};
            $scope.uiState = {mode:'select'};

            $scope.isCreationAllowed = angular.isDefined($scope.canCreate) ? $scope.canCreate : false;

            $scope.getDatasetCreationSettings = function () {
                let datasetCreationSetting = {
                    connectionId: ($scope.newDataset.connectionOption || {}).id,
                    specificSettings: {
                        overrideSQLCatalog: $scope.newDataset.overrideSQLCatalog,
                        overrideSQLSchema: $scope.newDataset.overrideSQLSchema,
                        formatOptionId: $scope.newDataset.formatOptionId,
                    },
                    partitioningOptionId: $scope.newDataset.partitioningOption,
                    inlineDataset: $scope.inlineDataset,
                    zone : $scope.zone,
                    markCreatedAsBuilt: $scope.markCreatedAsBuilt,
                };
                if ($scope.newDataset &&
                    $scope.newDataset.connectionOption &&
                    $scope.newDataset.connectionOption.fsProviderTypes &&
                    $scope.newDataset.connectionOption.fsProviderTypes.length > 1) {
                    datasetCreationSetting['typeOptionId'] = $scope.newDataset.typeOption;
                }
                return datasetCreationSetting;
            };

            $scope.createAndUseNewOutputDataset = function (force) {
                const projectKey = $stateParams.projectKey,
                    datasetName = $scope.newDataset.name,
                    settings = $scope.getDatasetCreationSettings();

                if (force) {
                    doCreateAndUseNewOutputDataset(projectKey, datasetName, settings);
                } else {
                    DataikuAPI.datasets.checkNameSafety(projectKey, datasetName, settings).success(data => {
                        $scope.uiState.backendWarnings = data.messages;
                        if (!data.messages || !data.messages.length) {
                            doCreateAndUseNewOutputDataset(projectKey, datasetName, settings);
                        }
                    }).error(setErrorInScope.bind($scope));
                }
            };

            function doCreateAndUseNewOutputDataset(projectKey, datasetName, settings) {
                DataikuAPI.datasets.newManagedDataset(projectKey, datasetName, settings)
                    .success(dataset => {
                        DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success((data) => {
                            $scope.uiState.mode = 'select';
                            if (!$scope.canSelectForeign) {
                                data = data.filter(fs => fs.localProject);
                            }
                            $scope.availableDatasets = data;
                            $scope.uiState.model = dataset.name;
                        });
                    }).error(setErrorInScope.bind($scope));
            }

            DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success((data) => {
                if (!$scope.canSelectForeign) {
                    data = data.filter(fs => fs.localProject);
                }
                $scope.availableDatasets = data;
            });

            DataikuAPI.datasets.getManagedDatasetOptionsNoContext($stateParams.projectKey).success(function (data) {
                $scope.managedDatasetOptions = data;
                if (!$scope.newDataset.connectionOption && data.connections.length) {
                    const fsConnection = data.connections.find(e => {
                        return 'Filesystem' === e.connectionType;
                    });
                    if (fsConnection) {
                        $scope.newDataset.connectionOption = fsConnection;
                    } else {
                        $scope.newDataset.connectionOption = data.connections[0];
                    }
                }
                if (!$scope.newDataset.formatOptionId && $scope.newDataset.connectionOption.formats.length) {
                    $scope.newDataset.formatOptionId = $scope.newDataset.connectionOption.formats[0].id;
                }
                $scope.partitioningOptions = [
                    {"id": "NP", "label": "Not partitioned"},
                ].concat(data.projectPartitionings);

            });

            $scope.fetchCatalogs = function(connectionName, connectionType) {
                SqlConnectionNamespaceService.listSqlCatalogs(connectionName, $scope, 'dataset-creator-selector', connectionType);
            };

            $scope.fetchSchemas = function(connectionName, connectionType) {
                const catalog = $scope.newDataset.overrideSQLCatalog ||
                    ($scope.newDataset.connectionOption ? $scope.newDataset.connectionOption.unoverridenSQLCatalog : '');
                SqlConnectionNamespaceService.listSqlSchemas(connectionName, $scope, catalog, 'dataset-creator-selector', connectionType);
            };

            $scope.$watch("newDataset.connectionOption.connectionName", function() {
                if ($scope.newDataset.connectionOption) {
                    SqlConnectionNamespaceService.setTooltips($scope, $scope.newDataset.connectionOption.connectionType);
                }

                SqlConnectionNamespaceService.resetState($scope, $scope.newDataset);
            });

        }],
        link: function (scope, el, attrs, ngModel) {
            scope.uiState = scope.uiState || {};
            scope.uiState.model = $parse(attrs.ngModel)(scope.$parent);
            scope.$watch("uiState.model", (nv, ov) => {
                if (ov === nv) return;
                if (nv === ngModel.$viewValue) return;
                // set the new value in the field and render
                ngModel.$setViewValue(nv);
                ngModel.$render();
            });
            scope.$watch("ngModel", (nv, ov) => {
                if (ov === nv) return;
                if (nv === scope.uiState.model) return;
                scope.uiState.model = ngModel.$viewValue;
            });
        }
    }
})

    /**
     * Display tags list if it fits else only one + the rest inside a popover.
     *
     * @param {Array}       items                   - List of the tags to display
     * @param {Array}       highlighted             - List of the tags to highlight (put in bold)
     * @param {Object}      tagsMap                 - Map of all the available tags. Used mainly to get the tag colors
     * @param {Array}       globalTagsCategories    - List of existing global tag categories applying to this object type
     * @param {string}      objectType              - Taggable type of the object on which the tags are applied
     * @param {boolean}     editable                - If editable, displays the add tag/category buttons
     * @param {Function}    onTagClick              - The method to call when clicking on a tag
     *                                                  - called with $event and tag when clicking on a tag -> onTagClick($event, tag)
     *                                                  - called with $event and category when clicking on a tag category -> onTagClick($event, category)
     *                                                  - called with just $event when clicking on the add tag button -> onTagClick($event)
     * @param {boolean}     displayPopoverOnHover   - Whether the popover should be displayed on hover on the ellipsis (not equivalent to !editable because you might still want to click on the tag if you can't edit, i.e. objects list page)
     *
     * <responsive-tags-list items="item.tags" tags-map="projectTagsMap" object-type="'OBJECT_TYPE'" on-tag-click="selectTag($event, tag)"></responsive-tags-list>
     */
    app.directive('responsiveTagsList', function ($timeout, Assert, translate) {
        return {
            templateUrl: '/templates/analysis/responsive-tags-list.html',
            scope: {
                items: '<',
                highlighted: '<',
                tagsMap: '<',
                globalTagsCategories: '<',
                objectType: '<',
                editable: '=?',
                onTagClick: '&?',
                displayPopoverOnHover: '<'
            },
            link: function ($scope, $element, attrs) {
                const ELLIPSIS_BUTTON_WIDTH = 45;
                let tagsListObserver;
                let tagsListContainer;
                $scope.canTagsFit = true;
                $scope.highlighted = $scope.highlighted || []; // default value
                $scope.translate = translate;

                // Bootstrap Popover's generated content sometimes looses AngularJS scope.
                // So we cannot rely on the later and therefore have to handle tag click through window object.
                // See https://github.com/angular/angular.js/issues/1323
                if (!window.onTagClick) {
                    window.onTagClick = (event) => {
                        if (!$scope.onTagClick) return;
                        const category = $(event.target).attr('data-category');
                        const tag = $(event.target).closest('.responsive-tags-list__tag').attr('data-tag');
                        $scope.onTagClick({$event: event, tag: tag, category: category});
                        $scope.$apply();
                    }
                }

                function computeFittingTags(tagsList) {
                    const fittingTags = { tags: [], categories: [] };
                    const containerHeight = tagsListContainer.offsetHeight;
                    const containerWidth = tagsListContainer.offsetWidth;

                    for (let index = 0; index < tagsList.length; index++) {
                        const tag = $(tagsList[index])[0];
                        const heightFits = tag.offsetTop + tag.offsetHeight < containerHeight;
                        const isLastLine = tag.offsetTop + 2 * tag.offsetHeight > containerHeight;
                        const widthFits = !isLastLine || tag.offsetLeft + tag.offsetWidth + ELLIPSIS_BUTTON_WIDTH < containerWidth;
                        if (heightFits && widthFits) {
                            if (index < $scope.items.length) {
                                fittingTags.tags.push($scope.items[index]);
                            } else if ($scope.globalTagsCategories && index < $scope.items.length + $scope.globalTagsCategories.length) {
                                fittingTags.categories.push($scope.globalTagsCategories[index - $scope.items.length]);
                            }
                        } else if (!heightFits) {
                            break;
                        }
                    }
                    return fittingTags;
                }

                if (!('IntersectionObserver' in window)) {
                    // Graceful degradation: will simply drop the tags in a scrollable area.
                    $scope.hasFitBeenChecked = true;
                } else {
                    const options = {
                        threshold: [0, 0.5, 1],
                        root: $element[0].parentElement
                    };
                    const intersectionCallback = (entries) => {
                        const entry = entries[0];
                        $scope.canTagsFit = entry.intersectionRatio === 1;

                        tagsListContainer = $element[0].parentElement;
                        if (!$scope.canTagsFit && !$scope.fittingTags) {
                            $scope.fittingTags = computeFittingTags(entry.target.children);
                        }
                        $scope.hasFitBeenChecked = true;
                        $scope.$apply();
                    }
                    tagsListObserver = new IntersectionObserver(intersectionCallback, options);
                    // Wait for template to be injected before checking
                    $timeout(() => {
                        tagsListObserver.observe($element[0].querySelector('#tag-list'));
                    }, 0);
                }

                // Detect element resize
                const resizeObserver = new ResizeObserver(() => {
                    // display #tag-list to trigger intersectionObserver
                    // and see which tags fit if it overflows
                    delete $scope.fittingTags;
                    $scope.canTagsFit = true;
                    $timeout(() => $scope.$apply());
                });
                resizeObserver.observe($element[0].parentElement);

                $scope.$watch("items", () => {
                    // display #tag-list to trigger intersectionObserver
                    // and see which tags fit if it overflows
                    delete $scope.fittingTags;
                    $scope.canTagsFit = true;
                    $timeout(() => $scope.$apply());
                }, true);

                $scope.$on('$destroy', () => {
                    tagsListObserver && tagsListObserver.disconnect();
                    resizeObserver && resizeObserver.disconnect();
                    delete window.onTagClick;
                });
            }
        }
    });

    app.controller("ResponsiveTagsListPopoverController", function($scope, translate) {
        $scope.translate = translate;
        $scope.getPlacement = function(tip, el) {
            let offset = $(el).offset();
            let top = offset.top;
            let height = $(document).outerHeight();
            return 0.5 * height - top > 0 ? 'bottom' : 'top';
        }
    })

    app.directive('codeViewer', function() {
        return {
            restrict: 'AE',
            replace: true,
            templateUrl : '/templates/widgets/code-viewer.html',
            scope : {
                code : '<',
                mimeType : '@'
            },
            controller : function($scope, CodeMirrorSettingService) {
                $scope.codeMirrorSettings = CodeMirrorSettingService.get(
                    $scope.mimeType || 'text/plain', {onLoad: function(cm){
                        cm.setOption("readOnly", true);
                        $scope.codeMirror = cm;
                    }}
                );

                $scope.uiRefreshToggle = false;

                function toggle() {
                    $scope.uiRefreshToggle = !$scope.uiRefreshToggle;
                }
                $scope.$on("codeViewerUIRefreshToggle", toggle);
            }
        }
    });

    /**
     * Directive for managing a list of steps
     *
     * Example usage:
     * <div stepper="stepperState.stepInfo" current-step="stepperState.currentStep" disable-steps="!disableSteps" />
     *
     * stepper (steps): An array of step objects
     * - step object:
     *   - label:       step header
     *   - description: step subtitle
     *   - getError(): function which returns an error message if there is one
     *   - getWarning(): function with returns a warning message if there is one (errors take precedence)
     *   - postAction(): function called right after leaving a step
     * currentStep: variable containing the current step (integer)
     * disableSteps: boolean for disabling step interaction (use a function to disable individual steps)
     */
    app.directive('stepper', function() {
        return {
            restrict: 'A',
            templateUrl: "/templates/widgets/stepper.html",
            scope: {
                steps: '<stepper',
                currentStep: '=',
                disableSteps: '<'
            },
            link : function(scope) {
                let previousStep = scope.currentStep;

                function goToStep() {
                    if (previousStep === scope.currentStep) return;

                    const stepInfo = scope.steps[previousStep];

                    // perform actions before leaving previous step
                    if (stepInfo && stepInfo.postAction) {
                        stepInfo.postAction();
                    }

                    previousStep = scope.currentStep;
                }

                scope.stepClicked = function(step) {
                    if (!scope.disableSteps && scope.currentStep !== step) {
                        scope.currentStep = step;
                    }
                }

                scope.$watch('currentStep', goToStep);
            }
        };
    });
    app.directive('stepperStep', function() {
        return {
            restrict: 'A',
            templateUrl: "/templates/widgets/stepper-step.html",
            scope: {
                stepNumber: '<',
                step: '<stepperStep',
                isLastStep: '<',
                isCurrentStep: '<',
                isCompletedStep: '<',
                disableStep: '<'
            },
            link: function(scope) {
                scope.step.getError = scope.step.getError || (() => '');
                scope.step.getWarning = scope.step.getWarning || (() => '');
            }
        };
    });

    /** Discrete progress ring with different sections (can be styled in css).
     * values           Array of values (each value is the number of sections for a given class)
     * maximum          Maximum number of sections
     * centerValue      Value to be shown in the center of the progress ring
     * classes          Array of classes for styling
     * classNotFilled   Class of the remaining sections if maximum > sum of values
     * radius           Radius of the progress ring (in px)
     * strokeWidth      Width of the stroke (in px)
     * maxAngle         Maximum angle of the progress ring (180 = half circle / 360 = full circle) (in deg)
     **/
    app.component('progressRing', {
        templateUrl: '/templates/widgets/progress-ring.html',
        bindings: {
            values: '<',
            maximum: '<',
            centerValue: '<',
            classes: '<',
            classNotFilled: '@',
            radius: '<',
            strokeWidth: '<',
            maxAngle: '<',
        },
        controller: [
            function () {
                const ctrl = this;

                ctrl.$onChanges = function () {
                    reDraw();
                };

                function reDraw() {
                    // Compute width and height of the svg depending on the radius and maxAngle
                    ctrl.svgWidth = ctrl.maxAngle > 180 ? 2 * ctrl.radius : (ctrl.maxAngle / 180) * 2 * ctrl.radius;

                    if (ctrl.maxAngle < 90) {
                        ctrl.svgHeight = (ctrl.maxAngle / 90) * ctrl.radius;
                    } else if (ctrl.maxAngle < 180) {
                        ctrl.svgHeight = ctrl.radius;
                    } else if (ctrl.maxAngle < 270) {
                        ctrl.svgHeight = (ctrl.maxAngle / 90 - 1) * ctrl.radius;
                    } else {
                        ctrl.svgHeight = 2 * ctrl.radius;
                    }
                    ctrl.svgHeight += 2; // To make room for the center value

                    const numberOfSections = ctrl.values.reduce((a, b) => a + b);
                    if (numberOfSections > ctrl.maximum) {
                        // If maximum is smaller than sum of values, use sum of values as maximum
                        ctrl.maximum = numberOfSections;
                    }

                    // 0.5px spacing between "sections"
                    let spacingInDegrees = 0.5 * ctrl.maxAngle / ctrl.radius;
                    // Single "section" size in degrees, taking into account the spacing between each one of them
                    let sectionInDegrees =
                        (ctrl.maxAngle - (ctrl.maximum - 1) * spacingInDegrees) / ctrl.maximum;
                    // If resulting section is smaller than 1px, reduce the spacing between them to half the section size
                    if (sectionInDegrees / 180 * ctrl.radius < 1) {
                        spacingInDegrees = 0.5 * sectionInDegrees;
                        sectionInDegrees = (ctrl.maxAngle - (ctrl.maximum - 1) * spacingInDegrees) / ctrl.maximum;
                    }

                    // Fill sections path data (describe and class)
                    ctrl.paths = [];
                    let sectionNumber = 0;
                    for (const [classValue, nSections] of ctrl.classes.map((c, i) => [c, ctrl.values[i]])) {
                        for (let i = 0; i < nSections; i++) {
                            ctrl.paths.push({
                                describe: describeArc(
                                    ctrl.radius,
                                    ctrl.radius,
                                    ctrl.radius - 2 * ctrl.strokeWidth,
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees),
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees) - sectionInDegrees
                                ),
                                class: classValue,
                            });
                            sectionNumber++;
                        }
                    }

                    // Fill not filled sections if there are some remaining
                    if (numberOfSections < ctrl.maximum) {
                        for (let i = 0; i < ctrl.maximum - numberOfSections; i++) {
                            ctrl.paths.push({
                                describe: describeArc(
                                    ctrl.radius,
                                    ctrl.radius,
                                    ctrl.radius - 2 * ctrl.strokeWidth,
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees),
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees) - sectionInDegrees
                                ),
                                class: ctrl.classNotFilled,
                            });
                            sectionNumber++;
                        }
                    }
                }

                function polarToCartesian(centerX, centerY, radius, angleInDegrees) {
                    const angleInRadians = (-angleInDegrees * Math.PI) / 180.0;

                    return {
                        x: centerX + radius * Math.cos(angleInRadians),
                        y: centerY + radius * Math.sin(angleInRadians),
                    };
                }

                function describeArc(x, y, radius, startAngle, endAngle) {
                    const start = polarToCartesian(x, y, radius, endAngle);
                    const end = polarToCartesian(x, y, radius, startAngle);

                    const largeArcFlag = Math.abs(endAngle - startAngle) <= 180 ? '0' : '1';

                    const d = ['M', start.x, start.y, 'A', radius, radius, 0, largeArcFlag, 0, end.x, end.y].join(' ');

                    return d;
                }
            },
        ],
    });

    /** Text that can be edited once clicked on
     * textValue        Current text value (two-way bound). Gets set to defaultValue on load if empty.
     * defaultValue     (optional) Default text shown when directive first loads (before editing) if textValue is empty
     *                  If defaultValue is passed in, title color will be grayed out if textValue === defaultValue
     * placeholder      (optional) Placeholder text shown in input field while editing
     * fontSize         (optional) Font size to be used in in both view and edit mode; default value matches style guide
     **/
    app.directive('editableTextField', function() {
        return {
            restrict: 'E',
            templateUrl: '/templates/widgets/editable-text-field.html',
            scope: {
                textValue: '=ngModel',
                defaultValue: '<?',
                placeholder: '<?',
                fontSize: '<?'
            },
            link: function(scope) {
                scope.item = {};
                scope.placeholder = scope.placeholder || 'New name';

                // Comparator item naming
                scope.startEdit = function(item) {
                    item.$editing = true;
                    item.$textValue = scope.textValue === scope.defaultValue ? '' : scope.textValue;
                };

                scope.cancelEdit = function(item) {
                    item.$editing = false;
                };

                scope.validateEdit = function(item) {
                    scope.textValue = item.$textValue || scope.textValue;
                    item.$editing = false;
                };

                scope.$watch('defaultValue', (nv, ov) => {
                    if (!scope.textValue || scope.textValue === ov) {
                        scope.textValue = nv;
                    }
                });
            }
        }
    });


    /** Text block that will be displayed on one line. If the text block does not fit on one line,
     * a show more/less button will be displayed.
     **/
    app.directive("truncateTextVertically", function () {
        const template = `
                <div ng-transclude ng-class="{ 'truncated': showMoreLess && !isOpen }"></div>
                <div ng-show="showMoreLess" class="mtop8">
                    <a ng-show="!isOpen" ng-click="isOpen = !isOpen">Show more&hellip;</a>
                    <a ng-show="isOpen" ng-click="isOpen = !isOpen">Show less&hellip;</a>
                </div>`;
        return {
            restrict: 'A',
            template: template,
            transclude: true,
            link: ($scope, element) => {
                const textElement = element.find('[ng-transclude]')[0];
                $scope.isOpen = false;

                const updateShouldShowMoreLess = function() {
                    const truncatedClass = 'truncated';

                    // ensure element has truncated class to calculate heights for showMoreLess correctly
                    const alreadyTruncated = textElement.classList.contains(truncatedClass);
                    if (!alreadyTruncated) {
                        textElement.classList.add(truncatedClass);
                    }
                    $scope.showMoreLess = textElement.offsetHeight < textElement.scrollHeight || textElement.offsetWidth < textElement.scrollWidth;
                    if (!alreadyTruncated) {
                        textElement.classList.remove(truncatedClass);
                    }
                };
                updateShouldShowMoreLess();

                const resizeObserver = new ResizeObserver(() => {
                    const oldValue = $scope.showMoreLess;
                    updateShouldShowMoreLess();
                    if (oldValue !== $scope.showMoreLess) {
                        // make sure Angular runs a digest to take into account the new value of showMoreLess
                        $scope.$applyAsync();
                    }
                });
                resizeObserver.observe(textElement);

                $scope.$on('$destroy', function() {
                    resizeObserver.disconnect();
                });
            }
        }
    });

    /** Basic chart (using only divs) to display percentages
     * data                 Format: [[categoryName, value (in decimal)], ...]
     * colors               Array of colors to use for each bar
     * showFilter           (optional) Add input for filtering if true
     * onCategoryClicked       (optional) Fires when category is clicked
     **/
    app.component('basicPercentChart', {
        templateUrl: '/templates/widgets/basic-percent-chart.html',
        bindings: {
            data: '<',
            colors: '<',
            showFilter: '<?',
            onCategoryClicked: '&?'
        },
        controller: function(Debounce) {
            const $ctrl = this;
            const DEFAULT_COLOR = '#999';
            const MAX_CHART_HEIGHT = 350;

            $ctrl.query = '';
            $ctrl.rowHeight = 20;

            $ctrl.$onChanges = () => {
                const maxValue = Math.max.apply(null, $ctrl.data.map(row => row[1]));
                $ctrl.chartData = $ctrl.data.map((row, index) => ({
                    category: row[0],
                    value: Math.round(row[1] * 100) + '%',
                    width: Math.round(row[1] / maxValue * 100) + '%',
                    color: $ctrl.colors[index] || DEFAULT_COLOR,
                }));
                filter();

                $ctrl.chartHeight = Math.min($ctrl.data.length * $ctrl.rowHeight + 1, MAX_CHART_HEIGHT);
            };

            const filter = () => {
                $ctrl.filteredChartData = $ctrl.chartData.filter(row => row.category.toLowerCase().indexOf($ctrl.query
                .toLowerCase()) >= 0);
            };
            $ctrl.filterDebounced = Debounce().withDelay(150, 150).wrap(filter);

            $ctrl.onCategoryClick = (category) => {
                if ($ctrl.onCategoryClicked) {
                    $ctrl.onCategoryClicked({ category });
                }
            };
        }
    });

    /**
     * A component to help displaying items that need to be filtered and paginated, but can also be reordered, added or removed.
     * Because of the limitations of angular transclusion, the contained items are not directly repeated, but you have to use a ng-repeat on the paginatedItems 'output' parameter
     * This component can be used with filteredPaginatedListItem that contains the buttons to manipulate items. Alternatively, you could design your own variant if the UI needs to be different.
     *
     * Example:
     * <filtered-paginated-list items="myArrayOfItems" page-size="20" new-item-template="defaultItemValue" paginated-items="tempvalues.paginatedItems">
            <li ng-repeat="item in tempvalues.paginatedItems"
                ng-class="{'blokc--highlighted': rule.highlighted">
                <filtered-paginated-list-controls item="item" />
                <my-custom-element item="item.model" />
            </li>
        </filtered-paginated-list>
     *
     * @param items = an array conaining the items to display
     * @param pageSize the number of items to display per page (default 50)
     * @param newItemTemplate the default value of a new item (will be cloned on item creation)
     * @param paginatedItems "Output" parameter, the list of items to display for the current page. Should be used to display content
     *      paginatedItems is an array of 'enriched' input items:
     *      - model is the original item (and can be modified in the transcluded elements)
     *      - rank is the position in the full list (starts at 1)
     *      - highlighted is an indicator that and the element has recently been moved or added, and should be highlighted in order for the user to keep track of it.
     */
    app.component('filteredPaginatedList', { // filtered-paginated-list
        templateUrl: '/templates/widgets/filtered-paginated-list.html',
        transclude: true,
        bindings: {
            configItems: "=items",
            paginatedItems: '=',
            pageSize: "<?",
            newItemTemplate: "<?",
        },
        controller: function($scope, ListFilter, CollectionFiltering, $timeout, Dialogs) {
            this.$onInit = function () {
                let filteredItems = [];
                let items = this.configItems.map((model, i) => ({
                    rank: i + 1,
                    model
                }));
                this.scrollToRank = -1;
                this.pageSize = this.pageSize || 50;

                this.pagination = new ListFilter.Pagination(items, this.pageSize);
                this.search = '';
                // this function update the ranks after an add, delete, move...
                // ranks start at 1 because we want the seach to find the right item.
                const updateRanks = (start, end = items.length + 1) => {
                    for(let i = start-1 ; i < end-1 ; i ++) {
                        items[i].rank = i + 1;
                    }
                };

                // refresh filter & pagination after any query change or list change (add, move, remove...)
                // not triggered for item content change (an item is not suddenly hidden because it's modified and doesn't match the filter anymore)
                const updateView = () => {
                    filteredItems = this.search
                        ? CollectionFiltering.filter(items, {
                             userQuery: this.search,
                        }, {
                            exactMatch: ['rank'],
                        })
                        : items;

                    this.pagination.updateAndGetSlice(filteredItems);
                    this.paginatedItems = this.pagination.slice;
                };


                const scrollTo = (item) => {
                    const filteredIndex = filteredItems.indexOf(item);
                    if(filteredIndex === -1) return; // when the item we move is filtered out, cancel the scroll out instead of going to page 0

                    const targetPage = Math.floor(filteredIndex / this.pageSize) + 1;
                    this.pagination.goToPage(targetPage);
                    this.scrollToRank = item.rank;

                    // the counter instead of simple boolean handles the fact that a same item could be clicked twice in less the 800ms
                    item.highlighted = (item.highlighted || 0) + 1;
                    $timeout(() => item.highlighted -= 1, 800);

                    // we need to reset scrollToRank so that the same element can trigger a scroll again
                    $timeout(() => this.scrollToRank = -1);
                };


                $scope.$watch(() => this.search, updateView);

                $scope.$watch(() => this.pagination.page, () => {
                    this.pagination.update();
                    this.paginatedItems = this.pagination.slice;
                });
                const moveToRank = (item, destRank) => {
                    const rank = item.rank;

                    items.splice(rank - 1, 1);
                    items.splice(destRank - 1, 0, item);

                    this.configItems.splice(rank - 1, 1);
                    this.configItems.splice(destRank - 1, 0, item.model);

                    updateRanks(
                        Math.min(rank, destRank),
                        Math.max(rank, destRank, items.length - 2) + 1
                    );
                    updateView();
                    scrollTo(item);
                };

                const moveTo = (item) => {
                    return Dialogs.prompt($scope, "Move to", `Move before the rule having rank (max ${items.length+1}):`, item.rank, {
                        minValue: 1,
                        maxValue: items.length+1,
                        type: 'number'
                    })
                    .then((newRank) => {
                        // if item.rank < newRank, we have to take into account the fact that removing item from the list will change target rule rank
                        if (newRank > item.rank) {
                            moveToRank(item, Math.max(0, Math.min(items.length, newRank - 1)));
                        } else {
                            moveToRank(item, Math.max(0, Math.min(items.length, newRank)));
                        }
                    })
                    .catch(() => {}) // reject = move canceled, nothing to do
                };
                const canMoveTo = () => items.length > 1;

                const moveUp = (item) => {
                    const filteredIndex = filteredItems.indexOf(item);
                    const previousVisibleRule = filteredItems[filteredIndex - 1];
                    moveToRank(item, previousVisibleRule.rank);
                };
                const canUp = (item) => filteredItems[0] && item.rank > filteredItems[0].rank;

                const moveDown = (item) => {
                    const filteredIndex = filteredItems.indexOf(item);
                    const nextVisibleRule = filteredItems[filteredIndex + 1];
                    moveToRank(item, nextVisibleRule.rank);
                };
                const canDown = (item) => filteredItems[filteredItems.length - 1] && item.rank < filteredItems[filteredItems.length - 1].rank;

                const moveTop = (item) => moveToRank(item, 1);
                const canTop = (item) => item.rank > 1;

                const moveBottom = (item) => moveToRank(item, items.length);
                const canBottom = (item) => item.rank < items.length;


                const insertNewItemAtRank = (rank) => {
                    const newValue = angular.copy(this.newItemTemplate || {});
                    const newItem = { model: newValue };
                    items.splice(rank - 1, 0, newItem);
                    this.configItems.splice(rank - 1, 0, newValue);

                    updateRanks(rank);
                    updateView();
                    scrollTo(newItem);
                };

                const insertNewItemBefore = (item) => insertNewItemAtRank(item.rank);
                const insertNewItemAfter = (item) => insertNewItemAtRank(item.rank + 1);
                const insertNewItemFirst = () => insertNewItemAtRank(1);
                const insertNewItemLast = () => insertNewItemAtRank(items.length + 1);

                const remove = (item) => {
                    items.splice(item.rank - 1, 1);
                    this.configItems.splice(item.rank - 1, 1);
                    updateRanks(item.rank);
                    updateView();
                };

                this.api = {
                    moveTop, canTop,
                    moveUp, canUp,
                    moveDown, canDown,
                    moveBottom, canBottom,
                    moveTo, canMoveTo,

                    insertNewItemAfter, insertNewItemBefore,
                    insertNewItemFirst, insertNewItemLast,
                    remove,
                };
            }
        }
    });

    /**
     * Optional child component for filteredPaginatedList
     * Displays buttons to move / add / delete items and the item rank
     * @param item current item (used for the actions)
     * @param rankLabel text used to label the rank display (default = 'Item rank:')
     */
    app.component('filteredPaginatedListControls', {
        transclude: true,
        templateUrl: '/templates/widgets/filtered-paginated-list-item.html',
        require: {
            list: '^^filteredPaginatedList',
        },
        bindings: {
            item: "=",
            rankLabel: "@?",
        },
    });

    app.component('messageAlert', {
         templateUrl : '/templates/widgets/message-alert.html',
         bindings: {
             text: '@',
             severity: '@',
         },
    });

    app.component('limitedVisibilityLock', { // limited-visibility-lock
        template: `
        <span class="limited-visibility-lock"
            toggle="tooltip" container="{{$ctrl.tooltipContainer || 'body'}}" title="You don't have full access to view this {{$ctrl.objectNiceName}}"
        >
            <i class="icon-lock"></i>
        </span>
        `,
        bindings: {
            objectType: '<',
            tooltipContainer: '<?'
        },
        controller: function($filter) {
            this.$onChanges = (changes) => {
                if(changes.objectType) {
                    this.objectNiceName = this.objectType === 'APP' ? 'application' : $filter('niceTaggableType')(this.objectType)
                }
            }
        }
    });

    /** Simple list for numerical values.
     * values: Array of numerical values
     **/
    app.component('numericalList', {
        template: `
        <div class="numerical-list__container">
            <div class="numerical-list__item" ng-repeat="value in $ctrl.values">{{value | smartNumber}}</div>
        </div>
        `,
        bindings: {
            values: '<',
        },
    });

    /**
     * Simple warning block when the currently selected connection for a managed folder doesn't allow managed folders to be defined on itself. Defined as a component simply to avoid code duplication.
     */
    app.component('fsProviderSettingsNoManagedFolderWarning', { // fs-provider-settings-no-managed-folder-warning
        template: `
        <div class="alert alert-warning" ng-if="$ctrl.selectedConnection && $ctrl.connections && !$ctrl.connections.includes($ctrl.selectedConnection)">
            <i class="icon-warning-sign"></i>
            The settings associated with {{$ctrl.selectedConnection}} prevent you to use this connection for managed folders.
        </div>`,
        bindings: {
            connections: '<', // the list of available connections, may be undef while loading from backend
            selectedConnection: '<' // the currently selected connection
        }
    });

    app.component("recipeAllAvailablePartitioningWarning", {
        templateUrl: '/templates/widgets/recipe-all-available-partitioning-warning.html',
    });

    app.component("variablesExpansionLoopConfig", {
        bindings: {
            config: '<',
            actionThatIsRepeated: '@',
            variablesUsageContext: '@',
            excludeDatasetFullName: "<?",
        },
        templateUrl: "/templates/widgets/variables-expansion-loop-config.html",
        controller: function($scope, $stateParams, DatasetsService, SmartId) {
            const ctrl = this;

            ctrl.formId = generateUniqueId();

            DatasetsService.listWithAccessible($stateParams.projectKey).success(function(data) {
                let datasets = data;
                if (ctrl.excludeDatasetFullName) {
                    datasets = datasets.filter(function(dataset) {
                        return dataset.projectKey + "." + dataset.name !== ctrl.excludeDatasetFullName;
                    });
                }
                datasets.forEach(function(dataset) {
                    dataset.smartName = SmartId.create(dataset.name, dataset.projectKey);
                    dataset.localProject = dataset.projectKey === $stateParams.projectKey;
                    dataset.usable = true;
                });
                ctrl.datasets = datasets;
            }).error(setErrorInScope.bind($scope));
        },
    });

    app.component('basicSearchBox', { // basic-search-box
        template: `
        <span class="basic-search-box__addon-icon">
            <i class="dku-icon-search-16"></i>
        </span>
        <input type="search"
            id="{{$ctrl.inputId || $ctrl.fallbackId}}"
            class="basic-search-box__input"
            ng-model="$ctrl.ngModel"
            ng-change="$ctrl.ngModelChange()"
            ng-disabled="$ctrl.disabled"
            placeholder="{{$ctrl.placeholder}}"
        />
        <i ng-if="$ctrl.ngModel" class="dku-icon-dismiss-16" ng-click="$ctrl.clear()"></i>
        `,
        require: { ngModelCtrl: 'ngModel' },
        bindings: {
            placeholder: '@',
            ngModel: '<',
            disabled: '<?',
            autofocus: '<?',
            inputId: '@',
            clickClear: '&?'
        },
        controller: function($element, $timeout) {
            const ctrl = this;

            ctrl.$onInit = function () {
                ctrl.fallbackId = 'basic-search-box-'+Math.round(Math.random()*100000);
                ctrl.placeholder = ctrl.placeholder === undefined ? "Search\u2026" : ctrl.placeholder;
            }

            ctrl.$onChanges = (changes) => {
                // forwarding autofocus directly as an html prop on the input doesn't work well with angularjs compilation
                if(changes.autofocus && ctrl.autofocus) {
                    $timeout(() => $element.find('input').focus());
                }
            }

            ctrl.clear = () => {
                ctrl.ngModelCtrl.$setViewValue('');
                ctrl.clickClear && ctrl.clickClear();
            }

            ctrl.ngModelChange = () => ctrl.ngModelCtrl.$setViewValue(ctrl.ngModel);
        }
    });

    app.component('externalLink', {
        bindings: {
            href: '@'
        },
        transclude: true,
        template: `<a target="_blank" ng-href="{{$ctrl.href}}" ng-click="$ctrl.removeFocus()">
                        <ng-transclude></ng-transclude>
                        <i class="dku-icon-arrow-external-link-12"></i>
                   </a>`,
        controller: function($element) {
            const ctrl = this;
            ctrl.removeFocus = function() {
                document.activeElement.blur();
            };
        }
    });

})();
