function gentleTruncate(str, len) {
    /**
     * Truncate a string to make sure it takes at most
     * n characters.
     * Whenever possible truncates on special chars.
     *
     * If str is not a string, returns str unchanged.
     */
    if ((typeof str != "string") || (str.length <= len)) {
        return str;
    }

    // eslint-disable-next-line no-useless-escape
    var cutOn = /[ ,\.,;\-\\\"\n\?\!\|]/g
    var truncated = str.substring(0, len-1);
    var lastSeparatorIdx = regexLastIndexOf(cutOn, truncated);

    // we don't want to cut more too much.
    if (lastSeparatorIdx > len / 2) {
        truncated = str.substring(0, lastSeparatorIdx);
    }
    return truncated + '…';
}

var now = new Date().getTime();
var NOW_REFRESH_INTERVAL = 1000 * 5; // 5 Seconds

setInterval(function(){
    now = new Date().getTime();
}, NOW_REFRESH_INTERVAL);

function friendlyDuration(diffInSeconds, translate) {
    const sec = Math.floor((diffInSeconds >= 60 ? diffInSeconds % 60 : diffInSeconds));
    const min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    const hrs = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 24 ? diffInSeconds % 24 : diffInSeconds);
    const days = Math.floor((diffInSeconds = (diffInSeconds / 24)) >= 30 ? diffInSeconds % 30 : diffInSeconds);
    const months = Math.floor((diffInSeconds = (diffInSeconds / 30)) >= 12 ? diffInSeconds % 12 : diffInSeconds);
    const years = Math.floor(diffInSeconds / 12);
    const values = { years: years, months: months, days: days, hours: hrs, minutes: min, seconds: sec };
    if (years > 0) {
        if (years <= 6 && months > 0) {
            return translate("FILTER.FRIENDLY_DURATION.YEARS_MONTHS", "{{years}} {{years === 1 ? 'year': 'years'}} and {{months}} {{months === 1 ? 'one month': (months + ' months')}}", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.YEARS", "{{years}} {{years === 1 ? 'year': 'years'}}", values);
        }
    } else if (months > 0) {
        if (months <= 6 && days > 0) {
            return translate("FILTER.FRIENDLY_DURATION.MONTHS_DAYS", "{{months}} {{months === 1 ? 'month': 'months'}} and {{days === 1 ? 'one day' : (days + ' days')}}", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.MONTHS", "{{months}} {{months === 1 ? 'month': 'months'}}", values);
        }
    } else if (days > 0) {
        if (days <= 3 && hrs > 0) {
            return translate("FILTER.FRIENDLY_DURATION.DAYS_HOURS", "{{days}} {{days === 1 ? 'day': 'days'}} and {{hours === 1 ? 'one hour' : (hours + ' hours')}}", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.DAYS", "{{days}} {{days === 1 ? 'day': 'days'}}", values);
        }
    } else if (hrs > 0) {
        if (min > 1) {
            return translate("FILTER.FRIENDLY_DURATION.HOURS_MINUTES", "{{hours}} {{hours === 1 ? 'hour': 'hours'}} and {{minutes}} minutes", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.HOURS", "{{hours}} {{hours === 1 ? 'hour': 'hours'}}", values);
        }
    } else if (min > 0) {
        if (sec > 1) {
            return translate("FILTER.FRIENDLY_DURATION.MINUTES_SECONDS", "{{minutes}} {{minutes === 1 ? 'minute': 'minutes'}} and {{seconds}} seconds", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.MINUTES", "{{minutes}} {{minutes === 1 ? 'minute': 'minutes'}}", values);
        }
    } else {
        return translate("FILTER.FRIENDLY_DURATION.ABOUT_SECONDS", "about {{seconds <= 1 ? 'a second' : (seconds + ' seconds')}}", values);
    }
}

/* eslint-disable-next-line no-redeclare */
function durationHHMMSS(diffInSeconds) {
    var sec = Math.floor((diffInSeconds >= 60 ? diffInSeconds % 60 : diffInSeconds));
    var min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    var hours = Math.floor( diffInSeconds / 60);
    var sb = "";
    if (hours > 0) {
        sb += (hours + "h ");
    }
    if (min > 0) {
        sb += (min + "m ");
    }
    sb += (sec + "s");
    return sb;
}

/* eslint-disable-next-line no-redeclare */
function durationHHMM(diffInSeconds) {
    var min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    var hours = Math.floor( diffInSeconds / 60);
    var sb = "";
    if (hours > 0) {
        sb += (hours + "h ");
    }
    if (min > 0) {
        sb += (min + "m ");
    }
    return sb;
}

function durationHHMMSSPadded(diffInSeconds) {
    if (diffInSeconds == 0) diffInSeconds = 1;
    var sec = Math.floor((diffInSeconds >= 60 ? diffInSeconds % 60 : diffInSeconds));
    var min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    var hours = Math.floor( diffInSeconds / 60);

    function pad(number) {
        if (number < 10) return "0" + number;
        else return number;
    }
    return pad(hours) + "h" + pad(min) + "m" + pad(sec) + "s";
}


function friendlyDurationShort(seconds, ref, noSeconds, translate) {
    const sec = Math.floor(seconds >= 60 ? seconds % 60 : seconds);
    const min = Math.floor((seconds = (seconds / 60)) >= 60 ? seconds % 60 : seconds);
    const hours = Math.floor((seconds = (seconds / 60)) >= 24 ? seconds % 24 : seconds);
    const days = Math.floor((seconds = (seconds / 24)) >= 30 ? seconds % 30 : seconds);
    const months = Math.floor((seconds = (seconds / 30)) >= 12 ? seconds % 12 : seconds);
    const years = Math.floor((seconds = (seconds / 12)));
    const values = { years: years, months: months, days: days, hours: hours, minutes: min, seconds: sec };
    let sb = "";

    if (years > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.YEARS", "{{years}} {{years === 1 ? 'year': 'years'}}", values);
    } else if (months > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.MONTHS", "{{months}} {{months === 1 ? 'month': 'months'}}", values);
    } else if (days > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.DAYS", "{{days}} {{days === 1 ? 'day': 'days'}}", values);
    } else if (hours > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.HOURS", "{{hours}} {{hours === 1 ? 'hour': 'hours'}}", values);
    } else if (min > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.MINUTES", "{{minutes}} {{minutes === 1 ? 'minute': 'minutes'}}", values);
    } else if(!noSeconds && sec > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.SECONDS", "{{seconds}} {{seconds === 1 ? 'second' : 'seconds'}}", values);
    }

    switch (ref) {
        case 'ago':
            return sb
                ? translate("FILTER.FRIENDLY_DURATION.DURATION_AGO", "{{duration}} ago", { duration: sb })
                : translate("FILTER.FRIENDLY_DURATION.JUST_NOW", "just now");
        case 'in':
            return sb
                ? translate("FILTER.FRIENDLY_DURATION.IN_DURATION", "in {{duration}}", { duration: sb })
                : translate("FILTER.FRIENDLY_DURATION.IMMEDIATELY", "immediately");
        default:
            return sb ? sb : (noSeconds
                ? translate("FILTER.FRIENDLY_DURATION.LESS_THAN_ONE_MIN", "< 1 minute")
                : translate("FILTER.FRIENDLY_DURATION.LESS_THAN_ONE_SEC", "< 1 second"));
    }
}

function dateDayDiff(date1, date2) {
    var d1 = new Date(date1);
    var d2 = new Date(date2);
    d1.setHours(0);
    d1.setMinutes(0);
    d1.setSeconds(0);
    d1.setMilliseconds(0);
    d2.setHours(0);
    d2.setMinutes(0);
    d2.setSeconds(0);
    d2.setMilliseconds(0);
    var dayLength = 24*60*60*1000;
    return Math.floor(d1.getTime()/dayLength) - Math.floor(d2.getTime()/dayLength);
}
function dateMinuteDiff(date1, date2) {
    var d1 = new Date(date1);
    var d2 = new Date(date2);
    d1.setSeconds(0);
    d1.setMilliseconds(0);
    d2.setSeconds(0);
    d2.setMilliseconds(0);
    var msToMin = 60*1000;
    return Math.floor(d1.getTime()/msToMin) - Math.floor(d2.getTime()/msToMin);
}


(function() {

'use strict';

var app = angular.module('dataiku.filters', [ 'dataiku.constants' ]);

app.filter('slugify', function(SlugifyService){
    return SlugifyService.transform;
});


app.filter('join', function(){
    return function(input){
        return input.join(",");
    };
});


app.filter('capitalize', function(){
    return function(input){
        if(input && input.length>0) {
            return input.charAt(0).toUpperCase() + input.slice(1);
        } else {
            return '';
        }
    };
});

app.filter('capitalizeWord', function() {
    return function(input) {
        input = input.toLowerCase();
        return (input || '').split(' ').reduce((result, word) => {
            return result + ' ' + word[0].toUpperCase() + word.slice(1);
        }, '');
    };
});


app.filter('pluralize', function() {
    /**
     * Pluralize an item name.
     * @param {number}            num         - The quantity of the item
     * @param {string}            singular    - The singular form of the item name (only used when num is worth 1)
     * @param {string}            plural      - The plural form of the item name
     * @param {d3 formatter}      [format]    - Optional d3.js formatter for num
     * @param {boolean | string}  [no]        - Optional indicator of the filter behavior when num is worth 0:
     *                                        If false (default), use '0 ' + plural
     *                                        If true, use 'no ' + plural
     *                                        If a string, use it
     */
    return function(num, singular, plural, format, no) {
        if (no && num == 0) return no === true ? 'no ' + plural : no;
        return (format ? d3.format(format)(num) : num) + " " + (num === 1 ? singular : plural);
    }
});


app.filter('plurify', function($translate) {
    return function(singular, num, plural) {
        const usePlural = num > 1 || (num === 0 && ($translate && $translate.proposedLanguage() || $translate.use() || "en") === 'en'); // in english zero uses plural mode (crazy guys)
        return usePlural ? (typeof (plural) !== "undefined" ? plural : singular + 's') : singular;
    };
});

app.filter('breakify', function (LoggerProvider) {
    return function (text, breakOnRegex) {
        try {
            const re = new RegExp(breakOnRegex, 'g');
            const indices = [];
            let m;
            do {
                m = re.exec(text);
                if (m) {
                    indices.push(m.index + m[0].length);
                }
            } while (m);
            indices.reverse().forEach(pos => {
                if (text) {
                    text = [text.slice(0, pos), '<wbr>', text.slice(pos)].join('');
                }
            });
            return text;
        } catch(err) {
            LoggerProvider.getLogger('DisplayFilters').error("Error ", err);
            return text;
        }
    };
});

app.filter('uncamel', function (LoggerProvider) {
    return function (text) {
        try {
            return text.replace(/[A-Z]/g, function(l) { return ' ' + l.toLowerCase(); })
        } catch(err) {
            LoggerProvider.getLogger('DisplayFilters').error("Error ", err);
            return text;
        }
    };
});

app.filter('objectSize', function() {
    return function(object) {
        if (!object) return 0;
        return Object.keys(object).length;
    };
});


app.filter("objectKeys", function() {
    return function(obj) {
        if (!obj) {
            return [];
        }
        return Object.keys(obj);
    }
});


app.filter('listSlice', function(){
    return function(input, from, to) {
        if (!input || input.length <= from) {
            return [];
        }
        return input.slice(from, to);
    }
});


// `[n] | range`        => 0 .. (n-1)
// `[from, to] | range` => from .. to (inclusive)
app.filter('range', function() {
    return function(input) {
        switch (input.length) {
            case  1:
                if (input[0] <= 0) return [];
                input = [0, input[0] - 1]; break;
            case  2:
                if (input[1] < input[0]) return [];
                break;
            default: return input;
        }
        var len = input[1] - input[0] + 1,
            result = new Array(len);
        for (var i = 0; i < len; i++) {
            result[i] = i + input[0];
        }
        return result;
    };
});


app.filter('gentleTruncate', function () {
    return function (text, length, end) {

        if (isNaN(length))
            length = 10;

        return gentleTruncate(text, length);

    };
});


app.filter('onlyNumbers', function(){
    return function(text){
        if (!text) { return ''; }
        return text.replace(/[^0-9]/g, '');
    };
});


app.filter('ordinal', function(){
    return function(number){
        return number < 14 && number > 10 ? 'th' : ['th','st','nd','rd','th','th','th','th','th','th'][number % 10];
    };
});


app.filter('ordinalWithNumber', function(){
    return function(number){
        return number + (number < 14 && number > 10 ? 'th' : ['th','st','nd','rd','th','th','th','th','th','th'][number % 10]);
    };
});

// ratio to % w/ fixed precision & optional non-breaking space
// special cases: '<1%' and '>99%' when <.01 but >0 (for precision == 2, '<0.01%'...)
app.filter('smartPercentage', function() {
    return function(ratio, precision, spaces) {
        precision = Math.max(+(precision || 0), 0);
        var tens = Math.pow(10, precision),
            min = 1 / tens / 100, // e.g. precision = 2 =>  0.01
            max = 1 - min,        //                    => 99.99
            out = [];
        if (ratio < 1 && ratio > max) {
            ratio = max;
            out.push('>');
        } else if (ratio > 0 && ratio < min) {
            ratio = min;
            out.push('<');
        }
        out.push((Math.round(ratio * 100 * tens) / tens).toFixed(precision), '%');
        return out.join(spaces ? '\u00A0' : '');
    }
});


app.filter('processorByType', function(){
    return function(processors, type){
        if (processors) {
            for (var i = 0; i < processors.processors.length; i++) {
                if (processors.processors[i].type == type) {
                    return processors.processors[i];
                }
            }
        }
        throw Error("Unknown processor " +type);
    }
});


app.filter('cleanFacetValue', function(DKU_NO_VALUE, DKU_OTHERS_VALUE) {
    return function(input) {
        if (input === DKU_NO_VALUE) {
            return "<em>No value</em>";
        } else if (input === DKU_OTHERS_VALUE) {
            return "<em>Others</em>";
        } else {
            return input;
        }
    };
});

app.filter('chartLabelValue', function(ChartLabels) {
    /**
     * @param {AxisElt}                             input the label object from the backend.
     * @param {NumberFormattingOptions | undefined} formattingOptions the number formatting options.
     * @param {number | undefined}                  minValue the minimum value.
     * @param {number | undefined}                  maxValue the maximum value.
     * @param {number | undefined}                  numValues the number of values.
     */
    return function(input, formattingOptions, minValue, maxValue, numValues) {
        return ChartLabels.getFormattedLabel(input, formattingOptions, minValue, maxValue, numValues);
    };
});


app.filter("prettyjson", function(){
    return function(input) {
         return JSON.stringify(input,undefined,3);
    }
});


app.filter("jsonOrString", function(){
    return function(input) {
        if (typeof input == "string") {
            return input;
        }
        else {
            return JSON.stringify(input);
        }
    }
});


app.filter('friendlyTime', function(translate) {
    return function(input) {
        var diffInSeconds = parseInt(input, 10) / 1000;
        return friendlyDuration(diffInSeconds, translate);
    };
});


app.filter('friendlyTimeDeltaForward', function(translate) {
    const filter =  function(input) {
        var diffInSeconds = (parseInt(input, 10) - now) / 1000;
        return translate("FILTER.FRIENDLY_TIME_DELTA_FORWARD", "in {{period}}", {period: friendlyDuration(diffInSeconds, translate)});
    };
    filter.$stateful = true;
    return filter;
});


app.filter('friendlyTimeDelta', function(translate) {
    const filter = input => {
        var diffInSeconds = (now - parseInt(input, 10)) / 1000;
        return translate("FILTER.FRIENDLY_TIME_DELTA", "{{period}} ago", {period: friendlyDuration(diffInSeconds, translate)});
    };
    filter.$stateful = true;
    return filter;
});


app.filter('friendlyTimeDeltaShort', function(translate) {
    const filter = function(input) {
        var diffInSeconds = (now - parseInt(input, 10)) / 1000;
        return friendlyDurationShort(diffInSeconds, 'ago', true, translate);
    };
    filter.$stateful = true;
    return filter;
});

app.filter('friendlyTimeDeltaHHMMSS', function() {
    const filter = function(input) {
        var diffInSeconds = Math.max(0, (now - parseInt(input, 10)) / 1000);
        return durationHHMMSS(diffInSeconds);
    };
    filter.$stateful= true;
    return filter;
});

app.filter('friendlyDuration', function(translate){
    return function (input) { return friendlyDuration(parseInt(input, 10) / 1000, translate); };
});


app.filter('friendlyDurationShort', function(translate) {
    return function(input, ref, noSeconds) {
        return friendlyDurationShort(parseInt(input, 10) / 1000, ref, noSeconds, translate);
    };
});


app.filter('friendlyDurationSec', function(translate) {
    return function(input) {
        return friendlyDuration(parseInt(input, 10), translate);
    };
});


app.filter('durationHHMMSS', function() {
    return function(input) {
        return durationHHMMSS(parseInt(input, 10)/1000);
    };
});


app.filter('durationHHMMSSPadded', function() {
    return function(input) {
        return durationHHMMSSPadded(parseInt(input, 10)/1000);
    };
});


app.filter('yesNo', function(){
    return function(input) {
        if (input) return "Yes";
        else return "No";
    }
});

app.filter('friendlyDate', function($filter, translate) {
    return function(time, format) {
        var today = new Date();
        var date = new Date(time);
        const defaultFormat = date.getFullYear() === today.getFullYear()
            ? 'EEEE, d MMMM'
            : 'EEEE, d MMMM y';
        format = format || defaultFormat;
        if(dateDayDiff(date, today)===0){
            return translate('FILTER.FRIENDLY_DATE.TODAY', 'Today')
        } else if(dateDayDiff(date, today)===-1){
            return translate('FILTER.FRIENDLY_DATE.YESTERDAY', 'Yesterday')
        } else if(dateDayDiff(date, today)===1){
            return translate('FILTER.FRIENDLY_DATE.TOMORROW', 'Tomorrow')
        } else {
            return $filter('date')(date, format);
        }
    };
});

app.filter('dateDayDiff', function() {
    return function(time) {
        var today = new Date();
        var date = new Date(time);
        return dateDayDiff(date, today);
    };
});


app.filter('friendlyDateRange', function($filter) {
    const day = $filter('friendlyDate');
    const time = date => $filter('date')(date, 'HH:mm');
    function duration(date1, date2, options) {
        if (options && options.noDuration || !date2){
            return '';
        }
        let delta = (new Date(date2) - new Date(date1));
        return ' (duration: ' + $filter('durationHHMMSS')(delta) + ')';
    }

    function unavailableEndDate(date) {
        return day(date) + ', started at ' +  time(date);
    }
    function sameMinute(date1, date2, options) {
        return day(date1) + ' ' +  time(date1) + duration(date1, date2, options);
    }
    function sameDay(date1, date2, options) {
        return day(date1) + ', ' +  time(date1) + ' to ' + time(date2) + duration(date1, date2, options);
    }
    function differentDays(date1, date2, options) {
        return 'From ' + day(date1) + ' ' +  time(date1)+ ' to ' + day(date2) + ' ' +  time(date2) + duration(date1, date2, options);
    }

    return function(date1, date2, options) {
        options = options || {};
        if (!date2) {
            return unavailableEndDate(date1);
        } else if (dateMinuteDiff(date1, date2) == 0) {
            return sameMinute(date1, date2, options);
        } else if (dateDayDiff(date1, date2) == 0) {
            return sameDay(date1, date2, options);
        } else {
            return differentDays(date1, date2, options);
        }
    };
});


app.filter('friendlyDateTime', function($filter, $translate, translate) {
    var sameDay = function (date1, date2) {
        return date1.getFullYear() === date2.getFullYear() &&
               date1.getMonth() === date2.getMonth() &&
               date1.getDate() === date2.getDate();
    }
    return function(time, format, timeFormat) {
        var today = new Date(),
            yesterday = new Date(),
            tomorrow = new Date();
        yesterday.setDate(yesterday.getDate() - 1);
        tomorrow.setDate(tomorrow.getDate() + 1);

        const date = new Date(time);
        let datePart;
        if (sameDay(date, today)) {
            datePart = translate('FILTER.FRIENDLY_DATE.TODAY', 'Today')
        } else if (sameDay(date, yesterday)) {
            datePart = translate('FILTER.FRIENDLY_DATE.YESTERDAY', 'Yesterday')
        } else if (sameDay(date, tomorrow)) {
            datePart = translate('FILTER.FRIENDLY_DATE.TOMORROW', 'Tomorrow')
        } else {
            if (!format) {
                const lang = $translate && $translate.proposedLanguage() || $translate.use() || "en-US";
                datePart = date.toLocaleString(lang, { weekday: "long", year: "numeric", month: "long", day: "numeric", });
            } else {
                datePart = $filter('date')(date, format);
            }
        }

        const timePart = $filter('date')(date, timeFormat || 'HH:mm');
        return translate("FILTER.FRIENDLY_DATE.DATE_TIME_JOINER", "{{date}} at {{time}}", {date:datePart, time:timePart});
    };
});

app.filter('dateUTC', function($filter) {
    return function(time, format) {
        return $filter('date')(time, format, 'UTC');
    }
});

app.filter('utcDate', function() {
    return function(time, format) {
        format = format || 'EEEE, d MMMM';
        return moment.utc(time).format(format);
    };
});

app.filter('typeToName', function($filter) {
    return function(input) {
        return $filter('capitalizeWord')($filter('niceConst')(input));
    }
});

app.filter('groupToName', function() {
    return function(input) {
        return input === "$$ALL_USERS$$" ? "All Users" : input;
    }
});


app.filter('recipeTypeToName', function(RecipeDescService) {
    return RecipeDescService.getRecipeTypeName;
});


app.filter('datasetTypeToName', function(TypeMappingService) {
    return function(type) {
        return TypeMappingService.mapDatasetTypeToName(type)
    };
});

app.filter('columnTypeToName', function(TypeMappingService) {
    return function(type) {
        if (type === 'date') {
            return 'datetime\u00A0with\u00A0tz'; // yes, it's a nbsp. Leave it there, otherwise these types get line-wrapped in small spaces
        } else if (type === 'dateonly') {
            return 'date\u00A0only'; // yes, it's a nbsp. Leave it there, otherwise these types get line-wrapped in small spaces
        } else if (type === 'datetimenotz') {
            return 'datetime\u00A0no\u00A0tz'; // yes, it's a nbsp. Leave it there, otherwise these types get line-wrapped in small spaces
        } else {
            return type;
        }
    };
});


app.filter('pluginIdToName', function(LoggerProvider, PluginsService) {
    return function(pluginId) {
        if(!pluginId) {
            return '';
        }
        let desc = PluginsService.getPluginDesc(pluginId);
        if (desc) {
            return desc.label;
        }
        return pluginId;
    };
});


app.filter("niceProfileName", function(){
    var dict = {
        /* Generic */
        "READER" : "Reader",
        "AI_CONSUMER" : "AI Consumer",
        "AI_ACCESS_USER" : "AI Access User",
        "NONE": "None",

        /* Legacy (2019 and before) profiles */
        "DATA_SCIENTIST": "Data scientist",
        "DATA_ANALYST": "Data analyst",

        /* 2020-FY2024 */
        "DESIGNER" : "Designer",
        "VISUAL_DESIGNER": "Visual Designer",
        "PLATFORM_ADMIN": "Platform admin",
        "EXPLORER": "Explorer",

        /* FY2025 */
        "DATA_DESIGNER" : "Data Designer",
        "ADVANCED_ANALYTICS_DESIGNER": "Advanced Analytics Designer",
        "FULL_DESIGNER": "Full Designer",
        "GOVERNANCE_MANAGER": "Governance Manager",
        "TECHNICAL_ACCOUNT": "Technical Account"
    };
    return function(input) {
        return dict[input] || input;
    };
});

app.filter('itemToColor', function(typeToColorFilter) {
    return function(item) {
        return typeToColorFilter(item.type);
    };
});


app.filter('typeToColor', function($filter) {
    const supportedTypes = [
        "project",
        "dataset",
        "streaming_endpoint",
        "recipe",
        "analysis",
        "notebook",
        "scenario",
        "saved_model",
        "model_evaluation_store",
        "model_comparison",
        "managed_folder",
        "web_app",
        "code_studio",
        "report",
        "dashboard",
        "insight",
        "article",
        "app",
        "labeling_task",
        "prompt_studio",
        "agent_tool",
        "retrievable_knowledge"
    ];

    function getStandardType(type) {
        if (!type) return;
        if (type.endsWith("_NOTEBOOK")) {
            return "notebook";
        }
        return type.toLowerCase();
    }
    return function(type, background) {
        if (!type) return "";
        const recipeColor = $filter('recipeTypeToColor')(type);
        if (recipeColor) {
            return recipeColor;
        }

        const stdType = getStandardType(type);
        if (supportedTypes.indexOf(stdType) >= 0) {
            return background ? "universe-background " + stdType : "universe-color " + stdType;
        }
        return "";
    };
});


app.filter('insightTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapInsightTypeToIcon;
});


app.filter('insightTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapInsightTypeToColor;
});

app.filter('chartTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapChartTypeToColor;
});

app.filter('insightTypeToDisplayableName', function(TypeMappingService) {
    return TypeMappingService.mapInsightTypeToDisplayableName;
});

app.filter('webappTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapWebappTypeToIcon;
});

app.filter('webappTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapWebappTypeToColor;
});

app.filter('webappTypeToName', function(TypeMappingService) {
   return TypeMappingService.mapWebappTypeToName;
});

app.filter('connectionTypeToNameForList', function(TypeMappingService) {
    return TypeMappingService.mapConnectionTypeToNameForList;
});
app.filter('connectionTypeToNameForItem', function(TypeMappingService) {
    return TypeMappingService.mapConnectionTypeToNameForItem;
});

app.filter("connectionTypeToIcon", function(TypeMappingService ){
     return TypeMappingService.mapConnectionTypeToIcon;
});

app.filter("credentialTypeToIcon", function(TypeMappingService ){
    return TypeMappingService.mapCredentialTypeToIcon;
});

app.filter('datasetTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapDatasetTypeToIcon;
});

app.filter('recipeTypeToColorClass', function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToColorClass;
});

app.filter('recipeTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToColor;
});
app.filter('recipeTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToIcon;
});


app.filter('modelTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapModelTypeToIcon;
});

app.filter("recipeTypeToLanguage", function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToLanguage;
});

app.filter('typeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapTypeToIcon;
});

/* used to translate old icons names provided by TypeMappingService to new icon collection name */
app.filter('toModernIcon', function(TypeMappingService) {
    return TypeMappingService.mapOldIconTypeToModernIcon;
})

app.filter('subTypeToIcon', function(TypeMappingService) {
  return TypeMappingService.mapSubtypeToIcon;
});

app.filter('subTypeToColor', function(TypeMappingService) {
   return TypeMappingService.mapSubtypeToColor;
});

app.filter('mlTaskTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapMlTaskTypeToIcon;
});


app.filter('backendTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapBackendTypeToIcon;
});

app.filter('analysisTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapAnalysisTypeToIcon;
});

app.filter('savedModelSubtypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapSavedModelSubtypeToIcon;
});

app.filter('savedModelTypeToClassColor', function(TypeMappingService) {
    return TypeMappingService.mapSavedModelTypeToClassColor;
});

app.filter('savedModelMLCategoryToClassColor', function(TypeMappingService) {
    return TypeMappingService.mapSavedModelMLCategoryToColor;
});

app.filter('agentToolTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapAgentToolTypeToIcon;
});

// Boldify a pattern in the input.
//
// Note:
// - Input is plain text (and not HTML, because it will be escaped)
// - Output is HTML (sanitized, ready for display)
app.filter('boldify', function(){
    function preg_quote( str ) {
        // eslint-disable-next-line no-useless-escape
        return (str+'').replace(/([\\\.\+\*\?\[\^\]\$\(\)\{\}\=\!\<\>\|\:])/g, "\\$1");
    }

    return function(input, replacements) {
        if (!replacements || replacements.length == 0 || !input) return sanitize(input);
        // Implementation notes:
        // - It is not possible to escape HTML entities and then boldify the pattern: this may produce invalid HTML if the pattern matches)
        // - It is not possible to boldify the pattern and then escape HTML entities: the boldification will be escaped
        // => Strategy is to split the string into tokens. Tokens are sanitized individually, and then enclosed in <b>...</b> if they match the pattern.

        var regex = new RegExp(replacements.map(function(e){
            if (e instanceof RegExp) {
                return "(?:"+e.source+")";
            } else {
                return "(?:"+preg_quote(e)+")";
            }
        }).join("|"), "gi");

        const highlightedSections = [];
        const rawTokenBoundaries = [0, input.length];

        input.replace(regex, function(val, pos) {
            highlightedSections.push([pos, pos + val.length]);
            rawTokenBoundaries.push(pos);
            rawTokenBoundaries.push(pos + val.length);
        });

        const tokenBoundaries = _.chain(rawTokenBoundaries).uniq().sortBy().value();

        let output = '';
        for(let i = 1; i < tokenBoundaries.length; i++) {
            const tokStartPos = tokenBoundaries[i-1];
            const tokEndPos = tokenBoundaries[i];
            let tokHighlighted = false;

            for(let j = 0; j < highlightedSections.length; j++) {
                const hlStartPos = highlightedSections[j][0];
                const hlEndPos = highlightedSections[j][1];

                if(hlStartPos < tokEndPos && tokStartPos < hlEndPos) {
                    tokHighlighted = true;
                    break;
                }
            }

            const token = input.substring(tokStartPos, tokEndPos);

            if(tokHighlighted) output += '<b>';
            output += sanitize(token);
            if(tokHighlighted) output += '</b>';
        }

        return output;
    }
});

app.filter('connectionNameFormatter', function () {
    const virtualConnectionRegex = "^@virtual\\((.+)\\):(connection:)?(.+)$";

    return function (input) {
        if (!input) return input;
        const match = input.match(virtualConnectionRegex);
        return match ? `Hive (${match[3]})` : input;
    }
});

app.filter('niceType', function(TypeMappingService) {
    return TypeMappingService.mapToNiceType;
});

app.filter('niceLogin', function() {
    return (login) => login === 'no:auth' ? 'Dataiku' : login;
});

app.filter('nicePrecision', function() {
    return function(val, p) {
        if (val == undefined || val != val || val == null || !val.toFixed)
            return undefined;
        if (Math.abs(val) < Math.pow(10, p)) {
            if (Math.round(val) == val) {
                /* Don't add stuff to integers */
                return val.toFixed(0);
            } else {
                return val.toPrecision(p);
            }
        } else {
            return val.toFixed(0);
        }
    };
});

app.filter('ifEmpty', function() {
    return (value, defaultValue) => value == null ? defaultValue : value;
});

app.filter('niceConst', function(Ng2NiceConstService) {
    return Ng2NiceConstService.transform.bind(Ng2NiceConstService);
});

app.filter('niceMLBackendType', function(LoggerProvider) {
    var niceNames = {
        'PY_MEMORY': 'Python (in memory)',
        'MLLIB': 'Spark MLLib',
        'H2O': 'Sparkling Water (H2O)',
        'VERTICA': 'Vertica Advanced Analytics',
        'DEEP_HUB': 'Deep Learning (PyTorch)'
    };

    var Logger = LoggerProvider.getLogger('DisplayFilters');

    return function(input) {
        if (!niceNames[input]) {
            Logger.warn("ML backend has no display name: "+input);
            return input;
        } else {
            return niceNames[input];
        }
    }
});


// input | replace:search_str:replacement
// input | replace:search_regexp:flags:replacement
app.filter('replace', function() {
    return function(input, regexp, flags, replace) {
        if (typeof replace === 'undefined') {
            replace = flags;
        } else {
            regexp = new RegExp(regexp, flags);
        }
        return regexp ? input.replace(regexp, replace) : input;
    }
});


app.filter('startAt', function() {
    return function(input, first) {
        first = parseInt(first, 10);
        var out = [];
        if (first >= input.length) {
            return out;
        }
        for (let i = first; i<input.length; i++) {
            out.push(input[i]);
        }
        return out;
    };
});


app.filter('shakerStepIcon', function(ShakerProcessorsUtils) {
    return function(step, size) {
        return ShakerProcessorsUtils.getStepIcon(step.type, step.params, size);
    };
});


app.filter('shakerStepDescription', function(ShakerProcessorsUtils, $filter) {
    return function(step, processors) {
        return ShakerProcessorsUtils.getHtmlStepDescriptionOrError(step, processors);
    };
});


app.filter('objToParamsList', function(){
    return function(obj) {
        var arr = [];
        $.each(obj, function(key, val){
            arr.push(key + ": " + val);
        });
        return arr.join(', ');
    }
});


app.filter('filesize', function(){
    return function(size){
        if (size >= 1024*1024*1024) {
            return Math.round(size / 1024 / 1024 / 1024 * 100)/100 + ' GB';
        } else if (size >= 1024*1024){
            return Math.round(size / 1024 / 1024*100)/100 + ' MB';
        } else {
            return Math.round(size / 1024 *100)/100 + ' KB';
        }
    };
});


app.filter('fileSizeOrNA', function(){
    return function(size){
        if (size < 0) {
            return "N/A";
        } else if (size >= 1024*1024*1024) {
            return Math.round(size / 1024 / 1024 / 1024 * 100)/100 + ' GB';
        } else if (size >= 1024*1024){
            return Math.round(size / 1024 / 1024*100)/100 + ' MB';
        } else {
            return Math.round(size / 1024 *100)/100 + ' KB';
        }
    };
});


app.filter('fileSizeInMB', function(){
    return function(size){
        if (size < 0) {
            return "N/A";
        } else {
            return Math.round(size / 1024 / 1024*100)/100 + ' MB';
        }
    };
});


app.filter("displayMeasure", function() {
    return function(measure) {
        if (measure.function == 'SUM') {
            return 'Sum of ' + measure.column.name;
        } else if (measure.function == 'AVG') {
            return 'Average of ' + measure.column.name;
        } else {
            return "I have no idea what this is about " + measure.column;
        }
    };
});


app.filter("percentage", function() {
    return function(val) {
        return Math.round(val * 100) + '%';
    };
});


app.filter('toKVArray', function(){
    return function(dict) {
        if (dict){
            return $.map(dict, function(v, k){
                return {k:k, v:v};
            });
        } else {
            return [];
        }
    };
});


function objectToArray(dict, saveKey) {
    if (dict) {
        return $.map(dict, function (v, k) {
            if (saveKey) {
                v['_key'] = k;
            }
            return v;
        });
    }
    return [];
}

app.filter('toArray', function(){
    return (dict) => objectToArray(dict);
});

app.filter('toArrayWithKey', function () {
    return (dict) => objectToArray(dict, true);
});


app.filter('datasetPartition', function() {
    return function(val) {
        if (val.partition && val.partition != 'NP') {
            return val.dataset + " (" + val.partition + ")";
        } else {
            return val.dataset;
        }
    };
});


app.filter('breakText', function(){
    return function(text, breakon){
        var bo = breakon || '_';
        return text.replace(new RegExp(bo, 'g'), bo + '&#8203;');
    };
});


app.filter('truncateText', function(){
   return function(text, val){
        if (val == null) val = 30;
       return text.substring(0, val);
   };
});

app.filter('subString', function(){
    return function(text, start, end){
        if (start > end) {
            let s = start;
            start = end;
            end = s;
        }
        return text.substring(start, end);
    };
 });

app.filter('sanitize', function() {
    return function(x) {
        return sanitize(x);
    };
});

app.filter('sanitizeHighlighted', function() {
    return function(x) {
        return sanitizeHighlighted(x);
    };
});

app.filter("stripHtml", function($sanitize) {
    return (htmlString) => $("<div>").html($sanitize(htmlString)).text();
})

app.filter('escape', function() {
   return function(x) {
       return escape(x);
   };
});


app.filter('escapeHtml', function() {
    var chars = /[<>&'"]/g,
        esc = (function(_) { return this[_]; }).bind({
            '<': '&lt;', '>': '&gt;', '&': '&amp;', '"': '&quot;', "'": '&apos;' });
    return function(s) { return s.replace(chars, esc); };
});


app.filter('unescapeHtml', function() {
    var chars = /&(lt|gt|amp|quot|apos|#(\d+)|#x([0-9a-fA-F]+));?/g,
        esc = (function(_, code, dec, hex) {
            if (code in this) return this[code];
            if (dec || hex) return String.fromCharCode(parseInt(dec || hex, dec ? 10 : 16));
            return _;
        }).bind({ lt: '<', gt: '>', amp: '&', quot: '"', apos: "'" });
    return function(s) { return s.replace(chars, esc); };
});


app.filter('quotedString', function() {
    return function(x) {
        return `'${x.replaceAll('\\', '\\\\').replaceAll("'", "\\'")}'`;
    };
});

app.filter('meaningLabel', function($rootScope) {
    return function(input) {
        return $rootScope.appConfig.meanings.labelsMap[input] || input;
    };
});


app.filter('buildPartitionsDesc', function() {
    return function(input) {
        if (input.useExplicit) {
            return input.explicit;
        } else if (input.start) {
            if (input.start == input.end) {
                return input.start;
            } else {
                return input.start + " / " +input.end;
            }
        } else {
            return input;
        }
    };
});


// Similar to linky for doesn't detect emails
app.filter('parseUrlFilter', function() {
    // eslint-disable-next-line no-useless-escape
    var urlPattern = /(\b(https?|ftp|file):\/\/[-A-Z0-9+&@#\/%?=~_|!:,.;]*[-A-Z0-9+&@#\/%=~_|])/ig;

    return function(text, target) {
        if(!target) {
            target = '_blank';
        }
        if(text) {
            return text.replace(urlPattern,'<a href="$1" target="'+target+'">$1</a>');
        }
        else {
            return '';
        }
    };
});


app.filter('dynamicFormat', function($filter) {
      return function(value, filterName) {
        return $filter(filterName)(value);
      };
});


app.factory('$localeDurations', [function () {
    return {
        'one': {
            year: '{} year',
            month: '{} month',
            week: '{} week',
            day: '{} day',
            hour: '{} hour',
            minute: '{} minute',
            second: '{} second'
        },
        'other': {
            year: '{} years',
            month: '{} months',
            week: '{} weeks',
            day: '{} days',
            hour: '{} hours',
            minute: '{} minutes',
            second: '{} seconds'
        }
    };
}]);


app.filter('duration', ['$locale', '$localeDurations', function ($locale, $localeDurations) {
    return function duration(value, unit, precision) {
        var unit_names = ['year', 'month', 'week', 'day', 'hour', 'minute', 'second'],
            units = {
                year: 86400*365.25,
                month: 86400*31,
                week: 86400*7,
                day: 86400,
                hour: 3600,
                minute: 60,
                second: 1
            },
            words = [],
            max_units = unit_names.length;


        precision = parseInt(precision, 10) || units[precision || 'second'] || 1;
        value = (parseInt(value, 10) || 0) * (units[unit || 'second'] || 1);

        if (value >= precision) {
            value = Math.round(value / precision) * precision;
        } else {
            max_units = 1;
        }

        var i, n;
        for (i = 0, n = unit_names.length; i < n && value !== 0; i++) {
            var unit_name = unit_names[i],
                unit_value = Math.floor(value / units[unit_name]);

            if (unit_value !== 0) {
                words.push(($localeDurations[unit_value] || $localeDurations[$locale.pluralCat(unit_value)] || {unit_name: ('{} ' + unit_name)})[unit_name].replace('{}', unit_value));
                if (--max_units === 0) break;
            }

            value = value % units[unit_name];
        }

        if (words.length){
            return words.join(" ");
        }
        return '0s';
    };
}]);

app.filter('fsProviderDisplayName', function(TypeMappingService) {
    return TypeMappingService.mapToFsProviderDisplayName;
});

app.filter('cleanConnectionName', function() {
    return function(input) {
        if (input && input.startsWith("@virtual(impala-jdbc):")) {
            return "Impala builtin";
        } else {
            return input;
        }
    };
});

app.filter('uniqueStrings', function() {
    return function(x) {
        if (!x) return x;
        var uniqueValues = [];
        x.forEach(function(v) {
           if (uniqueValues.indexOf(v) < 0) {
               uniqueValues.push(v);
           }
        });
        return uniqueValues;
    };
 });

app.filter('encodeHTML', function() {
    return rawStr => String(rawStr).replace(/[\u00A0-\u9999<>&]/gim, i => '&#'+i.charCodeAt(0)+';');
});

app.filter('map2Object', function() {
    return function(input) {
      var output = {};
      input.forEach((value, key) => output[key] = value);
      return output;
    };
});

app.filter("bundleProjectContent", function() {
    const bundleContentConfigMap = {
        datasets: 'Datasets',
        recipes: 'Recipes',
        savedModels: 'Saved models',
        modelEvaluationStores: 'Evaluation stores',
        managedFolders: 'Managed folders',
        scenarios: 'Scenarios',
        analysis: 'Analyses',
        jupyterNotebooks: 'Jupyter notebooks',
        labelingTasks: 'Labeling tasks',
        sqlNotebooks: 'SQL notebooks',
        insights: 'Insights',
        dashboards: 'Dashboards',
        knowledgeBanks: 'Knowledge Banks',
        promptStudios: 'Prompt Studios'
    }
    return input => bundleContentConfigMap[input];
});

app.filter('displayablePredictionType', function() {
    const map = {
        "BINARY_CLASSIFICATION": "binary classification",
        "REGRESSION": "regression",
        "MULTICLASS": "multiclass",
        "DEEP_HUB_IMAGE_OBJECT_DETECTION": "object detection",
        "DEEP_HUB_IMAGE_CLASSIFICATION": "image classification"
    }
    return input => map[input];
});

app.filter("targetRoleName", function() {
    return function(isCausal) {
        return isCausal ? "outcome" : "target";
    };
})

/**
 * Sample input: config_option-1 or config-option_1
 * Sample ouptut: Config option 1
 */
app.filter('prettyConfigOption', function(){
    return function(input){
        if(!input || !input.length) {
            return '';
        }
        let output = input.replace(/[-_]/g,' ');
        return output.charAt(0).toUpperCase() + output.slice(1);
    };
});

app.filter('formatCategoricalValue', function() {
    return function(input) {
        return input === '' ? '<empty>' : input;
    }
})

function andOrList(andor) {
    return function(input) {
        if (!input || !input.length) {
            return '';
        }

        if (input.length > 2) {
            input = [input.slice(0, -1).join(', '), input.slice(-1)[0]];
        }

        return input.join(' '+andor+' ');
    }
}

/*
    Input: array of strings
    Returns a comma-separated list with "and" before the last item (when list length is > 1)
*/
app.filter('andList', function() {
    return andOrList('and');
});

/*
    Input: array of strings
    Returns a comma-separated list with "or" before the last item (when list length is > 1)
*/
app.filter('orList', function(translate) {
    return andOrList(translate('GLOBAL.FILTERS.OR_LIST.OR', 'or'));
});

/*
    Returns a light or dark hex code given background color hex code string as input
*/
app.filter('colorContrast', function() {
    const DARK = '#000000';
    const LIGHT = '#FFFFFF';
    const YIQ_THRESHOLD = 128;

    return function(hexCode) {
        // https://24ways.org/2010/calculating-color-contrast/
        const [r, g, b] = extractColorValues(hexCode);
        const yiq = ((r * 299) + (g * 587) + (b * 114)) / 1000;

        return isNaN(yiq) ? DARK : (yiq >= YIQ_THRESHOLD) ? DARK : LIGHT;
    }

    function extractColorValues(color) {
        color = (color.charAt(0) === '#') ? color.substring(1, color.length) : color;
        return [
            parseInt(color.substring(0, 2), 16),
            parseInt(color.substring(2, 4), 16),
            parseInt(color.substring(4, 6), 16)
        ];
    }
});

 /**
 * Parses a string and converts it to an array
 *
 * If string is array-like ('["item1","item2"]') return as array object (["item1","item2"]). If regular string, return single element array
 * If not a string, return null
 * @param string str
 */
app.filter('stringToArray', function(LoggerProvider) {
    return function(str) {
        if (typeof str === 'string') {
            if (str.startsWith('[') && str.endsWith(']')) {
                try {
                    const arr = JSON.parse(str);

                    if (Array.isArray([arr])) {
                        return arr;
                    }
                } catch (err) {
                    // was not valid JSON string
                    LoggerProvider.getLogger('DisplayFilters').error("Error ", err);
                    return null;
                }
            }

            // if normal string, return as singleton array
            return [str];
        }
    }
});

app.filter('platformKey', function(DetectUtils) {
    return function(key) {
        switch (key) {
            case 'mod':
            case 'cmd':
            case 'command':
                return DetectUtils.isMac() ? "⌘" : "Ctrl";
            case 'ctrl':
                return DetectUtils.isMac() ? "⌃" : "Ctrl";
            case 'shift':
                return DetectUtils.isMac() ? "⇧" : "Shift";
            case 'alt':
                return DetectUtils.isMac() ? "⌥" : "Alt";
            default:
                return key;
        }
    }
});

 /**
 * Extracts the value of a specified key for each element in an array of objects
 * and returns an array of these values
 *
 * Example: [{id: 5, id: 6, id: 7}] -> [5, 6, 7]
 * @param Array array
 * @param string key
 */
app.filter('extractKey', function() {
    return function(array, key) {
        return (array || []).map(item => item[key]);
    };
});

app.filter('prettifyScenarioType', function(ScenarioUtils) {
    /**
     * Returns the display name for a given scenario step. Also handle plugin steps.
     * @param step: the step object containing the type attribute in technical format
     * @returns the display name of the step type, translated
     */
    return function(step) {
        return ScenarioUtils.getStepDisplayType(step);
    }
});

 /**
 * Displays a fallback string if the input string is null
 * 
 * @param string input containing string
 * @param string fallback string if input is null/undefined
 * @returns the string or the default value
 */
app.filter('stringWithFallback', function() {
    return function(input, fallback = 'N/A') {
        return input == null ? fallback : input;
    }
});

})();
