(function() {
'use strict';

const app = angular.module('dataiku.lambda');


app.constant('ENDPOINT_TYPES', {
    STD_PREDICTION: "Prediction",
    STD_CAUSAL_PREDICTION: "Causal Prediction",
    STD_FORECAST: "Time Series Forecasting",
    CUSTOM_PREDICTION: "Custom prediction (Python)",
    CUSTOM_R_PREDICTION: "Custom prediction (R)",
    DATASETS_LOOKUP: "Lookup",
    SQL_QUERY: "SQL Query",
    PY_FUNCTION: "Python function",
    R_FUNCTION: "R function",
    STD_CLUSTERING: "Clustering",
});


app.controller("LambdaServiceEndpointsController", function($stateParams, $scope, $state, DataikuAPI, WT1, TopNav, Dialogs, CreateModalFromTemplate,
        ENDPOINT_TYPES, SavedModelsService, $filter) {
    TopNav.setLocation(TopNav.TOP_HOME, "lambda", TopNav.TABS_LAMBDA, "endpoints");
    $scope.endpointTypes = ENDPOINT_TYPES;
    $scope.uiState = {
        activeEndpoint: null
    };
    $scope.$watch("uiState.activeEndpoint", function(nv, ov) {
        $scope.endpoint = nv;
        if (!$scope.endpoint) {
            return;
        }
        if ($scope.endpoint.testQueries.length > 0) {
            $scope.endpoint.testQueries.map((tq, i) => tq.$queryIndex = i);
        }
    });
    $scope.$watch("service", function(nv, ov) {
        if (nv && $scope.uiState.activeEndpoint == null && nv.endpoints.length) {
            $scope.uiState.activeEndpoint = nv.endpoints[0];
        }
        DataikuAPI.lambda.services.getSummary($stateParams.projectKey, $stateParams.serviceId).success(function(data) {
            $scope.authMethod = data.object.authMethod
       });
        SavedModelsService.listAPIServiceableModels($stateParams.projectKey).then(function(serviceableModels) {
            $scope.savedModels = serviceableModels.filter(sm => sm.miniTask && sm.miniTask.taskType === 'PREDICTION');
            $scope.clusteringSavedModels = serviceableModels.filter(sm => sm.miniTask && sm.miniTask.taskType === 'CLUSTERING');
        });
    });

    $scope.getEndpointIcon = function(endpoint) {
        if (!$scope.savedModels) return;
        const savedModel = $scope.savedModels.find(sm => sm.id === endpoint.modelRef) || {};
        const miniTask = savedModel.miniTask || {};
        return $filter("savedModelSubtypeToIcon")(miniTask.taskType, miniTask.backendType, miniTask.predictionType, savedModel.savedModelType, undefined, 16);
    }

    $scope.renameEndpoint = function() {
        Dialogs.prompt($scope, "Rename endpoint", "New endpoint name", $scope.endpoint.id, { pattern: "[\\w-_]+" }).then(function(newName) {
            if (!newName || newName === $scope.endpoint.id) return;
            if ($scope.service.endpoints.some(ep => ep.id === newName)) {
                Dialogs.ack("This name is already taken by another endpoint of this API service.");
                return;
            }
            $scope.endpoint.id = newName;
            $scope.saveService();
        });
    };

    $scope.deleteEndpoint = function() {
        const endpoints = $scope.service.endpoints;
        Dialogs.confirm($scope, "Confirm deletion", 'Are you sure you want to delete API endpoint "'+$scope.endpoint.id+'"?').then(function() {
            endpoints.splice(endpoints.indexOf($scope.uiState.activeEndpoint), 1);
            $scope.saveService();
            $scope.uiState.activeEndpoint = endpoints.length ? endpoints[0] : null;
        });
    };

    $scope.createNewEndpoint = function() {
        CreateModalFromTemplate("/templates/lambda/new-endpoint-modal.html", $scope, null, function(modalScope) {
            modalScope.selection = {type: "STD_PREDICTION"};
            modalScope.endpointsIds = $scope.service.endpoints.map(ep => ep.id);
            modalScope.ep = {
                projectKey: $stateParams.projectKey, id: null, code: '', type: modalScope.selection.type,
                modelType: 'CLASSIFICATION', testQueries: [], enrichMapping: [], useJava: true, computePropensity: true
            };
            modalScope.savedModels = $scope.savedModels;
            modalScope.clusteringSavedModels = $scope.clusteringSavedModels;

            const previousSMSelection = {};
            modalScope.$watch("selection.type", (newVal, oldVal) => {
                if (oldVal) {
                    previousSMSelection[oldVal] = modalScope.ep.modelRef;
                }
                if (newVal) {
                    modalScope.ep.modelRef = previousSMSelection[newVal];
                }
                // Update the endpoint type based on the selected type
                modalScope.ep.type = modalScope.selection.type
            });
        }).then(function(endpoint) {
            if (!endpoint) {
                return;
            }
            if (endpoint.type === 'CUSTOM_PREDICTION') {
                const tpl = document.getElementById('custPredEndpointTpl/PYTHON_' + endpoint.modelType);
                if (tpl) {
                    endpoint.code = tpl.innerHTML;
                }
            } else if (endpoint.type === 'CUSTOM_R_PREDICTION') {
                const tpl = document.getElementById('custPredEndpointTpl/R_' + endpoint.modelType);
                if (tpl) {
                    endpoint.code = tpl.innerHTML;
                }
                endpoint.userFunctionName = "prediction_function";
            } else if (endpoint.type === 'R_FUNCTION') {
                const tpl = document.getElementById('custEndpointTpl/R_FUNCTION');
                if (tpl) {
                    endpoint.code = tpl.innerHTML;
                }
                endpoint.userFunctionName = "api_r_function";
            } else if (endpoint.type === 'PY_FUNCTION') {
                const tpl = document.getElementById('custEndpointTpl/PY_FUNCTION');
                if (tpl) {
                    endpoint.code = tpl.innerHTML;
                }
                endpoint.userFunctionName = "api_py_function";
            } else if (endpoint.type == "SQL_QUERY") {
                endpoint.queries = [{ query: "-- Insert your SQL code here", inputParameters: []}]
            } else if (endpoint.type == "DATASETS_LOOKUP") {
                endpoint.lookups = [];
            }

            $scope.service.endpoints.push(endpoint);
            WT1.event('create-api-endpoint', {type: endpoint.type, nEndpints: $scope.service.endpoints.length});
            $scope.saveService();
            $scope.uiState.activeEndpoint = $scope.service.endpoints[$scope.service.endpoints.length - 1];
        });
    };

    $scope.canDevServer = function(endpoint) {
        if (!endpoint) {
            return false;
        }
        const t = endpoint.type;
        return t.substring(t.length - 11) == '_PREDICTION'
            || t === 'STD_FORECAST'
            || t.substring(t.length-9) == '_FUNCTION'
            || t == 'DATASETS_LOOKUP'
            || t == 'SQL_QUERY'
            || t == 'STD_CLUSTERING';
    };

    $scope.sendPlayTestQueriesMsg = function() {
        $scope.$broadcast("playTestQueries");
    };
});


app.controller("_EndpointController", function($scope, $stateParams, DataikuAPI, Fn, DatasetUtils, Logger, WT1) {
    $scope.$on("devServerDeploymentStarted", function(evt, promise) {
        if ($scope.testSettingsPane) {
            promise.success(function(data) {
                $scope.deployResult = data;
                $scope.uiState.settingsPane = "test";
            // eslint-disable-next-line no-console
            }).catch(console.info.bind(console));   /*@console*/
        }
    });

    $scope.$on("playTestQueries", function() {
        WT1.event('api-designer-test-queries', {nQueries: $scope.endpoint.testQueries.length, outputExplanations: $scope.endpoint.outputExplanations});
        if ($scope.testQueriesMode) {
            Logger.info("Playing testQueries:", $scope.endpoint.testQueries);
            $scope.deployToDevServer().then(function(result) {
                if (result.data && result.data.error) return; // not trying to run queries if deployment returned an error
                DataikuAPI.lambda.services.playTestQueries($stateParams.projectKey, $stateParams.serviceId,
                        $scope.endpoint.id, $scope.testQueriesMode, $scope.endpoint.testQueries).success(function(data) {
                            $scope.testQueriesResult = data;
                            for (let curResult of $scope.testQueriesResult.responses) {
                                const contentTypeHeader = curResult.headers.find(h => h.first.toLowerCase() === "content-type");
                                if (contentTypeHeader && contentTypeHeader.second.startsWith("image/")) {
                                    curResult.imageType = contentTypeHeader.second;
                                    curResult.url = `data:${curResult.imageType};base64,${curResult.response}`;
                                }
                            }
                        }).error(setErrorInScope.bind($scope));
            });
        }
    });
});


app.controller("_WithEnrichmentsController", function($scope, $controller, $stateParams, Assert, DataikuAPI, Fn, Logger) {
    $scope.enrichmentsList = null;

    if ($scope.endpoint.type == "DATASETS_LOOKUP") {
        $scope.enrichmentsList = $scope.endpoint.lookups;
    } else {
        $scope.enrichmentsList = $scope.endpoint.enrichMapping;
    }
    Assert.inScope($scope, 'enrichmentsList');

    $scope.uiState = {};
    if ($scope.enrichmentsList && $scope.enrichmentsList.length>0) {
        $scope.enrichmentIndex = 0;
    }
    DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {datasetsOnly:true}).success(function(datasets) {
        $scope.datasets = datasets;
        $scope.datasets.forEach(function(ds) {
            ds.localProject = ds.projectKey === $stateParams.projectKey;
        });

        $scope.defaultDataset = datasets.filter(Fn.prop('localProject'))[0] || datasets[0];
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.admin.getGeneralSettings().success(function(gs) {
        $scope.devBundledConnection = gs.lambdaDevBundledConnection;
    });

    $scope.showEnrichment = function(index) {
        $scope.enrichmentIndex = index;
    };

    $scope.enrichMappingTemplate = function() { return ; };

    $scope.addEnrichment = function() {
        $scope.enrichmentsList.push({
            packagingType: 'BUNDLED_TOCONNECTION',
            on: [],
            columnsMapping: {},
            missingLookupKeyBehavior: 'IGNORE',
            notFoundBehavior: 'IGNORE',
            multiMatchBehavior: 'KEEP_FIRST'
        });
        $scope.enrichmentIndex = $scope.enrichmentsList.length - 1;
    };

    $scope.deleteEnrichment = function(index) {
        $scope.enrichmentsList.splice(index, 1);
        if (index < $scope.enrichmentIndex) {
            $scope.enrichmentIndex--;
        } else if (index === $scope.enrichmentIndex) {
            $scope.enrichmentIndex = -1;
        }
    };
});


app.controller("SingleEnrichmentController", function($scope, Logger, AnyLoc, $stateParams, DataikuAPI, DatasetUtils) {
    $scope.uiState = {};
    $scope.selection = {};
    $scope.currentEnrichment = $scope.enrichmentsList[$scope.enrichmentIndex];

    function getColumnsUIState(datasetColumns, columnsMapping) {
        if (datasetColumns) {
            datasetColumns
                .filter(column => column.name in columnsMapping)
                .forEach(column => {
                    column.$selected = true;
                    if (columnsMapping[column.name] != column.name) {
                        column.modelFeature = columnsMapping[column.name];
                    }
                });
        }
        return datasetColumns;
    }

    function getColumnsMapping(columns) {
        return columns
            .filter(col => col.$selected)
            .reduce((dict, col) => {
                dict[col.name] = col.modelFeature || col.name;
                return dict;
            }, {});
    }

    $scope.$watch("currentEnrichment.datasetRef", function(nv) {
        if (nv) {
            const datasetLoc = AnyLoc.getLocFromSmart($stateParams.projectKey, nv);
            DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.localId, $stateParams.projectKey)
                .success(function(data) {
                    $scope.uiState.columns = getColumnsUIState(data.schema.columns, $scope.currentEnrichment.columnsMapping);
                    // new uiState.columns => new columnsMapping thanks to watch on uiState.columns
                    // still need to update "on"
                    const columnNames = $scope.uiState.columns.map(x => x.name);
                    $scope.currentEnrichment.on = $scope.currentEnrichment.on.filter(x => (columnNames.indexOf(x.resourceLookupCol) > -1));
                }).error(setErrorInScope.bind($scope));
        }
    });

    $scope.$watch("currentEnrichment.packagingType", (newValue) => {
        if (newValue === 'REFERENCED') {
            $scope.datasets.forEach(ds => ds.usable = DatasetUtils.isSQL(ds.dataset));
        } else if (newValue == 'BUNDLED_TOCONNECTION') {
            $scope.datasets.forEach(ds => ds.usable = true);
        }
    });

    // if uiState.columns changes, update columnsMapping - this watches the change on feature name
    $scope.$watch("uiState.columns", function(nv) {
        if (nv) {
            $scope.currentEnrichment.columnsMapping = getColumnsMapping(nv);
        }
    }, true);
    // We need also to watch $selected of each item - since it's not watchable ($-prefixed),
    // instead we watch selection.selectedObjects in non-deep mode
    $scope.$watch("selection.selectedObjects", function(nv) {
        if (nv && $scope.uiState.columns) {
            $scope.currentEnrichment.columnsMapping = getColumnsMapping($scope.uiState.columns);
        }
    });
});

app.controller("PredictionEndpointController", function($scope, $controller, $stateParams, DataikuAPI, Fn, DatasetUtils, Logger, SavedModelsService,
                                                        FullModelLikeIdUtils) {
    $controller("_EndpointController", {$scope:$scope});
    $controller("_WithEnrichmentsController", {$scope:$scope});
    $scope.testSettingsPane = 'test';

    $scope.uiState.settingsPane = 'loading';
    $scope.endpoint.outputExplanations = $scope.endpoint.outputExplanations || false;

    // Note: the compatibility is not updated for clustering, it's ok because all clustering models are API node compatible
    function resetCompatibility() {
        $scope.pythonCompatibility = { compatible: true, reason: null };
        $scope.javaCompatibility = { compatible: true, reason: null };
        $scope.apiNodeCompatibility = { compatible: true, reason: null };
    }

    $scope.$watch('endpoint', function(nv) {
        if (!nv) return;
        // Set the correct default pane
        if (['loading', 'model', 'settings', 'code'].indexOf($scope.uiState.settingsPane) >= 0) {
            $scope.uiState.settingsPane = ['STD_PREDICTION', 'STD_CAUSAL_PREDICTION', 'STD_CLUSTERING', 'STD_FORECAST'].includes(nv.type) ? 'model' : 'settings';
        }
        if (["CUSTOM_R_PREDICTION", "CUSTOM_PREDICTION"].indexOf($scope.endpoint.type) >= 0) {
            $scope.endpoint.envSelection = $scope.endpoint.envSelection || {envMode: 'INHERIT'};
        }
    });

    $scope.isTimeseriesForecast = function() {
        return $scope.savedModel && $scope.savedModel.miniTask && $scope.savedModel.miniTask.predictionType == "TIMESERIES_FORECAST";
    }

    $scope.isCausalPrediction = function () {
        return $scope.savedModel &&
               $scope.savedModel.miniTask &&
               ["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.savedModel.miniTask.predictionType);
    }

    $scope.canComputePropensity = function() {
        if (!$scope.isCausalPrediction()) {
            return false;
        }
        return $scope.savedModel.miniTask.modeling && $scope.savedModel.miniTask.modeling.propensityModeling.enabled;
    }

    $scope.isClusteringModel = function() {
        return $scope.savedModel && $scope.savedModel.miniTask && $scope.savedModel.miniTask.taskType == "CLUSTERING";
    }

    function setTestQueriesMode() {
        if ($scope.isTimeseriesForecast()) {
            $scope.testQueriesMode = 'forecast';
        } else if ($scope.isCausalPrediction()) {
            $scope.testQueriesMode = 'predict-effect';
        } else {
            $scope.testQueriesMode = 'predict';
        }
    }
    setTestQueriesMode();

    // STD_PREDICTION | STD_CLUSTERING | STD_CAUSAL_PREDICTION | STD_FORECAST / Saved models
    function setSM(ref) {
        if (!ref || !$scope.savedModels) return;
        $scope.savedModel = $scope.savedModels.find(sm => sm.id == ref); // can be null if model has been deleted
        $scope.endpoint.type = "STD_PREDICTION";
        if ($scope.isTimeseriesForecast()) {
            $scope.endpoint.type = "STD_FORECAST";
        } else if ($scope.isCausalPrediction()) {
            $scope.endpoint.type = "STD_CAUSAL_PREDICTION";
            if (!$scope.canComputePropensity()) {
                $scope.endpoint.computePropensity = false;
            }
        } else if ($scope.isClusteringModel()) {
            $scope.endpoint.useJava = false;
            $scope.endpoint.outputExplanations = false;
            $scope.endpoint.type = "STD_CLUSTERING";
        } else if (!$scope.endpoint.individualExplanationParams) {
            $scope.endpoint.individualExplanationParams = {
                method: "ICE",
                nbExplanations: 3,
                shapleyBackgroundSize: 100
            };
        }
        resetCompatibility();
        if ($scope.savedModel && $scope.savedModel.miniTask.taskType === 'PREDICTION') {
            const fmiComponents = {
                projectKey: $stateParams.projectKey,
                savedModelId: $scope.savedModel.id,
                versionId: $scope.savedModel.activeVersion
            }
            if (fmiComponents.versionId !== undefined) {
                const fmi = FullModelLikeIdUtils.buildSavedModelFmi(fmiComponents);
                DataikuAPI.ml.prediction.getModelDetails(fmi).success(data => {
                    $scope.apiNodeCompatibility = data.apiNodeScoreCompatibility;
                    $scope.pythonCompatibility = data.pythonApiNodeScoreCompatibility;
                    $scope.javaCompatibility = data.javaScoreCompatibility;
                    if (!$scope.canComputeExplanations()) {
                        $scope.endpoint.outputExplanations = false;
                    }
                    if (!$scope.pythonCompatibility.compatible) {
                        $scope.endpoint.useJava = true; // not python compatible -> force java
                    }
                    if (!$scope.javaCompatibility.compatible) {
                        $scope.endpoint.useJava = false; // not java compatible -> force python
                    }
                }).error(setErrorInScope.bind($scope))
            }
        }
        if ($scope.savedModel && !$scope.savedModel.miniTask.predictionType) {
            $scope.endpoint.mlFlowOutputStyle = "RAW";
        }
        setTestQueriesMode();
    }

    resetCompatibility();

    $scope.$watch("endpoint.modelRef", setSM);
    SavedModelsService.listAPIServiceableModels($stateParams.projectKey).then(function(serviceableModels) {
        $scope.savedModels = serviceableModels;
        setSM($scope.endpoint.modelRef);
    });

    $scope.isMLflowModel = function() {
        return SavedModelsService.isMLflowModel($scope.savedModel);
    }

    $scope.hasAPredictionType = function(){
       return $scope.savedModel && $scope.savedModel.miniTask && $scope.savedModel.miniTask.predictionType;
    };

    $scope.isProxyModel = function() {
        return SavedModelsService.isProxyModel($scope.savedModel);
    }

    $scope.isExternalMLflowModel = function() {
        return SavedModelsService.isExternalMLflowModel($scope.savedModel);
    }

    $scope.isOptimizedEngineSelectionSupported = function() {
        return !$scope.isExternalMLflowModel() && !$scope.isClusteringModel();
    }

    $scope.getOptimizedEngineSelectionDisableMessage = function () {
        if ($scope.endpoint.outputExplanations) {
            return "Java scoring is not available when explanations are enabled";
        }
        if (!$scope.javaCompatibility.compatible) {
            return $scope.javaCompatibility.reason;
        }
        if (!$scope.pythonCompatibility.compatible) {
            return $scope.pythonCompatibility.reason;
        }
        return null;
    }

    $scope.getExplanationsDisableMessage = function () {
        if ($scope.isExternalMLflowModel() ) {
            if ($scope.endpoint.mlFlowOutputStyle !== "PARSED") {
                return "Cannot compute explanations with RAW MLflow Model output";
            } else {
                return null; // explanations supported
            }
        }
        if ($scope.isTimeseriesForecast()) {
            return "Explanations are not available for time series forecasting";
        }
        if ($scope.isCausalPrediction()) {
            return "Explanations are not available for causal inference models";
        }
        if (!$scope.pythonCompatibility.compatible) {
            return $scope.pythonCompatibility.reason;
        }
        if (!$scope.savedModel || !$scope.savedModel.miniTask || $scope.savedModel.savedModelType !== "DSS_MANAGED" || $scope.savedModel.miniTask.backendType !== 'PY_MEMORY') {
            return "Cannot compute explanations for this model";
        }
        return null;
    }

    $scope.canComputeExplanations = () => $scope.getExplanationsDisableMessage() === null;

    $scope.onOutputExplanationsChange = function() {
        if ($scope.endpoint.outputExplanations) {
            $scope.endpoint.useJava = false;
        }
    }

    $scope.onMLFlowOutputStyleChange = function() {
        if ($scope.endpoint.mlFlowOutputStyle !== "PARSED") {
            $scope.endpoint.outputExplanations = false;
        }
    }

    $scope.hasInconsistentOpenAPI = function() {
        return $scope.endpoint.openAPI && $scope.endpoint.openAPI.enabled && !$scope.endpoint.openAPI.isManual && $scope.endpoint.type !=="STD_PREDICTION";
    }

    // CUSTOM_PREDICTION / Managed folders
    function setMF(ref) {
        if (!ref || !$scope.managedFolders) return;
        $scope.managedFolder = $scope.managedFolders.filter(Fn(Fn.prop('id'), Fn.eq(ref)))[0]; // possibly undefined
    }
    $scope.$watch("endpoint.inputFolderRef", setMF);
    DataikuAPI.managedfolder.list($stateParams.projectKey).success(function(managedFolders) {
        $scope.managedFolders = managedFolders;
        setMF($scope.endpoint.inputFolderRef);
    });
    $scope.customCodeSnippetCategories = [''];
    $scope.$watch("endpoint.modelType", function(nv) {
        if (!nv) return;
        $scope.customCodeSnippetCategories = [nv === 'REGRESSION' ? 'py-regressor' : 'py-classifier'];
    });
});

app.component("apiServiceMonitoring", {
    templateUrl: '/templates/lambda/monitoring.html',
    bindings: {
        endpoint: '<',
        serviceId: '<',
        savedModel: '<',
    },
    controller: function($stateParams, DataikuAPI, SpinnerService, WT1) {
        const $ctrl = this;

        $ctrl.isServicePublished = false;
        $ctrl.deployments = []
        $ctrl.selectedDeploymentId = null;

        $ctrl.settingsPane = 'loading';

        WT1.event('deployment-monitoring-tab-visited');

        $ctrl.$onChanges = function(){
            $ctrl.refreshServiceStatus();
        }

        $ctrl.refreshServiceStatus = function() {
            SpinnerService.lockOnPromise(DataikuAPI.apideployer.client.getServiceEndpointMonitoring($ctrl.serviceId, $ctrl.endpoint.id)
                .success(resp => {
                    $ctrl.isServicePublished = resp.isServicePublished;

                    const deployments = resp.deployments
                        .filter(deployment => deployment.smv.originalProjectKey === $stateParams.projectKey) // Only display deployments that come from the current project
                        .map(deployment => ({
                            serviceVersion: deployment.publishedServiceVersion,
                            stage : deployment.infrastructure.stage,
                            id: deployment.id,
                            enabled: deployment.enabled,
                            savedModel: deployment.smv.originalSavedModelName,
                            savedModelVersion: deployment.smv.originalSavedModelVersionName,
                            healthMessages : '',
                            loadingHealthStatus: deployment.enabled,
                            monitoring: {isMonitored: false}
                        }));

                    if (!deployments.length) {
                        $ctrl.deployments = [];
                        return;
                    }

                    DataikuAPI.lambda.services.monitoring.list($stateParams.projectKey, resp.deployments.map(deployment => deployment.id), $ctrl.endpoint.id)
                        .success(function(data) {
                            $ctrl.deployments = deployments.map(deployment => {
                                for(const monitoring in data){
                                    if (data[monitoring].deploymentId === deployment.id && data[monitoring].endpointId === $ctrl.endpoint.id){
                                        return {
                                            ...deployment,
                                            "monitoring": data[monitoring]
                                        }
                                    }
                                }
                            });
                            $ctrl.deployments
                                .filter(deployment => deployment.enabled)
                                .forEach(deployment => { DataikuAPI.apideployer.client.getDeploymentHeavyStatus(deployment.id) // This call updates the deployments separately as it takes some time
                                    .success(heavyStatus => {
                                        deployment["health"] = heavyStatus.health;
                                        deployment["healthMessages"] = heavyStatus.healthMessages.messages.map(message => message.title).join(". ");
                                        deployment["loadingHealthStatus"] = false;
                                    })
                                    .error(setErrorInScope.bind($ctrl));
                                });
                        })
                        .error(setErrorInScope.bind($ctrl));

                }).error(setErrorInScope.bind($ctrl)));
        };
    }
});

app.component("monitoringDeployments", {
    templateUrl: '/templates/lambda/monitoring-deployments.html',
    bindings: {
        endpoint:'<',
        serviceId: '<',
        isServicePublished: '<',
        deployments: '<',
        reloadDeployments:'&'
    },
    controller: function($state, LambdaServicesService, $stateParams, $rootScope, WT1){
        const $ctrl = this;

        $ctrl.projectKey = $stateParams.projectKey;
        $ctrl.$state = $state;

        $ctrl.serviceOnDeployerHref = function(){
            if ($rootScope.appConfig.deployerMode === 'LOCAL') {
                return `/api-deployer/services/${$ctrl.serviceId}/`;
            } else if ($rootScope.appConfig.deployerURL) {
                return `${$rootScope.appConfig.deployerURL}/api-deployer/services/${$ctrl.serviceId}/`;
            }
        }

        $ctrl.deploymentOnDeployerHref = function(deploymentId) {
            if ($rootScope.appConfig.deployerMode === 'LOCAL') {
                return `/api-deployer/deployments/${deploymentId}/`;
            } else if ($rootScope.appConfig.deployerURL) {
                return `${$rootScope.appConfig.deployerURL}/api-deployer/deployments/${deploymentId}/`;
            }
        }

        $ctrl.openConfigureMonitoringModal = (deploymentId) => LambdaServicesService.openConfigureMonitoringModal($stateParams.serviceId, $ctrl.endpoint, deploymentId, $ctrl.reloadDeployments);

        $ctrl.openMonitoringDetailsModal = (deploymentMonitoring) => LambdaServicesService.openMonitoringDetailsModal(deploymentMonitoring, $stateParams.projectKey);

        $ctrl.deleteMonitoringModal = (deploymentId) => LambdaServicesService.deleteMonitoringModal(deploymentId, $ctrl.endpoint.id, $ctrl.reloadDeployments);

        $ctrl.wt1Event = () => WT1.event('deployment-monitoring-visit-mes');
    }
});

app.controller("FunctionEndpointController", function($scope, $controller, $stateParams, DataikuAPI) {
    $controller("_EndpointController", {$scope:$scope});
    $scope.testSettingsPane = 'test';
    $scope.testQueriesMode = 'function';

    $scope.uiState.settingsPane = 'loading';

    $scope.$watch('endpoint', function(nv) {
        if (!nv) return;
        if (['loading', 'settings', 'code'].indexOf($scope.uiState.settingsPane) >= 0) {
            $scope.uiState.settingsPane = "code"
        }
        $scope.endpoint.envSelection = $scope.endpoint.envSelection || {envMode:'INHERIT'};
        if ($scope.endpoint.type == 'R_FUNCTION') {
            $scope.customCodeSnippetCategories = ['apinode-r-function-endpoint'];
            $scope.envLang = 'R';
            $scope.envName = $stateParams.envName;
            $scope.sampleType = 'R';
        } else if ($scope.endpoint.type == 'PY_FUNCTION') {
            $scope.customCodeSnippetCategories = ['apinode-py-function-endpoint'];
            $scope.envLang = 'PYTHON';
            $scope.envName = $stateParams.envName;
            $scope.sampleType = 'python';
        }
    });

    DataikuAPI.managedfolder.list($stateParams.projectKey).success(function(managedFolders) {
        $scope.managedFolders = managedFolders;
        // setMF($scope.endpoint.inputFolderRef);
    });
});


app.controller("SQLQueryEndpointController", function($scope, $controller, $stateParams, DataikuAPI, Fn, DatasetUtils, Logger, CodeMirrorSettingService) {
    $controller("_EndpointController", {$scope:$scope});
    $scope.testSettingsPane = 'test';
    $scope.testQueriesMode = 'query';
    $scope.sqlQueryIndex = 0;

    $scope.uiState.settingsPane = 'loading';

    $scope.editorOptions =  CodeMirrorSettingService.get('text/x-sql');
    $scope.customCodeSnippetCategories = ['apinode-sql-query-endpoint'];
    $scope.sampleType = 'sql';

    DataikuAPI.connections.getNames('SQL').success(function (data) { $scope.sqlConnections = data; }).error(setErrorInScope.bind($scope));

    $scope.showSQLQuery = function(index) {
        $scope.sqlQueryIndex = index;
    };

    $scope.addSQLQuery = function() {
        $scope.endpoint.queries.push({query: "-- Insert your SQL code here", inputParameters: [], maxResults: 0});
    };

    $scope.deleteSQLQuery = function(index) {
        $scope.endpoint.queries.splice(index, 1);
        if (index < $scope.sqlQueryIndex) {
            $scope.sqlQueryIndex--;
        } else if (index === $scope.sqlQueryIndex) {
            $scope.sqlQueryIndex = -1;
        }
    };

    $scope.$watch('endpoint', function(nv) {
        if (!nv) return;
        // fromInputParameterNames();
        if (['loading', 'settings', 'query'].indexOf($scope.uiState.settingsPane) >= 0) {
            $scope.uiState.settingsPane = "query"
        }
    });
});


app.controller("DatasetsLookupEndpointController", function($scope, $controller, $stateParams, DataikuAPI, Fn, DatasetUtils, Logger) {
    $controller("_EndpointController", {$scope:$scope});
    $controller("_WithEnrichmentsController", {$scope:$scope});
    $scope.testSettingsPane = 'test';
    $scope.testQueriesMode = 'lookup';

    $scope.uiState.settingsPane = 'loading';

    $scope.$watch('endpoint', function(nv) {
        if (!nv) return;
        $scope.uiState.settingsPane = "lookups";
        $scope.endpoint.lookups = $scope.endpoint.lookups || [];
    });
});


app.component('endpointOpenApi',  {
    templateUrl: '/templates/lambda/open-api-editor.html',
    bindings: {
        endpoint: '='
    },
    controller: function($scope, DataikuAPI, Dialogs, $q,  $state, CodeMirrorSettingService, Logger, $stateParams, ClipboardUtils, WT1) {
        let $ctrl = this;
        $scope.CodeMirrorSettingService = CodeMirrorSettingService;
        const MANUAL = "Manual";
        const AUTO = "Automated";

        $scope.uiState = {
            jsonContent: null,
            genOptions: [
                { name: MANUAL, enabled: true, description: "In Manual Mode, the document is generated from a template and is fully editable"},
                { name: AUTO, enabled: false, description: "In Automated mode, The document is not editable and will be filled automatically"} // pay attention to tweaking of enabled in initOpenAPIEndpoint
            ],
            currentOption: null,
            enableOpenAPI: false,
            savedModelName: null,
            description: "",
        }

        function generateOpenAPI(isManual) {
            // prevent deserialization error if content is empty
            if ($ctrl.endpoint.openAPI && !$ctrl.endpoint.openAPI.content) {
                $ctrl.endpoint.openAPI.content = "";
            }
            DataikuAPI.lambda.services.generateEndpointOpenAPI($stateParams.projectKey, $ctrl.endpoint, $ctrl.endpoint.openAPI.enabled, isManual).success(function(data) {
                $ctrl.endpoint.openAPI = data;
                $scope.uiState.jsonContent = JSON.parse($ctrl.endpoint.openAPI.content);
            });
        }

        $scope.copyJsonToClipboard = function() {
            ClipboardUtils.copyToClipboard(JSON.stringify($scope.uiState.jsonContent, null, 4));
        }

        $scope.resetOpenAPIDocContent = function() {
            Dialogs.confirm($scope, 'Reset JSON', 'Are you sure you want to reset JSON? Saving changes will erase your manual edition.').then(function() {
                generateOpenAPI(true);
            });
        }

        function applyModeUpdate(nv) {
            $scope.uiState.currentOption = nv;
            if (nv !== MANUAL) {
                generateOpenAPI(false);
            } else {
                $ctrl.endpoint.openAPI.isManual = true;
            }
            if ($scope.uiState.currentOption !== undefined) WT1.event('api-designer-openapi-select-mode', { mode: nv });
        }

        $scope.updateMode = function(nv) {
            if (!angular.equals(nv, $scope.uiState.currentOption)){
                if (nv === AUTO) {
                    Dialogs.confirm($scope, 'Change to Automated Mode', 'Are you sure you want to change to Automated mode? Saving change will erase your manual edition.').then(function() {
                        applyModeUpdate(nv);
                    });
                } else {
                    applyModeUpdate(nv);
                }
            }
        }

        $scope.isInconsistentAutomatedMode = function() {
            return $scope.uiState.enableOpenAPI && !automatedOpenAPIIsAvailable($ctrl.endpoint.type) && $scope.uiState.currentOption === AUTO
        }

        let unbindJsonContent;
        function initOpenAPIEndpoint() {
            $scope.uiState.genOptions[1].enabled = automatedOpenAPIIsAvailable($ctrl.endpoint.type);
            $scope.uiState.currentOption = $ctrl.endpoint.openAPI.isManual ? MANUAL: AUTO;
            try {
                $scope.uiState.jsonContent = JSON.parse($ctrl.endpoint.openAPI.content);
            } catch (e) {
                Logger.warn("Unable to parse saved OpenAPI documentation, leaving it undefined: ", $ctrl.endpoint.openAPI.content);
                $scope.uiState.jsonContent = undefined;
            }

            if ($ctrl.endpoint.modelRef) {
                DataikuAPI.savedmodels.get($stateParams.projectKey, $ctrl.endpoint.modelRef).success(function(data) {
                    $scope.uiState.savedModelName = data.name;
                });
            }

            unbindJsonContent = $scope.$watch("uiState.jsonContent", function(nv) {
                if ($ctrl.endpoint.openAPI.isManual) {
                    $ctrl.endpoint.openAPI.content = JSON.stringify($scope.uiState.jsonContent);
                }
            });
        }

        $scope.$watch("uiState.description", function(nv) {
            if ($ctrl.endpoint.openAPI && $ctrl.endpoint.openAPI.description !== nv && !$ctrl.endpoint.openAPI.isManual) {
                $ctrl.endpoint.openAPI.description = nv;
            }
        });

        $scope.$watch("uiState.description", function(nv) {
            if ($ctrl.endpoint.openAPI && ($ctrl.endpoint.openAPI.description !== nv && !$ctrl.endpoint.openAPI.isManual)) {
                $ctrl.endpoint.openAPI.description = nv;
            }
            if ($ctrl.endpoint?.openAPI?.enabled && !$ctrl.endpoint?.openAPI?.isManual) {
                if (!nv) {
                    nv = "Describe what the endpoint call does, what input it takes, what it returns ...";
                }
                if ($scope.uiState?.jsonContent?.paths) {
                    Object.keys($scope.uiState?.jsonContent.paths).forEach(pathKey => {
                        if ($scope.uiState?.jsonContent.paths[pathKey].post) {
                            $scope.uiState.jsonContent.paths[pathKey].post.description = nv;
                        }
                    });
                }
            }
        });

        $scope.$watch(
            function() {
                return $ctrl.endpoint.openAPI?.content;
            },
            function(nv, ov) {
                if (nv !== ov) {
                    try {
                        $scope.uiState.jsonContent = JSON.parse($ctrl.endpoint.openAPI.content);
                    } catch (e) {
                        Logger.warn("Unable to parse saved OpenAPI documentation, leaving it undefined: ", $ctrl.endpoint.openAPI.content);
                        $scope.uiState.jsonContent = undefined;
                    }
                }
            },
            true
        )

        $ctrl.$onChanges = () => {
            if ($ctrl.endpoint) {
                $scope.uiState.description = $ctrl.endpoint.openAPI?.description || "";
                $scope.uiState.genOptions[1].enabled = automatedOpenAPIIsAvailable($ctrl.endpoint.type);
                $scope.uiState.enableOpenAPI = !!($ctrl.endpoint.openAPI && $ctrl.endpoint.openAPI.enabled);
                if ($ctrl.endpoint.openAPI && $ctrl.endpoint.openAPI.enabled) {
                    initOpenAPIEndpoint();
                    if ($scope.uiState.currentOption === AUTO) generateOpenAPI(false);
                }
            }
        }

        $scope.$watch("uiState.enableOpenAPI", function(nv,ov) {
            if (!angular.equals(nv, ov)) {
                if ($scope.uiState.enableOpenAPI) {
                    if (!$ctrl.endpoint.openAPI) {
                        Logger.info("Generate OpenAPI obj for old endpoint " + $ctrl.endpoint.id);
                        DataikuAPI.lambda.services.generateEndpointOpenAPI($stateParams.projectKey, $ctrl.endpoint, true, !automatedOpenAPIIsAvailable($ctrl.endpoint.type)).success(function(data) {
                            $ctrl.endpoint.openAPI = data;
                            initOpenAPIEndpoint();
                        });
                    } else {
                        $ctrl.endpoint.openAPI.enabled = true;
                        initOpenAPIEndpoint();
                    }
                } else {
                    if ($ctrl.endpoint.openAPI) $ctrl.endpoint.openAPI.enabled = false;
                    unbindJsonContent();
                }
                WT1.event('api-designer-openapi-toggle', { enabled: $ctrl.endpoint.openAPI && $ctrl.endpoint.openAPI.enabled});
            }
        });

        function automatedOpenAPIIsAvailable(endpointType) {
            return endpointType === "STD_PREDICTION";
        }
    }
});


app.directive('endpointTestQueries', function(DataikuAPI, $q, CreateModalFromTemplate, $state, $stateParams, TopNav, Dialogs, Fn, $filter, Logger, CodeMirrorSettingService, StringUtils, PromptUtils) {
    return {
        restrict: 'A',
        templateUrl: '/templates/lambda/endpoint-test-queries.html',
        replace: true,
        scope: {
            endpoint: '=',
            deployResult: '=',
            testQueriesResult: '=',
            type: '@',
            datasets: '=',
            inputDatasetSmartName: '=?',
            run: '&',
            smBackendType : "<?"
        },
        link: function($scope, element, attrs) {
            $scope.isTimeseriesForecast = function () {
                return $scope.endpoint && $scope.endpoint.type === "STD_FORECAST";
            }
            $scope.isCausalPrediction = function () {
                return $scope.endpoint && $scope.endpoint.type === "STD_CAUSAL_PREDICTION";
            }
            $scope.isClustering = function () {
                return $scope.endpoint && $scope.endpoint.type === "STD_CLUSTERING";
            }


            $scope.emptyTestQueryTemplate = function(index) {
                let qName = StringUtils.transmogrify(
                    "Query", $scope.endpoint.testQueries.map(tq => tq.name), (count, name) => `${name} #${count}`, 1, true
                );
                if ($scope.type === "predict") {
                    if ($scope.smBackendType === "DEEP_HUB") {
                        return { q: { features: { input: "replace by image file encoded as base64" }}, name: qName, $queryIndex: index};
                    } else if ($scope.isTimeseriesForecast()) {
                        return { q: { items : [{features: {}}] }, name: qName, $queryIndex: index};
                    } else {
                        return enrichTestQueryWithExplanations({ q: { features: {} }, name: qName, $queryIndex: index});
                    }
                }
                else if ($scope.type == "prompt") {
                    const inputs = {}
                    PromptUtils.getInputs($scope.endpoint.prompt).forEach(input => inputs[input.name] = "Replace value here");
                    return {q: { inputs }, name: qName, $queryIndex: index};
                } else if ($scope.type === "lookup") {
                    return { q: { data: {} }, name: qName, $queryIndex: index};
                } else {
                    return { q: { paramNameToReplace: "paramValueToReplace" }, name: qName, $queryIndex: index};
                }
            };

            $scope.getHeight =  function(q) { return Object.keys(q.features).length + 4; };
            $scope.uiState = {
                requestType: "EMPTY",
                queriesBatchSize: 1,
                inputDatasetSmartName: null
            }
            if ($scope.endpoint.testQueries.length > 0) {
                $scope.uiState.testQueryIndex = 0;
            }

            $scope.createNewQueries = function() {
                if ($scope.type === "predict" && $scope.smBackendType !== "DEEP_HUB") {
                    $scope.showAddQueriesModal();
                } else {
                    $scope.endpoint.testQueries.push($scope.emptyTestQueryTemplate($scope.endpoint.testQueries.length));
                    $scope.uiState.testQueryIndex  = $scope.endpoint.testQueries.length - 1;
                }
            };
            $scope.addQueries = function(requestType, queriesBatchSize, inputDatasetSmartName, shouldIncludeNulls) {
                const newIndex = $scope.endpoint.testQueries.length;
                if (requestType === "EMPTY") {
                    for (let i = 0; i < queriesBatchSize; i++) {
                        $scope.endpoint.testQueries.push($scope.emptyTestQueryTemplate(newIndex+i));
                    }
                } else if (requestType === "DATASET") {
                    let testQueriesName = []
                    if ($scope.endpoint.testQueries.length > 0) {
                        testQueriesName = $scope.endpoint.testQueries.map(tq => tq.name)
                    }
                    DataikuAPI.lambda.services.getSampleQueriesFromDataset($stateParams.projectKey,
                                                                           testQueriesName,
                                                                           inputDatasetSmartName,
                                                                           $scope.endpoint.modelRef,
                                                                           queriesBatchSize,
                                                                           "HEAD_SEQUENTIAL",
                                                                           shouldIncludeNulls).success(function(data) {
                        data.forEach(enrichTestQueryWithExplanations);
                        data.map((tq, i) => tq.$queryIndex = newIndex+i);
                        $scope.endpoint.testQueries.push.apply($scope.endpoint.testQueries, data);
                    }).error(setErrorInScope.bind($scope));
                } else {
                    setErrorInScope.bind($scope);
                }
                $scope.uiState.testQueryIndex = newIndex;
            }

            $scope.deleteTestQuery = function(index) {
                $scope.endpoint.testQueries.splice(index, 1);
                if ($scope.testQueriesResult && $scope.testQueriesResult.responses) {
                    $scope.testQueriesResult.responses.splice(index, 1);
                }
                if (index < $scope.uiState.testQueryIndex) {
                    $scope.uiState.testQueryIndex--;
                } else if (index === $scope.uiState.testQueryIndex) {
                    $scope.uiState.testQueryIndex = -1;
                }
                $scope.endpoint.testQueries.map((tq, i) => tq.$queryIndex = i);
            };

            $scope.duplicateTestQuery = function(index) {
                if (index < 0 || index >= $scope.endpoint.testQueries.length) return;
                const copied = angular.copy($scope.endpoint.testQueries[index]);
                if (copied.name) {
                    copied.name = 'Copy of ' + copied.name;
                }
                const newIndex = $scope.endpoint.testQueries.length;
                copied.$queryIndex = newIndex;
                $scope.uiState.testQueryIndex = newIndex;
                $scope.endpoint.testQueries.push(copied);
            };

            $scope.showTestQuery = function(index) {
                $scope.uiState.testQueryIndex = index;
            };

            $scope.showAddQueriesModal = function() {
                $scope.uiState.shouldIncludeNulls = false;
                CreateModalFromTemplate("/templates/lambda/add-queries-modal.html", $scope)
            };

            $scope.getCollectedColumnMappings = function() {
                const mappings = {};
                $scope.endpoint.lookups.forEach(function(lookup) {
                    angular.forEach(lookup.columnsMapping, function(v,k) {
                        mappings[k] = v;
                    });
                });
                return mappings;
            };

            function enrichTestQueryWithExplanations(query) {
                if ($scope.endpoint.outputExplanations) {
                    const tq = {
                        enabled: true,
                        method: $scope.endpoint.individualExplanationParams.method,
                        nExplanations: $scope.endpoint.individualExplanationParams.nbExplanations,
                    };
                    if (tq.method === "SHAPLEY") {
                        tq.nMonteCarloSteps = $scope.endpoint.individualExplanationParams.shapleyBackgroundSize;
                    }
                    query.q.explanations = tq;
                }
                return query;
            }
        }
    };
});


app.directive('authMethodSettings', function($stateParams, CreateModalFromTemplate) {
      return {
           restrict : 'A',
           templateUrl : '/templates/lambda/lambda-auth-method.html',
           replace : true,
           scope : {
                   authMethod : '=',
                   apiKeys : '=',
                   oauth2Config : '=',
                   userLogin : '=',
                   editMode : '=',
           },
           link : function($scope, element, attrs) {

               function editApiKey(isNew, apiKey) {
                    return CreateModalFromTemplate("/templates/lambda/api-key-modal.html", $scope, null, function(modalScope) {
                        modalScope.apiKey = apiKey;
                    });
                };

               $scope.exampleJwkSet = JSON.stringify({
                                        "keys": [
                                          {
                                            "kty": "RSA",
                                            "e": "AQAB",
                                            "use": "sig",
                                            "kid": "CXup",
                                            "n": "hrwD-lc-IwzwidCANmy4qsiZk11yp9kHykOuP0yOnwi36VomYTQVEzZXgh2sDJpGgAutdQudgwLoV8tVSsTG9SQHgJjH9Pd_9V4Ab6PANyZNG6DSeiq1QfiFlEP6Obt0JbRB3W7X2vkxOVaNoWrYskZodxU2V0ogeVL_LkcCGAyNu2jdx3j0DjJatNVk7ystNxb9RfHhJGgpiIkO5S3QiSIVhbBKaJHcZHPF1vq9g0JMGuUCI-OTSVg6XBkTLEGw1C_R73WD_oVEBfdXbXnLukoLHBS11p3OxU7f4rfxA_f_72_UwmWGJnsqS3iahbms3FkvqoL9x_Vj3GhuJSf97Q"
                                          },
                                          {
                                            "kty": "EC",
                                            "use": "sig",
                                            "crv": "P-256",
                                            "kid": "yGvt",
                                            "x": "pvgdqM3RCshljmuCF1D2Ez1w5ei5k7-bpimWLPNeEHI",
                                            "y": "JSmUhbUTqiFclVLEdw6dz038F7Whw4URobjXbAReDuM"
                                          }
                                        ]
                                      }, null, 2)
               $scope.editAPIKey = editApiKey.bind(null, false);
               $scope.newAPIKey = function() {
                   const apiKey = {
                       createdOn: Date.now(),
                       createdBy: $scope.userLogin
                   };
                   editApiKey(true, apiKey).then(function(k) {
                       $scope.apiKeys.push(apiKey);
                   });
               };
           }
       };
});

app.filter('endpointTypeToName', function(ENDPOINT_TYPES) {
    return function(type) {
        if (!type) {
            return;
        }
        return ENDPOINT_TYPES[type] || type;
    };
});


app.filter('endpointTypeToIcon', function(ENDPOINT_TYPES) {
    return function(type) { // @TODO lambda: improve endpoint icons
        if (!type) {
            return;
        } else if (['STD_PREDICTION', 'STD_FORECAST', 'STD_CAUSAL_PREDICTION', 'CUSTOM_PREDICTION', 'CUSTOM_R_PREDICTION'].includes(type)) {
            return 'dku-icon-machine-learning-regression';
        } else if (type == 'STD_CLUSTERING') {
            return 'dku-icon-machine-learning-clustering';
        } else if (type == 'DATASETS_LOOKUP') {
            return 'dku-icon-dataset';
        } else if (type == 'SQL_QUERY') {
            return 'dku-icon-sql';
        } else if (type == 'PY_FUNCTION') {
            return 'dku-icon-python';
        } else if (type == 'R_FUNCTION') {
            return 'dku-icon-r';
        } else {
            return;
        }
    };
});
})();
