/* eslint-disable no-console */
/* global _wt1Q */
(function() {
'use strict';

const app = angular.module('dataiku.logger', []);


// This is an "instantiable" => it is NOT a singleton !
// DIContext contains some details related to the service who requested the logger instance.
app.instantiable('Logger', function(DIContext, LoggerProvider) {
    const fullname = getSimpleFullname(DIContext);
    return LoggerProvider.getLogger(fullname);
});


app.service('LoggerProvider',function($log, Notification, $injector) {
    const svc = this;

    function handleLog(type, namespace) {
        return function(...loggedObjects) {
            const timestamp = moment().utc().valueOf();
            const formattedDate = moment().format('HH:mm:ss.SSS');
            const formattedType = type.toUpperCase();

            // Print in console
            let prefix = '['+ formattedDate + ']';
            if(namespace) {
                prefix += ' [' + namespace + ']';
            }
            prefix += ' -';

            if (type == "debug" && loggedObjects && loggedObjects.length == 1) {
                $log.info("%c" + prefix + " " + loggedObjects[0], "color: #777");
            } else {
                const args = [prefix, ...loggedObjects];
                $log[type].apply($log, args);
            }

            if($injector.has('WebSocketService')) {
                if($injector.get("WebSocketService").isAvailable()) {
                    const stringifiedLoggedObjects = [];
                    // Websocket connection fails if the message is too big, so we truncate - #3821
                    // To work around the fact that might it be UTF8, we limit messages to 63K / 3
                    const MAX_LENGTH = 64000/3;
                    let remainingLen = MAX_LENGTH;

                    angular.forEach(loggedObjects,function(obj) {
                        let ret = '';
                        if(!obj) {
                            ret = ''+obj;
                        } else {
                            if(typeof obj == 'object') {
                                try {
                                    ret = JSON.stringify(obj);
                                } catch(e) {
                                    ret = ''+obj;
                                }
                            } else {
                                 ret = ''+obj;
                            }
                        }
                        if (ret.length > remainingLen) {
                            ret = "TRUNCATED: " + ret.substring(0, remainingLen);
                            console.warn("Truncated log message on Websocket (too long)"); /*@console*/ // NOSONAR: OK to use console.
                        }
                        remainingLen -= ret.length;
                        stringifiedLoggedObjects.push(ret)
                    });
                    try {
                        Notification.publishToBackend('log-event', {
                            messages: stringifiedLoggedObjects,
                            timestamp: timestamp,
                            type: formattedType,
                            namespace: namespace
                        });
                    } catch (e2) {
                        console.warn("Failed to send log event to backend", e2); /*@console*/ // NOSONAR: OK to use console.
                    }
                }
            }
        };
    };

    svc.getLogger = function(namespace) {
        return {
            log: handleLog("log", namespace),
            warn: handleLog("warn", namespace),
            debug: handleLog("debug", namespace),
            info: handleLog("info", namespace),
            error: handleLog("error", namespace)
        };
    };
});


/*
Instantialble assert:
    - throws an error if the condition is not met
    - prefixes the error message with the component
    - enriches the error with the component full name and adds a "js-assert" orign

This is useful in particular to easily track errors reported to WT1
*/
app.instantiable('Assert', function(DIContext, AssertProvider) {
    const namespace = getSimpleFullname(DIContext);
    return AssertProvider.getChecker(namespace);
});


app.service('AssertProvider',function() {
    const svc = this;

    function _fail(namespace, message) {
        const prefix = namespace ? `[${namespace}] ` : '';
        throw new Error('[Assert]' + prefix + message);
    }

    function fail(namespace) {
        return function(message) {
            _fail(namespace, message);
        };
    }

    function trueish(namespace) {
        return function(condition, message) {
            if (!condition) {
                _fail(namespace, message);
            }
        };
    }

    function inScope(namespace) {
        return function(scope, attribute) {
            if (!resolveValue(scope, attribute)) {
                _fail(namespace, attribute + ' is not in scope');
            }
        };
    }

    svc.getChecker = function(namespace) {
        return {
            trueish: trueish(namespace),
            inScope: inScope(namespace),
            fail: fail(namespace)
        };
    };
});


// Utils for prefixing based on the angular component
function getSimpleModuleName(DIContext) {
    let sname = DIContext.serviceName;
    if (sname) {
        sname = sname.replace("dataiku.", "d.")
            .replace(".services", ".s")
            .replace(".directive", ".dir")
            .replace(".controllers", ".ctrl")
            .replace(".recipes", ".r")
            .replace(".savedmodels", ".sm")
            .replace(".managedfolder", ".mf")
            ;
    }
    return sname;
}

function getSimpleObjectName(DIContext) {
    let oname = DIContext.objectName;
    if (oname) {
        oname = oname.replace("APIXHRService", "API")
            .replace("Controller", "Ctrl");
    }
    return oname;
}

function getSimpleFullname(DIContext) {
    const sname = getSimpleModuleName(DIContext);
    const oname = getSimpleObjectName(DIContext);
    const fullname = sname + (sname ? '.' : '') + oname;
    return fullname;
}


const EXPECTED_EXCEPTIONS=[
    "Possibly unhandled rejection: dismissed modal"
];

app.factory('$exceptionHandler', function(Logger, ErrorReporting) {
    return function(exception, cause) {
        /* Swallow "expected exceptions", such as the one we do not catch when dismissing a modal */
        if (EXPECTED_EXCEPTIONS.includes(exception)) {
            return;
        }
        /* A string was thrown, so at least, use it as a message */
        if (exception !== undefined && exception.message === undefined) {
            console.warn("Got weird exception", exception, printStackTrace()); /*@console*/ // NOSONAR: OK to use console.
            exception = {
                stack: "no stack - string thrown ?",
                message: exception
            }
        }

        /* Send to console and frontend log */
        if (exception === undefined) {
            Logger.error("Caught undefined exception", printStackTrace());
        } else {

            const typename = ({}).toString.call(exception);
            /* Firefox specific error */
            if (typename == "[object Exception]") {
                console.info("Changing Firefox exception", exception); /*@console*/ // NOSONAR: OK to use console.
                let newMessage = exception.message ? exception.message : "No message";
                newMessage += " - FF modified";
                if (exception.name) newMessage += " - Name=" + exception.name;
                if (exception.result) newMessage += " - result=" + exception.result;

                let newException = new Error(newMessage);
                newException.stack = exception.stack;
                exception = newException;
            }

            Logger.error("Caught exception: " + exception,
                "\nStack: ", exception.stack,
                '\nCaused by : ', cause,
                '\nMessage :', exception.message);
        }
        /* Send to WT1 */
        ErrorReporting.reportJSException(exception, cause);
    };
});


app.factory("ErrorReporting", function() {
    // Must not depend on rootScope else circular dependency

    function errorReportingEnabled() {
        return window.dkuAppConfig && window.dkuAppConfig.udrMode == 'DEFAULT' && window.dkuAppConfig.errorsReporting;
    }

    const svc = {
        // if you change this method, make sure it works with its usage in Angular (global-error-handler.ts)
        reportJSException: function(exception, cause) {
            if (window.devInstance) {
                console.info("Dev instance, not reporting JS error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            if (!errorReportingEnabled()) {
                console.info("Reporting is disabled, not reporting JS error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            console.info("Reporting JS exception", exception); /* @console */ // NOSONAR: OK to use console.

            try {
                const params = {
                    type: "js-error",
                    message: exception.message,
                    stack: exception.stack
                };
                _wt1Q.push(["trackEvent", params]);
            } catch (e) {
                console.info("WT1 failure", e); /*@console*/ // NOSONAR: OK to use console.
            }
        },
        // if you change this method, make sure it works with its usage in Angular (api-error.ts)
        reportBackendAPIError: function(apiError) {
            if (window.devInstance) {
                console.info("Dev instance, not reporting API error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            if (!errorReportingEnabled()) {
                console.info("Reporting is disabled, not reporting API error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            if (apiError.httpCode == 0) {
                console.info("HTTP status 0 (network error), not reporting it", apiError); /*@console */ // NOSONAR: OK to use console.
                return;
            }
            if (apiError.httpCode == 502) {
                console.info("HTTP Gateway Error, not reporting it", apiError); /*@console */ // NOSONAR: OK to use console.
                return;
            }

            /* Report to WT1 */
            try {
                const params = {
                    type: "api-error",
                    httpCode: apiError.httpCode,
                    errorType: apiError.errorType,
                    message: apiError.message,
                    stack: apiError.details
                }
                _wt1Q.push(["trackEvent", params]);
            } catch (e) {
                console.info("WT1 failure", e); /*@console*/ // NOSONAR: OK to use console.
            }
        },
        configure: function() {
            const appConfig = window.dkuAppConfig;
            if (!!appConfig.offlineFrontend) {
                return; // We're offline, can't report errors.
            }

            window.DKUErrorReporting = svc;
        }
    };
    return svc;
});

/*
WT1 is the service we use to send back usage statistics to Dataiku
*/
app.factory("WT1", function($rootScope, Logger) {
    let mode = "DEFAULT";
    let configured = false;
    let eventsBeforeConfiguration = [];

    window._wt1Q = [];

    // $stateParams keys to report in WT1, in the form of hashed objectId / objectType
    const dssObjectsToLog = {
        "analysisId": "analysis",
        "datasetName": "dataset",
        "recipeName": "recipe",
        "smId": "savedModel",
        "odbId":"odb",
        "dashboardId": "dashboard",
        "scenarioId": "scenario",
        "notebookId": "notebook",
        "modelComparisonId": "modelComparison",
        "mesId": "modelEvaluationStore",
        "webAppId": "webApp",
        "reportId": "report",
        "insightId": "insight",
        "serviceId": "apiService",
        "labelingTaskId": "labelingTask"
    };

    const svc = {
        configure: function() {
            const appConfig = $rootScope.appConfig;
            window.dkuUsageReportingUtils._configure(appConfig);

            if (window.devInstance) {
                mode = 'NO';
            } else if (appConfig.udrMode) {
                mode = appConfig.udrMode;
            }
            configured = true;
            if (mode != 'NO') {
                window.dkuUsageReportingUtils._loadWT1JS();
                /* Now that we are configured, replay the events that already happened */
                if (window.devInstance) {
                    console.debug("WT1 configured, replaying", eventsBeforeConfiguration.length, "events"); /*@console*/ // NOSONAR: OK to use console.
                }
                for (const eventBeforeConfiguration of eventsBeforeConfiguration) {
                    this.event(eventBeforeConfiguration.type, eventBeforeConfiguration.params);
                }
            }

            if (window.devInstance) {
                console.debug("WT1 dev mode startup complete", _wt1Q); /*@console*/ // NOSONAR: OK to use console.
            }

            // for non-angular stuff :/
            window.WT1SVC = this;
        },
        tryEvent: function(type, paramsGetter, errorMessage) {
            try {
                svc.event(type, paramsGetter());
            } catch(e) {
                Logger.error(typeof errorMessage === 'string' ? errorMessage : `Failed to report '${type}' event`, e);
            }
        },
        event: function(type, params) {
            if (angular.isUndefined(params)) {
                params = {};
            }

            if (!configured) {
                if (window.devInstance) {
                    console.debug("WT1: pre-configuration-event: " + type, params); /*@console*/ // NOSONAR: OK to use console.
                }
                // While it's not configured, we enqueue events, so that:
                // * the session and visitor params are set BEFORE we track the first state change event
                // * the proper privacy and minimal mode are taken into account
                eventsBeforeConfiguration.push({ "type": type, "params": params});
                return;
            }

            /* Add login, email and profile according to privacy rules */
            if ($rootScope && $rootScope.appConfig) {
                window.dkuUsageReportingUtils._addUserPropertiesToEventPayload($rootScope.appConfig, params);
            }

            /* Add and hash the project key, the objects id in the current state */
            if ($rootScope && $rootScope.$stateParams && $rootScope.$stateParams.projectKey) {
                params.projecth = md5($rootScope.$stateParams.projectKey);
                for (const objectKey in dssObjectsToLog) {
                    if (objectKey in $rootScope.$stateParams) {
                        const objectId = $rootScope.$stateParams[objectKey];
                        const objectType = dssObjectsToLog[objectKey];
                        params.objectType = objectType;
                        params[objectType + "h"] = md5($rootScope.$stateParams.projectKey + "." + objectId);
                        break;
                    }
                }
            }
            /* Add and hash the application id */
            if($rootScope && $rootScope.$stateParams && $rootScope.$stateParams.appId){
                params.objectType = 'App';
                params.applicationh = md5($rootScope.$stateParams.appId);
            }
            
            window.dkuUsageReportingUtils._trackEvent($rootScope.appConfig, mode, type, params);
        },
        setVisitorParam: function(key, value) {
            _wt1Q.push(["setVisitorParam", key, value]);
        },
        delVisitorParam: function(key) {
            _wt1Q.push(["delVisitorParam", key]);
        },
        setSessionParam: function(key, value) {
            _wt1Q.push(["setSessionParam", key, value]);
        },
        delSessionParam: function(key) {
            _wt1Q.push(["delSessionParam", key]);
        },
    };

    return svc;
});


app.directive("wt1ClickId", function(WT1) {
    return {
        restrict: 'A',
        link: function ($scope, element, attrs) {
            if (attrs.wt1ClickId && attrs.wt1ClickId !== ""){
                element.bind('click', function() {
                    WT1.event("clicked-item", {"item-id": attrs.wt1ClickId});
                });
            }
        }
    };
});
app.directive("wt1ClickEvent", function(WT1) {
    return {
        restrict: 'A',
        link: function ($scope, element, attrs) {
            element.bind('click', function() {
                WT1.event(attrs.wt1ClickEvent);
            });
        }
    };
});


app.factory("BackendReportsService", function($rootScope, $timeout, DataikuAPI, WT1, Logger, ErrorReporting) {
    $timeout(function() {
        if ($rootScope.appConfig && $rootScope.appConfig.loggedIn && $rootScope.appConfig.udrMode != 'NO' && !window.devInstance) {
            DataikuAPI.usage.popNextReport().success(function(data) {
                if (data.reportPublicId) {
                    Logger.info("sending report", data.reportType);
                    /* Something to send */
                    window.WT1SVC.event("v3-report", {
                        "reportType": data.reportType,
                        "reportId": data.reportPublicId,
                        "reportData": JSON.stringify(data.reportData)
                    });
                } else {
                    Logger.info("No report available");
                }
            });
        } else {
            Logger.debug("Reports reporting disabled")
        }

        if ($rootScope.appConfig && $rootScope.appConfig.loggedIn && $rootScope.appConfig.admin &&
                 $rootScope.appConfig.udrMode == 'DEFAULT' && $rootScope.appConfig.errorsReporting) {
            DataikuAPI.usage.popReflectedEvents().success(function(data) {
                data.events.forEach(function(evt) {
                    Logger.info("Reflecting event", evt);
                    window.WT1SVC.event("reflected-event", {
                        "reflectedEventData": JSON.stringify(evt)
                    });
                });
            });
        } else {
            Logger.debug("Reflected events reporting disabled")
        }

    }, 10000);
    return {
        //TODO this is actually not a service, it has no functionality...
    };
});

})();



/// =================== Global ===================

/* eslint-disable no-unused-vars, no-redeclare */
window.WT1SVC = {
    event: function() {} // temporary before load
};

window.DKUErrorReporting = {
    reportBackendAPIError: function() {} // temporary before load
};

function setErrorInScope(data, status, headers, config, statusText, xhrStatus) {
    // Explicitly ignore JS errors by re-throwing
    // Allow using the function in catch() blocks like: .catch(setErrorInScope.bind($scope)) while having a clear stack trace
    // Whereas using .error(setErrorInScope.bind($scope)) expects `data` to be an http response
    if (data instanceof Error) {
        throw data;
    }

    if (status === undefined && headers === undefined) {
        status = data.status;
        headers = data.headers;
        statusText = statusText || data.statusText;
        xhrStatus = xhrStatus || data.xhrStatus;
        data = data.data;
    }
    /* Put in bound scope */
    this.fatalAPIError = getErrorDetails(data, status, headers, statusText);
    this.fatalAPIError.html = getErrorHTMLFromDetails(this.fatalAPIError);

    /* Report to WT1 */
    window.APIErrorLogger.error("API error", this.fatalAPIError);
    window.DKUErrorReporting.reportBackendAPIError(this.fatalAPIError);
}

/**
 * Variant of setErrorInScope that takes as input an already parsed error.
 * Designed to be able to set in scope errors coming from an Angular context (where the http service internally parses the error already)
 * @param errorDetails result from getErrorDetails or similar
 */
function setErrorDetailsInScope(errorDetails) {
    /* Put in bound scope */
    this.fatalAPIError = errorDetails;
    this.fatalAPIError.html = getErrorHTMLFromDetails(this.fatalAPIError);

    /* Report to WT1 */
    window.APIErrorLogger.error("API error", this.fatalAPIError);
    window.DKUErrorReporting.reportBackendAPIError(this.fatalAPIError);
}


function setErrorInScope2(payload) {
    setErrorInScope.call(this, JSON.parse(payload.response || '{}'), payload.status, h => payload.getResponseHeader(h));
}

function resetErrorInScope(scope) {
    if (scope.fatalAPIError) {
        scope.fatalAPIError.httpCode = null;
        scope.fatalAPIError.errorType = null;
    }
}

/* API error struct : {code, httpCode, message, details} */
function getErrorDetails(data, status, headers, statusText) {
    /* Network / HTTP error */
    if (data == null && status == -1) {
        return {
            httpCode: status,
            message: "Network error: " + (statusText === undefined ? "" : statusText),
            errorType: "XHRNetworkError"
        };
    }
    if (status == 413) {
        return {
            httpCode: status,
            message: data && data.message || 'No message',
            details: data && data.details || 'No details',
            errorType: "HTTPError413"
        };
    }

    if (data && data.$customMessage) {
        return {
            httpCode: status,
            code: 0,
            message: data.message || "Unknown error",
            details: data.details,
            errorType: data.errorType || "unknown"
        };
    }

    const ctype = headers("Content-type");
    if (ctype && ctype.startsWith("application/json") && data && data.errorType) {
        const apiError = data;
        apiError.httpCode = status;
        return apiError;
    } else {
        let errorType = "unknown";
        if (status == 502) {
            errorType = "Gateway error";
        }
        return {
            httpCode: status,
            code: 0,
            message: 'Unknown error',
            details: data && data.details || 'No details',
            errorType: errorType
        };
    }
}

function getErrorHTMLFromDetails(apiError) {
    let html = "<strong>Error " + apiError.code +": "+apiError.message +"</strong>";
    if (apiError.details) {
        html += "<pre>" + apiError.details +"</pre>";
    }
    return html;
}

function getErrorHTML(data, status, headers, statusText) {
    const apiError = getErrorDetails(data, status, headers, statusText);
    return getErrorHTMLFromDetails(apiError);
}

/* eslint-enable no-unused-vars, no-redeclare */
