(function() {
'use strict';

const app = angular.module('dataiku.controllers');


app.controller('ProjectBaseController', function($scope, $controller, $state, $stateParams, $timeout, $rootScope, $q, $filter,
            Assert, DataikuAPI, WT1, TopNav, Breadcrumb, CreateModalFromTemplate, TaggingService, FlowGraphSelection,
            GlobalProjectActions, Notification, HistoryService, FutureWatcher, ProgressStackMessageBuilder, ProjectActionsService,
            AlationCatalogChooserService, WebAppsService, DatasetUtils, FlowGraph, TaggableObjectsService, ComputableSchemaRecipeSave,
            TypeMappingService, TYPE_MAPPING, Logger, Ng1SamplesService) {
    $controller('_CreateRecipesBehavior', {$scope: $scope});

    $scope.standardizedSidePanel = {};
    $scope.noAccessPermission = false;
    $scope.standardizedSidePanel.slidePanel = function() {
        $scope.standardizedSidePanel.opened = !$scope.standardizedSidePanel.opened;
    }
    $scope.standardizedSidePanel.toggleTab = function(tabName) {
        $scope.standardizedSidePanel.tabToToggle = '';
        $timeout(() => { $scope.standardizedSidePanel.tabToToggle = tabName; }, 50);
    }

    $scope.$on('$stateChangeStart', function (event, toState, toParams, fromState) {
        if (fromState.name === 'projects.project.flow') {
            toParams.fromFlow = true;
        }
    });

    const getProjectAccessInfo = () => {
        DataikuAPI.projects.getProjectAccessInfo($stateParams.projectKey).success(function(accessInfo) {
            delete $rootScope.projectSummary;
            delete $scope.projectSummary;
            $scope.accessInfo = accessInfo;
            if (accessInfo.hasAnyProjectAccess) {
                $scope.refreshProjectData();
                updateTagList("page-reload");
            }
            $scope.noAccessPermission = false;
        }).error((data, status, headers) => {
            if (data && data.errorType) { // to check that the error is from the api call and not for example a proxy error
                $scope.noAccessPermission = status == 403;
            }
            setErrorInScope.bind($scope)(data, status, headers);
        });
    };
    getProjectAccessInfo();

    $scope.refreshProjectData = function() {
        DataikuAPI.projects.getSummary($stateParams.projectKey).success(function(data) {
            $scope.projectCurrentBranch = data.projectCurrentBranch;
            $scope.projectSummary = data.object;
            $rootScope.projectSummary = $scope.projectSummary;
            $scope.objectInterest = data.interest;
            $scope.objectTimeline = data.timeline;
            $scope.projectSummaryStatus = data.objectsCounts;

            TopNav.setProjectData($scope.projectSummary, $scope.projectCurrentBranch);
            $scope.topNav.isProjectAnalystRO = $scope.isProjectAnalystRO();
            $scope.topNav.isProjectAnalystRW = $scope.isProjectAnalystRW();
            $scope.topNav.isCurrentProjectAdmin = $scope.isProjectAdmin();
            $scope.topNav.canReadDashboards = $scope.canReadDashboards();
            $scope.topNav.canAccessProjectSettings = $scope.isProjectAdmin();
            $scope.topNav.canAccessProjectSecurity = $scope.canAccessProjectSecurity();
            $scope.topNav.isAppInstance = $scope.isAppInstance();
            $scope.topNav.showFlowNavLink = $scope.showFlowNavLink();
            $scope.topNav.showGenAiNavLink = $scope.showGenAiNavLink();
            $scope.topNav.showCodeNavLink = $scope.showCodeNavLink();
            $scope.topNav.showLabNavLink = $scope.showLabNavLink();
            $scope.topNav.showVersionControlFeatures = $scope.showVersionControlFeatures();

            if ($scope.projectSummary.tutorialProject) {
                WT1.setVisitorParam("tutorial-project", "true");
                WT1.setVisitorParam("tutorial-id", $scope.projectSummary.tutorialId);
                WT1.event("tutorial-project-open");
            }

            // number app tiles, if any
            if ($scope.projectSummary.appManifest && $scope.projectSummary.appManifest.homepageSections) {
                $scope.projectSummary.appManifest.homepageSections.forEach(function(section, sectionIdx) {
                    if (section.tiles) {
                        section.tiles.forEach(function(tile, tileIdx) {
                            tile.$sectionIdx = sectionIdx;
                            tile.$tileIdx = tileIdx;
                        });
                    }
                });
            }

            $scope.isObjectSharingRequestEnabled = $rootScope.appConfig.objectSharingRequestsMode === 'ALL_ENABLED' ||
                                            ($rootScope.appConfig.objectSharingRequestsMode !== 'ALL_DISABLED' && $scope.projectSummary.sharingRequestsEnabled);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.$on("$destroy", function(){
        WT1.delVisitorParam("tutorial-project");
        WT1.delVisitorParam("tutorial-id");
        $scope.topNav.isCurrentProjectAdmin = false;
    });

    function profileMayReadProjectContent() {
        return $rootScope.appConfig && $rootScope.appConfig.userProfile && $rootScope.appConfig.userProfile.mayReadProjectContent;
    }

    // The Pinboard controller puts stuff here for use by the pinboard page top level nav bar.
    $scope.dashboardContext = {};

    $scope.isProjectAdmin = function() {
        return profileMayReadProjectContent() && $scope.projectSummary != null && $scope.projectSummary.isProjectAdmin;
    };

    $scope.canManageBundles = function() {
        return ($rootScope.appConfig && $scope.topNav
            && ($rootScope.appConfig.requireProjectAdminPermissionToExportAndBundleProjects && $scope.topNav.isCurrentProjectAdmin)
                || (!$rootScope.appConfig.requireProjectAdminPermissionToExportAndBundleProjects && $scope.topNav.isProjectAnalystRW));
    };

    $scope.canWriteProject = function() {
        // Alias
        return $scope.isProjectAnalystRW();
    };
    $scope.isProjectAnalystRW = function() {
        return profileMayReadProjectContent() && $scope.projectSummary != null && $scope.projectSummary.canWriteProjectContent;
    };
    // To be in sync with PagesSettingsCatalogPermissionService:canAccessProjectSecurity
    $scope.canAccessProjectSecurity = function() {
        return $scope.projectSummary != null &&
            ($scope.projectSummary.canManageDashboardAuthorizations ||
             $scope.projectSummary.canManageExposedElements ||
             $scope.projectSummary.canManageAdditionalDashboardUsers ||
             $scope.projectSummary.canEditPermissions);
    };
    $scope.isProjectAnalystRO = function() {
        return profileMayReadProjectContent() && $scope.projectSummary != null && $scope.projectSummary.canReadProjectContent;
    };
    $scope.canModerateDashboards = function() {
        return $scope.projectSummary != null && $scope.projectSummary.canModerateDashboards;
    };
    $scope.canWriteDashboards = function() {
        return $scope.projectSummary != null && $scope.projectSummary.canWriteDashboards;
    };
    $scope.canReadDashboards = function() {
        return profileMayReadProjectContent() && $scope.projectSummary != null && $scope.projectSummary.canReadDashboards;
    };
    $scope.canRunScenarios = function() {
        return $scope.projectSummary != null && $scope.projectSummary.canRunScenarios;
    };
    $scope.canManageExposedElements = function() {
        return $scope.projectSummary != null && $scope.projectSummary.canManageExposedElements;
    };
    $scope.canShareToWorkspaces = function() {
        return $scope.mayShareToWorkspaces() && $scope.projectSummary != null && $scope.projectSummary.canShareToWorkspaces;
    };
    $scope.mayShareToWorkspaces = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn && ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayShareToWorkspaces);
    };
    $scope.canExportDatasetsData = function() {
        return $scope.projectSummary != null && $scope.projectSummary.canExportDatasetsData;
    };
    $scope.canExecuteApp = function() {
        return $scope.projectSummary != null && $scope.projectSummary.canExecuteApp;
    };
    $scope.isAppInstance = function() {
        return $scope.projectSummary != null && $scope.projectSummary.projectAppType == 'APP_INSTANCE';
    };
    $scope.showFlowNavLink = function() {
        return !$scope.isAppInstance() || $scope.projectSummary.appManifest.instanceFeatures.showFlowNavLink;
    };
    $scope.showGenAiNavLink = function() {
        return !$scope.isAppInstance() || $scope.projectSummary.appManifest.instanceFeatures.showGenAiNavLink;
    };
    $scope.showCodeNavLink = function() {
        return !$scope.isAppInstance() || $scope.projectSummary.appManifest.instanceFeatures.showCodeNavLink;
    };
    $scope.showLabNavLink = function() {
        return !$scope.isAppInstance() || $scope.projectSummary.appManifest.instanceFeatures.showLabNavLink;
    };
    $scope.showVersionControlFeatures = function() {
        return !$scope.isAppInstance() || $scope.projectSummary.appManifest.instanceFeatures.showVersionControlFeatures;
    };


    /**
     * Promise indicating if node allows delete and reconnect
     * @param {*} nodeObject from flow possibly representing recipe which may allow delete and reconnect
     * @returns a promise resolving to true or false (or rejecting in case of error) 
     */
    $scope.canDeleteAndReconnectObject = function(nodeObject) {
        var deferred = $q.defer();

        // Some utility functions as it's a bit complicated

        function isRecipe(objectId) {
            if (!objectId) return false; 
            const node = FlowGraph.node(objectId);
            if (!node) return false;
            return node.nodeType === 'RECIPE';
        }

        function toDatasetNode(objectId) {
            if (!objectId) return undefined; 
            const node = FlowGraph.node(objectId);
            if (!node) return undefined;
            const nodeType = node.nodeType;
            if (nodeType === 'LOCAL_DATASET' || nodeType === 'FOREIGN_DATASET') {
                return node;
            }
        }

        function isPartitioned(dsNode) {
            return dsNode.partitioned;
        } 

        function negate(predicate) {
            return (dsNode) => !predicate(dsNode);
        }

        function allSuccessorsAreRecipes(dsNode) {
            return !dsNode.successors || dsNode.successors.every(successor => isRecipe(successor))
        }

        function isEligibleRecipe(recipeType) {
            // generate statistics recipes -> completely different schema, could not possibly want to reconnect
            // eval recipes -> makes no sense to reconnect
            const excluded_categories = [TYPE_MAPPING.RECIPE_CATEGORIES.EVALUATION, TYPE_MAPPING.RECIPE_CATEGORIES.STANDALONE, TYPE_MAPPING.RECIPE_CATEGORIES.EDA];
            const excluded_types = ['nlp_llm_evaluation'];
            return !excluded_categories.includes(TypeMappingService.mapRecipeTypeToCategory(recipeType)) && !excluded_types.includes(recipeType);
        }

        function hasSuccessors(dsNode) {
            return dsNode && dsNode.successors && dsNode.successors.length;
        }

        /**
         * Checks 
         * - predecessor dataset and successor dataset types match
         * - no datasets will be rewired as inputs to the same recipe after the snip (deleteAndReconnect) operation 
         * - the conditions on the successor datasets passed in as predicates
         *  Returns true if the conditions are met
         */
        function meetsSuccessorDatasetConditions(recipeNodeObject, predecessorDsNode, ...dsNodePredicates) {
            const successorRecipesOfPredecessor = predecessorDsNode.successors.filter(isRecipe);
            // We have to ensure both the predecessor dataset at the moment and the successor datasets that 
            // will be deleted do not share any of the same output recipes
            // So we collect the successors so far to check for collisions
            var successorRecipesSoFar = new Set(successorRecipesOfPredecessor); 
            for (const successorNodeId of recipeNodeObject.successors) {
                const dsNode = toDatasetNode(successorNodeId);
                if (!dsNode) {
                    return false;
                }

                if (!(predecessorDsNode.datasetType && dsNode.datasetType && predecessorDsNode.datasetType === dsNode.datasetType)) {
                    return false;
                }

                const successorRecipes = new Set(dsNode.successors);
                if (!successorRecipesSoFar.isDisjointFrom(successorRecipes)) {
                    return false;
                }
                successorRecipesSoFar = successorRecipesSoFar.union(successorRecipes);

                // All the other conditions - ignore nulls
                if (!dsNodePredicates.every(p => !p || p(dsNode))) {
                    return false;
                }
            }
            return true;
        }

        // Returns null if it not a DS with partitioning, otherwise returns a promise that resolves if the successor node
        // has the same partition schema as predecessorPartitionSchema, rejects if it does not match
        function successorNodePartitioningMatchPromise(successorNodeName, predecessorPartitionSchema) {
            const successorDsNode = toDatasetNode(successorNodeName);
            if (!predecessorPartitionSchema || !predecessorPartitionSchema.dimensions || !successorDsNode
                || !successorDsNode.partitioned || !allSuccessorsAreRecipes(successorDsNode)) {
                return null;
            }
            const successorDeferred = $q.defer();
            DataikuAPI.datasets.getFullInfo(nodeObject.projectKey, successorDsNode.projectKey, successorDsNode.name)
                .then(function (resp) {
                    const successorDataset = resp.data;
                    if (successorDataset.dataset && successorDataset.dataset.partitioning && 
                            angular.equals(successorDataset.dataset.partitioning.dimensions, predecessorPartitionSchema.dimensions)) {
                        successorDeferred.resolve();
                    } else {
                        successorDeferred.reject();
                    }
                }).catch(function(error) {
                    successorDeferred.reject(error);
                });
            return successorDeferred.promise;
        }

        function checkPartitionSchemasMatch(recipeNodeObject, predecessorDsNode, deferred) {
            // Check successor conditions when there is partitioning - first we need the predecessor partition schema
            DataikuAPI.datasets.getFullInfo(recipeNodeObject.projectKey, predecessorDsNode.projectKey, predecessorDsNode.name)
                .then(function(resp) {
                    const predecessorDataset = resp.data;
                    if (!predecessorDataset || !predecessorDataset.dataset) {
                        deferred.resolve(false);
                        return;
                    }
                    const predecessorPartitionSchema = predecessorDataset.dataset.partitioning;
                    const partitioningMatchPromises = []; 
                    for (const successorNodeName of recipeNodeObject.successors) {
                        const partitioningMatchPromise = successorNodePartitioningMatchPromise(successorNodeName, predecessorPartitionSchema);
                        if (partitioningMatchPromise === null) {
                            // Bail out early as the successor node not a DS or not partitioned (avoid unneeded API calls)
                            deferred.resolve(false);
                            return;
                        } else {
                            partitioningMatchPromises.push(partitioningMatchPromise);
                        }
                    }
                    $q.all(partitioningMatchPromises)
                        .then(() => deferred.resolve(true))
                        .catch((error) => error ? deferred.reject(error) : deferred.resolve(false));
            }).catch(function(error) {
                deferred.reject(error);
            });
        }

        // Main logic
        const mainConditionsMet = nodeObject && nodeObject.nodeType == 'RECIPE'
            && isEligibleRecipe(nodeObject.recipeType)
            && nodeObject.predecessors && nodeObject.predecessors.length === 1
            && nodeObject.successors && nodeObject.successors.length;

        if (!mainConditionsMet) {
            deferred.resolve(false);
            return deferred.promise;
        }

        // The other conditions depend on the predecessor node - if it is partitioned there's different logic
        const predecessorDsNode = toDatasetNode(nodeObject.predecessors[0]);
        if (!predecessorDsNode) {  
            //If it's not a dataset
            deferred.resolve(false);
            return deferred.promise;
        }

        // Successor dataset conditions
        // Checks predecessor DS and successor DS datasetType matches
        // Ensure no datasets will be rewired as inputs to the same recipe after the snip (deleteAndReconnect) operation
        // (e.g. if there is a split recipe and then the outputs are fed into the same recipe)
        // Plus some simple conditions
        if (!meetsSuccessorDatasetConditions(nodeObject, predecessorDsNode, 
                allSuccessorsAreRecipes, hasSuccessors, predecessorDsNode.partitioned ? isPartitioned : negate(isPartitioned) )) {
            deferred.resolve(false);
            return deferred.promise;
        }
        
        if (predecessorDsNode.partitioned) {
            // Async call is needed to check if all the partitions match - will resolve `deferred` as appropriate
            checkPartitionSchemasMatch(nodeObject, predecessorDsNode, deferred)
        } else {
            deferred.resolve(true);
        }
        
        return deferred.promise;
    }

    /**
     * Show dialog for delete and reconnect style deletion, checking the schemas for consistency afterwards
     * @param {*} recipeObject 
     */
    $scope.doDeleteAndReconnectRecipe = function(recipeObject, wt1_event_name) {
        // Build array of deletionRequests for operation with deleteAndReconnect true and essentially treat as a std deletion event
        const deletionRequests = [];

        // We also need to gather up the new successor recipes that will have their inputs rewired, as we will have to trigger a schema update on each of them. 
        // Code recipes are excluded from this as it doesn't work for them
        const newSuccessorRecipesToCheckSchema = [];

        // Add a delete for the selected recipe object
        deletionRequests.push({type: recipeObject.nodeType, projectKey: recipeObject.projectKey , id: recipeObject.name, options : { deleteAndReconnect : true}} );

        // Add deletes for successor datasets and gather new successors info
        for (const successor of recipeObject.successors) {
            const successorNode = FlowGraph.node(successor);
            if (successorNode.nodeType === 'LOCAL_DATASET') { //foreign DS not possible here
                const dr = {type: 'DATASET', projectKey: successorNode.projectKey , id: successorNode.name, options : { deleteAndReconnect : true}};
                deletionRequests.push(dr);

                //When the successor datasets are deleted, the new successors will be the recipes after that
                for (const newSuccessor of successorNode.successors ) {
                    const newSuccessorNode = FlowGraph.node(newSuccessor);
                    if ('RECIPE' === newSuccessorNode.nodeType 
                            && TypeMappingService.mapRecipeTypeToCategory(newSuccessorNode.recipeType) !== TYPE_MAPPING.RECIPE_CATEGORIES.CODE) 
                    {
                        newSuccessorRecipesToCheckSchema.push(newSuccessorNode);
                    }
                }
            }
        }
        
        //Fallback event name - just a precaution
        if (!wt1_event_name) { 
            wt1_event_name = 'flow-context-menu-snipandreconnect'; 
        }
 
        TaggableObjectsService.delete(deletionRequests)
            .then(() => {
                FlowGraphSelection.clearSelection();
                WT1.tryEvent(wt1_event_name, () => { return { recipe: recipeObject.recipeType } });
                newSuccessorRecipesToCheckSchema.forEach(newSuccessorRecipe =>
                    //This will prompt an update if any schema change detected for input into the new downstream recipes
                    ComputableSchemaRecipeSave.handleSchemaUpdateForDeleteAndReconnect($scope, newSuccessorRecipe.projectKey, newSuccessorRecipe.name));
            });
    }

    $scope.newManagedDataset = function() {
        CreateModalFromTemplate("/templates/flow-editor/new-managed-dataset.html",
            $scope, "NewManagedDatasetController");
    };
    $scope.newManagedFolder = function() {
        CreateModalFromTemplate("/templates/managedfolder/new-box-modal.html", $scope);
    };
    $scope.newModelEvaluationStore = function() {
        CreateModalFromTemplate("/templates/modelevaluationstores/new-model-evaluation-store-modal.html", $scope);
    };
    $scope.newSample = function() {
        Ng1SamplesService.openSampleModal($scope.projectSummary.projectKey);
    };

    $scope.importFromAlation = function(){
        AlationCatalogChooserService.openChooser();
    }

    $scope.displayPluginInfo = function(pluginId, showDatasets, showRecipes) {
        var newScope = $scope.$new();
        newScope.showDatasets = showDatasets;
        newScope.showRecipes = showRecipes;
        newScope.pluginDesc = $scope.appConfig.loadedPlugins.filter(function(x){
            return x.id == pluginId;
        })[0];
        CreateModalFromTemplate("/templates/plugins/modals/plugin-learn-more.html", newScope);
    };

    $scope.getRelevantZoneId = function(zoneId) {
        $scope.relevantZoneId = zoneId;
        let selectedItems = FlowGraphSelection.getSelectedNodes();
        if (selectedItems.length == 1 && selectedItems[0].nodeType == "ZONE") {
            $scope.relevantZoneId = selectedItems[0].cleanId;
        } else if (selectedItems.length == 1) {
            $scope.relevantZoneId = selectedItems[0].ownerZone;
        }
        return $scope.relevantZoneId;
    }

    $scope.getDatasetWithStatus = function(datasetName, target) {
        DataikuAPI.datasets.getWithMetricsStatus($stateParams.projectKey, datasetName).success(function(data){
            target.dataset = data.dataset;
            target.datasetShortStatus = data.shortStatus;
        }).error(function(){
            // if an error occurs consider the dataset was removed. TODO discard also relatedItems? (analyses)
            HistoryService.notifyRemoved({ type: "DATASET", id: datasetName });
            setErrorInScope.apply('$id' in target ? target : $scope, arguments);
        });
    };
    $scope.showLabModal = function(datasetSmartName, datasetFullInfo) {
        CreateModalFromTemplate("/templates/datasets/lab-modal.html", $scope, null, function(newScope) {
            newScope.datasetFullInfo = datasetFullInfo;
            newScope.datasetSmartName = datasetSmartName;
        });
    };

    $scope.showSparkNotLicensedModal = function() {
        CreateModalFromTemplate("/templates/widgets/spark-not-licensed-modal.html", $scope);
    };
    $scope.showCERestrictionModal = function(feature) {
        CreateModalFromTemplate("/templates/profile/community-vs-enterprise-modal.html",
            $scope, null, function(newScope){ newScope.lockedFeature = feature; });
    };

    $rootScope.showCERestrictionModal = $scope.showCERestrictionModal;
    $scope.$on("$destroy", function(){
        $rootScope.showCERestrictionModal = null;
    })

    $scope.GlobalProjectActions = GlobalProjectActions;

    /* ********************** Tags handling **************** */

    $scope.tagColor = TaggingService.getTagColor;
    $rootScope.activeProjectTagColor = TaggingService.getTagColor;
    $rootScope.activeGlobalTagsCategory = TaggingService.getGlobalTagCategory;

    $scope.projectTagsMap = {};
    var refreshTagMapRefs = function () {
        $scope.projectTagsMap = TaggingService.getProjectTags();
        $scope.projectTagsMapDirty = angular.copy($scope.projectTagsMap); // For settings, we want to be able to change and cancel/save then
    }
    TaggingService.fetchGlobalTags();

    function updateTagList(type) {
        TaggingService.update(type=="page-reload").success(refreshTagMapRefs);
    }

    function refreshForTagListChanged() {
        updateTagList();
        TaggingService.fetchGlobalTags(true);
        $rootScope.$broadcast('taggableObjectTagsChanged');
    }

    var deregister = Notification.registerEvent("tags-list-changed", refreshForTagListChanged);

    $scope.$on("projectDescriptionChanged", function(_, description) {
        $scope.projectSummary.description = description;
    });

    $scope.$on('projectTagsUpdated', refreshTagMapRefs);

    $scope.getTagsMap = function () {
        return $scope.projectTagsMap;
    }

    $scope.getAllProjectTags = function () {
        var deferred = $q.defer();
        if (!$scope.hasOwnProperty("allProjectLevelTags")) {
            $scope.allProjectLevelTags = [];
            DataikuAPI.projects.listAllTags()
            .success(function(data) {
                $scope.allProjectLevelTags = TaggingService.fillTagsMapFromArray(data);
                deferred.resolve($scope.allProjectLevelTags);
            })
            .error(() => {
                setErrorInScope.bind($scope);
                deferred.resolve($scope.allProjectLevelTags);
            });
        }
        else {
            deferred.resolve($scope.allProjectLevelTags);
        }
        return getRewrappedPromise(deferred);
    }

    $scope.getTags = function(global){
        var deferred = $q.defer();
        deferred.resolve($scope.projectTagsMap);
        return getRewrappedPromise(deferred);
    };

    $scope.$on("$destroy", function(){
        $rootScope.activeProjectTagColor = null;
        $rootScope.activeGlobalTagsCategory = null;
        deregister();
    });

    /* Macro roles mapping */

    $scope.macroRoles = {};
    $scope.webappRoles = {};

    const pluginsById = $rootScope.appConfig.loadedPlugins.reduce(function (map, obj) {
        map[obj.id] = obj;
        return map;
    }, {});

    $rootScope.appConfig.customRunnables.forEach(function(runnable) {
        if (!runnable.desc.macroRoles) return;

        const plugin = pluginsById[runnable.ownerPluginId];
        if (!plugin || plugin.hideComponents) return; // plugin might have been deleted or the plugin components visibility settings

        runnable.desc.macroRoles.forEach(function(macroRole) {
            $scope.macroRoles[macroRole.type] = $scope.macroRoles[macroRole.type] || [];

            $scope.macroRoles[macroRole.type].push({
                label: runnable.desc.meta.label || runnable.id,
                icon: runnable.desc.meta.icon || plugin.icon,
                roleTarget: macroRole.targetParamsKey || macroRole.targetParamsKeys,
                roleType: macroRole.type,
                applicableToForeign: macroRole.applicableToForeign,
                runnable: runnable
            });
        });
    });
    $scope.showCreateRunnable = function(runnable, targetKey, targetValue) {
        CreateModalFromTemplate('/templates/macros/runnable-modal.html', $scope, null, function(newScope) {
            newScope.runnable = runnable;
            newScope.targetKey = targetKey;
            newScope.targetValue = targetValue;
        });
    };


    $rootScope.appConfig.customWebApps.forEach(function(loadedWebApp) {
        if (!loadedWebApp.desc.roles) {
            return;
        }

        const plugin = pluginsById[loadedWebApp.ownerPluginId];
        if (!plugin || plugin.hideComponents) {
            return; // plugin might have been deleted or the plugin components visibility settings
        }

        loadedWebApp.desc.roles.forEach(function(role) {
            $scope.webappRoles[role.type] = $scope.webappRoles[role.type] || [];

            $scope.webappRoles[role.type].push({
                label: loadedWebApp.desc.meta.label || loadedWebApp.id,
                icon: loadedWebApp.desc.meta.icon || plugin.icon,
                roleTarget: role.targetParamsKey || role.targetParamsKeys,
                roleType: role.type,
                applicableToForeign: role.applicableToForeign,
                loadedWebApp: loadedWebApp
            });
        });
    });

    $scope.showCreateWebAppModal = function(webappCategory, loadedWebApp, targetKey, targetValue, defaultName) {

        if (webappCategory !== 'code' && webappCategory !== 'visual') {
           return;
        }

        let templateName;

        $scope.webappCategory = webappCategory;

        if (webappCategory === 'code') {
            templateName = '/templates/webapps/new-code-webapp-modal.html';
        } else {
            templateName = '/templates/webapps/new-visual-webapp-modal.html';
        }

        CreateModalFromTemplate(templateName, $scope, null, function(modalScope) {
            if (loadedWebApp) {
                modalScope.loadedWebApp = loadedWebApp;
                modalScope.app.type = loadedWebApp.webappType;
                modalScope.loadedDesc = WebAppsService.getWebAppLoadedDesc(modalScope.app.type) || {};
                modalScope.desc = modalScope.loadedDesc.desc;
                modalScope.pluginDesc = WebAppsService.getOwnerPluginDesc(modalScope.app.type);
            }
            modalScope.app.name = defaultName;
            modalScope.app.configFromRole = modalScope.app.configFromRole || {};
            modalScope.app.configFromRole[targetKey] = targetValue;
            modalScope.app.config = modalScope.app.config || {};
            modalScope.app.config[targetKey] = targetValue;

        }).then(function(webapp) {
            if (webapp.backendReadyOrNoBackend) {
                // backend up and running, go directly to view
                $state.go("projects.project.webapps.webapp.view", {projectKey : $stateParams.projectKey, webAppId: webapp.id, webAppName: $filter('slugify')(webapp.name)});
            } else {
                $state.go("projects.project.webapps.webapp.edit", {projectKey : $stateParams.projectKey, webAppId: webapp.id, webAppName: $filter('slugify')(webapp.name)});
            }
        });
    }

    $scope.showCreateCodeWebAppModal = function(loadedWebApp, targetKey, targetValue, defaultName) {
        $scope.showCreateWebAppModal('code', loadedWebApp, targetKey, targetValue, defaultName);
    };

    $scope.showCreateVisualWebAppModal = function(loadedWebApp, targetKey, targetValue, defaultName) {
        $scope.showCreateWebAppModal('visual', loadedWebApp, targetKey, targetValue, defaultName);
    };

    $scope.showCreateCodeStudioModal = function(loadedCodeStudio, targetKey, targetValue, defaultName) {
        CreateModalFromTemplate('/templates/code-studios/new-code-studio-modal.html', $scope, null, function(modalScope) {
            modalScope.newCodeStudio.name = defaultName;

        }).then(function(codeStudioObject) {
            $state.go("projects.project.code-studios.code-studio.view", {projectKey : $stateParams.projectKey, codeStudioObjectId: codeStudioObject.id});
        });
    }

    /* Global actions */

    $scope.duplicateThisProject = () => ProjectActionsService.duplicateThisProject($scope, $scope.projectSummary);
    $scope.exportThisProject = () => ProjectActionsService.exportThisProject($scope, $scope.projectSummary);
    $scope.moveThisProject = () => ProjectActionsService.moveThisProject($scope, $scope.projectSummary);
    $scope.deleteThisProject = () => ProjectActionsService.deleteThisProject($scope, $scope.projectSummary);

    $scope.saveCustomFields = function(newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'PROJECT'});
        let oldCustomFields = angular.copy($scope.projectSummary.customFields);
        $scope.projectSummary.customFields = newCustomFields;
        return DataikuAPI.projects.saveSummary($stateParams.projectKey, $scope.projectSummary)
            .success(function() {
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), $scope.projectSummary.customFields);
            })
            .error(function(a, b, c) {
                $scope.projectSummary.customFields = oldCustomFields;
                setErrorInScope.bind($scope)(a, b,c);
            });
    };

    $scope.editCustomFields = function() {
        if (!$scope.projectSummary) {
            return;
        }
        let modalScope = angular.extend($scope, {objectType: 'PROJECT', objectName: $scope.projectSummary.name, objectCustomFields: $scope.projectSummary.customFields});
        CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
            $scope.saveCustomFields(customFields);
        });
    };

    $scope.isAV2ProjectPermissions = () => $scope.projectSummary && $scope.projectSummary.permissionsVersion === 'V2';

    /**
     * Create and display a modal to insert a recipe between the dataset and the recipe
     * @param {Object} selectedDataset dataset node, same schema as return type from `FlowGraph.node(nodeId)`
     * @param {Array.<Object>?} selectedRecipes recipe nodes, same schema as return type from `FlowGraph.node(nodeId)`
    */
    $scope.showInsertRecipeModal = function(selectedDataset, selectedRecipes) {
        return CreateModalFromTemplate('/templates/recipes/insert-recipe-modal.html', $scope, null, (modalScope) => {
            modalScope.dataset = selectedDataset;

            // By default, if the dataset is not shared then insert recipe into the owner zone (which is the only zone the dataset is actually located in)
            let datasetZone = selectedDataset.ownerZone;
            // But if that selection target is the shared instance of that dataset in another zone then we will use that zone for the insertion
            if (selectedDataset.usedByZones && selectedDataset.usedByZones.length) {
                datasetZone = selectedDataset.usedByZones[0];
            }
            const datasetSmartName = DatasetUtils.makeSmart(DatasetUtils.makeLoc(selectedDataset.projectKey, selectedDataset.name), $stateParams.projectKey);
            const datasetRef = {
                projectKey: selectedDataset.projectKey,
                name: selectedDataset.name,
                type: selectedDataset.datasetType
            };
            const datasetStatus = GlobalProjectActions.getAllStatusForDataset(datasetRef);
            modalScope.recipes = GlobalProjectActions.getInsertableRecipesBySection($scope, datasetSmartName, datasetZone, datasetStatus);

            modalScope.handleRecipeSelect = function(recipe){
                modalScope.recipeToInsert = recipe;
            }

            // Default properties
            modalScope.originDatasetName = selectedDataset.name;
            modalScope.downstreamRecipes = selectedDataset.successors.map(recipe => FlowGraph.node(recipe));
            modalScope.hasMultipleDownstreamRecipes = modalScope.downstreamRecipes.length > 1;
            // Preselect all downstream recipes if there is no recipe selected
            modalScope.insertToAllExistingBranch = !selectedRecipes || selectedRecipes.length === modalScope.downstreamRecipes.length;
            modalScope.downstreamRecipesSelection = {};
            modalScope.selectedSuccessorRecipeCount = 0;
            for (let downstreamRecipe of modalScope.downstreamRecipes) {
                if (!selectedRecipes ||     // If no recipe is selected by the user, select all downstream recipes by default
                    (selectedRecipes && selectedRecipes.find(selectedRecipe => selectedRecipe.id === downstreamRecipe.id))      // Otherwise select the one that are selected by the user
                ) {
                    modalScope.downstreamRecipesSelection[downstreamRecipe.name] = true;
                    modalScope.selectedSuccessorRecipeCount++;
                }
            }
            updateAnimation();

            modalScope.onSuccessorRecipeToggle = (recipeName) => {
                // Set the value even though later on ng-model will update it. We need it to update selectedSuccessorRecipeCount
                modalScope.downstreamRecipesSelection[recipeName] = !modalScope.downstreamRecipesSelection[recipeName];
                // Update the selection counter
                modalScope.downstreamRecipesSelection[recipeName] ? modalScope.selectedSuccessorRecipeCount++ : modalScope.selectedSuccessorRecipeCount--;
                // Update the select all checkbox to match with selection
                modalScope.insertToAllExistingBranch = modalScope.selectedSuccessorRecipeCount === modalScope.downstreamRecipes.length;
                updateAnimation();
            };

            modalScope.onInsertToAllExistingBranchToggle = () => {
                // There is a "bug" in angular where for <select> the ng-change callback is triggered after the value is updated
                // but for <input> it is triggered before
                modalScope.insertToAllExistingBranch = !modalScope.insertToAllExistingBranch;
                modalScope.selectedSuccessorRecipeCount = modalScope.insertToAllExistingBranch ? modalScope.downstreamRecipes.length : 0;
                angular.forEach(modalScope.downstreamRecipes, (recipe) => {
                    modalScope.downstreamRecipesSelection[recipe.name] = modalScope.insertToAllExistingBranch
                });
                updateAnimation();
            };

            modalScope.getConfirmDisabledReason = () => {
                if (!modalScope.recipeToInsert) {
                    return 'Please select a recipe type to insert';
                }
                if (!modalScope.selectedSuccessorRecipeCount) {
                    return 'At least 1 subsequent recipe must be selected';
                }
                return '';
            }

            modalScope.confirm = () => {
                if(modalScope.recipeToInsert) {
                    const selectedDownstreamRecipes = Object.keys(modalScope.downstreamRecipesSelection).filter(recipe => modalScope.downstreamRecipesSelection[recipe] == true);
                    const additionalParams = {
                        'originDataset': datasetSmartName,
                        'originRecipes': selectedDownstreamRecipes
                    };
                    modalScope.recipeToInsert.fn(additionalParams);
                    modalScope.resolveModal(modalScope.recipeToInsert);
                }
            };

            modalScope.cancel = () => {
                modalScope.dismiss();
            };

            function updateAnimation() {
                if (!modalScope.hasMultipleDownstreamRecipes) {
                    modalScope.animation = '/static/dataiku/images/recipes/insert/insert-unique-downstream.json';
                    return;
                }
                if (modalScope.selectedSuccessorRecipeCount == 1) {
                    modalScope.animation = '/static/dataiku/images/recipes/insert/insert-check-one.json';
                    return;
                }
                if (modalScope.insertToAllExistingBranch) {
                    modalScope.animation = '/static/dataiku/images/recipes/insert/insert-check-all.json';
                    return;
                }
                if (modalScope.selectedSuccessorRecipeCount > 1) {
                    modalScope.animation = '/static/dataiku/images/recipes/insert/insert-check-many.json';
                    return;
                }
            }
        });
    };

    $scope.getGovernanceStatus = function() {
        $scope.projectGovernanceStatus = undefined;
        if (!$rootScope.appConfig.governEnabled) return;
        $scope.projectGovernanceStatus = { loading: true };
        DataikuAPI.projects.getProjectGovernanceStatus($stateParams.projectKey).success(function(data) {
            $scope.projectGovernanceStatus = { loading: false, data: data };
        }).error(function(a,b,c,d) {
            const fatalAPIError = getErrorDetails(a,b,c,d);
            fatalAPIError.html = getErrorHTMLFromDetails(fatalAPIError);
            $scope.projectGovernanceStatus = { loading: false, error: fatalAPIError };
        });
    };
});

app.controller("MoveProjectModalController", function($scope, $rootScope, $controller, DataikuAPI, $stateParams, translate, ActivityIndicator) {
    $controller("BrowseProjectsCommonController", { $scope });

    $scope.canBrowse = item => item.directory && $scope.movingFolders.findIndex(mf => mf.id === item.id) === -1;
    $scope.canSelect = () => false;

    function updateProjectBreadcrumb(newLocation) {
        $rootScope.$broadcast("projectLocationChanged", { projectKey: $stateParams.projectKey, location: newLocation });

        const destinationFolderName = newLocation[0]?.name || translate("HOME.PROJECTS_PAGE.ROOT_DISPLAY_NAME", "Projects");
        ActivityIndicator.success(translate("PROJECT.ACTIONS.MOVE_TO_FOLDER.SUCCESS", "The project has been successfully moved to <strong>{{folder}}</strong>",
            {folder: destinationFolderName}));

        $scope.resolveModal(newLocation);
    }

    $scope.confirm = () => {
        DataikuAPI.projectFolders
            .moveItems(
                $scope.destination,
                $scope.movingFolders.map(f => f.id),
                $scope.movingProjects.map(p => p.projectKey))
            .success(() => {
                // No need to retrieve all these from the project, we just need the new location
                const options = {
                    includeObjectsCounts: false,
                    includeTimeline: false,
                    includeContributors: false,
                    includeInterests: false,
                    includeGitInfo: false
                };
                DataikuAPI.projects.getSummary($stateParams.projectKey, options)
                    .success(data => {
                        updateProjectBreadcrumb(data.object.projectLocation);
                    }).error(setErrorInScope.bind($scope));
            }).error(setErrorInScope.bind($scope));
    };
});

app.service('ProjectActionsService', function($rootScope, $stateParams, CreateModalFromTemplate, DataikuAPI, FutureWatcher, $timeout, $state, WT1, Assert, ProgressStackMessageBuilder, AI_EXPLANATION_MODAL_MODES, ProjectFolderContext)  {
    this.deleteThisProject = function(scope, projectSummary, projectKey = $stateParams.projectKey, transition = 'home', transitionParams = {}) {
        DataikuAPI.projects.checkDeletability(projectKey).success((data) => {
            if (data.anyMessage) {
                // Some error happened!
                CreateModalFromTemplate("/templates/projects/delete-project-results.html", scope, null, function(newScope) {
                    newScope.beforeDeletion = true;
                    newScope.results = data.messages;
                });
            } else {
                CreateModalFromTemplate("/templates/projects/delete-project-confirm-dialog.html", scope, null, function(newScope) {
                    newScope.clearManagedDatasets = false;
                    newScope.clearOutputManagedFolders = false;
                    newScope.clearJobAndScenarioLogs = true;
                    newScope.projectSummary = projectSummary;

                    newScope.confirmProjectDeletion = function(clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs) {
                        DataikuAPI.projects.delete(projectKey, clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs).success(function(deletionResult) {
                            if(deletionResult.anyMessage) {
                                CreateModalFromTemplate("/templates/projects/delete-project-results.html", scope, null, function(newScope) {
                                    newScope.beforeDeletion = false;
                                    newScope.results = deletionResult.messages;
                                    newScope.$on('$destroy',function() {
                                        $timeout(function() {
                                            $state.transitionTo(transition, transitionParams);
                                        });
                                    });
                                });
                            } else {
                                $state.transitionTo(transition, transitionParams);
                            }

                        }).error(setErrorInScope.bind($rootScope));
                        WT1.event("project-delete", {clearManagedDatasets, clearOutputManagedFolders, clearJobAndScenarioLogs});
                    }
                });
            }
        }).error(setErrorInScope.bind(scope));
    };


    this.moveThisProject = (scope, projectSummary) => {
        return CreateModalFromTemplate("/templates/projects-list/modals/move-project-items-modal.html", scope, "MoveProjectModalController", newScope => {
            newScope.movingProjects = [projectSummary];
            newScope.movingFolders = [];
            newScope.newFolderId = projectSummary.projectLocation[0].id;
        }).then(function(newProjectLocation){
            // Directly update the location of the new project. No need to do a backend call for that.
            projectSummary.projectLocation = newProjectLocation;
        });
    };

    this.exportThisProject = function(scope, projectSummary) {
        CreateModalFromTemplate("/templates/projects/export-project-dialog.html", scope, null, function(newScope) {
            const canExportGitRepository = projectSummary != null && projectSummary.canExportGitRepository;
            newScope.exportOptions = {
                exportUploads: true,
                exportManaged: true,
                exportAnalysisModels: true,
                exportKnowledgeBanks: false,
                exportPromptStudioHistories: false,
                exportSavedModels: true,
                exportInsights: true,
                exportNotebooksWithOutputs: true,
                exportEditableDatasets: true,
                exportGitRepository: canExportGitRepository
            };

            newScope.uiState = {
                showAdvancedOptions: false,
                canExportGitRepository: canExportGitRepository
            };

            newScope.export = function() {
                DataikuAPI.projects.startProjectExport($stateParams.projectKey,
                    newScope.exportOptions).error(function() {newScope.dismiss();}).success(function(initialResponse){

                    CreateModalFromTemplate("/templates/projects/export-progress-modal.html", scope, null, function(progressScope) {
                        progressScope.modalName = "Project";
                        newScope.dismiss();
                        progressScope.download = function(){
                            Assert.trueish(progressScope.finalResponse, 'no future final response');
                            downloadURL(DataikuAPI.projects.getProjectExportURL(progressScope.finalResponse.projectKey,
                                progressScope.finalResponse.exportId));
                            progressScope.dismiss();
                            WT1.event("project-download");
                        }

                        progressScope.abort = function() {
                            DataikuAPI.futures.abort(initialResponse.jobId).error(setErrorInScope.bind(progressScope));
                        }

                        progressScope.done = false;
                        progressScope.aborted = false;
                        FutureWatcher.watchJobId(initialResponse.jobId)
                        .success(function(data) {
                            progressScope.done = data.hasResult;
                            progressScope.aborted = data.aborted;
                            progressScope.futureResponse = null;
                            progressScope.finalResponse = data.result;
                        }).update(function(data){
                            progressScope.percentage =  ProgressStackMessageBuilder.getPercentage(data.progress);
                            progressScope.futureResponse = data;
                            progressScope.stateLabels = ProgressStackMessageBuilder.build(progressScope.futureResponse.progress, true);
                        }).error(function(data, status, headers) {
                            progressScope.done = true;
                            progressScope.futureResponse = null;
                            setErrorInScope.bind(progressScope)(data, status, headers);
                        });
                    });
                }).error(setErrorInScope.bind(scope));
                WT1.event("project-export", newScope.exportOptions);
            };
        });
    };

    this.duplicateThisProject = function(scope, projectSummary) {
        scope.projectSummary = projectSummary;
        CreateModalFromTemplate("/templates/projects/duplicate-project-dialog.html", scope, "DuplicateProjectController", (modalScope) => {
            modalScope.setCurrentProjectFolderId(ProjectFolderContext.getCurrentProjectFolderId() || '');
        });
    };

    this.explainThisProject = function(scope, projectSummary) {
        scope.canWriteProject = () => projectSummary.canWriteProjectContent;
        CreateModalFromTemplate(
            "/static/dataiku/ai-explanations/explanation-modal/explanation-modal.html",
            scope,
            "AIExplanationModalController",
            (newScope) => {
                newScope.objectType = "PROJECT";
                newScope.object = projectSummary;
                newScope.mode = AI_EXPLANATION_MODAL_MODES.EXPLAIN;
            }
        );
    }
});

app.controller('_CreateRecipesBehavior', function($scope, CreateModalFromTemplate, RecipeDescService, SmartId) {
    function preselect(inputDatasetSmartName,zone,recipeAdditionalParams) {
        return function(newScope) {
            newScope.zone = zone;
            newScope.recipeAdditionalParams = recipeAdditionalParams;
            if (inputDatasetSmartName) {
                newScope.$broadcast('preselectInputDataset', inputDatasetSmartName);
            }
        };
    }
    $scope.showCreateShakerModal = function(inputDatasetSmartName, zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'ShakerRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateWindowRecipeModal = function(inputDatasetSmartName, zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'WindowRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateSamplingModal = function(inputDatasetSmartName, zone, recipeAdditionalParams) {
        $scope.recipeAdditionalParams = recipeAdditionalParams; // Necessary for SamplingRecipeController to check if filter is enabled
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'SamplingRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateSyncModal = function(inputDatasetSmartName,zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'SyncRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateCSyncModal = function(inputDatasetSmartName, zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'CsyncRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateUpdateModal = function(inputDatasetSmartName, zone) {
        CreateModalFromTemplate('/templates/recipes/update-recipe-creation.html', $scope, null, preselect(inputDatasetSmartName, zone));
    };
    $scope.showCreateExportModal = function(inputDatasetSmartName, zone) {
        CreateModalFromTemplate('/templates/recipes/export-recipe-creation.html', $scope, null, preselect(inputDatasetSmartName, zone));
    };
    $scope.showExtractFailedRowsModal = function(inputDatasetSmartName, zone) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'ExtractFailedRowsRecipeCreationController', function (newScope) {
            preselect(inputDatasetSmartName, zone)(newScope);
            newScope.wt1EventAdditionalParams = {
                'source': 'right-hand-panel',
            };
        });
    };
    $scope.showCreateDownloadModal = function(preselectedOutput, zone) {
        CreateModalFromTemplate('/templates/recipes/download-recipe-creation.html', $scope, null, function (newScope) {
            newScope.zone = zone;
            if (preselectedOutput) {
                newScope.io.newOutputTypeRadio = 'select';
                newScope.io.existingOutputDataset = preselectedOutput;
            }
        });
    };
    $scope.showCreateGroupingModal = function(inputDatasetSmartName, zone) {
        CreateModalFromTemplate('/templates/recipes/grouping-recipe-creation.html', $scope, null, preselect(inputDatasetSmartName, zone));
    };
    $scope.showCreateUpsertModal = function(inputDatasetSmartName,zone) {
        CreateModalFromTemplate('/templates/recipes/upsert-recipe-creation.html', $scope, null, preselect(inputDatasetSmartName, zone));
    };
    $scope.showCreateDistinctModal = function(inputDatasetSmartName,zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'DistinctRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateSplitModal = function(inputDatasetSmartName,zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/split-recipe-creation.html', $scope, null, preselect(inputDatasetSmartName,zone,recipeAdditionalParams));
    };
    $scope.showCreateTopNModal = function(inputDatasetSmartName,zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'TopNRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateSortModal = function(inputDatasetSmartName,zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/single-output-recipe-creation.html', $scope, 'SortRecipeCreationController', preselect(inputDatasetSmartName, zone, recipeAdditionalParams));
    };
    $scope.showCreateLabelingTaskModal = function(inputDatasetSmartName, zone) {
        CreateModalFromTemplate('/templates/labelingtasks/labeling-task-creation.html', $scope, "LabelingTaskCreationController", preselect(inputDatasetSmartName, zone));
    };
    $scope.showCreatePivotModal = function(inputDatasetSmartName,zone) {
        CreateModalFromTemplate("/templates/recipes/pivot-recipe-creation.html", $scope, null, preselect(inputDatasetSmartName, zone));
    };
    $scope.showCreatePredictionModal = function() {
        CreateModalFromTemplate('/templates/models/prediction/create-scoring-recipe-modal.html', $scope, null);
    };
    $scope.showCreateAssignClustersModal = function() {
        CreateModalFromTemplate('/templates/models/clustering/create-scoring-recipe-modal.html', $scope, null);
    };

    $scope.showCreateJoinModal = function(preselectedInputs, zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/2to1-recipe-creation.html', $scope, 'JoinRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (preselectedInputs && preselectedInputs.length >= 1) {
                newScope.io.inputDataset = preselectedInputs[0]
            }
            if (preselectedInputs && preselectedInputs.length >= 2) {
                newScope.io.inputDataset2 = preselectedInputs[1];
            }
            newScope.recipeAdditionalParams = recipeAdditionalParams;
        });
    };
    $scope.showCreateFuzzyJoinModal = function(preselectedInputs, zone, recipeAdditionalParams) {
            CreateModalFromTemplate('/templates/recipes/2to1-recipe-creation.html', $scope, 'FuzzyJoinRecipeCreationController', function(newScope) {
                newScope.zone = zone;
                if (preselectedInputs && preselectedInputs.length >= 1) {
                    newScope.io.inputDataset = preselectedInputs[0];
                }
                if (preselectedInputs && preselectedInputs.length >= 2) {
                    newScope.io.inputDataset2 = preselectedInputs[1];
                }
                newScope.recipeAdditionalParams = recipeAdditionalParams;
            });
        };
    $scope.showCreateGeoJoinModal = function(preselectedInputs, zone, recipeAdditionalParams) {
            CreateModalFromTemplate('/templates/recipes/2to1-recipe-creation.html', $scope, 'GeoJoinRecipeCreationController', function (newScope) {
                newScope.zone = zone;
                if (preselectedInputs) {
                    preselectedInputs.forEach(function (input, idx) {
                        if (idx < 2) {
                            newScope.io[`inputDataset${idx + 1}`] = input;
                        }
                        newScope.recipe.inputs.main.items.push({ref: input});
                    });
                }
                newScope.recipeAdditionalParams = recipeAdditionalParams;
            });
        };
    $scope.showCreateVStackModal = function(preselectedInputs, zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/Nto1-recipe-creation.html', $scope, 'VStackRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (preselectedInputs) {
                preselectedInputs.forEach(function(input) {
                    newScope.recipe.inputs.main.items.push({ref: input});
                });
            }
            newScope.recipeAdditionalParams = recipeAdditionalParams;
        });
    };
    $scope.showCreateAfgModal = function(preselectedInput, zone) {
        CreateModalFromTemplate('/templates/recipes/afg-recipe-creation.html', $scope, 'AfgRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (preselectedInput) {
                newScope.recipe.inputs.main.items.push({ref: preselectedInput});
            }
        });
    };

    $scope.showCreateEdaRecipeModal = function(preselectedInput, zone) {
        CreateModalFromTemplate('/templates/recipes/eda-recipe-creation.html', $scope, 'EdaRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (preselectedInput) {
                newScope.preselectedInput = preselectedInput;
            }
        });
    };

    $scope.showCreateMergeFolderModal = function(preselectedInputs, zone) {
        CreateModalFromTemplate('/templates/recipes/merge_folder-recipe-creation.html', $scope, 'MergeFolderRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (preselectedInputs) {
                preselectedInputs.forEach(function(input) {
                    newScope.recipe.inputs.main.items.push({ref: input});
                });
            }
        });
    };

    $scope.showCreateListFolderContentsModal = function(inputFolderSmartName, inputFolderName, zone) {
        CreateModalFromTemplate('/templates/recipes/Nto1-recipe-creation.html', $scope, 'ListFolderContentsRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (inputFolderSmartName && inputFolderName) {
                newScope.recipe.inputs.main.items.push({ref: inputFolderSmartName});
                newScope.$broadcast('preselectInputFolder', {smartName: inputFolderSmartName, name: inputFolderName});
            }
        });
    };

    $scope.showCreateListAccessModal = function(inputFolderSmartName, inputFolderName, zone) {
        CreateModalFromTemplate('/templates/recipes/list_access-recipe-creation.html', $scope, 'ListAccessRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (inputFolderSmartName && inputFolderName) {
                newScope.recipe.inputs.main.items.push({ ref: inputFolderSmartName });
                newScope.$broadcast('preselectInputFolder', { smartName: inputFolderSmartName, name: inputFolderName });
            }
        });
    };

    $scope.showCreateEmbedDocumentsModal = function(folderId, folderProjectKey, folderName, zone, datasetProjectKey, datasetName) {
        CreateModalFromTemplate('/templates/recipes/embed_documents-recipe-creation.html', $scope, 'EmbedDocumentsRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            if (folderId && folderProjectKey && folderName) {
                let inputFolderSmartName = SmartId.create(folderId, folderProjectKey);
                newScope.recipe.inputs.main.items.push({ref: inputFolderSmartName});
                newScope.$broadcast('preselectInputFolder', {smartName: inputFolderSmartName, name: folderName});
            }
            if(datasetProjectKey && datasetName) {
                newScope.recipe.inputs.metadata_dataset = {items: [{ref: SmartId.create(datasetName, datasetProjectKey)}]};
            }
        });
    };

    $scope.showSQLRecipeModal = function(inputDatasetSmartName, zone, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/flow-editor/new-sql-recipe-box.html', $scope, null, function(newScope) {
            newScope.zone = zone;
            newScope.preselectedInputDataset = inputDatasetSmartName;
            newScope.recipeAdditionalParams = recipeAdditionalParams;
        }, 'new-sql-recipe-box');
    };

    $scope.showCreateRecipeFromNotebookModal = function(notebookName, recipeType, analyzedDataset) {
        CreateModalFromTemplate("/templates/recipes/recipe-from-notebook-creation.html", $scope, null, function(newScope) {
            newScope.notebookName = notebookName;
            newScope.newRecipeType = recipeType;
            newScope.analyzedDataset = analyzedDataset;
        });
    };

    // preselectedInputs can be a computable smartName or an array of smartNames
    $scope.showCreateCodeBasedModal = function(recipeType, preselectedInputs, zone, prefillKey, recipeAdditionalParams) {
        CreateModalFromTemplate('/templates/recipes/code-based-recipe-creation.html', $scope, 'CodeBasedRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            newScope.newRecipeType = recipeType;
            newScope.preselectedInputs = preselectedInputs;
            newScope.recipePrefillKey = prefillKey;
            newScope.recipeAdditionalParams = recipeAdditionalParams;
        });
    };

    $scope.showCreatePromptModal = function(preselectedInput, preselectedImageFolder, initialPayload, zone) {
        CreateModalFromTemplate('/templates/recipes/prompt-recipe-creation.html', $scope, 'PromptRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            newScope.initialPayload = initialPayload;
            if (preselectedInput) {
                newScope.io.inputDataset = preselectedInput;
            }
            if (preselectedImageFolder) {
                // show selector iff there's a selected image folder when creating from flow
                newScope.showImageFolderSelector = true;
                newScope.io.inputImageFolder = preselectedImageFolder;
            }
        });
    };

    $scope.showCreateNLPClassificationRecipeSuperModal = function(preselectedInput, zone) {
        CreateModalFromTemplate('/templates/recipes/nlp/new-text-classification-recipe.html', $scope, "NLPLLMTextClassificationRecipeCreationController", function(newScope) {
            newScope.recipePresets = {
                preselectedInput, zone,
            };
        });
    };

    $scope.showCreateNLPLLMSummarizationRecipeModal = function(preselectedInput, initialPayload, zone) {
        CreateModalFromTemplate('/templates/recipes/nlp-based-recipe-creation.html', $scope, 'NLPLLMSummarizationRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            newScope.initialPayload = initialPayload;
            if (preselectedInput) {
                newScope.io.inputDataset = preselectedInput;
            }
        });
    };

    $scope.showCreateNLPFineTuningRecipeModal = function(preselectedInput, initialPayload, zone) {
        if ($scope.appConfig.licensedFeatures.advancedLLMMeshAllowed) {
            CreateModalFromTemplate('/templates/recipes/fine-tuning/finetuning-recipe-creation.html', $scope, 'FineTuningRecipeCreationController', function(newScope) {
                newScope.zone = zone;
                newScope.initialPayload = initialPayload;
                if (preselectedInput) {
                    newScope.io.inputDataset = preselectedInput;
                }
            });
        } else {
            CreateModalFromTemplate("/templates/llm/advanced-llm-mesh-required.html", $scope, null, function(modalScope) {
                modalScope.uiState = {
                    "forbiddenFeature" : "Fine tuning"
                }
            });
        }
    };

     $scope.showCreateNLPRAGEmbeddingRecipeModal = function(preselectedInput, initialPayload, zone) {
        CreateModalFromTemplate('/templates/recipes/rag_embedding-recipe-creation.html', $scope, 'RAGEmbeddingRecipeCreationController', function(newScope) {
            newScope.zone = zone;
            newScope.initialPayload = initialPayload;
            if (preselectedInput) {
                newScope.io.inputDataset = preselectedInput;
            }
        });
    };

    $scope.showCreateRetrievalAugmentedLLMModal = function (preselectedInput) {
        CreateModalFromTemplate('/templates/savedmodels/retrieval-augmented-llm/create-retrieval-augmented-llm-modal.html', $scope, 'CreateRetrievalAugmentedLLMModalController', function (newScope) {
            newScope.input.preselectedInput = preselectedInput;
        });
    };

    $scope.showCreateNLPLLMEvaluationRecipeModal = function(preselectedInput, initialPayload, zone) {
        if ($scope.appConfig.licensedFeatures.advancedLLMMeshAllowed) {
            CreateModalFromTemplate('/templates/recipes/new-llm-evaluate-recipe.html', $scope, 'NLPLLMEvaluationRecipeCreationController', function(newScope) {
                newScope.zone = zone;
                newScope.initialPayload = initialPayload;
                if (preselectedInput) {
                    newScope.uiState.inputDs = preselectedInput;
                }
            });
        } else {
            CreateModalFromTemplate("/templates/llm/advanced-llm-mesh-required.html", $scope, null, function(modalScope) {
                modalScope.uiState = {
                    "forbiddenFeature" : "LLM evaluation"
                }
            });
        }
    };

    $scope.showCreateCustomCodeRecipeModal = function(recipeType, inputRefs, inputRole, zone){
        CreateModalFromTemplate('/templates/recipes/custom-code-recipe-creation.html', $scope, null, function(newScope) {
            newScope.zone = zone;
            newScope.newRecipeType = recipeType;
            // there can be more than one preselected input,
            // but they have to be for the same role as there can only be one preselected role.
            newScope.preselectedInputs = inputRefs;
            newScope.preselectedInputRole = inputRole;
        });
    };

    $scope.showCreateRecipeFromPlugin = function(pluginId, inputRefs, zone) {
        let modalScope;
        CreateModalFromTemplate('/templates/recipes/recipe-from-plugin-creation.html', $scope, null, function(newScope) {
            newScope.zone = zone;
            newScope.pluginId = pluginId;
            newScope.inputs = inputRefs;
            if (inputRefs) {
                newScope.inputCount = {}
                for (const key in inputRefs) {
                    newScope.inputCount[key] = inputRefs[key].length
                }
            }
            modalScope = newScope;
            modalScope.$on('$destroy', () => modalScope = null);
        });
        // on opener scope destroy, dismiss the modal
        // e.g. recipe was created, going on recipe page
        this.$on('$destroy', () => modalScope && modalScope.dismiss());
    };

    $scope.showCreateAppRecipeModal = function(recipeType, inputRefs, inputRole){
        CreateModalFromTemplate('/templates/recipes/app-recipe-creation.html', $scope, null, function(newScope) {
            newScope.newRecipeType = recipeType;
            // there can be more than one preselected input,
            // but they have to be for the same role as there can only be one preselected role.
            newScope.preselectedInputs = inputRefs;
            newScope.preselectedInputRole = inputRole;
        });
    };

    $scope.showCreateDatasetFromPlugin = function(pluginId) {
        CreateModalFromTemplate('/templates/datasets/dataset-from-plugin-creation.html', $scope, null, function(newScope) {
            newScope.pluginId = pluginId;
        });
    };

    $scope.showCreateUrlDownloadToFolderDataset = function(projectKey) {
        CreateModalFromTemplate('/templates/recipes/download-url-to-folder-dataset.html', $scope, null, function(newScope) {
            newScope.params.projectKey = projectKey;
        });
    };

    $scope.showCreateStreamingEndpointModal = function(type) {
        CreateModalFromTemplate('/templates/streaming-endpoints/new-streaming-endpoint-modal.html', $scope, "NewStreamingEndpointController", function(newScope) {
            newScope.newStreamingEndpoint.type = type;
        });
    }

    // --- Copy

    $scope.recipeTypeIsCopiable = function(recipeType) {
        if (!recipeType) return false;
        const desc = RecipeDescService.getDescriptor(recipeType);
        if (!desc) {
            throw Error(`Could not find descriptor for recipe type ${recipeType}`);
        }
        return desc.copiable;
    };

    $scope.showCopyRecipeModal = function(recipe) {
        const newScope = $scope.$new();
        newScope.recipe = recipe;
        newScope.newInputs = angular.copy(recipe.inputs);
        newScope.newOutputs = {};
        newScope.zone = recipe.zone;
        CreateModalFromTemplate('/templates/recipes/recipe-copy-modal.html', newScope);
    };

    // method unused now (it doesn't copy data)
    // to be deleted along with modal controller code related to copy (SC-203279)
    $scope.showCopyLabelingTaskModal = function(labelingTask) {
        const newScope = $scope.$new();
        newScope.newTask = labelingTask;
        newScope.newTask.id = null;
        newScope.zone = labelingTask.zone;
        CreateModalFromTemplate('/templates/labelingtasks/labeling-task-creation.html', newScope);
    };
});

app.controller('ContextualProjectHomeController', function($scope, $state, $stateParams, TopNav, HistoryService, $timeout, ProjectHomeContextService) {
    TopNav.setItem(TopNav.ITEM_PROJECT, $stateParams.projectKey);

    $scope.$watch("projectSummary", nv => {
        if (!nv)  return;
        if (nv.projectKey === $stateParams.projectKey) {
            let stateToRedirectTo = 'projects.project.home.summary';
            HistoryService.recordProjectOpen($scope.projectSummary);
            $timeout(() => ProjectHomeContextService.recordProjectViewed($stateParams.projectKey));
            if (!$stateParams.discussionId) {
                stateToRedirectTo = ProjectHomeContextService.getProjectRedirectState(nv, $scope.projectSummaryStatus);
            }
            $state.go(stateToRedirectTo, {projectKey: nv.projectKey}, {location: 'replace'});
        }
    });
});

app.component('projectAccessRequests', {
    bindings: {
        projectKey: '<',
        accessInfo: '<'
    },
    templateUrl: '/templates/projects/request-access.html',
    controller: function ctrlProjectAccessRequests($scope, TopNav, DataikuAPI) {
        const ctrl = this;

        TopNav.setLocation(TopNav.TOP_ACCESS_REQUEST, undefined, TopNav.TABS_NONE, undefined);

        const getVisibleSummary = () => {
            DataikuAPI.projects.getVisibleSummary(ctrl.projectKey).success((visibleSummary) => {
                ctrl.projectSummary = visibleSummary;
                ctrl.projectNameWithLockIcon = `${sanitize(ctrl.projectSummary.name)} <i class="icon-lock"></i>`;
            }).error(setErrorInScope.bind($scope));
        };

        ctrl.$onChanges = (changes) => {
            if (!changes || !changes.accessInfo || !changes.accessInfo.currentValue) return;
            ctrl.projectNameWithLockIcon = `${sanitize(ctrl.projectKey)} <i class="icon-lock"></i>`;
            if (ctrl.accessInfo.isVisible) {
                getVisibleSummary();
            }
        }
    }
});

app.controller('ProjectHomeTabController', function($scope, $state) {
    $scope.tabUiState = {
        projectAppView : 'REGULAR'
    };
    $scope.$state = $state;

    $scope.$watch("projectSummary", function(nv) {
        if (!nv)  return;

        if ($scope.projectSummary.projectAppType == 'APP_INSTANCE') {
            $scope.tabUiState.projectAppView = "APP_TILES";
        } else {
            $scope.tabUiState.projectAppView = "REGULAR";
        }
    });

    $scope.switchToAppView = function(){
        $scope.tabUiState.projectAppView = "APP_TILES";
    }
    $scope.switchToRegularProjectView = function(){
        $scope.tabUiState.projectAppView = "REGULAR";
    }
});

app.controller('ProjectHomeController', function($scope, $state, $stateParams, $timeout, WT1, TopNav, DataikuAPI, Dialogs, ActivityIndicator, FutureProgressModal, HistoryService, StateUtils, WatchInterestState, CreateModalFromTemplate) {
    TopNav.setLocation(TopNav.TOP_HOME, null, null, "summary");
    TopNav.setItem(TopNav.ITEM_PROJECT, $stateParams.projectKey);

    $scope.$watch("projectSummary", function(nv) {
        if (!nv)  return;
        if (nv.projectKey == $stateParams.projectKey) {
            if ($scope.projectSummary.name) {
                HistoryService.recordProjectOpen($scope.projectSummary);
                TopNav.setPageTitle($scope.projectSummary.name);
            }
            if ($scope.projectSummary.projectAppType == 'APP_INSTANCE' && nv.projectKey) {
                DataikuAPI.apps.getInstanceSummary(nv.projectKey).success(function (data) {
                    $scope.appSummary = data;
                    TopNav.setItem(TopNav.ITEM_PROJECT, nv.projectKey, $scope.appSummary);
                }).error(setErrorInScope.bind($scope));
            }
        }
    });

    $scope.skipAppUpdate = () => {
        $scope.appSummary.obsolete = false;
        $scope.appSummary.instanceSkippedVersion = $scope.appSummary.templateVersion;
        DataikuAPI.apps.skipInstanceUpdate($scope.projectSummary.appManifest.id, $scope.projectSummary.projectKey, $scope.appSummary.templateVersion)
            .success(() => {
                ActivityIndicator.info("Update skipped");
            })
            .error(setErrorInScope.bind($scope));
    };

    $scope.reInstantiateApp = () => {
        CreateModalFromTemplate("/templates/apps/app-re-instantiation-modal.html", $scope, null, function(newScope) {
            newScope.projectSummary = $scope.projectSummary;
            newScope.keepProjectDescription = true;
            newScope.keepProjectVariables = true;
            newScope.confirmAppReInstantiation = function(keepProjectDescription, keepProjectVariables) {
                DataikuAPI.apps.updateInstance($scope.projectSummary.appManifest.id, $scope.projectSummary.projectKey, keepProjectDescription, keepProjectVariables).success(function(future) {
                    const appPageScope = $scope.$parent.$parent;
                    FutureProgressModal.show(appPageScope, future, "Recreating your application").then(function(result) {
                        if (!result.done) {
                            Dialogs.infoMessagesDisplayOnly(appPageScope, "Creation result", result, undefined, undefined, 'static', false);
                        } else {
                            WT1.event("app-recreated", { keepProjectVariables: keepProjectVariables });
                            $state.reload();
                        }
                    });
                }).error(setErrorInScope.bind(newScope));
            };
        });
    };

    $scope.uiState = {
        editSummary: false,
        activeTimelineTab : 'full'
    };

    $scope.projectRecentItems = HistoryService.getRecentlyViewedItems(5, null, $stateParams.projectKey);

    $scope.refreshTimeline = function() {
        if ($scope.isProjectAnalystRO()) {
            DataikuAPI.timelines.getForProject($stateParams.projectKey).success(function(data) {
              $scope.objectTimeline = data;
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.historyItemHref = function(item) {
        return StateUtils.href.dssObject(item.type, item.id, item.projectKey);
    };

    $scope.isProjectStatusSelected = function(projectStatus) {
        return projectStatus.name == $scope.projectSummary.projectStatus;
    };

    $scope.showSwitchToProjectViewButton = function() {
        return $scope.projectSummary.appManifest.instanceFeatures.showSwitchToProjectViewButton;
    };

    /* For update of name/tags/description */
    function save(){
        DataikuAPI.projects.saveSummary($stateParams.projectKey, $scope.projectSummary).success(function(data){
            ActivityIndicator.success("Saved!");
        }).error(setErrorInScope.bind($scope));
    };

    $scope.$on("objectSummaryEdited", function(event, currentEditing){
        save();
        // remove pattern image from cache to update initials on project image (for graph/list view on project explorer page)
        if (currentEditing === 'name' && $scope.projectSummary.showInitials && !$scope.projectSummary.isProjectImg) {
            DataikuAPI.images.removeImage($stateParams.projectKey, 'PROJECT', $stateParams.projectKey);
        }
        $scope.refreshTimeline();
    });

    $scope.$on('customFieldsSummaryEdited', function(event, customFields) {
        $scope.saveCustomFields(customFields);
    });

    $scope.$on("projectImgEdited", function(ev, newState){
        $scope.projectSummary.imgColor = newState.imgColor;
        $scope.projectSummary.isProjectImg = newState.isProjectImg;
        $scope.projectSummary.imgPattern = parseInt(newState.imgPattern, 10);
        $scope.projectSummary.showInitials = newState.showInitials;
        save();
        $scope.refreshTimeline();
    });

    const { isWatching, isShallowWatching, isFullyWatching } = WatchInterestState;
    $scope.isWatching = isWatching;
    $scope.isShallowWatching = isShallowWatching;
    $scope.isFullyWatching = isFullyWatching;

    $scope.getGovernanceStatus();
});

app.controller('ProjectActivityViewCommonController', function($scope, Fn) {
    $scope.prepareData = function(data) {
        $scope.activitySummary = data;
        $scope.dailyData = {
            commits: data.totalCommits.dayTS.data.map(function(ts, i) {
                return { date: new Date(ts), value: data.totalCommits.value.data[i] };
            }),
            writeHours: data.totalHoursWithWrites.dayTS.data.map(function(ts, i) {
                return { date: new Date(ts), value: data.totalHoursWithWrites.value.data[i] };
            }),
            presenceHours: data.totalPresence.dayTS.data.map(function(ts, i) {
                return { date: new Date(ts), value: Math.round(data.totalPresence.value.data[i] / 1000) };
            })
        };

        data.contributorsChart.dates = data.contributorsChart.bucketsTS.data
            .map(function (ts) { return new Date(ts); });
        data.contributorsSummaryAllTime.forEach(function(c) {
            if (c.user in data.contributorsChart.perContributor) {
                data.contributorsChart.perContributor[c.user].totalCommits = c.commits;
                data.contributorsChart.perContributor[c.user].totalAddedLines = c.addedLines;
                data.contributorsChart.perContributor[c.user].totalRemovedLines = c.removedLines;
            }
        });
        // To array for sortability
        data.contributorsChart.perContributor = Object.keys(data.contributorsChart.perContributor)
            .map(Fn.from(data.contributorsChart.perContributor));
        // Scope is computed by first 'global' chart, then copy it to inididual charts
        // (global charts must render first)
        data.contributorsChart.scale = null;
        $scope.setContributorsChartScale = function(scale) {
            data.contributorsChart.scale = scale;
            return scale;
        };

        data.contributorsSummary.forEach(function (c) {
            c.presenceHours = c.totalPresence / 3600 / 1000; // precence in hours
        });

        $scope.totalCommitsPerHour = Array.reshape2d(data.totalCommitsPerHour.matrix, 24);
    }
});

app.controller('ProjectActivityDashboardController', function($scope, DataikuAPI, TopNav, $stateParams, $controller) {
    $controller('ProjectActivityViewCommonController', {$scope: $scope});

    TopNav.setLocation(TopNav.TOP_HOME, null, null, "activity");
    TopNav.setItem(TopNav.ITEM_PROJECT, $stateParams.projectKey);
    $scope.uiState = {
        settingsPane : "summary",
        summaryChart: 'commits',
        contributorsChart: 'commits'
    };

    $scope.niceHours = function(sec) {
        var h = (sec/3600);
        return h.toFixed(1).replace(/\.0$/, '') + (h >= 2 ? ' hrs' : ' hr');
    };

    $scope.$watch('uiState.timeSpan', function(timeSpan) {
        DataikuAPI.projects.activity.getActivitySummary($stateParams.projectKey, timeSpan).success(function(data){
            $scope.prepareData(data);
        }).error(setErrorInScope.bind($scope));
    });
    $scope.uiState.timeSpan = 'year';
});

app.controller('ProjectMetricsController', function($scope, DataikuAPI, TopNav, $stateParams, $controller) {
    TopNav.setLocation(TopNav.TOP_HOME, null, null, "status");
    TopNav.setItem(TopNav.ITEM_PROJECT, $stateParams.projectKey);

    $scope.$watch("projectSummary", function(nv, ov) {
        if (!nv)  return;
        TopNav.setPageTitle($scope.projectSummary.name);
    });
});


app.controller('ProjectMetricsEditionController', function($scope, DataikuAPI, TopNav, $stateParams, $controller, WT1) {
    $scope.newMetric = {};

    $scope.addMetricPoint = function(newMetric) {
        var metricsData = {};
        metricsData[newMetric.name] = newMetric.value;
        DataikuAPI.projects.saveExternalMetricsValues($stateParams.projectKey, metricsData, {}).success(function(data){
            WT1.event("project-metric-inserted");
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller('ProjectChecksEditionController', function($scope, DataikuAPI, TopNav, $stateParams, $controller, WT1) {
    $scope.newCheck = {};

    $scope.addCheckPoint = function(newCheck) {
        var checksData = {};
        if (newCheck.message && newCheck.message.length > 0) {
            checksData[newCheck.name] = [newCheck.value, newCheck.message];
        } else {
            checksData[newCheck.name] = newCheck.value;
        }
        DataikuAPI.projects.saveExternalChecksValues($stateParams.projectKey, checksData).success(function(data){
            WT1.event("project-check-inserted");
        }).error(setErrorInScope.bind($scope));
    };
});

app.directive('projectScenariosRuns', function($controller, $state){
    return {
        templateUrl: '/templates/projects/home/project-scenarios-runs.html',
        scope: {
            scenariosDays: '<',
            activeScenarios: '=',
            totalScenarios: '=',
            projectKey: '='
        },
        link: function($scope, element, attrs){
            $controller('OutcomesBaseController', {$scope: $scope});

            $scope.uiState = {};

            $scope.$watch('scenariosDays', function(nv, ov) {
                if (!nv) return;
                $scope.fixupOutcomes($scope.scenariosDays.columns, 14);
                computeLastRuns();
                sortRows();
                fixupRows();
                $scope.hasNoScenario = computeHasNoScenario();
                $scope.scenariosDays.rows = $scope.scenariosDays.rows.slice(0,4);
                $scope.displayedColumns = $scope.scenariosDays.columns; // b/c the underlying directive, outcomeCells, needs it in its scope
            });

            function computeLastRuns() {
                $scope.scenariosDays.rows.forEach(function(row) {
                    const id = row.uniqueId;
                    for (let i = $scope.scenariosDays.columns.length - 1; i>=0; i--) {
                        const column = $scope.scenariosDays.columns[i];
                        if (column.actions && column.actions[id]) {
                            const actions = column.actions[id];
                            row.lastRun = {
                                date: column.date,
                                outcome: actions[actions.length - 1].outcome.toLowerCase()
                            };
                            break;
                        }
                    }
                });
            }

            function sortRows() {
                $scope.scenariosDays.rows.sort(function(r1, r2) {
                    if (angular.equals({}, r2)) {
                        return -1;
                    }
                    if (angular.equals({}, r1)) {
                        return 1;
                    }
                    if (r1.lastRun.date == r2.lastRun.date) {
                        return r1.info.name.localeCompare(r2.info.name)
                    }
                    return r1.lastRun.date.localeCompare(r2.lastRun.date);
                });
            }

            function fixupRows() {
                while($scope.scenariosDays.rows.length < 4) {
                    $scope.scenariosDays.rows.push({});
                }
            }

            function computeHasNoScenario() {
                for (let i = 0; i<$scope.scenariosDays.rows.length; i++) {
                    const row = $scope.scenariosDays.rows[i];
                    if (!angular.equals({}, row)) {
                        return false;
                    }
                }
                return true;
            }

            $scope.hover = function(evt, column, row, localScope) {
                if (!$scope.hasNoScenario) {
                    $scope.hovered.date = column.date;
                    if (row && row.uniqueId) {
                        $scope.hovered.row = row;
                    }
                    $scope.hovered.actions = row && row.uniqueId ? column.actions[row.uniqueId] : null;
                }
            };

            $scope.unhover = function(evt, column, row, localScope) {
                $scope.hovered.date = null;
                $scope.hovered.row = null;
                $scope.hovered.actions = null;
            };

            $scope.select = function(evt, column, row, localScope) {
                if (row && row.uniqueId) {
                    evt.stopPropagation();
                    $state.go('projects.project.automation.outcomes.scoped', {
                        projectKey: $scope.projectKey,
                        scopeToDay: column.date,
                        scenarioQuery: row.info.name
                    });
                }
            }
        }
    };
});

app.directive('projectLocation', function($timeout, translate){
    return {
        templateUrl: '/templates/projects/project-location.html',
        scope: {
            summary: '<', // ProjectSummary object containing the location of the project
            collapseOnOverflow: '<', // Boolean indicating whether breadcrumb should hide the middle folders in case of horizontal overflow
            rootIcon: '<'
        },
        link: function($scope, element){
            const breadcrumbElement = element[0];

            function hasOverflow(breadcrumbElement) {
                let items = breadcrumbElement.getElementsByClassName("folder");
                let firstItemHeight = items[0].offsetHeight;
                let totalHeight = breadcrumbElement.offsetHeight;
                return totalHeight > firstItemHeight;
            }

            function updateBreadcrumb(newBreadcrumb) {
                $scope.projectBreadcrumb = newBreadcrumb;
                $scope.showExpanded = true;
                // If project is at the root level or just under, no need to try to collapse the intermediate levels
                if ($scope.collapseOnOverflow && $scope.projectBreadcrumb.length > 1) {
                    observer.observe(breadcrumbElement, { childList: true, subtree: true });
                }
            }

            const observer = new MutationObserver(() => {
                $scope.showExpanded = !hasOverflow(breadcrumbElement);
                $scope.hiddenHierarchy = $scope.showExpanded ? "" : $scope.projectBreadcrumb.slice(0, -1).map(folder => folder.name).join(" > ");
                observer.disconnect();
            });

            const projectLocationChangedWatcher = $scope.$on("projectLocationChanged", function(event, args) {
                observer.disconnect(); // Cancel any pending observation
                const newBreadcrumb = [...(args.location)].reverse().slice(1);
                if (args.projectKey === $scope.summary.projectKey && JSON.stringify(newBreadcrumb) !== JSON.stringify($scope.projectBreadcrumb)) {
                    updateBreadcrumb(newBreadcrumb);
                }
            });

            $scope.$on('$destroy', function() {
                observer.disconnect();  // Cancel any pending observation
                projectLocationChangedWatcher(); // Stop watching for "projectLocationChanged" events
            });

            $scope.expand = function() {
                $scope.showExpanded = true;
            }

            // Initialize component
            $scope.translate = translate;
            updateBreadcrumb([...$scope.summary.projectLocation].reverse().slice(1));
        }
    };
});

}());
