(function(){
'use strict';

const app = angular.module('dataiku.projects.settings',[]);

//TODO: common controller
app.controller("ProjectSettingsVariablesController", function($scope, $stateParams, DataikuAPI, Logger, ActivityIndicator, TopNav) {
    TopNav.setLocation(TopNav.TOP_MORE, "variables", "NONE", null);
    $scope.projectVariables = {};

    function getSerialized() {
        return {
            standard : $scope.projectVariables.standardAsJSON || '{}',
            local : $scope.projectVariables.localAsJSON || '{}'
        };
    }

    $scope.dirtyVariables = function() {
        if (!$scope.projectVariables.saved) {return false;}
        try {
            return !angular.equals(getSerialized(), $scope.projectVariables.saved);
        } catch (err) {
            Logger.error(err);
            return true; // Always dirty if invalid
        }
    };

    $scope.saveVariables = function(){
        try {
            const serialized = getSerialized();
            return DataikuAPI.projects.variables.save($stateParams.projectKey, serialized).success(function() {
                $scope.projectVariables.saved = serialized;
                ActivityIndicator.success("Saved variables");
            }).error(setErrorInScope.bind($scope));
        } catch (err) {
            ActivityIndicator.error("Invalid format: "+err.message);
        }
    };

    DataikuAPI.projects.variables.get($stateParams.projectKey).success(function(data) {
        $scope.projectVariables.saved = angular.copy(data);
        $scope.projectVariables.standardAsJSON = data.standard;
        $scope.projectVariables.localAsJSON = data.local;

    }).error(setErrorInScope.bind($scope));

    checkChangesBeforeLeaving($scope, $scope.dirtyVariables);
});

//TODO: common controller
app.controller("ProjectSettingsSettingsController", function($scope, $controller, $stateParams,$timeout, DataikuAPI, WT1, TopNav,
                                                             Dialogs, $q, ActivityIndicator,
                                                             ProjectIntegrations, FutureProgressModal, TaggingService, PluginConfigUtils) {

    $scope.uiState = {
        settingsPane : $stateParams.selectedTab || 'tags',
        selectedPlugin: null
    };

    $scope.sqlLikeRecipesInitializationModes = [
        ["RESOLVED_TABLE_REFERENCES", "Fully-resolved table references"],
        ["VARIABILIZED_TABLE_REFERENCES", "Table references with variables"],
        ["DATASET_REFERENCES", "Virtual dataset references"]
    ]
    $scope.sqlLikeRecipesInitializationModesDesc = [
        "Like MYPROJECT_mytable. The most 'understandable' form, it does not permit relocation to another project since the recipe will not 'follow'",
        "Like ${projectKey}_mytable. This form permits relocation to another project key, but does not support changing the name of the table in the datasets",
        "Like ${tbl:datasetName}. This is the most versatile form but is slightly less familiar for SQL developers"
    ]

    $scope.virtualWebAppBackendSettingsModes = [{id:"USE_DEFAULT", label:"Run as local processes"}, {id:"INHERIT", label:"Inherit instance-level settings"}, {id:"EXPLICIT", label:"Run in container"}];
    
    DataikuAPI.security.listUsers().success(function(data) {
        $scope.allUsers = data;
    }).error(setErrorInScope.bind($scope));

    var savedSettings = null;

    $scope.invalidTabs = new Set();

    $scope.$watch("uiState.settingsPane", function(nv, ov) {
        if (nv === ov) return;
        // We do not set 'Resource control' tab as invalid to avoid weird UI behavior. For this tab, a ng-model is not
        // changed if the new input value is not valid. Hence if a user exits the 'Resource control' tab with some
        // invalid fields and then switch back to it, the fields will no longer be invalid, which can be confusing.
        if ($scope.projectSettingsForms.$invalid && ov !== 'limits') {
            $scope.invalidTabs.add(ov);
        }
        $scope.invalidTabs.delete(nv);
    });

    $scope.isProjectSettingsFormInvalid = function() {
        return $scope.projectSettingsForms.$invalid || $scope.invalidTabs.size;
    }

    $scope.tagsAreDirty = function(projectTagsMapDirty, projectTagsMap) {
        // check for renamed tags

        const tags1 = Object.keys(projectTagsMapDirty);
        const tags2 = Object.keys(projectTagsMap);

        // if the number of tags is different, return false
        if (tags1.length !== tags2.length) {
            return true;
        }

        for (let i = 0; i < tags1.length; i++) {
            const key = tags1[i];
            // if key is not in the other map, return true
            if(projectTagsMapDirty[key].isEdited || projectTagsMap[key] === undefined){            
                return true;
            }
            // if the color is different, return true
            if (projectTagsMapDirty[key].color !== projectTagsMap[key].color) {
                return true;
            }
        }
        return false;
    }

    $scope.dirtySettings = function() {
        return !$scope.projectSettings
            || !angular.equals($scope.projectSettings.settings, savedSettings)
            || $scope.tagsAreDirty($scope.projectTagsMapDirty, $scope.projectTagsMap)
            || ($scope.originalPluginSettings !== null && !angular.equals($scope.originalPluginSettings, $scope.pluginSettings));
    };

    $scope.validIntegration = ProjectIntegrations.getValidity;

    loadSettings();

    function loadSettings() {
        $scope.$emit('projectTagsUpdated'); // triggers a project tag update, which is required for dirtiness detection because other places edit them in-place
        DataikuAPI.projects.getSettings($stateParams.projectKey).success(function(projectSettings) {
            $scope.projectSettings = projectSettings;
            savedSettings = angular.copy($scope.projectSettings.settings);
            $scope.savedSettings = savedSettings;
        }).error(setErrorInScope.bind($scope));
    }

    // Settings mgmt
    $scope.saveSettings = function() {
        // filter out rules of exposed datasets where the projectKey is still null
        const settings = angular.copy($scope.projectSettings.settings);

        let promises = [];
        promises.push(TaggingService.saveToBackend($scope.projectTagsMapDirty));

        if (!angular.equals($scope.projectSettings.settings, savedSettings)) {
            promises.push(DataikuAPI.projects.saveSettings($stateParams.projectKey, settings));
        }

        $q.all(promises)
            .then(loadSettings)
            .then($scope.refreshProjectData)
            .then(() => {
                if (!$scope.validIntegration() || $scope.projectSettingsForms.$invalid) {
                    ActivityIndicator.warning("Saved with some invalid fields");
                } else {
                    ActivityIndicator.success("Saved!");
                }
            })
            .catch(setErrorInScope.bind($scope));

        // Save plugins at project level by providing the project key
        $scope.dirtyPluginSettings() && $scope.savePluginSettings($stateParams.projectKey);
    };

    $scope.availableIntegrationTypes = ProjectIntegrations.integrationTypes;
    $scope.getIntegrationTypeLabel = function(type) {
        const integration = $scope.availableIntegrationTypes.find(element => element.id === type);
        return integration === undefined ? type : integration.label;
    };

    $scope.addIntegration = function(type) {
        WT1.event("project-integration-add", {type: type});
        var intConf = {};
        var integration = {
            active : true,
            hook : {
                type : type,
                configuration : intConf
            },
            $expanded:true
        };

        switch (type) {
            case "slack-project":
                intConf.mode = 'WEBHOOK';
                intConf.useProxy = true;
                intConf.selection = {
                    timelineEditionItems: true,
                    timelineItemsExceptEditions: true,
                    watchStar: true
                };
                break;
            case "msft-teams-project":
                intConf.useProxy = true;
                intConf.webhookType = "WORKFLOWS";
                intConf.selection = {
                    timelineEditionItems: true,
                    timelineItemsExceptEditions: true,
                    watchStar: true
                };
                break;
            case "google-chat-project":
                intConf.useProxy = true;
                intConf.selection = {
                    timelineEditionItems: true,
                    timelineItemsExceptEditions: true,
                    watchStar: true
                };
                break;
        }

        $scope.projectSettings.settings.integrations.integrations.push(integration);
    };

    $scope.removeIntegration = function(index) {
        $scope.projectSettings.settings.integrations.integrations.splice(index, 1);
        ProjectIntegrations.removeIntegration(index);
    };

    $timeout(function() {
        checkChangesBeforeLeaving($scope, $scope.dirtySettings);
    });

    $scope.resyncHDFSDatasetPermissions = function(){
        DataikuAPI.projects.resyncHDFSDatasetPermissions($stateParams.projectKey).success(function(data){
            FutureProgressModal.show($scope, data, "ACLs sync").then(function(result){
                Dialogs.infoMessagesDisplayOnly($scope, "ACLs sync result", result);
            });
        });
    };

    $scope.governIntegrationSync = function() {

        DataikuAPI.projects.governIntegrationSync($stateParams.projectKey).success(function (data) {
            FutureProgressModal.show($scope, data, "Synchronizing DSS items of this project on Dataiku Govern", null, 'static', false, true).then(function(result){
                if (result) {
                    $scope.governIntegrationProjectSyncResult = {
                        status: 'SUCCESS',
                        data: result
                    };
                }
            })
        }).error(function(data, status, headers) {
            $scope.governIntegrationProjectSyncResult = {
                status: 'ERROR',
                error: getErrorDetails(data, status, headers)
            };
        });
    };
    
    DataikuAPI.admin.clusters.listAccessible('HADOOP').success(function(data){
        $scope.clusterIds = data.map(function(c) {return c.id;});
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.admin.clusters.listAccessible('KUBERNETES').success(function(data){
        $scope.k8sClusterIds = data.map(function(c) {return c.id;});
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.containers.listNames(null, "USER_CODE").success(function(data){
        $scope.containerNames = data;
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.containers.listNames(null, "VISUAL_RECIPES").success(function(data){
        $scope.containerNamesForVisualRecipesWorkloads = data;
    }).error(setErrorInScope.bind($scope));

    // Plugins presets
    $controller("PluginsExploreController", { $scope: $scope });
    $controller("PluginSettingsController", { $scope: $scope });

    $scope.refreshPluginsList = function() {
        DataikuAPI.plugins.listPluginsWithPresets().success(function(data) {
            $scope.projectPluginsList = { plugins: data };
        }).error(setErrorInScope.bind($scope));
    };

    $scope.selectPlugin = function(pluginId) {
        if (!pluginId) {
            return;
        }
        // Get plugin with project-level settings (by providing the project key)
        DataikuAPI.plugins.get(pluginId, $stateParams.projectKey).success(function(data) {
            $scope.pluginData = data;
            $scope.installed = data.installedDesc;

            if ($scope.installed.desc.params && data.settings.config) {
                PluginConfigUtils.setDefaultValues($scope.installed.desc.params, data.settings.config);
            }
            $scope.setPluginSettings(data.settings);
            $scope.uiState.selectedPlugin = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshPluginsList();

    TopNav.setLocation(TopNav.TOP_MORE, "settings", "NONE", "config");
});

app.service("ProjectIntegrations", function(){
    var validIntegrations = [];
    return {
        integrationTypes :[
            {"id" : "slack-project", "label" : "Slack"},
            {"id" : "msft-teams-project", "label" : "Microsoft Teams"},
            {"id" : "google-chat-project", "label" : "Google Chat"},
            {"id" : "github", "label" : "Github"}
        ],
        getValidity : () => !validIntegrations.includes(false),
        setValidity : (index, flag) => {
            validIntegrations[index] = flag;
        },
        removeIntegration : (index) => validIntegrations.splice(index,1)
    };
});

app.directive("projectIntegrationEditor", function(ProjectIntegrations){
    return {
        scope : true,
        templateUrl : '/templates/projects/project-integration-editor.html',
        link : function($scope, element, attrs) {
            $scope.integrationTypes = ProjectIntegrations.integrationTypes;
        }
    }
});

app.directive("projectIntegrationParams", function(ProjectIntegrations){
    return {
        scope : {
            hook : '=',
            form : '=',
            index : '='
        },
        link : function($scope, element) {
            $scope.$watch("form.$valid", () => ProjectIntegrations.setValidity($scope.index, $scope.form.$valid)); 
        },
        templateUrl : '/templates/projects/project-integration-params.html',
    }
});

app.controller("NotificationsReporterController", function($scope, $timeout) {
    $scope.noStartMessage = $scope.noStartMessage || false;
    $scope.showItemHeader = $scope.showItemHeader == undefined || $scope.showItemHeader;

    $scope.conditionEditorOptions = {
            mode:'text/grel',
            theme:'elegant',
            indentUnit: 4,
            lineNumbers : false,
            lineWrapping : true,
            autofocus: true,
            onLoad : function(cm) {$scope.codeMirror = cm;}
        };

    // because otherwise the codemirror pops up shrunk when the ng-show on reporter.messaging.channelId changes state
    $scope.$watch("reporter.messaging.channelId", function() {
        if ( $scope.codeMirror ) {
            $timeout(function() {$scope.codeMirror.refresh();});
        }
    }, true);

    if ($scope.reporter.messaging == null) {
        $scope.reporter.messaging = {};
    }
});

/***********************************
 * Security
 ***********************************/

 //TODO: common controller
app.controller("ProjectSettingsSecurityController", function($scope, TopNav, $state) {
    TopNav.setLocation(TopNav.TOP_MORE, "security", "NONE", null);
    $scope.uiState = {
        securityPane: $state.params.selectedTab || "permissions",
    };
    $scope.$watch("projectSummary", function(newValue) {
        if (!newValue || $state.params.selectedTab) {
            return;
        }

        let selectedTab;
        if ($scope.projectSummary.canEditPermissions) {
            selectedTab = "permissions";
        } else if ($scope.projectSummary.canManageDashboardAuthorizations) {
            selectedTab = "dashboard";
        } else if ($scope.projectSummary.canManageAdditionalDashboardUsers) {
            selectedTab = "dashboardUsers";
        } else {
            selectedTab = "exposed";
        }
        $state.go(".", { selectedTab }, { location: "replace" });
    });
})

app.controller("ProjectSettingsAPIController", function($scope, $stateParams,
               DataikuAPI, Dialogs,CreateModalFromTemplate) {

    $scope.isExpired = function (key) {
        return key.expiresOn != 0 && key.expiresOn < new Date().getTime();
    };

    $scope.canAPI = function(){
        if ($scope.appConfig.communityEdition && !$scope.appConfig.licensing.ceEntrepriseTrial) return false;
        if ($scope.appConfig.licensingMode == "SAAS") return false;
        return true;
    }

    $scope.refreshProjectApiKeysList = function() {
        DataikuAPI.projects.publicApi.listProjectApiKeys($stateParams.projectKey).success(function(data) {
            $scope.apiKeys = data;
        }).error(setErrorInScope.bind($scope));
    };


    $scope.createProjectApiKey = function() {
        CreateModalFromTemplate("/templates/projects/project-api-key-modal.html", $scope, null, function(newScope) {
            newScope.apiKey = {
                projectKey : $stateParams.projectKey,
                label : "New key",
                localDatasets : [ {
                    datasets : ['__rw__dataset1__', '__rw__dataset2__'],
                    privileges : [
                        'READ_DATA',
                        'WRITE_DATA',
                        'READ_METADATA',
                        'WRITE_METADATA',
                        'READ_SCHEMA',
                        'WRITE_SCHEMA'
                    ]
                }, {
                    datasets : ['__r__dataset__'],
                    privileges : [
                        'READ_DATA',
                        'READ_METADATA',
                        'READ_SCHEMA'
                    ]
                }],
                projectPrivileges : {
                    admin: false,
                    readProjectContent: true,
                    writeProjectContent: false,
                    shareToWorkspaces: true,
                    exportDatasetsData: true,
                    readDashboards: true,
                    writeDashboard: false,
                    moderateDashboards: false,
                    runScenarios: false,
                    manageDashboardAuthorizations: false,
                    manageExposedElements: false,
                    executeApp: false
                },
                execSQLLike: false
            };
            newScope.creation = true;
        });
    };

    $scope.editProjectApiKey = function(key) {
        CreateModalFromTemplate("/templates/projects/project-api-key-modal.html", $scope, null, function(newScope) {
            newScope.apiKey = angular.copy(key);
            newScope.creation = false;
        });
    };

    $scope.deleteProjectApiKey = function(keyId) {
        Dialogs.confirm($scope, "Remove API key", "Are you sure you want to remove this API key?").then(function() {
            DataikuAPI.projects.publicApi.deleteProjectApiKey($stateParams.projectKey, keyId).success(function(data){
               $scope.refreshProjectApiKeysList();
            }).error(setErrorInScope.bind($scope));
        });
    };

    $scope.keyIsProjectAdmin = function(key) {
        return key && key.projectPrivileges && key.projectPrivileges.admin;
    };

    $scope.refreshProjectApiKeysList();
});

app.controller("ProjectPermissionsCommonController", function($scope, DataikuAPI, WT1, Notification, $stateParams) {
    DataikuAPI.security.listGroups(false).success(function(allGroups) {
        if (allGroups) {
            allGroups.sort();
        }
        $scope.allGroups = allGroups;
        DataikuAPI.security.listUsers().success(function(data) {
            $scope.allUsers = data;

            $scope.allUsers.sort(function(a, b){
                if (a.displayName < b.displayName) return -1;
                if (a.displayName > b.displayName) return 1;
                return 0;
            });

            $scope.refreshPermissions();
        }).error(setErrorInScope.bind($scope));
    }).error(setErrorInScope.bind($scope));

    const isNewPendingPermissionEmail = (permission, oldPendingEmailPermissionsSet) => {
        return permission && permission.pendingUserEmail && !oldPendingEmailPermissionsSet.has(permission.pendingUserEmail.toLowerCase());
    }

    /* Check whether some pendingUserEmail permissions were added since last save */
    $scope.hasNewEmailPermission = (oldPermissions, newPermissions) => {
        const oldPendingEmailPermissionsSet = oldPermissions ? new Set(oldPermissions.filter(perm => perm && perm.pendingUserEmail).map(perm => perm.pendingUserEmail.toLowerCase())) : new Set();
        return newPermissions && newPermissions.some(
            perm => isNewPendingPermissionEmail(perm, oldPendingEmailPermissionsSet)
        );
    }

    // Send WT1 about the newly created pending email permissions
    $scope.sendWT1ForNewPendingEmailPermissions = (oldPermissions, newPermissions) => {
        const oldPendingEmailPermissionsSet = oldPermissions ? new Set(oldPermissions.filter(perm => perm && perm.pendingUserEmail).map(perm => perm.pendingUserEmail.toLowerCase())) : new Set();
        for (let perm of newPermissions) {
            if (isNewPendingPermissionEmail(perm, oldPendingEmailPermissionsSet)) {
                WT1.tryEvent('sharing-assets-invite-by-email', () => ({
                    "forEmailh": md5(perm.pendingUserEmail.toLowerCase()),
                }));
            }
        }
    };

    /* Display info messages and refresh permissions when email status is updated */
    var deregister = Notification.registerEvent("project-invitation-sent", (_, data) => {
        if ($stateParams.projectKey === data.projectKey) {
            if (!$scope.isDirty()) {
                $scope.refreshPermissions();
            } else {
                // if there are some unsaved changes, just refresh the statuses instead of all permissions
                $scope.getPermissions().filter(perm => perm && perm.pendingUserEmail).forEach((perm) => {
                    if (data.result && Array.isArray(data.result.sentEmails) && data.result.sentEmails.some(sentEmail => sentEmail && sentEmail.toLowerCase() === perm.pendingUserEmail.toLowerCase())) {
                        perm.invitationEmailStatus = "SENT";
                    } else if (data.result && Array.isArray(data.result.failedEmails) && data.result.failedEmails.some(failedEmail => failedEmail && failedEmail.toLowerCase() === perm.pendingUserEmail.toLowerCase())) {
                        perm.invitationEmailStatus = "FAILED";
                    }
                });
            }
        }
    });

    $scope.$on("$destroy", function() {
        deregister();
    });
});

app.controller("ProjectSettingsPermissionsController", function($rootScope, $scope, $stateParams, DataikuAPI, Dialogs,
    $controller, ActivityIndicator, PermissionsService, RequestCenterService, WT1) {

    $scope.ui = {};
    $controller("ProjectPermissionsCommonController", {$scope: $scope});

    $scope.projectVisibilityOptions = [
        { label: 'Discoverable', value: 'ENABLED' },
        { label: 'Private', value: 'DISABLED' },
        { label: 'Inherit global settings (' + ($rootScope.appConfig.projectVisibility.visibilityMode === 'ENABLED_BY_DEFAULT' ? 'Discoverable' : 'Private') + ')', value: 'INHERIT' }
    ];

    $scope.projectRequestAccessOptions = [
        { label: 'Enabled', value: 'ENABLED' },
        { label: 'Disabled', value: 'DISABLED' },
        { label: 'Inherit global settings (' + ($rootScope.appConfig.projectVisibility.accessRequestsMode === 'ENABLED_BY_DEFAULT' ? 'Enabled' : 'Disabled') + ')', value: 'INHERIT' },
    ];

    $scope.isDirty = function() {
        return $scope.projectSettings && !angular.equals($scope.lastProjectSettings, $scope.projectSettings);
    };
    checkChangesBeforeLeaving($scope, $scope.isDirty);

    $scope.lastProjectSettings = {};
    $scope.getPermissions = () => $scope.projectSettings.permissions;
    $scope.refreshPermissions = function() {
        DataikuAPI.projects.getSettings($stateParams.projectKey).success(function(projectSettings) {
            $scope.projectSettings = projectSettings;
            $scope.lastProjectSettings = angular.copy($scope.projectSettings);
            if ($scope.ui) {
                $scope.ui.ownerLogin = projectSettings && projectSettings.owner;
            }
        }).error(setErrorInScope.bind($scope));
    }
    $scope.onPermissionUpgraded = function() {
        $scope.refreshProjectData();
        $scope.refreshPermissions();
        location.reload();
    }

    $scope.save = function() {
        return DataikuAPI.projects.savePermissions($stateParams.projectKey, $scope.projectSettings).success(function(data) {
            if ($scope.lastProjectSettings.settings.limitedVisibilityEnabled !== $scope.projectSettings.settings.limitedVisibilityEnabled) {
                WT1.event('item-accessibility-update', {
                    itemType: 'project-accessibility',
                    accessibility: ({
                    ENABLED: 'limited', DISABLED: 'private', INHERIT: 'inherit-' + ($rootScope.appConfig.projectVisibility.visibilityMode == 'ENABLED_BY_DEFAULT' ? 'limited' : 'private')
                    })[$scope.projectSettings.settings.limitedVisibilityEnabled],
                });
            }
            if ($scope.lastProjectSettings.settings.accessRequestsEnabled !== $scope.projectSettings.settings.accessRequestsEnabled){
                RequestCenterService.WT1Events.onProjectAccessRequestsSettingChanged(
                  $scope.projectSettings.settings.accessRequestsEnabled,
                  $rootScope.appConfig.projectVisibility.accessRequestsMode
                );
            }
            if (data.anyMessage) {
                Dialogs.infoMessagesDisplayOnly($scope, "Permissions update", data);
            }
            if ($rootScope.appConfig.emailChannelId && $scope.hasNewEmailPermission($scope.lastProjectSettings.permissions, $scope.projectSettings.permissions)) {
                ActivityIndicator.success("Saved! Sending invitation emails.");
            } else {
                ActivityIndicator.success("Saved!");
            }
            $scope.sendWT1ForNewPendingEmailPermissions($scope.lastProjectSettings.permissions, $scope.projectSettings.permissions);
            $scope.lastProjectSettings = angular.copy($scope.projectSettings);
        }).error(setErrorInScope.bind($scope));
    };

    // Ownership mgmt
    $scope.$watch("ui.ownerLogin", function() {
        PermissionsService.transferOwnership($scope, $scope.projectSettings, "project");
    });
    $scope.$watch("projectSettings.owner", function(newOwnerLogin, oldOwnerLogin) {
        if (!newOwnerLogin) {
            return;
        }
        // If owner changed, keep individual permissions for the old owner
        if (oldOwnerLogin && oldOwnerLogin !== newOwnerLogin) {
            $scope.projectSettings.permissions = [{user: oldOwnerLogin, admin: true }, ...$scope.projectSettings.permissions];
        }
        // Remove individual permissions for this user, if any
        $scope.projectSettings.permissions = $scope.projectSettings.permissions.filter(p => p != null && p.user !== newOwnerLogin);
    });
    $scope.$watch("allUsers", function() {
        if ($scope.allUsers) {
            $scope.allUsers.forEach(user => user.annotation = '@' + user.login);
        }
    });
});

app.controller("NonAdminProjectPermissionsController", function($rootScope, $scope, $stateParams, DataikuAPI, Dialogs,
    ActivityIndicator, $controller) {

    $controller("ProjectPermissionsCommonController", {$scope: $scope});
    $scope.lastProjectPermissions = {};
    $scope.getPermissions = () => $scope.projectPermissions.permissions;
    $scope.refreshPermissions = () => {
        DataikuAPI.projects.getPermissions($stateParams.projectKey).success(function(projectPermissions) {
            $scope.projectPermissions = projectPermissions;
            $scope.lastProjectPermissions = angular.copy($scope.projectPermissions);
        }).error(setErrorInScope.bind($scope));
    }
    $scope.refreshPermissions();

    $scope.isDirty = () => {
        return $scope.projectPermissions && !angular.equals($scope.lastProjectPermissions, $scope.projectPermissions);
    }
    checkChangesBeforeLeaving($scope, $scope.isDirty);

    $scope.save = function() {
        return DataikuAPI.projects.savePermissionsNonAdmin($stateParams.projectKey, $scope.projectPermissions.permissions).success(function(data) {
            if (data.anyMessage) {
                Dialogs.infoMessagesDisplayOnly($scope, "Permissions update", data);
            }
            if ($rootScope.appConfig.emailChannelId && $scope.hasNewEmailPermission($scope.lastProjectPermissions.permissions, $scope.projectPermissions.permissions)) {
                ActivityIndicator.success("Saved! Sending invitation emails.");
            } else {
                ActivityIndicator.success("Saved!");
            }
            $scope.sendWT1ForNewPendingEmailPermissions($scope.lastProjectPermissions.permissions, $scope.projectPermissions.permissions);
            $scope.lastProjectPermissions = angular.copy($scope.projectPermissions);
        }).error(setErrorInScope.bind($scope));
    };
});

app.component('projectSecurityPermissions', {
    templateUrl: "/templates/projects/settings/project-security-permissions.html",
    bindings: {
        owner: '<',
        permissions: '<',
        projectSummary: '<',
        hasNewEmailPermission: '<',
        allUsers: '<',
        allGroups: '<'
    },
    controller: function($scope, $rootScope, $state, ActivityIndicator, $stateParams, DataikuAPI) {
        const $ctrl = this;
        $ctrl.$state = $state;
        const invitationStatusesByEmail = new Map();

        $ctrl.$onInit = () => {
            $scope.$watch("$ctrl.permissions", function () {
                updatePermissionsWithOwner();
                updateUnassignedGroupsAndUsers();
                $scope.handleImpliedPermissions();
            }, true);
        }

        $ctrl.resendProjectInvitationEmail = (perm) => {
            perm.$resendEmailDisabled = true;
            DataikuAPI.projects.resendProjectInvitationEmail($ctrl.projectSummary.projectKey, perm.pendingUserEmail).then(({data}) => {
                // if there are some failed emails but no info message (i.e. when there is no email channel), display a notification
                if (!data.infoMessages.anyMessage && data.failedEmails.length) {
                    ActivityIndicator.error("Failed to resend invitation email");
                }
            }).catch(setErrorInScope.bind($scope.$parent))
            .finally(() => delete perm.$resendEmailDisabled);
        }

        $ctrl.$onChanges = (changes) => {
            if (changes.permissions || changes.owner) {
                $scope.handleImpliedPermissions(); // needed in case permissions are changed to the same values, in which case it's not detected by the watcher
            }
            if (changes.permissions) {
                if ($ctrl.permissions) {
                    // store the original invitation statuses for each email so email permissions don't stay pending when we delete and re-add them. If it's a permission without status (pre 13.5) we mark it as SENT.
                    $ctrl.permissions.filter(perm => perm.pendingUserEmail).forEach(perm => invitationStatusesByEmail.set(perm.pendingUserEmail.toLowerCase(), perm.invitationEmailStatus || 'SENT'));
                }
            }
            if (changes.permissions || changes.allUsers || changes.allGroups || changes.owner) {
                updatePermissionsWithOwner();
                updateUnassignedGroupsAndUsers();
                $scope.handleImpliedPermissions();
            }
        }

        const buildDefaultPermissions = () => {
            return {
                editPermissions: true,
                writeProjectContent: $ctrl.projectSummary && $ctrl.projectSummary.canWriteProjectContent,
                readProjectContent: $ctrl.projectSummary && $ctrl.projectSummary.canReadProjectContent,
                shareToWorkspaces: $ctrl.projectSummary && $ctrl.projectSummary.canShareToWorkspaces,
                publishToDataCollections: $ctrl.projectSummary && $ctrl.projectSummary.canPublishToDataCollections,
                exportDatasetsData: $ctrl.projectSummary && $ctrl.projectSummary.canExportDatasetsData
            }
        };

        $scope.onUsersAdded = (usersLogin) => {
            $ctrl.permissions.push(...usersLogin.map(user => ({... buildDefaultPermissions(), user: user })));
            $scope.$apply();
        }

        $scope.onGroupsAdded = (groups) => {
            $ctrl.permissions.push(...groups.map(group => ({... buildDefaultPermissions(), group })));
            $scope.$apply();
        }

        $scope.removePermission = (index) => {
            $ctrl.permissions.splice(index, 1);
        }

        $ctrl.hasMoreRights = (permission) => {
            if ($ctrl.projectSummary.isProjectAdmin) {
                return false;
            }
            // keep in sync with permissions
            if (permission.admin ||
                (permission.editPermissions && !$ctrl.projectSummary.canEditPermissions) ||
                (permission.writeProjectContent && !$ctrl.projectSummary.canWriteProjectContent) ||
                (permission.readProjectContent && !$ctrl.projectSummary.canReadProjectContent) ||
                (permission.executeApp && !$ctrl.projectSummary.canExecuteApp) ||
                (permission.publishToDataCollections && !$ctrl.projectSummary.canPublishToDataCollections) ||
                (permission.shareToWorkspaces && !$ctrl.projectSummary.canShareToWorkspaces) ||
                (permission.exportDatasetsData && !$ctrl.projectSummary.canExportDatasetsData) ||
                (permission.readDashboards && !$ctrl.projectSummary.canReadDashboards) ||
                (permission.writeDashboards && !$ctrl.projectSummary.canWriteDashboards) ||
                (permission.moderateDashboards && !$ctrl.projectSummary.canModerateDashboards) ||
                (permission.runScenarios && !$ctrl.projectSummary.canRunScenarios) ||
                (permission.manageDashboardAuthorizations && !$ctrl.projectSummary.canManageDashboardAuthorizations) ||
                (permission.manageExposedElements && !$ctrl.projectSummary.canManageExposedElements) ||
                (permission.manageAdditionalDashboardUsers && !$ctrl.projectSummary.canManageAdditionalDashboardUsers)
            ) {
                return true;
            }
            return false;
        }

        $scope.getUserDisplayName = function(login) {
            const user = $ctrl.allUsers && $ctrl.allUsers.find(u => u.login === login);
            // If we do not find the login in the list of users, it is either because the user is disabled, deleted or the current user is restricted from seeing them.
            return user ? user.displayName : null;
        }

        /** Display the owner first as an individual permission in the list (only visual and not saved in the backend) **/
        const updatePermissionsWithOwner = () => {
            if ($ctrl.owner) {
                $ctrl.permissionsWithOwner = [{ user: $ctrl.owner, admin: true, $adminDisabled: true }, ...$ctrl.permissions];
            } else {
                $ctrl.permissionsWithOwner = $ctrl.permissions;
            }
        }

        const updateUnassignedGroupsAndUsers = () => {
            if (!$ctrl.permissions || !$ctrl.allUsers || !$ctrl.allGroups) {
                return;
            }
            $scope.unassignedGroups = $ctrl.allGroups.filter(groupName => !$ctrl.permissions.some(perm => perm != null && perm.group === groupName));
            $scope.unassignedUsers = $ctrl.allUsers.filter(user => user.login !== $ctrl.owner && !$ctrl.permissions.some(perm => perm != null && perm.user === user.login));
        }

        $scope.handleImpliedPermissions = () => {
            if (!$ctrl.permissionsWithOwner) {
                return;
            }

            /* Handle implied permissions
                Keep in sync with PermissionsService::makeEffective
            */
            $ctrl.permissionsWithOwner.forEach((p) => {
                if (p == null) {
                    return;
                }
                p.$adminDisabled = p.user === $ctrl.owner;
                p.$readProjectContentDisabled = false;
                p.$editPermissionsDisabled = false;
                p.$writeProjectContentDisabled = false;
                p.$publishToDataCollectionsDisabled = false;
                p.$shareToWorkspacesDisabled = false;
                p.$exportDatasetsDataDisabled = false;
                p.$readDashboardsDisabled = false;
                p.$writeDashboardsDisabled = false;
                p.$moderateDashboardsDisabled = false;
                p.$runScenariosDisabled = false;
                p.$manageDashboardAuthorizationsDisabled = false;
                p.$manageExposedElementsDisabled = false;
                p.$manageAdditionalDashboardUsersDisabled = false;
                p.$executeAppDisabled = false;

                if (p.admin) {
                    p.$editPermissionsDisabled = true;
                    p.$readProjectContentDisabled = true;
                    p.$writeProjectContentDisabled = true;
                    p.$publishToDataCollectionsDisabled = true;
                    p.$shareToWorkspacesDisabled = true;
                    p.$exportDatasetsDataDisabled = true;
                    p.$readDashboardsDisabled = true;
                    p.$writeDashboardsDisabled = true;
                    p.$moderateDashboardsDisabled = true;
                    p.$runScenariosDisabled = true;
                    p.$manageDashboardAuthorizationsDisabled = true;
                    p.$manageExposedElementsDisabled = true;
                    p.$manageAdditionalDashboardUsersDisabled = true;
                    p.$executeAppDisabled = true;
                }
                if (p.editPermissions) {
                    p.$readDashboardsDisabled = true;
                }
                if (p.writeProjectContent) {
                    p.$readProjectContentDisabled = true;
                    p.$readDashboardsDisabled = true;
                    p.$writeDashboardsDisabled = true;
                    p.$moderateDashboardsDisabled = true;
                    p.$runScenariosDisabled = true;
                    p.$executeAppDisabled = true;
                }
                if (p.shareToWorkspaces || p.publishToDataCollections) {
                    p.$readProjectContentDisabled = true;
                    p.$readDashboardsDisabled = true;
                    p.$manageDashboardAuthorizationsDisabled = true;
                    p.$executeAppDisabled = true;
                }
                if (p.moderateDashboards) {
                    p.$readDashboardsDisabled = true;
                    p.$writeDashboardsDisabled = true;
                }
                if (p.readProjectContent) {
                    p.$readDashboardsDisabled = true;
                    p.$executeAppDisabled = true;
                }
                if (p.writeDashboards) {
                    p.$readDashboardsDisabled = true;
                }
            });

        };

        $scope.areEmailPermissionsPending = () => {
            return $ctrl.permissions && $ctrl.permissions.some(perm => perm != null && perm.pendingUserEmail);
        }

        $ctrl.canResendInvitationEmails = () => {
            return $ctrl.permissions && $ctrl.permissions.some(perm => perm != null && perm.pendingUserEmail && perm.invitationEmailStatus !== 'PENDING');
        }

        $scope.inviteUserByEmail = (pendingEmail) => {
            if (!pendingEmail || !$rootScope.appConfig.permissionsByEmailEnabled) return;
            pendingEmail = pendingEmail.toLowerCase();

            const addExistingUsersIfNotPresent = (logins) => {
                let permissionAdded = false;
                logins.forEach(login => {
                    const matchingPermission = $ctrl.permissions.find(perm => perm != null && perm.user === login) || login === $ctrl.owner;
                    if (!matchingPermission) {
                        permissionAdded = true;
                        // found an existing user with that email, adding it to the project
                        const permissionItem = buildDefaultPermissions();
                        permissionItem.user = login;
                        $ctrl.permissions.push(permissionItem);
                    };
                });

                if (permissionAdded) {
                    // users were added to project permissions, save
                    ActivityIndicator.success("Added users with matching email to project permissions.");
                } else {
                    // users are already in project permissions, do nothing
                    ActivityIndicator.warning("Users with matching email already have access to this project");
                }
            };

            const addPendingEmailIfNotPresent = (pendingEmail) => {
                const existingPendingEmail = $ctrl.permissions.some(perm => perm != null && perm.pendingUserEmail && perm.pendingUserEmail.toLowerCase() === pendingEmail);
                if (existingPendingEmail) {
                    ActivityIndicator.warning("An invitation is already pending for this email");
                } else {
                    const permissionItem = buildDefaultPermissions();
                    permissionItem.pendingUserEmail = pendingEmail;
                    permissionItem.invitationEmailStatus = invitationStatusesByEmail.get(permissionItem.pendingUserEmail) || "PENDING";  // reuse the status if it's a permission that was removed and added back
                    $ctrl.permissions.push(permissionItem);
                }
            };

            // check if there are any users with an email or login matching this email
            DataikuAPI.security.listUsersMatchingEmail($ctrl.projectSummary.projectKey, null, pendingEmail).success((matchingUserLogins) => {
                if (matchingUserLogins.length) {
                    addExistingUsersIfNotPresent(matchingUserLogins);
                } else {
                    addPendingEmailIfNotPresent(pendingEmail);
                }
            }).error(setErrorInScope.bind($scope.$parent));
            $("#permissionByEmailEnabled").val("").trigger("change"); // clear email input and trigger change
        };
    }
});


app.controller("EditProjectAPIKeyModalController", function($scope, DataikuAPI, CreateModalFromTemplate, ClipboardUtils) {
    $scope.create = function(){
        DataikuAPI.projects.publicApi.createProjectApiKey($scope.apiKey).success(function(data){
            CreateModalFromTemplate("/templates/admin/security/new-api-key-modal.html", $scope, null, function(newScope) {
                newScope.hashedApiKeysEnabled = $scope.appConfig.hashedApiKeysEnabled;
                newScope.key = data;

                newScope.copyKeyToClipboard = function() {
                    ClipboardUtils.copyToClipboard(data.key, 'Copied to clipboard.');
                };

                newScope.viewQRCode = function() {
                    CreateModalFromTemplate("/templates/admin/security/api-key-qrcode-modal.html", $scope, null, function (newScope) {
                        newScope.apiKeyQRCode = JSON.stringify({
                            k : data.key,
                            u : $scope.appConfig.dssExternalURL
                        });
                    });
                };

                newScope.$on("$destroy",function() {
                    $scope.dismiss();
                    $scope.refreshProjectApiKeysList();
                });
            });
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function(){
        DataikuAPI.projects.publicApi.saveProjectApiKey($scope.apiKey).success(function(data){
            $scope.dismiss();
            $scope.refreshProjectApiKeysList();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.makeAdmin = function(key) {
        if (angular.isObject(key.projectPrivileges)) {
            key.projectPrivileges.admin = true;
            key.projectPrivileges = angular.copy(key.projectPrivileges);
        } else {
            key.projectPrivileges = {'admin': true};
        }
        key.localDatasets = [];
    };
});


app.factory("ProjectSettingsObjectsListService", function(DataikuAPI, $stateParams, Fn) {
    var svc = {
        addOnScope: function($scope) {
            var projectKey = $stateParams.projectKey;

            function idsAndNames(data) {
                return data.map(function(x){return [x.id, x.name]});
            }

            DataikuAPI.savedmodels.list(projectKey).success(function(data){
                $scope.savedModels = idsAndNames(data);
            });
            DataikuAPI.modelevaluationstores.list(projectKey).success(function(data){
                $scope.modelEvaluationStores = idsAndNames(data);
            });
            DataikuAPI.managedfolder.list(projectKey).success(function(data){
                $scope.managedFolders = idsAndNames(data);
            });
            DataikuAPI.webapps.list(projectKey).success(function(data){
                $scope.webApps = idsAndNames(data);
            });
            DataikuAPI.reports.list(projectKey).success(function(data){
                $scope.reports = idsAndNames(data);
            });

            DataikuAPI.datasets.listNames(projectKey).success(function(data){
                $scope.datasetNames = data;
            });

            DataikuAPI.jupyterNotebooks.listHeads(projectKey, {}).success(function(data){
                $scope.jupyterNotebooks = data.items.map(Fn.prop('name'));
            });

            DataikuAPI.projects.list().success(function(data) {
                $scope.projectsList = data;
            }).error(setErrorInScope.bind($scope));
        }
    }
    return svc;
})

app.controller("ProjectSettingsExposedController", function($rootScope, $scope, $state, $stateParams, $filter, $timeout,
               DataikuAPI, ActivityIndicator, FLOW_EXPOSABLE_TYPES, RequestCenterService, QuickSharingWT1EventsService,
               MLUtilsService, AccessibleObjectsCacheService) {

    $scope.exposableTypes = FLOW_EXPOSABLE_TYPES;

    // object links
    $scope.openObject = function(object) {
        switch (object.type) {
            case 'DATASET':
                $state.go('projects.project.datasets.dataset.explore',{datasetName: object.localName});
                break;
            case 'SAVED_MODEL':
                $state.go('projects.project.savedmodels.savedmodel.versions',{smId: object.localName});
                break;
            case 'MODEL_EVALUATION_STORE':
                $state.go('projects.project.modelevaluationstores.modelevaluationstore.evaluations',{mesId: object.localName});
                break;
            case 'MANAGED_FOLDER':
                $state.go('projects.project.managedfolders.managedfolder.view',{odbId: object.localName});
                break;
            case 'JUPYTER_NOTEBOOK':
                $state.go('projects.project.notebooks.jupyter_notebook',{notebookId: object.localName});
                break;
            case 'WEB_APP':
                $state.go('projects.project.webapps.webapp.view',{webAppId: object.localName, webAppName:$filter('slugify')(object.displayName)});
                break;
            case 'REPORT':
                $state.go('projects.project.reports.report.view',{reportId: object.localName});
                break;
            case 'SCENARIO':
                $state.go('projects.project.scenarios.scenario.steps',{scenarioId: object.localName});
                break;
        }
    };

    function removeItemFromArray(item, arr) {
        var idx = arr.indexOf(item);
        if (idx > -1) {
            arr.splice(idx, 1);
        }
    }

    $scope.uiState = $scope.uiState || {};
    $scope.uiState.view = 'objects';

    $scope.projectSharingRequestsOptions = [
        { n: 'Enabled', v: 'ENABLED' },
        { n: 'Disabled', v: 'DISABLED' },
        { n: 'Inherit global settings (' + ($rootScope.appConfig.objectSharingRequestsMode === 'ENABLED_BY_DEFAULT' ? 'Enabled' : 'Disabled') + ')', v: 'INHERIT' },
    ];

    $scope.newSource = {};
    $scope.projects = [];

    // Global indices to disable existing objects / projects when adding to list (per-object and per-project indices are stored on the object/project itself)
    $scope.projectsIndex = {};
    $scope.objectsIndex = {};
    $scope.quickSharedElementIndex = {};

    if ($stateParams.projectKey) {
        $scope.projectsIndex[$stateParams.projectKey] = true;
    }

    // Cache for object-picker objects
    $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', setErrorInScope.bind($scope));

    $scope.isDirty = function() {
        return !angular.equals($scope.exposedObjects, $scope.origExposedObjects);
    };

    checkChangesBeforeLeaving($scope, $scope.isDirty);

    function loadSettings() {
        // Exposed objects is the old name for Shared objects
        DataikuAPI.projects.getEnrichedExposedObjects($stateParams.projectKey, true).success(function(exposedObjects) {
            $scope.exposedObjects = exposedObjects;
            $scope.origExposedObjects = dkuDeepCopy($scope.exposedObjects, function(key) { return !key.startsWith('$'); });

            $scope.exposedObjects.objects.forEach(function(object) {
                object.$open = object.rules.length === 0 && !($scope.appConfig.quickSharingElementsEnabled && object.quickSharingEnabled);
                for (var i = 0; i < object.rules.length; i++) {

                    var project = object.rules[i];
                    if (!$scope.projectsIndex[project.targetProject]) {
                        $scope.projectsIndex[project.targetProject] = project;
                        $scope.projects.push(project);
                    } else {
                        project = $scope.projectsIndex[project.targetProject];
                        object.rules[i] = project;
                    }

                    $scope.getProjectKeysForObject(object)[project.targetProject] = true;
                    $scope.getObjectsForProject(project).push(object);
                    $scope.getObjectIdsForProject(project, object.type)[object.localName] = true;
                }
                $scope.getObjectsIndex(object.type)[object.localName] = object;
            });
            $scope.projects.forEach(function(project){
                project.$open = project.$exposedObjects.length === 0;
            });
            refreshHasObjectsSharedToDeletedProjects();
        }).error(setErrorInScope.bind($scope));
    }

    loadSettings();

    const saveExposedObjects = function(msg) {
        var copy = dkuDeepCopy($scope.exposedObjects, function(key) {
            return !key.startsWith('$');
        });
        DataikuAPI.projects.saveExposedObjects($stateParams.projectKey, $scope.exposedObjects).success(function() {
            if($scope.origExposedObjects.sharingRequestsEnabled !== $scope.exposedObjects.sharingRequestsEnabled) {
                RequestCenterService.WT1Events.onObjectAccessRequestsSettingChanged(
                  $scope.exposedObjects.sharingRequestsEnabled,
                  $rootScope.appConfig.objectSharingRequestsMode
                );
            }

            const added = copy.objects.filter(newObject =>
                !$scope.origExposedObjects.objects.find(oldObject => newObject.type === oldObject.type && newObject.localName === oldObject.localName)
            );

            const changed = [];
            const removed = [];
            $scope.origExposedObjects.objects.forEach((oldObject) => {
                const newObject = copy.objects.find(newObject => newObject.type === oldObject.type && newObject.localName === oldObject.localName);
                if (!newObject) {
                    removed.push(oldObject);
                } else if (!angular.equals(oldObject, newObject)) {
                    changed.push({oldObject, newObject});
                }
            });
            QuickSharingWT1EventsService.onObjectsAdded($stateParams.projectKey, added);
            QuickSharingWT1EventsService.onObjectsRemoved($stateParams.projectKey, removed);
            changed.forEach((change) => {
                QuickSharingWT1EventsService.onObjectChanged($stateParams.projectKey, change.oldObject, change.newObject);
            });

            $scope.origExposedObjects = copy;
            refreshHasObjectsSharedToDeletedProjects();
            ActivityIndicator.success(msg);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.saveAndMaybePerformChanges = function() {
        if (!angular.equals($scope.exposedObjects, $scope.origExposedObjects)) {
            saveExposedObjects("Shared objects saved");
        }
    };

    function scrollToBottom(containerId) {
        $timeout(function() {
            const element = $(containerId);
            if (element[0]) {
                element[0].scrollTop = element[0].scrollHeight;
            }
        });
    }

    function scrollObjectsToBottom() {
        scrollToBottom('#objects-scrollable-container');
    }

    function scrollProjectsToBottom() {
        scrollToBottom('#projects-scrollable-container');
    }

    function refreshHasObjectsSharedToDeletedProjects() {
        $scope.hasObjectsSharedToDeletedProjects = $scope.projects.some(project => project.isProjectDeleted);
    }

    $scope.$watch('newSource.type', function(nv, ov) {
        if (nv && nv !== ov) {
            if ($scope.uiState.view === 'objects') {
                scrollObjectsToBottom();
            } else if ($scope.uiState.view === 'projects') {
                scrollProjectsToBottom();
            }
        }
    });

    $scope.addObject = function(newObject) {
        if (!newObject) return
        var object = {
            type: newObject.type,
            localName: newObject.id,
            displayName: newObject.label,
            rules: [],
            $open: true
        };
        if (object.type === "SAVED_MODEL") {
            const miniTask = newObject.object.miniTask || {};
            object.savedModelMLCategory = MLUtilsService.getMLCategory(
                miniTask.taskType,
                miniTask.backendType,
                miniTask.predictionType,
                newObject.object.savedModelType,
                (newObject.object.proxyModelConfiguration || {}).protocol
            );
        }
        $scope.exposedObjects.objects.push(object);
        $scope.getObjectsIndex(object.type)[object.localName] = object;
        scrollObjectsToBottom();
    };

    $scope.addProject = function(newProject) {
        var project = {
            targetProject : newProject.id,
            targetProjectDisplayName: newProject.label,
            appearOnFlow : true,
            isProjectDeleted: false,
            $exposedObjects: [],
            $open: true
        };

        $scope.projects.push(project);
        $scope.projectsIndex[project.targetProject] = project;
        scrollProjectsToBottom();
    };

    $scope.removeObject = function(object) {
        // Remove from every project
        const projectsForObject = [...$scope.getProjectsForObject(object)]; // Duplicate the array to avoid side effect when removing rules
        projectsForObject.forEach(function(project) {
            $scope.removeRule(project, object);
            if (!$scope.getObjectsForProject(project).length) {
                $scope.removeProject(project);
            }
        });
        removeItemFromArray(object, $scope.exposedObjects.objects);
        delete $scope.getObjectsIndex(object.type)[object.localName];
    };

    $scope.removeProject = function(project) {
        // Remove from every object
        const objectsForProject = [...$scope.getObjectsForProject(project)]; // Duplicate the array to avoid side effect when removing rules
        objectsForProject.forEach(function(object) {
            $scope.removeRule(project, object);
            if (!object.quickSharingEnabled && $scope.getProjectsForObject(object).length === 0) {
                $scope.removeObject(object);
            }
        });

        removeItemFromArray(project, $scope.projects);
        delete $scope.projectsIndex[project.targetProject];
        refreshHasObjectsSharedToDeletedProjects();
    };

    $scope.removeSharingToDeletedProjects = () => {
        for(const project of $scope.projects) {
            if (project.isProjectDeleted) {
                $scope.removeProject(project);
            }
        }
    }

    $scope.addObjectToProject = function(newObject, project) {
        if (!$scope.getObjectsIndex(newObject.type)[newObject.id]) {
            $scope.addObject(newObject);
        }

        var object = $scope.getObjectsIndex(newObject.type)[newObject.id];
        $scope.addRule(project, object);
    };

    $scope.addProjectToObject = function(newProject, object) {
        if (!$scope.projectsIndex[newProject.id]) {
            $scope.addProject(newProject);
        }

        var project = $scope.projectsIndex[newProject.id];
        $scope.addRule(project, object);
    };

    $scope.addRule = function(project, object) {
        $scope.getProjectsForObject(object).push(project);
        $scope.getProjectKeysForObject(object)[project.targetProject] = true;
        $scope.getObjectsForProject(project).push(object);
        $scope.getObjectIdsForProject(project, object.type)[object.localName] = true;
    };

    $scope.removeRule = function(project, object) {
        removeItemFromArray(project, $scope.getProjectsForObject(object));
        removeItemFromArray(object, $scope.getObjectsForProject(project));

        delete $scope.getProjectKeysForObject(object)[project.targetProject];
        delete $scope.getObjectIdsForProject(project, object.type)[object.localName];
    };

    // Getters with empty defaults for object lists / maps

    $scope.getProjectsForObject = function(object) {
        if (!object.rules) {
            object.rules = [];
        }
        return object.rules;
    };

    $scope.getProjectKeysForObject = function(object) {
        if (!object.$targetProjectKeys) {
            object.$targetProjectKeys = {};
            if ($stateParams.projectKey) {
                object.$targetProjectKeys[$stateParams.projectKey] = true;
            }
        }
        return object.$targetProjectKeys;
    };

    $scope.getObjectsForProject = function(project) {
        if (!project.$exposedObjects) {
            project.$exposedObjects = [];
        }
        return project.$exposedObjects;
    };

    $scope.getObjectIdsForProject = function(project, objectType) {
        if (!project.$exposedObjectIds) {
            project.$exposedObjectIds = {};
        }
        if (!project.$exposedObjectIds[objectType]) {
            project.$exposedObjectIds[objectType] = {};
        }
        return project.$exposedObjectIds[objectType];
    };

    $scope.getObjectsIndex = function(objectType) {
        if (!$scope.objectsIndex[objectType]) {
            $scope.objectsIndex[objectType] = {};
        }
        return $scope.objectsIndex[objectType];
    };
});

app.controller("ProjectSettingsDashboardController", function($scope, $stateParams, $timeout,
        DataikuAPI, ActivityIndicator, SmartId, ProjectSettingsObjectsListService, ALL_EXPOSABLE_TYPES, MLUtilsService) {

    ProjectSettingsObjectsListService.addOnScope($scope);

    $scope.exposableTypes = ALL_EXPOSABLE_TYPES;

    $scope.allModes = ['RUN', 'DISCOVER', 'READ', 'WRITE']; // warning, order is important for UI
    $scope.allModeIcons = { // TODO @datacollections use better icons
        'DISCOVER': 'icon-compass',
        'READ': 'icon-search',
        'RUN': 'icon-play',
        'WRITE': 'icon-pencil',
    }
    $scope.availableModesForType = {
        'DATASET':  ['DISCOVER', 'READ', 'WRITE'],
        'SCENARIO': ['READ', 'RUN']
    };
    const getAvailableModesForType = (type) => ($scope.availableModesForType[type] || []);


    function loadSettings() {
        DataikuAPI.projects.getDashboardAuthorizations($stateParams.projectKey, true).success(function(authorizations) {

            $scope.dashboardAuthorizations = authorizations;
            $scope.initialDashboardAuthorizations = angular.copy($scope.dashboardAuthorizations);

            $scope.readerAuthorizationsByType = {};
            $scope.dashboardAuthorizations.authorizations.forEach(function(ref) {
                if (!$scope.readerAuthorizationsByType[ref.objectRef.objectType]) {
                    $scope.readerAuthorizationsByType[ref.objectRef.objectType] = [SmartId.fromRef(ref.objectRef)];
                } else {
                    $scope.readerAuthorizationsByType[ref.objectRef.objectType].push(SmartId.fromRef(ref.objectRef));
                }
            });
            $timeout(function(){
                $scope.$broadcast('redrawFatTable')
            });
        }).error(setErrorInScope.bind($scope));
    }

    loadSettings();

    // Go through all authorization, and ensures that every mode that is implied by a checked mode is checked.
    function fixupImpliedModes(authorizations) {
        authorizations.forEach(o => {
            const availableModes = getAvailableModesForType(o.objectRef.objectType);
            availableModes.forEach(mode => {
                if(!o.modes.includes(mode) && $scope.isImplicitlyChecked(mode, o.modes)) {
                    o.modes.push(mode);
                }
            })
        });
    }

    $scope.saveAuthorizations = function() {
        fixupImpliedModes($scope.dashboardAuthorizations.authorizations);
        DataikuAPI.projects.saveDashboardAuthorizations($stateParams.projectKey, $scope.dashboardAuthorizations).success(function(){
            ActivityIndicator.success("Saved!");
            $scope.initialDashboardAuthorizations = angular.copy($scope.dashboardAuthorizations);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.newSource = {
        modes: ['READ'],
        type: null
    };

    $scope.isDashboardAuthorizationDirty = function() {
        return !angular.equals($scope.dashboardAuthorizations, $scope.initialDashboardAuthorizations);
    }
    checkChangesBeforeLeaving($scope, $scope.isDashboardAuthorizationDirty);

    $scope.addReaderAuthorization = function(object) {
        if ($scope.dashboardAuthorizations.allAuthorized || !object || !object.id
            || ($scope.readerAuthorizationsByType[object.type] || []).indexOf(object.smartId) != -1) {
            return;
        }

        var readerAuth = {
            modes: angular.copy($scope.newSource.modes),
            objectRef: {
                objectId: object.id,
                objectType: object.type,
                objectDisplayName: object.label
            }
        };

        if (object.type === 'SAVED_MODEL') {
            const miniTask = object.object.miniTask || {};
            readerAuth.objectRef.savedModelMLCategory = MLUtilsService.getMLCategory(
                miniTask.taskType,
                miniTask.backendType,
                miniTask.predictionType,
                object.object.savedModelType,
                (object.object.proxyModelConfiguration || {}).protocol
            );
        }

        if (!object.localProject) {
            readerAuth.objectRef.projectKey = object.projectKey;
        }

        $scope.dashboardAuthorizations.authorizations.push(readerAuth);

        if (!$scope.readerAuthorizationsByType[readerAuth.objectRef.objectType]) {
            $scope.readerAuthorizationsByType[readerAuth.objectRef.objectType] = [SmartId.fromRef(readerAuth.objectRef)];
        } else {
            $scope.readerAuthorizationsByType[readerAuth.objectRef.objectType].push(SmartId.fromRef(readerAuth.objectRef));
        }

        $scope.clearFilters();
        $timeout(function(){
            $scope.$broadcast('redrawFatTable');
            $scope.$broadcast('scrollToLine', -1);
        },10); // wait for clearfilters
    };

    $scope.removeReaderAuthorization = function(ref) {
        var idx1 = $scope.dashboardAuthorizations.authorizations.indexOf(ref);
        $scope.dashboardAuthorizations.authorizations.splice(idx1, 1);
        var idx2 = $scope.readerAuthorizationsByType[ref.objectRef.objectType].indexOf(SmartId.fromRef(ref.objectRef));
        if (idx2 > -1) {
            $scope.readerAuthorizationsByType[ref.objectRef.objectType].splice(idx2, 1);
        }
    };

    $scope.removeReaderAuthorizations = function(refs) {
        refs.forEach(function(ref) {
            $scope.removeReaderAuthorization(ref);
        })
    };

    $scope.getReaderAuthUrl = function(readerAuth) {
        if (readerAuth&&$stateParams.projectKey) {
            return $scope.$root.StateUtils.href.dssObject(readerAuth.objectRef.objectType, SmartId.fromRef(readerAuth.objectRef))
        } else {
            return "";
        }
    };

    $scope.canMassEnableReaderAuth = function(selection, mode) {
        if (!selection || !selection.selectedObjects) {return false;}
        return selection.selectedObjects.some((o) => {
            return getAvailableModesForType(o.objectRef.objectType).includes(mode)
                && !o.modes.includes(mode) && !$scope.isImplicitlyChecked(mode, o.modes);
        });
    };

    $scope.canMassDisableReaderAuth = function(selection, mode) {
        if (!selection || !selection.selectedObjects) {return false;}
        return selection.selectedObjects.some((o) => {
            return getAvailableModesForType(o.objectRef.objectType).includes(mode)
                && o.modes.includes(mode) && !$scope.isImplicitlyChecked(mode, o.modes);
        });
    };

    $scope.setReaderAuth = function(objects, val, mode) {
        objects.forEach(function(o){
            if (($scope.availableModesForType[o.objectRef.objectType] || []).indexOf(mode) > -1) {
                if (val&&o.modes.indexOf(mode)===-1) {
                    o.modes.push(mode);
                }
                if (!val&&o.modes.indexOf(mode)>-1) {
                    o.modes.splice(o.modes.indexOf(mode),1);
                }
            }
        });
    };

    $scope.isImplicitlyChecked = (mode, activeModes) => {
        switch (mode) {
            case 'DISCOVER':
                return activeModes.includes('READ') || activeModes.includes('WRITE');
            case 'READ':
                return activeModes.includes('WRITE');
            default:
                return false;
        }
    }

    $scope.setNewSourceType = function(sourceType) {
        $scope.newSourceType = sourceType;
        this.hidePopover();
    }

    $scope.$watch("newSource.type", function(nv) {
        if (!nv) return;
        $scope.newSource.modes = ['READ'];
        if(getAvailableModesForType(nv).includes('DISCOVER')) {
            $scope.newSource.modes.push('DISCOVER');
        }
    });
});

app.controller("ProjectSettingsDashboardUsersController", function($scope, $state, $stateParams,$timeout, DataikuAPI, ActivityIndicator) {

    function loadSettings() {
        DataikuAPI.projects.getAdditionalDashboardUsers($stateParams.projectKey).success(function(data) {
            $scope.additionalDashboardUsers = data;
            $scope.initialAdditionalDashboardUsers = angular.copy($scope.additionalDashboardUsers);
            $timeout(function(){
                $scope.$broadcast('redrawFatTable')
            });
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.security.listUsers().success(function(data) {
            $scope.userLogins = data.map(function(x) { return x.login });
        }).error(setErrorInScope.bind($scope));
    }

    loadSettings();

    $scope.save = function() {
        DataikuAPI.projects.saveAdditionalDashboardUsers($stateParams.projectKey, $scope.additionalDashboardUsers).success(function(){
            ActivityIndicator.success("Saved!");
            $scope.initialAdditionalDashboardUsers = angular.copy($scope.additionalDashboardUsers);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.isDirty = function() {
        return !angular.equals($scope.additionalDashboardUsers, $scope.initialAdditionalDashboardUsers);
    }
    checkChangesBeforeLeaving($scope, $scope.isDirty);

    $scope.add = function() {
        $scope.additionalDashboardUsers.users.push({
            login: ""
        });
        $scope.clearFilters();
        $timeout(function(){
            $scope.$broadcast('redrawFatTable');
            $scope.$broadcast('scrollToLine', -1);
        },10); // wait for clearfilters
    };

    function removeOne(user) {
        var idx = $scope.additionalDashboardUsers.users.indexOf(user);
        if (idx >= 0) {
            $scope.additionalDashboardUsers.users.splice(idx, 1);
        }
    }

    $scope.remove = function(users) {
        users.forEach(removeOne);
    }

    $scope.onPermissionUpgraded = () => {
        $scope.refreshProjectData();
        $state.go('projects.project.security', { selectedTab: 'permissions' });
    };
});


// expects dates corresponding to UTC days
app.directive("weekDayHeatmap", function(Fn){
    function previousSunday(date) { return d3.time.week.utc(date); }
    function niceDate(date, year) {
        return date.toUTCString().replace(/^\w+,? 0?(\d+) (\w+) (\d+) .+$/,
            year ? '$2 $1, $3' : '$2 $1');
    }

    function monthTickFormat(date) {
        var endOfWeek = new Date(date.getTime() + 7*24*60*60*1000);
        var range = d3.time.days.utc(date, endOfWeek);
        for (var i = 0; i < range.length; i++) {
            if (range[i].getDate() == 1) {
                return range[i].toUTCString().replace(/^\w+,? 0?(\d+) (\w+) (\d+) .+$/, '$2').toUpperCase();
            }
        }
        return '';
    }

    return {
        scope: { data: '=', formatter: '=?', light: '=?'},
        require: ['?svgTitles'],
        link: function(scope, element, attrs, controllers) {
            var itemSize = 16, // ajusted for project summary (1 year w/o scroll @ 925px container)
                gap = 2,
                cellSize = itemSize - gap,
                margin = scope.light ? {top:15,right:0,bottom:0,left:0} : {top:40,right:20,bottom:20,left:30},
                data = [],
                weeks = 0,
                dayScale = d3.scale.ordinal().domain([0, 1, 2, 3, 4, 5, 6])
                    .rangeBands([0, 7 * itemSize - gap], gap / (7 * itemSize - gap), 0),
                weekScale = d3.scale.ordinal(),
                heatScale = d3.scale.linear().range(['#bbdefb', '#0a69b5']),
                svg = d3.select(element.get(0)),
                svgTitles = controllers[0],
                minWidth = 140;

            scope.$watch('data', function() {
                svg.selectAll('g').remove();
                data = scope.data && scope.data.length ? scope.data.concat() : [{date: new Date(), value:0}];
                data.sort(function(a, b) { return Fn.CMP(a.date, b.date); });

                weeks = d3.time.weeks.utc(data[0].date, data[data.length-1].date).length + 1;
                var previousSundayOfDate = previousSunday(data[0].date);
                var weekDomain = [];
                for (var i=0;i<weeks;i++) {weekDomain.push(previousSundayOfDate);}
                weekScale = weekScale.domain(
                        weekDomain
                            .map(function(first, i){ return d3.time.week.utc.offset(first, i); })
                    ).rangeBands([0, weeks * itemSize - gap], gap / (weeks * itemSize - gap), 0);
                heatScale = heatScale.domain([d3.min(data.map(Fn.prop('value'))), d3.max(data.map(Fn.prop('value')))]);

                var xAxis = d3.svg.axis().orient('top').scale(weekScale)
                        .tickFormat(function(d){ return scope.light ? monthTickFormat(new Date(d)) : niceDate(new Date(d), false); }),
                    yAxis = d3.svg.axis().orient('left').scale(dayScale).tickSize(1)
                        .tickFormat(Fn.from(['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'])),
                    years = weeks > 26,
                    width  = Math.max(minWidth, xAxis.scale().rangeExtent()[1] + margin.left + margin.right),
                    height = yAxis.scale().rangeExtent()[1] + margin.top + margin.bottom + (years ? 16 : 0);
                svg .attr('viewBox', [0, 0, width, height].join(' '))
                    .attr('width', width).attr('height', height)
                    .classed('crisp-edges svg-defaults', true);

                svg.selectAll("text").remove();
                svg.selectAll("defs").remove();

                //declaring diagonal stripes pattern
                svg.append('defs').append('pattern')
                    .attr("id","pattern-stripes")
                    .attr("width","4")
                    .attr("height","4")
                    .attr("patternUnits","userSpaceOnUse")
                    .attr("patternTransform","rotate(45)")
                    .append("rect")
                        .attr("width","2")
                        .attr("height","4")
                        .attr("transform","translate(0,0)")
                        .attr("fill","#eee")

                //render axes
                svg.append('g')
                    .attr('transform','translate('+margin.left+','+margin.top+')')
                    .attr('class','x axis')
                    .call(xAxis)
                if (scope.light) {
                    svg.selectAll('g.tick text').style("text-anchor", "start").attr('transform', 'translate('+ -cellSize/2 +','+ xAxis.innerTickSize() +')');
                } else {
                    svg.selectAll('g.tick text').attr('transform', 'translate(18,-8) rotate(-45)');
                }
                if (years) {
                    var extent = d3.extent(weekScale.domain());
                    extent[1] = d3.time.week.utc.offset(extent[1], 1);   // end of axis = last week + 1
                    var yearScale = d3.time.scale.utc().domain(extent).range(weekScale.rangeExtent());
                    svg.append('g')
                        .attr('transform','translate('+margin.left+','+ (height - margin.bottom - 18) +')')
                        .attr('class','x axis')
                        .call(d3.svg.axis().orient('bottom').scale(yearScale)
                            .tickValues(yearScale.ticks(d3.time.year.utc, 1).map(d3.time.week.utc.ceil))
                            .tickFormat(d3.time.format('%Y')));
                }
                if (!scope.light) {
                    svg.append('g')
                        .attr('transform','translate(' + margin.left + ',' + margin.top + ')')
                        .attr('class','y axis')
                        .call(yAxis);
                }
                svg.selectAll('path.domain, g.tick line').remove();

                svg.append('g')
                    .attr('transform','translate(' + (margin.left + gap) + ',' + (margin.top + gap) + ')')
                    .attr('class','heatmap')
                    .selectAll('rect').data(data).enter().append('rect')
                        .attr('width', cellSize)
                        .attr('height', cellSize)
                        .attr('x', Fn(Fn.prop('date'), previousSunday, weekScale))
                        .attr('y', Fn(Fn.prop('date'), Fn.method('getUTCDay'), dayScale))
                        .attr('fill', function(d) {
                            if (d.value === 0) {
                                return '#eee';
                            } else if (d.value === -1) { //meaning project did not exist at this time
                                return 'url(#pattern-stripes)';
                            } else {
                                return heatScale(d.value);
                            }
                        })
                        .attr('data-title', function(d) {
                            return d.value === -1 ? "Project did not exist on {0}".format(niceDate(d.date, true)) : "<strong>{0}</strong> on {1}".format((scope.formatter || Fn.SELF)(d.value), niceDate(d.date, true));
                        });

                if (svgTitles) {
                    svgTitles.update();
                }

                //If no activity at all adding overlaying text entitled 'No activity'
                var noActivity = data.every(function(d) {
                    return d.value <= 0;
                });

                if (noActivity) {
                    svg.append('text')
                        .attr('transform','translate(' + margin.left + ',' + margin.top + ')')
                        .attr('x', (width - margin.left - margin.right)/2)
                        .attr('y', (height - margin.top - margin.bottom)/2)
                        .attr("alignment-baseline", "middle")
                        .attr("text-anchor", "middle")
                        .attr("style", "text-transform:uppercase")
                        .attr("fill", "#999")
                        .text("no activity");
                }
            });
        }
    };
});



app.directive("simpleTimeAreaChart", function(Fn, D3ChartAxes){

    return {
        scope: { ts: '=', values: '=', scale: '=?', color: '@?', width: '@?', height: '@?' },
        link: function(scope, element, attrs, controllers) {
            var margin = {top:20,right:20,bottom:30,left:40},
                svg = d3.select(element.get(0)),
                height, width;

            var tScale = d3.time.scale(),
                yScale = d3.scale.linear(),
                viz = svg.append('g').attr('transform','translate(' + margin.left + ',' + margin.top + ')'),
                xAxis = d3.svg.axis().orient('bottom').scale(tScale).outerTickSize(0),
                yAxis = d3.svg.axis().orient('left').scale(yScale).ticks(3),
                yAxisG = viz.append('g').attr('class','y axis crisp-edges').style('stroke', '#bdbdbd'),
                area = viz.append('g').attr('class', 'area').append('path').attr('fill', scope.color || '#64B5F6'),
                xAxisG = viz.append('g').attr('class','x axis crisp-edges stroke-cc').style('color', '#bdbdbd');

            var resize = function() {
                height = scope.height == '100%' ? element.parent().height() : (parseInt(scope.height) || 160);
                width = scope.width === '100%' ? element.parent().width() : (parseInt(scope.width) || 400);

                svg .attr('viewBox', [0, 0, width, height].join(' '))
                    .attr('width', width).attr('height', height)
                    .classed('svg-defaults', true);

                xAxisG.attr('transform','translate(0,' + (height - margin.bottom - margin.top) + ')');
                xAxis.ticks(width/100);
                yAxis.tickSize(-width + margin.left + margin.right, 0);
            };

            resize();
            scope.$on('resize', function() {
                resize();
                draw();
            });


            var draw = function() {
                if (!scope.ts || !scope.ts.length || !scope.values.length) return;

                var xs = scope.ts,
                    ys = scope.values;

                tScale.domain([d3.min(xs, Fn.method('valueOf')), d3.max(scope.ts, Fn.method('valueOf'))])
                      .range([0, width - margin.left - margin.right]);

                yScale.range([height - margin.top - margin.bottom, 0]);

                if (d3.time.weeks.utc.apply(null, tScale.domain()).length > 4) {
                    // aggregate by week to avoid jigsaw graph
                    xs = xs.map(d3.time.week.utc).filter(Fn(Fn.method('valueOf'), Fn.unique()));
                    ys = xs.map(Fn.cst(0));
                    scope.values.forEach(function(y, i) {
                        ys[this.indexOf(d3.time.week.utc(scope.ts[i]).valueOf())] += y;
                    }, xs.map(Fn.method('valueOf')));
                    xs = xs.map(d3.time.thursday.utc.ceil); // place data points on thursdays, 12am (~middle of the week)
                    tScale.domain([d3.min(xs, Fn.method('valueOf')), d3.max(xs, Fn.method('valueOf'))]); // readjust axis
                }

                yScale.domain([d3.min(ys.concat(0)), d3.max(ys)]);
                if (typeof scope.scale === 'function') { // scale callback from computed scale
                    yScale.domain(scope.scale(yScale.domain().concat()));
                } else if (scope.scale) {   // fixed
                    yScale.domain(scope.scale);
                }

                D3ChartAxes.addNumberFormatterToAxis(yAxis);
                yAxisG.transition().call(yAxis);
                yAxisG.selectAll('path.domain').remove();

                area.datum(xs.map(Fn.INDEX))
                    .transition()
                    .attr('d', d3.svg.area().x(Fn(Fn.from(xs), tScale))
                        .y0(yScale.range()[0])
                        .y1(Fn(Fn.from(ys), yScale))
                        .interpolate("monotone"));

                xAxisG.call(xAxis);
                xAxisG.select('path.domain').style('stroke', 'black');
       };

            scope.$watch('values', draw);
        }
    };
});

app.directive("weekPunchCard", function(Fn){
    return {
        scope: { data: '=' }, // data: 7 by 24 numbers array
        link: function(scope, element, attrs) {
            scope.cellSize = scope.cellSize || 25;
            if (!scope.data || !scope.data.length) return;

            var rowHeight = 50,
                colWidth = 35,
                margin = {top: 0, right: 20, bottom: 20, left: 75},
                vizWidth = 24 * colWidth,
                vizHeight = 7 * rowHeight,
                data = scope.data.concat(),
                xScale = d3.scale.ordinal().domain(Array.range(24)).rangeRoundBands([0, vizWidth], 0.1, 0),
                yScale = d3.scale.ordinal().domain(Array.range(7)).rangeRoundBands([0, vizHeight], 0.1, 0),
                maxRadius = Math.min(xScale.rangeBand()/ 2, yScale.rangeBand()/2),
                sizeScale = d3.scale.sqrt().range([2, maxRadius]).domain([0, d3.max(data, Fn.passFirstArg(d3.max))]),
                svg = d3.select(element.get(0)),
                viz = svg.classed('svg-defaults', true).style('color', '#ccc')
                    .attr('viewBox', [0, 0, vizWidth + margin.left + margin.right, vizHeight + margin.top + margin.bottom].join(' '))
                    .append("g")
                    .attr("transform", "translate(" + margin.left + "," + margin.top + ")");

            /* Draw x axis */
            var xAxis = d3.svg.axis().scale(xScale)
                .tickSize(0)
                .tickFormat(function(d) { return (d % 12 || 12) + (d > 12 ? 'p': 'a'); });
            viz.append("g").call(xAxis)
                .attr('class', 'x axis crisp-edges')
                .attr('transform', 'translate(0,' + vizHeight + ')')
                .select('.domain').remove();

            /* Draw all horizontal lines with ticks */
            var evenHoursAxis = d3.svg.axis().scale(xScale)
                .orient('top')
                .tickFormat('')
                .tickSize(0.1 * scope.cellSize, 0)
                .tickValues(Array.range(12).map(function(_, i) { return 2*i; }));
            var oddHoursAxis = d3.svg.axis().scale(xScale)
                .orient('top')
                .tickFormat('')
                .tickSize(0.2 * scope.cellSize, 0)
                .tickValues(Array.range(12).map(function(_, i) { return 2*i + 1; }));
            var xLines = viz.append("g")
                .attr('class', 'line axis stroke-cc crisp-edges')
                .selectAll('g.line').data(data).enter()
                    .append('g')
                    .attr('transform', function(d,i) { return 'translate(0, ' + (yScale(i) + yScale.rangeBand()) + ')'; });
            xLines.append('g').call(evenHoursAxis);
            xLines.append('g').call(oddHoursAxis);

            // move all ticks to the left by half a column so that circle are in-between full hours
            viz.selectAll('.tick').each(function() { // may already have transform
                this.setAttribute('transform', 'translate(-' + (colWidth/2) + ', 0) ' + this.getAttribute('transform'));
            });

            /* Draw day labels on the left and extend horizontal lines */
            var yLabels = svg.append("g")
                .attr("class", "day-labels stroke-cc crisp-edges")
                .selectAll("g")
                    .data(['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'])
                    .enter().append("g").attr('class', 'y axis');
            yLabels.append('text')
                .text(Fn.SELF)
                .attr('x', 5)
                .attr('y', function(d,i) { return yScale(i) + yScale.rangeBand() * 0.55; });
            yLabels.append('line')
                .attr('x1', 5)
                .attr('x2', margin.left)
                .attr('y1', function(d,i) { return yScale(i) + yScale.rangeBand(); })
                .attr('y2', function(d,i) { return yScale(i) + yScale.rangeBand(); });

            /* Draw circles */
             var circles = viz.selectAll('g.day').data(data).enter().append('g')
                .attr('class', 'day')
                .attr('transform', function (d, i) { return 'translate(0, ' + (yScale(i) + yScale.rangeBand()/2 - 0.1 * scope.cellSize) + ')'; })
                .selectAll('circle').data(Fn.SELF).enter().append('circle')
                    .attr('cx', function(d, i, j) { return xScale(i) + xScale.rangeBand()/2; })
                    .attr('r', sizeScale)
                    .attr('fill', 'grey').classed('hover-fill', true)
                    .attr('data-title', function(d) { return "{0} commit{1}".format(d, d > 1 ? "s" : ""); });
        }
    }
});

app.directive("userLeaderboard", function(Fn, $state, UserImageUrl){
    return {
        scope: { data: '=', prop: '=?' },
        require: ['?svgTitles'],
        link: function(scope, element, attrs, controllers) {

            var margin = {top: 10, right: 20, bottom: 40, left: 40},
                vizHeight = 120,
                barWidth = 24,
                padding = 5,
                vizMaxWidth = 800,
                maxBars = vizMaxWidth / (barWidth + padding),
                xScale = function(i) { return padding + (padding + barWidth) * i; },
                yScale = d3.scale.linear().range([vizHeight, 0]),
                svg = d3.select(element.get(0)).classed('svg-defaults crisp-edges', true),
                svgTitles = controllers[0],
                format = d3.format('.3s');

            /* Draw x axis */

            function redraw() {
                svg.select('g').remove();

                if (!scope.data || !scope.prop || !scope.data.length || !(scope.prop in scope.data[0])) return;

                var extract = Fn.prop(scope.prop),
                    n = Math.min(maxBars, scope.data.length),
                    vizWidth = n * (barWidth + padding),
                    svgWidth = vizWidth + margin.left + margin.right,
                    sorted = scope.data.concat(),
                    viz = svg.append("g").attr("transform", "translate(" + margin.left + "," + margin.top + ")");
                sorted.sort(function(a,b) { // descending order
                    return extract(a) > extract(b) ? -1 : (extract(a) < extract(b)) ? 1 : 0;
                });
                sorted = sorted.slice(0, n);

                viz.append('clipPath').attr('id', 'user-clip-path')
                    .append('circle').attr('cx', barWidth/2).attr('cy', barWidth/2).attr('r', barWidth/2).attr('fill', 'black');

                svg .attr('viewBox', [0, 0, svgWidth, vizHeight + margin.top + margin.bottom].join(' '))
                    .attr('width', svgWidth);
                yScale.domain([0, Math.max(1, d3.max(sorted, extract))]);
                var yAxis = d3.svg.axis().scale(yScale).orient('left').tickSize(-vizWidth, 0).ticks(3);

                viz.datum(sorted);
                viz.append("g").attr('class', 'y axis').call(yAxis)
                    .select('.domain').remove();
                viz.selectAll('.tick line').attr('stroke', '#ddd');

                var tooltip = function(d) { return '<strong>{0}</strong> for {1}'.format(format(extract(d)), d.user); };
                viz.selectAll('rect.user').data(Fn.SELF).enter().append('rect')
                    .data(Fn.SELF)
                    .attr('class', 'user')
                    .attr('x', function(d,i) { return xScale(i); })
                    .attr('y', Fn(extract, yScale))
                    .attr('fill', '#64B5F6')
                    .attr('height', function(d) { return vizHeight - yScale(extract(d)); })
                    .attr('width', barWidth)
                    .attr('data-title', tooltip);

                viz.append('g').attr('class', 'users')
                    .selectAll('g').data(Fn.SELF).enter().append('g')
                    .attr('clip-path', 'url(#user-clip-path)')
                    .attr('transform', function(d, i) { return 'translate(' + xScale(i) + ',' + (vizHeight+5) + ')'; })
                    .append('a').attr('xlink:href', function(d) { return '/profile/' + d.user + '/'; })
                    .attr('xlink-href', function(d) { return '/profile/' + d.user + '/'; })
                    .on('click', function(d) { // work around xlink:href not fully working in Chrome
                        if (d3.event.which == 1 && !d3.event.metaKey) {
                            $state.go('profile.user.view', {userLogin: d.user});
                        }
                    })
                    .attr('data-title', tooltip)
                    .append('image')
                        .attr ('xlink:href', function(d) {
                            return UserImageUrl(d.user, barWidth);
                        })
                        .attr('x', 0)
                        .attr('y', 0)
                        .attr('width', barWidth).attr('height', barWidth);

                if (svgTitles) {
                    svgTitles.update();
                }
            }
            scope.$watch("data", redraw);
            scope.$watch("prop", redraw);
        }
    }
});

app.component('upgradePermissionsInfo', {
    templateUrl: '/templates/projects/security/upgrade-permissions-info.html',
    bindings: {
        projectSummary: '<',
        canUpgrade: '<',
        onComplete: '&',
    },
    controller: function UpgradePermissionsInfoController($scope, DataikuAPI, CreateModalFromTemplate, ActivityIndicator) {
        this.upgradePermissions = () => {
            Promise.all([
                DataikuAPI.dashboards.list(this.projectSummary.projectKey),
                DataikuAPI.projects.getSettings(this.projectSummary.projectKey)
            ]).then(([dashboards, projectSettings]) => {
                CreateModalFromTemplate('/templates/projects/security/upgrade-permissions-modal.html', $scope, null, newScope => {
                    newScope.dashboardUsers = projectSettings.data.additionalDashboardUsers.users;
                    newScope.writeDashboardsUsers = projectSettings.data.permissions.filter(p => p.writeDashboards && !p.moderateDashboards);
                    newScope.dashboards = dashboards.data.filter(dashboard => dashboard.listed);

                    newScope.doUpgradePermissions = () => {
                        DataikuAPI.projects.upgradePermissionsVersion(this.projectSummary.projectKey).success(() => {
                            ActivityIndicator.success('Permissions successfully upgraded!');
                            newScope.dismiss();
                            this.onComplete();
                        }).error(setErrorInScope.bind(newScope));
                    };
                });
            }).catch(setErrorInScope.bind($scope));
        }
    }
});

})();
