(function() {
'use strict';

const app = angular.module('dataiku.recipes');


app.service("RecipesUtils", function(Assert, Logger, $filter){
    var svc = {
        isMLRecipe: function(recipe){
            return recipe.type in {
                "prediction_training" : true,
                "prediction_scoring" : true,
                "clustering_training" : true,
                "clustering_scoring" : true,
                "clustering_complete" : true,
                "evaluation": true,
                "standalone_evaluation": true
            };
        },

        getInputsForRole: function(recipe, role) {
            if (recipe.inputs[role] != null) return recipe.inputs[role].items;
            return [];
        },
        getOutputsForRole: function(recipe, role) {
            if (recipe.outputs[role] != null) return recipe.outputs[role].items;
            return [];
        },

        getFlatInputsList: function(recipe) {
            var flat =[];
            $.each(recipe.inputs, function(roleName, roleData){
                flat = flat.concat(roleData.items);
            });
            return flat;
        },
        getFlatOutputsList: function(recipe) {
            Assert.trueish(recipe, 'no recipe');
            var flat =[];
            $.each(recipe.outputs, function(roleName, roleData){
               flat = flat.concat(roleData.items);
            });
            return flat;
        },
        getFlatIOList: function(recipe) {
            return svc.getFlatInputsList(recipe).concat(svc.getFlatOutputsList(recipe));
        },

        hasAnyPartitioning: function(recipe, computablesMap) {
            if (!recipe || !computablesMap)
                return false;
            var hap = svc.getFlatIOList(recipe).some(function(input){
                var computable = computablesMap[input.ref];
                if (computable == null) {
                    Logger.error("Computable not found for " + input.ref);
                    return false;
                }
                const partitioning = $filter('retrievePartitioning')(computable);
                return partitioning && partitioning.dimensions && partitioning.dimensions.length;
            });
            return hap;
        },

        addInput: function(recipe, role, ref) {
            if (recipe.inputs == null) recipe.inputs = {};
            if (recipe.inputs[role] == null) recipe.inputs[role] = { items : [] }
            recipe.inputs[role].items.push({ref:ref, deps : []});
        },
        addOutput: function(recipe, role, ref) {
            if (recipe.outputs == null) recipe.outputs = {};
            if (!recipe.outputs[role]) recipe.outputs[role] = { items : [] }
            recipe.outputs[role].items.push({ref:ref, deps : []});
        },

        getSingleInput: function(recipe, role) {
            if (recipe.inputs[role] == null) throw Error("No input found in role " + role);
            if (recipe.inputs[role].items.length  == 0) throw Error("No input found in role " + role);
            if (recipe.inputs[role].items.length > 1) throw Error("Multiple inputs found in role " + role);
            return recipe.inputs[role].items[0]
        },
        getSingleOutput: function(recipe, role) {
            if (recipe.outputs[role] == null) throw Error("No output found in role " + role);
            if (recipe.outputs[role].items.length  == 0) throw Error("No output found in role " + role);
            if (recipe.outputs[role].items.length > 1) throw Error("Multiple outputs found in role " + role);
            return recipe.outputs[role].items[0]
        },

        getInput: function(recipe, role, ref) {
            if (recipe.inputs[role] == null) return null;
            if (recipe.inputs[role].items.length  == 0) return null;
            var i = 0;
            for (i = 0; i < recipe.inputs[role].items.length; i++) {
                var input = recipe.inputs[role].items[i];
                if (input.ref == ref) return input;
            }
            return null;
        },
        removeInput: function(recipe, role, ref) {
            if (recipe.inputs[role] == null) return;
            if (recipe.inputs[role].items.length  == 0) return;
            recipe.inputs[role].items = recipe.inputs[role].items.filter(function(x){
                return x.ref != ref;
            })
        },
        removeInputsForRole: function(recipe, role) {
            if (recipe.inputs[role]) {
                recipe.inputs[role].items = [];
            }
        },
        removeOutputsForRole: function(recipe, role) {
            if (recipe.outputs[role]) {
                recipe.outputs[role].items = [];
            }
        },
        parseScriptIfNeeded: function(recipeData) {
            if (['shaker', 'join', 'grouping', 'upsert', 'sampling', 'split', 'clustering_training', 'prediction_training', 'vstack', 'grouping'].indexOf(recipeData.recipe.type) > -1 && typeof recipeData.script === 'string') {
                recipeData.script = JSON.parse(recipeData.script);
            }
        },
        getWT1LoggableRecipeEventParams: function(recipe, recipeDesc) {
            try {
                if (!recipe || !recipe.params || !recipe.params.customConfig || !recipeDesc || !recipeDesc.params){
                    return {}
                }
                const config = recipe.params.customConfig;
                const entries = recipeDesc.params.filter(p => p.wt1Loggable)
	                .map(p => [ `recipeEventParam-${p.name}`, config[p.name] ]);
                return Object.fromEntries(entries);
            } catch (e) {
                Logger.error("Failed to get wt1 loggable recipe event params", e);

                return {}
            }
        },
        getWT1OutputAppendModeTracking: function(recipe) {
            try {
                let recipeAnyOutputAppendMode = false;
                Assert.trueish(typeof recipe.outputs === 'object', 'recipe outputs is not an object');
                for (const output of Object.values(recipe.outputs)) {
                    for (const o of output.items) {
                        recipeAnyOutputAppendMode = recipeAnyOutputAppendMode || !!o.appendMode;
                    }
                }
                return { recipeAnyOutputAppendMode };
            } catch (e) {
                Logger.error("Failed to get wt1 recipe output append mode usage", e);
                return {};
            }
        },
    };
    return svc;
});


app.service("RecipeRunJobService", function(Assert, DataikuAPI, $stateParams, RecipesUtils, JobDefinitionComputer, $filter, CreateModalFromTemplate, $timeout) {

    function getInputAndPartitioning(recipe, computablesMap) {
        if (!computablesMap) {
            throw new Error("computablesMap not ready");
        }
        const inputs = RecipesUtils.getFlatInputsList(recipe);
        for (const input of inputs) {
            const computable = computablesMap[input.ref];
            const partitioning = $filter('retrievePartitioning')(computable);

            if (partitioning && partitioning.dimensions.length > 0) {
                return { input, partitioning };
            }
        }
        return { input: inputs[0] };
    }

    function getOutputAndPartitioning(recipe, computablesMap) {
        if (!computablesMap) {
            throw new Error("computablesMap not ready");
        }
        const outputs = RecipesUtils.getFlatOutputsList(recipe);
        for (const output of outputs) {
            const computable = computablesMap[output.ref];
            const partitioning = $filter('retrievePartitioning')(computable);

            if (partitioning && partitioning.dimensions.length > 0) {
                return { output, partitioning };
            }
        }
        return { output: outputs[0] };
    }

    // Ugly fix to prevent the modal from focusing the first button in its body by default
    function focusRunDownstreamButton() {
        var element = document.getElementById("build-mode-selector__recursive-downstream-btn");
        $timeout(function() {
            element.focus();
        }, 50);
    }

    var svc = {
        getInputAndPartitioning,
        getOutputAndPartitioning,

        getInputDimensions: function(recipe, computablesMap) {
            const partitioning = getInputAndPartitioning(recipe, computablesMap).partitioning
            return partitioning ? partitioning.dimensions : [];
        },

        getOutputDimensions: function(recipe, computablesMap) {
            const partitioning = getOutputAndPartitioning(recipe, computablesMap).partitioning
            return partitioning ? partitioning.dimensions : [];
        },

        getTargetPartition: function($scope) {
            if ( $scope.testRun == null ) return "";
            var outputs = RecipesUtils.getFlatOutputsList($scope.recipe);
            /* First find a dataset */
            for (var i = 0; i < outputs.length; i++) {
                var output = outputs[i];
                if (!$scope.computablesMap) {
                    throw new Error("computablesMap not ready");
                }
                var computable = $scope.computablesMap[output.ref];
                if (computable.type == "DATASET") {
                    var jd = JobDefinitionComputer.computeJobDefForSingleDataset($stateParams.projectKey,
                                $scope.testRun.runMode,
                                computable.dataset,
                                $scope.testRun.build_partitions);
                    return jd.outputs[0].targetPartition;
                }
            }
            /* No dataset found ... */
            return "";
        },

        isRunning: function(startedJob) {
            if (!startedJob || !startedJob.jobStatus || !startedJob.jobStatus.baseStatus){
                return false;
            }
            var state = startedJob.jobStatus.baseStatus.state;
            return ["RUNNING", "COMPUTING_DEPS", "STARTING"].includes(state);
        },

        isPending: function(startedJob) {
            if (!startedJob || !startedJob.jobStatus || !startedJob.jobStatus.baseStatus){
                return false;
            }
            var state = startedJob.jobStatus.baseStatus.state;
            return ["PENDING", "NOT_STARTED"].includes(state);
        },

        run: function (recipe, computablesMap, testRun, startedJob, callbackOnceStarted, runRecipeOnly, errorScope) {
            Assert.trueish(recipe, 'no recipe');
            Assert.trueish(computablesMap, 'no computablesMap');
            Assert.trueish(RecipesUtils.getFlatOutputsList(recipe).length > 0, 'no outputs');
            return DataikuAPI.flow.recipes.runFromRecipeUI(
                $stateParams.projectKey, recipe.name, runRecipeOnly ? "RUN_RECIPE_ONLY" : "RUN_ONLY_IF_NO_SUCCESSORS", testRun.build_partitions).success(function (data) {

                    if (data.started) {
                        startedJob.starting = false;
                        startedJob.jobId = data.jobId;
                        callbackOnceStarted();
                    } else if (data.hasSuccessorsAndCanDoAReverseBuild) {

                        CreateModalFromTemplate("/templates/recipes/run-recipe-modal.html", errorScope, "RunRecipeBuildOptionsController", function (modalScope) {
                            modalScope.selectedBuildMode = "RECURSIVE_DOWNSTREAM";
                            modalScope.isRunningFromRecipe = true;
                            // Forcing focus on the default selected build mode button, as the unselected option is focused at modal creation otherwise
                            focusRunDownstreamButton()
                            modalScope.confirm = function () {
                                modalScope.jobStarted = true;
                                DataikuAPI.flow.recipes.runFromRecipeUI(
                                    $stateParams.projectKey, recipe.name, modalScope.selectedBuildMode === "RUN_RECIPE_ONLY" ? "RUN_RECIPE_ONLY" : "RUN_REVERSE_BUILD", testRun.build_partitions).success(function (data) {
                                        Assert.trueish(data.started);
                                        modalScope.dismiss();
                                        startedJob.starting = false;
                                        startedJob.jobId = data.jobId;
                                        callbackOnceStarted();
                                    }).error((data, status, headers) => {
                                        setErrorInScope.bind(modalScope)(data, status, headers)
                                        modalScope.jobStarted = false;
                                    });
                            };
                            modalScope.$on("$destroy", function () {
                                startedJob.starting = modalScope.jobStarted;
                            });
                        });
                    } else {
                        Assert.trueish(false); // One of the two flags must be true
                    }

                }).error(setErrorInScope.bind(errorScope));

        }
    }
    return svc;
});
                        
app.service("RecipeDescService", function($stateParams, $translate, DataikuAPI, Logger, WT1, $q) {

    var descriptors;

    function capitalize(str) {
        if (!str) return '';
        if(str.length>0) {
            str = str.substring(0,1).toUpperCase() + str.substring(1);
        }
        return str;
    }
    function formatNice(str) {
        if (!str) return '';
        str = str.toLowerCase();
        str = str.replace(/[_ ]+/g,' ');
        return str;
    }

    function isOverallUnary(roles, editableFilterLocation) {
        if (!roles) return false;
        var unary = null;
        if (editableFilterLocation) {
            const isRoleEditableFn = editableFilterLocation === "modal" ? (r) => r.editableInModal : (r) => r.editableInEditor;
            roles = roles.filter(isRoleEditableFn);
        }
        roles.forEach(function(r) {
            if (r.arity != 'UNARY') {
                unary = false;
            } else if (unary === null) {
                // first one
                unary = true;
            } else {
                // 2 unaries => not overall unary
                unary = false;
            }
        });
        return unary;
    }

    const recipeDescriptorsDeferred = $q.defer();
    var svc = {
        recipeDescriptorsAsPromise: recipeDescriptorsDeferred.promise,
        load: function(errorScope) {
            DataikuAPI.flow.recipes.getTypesDescriptors($translate.proposedLanguage()).success(function(data) {
                Logger.info("Received recipe descriptors");
                descriptors = data;
                recipeDescriptorsDeferred.resolve(data);
            }).error(function(a, b, c) {
                Logger.error("Failed to get recipe descriptors");
                setErrorInScope.bind(errorScope)(a, b, c);
                recipeDescriptorsDeferred.reject(a);
                WT1.event("get-recipe-descriptors-failed");
            });
        },
        getDescriptors: function() {
            if (!descriptors) {
                Logger.error("Recipes descriptors not ready");
            }
            return descriptors;
        },
        isRecipeType: function(recipeType) {
            var desc;
            if (!descriptors) {
                Logger.error("Recipe descriptors are not ready");
            } else {
                return !!descriptors[recipeType];
            }
        },
        getDescriptor: function(recipeType) {
            var desc;
            if (!descriptors) {
                Logger.error("Recipe descriptors are not ready");
            } else {
                angular.forEach(descriptors, function(d, t) {
                    if (t == recipeType || t.toLowerCase() == recipeType) desc = d;
                });
                if (!desc) {
                    // This should not happen (maybe a new plugin recipe type)
                    Logger.error("Recipe descriptor not found for type: "+recipeType);
                }
                return angular.copy(desc);
            }
        },
        getRolesIndex: function(recipeType) {
            var desc = svc.getDescriptor(recipeType);
            var inputRolesIndex = {};
            var outputRolesIndex = {};
            desc.inputRoles.forEach(function(r) {
                inputRolesIndex[r.name] = r;
            });
            desc.outputRoles.forEach(function(r) {
                r.editing = true; //Ugly, to be compatible with current templates
                outputRolesIndex[r.name] = r;
            });
            return {inputs: inputRolesIndex, outputs: outputRolesIndex};
        },

        getInputRoleDesc: function(recipeType, roleName) {
            return svc.getRolesIndex(recipeType).inputs[roleName];
        },

        getRecipeTypeName: function(recipeType, capitalize) {
            if(!recipeType) return '';
            var desc = svc.getDescriptor(recipeType);
            var name = (desc && desc.meta && desc.meta.label) || formatNice(recipeType);
            return capitalize ? capitalize(name) : name;
        },
        isSingleInputRecipe: function(recipeType, editableFilterLocation) {
            return (
                !recipeType.startsWith('CustomCode_') &&
                !recipeType.startsWith('App_') &&
                isOverallUnary(svc.getDescriptor(recipeType).inputRoles, editableFilterLocation)
            );
        },
        isSingleOutputRecipe: function(recipeType, editableFilterLocation) {
            return (
                !recipeType.startsWith('CustomCode_') &&
                !recipeType.startsWith('App_') &&
                recipeType !== 'standalone_evaluation' &&
                isOverallUnary(svc.getDescriptor(recipeType).outputRoles, editableFilterLocation)
            );
        },
        hasValidRequiredRoles: function(recipe) {
            const desc = svc.getDescriptor(recipe.type);
            return !desc.inputRoles.some(role => role.required && (!recipe.inputs[role.name] || recipe.inputs[role.name].items.length == 0))
                && !desc.outputRoles.some(role => role.required && (!recipe.outputs[role.name] || recipe.outputs[role.name].items.length == 0));
        }
    };

    return svc;
});

app.service("SelectablePluginsService", function($rootScope) {
    var svc = {

        listSelectablePlugins : function (inputTypesCount) {

            const pluginsById = $rootScope.appConfig.loadedPlugins.reduce( function (map, obj) {
                map[obj.id] = obj;
                return map;
            }, {});

            var selectablePlugins = [];
            var alreadySelectedPlugins = {};

            $rootScope.appConfig.customCodeRecipes.forEach( (recipe) => {
                const plugin = pluginsById[recipe.ownerPluginId];
                if (!alreadySelectedPlugins[recipe.ownerPluginId] && svc.canBeBuildFromInputs(recipe, inputTypesCount).ok && pluginsById.hasOwnProperty(recipe.ownerPluginId) && !plugin.hideComponents) {
                    plugin.pluginId = plugin.id;
                    selectablePlugins.push(plugin);
                    alreadySelectedPlugins[plugin.id] = true;
                }
            });

            return selectablePlugins;
        },

        canBeBuildFromInputs: function (recipe, inputTypesCount) {
            if (!inputTypesCount)
                return { ok: true};

            const allTypes = {
                "DATASET": {
                    "selectableFromRole": 'selectableFromDataset',
                    "inputName" : recipe.desc.selectableFromDataset,
                    "inputTypeCount": inputTypesCount.DATASET
                },
                "MANAGED_FOLDER":  {
                    "selectableFromRole": 'selectableFromFolder',
                    "inputName": recipe.desc.selectableFromFolder,
                    "inputTypeCount": inputTypesCount.MANAGED_FOLDER
                },
                "SAVED_MODEL": {
                    "selectableFromRole": 'selectableFromSavedModel',
                    "inputName": recipe.desc.selectableFromSavedModel,
                    "inputTypeCount": inputTypesCount.SAVED_MODEL
                }
            };
            for (let [inputType, nb_input] of Object.entries(inputTypesCount)) {
                const typeParams = allTypes[inputType];
                var role = recipe.desc.inputRoles.find( (el) => { return el.name == typeParams.inputName; });
                // role does not exist
                if (!role)
                    return { ok: false, reason: "InputRole selectable from " + inputType + " doesn't exist in this recipe."};

                // recipe does not accept selectable inputs of this type
                if (!recipe.desc.hasOwnProperty(typeParams.selectableFromRole)) {
                    return { ok: false, reason: "Input role type " + inputType + " of (" + role.name + ") is not selectable for this recipe."};
                }
                // recipe does not accept inputs of type of the role
                if ((inputType === "DATASET" && !role.acceptsDataset) ||
                    (inputType === "MANAGED_FOLDER" && !role.acceptsManagedFolder) ||
                    (inputType === "SAVED_MODEL" && !role.acceptsSavedModel)) {
                    return { ok: false, reason: "Recipe doesn't accept " + inputType + " as input for the selectable role (" + role.name + ")."};
                }

                if (nb_input > 1 && role.arity == 'UNARY')
                    return { ok: false, reason: "Recipe only accepts unary inputs for the selectable role (" + role.name + ")."};
            }
            return {ok: true};
        }

    };
    return svc;
});

app.service("RecipeComputablesService", function(Assert, DataikuAPI, $stateParams, RecipesUtils, Logger) {
    var svc = {

        isUsedAsInput: function(recipe, ref) {
            var found = false;
            $.each(recipe.inputs, function(role, roleData){
                 found |= roleData.items.some(function(input) {
                    return input.ref == ref;
                 });
            })
            return found;
        },
        isUsedAsOutput: function(recipe, ref) {
            var found = false;
            $.each(recipe.outputs, function(role, roleData){
                 found |= roleData.items.some(function(x) {
                    return x.ref == ref;
                 });
            })
            return found;
        },
        getComputablesMap: function(recipe, errorScope) {
            return DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {
                forRecipeType : recipe.type
            }).then(function(data) {
                var computablesMap ={};
                $.each(data.data, function(idx, elt) { computablesMap[elt.smartName]  = elt;});
                // added bonus: lots of places expect computablesMap to contain all the input/outputs of the recipe
                // so that if one of the elements is removed or stopped being exposed, the UI throws gobs of js
                // errors
                // => we pad the map with fake elements to cover the input/outputs
                var unusable = {};
                angular.forEach(recipe.inputs, function(x, role) {unusable[role] = {};});
                angular.forEach(recipe.outputs, function(x, role) {unusable[role] = {};});
                RecipesUtils.getFlatIOList(recipe).forEach(function(io){
                    var computable = computablesMap[io.ref];
                    if (computable == null) {
                        Logger.warn("Computable not found for " + io.ref + ". Inserting dummy computable.");
                        computablesMap[io.ref] = {type:'MISSING', name:io.ref, usableAsInput:unusable, usableAsOutput:unusable};
                    }
                });
                return computablesMap;
            }, function(resp){
                setErrorInScope.bind(errorScope)(resp.data, resp.status, resp.headers)
            });
        },

        /* returns the name of an output computable */
        getAnyOutputName: function(recipe) {
            if (!recipe || !recipe.outputs) return;
            var roles = Object.keys(recipe.outputs);
            for (var r = 0; r < roles.length; r++) {
                var items = recipe.outputs[roles[r]].items;
                if (items && items.length > 0) {
                    return items[0].ref;
                }
            }
        },

        /**
         * Builds the list of usable inputs for a given role.
         * Note: it actually also includes some non-usable inputs ...
         */
        buildPossibleInputList: function(recipe, computablesMap, role, filter){
            var usableInputs = [];
            var filterStr = null;
            if (filter && filter.length) {
                filterStr = filter.toLowerCase();
            }
            Assert.trueish(computablesMap, 'no computablesMap');
            $.each(computablesMap, function(k, v) {
                if (!filter || !filter.length || v.smartName.toLowerCase().indexOf(filterStr) >= 0 || (v.label && v.label.toLowerCase().indexOf(filterStr) >= 0)) {
                    if (!svc.isUsedAsInput(recipe, v.smartName)) {
                        usableInputs.push(v);
                    }
                }
            });
            return usableInputs;
        },
        /**
         * Builds the list of usable outputs for a given role.
         * Note: it actually also includes some non-usable outputs ...
         * It removes the ones that are currently used in the recipe, and executes the filter
         */
        buildPossibleOutputList: function(recipe, computablesMap, role, filter){
            if (!computablesMap) {
                throw Error("No computablesMap");
            }
            var usable = [];
            var filterStr = null;
            if (filter && filter.length) {
                filterStr = filter.toLowerCase();
            }

            $.each(computablesMap, function(k, v) {
                if (!filter || !filter.length ||
                    v.smartName.toLowerCase().indexOf(filterStr) >= 0 ||
                    (v.label && v.label.toLowerCase().indexOf(filterStr) >= 0)) {
                    if (!svc.isUsedAsOutput(recipe, v.smartName)) {
                        usable.push(v);
                    }
                }
            });
            return usable;
        }
    };
    return svc;
});

app.service('RecipeRenameService', function($stateParams, $state, Dialogs, CreateModalFromTemplate, DataikuAPI, HistoryService) {

    function startRenamingRecipe($scope, projectKey, recipeName) {
        // check if we need to save the recipe before renaming (should only apply on the recipe page)
        if ($scope.hooks && $scope.hooks.recipeIsDirty()) {
            Dialogs.error($scope, "Save the recipe", "You must save the recipe before renaming it");
            return;
        }
        openRecipeRenameModal($scope, projectKey, recipeName);
    };

    function openRecipeRenameModal($scope, projectKey, recipeName) {
        CreateModalFromTemplate("/templates/recipes/rename-recipe-box.html", $scope, null, function(newScope){
            newScope.recipeName = recipeName;
            newScope.uiState = {
                newName: recipeName
            };

            newScope.go = function(){
                DataikuAPI.flow.recipes.rename(projectKey, recipeName, newScope.uiState.newName).success(function() {
                    HistoryService.notifyRenamed({
                        type: "RECIPE",
                        id: recipeName,
                        projectKey: projectKey
                    }, newScope.uiState.newName);
                    newScope.dismiss();
                    redirectAfterRename(recipeName, newScope.uiState.newName);
                }).error(setErrorInScope.bind(newScope));
            }
        });
    }
    
    function redirectAfterRename(oldName, newName) {
        if($stateParams.recipeName === oldName) {
            // reload the page with the new name if the recipeName param is in the current route
            $state.go($state.current, {
                ...$stateParams,
                recipeName : newName
            }, {
                location: 'replace'
            });
        } else {
            // else still reload to refresh content that may reference the old name
            $state.reload();
        }
    }

    return {
        startRenamingRecipe
    };
});

app.service("CodeEnvsService", function(DataikuAPI, $stateParams, RecipeDescService, Logger, CreateModalFromTemplate) {
    var svc = {
        canPythonCodeEnv: function(recipe) {
            if (recipe.nodeType !== 'RECIPE' && recipe.interest && recipe.interest.objectType !== 'RECIPE') {
                return false;
            }
            var t = recipe.recipeType || recipe.type;
            if(['python','pyspark'].indexOf(t) >= 0) {
                return true;
            }
            if (t.startsWith('CustomCode_')) {
                var desc = RecipeDescService.getDescriptor(t);
                if (!desc) return;
                if (['PYTHON'].indexOf(desc.kind) >= 0) {
                    return true;
                }
            }
            return false;
        },

        canRCodeEnv: function(recipe) {
            if (recipe.nodeType !== 'RECIPE' && recipe.interest && recipe.interest.objectType !== 'RECIPE') {
                return false;
            }
            var t = recipe.recipeType || recipe.type;
            if(['r','sparkr'].indexOf(t) >= 0) {
                return true;
            }
            if (t.startsWith('CustomCode_')) {
                var desc = RecipeDescService.getDescriptor(t);
                if (!desc) return;
                if (['R'].indexOf(desc.kind) >= 0) {
                    return true;
                }
            }
            return false;
        },
        startChangeCodeEnv: function(selectedRecipes, envLang, $scope) {
            return CreateModalFromTemplate('/templates/recipes/fragments/change-code-env-modal.html', $scope, null, function(modalScope) {
                modalScope.uiState = {envSelection : {envMode : 'INHERIT'}};
                modalScope.envLang = envLang;
                modalScope.selectedObjects = selectedRecipes;

                modalScope.change = function() {
                    DataikuAPI.flow.recipes.massActions.changeCodeEnv(modalScope.selectedObjects, modalScope.uiState.envSelection).success(function() {
                        modalScope.resolveModal();
                    }).error(setErrorInScope.bind(modalScope));
                };
            })
        }
    };
    return svc;
});

})();
