(function() {
    'use strict';

    const app = angular.module('dataiku.directives.simple_report');

    /*
     * (!) This directive previously was in static/dataiku/js/simple_report/chart_logic.js
     *
     * Need in the scope :
     * - response (containing the pivotResponse - among other things)
     * - "chart" : the chart object, which must contain at least
     *       - data, a ChartSpec object
     *       - summary
     * - 'getExecutePromise(request)' : a function that returns a promise
     * DO NOT USE AN ISOLATE SCOPE as there is some communication with drag-drop
     * stuff
     */
    app.directive('chartConfiguration', function(MonoFuture, Debounce, DataikuAPI, translate,
        ChartDimension, ChartLabels, ChartRequestComputer, $state, $timeout, PluginsService, Logger, ActivityIndicator,
        ChartTypeChangeHandler, ChartDefinitionChangeHandler, ChartUADimension, _MapCharts, ChartsStaticData, ChartConfigurationCopyPaste, DetectUtils, ChartIconUtils, ChartSetErrorInScope, ChartFeatures, PolygonSources, ChartAxesUtils,
        ChartZoomControlAdapter, ChartDataUtils, WT1, CHART_TYPES, CHART_VARIANTS, ChartStoreFactory, ColorUtils, $rootScope, $stateParams, ChartsAvailableTypes, RegressionTypes, ChartFilterUtils, ChartFormattingPane, ChartFormattingPaneSections,
        DataGroupingMode, ChartColorSelection, ConditionalFormattingOptions, ChartDataWrapperFactory, DSSVisualizationThemeUtils) {
        return {
            restrict: 'AE',
            templateUrl: '/static/dataiku/js/simple_report/directives/chart-configuration/chart-configuration.directive.html',
            link: function(scope, element) {

                //needs to be put in the scope so these functions are accessible in the dashboards
                scope.canHaveZoomControls = ChartFeatures.canHaveZoomControls;
                scope.canDisplayLegend = ChartFeatures.canDisplayLegend;
                scope.isStackedLegend = ChartFeatures.isStackedLegend;

                ChartSetErrorInScope.defineInScope(scope);
                scope.isInAnalysis = $state.current.name.indexOf('analysis') != -1;
                scope.isInPredicted = $state.current.name.indexOf('predicted') != -1;
                scope.isInInsight = $state.current.name.indexOf('insight') != -1;
                scope.shouldDisplaySamplingStatus = shouldDisplaySamplingStatus;
                scope.RegressionTypes = RegressionTypes;
                scope.PolygonSources = PolygonSources;

                scope.hasLegacyBadge = ChartFeatures.hasLegacyBadge;
                scope.hasBetaBadge = ChartFeatures.hasBetaBadge;

                scope.legacyBadge = {
                    tooltip: `You are using the legacy engine.
                    Click to go back to the new one.`
                };
                scope.editingTitle = false;

                scope.warningBadge = ChartsStaticData.WARNING_BADGE;

                scope.chartTypePickerOptions = {
                    chartVariant: scope.chart.def.variant,
                    chartType: scope.chart.def.type,
                    webappType: scope.chart.def.webAppType
                };

                scope.availableChartTypes = ChartsAvailableTypes.getAvailableChartTypes();
                scope.chartTypePickerOptions = { ...scope.chartTypePickerOptions, chartTypes: scope.availableChartTypes };

                if (!scope.isInPredicted && !scope.isInAnalysis) {
                    DataikuAPI.explores.listPluginChartDescs($stateParams.projectKey)
                        .success((data) => {
                            scope.webapps = parseWebapps(data);
                            ChartConfigurationCopyPaste.setWebApps(scope.webapps);
                            scope.chartTypePickerOptions = { ...scope.chartTypePickerOptions, webappTypes: scope.webapps };
                        }).error(setErrorInScope.bind(scope));
                };

                if (!scope.chartBottomOffset) {
                    scope.chartBottomOffset = 0;
                }

                scope.optionsFolds = {
                    legend: true,
                    chartMode: true,
                    showTopBar: true
                };
                scope.PluginsService = PluginsService;

                scope.getPivotResponseOptions = () => ({
                    projectKey: $stateParams.projectKey,
                    dataSpec: scope.getDataSpec(),
                    requestedSampleId: scope.chart.summary.requiredSampleId,
                    onError: onPivotRequestError(scope),
                    visualAnalysisFullModelId: $stateParams.fullModelId
                });

                Mousetrap.bind('s h h', function() {
                    scope.$apply(function() {
                        scope.bigChartSwitch();
                    });
                });

                Mousetrap.bind('s v', function() {
                    scope.$apply(function() {
                        scope.switchCharts();
                    });
                });

                scope.$on('$destroy', function() {
                    Mousetrap.unbind('s h h');
                    Mousetrap.unbind('s v');
                });

                scope.switchCharts = function() {
                    if (ChartFeatures.hasEChartsDefinition(scope.chart.def.type) && ChartFeatures.hasD3Definition(scope.chart.def.type)) {
                        if (ChartFeatures.hasDefaultEChartsDisplay(scope.chart.def.type)) {
                            scope.chart.def.displayWithEChartsByDefault = !scope.chart.def.displayWithEChartsByDefault;
                        } else {
                            scope.chart.def.displayWithECharts = !scope.chart.def.displayWithECharts;
                        }
                    } else if (!ChartFeatures.hasD3Definition(scope.chart.def.type)) {
                        Logger.warn('No d3 definition for this chart');
                    } else {
                        Logger.warn('No echarts definition for this chart');
                    }
                };

                scope.openSection = function(section, properties) {
                    const readOnly = $state.current.name === 'projects.project.dashboards.insights.insight.view';
                    if (!readOnly && !(section === ChartFormattingPaneSections.MISC && scope.isInPredicted)) {
                        ChartFormattingPane.open(section, true, true);
                        if (properties && typeof properties === 'object'){
                            Object.keys(properties).forEach(propertyName => {
                                ChartFormattingPane.setData(propertyName, properties[propertyName]);
                            });
                        }
                    }
                };

                scope.onFoldableSectionLoaded = function(section) {
                    let sections = [];

                    if ($stateParams.sections) {
                        sections = JSON.parse($stateParams.sections);
                    }

                    sections.forEach(sectionId => {
                        if (section.getId() === sectionId) {
                            section.open();
                            setTimeout(() => section.scrollIntoView());
                        }
                    });
                };

                scope._MapCharts = _MapCharts;

                const addBigChartClass = (addClass) => {
                    if (addClass) {
                        $('.charts-container').addClass('big-chart');
                    } else {
                        $('.charts-container').removeClass('big-chart');
                    }
                };

                scope.fixupCurrentChart = function() {
                    ChartTypeChangeHandler.fixupChart(scope.chart.def, scope.chart.theme);
                };

                addBigChartClass(false); // remove class in case user is switching from a big chart
                scope.bigChart = false;
                scope.bigChartSwitch = function() {
                    scope.bigChart = !scope.bigChart;
                    scope.isBigChartSwitch = true;
                    $('.graphWrapper').fadeTo(0, 0);
                    addBigChartClass(scope.bigChart);
                    //waiting for the css transition to finish (0.25s, we use 300ms, extra 50ms is fore safety)
                    $timeout(function() {
                        //for binned_xy_hex we need to recompute because width and height are taken into account in chart data computing
                        if (scope.chart.def.type == CHART_TYPES.BINNED_XY && scope.chart.def.variant == CHART_VARIANTS.binnedXYHexagon) {
                            scope.recomputeAndUpdateData();
                            scope.executeIfValid();
                        } else {
                            scope.redraw();
                        }
                        $('.graphWrapper').fadeTo(0, 1);
                    }, 250);
                };

                scope.chartSpecific = {};

                scope.droppedData = [];

                scope.ChartTypeChangeHandler = ChartTypeChangeHandler;
                scope.ChartsStaticData = ChartsStaticData;
                scope.ChartFormattingPane = ChartFormattingPane;
                scope.ChartFormattingPaneSections = ChartFormattingPaneSections;

                let frontImportantChangeTimeoutId;
                let noRedrawDelayedChangeTimeoutId;
                let invalidChangeTimeoutId;
                let themeChangeTimeoutId;

                /**
                 * A hack to redraw the scatter MP when a new dimension pair placeholder is added.
                 * We do not recompute the chart until the pair is complete but we do add a new placeholder
                 * resulting in the chart decreasing in size but not redrawing. This launches a redraw.
                 */
                const resizeObserver = new ResizeObserver(function(entries) {
                    const newHeight = entries[0].contentRect.height;
                    const newWidth = entries[0].contentRect.width;
                    const deltaHeight = Math.abs(scope.chartParamBarHeight - newHeight);
                    const deltaWidth = Math.abs(scope.chartParamBarWidth - newWidth);
                    // 30 is a treshold to avoid redrawing too frequently. Can be increased if needed.
                    if ((scope.chartParamBarHeight || scope.chartParamBarWidth) && Math.max(deltaHeight, deltaWidth) > 30) {
                        if (!scope.isBigChartSwitch) {
                            handleResize(scope);
                        } else {
                            scope.isBigChartSwitch = false;
                        }
                    }
                    scope.chartParamBarHeight = newHeight;
                    scope.chartParamBarWidth = newWidth;
                });

                /*
                 * ------------------------------------------------------
                 * only trigger this code once, when the chart is initialized
                 */
                const unregister = scope.$watch('chart', function(nv) {
                    if (nv == null) {
                        return;
                    }
                    unregister();

                    scope.executedOnce = false;

                    if (angular.isUndefined(scope.chart.def)) {
                        Logger.warn('!! BAD CHART !!');
                    }

                    // scope.chart.spec.unregisterWatch = 1;

                    scope.fixupCurrentChart();

                    // STATIC DATA
                    scope.staticData = {};
                    scope.staticData.multiplotDisplayModes = [
                        { id: 'column', name: translate('CHARTS.DROPPED_MEASURE.DISPLAY_AS.BAR') },
                        { id: 'line', name: translate('CHARTS.DROPPED_MEASURE.DISPLAY_AS.LINE') }
                    ];

                    scope.chartTypes = [
                        {
                            type: CHART_TYPES.GROUPED_COLUMNS,
                            title: 'Grouped columns',
                            description: 'Use to create a grouped bar chart.<br/> Break down once to create one group of bars per category. Measures provide bars.<br/> Break down twice to create one group of bars per category and one bar for each subcategory.'
                        },
                        {
                            type: CHART_TYPES.STACKED_COLUMNS,
                            title: 'Stacked columns',
                            description: 'Use to display data that can be summed.<br/> Break down once with several measures to stack the measures.<br/>  Break down twice to create one stack element per value of the second dimension.'
                        },
                        {
                            type: CHART_TYPES.STACKED_AREA,
                            title: 'Stacked area',
                            description: 'Use to display data that can be summed.<br/> Break down once with several measures to stack the measures.<br/>  Break down twice to create one stack element per value of the second dimension.'
                        },
                        {
                            type: CHART_TYPES.LINES,
                            title: 'Lines',
                            description: 'Use to compare evolutions.<br/> Break down once with several measures to create one line per measure.<br/>  Break down twice to create one line per value of the second dimension.'
                        },
                        {
                            type: CHART_TYPES.SCATTER,
                            title: 'Scatter plot',
                            description: 'Scatterize'
                        }
                    ];
                    if (PluginsService.isPluginLoaded('geoadmin')) {
                        scope.chartTypes.push({
                            type: CHART_TYPES.MAP,
                            title: 'World map (BETA)',
                            description: 'Use to plot and aggregate geo data'
                        });
                    } else {
                        scope.chartTypes.push({
                            type: CHART_TYPES.MAP,
                            title: 'World map (BETA)',
                            description: 'Use to plot and aggregate geo data',
                            disabled: true,
                            disabledReason: 'You need to install the \'geoadmin\' plugin. Please see documentation'
                        });
                    }

                    scope.dataGroupingModes = DataGroupingMode;
                    scope.dataGroupingLabels = ChartLabels.DATA_GROUPING_LABELS;
                    scope.allYAxisModes = {
                        'NORMAL': { value: 'NORMAL', label: 'Normal', shortLabel: 'Normal' },
                        'LOG': { value: 'LOG', label: 'Logarithmic scale', shortLabel: 'Log' },
                        'PERCENTAGE_STACK': { value: 'PERCENTAGE_STACK', label: 'Normalize stacks at 100%', shortLabel: '100% stack' }
                    };
                    scope.allXAxisModes = {
                        'NORMAL': { value: 'NORMAL', label: 'Normal', shortLabel: 'Normal' },
                        'CUMULATIVE': { value: 'CUMULATIVE', label: 'Cumulative values', shortLabel: 'Cumulative' },
                        'DIFFERENCE': {
                            value: 'DIFFERENCE', label: 'Difference (replace each value by the diff to the previous one)',
                            shortLabel: 'Difference'
                        }
                    };
                    scope.allComputeModes = {
                        'NONE': { value: 'NONE', label: 'No computation', shortLabel: 'None' },
                        'LIFT_AVG': {
                            value: 'LIFT_AVG', shortLabel: 'Ratio to AVG',
                            label: 'Compute ratio of each value relative to average of values'
                        },
                        'AB_RATIO': {
                            value: 'AB_RATIO', shortLabel: 'a/b ratio',
                            label: 'Compute ratio of measure 1 / measure 2'
                        },
                        'AB_RATIO_PCT': {
                            value: 'AB_RATIO_PCT', shortLabel: 'a/b ratio (%)',
                            label: 'Compute ratio of measure 1 / measure 2, as percentage'
                        }
                    };

                    scope.initChartCommonScopeConfig(scope);
                    scope.graphError = { error: null };

                    scope.dateFilterParts = ChartFilterUtils.getDateChartFilterParts();
                    scope.dateFilterTypes = ChartFilterUtils.getDateFilterTypes();

                    scope.temporalBinningModes = ChartsStaticData.dateModes.concat(ChartsStaticData.UNBINNED_TREAT_AS_ALPHANUM);

                    scope.familyToTypeMap = {
                        'basic': [CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.PIE],
                        'table': [CHART_TYPES.PIVOT_TABLE],
                        'scatter': [CHART_TYPES.SCATTER, CHART_TYPES.GROUPED_XY, CHART_TYPES.BINNED_XY],
                        'map': [CHART_TYPES.SCATTER_MAP, CHART_TYPES.ADMINISTRATIVE_MAP, CHART_TYPES.GRID_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.DENSITY_HEAT_MAP],
                        'other': [CHART_TYPES.BOXPLOTS, CHART_TYPES.LIFT, CHART_TYPES.DENSITY_2D, CHART_TYPES.TREEMAP, CHART_TYPES.KPI, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.GAUGE],
                        'webapp': [CHART_TYPES.WEBAPP]
                    };

                    scope.canExportToExcel = ChartFeatures.canExportToExcel;
                    scope.canExportToImage = ChartFeatures.canExportToImage;
                    scope.appVersion = $rootScope.appConfig.version.product_version;

                    scope.getDownloadDisabledReason = function() {
                        return ChartFeatures.getExportDisabledReason(scope.chart.def);
                    };

                    scope.canDownloadChart = function() {
                        return scope.validity.valid && (scope.canExportToExcel(scope.chart.def) || scope.canExportToImage(scope.chart.def));
                    };

                    scope.typeAndVariantToImageMap = ChartIconUtils.typeAndVariantToImageMap;

                    scope.computeChartOptionsPreview = function() {
                        const macos = DetectUtils.getOS() === 'macos';
                        return macos ? '/static/dataiku/images/charts/previews/cmd.svg' : '/static/dataiku/images/charts/previews/ctrl.svg';
                    };

                    scope.computeChartPreview = function(type, variant) {
                        let imageName = '';
                        if (typeof (variant) === 'undefined') {
                            variant = 'normal';
                        }
                        if (typeof (scope.typeAndVariantToImageMap[type]) !== 'undefined'
                            && typeof (scope.typeAndVariantToImageMap[type][variant]) !== 'undefined'
                            && typeof (scope.typeAndVariantToImageMap[type][variant].preview) !== 'undefined') {
                            imageName = scope.typeAndVariantToImageMap[type][variant].preview;
                        }
                        if (imageName != '') {
                            return '/static/dataiku/images/charts/previews/' + imageName + '.png';
                        }
                        return false;
                    };

                    scope.request = {};

                    /*
                     * ------------------------------------------------------
                     * Property accessors and helpers
                     */

                    scope.ChartDimension = ChartDimension;
                    scope.isGroupedNumericalDimension = ChartDimension.isGroupedNumerical.bind(ChartDimension);
                    scope.isTimelineable = ChartDimension.isTimelineable.bind(ChartDimension);
                    scope.isUngroupedNumericalDimension = ChartDimension.isUngroupedNumerical.bind(ChartDimension);
                    scope.isAlphanumLikeDimension = ChartDimension.isAlphanumLike.bind(ChartDimension);
                    scope.isNumericalDimension = ChartDimension.isTrueNumerical.bind(ChartDimension);

                    scope.ChartUADimension = ChartUADimension;

                    scope.acceptStdAggrTooltipMeasure = function(data) {
                        return ChartTypeChangeHandler.stdAggregatedAcceptMeasureWithAlphanumResults(data);
                    };

                    scope.acceptUaTooltip = function(data) {
                        return ChartTypeChangeHandler.uaTooltipAccept(data);
                    };

                    scope.acceptFilter = function(data) {
                        const ret = ChartTypeChangeHandler.stdAggregatedAcceptDimension(data);
                        if (!ret.accept || !data) {
                            return ret;
                        }
                        if (data.type == 'GEOMETRY' || data.type == 'GEOPOINT') {
                            return {
                                accept: false,
                                message: 'Cannot filter on Geo dimensions'
                            };
                        }
                        return ret;
                    };

                    scope.dimensionBinDescription = function(dimension) {
                        return ChartDimension.getDimensionBinDescription(dimension, scope.chart.def);
                    };

                    scope.dateModeSuffix = function(mode) {
                        return `(${ChartDimension.getDateModeDescription(mode)})`;
                    };

                    scope.geoDimDescription = function(dim) {
                        for (let i = 0; i < ChartsStaticData.mapAdminLevels.length; i++) {
                            if (dim.adminLevel == ChartsStaticData.mapAdminLevels[i][0]) {
                                return ChartsStaticData.mapAdminLevels[i][2];
                            }
                        }
                        return translate('CHARTS.DROPPED_DIMENSION.UNKNOWN', 'Unknown');
                    };

                    scope.isDateRangeFilter = ChartFilterUtils.isDateRangeFilter.bind(ChartFilterUtils);
                    scope.isDatePartFilter = ChartFilterUtils.isDatePartFilter.bind(ChartFilterUtils);

                    /*
                     * ------------------------------------------------------
                     * Response handling / Facets stuff
                     */

                    scope.filterTmpDataWatchDeregister = null;

                    scope.onResponse = function() {
                        scope.setValidity({ valid: true });
                        scope.uiDisplayState = scope.uiDisplayState || {};
                        scope.uiDisplayState.chartTopRightLabel = ChartDataUtils.computeRecordsStatusLabel(
                            scope.response.result.pivotResponse.beforeFilterRecords,
                            scope.response.result.pivotResponse.afterFilterRecords,
                            ChartDimension.getComputedMainAutomaticBinningModeLabel(scope.response.result.pivotResponse, scope.chart.def),
                            undefined,
                            scope.chart.def.type
                        );

                        scope.uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(
                            scope.chart.def.type,
                            scope.response.result.pivotResponse.afterFilterRecords,
                            scope.response.result.pivotResponse.axesPairs,
                            undefined
                        );

                        scope.uiDisplayState.samplingSummaryMessage = ChartDataUtils.getSamplingSummaryMessage(scope.response.result.pivotResponse, scope.chart.def.type, scope.readOnly ? null : translate('CHARTS.HEADER.DATA_SAMPLING.CLICK_TO_OPEN', 'Click to open sampling settings'), undefined);

                        if (scope.chart.summary && scope.response.result.updatedSampleId) {
                            scope.chart.summary.requiredSampleId = scope.response.result.updatedSampleId;
                        }

                        // For the facet indexes to match the filter indexes, we introduce null entries in front of the filters using the minimal UI, as they don't have facets.
                        const responseFacetsStack = scope.response.result.pivotResponse.filterFacets.reverse();
                        scope.filterFacets = scope.chart.def.filters.map((filter) => {
                            if (filter.useMinimalUi) {
                                return null;
                            }
                            return responseFacetsStack.pop();
                        });
                        /*
                         * Filters need to compare the previous and current values of engineType and refreshableSelection to detect sampling changes.
                         * Because they are mutated on the chart object, we need to duplicate them for the change detection to work properly.
                         */
                        scope.samplingParams = {
                            engineType: scope.chart.engineType || 'LINO',
                            refreshableSelection: angular.copy(scope.chart.refreshableSelection)
                        };
                        scope.recordsMetadata = ChartFeatures.getRecordsMetadata(scope.chart.def, scope.response.result.pivotResponse.sampleMetadata, scope.response.result.pivotResponse.beforeFilterRecords, scope.response.result.pivotResponse.afterFilterRecords);
                        scope.display0Warning = ChartFeatures.shouldDisplay0Warning(scope.chart.def.type, scope.response.result.pivotResponse);
                        scope.colorColumn = ChartColorSelection.getColorDimensionOrMeasure(scope.chart.def, scope.chart.def.geoLayers[0]);

                        const axesDef = ChartDimension.getAxesDef(scope.chart.def, ChartDimension.getChartDimensions(scope.chart.def));
                        scope.chartData = ChartDataWrapperFactory.chartTensorDataWrapper(scope.response.result.pivotResponse, axesDef);
                    };

                    scope.allFilteredOut = false;

                    /*
                     * Wraps scope.getExecutePromise
                     * and add supports for automatic abortion
                     */
                    const executePivotRequest = MonoFuture(scope).wrap(scope.getExecutePromise);

                    scope.executeIfValid = function() {
                        const validity = ChartTypeChangeHandler.getValidity(scope.chart);
                        scope.setValidity(validity);
                        /*
                         * clear the response as well, otherwise when changing the chart, it will
                         * first run once with the new settings and the old response, producing
                         * js errors when something drastic (like chart type) changes
                         */
                        scope.previousResponseHadResult = (scope.response && scope.response.hasResult);
                        scope.response = null;

                        if (validity.valid) {
                            Logger.info('Chart is OK, executing');
                            scope.execute();
                        } else {
                            scope.allFilteredOut = false;
                            Logger.info('Chart is NOK, not executing', scope.validity);
                        }
                    };

                    // fetch the response
                    scope.execute = Debounce()
                        .withDelay(1, 300)
                        .withScope(scope)
                        .withSpinner(true)
                        .wrap(function() {

                            Logger.info('Debounced, executing');
                            scope.executedOnce = true;

                            let request = null;

                            try {
                                const wrapper = element.find('.chart-zone');
                                const width = wrapper.width();
                                const height = wrapper.height();
                                scope.chartSpecific = {
                                    ...scope.chartSpecific,
                                    datasetProjectKey: scope.getDataSpec().datasetProjectKey,
                                    datasetName: scope.getDataSpec().datasetName,
                                    context: scope.getCurrentChartsContext()
                                };
                                request = ChartRequestComputer.compute(scope.chart.def, width, height, scope.chartSpecific);
                                request.useLiveProcessingIfAvailable = scope.chart.def.useLiveProcessingIfAvailable;
                                Logger.info('Request is', request);
                                scope.graphError.error = null;
                            } catch (error) {
                                Logger.info('Not executing, chart is not ready', error);
                                scope.graphError.error = error;
                            }
                            // We are sure that request is valid so we can generate the name
                            if (!scope.chart.def.userEditedName) {
                                const newName = ChartTypeChangeHandler.computeAutoName(scope.chart.def);
                                if (newName.length > 0) {
                                    scope.chart.def.name = newName;
                                }
                            }

                            scope.filter = { 'query': undefined };
                            resetErrorInScope(scope);

                            scope.excelExportableChart = undefined;
                            const chartDefCopy = angular.copy(scope.chart.def);

                            executePivotRequest(request, undefined, false).update(function(data) {
                                scope.request = request;
                                scope.response = data;

                            }).success(function(data) {
                                // For Excel export
                                scope.excelExportableChart = {
                                    pivotResponse: data.result.pivotResponse,
                                    chartDef: chartDefCopy
                                };

                                scope.request = request;
                                scope.response = data;
                                scope.allFilteredOut = false;
                                scope.onResponse();

                            }).error(function(data, status, headers) {
                                onPivotRequestError(scope)(data, status, headers);
                            });
                        });

                    function getChangedDefinitionMessage(changedDefinition) {
                        const { name, nv, ov } = changedDefinition;
                        return `${name} \nbefore: ${JSON.stringify(ov)} \nafter: ${JSON.stringify(nv)}`;
                    }

                    function onChartImportantDefinitionChanged(changedDefinition, newChartDef) {
                        Logger.info(`Chart important change: ${getChangedDefinitionMessage(changedDefinition)}`);

                        if (newChartDef) {
                            const oldChartDef = angular.copy(scope.chart.def);
                            scope.recomputeAndUpdateData(changedDefinition);

                            if (!angular.equals(oldChartDef, scope.chart.def)) {
                                Logger.info('Data has been modified, not executing --> will execute at next cycle');
                                return;
                            }
                            Logger.info('Triggering executeIfValid');
                            scope.executeIfValid();
                        }
                    }

                    function onChartFrontImportantDefinitionChanged(changedDefinition, newChartDef) {
                        Logger.info(`Chart front important change: ${getChangedDefinitionMessage(changedDefinition)}`);
                        if (newChartDef) {
                            if (!scope.insight) {
                                scope.saveChart();
                            }
                            scope.redraw({ updateThumbnail: true });
                        }
                    }

                    function onChartFrontImportantNoRedrawDefinitionChanged(changedDefinition, newChartDef) {
                        Logger.info(`Chart front important no redraw change: ${getChangedDefinitionMessage(changedDefinition)}`);
                        if (newChartDef && !scope.insight) {
                            scope.saveChart();
                        }
                    }

                    function formatDefinition(chartDef) {
                        if (chartDef.type === CHART_TYPES.GEOMETRY_MAP) {
                            const flattenedChartDef = angular.copy(chartDef);
                            const listPropertiesToFlatten = ['geometry', 'uaColor'];
                            for (const property of listPropertiesToFlatten) {
                                flattenedChartDef[property] = [];
                                for (const geoLayer of chartDef.geoLayers) {
                                    flattenedChartDef[property].push(getFirstValue(geoLayer[property]));
                                }
                            }

                            flattenedChartDef['colorOptions'] = [];
                            for (const geoLayer of chartDef.geoLayers) {
                                flattenedChartDef['colorOptions'].push(geoLayer['colorOptions']);
                            }
                            return flattenedChartDef;
                        }
                        return chartDef;
                    }

                    function getFirstValue(array) {
                        if (array.length > 0) {
                            return array[0];
                        }
                        return {};
                    }

                    //This watch needs to be set before the scope.chart.def watch
                    scope.$watch('chart.theme', (nv, ov) => {
                        Logger.info('Chart theme change');
                        scope.immutableTheme = nv ? angular.copy(nv) : undefined;
                        $timeout.cancel(themeChangeTimeoutId);
                        if (!scope.insight) {
                            themeChangeTimeoutId = $timeout(() => {
                                scope.saveChart();
                            }, 600);
                        }
                        /*
                         * Updating the theme palettes doesn't trigger a change detection cycle on
                         * chart.def as only the theme changes. The chart still needs to be redrawn.
                         */
                        if (nv && (!ov || !angular.equals(nv.themePalettes, ov.themePalettes))) {
                            scope.redraw({ updateThumbnail: true });
                        }
                    });

                    // thumbnailData changes are handled by a dedicated watcher
                    const ignoredChartDefFields = ['thumbnailData'];

                    scope.$watch(() => _.omit(scope.chart.def, ignoredChartDefFields), function(nv, ov) {
                        if (!nv) {
                            return;
                        }
                        if (!ov) {
                            onChartImportantDefinitionChanged({ name: 'initial', nv, ov }, nv);
                        }

                        if (nv.tooltipOptions && nv.tooltipOptions.display == false && ov.tooltipOptions.display == true) {
                            $rootScope.$emit('unfixTooltip');
                        }

                        $timeout.cancel(invalidChangeTimeoutId);

                        const invalidMessage = ChartDefinitionChangeHandler.getInvalidChangeMessage(nv);
                        if (invalidMessage !== null) {
                            // Timeouting to prevent displaying warning while user is still typing in a custom input.
                            invalidChangeTimeoutId = $timeout(() => {
                                ActivityIndicator.hide();
                                ActivityIndicator.warning('Not refreshing: ' + invalidMessage);
                            }, ChartsStaticData.SAVE_DEBOUNCE_DURATION);
                            return;
                        }

                        scope.immutableChartDef = angular.copy(nv);

                        const newValue = formatDefinition(nv);
                        const oldValue = formatDefinition(ov);

                        // Check for a change which triggers save + recompute + redraw
                        const importantChangedDefinition = ChartDefinitionChangeHandler.getRecomputeChange(newValue, oldValue);
                        if (importantChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            $timeout.cancel(frontImportantChangeTimeoutId); // apply immediate change, prevail on delayed changes
                            onChartImportantDefinitionChanged(importantChangedDefinition, newValue);
                            return;
                        }
                        // Check for a change which triggers save + redraw
                        const frontChangedDefinition = ChartDefinitionChangeHandler.getFrontImportantChange(newValue, oldValue);
                        if (frontChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            $timeout.cancel(frontImportantChangeTimeoutId); // apply immediate change, prevail on delayed changes
                            onChartFrontImportantDefinitionChanged(frontChangedDefinition, newValue);
                            return;
                        }
                        // Check for a change which triggers save + redraw after a timeout
                        const delayedChangedDefinition = ChartDefinitionChangeHandler.getDelayedFrontImportantChange(newValue, oldValue);
                        if (delayedChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            // Timeouting to prevent excessive refresh while user is still typing in a custom input.
                            $timeout.cancel(frontImportantChangeTimeoutId);
                            frontImportantChangeTimeoutId = $timeout(() => {
                                onChartFrontImportantDefinitionChanged(delayedChangedDefinition, newValue);
                            }, ChartsStaticData.SAVE_DEBOUNCE_DURATION);
                            return;
                        }
                        // Check for a change which triggers save
                        const noRedrawChangedDefinition = ChartDefinitionChangeHandler.getNoRedrawChange(newValue, oldValue);
                        if (noRedrawChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            $timeout.cancel(noRedrawDelayedChangeTimeoutId); // apply immediate change, prevail on delayed changes
                            onChartFrontImportantNoRedrawDefinitionChanged(noRedrawChangedDefinition, newValue);
                            return;
                        }
                        // Check for a change which triggers save after a timeout
                        const noRedrawDelayedChangedDefinition = ChartDefinitionChangeHandler.getDelayedNoRedrawChange(newValue, oldValue);
                        if (noRedrawDelayedChangedDefinition) {
                            $timeout.cancel(themeChangeTimeoutId);
                            // Timeouting to prevent excessive refresh while user is still typing in a custom input.
                            $timeout.cancel(noRedrawDelayedChangeTimeoutId);
                            noRedrawDelayedChangeTimeoutId = $timeout(() => {
                                onChartFrontImportantNoRedrawDefinitionChanged(noRedrawDelayedChangedDefinition, newValue);
                            }, ChartsStaticData.SAVE_DEBOUNCE_DURATION);
                            return;
                        }
                    }, true);

                    const resetImmutableChartDef = function(nv, ov) {
                        if (!angular.equals(nv, ov)) {
                            scope.immutableChartDef = angular.copy(scope.chart.def);
                        }
                    };

                    scope.$watch('chart.summary', () => {
                        scope.chartUsableColumns = angular.copy(scope.chart.summary ? scope.chart.summary.usableColumns : []);
                    });

                    scope.$watch('chart.def.$axisSpecs', resetImmutableChartDef);

                    scope.$watch('chart.def.xCustomExtent.$autoExtent', resetImmutableChartDef);

                    //  Checks only the first y axis formatting options for $autoExtent, if one is updated, others are.
                    scope.$watch('chart.def.yAxesFormatting[0].customExtent.$autoExtent', resetImmutableChartDef);

                    scope.$watch('chart.def.$zoomControlInstanceId', resetImmutableChartDef);

                    scope.$watch('chart.def.thumbnailData', function(nv, ov) {
                        if (nv !== ov && !scope.insight) {
                            scope.saveChart(true);
                        }
                    });

                    $(window).on('resize.chart_logic', function(e) {
                        if (e.detail && e.detail.skipInCharts) {
                            return;
                        }
                        handleResize(scope);
                    });
                    scope.$on('$destroy', function() {
                        $(window).off('resize.chart_logic');
                        resizeObserver.disconnect();
                    });
                    const chartParamBar = document.querySelector('.chart-param-bar');
                    scope.chartParamBarHeight = chartParamBar.offsetHeight;
                    scope.chartParamBarWidth = chartParamBar.offsetWidth;
                    resizeObserver.observe(chartParamBar);

                    scope.forceExecute = function(options) {
                        if (options !== undefined) {
                            const { store, id } = ChartStoreFactory.getOrCreate(scope.chart.def.$chartStoreId);
                            scope.chart.def.$chartStoreId = id;
                            store.setRequestOptions(options);
                        }
                        scope.recomputeAndUpdateData();
                        scope.executeIfValid();
                    };

                    scope.rebuildSampling = function() {
                        const refreshTrigger = new Date().getTime();
                        if (scope.chart.copySelectionFromScript) {
                            scope.shaker.explorationSampling._refreshTrigger = refreshTrigger;
                        } else {
                            scope.chart.refreshableSelection._refreshTrigger = refreshTrigger;
                        }
                        scope.$emit('chartSamplingChanged', { chart: scope.chart });
                    };

                    scope.$on('forceExecuteChart', function() {
                        scope.forceExecute();
                    });

                    scope.$emit('listeningToForceExecuteChart'); // inform datasetChartBase directive that forceExecute() can be triggered through broadcast

                    scope.redraw = function(options) {
                        scope.$broadcast('redraw', options);
                    };

                    scope.revertToLinoEngineAndReload = function() {
                        scope.chart.engineType = 'LINO';
                        scope.forceExecute();
                    };

                    scope.craftBinningFormChart = function(params) {
                        $rootScope.globallyOpenContextualMenu = undefined; // close currently opened one
                        scope.$emit('craftBinningFormChart', { dimension: angular.copy(params.dimension), dimensionRef: params.dimension, isEditMode: params.isEditMode, hideOneTickPerBin: params.hideOneTickPerBin });
                    };

                    /*
                     * ------------------------------------------------------
                     * Recompute/Update handlers
                     */
                    scope.recomputeAndUpdateData = function(changedDefinition) {
                        ChartTypeChangeHandler.fixupSpec(scope.chart, scope.chart.theme, changedDefinition);

                        scope.canHasTooltipMeasures = [
                            CHART_TYPES.MULTI_COLUMNS_LINES,
                            CHART_TYPES.GROUPED_COLUMNS,
                            CHART_TYPES.STACKED_COLUMNS,
                            CHART_TYPES.STACKED_BARS,
                            CHART_TYPES.GRID_MAP,
                            CHART_TYPES.LINES,
                            CHART_TYPES.STACKED_AREA,
                            CHART_TYPES.ADMINISTRATIVE_MAP,
                            CHART_TYPES.PIE,
                            CHART_TYPES.BINNED_XY
                        ].includes(scope.chart.def.type);

                        scope.canAnimate = ChartFeatures.canAnimate(scope.chart.def.type);
                        scope.canFacet = ChartFeatures.canFacet(scope.chart.def.type, scope.chart.def.webAppType);
                        scope.canFilter = ChartFeatures.canFilter(scope.chart.def.type, scope.chart.def.webAppType);

                        return;
                    };

                    scope.setChartType = function(chartId) {
                        let newChartType;
                        if (scope.availableChartTypes || scope.webapps) {
                            newChartType = [...(scope.availableChartTypes || []), ...(scope.webapps || [])].find(item => item.id === chartId);
                        }
                        if (!newChartType) {
                            return;
                        }
                        WT1.event('chart-type-change', {
                            chartId: `${$stateParams.projectKey.dkuHashCode()}.${getWT1ContextName().dkuHashCode()}.${scope.chart.def.name.dkuHashCode()}`,
                            chartType: newChartType.type,
                            chartVariant: newChartType.variant
                        });

                        const oldChartType = scope.chart.def.type;
                        element.find('.pivot-charts .mainzone').remove(); // avoid flickering
                        Logger.info('Set chart type');
                        ChartTypeChangeHandler.onChartTypeChange(scope.chart.def, newChartType.type, newChartType.webappType);
                        Logger.info('AFTER chart type', scope.chart.def);
                        scope.chart.def.type = newChartType.type;
                        scope.chart.def.variant = newChartType.variant;
                        scope.chart.def.webAppType = newChartType.webappType;
                        if (!scope.validity.valid) {
                            // If chart is not ready and we changed, we can apply things related to new chart type
                            DSSVisualizationThemeUtils.applyToChart({ chart: scope.chart.def, theme:scope.chart.theme });
                        }

                        if (scope.chart.def.$zoomControlInstanceId) {
                            ChartZoomControlAdapter.clear(scope.chart.def.$zoomControlInstanceId);
                            scope.chart.def.$zoomControlInstanceId = null;
                        }

                        onChartImportantDefinitionChanged({ name: 'type', nv: newChartType.type, ov: oldChartType }, scope.chart.def);
                        if (!scope.$$phase) {
                            scope.$apply(); // sc-115878
                        }
                    };
                });

                scope.exportToImage = function() {
                    scope.$broadcast('export-chart');
                    WT1.event('chart-download', {
                        chartId: `${$stateParams.projectKey.dkuHashCode()}.${getWT1ContextName().dkuHashCode()}.${scope.chart.def.name.dkuHashCode()}`,
                        format: 'image',
                        chartType: scope.chart.def.type,
                        chartVariant: scope.chart.def.variant
                    });
                    if (scope.displayChartOptionsMenu) {
                        scope.toggleChartOptionsMenu();
                    }
                };

                scope.exportToExcel = function() {
                    if (scope.excelExportableChart) {
                        let animationFrameIdx;
                        const chartDef = scope.chart.def;
                        const colorMapsArray = scope.getColorMap();
                        const pivotResponse = scope.excelExportableChart.pivotResponse;

                        if (chartDef.animationDimension.length) {
                            animationFrameIdx = scope.animation.currentFrame;
                        }

                        DataikuAPI.shakers.charts.exportToExcel(
                            chartDef,
                            pivotResponse,
                            animationFrameIdx,
                            colorMapsArray
                        ).success(data => {
                            WT1.event('chart-download', {
                                chartId: `${$stateParams.projectKey.dkuHashCode()}.${getWT1ContextName().dkuHashCode()}.${scope.chart.def.name.dkuHashCode()}`,
                                format: 'excel',
                                chartType: chartDef.type,
                                chartVariant: chartDef.variant
                            });
                            downloadURL(DataikuAPI.shakers.charts.downloadExcelUrl(data.id));
                        }).error(setErrorInScope.bind(scope));
                        if (scope.displayChartOptionsMenu) {
                            scope.switchchartOptionsMenu();
                        }
                    }
                };

                scope.getColorMap = function() {
                    const colorMapsArray = [];

                    if (scope.chart.def.colorMode === 'COLOR_GROUPS') {
                        scope.getColorFromConditionalFormatting(colorMapsArray);
                    } else {
                        scope.getColorFromColorScale(colorMapsArray);
                    }
                    return colorMapsArray;
                },

                scope.getColorFromColorScale = function(colorMapArray) {
                    const pivotResponse = scope.excelExportableChart.pivotResponse;
                    if (scope.chart.def.colorMeasure.length && pivotResponse.aggregations.length > 1) {
                        const colorScale = scope.legendsWrapper.getLegend(0).scale;
                        const backgroundColorMap = {};
                        const fontColorMap = {};
                        pivotResponse.aggregations[pivotResponse.aggregations.length - 1].tensor.forEach((value, index) => {
                            const color = colorScale(value, index);

                            if (!(value in backgroundColorMap) && color) {
                                backgroundColorMap[value] = ColorUtils.toHex(color);
                                fontColorMap[value] = ColorUtils.getFontContrastColor(color, scope.chart.theme && scope.chart.theme.generalFormatting.fontColor);
                            }
                        });
                        colorMapArray.push({ backgroundColorMap, fontColorMap });
                    }
                },

                scope.getColorFromConditionalFormatting = function(colorMapsArray) {
                    scope.chart.def.genericMeasures.forEach((measure, measureIndex) => {
                        colorMapsArray.push(scope.retrieveColorMaps(scope.chart.def.colorGroups, measure, measureIndex));
                    });
                    return colorMapsArray;
                },

                scope.retrieveColorMaps = function(colorGroups, measure, measureIndex) {
                    const backgroundColorMap = {};
                    const fontColorMap = {};
                    const pivotResponse = scope.excelExportableChart.pivotResponse;
                    const measureId = ConditionalFormattingOptions.getMeasureId(measure);
                    const measureGroup = colorGroups.find(group => group.appliedColumns && group.appliedColumns.map(column => ConditionalFormattingOptions.getMeasureId(column)).includes(measureId));

                    pivotResponse.aggregations[measureIndex].tensor.forEach((value, valueIndex) => {
                        if (!_.isNil(measureGroup)) {
                            value = scope.chartData.getNonNullCount(valueIndex, measureIndex) > 0 ? value : '';

                            const ruleClass = ConditionalFormattingOptions.getColorRuleClass(value, measureGroup.rules, measure, scope.chart.theme);
                            const colorFormattingClass = ruleClass.class;

                            let fontColor = '#000000';
                            let backgroundColor = '#FFFFFF';
                            if (colorFormattingClass.includes('text')) {
                                fontColor = ConditionalFormattingOptions.getStripeColor(colorFormattingClass, {});
                            } else if (colorFormattingClass.includes('background')) {
                                fontColor = '#FFFFFF'; // fontcolor always white with background color.
                                backgroundColor = ConditionalFormattingOptions.getStripeColor(colorFormattingClass, {});
                            } else if (colorFormattingClass.includes('custom')) {
                                fontColor = ruleClass.customColors.customFontColor || fontColor;
                                backgroundColor = ruleClass.customColors.customBackgroundColor || backgroundColor;
                            }
                            scope.retrieveColor(fontColorMap, fontColor, value);
                            scope.retrieveColor(backgroundColorMap, backgroundColor, value);
                        }
                    });
                    return { backgroundColorMap, fontColorMap };
                };

                scope.retrieveColor = function(colorMap, color, measureValue) {
                    if (!(measureValue in colorMap && color)) {
                        colorMap[measureValue] = color;
                    }
                },

                scope.openPasteModalFromKeydown = function(data) {
                    !scope.readOnly && ChartConfigurationCopyPaste.pasteChartFromClipboard({ valid: scope.validity.valid, isInInsight: scope.isInInsight }, data).then(res => scope.pasteChart(res, scope.currentChart.index));
                };

                scope.keydownCopy = function() {
                    // copy chart only if the selection is empty
                    if (window.getSelection().toString() === '') {
                        ChartConfigurationCopyPaste.copyChartToClipboard(scope.chart.def, scope.appVersion, scope.chart.theme);
                    }
                };

                scope.deleteCurrentChart = function() {
                    scope.deleteChart(scope.currentChart.index);
                };

                scope.duplicateChart = function() {
                    scope.addChart({ chart: scope.chart, index: scope.currentChart.index, copyOfName: true, wt1Event: 'chart-duplicate' });
                };

                scope.displayChartOptionsMenu = false;
                scope.toggleChartOptionsMenu = function() {
                    scope.displayChartOptionsMenu = !scope.displayChartOptionsMenu;
                    if (scope.displayChartOptionsMenu) {
                        $timeout(function() {
                            $(window).on('click', scope.switchChartOptionsMenuOnClick);
                        });
                    } else {
                        $(window).off('click', scope.switchChartOptionsMenuOnClick);
                    }
                };

                scope.switchChartOptionsMenuOnClick = function(e) {
                    const clickedEl = e.target;
                    if ($(clickedEl).closest('.chart-options-wrapper').length <= 0 && scope.displayChartOptionsMenu) {
                        scope.toggleChartOptionsMenu();
                        scope.$apply();
                    }
                };

                scope.blurElement = function(inputId) {
                    $timeout(function() {
                        $(inputId).blur();
                    });
                };

                scope.blurTitleEdition = function() {
                    scope.editingTitle = false;
                    scope.chart.def.userEditedName = true;
                    $timeout(scope.saveChart);
                    if (scope.excelExportableChart) {
                        scope.excelExportableChart.chartDef.name = scope.chart.def.name;
                    }
                };

                scope.editChartTitle = function() {
                    scope.editingTitle = !(scope.isInInsight && $rootScope.topNav.tab === 'view');
                };

                scope.openChartSamplingTab = function() {
                    $rootScope.$broadcast('tabSelect', 'sampling-engine');
                };

                function getWT1ContextName() {
                    const context = scope.analysisCoreParams || scope.insight || scope.dataset || { name: 'unknown' };
                    return context.name;
                }

                function shouldDisplaySamplingStatus() {
                    const pivotResponse = scope.response && scope.response.result && scope.response.result.pivotResponse;
                    const sampleMetadata = pivotResponse && pivotResponse.sampleMetadata;
                    return (sampleMetadata && !sampleMetadata.sampleIsWholeDataset);
                }

                scope.shouldDisplayPointRangeWarning = () => {
                    return scope.chart.def.type === CHART_TYPES.SCATTER && scope.uiDisplayState.lowPointSizeRangeWarning;
                };

                scope.hasBetaFormattingPane = function(chartType) {
                    const charts = [
                        CHART_TYPES.STACKED_COLUMNS,
                        CHART_TYPES.GROUPED_COLUMNS,
                        CHART_TYPES.STACKED_BARS,
                        CHART_TYPES.STACKED_AREA,
                        CHART_TYPES.PIVOT_TABLE,
                        CHART_TYPES.SCATTER,
                        CHART_TYPES.SCATTER_MULTIPLE_PAIRS,
                        CHART_TYPES.LINES,
                        CHART_TYPES.TREEMAP,
                        CHART_TYPES.GEOMETRY_MAP,
                        CHART_TYPES.SCATTER_MAP,
                        CHART_TYPES.ADMINISTRATIVE_MAP,
                        CHART_TYPES.GRID_MAP,
                        CHART_TYPES.DENSITY_HEAT_MAP,
                        CHART_TYPES.MULTI_COLUMNS_LINES,
                        CHART_TYPES.PIE,
                        CHART_TYPES.SANKEY,
                        CHART_TYPES.GROUPED_XY,
                        CHART_TYPES.BINNED_XY,
                        CHART_TYPES.LIFT,
                        CHART_TYPES.BOXPLOTS,
                        CHART_TYPES.DENSITY_2D,
                        CHART_TYPES.RADAR,
                        CHART_TYPES.KPI,
                        CHART_TYPES.GAUGE
                    ];
                    return charts.includes(chartType);
                };

                scope.assignToChartDef = function(propertyPath, value, merge) {
                    let assignedValue = value;

                    //  If we use the merge strategy, we assign properties from the incoming value to the existing object, otherwise, we just replace it
                    if (merge) {
                        assignedValue = _.assign(_.get(scope.chart.def, propertyPath), value);
                    }

                    _.set(scope.chart.def, propertyPath, assignedValue);
                    $timeout(() => scope.$apply());
                };

                scope.onChartDefPropertyChange = function(event) {
                    if (event.id) {
                        //  If we provide an id, it targets yAxesFormatting
                        scope.updateYAxisFormatting(event.key, event.value, event.id);
                    } else {
                        scope.assignToChartDef(event.key, event.value, event.merge);
                    }
                };

                scope.onFormattingPaneTabChange = function() {
                    $rootScope.$broadcast('reflow');
                };

                scope.onFetchColumnsSummaryForPalette = function() {
                    if ($rootScope.fetchColumnsSummaryForCurrentChart) {
                        $rootScope.fetchColumnsSummaryForCurrentChart(true).then($rootScope.redraw);
                    } else {
                        $rootScope.fetchColumnsSummary().then($rootScope.redraw);
                    }
                };

                scope.updateYAxisFormatting = (key, value, id) => {
                    const yAxesFormatting = _.clone(scope.chart.def.yAxesFormatting);
                    const axisFormatting = ChartAxesUtils.getFormattingForYAxis(yAxesFormatting, id);
                    _.set(axisFormatting, key, value);
                    scope.assignToChartDef('yAxesFormatting', yAxesFormatting);
                };
            }
        };

        function onPivotRequestError(scope) {
            return (data, status, headers) => {
                scope.response = undefined;
                scope.allFilteredOut = false;
                scope.display0Warning = false;

                if (data.code === 'FILTERED_OUT') {
                    scope.allFilteredOut = true;
                } else if (data.code === 'CANNOT_DISPLAY_WITH_EMPTY_AXES') {
                    scope.display0Warning = true;
                } else {
                    if (ChartTypeChangeHandler.hasRequestResponseWarning(scope.chart.def, data)) {
                        scope.setValidity(ChartTypeChangeHandler.getRequestResponseWarning(scope.chart.def, data));
                    } else if (ChartDataUtils.isPivotRequestAborted(data)) {
                        // Manually aborted => do not report as error
                    } else {
                        scope.chartSetErrorInScope(data, status, headers);
                    }
                }
            };
        }

        function parseWebapps(data) {
            const webapps = [];
            data.forEach(w => {
                webapps.push({
                    id: w.desc.id,
                    displayName: w.desc.meta.label || w.desc.id,
                    type: CHART_TYPES.WEBAPP,
                    variant: CHART_VARIANTS.normal,
                    webappType: w.webappType,
                    isWebapp: true
                });
            });
            if (webapps) {
                webapps.sort((a, b) => a.webappType.localeCompare(b.webappType));
            }
            return webapps;
        }

        function handleResize(scope) {
            const debouncedRedrawAfterResize = Debounce()
                .withDelay(1, 300)
                .withScope(scope)
                .withSpinner(false)
                .wrap(f => {
                    Logger.debug('Redrawing chart after resize (debounced)');
                    scope.redraw();
                });

            Logger.debug('Window was resized, will redraw chart a bit later');
            if (scope.chart.def.type == CHART_TYPES.BINNED_XY && scope.chart.def.variant == CHART_VARIANTS.binnedXYHexagon) {
                scope.recomputeAndUpdateData();
                scope.executeIfValid();
            } else {
                debouncedRedrawAfterResize();
            }
            scope.$apply();
        }
    });

})();
