(function () {
    "use strict";

    const nestedFilter = {
        bindings: {
            nestedFilters: '<',
            filterDesc: '=',
            schema: '<',
            filterHeader: '<',
            fromPrepare: '<',
            columnName: '<',
            onEnterKeydown:'<?',
            enumValuesProvider:'<?',
            supportedBooleanOperators: '<?',
            supportedFiltersOperators: '<?',
        },
        templateUrl: "/static/dataiku/nested-filters/nested-filter/nested-filter.component.html",
        controller: function nestedFilterController($scope, $element, $attrs, FilterConditionsEditorService, FiltersService, translate) {
        /*
         * This component implements recursion that allows for recursive filter conditions handling.
         * Each nestedFilter component can have two types of children :
         *      - filterConditionsEditor components, defining a list of traditional filter conditions.
         *      - other nestedFilter components, in case of presence of subfilters. This implements recursion.
         * Each nestedFilter component of the recursion tree works on its own level $scope.nestedFilters list.
         * The rootNestedFilter variable contains all the information describing a complete filter, with recursive parts. It is defined at the
         * top level nestedFilter component and passed as is through each children.
         * As the rootNestedFilter completely describes a filter, it is used to update the filterDesc backend in the updateFilterDesc method.
         * Similarly, when creating a nestedFilter, we read the filterDesc from the backend to initialize it properly
         */
            const ctrl = this;
            ctrl.$onChanges = function() {
                $scope.enumValuesProvider = ctrl.enumValuesProvider;
                $scope.supportedBooleanOperators = ctrl.supportedBooleanOperators || ["&&", "||"]
                const mapToString = {
                    "&&": ["FILTER.AND", "And"],
                    "||": ["FILTER.OR", "Or"]
                }
                $scope.booleanOptions = $scope.supportedBooleanOperators.map(so =>  [so, translate(mapToString[so][0], mapToString[so][1])]); 
                $scope.supportedFiltersOperators = ctrl.supportedFiltersOperators;
            }
            
            ctrl.$onInit = function () {
                $scope.onEnterKeydown = ctrl.onEnterKeydown;
                // $scope.columns = $scope.$parent.columns;
                $scope.schema = ctrl.schema || $scope.$parent.schema;
                $scope.filterDescToNestedFilters = function(filterDesc, recursionDepth, position) {
                /*
                 Used to reload to the frontend a saved filter.
                 */
                    let filter = {}
                    filter.conditions = filterDesc.uiData.conditions.filter(condition => !("subCondition" in condition));
                    if (filterDesc.uiData && ["&&", "||"].includes(filterDesc.uiData.mode)) {
                        filter.mode = filterDesc.uiData.mode;
                    } else if (filterDesc.uiData && filterDesc.uiData.$latestOperator) {
                        filter.mode = filterDesc.uiData.$latestOperator;
                    } else {
                        filter.mode = "&&";
                    }
                    filter.recursionDepth = recursionDepth;
                    filter.position = position;
                    let subConditions = filterDesc.uiData.conditions.filter(condition => "subCondition" in condition)
                    filter.nestedFilters = []
                    for (let i=0; i<subConditions.length; i++) {
                        filter.nestedFilters.push($scope.filterDescToNestedFilters(subConditions[i].subCondition, recursionDepth+1, i));
                    }
                    return filter;
                }

                $scope.initNestedFilters = function(ctrl) {

                    if (!$scope.filterDesc.uiData) {
                        $scope.filterDesc.uiData = { mode: '&&', conditions: [] };
                    }

                    if (["&&", "||"].includes($scope.filterDesc.uiData.mode)) {
                        $scope.mode = $scope.filterDesc.uiData.mode
                    } else if ($scope.filterDesc.uiData.$latestOperator) {
                        $scope.mode = $scope.filterDesc.uiData.$latestOperator;
                    }
                    else if ($scope.$parent.mode) {
                        if (["&&", "||"].includes($scope.$parent.mode)) {
                            $scope.mode = $scope.$parent.mode;
                        }
                    } else {
                        $scope.mode = "&&"
                    }

                    if ("nestedFilters" in ctrl) {
                        $scope.nestedFilters = ctrl["nestedFilters"] || [$scope.filterDescToNestedFilters($scope.filterDesc, 0, 0)];
                    } else {
                        $scope.nestedFilters = [$scope.filterDescToNestedFilters($scope.filterDesc, 0, 0)];
                    }
                    if ("filterHeader" in ctrl) {
                        $scope.filterHeader = ctrl["filterHeader"] || "Where";
                    } else {
                        $scope.filterHeader = "Where";
                    }
                    $scope.filterHeader = ("filterHeader" in ctrl) ? (ctrl["filterHeader"] || "Where") : "Where";
                    $scope.rootNestedFilters = ($scope.$parent.rootNestedFilters) ? $scope.$parent.rootNestedFilters : $scope.nestedFilters;
                    $scope.isTopNestedFilter = !($scope.$parent.rootNestedFilters);
                }
                $scope.filterDesc = ctrl["filterDesc"];
                $scope.initNestedFilters(ctrl);
                $scope.fromPrepare = ctrl["fromPrepare"] || false;
                $scope.columnName = ctrl["columnName"] || "";
                /*
                    Scope methods
                */
                $scope.addCondition = function() {
                    let newCondition = {};
                    if (!$scope.columnName) {
                        if ($scope.nestedFilters[0].conditions.length > 0) {
                            const lastConditionIndex = $scope.nestedFilters[0].conditions.length - 1;
                            newCondition = {num: 0, "input": $scope.nestedFilters[0].conditions[lastConditionIndex].input,
                                                    "operator": $scope.nestedFilters[0].conditions[lastConditionIndex].operator};
                        } else {
                            newCondition = {num: 0};
                        }
                    } else {
                        newCondition = {"num":0, "operator":"== [number]", "input":$scope.columnName};
                    }
                    FilterConditionsEditorService.update($scope,newCondition, $scope.fromPrepare);
                    $scope.nestedFilters[0].conditions.push(newCondition);
                }

                $scope.addGroup = function() {
                    let lastConditionColumn = $scope.nestedFilters[0].conditions[$scope.nestedFilters[0].conditions.length - 1].input;
                    $scope.nestedFilters[0].nestedFilters.push({
                          "recursionDepth": $scope.nestedFilters[0].recursionDepth+1,
                          "conditions": [{num: 0, "input": lastConditionColumn}],
                          "nestedFilters": [],
                          "mode": FiltersService.getOtherMode($scope.mode),
                          "position": $scope.nestedFilters[0].nestedFilters.length
                           });
                }

                $scope.canRemoveGroup = function(nestedFilter, recursionDepth){
                    return recursionDepth > 0 && (nestedFilter.conditions.length!==0 || nestedFilter.nestedFilters.length > 1 || recursionDepth > 1);
                }

                $scope.removeGroup = function(position) {
                    $scope.$parent.nestedFilters[0].nestedFilters.splice(position,1);
                    for(let i=0; i< $scope.$parent.nestedFilters[0].nestedFilters.length; i++){
                         $scope.$parent.nestedFilters[0].nestedFilters[i].position = i;
                    }
                }

                $scope.unGroup = function(position) {
                    let parentNestedFiltersCopy = angular.copy($scope.$parent.nestedFilters[0]);
                    let nestedFiltersCopy = angular.copy($scope.nestedFilters[0]);
                    let newConds = parentNestedFiltersCopy.conditions.concat(nestedFiltersCopy.conditions);
                    let newNestedFilters = parentNestedFiltersCopy.nestedFilters.concat(nestedFiltersCopy.nestedFilters);
                    $scope.$parent.nestedFilters[0].nestedFilters = newNestedFilters;
                    $scope.$parent.nestedFilters[0].conditions = newConds;
                    $scope.removeGroup(position);
                }

                $scope.duplicateGroup = function(position) {
                    let newGroup = {
                        "conditions": $scope.nestedFilters[0].conditions,
                        "nestedFilters": $scope.nestedFilters[0].nestedFilters,
                        "mode": $scope.nestedFilters[0].mode,
                        "position": position+1,
                        "recursionDepth": $scope.nestedFilters[0].recursionDepth
                    };
                    $scope.$parent.nestedFilters[0].nestedFilters.splice(position+1,0,angular.copy(newGroup));
                };

                $scope.simplifyTree = function(curNestedFilterList, curPosition) {
                /*
                Use to un-nest a group inside an empty group
                */
                    if(curNestedFilterList !==$scope.rootNestedFilters && curNestedFilterList[curPosition].conditions.length===0 && curNestedFilterList[curPosition].nestedFilters !== []){
                        for(let i=0;i<curNestedFilterList[curPosition].nestedFilters.length; i++){
                            let newConds = angular.copy(curNestedFilterList[curPosition].conditions).concat(angular.copy(curNestedFilterList[curPosition].nestedFilters[i].conditions))
                            let newNestedFilters = angular.copy(curNestedFilterList[curPosition].nestedFilters[i].nestedFilters)
                            curNestedFilterList.push({
                            "recursionDepth":curNestedFilterList[curPosition].nestedFilters[i].recursionDepth,
                            "mode": curNestedFilterList[curPosition].nestedFilters[i].mode,
                            "position": curNestedFilterList[curPosition].nestedFilters[i].position,
                            "conditions": newConds,
                            "nestedFilters": newNestedFilters});
                        }
                        curNestedFilterList.splice(curPosition,1);
                    }

                }

                $scope.updateRootNestedFilters = function(curNestedFilterList,curPosition,curRecursionDepth) {
                /*
                 Recursively explore rootNestedFilters to update the indices (position, recursionDepth) of each element.
                 Simplify the rootNestedFilters tree if necessary.
                 */
                    curNestedFilterList[curPosition].position = curPosition;
                    curNestedFilterList[curPosition].recursionDepth = curRecursionDepth;
                    $scope.simplifyTree(curNestedFilterList, curPosition)
                    for (let i=0; i<curNestedFilterList[curPosition].nestedFilters.length; i++){
                        $scope.updateRootNestedFilters(curNestedFilterList[curPosition].nestedFilters, i, curRecursionDepth+1);
                    }
                }

                $scope.updateFilterDesc = function(inputFilterDesc, inputFilter) {
                /*
                 Recursively retrieve information in the rootNestedFilters, and add it with
                 correct format to the backend filterDesc object
                 */
                    let normalConditions = inputFilter.conditions;
                    let subConditions = [];
                    for (const element of inputFilter.nestedFilters){
                        subConditions.push(
                           {
                                "subCondition":{
                                    "distinct":inputFilterDesc.distinct,
                                    "enabled":inputFilterDesc.enabled,
                                    "uiData":
                                        {
                                        "conditions":element.conditions,
                                        "mode":element.mode
                                        }
                                }
                           }
                       );
                    }
                    inputFilterDesc.uiData.conditions = normalConditions.concat(subConditions);
                    inputFilterDesc.uiData.mode = inputFilter.mode;

                    let filterDescSubConditions = inputFilterDesc.uiData.conditions.filter(cond => "subCondition" in cond);
                    for(let i = 0; i<inputFilter.nestedFilters.length; i++) {
                        $scope.updateFilterDesc(filterDescSubConditions[i]["subCondition"], inputFilter.nestedFilters[i]);
                    }
                }

                $scope.getColumns = function() {
                    return FiltersService.getColumns($scope);
                };

                // for the visual if processor, make sure there is at least one condition at init
                if ($scope.isTopNestedFilter && $scope.nestedFilters[0].conditions.length==0 && $scope.nestedFilters[0].nestedFilters.length===0) {
                    const newCondition = {num: 0, items: []};
                    FilterConditionsEditorService.update($scope,newCondition);
                    $scope.rootNestedFilters[0].conditions.push(newCondition);
                    $scope.updateFilterDesc($scope.filterDesc, $scope.rootNestedFilters[0]);
                }

                /*
                Watchers
                */

                $scope.$watch("nestedFilters[0].mode", function(nv, ov) {
                    if(nv) {
                        $scope.mode = nv;
                    }
                });

                $scope.$parent.$watch($attrs.schema, function(nv, ov) {
                    if(nv!==ov) {
                        $scope.schema = nv;
                    }
                });

                $scope.$watch("rootNestedFilters", function(nv, ov) {
                    if ($scope.isTopNestedFilter && nv!==ov) {
                        $scope.updateRootNestedFilters($scope.rootNestedFilters,0,0);
                        $scope.updateFilterDesc($scope.filterDesc, $scope.rootNestedFilters[0]);
                    }
                }, true);


                $scope.$watch(() => this.filterDesc, function (nv, ov) {
                    if (nv && nv !== ov) {
                        $scope.filterDesc = nv;
                        $scope.initNestedFilters(ctrl);
                    }
                });
            }
        },
    };

    angular.module("dataiku.nestedFilters").component("nestedFilter", nestedFilter);
})();
