import os
import shutil


def download_sentence_transformers_model(cache_dir, model_repo, revision):
    """ Download a sentence transformers model to a local directory
    """
    import sentence_transformers

    model_path = os.path.join(cache_dir, model_repo.replace("/", "_"))
    if os.path.exists(model_path):
        return
    kwargs = dict()
    files_to_ignore = ["flax_model.msgpack", "rust_model.ot", "tf_model.h5"]
    if sentence_transformers.__version__[:3] < "2.3":
        kwargs["ignore_files"] = files_to_ignore
    model_path_tmp = sentence_transformers.util.snapshot_download(
        repo_id=model_repo,
        revision=revision,
        cache_dir=cache_dir,
        library_name="sentence-transformers",
        library_version=sentence_transformers.__version__,
        **kwargs,
    )
    if sentence_transformers.__version__[:3] >= "2.3":
        _delete_ignored_files(model_path_tmp, files_to_ignore)
        _replace_symlinks_with_files(model_path_tmp)
        os.rename(model_path_tmp, model_path)
        tmp_model_root_directory = os.path.abspath(os.path.join(model_path_tmp, '..', '..'))
        shutil.rmtree(tmp_model_root_directory)
    else:
        os.rename(model_path_tmp, model_path)


def _delete_ignored_files(root_dir, files_to_ignore):
    for dirpath, dirnames, filenames in os.walk(root_dir, topdown=False):
        for filename in filenames:
            if filename in files_to_ignore:
                filepath = os.path.join(dirpath, filename)
                os.remove(filepath)


def _replace_symlinks_with_files(root_dir):
    for dirpath, dirnames, filenames in os.walk(root_dir, topdown=False):
        for filename in filenames:
            filepath = os.path.join(dirpath, filename)
            if not os.path.islink(filepath):
                continue

            target = os.readlink(filepath)
            target_path = os.path.abspath(os.path.join(dirpath, target))
            if not os.path.exists(target_path):
                continue

            os.remove(filepath)
            shutil.move(target_path, filepath)
