class GeneratedFeaturesMapping:
    SINGLE_ORIGIN_COLUMN = "single_origin_column"
    SINGLE_PROCESSED_FEATURE = "single_processed_feature"

    def __setstate__(self, state):
        """
        We have this for backwards compatibility reasons. Ensemble models are pickled with their pipelines,
        and before version 12.4 models didn't have the "feature_to_block" property.
        Therefore, once they are unserialized,  they are missing the "feature_to_block" property.
        We create it during the unpickling so that the generated_features_mapping
        is populated during `pipeline.process`.
        """
        self.__dict__.update(state)
        if not hasattr(self, "feature_to_block"):
            self.feature_to_block = {}

    def __init__(self):
        self.mapping = {}
        self.feature_to_block = {}

    def add_whole_block_mapping(self, block_name, original_names):
        self.mapping[block_name] = {
            "type": self.SINGLE_ORIGIN_COLUMN,
            "original_names": original_names,
        }

    def add_per_column_mapping(self, block_name, original_names, new_name):
        if block_name not in self.mapping:
            block_type = self.SINGLE_PROCESSED_FEATURE
            self.mapping[block_name] = {"type": block_type, "values": {}, }
        self.feature_to_block[new_name] = block_name
        self.mapping[block_name]["values"][new_name] = original_names

    def add_features_to_block(self, features, block_name):
        """
        Map all features to their original block name
        :param features: list of features
        :param block_name: name of the block containing features
        """
        if features is None:
            return
        for feature in features:
            self.feature_to_block[feature] = block_name

    def get_block_from_feature(self, feature_name):
        """
        This method has 2 distinct logics.
        For sparse features whose block is easy to track back we infer the block.
        For other features we use a simple dictionary
        :param feature_name: str
        :return: name of the block
        """
        feature_generation = feature_name.split(":")[0]
        # The safe preprocessing are because the last part is an idx and therefore not a user inputted value that can contain ":"
        # We do this for blocks that can have a large number of features
        if feature_generation in ["hashing", "unfold", "hashvect"]:
            return ":".join(feature_name.split(":")[:-1])
        return self.feature_to_block[feature_name]

    def should_map_whole_block_to_one_origin_column(self, block_name):
        if block_name not in self.mapping:
            return False
        else:
            return self.mapping[block_name]["type"] == self.SINGLE_ORIGIN_COLUMN

    def get_whole_block_original(self, block_name):
        return self.mapping[block_name]["original_names"]

    def get_origin_columns_from_mapping(self, block_name, feature_name):
        block_mapping = self.mapping[block_name]
        if block_mapping.get("type") == self.SINGLE_ORIGIN_COLUMN:
            return block_mapping["original_names"]
        else:
            return block_mapping["values"][feature_name]

    def get_origin_columns_from_feature(self, feature_name, columns):
        # Feature wasn't generated
        if feature_name in columns:
            return [feature_name]

        block_name = self.get_block_from_feature(feature_name)
        if block_name in self.mapping:
            return self.get_origin_columns_from_mapping(block_name, feature_name)
        raise ValueError("Unsupported feature mapping {}".format(feature_name))
