"""Scikit-Learn compatibility module

Description: The purpose of this module is to encapsulate the code that handles
    the differences across sklearn version.

Differences tend to be in 4 categories :
1. functions have changed names/have changed their argument names.
This is handled by importing a different function depending on scikit version (e.g get_feature_names)

2. model parameters have changed names.
This is handled by remapping the models HP spaces and final params (e.g gbt_skcompat_hp_space/gbt_skcompat_actual_params )

3. classes have changed modules and/or names between scikit versions, but we want to unpickle models using an old version and find the new classes.
This is handled in dku_pickle.py with class names/modules remapping

4. model parameters have changed names but we want to unpickle models using an old version and find the new parameters
This is handled by mapping the scikit class to a custom DSS class, and having a custom unpickling there (e.g UnpickableKMeans)

Note that in a lot of cases we handle point 2. but not point 4. Meaning the code is compatible with all scikit versions but
you can't change your code env from 0.24 to 1.2 and unpickle old models. Mostly, we make sure you can go from 0.20 to 0.24/1.0
but that's it.

Motivation: DSS must necessarily support sklearn 0.20 because it is the last
    version of sklearn that supports Python 2. But DSS should also support
    newer versions of sklearn as much as possible.
"""
import sklearn

from dataiku.base.utils import package_is_at_least
from ._dku_fit import dku_fit
from ._gbt import get_gbt_binary_classification_baseline, get_gbt_multiclass_classification_baseline, get_gbt_regression_baseline, update_gradient_boosting_model_state

# Beware, classes needed for custom unpickling are referenced by name in dku_pickle,
# standard find usages in an IDE won't find that. 
# /!\ Think twice before deleting seemingly unused imports here /!\
if package_is_at_least(sklearn, '0.23') and not package_is_at_least(sklearn, '1.4'):
    from ._gbt import Sk020MultinomialDeviance
if package_is_at_least(sklearn, '0.23'):
    from ._gbt import LogOddsEstimatorDummyClassifier, ScaledLogOddsEstimatorDummyClassifier, MeanEstimatorDummyRegressor, \
        PriorProbabilityEstimatorDummyClassifier, QuantileEstimatorDummyRegressor
from ._gbt import UnpicklableGradientBoostingClassifier, UnpicklableGradientBoostingRegressor, RemovedGradientBoostingLoss, update_gradient_boosting_model_state
from ._calibrated_classifier import get_calibrators, get_base_estimator, UnpicklableCalibratedClassifier
from ._preprocessor import get_feature_names, get_n_stop_words_and_remove
from ._hp_space import gbt_skcompat_hp_space, gbt_skcompat_actual_params, sgd_skcompat_hp_space, sgd_skcompat_actual_params, SQUARED_LOSS_NAME
from ._isolation_forest import handle_behaviour_param_of_isolation_forest
from ._isotonic_calibration import extract_X_y_from_isotonic_regressor, UnpicklableIsotonicRegression
from ._kmeans import get_kmeans_estimator, get_kmeans_n_init_value, get_minibatch_n_init_value
from ._kmeans import UnpickableKMeans
from ._kneighbors import UnpickableKNeighborsClassifier
from ._recursive_partial_dependence import dku_recursive_partial_dependence
from ._confusion_matrix import confusion_matrix
from ._stratified_group_kfold import instantiate_stratified_group_kfold
from ._roc_curve import roc_curve
from ._sgd import UnpicklableSGDClassifier, update_sgd_model_state
from ._svm import UnpicklableSVR, UnpicklableSVC
from ._trees import UnpicklableTree, UnpicklableDecisionTreeRegressor, UnpicklableDecisionTreeClassifier, UnpicklableRandomForestClassifier, UnpicklableExtraTreeClassifier, UnpicklableExtraTreeRegressor
if package_is_at_least(sklearn, "0.22"):
    from sklearn.linear_model._base import _preprocess_data as dku_preprocess_data
else:
    from sklearn.linear_model.base import _preprocess_data as dku_preprocess_data
