import glob
import shutil
import tarfile
import tempfile
import logging
import uuid
from abc import ABCMeta, abstractmethod
import os.path as osp
import os
from six import add_metaclass

from dataiku.base.utils import tar_extractall
from dataiku.core import intercom

logger = logging.getLogger(__name__)

TAR_BUFFER_SIZE = 10 * 16 * 1024


@add_metaclass(ABCMeta)
class AbstractFileExchanger(object):

    def __init__(self, execution_id, file_kind):
        self._execution_id = execution_id
        self._file_kind = file_kind

    @abstractmethod
    def _fetch_file_api_call(self, path):
        pass

    @abstractmethod
    def _put_file_api_call(self, params, _file):
        pass

    @abstractmethod
    def _delete_file_api_call(self, path):
        pass

    def send_file(self, file, path=None):
        if path is None:
            path = file
        logging.info("Sending %s to %s" % (file, path))
        with open(file, 'rb') as f:
            params = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path, 'expand': 'false'}
            self._put_file_api_call(params, f)

    def delete_path(self, path):
        logging.info("Deleting path %s" % path)
        self._delete_file_api_call(path)

    def send_files(self, files, path=None, archive_root=None):
        if path is None:
            path = "_dku_send.tgz"
        logger.info("Sending %d globs to %s" % (len(files), path))
        all_files= []
        for g in files:
            all_files.extend(glob.glob(g))
        logging.info("Sending files to %s: %s" % (path, all_files))
        tmp_archive = '/tmp/_dku_send.%s.tgz' % str(uuid.uuid4())
        with tarfile.open(tmp_archive, 'w:gz') as tar:
            if archive_root is not None:
                # compute path from archive_root to use as name in the tar
                prefix_length = len(osp.abspath(archive_root))
                for f in all_files:
                    arc_f = osp.join('.', osp.abspath(f)[prefix_length:])
                    tar.add(f, arcname=arc_f)
            else:
                for f in all_files:
                    tar.add(f)
        with open(tmp_archive, 'rb') as f:
            params = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path, 'expand': 'true'}
            self._put_file_api_call(params, f)

        os.remove(tmp_archive)

    def fetch_dir(self, path, dest=None):
        if dest is None:
            dest = path
        logger.info("Fetching dir %s into %s" % (path, dest))
        resp = self._fetch_file_api_call(path)

        if resp.status_code == 200:
            dest = os.path.abspath(dest)
            parent_dir = os.path.dirname(dest)
            if not os.path.isdir(parent_dir):
                os.makedirs(parent_dir)
            tmp_dir = tempfile.mkdtemp(dir=parent_dir)
    
            with tarfile.open(fileobj=resp.raw, mode='r|', bufsize=TAR_BUFFER_SIZE) as tar:
                tar_extractall(tar, tmp_dir)

            if os.path.isdir(dest):
                shutil.rmtree(dest, ignore_errors=True)
            elif os.path.exists(dest):
                os.remove(dest)

            os.renames(tmp_dir, dest)
        elif resp.status_code != 404:
            raise Exception("Error fetching dir %s, HTTP status %d: %s" % (path, resp.status_code, resp.text))
        else:
            logger.warning("Could not fetch dir %s, not found" % path)

    def fetch_file(self, path, dest=None):
        if dest is None:
            dest = path
        logger.info("Fetching file %s into %s" % (path, dest))
        resp = self._fetch_file_api_call(path)

        if resp.status_code == 200:
            tmp_fetch_file = '_dku_fetch_file.%s' % str(uuid.uuid4())
            with open(tmp_fetch_file, 'wb') as f:
                for chunk in resp.iter_content(chunk_size=4096):
                    f.write(chunk)
            shutil.move(tmp_fetch_file, dest)
        elif resp.status_code != 404:
            raise Exception("Error fetching file %s, HTTP status %d: %s" % (path, resp.status_code, resp.text))
        else:
            logger.warning("Could not fetch file %s, not found" % path)


class FileExchanger(AbstractFileExchanger):

    def _put_file_api_call(self, params, _file):
        intercom.jek_or_backend_void_call('containers/put-file', params=params, data=_file)

    def _delete_file_api_call(self, path):
        params = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path}
        intercom.jek_or_backend_void_call('containers/delete-file', params=params)

    def _fetch_file_api_call(self, path):
        data = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path, 'compress': False}
        return intercom.jek_or_backend_post_call('containers/get-file', stream=True, data=data)


@add_metaclass(ABCMeta)
class AbstractReadonlyFileExchanger(AbstractFileExchanger):

    def _put_file_api_call(self, params, _file):
        raise ValueError("Cannot send file with readonly file exchanger")

    def _delete_file_api_call(self, path):
        raise ValueError("Cannot send file with readonly file exchanger")


class LabReadonlyFileExchanger(AbstractReadonlyFileExchanger):

    def __init__(self, execution_id, project_key, analysis_id, mltask_id):
        super(LabReadonlyFileExchanger, self).__init__(execution_id, "ML_FILES")
        self._project_key = project_key
        self._analysis_id = analysis_id
        self._mltask_id = mltask_id

    def _fetch_file_api_call(self, path):
        data = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path,
                "projectKey": self._project_key, "analysisId": self._analysis_id, "mlTaskId": self._mltask_id}
        return intercom.jek_or_backend_post_call('containers/get-lab-file', stream=True, data=data)


class SavedModelFileExchanger(AbstractFileExchanger):

    def __init__(self, execution_id, project_key, saved_model_id):
        super(SavedModelFileExchanger, self).__init__(execution_id, "ML_FILES")
        self._project_key = project_key
        self._saved_model_id = saved_model_id

    def _put_file_api_call(self, params, _file):
        params = params.copy()
        params.update({"projectKey": self._project_key, "savedModelId": self._saved_model_id})
        intercom.jek_or_backend_void_call('containers/put-sm-file', params=params, data=_file)

    def _delete_file_api_call(self, path):
        params = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path,
                  "projectKey": self._project_key, "savedModelId": self._saved_model_id}
        intercom.jek_or_backend_void_call('containers/delete-sm-file', params=params)

    def _fetch_file_api_call(self, path):
        data = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path,
                "projectKey": self._project_key, "savedModelId": self._saved_model_id}
        return intercom.jek_or_backend_post_call('containers/get-sm-file', stream=True, data=data)


class LocalHuggingfaceFileExchanger(AbstractReadonlyFileExchanger):
    """
    Identical to SavedModelFileExchanger, but filters out the "output" folder.
    """
    def __init__(self, execution_id, project_key, saved_model_id):
        super(LocalHuggingfaceFileExchanger, self).__init__(execution_id, "LOCAL_HF_FILE")
        self._project_key = project_key
        self._saved_model_id = saved_model_id

    def _fetch_file_api_call(self, path):
        data = {'executionId': self._execution_id, 'fileKind': self._file_kind, 'path': path,
                "projectKey": self._project_key, "savedModelId": self._saved_model_id}
        return intercom.jek_or_backend_post_call('containers/get-local-hf-file', stream=True, data=data)
