import sys
from dataiku.core import base
from dateutil import parser as date_iso_parser
import numpy as np

def parse_iso_date(s):
    if s == "":
        return None
    else:
        return date_iso_parser.parse(s)

DKU_PANDAS_TYPES_MAP = {
    'int': np.int32,
    'bigint': np.int64,
    'float': np.float32,
    'double': np.float64,
    'boolean': bool
}

# The previous map is incomplete ... not touching it in order to avoid changing behaviors
DKU_RICHER_PANDAS_TYPES_MAP = {
    "tinyint": np.int8,
    "smallint": np.int16,
    "int": np.int32,
    "bigint": np.int64,
    "float": np.float32,
    "double": np.float64,
    "boolean": bool
}

def str_to_bool(s):
    if s is None:
        return False
    return s.lower() == "true"

# used to stream typed fields in iter_tuples.
CASTERS = {
    "tinyint" : int,
    "smallint" : int,
    "int": int,
    "bigint": int,
    "float": float,
    "double": float,
    "date": parse_iso_date,
    "dateonly": parse_iso_date,
    "datetimenotz": parse_iso_date,
    "boolean": str_to_bool
}


PANDAS_DKU_TYPES_MAP = {
    'int64': 'bigint',
    'float64': 'double',
    'float32': 'float',
    'int32': 'int',
    'object': 'string',
    'int': 'int',
    'float': 'float',
    'bool': 'boolean',
    'datetime64[ns]': 'datetimenotz', # used to be date, but now should be caught by is_datetime64tz_dtype() below
    'Int64': 'bigint',
    'boolean': 'boolean'
}

def pandas_dku_type(dtype):
    '''Return the DSS type for a Pandas dtype'''
    from pandas.api.types import is_datetime64_any_dtype, is_datetime64tz_dtype
    if is_datetime64tz_dtype(dtype):
        return 'date'
    elif is_datetime64_any_dtype(dtype):
        return 'datetimenotz'
    else:
        return PANDAS_DKU_TYPES_MAP.get(str(dtype), "string")

def get_schema_from_df(df):
    ''' A simple function that returns a DSS schema from
    a Pandas dataframe, to be used when writing to a dataset
    from a data frame'''
    schema = []

    if len(set(df.columns)) != len(list(df.columns)):
        raise Exception("DSS doesn't support dataframes containing multiple columns with the same name.")

    for col_name in df.columns:
        # We convert the name of the column to a string to be able to insert it in the
        # schema object.
        # However, since Pandas 0.17, we keep the "pristine" name to lookup because you
        # can't anymore lookup 123 by "123"
        if not isinstance(col_name, base.dku_basestring_type):
            col_name_str = str(col_name)
        else:
            col_name_str = col_name

        dss_type = pandas_dku_type(df[col_name].dtype)
        if dss_type == 'datetimenotz':
            # check if the values have non-00:00:00 time components, otherwise it's very likely to be dateonly,
            # and anyway, it'll be representable with dateonly
            s = df[col_name]
            t = (s - s.dt.normalize()).dt.total_seconds()
            if not t.isna().all() and (t.fillna(0.0) == 0.0).all():
                dss_type = 'dateonly'
        column_type = {
            'name': col_name_str,
            'type': dss_type
        }
        schema.append(column_type)
    return schema
