(function() {
    'use strict';

    const app = angular.module('dataiku.fm.forms', []);

    /*
      generate on hover tooltip with all invalid form fields.
      it's typically used on form submit button.
      by default, the first parent form is considered but you can give a form name as attribute value.
      tooltip position can be given with the `toggle-form-errors-position` attribute, with as default, `tooltip-bottom`.

      eg:
      ```html
        #from inside a form
        <button on-click="save()" 
                toggle-form-errors>
            save
        </button>

        #from outside a form and left'ed
        <button on-click="save()" 
                toggle-form-errors="myFormName"
                toggle-form-errors-position="tooltip-left">
            save
        </button>
      ```
    */  
    app.directive('toggleFormErrors', function(getBootstrapTooltipPlacement) {
        
        function getFormFieldLabel(error) {
            const attr = error.$$attr || {};
            const fieldName = error.$name || attr.id;
            if (!fieldName) return '';
            const labelFromInput = $("label[for='" + fieldName + "']").first().text();
            if (labelFromInput) {
                return labelFromInput.trim().replace(/\s*\*\s*$/,'');
            }
            //for "subform" such as tag list
            const labelFromForm = error.$$element.attr("label");
            return labelFromForm || fieldName;
        }

        const MAX_DISPLAYED_LINES = 6;
        const MAX_DISPLAYED_ERRORS = MAX_DISPLAYED_LINES - 1;

        function getFormErrors(form) {
            if (form && form.$error) {
                const errors = []
                    .concat(...Object.values(form.$error)) //flatten + concat 
                    .map(getFormFieldLabel)                //transform to readable label
                    .reduce((errors, label) => {           //dedup with a counter
                        errors[label] = (errors[label] || 0) + 1;
                        return errors;
                    }, {});
                const errorLabels = Object.keys(errors);
                if (errorLabels.length > 0) {
                    const truncate = errorLabels.length > MAX_DISPLAYED_LINES;
                    let errorsList = errorLabels
                        .slice(0, truncate ? MAX_DISPLAYED_ERRORS : MAX_DISPLAYED_LINES)
                        .map(label => '&bullet; ' + label + (errors[label] > 1 ? ` (x${errors[label]})` : ''))
                        .join('<br/>');
                    if (truncate) {
                        errorsList += `<br/> ... and ${errorLabels.length - MAX_DISPLAYED_ERRORS} more`;
                    }
                    return `<div style="text-align: left;">
                                Missing&nbsp;or&nbsp;invalid&nbsp;field${errorLabels.length>1 ? "s" : ""}:<br>
                                ${errorsList}
                            </div>`;
                }
            }
            return '';
        };

        return {
            restrict: 'A',
            require: '?^form',
            scope: true,
            link: function (scope, element, attrs, form) {

                if (attrs.toggleFormErrors) {
                    //by default `form` is the controller of the first parent FORM element
                    //but for usage of this directive from outside the form,
                    //you can specify the form name as attribute value
                    form = angular.element(document[attrs.toggleFormErrors]).controller('form');
                }

                if (form) {
                    //setup tooltip
                    const params = {
                        placement: attrs.placement ? attrs.placement 
                                                   : getBootstrapTooltipPlacement(attrs.toggleFormErrorsPosition || 'tooltip-bottom'),
                        animation: attrs.animation ? scope.$eval(attrs.animation) 
                                                   : false,
                        html: true
                    };
                    if (attrs.container) params.container = attrs.container;
                    if (attrs.trigger) params.trigger = attrs.trigger;
                    element.on('hide.bs.modal', e => e.stopPropagation());
                    element.tooltip(params);
                    element.hover(
                        () => $(this).tooltip('show'),
                        () => $('.tooltip').not(element.next()).remove() //remove other tooltips on hover out
                    );

                    //handle error changes
                    const stopWatchErrorFn = scope.$watch(
                        () => getFormErrors(form), 
                        (newFormErrors) => element.attr('data-original-title', newFormErrors)
                    );

                    element.on('$destroy', stopWatchErrorFn);
                }
            }
        }
    });

})();