(function() {
    'use strict';
    /** @typedef {import('../../simple_report/types').GeneratedSources.ChartFilter} ChartFilter */
    /** @typedef {import('../../simple_report/types').GeneratedSources.DashboardPage} DashboardPage */

    const app = angular.module('dataiku.dashboards');

    /**
     * Everything to store and manipulate dashboard filters.
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.factory('DashboardFilters', function(CHART_FILTERS, FILTERABLE_INSIGHT_TYPES, DashboardFiltersUrlParams, ChartFilterUtils, FiltersPanelPosition, ActivityIndicator, ChartTypeChangeHandler, ChartFilters, FilterFacetsService, Logger, DataikuAPI, ChartUsableColumns, FilterPanelContext, $stateParams, CachedAPICalls, DashboardPageUtils, TileUtils) {
        const FILTERS_ON_MSG = 'Data is filtered';
        const SAMPLING_PROPS = ['refreshableSelection', 'engineType'];
        const MAX_DATA_BYTES = -1; // default to no memory limit (for sampling)
        const filtersSourceTypes = Object.freeze({ DATASET: 'dataset', TILE: 'tile' });

        /** @type {Map<string, ChartFilter[]>} */
        const globalFiltersByPageId = new Map();
        /** @type {Map<string, boolean>} */
        const enableCrossFiltersByPageId = new Map();
        /** @type {Map<string, boolean>} */
        const isFilterPanelDeactivatedByPageId = new Map();
        let insightsMap; // get all insights on dashboard mapped by id

        const lowerCaseStatusByDatasetKey = new Map(); // store the lower case status of each dataset
        const lowerCaseStatusRequestsQueues = new Map(); // used to handle simultaneous requests to fetch the lower case status of the same new dataset
        let datasetTypes;


        function getSamplingProps(src) {
            if (!src) {
                return {};
            }
            const samplingProps = {};
            for (const prop of SAMPLING_PROPS) {
                if (!src[prop]) {
                    continue;
                }
                if (prop === 'refreshableSelection') {
                    const newSelection = angular.copy(src[prop]);
                    delete newSelection._refreshTrigger; // ignore front internal prop
                    samplingProps[prop] = { ...samplingProps[prop], ...newSelection };
                } else {
                    samplingProps[prop] = angular.copy(src[prop]);
                }
            }
            return samplingProps;
        }

        function getSamplingRefreshTrigger(src) {
            if (!src || src.refreshableSelection == null) {
                return null;
            }
            return src.refreshableSelection._refreshTrigger;
        }

        function resetAllDashboardPagesFilterPanelStates() {
            globalFiltersByPageId.clear();
            isFilterPanelDeactivatedByPageId.clear();
            enableCrossFiltersByPageId.clear();
        }

        function resetGlobalFilters(pageId) {
            globalFiltersByPageId.delete(pageId);
        }

        function setGlobalFilters(filters, pageId) {
            if (filters !== globalFiltersByPageId.get(pageId) && pageId) {
                globalFiltersByPageId.set(pageId, filters);
            }
        }

        function getGlobalFilters(pageId) {
            return globalFiltersByPageId.get(pageId);
        }

        function getActiveGlobalFilters(pageId) {
            if (getIsFilterPanelDeactivated(pageId)) {
                return [];
            }
            return globalFiltersByPageId.get(pageId);
        }

        function setInsightsMap(newInsightsMap) {
            insightsMap = newInsightsMap;
        }

        function setCrossFilterStatusToPageId(pageId, enableCrossFilters) {
            enableCrossFiltersByPageId.set(pageId, enableCrossFilters);
        }

        function getSlideFilterableTiles(dashboardPageId, dashboardPages) {
            const filterableTiles = [];
            const dashboardPageIdx = findDashboardPageIdx(dashboardPageId, dashboardPages);
            if (dashboardPageIdx > -1 && insightsMap) {
                TileUtils.getFlattenTileList(dashboardPages[dashboardPageIdx].grid.tiles).forEach(tile => {
                    if (FILTERABLE_INSIGHT_TYPES.includes(tile.insightType)) {
                        const mappedInsight = insightsMap[tile.insightId];
                        if (!mappedInsight) {
                            return;
                        }
                        filterableTiles.push({
                            id: mappedInsight.id,
                            name: tile.titleOptions.title ? tile.titleOptions.title : mappedInsight.name,
                            type: mappedInsight.type,
                            datasetSmartName: mappedInsight.params.datasetSmartName,
                            refreshableSelection: mappedInsight.params.refreshableSelection,
                            script: mappedInsight.params.shakerScript,
                            engineType: mappedInsight.params.engineType
                        });
                    }
                });
            }

            return filterableTiles;
        }

        function findDashboardPageIdx(dashboardPageId, dashboardPages) {
            // new page does not have an id until it's saved, its temporary id is its index
            if (dashboardPages[dashboardPageId] && !dashboardPages[dashboardPageId].id) {
                return dashboardPageId;
            } else {
                return dashboardPages.findIndex(page => page.id === dashboardPageId);
            }
        }

        /**
         * Checks if the filter is actually filtering (for example a filter set to exclude nothing does nothing).
         *
         * @param {ChartFilter} filter
         * @param {DashboardPage.DashboardPageFiltersParams} filtersParams
         * @returns
         */
        function isFilterEffective(filter, filtersParams) {
            if (ChartFilterUtils.isAlphanumericalFilter(filter)) {
                // If excludeOtherValues is true, even if all values are selected the filter can be effective (for example in a dashboard if the filter insight sample contains less values than in the dataset insight sample).
                return filter.excludeOtherValues || (Object.keys(filter.excludedValues || {}).length > 0);
            }
            if (ChartFilterUtils.isNumericalFilter(filter)) {
                /*
                 * In dashboards, the samples of the filters tile and of the filtered insight can be different.
                 * As a result, if the min and max values haven't been changed by the user, a sampled filter may still be effective.
                 */
                return areFiltersSampled(filtersParams) || filter.minValue != null ||filter.maxValue != null || !filter.includeEmptyValues;
            }
            if (ChartFilterUtils.isRelativeDateFilter(filter)) {
                return ChartFilterUtils.isRelativeDateFilterEffective(filter.dateFilterPart, filter.dateFilterOption, filter.includeEmptyValues);
            }
            if (ChartFilterUtils.isExplicitFilter(filter)) {
                return true;
            }
            if (ChartFilterUtils.isCustomFilter(filter)) {
                return filter.selectedValues && Object.keys(filter.selectedValues).length || filter.excludedValues && Object.keys(filter.excludedValues).length;
            }
            return false;
        }

        function areFiltersSampled(filtersParams) {
            if (filtersParams.refreshableSelection == null) {
                return false;
            }
            if (filtersParams.refreshableSelection.selection != null && filtersParams.refreshableSelection.selection.filter != null && filtersParams.refreshableSelection.selection.filter.enabled) {
                return true;
            }
            return filtersParams.refreshableSelection.selection.samplingMethod !== 'FULL';
        }

        function hasGlobalFiltersApplied(pivotResponse, appliedFilters) {
            const hasDashboardFiltersApplied = appliedFilters.some(filter => filter.isAGlobalFilter);
            if (!hasDashboardFiltersApplied) {
                return false;
            }
            /*
             * We do not check what each filter has filtered out as it's the result of a global query
             * so we consider dashboard filters are applied when there are dashboard filters and filtered out rows.
             * => It means there can be false positive for chart/dataset insight with internal filters with inefective dashboard filters applied.
             */
            if (pivotResponse.totalKeptRows != null && pivotResponse.totalRows != null) {
                return pivotResponse.totalRows > pivotResponse.totalKeptRows;
            } else if (pivotResponse.beforeFilterRecords != null && pivotResponse.afterFilterRecords != null) {
                return pivotResponse.beforeFilterRecords > pivotResponse.afterFilterRecords;
            }
            return hasDashboardFiltersApplied;
        }

        function isSameSampling(tileSampling, filterSampling) {
            // Exclude check of backend internal properties
            const excludedProperties = ['useMemTable', 'maxRecordsForDisplay', 'maxStoredBytes', 'timeout'];

            for (const property in { ...tileSampling, ...filterSampling }) {
                if (excludedProperties.includes(property)) {
                    continue;
                }
                if (!angular.equals(filterSampling[property], tileSampling[property])) {
                    return false;
                }
            }
            return true;
        }

        function canCrossFilter(pageId) {
            return !!enableCrossFiltersByPageId.get(pageId);
        }

        function getInconsistentFilteringCause(tileData, filtersParams) {
            if (filtersParams == null){
                return '';
            }
            if (tileData.sourceDataset !== filtersParams.datasetSmartName) {
                return 'datasets are different';
            }
            // no sense on checking for samples if datasets are different
            if (!isSameSampling(tileData.sampleSettings.selection, filtersParams.refreshableSelection.selection)) {
                return 'samples are different';
            }
            return '';
        }

        function getUnusableFiltersMessage(unusableFilters, warningRootCause) {
            if (!unusableFilters || !unusableFilters.length) {
                return '';
            }
            const MAX_COLUMNS = 3; // display a maximum of 3 columns names in message
            const hasWarningRootCause = warningRootCause && warningRootCause.length > 0;
            const otherColumnsLength = unusableFilters.length - MAX_COLUMNS;
            const columnsNames = unusableFilters.slice(0, MAX_COLUMNS).map(uf => uf.columnName).join(', ');
            const isPluralColumns = unusableFilters.length > 1 ? 's' : '';
            const isPluralOthers = otherColumnsLength > 1 ? 's' : '';
            const hasDifferentReasons = unusableFilters.slice(1).some(uf => uf.reason !== unusableFilters[0].reason);
            return `${FILTERS_ON_MSG} - except for column${isPluralColumns} ${columnsNames}\
                    ${otherColumnsLength > 0 ? ` + ${otherColumnsLength} other${isPluralOthers}` : ''} (\
                        ${hasDifferentReasons ? 'various reasons' : unusableFilters[0].reason}\
                        ${hasWarningRootCause ? `as ${warningRootCause}` : ''}\
                    )`;
        }

        function setFilteringMessages(tileData, filtersParams) {
            if (!tileData.filteringStatus.isInsightFiltered) {
                tileData.filteringStatus.warningMessage = '';
                tileData.filteringStatus.infoMessage = '';
                return;
            }
            const inconsistentFilteringCause = getInconsistentFilteringCause(tileData, filtersParams);
            if (!inconsistentFilteringCause.length && !tileData.unusableColumnsFilters.length) {
                tileData.filteringStatus.warningMessage = ''; // no warning message if no inconsistent filtering detected cause detected
                tileData.filteringStatus.infoMessage = FILTERS_ON_MSG;
            } else {
                tileData.filteringStatus.warningMessage = getUnusableFiltersMessage(tileData.unusableColumnsFilters, inconsistentFilteringCause);
                tileData.filteringStatus.infoMessage = tileData.filteringStatus.warningMessage.length > 0 ? ''
                    : `${FILTERS_ON_MSG} - risk of inconsistency (${inconsistentFilteringCause})`;
            }
        }

        function hasUsableColumn(usableColumns, filter, hasCaseInsensitiveColumnNames) {
            if (!usableColumns || !filter) {
                return false;
            }
            const usableColumnMap = new Map(usableColumns.map(usableColumn => [hasCaseInsensitiveColumnNames? usableColumn.column.toLowerCase() : usableColumn.column, usableColumn.type]));
            if (ChartFilterUtils.isExplicitFilter(filter)) {
                return filter.explicitConditions.every(condition => {
                    const filterColumnName = hasCaseInsensitiveColumnNames? condition.column.toLowerCase() : condition.column;
                    return usableColumnMap.has(filterColumnName) && ChartFilters.isExplicitConditionCompatibleWithColumnType(usableColumnMap.get(filterColumnName), condition);
                });
            } else {
                const filterColumnName = hasCaseInsensitiveColumnNames? filter.column.toLowerCase() : filter.column;
                return usableColumnMap.has(filterColumnName) && usableColumnMap.get(filterColumnName) === filter.columnType;
            }
        }

        function adaptPristineRangeFilter(filter) {
            if (!filter || !ChartFilterUtils.isNumericalFilter(filter)) {
                return filter;
            }
            // minValue and maxValue can have a value of null if they haven't been changed by the user, but because of sampling they may still affect the dataset so we fallback to the filters tile global min and max values.
            const minValue = filter.minValue != null ? filter.minValue : filter.$globalMinValue;
            const maxValue = filter.maxValue != null ? filter.maxValue : filter.$globalMaxValue;
            return {
                ...filter,
                minValue,
                maxValue
            };
        }

        function getUsableFilters(filters, usableColumns, hasCaseInsensitiveColumnNames, filtersParams) {
            if (usableColumns && usableColumns.length && filters && filters.length) {
                return filters.filter(filter => isFilterEffective(filter, filtersParams) && hasUsableColumn(usableColumns, filter, hasCaseInsensitiveColumnNames)).map(filter => adaptPristineRangeFilter(filter));
            }
            return [];
        }

        function getReason(filter, usableColumns) {
            if (filter && usableColumns && usableColumns.length) {
                const columnFound = usableColumns.find(e => e.column === filter.column);
                if (columnFound && columnFound.type !== filter.type) {
                    return 'type mismatch';
                }
            }
            return 'not found';
        }

        function getUnusableColumnFilters(filters, usableColumns) {
            if (usableColumns && usableColumns.length && filters && filters.length) {
                return filters
                    .filter(filter => !hasUsableColumn(usableColumns, filter))
                    .map(filter => ({ columnName: filter.column, reason: getReason(filter, usableColumns) }));
            }
            return [];
        }

        function findFiltersTileInPage(page) {
            if (page && page.grid && page.grid.tiles) {
                return findFiltersTileList(page.grid.tiles);
            }
        }

        function findFiltersTileList(tiles) {
            return tiles.find(tile => TileUtils.isFilterTile(tile));
        }

        /**
         * @param {DashboardPage} page
         * @returns {ChartFilter[]}
         */
        function getPageFilters(page) {
            return (page.filters == null) ? [] : page.filters.filter(filter => !filter.useMinimalUi);
        }

        function onApplyFiltersSuccess(tileData, filtersParams, response) {
            tileData.filteringStatus.isApplyingFilters = false;
            tileData.filteringStatus.isInsightFiltered = hasGlobalFiltersApplied(response, tileData.appliedFilters);
            setFilteringMessages(tileData, filtersParams);
        }

        function onApplyFiltersError(error, filteringStatus, loadingData) {
            filteringStatus.isApplyingFilters = false;

            if (!error || !error.data) {
                return;
            }
            // If the error contains a filtered out message, the insight is considered as filtered (empty state)
            filteringStatus.isInsightFiltered = error.data.message.includes('filtered out');


            if (!filteringStatus.isInsightFiltered && error.data.message.includes('To finish your chart')) {
                loadingData.unconfiguredRoutine();
            }
        }

        function applyFilters(filters, filtersParams, tileData, loadingData, hasCaseInsensitiveColumnNames) {
            try {
                if (!tileData.usableColumns) {
                    return;
                }

                // Remove non usable columns
                let matchingFilters = getUsableFilters(filters, tileData.usableColumns, hasCaseInsensitiveColumnNames, filtersParams);
                if (hasCaseInsensitiveColumnNames && matchingFilters.length > 0) {
                    matchingFilters = getFiltersWithCorrectCaseColumnNames(matchingFilters, tileData.usableColumns);
                }

                tileData.appliedFilters = angular.copy(matchingFilters);
                tileData.unusableColumnsFilters = getUnusableColumnFilters(filters, tileData.usableColumns);
                tileData.filteringStatus.isApplyingFilters = !!matchingFilters && !!matchingFilters.length;
                return tileData.applyFilters(matchingFilters, loadingData)
                    .then(data => tileData.parseResponse ? onApplyFiltersSuccess(tileData, filtersParams, tileData.parseResponse(data)) : data)
                    .catch((error) => {
                        onApplyFiltersError(error, tileData.filteringStatus, loadingData);
                        // Let the insight handle the error
                        throw error;
                    });
            } catch (error) {
                return Promise.reject(error);
            }
        }

        function getFiltersWithCorrectCaseColumnNames(matchingFilters, usableColumns) {
            return matchingFilters.map(filter => {
                if (ChartFilterUtils.isExplicitFilter(filter)) {
                    return {
                        ...filter,
                        explicitConditions: filter.explicitConditions.map(condition => ({
                            ...condition,
                            column: usableColumns.find(column => column.column.toLowerCase() === condition.column.toLowerCase()).column
                        }))
                    };
                } else {
                    return {
                        ...filter,
                        column: usableColumns.find(column => column.column.toLowerCase() === filter.column.toLowerCase()).column
                    };
                }
            });
        }

        /**
         * Get explorations filters from charts filters
         * @param  {List<ChartFilter.java>} usableFilters - Filters to convert from
         * @return {List<FilterElement.java>}
         */
        function getDashboardExplorationFilters(usableFilters) {
            if (!usableFilters || !usableFilters.length) {
                return [];
            }
            const explorationFilters = [];
            usableFilters.forEach(filter => {
                const facet = {
                    type: 'facet',
                    column: filter.column,
                    columnType: filter.columnType,
                    currentMode: filter.filterType.split('_')[0], // remove "_FACET" part
                    sort: 'count',
                    minValue: filter.minValue,
                    maxValue: filter.maxValue,
                    isAGlobalFilter: true,
                    active: filter.active,
                    includeEmptyValues: filter.includeEmptyValues
                };

                if (filter.explicitConditions) {
                    facet.explicitConditions = filter.explicitConditions;
                }
                if (filter.explicitExclude) {
                    facet.explicitExclude = filter.explicitExclude;
                }

                if (filter.excludeOtherValues) {
                    facet.selectedValues = filter.selectedValues;
                } else {
                    facet.excludedValues = filter.excludedValues;
                }

                if (filter.dateFilterType === CHART_FILTERS.DATE_TYPES.RANGE) {
                    facet.dateFilterType = filter.dateFilterType;
                    facet.timezone = 'UTC';
                } else if (filter.dateFilterType === CHART_FILTERS.DATE_TYPES.RELATIVE) {
                    facet.dateFilterType = filter.dateFilterType;
                    facet.dateFilterPart = filter.dateFilterPart;
                    facet.dateFilterRelativeOption = filter.dateFilterOption;
                    facet.dateFilterRelativePreset = filter.dateFilterRelativePreset;
                } else if (filter.dateFilterType === CHART_FILTERS.DATE_TYPES.PART) {
                    facet.dateFilterType = filter.dateFilterType;
                    facet.dateFilterPart = filter.dateFilterPart;
                }
                const alphanumFilter = {
                    isAGlobalFilter: filter.isAGlobalFilter,
                    type: 'alphanum',
                    column: filter.column,
                    params: {
                        mode: (filter.customOptions && filter.customOptions.matchingMode) || 'FULL_STRING',
                        normalization: (filter.customOptions && filter.customOptions.normalizationMode) || 'EXACT',
                        ...(filter.customOptions && filter.customOptions.decimalPlaces ? { decimalPlaces: filter.customOptions.decimalPlaces } : {})
                    }
                };
                if (filter.explicitConditions) {
                    alphanumFilter.explicitConditions = filter.explicitConditions;
                }
                if (filter.explicitExclude) {
                    alphanumFilter.explicitExclude = filter.explicitExclude;
                }
                if (filter.selectedValues) {
                    alphanumFilter.selectedValues = Object.keys(filter.selectedValues);
                }
                if (filter.excludedValues) {
                    alphanumFilter.excludedValues = Object.keys(filter.excludedValues);
                }
                explorationFilters.push({
                    column: filter.column,
                    type: 'columnFilter',
                    currentMode: filter.filterSelectionType === 'CUSTOM' ? 'SIMPLE_ALPHANUM' : 'FACET',
                    active: filter.active,
                    facet,
                    alphanumFilter
                });
            });
            return explorationFilters;
        }

        /**
         * Computes the initial state of a dashboard slide filters, store them in the service and returns them.
         * @param {DashboardPage} page
         * @param {Record<string, string>} urlQueryParams
         * @param {any[]} insights
         * @returns {Promise<FrontendChartFilter[]>}
         */
        function initPageFilters(page, urlQueryParams, insights) {
            const filtersAreInSearchParams = DashboardFiltersUrlParams.areFiltersInSearchParams(urlQueryParams);
            const previousFilters = getGlobalFilters(page.id) || getPageFilters(page);

            if (filtersAreInSearchParams) {
                return getFilterPanelUsableColumns(page.filtersParams, urlQueryParams, insights)
                    .then(usableColumns => applyUrlFiltersOnPageFilters(urlQueryParams, previousFilters, usableColumns))
                    .then((filters) => {
                        if (filters) {
                            setGlobalFilters(filters, page.id);
                            return { activeFilters: filters, filters };
                        }
                        return [];
                    });
            }
            setGlobalFilters(previousFilters, page.id);
            return Promise.resolve({
                activeFilters: getActiveGlobalFilters(page.id) || previousFilters,
                filters: previousFilters
            });
        }

        /**
         * @param {any[]} insights
         * @returns {{ datasetSmartName: string, engineType: string }}
         */
        function findMostUsedDatasetWithEngine(insights) {
            const filterableInsights = (insights || []).filter(({ type }) => type === 'chart' || type === 'dataset_table');
            if (filterableInsights.length == 0) {
                return null;
            }
            const occurrencesByDatasetName = new Map();
            /** @type {{ datasetSmartName: string, engineType: string }} */
            let mostUsedDatasetWithEngine = { datasetSmartName: filterableInsights[0].params.datasetSmartName, engineType: filterableInsights[0].params.engineType || 'LINO' };
            let maxCount = 1;

            for (const insight of filterableInsights) {
                const currentDatasetOccurrences = occurrencesByDatasetName.get(insight.params.datasetSmartName);
                const newDatasetOccurrences = (currentDatasetOccurrences || 0) + 1;
                occurrencesByDatasetName.set(insight.params.datasetSmartName, newDatasetOccurrences);

                if (newDatasetOccurrences > maxCount) {
                    mostUsedDatasetWithEngine = { datasetSmartName: insight.params.datasetSmartName, engineType: insight.params.engineType || 'LINO' };
                    maxCount = newDatasetOccurrences;
                }
            }
            return mostUsedDatasetWithEngine;
        }

        /**
         * Retrieve the usable columns of the filter panel dataset
         * @param {Object} filtersParams
         * @param {Record<string, string>} urlQueryParams
         * @param {any[]} insights
         * @returns {Promise<UsableColumn[]>}
         */
        function getFilterPanelUsableColumns(filtersParams, urlQueryParams, insights) {
            if (DashboardFiltersUrlParams.shouldUseV1Strategy(urlQueryParams)) {
                // When filters from query parameters v1 are present, a dataset is required for column type resolution of the parsed filters. If the filter panel does not contain one, we utilize the most frequently used dataset from the dashboard.
                const datasetParams = filtersParams.datasetSmartName ?
                    { datasetSmartName: filtersParams.datasetSmartName, engineType: filtersParams.engineType }:
                    findMostUsedDatasetWithEngine(insights);
                if (datasetParams == null) {
                    // No dataset found neither in the filter panel nor in the dashbaord tiles
                    return Promise.resolve([]);
                }
                return getDatasetUsableColumns(datasetParams.datasetSmartName, datasetParams.engineType);
            } else {
                /*
                 * If there's no dataset in the filter panel for query parameters v2, there's no problem. We can still resolve the column types thanks to the v2 syntax.
                 * If there is a dataset in the filter panel, we check if the cross-filters from the URL belong to this dataset thanks to the usable columns. If they do, we consider them to compute the filter facets.
                 */
                if (filtersParams.datasetSmartName == null) {
                    // No dataset found in the filter panel
                    return Promise.resolve([]);
                } else {
                    return getDatasetUsableColumns(filtersParams.datasetSmartName, filtersParams.engineType);
                }
            }
        }

        /**
         * @param {string} datasetSmartName
         * @returns {Promise<UsableColumn[]>}
         */
        function getDatasetUsableColumns(datasetSmartName, engineType) {
            const dataSpec = getDataSpec($stateParams.projectKey, datasetSmartName, engineType);
            return fetchColumnsSummary(dataSpec.datasetProjectKey, dataSpec, { datasetSmartName })
                .then(({ usableColumns }) => usableColumns);
        }

        /**
         * @param {string} urlQueryParams
         * @param {FrontendChartFilter[]} filters
         * @param {UsableColumn[]} usableColumns
         * @returns {FrontendChartFilter[]}
         */
        function applyUrlFiltersOnPageFilters(urlQueryParams, filters, usableColumns) {
            const parsedFiltersData = DashboardFiltersUrlParams.getFiltersFromSearchParams(urlQueryParams, filters, usableColumns);
            const newFilters = parsedFiltersData.filters.length > 0 ? parsedFiltersData.filters : filters;
            if (parsedFiltersData.error) {
                ActivityIndicator.warning(`Some filters from URL are not applied (${parsedFiltersData.error})`, 5000);
            }
            return newFilters;
        }

        function setIsFilterPanelDeactivated(pageId, isDeactivated) {
            isFilterPanelDeactivatedByPageId.set(pageId, isDeactivated);
        }

        function getIsFilterPanelDeactivated(pageId) {
            return !!isFilterPanelDeactivatedByPageId.get(pageId);
        }

        function acceptColumnAsFilter(data) {
            const ret = ChartTypeChangeHandler.stdAggregatedAcceptDimension(data);
            return ret.accept && data.type !== 'GEOMETRY' && data.type !== 'GEOPOINT';
        }

        /**
         * Compute and return DataSpec (describes the data source for charts request)
         * @return {DataSpec.java}
         */
        function getDataSpec(projectKey, datasetSmartName, engineType, refreshableSelection, shaker) {
            const resolvedDataset = resolveDatasetFullName(datasetSmartName, projectKey);

            if (refreshableSelection == null) {
                refreshableSelection = {
                    _refreshTrigger: Date.now(),
                    selection: {
                        samplingMethod: 'FULL'
                    }
                };
            }

            return {
                datasetProjectKey: resolvedDataset.projectKey,
                datasetName:  resolvedDataset.datasetName,
                sampleSettings : refreshableSelection,
                engineType: engineType || 'LINO',
                ...(shaker == null ? {
                    copyScriptFromExplore: true,
                    copySelectionFromScript: false
                } : {
                    script: { ...shaker, origin: 'DATASET_EXPLORE' }
                })
            };
        };

        /**
         * Prepare request and call get-pivot-response
         * @param  {List<ChartFilters>} filters                         - filters to use for this request
         * @param  {Object}             samplingParams                  - sampling & engine configuration to use for this request
         * @param  {Object}             loadingData                     - callbacks to use for success & errors
         * @param  {Boolean}            samplingHasChanged              - flag to tell if the sampling has changed
         * @param  {MonoFuture}         filtersPivotRequestMonoFuture   - mono future to use for the request
         */
        function executeFiltersPivotRequest(projectKey, summary, filters, dataSpec, filtersPivotRequestMonoFuture) {
            try {
                const requestParams = {};
                /*
                 * if ($scope.insight.$chart) {
                 *     requestParams.maxDataBytes = $scope.insight.$chart.maxDataBytes;
                 * }
                 */
                const filtersWithMatchingColumn = filters.filter(filter => !filter.useMinimalUi || filter.hasMatchingColumn);
                if (!filtersWithMatchingColumn.length) {
                    return Promise.resolve([]);
                }
                const request = ChartFilters.buildFiltersRequest(filtersWithMatchingColumn, requestParams);

                const requestedSampleId = summary.requiredSampleId;
                return FilterFacetsService.getFilterFacets(projectKey, dataSpec, request, requestedSampleId, filtersPivotRequestMonoFuture);
            } catch (error) {
                Logger.info('Not executing filter request', error);
            }
        }

        function getShaker(projectKey, datasetSmartName) {
            return DataikuAPI.explores.get(projectKey, datasetSmartName).noSpinner()
                .then(data => {
                    return data.script || data.data.script;
                });
        }

        function fetchColumnsSummary(projectKey, dataSpec, filtersParams) {
            return DashboardPageUtils.getColumnSummary(projectKey, dataSpec)
                .then(data => {
                    let usableColumnsForEngine = ChartUsableColumns.makeUsableColumns(data, getCacheableContext(projectKey, filtersParams.datasetSmartName));
                    if (usableColumnsForEngine && usableColumnsForEngine.length) {
                        usableColumnsForEngine = usableColumnsForEngine.filter(acceptColumnAsFilter);
                        usableColumnsForEngine.forEach(filter => ChartFilters.autocompleteFilter(filter, usableColumnsForEngine, true));
                    }

                    return { summary: data, usableColumns: usableColumnsForEngine };
                });
        }

        function getCacheableContext(datasetProjectKey, datasetName) {
            return {
                datasetProjectKey,
                datasetName,
                context: FilterPanelContext.FILTER_PANEL
            };
        }

        function canFilter(editable, filters, crossFiltersEnabled) {
            return editable || (filters && filters.length || crossFiltersEnabled);
        }

        function isFiltersPanelDocked(panelPosition) {
            return panelPosition !== FiltersPanelPosition.TILE;
        }

        /**
         * Returns the dataset types
         * @returns {Promise<DatasetType[]>}
         */
        function getDatasetTypes() {
            if (!_.isNil(datasetTypes)) {
                return Promise.resolve(datasetTypes);
            }
            return CachedAPICalls.datasetTypes.then(data => {
                datasetTypes = data;
                return datasetTypes;
            });

        }

        function hasCaseInsensitiveColumnNames(projectKey, datasetName, contextProjectKey) {
            const datasetKey = `${projectKey}/${contextProjectKey}/${datasetName}`;
            if (lowerCaseStatusByDatasetKey.has(datasetKey)) {
                return Promise.resolve(lowerCaseStatusByDatasetKey.get(datasetKey));
            }
            return new Promise((resolve, reject) => {
                if (lowerCaseStatusRequestsQueues.has(datasetKey)) {
                    lowerCaseStatusRequestsQueues.get(datasetKey).push( { queuedResolve: resolve, queuedReject: reject } );
                } else {
                    lowerCaseStatusRequestsQueues.set(datasetKey, [{ queuedResolve: resolve, queuedReject: reject }]);
                    DataikuAPI.datasets.get(projectKey, datasetName, contextProjectKey).then(({ data }) => {
                        getDatasetTypes().then((datasetTypes) => {
                            const hasCaseInsensitiveColumnNamesStatus = datasetTypes[data.type] && datasetTypes[data.type]['hasCaseInsensitiveColumnNames'];
                            lowerCaseStatusByDatasetKey.set(datasetKey, hasCaseInsensitiveColumnNamesStatus);
                            lowerCaseStatusRequestsQueues.get(datasetKey).forEach(({ queuedResolve }) =>
                                queuedResolve(hasCaseInsensitiveColumnNamesStatus));
                            lowerCaseStatusRequestsQueues.delete(datasetKey);
                        });
                    }).catch(error => {
                        // reject all promises from the queue
                        lowerCaseStatusRequestsQueues.get(datasetKey).forEach(({ queuedReject }) =>
                            queuedReject(error));
                        lowerCaseStatusRequestsQueues.delete(datasetKey);
                    });
                }
            });
        }

        function flushDatasetLowerCaseStatus() {
            lowerCaseStatusByDatasetKey.clear();
        }

        return {
            SAMPLING_PROPS,
            MAX_DATA_BYTES,
            FILTERS_ON_MSG,
            filtersSourceTypes,
            getSamplingProps,
            getSamplingRefreshTrigger,
            resetGlobalFilters,
            setGlobalFilters,
            getGlobalFilters,
            getActiveGlobalFilters,
            hasGlobalFiltersApplied,
            getUsableFilters,
            getUnusableColumnFilters,
            findFiltersTileInPage,
            findFiltersTileList,
            getPageFilters,
            applyFilters,
            onApplyFiltersSuccess,
            setInsightsMap,
            getSlideFilterableTiles,
            getDashboardExplorationFilters,
            initPageFilters,
            isSameSampling,
            setCrossFilterStatusToPageId,
            canCrossFilter,
            setIsFilterPanelDeactivated,
            getIsFilterPanelDeactivated,
            acceptColumnAsFilter,
            getDataSpec,
            executeFiltersPivotRequest,
            fetchColumnsSummary,
            getShaker,
            canFilter,
            isFiltersPanelDocked,
            hasCaseInsensitiveColumnNames,
            flushDatasetLowerCaseStatus,
            resetAllDashboardPagesFilterPanelStates
        };
    });
})();
