const app = angular.module('dataiku.labeling', ['platypus.utils']);

app.controller("LabelingTaskController", function($scope, $state, $stateParams, $timeout, DataikuAPI, TopNav, $q, Dialogs, DKUtils, ActivityIndicator) {
    $scope.identifier = $stateParams.identifier;

    $q.all([
        DataikuAPI.labelingtasks.get($stateParams.projectKey, $stateParams.labelingTaskId),
        DataikuAPI.labelingtasks.getLabelingTaskPrivileges($stateParams.projectKey, $stateParams.labelingTaskId),
    ]).then(function([labelingTaskData, privileges]) {
        $scope.labelingTask = labelingTaskData.data;
        $scope.originLabelingTask = angular.copy($scope.labelingTask);

        $scope.privilegesOnTask = {
            canReadConf: privileges.data.includes("READ_CONF"),
            canAnnotate: privileges.data.includes("ANNOTATE"),
            canReview: privileges.data.includes("REVIEW"),
            canWriteConf: privileges.data.includes("WRITE_CONF"),
        };

        TopNav.setItem(TopNav.ITEM_LABELING_TASK, $stateParams.labelingTaskId, {name: $scope.labelingTask.name});
        TopNav.setPageTitle($scope.labelingTask.name + " - Labeling Task");
        
        // Need to set the top location as it depends on the user rights and those rights might have not be loaded when top has been set
        TopNav.setTopLocation($scope.isProjectAnalystRO() ? TopNav.TOP_FLOW : TopNav.TOP_LABELING_TASK);

        const isLabelingTaskSet = !!$scope.labelingTask.idColumn;
        if ($scope.privilegesOnTask.canWriteConf && !$scope.labelingTask.idColumn) {
            // Guess the id column and save the labeling task if such a column is found
            guessIdColumnAndSaveTask(() => redirectToAccessibleTabIfNeeded(isLabelingTaskSet));
        } else {
            redirectToAccessibleTabIfNeeded(isLabelingTaskSet);
        }

    }).catch(setErrorInScope.bind($scope));

    function guessIdColumnAndSaveTask(callback) {
        // only guessing ids for image tasks for now
        if (!['IMAGE_CLASSIFICATION', 'OBJECT_DETECTION'].includes($scope.labelingTask.type)) {
            callback();
            return;
        }

        const { projectKey: datasetProjectKey, datasetName: notSmartDatasetName} = resolveDatasetFullName($scope.labelingTask.inputs.metadata.items[0].ref, $stateParams.projectKey);

        DataikuAPI.datasets.get(datasetProjectKey, notSmartDatasetName, $stateParams.projectKey).success(async (dataset) => { 
        
            if ($scope.privilegesOnTask.canWriteConf && !$scope.labelingTask.idColumn) {
                const columns = dataset.schema.columns.map(c => c.name);
                let idColumn;

                if (columns.length === 1) {
                    idColumn = columns[0];
                } else {
                    const pathRelatedStr = ['path', 'file', 'image'];
                    const regex = new RegExp(pathRelatedStr.join('|'), 'i');
                    idColumn = columns.find((column) => regex.test(column));
                }
    
                if (idColumn) {
                    $scope.labelingTask.idColumn = idColumn;
                    await saveAfterConflictCheck("Auto saved, new guessed id column: " + idColumn); // don't need to check conflicts
                }
            }

            callback();
            
        }).error(function() {
            $scope.noInputReadAccess = true;
            setErrorInScope.apply($scope, arguments)
        });
    }

    const requiredPrivilegeForTab = {
        settings: "canReadConf",
        annotate: "canAnnotate",
        review: "canReview",
        overview: "canReadConf"
    };

    function defautTabDependingOnPerm(isLabelingTaskSet) {
        if (!$scope.identifier && $scope.privilegesOnTask.canWriteConf) {
            if (isLabelingTaskSet) {
                return "overview"
            } else {
                return "settings";
            }
        } else if ($scope.privilegesOnTask.canReview) {
            return "review";
        } else if ($scope.privilegesOnTask.canAnnotate) {
            return "annotate";
        } else if ($scope.privilegesOnTask.canReadConf) {
            return "overview";
        }
    }

    function redirectToAccessibleTabIfNeeded(isLabelingTaskSet) {
        const stateBaseName = "projects.project.labelingtasks.labelingtask"
        let tabToGoTo;
        if ($state.current.name === stateBaseName) {
            // This state is a transition, it's used as link in the labeling tasks list. In this controller
            // DSS will decide where the user should be redirected using its rights
            tabToGoTo = defautTabDependingOnPerm(isLabelingTaskSet);
        } else {
            const tab = $state.current.name.replace(stateBaseName+".", "");
            if (!$scope.privilegesOnTask[requiredPrivilegeForTab[tab]]) {
                // user doesn't have perm for this tab, redirecting him/her
                tabToGoTo = defautTabDependingOnPerm(isLabelingTaskSet);
            }
        }

        if (tabToGoTo) {
            $state.go([stateBaseName, tabToGoTo].join("."), {
                projectKey: $stateParams.projectKey, 
                labelingTaskId: $stateParams.labelingTaskId,
                identifier: $scope.identifier
            }, {location: 'replace'});
        }
    }

    $scope.isPartialTaskSettingsValid = true;
    $scope.identifier = $stateParams.identifier;
    $scope.dirtyLabel = false;

    $scope.postSave = function(savedLabelingTask) {
        $scope.labelingTask = savedLabelingTask;
        $scope.originLabelingTask = angular.copy($scope.labelingTask);
        $scope.currentSaveCommitMessage = null;
        ActivityIndicator.success("Saved");
    }

    $scope.canSave = function() {
        return $scope.labelingTask && $scope.labelingTask.name && $scope.labelingTask.name.length && $scope.isPartialTaskSettingsValid;
    };

    $scope.isLabelingTaskDirty = function() {
        return !angular.equals($scope.labelingTask, $scope.originLabelingTask);
    }

    checkChangesBeforeLeaving($scope, () => $scope.isLabelingTaskDirty() || $scope.dirtyLabel, null, null, () => {
        // If tab is changed and changes were not saved we should reset the LT to the original state
        $scope.labelingTask = angular.copy($scope.originLabelingTask);
        $scope.dirtyLabel = false;
    });

    // comes from Angular - refers to data under Settings tab
    $scope.partialTaskChange = function(partialTask) {
        $timeout(function() {
            Object.keys(partialTask).forEach((k) => {
                if (!$scope.labelingTask[k] && partialTask[k] === null) {
                    delete partialTask[k];
                }
            });
            Object.assign($scope.labelingTask, partialTask);
        });
    };

    $scope.closeError = function() {
        $scope.fatalAPIError = null;
        $scope.saveError = null;
    }

    $scope.partialTaskValidityChange = function(isValid) {
        $timeout(function() {
            $scope.isPartialTaskSettingsValid = isValid;
        });
    }

    $scope.dirtyLabelChange = function(dirtyLabel) {
        $timeout(function() {
            $scope.dirtyLabel = dirtyLabel;
        });
    }

    function saveAfterConflictCheck(commitMessage) {
        return new Promise((resolve, reject) => {
            $scope.currentSaveCommitMessage = commitMessage;
            DataikuAPI.labelingtasks.save($scope.labelingTask, null, {}, commitMessage).success((savedLabelingTask) => {
                $scope.postSave(savedLabelingTask);
                resolve();
            }).error(function() {
                setErrorInScope.apply($scope, arguments);
                $scope.saveError = getErrorDetails.apply(this, arguments);
                reject();
            });
        });
    };

    $scope.saveLabelingTask = function(commitMessage) {
        const deferred = $q.defer();

        DataikuAPI.labelingtasks.checkSaveConflict($scope.labelingTask).success(function(conflictResult) {
            if (!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope,conflictResult).then((resolutionMethod) => {
                        if (resolutionMethod === 'erase') {
                            saveAfterConflictCheck(commitMessage);
                        } else if (resolutionMethod === 'ignore') {
                            $scope.labelingTask = $scope.originLabelingTask; // avoid checkChangesBeforeLeaving to display warning when state is reloaded
                            DKUtils.reloadState();
                        }
                    }
                );
            } else {
                saveAfterConflictCheck(commitMessage);
            }
        }).error((data, status, headers, config, statusText) => {
            setErrorInScope.call($scope, data, status, headers, config, statusText);
            $scope.saveError = getErrorDetails(data, status, headers, statusText);
        });
        
        return deferred.promise;
    }
});

app.controller("LabelingTasksListController", function($scope, $controller, $stateParams, $filter, TopNav, DataikuAPI) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    TopNav.setNoItem();
    TopNav.setLocation($scope.isProjectAnalystRO() ? TopNav.TOP_FLOW : TopNav.TOP_LABELING_TASK, TopNav.ITEM_LABELING_TASK, TopNav.TABS_NONE, null);

    $scope.sortBy = [
        { value: 'realName', label: 'Name' },
        { value: 'type', label: 'Type' },
        { value: 'owner', label: 'Owner'},
        { value: '-lastModifiedOn', label: 'Last modified'},
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            }
        },
        filterParams: {
            userQueryTargets: ["realName", "owner", "tags", "type"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.sortCookieKey = 'labelingtasks';
    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.labelingtasks.listHeads($stateParams.projectKey).success(function (data) {
            data.forEach(labelingTask => {
                labelingTask.realName = labelingTask.name;
                labelingTask.name = labelingTask.id;
                labelingTask.description = `${labelingTask.realName} (${labelingTask.id})`;
            });
    
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.list();
});

app.controller("LabelingTaskSettingsController", function(TopNav) {
    TopNav.setLocation(TopNav.TOP_FLOW, TopNav.ITEM_LABELING_TASK, TopNav.TABS_LABELING_TASK, "settings");
});

app.controller("LabelingTaskAnnotateController", function(TopNav, $scope) {
    TopNav.setLocation($scope.isProjectAnalystRO() ? TopNav.TOP_FLOW : TopNav.TOP_LABELING_TASK, TopNav.ITEM_LABELING_TASK, TopNav.TABS_LABELING_TASK, "annotate");
});

app.controller("LabelingTaskReviewController", function(TopNav, $scope) {
    TopNav.setLocation($scope.isProjectAnalystRO() ? TopNav.TOP_FLOW : TopNav.TOP_LABELING_TASK, TopNav.ITEM_LABELING_TASK, TopNav.TABS_LABELING_TASK, "review");
});

app.controller("LabelingTaskOverviewController", function(TopNav, $scope) {
    TopNav.setLocation($scope.isProjectAnalystRO() ? TopNav.TOP_FLOW : TopNav.TOP_LABELING_TASK, TopNav.ITEM_LABELING_TASK, TopNav.TABS_LABELING_TASK, "overview");
});

app.controller("LabelingTaskCreationController", function(Logger, $scope, $rootScope, DataikuAPI, StateUtils, $stateParams, LabelingTaskIOService){
    // new task can be already set in case of copy
    if ($scope.newTask) {
        $scope.creationStep = "IO";
        $scope.newTask.name = $scope.newTask.name + "_copy";
        $scope.newTask.outputs.main.items = [];
    } else {
        $scope.creationStep = "type";
        $scope.newTask = {
            projectKey: $stateParams.projectKey,
            inputs: {
                metadata: {
                    items: [{ ref: $scope.preselectedInputDataset }]},
            },
            outputs: {
                main: {
                    items: [],
                },  
            }
        };
    }

    $scope.outputDataset = null;
    $scope.newLabelsDataset = {};
    $scope.editing = false;
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.newTask.inputs.metadata.items[0].ref = preselectedInputDataset;
        $scope.preselectedInputDataset = preselectedInputDataset;
    });
    addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

    LabelingTaskIOService.listUsableComputables($stateParams.projectKey).then((usableInputs) => {
        $scope.usableInputDatasets = Object.values(usableInputs.usableInputDatasetsMap);
        $scope.usableInputManagedFolders = Object.values(usableInputs.usableInputManagedFoldersMap);
    });

    $scope.chooseType = function(taskType) {
        $scope.creationStep = "IO";
        $scope.newTask.type = taskType;
        if ($scope.taskNeedFolder()) {
            $scope.newTask.inputs.data = {
                items: [{ ref: null }]
            }
        }
    }

    $scope.requiredInputsSentence = function() {
        const needsFolder = $scope.taskNeedFolder();
        const hasFolder =  $scope.newTask.inputs.data && $scope.newTask.inputs.data.items[0].ref;
        const hasMetadata = $scope.newTask.inputs.metadata.items[0].ref;

        const ret  = [];

        if (!hasMetadata) {
            ret.push("an input dataset")
        }

        if (needsFolder && !hasFolder) {
            ret.push("an input folder");
        }
        return ret.length > 0 ? "This labeling task requires " +  ret.join(" and ") : "";
    }

    $scope.taskNeedFolder = function() {
        return ['IMAGE_CLASSIFICATION', 'OBJECT_DETECTION'].includes($scope.newTask.type);
    }

    $scope.createLabelingTask = function() {
        // TODO @labeling let the user choose the name
        if (!$scope.newTask.name) {
            $scope.newTask.name = "label_" + $scope.newTask.inputs.metadata.items[0].ref;
        } 

        $scope.newTask.outputs.main.items = $scope.outputDataset ? [{ref: $scope.outputDataset.name}] : []
        DataikuAPI.labelingtasks.save($scope.newTask, $scope.zone).success(function(data) {
            $scope.creatingLabelingTask = false;
            StateUtils.go.labelingTask(data.id);
            $scope.dismiss();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.startAddingOuput = function() {
        $scope.newLabelsDataset.name = '';
        $scope.editing = true;
    };

    $scope.cancelAddOutput = function() {
        $scope.editing = false;
    };
    
    $scope.removeOutput = function() {
        $scope.outputDataset = null;
    };

    DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {datasetsOnly: true}).success(function(data) {
        let labelDatasets = data.filter(d => d.dataset.type === "Labels");
        $scope.editOutput = {"usable": labelDatasets};
    }).error(setErrorInScope.bind($scope));
    
    $scope.createAndUseLabelsDataset = function() {
        Logger.info("Create and use labels dataset", $scope);
        let dataset = {
            type: "Labels",
            name: $scope.newLabelsDataset.name,
            projectKey: $stateParams.projectKey,
            params: {},
            schema: {}
        };
        DataikuAPI.datasets.create($stateParams.projectKey, dataset, $scope.zone).success(function(dataset) {
            $scope.outputDataset = dataset;
            $scope.editing = false;

            $rootScope.$emit('datasetsListChangedFromModal');
        }).error(setErrorInScope.bind($scope));
    };
});


app.service("LabelingTasksCustomFieldsService", function ($rootScope, TopNav, DataikuAPI, ActivityIndicator, WT1) {
    let svc = {};

    svc.saveCustomFields = function(labelingTask, newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'LABELING_TASK'});
        let oldCustomFields = angular.copy(labelingTask.customFields);
        labelingTask.customFields = newCustomFields;
        return DataikuAPI.labelingtasks.save(labelingTask, null, {summaryOnly: true})
            .success(function() {
                ActivityIndicator.success("Saved");
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), labelingTask.customFields);
                $rootScope.$broadcast('reloadGraph');
            })
            .error(function(a, b, c) {
                labelingTask.customFields = oldCustomFields;
                setErrorInScope.bind($rootScope)(a, b, c);
            });
    };

    return svc;
});

app.controller("LabelingTaskPageRightColumnActions", async function($controller, $scope, $rootScope, $stateParams, $state, DataikuAPI, ActiveProjectKey) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.labelingTaskData = (await DataikuAPI.labelingtasks.getFullInfo(ActiveProjectKey.get(), $stateParams.labelingTaskId)).data;
    $scope.labelingTask = ($scope.labelingTaskData || {}).labelingTask;
    
    // Needed for compatibility with serialized node model. When serializing a node the name of the node is actually the id and the description is the name
    // The right column summary in the flow view expects the description to be the name of the labeling task
    $scope.labelingTask.description = $scope.labelingTask.name;
    // The call to tasks.get in labelingTaskRightColumnSummary expects the name to be the id of the object. 
    $scope.labelingTask.name = $scope.labelingTask.id;
    $scope.labelingTask.nodeType = "LABELING_TASK";
    $scope.labelingTask.interest = $scope.labelingTaskData.interest;

    $scope.selection = {
        selectedObject : $scope.labelingTask,
        confirmedItem : $scope.labelingTask
    };

    $scope.renameObjectAndSave = function(newName) {
        $scope.labelingTask.name = newName;
        return DataikuAPI.labelingtasks.rename($scope.labelingTask.projectKey, $scope.labelingTask.id, newName);
    };
});

app.controller("LabelingTaskIOController", function($scope, DataikuAPI, TopNav, $stateParams, LabelingTaskIOService, ) {
    TopNav.setLocation(TopNav.TOP_FLOW, TopNav.ITEM_LABELING_TASK, TopNav.TABS_LABELING_TASK, "io");
    addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);
    $scope.newLabelsDataset = {};

    LabelingTaskIOService.listUsableComputables($stateParams.projectKey).then((computables) => {
        $scope.usableOutputLabelsDatasetsMap = computables.usableOutputLabelsDatasetsMap;
        if ($scope.labelingTask.inputs.data && $scope.labelingTask.inputs.data.items.length) {
            $scope.usedManagedFolder = computables.usableInputManagedFoldersMap[$scope.labelingTask.inputs.data.items[0].ref];
        }
        $scope.usedMetadataDataset = computables.usableInputDatasetsMap[$scope.labelingTask.inputs.metadata.items[0].ref];
    });

    $scope.startAddingOuput = function() {
        $scope.editing = true;
    }

    $scope.cancelAddOutput = function() {
        $scope.editing = false;
    }
    
    $scope.removeOutput = function(index) {
        $scope.labelingTask.outputs.main.items.splice(index, 1);
    }

    $scope.invalidateMetadataTableCache = function() {
        return DataikuAPI.labelingtasks.invalidateMetadataTableCache($stateParams.projectKey, $stateParams.labelingTaskId);
    };

    $scope.createAndUseLabelsDataset = function() {
        let dataset = {
            type: "Labels",
            name: $scope.newLabelsDataset.name,
            projectKey: $stateParams.projectKey,
            params: {},
            schema: {}
        };
        DataikuAPI.datasets.create($stateParams.projectKey, dataset, $stateParams.zoneId).success(function(savedDataset) {
            LabelingTaskIOService.listUsableComputables($stateParams.projectKey).then((computables) => {

                $scope.editing = false;
                $scope.usableOutputLabelsDatasetsMap = computables.usableOutputLabelsDatasetsMap;
                $scope.labelingTask.outputs.main.items.push({ref: savedDataset.name});
                $scope.newLabelsDataset = {};

                // refresh list of dataset for uniqueness check
                addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);

                // Force save to take into account added output to the labeling task params
                $scope.saveLabelingTask();
            }).catch(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    }
});

app.service("LabelingTaskIOService", function(DataikuAPI, $q) {
    return {
        listUsableComputables: function(projectKey) {
            var deferred = $q.defer();
            DataikuAPI.flow.listUsableComputables(projectKey, {
                datasetsOnly : false,
            }).success(function(data) {
                const usableInputDatasetsMap = {};
                const usableInputManagedFoldersMap = {};
                const usableOutputLabelsDatasetsMap = {};
                for (const computable of Object.values(data)) {
                    if (computable.type == "DATASET") {
                        usableInputDatasetsMap[computable.smartName] = computable;
                        if (computable.datasetType === "Labels") {
                            usableOutputLabelsDatasetsMap[computable.smartName] = computable;
                        }
                    } else if (computable.type == "MANAGED_FOLDER") {
                        usableInputManagedFoldersMap[computable.smartName] = computable;
                    }
                }
                deferred.resolve({usableInputDatasetsMap, usableInputManagedFoldersMap, usableOutputLabelsDatasetsMap});
            }).error(deferred.reject);
            return deferred.promise;
        }
    }

    
});

app.directive('labelingTaskRightColumnSummary', function ($rootScope, $controller, DataikuAPI, LabelingTasksCustomFieldsService, ActiveProjectKey, CreateModalFromTemplate, ActivityIndicator) {
    return {
        templateUrl :'/templates/labelingtasks/right-column-summary.html',
    
        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});
            
            scope.refreshData = function() {
                DataikuAPI.labelingtasks.getFullInfo(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name).success(function(data){
                    scope.labelingTaskData = data;
                    scope.labelingTask = data.labelingTask;
                    scope.labelingTask.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;
                }).error(setErrorInScope.bind(scope));
            };
            
            scope.$watch("selection.selectedObject",function() {
                if(scope.selection && scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.labelingTaskData = null;
                }
            });
            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();

            });

            scope.editCustomFields = function() {
                if (!scope.labelingTask) {
                    return;
                }
                let modalScope = angular.extend(scope, {objectType: 'LABELING_TASK', objectName: scope.labelingTask.name, objectCustomFields: scope.labelingTask.customFields});
                CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                    LabelingTasksCustomFieldsService.saveCustomFields(scope.labelingTask, customFields);
                });
            };

            scope.$on("objectSummaryEdited", function() {
                // @todo labeling: change permission check to match future labeling task permission system
                if (scope.$root.projectSummary && scope.$root.projectSummary.canWriteProjectContent) {
                    // @todo labeling: update save with new parameter
                    DataikuAPI.labelingtasks.save(scope.labelingTask, null, {summaryOnly: true})
                    .success(function() {
                        ActivityIndicator.success("Saved");
                    }).error(setErrorInScope.bind(scope));
                }
            });

            scope.$on('taggableObjectTagsChanged', () => scope.refreshData());        
            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);
            
            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "LABELING_TASK", ActiveProjectKey.get(), scope.selection.selectedObject.name).success(function(data) {
                    scope.selection.selectedObject.interest = data;
                    scope.labelingTaskData.interest = data;
                }).error(setErrorInScope.bind(scope));
            }
            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            scope.$on("$destroy", interestsListener);
        }
    }
});

app.controller("LabelingTaskHistoryController", function(TopNav, $scope) {
    TopNav.setLocation($scope.isProjectAnalystRO() ? TopNav.TOP_FLOW : TopNav.TOP_LABELING_TASK, TopNav.ITEM_LABELING_TASK, TopNav.TABS_LABELING_TASK, "history");
});

