// @ts-check
(function() {
    'use strict';

    // @ts-ignore
    angular.module('dataiku.directives.simple_report')
        .factory('ChartFeatures', chartFeatures);

    /**
     * ChartFeatures service
     * Defines which UI option is available based on chart type.
     * (!) This service previously was in static/dataiku/js/simple_report/services/chart-features.service.js
     * @typedef {import("../../../../../../../../server/src/frontend/src/app/features/simple-report/services/chart-features/chart-features.type").ChartFeatureReturnType} ChartFeatureReturnType
     * @returns {ChartFeatureReturnType}
     */
    function chartFeatures(WebAppsService, ChartStoreFactory, ChartDimension, ChartsStaticData, ChartUADimension, ChartTypeChangeUtils, D3ChartZoomControl, CHART_TYPES, CHART_VARIANTS, CHART_AXIS_TYPES, NumberFormatter, AxisTicksConfigMode, VALUES_DISPLAY_MODES, translate, $rootScope) {

        function isChartTypeInList(chartType, chartTypeList) {
            return chartTypeList.includes(chartType);
        }

        function canDisplayTotalValues(chartDef) {
            return chartDef.type === CHART_TYPES.STACKED_COLUMNS && chartDef.variant !== CHART_VARIANTS.stacked100;
        }

        const canSetCustomAxisRange = chartDef => chartDef && chartDef.variant !== CHART_VARIANTS.binnedXYHexagon;
        const axisTicksModes = [
            { label: translate('CHARTS.SETTINGS.FORMAT.AXIS_TICKS.NUMBER', 'Number'), value: AxisTicksConfigMode.NUMBER },
            { label: translate('CHARTS.SETTINGS.FORMAT.AXIS_TICKS.INTERVAL', 'Interval'), value: AxisTicksConfigMode.INTERVAL }
        ];

        /** @type {ChartFeatureReturnType} */
        const svc = {

            canDisplayTotalValues,

            canExportToExcel: chartDef => {
                if (chartDef.facetDimension.length) {
                    return false;
                }
                return (isChartTypeInList(chartDef.type, [
                    CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_AREA,
                    CHART_TYPES.LINES, CHART_TYPES.PIVOT_TABLE, CHART_TYPES.GROUPED_XY, CHART_TYPES.KPI
                ]) || (chartDef.type === CHART_TYPES.GROUPED_COLUMNS && chartDef.variant !== CHART_VARIANTS.waterfall));
            },

            canExportToImage: chartDef => {
                if (chartDef.facetDimension.length) {
                    return false;
                }
                return !isChartTypeInList(chartDef.type, [
                    CHART_TYPES.PIVOT_TABLE, CHART_TYPES.KPI, CHART_TYPES.SCATTER_MAP, CHART_TYPES.ADMINISTRATIVE_MAP,
                    CHART_TYPES.GRID_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.WEBAPP, CHART_TYPES.DENSITY_HEAT_MAP
                ]);
            },

            isMap: chartType => {
                return isChartTypeInList(chartType, [
                    CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.GRID_MAP, CHART_TYPES.SCATTER_MAP, CHART_TYPES.ADMINISTRATIVE_MAP, CHART_TYPES.DENSITY_HEAT_MAP
                ]);
            },

            canDisplayDimensionValuesInChart: chartType => {
                return chartType === CHART_TYPES.TREEMAP;
            },

            getExportDisabledReason: chartDef => {
                if (chartDef.facetDimension.length) {
                    return 'Download is not available for subcharts.';
                }
            },

            canDrillHierarchy: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.RADAR,
                CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.PIE, CHART_TYPES.STACKED_AREA, CHART_TYPES.BOXPLOTS,
                CHART_TYPES.BINNED_XY, CHART_TYPES.GROUPED_XY, CHART_TYPES.LIFT, CHART_TYPES.PIVOT_TABLE, CHART_TYPES.TREEMAP
            ]),

            canDisplayBreadcrumb: chartType => svc.canDrillHierarchy(chartType) && chartType !== CHART_TYPES.PIVOT_TABLE,

            shouldDisplayBreadcrumb: chartDef => svc.canDisplayBreadcrumb(chartDef.type) && chartDef.drilldownOptions.displayBreadcrumb,

            canAnimate: (chartType, chartVariant) => isChartTypeInList(chartType, [
                CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_COLUMNS,
                CHART_TYPES.STACKED_BARS, CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.PIE,
                CHART_TYPES.BINNED_XY, CHART_TYPES.GROUPED_XY, CHART_TYPES.LIFT
            ]) && chartVariant !== CHART_VARIANTS.waterfall,

            canFacet: (chartType, webAppType, chartVariant) => {
                if (chartType === CHART_TYPES.GROUPED_COLUMNS) {
                    return chartVariant !== CHART_VARIANTS.waterfall;
                }
                if (chartType === CHART_TYPES.WEBAPP) {
                    const loadedDesc = WebAppsService.getWebAppLoadedDesc(webAppType) || {};
                    const pluginChartDesc = (loadedDesc.desc && loadedDesc.desc.chart) || {};
                    return pluginChartDesc.canFacet === true;
                } else {
                    return isChartTypeInList(chartType, [
                        CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.STACKED_COLUMNS,
                        CHART_TYPES.STACKED_BARS, CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.PIE,
                        CHART_TYPES.BINNED_XY, CHART_TYPES.GROUPED_XY, CHART_TYPES.LIFT
                    ]);
                }
            },

            canSetSubchartsCommonXAxis: chartType => chartType !== CHART_TYPES.PIE,

            canFilter: (chartType, webAppType) => {
                if (chartType === CHART_TYPES.WEBAPP) {
                    const loadedDesc = WebAppsService.getWebAppLoadedDesc(webAppType) || {};
                    const pluginChartDesc = (loadedDesc.desc && loadedDesc.desc.chart) || {};
                    return pluginChartDesc.canFilter === true;
                } else {
                    return true;
                }
            },

            canHaveZoomControls: (chartDef, zoomControlInstanceId) => {
                // @ts-ignore
                return !_.isNil(chartDef) && !_.isNil(zoomControlInstanceId) && (svc.canHaveEChartsZoomControls(chartDef) || svc.hasScatterZoomControlEnabled(chartDef.type, zoomControlInstanceId) || svc.canHaveMapsZoomControls(chartDef));
            },

            canHaveEChartsZoomControls: chartDef => chartDef.type === CHART_TYPES.TREEMAP && !!ChartDimension.getYDimensions(chartDef).length && !!chartDef.genericMeasures.length,

            canHaveMapsZoomControls: chartDef => svc.isMap(chartDef.type) && (chartDef.geometry.length > 0 || chartDef.geoLayers.length > 1),

            isScatterZoomed: (chartType, zoomControlInstanceId) => {
                return svc.isUnaggregated(chartType) && D3ChartZoomControl.isZoomed(zoomControlInstanceId);
            },

            isUnaggregated: (chartType) => {
                return chartType === CHART_TYPES.SCATTER || chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
            },

            hasScatterZoomControlEnabled: (chartType, zoomControlInstanceId) => {
                return svc.isUnaggregated(chartType) && D3ChartZoomControl.isEnabled(zoomControlInstanceId);
            },

            hasScatterZoomControlActivated: (chartType, zoomControlInstanceId) => {
                return svc.isUnaggregated(chartType) && D3ChartZoomControl.isActivated(zoomControlInstanceId);
            },

            canSmooth: chartType => isChartTypeInList(chartType, [CHART_TYPES.LINES, CHART_TYPES.STACKED_AREA, CHART_TYPES.MULTI_COLUMNS_LINES]),

            canSetStrokeWidth: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GEOMETRY_MAP
            ]),

            canSetFillOpacity: chartDef => {
                return (chartDef.type === CHART_TYPES.GEOMETRY_MAP || (chartDef.type === CHART_TYPES.ADMINISTRATIVE_MAP && chartDef.variant === CHART_VARIANTS.filledMap));
            },

            canHideXAxis: chartType => chartType === CHART_TYPES.STACKED_BARS,

            canDisplayValuesInChart: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GAUGE, CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.PIE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.STACKED_BARS, CHART_TYPES.TREEMAP
            ]),

            canDisplayBrush: chartDef => {
                return chartDef && chartDef.linesZoomOptions && chartDef.linesZoomOptions.enabled && ChartDimension.isInteractiveChart(chartDef);
            },

            canDrawIdentityLine: chartDef => {
                const yAxisFormatting = chartDef.yAxesFormatting.find(formatting => formatting.id === ChartsStaticData.LEFT_AXIS_ID);
                return chartDef && chartDef.type === CHART_TYPES.SCATTER && !(chartDef.xAxisFormatting.isLogScale || (yAxisFormatting && yAxisFormatting.isLogScale));
            },

            canCustomizeValuesInChart: chartDef => {
                return isChartTypeInList(chartDef.type, [
                    CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.GAUGE, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.TREEMAP])
                    || (chartDef.type === CHART_TYPES.PIE && svc.isEChart(chartDef));
            },

            canCustomizeLabelsInChart: chartDef => chartDef.type === CHART_TYPES.PIE && svc.isEChart(chartDef),

            canCustomizeValuesInChartOverlap: chartDef => isChartTypeInList(chartDef.type, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS
            ]) && svc.canCustomizeValuesInChart(chartDef),

            canAddMeasuresToLabelsInChart: chartDef => isChartTypeInList(chartDef.type, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.PIE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.TREEMAP, CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES]) && chartDef.variant !== CHART_VARIANTS.stacked100,

            shouldDisplayTotalValues: chartDef => canDisplayTotalValues(chartDef) && chartDef.valuesInChartDisplayOptions
                && chartDef.valuesInChartDisplayOptions.displayValues
                && chartDef.valuesInChartDisplayOptions.displayMode === VALUES_DISPLAY_MODES.VALUES_AND_TOTALS,

            shouldDisplay0Warning: (chartType, pivotResponse) => [CHART_TYPES.SANKEY, CHART_TYPES.TREEMAP].includes(chartType)
                && (pivotResponse && pivotResponse.aggregations.every(agg => agg.tensor.every(el => el === 0)))
            ,

            canDisplayLabelsInChart: chartType => isChartTypeInList(chartType, [CHART_TYPES.PIE, CHART_TYPES.GAUGE]),

            shouldDrawRegressionLine: (chartDef) => {
                return ChartUADimension.areAllNumericalOrDate(chartDef) && chartDef.type === CHART_TYPES.SCATTER && chartDef.scatterOptions
                && chartDef.scatterOptions.regression && chartDef.scatterOptions.regression.show;
            },

            canSelectPivotRowOrColumnDisplayMode: chartDef => chartDef != null && chartDef.type === CHART_TYPES.PIVOT_TABLE &&
                !!ChartDimension.getYDimensions(chartDef).length && !!ChartDimension.getXDimensions(chartDef).length,

            canCustomizePivotFormatting: chartDef => chartDef != null && chartDef.type === CHART_TYPES.PIVOT_TABLE &&
                (!!ChartDimension.getYDimensions(chartDef).length || !!ChartDimension.getXDimensions(chartDef).length)
                && (chartDef.genericMeasures && chartDef.genericMeasures.length),

            canCustomizePivotRowHeaders: chartDef => svc.canCustomizePivotFormatting(chartDef) &&
                !!ChartDimension.getYDimensions(chartDef).length || (chartDef.genericMeasures && chartDef.genericMeasures.length > 1),

            isColoredPivotTable: chartDef => chartDef != null && chartDef.type === CHART_TYPES.PIVOT_TABLE && (
                chartDef.colorMeasure && chartDef.colorMeasure.length ||
                chartDef.colorMode === 'COLOR_GROUPS'
            ),

            canSetMeasureColumnsTooltips: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GROUPED_COLUMNS,
                CHART_TYPES.STACKED_COLUMNS, CHART_TYPES.STACKED_BARS, CHART_TYPES.STACKED_AREA,
                CHART_TYPES.GRID_MAP, CHART_TYPES.LINES, CHART_TYPES.ADMINISTRATIVE_MAP, CHART_TYPES.PIE, CHART_TYPES.BINNED_XY
            ]),

            canSetUnaggregatedColumnsTooltips: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.SCATTER_MAP, CHART_TYPES.GEOMETRY_MAP, CHART_TYPES.DENSITY_HEAT_MAP
            ]),

            canDisplayTooltips: chartType => !isChartTypeInList(chartType, [
                CHART_TYPES.KPI, CHART_TYPES.GAUGE, CHART_TYPES.DENSITY_2D, CHART_TYPES.WEBAPP
            ]),

            shouldDisplayTooltips: (noTooltips, tooltipOptions, fixed) => {
                // @ts-ignore
                const isInADashboard = !_.isNil(noTooltips);

                // If the Show tooltip checkbox is unchecked at chart level, it can still be forced for this chart in a dashboard with the one set at dashboard level
                return (isInADashboard ? noTooltips === false : tooltipOptions.display) && !fixed;
            },

            canHandleEmptyBins: chartType => isChartTypeInList(chartType, [CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES]),

            canSetAxisTextColor: chartType => chartType !== CHART_TYPES.SCATTER_MULTIPLE_PAIRS,

            chartSupportOneTickPerBin: chartDef =>
                !(isChartTypeInList(chartDef.type,
                    [CHART_TYPES.PIE, CHART_TYPES.PIVOT_TABLE, CHART_TYPES.RADAR, CHART_TYPES.SANKEY, CHART_TYPES.TREEMAP, CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS, CHART_TYPES.BOXPLOTS, CHART_TYPES.GROUPED_XY]) || chartDef.hexbin)
                && chartDef.variant !== CHART_VARIANTS.binnedXYHexagon,

            chartOnlySupportsOneTickPerBin: (chartDef) => {
                return chartDef?.type === CHART_TYPES.GROUPED_COLUMNS && chartDef?.variant === CHART_VARIANTS.waterfall;
            },

            canModifyAxisRange: (chartDef, axis) => {
                if (!canSetCustomAxisRange(chartDef)) {
                    return false;
                }

                let dimensionsAndMeasures;

                if (axis === 'x') {
                    dimensionsAndMeasures = ChartTypeChangeUtils.takeAllInX(chartDef);
                } else if (axis === 'y') {
                    dimensionsAndMeasures = ChartTypeChangeUtils.takeAllInY(chartDef);
                } else {
                    return false;
                }

                const hasOneTickPerBin = svc.hasOneTickPerBinSelected(chartDef.$chartStoreId, axis);
                /*
                 * Only check the first value as the others are guaranteed to be of the same type.
                 * Dimensions always have one value, but measures can have multiple.
                 */
                return dimensionsAndMeasures.some(([firstOne]) =>
                    firstOne && (firstOne.isA === 'measure' || (ChartDimension.isTrueNumerical(firstOne) && !hasOneTickPerBin)));
            },

            canIncludeZero: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.LINES, CHART_TYPES.MULTI_COLUMNS_LINES,
                CHART_TYPES.GROUPED_XY
            ]),

            isPivotTableWithNoDimension: (chartDef) => {
                return chartDef.type === CHART_TYPES.PIVOT_TABLE && !ChartDimension.getXDimensions(chartDef).length && !ChartDimension.getYDimensions(chartDef).length;
            },

            canHaveConditionalFormatting: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.PIVOT_TABLE, CHART_TYPES.KPI
            ]),

            hasLogarithmicRegressionWarning: (chartDef) => {
                if (chartDef == null) {
                    return false;
                }

                const areValuesLessOrEqual0 = chartDef.xAxisFormatting && chartDef.xAxisFormatting.customExtent && chartDef.xAxisFormatting.customExtent.$autoExtent && chartDef.xAxisFormatting.customExtent.$autoExtent[0] <= 0;
                const isAxisMinAbove0 = chartDef.xAxisFormatting && chartDef.xAxisFormatting.customExtent && chartDef.xAxisFormatting.customExtent.manualExtent && chartDef.xAxisFormatting.customExtent.manualExtent[0] > 0;

                return areValuesLessOrEqual0 && !isAxisMinAbove0 && chartDef.type === CHART_TYPES.SCATTER;
            },

            hasExponentialRegressionWarning: (chartDef, yAxisID = ChartsStaticData.LEFT_AXIS_ID) => {
                if (chartDef == null) {
                    return false;
                }

                const yAxisFormatting = chartDef.yAxesFormatting.find(v => v.id === yAxisID);
                const areValuesLessOrEqual0 = yAxisFormatting && yAxisFormatting.customExtent && yAxisFormatting.customExtent.$autoExtent && yAxisFormatting.customExtent.$autoExtent[0] <= 0;
                const isAxisMinAbove0 = yAxisFormatting && yAxisFormatting.customExtent && yAxisFormatting.customExtent.manualExtent && yAxisFormatting.customExtent.manualExtent[0] > 0;

                return areValuesLessOrEqual0 && !isAxisMinAbove0 && chartDef.type === CHART_TYPES.SCATTER;
            },

            canEditAxisRange: (chartDef) => {
                return chartDef != null && !(ChartUADimension.areAllNumericalOrDate(chartDef) && chartDef.type === CHART_TYPES.SCATTER
                    && chartDef.scatterOptions && chartDef.scatterOptions.equalScales);
            },

            canShowXAxisTitle: (chartDef) => {
                const pairs = chartDef.uaDimensionPair.filter(pair => pair.uaXDimension && pair.uaXDimension.length && pair.uaXDimension[0].column && pair.uaYDimension && pair.uaYDimension.length && pair.uaYDimension[0].column);
                return !((chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS && pairs.length > 1) || chartDef.type === CHART_TYPES.BOXPLOTS);
            },

            hasMultipleXDim: chartType => chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS,

            canSetLogScale: (chartDef, axisName, id) => {
                if (chartDef == null) {
                    return false;
                }

                switch (axisName) {
                    case 'x':
                        if (chartDef.type === CHART_TYPES.SCATTER) {
                            return ChartUADimension.isTrueNumerical(chartDef.uaXDimension[0]);
                        } else if (chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                            return chartDef.uaDimensionPair.every(pair => pair.uaXDimension && ChartUADimension.isTrueNumerical(pair.uaXDimension[0]));
                        }
                        return chartDef.type === CHART_TYPES.STACKED_BARS && chartDef.variant !== CHART_VARIANTS.stacked100;
                    case 'y':
                        if (chartDef.type === CHART_TYPES.SCATTER) {
                            return ChartUADimension.isTrueNumerical(chartDef.uaYDimension[0]);
                        } else if (chartDef.type === CHART_TYPES.BINNED_XY) {
                            return chartDef.variant !== CHART_VARIANTS.binnedXYHexagon
                                && chartDef.yDimension[0]
                                && !ChartDimension.hasOneTickPerBin(chartDef.yDimension[0])
                                && ChartDimension.isTrueNumerical(chartDef.yDimension[0]);
                        } else if (chartDef.type === CHART_TYPES.SCATTER_MULTIPLE_PAIRS) {
                            const axisPair = chartDef.uaDimensionPair.find(pair => pair.id === id);
                            return axisPair && ChartUADimension.isTrueNumerical(axisPair.uaYDimension[0]);
                        }
                        return chartDef.type !== CHART_TYPES.STACKED_BARS
                            && chartDef.variant !== CHART_VARIANTS.stacked100 && chartDef.variant !== CHART_VARIANTS.waterfall;
                    default:
                        return false;
                }
            },

            canSetLeftRightYAxes: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.LINES
            ]),

            canSelectGridlinesAxis: chartType => svc.canSetLeftRightYAxes(chartType) || chartType === CHART_TYPES.SCATTER_MULTIPLE_PAIRS,

            canHaveVerticalGridlines: chartDef => !(chartDef.facetDimension.length && chartDef.singleXAxis)
                && !(chartDef.type === CHART_TYPES.BOXPLOTS && !chartDef.boxplotBreakdownDim.length && !chartDef.boxplotBreakdownHierarchyDimension.length),

            getVLinesDisabledTooltip: chartType => chartType === CHART_TYPES.BOXPLOTS
                ? translate('CHARTS.SETTINGS.FORMAT.GRIDLINES.NO_X_DIMENSION', 'Add an X dimension to be able to display vertical gridlines')
                : translate('CHARTS.SETTINGS.FORMAT.GRIDLINES.COMMON_X_AXIS_NOT_SUPPORTED', 'Vertical gridlines cannot be drawn on subcharts using the common X axis option.'),

            canSetMultiPlotDisplayMode: chartType => chartType === CHART_TYPES.MULTI_COLUMNS_LINES,

            isKPIChart: chartType => chartType === CHART_TYPES.KPI,

            canUseUnaggregatedMeasures: (chartDef, column, measureType, measureDisplayType, contextualMenuMeasureType) => {
                return $rootScope?.featureFlagEnabled?.('unaggregatedMeasures') && !!column // can't use unaggregated mode for count of records
                    && ([
                        // CHART_TYPES.MULTI_COLUMNS_LINES, CHART_TYPES.LINES,
                        CHART_TYPES.GROUPED_COLUMNS].includes(chartDef.type)
                        || (chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES && measureDisplayType === 'column')
                        || (chartDef.type === CHART_TYPES.STACKED_COLUMNS && chartDef.variant !== CHART_VARIANTS.stacked100)
                    )
                    && measureType === 'NUMERICAL' && (!contextualMenuMeasureType || contextualMenuMeasureType === 'generic' || contextualMenuMeasureType === 'valuesInChart' || contextualMenuMeasureType === 'tooltip');
            }, // for now unaggregated mode available only in chart def measures

            isGaugeChart: chartType => chartType === CHART_TYPES.GAUGE,

            isKPILikeChart: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.KPI, CHART_TYPES.GAUGE
            ]),

            canShowTextFormattingFromDropdown: chartType => isChartTypeInList(chartType, [
                CHART_TYPES.GAUGE
            ]),

            canDisplayNumberFormatting: measure => {
                return !((measure.type === 'CUSTOM' && measure.inferredType !== 'NUMERICAL')
                    || (measure.type === 'DATE' && !ChartsStaticData.ANY_AND_NUMERICAL_RESULT_AGGREGATION.includes(measure.function))
                    || ChartsStaticData.ALPHANUM_AND_ALPHANUMERICAL_RESULT_AGGREGATION.includes(measure.function));
            },

            canHaveThumbnail: chartDef => {
                if (chartDef.facetDimension.length) {
                    return false;
                }
                return !isChartTypeInList(chartDef.type, [CHART_TYPES.PIVOT_TABLE, CHART_TYPES.KPI, CHART_TYPES.WEBAPP]);
            },

            getMeasureComputationModes: chartDef => {
                const measureComputationModes = [];

                function addMeasureComputationMode(str) {
                    measureComputationModes.push(ChartsStaticData.stdAggrMeasureComputeModes[str]);
                }

                addMeasureComputationMode('NORMAL');

                const nbGDims = chartDef.genericDimension0.length + chartDef.genericHierarchyDimension.length + chartDef.genericDimension1.length;

                switch (chartDef.type) {
                    case CHART_TYPES.GROUPED_COLUMNS:
                    case CHART_TYPES.LINES:
                    case CHART_TYPES.MULTI_COLUMNS_LINES:
                        addMeasureComputationMode('PERCENTAGE');
                        addMeasureComputationMode('AVG_RATIO');
                        addMeasureComputationMode('CUMULATIVE');
                        addMeasureComputationMode('CUMULATIVE_PERCENTAGE');

                        if (nbGDims === 1) {
                            addMeasureComputationMode('DIFFERENCE');
                        }
                        break;
                    case CHART_TYPES.STACKED_COLUMNS:
                    case CHART_TYPES.STACKED_BARS:
                    case CHART_TYPES.STACKED_AREA:
                        addMeasureComputationMode('PERCENTAGE');
                        addMeasureComputationMode('CUMULATIVE');
                        addMeasureComputationMode('CUMULATIVE_PERCENTAGE');
                        break;
                    case CHART_TYPES.PIVOT_TABLE:
                        if (ChartDimension.getXDimensions(chartDef).length <= 1 && ChartDimension.getYDimensions(chartDef).length <= 1) {
                            addMeasureComputationMode('PERCENTAGE');
                            addMeasureComputationMode('AVG_RATIO');
                            addMeasureComputationMode('CUMULATIVE');
                            addMeasureComputationMode('CUMULATIVE_PERCENTAGE');
                            addMeasureComputationMode('DIFFERENCE');
                        }
                        break;
                    case CHART_TYPES.PIE:
                        addMeasureComputationMode('PERCENTAGE');
                        break;
                    case CHART_TYPES.TREEMAP:
                        if (ChartDimension.getYDimensions(chartDef).length <= 2) {
                            addMeasureComputationMode('PERCENTAGE');
                        }
                        break;
                }

                return measureComputationModes;
            },

            canSetAlongDimension: function(chartDef, computeMode) {
                const genericDimension0 = ChartDimension.getGenericDimension(chartDef);
                const xDimension = ChartDimension.getXDimension(chartDef);
                const yDimensions = ChartDimension.getYDimensions(chartDef);
                if (((genericDimension0 && chartDef.genericDimension1.length) || (xDimension && yDimensions.length) || (chartDef.type == CHART_TYPES.TREEMAP && yDimensions.length == 2))
                    && (computeMode == 'PERCENTAGE' || computeMode == 'AVG_RATIO' || computeMode == 'CUMULATIVE_PERCENTAGE')) {
                    return true;
                }
                return false;
            },

            getDimensionsForComputeAlong: chartDef => {
                if (chartDef.type === CHART_TYPES.PIVOT_TABLE) {
                    const xDimension = ChartDimension.getXDimension(chartDef);
                    const yDimension = ChartDimension.getYDimension(chartDef);
                    if (!xDimension || !yDimension) {
                        return [];
                    }
                    // PIVOT order of insertion of the axes is defined as follows: [...xDimension, ...yDimension]
                    return [[0, xDimension.column], [1, yDimension.column]];
                }
                if (chartDef.type === CHART_TYPES.SANKEY || chartDef.type === CHART_TYPES.TREEMAP) {
                    const yDimensions = ChartDimension.getYDimensions(chartDef);
                    if (yDimensions.length < 2) {
                        return [];
                    }
                    return [[0, yDimensions[0].column], [1, yDimensions[1].column]];
                }
                const genericDimension = ChartDimension.getGenericDimension(chartDef);
                if (!genericDimension || !chartDef.genericDimension1 || !chartDef.genericDimension1[0]) {
                    return [];
                }
                // default order of insertion of the axes is defined as follows: [...genericDimension0, ...genericDimension1]
                return [[0, genericDimension.column], [1, chartDef.genericDimension1[0].column]];
            },

            isEChart: (chartDef) => {
                const displayEchartsByDefault = svc.hasDefaultEChartsDisplay(chartDef.type) && chartDef.displayWithEChartsByDefault;
                const displayEchartsByFeatureFlag = !svc.hasDefaultEChartsDisplay(chartDef.type) && chartDef.displayWithECharts;

                return svc.hasEChartsDefinition(chartDef.type) && (displayEchartsByDefault || displayEchartsByFeatureFlag || !svc.hasD3Definition(chartDef.type));
            },

            isAxisNumericalAndFormattable: function(chartDef, axisId) {
                if (chartDef && chartDef.$chartStoreId && axisId) {
                    const chartStore = ChartStoreFactory.get(chartDef.$chartStoreId);
                    const axisType = chartStore.getAxisType(axisId);

                    if (axisType === CHART_AXIS_TYPES.MEASURE) {
                        return true;
                    } else {
                        const dimensionType = chartStore.getAxisDimensionOrUADimensionType(axisId);
                        const numParamsMode = chartStore.getAxisDimensionOrUADimensionNumParamsMode(axisId);
                        return dimensionType === 'NUMERICAL' && numParamsMode !== 'TREAT_AS_ALPHANUM';
                    }
                }

                return false;
            },

            getAxisTicksModes: () => {
                return axisTicksModes;
            },

            getAxisTicksDisabledMessage: (type, zoomControlInstanceId) => {
                if (svc.isScatterZoomed(type, zoomControlInstanceId)) {
                    return translate(
                        'CHARTS.SETTINGS.FORMAT.AXIS_TICKS.SCATTER_ZOOMED_TOOLTIP',
                        'Option is disabled as it is not compatible with the current zoom state of the chart. To enable it, reset the zoom state.'
                    );
                } else {
                    return translate(
                        'CHARTS.SETTINGS.FORMAT.AXIS_TICKS.DEFAULT_DISABLED_TOOLTIP',
                        'Option is disabled as "{{generate_one_tick_per_bin}}" is selected in the dimension',
                        { generate_one_tick_per_bin: translate('CHARTS.SHARED.GENERATE_ONE_TICK_PER_BIN', 'Generate one tick per bin') }
                    );
                }
            },

            canSetTicksConfig: (axisFormatting, chartDef, axisId) => {
                return axisFormatting && axisFormatting.displayAxis && svc.isAxisNumericalAndFormattable(chartDef, axisId);
            },

            hasOneTickPerBinSelected: (chartStoreId, axisId) => {
                if (chartStoreId) {
                    const chartStore = ChartStoreFactory.get(chartStoreId);
                    const axisSpec = chartStore.getAxisSpec(axisId);
                    return ChartDimension.hasOneTickPerBin(axisSpec && axisSpec.dimension) || false;
                }
            },

            supportSorting: (dimension, chartDef, isSecondDimension) => {
                if (!dimension) {
                    return false;
                }

                if (ChartDimension.isTrueNumerical(dimension)) {
                    return false;
                }

                if (svc.chartSupportOneTickPerBin(chartDef)) {
                    return !ChartDimension.supportOneTickPerBin(dimension) || ChartDimension.isAlphanumLike(dimension) || ChartDimension.isSortableDateDimension(dimension, isSecondDimension);
                }

                return true;
            },

            shouldRemoveOverlappingTicks: (axisFormatting, chartDef, axisId) => {
                const ticksConfig = axisFormatting.ticksConfig;
                return ticksConfig.number !== undefined && !svc.hasOneTickPerBinSelected(chartDef && chartDef.$chartStoreId, axisId) && svc.canSetTicksConfig(axisFormatting, chartDef, axisId);
            },

            canRealiasMeasuresAndDimensions: chartDef => {
                return chartDef.type !== 'boxplots';
            },

            canHaveBinningOptions: chartDef => {
                return chartDef.variant !== CHART_VARIANTS.binnedXYHexagon;
            },

            hasDefaultEChartsDisplay: chartType => {
                return chartType != null && [
                    CHART_TYPES.PIE
                ].includes(chartType);
            },

            hasEChartsDefinition: chartType => {
                return chartType != null && [
                    CHART_TYPES.GAUGE,
                    CHART_TYPES.PIE,
                    CHART_TYPES.RADAR,
                    CHART_TYPES.SANKEY,
                    CHART_TYPES.TREEMAP
                ].includes(chartType);
            },

            hasD3Definition: chartType => {
                return chartType != null && [
                    CHART_TYPES.GROUPED_COLUMNS,
                    CHART_TYPES.STACKED_BARS,
                    CHART_TYPES.STACKED_COLUMNS,
                    CHART_TYPES.MULTI_COLUMNS_LINES,
                    CHART_TYPES.LINES,
                    CHART_TYPES.STACKED_AREA,
                    CHART_TYPES.PIVOT_TABLE,
                    CHART_TYPES.SCATTER,
                    CHART_TYPES.GROUPED_XY,
                    CHART_TYPES.BINNED_XY,
                    CHART_TYPES.DENSITY_2D,
                    CHART_TYPES.SCATTER_MAP,
                    CHART_TYPES.DENSITY_HEAT_MAP,
                    CHART_TYPES.GEOMETRY_MAP,
                    CHART_TYPES.ADMINISTRATIVE_MAP,
                    CHART_TYPES.GRID_MAP,
                    CHART_TYPES.BOXPLOTS,
                    CHART_TYPES.PIE,
                    CHART_TYPES.LIFT,
                    CHART_TYPES.WEBAPP
                ].includes(chartType);

            },

            hasLegacyBadge: (chartType, isEChartsToggled) => {
                return svc.hasEChartsDefinition(chartType) && svc.hasD3Definition(chartType) && svc.hasDefaultEChartsDisplay(chartType) && isEChartsToggled === false;
            },

            hasBetaBadge: (chartType, isEChartsToggled) => {
                return svc.hasEChartsDefinition(chartType) && svc.hasD3Definition(chartType) && !svc.hasDefaultEChartsDisplay(chartType) && !!isEChartsToggled;
            },

            canDisplayLegend: (chartType) => {
                return !isChartTypeInList(chartType, [
                    CHART_TYPES.PIVOT_TABLE,
                    CHART_TYPES.LIFT,
                    CHART_TYPES.DENSITY_2D,
                    CHART_TYPES.KPI,
                    CHART_TYPES.GAUGE,
                    CHART_TYPES.SANKEY,
                    CHART_TYPES.DENSITY_HEAT_MAP
                ]);
            },

            isStackedLegend: (chartDef) => {
                return chartDef.type === CHART_TYPES.GEOMETRY_MAP && chartDef.geoLayers.length > 2;
            },

            shouldComputeLegend: (chartType) => {
                // compute sankey legend to have color menu, and compute pivot table legend for excel export
                return svc.canDisplayLegend(chartType) || isChartTypeInList(chartType, [CHART_TYPES.PIVOT_TABLE, CHART_TYPES.SANKEY]);
            },

            canConfigureForceLastPositionOthers: (chartType, dimension) => {
                return isChartTypeInList(chartType, [
                    CHART_TYPES.DONUT,
                    CHART_TYPES.PIE
                ]) && ChartDimension.isAlphanumLike(dimension) && dimension.generateOthersCategory;
            },

            canDisplayAxes: chartType => {
                return !isChartTypeInList(chartType, [
                    CHART_TYPES.PIE,
                    CHART_TYPES.PIVOT_TABLE,
                    CHART_TYPES.KPI,
                    CHART_TYPES.GAUGE,
                    CHART_TYPES.TREEMAP,
                    CHART_TYPES.SANKEY,
                    CHART_TYPES.RADAR,
                    CHART_TYPES.SCATTER_MAP,
                    CHART_TYPES.DENSITY_HEAT_MAP,
                    CHART_TYPES.GEOMETRY_MAP,
                    CHART_TYPES.ADMINISTRATIVE_MAP,
                    CHART_TYPES.GRID_MAP
                ]);
            },

            canHideAxes: chartType => {
                return svc.canDisplayAxes(chartType) && !isChartTypeInList(chartType, [
                    CHART_TYPES.DENSITY_2D,
                    CHART_TYPES.BOXPLOTS
                ]);
            },

            canUseSQL: chartDef => {
                return !chartDef.hexbin;
            },

            canShowTooltips: chartType => {
                return !isChartTypeInList(chartType, [CHART_TYPES.GAUGE, CHART_TYPES.KPI]);
            },

            canPinTooltip: (chartType) => {
                return chartType !== CHART_TYPES.SCATTER && chartType !== CHART_TYPES.SCATTER_MULTIPLE_PAIRS;
            },

            canFilterInTooltip: (chartDef) => {
                return ![CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS].includes(chartDef.type);
            },

            canIgnoreEmptyBinsForColorDimension: (chartDef) => {
                return [CHART_TYPES.GROUPED_COLUMNS, CHART_TYPES.MULTI_COLUMNS_LINES].includes(chartDef.type);
            },

            getRecordsMetadata: (chartDef, sampleMetadata, beforeFilterRecords, afterFilterRecords) => {
                if ([CHART_TYPES.SCATTER, CHART_TYPES.SCATTER_MULTIPLE_PAIRS].includes(chartDef.type)) {
                    const datasetRecordCount = sampleMetadata ? sampleMetadata.datasetRecordCount : beforeFilterRecords;
                    const sampleRecordCount = sampleMetadata ? sampleMetadata.sampleRecordCount : beforeFilterRecords;
                    const scatterOptions = chartDef.type === CHART_TYPES.SCATTER ? chartDef.scatterOptions : chartDef.scatterMPOptions;
	                const numberOfRecords = scatterOptions.numberOfRecords || 1000000;
                    const numberOfRecordsPerPair = Math.floor(numberOfRecords / (chartDef.uaDimensionPair.length || 1));

                    // By default, we compare it to the full dataset
                    let hasLessRecordsThanSampling = numberOfRecordsPerPair < datasetRecordCount;

                    // If we're using a sampling, datasetRecordCount equals -1, so we compare it to the sample
                    if (datasetRecordCount < 0) {
                        hasLessRecordsThanSampling = numberOfRecordsPerPair < sampleRecordCount;
                    }

                    if (afterFilterRecords < numberOfRecordsPerPair) {
                        // If there are less points after filtering than the authorized number of points per pair, we don't display the warning
                        hasLessRecordsThanSampling = false;
                    }


                    return {
                        hasLessRecordsThanSampling,
                        message: translate(
                            'CHARTS.HEADER.BADGE.TRUNCATED.TOOLTIP',
                            `${NumberFormatter.longSmartNumberFilter()(numberOfRecords)} points limit reached. You can modify the limit in the "Misc" section of your chart.`,
                            { maxNbPoints: NumberFormatter.longSmartNumberFilter()(numberOfRecords) }
                        )
                    };
                } else {
                    return null;
                }
            },

            shouldHideFormatTab: (chartType, readOnly) => {
                return readOnly;
            }
        };

        return svc;
    }
})();
