(function() {
    'use strict';

    const app = angular.module("dataiku.fm.alerting", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs"]);

    app.controller("AlertingController", function($scope, FMAPI, LoggerProvider) {
        const DEFAULT_GRACE_PERIOD_IN_MIN = 5;
        const DEFAULT_RESEND_NOTIFICATION_PERIOD_IN_MIN = 60;
        const DEFAULT_THRESHOLD = 80; // aligned on the value above which the disk consumption progress bar becomes red in the instances dashboard

        const Logger = LoggerProvider.getLogger('AlertingController');

        const convertSecondsToMinutes = function(minutes) {
            return Math.ceil(minutes / 60);
        }

        const convertMinutesToSeconds = function(seconds) {
            return seconds * 60;
        }

        const updateStateWithMessagingChannel = function(state, messagingChannel) {
            if (!messagingChannel) {
                return;
            }

            const configuration = JSON.parse(messagingChannel.configuration);

            switch (messagingChannel.configurationVersion) {
            case 1:
                state.smtpConfiguration.messagingChannelId = messagingChannel.id;
                state.smtpConfiguration.host = configuration.host;
                state.smtpConfiguration.port = configuration.port;
                state.smtpConfiguration.useSSL = configuration.useSSL;
                state.smtpConfiguration.useTLS = configuration.useTLS;
                state.smtpConfiguration.login = configuration.login;
                state.smtpConfiguration.password = configuration.password;
                break;
            default:
                throw `Unknown messaging channel configuration version: ${messagingChannel.configurationVersion}`;
            }
        }

        const updateStateWithCheckReporter = function(state, checkReporter) {
            if (!checkReporter) {
                return;
            }

            const configuration = JSON.parse(checkReporter.configuration);

            switch (checkReporter.configurationVersion) {
            case 1:
                state.smtpConfiguration.checkReporterId = checkReporter.id;
                state.smtpConfiguration.sender = configuration.sender;
                state.smtpConfiguration.recipients = configuration.recipients.length > 0
                    ? configuration.recipients.split(",").map(value => ({ value }))
                    : [];
                break;
            default:
                throw `Unknown check reporter configuration version: ${checkReporter.configurationVersion}`;
            }
        }

        const updateStateWithCheckDefinitions = function(state, checkDefinitions) {
            const invalidStatusCheckDefinition = checkDefinitions.find((checkDefinition) => checkDefinition.type === "LOGICAL_INSTANCE_INVALID_STATUS");
            if (invalidStatusCheckDefinition) {
                switch (invalidStatusCheckDefinition.configurationVersion) {
                case 1:
                    state.checks.invalidStatus = {
                        id: invalidStatusCheckDefinition.id,
                        enabled: true,
                        gracePeriodInMin: convertSecondsToMinutes(invalidStatusCheckDefinition.gracePeriodInSec),
                        resendNotificationPeriodInMin: convertSecondsToMinutes(invalidStatusCheckDefinition.resendNotificationPeriodInSec),
                    };
                    break;
                default:
                    throw `Unknown invalid status check definition configuration version: ${invalidStatusCheckDefinition.configurationVersion}`;
                }
            } else {
                state.checks.invalidStatus = {
                    enabled: false,
                    gracePeriodInMin: DEFAULT_GRACE_PERIOD_IN_MIN,
                    resendNotificationPeriodInMin: DEFAULT_RESEND_NOTIFICATION_PERIOD_IN_MIN,
                };
            }

            const dataDiskConsumptionCheckDefinition = checkDefinitions.find((checkDefinition) => checkDefinition.type === "LOGICAL_INSTANCE_DATA_DISK_CONSUMPTION");
            if (dataDiskConsumptionCheckDefinition) {
                switch (dataDiskConsumptionCheckDefinition.configurationVersion) {
                case 1:
                    state.checks.dataDiskConsumption = {
                        id: dataDiskConsumptionCheckDefinition.id,
                        enabled: true,
                        gracePeriodInMin: convertSecondsToMinutes(dataDiskConsumptionCheckDefinition.gracePeriodInSec),
                        resendNotificationPeriodInMin: convertSecondsToMinutes(dataDiskConsumptionCheckDefinition.resendNotificationPeriodInSec),
                        threshold: JSON.parse(dataDiskConsumptionCheckDefinition.configuration).threshold,
                    };
                    break;
                default:
                    throw `Unknown data disk consumption check definition configuration version: ${dataDiskConsumptionCheckDefinition.configurationVersion}`;
                }
            } else {
                state.checks.dataDiskConsumption = {
                    enabled: false,
                    gracePeriodInMin: DEFAULT_GRACE_PERIOD_IN_MIN,
                    resendNotificationPeriodInMin: DEFAULT_RESEND_NOTIFICATION_PERIOD_IN_MIN,
                    threshold: DEFAULT_THRESHOLD,
                };
            }

            const osDiskConsumptionCheckDefinition = checkDefinitions.find((checkDefinition) => checkDefinition.type === "LOGICAL_INSTANCE_OS_DISK_CONSUMPTION");
            if (osDiskConsumptionCheckDefinition) {
                switch (osDiskConsumptionCheckDefinition.configurationVersion) {
                case 1:
                    state.checks.osDiskConsumption = {
                        id: osDiskConsumptionCheckDefinition.id,
                        enabled: true,
                        gracePeriodInMin: convertSecondsToMinutes(osDiskConsumptionCheckDefinition.gracePeriodInSec),
                        resendNotificationPeriodInMin: convertSecondsToMinutes(osDiskConsumptionCheckDefinition.resendNotificationPeriodInSec),
                        threshold: JSON.parse(osDiskConsumptionCheckDefinition.configuration).threshold,
                    };
                    break;
                default:
                    throw `Unknown OS disk consumption check definition configuration version: ${osDiskConsumptionCheckDefinition.configurationVersion}`;
                }
            } else {
                state.checks.osDiskConsumption = {
                    enabled: false,
                    gracePeriodInMin: DEFAULT_GRACE_PERIOD_IN_MIN,
                    resendNotificationPeriodInMin: DEFAULT_RESEND_NOTIFICATION_PERIOD_IN_MIN,
                    threshold: DEFAULT_THRESHOLD,
                };
            }
        }

        const convertStateToMessagingChannel = function(state) {
            const {
                host,
                port,
                login,
                password,
                useSSL,
                useTLS,
                messagingChannelId,
            } = state.smtpConfiguration;

            let configuration = { host, port, useSSL, useTLS };

            if (login) {
                configuration.login = login;
            }

            if (password) {
                configuration.password = password;
            }

            return {
                id: messagingChannelId,
                type: "SMTP",
                configuration: JSON.stringify(configuration),
                configurationVersion: 1,
            };
        }

        const convertStateToCheckReporter = function(state) {
            const { messagingChannelId, sender, recipients, checkReporterId } = state.smtpConfiguration;

            return {
                id: checkReporterId,
                messagingChannelId,
                messagingChannelType: "SMTP",
                configuration: JSON.stringify({
                    sender,
                    recipients: recipients.map(it => it.value).join(","),
                }),
                configurationVersion: 1,
            };
        }

        const convertStateToCheckDefinitions = function(state) {
            const checks = state.checks;
            const checkReporterId = state.smtpConfiguration.checkReporterId;
            let checkDefinitions = [];

            // Convert invalid status settings to a check definition
            if (checks.invalidStatus.enabled) {
                checkDefinitions.push({
                    id: checks.invalidStatus.id,
                    type: "LOGICAL_INSTANCE_INVALID_STATUS",
                    configuration: "{}",
                    configurationVersion: 1,
                    checkReporterIds: [checkReporterId],
                    gracePeriodInSec: convertMinutesToSeconds(checks.invalidStatus.gracePeriodInMin),
                    resendNotificationPeriodInSec: convertMinutesToSeconds(checks.invalidStatus.resendNotificationPeriodInMin),
                });
            }

            // Convert data disk consumption settings to a check definition
            if (checks.dataDiskConsumption.enabled) {
                checkDefinitions.push({
                    id: checks.dataDiskConsumption.id,
                    type: "LOGICAL_INSTANCE_DATA_DISK_CONSUMPTION",
                    configuration: JSON.stringify({ threshold: checks.dataDiskConsumption.threshold }),
                    configurationVersion: 1,
                    checkReporterIds: [checkReporterId],
                    gracePeriodInSec: convertMinutesToSeconds(checks.dataDiskConsumption.gracePeriodInMin),
                    resendNotificationPeriodInSec: convertMinutesToSeconds(checks.dataDiskConsumption.resendNotificationPeriodInMin),
                });
            }

            // Convert OS disk consumption settings to a check definition
            if (checks.osDiskConsumption.enabled) {
                checkDefinitions.push({
                    id: checks.osDiskConsumption.id,
                    type: "LOGICAL_INSTANCE_OS_DISK_CONSUMPTION",
                    configuration: JSON.stringify({ threshold: checks.osDiskConsumption.threshold }),
                    configurationVersion: 1,
                    checkReporterIds: [checkReporterId],
                    gracePeriodInSec: convertMinutesToSeconds(checks.osDiskConsumption.gracePeriodInMin),
                    resendNotificationPeriodInSec: convertMinutesToSeconds(checks.osDiskConsumption.resendNotificationPeriodInMin),
                });
            }

            return checkDefinitions;
        }

        const saveAlerting = function(state) {
            let messagingChannel = convertStateToMessagingChannel(state);
            let checkReporter = convertStateToCheckReporter(state);

            // Update messaging channel, check reporter, and check definitions
            const alerting = {
                messagingChannels: [messagingChannel],
                checkReporters: [checkReporter],
                checkDefinitions: convertStateToCheckDefinitions(state),
            }
            return FMAPI.alerting.updateAlerting(alerting)
                .then(alertingResponse => {
                    try {
                        updateStateWithMessagingChannel(state, alertingResponse.data.messagingChannels[0]);
                        updateStateWithCheckReporter(state, alertingResponse.data.checkReporters[0]);
                        updateStateWithCheckDefinitions(state, alertingResponse.data.checkDefinitions);
                    } catch (err) {
                        Logger.error(err);
                    }
                });
        }

        const afterSave = function(scope) {
            scope.alerting.$lastSaved = dkuDeepCopy(scope.alerting, filterDollarKey);
            scope.alerting.$dirty = false;
            resetErrorInScope(scope);
        }

        $scope.alerting = {
            smtpConfiguration: {
                recipients: [],
            },
            checks: {
                invalidStatus: {},
                dataDiskConsumption: {},
                osDiskConsumption: {},
            },
        };

        // Retrieve current settings
        FMAPI.alerting.getAlerting()
            .success(alertingResponse => {
                try {
                    updateStateWithMessagingChannel($scope.alerting, alertingResponse.messagingChannels[0]);
                    updateStateWithCheckReporter($scope.alerting, alertingResponse.checkReporters[0]);
                    updateStateWithCheckDefinitions($scope.alerting, alertingResponse.checkDefinitions);
                    afterSave($scope);
                } catch (err) {
                    Logger.error(err);
                }
            })
            .error(setErrorInScope.bind($scope));

        $scope.save = function() {
            if ($scope.alerting.$dirty) {
                // Save settings
                saveAlerting($scope.alerting)
                    .then(() => afterSave($scope))
                    .catch(setErrorInScope.bind($scope));
            }
        }

        $scope.$watch("alerting", function() {
            $scope.alerting.$dirty = !angular.equals($scope.alerting, $scope.alerting.$lastSaved);
        }, true);
    });
}());
