(function() {
    'use strict';

    const app = angular.module("dataiku.fm.instancesettingstemplates", ["dataiku.services", "dataiku.filters", "dataiku.fm.dialogs"]);

    app.controller("InstanceTemplateListController", function($scope, $state, FMAPI, Dialogs, CreateModalFromTemplate, WT1) {
        $scope.refreshInstanceTemplateList = function() {
            FMAPI.instancesettingstemplates.list().success(function(data) {
                resetErrorInScope($scope);
                $scope.instancetemplates = data.filter(ist => !ist.id.startsWith("ist-snap"));
            }).error(setErrorInScope.bind($scope));
        };

        $scope.newInstanceTemplate = function() {
            $state.go("instancetemplates.edit", {instanceSettingsTemplateId: "new"});
        };
        $scope.copyInstanceTemplate = function(template) {
            $state.go("instancetemplates.copy", {instanceSettingsTemplateId: "copy", copyTarget: template.id});
        }

        $scope.deleteInstanceTemplate = function(template) {
            if (template.nbInstances === 0) {
                // Display an information message for templates that are not used, thus not greyed.
                Dialogs.confirmInfoMessages($scope, 'Confirm instance template deletion', null, 'Are you sure you want to delete the Instance template "' + sanitize(template.label) + '"?', false).then(function() {
                    $scope.deleteInstanceNoConfirmation(template);
                });
            } else {
                // For templates where a grey icon is displayed, no confirmation message as deletion will fail.
                // This way users will get the reason why this template cannot be deleted.
                $scope.deleteInstanceNoConfirmation(template);
            }
        };

        $scope.deleteInstanceNoConfirmation = function(template) {
            WT1.event("fm-instance-template-delete", {});
            FMAPI.instancesettingstemplates.delete(template.id).success(function(commandResult) {
                resetErrorInScope($scope);
                if (commandResult.success) {
                    $scope.refreshInstanceTemplateList();
                } else {
                    CreateModalFromTemplate("/templates/dialogs/info-messages-dialog.html", $scope, null, function(newScope) {
                        newScope.modalTitle = "Cannot delete Instance template";
                        newScope.data = commandResult.statusMessages;
                    }, true, 'static', true);
                }
            }).error(setErrorInScope.bind($scope));
        };

        $scope.refreshInstanceTemplateList();
    });

    app.controller("InstanceTemplateEditController", function($scope, $stateParams, $state, FMAPI, Dialogs, CreateModalFromTemplate, ActivityIndicator, WT1) {
        $scope.basicCodeMirrorSetting = function(lineWrapping, mode="text/plain") {
            return {
                mode: mode,
                lineNumbers: true,
                lineWrapping: lineWrapping,
                foldGutter: true,
                gutters: ["CodeMirror-linenumbers", "CodeMirror-foldgutter"],
                autoCloseBrackets: false,
                autoCloseTags: false,
                matchBrackets: false,
                matchTags: false,
                highlightSelectionMatches: true,
                styleSelectedText: true,
                styleActiveLine: false
            }
        };

        $scope.allActions = [
            { type: "INSTALL_SYSTEM_PACKAGES", enabled: true, params:{packages:''} },
            { type: "ADD_AUTHORIZED_KEY", enabled: true, params:{sshKey:''} },
            { type: "SETUP_ADVANCED_SECURITY", enabled: true, params:{basic_headers: true, hsts: false} },
            { type: "INSTALL_JDBC_DRIVER", enabled: true, params:{url:'',} },
            { type: "RUN_ANSIBLE_TASKS", enabled: true, params:{stage:'after_dss_startup',ansibleTasks:'---\n'} },
            { type: "SETUP_K8S_AND_SPARK", enabled: true, params:{} },
            { type: "ADD_ENV_VARIABLES", enabled: true, params:{variables:[]} },
            { type: "ADD_PROPERTIES", enabled: true, params:{properties:[]} },
            { type: "ADD_SSH_KEY", enabled: true, params:{storageMode:'INLINE_ENCRYPTED',knownHosts:[], defaultKey: true, keyType: 'RSA'} },
            { type: "SETUP_PROXY", enabled: true, params:{proxy_host: '', proxy_port: '', proxy_username: '', proxy_password: '', http_proxy: '', https_proxy: '', no_proxy: ''} },
            { type: "ADD_CA_TO_TRUSTSTORE", enabled: true, params:{file_name:''} },
            { type: "INSTALL_CODE_ENV_WITH_PRESET", enabled: true, params:{allow_update:true, use_gpu:false} },
            { type: "INSTALL_DEPRECATED_PYTHON", enabled: true, params:{install_py36: false, install_py37: false, install_py38: false} }
        ];

        $scope.formatActionLabel = function(actionType) {
            switch (actionType) {
                case "INSTALL_SYSTEM_PACKAGES":
                    return "Install system packages";
                case "ADD_AUTHORIZED_KEY":
                    return "Add authorized SSH Key";
                case "SETUP_ADVANCED_SECURITY":
                    return "Setup advanced security";
                case "INSTALL_JDBC_DRIVER":
                    return "Install a JDBC driver";
                case "RUN_ANSIBLE_TASKS":
                    return "Run Ansible tasks";
                case "ADD_ENV_VARIABLES":
                    return "Add environment variables";
                case "ADD_PROPERTIES":
                    return "Add properties";
                case "ADD_SSH_KEY":
                    return "Add SSH key";
                case "SETUP_PROXY":
                    return "Setup proxy";
                case "ADD_CA_TO_TRUSTSTORE":
                    return "Add Certificate Authority to DSS truststore";
                case "SETUP_K8S_AND_SPARK":
                    return "Setup Kubernetes and Spark-on-Kubernetes";
                case "INSTALL_CODE_ENV_WITH_PRESET":
                    return "Install code env with VisualML preset";
                case "INSTALL_DEPRECATED_PYTHON":
                    return "Install deprecated Python versions (DSS 14 or later)";
                default:
                    return actionType;
            }
        };

        $scope.addAction = function(actionTemplate) {
            // eslint-disable-next-line no-undef
            let action = dkuDeepCopy(actionTemplate, filterDollarKey);
            if (!$scope.editedInstanceTemplate.setupActions) {
                $scope.editedInstanceTemplate.setupActions = [];
            }
            action.$folded = false;
            $scope.editedInstanceTemplate.setupActions.push(action);
            $scope.availableActions = $scope.listAvailableActions();
        };

        $scope.foldAction = function(index) {
            if ($scope.editedInstanceTemplate.setupActions) {
                $scope.editedInstanceTemplate.setupActions[index].$folded = !$scope.editedInstanceTemplate.setupActions[index].$folded;
            }
        };

        $scope.removeAction = function(index) {
            if ($scope.editedInstanceTemplate.setupActions) {
                $scope.editedInstanceTemplate.setupActions.splice(index, 1);
                $scope.availableActions = $scope.listAvailableActions();
            }
        };

        $scope.hasAction = function(actionName) {
            if (!$scope.editedInstanceTemplate || !$scope.editedInstanceTemplate.setupActions) {
                return false;
            }
            return $scope.editedInstanceTemplate.setupActions.some(e => e.type === actionName);
        };

        $scope.listAvailableActions = function () {
            let allActions = $scope.allActions;

            function enrichAction(action) {
                return {
                    label: $scope.formatActionLabel(action.type),
                    data: action
                };
            };

            if (!$scope.editedInstanceTemplate || !$scope.editedInstanceTemplate.setupActions) {
                return allActions.map(enrichAction);
            }

            let availableActions = allActions.filter(action =>
                !$scope.isUniqueAction(action.type) ||
                !$scope.editedInstanceTemplate.setupActions.some(setupAction => setupAction.type === action.type));

            return availableActions.map(enrichAction);
        };

        $scope.availableActions = $scope.listAvailableActions();

        $scope.isUniqueAction = function (actionType) {
            switch (actionType) {
                case "SETUP_ADVANCED_SECURITY":
                case "ADD_ENV_VARIABLES":
                case "ADD_PROPERTIES":
                case "SETUP_PROXY":
                case "SETUP_K8S_AND_SPARK":
                case "INSTALL_CODE_ENV_WITH_PRESET":
                case "INSTALL_DEPRECATED_PYTHON":
                    return true;
                default:
                    return false;
            }
        };

        $scope.hasMultipleUniqueActions = function() {
            if (!$scope.editedInstanceTemplate || !$scope.editedInstanceTemplate.setupActions) {
                return false;
            }

            let uniqueActions = $scope.editedInstanceTemplate.setupActions.filter(action => $scope.isUniqueAction(action.type));

            function countUniqueActionsByType(uniqueActions) {
                let uniqueActionsCountByType = new Map();

                uniqueActions.forEach(action => {
                    let isUniqueAction = $scope.isUniqueAction(action.type);

                    if (action && action.type && $scope.isUniqueAction(action.type)) {
                        if (uniqueActionsCountByType.has(action.type)) {
                            let currentCount = uniqueActionsCountByType.get(action.type);
                            uniqueActionsCountByType.set(action.type, currentCount + 1);
                        }
                        else {
                            uniqueActionsCountByType.set(action.type, 1);
                        }
                    }
                });

                return uniqueActionsCountByType;
            };

            let uniqueActionsCountByType = countUniqueActionsByType(uniqueActions);
            let multiUniqueActions = [...uniqueActionsCountByType]
                .filter(([actionType, count]) => count > 1)
                .map(([actionType, count]) => actionType);

            // Once the unique actions appearing multiple times have been identified,
            // We prepare the list for display.
            $scope.multiUniqueSetupActions = multiUniqueActions
                .map(actionType => "\"" + $scope.formatActionLabel(actionType) + "\"")
                .join(", ");

            return multiUniqueActions.length > 0;
        };

        $scope.openLicenseEditor = function() {
            $scope.showLicenseEditor = true;
        };
        $scope.closeLicenseEditor = function() {
            $scope.showLicenseEditor = false;
            $scope.license = $scope.parseLicense($scope.editedInstanceTemplate.license);
        };

        $scope.parseLicense = function(licenseText) {
            if (!licenseText) {
                return {empty: true};
            }
            try {
                const license = JSON.parse(licenseText);
                if (!license || !license.content.licenseId) {
                    return {empty: false, error: true};
                }
                let expirationDate = undefined;
                if (license.content.expiresOn) {
                    const year = parseInt(license.content.expiresOn.substring(0, 4), 10);
                    const month = parseInt(license.content.expiresOn.substring(4, 6), 10);
                    const day = parseInt(license.content.expiresOn.substring(6, 8), 10);
                    expirationDate = new Date(year, month - 1, day).getTime();
                }
                return {empty: false, valid: true, expirationDate: expirationDate};
            } catch (e) {
                return {empty: false, error: true};
            }
        };

        $scope.initializeInstanceTemplate = function(ist) {
            $scope.instanceTemplate = ist;

            // Generate the original (i.e. non-modified) template in an editable format:
            //   - For INSTALL_SYSTEM_PACKAGES actions, turn the array into a string separated by new lines
            //   - For INSTALL_JDBC_DRIVER, turn pathsInArchive and headers in multiline strings
            // eslint-disable-next-line no-undef
            $scope.originalInstanceTemplate = dkuDeepCopy(ist, filterDollarKey);
            if ($scope.originalInstanceTemplate.setupActions) {
                $scope.originalInstanceTemplate.setupActions.filter(action => action.type === 'INSTALL_SYSTEM_PACKAGES').forEach(action => {
                    action.params.packages = action.params.packages ? action.params.packages.join('\n') : "";
                });
                $scope.originalInstanceTemplate.setupActions.filter(action => action.type === 'INSTALL_JDBC_DRIVER').forEach(action => {
                    if (!action.params.url) {
                        action.params.url = "";
                    }
                    action.params.pathsInArchive = action.params.pathsInArchive ? action.params.pathsInArchive.join('\n') : "";
                    action.params.headers = action.params.headers ? Object.entries(action.params.headers).map(entry => entry[0] + ': ' + entry[1]).join('\n') : "";
                    if (!action.params.username) {
                        action.params.username = "";
                    }
                    if (!action.params.password) {
                        action.params.password = "";
                    }
                    if (!action.params.subpathInDatadir) {
                        action.params.subpathInDatadir = "";
                    }
                    if (!action.params.dbType) {
                        action.params.dbType = "";
                    }
                });
            }

            // Generate the working copy of the template that is being edited
            // eslint-disable-next-line no-undef
            $scope.editedInstanceTemplate = dkuDeepCopy($scope.originalInstanceTemplate, filterDollarKey);

            if ($scope.editedInstanceTemplate.setupActions) {
                $scope.editedInstanceTemplate.setupActions.forEach((a) => a.$folded = true);
            }

            $scope.license = $scope.parseLicense($scope.editedInstanceTemplate.license);
        };

        $scope.loadInstanceTemplate = function(instanceTemplateId, copyTarget) {
            if (instanceTemplateId === "copy") {
                FMAPI.instancesettingstemplates.get(copyTarget).success(function(instanceTemplate) {
                    resetErrorInScope($scope);
                    instanceTemplate.id = "";
                    instanceTemplate.label = "";
                    $scope.initializeInstanceTemplate(instanceTemplate);
                    $scope.availableActions = $scope.listAvailableActions();
                }).error(setErrorInScope.bind($scope));
                return;
            }
            if (instanceTemplateId === "new") {
                // Creation mode
                $scope.initializeInstanceTemplate({id: "", label: "", gcpBlockProjectWideKeys: true});
            } else {
                // Edition mode
                FMAPI.instancesettingstemplates.get(instanceTemplateId).success(function(instanceTemplate) {
                    resetErrorInScope($scope);
                    $scope.initializeInstanceTemplate(instanceTemplate);
                    $scope.availableActions = $scope.listAvailableActions();
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.saveInstanceTemplate = function() {
            // Duplicate the template and post-process it before sending to the backend
            let ist = JSON.parse(JSON.stringify($scope.editedInstanceTemplate));
            if (ist.setupActions) {
                ist.setupActions.filter(action => action.type === 'INSTALL_SYSTEM_PACKAGES').forEach(action => {
                    action.params.packages = action.params.packages.split('\n').map(line => line.trim()).filter(line => line.length > 0);
                });
                ist.setupActions.filter(action => action.type === 'INSTALL_JDBC_DRIVER').forEach(action => {
                    // It is required to force a default value here or they could be stored as strings in some situations
                    action.params.pathsInArchive = action.params.pathsInArchive ? action.params.pathsInArchive.split('\n').map(line => line.trim()).filter(line => line.length > 0) : [];
                    action.params.headers = action.params.headers ? Object.fromEntries(action.params.headers
                        .split('\n')
                        .map(line => line.trim())
                        .filter(line => 0 < line.indexOf(':') && line.indexOf(':') < (line.length - 1))
                        .map(line => [line.substr(0, line.indexOf(':')), line.substr(line.indexOf(':') + 1).trim()])
                    ) : {};
                });
            }

            if (ist.id) {
                // Save existing instance template
                FMAPI.instancesettingstemplates.save(ist).success(function(cmdResult) {
                    if (cmdResult.requiredAction !== "NO_ACTION_REQUIRED") {
                        CreateModalFromTemplate("/app/instance-templates/dialogs/instances-update-dialog.html", $scope, null, function(newScope) {
                            newScope.impactedInstances = cmdResult.impactedInstances;
                            newScope.requiredAction = cmdResult.requiredAction;
                        });
                    } else {
                        ActivityIndicator.success("Instance template updated");
                    }
                    $scope.initializeInstanceTemplate(cmdResult.ist);
                    $scope.instanceTemplateEditionState.dirty = false;
                }).error(setErrorInScope.bind($scope));
            } else {

                WT1.event("fm-instance-template-create", {});

                // Create new instance template
                FMAPI.instancesettingstemplates.create(ist).success(function(newTemplate) {
                    ActivityIndicator.success("Instance template created");
                    $state.go("instancetemplates.edit", {instanceSettingsTemplateId: newTemplate.id});
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.$watch("editedInstanceTemplate", function() {
            $scope.instanceTemplateEditionState.dirty = !angular.equals($scope.editedInstanceTemplate, $scope.originalInstanceTemplate);
        }, true);

        // Define scope variables
        $scope.instanceTemplateEditionState = {
            dirty: false
        };

        // Initialize screen
        $scope.loadInstanceTemplate($stateParams.instanceSettingsTemplateId, $stateParams.copyTarget);
    });
}());
