/* eslint-disable no-console */
/* global _wt1Q */
(function() {
'use strict';

const app = angular.module('dataiku.logger', []);


// This is an "instantiable" => it is NOT a singleton !
// DIContext contains some details related to the service who requested the logger instance.
app.instantiable('Logger', function(DIContext, LoggerProvider) {
    const fullname = getSimpleFullname(DIContext);
    return LoggerProvider.getLogger(fullname);
});


app.service('LoggerProvider',function($log, Notification, $injector) {
    const svc = this;

    function handleLog(type, namespace) {
        return function(...loggedObjects) {
            const timestamp = moment().utc().valueOf();
            const formattedDate = moment().format('HH:mm:ss.SSS');
            const formattedType = type.toUpperCase();

            // Print in console
            let prefix = '['+ formattedDate + ']';
            if(namespace) {
                prefix += ' [' + namespace + ']';
            }
            prefix += ' -';

            if (type == "debug" && loggedObjects && loggedObjects.length == 1) {
                $log.info("%c" + prefix + " " + loggedObjects[0], "color: #777");
            } else {
                const args = [prefix, ...loggedObjects];
                $log[type].apply($log, args);
            }

            if($injector.has('WebSocketService')) {
                if($injector.get("WebSocketService").isAvailable()) {
                    const stringifiedLoggedObjects = [];
                    // Websocket connection fails if the message is too big, so we truncate - #3821
                    // To work around the fact that might it be UTF8, we limit messages to 63K / 3
                    const MAX_LENGTH = 64000/3;
                    let remainingLen = MAX_LENGTH;

                    angular.forEach(loggedObjects,function(obj) {
                        let ret = '';
                        if(!obj) {
                            ret = ''+obj;
                        } else {
                            if(typeof obj == 'object') {
                                try {
                                    ret = JSON.stringify(obj);
                                } catch(e) {
                                    ret = ''+obj;
                                }
                            } else {
                                 ret = ''+obj;
                            }
                        }
                        if (ret.length > remainingLen) {
                            ret = "TRUNCATED: " + ret.substring(0, remainingLen);
                            console.warn("Truncated log message on Websocket (too long)"); /*@console*/ // NOSONAR: OK to use console.
                        }
                        remainingLen -= ret.length;
                        stringifiedLoggedObjects.push(ret)
                    });
                    try {
                        Notification.publishToBackend('log-event', {
                            messages: stringifiedLoggedObjects,
                            timestamp: timestamp,
                            type: formattedType,
                            namespace: namespace
                        });
                    } catch (e2) {
                        console.warn("Failed to send log event to backend", e2); /*@console*/ // NOSONAR: OK to use console.
                    }
                }
            }
        };
    };

    svc.getLogger = function(namespace) {
        return {
            log: handleLog("log", namespace),
            warn: handleLog("warn", namespace),
            debug: handleLog("debug", namespace),
            info: handleLog("info", namespace),
            error: handleLog("error", namespace)
        };
    };
});


/*
Instantialble assert:
    - throws an error if the condition is not met
    - prefixes the error message with the component
    - enriches the error with the component full name and adds a "js-assert" orign

This is useful in particular to easily track errors reported to WT1
*/
app.instantiable('Assert', function(DIContext, AssertProvider) {
    const namespace = getSimpleFullname(DIContext);
    return AssertProvider.getChecker(namespace);
});


app.service('AssertProvider',function() {
    const svc = this;

    function _fail(namespace, message) {
        const prefix = namespace ? `[${namespace}] ` : '';
        throw new Error('[Assert]' + prefix + message);
    }

    function fail(namespace) {
        return function(message) {
            _fail(namespace, message);
        };
    }

    function trueish(namespace) {
        return function(condition, message) {
            if (!condition) {
                _fail(namespace, message);
            }
        };
    }

    function inScope(namespace) {
        return function(scope, attribute) {
            if (!resolveValue(scope, attribute)) {
                _fail(namespace, attribute + ' is not in scope');
            }
        };
    }

    svc.getChecker = function(namespace) {
        return {
            trueish: trueish(namespace),
            inScope: inScope(namespace),
            fail: fail(namespace)
        };
    };
});


// Utils for prefixing based on the angular component
function getSimpleModuleName(DIContext) {
    let sname = DIContext.serviceName;
    if (sname) {
        sname = sname.replace("dataiku.", "d.")
            .replace(".services", ".s")
            .replace(".directive", ".dir")
            .replace(".controllers", ".ctrl")
            .replace(".recipes", ".r")
            .replace(".savedmodels", ".sm")
            .replace(".managedfolder", ".mf")
            ;
    }
    return sname;
}

function getSimpleObjectName(DIContext) {
    let oname = DIContext.objectName;
    if (oname) {
        oname = oname.replace("APIXHRService", "API")
            .replace("Controller", "Ctrl");
    }
    return oname;
}

function getSimpleFullname(DIContext) {
    const sname = getSimpleModuleName(DIContext);
    const oname = getSimpleObjectName(DIContext);
    const fullname = sname + (sname ? '.' : '') + oname;
    return fullname;
}


const EXPECTED_EXCEPTIONS=[
    "Possibly unhandled rejection: dismissed modal",
    'Possibly unhandled rejection: {"data":{"aborted":true,"reason":"Superseded by new call"},"status":-1}'
];

app.factory('$exceptionHandler', function(Logger, ErrorReporting) {
    return function(exception, cause) {
        /* Swallow "expected exceptions", such as the one we do not catch when dismissing a modal */
        if (EXPECTED_EXCEPTIONS.includes(exception)) {
            return;
        }
        /* A string was thrown, so at least, use it as a message */
        if (exception !== undefined && exception.message === undefined) {
            console.warn("Got weird exception", exception, printStackTrace()); /*@console*/ // NOSONAR: OK to use console.
            exception = {
                stack: "no stack - string thrown ?",
                message: exception
            }
        }

        /* Send to console and frontend log */
        if (exception === undefined) {
            Logger.error("Caught undefined exception", printStackTrace());
        } else {

            const typename = ({}).toString.call(exception);
            /* Firefox specific error */
            if (typename == "[object Exception]") {
                console.info("Changing Firefox exception", exception); /*@console*/ // NOSONAR: OK to use console.
                let newMessage = exception.message ? exception.message : "No message";
                newMessage += " - FF modified";
                if (exception.name) newMessage += " - Name=" + exception.name;
                if (exception.result) newMessage += " - result=" + exception.result;

                let newException = new Error(newMessage);
                newException.stack = exception.stack;
                exception = newException;
            }

            Logger.error("Caught exception: " + exception,
                "\nStack: ", exception.stack,
                '\nCaused by : ', cause,
                '\nMessage :', exception.message);
        }
        /* Send to WT1 */
        ErrorReporting.reportJSException(exception, cause);
    };
});


app.factory("ErrorReporting", function() {
    // Must not depend on rootScope else circular dependency

    function errorReportingEnabled() {
        return window.dkuAppConfig && window.dkuAppConfig.udrMode == 'DEFAULT' && window.dkuAppConfig.errorsReporting;
    }

    const svc = {
        // if you change this method, make sure it works with its usage in Angular (global-error-handler.ts)
        reportJSException: function(exception, cause) {
            if (window.devInstance) {
                console.info("Dev instance, not reporting JS error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            if (!errorReportingEnabled()) {
                console.info("Reporting is disabled, not reporting JS error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            console.info("Reporting JS exception", exception); /* @console */ // NOSONAR: OK to use console.

            try {
                const params = {
                    type: "js-error",
                    message: exception.message,
                    stack: exception.stack
                };
                _wt1Q.push(["trackEvent", params]);
            } catch (e) {
                console.info("WT1 failure", e); /*@console*/ // NOSONAR: OK to use console.
            }
        },
        // if you change this method, make sure it works with its usage in Angular (api-error.ts)
        reportBackendAPIError: function(apiError) {
            if (window.devInstance) {
                console.info("Dev instance, not reporting API error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            if (!errorReportingEnabled()) {
                console.info("Reporting is disabled, not reporting API error"); /*@console*/ // NOSONAR: OK to use console.
                return;
            }
            if (apiError.httpCode == 0) {
                console.info("HTTP status 0 (network error), not reporting it", apiError); /*@console */ // NOSONAR: OK to use console.
                return;
            }
            if (apiError.httpCode == 502) {
                console.info("HTTP Gateway Error, not reporting it", apiError); /*@console */ // NOSONAR: OK to use console.
                return;
            }

            /* Report to WT1 */
            try {
                const params = {
                    type: "api-error",
                    httpCode: apiError.httpCode,
                    errorType: apiError.errorType,
                    message: apiError.message,
                    stack: apiError.details
                }
                _wt1Q.push(["trackEvent", params]);
            } catch (e) {
                console.info("WT1 failure", e); /*@console*/ // NOSONAR: OK to use console.
            }
        },
        configure: function() {
            const appConfig = window.dkuAppConfig;
            if (!!appConfig.offlineFrontend) {
                return; // We're offline, can't report errors.
            }

            window.DKUErrorReporting = svc;
        }
    };
    return svc;
});

/*
WT1 is the service we use to send back usage statistics to Dataiku
*/
app.factory("WT1", function($rootScope, Logger) {
    let mode = "DEFAULT";
    let configured = false;
    let eventsBeforeConfiguration = [];

    window._wt1Q = [];

    // $stateParams keys to report in WT1, in the form of hashed objectId / objectType
    const dssObjectsToLog = {
        "analysisId": "analysis",
        "datasetName": "dataset",
        "recipeName": "recipe",
        "smId": "savedModel",
        "odbId":"odb",
        "dashboardId": "dashboard",
        "scenarioId": "scenario",
        "notebookId": "notebook",
        "modelComparisonId": "modelComparison",
        "mesId": "modelEvaluationStore",
        "webAppId": "webApp",
        "reportId": "report",
        "insightId": "insight",
        "serviceId": "apiService",
        "labelingTaskId": "labelingTask"
    };

    const svc = {
        configure: function() {
            const appConfig = $rootScope.appConfig;
            window.dkuUsageReportingUtils._configure(appConfig);

            if (window.devInstance) {
                mode = 'NO';
            } else if (appConfig.udrMode) {
                mode = appConfig.udrMode;
            }
            configured = true;
            if (mode != 'NO') {
                window.dkuUsageReportingUtils._loadWT1JS();
                /* Now that we are configured, replay the events that already happened */
                if (window.devInstance) {
                    console.debug("WT1 configured, replaying", eventsBeforeConfiguration.length, "events"); /*@console*/ // NOSONAR: OK to use console.
                }
                for (const eventBeforeConfiguration of eventsBeforeConfiguration) {
                    this.event(eventBeforeConfiguration.type, eventBeforeConfiguration.params);
                }
            }

            if (window.devInstance) {
                console.debug("WT1 dev mode startup complete", _wt1Q); /*@console*/ // NOSONAR: OK to use console.
            }

            // for non-angular stuff :/
            window.WT1SVC = this;
        },
        tryEvent: function(type, paramsGetter, errorMessage) {
            try {
                svc.event(type, paramsGetter());
            } catch(e) {
                Logger.error(typeof errorMessage === 'string' ? errorMessage : `Failed to report '${type}' event`, e);
            }
        },
        event: function(type, params) {
            if (angular.isUndefined(params)) {
                params = {};
            }

            if (!configured) {
                if (window.devInstance) {
                    console.debug("WT1: pre-configuration-event: " + type, params); /*@console*/ // NOSONAR: OK to use console.
                }
                // While it's not configured, we enqueue events, so that:
                // * the session and visitor params are set BEFORE we track the first state change event
                // * the proper privacy and minimal mode are taken into account
                eventsBeforeConfiguration.push({ "type": type, "params": params});
                return;
            }

            /* Add login, email and profile according to privacy rules */
            if ($rootScope && $rootScope.appConfig) {
                window.dkuUsageReportingUtils._addUserPropertiesToEventPayload($rootScope.appConfig, params);
            }

            /* Add and hash the project key, the objects id in the current state */
            if ($rootScope && $rootScope.$stateParams && $rootScope.$stateParams.projectKey) {
                params.projecth = md5($rootScope.$stateParams.projectKey);
                for (const objectKey in dssObjectsToLog) {
                    if (objectKey in $rootScope.$stateParams) {
                        const objectId = $rootScope.$stateParams[objectKey];
                        const objectType = dssObjectsToLog[objectKey];
                        params.objectType = objectType;
                        params[objectType + "h"] = md5($rootScope.$stateParams.projectKey + "." + objectId);
                        break;
                    }
                }
            }
            /* Add and hash the application id */
            if($rootScope && $rootScope.$stateParams && $rootScope.$stateParams.appId){
                params.objectType = 'App';
                params.applicationh = md5($rootScope.$stateParams.appId);
            }
            
            window.dkuUsageReportingUtils._trackEvent($rootScope.appConfig, mode, type, params);
        },
        setVisitorParam: function(key, value) {
            _wt1Q.push(["setVisitorParam", key, value]);
        },
        delVisitorParam: function(key) {
            _wt1Q.push(["delVisitorParam", key]);
        },
        setSessionParam: function(key, value) {
            _wt1Q.push(["setSessionParam", key, value]);
        },
        delSessionParam: function(key) {
            _wt1Q.push(["delSessionParam", key]);
        },
    };

    return svc;
});


app.directive("wt1ClickId", function(WT1) {
    return {
        restrict: 'A',
        link: function ($scope, element, attrs) {
            if (attrs.wt1ClickId && attrs.wt1ClickId !== ""){
                element.bind('click', function() {
                    WT1.event("clicked-item", {"item-id": attrs.wt1ClickId});
                });
            }
        }
    };
});
app.directive("wt1ClickEvent", function(WT1) {
    return {
        restrict: 'A',
        link: function ($scope, element, attrs) {
            element.bind('click', function() {
                WT1.event(attrs.wt1ClickEvent);
            });
        }
    };
});


app.factory("BackendReportsService", function($rootScope, $timeout, DataikuAPI, WT1, Logger, ErrorReporting) {
    $timeout(function() {
        if ($rootScope.appConfig && $rootScope.appConfig.loggedIn && $rootScope.appConfig.udrMode != 'NO' && !window.devInstance) {
            DataikuAPI.usage.popNextReport().success(function(data) {
                if (data.reportPublicId) {
                    Logger.info("sending report", data.reportType);
                    /* Something to send */
                    window.WT1SVC.event("v3-report", {
                        "reportType": data.reportType,
                        "reportId": data.reportPublicId,
                        "reportData": JSON.stringify(data.reportData)
                    });
                } else {
                    Logger.info("No report available");
                }
            });
        } else {
            Logger.debug("Reports reporting disabled")
        }

        if ($rootScope.appConfig && $rootScope.appConfig.loggedIn && $rootScope.appConfig.admin &&
                 $rootScope.appConfig.udrMode == 'DEFAULT' && $rootScope.appConfig.errorsReporting) {
            DataikuAPI.usage.popReflectedEvents().success(function(data) {
                data.events.forEach(function(evt) {
                    Logger.info("Reflecting event", evt);
                    window.WT1SVC.event("reflected-event", {
                        "reflectedEventData": JSON.stringify(evt)
                    });
                });
            });
        } else {
            Logger.debug("Reflected events reporting disabled")
        }

    }, 10000);
    return {
        //TODO this is actually not a service, it has no functionality...
    };
});

})();



/// =================== Global ===================

/* eslint-disable no-unused-vars, no-redeclare */
window.WT1SVC = {
    event: function() {} // temporary before load
};

window.DKUErrorReporting = {
    reportBackendAPIError: function() {} // temporary before load
};

function setErrorInScope(data, status, headers, config, statusText, xhrStatus) {
    // Explicitly ignore JS errors by re-throwing
    // Allow using the function in catch() blocks like: .catch(setErrorInScope.bind($scope)) while having a clear stack trace
    // Whereas using .error(setErrorInScope.bind($scope)) expects `data` to be an http response
    if (data instanceof Error) {
        throw data;
    }

    if (status === undefined && headers === undefined) {
        status = data.status;
        headers = data.headers;
        statusText = statusText || data.statusText;
        xhrStatus = xhrStatus || data.xhrStatus;
        data = data.data;
    }
    /* Put in bound scope */
    this.fatalAPIError = getErrorDetails(data, status, headers, statusText);
    this.fatalAPIError.html = getErrorHTMLFromDetails(this.fatalAPIError);

    /* Report to WT1 */
    window.APIErrorLogger.error("API error", this.fatalAPIError);
    window.DKUErrorReporting.reportBackendAPIError(this.fatalAPIError);
}

/**
 * Variant of setErrorInScope that takes as input an already parsed error.
 * Designed to be able to set in scope errors coming from an Angular context (where the http service internally parses the error already)
 * @param errorDetails result from getErrorDetails or similar
 */
function setErrorDetailsInScope(errorDetails) {
    /* Put in bound scope */
    this.fatalAPIError = errorDetails;
    this.fatalAPIError.html = getErrorHTMLFromDetails(this.fatalAPIError);

    /* Report to WT1 */
    window.APIErrorLogger.error("API error", this.fatalAPIError);
    window.DKUErrorReporting.reportBackendAPIError(this.fatalAPIError);
}


function setErrorInScope2(payload) {
    setErrorInScope.call(this, JSON.parse(payload.response || '{}'), payload.status, h => payload.getResponseHeader(h));
}

function resetErrorInScope(scope) {
    if (scope.fatalAPIError) {
        scope.fatalAPIError.httpCode = null;
        scope.fatalAPIError.errorType = null;
    }
}

/* API error struct : {code, httpCode, message, details} */
function getErrorDetails(data, status, headers, statusText) {
    /* Network / HTTP error */
    if (data == null && status == -1) {
        return {
            httpCode: status,
            message: "Network error: " + (statusText === undefined ? "" : statusText),
            errorType: "XHRNetworkError"
        };
    }
    if (status == 413) {
        return {
            httpCode: status,
            message: data && data.message || 'No message',
            details: data && data.details || 'No details',
            errorType: "HTTPError413"
        };
    }

    if (data && data.$customMessage) {
        return {
            httpCode: status,
            code: 0,
            message: data.message || "Unknown error",
            details: data.details,
            errorType: data.errorType || "unknown"
        };
    }

    const ctype = headers("Content-type");
    if (ctype && ctype.startsWith("application/json") && data && data.errorType) {
        const apiError = data;
        apiError.httpCode = status;
        return apiError;
    } else {
        let errorType = "unknown";
        if (status == 502) {
            errorType = "Gateway error";
        }
        return {
            httpCode: status,
            code: 0,
            message: 'Unknown error',
            details: data && data.details || 'No details',
            errorType: errorType
        };
    }
}

function getErrorHTMLFromDetails(apiError) {
    let html = "<strong>Error " + apiError.code +": "+apiError.message +"</strong>";
    if (apiError.details) {
        html += "<pre>" + apiError.details +"</pre>";
    }
    return html;
}

function getErrorHTML(data, status, headers, statusText) {
    const apiError = getErrorDetails(data, status, headers, statusText);
    return getErrorHTMLFromDetails(apiError);
}

/* eslint-enable no-unused-vars, no-redeclare */

;
(function() {
    'use strict';
    // Module declaration
    angular.module('dataiku.constants', []);
})();
;
(function() {
'use strict';

const app = angular.module('dataiku.services', [ 'dataiku.constants' ]);

// Mapping DOMRect to Object for better usability
function getBoundingClientRect(element) {
    const rect = element.getBoundingClientRect();
    return {
        top: rect.top,
        right: rect.right,
        bottom: rect.bottom,
        left: rect.left,
        width: rect.width,
        height: rect.height,
        x: rect.x,
        y: rect.y
    };
}

app.factory("DKUConstants", function($rootScope) {
    const cst = {
        ARCHIVED_PROJECT_STATUS: "Archived",
        design: {
            alertClasses: {
                SUCCESS: 'alert-success',
                FATAL: 'alert-danger',
                ERROR: 'alert-danger',
                WARNING: 'alert-warning',
                INFO: 'alert-info'
            }
        }
    };
    $rootScope.DKUConstants = cst;

    return cst;
});


app.service('AppConfig', function (Logger) {
    let config;
    this.set = function(cfg) {
        Logger.info('Set appConfig')
        config = cfg;
    };
    this.get = function() {
        return config;
    }
});

app.service('FeatureFlagsService', function($rootScope) {
    this.featureFlagEnabled = function(flagName) {
        return $rootScope.appConfig && $rootScope.appConfig.featureFlags.includes(flagName);
    };

    // put in rootScope for easy use in templates
    $rootScope.featureFlagEnabled = this.featureFlagEnabled;
});

app.service('uiCustomizationService', ['$rootScope', 'DataikuAPI', '$q', function($rootScope, DataikuAPI, $q) {
    const datasetTypeStatus = { HIDE: 'HIDE', SHOW: 'SHOW', NO_CONNECTION: 'NO_CONNECTION'};
    let computeDatasetTypeStatusCache = {}; // this is project-dependant, we cache for each project

    this.datasetTypeStatus = datasetTypeStatus;

    // returns a promise resolving into a function (type) => datasetTypeStatus
    this.getComputeDatasetTypesStatus = (scope, projectKey) => {
        if(! computeDatasetTypeStatusCache[projectKey]) {
            const deferred = $q.defer();
            DataikuAPI.datasets.listCreatableDatasetTypes(projectKey).success((datasetTypes) => {
                const uiCustomization = $rootScope.appConfig.uiCustomization; // just a shortcut

                deferred.resolve((type) => {
                    const typeIsIn = (array) => array.indexOf(type) !== -1;

                    // Hive is always hidden if showTraditionalHadoop is unchecked
                    if(!uiCustomization.showTraditionalHadoop && type === 'hiveserver2') {
                        return datasetTypeStatus.HIDE;
                    }

                    if(uiCustomization.hideDatasetTypes && uiCustomization.hideDatasetTypes.length !== 0) {
                        // we hide datasets that are blackListed except if they are allowed by personal connections
                        const isBlackListed = typeIsIn(uiCustomization.hideDatasetTypes);
                        const fromPersonalConnections = typeIsIn(datasetTypes.fromPersonalConnections);
                        if(isBlackListed && !fromPersonalConnections) {
                            return datasetTypeStatus.HIDE;
                        }
                    }

                    if(! uiCustomization.showDatasetTypesForWhichThereIsNoConnection) {
                        // we hide / disable dataset that have no adequate connection available
                        let hasConnection = typeIsIn(datasetTypes.fromAllConnections) || typeIsIn(datasetTypes.outsideOfConnections);

                        // if user is admin we consider all connections are available since he could create them (except UploadedFiles)
                        if($rootScope.isDSSAdmin() && type !== 'UploadedFiles') {
                            hasConnection = true;
                        }
                        // if user can create personal connections, we consider all connections (except UploadedFiles, Filesystem and hiveserver2) are available since he could create them
                        if($rootScope.appConfig.globalPermissions.mayCreateAuthenticatedConnections
                                && type !== 'UploadedFiles' && type !== 'Filesystem' && type !== 'hiveserver2') {
                            hasConnection = true;
                        }

                        if(!hasConnection) {
                            return datasetTypeStatus.NO_CONNECTION;
                        }

                    }

                    return datasetTypeStatus.SHOW;
                });
            }).error(setErrorInScope.bind(scope))
            .catch(() => {
                // In case of API error, we resolve the promise with a filter that will show every dataset type
                // not ideal, but better than showing nothing
                deferred.resolve(() => datasetTypeStatus.SHOW);
            });
            computeDatasetTypeStatusCache[projectKey] = deferred.promise;
        }
        return computeDatasetTypeStatusCache[projectKey];
    }

    // clear the cache when settings are changed.
    $rootScope.$watch(() => $rootScope.appConfig.uiCustomization, () => computeDatasetTypeStatusCache = {});
}]);

    /**
     * ActiveProjectKey is intended to replace explicit references to $stateParams.projectKey references
     * that litter the current code base.  These references make it impossible to use many directives outside
     * of an opened project.
     * The service get() method will return $stateParams.projectKey whenever possible, but when this is not
     * defined it will return a value previously saved via the set() method.
     */
    app.service('ActiveProjectKey', function($stateParams) {
    let explicitProjectKey;
    this.set = function(projectKey) {
        explicitProjectKey = projectKey;
    };
    this.get = function() {
        let key = $stateParams.projectKey;
        if (typeof(key) === 'undefined') key = explicitProjectKey;
        return key;
    }
});

app.factory("TableChangePropagator", function() {
    var svc = {seqId: 0};
    svc.update = function() {
        svc.seqId++;
    };
    return svc;
});

app.factory("executeWithInstantDigest", function($timeout) {
    return function executeWithInstantDigest(fn, scope) {
        return $timeout(function(){
            scope.$apply(fn);
        });
    }
});




app.factory("textSize", function() {
    // if text is a string returns its length
    // if text is a list of texts returns the maximum width.
    // if container is not undefined, append it to container
    // element will be added and then remove before any repaint.
    return function(texts, tagName, className, container) {
        if (texts.length === undefined) {
            texts = [texts];
        }
        if (className === undefined) {
            className = "";
        }
        className += " text-sizer";
        if (container === undefined) {
            container = $("body");
        }
        else {
            container = $(container);
        }
        var maxWidth = 0;
        var elements = [];
        for (let i = 0; i < texts.length; i++) {
            let text = texts[i];
            let el = document.createElement(tagName);
            el.className = className;
            let $el = $(el);
            $el.text(text);
            container.append($el);
            elements.push($el);
        }
        ;
        for (let i = 0; i < elements.length; i++) {
            let $el = elements[i];
            maxWidth = Math.max(maxWidth, $el.width());
            $el.remove();
        }
        ;
        return maxWidth;
    }
});


app.factory("CollectionFiltering", function(Fn) {
    // service to filter collection with complex filters
    // i.e. magic around $filter('filter')

    // usage : CollectionFiltering.filter([obj,obj,obj], filterQuery, params)

    // filterQuery is an object that's matched against objects from the list
    // JS objects : {} means a logical AND
    // JS arrays : [] means a logical OR
    // string/Regex properties are then matched against each others with a substr logic

    // ** Parameters **
    // * filterQuery.userQuery : special str property that's matched against all properties of the object
    // *        this filter is cumulative (AND behavior) with regard to other fields in filterQuery,
    // *        but has an OR behavior with regard to the object properties (only needs to match a single field to validate)
    // *        property:value strings are extracted from filterQuery.userQuery and added to the filterQuery object
    // * params.userQueryTargets : single (or list of) dotted property paths to restrict the match of userQuery
    // *                            (except specified property:value patterns in the userQuery, which will bypass this restriction)
    // * params.propertyRules : if you want to give the user shortcuts for properties (replace dict)
    // *                        for example, if propertyRules = { 'name': 'smartId' }, a userQuery 'name:toto' would look for toto in the smartId field
    // * params.exactMatch: array of keys for which the search will require an exact match

    // params are enriched with :
    // * params.userQueryResult : a list of Regex+str that where used in the matching (for highlight in list)
    // * params.userQueryErrors : a dict : {str:errMessage} that occurred during parsing

    /**
     * Returns true if the query and obj matches
     *
     * @param query the search term to look for in the object (can be a regex or string)
     * @param obj the object to match
     * @param exactMatch if the string comparison requires an exact match
     *
     * @return {Boolean} true if the search term matches the object
     */
    var matchString = function(query, obj, exactMatch) {
        if (query instanceof RegExp) {
            return query.test(obj);
        } else {
            return !exactMatch ? ('' + obj).toLowerCase().indexOf(query) > -1 : ('' + obj).toLowerCase() === query;
        }
    }

    /**
     * Returns true if the searchTerm matches any allowed property from the specified object
     *
     * @param searchTerm the search term to look for in the object properties
     * @param obj the object on which properties to look
     * @param searchInFields list of allowed fields to search in (absolute paths). if undefined, allows any path
     * @param currentPath current absolute path of the object being tested (used to compare with the allowed paths when matching)
     * @return {Boolean} true if the search term matches any property from the object
     */
    var searchTermMatchesAnyProperty = function(searchTerm, obj, searchInFields, currentPath = '') {
        if (Array.isArray(obj)) {
            // ok if any of the array item matches. array dereference does not 'consume' path
            return obj.some(item => searchTermMatchesAnyProperty(searchTerm, item, searchInFields, currentPath));
        } else if ($.isPlainObject(obj)) {
            // ok if any of the field matches, but if the subpath matches the allowed paths.
            return Object.keys(obj)
                .filter((objKey) => objKey.charAt(0) !== '$' && hasOwnProperty.call(obj, objKey))
                .some((objKey) => {
                    const newPath = currentPath + '.' + objKey;
                    return searchTermMatchesAnyProperty(searchTerm, obj[objKey], searchInFields, newPath);
                });
        } else if (searchTerm instanceof RegExp || ((typeof(searchTerm) === "string" || typeof(searchTerm) === "number") && searchTerm !== "")) {
            return (searchInFields === undefined || searchInFields.some(allowedPath => currentPath.startsWith('.'+allowedPath))) // startsWith because if the match is deep under an allowed path, it's ok
                && matchString(searchTerm, obj);
        } else {
            return true;
        }
    };

    /**
     * Returns true if each word the query matches any property from the specified object (with an AND behaviour)
     * For special syntax (i.e. when you search for tag:tag_name) it must match any value for that specific property
     *
     * @param query an object containing the prepared and clean query of the user
     *        The object usually looks something like :
     *          {
     *             userQuery: '',
     *             tags: [],
     *             interest: {
     *                 starred: '',
     *             },
     *          }
     * @param params the filter params for that query. Contains the properties we want to target, those on which we want an exact match, property shortcuts, etc.
     * @param obj the object we want to search on
     * @param queryArrayIsAND if true and the query is an array, all terms must match a property of obj to return true
     * @param exactMatch true if the two compared strings must be exact match
     *
     * @return {Boolean} true if the query matches any property from the specified object
     */
    var filterFilter = function(query, params, obj, queryArrayIsAND, exactMatch) {
        if (Array.isArray(query)) {
            if (query.length == 0) {
                return true
            }
            if (queryArrayIsAND) {
                return query.every(searchTerm => filterFilter(searchTerm, params, obj, undefined, exactMatch));
            } else {
                return query.some(searchTerm => filterFilter(searchTerm, params, obj, undefined, exactMatch));
            }
        } else if (Array.isArray(obj)) {
            if (obj.length == 0 && query != "") {
                return false;
            }
            return obj.some(value => filterFilter(query, params, value, undefined, exactMatch));
        } else if ($.isPlainObject(obj) || $.isPlainObject(query)) {
            if (!$.isPlainObject(obj)) {
                return false
            }
            if (!$.isPlainObject(query)) {
                return true
            }

            // $requiredSearchTerms must all appear in the object, though not necessarily all in the same field
            if (query.$requiredSearchTerms) {
                const userQueryTargets = params.userQueryTargets
                        ? Array.isArray(params.userQueryTargets) ? params.userQueryTargets : [params.userQueryTargets]
                        : undefined;
                if(!query.$requiredSearchTerms.every(
                    searchTerm => searchTermMatchesAnyProperty(searchTerm, obj, userQueryTargets)
                )) {
                    return false;
                }
            }

            for (var objKey in query) {
                const requiresExactMatch = params.exactMatch && params.exactMatch.includes(objKey);

                if (objKey.charAt(0) !== '$' && objKey !== 'userQuery') {
                    const queryIsNOT = objKey.endsWith("__not");
                    const queryIsAND = objKey.endsWith("__and");
                    if (!queryIsNOT && !queryIsAND
                        && !filterFilter(query[objKey], params, Fn.propStr(objKey)(obj), undefined, requiresExactMatch)) {
                        return false;
                    }
                    if (!queryIsNOT && queryIsAND) {
                        const objValue = Fn.propStr(objKey.substr(0,objKey.length-5))(obj)
                        if (!objValue || !filterFilter(query[objKey], params, objValue, true, requiresExactMatch)) {
                            return false;
                        }
                    }
                    if (queryIsNOT && !queryIsAND) {
                        const objValue = Fn.propStr(objKey.substr(0,objKey.length-5))(obj)
                        if (objValue && filterFilter(query[objKey], params, objValue, undefined, requiresExactMatch)) {
                            return false;
                        }
                    }
                }
            }
            return true;
        } else if (query instanceof RegExp || ((typeof(query) === "string" || typeof(query) === "number") && query !== "")) {
            return matchString(query, obj, exactMatch);
        } else {
            return true;
        }
    };

    var prepareStringKey = function(params, a) {
        if (a.endsWith("/") && a.startsWith("/") && a.length > 2) {
            try {
                return new RegExp(a.substr(1, a.length - 2));
            } catch (err) {
                if (params) {
                    params.userQueryErrors = params.userQueryErrors || {};
                    params.userQueryErrors[a] = err;
                }
                return a;
            }
        } else {
            return a.toLowerCase();
        }
    };

    var cleanFilterQuery = function(obj, params) {
        if ($.isPlainObject(obj)) {
            for (let k in obj) {
                obj[k] = cleanFilterQuery(obj[k], params);
                if (($.isEmptyObject(obj[k]) && !(obj[k] instanceof RegExp) && (typeof obj[k] !== 'boolean')) || obj[k] === "") {
                    delete obj[k]
                }
            }
        } else if ($.isArray(obj)) {
            for (let k = obj.length - 1; k >= 0; k--) {
                obj[k] = cleanFilterQuery(obj[k], params);
                if (($.isEmptyObject(obj[k]) && !(obj[k] instanceof RegExp) && (typeof obj[k] !== 'boolean')) || obj[k] === "") {
                    obj.splice(k, 1)
                }
            }
        } else if (typeof(obj) === "string" || typeof(obj) === "number" || typeof(obj) === "boolean") {
            obj = prepareStringKey(params, ''+obj);
        }
        return obj;
    };


    var translatePropertyName = function(params, propName) {
        if (propName === 'not') {
            return propName
        }
        propName = (params.propertyRules || {})[propName] || propName;
        if (propName.charAt(0) === '-') {
            propName = ((params.propertyRules || {})[propName.substr(1)] || propName.substr(1)) + "__not"
        } else {
            propName = propName + "__and"
        }
        return propName;
    }

    /**
     * Adds the value to the propStr entry (propStr -> value) of obj
     * If it's not an array convert it to an array and add value
     * If value is an array, add each value of array
     *
     * @param obj the object to add the value to
     * @param propStr the property on which to add the value
     * @param value the value to add
     *
     * @return void
     */
    var safePushProp = function(obj, propStr, value) {
        if (!obj || !propStr) {
            return
        }
        var initialValue = Fn.propStrSafe(propStr)(obj);
        if (!$.isArray(initialValue)) {
            if (initialValue) {
                Fn.setPropStr([initialValue], propStr)(obj)
            }
            else {
                Fn.setPropStr([], propStr)(obj)
            }
            initialValue = Fn.propStr(propStr)(obj);
        }
        if ($.isArray(value)) {
            value.forEach(function(o) {
                initialValue.push(o)
            })
        }
        else {
            initialValue.push(value)
        }
    }

    var handleUserQuery = function(filterQuery, params) {
        if (!filterQuery.userQuery) {
            return filterQuery
        }
        var remainingUserQuery = [];
        filterQuery.userQuery.split(" ").filter(function(o) {
            return o !== ""
        }).forEach(function(dottedProp) {
            if (dottedProp.indexOf(':') > -1 && !( dottedProp.startsWith("/") && dottedProp.endsWith("/") )) {
                var fqelem = dottedProp.split(":");
                var propName = translatePropertyName(params, fqelem.shift());
                safePushProp(filterQuery, propName, fqelem.join(":"));
            } else {
                remainingUserQuery.push(dottedProp);
            }
        });
        filterQuery.userQuery = remainingUserQuery.join(" ").trim();

        // prepare the remaining search terms so that we can match them on the object properties
        filterQuery.$requiredSearchTerms = ('' + filterQuery.userQuery).split(" ").filter(a => a != "").map(prepareStringKey.bind(null, null));

        params.userQueryResult = ('' + filterQuery.userQuery).split(" ").filter(a => a).map(prepareStringKey.bind(null, null));
        return filterQuery;
    };

    var handleUserQueryTargets = function(filterQuery, params) {
        if (!params.userQueryTargets || !filterQuery) {
            return filterQuery
        }
        var userQueryTargets = $.isArray(params.userQueryTargets) ? params.userQueryTargets : [params.userQueryTargets];
        // TODO : there should be a proper "summary of the query after the whole processing"

        // Add appropriate filters if user imputed not:something (only applies to userQueryTargets)
        if (filterQuery.not) {
            userQueryTargets.forEach(function(userQueryTarget) {
                safePushProp(filterQuery, userQueryTarget + "__not", filterQuery.not)
            });
            delete filterQuery.not;
        }

        return filterQuery
    };

    var filterWrapper = function(collection, query, params) {
        var filterQuery = angular.copy(query || {});
        if (!params) {
            params = {}
        } else {
            delete params.userQueryErrors;
            delete params.userQueryResult;
        }
        filterQuery = handleUserQuery(filterQuery, params);
        // console.info('HANDLE',JSON.stringify(filterQuery));
        filterQuery = handleUserQueryTargets(filterQuery, params);
        // console.info('TARGETS',JSON.stringify(filterQuery));
        filterQuery = cleanFilterQuery(filterQuery, params);
        // console.info('CLEANED',JSON.stringify(filterQuery));
        return collection.filter(function(o) {
            return filterFilter(filterQuery, params, o);
        });
    };

    return {
        filter: filterWrapper,
    }
});


let algorithmsPaletteColors = [
    "#f07c48", // poppy red (no, it's actually orange)
    "#fdc766", // chicky yellow
    "#7bc9a6", // turquoisy green
    "#4ec5da", // sky blue
    "#548ecb", // sea blue
    "#d848c0", // lilas
    "#41bb15" // some no name green
    //"#97668f", // lilas
    //"#5e2974", // dark purple
];

app.factory("algorithmsPalette", function() {
    var COLORS = algorithmsPaletteColors;
    return function(i) {
        return COLORS[i % COLORS.length];
    }
});


app.factory("categoricalPalette", function() {  // Keep in sync with PaletteFactory/categorical
    var COLORS = [
        "#f06548", // poppy red
        "#fdc766", // chicky yellow
        "#7bc9a6", // turquoisy green
        "#4ec5da", // sky blue
        "#548ecb", // sea blue
        "#97668f", // lilas
        "#5e2974", // dark purple
    ];
    return function(i) {
        return COLORS[i % COLORS.length];
    }
});

app.factory("gradientGenerator", function() {
    function toHexString(n) {
        return Math.round(n).toString(16);
    }
    function toHexColor(c) {
        return `#${toHexString(c.r)}${toHexString(c.g)}${toHexString(c.b)}`;
    }

    return function(hexColor, nbColors) {
        let currentColor = d3.rgb(hexColor);
        const colors = [];
        for (let i = 0; i < nbColors; i++) {
            colors.push(currentColor);
            currentColor = currentColor.brighter(0.05);
        }

        return colors.map(toHexColor);
    }
})

app.factory("divergingPalette", function() {
    /* given a value going from -1 to 1
     returns the color associated by a diverging palette
     going form blue to red. */
    var rgbToHsl = function(s) {
        return d3.rgb(s).hsl();
    };
    var RED_SCALE = ["#fbefef", "#FBDAC8", "#F3A583", "#D75D4D", "#B11F2C"].map(rgbToHsl);
    var BLUE_SCALE = ["#f0f1f1", "#CFE6F1", "#94C5DE", "#4793C3", "#2966AB"].map(rgbToHsl);

    var divergingPalette = function(r) {
        var SCALE = (r > 0) ? RED_SCALE : BLUE_SCALE;
        if (r < 0) {
            r = -r;
        }
        if (r >= 1.) r = 0.99;
        var N = SCALE.length - 1;
        var bucket = r * N | 0;
        var low_color = SCALE[bucket];
        var high_color = SCALE[bucket + 1];
        r = r * N - bucket;
        var h = high_color.h;
        var l = low_color.l * (1 - r) + high_color.l * r;
        var s = low_color.s * (1 - r) + high_color.s * r;
        return d3.hsl(h, s, l);
    };
    return divergingPalette;
});

app.constant("COLOR_PALETTES", {
    highContrast: ['#FFD502', '#02FFFF', '#FF0256', '#02FFD5', '#FF02AB', '#02D5FF', '#2C02FF', '#8102FF', '#022CFF', '#02ABFF', '#FF02D5', '#FF02FF', '#02FF81', '#FF0281', '#F0E704', '#02FFAB', '#5602FF', '#AB02FF', '#CDF600', '#D502FF', '#0256FF', '#FF5602', '#FF8102', '#0281FF', '#FFAB02', '#ABFF02'],
    algorithms: algorithmsPaletteColors
});

app.factory("AnyLoc", function(Assert) {
    // Caution: not in line with backend type AnyLoc
    // On the angular side, to keep in line with server/src/frontend/src/app/utils/loc.ts
    return {
        makeLoc : function(projectKey, localId) {
            return {
                    projectKey: projectKey,
                    localId: localId,
                    fullId: projectKey + "." + localId
            }
        },
        getLocFromSmart: function(contextProjectKey, name) {
            if (name.indexOf(".") >= 0) {
                return {
                    projectKey: name.split(".")[0],
                    localId: name.split(".")[1],
                    fullId: name
                }
            } else {
                return {
                    projectKey: contextProjectKey,
                    localId: name,
                    fullId: contextProjectKey + "." + name
                }
            }
        },
        getLocFromFull: function(fullName) {
            Assert.trueish(fullName.includes('.'), 'no dot in fullname');
            return {
                projectKey: fullName.split(".")[0],
                localId: fullName.split(".")[1],
                fullId: fullName
            };
        },
    }
});

app.factory("AutomationUtils", function() {
    const svc = {
        pythonEnvImportTimeModes: [
            ["INSTALL_IF_MISS", "Create new if not present"],
            ["FAIL_IF_MISS", "Fail if not present"],
            ["DO_NOTHING", "Ignore"]
        ],
        pythonEnvImportTimeModeDescs: [
            "When the bundle declares a dependency on a code env that does not exist on this instance, create a new code env with that name",
            "When the bundle declares a dependency on a code env that does not exist on this instance, stop the preloading",
            "Do not take action if a code env is missing"
        ],
        envImportSpecificationModes: [
            ["SPECIFIED", "User-specified list of packages"],
            ["ACTUAL", "Actual list of packages"]
        ],
        envImportSpecificationModeDescs: [
            "Use the list of packages that the user specified, and the default list of required packages",
            "Use the list of packages built by inspecting the environment"
        ]
    }

    return svc;
});

app.factory("ConnectionUtils", function() {
    const connectionSqlTypes =
        ["hiveserver2", "MySQL", "PostgreSQL", "AlloyDB", "Vertica", "Greenplum", "Redshift", "Teradata", "Oracle", "SQLServer", "Synapse", "FabricWarehouse", "Netezza", "BigQuery", "SAPHANA", "Snowflake", "JDBC", "Athena", "Trino", "Databricks", "DatabricksLakebase", "TreasureData"]; //TODO @datasets remove
    return {
        isIndexable: function(connection) {
            return connection && connection.type && connectionSqlTypes.includes(connection.type);
        }
    }
});

app.factory("DatasetUtils", function(Assert, DataikuAPI, $q, Logger, $rootScope, SmartId, RecipesUtils) {
    const sqlTypes = ["MySQL", "PostgreSQL", "AlloyDB", "Vertica", "Greenplum", "Redshift", "Teradata", "Oracle", "SQLServer", "Synapse", "FabricWarehouse", "Netezza", "BigQuery", "SAPHANA", "Snowflake", "JDBC", "Athena", "Trino", "Denodo", "Databricks", "DatabricksLakebase", "TreasureData"]; //TODO @datasets remove
    const sqlAbleTypes = ["S3"];
    const svc = {
        canUseSQL: function(dataset) {
            if (sqlTypes.indexOf(dataset.type) >= 0 &&
                dataset.params.mode == "table") {
                return true;
            }
            if (sqlAbleTypes.indexOf(dataset.type) >= 0) {
                return true;
            }
            if (dataset.type == "HDFS" || dataset.type == "hiveserver2") {// && $scope.appConfig.impalaEnabled) {
                return true;
            }
            Logger.info("Dataset is not SQL-capable: " + dataset.type);
            return false;
        },

        canUseSparkSQL: function(dataset) {
            if ($rootScope.appConfig.interactiveSparkEngine == "DATABRICKS") {
                return dataset.type == "HDFS";
            } else {
                if (sqlTypes.indexOf(dataset.type) >= 0 && dataset.params.mode == "table") {
                    return false;
                }
                return true;
            }
        },

        hasSizeStatus: function(type) {
            return svc.getKindForConsistency({type: type}) == "files";
        },

        isSQL: function(dataset) {
            return sqlTypes.indexOf(dataset.type) >= 0;
        },

        isSQLTable: function(dataset) {
            return sqlAbleTypes.indexOf(dataset.type) >= 0 ||
                sqlTypes.indexOf(dataset.type) >= 0 && (!dataset.params || dataset.params.mode == "table"); // A bit hackish, when we don't have the params, let's not block functionality
        },

        // Can we run SQL queries on this dataset
        isSQLQueryAble: function(dataset) {
            return sqlAbleTypes.indexOf(dataset.type) >= 0 ||
                sqlTypes.indexOf(dataset.type) >= 0;
        },

        supportsReadOrdering : function(dataset) {
            return svc.isSQLTable(dataset);
        },

        getKindForConsistency : function(dataset) {
            if (sqlTypes.indexOf(dataset.type) >= 0) {
                return "sql";
            } else if (dataset.type == "MongoDB") {
                return "mongodb";
            } else if (dataset.type == "SharePointOnlineList") {
                return "sharepointonlinelist";
            } else if (dataset.type == "DynamoDB") {
                return "dynamodb";
            } else if (dataset.type == "Cassandra") {
                return "cassandra";
            } else if (dataset.type == "Twitter") {
                return "generic";
            } else if (dataset.type == "ElasticSearch") {
                return "generic";
            } else if (dataset.type == "Kafka") {
                return "generic";
            } else if (dataset.type == "SQS") {
                return "generic";
            } else {
                return "files";
            }
        },

        getLocFromSmart: function(contextProjectKey, name) {
            if (name.indexOf(".") >= 0) {
                return {
                    projectKey: name.split(".")[0],
                    name: name.split(".")[1],
                    fullName: name
                };
            } else {
                return {
                    projectKey: contextProjectKey,
                    name: name,
                    fullName: contextProjectKey + "." + name
                };
            }
        },

        getLocFromFull: function(fullName) {
            Assert.trueish(fullName.includes('.'), 'no dot in fullname');
            return {
                projectKey: fullName.split(".")[0],
                name: fullName.split(".")[1],
                fullName: fullName
            };
        },

        getSchema: function(scope, datasetName) {
            return svc.getSchemaFromComputablesMap(scope.computablesMap,datasetName)
        },

        getSchemaFromComputablesMap: function(computablesMap, datasetName) {
            if (!computablesMap || !computablesMap[datasetName]) {
                return;
            }
            var it = computablesMap[datasetName];
            if (!it || !it.dataset) {
                throw Error('dataset is not in computablesMap');
            }
            return it.dataset.schema;
        },

        makeLoc: function(datasetProjectKey, datasetName) {
            return {
                projectKey: datasetProjectKey,
                name: datasetName,
                fullName: datasetProjectKey + "." + datasetName
            };
        },

        makeSmart: function(loc, contextProjectKey) {
            if (loc.projectKey == contextProjectKey) {
                return loc.name;
            } else {
                return loc.fullName;
            }
        },

        makeHeadSelection: function(lines) {
            return {
                partitionSelectionMethod: "ALL",
                samplingMethod: "HEAD_SEQUENTIAL",
                maxRecords: lines
            };
        },

        updateRecipeComputables: function(scope, recipe, projectKey, contextProjectKey) {
            if (!scope.computablesMap) return Promise.resolve();
            let references = new Set(RecipesUtils.getFlatIOList(recipe).map(role => role.ref));
            references = [...references].filter(ref => (ref in scope.computablesMap) && ((scope.computablesMap[ref].dataset && !scope.computablesMap[ref].dataset.schema) || (scope.computablesMap[ref].streamingEndpoint && !scope.computablesMap[ref].streamingEndpoint.schema)));
            return $q.all(references
                .map(name => {
                    let resolvedSmartId = SmartId.resolve(name, contextProjectKey);
                    if (scope.computablesMap[name].dataset) {
                        return DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                            scope.computablesMap[name].dataset = data;
                        }).error(setErrorInScope.bind(scope));
                    } else if (scope.computablesMap[name].streamingEndpoint) {
                        return DataikuAPI.streamingEndpoints.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                            scope.computablesMap[name].streamingEndpoint = data;
                        }).error(setErrorInScope.bind(scope));
                    }
                })
            );
        },

        updateDatasetInComputablesMap: function(scope, dsName, projectKey, contextProjectKey) {
            let resolvedSmartId = SmartId.resolve(dsName, contextProjectKey);
            return DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                scope.computablesMap[dsName].dataset = data;
            }).error(setErrorInScope.bind(scope));
        },

        listDatasetsUsabilityForAny: function(contextProjectKey) {
            return DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly: true
            });
        },

        listFoldersUsabilityForOutput : function(contextProjectKey, recipeType) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly : false,
                forRecipeType : recipeType,
            }).success(function(data) {
                data.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                d.resolve(data);
            });
            return d.promise;
        },

        listDatasetsUsabilityForInput: function(contextProjectKey, recipeType) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly: true,
                forRecipeType: recipeType,
            }).success(function(data) {
                data.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                d.resolve(data);
            });
            return d.promise;
        },

        /**
         * Returns a promise on an arry of two array, "availableInputDatasets" and "availableOutputDataset"
         * On each, the "usable" and "usableReason" are set
         */
        listDatasetsUsabilityInAndOut: function(contextProjectKey, recipeType, datasetsOnly) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                datasetsOnly: datasetsOnly == null ? true : datasetsOnly,
                forRecipeType: recipeType,
            }).success(function(data) {
                var avlIn = angular.copy(data);
                var avlOut = angular.copy(data);
                avlIn.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                avlOut.forEach(function(x) {
                    x.usable = x.usableAsOutput;
                    x.usableReason = x.outputReason;
                });
                d.resolve([avlIn, avlOut]);
            });
            return d.promise;
        },

        /**
         * Returns a promise on an arry of two array, "availableInputDatasets" and "availableOutputDataset"
         * On each, the "usable" and "usableReason" are set
         */
        listUsabilityInAndOut: function(contextProjectKey, recipeType) {
            var d = $q.defer();
            DataikuAPI.flow.listUsableComputables(contextProjectKey, {
                forRecipeType: recipeType
            }).success(function(data) {
                var avlIn = angular.copy(data);
                var avlOut = angular.copy(data);
                avlIn.forEach(function(x) {
                    x.usable = x.usableAsInput;
                    x.usableReason = x.inputReason;
                });
                avlOut.forEach(function(x) {
                    x.usable = x.usableAsOutput;
                    x.usableReason = x.outputReason;
                });
                d.resolve([avlIn, avlOut]);
            });
            return d.promise;
        },

        isOnDifferentConnection: function (dataset, primaryDatasetConnection) {
            return primaryDatasetConnection && dataset.dataset.params.connection !== primaryDatasetConnection;
        },

        isOutputDataset: function (dataset, recipe, outputDatasetName) {
            return (dataset.dataset.name === outputDatasetName && recipe.projectKey && dataset.dataset.projectKey === recipe.projectKey);
        },

        INVALID_INPUT_DATASET_REASONS: {
            DIFF_CONN: "Dataset connection is different from primary one",
            IS_RECIPE_OUTPUT: "Dataset cannot be the recipe's output"
        },

        /**
         * Mark the recipe output dataset as unusable.
         * Currently used for join input dataset select options.
         * @param datasets
         * @returns Copy of input datasets with the recipe output dataset marked as unusable.
         */
        setInputDatasetsUsability: function (datasets, recipe, outputDatasetName) {
            const availableInputDatasetsForJoin = [];
            for (const originalDataset of datasets) {
                const dataset = angular.copy(originalDataset);
                if (this.isOutputDataset(dataset, recipe, outputDatasetName)) {
                    dataset.usable = false;
                    dataset.usableReason = this.INVALID_INPUT_DATASET_REASONS.IS_RECIPE_OUTPUT;
                }
                availableInputDatasetsForJoin.push(dataset);
            }
            return availableInputDatasetsForJoin;
        }
    };
    return svc;
});


app.factory("Breadcrumb", function($stateParams, $rootScope) {
    var ret = {}
    $rootScope.masterBreadcrumbData = {}

    ret.setProjectSummary = function(projectSummary) {
        $rootScope.currentProjectSummary = projectSummary;
    }

    ret.setData = function(k, v) {
        $rootScope.masterBreadcrumbData[k] = v;
    }

    ret.projectBreadcrumb = function() {
        return [
            //{ "type" : "home" },
            {"type": "project", "projectKey": $stateParams.projectKey}
        ]
    }
    ret.datasetBreadcrumb = function() {
        return ret.projectBreadcrumb().concat([
            //{"type" : "datasets", projectKey : $stateParams.projectKey},
            {
                "type": "dataset", projectKey: $stateParams.projectKey, id: $stateParams.datasetName,
                displayName: $stateParams.datasetName
            }
        ]);
    }
    ret.recipeBreadcrumb = function() {
        return ret.projectBreadcrumb().concat([
            {"type": "recipes", projectKey: $stateParams.projectKey},
            {
                "type": "recipe", projectKey: $stateParams.projectKey, id: $stateParams.recipeName,
                displayName: $stateParams.recipeName
            }
        ]);
    }
    ret.insightBreadcrumb = function(insightName) {
        return ret.projectBreadcrumb().concat([
            {"type": "insights", "projectKey": $stateParams.projectKey},
            {
                "type": "insight",
                "projectKey": $stateParams.projectKey,
                "id": $stateParams.insightId,
                displayName: insightName
            }
        ]);
    }

    ret.set = function(array) {
        $rootScope.masterBreadcrumb = array;
    }

    ret.setWithProject = function(array) {
        ret.set(ret.projectBreadcrumb().concat(array));
    }
    ret.setWithDataset = function(array) {
        ret.set(ret.datasetBreadcrumb().concat(array));
    }
    ret.setWithInsight = function(insightName, array) {
        ret.set(ret.insightBreadcrumb(insightName).concat(array));
    }
    ret.setWithRecipe = function(array) {
        ret.set(ret.recipeBreadcrumb().concat(array));
    }
    ret.setWith
    return ret;
});


app.service('LocalStorage', ['$window', function($window) {
    return {
        set: function(key, value) {
            if (value !== undefined) {
                $window.localStorage[key] = JSON.stringify(value);
            }
        },
        get: function(key) {
            var ret = $window.localStorage[key];
            if (ret !== undefined) {
                ret = JSON.parse(ret);
            }
            return ret;
        },
        clear: function(key) {
            delete $window.localStorage[key];
        }
    }
}]);


app.service('RemoteResourcesLinksUtils', function() {
    const svc = this;

    svc.getSageMakerResourceLink = function(resource, region, resourceName) {
        return `https://${region}.console.aws.amazon.com/sagemaker/home#/${resource}/${resourceName}`;
    }

    svc.getVertexAIResourceLink = function(resource, project, location, resourceId) {
        return `https://console.cloud.google.com/vertex-ai/locations/${location}/${resource}/${resourceId}?project=${project}`;
    }

    const getWsid = function(workspace, resourceGroup, subscription) {
        return `wsid=/subscriptions/${subscription}/resourcegroups/${resourceGroup}/providers/Microsoft.MachineLearningServices/workspaces/${workspace}`;
    }

    const getTid = function(tenantId) {
        return (tenantId) ? `tid=${tenantId}` : '';
    }

    svc.getAzureMLReference = function(resourceName, resourceVersion) {
        return `azureml:${resourceName}:${resourceVersion}`;
    }

    svc.getAzureMLModelLink = function(resourceInfo) {
        const wsid = getWsid(resourceInfo.azWorkspace, resourceInfo.azResourceGroup, resourceInfo.azSubscription);
        const tid = '&' + getTid(resourceInfo.azTenantId);
        return `https://ml.azure.com/model/${resourceInfo.azModelName}:${resourceInfo.azModelVersion}/details?${wsid}${tid}`;
    }

    svc.getAzureMLEnvironmentLink = function(resourceInfo) {
        const wsid = getWsid(resourceInfo.azWorkspace, resourceInfo.azResourceGroup, resourceInfo.azSubscription);
        const tid = '&' + getTid(resourceInfo.azTenantId);
        return `https://ml.azure.com/environments/${resourceInfo.azEnvironmentName}/version/${resourceInfo.azEnvironmentVersion}?${wsid}${tid}`;
    }

    svc.getAzureMLOnlineEndpointLink = function(resourceInfo) {
        const wsid = getWsid(resourceInfo.azWorkspace, resourceInfo.azResourceGroup, resourceInfo.azSubscription);
        const tid = '&' + getTid(resourceInfo.azTenantId);
        return `https://ml.azure.com/endpoints/realtime/${resourceInfo.azOnlineEndpointName}?${wsid}${tid}`;
    }

    svc.getDatabricksEndpointLink = function(host, endpointName) {
        return `https://${host}/ml/endpoints/${endpointName}`;
    }

    svc.getDatabricksModelLink = function(host, isUnityCatalogUsed, modelName) {
        if (isUnityCatalogUsed) {
            const [catalogName, schemaName, unityCatalogModelName] = modelName.split(".");
            return `https://${host}/explore/data/models/${catalogName}/${schemaName}/${unityCatalogModelName}`;
        }
        return `https://${host}/ml/models/${modelName}`;
    }

    svc.getDatabricksModelVersionLink = function(host, isUnityCatalogUsed, modelName, modelVersion) {
        const modelLink = svc.getDatabricksModelLink(host, isUnityCatalogUsed, modelName);
        if (isUnityCatalogUsed) {
            return `${modelLink}/version/${modelVersion}`;
        }
        return `${modelLink}/versions/${modelVersion}`;
    }

    svc.getDatabricksExperimentLink = function(host, expId) {
        return `https://${host}/ml/experiments/${expId}`;
    }

    svc.getDatabricksExperimentRunLink = function(host, expId, runId) {
        return `https://${host}/ml/experiments/${expId}/runs/${runId}`;
    }
});

app.factory("ContextualMenu", function($compile, $rootScope, $templateCache, $window, $http) {
        // Class describing a menu.
        // Can be used for both contextual menu and
        // regular menues.
        //
        function Menu(params) {
            /*
             Contextual or not, only one menu can be visible at the same time
             on the screen.
             The contextual menu content does not live in the DOM until it is displayed
             to the user.

             Parameters contains the following options
             - template (required) : template path for the content of the menu.
             - controller (optional) : name of the controller
             - scope  (optional) : if not added, a new scope will be created.
             - contextual (option, true|false, default:true
             in contextual menu mode, all clicks outside of the
             popup is captured.
             - onOpen (optional): called on menu open
             - onClose (optional): called on menu close
             - cssClass: CSS class on the ul
             */
            this.template = params.template;
            if (typeof this.template != "string") {
                throw "Template parameter is required";
            }
            this.cssClass = params.cssClass;
            this.controller = params.controller;
            this.contextual = params.contextual;
            if (this.contextual === undefined) {
                this.contextual = true;
            }
            this.enableClick = params.enableClick;
            if (this.enableClick === undefined) {
                this.enableClick = false;
            }
            this.handleKeyboard = params.handleKeyboard;
            if (this.handleKeyboard === undefined) {
                this.handleKeyboard = true;
            }
            this.scope = params.scope;
            this.tmplPromise = $http.get(this.template, {cache: $templateCache});
            this.onClose = params.onClose || function() {
                };
            this.onOpen = params.onOpen || function() {
                };
        }

        Menu.prototype.newScope = function() {
            if (this.scope) {
                return this.scope.$new();
            }
            else {
                return $rootScope.$new(true);
            }
        };

        Menu.prototype.globalOnClose = function() {
        };
        Menu.prototype.globalOnOpen = function() {
        };

        // close any popup currently visible on the screen.
        Menu.prototype.closeAny = function(e) {
            // remove and unbind any overlays
            Menu.prototype.$overlay.unbind("click");
            Menu.prototype.$overlay.unbind("contextmenu");
            Menu.prototype.$overlay.remove();

            // remove the document click
            $(document).off(".closeMenu");

            Menu.prototype.$menu.remove();
            Menu.prototype.globalOnClose();
            Menu.prototype.globalOnClose = function() {
            };
            Menu.prototype.globalOnOpen = function() {
            };
            Menu.prototype.$menu.removeClass();
            Menu.prototype.$menu.addClass('dropdown-menu');
            if (e) e.preventDefault();
            return false;
        };

        Menu.prototype.setup = function($menu) {
            var me = this;

            Menu.prototype.globalOnClose = this.onClose;
            Menu.prototype.globalOnOpen = this.onOpen;
            var index = -1;
            var currentMenu = Menu.prototype.$menu;

            if (me.contextual) {
                $menu.before(Menu.prototype.$overlay);
                $(Menu.prototype.$overlay).bind("contextmenu", me.closeAny.bind(me));
                Menu.prototype.$overlay.click(me.closeAny.bind(me));
            } else {
                window.setTimeout(function() {
                    // handle click when menu is open
                    $(document)
                        .on('click.closeMenu', function(evt) {
                            const targetParents = $(evt.target).parents();
                            const isClickedOutsideMenu = targetParents.index(Menu.prototype.$menu) === -1;
                            // Dropdown panels can be appended in the body but clicking in a dropdown panel shouldn't close a contextual menu.
                            const getClassList = (parent) => parent.classList ? Array.from(parent.classList) : [];
                            const isClickedElementADropdownItem = targetParents.toArray().some(parent => getClassList(parent).includes('ng-dropdown-panel'));
                            if (isClickedOutsideMenu && !isClickedElementADropdownItem) {
                                Menu.prototype.closeAny();
                            }
                        });
                }, 0);
            }
            if (!me.enableClick) {
                $menu.on('click.ctxmenu', function(e) {
                    me.closeAny();
                });
            }

            window.setTimeout(function() {
                // makes the links focusable
                Menu.prototype.$menu.find('a').attr('tabindex', -1);
                // focus on the first link
                const items = Menu.prototype.$menu.find('a');
                if (items.length > 0) {
                    const item = items.eq(0);
                    const parentMenus = item.parents('ul');
                    if (parentMenus.length > 0) {
                        currentMenu = parentMenus.eq(0);
                        index = 0;
                        item.focus();
                    }
                }

                var handleKey = function(evt) {
                    if (Menu.prototype.$menu.height()) {
                        if (evt.which === 27) {
                            // esc
                            Menu.prototype.closeAny();
                        }

                        const isTargetInMenu = $(evt.target).parents().index(currentMenu) !== -1; // Only process events when the target is an item of the menu
                        if (isTargetInMenu) {
                            if (evt.which === 40) {
                                // down
                                const items = currentMenu.find('>li>a');
                                if (items.length) {
                                    index = Math.min(items.length - 1, index + 1);
                                    items[index].focus();
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }
                            if (evt.which === 38) {
                                // up
                                index = Math.max(0, index - 1);
                                const items = currentMenu.find('>li>a, >*>li>a');
                                if (items.length) {
                                    items[index].focus();
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }

                            if (evt.which === 37) {
                                // left
                                // Go up one menu
                                if (currentMenu != Menu.prototype.$menu) {
                                    index = currentMenu.parents('ul').eq(0).find('>li>a').index(currentMenu.parent('li.hover').removeClass('hover').find('>a'));
                                    currentMenu = currentMenu.parents('ul').eq(0);
                                    const items = currentMenu.find('>li>a');
                                    if (items.length) {
                                        items[index].focus();
                                    }
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }
                            if (evt.which === 39) {
                                // right
                                // go into submenu
                                const items = currentMenu.find('>li>a');
                                const submenus = items.eq(index).siblings('ul');
                                if (submenus.length) {
                                    items.eq(index).parent().addClass('hover');
                                    currentMenu = submenus.eq(0);
                                    index = 0;
                                    currentMenu.find('a').eq(0).focus();
                                }
                                evt.preventDefault();
                                evt.stopPropagation();
                            }
                            if (evt.which === 13) {
                                // enter
                                currentMenu.find('>li>a').eq(index).trigger('click');
                                Menu.prototype.closeAny();
                            }
                        }
                    }
                };
                if (me.handleKeyboard) {
                    $(document).on('keydown.closeMenu', handleKey); // handle keypress while the menu doesn't have the focus
                    Menu.prototype.$menu.on('keydown', handleKey);
                }
                Menu.prototype.$menu.on('mouseenter', 'a', function(e) {
                    Menu.prototype.$menu.find('.hover').removeClass('hover');
                    currentMenu = $(this).parents('ul').eq(0);
                    const items = currentMenu.find('>li>a')
                    index = items.index($(this));
                    if (index !== -1) {
                        items[index].focus(); // Set focus to current item, making it the target for keyboard events
                    }
                });

            });
        };

        // Fill the shared menu element with menu instance
        // content.
        //
        // Template is compiled against the scope
        // at each call.
        //
        Menu.prototype.fill = function(cb) {
            var me = this;
            this.tmplPromise.success(function(tmplData) {
                if (me.controller !== undefined) {
                    me.$menu.attr("ng-controller", me.controller);
                }
                else {
                    me.$menu.removeAttr("ng-controller");
                }
                me.$menu.html(tmplData);
                Menu.prototype.destroyCurrentScope();
                var newScope = me.newScope();
                $compile(me.$menu)(newScope);
                Menu.prototype.currentScope = newScope;
                if (cb !== undefined) {
                    cb(me.$menu);
                }
                if (me.cssClass) {
                    me.$menu.addClass(me.cssClass);
                }
            });
        };

        Menu.prototype.destroyCurrentScope = function() {
            if (Menu.prototype.currentScope != undefined) {
                Menu.prototype.currentScope.$destroy();
            }
            Menu.prototype.currentScope = undefined;
        };

        Menu.prototype.openAlignedWithElement = function(alignElement, callback, followScroll, exactAlignment) {
            var me = this;
            me.closeAny();
            me.fill(function($menu) {
                // place the element.
                var $body = $("body");
                var $alignElement = $(alignElement);
                var alignElementOffset = $alignElement.offset();
                var scrollOffsetLeft = alignElementOffset.left;
                var scrollOffsetTop = alignElementOffset.top;

                var box = $alignElement.offset();
                box.width = $alignElement.width();
                box.height = $alignElement.outerHeight();
                $body.append($menu);
                var left = Math.max(0, box.left - (exactAlignment ? 0 : 10));

                // we also want to move the dropdown menu to the left
                // to stay on the screen.
                var menuWidth = $menu.width();
                var bodyWidth = $body.width();

                $menu.detach();

                left = Math.min(left, bodyWidth - menuWidth - 10);

                if (bodyWidth - left - menuWidth < 180) {
                    // let's step into bizarro land
                    // where submenues open to the left.
                    $menu.addClass("bizarro");
                }
                else {
                    $menu.removeClass("bizarro");
                }

                var position = {
                    left: left,
                    top: box.top + box.height,
                    bottom: "auto",
                    right: "auto"
                };

                let containerElement = $body;

                if (followScroll) {
                    containerElement = $(alignElement).offsetParent();
                    const alignElementPosition = $alignElement.position();
                    scrollOffsetLeft = alignElementPosition.left - box.left;
                    scrollOffsetTop = alignElementPosition.top - box.top;
                    position.left += scrollOffsetLeft;
                    position.top += scrollOffsetTop;
                }

                $menu.appendTo(containerElement);
                $menu.css(position);

                me.setup($menu);
                if (callback !== undefined) {
                    callback($menu);
                }
                Menu.prototype.globalOnOpen();
            });
        };

        Menu.prototype.openAtXY = function(left, top, callback, dummyLateralPosition, dummyVerticalPosition) {
            var me = this;
            me.closeAny();
            me.fill(function($menu) {
                $("body").append($menu);
                var offset = {};
                if (left < $($window).width() / 2 || dummyLateralPosition) {
                    offset.left = left;
                    offset.right = 'auto';
                } else {
                    offset.left = 'auto';
                    offset.right = $($window).width() - left;
                }
                if (top < $($window).height() / 2 || dummyVerticalPosition) {
                    offset.top = top;
                    offset.bottom = 'auto';
                } else {
                    offset.top = 'auto';
                    offset.bottom = $($window).height() - top;
                }
                $menu.css(offset);
                me.setup($menu);
                if (callback !== undefined) {
                    callback($menu);
                }
                Menu.prototype.globalOnOpen();
            });
        };

        Menu.prototype.openAtEventLoc = function (evt) {
            if (!evt) return;
            this.openAtXY(evt.pageX, evt.pageY);
        }

        // TODO get rid of the id
        Menu.prototype.$menu = $('<ul id="dku-contextual-menu" class="dropdown-menu" style="position:absolute" role="menu">');
        // overlay element that helps capturing any click
        // outside of the menu.
        // Used in ContextualMenu mode.
        Menu.prototype.$overlay = $('<div class="contextualMenuOverlay"></div>');
        // Menu.prototype.$overlay

        return Menu;
    });


app.factory("ActivityIndicatorManager", ["$timeout", function($timeout) {
    function hide(activityIndicator) {
        activityIndicator.hidden = true;
    }

    function getActivityIndicatorType(type) {
        switch (type) {
            case 'waiting':
            case 'info':
                return 'progress';
            case 'success':
                return 'success';
            case 'warning':
                return 'warning';
            case 'error':
                return 'error';
            default:
                throw new Error('Unknown type: ' + type);
        }
    }

    return {
        hide,
        /**
         *
         * @param {ActivityIndicator} activityIndicator
         * @param {'waiting' | 'success' | 'warning' | 'info' | 'error'} type
         * @param {string} text
         * @param {number} time
         * @param {boolean?} faded
         * @param {{ label: string, callback: () => void, icon?: string }} action
         */
        configureActivityIndicator: function (activityIndicator, type, text, time, faded = true, action) {
            activityIndicator.hidden = false;
            activityIndicator.text = text;
            activityIndicator.type = getActivityIndicatorType(type);
            activityIndicator.faded = faded;
            activityIndicator.action = action;
            activityIndicator.hide = () => hide(activityIndicator);

            if (type === 'waiting') {
                activityIndicator.spinner = true;
            } else {
                activityIndicator.spinner = false;
                if (!time) {
                    time = 2000;
                }
                if(activityIndicator.pending) {
                    $timeout.cancel(activityIndicator.pending);
                }
                activityIndicator.pending = $timeout(function () {
                    activityIndicator.pending = undefined;
                    hide(activityIndicator);
                }, time);
            }
        },
        buildDefaultActivityIndicator: function() {
            return {
                pending: undefined,
                hidden: true,
                text: '',
                type: 'progress',
                spinner: false,
                faded: true
            };
        },
        isDisplayed: function(activityIndicator) {
            return !activityIndicator.hidden;
        }
    };
}]);

app.factory("ActivityIndicator", ["$rootScope", "ActivityIndicatorManager", function($rootScope, ActivityIndicatorManager) {
    $rootScope.activityIndicator = ActivityIndicatorManager.buildDefaultActivityIndicator();
    return {
        isDisplayed: function() {
            return ActivityIndicatorManager.isDisplayed($rootScope.activityIndicator)
        },
        waiting: function(text) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator, 'waiting', text);
        },
        hide: function() {
            ActivityIndicatorManager.hide($rootScope.activityIndicator);
        },
        success: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'success', text, time, true, action);
        },
        warning: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'warning', text, time, true, action);
        },
        info: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'info', text, time, true, action);
        },
        error: function(text, time, action) {
            ActivityIndicatorManager.configureActivityIndicator($rootScope.activityIndicator,'error', text, time, true, action);
        }
    };
}]);

app.factory("ChartActivityIndicator", ["ActivityIndicatorManager", function(ActivityIndicatorManager) {
    return {
        buildDefaultActivityIndicator: function () {
            return ActivityIndicatorManager.buildDefaultActivityIndicator()
        },
        displayBackendError: function (chartActivityIndicator, errorMessage) {
            ActivityIndicatorManager.configureActivityIndicator(chartActivityIndicator, 'error', errorMessage, 5000, false);
        }
    };
}]);

app.factory("APIXHRService", ["$rootScope", "$http", "$q", "Logger", "HistoryService", function($rootScope, $http, $q, Logger, HistoryService) {
    $rootScope.httpRequests = [];

    var unloadingState = false;

    $(window).bind("beforeunload", function() {
        unloadingState = true;
    });

    // Return a proxified promise that can be disabled
    function disableOnExit(promise) {

        function isEnabled() {
            return !unloadingState;
        }

        var deferred = $q.defer();

        // $q promises
        promise.then(function(data) {

                if (isEnabled()) {
                    deferred.resolve(data);
                }
            },
            function(data) {
                if (isEnabled()) {
                    deferred.reject(data);
                }
            },
            function(data) {
                if (isEnabled()) {
                    deferred.notify(data);
                }
            });

        // $http specific
        if (promise.success) {
            deferred.promise.success = function(callback) {
                promise.success(function(data, status, headers, config, statusText, xhrStatus) {
                    if (isEnabled()) {
                        callback(data === 'null' ? null : data, status, headers, config, statusText, xhrStatus);
                    }
                });
                return deferred.promise;
            };
        }

        if (promise.error) {
            promise.error(function(data, status, headers) {
                var apiError = getErrorDetails(data, status, headers);
                Logger.error("API error: ", apiError.errorType + ": " + apiError.message);
            })
            deferred.promise.error = function(callback) {
                promise.error(function(data, status, headers, config, statusText, xhrStatus) {
                    if (isEnabled()) {
                        callback(data, status, headers, config, statusText, xhrStatus);
                    }
                });
                return deferred.promise;
            };
        }

        if (promise.noSpinner) {
            deferred.promise.noSpinner = promise.noSpinner;
        }

        return deferred.promise;
    }

    return function(method, url, data, spinnerMode, contentType, timeoutPromise) {
        var headers = {
            'Content-Type': contentType === 'json' ? 'application/json;charset=utf-8' : 'application/x-www-form-urlencoded;charset=utf-8'
        };
        var versionIdHeader = "version-id";

        if ($rootScope.versionId && !isAbsoluteURL(url)) {
            headers[versionIdHeader] = $rootScope.versionId;
        }

        var start = new Date().getTime();
        Logger.debug("[S] " + method + ' ' + url);

        var params = {
            method: method,
            url: url,
            headers: headers,
            transformRequest: function(data) {
                // Transform data based on the content type
                if (contentType === 'json') {
                    return angular.isObject(data) ? JSON.stringify(data) : data;
                } else {
                    return angular.isObject(data) && String(data) !== '[object File]' ? jQuery.param(data) : data;
                }
            },
        };
        if ($rootScope.appConfig) {
            params.xsrfCookieName = $rootScope.appConfig.xsrfCookieName;
        }
        if (method == 'GET') {
            params.params = data;
        } else {
            params.data = data;
        }
        if (timeoutPromise) {
            params.timeout = timeoutPromise;
        }

        var promise = $http(params);
        var disableSpinner = spinnerMode && spinnerMode == "nospinner";

        const logDone = function (result) {
            const end = new Date().getTime();
            const details = [];
            if (result && result.status) {
                details.push("s=" + result.status);
            }
            details.push("f=" + (end - start) + "ms");
            if (result != null && result.headers != null) {
                const backendTime = result.headers("DKU-Call-BackendTime");
                if (backendTime != null) {
                    details.push("b=" + backendTime + "ms");
                }
                const versionId = result.headers(versionIdHeader);
                // The "Version-Id" header is *generally* present but it is not always guaranteed (e.g. 'hideVersionStringsWhenNotLogged')
                // The absence of this header can help diagnose spoofed requests (e.g. blocked by WAF, reverse proxy issues, etc)
                details.push("v=" + (versionId == null ? "?" : versionId));
            }
            Logger.debug("[D] " + method + " " + url + " (" + details.join(", ") + ")");
        };
        promise.then(logDone, logDone);

        if (!disableSpinner) {
            promise.spinnerMode = spinnerMode;
            $rootScope.httpRequests.push(promise);

            var removeRequest = function(result) {
                var idx = $rootScope.httpRequests.indexOf(promise);
                if (idx != -1) $rootScope.httpRequests.splice(idx, 1);
            };

            promise.noSpinner = function() {
                removeRequest();
                safeApply($rootScope);
                return promise;
            };

            promise.spinner = function(enable = true) {
                if (enable === false) {
                    promise.noSpinner();
                }
                return promise;
            };

            promise.then(removeRequest, removeRequest);
        }

        function isAbsoluteURL(url) {
            try {
                new URL(url);
            } catch (e) {
                if (e instanceof TypeError) {
                    return false;
                } else {
                    throw e;
                }
            }
            return true;
        }

        function retrieveVersionId(result) {
            if (result.headers) {
                const versionId = result.headers(versionIdHeader)

                if (versionId && !$rootScope.versionId) {
                    $rootScope.versionId = versionId;
                }
            }
        }

        promise.then(retrieveVersionId, retrieveVersionId);

        function promptRefreshOnVersionMismatch(result) {
            if (result.headers) {
                const promptRefresh = Boolean(result.headers("version-mismatch-refresh-prompt"));
                const backendVersionId = result.headers(versionIdHeader);

                if (promptRefresh && $rootScope.versionId &&
                    $rootScope.versionId !== $rootScope.promptRefreshFromVersionId &&
                    backendVersionId !== $rootScope.promptRefreshFromVersionId) {
                    $rootScope.promptRefresh = promptRefresh;
                    $rootScope.promptRefreshFromVersionId=backendVersionId;
                }
            }
        }

        promise.then(promptRefreshOnVersionMismatch, promptRefreshOnVersionMismatch);


        if (method=="POST") HistoryService.recordItemPost(url, data);

        return app.addSuccessErrorToPromise(promise);
    };
}]);


app.factory("CreateModalFromDOMElement", function(CreateModalFromHTML) {
    return function(selector, scope, controller, afterCompileCallback) {
        return CreateModalFromHTML($(selector).html(), scope, controller, afterCompileCallback, false);
    };
});

app.factory("CreateModalFromTemplate", function(CreateModalFromHTML, $http, $templateCache, $q) {
    /**
     * Callback for preparing the modal scope
     *
     * @callback afterCompileCallback
     * @param {Object} modalScope the initialized modal scope
     * @returns {void}
     */


    /**
     * Create a modal popup from a template
     *
     * @param {String} location path to the modal HTML template
     * @param {Object} scope The scope variable that can be passed along and used in the controller
     * @param {String} controller The name of the controller that the modal scope should inherit from during initialization
     * @param {afterCompileCallback} afterCompileCallback an anonymous function that will be executed after the controller initialization
     * @param noFocus
     * @param backdrop can be 'true', 'false' or 'static' (check bootstrap documentation) or custom 'confirm' value (in this case a click on the backdrop will be treated as a confirmation on the modal)
     * @param keyboard
     */
    return function(location, scope, controller, afterCompileCallback, noFocus, backdrop, keyboard) {
        var deferred = $q.defer();
        $q.when($templateCache.get(location) || $http.get(location, {cache: true})).then(function(template) {
            if (angular.isArray(template)) {
                template = template[1];
            } else if (angular.isObject(template)) {
                template = template.data;
            }
            deferred.resolve(CreateModalFromHTML(template, scope, controller, afterCompileCallback, noFocus, backdrop, keyboard));
        });
        return deferred.promise;
    }
});

/**
 * Create a modal from an AngularJS component
 *
 * Usage:
 *
 *  CreateModalFromComponent(injectedComponentDirective, {
 *       paramOne: ...,
 *       paramTwo: ...,
 *  }, ['custom-modal-class-1', 'custom-modal-class-2']);
 *
 *  1) 'paramOne' & 'paramTwo' will be respectively bound to the 'param-one' and 'param-two' inputs of the component
 *  2) 'injectedComponentDirective' is the component's directive
 *     /!\ For a component('myComponent') you need to inject 'myComponentDirective' (AngularJS magically appends a 'Directive' suffix to all components names)
 *  3) the 3rd parameter is an optional array of CSS classes to add to the modal (for example, 'modal-wide')
 */
app.factory("CreateModalFromComponent", function(CreateModalFromHTML, $rootScope) {
    function camelToKebab(input) {
        return input.replace(/([A-Z])/g, function($1){return "-"+$1.toLowerCase();});
    }

    function getComponentTagName(component) {
        return camelToKebab(component[0].name); // Lol
    }

    return function(component, params, customClasses) {
        const newScope = $rootScope.$new(true);
        const tagName = getComponentTagName(component);
        let html = '<div class="modal modal3';
        if (customClasses) {
            for (let customClass of customClasses) {
                html += ` ${customClass}`;
            }
        }
        html += '"><' + tagName + ' ';
        for (let p in params) {
            newScope[p] = params[p];
            html += camelToKebab(p) + "=" + '"' + p + '" ';
        }
        html += 'modal-control="modalControl"'
        html += '></'+tagName+ '><div>';
        const promise = CreateModalFromHTML(html, newScope, null, function($scope, element) {
            newScope.modalControl = {
                resolve: value => $scope.resolveModal(value),
                dismiss: () => $scope.dismiss()
            }
        });
        return promise;
    }
});

app.factory("CreateModalFromHTML", function($timeout, $compile, $q, $rootScope) {
    let activeModals = [];

    $rootScope.$on('dismissModals', function() {
        activeModals.forEach((modalScope)=> unregisterModal(modalScope));
    });

    function registerModal(modalScope) {
        activeModals.unshift(modalScope);
    }

    function unregisterModal(modalScope) {
        activeModals = activeModals.filter((activeModalScope)=> {
            if(modalScope == activeModalScope) {
                modalScope.dismiss();
                return false;
            }
            return true;
        });
    }

    return function(template, scope, controller, afterCompileCallback, noFocus, backdrop, keyboard) {
        var deferred = $q.defer();
        var newDOMElt = $(template);
        if (controller != null) {
            newDOMElt.attr("ng-controller", controller);
        }
        newDOMElt.addClass("ng-cloak");

        const $existingModal = $('div.modal-container');
        let stackedClass = "";
        let waitForTransition = 0;
        if ($existingModal.length>0) {
            $existingModal.addClass('aside').removeClass('restored'); //move aside any existing modal in case of stacking
            waitForTransition = 250;
            stackedClass = "new-stacked"
        }

        var wrapper = $("<div>").addClass("modal-container " + stackedClass).append(newDOMElt);
        $("body").append(wrapper);

        if(scope.fromAngularContext === true) {
            wrapper.addClass("modal-container--from-angular");
        }

        $timeout(function() {
            var newScope = scope.$new();
            $compile(newDOMElt)(newScope);

            var modalScope = angular.element(newDOMElt).scope();

            if (afterCompileCallback) {
                modalScope.$apply(afterCompileCallback(modalScope, newDOMElt));
            }
            newDOMElt.on('hidden', function(e){
                if (e.target == newDOMElt.get(0)) {
                    unregisterModal(modalScope);
                    wrapper.remove();
                    modalScope.$destroy();
                    if (deferred != null) {
                        deferred.reject("modal hidden");
                        deferred = null;
                    }
                }
            });

            var prepareForModalStack = function () {
                $('div.modal-backdrop').addClass('modal-rollup').removeClass('non-see-through'); //mjt in the event of stacking a modal
                $("div.modal-container.new-stacked").addClass("modal-stacked-on-top").removeClass("new-stacked");
            }
            prepareForModalStack();

            if (backdrop) {
                 newDOMElt.attr('data-backdrop', backdrop === 'confirm' ? 'static' : backdrop);
            }

            newDOMElt.modal("show");
            $rootScope.$broadcast("dismissPopovers");

            modalScope.unwindModalStack = function (newDOMElt) {
                $('div.modal-backdrop.modal-rollup').removeClass('modal-rollup').click(modalScope.dismiss);
                $('div.modal-container.aside').removeClass('aside').addClass('restored'); //move aside any existing modal in case of stacking
            };
            modalScope.dismiss = function() {
                newDOMElt.modal("hide");
                if (deferred != null) {
                    deferred.reject("dismissed modal");
                    deferred = null;
                }
            };

            registerModal(modalScope);

            modalScope.resolveModal = function(value) {
                if (deferred != null) {
                    deferred.resolve(value);
                    deferred = null;
                }
                newDOMElt.modal("hide");
            };
            modalScope.$modalScope = modalScope;

            $(newDOMElt).on('hide.bs.modal', function (e) {
                if (modalScope && modalScope.canCloseModal && typeof modalScope.canCloseModal === 'function') {
                    if (!modalScope.canCloseModal()) {
                        e.preventDefault();
                        e.stopImmediatePropagation();
                        return false;
                    }
                }
                modalScope.unwindModalStack(newDOMElt);
            });

            if (backdrop === 'confirm') {
                // a click on the backdrop will be treated as a confirmation
                     $(".modal-backdrop").on('click', function() {
                            modalScope.confirm();
                    });
            }

            modalScope.$on("dismissModal", modalScope.dismiss);

            if (!noFocus) {
                // the first form of the modal, should contain the modal-body in 99% of cases
                var firstForm = newDOMElt.find('form').first();
                // list of focusable elements we want to try, in order of preference
                var focusCandidateFinders = [];
                focusCandidateFinders.push(function() {
                    return firstForm.find('input[type="text"]:not([readonly])').first();
                });
                focusCandidateFinders.push(function() {
                    return firstForm.find('button:submit').first();
                });
                focusCandidateFinders.push(function() {
                    return firstForm.find('button:button').first();
                });
                // if the modal has no form, or footer buttons are not in the form, look in the full modal
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('input[type="text"]:not([readonly])').first();
                });
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('button:submit').first();
                });
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('button:button').first();
                });
                focusCandidateFinders.push(function() {
                    return newDOMElt.find('.close').first();
                });

                var focusCandidate;
                for (var i = 0; i < focusCandidateFinders.length; i++) {
                    var focusCandidateFinder = focusCandidateFinders[i];
                    focusCandidate = focusCandidateFinder().not('.no-modal-autofocus');
                    if (focusCandidate.length > 0) {
                        focusCandidate.focus();
                        // in some cases the element is disabled by a ng-disabled, and the focus behavior becomes a bit erratic
                        // so for safety we focus once more
                        $timeout(function() {
                            focusCandidate.focus();
                        });
                        break;
                    }
                }

                // in case the submit button is dangerous, prevent submit-on-enter
                if (firstForm.length > 0 && focusCandidate.hasClass('btn--danger')) { //NOSONAR: focusCandidate always initialized thanks to jquery first() specs
                    focusCandidate.bind("keydown keypress", function(event) {
                        if (event.which === 13) {
                            event.preventDefault();
                        }
                    });
                }
            }
        }, waitForTransition);

        return deferred.promise;
    };
});


/**
 * Create a custom body-attached DOM element within a new scope.
 * The new scope is fitted with a "dismiss" function, which destroys the DOM
 * element and the scope.
 */
app.factory("CreateCustomElementFromTemplate", ["$http", "$timeout", "$compile", "$templateCache", "$q", "$window",
    function($http, $timeout, $compile, $templateCache, $q, $window) {
        return function(location, scope, controller, afterCompileCallback, domInsertionCallback) {
            $q.when($templateCache.get(location) || $http.get(location, {cache: true})).then(function onSuccess(template) {
                if (angular.isArray(template)) {
                    template = template[1];
                } else if (angular.isObject(template)) {
                    template = template.data;
                }
                var newDOMElt = $(template);
                if (controller != null) {
                    newDOMElt.attr("ng-controller", controller);
                }

                if (domInsertionCallback != null) {
                    domInsertionCallback(newDOMElt);
                } else {
                    $("body").append(newDOMElt);
                }

                /* Now, compile the element, set its scope, call the callback */
                $timeout(function() {
                    var newScope = scope.$new();
                    $compile(newDOMElt)(newScope);
                    var newScope2 = angular.element(newDOMElt).scope();

                    if (afterCompileCallback) {
                        newScope2.$apply(afterCompileCallback(newScope2));
                    }
                    newScope2.$on("dismissModalInternal_", function() {
                        $timeout(function() {
                            newScope2.dismiss();
                        }, 0);
                    });
                    newScope2.dismiss = function() {
                        newDOMElt.remove();
                        newScope2.$destroy();
                    };
                    scope.$on("$destroy", newScope2.dismiss);
                });
            });
        };
    }
]);


/** Keeps a map of promises for static API calls */
app.factory("CachedAPICalls", function(DataikuAPI, Assert, $http, $rootScope, $q, translate, $translate) {
    var deferredLogin = $q.defer();
    var whenLoggedIn = deferredLogin.promise;

    let staticData = app.addSuccessErrorToPromise(whenLoggedIn.then(() => DataikuAPI.staticData.getStaticData($translate.proposedLanguage())));

    return {
        notifyLoggedIn: () => {
            deferredLogin.resolve();
        },
        processorsLibrary: app.addSuccessErrorToPromise(whenLoggedIn.then(() => DataikuAPI.shakers.getProcessorsLibrary($translate.proposedLanguage()).success(function(processors) {
            processors.deprecatedTypes = [];

            // Inject the doc link at this point so it is only done once.
            processors.processors.forEach(function(p) {
                if (p.docPage) {
                    p.help += "\n\n" + translate("SHAKER.PROCESSORS.HELP.MORE_INFO",
                        "For more info, <a target=\"_blank\" href=\"{{docPageUrl}}\">please see the processor's reference</a>",
                        { docPageUrl: $rootScope.versionDocRoot + "preparation/processors/" + p.docPage + ".html"})
                        + "\n";
                }

                if (p.deprecated) {
                    processors.deprecatedTypes.push(p.type);

                    var deprecationMsg = "### " + translate("SHAKER.PROCESSORS.HELP.DEPRECATED", "This processor is deprecated.");
                    if (p.replacementDocLink) {
                        var replacementLink = $rootScope.versionDocRoot + p.replacementDocLink + ".html";
                        var replacementOptions = p.replacementName ? (p.replacementName + " options") : "options";
                        deprecationMsg += ` Please see <a target="_blank" href="${replacementLink}">our documentation</a> for our newer ${replacementOptions}.`;
                    }
                    p.help = deprecationMsg + "\n\n---\n\n" + p.help;
                    p.enDescription = "<span class=\"deprecation-tag\">"+ translate("SHAKER.PROCESSORS.DESCRIPTION.DEPRECATED", "[DEPRECATED]") + " </span>" + p.enDescription;
                }
            });

            return processors
        }))),
        staticData: staticData,
        customFormulasFunctions: staticData.then(resp => resp.data.customFormulasFunctions),
        udafCustomFormulasFunctions: staticData.then(resp => resp.data.udafCustomFormulasFunctions),
        customFormulasReference: staticData.then(resp => resp.data.customFormulasReference),
        udafCustomFormulasReference: staticData.then(resp => resp.data.udafCustomFormulasReference),
        datasetCommonCharsets: staticData.then(resp => resp.data.datasetCommonCharsets),
        datasetFormatTypes: staticData.then(resp => resp.data.datasetFormatTypes),
        datasetTypes: staticData.then(resp => resp.data.datasetTypes),
        pmlGuessPolicies: staticData.then(resp => resp.data.pmlGuessPolicies),
        cmlGuessPolicies: staticData.then(resp => resp.data.cmlGuessPolicies),
        timezonesList: staticData.then(resp => resp.data.timezonesList),
        timezonesShortList: staticData.then(resp => resp.data.timezonesShortList),
        webAppTypes: staticData.then(resp => resp.data.webAppTypes),
        mlCommonDiagnosticsDefinition: staticData.then(resp => resp.data.mlCommonDiagnosticsDefinition),
        pmlDiagnosticsDefinition: staticData.then(resp => function(backendType, predictionType) {
            const backendTypeNullable = backendType === null ? undefined : backendType;
            const matchingItem = resp.data.pmlDiagnosticsDefinition.find((item) => item.backendType === backendTypeNullable && item.predictionType === predictionType);
            return !matchingItem ? {} : matchingItem.definitions;
        }),
        cmlDiagnosticsDefinition: staticData.then(resp => function(backendType) {
            const backendTypeNullable = backendType === null ? undefined : backendType;
            const matchingItem = resp.data.cmlDiagnosticsDefinition.find((item) => item.backendType === backendTypeNullable);
            return !matchingItem ? {} : matchingItem.definitions;
        }),
        flowIcons: $http.get("/static/dataiku/flow-iconset.json"),
        emojisTable: $http.get("/static/third/emoji.json").then(function(response) {
            Assert.trueish(response.data, 'No emoji returned');
            Assert.trueish(angular.isArray(response.data), 'Emojis were not returned as an array');
            const emojisTable = {};
            response.data.forEach(function(x) {
                emojisTable[x['sn']] = x['code'].split('-').map(x => '&#x' + x + ';').join('');
            });
            return emojisTable;
        })
    };
});


app.service('ComputablesService', function(CreateModalFromTemplate, DataikuAPI, TaggableObjectsUtils, Dialogs) {
    this.clear = function(scope, taggableItems) {
        return CreateModalFromTemplate("/templates/taggable-objects/clear-data-modal.html", scope, null, function(modalScope) {
            modalScope.taggableItems = taggableItems;
            modalScope.itemsType = TaggableObjectsUtils.getCommonType(taggableItems, it => it.type);

            modalScope.confirm = function() {
                DataikuAPI.taggableObjects.clear(taggableItems).success(function(data) {
                    if (data.anyMessage && !data.success) {
                        modalScope.dismiss();
                        Dialogs.infoMessagesDisplayOnly(scope, "Clear result", data);
                    } else {
                        modalScope.resolveModal(data);
                    }
                }).error(setErrorInScope.bind(scope));
            }
        });
    };
});


app.service('DatasetsService', function($rootScope, $q, DataikuAPI, Logger, Notification, ComputablesService, FutureProgressModal, CreateModalFromTemplate, FlowGraph) {
    const svc = this;

    const listsWithAccessiblePerProject = {};

    svc.listWithAccessible = function(projectKey) {
        if (listsWithAccessiblePerProject[projectKey] == null) {
            listsWithAccessiblePerProject[projectKey] = DataikuAPI.datasets.listWithAccessible(projectKey);
        }
        return listsWithAccessiblePerProject[projectKey];
    };

    svc.clear = function(scope, projectKey, datasetName) {
        return ComputablesService.clear(scope, [{
            type: 'DATASET',
            projectKey: projectKey,
            id: datasetName,
            displayName: datasetName
        }]);
    };

    svc.refreshSummaries = function (scope, selectedItems, computeRecords = true, forceRecompute = false, displayErrorsInFlow = false) {
        const deferred = $q.defer();
        DataikuAPI.datasets.refreshSummaries(selectedItems, computeRecords, forceRecompute).success(function (data) {
            FutureProgressModal.show(scope, data, "Refresh datasets status")
                .then(data => deferred.resolve(data), data => deferred.reject(data));
        }).error(displayErrorsInFlow ? FlowGraph.setError() : setErrorInScope.bind(scope));
        return deferred.promise;
    };

    svc.setVirtualizable = function(scope, selectedItems, virtualizable) {
        const datasets = selectedItems.filter(it => it.type == 'DATASET');
        return DataikuAPI.datasets.setVirtualizable(datasets, !!virtualizable)
            .error(setErrorInScope.bind(scope));
    };

    svc.startSetAutoCountOfRecords = function(selectedItems) {
        return CreateModalFromTemplate("/templates/datasets/set-auto-count-of-records-modal.html", $rootScope, null, function(modalScope) {
            modalScope.autoCountOfRecords = false;

            modalScope.ok = function(vitualizable) {
                svc.setAutoCountOfRecords(selectedItems, modalScope.autoCountOfRecords)
                    .then(modalScope.resolveModal, setErrorInScope.bind(modalScope));
            };
        });
    };

    svc.setAutoCountOfRecords = function(selectedItems, autoCountOfRecords) {
        return DataikuAPI.datasets.setAutoCountOfRecords(selectedItems, autoCountOfRecords);
    };

    Notification.registerEvent("datasets-list-changed", function(evt, message) {
        Logger.info("Datasets list changed, updating");
        delete listsWithAccessiblePerProject[message.projectKey]; // just invalidate
    });
});

/** Cached access to datasets information */
app.factory("DatasetInfoCache", function($stateParams, DataikuAPI, $q, Notification, Logger) {
    // Cache for results of datasets/get
    var simpleCache = {}
    Notification.registerEvent("websocket-status-changed", function() {
        Logger.info("Websocket status change, dropping dataset cache");
        simpleCache = {};
    });
    Notification.registerEvent("datasets-list-changed", function(evt, message) {
        Logger.info("Datasets list changed, dropping cache for ", message.projectKey);
        delete simpleCache[message.projectKey];
    });
    var svc = {
        getSimple: function(projectKey, name) {
            var projectCache = simpleCache[projectKey];
            if (projectCache != null) {
                var data = projectCache[name];
                if (data != null) {
                    Logger.info("Cache hit: " + projectKey + "." + name);
                    return $q.when(data);
                }
            } else {
                simpleCache[projectKey] = {};
            }
            Logger.info("Cache miss: " + projectKey + "." + name);
            return DataikuAPI.datasets.get(projectKey, name, $stateParams.projectKey).then(function(data) {
                simpleCache[projectKey][name] = data.data;
                return data.data;
            });
        }
    }
    return svc;
});

// Queue
//
// var lockable = Queue();
// lockable.exec(function() { alert('A'); }); // Executed right now
// var unlock = lockable.lock(); // Lock the object
// lockable.exec(function() { alert('B'); }); // Not executed
// lockable.exec(function() { alert('C'); }); // Not executed
// unlock(); // Execute alert('A') & alert('B');
//
// The queue can be tied to a promise
// lockable.lockOnPromise(DataikuAPI.xxx(yyy,zzz).success(function() {
//     ...
// }));
//
app.factory("Queue", function() {

    return function() {

        var semaphore = 0;
        var queue = [];
        var destroyed = false;
        var scopeUnregisterer = undefined;
        var inside = false;

        var processQueue = function() {
            while (!destroyed && semaphore == 0 && queue.length > 0) {
                if (!inside) {
                    try {
                        inside = true;
                        queue.splice(0, 1)[0]();
                    } finally {
                        inside = false;
                    }
                } else {
                    break;
                }
            }
        };

        var exec = function(fn) {
            if (fn && !destroyed) {
                queue.push(fn);
                processQueue();
            }
        };

        var destroy = function() {
            destroyed = true;
            queue = [];
        };

        var wrap = function(func) {
            return function() {
                var args = arguments;
                exec(function() {
                    if (func) {
                        func.apply(null, args);
                    }
                });
            };
        };

        var lock = function() {
            if (destroyed) {
                return;
            }
            semaphore++;
            var unlocked = false;
            return function() {
                if (!unlocked) {
                    semaphore--;
                    unlocked = true;
                    processQueue();
                }
            };
        };

        var ret = {

            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                return ret;
            },
            locked: function() {
                return destroyed || semaphore > 0;
            },
            exec: function(fn) {
                exec(fn);
            },
            wrap: function(fn) {
                return wrap(fn);
            },
            lockOnPromise: function(promise) {
                if (promise && promise['finally']) {
                    var unlocker = lock();
                    promise['finally'](function() {
                        unlocker();
                    });
                }
                return promise;
            },
            lock: function() {
                return lock();
            }
        };
        return ret;
    };

});

function enrichFuturePromise(promise) {
    promise.success = function(fn) {
        promise.then(function(data) {
            fn(data.data, data.status, data.headers);
        });
        return promise;
    };

    promise.error = function(fn) {
        promise.then(null, function(data) {
            fn(data.data, data.status, data.headers);
        });
        return promise;
    };

    promise.update = function(fn) {
        promise.then(null, null,function(data) {
            fn(data.data, data.status, data.headers);
        });
        return promise;
    };
    return promise;
}

// MonoFuture
//
// - Wait for future result
// - Abort the previous future as soon as a new one is started
//
// Usage :
//
// var monoFuture = MonoFuture(scope); // If a scope is passed, life time of monofuture = life time of the scope
// monoFuture.exec(DataikuAPI.xxx.getYYYFuture(zzz)).success(function(data) {
// ...
// }).update(function(data) {
// ...
// }).error(...);
//
//  OR by wrapping the DataikuAPI function directly
//
//  var apiCall = MonoFuture().wrap(DataikuAPI.xxx.getYYYFuture);
//  apiCall(zzz).success(...);
//
//
//

app.factory("MonoFuture", ["$q", "DataikuAPI", "Throttle", function($q, DataikuAPI, Throttle) {

    return function(scope, spinner=true, delay=1000) {

        var promises = [];
        var destroyed = false;

        // Refresh the state of the last promise
        var refreshFutureState = Throttle().withDelay(delay).wrap(function() {

            updateInternalState();

            if (promises.length > 0) {
                var last = promises[promises.length - 1];
                if (!last.hasResult && !last.waiting) {
                    last.waiting = true;
                    DataikuAPI.futures.getUpdate(last.id).success(function(data, status, headers) {

                        last.waiting = false;
                        last.result = {data: data, status: status, headers: headers};
                        if (data.hasResult) {
                            last.hasResult = true;
                        } else {
                            if (!last.aborted) {
                                last.deferred.notify(last.result);
                            }
                            refreshFutureState();
                        }

                        updateInternalState();

                    }).error(function(data, status, headers) {

                        last.failed = true;
                        last.result = {data: data, status: status, headers: headers};
                        last.hasResult = true;
                        last.waiting = false;

                        updateInternalState();
                    });
                }
            }
        });

        // Update current state
        var updateInternalState = function() {
            // Abort all abortable futures & remove them
            var loop = false;
            do {
                loop = false;
                for (var i = 0; i < promises.length; i++) {
                    var isLast = i == (promises.length - 1);
                    var promise = promises[i];
                    promise.aborted |= !isLast;
                    if (promise.aborted && !promise.waiting) {

                        // We have the future id, and not the result meaning that the future is running
                        if (promise.id && !promise.hasResult) {
                            // Abort me
                            DataikuAPI.futures.abort(promise.id, spinner);
                        }

                        promises.splice(i, 1);
                        loop = true;
                        break;
                    }
                }
            } while (loop);

            // Check last one : finished?
            if (promises.length > 0) {
                var last = promises[promises.length - 1];
                if (last.hasResult && !last.aborted) {
                    promises.splice(promises.length - 1, 1);
                    if (last.failed) {
                        last.deferred.reject(last.result);
                    } else if (last.result && last.result.data && last.result.data.aborted) {
                        // The future has been aborted by someone
                        last.deferred.reject(last.result);
                    } else {
                        last.deferred.resolve(last.result);
                    }
                }
            }
        };

        var fakePromise = function() {
            var promise = $q.defer().promise;
            promise.success = function() {
                return promise;
            };
            promise.error = function() {
                return promise;
            };
            promise.update = function() {
                return promise;
            };
            return promise;
        };

        var exec = function(apiPromise) {
            if (destroyed || !apiPromise || !apiPromise.success) {
                return fakePromise();
            }
            var deferred = $q.defer();
            var promise = {
                id: null,
                hasResult: false,
                result: undefined,
                deferred: deferred,
                aborted: false,
                failed: false,
                waiting: true,
                noSpinner: apiPromise.noSpinner
            };
            promises.push(promise);
            updateInternalState();
            apiPromise.success(function(data, status, headers) {
                promise.waiting = false;
                promise.result = {data, status, headers};
                if (data) {
                    promise.id = data.jobId;
                }
                if (data.hasResult) {
                    promise.hasResult = true;
                } else {
                    refreshFutureState();
                }
                updateInternalState();
            }).error(function(data, status, headers) {
                promise.failed = true;
                promise.result = {data, status, headers};
                if (data) {
                    promise.id = data.jobId;
                }
                promise.hasResult = true;
                promise.waiting = false;
                updateInternalState();
            });

            enrichFuturePromise(deferred.promise);

            deferred.promise.noSpinner = function() {
                promises.forEach(function(p) {
                    if (p.noSpinner) p.noSpinner();
                });
                return deferred.promise;
            };

            return deferred.promise;
        };

        var abort = function() {
            if (promises.length > 0) {
                promises[promises.length - 1].aborted = true;
                updateInternalState();
            }
        };

        var destroy = function() {
            abort();
            destroyed = true;
        };

        const active = function() {
            return promises.length > 0;
        };

        if (scope && scope.$on) {
            scope.$on('$destroy', destroy);
        }

        return {

            exec: exec,
            wrap: function(func) {
                return function() {
                    return exec(func.apply(func, arguments));
                };
            },
            abort: abort,
            destroy: destroy,
            active: active,
            destroyed: () => destroyed
        };
    };
}]);


/* MonoFuturePool
    Wrapper around MonoFuture that allows a bunch of related monofutures to run sequencially, never running more than one at a time
    API is mostly the same as for the MonoFuture, except: exec makes no sense for the pool you have to use wrap, and the returned promise doesn't have the noSpinner method.

   Example: imagine you have this kind of stuff in a component:

    const wrapper = MonoFuture(scope).wrap(DataikuApi.blah.blah);
    $scope.$watch('someApiParams', (nv) => {
        $scope.something = wrapper(nv)
    });

    And there are 20 instances of this component => you may have many future running in parallel, one for each component

    Using:
    // in some shared place
    const pooledMonoFuture = MonoFuturePool(scope);

    ...

    //in each component
    const wrapper = pooledMonoFuture.createMonoFuture(scope, false).wrap(DataikuApi.blah.blah);
    $scope.$watch('someApiParams', (nv) => {
        $scope.something = wrapper(nv)
    });

    Future calls from each component will be queued and only one will run at once.
    If any of the components triggers a new call (someApiParams changed) while a future is running, it will be put on hold to be executed once the running query is done. If any of the component triggers a new call while it already has a call on hold, the previous one is dropped and replaced by the new one.

    Timeline example:
    - Component 1 triggers a future with call A1 => api call is done & progress is tracked
    - Component 2 triggers a future with call A2 => put on hold
    - Component 3 triggers a future with call A3 => put on hold
    - Component 3 triggers a future with call B3 => put on hold, A3 is dropped
    - API call A1 is done => API call A2 is triggered and and progress is tracked
    - API call A2 is done => API call B3 is triggered and and progress is tracked

    key points:
    - only one future is in progress at any time
    - A3 has been dropped before even started because it has been discarded by B3
    - The results obtained by each component is no different than what they would have got independantly, except for the longer delay.

    When the pool is created with a scope, all child MonoFutures are destroyed when this scope is destroyed
    When a child MonoFuture is created with a scope, it's destroyed when this scope is destroyed (and releases the lock for the others when applicable)
    It is possible to use both scope-based auto-destroy at the same time (if the Pool is bound to a component but each MonoFuture to one of many chilren for example)
*/

app.factory("MonoFuturePool", ["$q", "$timeout", "MonoFuture", "Queue", function($q, $timeout, MonoFuture, Queue) {
    return function MonoFuturePool(poolScope) {
        const allMonoFutures = new Set();
        const lockable = Queue().withScope(poolScope);
        poolScope?.$on?.('$destroy', destroy);

        return {
            createMonoFuture,
            destroy,
        };

        function createMonoFuture(monoFutureScope, spinner, delay) {
            const monoFuture = MonoFuture(undefined, spinner, delay); // we don't give the scope to the base monofuture because we need to manage destroy here to release the lock
            let waitingTask // the next scheduled task for this MonoFuture. When the monofuture is wrapping an API call, running waitingTask is what triggers the API call.
            let releaseThisMonoFutureLockIfAny = () => {}; // method to unlock the current query lock - only works when current monoFuture holds the lock, otherwise it's a noop

            const pooledMonoFuture = {
                wrap: function(func) {
                    return function(...args) {
                        const pendingMonoFuturePromiseDeferred = $q.defer(); // simple wrapper on the MonoFuture.exec output to handle the fact we don't have it synchronously, but we need to return it

                        if (waitingTask) {
                            // there is already a waiting task for this MonoFuture => we don't need to schedule anything, we just override the waitingTask & it will be executed in place of the previously scheduled one.
                            waitingTask = () => func.apply(func, args);
                        } else {
                            // otherwise, we schedule the task for when the lock is released. If the lock is free, lockable.exec will run synchronously
                            waitingTask = () => func.apply(func, args);
                            lockable.exec(() => {
                                releaseThisMonoFutureLockIfAny = lockable.lock();

                                // at this point, waitingTask may have been overridden by a later call, or the monoFuture may have be destroyed
                                // we run the task if the monoFuture has not be destroyed / the task has not been discarded in-between
                                const apiPromise = monoFuture.destroyed() ? undefined : waitingTask?.(); // let's not even make the call if destroyed
                                waitingTask = undefined;

                                if (!apiPromise || !apiPromise.success) {
                                    // cases when there isn't anything to query anymore: monoFuture is destroyed, task cancelled, wrapped method didn't return a promise
                                    // nothing is pending aymore => release the lock
                                    releaseThisMonoFutureLockIfAny();
                                } else {
                                    // normal case: track progress using MonoFuture then release lock
                                    const run = monoFuture.exec(apiPromise).finally(() => {
                                        releaseThisMonoFutureLockIfAny();
                                    });
                                    pendingMonoFuturePromiseDeferred.resolve(run); // forwards resolve, notify & rejects from the inner MonoFuture to the outer pooled one.
                                }
                            });
                        }

                        return enrichFuturePromise(pendingMonoFuturePromiseDeferred.promise);
                    };
                },
                abort() {
                    waitingTask = undefined; // before release otherwise a waiting task could be triggered
                    releaseThisMonoFutureLockIfAny();
                    monoFuture.abort();
                },
                destroy() {
                    waitingTask = undefined;
                    // when many monoFuture instances from the same pool are destroyed at once, we could have each destroy release the lock let the next query start, just for it to be destroyed after
                    // this micro-delay allows the next task to only be executed after the end of a synchronous bulk delete.
                    $timeout(releaseThisMonoFutureLockIfAny, 0);
                    allMonoFutures.delete(pooledMonoFuture);
                    monoFuture.destroy();
                },
                active() {
                    return monoFuture.active() || waitingTask !== undefined;
                },
            }

            if (monoFutureScope && monoFutureScope.$on) {
                monoFutureScope.$on('$destroy', () => pooledMonoFuture.destroy());
            }

            allMonoFutures.add(pooledMonoFuture);
            return pooledMonoFuture;
        }

        function destroy() {
            [...allMonoFutures].forEach(mf => mf.destroy());
        }
    }

}]);


// Provide a way to take/release the spinner
//
// Usage:
//
//  var spinner = SpinnerService();   // The spinner is permanently released automatically when scope is destroyed
//  spinner.acquire();
//  /* spinning... */
//  spinner.release(); // It's safe to release it multiple time
//
//
//  The spinner can be tied to a promise like that:
//  SpinnerService.lockOnPromise(promise);
app.factory("SpinnerService", ["$rootScope", function($rootScope) {
    let fakeReq = {}; // Doesn't matter what this object is, it's never used anyway...
    let actives = 0; // Number of active spinners
    let scopeUnregisterer = null;

    // TODO : implement this properly
    // (currently it's  is a little hack around $rootScope.httpRequests)
    function update() {
        // Reset
        let idx = $rootScope.httpRequests.indexOf(fakeReq);
        if (idx != -1) {
            $rootScope.httpRequests.splice(idx, 1);
        }
        // Activate
        if (actives > 0) {
            $rootScope.httpRequests.push(fakeReq);
        }
    }

    function fnr() {
        let acquired = false;
        let destroyed = false;
        function acquire() {
            if (destroyed) {
                return;
            }
            if (!acquired) {
                acquired = true;
                actives++;
                update();
            }
        }
        function release() {
            if (acquired) {
                acquired = false;
                actives--;
                update();
            }
        }
        function destroy() {
            destroyed = true;
            release();
        }

        const ret = {
            acquire: acquire,
            release: release,
            destroy: destroy,
            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                return ret;
            }
        };
        return ret;
    };

    fnr.lockOnPromise = function(promise) {
        if (promise && promise['finally']) {
            var lock = fnr();
            lock.acquire();
            promise['finally'](function() {
                lock.release();
            });
        }
    }
    return fnr;
}]);


// Ability to merge watch calls into one (old value is kept, new value is updated)
function wrapWatchHelper(ctrl, fn) {
    var isSet = false;
    var oldVal = undefined;
    var newVal = undefined;

    var trueExec = ctrl.wrap(function() {
        if (!angular.equals(newVal, oldVal)) {
            isSet = false;
            fn(newVal, oldVal);
        }
    });

    return function(nv, ov) {
        if (isSet) {
            newVal = angular.copy(nv);
        } else {
            isSet = true;
            oldVal = angular.copy(ov);
            newVal = angular.copy(nv);
        }
        trueExec();
    };
};

// Debounce
// API is similar to Throttle but it implements a behavior similar to the "onSmartChange" directive as a service
//
// var fn = Debounce
//    .withDelay(500,1000)   // initial delay = 500ms, then delay = 1000ms
//    .withSpinner(true)
//    .wrap(function() {
//        console.log('hello');
//     });
//
// Example 1:
//
// fn(); // Will be executed in 500ms
//
// Example 2 :
// fn();  // Will be dropped
// [sleep less than 100ms]
// fn();  // Will be executed in 1000s
//
//
app.factory("Debounce", ["SpinnerService", "$rootScope", function(SpinnerService, $rootScope) {

    return function(scope) {

        var initialDelay = 0;
        var delay = 0;
        var destroyed = false;
        var spinner = SpinnerService();
        var enableSpinner = false;
        var stdTimer = null;
        var initTimer = null;
        var scopeUnregisterer = null;

        var exec = function(func) {

            if (destroyed) {
                return;
            }

            var wrapped = function debounceExecWrapped() {
                var toBeExec;
                if (func) {
                    spinner.release();
                    toBeExec = func;
                    func = null;
                }
                if (toBeExec) {
                    $rootScope.$apply(function debounceExec() {
                        toBeExec();
                    });
                } else {
                    // Important because the activity state may have changed!
                    $rootScope.$digest();
                }
            };

            var isFirst = true;
            if (stdTimer) {
                clearTimeout(stdTimer.key);
                stdTimer = null;
                isFirst = false;
            }

            if (initTimer) {
                clearTimeout(initTimer.key);
                initTimer = null;
                isFirst = false;
            }

            if (enableSpinner) {
                spinner.acquire();
            }

            if (isFirst) {

                initTimer = {
                    key: setTimeout(function() {
                        initTimer = null;
                        wrapped();
                    }, initialDelay)
                };

                stdTimer = {
                    key: setTimeout(function() {
                        stdTimer = null;
                        wrapped();
                    }, delay)
                };

            } else {

                stdTimer = {
                    key: setTimeout(function() {
                        stdTimer = null;
                        wrapped();
                    }, delay)
                };
            }
        }

        var abort = function() {
            spinner.release();
            if (initTimer) {
                clearTimeout(initTimer.key);
                initTimer = null;
            }
            if (stdTimer) {
                clearTimeout(stdTimer.key);
                stdTimer = null;
            }
        };

        var wrap = function debounceWrap(func) {
            return function debounceWrapped() {
                var args = arguments;
                var argsArray = Array.prototype.slice.call(arguments);

                if (argsArray.indexOf("SkipDebounceAndRunImmediately") !== -1) {
                    // eslint-disable-next-line no-console
                    console.debug("Skipping debounce, running function immediately");
                    argsArray = argsArray.filter(arg => arg != "SkipDebounceAndRunImmediately");
                    func.apply(null, argsArray);
                } else {
                    exec(function debounceWrappedCB() {
                        func.apply(null, args);
                    });
                }
            };
        };

        var destroy = function() {
            // eslint-disable-next-line no-console
            console.info("Destroy debounce on scope", scope); // NOSONAR
            destroyed = true;
            abort();
        };

        var ret = {
            exec: function(func) {
                exec(func);
            },
            wrap: function(func) {
                return wrap(func);
            },
            wrapWatch: function(fn) {
                return wrapWatchHelper(ret, fn);
            },
            active: function() {
                return !!(initTimer || stdTimer);
            },
            abort: function() {
                abort();
            },
            destroy: function() {
                destroy();
            },
            withDelay: function(newInitialDelay, newDelay) {
                delay = newDelay;
                initialDelay = newInitialDelay;
                return ret;
            },
            withSpinner: function(enabled) {
                enableSpinner = enabled;
                return ret;
            },
            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                spinner.withScope(scope);
                return ret;
            }
        };
        return ret;

    };
}]);


// Limit the maximum update frequency to 1/delay
// Some calls will be dropped in order to limit update frequency
//
// Usage:
//
// var throttle = Throttle().withScope(scope).withDelay(1000);  // If a scope is passed, life time of monofuture = life time of the scope
//
// throttle.exec(func1); // Executed now (= after the current event loop)
// throttle.exec(func2); // Dropped (because of func3)
// throttle.exec(func3); // Dropped (because of func4)
// throttle.exec(func4); // Executed 1 second later
//
// It's also possible to permanently wrap a function :
// myFunc = Throttle().wrap(function() {
//    ...
// });
// myFunc(); // executed (= after the current event loop)
// myFunc(); // dropped
// myFunc(); // delayed
//
app.factory("Throttle", ["$timeout", function($timeout) {

    return function() {

        var delay = 0;
        var currentlyWaitingOn = null;
        var storedFunc = null;
        var destroyed = false;
        var scopeUnregisterer = null;

        var waitCallback = function() {
            var toBeExec;
            if (storedFunc) {
                toBeExec = storedFunc;
                storedFunc = null;
                currentlyWaitingOn = $timeout(waitCallback, delay);
            } else {
                currentlyWaitingOn = null;
            }
            // Re-entrant safe
            if (toBeExec) {
                toBeExec();
            }
        }
        var exec = function(func) {
            if (destroyed) {
                return;
            }
            if (!func) {
                func = function() {
                };
            }
            if (currentlyWaitingOn) {
                storedFunc = func;
                // It will be called later :)
            } else {
                // Execute now
                // ... and setup a timeout to drop further calls for 'delay' ms
                $timeout(func);
                storedFunc = null;
                currentlyWaitingOn = $timeout(waitCallback, delay);
            }
        };

        var wrap = function(func) {
            return function() {
                var args = arguments;
                exec(function() {
                    func.apply(null, args);
                });
            };
        };

        var abort = function() {
            if (currentlyWaitingOn) {
                $timeout.cancel(currentlyWaitingOn);
                currentlyWaitingOn = null;
            }
            storedFunc = null;
        };

        var destroy = function() {
            abort();
            destroyed = true;
        };


        var ret = {
            exec: function(func) {
                exec(func);
            },
            wrap: function(func) {
                return wrap(func);
            },
            wrapWatch: function(fn) {
                return wrapWatchHelper(ret, fn);
            },
            active: function() {
                return !!storedFunc;
            },
            abort: function() {
                abort();
            },
            destroy: function() {
                destroy();
            },
            withDelay: function(newDelay) {
                delay = newDelay;
                return ret;
            },
            withScope: function(scope) {
                if (scopeUnregisterer) {
                    scopeUnregisterer();
                }
                if (scope) {
                    scopeUnregisterer = scope.$on('$destroy', destroy);
                }
                return ret;
            }
        };
        return ret;
    };

}]);


app.factory("DKUtils", function($rootScope, $state, $stateParams, $timeout, Logger) {
    return {
        /* Reflows at current digest */
        reflowNow: function() {
            $rootScope.$broadcast("reflow");
        },

        /* Reflows at next digest */
        reflowNext: function() {
            $timeout(function() {
                $rootScope.$broadcast("reflow")
            }, 0);
        },
        /* It's probably bad if you need this */
        reflowLater: function() {
            $timeout(function() {
                Logger.info("delayed reflow");
                $rootScope.$broadcast("reflow")
            }, 400);
        },
        /* Reload current state. Works around broken $state.reload() */
        reloadState: function() {
            return $state.transitionTo($state.current,
                angular.copy($stateParams),
                {reload: true, inherit: true, notify: true});
        }
    }
});


app.factory("PluginsService", function($rootScope, DataikuAPI, WT1, StateUtils, $q, DataikuCloudService) {
    const namingConvention = '^[a-z][a-z0-9-]*$';

    const isUniqueInPlugin = function(newComponentId, pluginId, pluginComponentsOfSameType) {
        return !pluginComponentsOfSameType.some(existingComponent => existingComponent.id == newComponentId
                               || `${pluginId}_${existingComponent.id}` === newComponentId
                               || existingComponent.id === `${pluginId}_${newComponentId}`
                               || `${pluginId}_${existingComponent.id}` === `${pluginId}_${newComponentId}`);
    }

    const validateComponentId = function(newComponentId, pluginId, pluginComponentsOfSameType) {
        if (!newComponentId) return [];
        const warnings = [];
        if (!new RegExp(namingConvention).test(newComponentId)) {
            warnings.push("Component ID is invalid. It should start with a letter and continue with letters, numbers or hyphens.");
        }
        if (newComponentId.startsWith(pluginId)) {
            warnings.push("Component ID starts with plugin ID.");
        }
        if (!isUniqueInPlugin(newComponentId, pluginId, pluginComponentsOfSameType)) {
            warnings.push("Component ID is not unique.");
        }
        return warnings;
    }

    const isValidComponentId = function(newComponentId, pluginId, pluginComponentsOfSameType) {
        if (!newComponentId) return false;
        return validateComponentId(newComponentId, pluginId, pluginComponentsOfSameType).length == 0;
    };

    const getLoadedDescForDataset = function(svc, datasetType) {
        return svc.getDatasetLoadedDesc(datasetType) ||
            svc.getFSProviderLoadedDesc(datasetType) ||
            svc.getSampleLoadedDesc(datasetType);
    };

    var svc = {
        namingConvention: namingConvention,
        transformToDevPlugin: function(modalScope, convertAPIFunc, getAPICallParams, eventWT1Name, componentType, originalType) {
            DataikuAPI.plugindev.list().success(function(data) {
                modalScope.devPlugins = data;
            }).error(setErrorInScope.bind(modalScope));

            modalScope.convert = {
                mode: 'NEW',
                pattern: namingConvention
            };

            const noErrorMessage = {"display": false, "message": ""};
            const mustNotStartWithPluginId = {"display": true, "message": "New plugin recipe id should not start with the plugin id."};
            const mustBeUnique = {"display": true, "message": "New plugin recipe id must be unique across the recipe components of this plugin."};
            modalScope.matchesTargetFolder = function(pluginId) {
                return modalScope.convert.targetFolder.startsWith(pluginId);
            }
            modalScope.matchesOtherComponent = function(pluginId) {
                const plugin = modalScope.devPlugins.find(_ => _.desc.id === pluginId);
                return !isUniqueInPlugin(modalScope.convert.targetFolder, pluginId, plugin.content[componentType])
            }

            modalScope.modalWarning = function() {
            /**
            we add a special warning to explicitly say what the problem is, but only when it may be hard to understand
            **/
                if (modalScope.convert.mode === 'EXISTING') {
                    if (modalScope.convert.targetFolder && modalScope.matchesTargetFolder(modalScope.convert.targetPluginId)) {
                        return mustNotStartWithPluginId;
                    }
                    if (modalScope.convert.targetPluginId && modalScope.matchesOtherComponent(modalScope.convert.targetPluginId)) {
                        return mustBeUnique;
                    }
                } else {
                    if (modalScope.convert.targetFolder && modalScope.matchesTargetFolder(modalScope.convert.newPluginId)) {
                        return mustNotStartWithPluginId;
                    }
                }
                return noErrorMessage;
            };

            modalScope.isIdValid = function() {
                if (modalScope.convert.mode === 'EXISTING') {
                    if (!modalScope.convert.targetPluginId) return false;
                    const plugin = modalScope.devPlugins.find(_ => _.desc.id === modalScope.convert.targetPluginId);
                    return isValidComponentId(modalScope.convert.targetFolder,
                                                            modalScope.convert.targetPluginId,
                                                            plugin.content[componentType]);
                }
                if (!modalScope.convert.newPluginId) return false;
                return isValidComponentId(modalScope.convert.targetFolder,
                                                        modalScope.convert.newPluginId,
                                                        []);
            };

            modalScope.go = function() {
                resetErrorInScope(modalScope);
                convertAPIFunc(...getAPICallParams(modalScope)).success(function(data) {
                    WT1.event(eventWT1Name, {original: originalType});
                    modalScope.reloadPluginConfiguration();
                    StateUtils.go.pluginEditor(data.pluginId, data.relativePathToOpen);
                }).error(setErrorInScope.bind(modalScope));
            };
        },
        validateComponentId: validateComponentId,
        isValidComponentId: isValidComponentId,
        getPluginDesc(pluginId) {
            return $rootScope.appConfig.loadedPlugins.find(x => x.id == pluginId);
        },
        isPluginLoaded: function(pluginId) {
            var i, plugin;
            for (i = 0; i < $rootScope.appConfig.loadedPlugins.length; i++) {
                plugin = $rootScope.appConfig.loadedPlugins[i];
                if (plugin.id == pluginId) return true;
            }
            return false;
        },
        getSampleLoadedDesc : function(datasetType) {
            return $rootScope.appConfig.customSampleDatasets.find(x => x.datasetType === datasetType);
        },
        getDatasetLoadedDesc : function(datasetType) {
            return $rootScope.appConfig.customDatasets.find(x => x.datasetType == datasetType);
        },
        getFSProviderLoadedDesc : function(fsProviderType) {
            return $rootScope.appConfig.customFSProviders.find(x => x.fsProviderType == fsProviderType);
        },
        getRecipeLoadedDesc : function(recipeType) {
            return $rootScope.appConfig.customCodeRecipes.find(x => x.recipeType == recipeType);
        },
        getOwnerPluginDesc: function(loadedDesc) {
            if (loadedDesc != null) {
                return $rootScope.appConfig.loadedPlugins.find(x => x.id == loadedDesc.ownerPluginId);
            } else {
                return null; // plugin most likely removed
            }
        },
        getRecipeIcon: function(recipeType) {
            var loadedDesc = svc.getRecipeLoadedDesc(recipeType);
            if (loadedDesc && loadedDesc.desc && loadedDesc.desc.meta && loadedDesc.desc.meta.icon) {
                return loadedDesc.desc.meta.icon;
            } else {
                var pluginDesc = svc.getOwnerPluginDesc(loadedDesc);
                if (pluginDesc) {
                    return pluginDesc.icon || "icon-visual_prep_sync_recipe";
                } else {
                    return "icon-visual_prep_sync_recipe"; // plugin has been removed
                }
            }
        },
        getDatasetIcon: function(datasetType) {
            const loadedDesc = getLoadedDescForDataset(svc, datasetType);
            const defaultSampleDatasetIcon = "fas fa-flask";

            if (loadedDesc == null) {
                return datasetType.startsWith("Sample_") ? defaultSampleDatasetIcon : "icon-question-sign";
            }

            return loadedDesc.desc?.meta?.icon ||
                svc.getOwnerPluginDesc(loadedDesc)?.icon ||
                (datasetType.startsWith("Sample_") ? defaultSampleDatasetIcon : "icon-puzzle-piece");
        },
        getDatasetLabel : function(datasetType) {
            return getLoadedDescForDataset(svc, datasetType)?.desc?.meta?.label || datasetType;
        },
        checkInstalledStatus: function(pluginId, forceFetch) {
            return $q((resolve, reject) => {
                DataikuAPI.plugins.list(forceFetch)
                .success(function(data) {
                    const isInstalled = data.plugins.some(plugin => plugin.id === pluginId && plugin.installed);
                    resolve(isInstalled);
                })
                .error(reject)
            });
        },
        getPluginVersion: function(pluginId, forceFetch) {
            return $q((resolve, reject) => {
                DataikuAPI.plugins.list(forceFetch)
                .success(function(data) {
                    const plugin = data.plugins.find(plugin => plugin.id === pluginId);
                    if (plugin && plugin.installedDesc && plugin.installedDesc.desc) {
                        resolve(plugin.installedDesc.desc.version);
                    } else {
                        reject(undefined);
                    }
                })
                .error(reject)
            });
        },
        /**
         * Get the link to the plugin details store listing of the plugin with the specified ID
         * @param {String} pluginId
         * @param {Boolean} isPluginInstalled
         * @param {Boolean} isAppDeploymentCloud
         * @returns {String} If DSS cloud is detected the listing will be the public wordpress info page. Otherwise the local store listing will be returned\
         */
        getPluginDetailsLink: function(pluginId, isPluginInstalled, isAppDeploymentCloud) {
            if (isAppDeploymentCloud) return `https://www.dataiku.com/product/plugins/${pluginId}/`;
            if (isPluginInstalled) return `plugins/${pluginId}/summary/`;
            return `/plugins-explore/store/${pluginId}`;
        },
        /**
         * Get the link to the plugin store depending on how DSS is deployed (Cloud and On Prem.)
         */
        getPluginStoreLink: function(cloudInfo = {}, pluginName = '') {
            if (cloudInfo.isDataikuCloud && cloudInfo.isSpaceAdmin) {
                return DataikuCloudService.getLaunchpadUrl() + '/plugins/';
            }

            return (cloudInfo.isDataikuCloud ? 'https://www.dataiku.com/product/plugins/' : StateUtils.href.pluginStore()) + pluginName;
        }
    };
    return svc;
});


app.factory("PluginConfigUtils", function() {
    return {
        setDefaultValues: function(params, customConfig) {
            if (!customConfig) {
                return;
            }
            params.forEach(function(param) {
                if (customConfig[param.name] === undefined) {
                    if (param.defaultValue) {
                        // the type is not checked, so if the default is not of the right type, strange things can happen
                        customConfig[param.name] = param.defaultValue;
                    } else if (param.type == 'BOOLEAN') {
                        customConfig[param.name] = false;
                    } else if (param.type == 'INT' || param.type == 'DOUBLE') {
                        customConfig[param.name] = 0;
                    } else if (param.type == 'MAP') {
                        customConfig[param.name] = {};
                    } else if (param.type == 'KEY_VALUE_LIST') {
                        customConfig[param.name] = [];
                    } else if (param.type == 'ARRAY') {
                        customConfig[param.name] = [];
                    } else if (param.type == 'OBJECT_LIST') {
                        customConfig[param.name] = [];
                    }
                }
            });
        },
        shouldComponentBeVisible: function (loadedPlugins, selectedComponentId, componentIdGetter = (c) => c.id) {
            return (component) => {
                const ownerPlugin = loadedPlugins.find(p => p.id === component.ownerPluginId);

                const isHiddenByPlugin = ownerPlugin && ownerPlugin.hideComponents;
                const isSelected = selectedComponentId && selectedComponentId === componentIdGetter(component);

                return !isHiddenByPlugin || isSelected;
            };
        }
    }
});


app.factory("FutureProgressUtils", function() {
    var svc = {
        getTotalProgressPercentage: function(progress) {
            var percentage = 0;
            var fractionOf = 100;
            if (progress && progress.states) {
                angular.forEach(progress.states, function(state) {
                    if (state.target > 0) {
                        fractionOf = fractionOf / (state.target + 1);
                        percentage += fractionOf * state.cur;
                    }
                });
            }
            return percentage;
        }
    }
    return svc;
})

// Store large things, and give them a unique ID. It's backed by a very small LRU cache, and its purpose is
// to overcome the limitations of $stateParams when trying to pass big data in the URL
app.factory("BigDataService", ['$cacheFactory', function($cacheFactory) {
    var cache = $cacheFactory('BigDataServiceCache', {
        number: 20
    });
    return {
        store: function(bigdata) {
            var id = generateRandomId(10);
            cache.put(id, {data: bigdata});
            return id;
        },
        fetch: function(id) {
            var val = cache.get(id);
            if (!val) {
                return undefined;
            }
            return val.data;
        }
    };
}]);


app.factory("SQLExplorationService", ['$rootScope', '$cacheFactory', '$q', '$stateParams', 'Logger', 'MonoFuture', 'DataikuAPI',
            function($rootScope, $cacheFactory, $q, $stateParams, Logger, MonoFuture, DataikuAPI) {

    var cache = $cacheFactory('SQLExplorationServiceCache', {
        number: 50
    });

    function _cachedDataFetcher(promiseFunction, loggedName) {
       let runningPromise = {};
       return function(cacheKey, args) {
           if (cache.get(cacheKey)) {
               Logger.info("Loaded " + loggedName + " from cache");
               return $q.when(cache.get(cacheKey));
           }
           if (runningPromise[cacheKey]) {
               Logger.info("Loading " + loggedName + " leveraging already initiated promise");
               return runningPromise[cacheKey].promise;
           }
           const deferred = $q.defer();
           runningPromise[cacheKey] = deferred;

           Logger.info("Loading " + loggedName + " from backend");
           promiseFunction(...args)
               .success(function(data) {
                   cache.put(cacheKey, data);
                   deferred.resolve(data);
               })
               .error((data, status, headers) => {
                   setErrorInScope.bind($rootScope)(data, status, headers)
                   deferred.reject();
               })
               .finally(function() {
                   runningPromise[cacheKey] = null;
               });
           return deferred.promise;
       }
    }
    const listFieldDataFetcher = _cachedDataFetcher(DataikuAPI.connections.listSQLFields, "fields");
    const listTablesFromProjectDataFetcher = _cachedDataFetcher(DataikuAPI.connections.listSQLTablesFromProject, "project tables");
    const listTablesDataFetcher = _cachedDataFetcher(MonoFuture().wrap(DataikuAPI.connections.listSQLTables), "tables");

    var sep = '^~\n$* \\#';

    function connectionKey(connectionName) {
        return 'CNX' + sep + connectionName + sep;
    }

    function tableKey(projectKey, connection, schema, table) {
        if (!schema) {
            schema = '';
        }
        return 'TBL' + sep + schema + sep + table + sep + connection + sep + projectKey;
    }

    function listFieldsForTable(connection, table, projectKey) {
        const cacheKey = tableKey(projectKey, connection, table.schema, table.table);
        return listFieldDataFetcher(cacheKey, [connection, [table], projectKey]);
    }

    return {
        getCacheId: function(connectionName, projectKey, catalog, schema) {
            let id = connectionKey(connectionName) + '__all__' + projectKey;
            if (catalog) {
                id += '__catalog__' + catalog
            }
            if (schema) {
                id += '__schema__' + schema
            }
            return id;
        },
        getCacheIdFromProject: function(connectionName, projectKey) {
            return connectionKey(connectionName) + '__fromProject__' + projectKey;
        },
        isTableInCache: function(id) {
            return cache.get(id);
        },
        listTables: function(connectionName, projectKey, catalog, schema) {
            Logger.info(`Listing tables connection=${connectionName}, projectKey=${projectKey}, catalog=${catalog}, schema=${schema}`);
            const cacheKey = this.getCacheId(connectionName, projectKey, catalog, schema);
            return listTablesDataFetcher(cacheKey, [connectionName, projectKey, catalog, schema]).then(function(future) {
                return future.result;
            });
        },
        listTablesFromProject: function(connectionName, projectKey) {
            const cacheKey = this.getCacheIdFromProject(connectionName, projectKey);
            return listTablesFromProjectDataFetcher(cacheKey, [connectionName, projectKey]);
        },
        clearCache: function() {
            cache.removeAll();
        },
        listFields: function(connectionName, tables) {
            var promises = [];
            for (var i = 0; i < tables.length; i++) {
                promises.push(listFieldsForTable(connectionName, tables[i], $stateParams.projectKey));
            }
            var deferred = $q.defer();
            $q.all(promises).then(function(results) {
                var out = [];
                for (var i in results) {
                    out = out.concat(results[i]);
                }
                deferred.resolve(angular.copy(out));
            });
            return deferred.promise;
        }
    };
}]);


app.factory("SmartId", function($stateParams) {
    return {
        create: function(id, contextProjectKey) {
            if (contextProjectKey == $stateParams.projectKey) {
                return id;
            } else {
                return contextProjectKey + "." + id;
            }
        },

        resolve: function(smartId, contextProject) {
            if (contextProject === undefined) {
                contextProject = $stateParams.projectKey
            }
            if (smartId && smartId.indexOf(".") > 0) {
                var chunks = smartId.split(".");
                return {projectKey: chunks[0], id: chunks[1]}
            } else {
                return {projectKey: contextProject, id: smartId};
            }
        },

        fromRef: function(smartObjectRef, contextProject) {
            if (contextProject === undefined) {
                contextProject = $stateParams.projectKey
            }
            if (smartObjectRef.objectType == 'PROJECT') {
                return smartObjectRef.objectId;
            }
            if (!smartObjectRef.projectKey || !smartObjectRef.projectKey.length || smartObjectRef.projectKey == contextProject) {
                return smartObjectRef.objectId;
            } else {
                return smartObjectRef.projectKey + "." + smartObjectRef.objectId;
            }
        },

        fromTor: function(tor, contextProject) {
            if (contextProject === undefined) {
                contextProject = $stateParams.projectKey
            }
            if (tor.taggableType == 'PROJECT') {
                return tor.id;
            }
            if (!tor.projectKey || !tor.projectKey.length || tor.projectKey == contextProject) {
                return tor.id;
            } else {
                return tor.projectKey + "." + tor.id;
            }
        }
    }
});


app.service("FeatureNameUtils", function($filter, Fn) {

    /*
     Returns an object with :
     .elements, an array of strings representing
     consecutive elements of processed the feature name.
     .isCode, an array of booleans with same length indicating whether the corresponding
     element should be treated as "code"
     .value, an optional value for the feature. Defaults to null.
     .operator, the operator describing the operation when there is a value. Defaults to null.
     .no_operator, the inverse operator. Defaults to null.
     */
    var getAsElements = function(input, asHtml) {
        if (asHtml === undefined) {
            asHtml = false;
        }
        if (input == null) {
            input = "";
        }
        // Formatting
        var esc = asHtml ? $filter('escapeHtml') : Fn.SELF;
        var code = [];
        var els = [];
        var value = null, rawValue = null;
        var operator = null;
        var no_operator = null;
        var type = null;

        var addCode = function(c) {
            code.push(true);
            els.push(esc(c));
        };

        var addText = function(c) {
            code.push(false);
            els.push(esc(c));
        };

        let match;
        const elts = input.split(":");
        if ( (match = input.match(/^dummy:([^:]+):(.*)/)) ) {
            addCode(match[1]);
            switch (match[2].trim()) {
                case '':
                    value = "empty";
                    operator = "is";
                    no_operator = "is not";
                    break;
                case '__Others__':
                    value = "other";
                    operator = "is";
                    no_operator = "is not";
                    break;
                default:
                    value = esc(match[2]);
                    operator = "is";
                    no_operator = "is not";
            }
        } else if ( (match = input.match(/^(?:thsvd|hashvect):(.+):(\d+)$/)) ) {
            addCode(match[1]);
            addText('[text #' + match[2] + ']');
        } else if ( (match = input.match(/^unfold:([^:]+):(.*)$/)) ) {
            addCode(match[1]);
            addText('[element #' + match[2] + ']');
        } else if (input.startsWith("impact:")) {
            /*reg: impact:ft:all_target_values*/
            /*multi: impact:ft:target_value:<target_value>*/
            if (elts.length == 4) {
                addCode(elts[1]);
                addText('[impact #' + elts[3] + ']');
            } else {
                addCode(elts[1]);
                addText("[impact on target]");
            }
        } else if (input.startsWith("glmm:")) {
            /*reg: glmm:ft:all_target_values*/
            /*multi: glmm:ft:target_value:<target_value>*/
            if (elts.length == 4) {
                addCode(elts[1]);
                addText('[glmm #' + elts[3] + ']');
            } else {
                addCode(elts[1]);
                addText("[glmm on target]");
            }
        } else if (input.startsWith("frequency:")) {
            /*frequency: frequency:ft:frequency*/
            /*count: frequency:ft:count*/
            addCode(elts[1]);
            addText("[" + elts[2] + " encoded]");
        } else if (input.startsWith("ordinal:")) {
            /*lexicographic: ordinal:ft:lexicographic*/
            /*count: ordinal:ft:count*/
            addCode(elts[1]);
            addText("[ordinal encoded (" + elts[2] + ")]");
        } else if (input.startsWith("datetime_cyclical:")) {
            /* datetime_cyclical:ft:<period>:<cos|sin> */
            addCode(elts[1]);
            addText(`[${elts[2]} cycle (${elts[3]})]`);
        } else if ( (match = input.match(/^poly_int:(.*)$/)) ) {
            addCode(match[1]);
            addText("(computed)");
        } else if ( (match = input.match(/^pw_linear:(.*)$/)) ) {
            addCode(match[1]);
            addText("(computed)");
        } else if ( (match = input.match(/countvec:(.+):(.+)$/)) ) {
            addCode(match[1]);
            operator = 'contains';
            no_operator = "does not contain";
            value = match[2];
            type = "countvec";
        } else if ( (match = input.match(/tfidfvec:(.+):(.+):(.+)$/)) ) {
            addCode(match[1]);
            operator = 'contains';
            no_operator = "does not contain";
            value = match[3] + "(idf=" + match[2] + ")";
            rawValue = match[3];
            type = "tfidfvec";
        } else if(input.startsWith("hashing:")) {
            addCode(elts[1]);
            value = elts[2];
            operator = "hashes to";
            no_operator = "does not hash to";
            type = "hashing";
        } else if (input.startsWith("interaction")) {
            if (elts.length == 3) {
                addCode(elts[1]);
                addText("x");
                addCode(elts[2]);
            } else if (elts.length == 4) {
                addCode(elts[1]);
                addText("x");
                addCode(elts[2] + " = " + elts[3]);
            } else {
                addCode(elts[1] + " = " + elts[3]);
                addText("and");
                addCode(elts[2] + " = " + elts[4]);
            }
        } else {
            addCode(input);
        }

        return {
            elements: els,
            isCode: code,
            value: value,
            operator: operator,
            no_operator: no_operator,
            type : type,
            rawValue : rawValue
        };
    }

    var getAsHtmlString = function(feature) {
        var els = getAsElements(feature, true);
        var htmlArray = [];
        for (var i = 0; i < els.elements.length; i++) {
            if (els.isCode[i]) {
                htmlArray.push("<code>" + els.elements[i] + "</code>")
            } else {
                htmlArray.push(els.elements[i]);
            }
        }
        if (els.value != null) {
            htmlArray.push(els.operator, "<code>" + els.value + "</code>");
        }
        return htmlArray.join(" ");
    };

    var getAsTextElements = function(feature) {
        var els = getAsElements(feature);
        return {
            feature: els.elements.join(" "),
            operator: els.operator,
            no_operator: els.no_operator,
            value: els.value,
            type : els.type,
            rawValue : els.rawValue
        };
    };

    var getAsText = function(feature, negate) {
        if (negate === undefined) {
            negate = false;
        }
        var els = getAsElements(feature);
        if (els.value == null) {
            return els.elements.join(" ");
        } else {
            return els.elements.concat([negate ? els.no_operator : els.operator, els.value]).join(" ");
        }
    }

    return {
        getAsElements: getAsElements,
        getAsHtmlString: getAsHtmlString,
        getAsTextElements: getAsTextElements,
        getAsText: getAsText
    };

})

app.filter("getNameValueFromMLFeature", function(FeatureNameUtils) {
    return function(feature) {
        var els = FeatureNameUtils.getAsTextElements(feature);
        return {
            name: els.feature,
            value: els.value
        };
    }
});


app.factory("InfoMessagesUtils", function() {
    var svc = {
        /* Returns the first of the info messages with a given line, or null if there is none */
        getMessageAtLine: function(im, line) {
            if (!im || !im.messages) return null;
            for (var i = 0; i < im.messages.length; i++) {
                if (im.messages[i].line == line) return im.messages[i];
            }
            return null;
        },
        /* Filter the messages of all categories by line */
        filterForLine : function(im, line) {
            if (!im || !im.messages) return null;
            var fim = {};
            fim.messages = im.messages.filter(function(m) {return m.line == line;});
            fim.anyMessage = fim.messages.length > 0;
            fim.error = fim.messages.filter(function(m) {return m.severity == 'ERROR'}).length > 0;
            fim.warning = fim.messages.filter(function(m) {return m.severity == 'WARNING'}).length > 0;
            fim.maxSeverity = fim.error ? 'ERROR' : (fim.warning ? 'WARNING' : (fim.anyMessage ? 'INFO' : null));
            return fim;
        },
        getMessageAtColumn : function(im, column) {
            if (!im || !im.messages) return null;
            for (var i = 0; i < im.messages.length; i++) {
                if (im.messages[i].column == column) return im.messages[i];
            }
            return null;
        }
    }
    return svc;
});


app.factory("MessengerUtils", function($sanitize) {
    Messenger.options = {
        extraClasses: 'messenger-fixed messenger-on-bottom messenger-on-right',
        theme: 'dss'
    };

    const svc = {
        post: function(options) {
            let msg = null;
            options.actions = options.actions || {};
            if (options.showCloseButton) {
                options.actions.close = {
                    label: "Close",
                    action: function() {
                        msg.hide();
                    }
                };
                delete options.showCloseButton;
            }
            if (options.icon) {
                options.message = '<div style="width: 100%;"><div class="messenger-icon">' + $sanitize(options.icon) + '</div>' + $sanitize(options.message) + '</div>'
                delete options.icon;
            } else {
                options.message = '<div style="width: 100%;">' + $sanitize(options.message) + '</div>'
            }
            msg = Messenger().post(options);
        }
    }
    return svc;
});


// Front-end equivalent of StringNormalizationMode.java
app.factory("StringNormalizer", function() {

    var inCombiningDiatricalMarks = /[\u0300-\u036F]/g;
    var punct = /!"#\$%&'\(\)\*\+,-\.\/:;<=>\?@\[\]\^_`\{\|\}~/g;

    var svc = {
        get: function(stringNormalizationMode) {
            switch(stringNormalizationMode) {
                case 'EXACT':
                    return function(str) {
                        return str;
                    };

                case 'LOWERCASE':
                    return function(str) {
                        return str.toLowerCase();
                    };

                case 'NORMALIZED':
                default:
                    return function(str) {
                        return svc.normalize(str);
                    };
            }
        },

        normalize: function(str) {
            return str.normalize('NFD').replace(inCombiningDiatricalMarks, '');
        },

        removePunct: function(str) {
            return str.replace(punct, '');
        }
    };

    return svc;
});


app.service('HiveService', function($rootScope, Dialogs, ActivityIndicator, DataikuAPI, ToolBridgeService, CreateModalFromTemplate, $q) {
    this.convertToImpala = function(selectedRecipes) {
        var deferred = $q.defer();
        //TODO @flow need a dedicated modal or rather a generic confirm modal that can have errors in scope
        Dialogs.confirm($rootScope, "Convert recipes to Impala", `Are you sure you want to convert ${selectedRecipes.length} Hive recipes to Impala?`).then(function() {
            DataikuAPI.flow.recipes.massActions.convertToImpala(selectedRecipes, true)
                .success(function() {
                    deferred.resolve("converted");
                }).error(function(a,b,c) {
                    deferred.reject("conversion failed");
                    setErrorInScope.bind($rootScope)(a,b,c);
                });
        }, function() {deferred.reject("user cancelled");});
        return deferred.promise;
    };

    this.resynchronizeMetastore = function(selectedDatasets) {
        Dialogs.confirmPositive($rootScope,
            'Hive metastore resynchronization',
            'Are you sure you want to resynchronize datasets to the Hive metastore?')
        .then(function() {
            ActivityIndicator.waiting('Synchronizing Hive metastore...');
            DataikuAPI.datasets.synchronizeHiveMetastore(selectedDatasets).success(function(data) {
                if (data.anyMessage && (data.warning || data.error)) {
                    ActivityIndicator.hide();
                    Dialogs.infoMessagesDisplayOnly($rootScope, "Metastore synchronization", data);
                } else {
                    // nothing to show
                    ActivityIndicator.success('Hive metastore successfully synchronized');
                }
            }).error(function(data, status, headers) {
                ActivityIndicator.hide();
                setErrorInScope.call($rootScope, data, status, headers);
            });
        });
    };

    this.startChangeHiveEngine = function(selectedRecipes) {
        return CreateModalFromTemplate('/templates/recipes/fragments/hive-engine-modal.html', $rootScope, null, function(modalScope) {
            modalScope.options = {executionEngine: 'HIVESERVER2'};

            DataikuAPI.flow.recipes.massActions.startSetHiveEngine(selectedRecipes).success(function(data) {
                modalScope.messages = data;
                ToolBridgeService.emitRefreshView('HiveModeView');
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setHiveEngine(selectedRecipes, modalScope.options.executionEngine)
                    .success(function() {
                        $rootScope.$emit('recipesHiveEngineUpdated');
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('ImpalaService', function($rootScope, Dialogs, CreateModalFromTemplate, ToolBridgeService, DataikuAPI, $q) {

    this.convertToHive = function(selectedRecipes) {
        var deferred = $q.defer();
        //TODO @flow need a dedicated modal or rather a generic confirm modal that can have errors in scope
        Dialogs.confirm($rootScope, "Convert recipes to Hive", `Are you sure you want to convert ${selectedRecipes.length} Impala recipes to Hive?`).then(function() {
            DataikuAPI.flow.recipes.massActions.convertToHive(selectedRecipes, true)
                .success(function() {
                    deferred.resolve("converted");
                }).error(function(a,b,c) {
                    deferred.reject("conversion failed");
                    setErrorInScope.bind($rootScope)(a,b,c);
                });
        }, function() {deferred.reject("user cancelled");});
        return deferred.promise;
    };

    this.startChangeWriteMode = function(selectedRecipes) {
        return CreateModalFromTemplate('/templates/recipes/fragments/impala-write-flag-modal.html', $rootScope, null, function(modalScope) {
            modalScope.options = {runInStreamMode: true};

            DataikuAPI.flow.recipes.massActions.startSetImpalaWriteMode(selectedRecipes).success(function(data) {
                modalScope.messages = data;
                ToolBridgeService.emitRefreshView('ImpalaWriteModeView');
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setImpalaWriteMode(selectedRecipes, modalScope.options.runInStreamMode)
                    .success(function() {
                        $rootScope.$emit('recipesImpalaWriteModeUpdated');
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('SparkService', function($rootScope, CreateModalFromTemplate, DataikuAPI) {
    this.startChangeSparkConfig = function(selectedItems) {

        return CreateModalFromTemplate('/templates/recipes/fragments/spark-config-modal.html', $rootScope, null, function(modalScope) {
            modalScope.selectedRecipes = selectedItems.filter(it => it.type == 'RECIPE');
            modalScope.options = {};

            DataikuAPI.flow.recipes.massActions.startSetSparkConfig(modalScope.selectedRecipes).success(function(data) {
                modalScope.messages = data;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setSparkConfig(modalScope.selectedRecipes, modalScope.options.sparkConfig)
                    .success(function() {
                        $rootScope.$emit('recipesSparkConfigUpdated');
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('PipelineService', function($rootScope, CreateModalFromTemplate, DataikuAPI) {
    this.startChangePipelineability = function(selectedItems, pipelineType) {
        return CreateModalFromTemplate('/templates/recipes/fragments/pipelineability-modal.html', $rootScope, null, function(modalScope) {
            modalScope.selectedRecipes = selectedItems.filter(it => it.type === 'RECIPE');

            modalScope.pipelineTypeText = (pipelineType === 'SPARK' ? 'Spark' : 'SQL');

            modalScope.options = {
                allowStart: true,
                allowMerge: true
            };

            DataikuAPI.flow.recipes.massActions.startSetPipelineability(modalScope.selectedRecipes, pipelineType).success(function(data) {
                modalScope.messages = data;
            }).error(function(...args) {
                modalScope.fatalError = true;
                setErrorInScope.apply(modalScope, args);
            });

            modalScope.ok = function() {
                DataikuAPI.flow.recipes.massActions.setPipelineability(modalScope.selectedRecipes, pipelineType, modalScope.options.allowStart, modalScope.options.allowMerge)
                    .success(function() {
                        modalScope.resolveModal();
                    })
                    .error(setErrorInScope.bind(modalScope));
            };
        });
    };
});


app.service('ColorPalettesService', function() {

    const svc = this;

    const DEFAULT_COLORS = [
        "#1ac2ab",
        "#0f6d82",
        "#FFD83D",
        "#de1ea5",
        "#9dd82b",
        "#28aadd",
        "#00a55a",
        "#d66b9b",
        "#77bec2",
        "#94be8e",
        "#123883",
        "#a088bd",
        "#c28e1a"
    ];

    svc.fixedColorsPalette = function(name, colors=DEFAULT_COLORS) {
        const colorMap = {};
        return function(key) {
            key = key + ''; //force conversion
            if (colorMap[key]) {
                return colorMap[key];
            }
            colorMap[key] = colors[Object.keys(colorMap).length % colors.length];
        };
    };

});


/*
 * TODO: finally this service is a bit of a duplicate of what CodeBasedEditorUtils was supposed to be. Would be good to merge both at some point...
 */
app.service('CodeMirrorSettingService', function($rootScope) {

    const INDENT_MORE_SHORTCUT = "Tab";
    const INDENT_LESS_SHORTCUT = "Shift-Tab";
    const FIND_SHORTCUT = "Ctrl-F";
    const REPLACE_SHORTCUT = "Ctrl-Alt-F";
    const JUMP_TO_LINE_SHORTCUT = "Ctrl-L";
    const TOGGLE_COMMENT_SHORTCUT_QWERTY = "Cmd-/";
    const TOGGLE_COMMENT_SHORTCUT_AZERTY = "Shift-Cmd-/";
    const TOGGLE_COMMENT_SHORTCUT_RSTUDIO = "Shift-Ctrl-C";
    const AUTOCOMPLETE_SHORTCUT = "Ctrl-Space";
    const FULL_SCREEN_SHORTCUT = "F11";

    this.getShortcuts = function() {
        return {
            "INDENT_MORE_SHORTCUT": INDENT_MORE_SHORTCUT,
            "INDENT_LESS_SHORTCUT": INDENT_LESS_SHORTCUT,
            "FIND_SHORTCUT": FIND_SHORTCUT,
            "REPLACE_SHORTCUT": REPLACE_SHORTCUT,
            "JUMP_TO_LINE_SHORTCUT": JUMP_TO_LINE_SHORTCUT,
            "TOGGLE_COMMENT_SHORTCUT": TOGGLE_COMMENT_SHORTCUT_QWERTY,
            "AUTOCOMPLETE_SHORTCUT": AUTOCOMPLETE_SHORTCUT,
            "FULL_SCREEN_SHORTCUT": FULL_SCREEN_SHORTCUT}
    }

    this.get = function(mimeType, options) {
        var extraKeys = {};

        if (!$rootScope.appConfig.userSettings.codeEditor || !$rootScope.appConfig.userSettings.codeEditor.keyMap || $rootScope.appConfig.userSettings.codeEditor.keyMap == "default") {
            extraKeys[INDENT_MORE_SHORTCUT] = "indentMore";
            extraKeys[INDENT_LESS_SHORTCUT] = "indentLess";
            extraKeys[FIND_SHORTCUT] = "find";
            extraKeys[REPLACE_SHORTCUT] = "replace";
            extraKeys[JUMP_TO_LINE_SHORTCUT] = "jumpToLine";
            extraKeys[TOGGLE_COMMENT_SHORTCUT_QWERTY] = "toggleComment";
            extraKeys[TOGGLE_COMMENT_SHORTCUT_AZERTY] = "toggleComment";
            extraKeys[TOGGLE_COMMENT_SHORTCUT_RSTUDIO] = "toggleComment";
            extraKeys[AUTOCOMPLETE_SHORTCUT] = this.showHint(mimeType, options && options.words ? options.words : []);
        }
        if (!options || !options.noFullScreen) {
            extraKeys[FULL_SCREEN_SHORTCUT] = function(cm) {
                if (cm.getOption("fullScreen")) {
                    cm.setOption("fullScreen", false);
                } else {
                    cm.setOption("fullScreen", !cm.getOption("fullScreen"));
                }
            };
        }


        var settings =  {
            mode: mimeType,
            theme: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.theme ? $rootScope.appConfig.userSettings.codeEditor.theme : 'default',

            //left column
            lineNumbers : true,
            foldGutter: true,
            gutters: ["CodeMirror-linenumbers", "CodeMirror-foldgutter"],

            //indentation
            indentUnit: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.indentUnit ? $rootScope.appConfig.userSettings.codeEditor.indentUnit : 4,
            tabSize: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.tabSize ? $rootScope.appConfig.userSettings.codeEditor.tabSize : 4,
            indentWithTabs: $rootScope.appConfig.userSettings.codeEditor ? $rootScope.appConfig.userSettings.codeEditor.indentWithTabs : false,

            //edition
            autoCloseBrackets: $rootScope.appConfig.userSettings.codeEditor ? $rootScope.appConfig.userSettings.codeEditor.autoCloseBrackets : true,
            autoCloseTags: $rootScope.appConfig.userSettings.codeEditor ? $rootScope.appConfig.userSettings.codeEditor.autoCloseTags : true,

            // readonly
            readOnly: options && options.readOnly,

            //code reading
            matchBrackets: true,
            matchTags: true,
            highlightSelectionMatches: true,
            styleSelectedText: true,
            styleActiveLine: true,

            keyMap: $rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.keyMap ? $rootScope.appConfig.userSettings.codeEditor.keyMap : 'default',
            extraKeys: extraKeys,
            onLoad: function(cm) {
                if ($rootScope.appConfig.userSettings.codeEditor && $rootScope.appConfig.userSettings.codeEditor.fontSize) {
                    $($(cm.getTextArea()).siblings('.CodeMirror')[0]).css('font-size', $rootScope.appConfig.userSettings.codeEditor.fontSize + 'px');
                }
                if(options && options.readOnly){
                    $($(cm.getTextArea()).siblings('.CodeMirror')[0]).css('opacity', '0.3');
                }
                if (options && options.onLoad && typeof(options.onLoad) == "function") {
                    options.onLoad(cm);
                }
            }
        };

        return settings;
    }

    this.showHint = function(mode, words) {
        return function(cm) {
            const modes = {
                'text/x-python': CodeMirror.hintWords.python,
                'text/css': CodeMirror.hintWords.css
            };

            CodeMirror.showHint(cm, function(editor) {
                const anyWordHint = CodeMirror.hint.anyword(cm);
                const recipeWords = words || [];
                const codeKeywords = CodeMirror.hintWords[mode] || modes[mode] || [];

                let combinedWords = [recipeWords, [' '], codeKeywords, [' '], anyWordHint && anyWordHint.list ? anyWordHint.list : []]
                    .reduce((a, b) => a.concat(b.filter(_ => _ === ' ' || !a.includes(_)))); // deduplicates

                /*
                    Filter functionality based off of https://github.com/amarnus/ng-codemirror-dictionary-hint/blob/master/lib/ng-codemirror-dictionary-hint.js
                */
                var cur = editor.getCursor();
                var curLine = editor.getLine(cur.line);
                var start = cur.ch;
                var end = start;
                while (end < curLine.length && /[\w$]/.test(curLine.charAt(end))) ++end;
                while (start && /[\w$]/.test(curLine.charAt(start - 1))) --start;
                var curWord = start !== end && curLine.slice(start, end);
                return {
                    list: (!curWord ? combinedWords : combinedWords.filter(_ => _.startsWith(curWord) && _ !== curWord)),
                    from: CodeMirror.Pos(cur.line, start),
                    to: CodeMirror.Pos(cur.line, end)
                }
            }, { completeSingle: false });
        };
    }
});

app.service('CodeEnvService', function(DataikuAPI, $stateParams, FutureProgressModal, Dialogs, Deprecation, CreateModalFromTemplate, $q, FutureWatcher, FeatureFlagsService) {
    const allInterpreters = [
        ["PYTHON36", "Python 3.6"],
        ["PYTHON37", "Python 3.7"],
        ["PYTHON38", "Python 3.8"],
        ["PYTHON39", "Python 3.9"],
        ["PYTHON310", "Python 3.10"],
        ["PYTHON311", "Python 3.11"],
        ["PYTHON312", "Python 3.12"],
        ["PYTHON313", "Python 3.13"],
        ["PYTHON314", "Python 3.14"],
        ["CUSTOM", "Custom (lookup in PATH)"]
    ];
    if (FeatureFlagsService.featureFlagEnabled('allowPython2CodeEnv')) {
        allInterpreters.unshift(["PYTHON27", "Python 2.7"])
    }

    const experimentalInterpreters = [
        "PYTHON314",
    ]

    const condaInterpreters = [
        ["PYTHON36", "Python 3.6"],
        ["PYTHON37", "Python 3.7"],
        ["PYTHON38", "Python 3.8"],
        ["PYTHON39", "Python 3.9"],
        ["PYTHON310", "Python 3.10"],
        ["PYTHON311", "Python 3.11"],
        ["PYTHON312", "Python 3.12"],
        ["PYTHON313", "Python 3.13"],
        ["PYTHON314", "Python 3.14"],
    ];
    if (FeatureFlagsService.featureFlagEnabled('allowPython2CodeEnv')) {
        condaInterpreters.unshift(["PYTHON27", "Python 2.7"])
    }

    const condaInterpretersWithDeprecationNotice = dkuDeepCopy(condaInterpreters);
    condaInterpretersWithDeprecationNotice.filter(interpreter => Deprecation.isPythonDeprecated(interpreter[0])).forEach(interpreter => interpreter[1] += " - Deprecated");

    let availableInterpreters;
    let enrichedInterpreters;

    function getAvailableInterpretersPromise() {
        const deferred = $q.defer();
        if (availableInterpreters) {
            deferred.resolve(availableInterpreters);
        } else {
            DataikuAPI.codeenvs.listAvailablePythonInterpreters().success(function (data) {
                availableInterpreters = data;
                // CUSTOM should always be available
                availableInterpreters.push("CUSTOM");
                deferred.resolve(availableInterpreters)
            });
        }
        return deferred.promise;
    }

    function enrichInterpreterDescription(availableInterpretersList) {
        const enrichedInterpreters = [];
        // make sure the list of available interpreters has been retrieved from backend
        for (const interpreter of allInterpreters) {
            const interpreterId = interpreter[0];
            const interpreterDesc = interpreter[1];
            if (!availableInterpretersList.includes(interpreterId)) {
                enrichedInterpreters.push([interpreterId, interpreterDesc + " - Not available"]);
            } else if (Deprecation.isPythonDeprecated(interpreterId)) {
                enrichedInterpreters.push([interpreterId, interpreterDesc + " - Deprecated"]);
            } else if (experimentalInterpreters.includes(interpreterId)) {
                enrichedInterpreters.push([interpreterId, interpreterDesc + " - Experimental"]);
            } else {
                enrichedInterpreters.push([interpreterId, interpreterDesc]);
            }
        }
        return enrichedInterpreters;
    }

    const svc = {
        pythonInterpreters: allInterpreters, // these interpreters can be used as default, but will not contain enrichments
        getPythonInterpreters: function() {
            if (enrichedInterpreters) {
                return $q.when(enrichedInterpreters);
            }
            return getAvailableInterpretersPromise()
                .then((interpreters) => {
                    enrichedInterpreters = enrichInterpreterDescription(interpreters);
                    return enrichedInterpreters;
                });
        },
        pythonCondaInterpreters: condaInterpretersWithDeprecationNotice,
        getPackagesThatAreBothRequiredAndInstalled : function(spec, actual) {
            if (!spec || !actual) {
                return new Set();
            }
            return svc.intersection(svc.getPackageNames(spec), svc.getPackageNames(actual));
        },
        intersection : function(setA, setB) {
            let intersection = new Set();
            for (let elt of setB) {
                if (setA.has(elt)) {
                    intersection.add(elt)
                }
            }
            return intersection;
        },
        difference : function(setA, setB) {
            let difference = new Set();
            for (let elt of setA) {
                if (!setB.has(elt)) {
                    difference.add(elt)
                }
            }
            return difference;
        },
        getPackagesThatAreRequired : function(codeEnv, packageSystem) {
            let spec = null;
            let mandatory = null;
            if (packageSystem == 'pip') {
                spec = codeEnv.specPackageList;
                mandatory = codeEnv.mandatoryPackageList;
            } else if (packageSystem == 'conda') {
                spec = codeEnv.specCondaEnvironment;
                mandatory = codeEnv.mandatoryCondaEnvironment;
            }
            let packages = !spec ? new Set() : svc.getPackageNames(spec);
            if (mandatory) {
                svc.getPackageNames(mandatory).forEach(p => packages.add(p));
            }
            return packages;
        },
        getPackageNames : function(listStr) {
            /** Gets the set of package names mentioned in a code env listing */
            let ret = new Set();
            for (let line of listStr.split("\n")) {
                // Ignore packages that are not clean package names
                if (line.indexOf("-e") >= 0 || line.indexOf("git+") >= 0 || line.indexOf("ssh") >= 0) continue;

                const chunks = line.split(/[\s>=<,[\]]+/);
                const packageName = chunks[0].trim().replaceAll('"',"");
                if (packageName.length > 0) {
                    ret.add(packageName.toLowerCase());
                }
            }
            return ret;
        },
        listLogs : function($scope){
            return DataikuAPI.admin.codeenvs.design.listLogs($scope.envLang, $stateParams.envName).then(function(response) {
                $scope.logs = response.data;
            }).catch(setErrorInScope.bind($scope));
        },
        makeCurrentDesc : function(desc) {
            return {
                yarnPythonBin: angular.copy(desc.yarnPythonBin),
                yarnRBin: angular.copy(desc.yarnRBin),
                owner: angular.copy(desc.owner),
                installJupyterSupport : desc.installJupyterSupport,
                installCorePackages : desc.installCorePackages,
                corePackagesSet : desc.corePackagesSet,
                envSettings : angular.copy(desc.envSettings),
                dockerImageResources: desc.dockerImageResources,
                updateResourcesApiNode: desc.updateResourcesApiNode,
                dockerfileAtStart: desc.dockerfileAtStart,
                dockerfileBeforePackages: desc.dockerfileBeforePackages,
                dockerfileAfterCondaPackages: desc.dockerfileAfterCondaPackages,
                dockerfileAfterPackages: desc.dockerfileAfterPackages,
                dockerfileAtEnd: desc.dockerfileAtEnd,
                containerCacheBustingLocation: desc.containerCacheBustingLocation,
                predefinedContainerHooks: angular.copy(desc.predefinedContainerHooks)
            };
        },
        makeCurrentSpec : function(spec) {
            return {
                specPackageList: angular.copy(spec.specPackageList),
                desc: svc.makeCurrentDesc(spec.desc),
                specCondaEnvironment: angular.copy(spec.specCondaEnvironment),
                permissions: angular.copy(spec.permissions),
                usableByAll: spec.usableByAll,
                allContainerConfs: angular.copy(spec.allContainerConfs),
                containerConfs: angular.copy(spec.containerConfs),
                allSparkKubernetesConfs: angular.copy(spec.allSparkKubernetesConfs),
                sparkKubernetesConfs: angular.copy(spec.sparkKubernetesConfs),
                rebuildDependentCodeStudioTemplates: angular.copy(spec.rebuildDependentCodeStudioTemplates),
                resourcesInitScript: angular.copy(spec.resourcesInitScript)
                };
        },
        refreshEnv : function($scope) {
            const getEnvPromise = DataikuAPI.admin.codeenvs.design.get($scope.envLang, $stateParams.envName).then(function(response) {
                const codeEnv = response.data;
                $scope.codeEnv = codeEnv;
                $scope.codeEnv.pythonInterpreterFriendlyName = svc.pythonInterpreterFriendlyName($scope.codeEnv.desc.pythonInterpreter);
                $scope.previousSpec = svc.makeCurrentSpec(codeEnv);
                $scope.previousPackagesSetForRemovalWarning['pip'] = svc.getPackagesThatAreBothRequiredAndInstalled(codeEnv.specPackageList, codeEnv.actualPackageList);
                $scope.previousPackagesSetForRemovalWarning['conda'] = svc.getPackagesThatAreBothRequiredAndInstalled(codeEnv.specCondaEnvironment, codeEnv.actualCondaEnvironment);
            }).catch(setErrorInScope.bind($scope));
            const listLogsPromise = svc.listLogs($scope);
            return $q.all([getEnvPromise, listLogsPromise]);
        },
        updateEnv : function($scope, envName, upgradeAllPackages, updateResources, forceRebuildEnv, envLang, callback){
            var updateSettings = {
                upgradeAllPackages: upgradeAllPackages,
                updateResources: $scope.uiState.updateResources,
                forceRebuildEnv: $scope.uiState.forceRebuildEnv
            }

            var buildTemplates = function(templatesToRebuild) {
                DataikuAPI.codeStudioTemplates.massBuild(templatesToRebuild)
                    .then(response => FutureProgressModal.show($scope, response.data, "Build Code Studio templates", undefined, 'static', 'false', true)
                        .then(result => {
                            if (result) {
                                Dialogs.infoMessagesDisplayOnly($scope, "Build Code Studio templates result", result.messages, result.futureLog);
                            }
                        })
                    ).catch(function(a,b,c){
                        setErrorInScope.bind($scope)(a,b,c);
                    });
            }

            var doUpdateEnv = function(templatesToRebuild) {
                DataikuAPI.admin.codeenvs.design.update($scope.envLang, $stateParams.envName, updateSettings).success(data => {
                    FutureProgressModal.show($scope, data, "Env update", undefined, 'static', false)
                        .then(result => {
                            if (result) {
                                function buildTemplatesIfNeeded() {
                                    if (templatesToRebuild) {
                                        buildTemplates(templatesToRebuild);
                                    }
                                };
                                if(result.messages.maxSeverity == 'ERROR') { // 'ERROR' also in case of abort
                                    Dialogs.infoMessagesDisplayOnly($scope, "Update Code Env result", result.messages, result.futureLog);
                                    // no template building if error
                                } else if(result.messages.maxSeverity == 'WARNING') {
                                    // warning is not serious enough to imply no template building
                                    Dialogs.infoMessagesDisplayOnly($scope, "Update Code Env result", result.messages, result.futureLog).then(buildTemplatesIfNeeded);
                                } else {
                                    buildTemplatesIfNeeded();
                                }
                            }
                            svc.refreshEnv($scope);
                            $scope.$broadcast('refreshCodeEnvResources', {envName: $scope.codeEnv.envName});
                            if (callback) {
                                callback();
                            }
                        })
                }).error(setErrorInScope.bind($scope));
            }

            if (["ASK", "ALL"].includes($scope.codeEnv.rebuildDependentCodeStudioTemplates)){
                let getOnlyGlobalUsages = true; // We only need usages in CodeStudio templates
                DataikuAPI.admin.codeenvs.design.listUsages($scope.codeEnv.envLang, $scope.codeEnv.envName, getOnlyGlobalUsages).success(usagesList => {
                    const templatesToRebuild = usagesList
                        .filter(u => u.envUsage === 'CODE_STUDIO_TEMPLATE' && u.projectKey === '__DKU_ANY_PROJECT__' && u.objectId !== 'OBJECT_NOT_ACCESSIBLE') // will be empty if you don't have the rights
                        .map(u => u.objectId);
                    if ($scope.codeEnv.rebuildDependentCodeStudioTemplates === 'ASK' && templatesToRebuild.length > 0){
                        CreateModalFromTemplate("/templates/admin/code-envs/design/code-studio-rebuild-modal.html", $scope, null, newScope => {
                            newScope.templatesToRebuild = templatesToRebuild;
                            newScope.rebuildAll = () => {
                                newScope.dismiss();
                                doUpdateEnv(templatesToRebuild);
                            }
                            newScope.rebuildNone = () => {
                                newScope.dismiss();
                                doUpdateEnv();
                            }
                        });
                    } else {
                        doUpdateEnv(templatesToRebuild);
                    }
                }).error(() => {
                    setErrorInScope.bind($scope);
                    doUpdateEnv();
                });
            } else {
                doUpdateEnv();
            }
        },
        pythonInterpreterFriendlyName: function(pythonInterpreter) {
            let interpreter = allInterpreters.filter(interpreter => interpreter[0] == pythonInterpreter);
            if (interpreter.length) {
                return interpreter[0][1];
            } else {
                return pythonInterpreter;
            }
        }
    }
    return svc;
});

app.service('TimingService', function($rootScope) {
    return {
        wrapInTimePrinter: function(prefix, fn) {
            return function() {
                const before = performance.now();
                const retval = fn.apply(this, arguments);
                const after = performance.now();
                // eslint-disable-next-line no-console
                console.info("Timing: " + prefix + ": " + (after - before) + "ms");
                return retval;
            }
        }
    };
});

app.service('PromiseService', function() {
    const svc = this;

    /**
     * Wrap a $q promise in a $http promise (in order to keep the .success and .error methods)
     */
    svc.qToHttp = function(p) {
        return {
            then: p.then,
            success: function (fn) {
                return svc.qToHttp(p.then(fn));
            },
            error: function (fn) {
                return svc.qToHttp(p.then(null, fn));
            }
        }
    }

})

app.service('ProjectStatusService', function(TaggingService, $rootScope)  {
    const svc = this;
    let projectStatusMap = {};

    svc.getProjectStatusColor = function(status) {
        if(projectStatusMap && projectStatusMap[status]) {
            return projectStatusMap[status];
        } else {
            return TaggingService.getDefaultColor(status);
        }
    }

    function computeProjectStatusMap() {
        projectStatusMap = {};
        if ($rootScope.appConfig && $rootScope.appConfig.projectStatusList) {
            $rootScope.appConfig.projectStatusList.forEach(function(projectStatus) {
                projectStatusMap[projectStatus.name] = projectStatus.color;
            });
        }
    }
    $rootScope.$watch('appConfig.projectStatusList', computeProjectStatusMap, true);
});


app.service('ProjectFolderService', function(DataikuAPI, $q) {
    const svc = this;
    const ROOT_ID = 'ROOT';

    /**
     * Maps from ProjectFolderSummary to browse node information consisting
     * - children - an array of folder ids
     * - pathElts - an array of folder ids, from the root down to and including this node
     * - exists - always true
     * - directory - always true
     */
    svc.buildBrowseNode = function(folder) {
        const folders = folder.children.map(f => angular.extend({}, f, { directory: true, fullPath: f.id }))
        const pathElts = treeToList(folder, item => item.parent);
        return({
            children: folders,
            pathElts: pathElts.map(f => angular.extend({}, f, { toString: () => f.id })),
            exists: true,
            directory: true,
        });
    };

    /*
     * Adds pathElts to ProjectFolderSummary, a string like /Sandbox/marketing/summer
     */
    svc.addPathToFolder = (folder) => {
        const pathElts = treeToList(folder, item => item.parent);
        return angular.extend({}, folder, { pathElts: pathElts.map(f => f.name).join('/') });
    };

    // Return (a promise which returns) a ProjectFolderSummary object for the given folderId with minimal info
    svc.getFolder = function(folderId) {
        const requestFolderId = folderId || ROOT_ID;
        return DataikuAPI.projectFolders.getSummary(requestFolderId, 1, true, true).then(resp => {
            return resp.data;
        });
    };

    // Returns a function which sets the given path on the root folder if necessary
    const setRootPath = function(rootPath) {
        return function(folder) {
            if (folder.id === ROOT_ID) {
                folder.pathElts = rootPath;
            }
            return folder;
        }
    }

    svc.getDefaultFolder = function(folderId) {
        const requestFolderId = folderId || ROOT_ID;
        return DataikuAPI.projectFolders.getDefaultFolderSummary(requestFolderId).then(resp => {
            return resp.data;
        });
    };

    svc.getBrowseNode = function(folderId) {
        return svc.getFolder(folderId).then(svc.buildBrowseNode);
    }

    svc.getFolderWithPath = function(folderId) {
        return svc.getFolder(folderId).then(svc.addPathToFolder).then(setRootPath('/'));
    }

    // Return (a promise which returns) a folder with a path (in pathElts) to be used in a folder-path-input
    // If the folder cannot be found (or is not visible to the user) we fallback to the root folder
    // rootFolderDisplayText is the text or path to display when we fallback to the root
    svc.getFolderFallbackRoot = function(folderId, rootFolderDisplayText) {
        const deferred = $q.defer();
        this.getFolderWithPath(folderId).then((folder) => {
            deferred.resolve(folder);
        }).catch(() => {
            this.getFolder('').then(setRootPath(rootFolderDisplayText)).then((folder) => {
                deferred.resolve(folder);
            });
        });
        return deferred.promise;
    }

    // Return (a promise which returns) a folder (ProjectFolderSummary) with a path (pathElts) for initialising folder creation or duplication dialogs
    // If we are in the root folder, currentFolderId will be an empty string - meaning we will fallback to the default configured
    svc.getDefaultFolderForNewProject = function(currentFolderId) {
         // set rootpath to '/' to match what is added in browseDoneFn in folderPathInput (when show-root-folder-path="true")
        return svc.getDefaultFolder(currentFolderId).then(svc.addPathToFolder).then(setRootPath('/'));
    }

    svc.isInRoot = function(folderId) {
        return folderId === ROOT_ID;
    }

    // Return true if a folder is the SANDBOX folder or any SUBFOLDER of it
    svc.isUnderSandbox = function(folder) {
        if (!folder) {
            return false;
        }

        return folder.id === 'SANDBOX' || !!(folder.parent && svc.isUnderSandbox(folder.parent));
    }

});

/**
 * Enhance fattable elements with dragging capabilities.
 * Mandatory class fat-draggable should be added for parent draggable zone.
 * Mandatory class fat-draggable__item should be added for each items that can be dragged.
 * Mandatory class fat-draggable__handler should be added to trigger drag. It should be a child of an item or the item itself.
 * Mandatory data-column-name attribute should be added at fat-draggable__item level to keep a reference to the column when DOM is being recycled.
 *
 * To enable drag on an element, call the setDraggable() method with the following options :
 *
 * @param {Object}              options                         - The available options
 * @param {HTMLElement}         options.element                 - (Mandatory) element containing the draggable items.
 * @param {String}              [options.axis="x"]              - Define the dragging axis. Default to horizontal dragging.
 * @param {Function}            [options.onDrop]                - Drop callback
 * @param {Function}            [options.onPlaceholderUpdate]   - Placeholder dimensions update. Use it to reshape / position the placeholder. Called with the placeholder dimensions
 * @param {ScrollBarProxy}      [options.scrollBar]             - Fattable scrollbar to be updated if necessary
 *
 * @example
 *
 * <div class="fat-draggable">
 *  <div class="fat-draggable__item" data-column-name="{{column.name}}">
 *       <i class="fat-draggable__handler"></i>
 *       ...
 *  </div>
 * </div>
 */
app.factory('FatDraggableService', function() {
    const MAIN_CLASSNAME = 'fat-draggable';
    const HANDLER_CLASSNAME = MAIN_CLASSNAME + '__handler';
    const ITEM_CLASSNAME = MAIN_CLASSNAME + '__item';
    const PLACEHOLDER_CLASSNAME = MAIN_CLASSNAME + '__placeholder';
    const BAR_CLASSNAME = MAIN_CLASSNAME + '__bar';
    const DRAGGING_CLASSNAME = MAIN_CLASSNAME + '--dragging';
    const DRAGGED_CLASSNAME = ITEM_CLASSNAME + '--dragged';
    const COLUMN_NAME_ATTRIBUTE = 'data-column-name';
    const BAR_THICKNESS = 2;
    const MINIMAL_MOVE_TO_DRAG = 10;
    let classNamesToIgnore;
    let scrollBar;
    let element;
    let axis = 'x';
    let axisClassname;
    let placeholderDOM = document.createElement('div');
    let barDOM = document.createElement('div');
    let draggedItem;
    let draggedColumnName;
    let disabledTarget;
    let draggedItemDimensions = {};
    let placeholderDimensions = {};
    let barDimensions = {};
    let elementDimensions = {};
    let hoveredItem;
    let hoveredColumnName;
    let hoveredItemDimensions = {};
    let dragging = false;
    let downing = false;
    let onDrop;
    let onPlaceholderUpdate;
    let cursorInitialPosition = -1;
    let cursorPosition = -1;
    let gap = -1;

    // Ensures requestAnimationFrame cross-browsers support
    window.requestAnimationFrame = window.requestAnimationFrame || window.mozRequestAnimationFrame || window.webkitRequestAnimationFrame || window.msRequestAnimationFrame;

    /* HELPERS */

    // Gets the draggable item matching the mouse target
    function getDraggableItem(target) {
        return target.closest('.' + ITEM_CLASSNAME);
    }

    function getColumnName(columnDOM) {
        if (columnDOM) {
            return columnDOM.getAttribute(COLUMN_NAME_ATTRIBUTE) || (columnDOM.firstElementChild && columnDOM.firstElementChild.getAttribute(COLUMN_NAME_ATTRIBUTE));
        }
    }

    function getColumnDOM(columnName) {
        if (columnName) {
            return element.querySelector('[' + COLUMN_NAME_ATTRIBUTE + '="' + columnName + '"]');
        }
    }

    // Returns true if we are dragging before (on top in y axis or on left on x axis) the item being dragged
    function isDraggingBefore() {
        return cursorPosition && draggedItemDimensions && (cursorPosition <= draggedItemDimensions[axis === 'y' ? 'top' : 'left']);
    }

    // Retrieves the item around the cursor position
    function getHoveredItem() {
        let items = element.querySelectorAll('.' + ITEM_CLASSNAME);
        let vertical = (axis === 'y');
        for (let i = 0; i < items.length; i++) {
            let currentItem = items[i];
            let itemDimensions = getBoundingClientRect(currentItem);
            if (vertical) {
                if (itemDimensions.top <= cursorPosition && cursorPosition < itemDimensions.top + itemDimensions.height) {
                    return currentItem;
                }
            } else {
                if (itemDimensions.left <= cursorPosition && cursorPosition < itemDimensions.left + itemDimensions.width) {
                    return currentItem;
                }
            }
        }
        return null;
    }

    // Sets hover style for the draggable item matching the mouse target
    function updateHoveredItem() {
        let draggingBefore = isDraggingBefore();
        let target = getHoveredItem();

        if (!target) {
            return;
        }
        if (hoveredItem && hoveredItem === target) {
            return;
        }

        hoveredItem = target;
        hoveredColumnName = getColumnName(hoveredItem);

        if (hoveredItem !== draggedItem) {
            hoveredItemDimensions = getBoundingClientRect(hoveredItem);
            elementDimensions = getBoundingClientRect(element);

            if (axis === 'y') {
                barDimensions.top = draggingBefore ? hoveredItemDimensions.top : hoveredItemDimensions.top + hoveredItemDimensions.height;
                barDimensions.height = BAR_THICKNESS;
                barDimensions.width = placeholderDimensions.width;
                barDimensions.left = elementDimensions.left;
            } else {
                barDimensions.left = draggingBefore ? hoveredItemDimensions.left : hoveredItemDimensions.left + hoveredItemDimensions.width;
                barDimensions.height = placeholderDimensions.height;
                barDimensions.width = BAR_THICKNESS;
                barDimensions.top = elementDimensions.top;
            }
        } else {
            barDimensions.width = 0;
            barDimensions.height = 0;
        }

        updateBarBoundingBox(barDimensions);
    }

    // Redraw the placeholder according the mouse position
    function updatePlaceholderBoundingBox(dimensions) {

        if (typeof onPlaceholderUpdate === 'function') {
            onPlaceholderUpdate(dimensions);
        }

        if (dimensions.top >= 0) {
            if (dimensions.top <= elementDimensions.top - placeholderDimensions.height / 2) {
                // If overflowing on top
                // Putting at top of parent element - half of the placeholder (for better auto scroll)
                dimensions.top = elementDimensions.top - placeholderDimensions.height / 2;
            } else if (dimensions.top + placeholderDimensions.height - placeholderDimensions.height / 2 >= elementDimensions.top + elementDimensions.height) {
                // If overflowing on bottom
                // Putting at bottom of parent element + half of the placeholder (for better auto scroll)
                dimensions.top = elementDimensions.top + elementDimensions.height - placeholderDimensions.height / 2;
            }
            placeholderDOM.style.top = dimensions.top + 'px';
        }
        if (dimensions.left >= 0) {
            if (dimensions.left <= elementDimensions.left - placeholderDimensions.width / 2) {
                // If overflowing on the left
                // Putting at left of parent element - half of the placeholder (for better auto scroll)
                dimensions.left = elementDimensions.left - placeholderDimensions.width / 2;
            } else if (dimensions.left + placeholderDimensions.width - placeholderDimensions.width / 2 >= elementDimensions.left + elementDimensions.width) {
                // If overflowing on the right
                // Putting at right of parent element + half of the placeholder (for better auto scroll)
                dimensions.left = elementDimensions.left + elementDimensions.width - placeholderDimensions.width / 2;
            }
            placeholderDOM.style.left = dimensions.left + 'px';
        }

        if (dimensions.height >= 0) {
            placeholderDOM.style.height = dimensions.height + 'px';
        }
        if (dimensions.width >= 0) {
            placeholderDOM.style.width = dimensions.width + 'px';
        }
    }

    // Wrap the placeholder position update in a callback for requestAnimationFrame()
    function placeholderDOMRedraw() {
        updatePlaceholderBoundingBox(axis === 'y' ? {top: (cursorPosition - gap)} : {left: (cursorPosition - gap)});
    }

    // Redraw the bar according the given dimensions
    function updateBarBoundingBox(dimensions) {
        if (dimensions.top >= 0) {
            if (dimensions.top <= elementDimensions.top) {
                dimensions.top = elementDimensions.top;
            } else if (dimensions.top >= elementDimensions.top + elementDimensions.height) {
                dimensions.top = elementDimensions.top + elementDimensions.height;
            }
            barDOM.style.top = dimensions.top + 'px';
        }
        if (dimensions.left >= 0) {
            if (dimensions.left <= elementDimensions.left) {
                dimensions.left = elementDimensions.left;
            } else if (dimensions.left >= elementDimensions.left + elementDimensions.width) {
                dimensions.left = elementDimensions.left + elementDimensions.width;
            }
            barDOM.style.left = dimensions.left + 'px';
        }
        if (dimensions.height >= 0) {
            barDOM.style.height = dimensions.height + 'px';
        }
        if (dimensions.width >= 0) {
            barDOM.style.width = dimensions.width + 'px';
        }
    }

    // Generic fatTable scroll update
    function updateScroll () {
        if (!scrollBar) {
            return;
        }
        if (axis === 'y') {
            let elementTop = getBoundingClientRect(element).top;
            let elementBottom = elementTop + element.offsetHeight;
            let placeholderTop= placeholderDOM.offsetTop;
            let placeholderBottom = placeholderTop + placeholderDOM.offsetHeight;

            if (placeholderBottom > elementBottom) {
                scrollBar.setScrollXY(element.scrollLeft, scrollBar.scrollTop + placeholderBottom - elementBottom);
            } else if (placeholderTop < elementTop) {
                scrollBar.setScrollXY(element.scrollLeft, scrollBar.scrollTop + placeholderTop - elementTop);
            }
        } else {
            let elementLeft = getBoundingClientRect(element).left;
            let elementRight = elementLeft + element.offsetWidth;
            let placeholderLeft = placeholderDOM.offsetLeft;
            let placeholderRight = placeholderLeft + placeholderDOM.offsetWidth;

            if (placeholderRight > elementRight) {
                scrollBar.setScrollXY(scrollBar.scrollLeft + placeholderRight - elementRight, element.scrollTop);
            } else if (placeholderLeft < elementLeft) {
                scrollBar.setScrollXY(scrollBar.scrollLeft + placeholderLeft - elementLeft, element.scrollTop);
            }
        }
    }

    // Clean every dragging-related things
    function reset() {
        document.body.contains(placeholderDOM) && document.body.removeChild(placeholderDOM);
        barDOM.style.top = '';
        barDOM.style.left = '';
        barDOM.style.height = '';
        barDOM.style.width = '';
        document.body.removeChild(barDOM);
        draggedItem && draggedItem.classList.remove(DRAGGED_CLASSNAME);
        document.body.classList.remove(DRAGGING_CLASSNAME);

        window.removeEventListener('mousemove', onMouseMove);
        window.removeEventListener('mouseup', onMouseUp);
        disabledTarget.removeEventListener('mouseup', disableClick);
        Mousetrap.unbind("esc", onEscape);

        dragging = false;
        downing = false;
        draggedItem = null;
        draggedColumnName = null;
        hoveredItem = null;
        hoveredColumnName = null;
        disabledTarget = null;
        draggedItemDimensions = {};
        placeholderDimensions = {};
        barDimensions = {};
        hoveredItemDimensions = {};
        elementDimensions = {};
        cursorPosition = -1;
        cursorInitialPosition = -1;
        gap = -1;
    }

    // Prevent the beginning of the drag
    function cancel() {
        if (dragging) {
            reset();
        } else {
            cursorInitialPosition = -1;
            downing = false;
            draggedItem = null;
            draggedColumnName = null;
            window.removeEventListener('mousemove', onMouseMove);
        }
    }

    /* EVENTS LISTENERS */
    function onMouseUp() {

        if (!dragging) {
            cancel();
            return;
        }
        if (typeof onDrop === 'function' && draggedColumnName && hoveredColumnName && draggedColumnName !== hoveredColumnName) {
            onDrop(draggedItem, hoveredItem, draggedColumnName, hoveredColumnName);
        }
        reset();
    }

    // When the dragging has started we do not want any click to be triggered on the element
    function disableClick(event) {
        event.stopImmediatePropagation();
        event.preventDefault();
        onMouseUp();
    }

    // If moving the placeholder, update its position according the given axis
    function onMouseMove(event) {

        // If not yet dragging, initiate drag, else update the placeholder position
        if (!dragging && downing) {
            if (axis === 'y') {
                cursorPosition = event.clientY;
                gap = cursorInitialPosition - draggedItemDimensions.y;
            } else {
                cursorPosition = event.clientX;
                gap = cursorInitialPosition - draggedItemDimensions.x;
            }

            // Do not start drag if the mouse has not moved enough
            if (Math.abs(cursorPosition - cursorInitialPosition) < MINIMAL_MOVE_TO_DRAG) {
                return;
            }

            dragging = true;
            // Bind mouseup for dragging end and click on target to prevent other listeners to be triggered
            window.addEventListener('mouseup', onMouseUp);
            disabledTarget = event.target;
            disabledTarget.addEventListener('click', disableClick);

            // Inject placeholder and bar in DOM and add the drag-related class names
            document.body.appendChild(placeholderDOM);
            document.body.appendChild(barDOM);
            document.body.classList.add(DRAGGING_CLASSNAME);
            draggedItem.classList.add(DRAGGED_CLASSNAME);

            placeholderDimensions = angular.copy(draggedItemDimensions);
            updatePlaceholderBoundingBox(placeholderDimensions);

            Mousetrap.bind("esc", onEscape);
        } else {
            cursorPosition = axis === 'y' ? event.clientY : event.clientX;
            requestAnimationFrame(placeholderDOMRedraw);
            updateScroll();
            // If the dragged column DOM from fattable has been removed try to re-fetch it
            if (!document.body.contains(draggedItem)) {
                let newDraggedItem = getColumnDOM(draggedColumnName);
                if (newDraggedItem) {
                    draggedItem = newDraggedItem;
                    draggedItem.classList.add(DRAGGED_CLASSNAME);
                }
            // If the dragged column DOM and column name are inconsistent, invalidate it and try to refetch it
            } else if (!draggedItem.getAttribute(COLUMN_NAME_ATTRIBUTE) || draggedItem.getAttribute(COLUMN_NAME_ATTRIBUTE) !== draggedColumnName) {
                draggedItem.classList.remove(DRAGGED_CLASSNAME);
                draggedItem = null;
                let newDraggedItem = getColumnDOM(draggedColumnName);
                if (newDraggedItem) {
                    draggedItem = newDraggedItem;
                    draggedItem.classList.add(DRAGGED_CLASSNAME);
                }
            }
        }

        updateHoveredItem();
    }

    // Press escape to stop the current drag
    function onEscape() {
        if (!dragging) {
            return;
        }
        reset();
    }

    function onMouseDown(event) {

        // Do not drag if the selected element has at least one class marking it as not-draggable
        try {
            classNamesToIgnore.forEach(className => {
                if (event.target.closest('.' + className)) {
                    throw new Error("Ignoring drag, element not draggable.");
                }
            });
        } catch (e) {
            return;
        }

        if (!event.target.closest('.' + HANDLER_CLASSNAME)) {
            return;
        }

        // Do not consider right click
        if (event.which === 3) {
            return;
        }

        downing = true;
        draggedItem = getDraggableItem(event.target);
        draggedColumnName = getColumnName(draggedItem);

        window.addEventListener('mousemove', onMouseMove);

        // If a click occurred, prevent dragging
        element.addEventListener('click', cancel);

        // Prevent native drag
        event.preventDefault();

        // Ensure mandatory class name are here
        element.classList.add(MAIN_CLASSNAME);
        element.classList.add(axisClassname);

        draggedItemDimensions = getBoundingClientRect(draggedItem);

        if (axis === 'y') {
            cursorInitialPosition = event.clientY;
        } else {
            cursorInitialPosition = event.clientX;
        }
    }

    return {

        setDraggable: function(options) {
            if (!options || !options.element) { return }
            element = options.element;
            onDrop = options.onDrop;
            onPlaceholderUpdate = options.onPlaceholderUpdate;
            scrollBar = options.scrollBar;
            classNamesToIgnore = options.classNamesToIgnore;

            element.classList.add(MAIN_CLASSNAME);

            axis = options.axis && options.axis === 'y' ? 'y' : 'x';
            axisClassname = axis === 'y' ? 'fat-draggable-y-axis' : 'fat-draggable-x-axis';
            element.classList.add(axisClassname);

            placeholderDOM.className = PLACEHOLDER_CLASSNAME;
            barDOM.className = BAR_CLASSNAME;

            // If clicking on a drag handler, retrieve dragged item data and attach mouse move
            element.addEventListener('mousedown', onMouseDown);
        }
    }
});

/**
 * Enhance fattable elements with resize capabilities.
 * Mandatory class fat-resizable__item should be added for each items that can be dragged.
 * Mandatory class fat-resizable__handler should be added to trigger resize. It should be a child of an item or the item itself.
 * Mandatory data-column-name and data-column-index attributes should be added at fat-resizable__item level to keep a reference to the column when DOM is being recycled.
 *
 * To enable resize on an element, call the setResizable() method with the following options :
 *
 * @param {Object}              options                         - The available options
 * @param {HTMLElement}         options.element                 - (Mandatory) element containing the resizable items.
 * @param {Function}            options.onDrop               - Drop callback. Returns an object containing resized column data: index, name and width.
 *
 * @example
 *
 * <div fat-resizable>
 * ...
 *  <div class="fat-resizable__item" data-column-name="{{column.name}}" data-column-index="{{columnIndex}}">
 *      ...
 *      <span class="fat-resizable__handler"></span>
 *      ...
 *  </div>
 * ...
 * </div>
 */
app.factory('FatResizableService', function () {
    const MAIN_CLASSNAME = 'fat-resizable';
    const ITEM_CLASSNAME = MAIN_CLASSNAME + '__item';
    const HANDLER_CLASSNAME = MAIN_CLASSNAME + '__handler';
    const BAR_CLASSNAME = MAIN_CLASSNAME + '__bar';
    const COLUMN_INDEX_ATTRIBUTE_NAME = 'data-column-index';
    const COLUMN_NAME_ATTRIBUTE_NAME = 'data-column-name';
    const BAR_THICKNESS = 4;
    const ITEM_MIN_WIDTH = 60;

    let element,
        onDrop,
        resizing = false,
        downing = false,
        cursorPosition = -1,
        barDimensions = {},
        elementDimensions,
        resizableItem = null,
        resizedItemDimensions = {},
        draggingLowerBound,
        disabledTarget,
        barDOM = document.createElement('div');

    /* HELPERS */

    // Gets the resizable item matching the mouse target
    function getResizableItem(target) {
        return target.closest('.' + ITEM_CLASSNAME);
    }

    function updateBar() {
        elementDimensions = elementDimensions || getBoundingClientRect(element);
        barDimensions.left = cursorPosition;
        barDimensions.top = elementDimensions.top;
        barDOM.style.top = barDimensions.top + 'px';
        barDOM.style.left = barDimensions.left + 'px';
        barDOM.style.height = barDimensions.height + 'px';
        barDOM.style.width = barDimensions.width + 'px';
    }

    // Clean every resizing-related things
    function reset() {
        barDOM.style.top = '';
        barDOM.style.left = '';
        barDOM.style.height = '';
        barDOM.style.width = '';
        document.body.removeChild(barDOM);

        window.removeEventListener('mousemove', onMouseMove);
        window.removeEventListener('mouseup', onMouseUp);
        disabledTarget.removeEventListener('mouseup', disableClick);
        Mousetrap.unbind("esc", onEscape);

        resizing = false;
        downing = false;
        barDimensions.left = 0;
        barDimensions.right = 0;
        cursorPosition = -1;
        draggingLowerBound = null;
        resizableItem = null;
        resizedItemDimensions = {};
    }

    // Prevent the beginning of the drag
    function cancel() {
        if (resizing) {
            reset();
        } else {
            downing = false;
            resizableItem = null;
            resizedItemDimensions = {};
            window.removeEventListener('mousemove', onMouseMove);
        }
    }

    /* EVENTS LISTENERS */

    function onMouseUp() {

        if (!resizing) {
            cancel();
            return;
        }

        if (typeof onDrop === 'function') {
            let resizedWidth = cursorPosition - resizedItemDimensions.x;
            resizedWidth = resizedWidth > ITEM_MIN_WIDTH ? resizedWidth : ITEM_MIN_WIDTH;
            onDrop({
                index: resizableItem.getAttribute(COLUMN_INDEX_ATTRIBUTE_NAME),
                name: resizableItem.getAttribute(COLUMN_NAME_ATTRIBUTE_NAME),
                width: resizedWidth
            });
        }
        reset();
    }

    // When the dragging has started we do not want any click to be triggered on the element
    function disableClick(event) {
        event.stopImmediatePropagation();
        event.preventDefault();
        onMouseUp();
    }

    function onMouseMove(event) {

        // If not yet dragging, initiate drag, else update the placeholder position
        if (!resizing && downing) {

            resizing = true;
            // Bind mouseup for dragging end and click on target to prevent other listeners to be triggered
            window.addEventListener('mouseup', onMouseUp);
            disabledTarget = event.target;
            disabledTarget.addEventListener('click', disableClick);

            // Inject bar in DOM
            document.body.appendChild(barDOM);

            Mousetrap.bind("esc", onEscape);
        } else {
            cursorPosition = event.clientX;
        }

        if (!draggingLowerBound) {
            draggingLowerBound = getBoundingClientRect(resizableItem).x
        }

        // Prevent resizing beyond the previous column
        if (cursorPosition <= draggingLowerBound + ITEM_MIN_WIDTH) { return }

        updateBar();
    }

    function onMouseDown(event) {

        if (!event.target.closest('.' + HANDLER_CLASSNAME)) {
            return;
        }

        // Do not consider right click
        if (event.which === 3) {
            return;
        }

        downing = true;
        resizableItem = getResizableItem(event.target);

        window.addEventListener('mousemove', onMouseMove);

        // If a click occurred, prevent dragging
        element.addEventListener('click', cancel);

        // Prevent native drag
        event.preventDefault();

        resizedItemDimensions = getBoundingClientRect(resizableItem);
    }

    // Press escape to stop the current drag
    function onEscape() {
        if (!resizing) {
            return;
        }
        reset();
    }

    return {

        setResizable: function(options) {
            if (!options || !options.element) { return }

            onDrop = options.onDrop;
            element = options.element;

            // Prepare vertical bar used as feedback while resizing
            barDimensions.height = options.barHeight;
            barDimensions.width = BAR_THICKNESS;
            barDOM.className = BAR_CLASSNAME;

            // If clicking on a drag handler, retrieve dragged item data and attach mouse move
            element.addEventListener('mousedown', onMouseDown);
        }
    }
})

app.service('FatTouchableService', function($timeout) {
    /**
     * Make a fattable/fatrepeat scrollable through touch interaction
     * @param scope: scope of the fattable/fatrepeat directive
     * @param element: DOM element of the fattable/fatrepeat directive
     * @param fattable: fattable object of the fattable/fatrepeat directive
     * @returns function to remove event listeners added by this function
     */
    this.setTouchable = function(scope, element, fattable) {
        /**
         * Return an object wrapping callbacks. This function will be called each time a touchstart is emmited in order
         * to give callbacks to the added the touchmove and touchend event listeners.
         */
        let getOnTouchCallbacks = (function() {
            /*
             * Callbacks
             */

            /**
             * Turns the touchMoveEvent passed in parameter into a scrollOrder to the fattable
             * @param event
             */
            let startScroll;
            let lastScroll;
            let startTouch;
            let lastTouch;
            function scrollFattable(touchMoveEvent) {
                fattable.scroll.dragging = true; //otherwise fattable behaves as user had scrolled using scrollbars, which triggers multiple reflows, which is bad for performances
                touchMoveEvent.preventDefault();
                touchMoveEvent.stopPropagation();
                let newTouch = touchMoveEvent.originalEvent.changedTouches[0];

                // Tracking for direction change
                function getTouchedDistance(t) {
                    return {
                        x: startTouch.screenX - t.screenX,
                        y: startTouch.screenY - t.screenY
                    }
                }
                let touchedDistance = getTouchedDistance(newTouch);
                let lastTouchedDistance = getTouchedDistance(lastTouch);
                if (Math.abs(lastTouchedDistance) - Math.abs(touchedDistance) > 0) {
                    startTouch = lastTouch;
                    startScroll = lastScroll;
                    touchedDistance = getTouchedDistance(newTouch);
                }
                // Scrolling
                requestAnimationFrame(_ => fattable.scroll.setScrollXY(startScroll.x + touchedDistance.x, startScroll.y + touchedDistance.y));
                // Updating memory
                lastTouch = touchMoveEvent.originalEvent.changedTouches[0];
                lastScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                }
            }

            /**
             * Keeps track of touch velocity in order to generate momentum when touchmove will stop.
             * (Tracking will stop at touchEnd)
             * @type {number}
             */
            const SPEED_FILTER = 0.8; // Arbitrary chosen constant
            let prevScroll;
            let velocity;
            let lastTimestamp;
            let keepTrackingVelocity;
            function trackVelocity() {
                // current scroll position
                let scroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                // scrolled distance since last track
                let delta = {
                    x: scroll.x - prevScroll.x,
                    y: scroll.y - prevScroll.y
                };
                // if scroll changed direction then we do not take previous velocity into account
                let prevVelocityCoeff = {
                    x: delta.x * velocity.x > 0 ? 0.2 : 0,
                    y: delta.y * velocity.y > 0 ? 0.2 : 0,
                };
                // computing velocity
                let timeStamp = Date.now();
                velocity.x = SPEED_FILTER * delta.x * 1000 / (1 + timeStamp - lastTimestamp) + prevVelocityCoeff.x * velocity.x;
                velocity.y = SPEED_FILTER * delta.y * 1000 / (1 + timeStamp - lastTimestamp) + prevVelocityCoeff.y * velocity.y;
                // updating memory
                lastTimestamp = timeStamp;
                prevScroll = scroll;
                if (keepTrackingVelocity) {
                    $timeout(trackVelocity, 10);
                }
            }

            /**
             * Generates a momentum animation when touch stops
             * @param event
             */
            function animateMomentum() {
                keepTrackingVelocity = false;
                let endTime = Date.now();
                // Momentum appears only if velocity was greater than 10px/s
                if (Math.abs(velocity.x) > 10 || Math.abs(velocity.y) > 10) {
                    // Detach event listeners when momentum ends
                    let onMomentumEnd = function () {
                        element.off('touchstart', stopMomentumAnimation);
                        element.off('touchmove', stopMomentumAnimation);
                        fattable.scroll.dragging = false;
                    }

                    // Stop momentum on new touchevent
                    let interruptedMomentum = false;
                    let stopMomentumAnimation = function () {
                        interruptedMomentum = true;
                        onMomentumEnd();
                    }
                    element.on('touchstart', stopMomentumAnimation);
                    element.on('touchmove', stopMomentumAnimation);

                    // Compute if scroll can go further with inertia
                    let canScroll = function (delta) {
                        let canScrollH = (delta.x < 0 && fattable.scroll.scrollLeft > 0) || (delta.x > 0 && fattable.scroll.scrollLeft < fattable.scroll.maxScrollHorizontal);
                        let canScrollV = (delta.y < 0 && fattable.scroll.scrollTop > 0) || (delta.y > 0 && fattable.scroll.scrollTop < fattable.scroll.maxScrollVertical);
                        return canScrollH || canScrollV;
                    }

                    /*
                     * MOMENTUM (all formulas come from this article: https://ariya.io/2013/11/javascript-kinetic-scrolling-part-2)
                     */

                    // additional distance to scroll while momentum
                    let amplitude = {
                        x: SPEED_FILTER * velocity.x,
                        y: SPEED_FILTER * velocity.y
                    }
                    let previousScroll = {x: 0, y: 0};
                    let autoScroll = function () {
                        const TIME_CONSTANT = 325; // arbitrary constant chosen experimentally
                        let elapsedSinceStop = Date.now() - endTime; // elapsed time since touchend
                        let exponentialDecay = Math.exp(-elapsedSinceStop / TIME_CONSTANT);
                        // where scroll should be at this time (due to inertia)
                        let scroll = {
                            x: amplitude.x * (1 - exponentialDecay),
                            y: amplitude.y * (1 - exponentialDecay)
                        }
                        // missing scroll distance (where scroll should be minus where scroll is now)
                        let delta = {
                            x: scroll.x - previousScroll.x,
                            y: scroll.y - previousScroll.y
                        }
                        // scrolling of missing scrolled distance
                        fattable.scroll.setScrollXY(fattable.scroll.scrollLeft + delta.x, fattable.scroll.scrollTop + delta.y);
                        previousScroll = scroll;
                        // momentum keeps going on until amplitude is almost reached or animation got interrupted by a new touchevent
                        if ((Math.abs(amplitude.x - scroll.x) > 0.5 || Math.abs(amplitude.y - scroll.y) > 0.5) && !interruptedMomentum && canScroll(delta)) {
                            requestAnimationFrame(autoScroll);
                        } else {
                            onMomentumEnd()
                        }
                    }
                    requestAnimationFrame(autoScroll);
                }
            }

            /*
             * Actual getOnTouchCallbacks function
             */
            return function (touchStartEvent) {
                // Initialization of scrollFattable variables
                startScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                lastScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                startTouch = touchStartEvent.originalEvent.changedTouches[0];
                lastTouch = touchStartEvent.originalEvent.changedTouches[0];

                // Initialization of trackVelocity variables
                prevScroll = {
                    x: fattable.scroll.scrollLeft,
                    y: fattable.scroll.scrollTop
                };
                velocity = {x: 0, y: 0};
                lastTimestamp = Date.now();
                keepTrackingVelocity = true;

                return {
                    onTouchStart: function (e) {
                        trackVelocity(e);
                    },
                    onTouchMove: function (e) {
                        scrollFattable(e);
                    },
                    onTouchEnd: function (e) {
                        animateMomentum(e);
                    }
                };
            }
        })();

        function onTouchStart(event) {
            let currentOnTouchCallbacks = getOnTouchCallbacks(event, element);
            currentOnTouchCallbacks.onTouchStart();
            element.on("touchmove", currentOnTouchCallbacks.onTouchMove);
            let onTouchEnd = function(event) {
                currentOnTouchCallbacks.onTouchEnd(event);
                element.off("touchmove", currentOnTouchCallbacks.onTouchMove);
                element.off("touchend", onTouchEnd);
            }
            element.on("touchend", onTouchEnd);
        }

        element.on("touchstart", onTouchStart);

        let removeOnDestroy = scope.$on('$destroy', function() {
            element.off("touchstart", onTouchStart);
        });

        return function() {
            removeOnDestroy();
            element.off("touchstart", onTouchStart)
        }
    }
});

app.service('ClipboardUtils', function($q, $rootScope, translate, ActivityIndicator) {
    const svc = this;

    /**
     * Copy some text into the clipboard and notify users of success/failure with an ActivityIndicator banner message.
     *
     * @param text text to copy to the clipbard. If empty, does nothing and immediately resolves the promise (successfully).
     * @param successMessage Localized message to display with ActivityIndicator upon success. Use null/undefined to use the default message.
     *          Use an empty string to prevent the service from displaying any message.
     * @param errorMessage Localized message to display with ActivityIndicator in case of error. Use null/undefined to use the default message.
     *          Use an empty string to prevent the service from displaying any message in case of error.
     * @return {Promise<void>|*} A promise that resolves when 'text' has been copied to the clipboard, or if an error occurred.
     */
    svc.copyToClipboard = function(text, successMessage, errorMessage) {
        function showSuccess(triggerDigest = true) {
            if (successMessage === undefined || successMessage === null) {
                successMessage = translate('GLOBAL.CLIPBOARD.COPY_SUCCESS', 'Copied to clipboard!');
            }
            if (successMessage) {
                ActivityIndicator.success(successMessage, 5000);
                if (triggerDigest) {  // Trigger an immediate digest as we are outside AngularJS
                    $rootScope.$apply();
                }
            }
        }
        function showError(triggerDigest = true) {
            if (errorMessage === undefined || errorMessage === null) {
                errorMessage = translate('GLOBAL.CLIPBOARD.COPY_ERROR', 'Failed to copy to the clipboard!');
            }
            if (errorMessage) {
                ActivityIndicator.error(errorMessage, 5000);
                if (triggerDigest) {  // Trigger an immediate digest as we are outside AngularJS
                    $rootScope.$apply();
                }
            }
        }
        if (!text) {
            return Promise.resolve();
        }
        if (!navigator.clipboard) { // In case we are not in a secure context (for example a self-signed certificate)
            let tempInput = document.createElement("textarea");
            tempInput.style = "position: absolute; left: -1000px; top: -1000px";
            tempInput.value = text;
            document.body.appendChild(tempInput);
            tempInput.select();
            try {
                // noinspection JSDeprecatedSymbols
                if (document.execCommand("copy")) {
                    showSuccess(false);
                    return Promise.resolve();
                } else {
                    showError(false);
                    return Promise.reject(new Error('execCommand copy failed'));
                }
            } catch (err) {
                showError(false);
                return Promise.reject(err);
            } finally {
                document.body.removeChild(tempInput);
            }
        }
        return navigator.clipboard.writeText(text).then(showSuccess, showError);
    };
    // for pasting into non-editable element
    // called after capturing ctrl + v keydown event
    svc.pasteFromClipboard = function(event, callback) {
        let tempInput = document.createElement("textarea");
        tempInput.style = 'position: absolute; left: -1000px; top: -1000px';
        document.body.appendChild(tempInput);
        tempInput.select();
        // delay to capture imput value
        window.setTimeout(function() {
            let data = tempInput.value;

            callback(data);

            document.body.removeChild(tempInput);

            if (event) {
                event.currentTarget.focus();
            }
        }, 100);
    }
});

app.factory('DetectUtils', function() {
    const svc = {
        getOS: function() {
            let browser = '';

            if (navigator.appVersion.indexOf("Win") !== -1){
                browser = 'windows';
            }

            if (navigator.appVersion.indexOf("Mac")!=-1){
                browser = 'macos';
            }

            if (navigator.appVersion.indexOf("X11")!=-1){
                browser = 'unix';
            }

            if (navigator.appVersion.indexOf("Linux")!=-1){
                browser = 'linux';
            }

            return browser;
        },
        isMac: function() {
            return svc.getOS() === 'macos';
        },
        isWindows: function() {
            return svc.getOS() === 'windows';
        },
        isUnix: function() {
            return svc.getOS() === 'unix';
        },
        isLinux: function() {
            return svc.getOS() === 'linux';
        }
    };

    return svc;
});

app.constant("GRAPHIC_EXPORT_OPTIONS", {
    fileTypes: ['PDF','JPEG','PNG'],
    orientationMap: {
        'LANDSCAPE': 'Landscape',
        'PORTRAIT': 'Portrait'
    },
    paperSizeMap: {
        'A4': 'A4',
        'A3': 'A3',
        'US_LETTER': 'US Letter',
        'LEDGER': 'Ledger (ANSI B)',
        'SCREEN_16_9': '16:9 (Computer screen)',
        'CUSTOM': 'Custom'
    },
    paperSizeMapPage: {
        'A4': 'A4',
        'A3': 'A3',
        'US_LETTER': 'US Letter',
        'LEDGER': 'Ledger (ANSI B)'
    },
    paperInchesMap: {
        'A4': 11.6929,
        'A3': 16.5354,
        'US_LETTER': 11,
        'LEDGER': 17,
        'SCREEN_16_9': 11
    },
    ratioMap: {
        'A4': Math.sqrt(2),
        'A3': Math.sqrt(2),
        'US_LETTER': 11 / 8.5,
        'LEDGER': 17 / 11,
        'SCREEN_16_9': 16 / 9,
        'CUSTOM': 16 / 9
    }
});

app.service('GraphicImportService', function(GRAPHIC_EXPORT_OPTIONS) {
    const svc = this;
    svc.computeHeight = function (width, paperSize, orientation) {
        if (orientation == "PORTRAIT") {
            return Math.round(width * GRAPHIC_EXPORT_OPTIONS.ratioMap[paperSize]);
        } else {
            return Math.round(width / GRAPHIC_EXPORT_OPTIONS.ratioMap[paperSize]);
        }
    };
});

app.service('StringUtils',  function() {
    return {
        transmogrify: function(name, usedNames, makeName, offset = 2, alwaysAddSuffix = false) {
            if (! (usedNames instanceof Set)) {
                usedNames = new Set(usedNames);
            }
            if (! usedNames.has(name) && !alwaysAddSuffix) {
                return name;
            }
            if (! makeName) {
                makeName = i => `${name} ${i}`;
            }

            let i = offset;
            while (usedNames.has(makeName(i, name))) {
                i++;
            }
        return makeName(i, name);
        },
        /**
         * Compute the real width (in pixels) that a given text will take once rendered in DOM for the given font.
         */
        getTextWidth(text, font = '12px Arial', canvas) {
            if (!canvas) {
                canvas = document.createElement('canvas');
            }
            const context = canvas.getContext('2d');
            context.font = font;
            return context.measureText(text).width || 0;
        },
        /**
         * Compute the real height (in pixels) that a given text will take once rendered in DOM for the given font.
         */
        getTextHeight(text, font = '12px Arial', canvas) {
            if (!canvas) {
                canvas = document.createElement('canvas');
            }
            const context = canvas.getContext('2d');
            context.font = font;
            const metrics = context.measureText(text);
            return metrics.actualBoundingBoxAscent + metrics.actualBoundingBoxDescent || 0;
        },
        /**
         * Return a string truncated such that its length when encoded in bytes does not exceed maxBytes.
         * Useful for WT1 mixpanel event param values which have a max length of 255 bytes in utf-8.
         * @param {string} text - string to truncate
         * @param {num} maxBytes - length in bytes to not be exceeded
         * @param {string} encodingScheme - optional, to indicate encoding if not utf-8
         * @returns text truncated as necessary
         */
        truncateTextToLengthInBytes(text, maxBytes, encodingScheme = 'utf-8') {
            const encoder = new TextEncoder(encodingScheme);
            const encoded = encoder.encode(text);
            if (encoded.length <= maxBytes) {
                return text;
            }

            // We need to truncate - but we need to make sure it is still a valid string
            // if it's invalid, keep slicing off bytes until it is valid
            const decoder = new TextDecoder(encodingScheme, {fatal : true});
            let trimmedEncoded = encoded.slice(0, maxBytes);
            let decodedText = null;
            while (!decodedText){
                try {
                    decodedText = decoder.decode(trimmedEncoded);
                } catch (e) {
                    trimmedEncoded = trimmedEncoded.slice(0, trimmedEncoded.length - 1);
                }
            };
            return decodedText;
        },
        //Same as src/app/utils/string-utils.ts
        normalizeTextForSearch(str) {
            return this.stripAccents(str ?? '').toLowerCase().trim();
        },
        stripAccents(str) {
            return str.normalize("NFD").replace(/[\u0300-\u036f]/g, "");
        },
    };
});

app.service('PermissionsService', function(Dialogs, Logger) {
    return {
        buildUnassignedGroups: function(item, allGroups) {
            if (!item || !allGroups) return;

            return allGroups.filter(function(groupName) {
                return item.permissions.every(perm => perm == null || perm.group !== groupName);
            });
        },
        buildUnassignedUsers: function(item, allUsers) {
            return allUsers.filter(user =>
                item.owner !== user.login && item.permissions.every(perm => perm == null || perm.user !== user.login));
        },
        transferOwnership: function(scope, item, itemName, ownerUiField="ownerLogin") {
            const ownerUi = scope.ui && scope.ui[ownerUiField];
            if (!ownerUi || !item || ownerUi === item.owner) return;
            const newOwnerDisplayName = scope.allUsers.find(user => user.login === ownerUi).displayName || ownerUi;

            Dialogs.confirm(scope, 'Ownership transfer',
                `Are you sure you want to transfer ${itemName} ownership to '${newOwnerDisplayName}' ?`).then(function() {
                Logger.info(`Transferring ${itemName} ownership to ${ownerUi}`);
                item.owner = ownerUi;
            },function() {
                scope.ui[ownerUiField] = item.owner;
            });
        }
    }
});

    app.service('FullScreenService', ['$location', '$state', function ($location, $state) {
        return {
            isFullScreen: () => $location.search().fullScreen && $location.search().fullScreen !== "false"
        }
    }]);

    app.service('MLUtilsService', function () {
        return {
            getMLCategory: (taskType, backendType, predictionType, savedModelType, proxyModelProtocol) => {
                // Keep aligned with SavedModel.getMLCategory in `SavedModel.java`
                switch (savedModelType) {
                    case 'MLFLOW_PYFUNC':
                        return 'mlflow';
                    case 'LLM_GENERIC':
                        return 'fine_tuning';
                    case 'PYTHON_AGENT':
                        return 'python_agent';
                    case 'PLUGIN_AGENT':
                        return 'plugin_agent';
                    case 'TOOLS_USING_AGENT':
                        return "tools_using_agent";
                    case 'RETRIEVAL_AUGMENTED_LLM':
                        return "retrieval_augmented_llm";
                    case 'PROXY_MODEL':
                        switch (proxyModelProtocol) {
                            case 'sagemaker':
                                return 'sagemaker';
                            case 'vertex-ai':
                                return 'vertex';
                            case 'azure-ml':
                                return 'azureml';
                            case 'databricks':
                                return 'databricks';
                        }
                    default:
                        break;
                }

                switch (taskType) {
                    case 'CLUSTERING':
                        return 'clustering';
                    case 'PREDICTION':
                        if (backendType === 'KERAS') {
                            return 'keras';
                        }
                        switch (predictionType) {
                            case 'TIMESERIES_FORECAST':
                                return 'timeseries';
                            case 'CAUSAL_REGRESSION':
                            case 'CAUSAL_BINARY_CLASSIFICATION':
                                return 'causal';
                            case 'DEEP_HUB_IMAGE_CLASSIFICATION':
                                return 'deephub_image_classification';
                            case 'DEEP_HUB_IMAGE_OBJECT_DETECTION':
                                return 'deephub_object_detection';
                        }
                    default:
                        return 'prediction';
                }
            },
        };
    });

    app.service('AgentToolService', function($rootScope, PluginsService) {
        const AGENT_TOOL_TYPES_UI_DATA = [
            {
                type: "InlinePython",
                typeInUI: "Inline Python",
                label: "Inline Python",
                description: "Implement your own custom tool in Python code",
                documentationLink: "/agents/tools/custom-tools",
                icon: "dku-icon-python-20",
                codeEnvSelectionSupported: true,
                containerExecutionSupported: true,
                testQuery: {},
                showStopDevKernelBtn: true,
            },
            {
                type: "DatasetRowLookup",
                label: "Dataset Lookup",
                description: "Find a record in a dataset based on conditions on one or several columns",
                documentationLink: "/agents/tools/dataset-lookup",
                icon: "dku-icon-dataset-20",
                testQuery: {
                    "input": {
                        "filter": {
                            "column": "<COLUMN_NAME>",
                            "operator": "EQUALS",
                            "value": "<VALUE>"
                        }
                    }
                },
                showStopDevKernelBtn: false,
            },
            {
                type: "VectorStoreSearch",
                label: "Knowledge Bank Search",
                description: "Search a Knowledge Bank for relevant information",
                documentationLink:"/agents/tools/knowledge-bank-search",
                icon: 'dku-icon-cards-stack-20',
                testQuery: {
                    "input": {
                        "searchQuery": "Enter search query here"
                    }
                },
                showStopDevKernelBtn: true,
            },
            {
                type: "LLMMeshLLMQuery",
                label: "Query an LLM/Agent",
                description: "Send a request to an LLM or Agent registered in the LLM Mesh",
                documentationLink:"/agents/tools/llm-mesh-query",
                icon: 'dku-icon-ai-prompt-20',
                testQuery: {
                    "input": {
                        "question": "Ask your question here"
                    }
                },
                showStopDevKernelBtn: false,
            },
            {
                type: "DatasetRowAppend",
                label: "Dataset Append",
                description: "Write a record to a Dataiku dataset",
                documentationLink:"/agents/tools/dataset-append",
                icon: "dku-icon-data-table-row-20",
                testQuery: {
                    "input": {
                        "record": {
                            "<COLUMN1_NAME>": "<COLUMN1_VALUE>",
                            "<COLUMN2_NAME>": "<COLUMN2_VALUE>"
                        }
                    }
                },
                showStopDevKernelBtn: false,
            },
            {
                type: "DataikuReporter",
                label: "Send Message",
                description: "Send a message through a Dataiku Messaging Channel (email, Slack, Teams, ...)",
                icon: "dku-icon-data-user-email-20",
                testQuery: {
                    "input": {
                        "parameters": {
                            "message": "The message to send"
                        }
                    }
                },
                showStopDevKernelBtn: false,
            },
            {
                type: "ClassicalPredictionModelPredict",
                label: "Model Predict",
                description: "Predict a record with a Dataiku model",
                documentationLink:"/agents/tools/model-predict",
                icon: "dku-icon-automl-prediction-20",
                testQuery: {
                    "input": {
                        "record": {
                            "<FEATURE1_NAME>": "<FEATURE1_VALUE>",
                            "<FEATURE2_NAME>": "<FEATURE2_VALUE>"
                        }
                    }
                },
                showStopDevKernelBtn: false,
            },
            {
                type: "GRELCalculator",
                typeInUI: "Calculator",
                label: "Calculator",
                description: "Perform arithmetic, trigonometry, boolean, date, and geometry calculations",
                documentationLink: "/agents/tools/calculator",
                icon: "dku-icon-calculator-20",
                testQuery: {
                    "input": {
                        "formula": "sqrt(441) * 2"
                    }
                },
                showStopDevKernelBtn: false,
            },
            {
                type: "ApiEndpoint",
                label: "API Endpoint",
                description: "Use any API endpoint deployed with Dataiku",
                documentationLink: "/agents/tools/api-endpoint",
                icon: "dku-icon-api-service-deploy-20",
                testQuery: {},
                showStopDevKernelBtn: false,
            },
            {
                type: "VirtualTypeForMCPClients",
                label: "Use an MCP server",
                description: "Access and use tools exposed by a remote/local MCP server",
                icon: "dku-icon-mcp-20",
            },
            {
                type: "GenericStdioMCPClient",
                typeInUI: "Local MCP",
                label: "Run a local MCP server",
                description: "Use tools from any local MCP server",
                blueBoxDescription: "Configure and run a local MCP server in your environment. Set up the server manually or by pasting a JSON configuration",
                documentationLink: "/agents/tools/local-mcp",
                icon: "dku-icon-mcp-client-local-20",
                codeEnvSelectionSupported: true,
                containerExecutionSupported: true,
                testQuery: {
                    "input": {},
                    "subtoolName": "name of the subtool to use, must be enabled",
                },
                showStopDevKernelBtn: true,
                excludeFromNewAgentToolList: true,
            },
            {
                type: "RemoteMCPClient",
                typeInUI: "Remote MCP",
                label: "Use a remote MCP server",
                description: "Use tools from a remote MCP server",
                blueBoxDescription: "Access tools from a remote MCP server through an existing connection",
                documentationLink: "/agents/tools/remote-mcp",
                icon: "dku-icon-mcp-client-remote-20",
                codeEnvSelectionSupported: false,
                containerExecutionSupported: false,
                testQuery: {
                    "input": {},
                    "subtoolName": "name of the subtool to use, must be enabled",
                },
                showStopDevKernelBtn: false,
                excludeFromNewAgentToolList: true,
            }
        ];
        if ($rootScope.appConfig && $rootScope.appConfig.customAgentTools) {
            for (let agentTool of $rootScope.appConfig.customAgentTools) {
                AGENT_TOOL_TYPES_UI_DATA.push({
                    type: "Custom_" + agentTool.desc.id,
                    label: agentTool.desc.meta?.label ?? "Custom agent tool " + agentTool.desc.id,
                    description: agentTool.desc.meta?.description ?? "",
                    documentationLink: "/agents/tools/index",
                    icon: agentTool.desc.meta?.icon ?? "dku-icon-puzzle-piece-20",
                    containerExecutionSupported: true,
                    showStopDevKernelBtn: true,
                });
            }
        }

        this.listAgentToolTypesUiData = function() {
            return AGENT_TOOL_TYPES_UI_DATA.filter(toolTypeUIData => !toolTypeUIData.excludeFromNewAgentToolList);
        };

        this.getAgentToolTypesUiDataForType = function(agentToolType) {
            return AGENT_TOOL_TYPES_UI_DATA.find(toolTypeUiData => agentToolType === toolTypeUiData.type);
        };

        this.getAgentToolTypesUiData = function(agentTool) {
            return this.getAgentToolTypesUiDataForType(agentTool.type);
        };

        this.checkShowStopDevKernelBtn = function (agentToolType) {
            const toolUiData = AGENT_TOOL_TYPES_UI_DATA.find((item) => item.type === agentToolType);
            return toolUiData ? toolUiData.showStopDevKernelBtn : false;
        };

        this.getAgentToolQuickTestQueryForType = function(agentToolType) {
            const agentToolTypeUiData =  AGENT_TOOL_TYPES_UI_DATA.find(toolTypeUiData => agentToolType === toolTypeUiData.type);

            let testQuery = {"input": {}, "context": {}};
            if (agentToolTypeUiData) {
                Object.assign(testQuery, agentToolTypeUiData.testQuery);
            }

            return this.getAgentToolQuickTestQueryAsString(testQuery);
        };

        this.getAgentToolQuickTestQueryAsString = function(testQuery) {
            return JSON.stringify(testQuery, null, 2);
        };

        const CUSTOM_PREFIX = 'Custom_';
        this.getCustomAgentToolDesc = function (agentToolType) {
            let customAgentToolDesc = null;
            if (agentToolType && agentToolType.startsWith(CUSTOM_PREFIX)) {
                customAgentToolDesc = $rootScope.appConfig.customAgentTools.findLast((c) => c.agentType === agentToolType.substring(CUSTOM_PREFIX.length));
            }
            return customAgentToolDesc;
        };

        this.getCustomAgentPluginId = function (agentToolType) {
            let pluginId = null;
            let customAgentToolDesc = this.getCustomAgentToolDesc(agentToolType);
            if (customAgentToolDesc) {
                const pluginDesc = PluginsService.getOwnerPluginDesc(customAgentToolDesc);
                if (pluginDesc && pluginDesc.id) {
                    pluginId = pluginDesc.id;
                }
            }
            return pluginId;
        };
    });

    app.service('LLMUtilsService', function() {
        // keep aligned with LLMStructuredRef.decodeId
        this.mapIdToType = (id) => {
            if (id.startsWith('agent:')) {
                return 'SAVED_MODEL_AGENT';
            } else if (id.startsWith('savedmodel:finetuned_openai:')) {
                return 'SAVED_MODEL_FINETUNED_OPENAI';
            } else if (id.startsWith('savedmodel:finetuned_azureopenai:')) {
                return 'SAVED_MODEL_FINETUNED_AZURE_OPENAI';
            } else if (id.startsWith('savedmodel:finetuned_huggingfacelocal:')) {
                return 'SAVED_MODEL_FINETUNED_HUGGINGFACE_TRANSFORMER';
            } else if (id.startsWith('savedmodel:finetuned_bedrock:')) {
                return 'SAVED_MODEL_FINETUNED_BEDROCK';
            } else if (id.startsWith('openai:')) {
                return 'OPENAI'
            } else if (id.startsWith('azureopenai-model:')) {
                return 'AZURE_OPENAI_MODEL';
            } else if (id.startsWith('azureopenai:')) {
                return 'AZURE_OPENAI_DEPLOYMENT';
            } else if (id.startsWith('cohere')) {
                return 'COHERE';
            } else if (id.startsWith('mistralai:')) {
                return 'MISTRALAI';
            } else if (id.startsWith('anthropic')) {
                return 'ANTHROPIC';
            } else if (id.startsWith('vertex:')) {
                return 'VERTEX';
            } else if (id.startsWith('bedrock')) {
                return 'BEDROCK';
            } else if (id.startsWith('mosaicml')) {
                return 'MOSAICML';
            } else if (id.startsWith('stabilityai')) {
                return 'STABILITYAI';
            } else if (id.startsWith('nvidia-nim')) {
                return 'NVIDIA_NIM';
            } else if (id.startsWith('huggingfaceapi:')) {
                return 'HUGGINGFACE_API';
            } else if (id.startsWith('huggingfacelocal:')) {
                return 'HUGGINGFACE_TRANSFORMER_LOCAL';
            } else if (id.startsWith('databricksllm:')) {
                return 'DATABRICKS';
            } else if (id.startsWith('snowflakecortex:')) {
                return 'SNOWFLAKE_CORTEX';
            } else if (id.startsWith('retrievalaugmented:')) {
                return 'RETRIEVAL_AUGMENTED';
            } else if (id.startsWith('retrieval-augmented-llm:')) {
                return 'RETRIEVAL_AUGMENTED';
            } else if (id.startsWith('custom:')) {
                return 'CUSTOM';
            } else if (id.startsWith('sagemaker-generic:')) {
                return 'SAGEMAKER_GENERICLLM';
            } else if (id.startsWith('azure-llm:')) {
                return 'AZURE_LLM';
            }

            return '';
        };
    })

    app.service('AgentToolEnrichmentService', function(DataikuAPI, TypeMappingService, DatasetUtils) {
        this.enrichAgentToolObject = function(objectData, scope) {
            //$scope.object = objectData.agentTool;
            objectData.summary = {
                usedByAgents: [],
                usesKnowledgeBank: undefined, // Knowledge Bank Search
                usesSavedModel: undefined, // Model Predict
                usesDataset: undefined, // Dataset Append and Dataset Lookup
            };

            // find agents that use this tool
            DataikuAPI.agentTools.getUsage(objectData.projectKey, objectData.id).success(data => {
                objectData.summary.usedByAgents = data.agents.map(agent => {
                    agent.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(agent.type, agent.backendType, agent.predictionType, agent.savedModelType, null, 20);
                    return agent;
                });
            }).error(setErrorInScope.bind(scope));

            // load knowledge bank if any
            if (objectData.type === 'VectorStoreSearch' && objectData.params.knowledgeBankRef) {
                DataikuAPI.retrievableknowledge.get(objectData.projectKey, objectData.params.knowledgeBankRef).success(knowledgeBank => {
                    objectData.summary.usesKnowledgeBank = knowledgeBank;
                }).error(setErrorInScope.bind(scope));
            }

            // load saved model if any
            if (objectData.type === 'ClassicalPredictionModelPredict' && objectData.params.smRef) {
                DataikuAPI.savedmodels.get(objectData.projectKey, objectData.params.smRef).success(savedModel => {
                    objectData.summary.usesSavedModel = savedModel;
                    objectData.summary.usesSavedModel.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(savedModel.type, savedModel.backendType, savedModel.predictionType, savedModel.savedModelType, null, 20);
                }).error(setErrorInScope.bind(scope));
            }

            // load dataset if any
            if ((objectData.type === 'DatasetRowAppend' || objectData.type === 'DatasetRowLookup') && objectData.params.datasetRef) {
                const datasetLoc = DatasetUtils.getLocFromSmart(objectData.projectKey, objectData.params.datasetRef);
                DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.name, objectData.projectKey).success(dataset => {
                    objectData.summary.usesDataset = dataset;
                }).error(setErrorInScope.bind(scope));
            }
        }
    });

    app.service('TypeMappingService', function(PluginsService, RecipeDescService, TYPE_MAPPING, LoggerProvider, DashboardUtils, WebAppsService, MLUtilsService, AgentToolService, LLMUtilsService) {

        this.mapRecipeTypeToName = RecipeDescService.getRecipeTypeName;

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapDatasetTypeToName = (type) => typeProcessor(type, TYPE_MAPPING.DATASET_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, LoggerProvider.getLogger('DisplayFilters'),
        defaultValueForDataset);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapInsightTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.DASHBOARDS_OR_INSIGHTS_TYPES, undefined, defaultValueForInsight(type), size);


        /**
         * @param {string} type
         * @param {number} size - Only used for new icons only, like data collections.
         * @returns {string}
         */
        this.mapTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.ALL_TYPES, undefined, defaultValueForDataset, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapOldIconTypeToModernIcon = (oldIcon, size = 16) => {
            // sometimes the input is not defined, make sure we don't make an error in that case
            if(oldIcon === undefined || oldIcon === null || oldIcon === '') {
                return oldIcon;
            }

            // Note: oldIcon may contain multiple classes
            const iconClasses = oldIcon.split(' ');

            // Find the first class that is a modern icon and add a size if missing
            const index = iconClasses.findIndex(iconClass => iconClass.startsWith('dku-icon'));
            if (index > -1) {
                const modernIcon = iconClasses[index];
                // check if icon is missing a size suffix
                if (!/(12|16|20|24|32|48)$/.test(modernIcon)) {
                    iconClasses[index] = modernIcon + '-' + size;
                }
            } else {
                // Check each icon for a modern equivalent and convert the first valid one -- otherwise just return oldIcon
                const mappedIconClasses = iconClasses.map(iconClass => TYPE_MAPPING.NEW_ICON_TYPE_MAPPING[iconClass]);
                const mappedIndex = mappedIconClasses.findIndex(iconClass => !!iconClass);
                if (mappedIndex === -1) {
                    if (window.devInstance) {
                        LoggerProvider.getLogger('icons-migration').warn(`Old icon "${oldIcon}" mapping to modern icon unsupported."`)
                    }
                } else {
                    iconClasses[mappedIndex] = `${mappedIconClasses[mappedIndex]}-${size}`;
                }
            }

            return iconClasses.join(' ');
        }

        /**
         * @param {string} subtype
         * @param {string} type
         * @param {number} size - Only used for new icons only, like data collections.
         * @returns {string}
         */
        this.mapSubtypeToIcon = (subtype, type, size, suffix) => {
            var subtypeFilters = {
                'INSIGHT': this.mapInsightTypeToIcon,
                'DATASET': this.mapDatasetTypeToIcon,
                'RECIPE': this.mapRecipeTypeToIcon,
                'WEB_APP': this.mapWebappTypeToIcon,
                'STREAMING_ENDPOINT': this.mapDatasetTypeToIcon,
                'SAVED_MODEL': this.mapSavedModelMLCategoryToIcon
            };
            if (subtype && type && subtypeFilters[type.toUpperCase()]) {
                return subtypeFilters[type.toUpperCase()](subtype, size, suffix);
            } else {
                return this.mapTypeToIcon(type, size);
            }
        }

        /**
         * @param {string} type
         * @param (boolean) forDetailView
         * @returns {string}
         */
        this.mapConnectionTypeToName = (type, forDetailView) => {
            const conversionField = forDetailView ? TYPE_MAPPING.CONVERSION_FIELD_NAME : TYPE_MAPPING.CONVERSION_FIELD_OTHER_NAME;
            return typeProcessor(type, TYPE_MAPPING.CONNECTION_TYPES, conversionField, undefined, defaultValue);
        }

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapConnectionTypeToNameForList = (type) => this.mapConnectionTypeToName(type, false);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapConnectionTypeToNameForItem = (type) => this.mapConnectionTypeToName(type, true);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapConnectionTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.CONNECTION_TYPES, LoggerProvider.getLogger('connectionTypeToIcon'), defaultValue, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapCredentialTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.CREDENTIAL_TYPES, LoggerProvider.getLogger('credentialTypeToIcon'), defaultValue, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapDatasetTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.DATASET_TYPES, LoggerProvider.getLogger('datasetTypeToIcon'), defaultValueForDataset, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.RECIPE_TYPES, undefined, defaultValueForRecipe, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapModelTypeToIcon = (type, size) => this.iconProcessor(type, TYPE_MAPPING.ML_TYPES, undefined, defaultValueForRecipe, size);

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToLanguage = (type) => {
            function defaultToUndefined() {
                return undefined;
            }
            return typeProcessor(type,TYPE_MAPPING. RECIPE_TYPES, TYPE_MAPPING.CONVERSION_FIELD_LANGUAGE, undefined, defaultToUndefined);
        }

        /**
         * @param {string} type
         * @returns {string} one of TYPE_MAPPING.RECIPE_CATEGORIES.CODE or undefined if unknown or custom
         */
        this.mapRecipeTypeToCategory = (type) => {
            const recipeTypeObject = TYPE_MAPPING.RECIPE_TYPES[type];
            return recipeTypeObject ? recipeTypeObject.category : undefined;
        }


        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToColorClass = (type) => {
            const customRecipeColors = ['red', 'pink', 'purple', 'blue', 'green', 'sky', 'yellow', 'orange', 'brown', 'gray'];

            if (!type) {
                return 'recipe-custom';
            }
            if (!RecipeDescService.isRecipeType(type) && type !== 'labeling') {
                return;
            }

            const loadedDesc = PluginsService.getRecipeLoadedDesc(type);
            if (loadedDesc) {
                let colorClass = 'recipe-custom';

                if (loadedDesc && loadedDesc.desc && loadedDesc.desc.meta) {
                    const iconColor = loadedDesc.desc.meta.iconColor;

                    if (customRecipeColors.indexOf(iconColor) > -1) {
                        colorClass = colorClass + '-' + iconColor;
                    }
                }
                return colorClass;
            }

            if (type.startsWith('App_')) {
                return 'app';
            }

            return 'recipe-' + typeProcessor(type, TYPE_MAPPING.RECIPE_TYPES, TYPE_MAPPING.CONVERSION_FIELD_CATEGORY, undefined, () => '');
        }

        /**
         * @param {string} type
         * @param {boolean} background
         * @returns {string}
         */
        this.mapTypeToColor = (type, background = false) => {
            const supportedTypes = [
                'project',
                'dataset',
                'streaming_endpoint',
                'recipe',
                'analysis',
                'notebook',
                'scenario',
                'saved_model',
                'model_evaluation_store',
                'genai_evaluation_store',
                'model_comparison',
                'genai_comparison',
                'prompt_studio',
                'retrievable_knowledge',
                'agent_tool',
                'managed_folder',
                'web_app',
                'report',
                'dashboard',
                'insight',
                'article',
                'labeling_task',
                'app'];


            if (!type) return '';

            const recipeColor = this.mapRecipeTypeToColor(type);
            if (recipeColor) {
                return recipeColor;
            }

            const stdType = getStandardType(type);
            if (stdType && supportedTypes.indexOf(stdType) >= 0) {
                return background ? 'universe-background ' + stdType : 'universe-color ' + stdType;
            }
            return '';
        }

        this.mapEvaluationStoreToClassColor = (subType) => {
            const color = subType == "LLM" || subType == "AGENT" ? 'genai_evaluation_store' : 'model_evaluation_store';
            return `universe-color ${color}`;
        }

        this.mapComparisonToClassColor = (subType) => {
            const color = subType == "LLM" || subType == "AGENT" ? 'genai_comparison' : 'model_comparison';
            return `universe-color ${color}`;
        }

        /**
         *
         * @param {string} subtype a subtype to be mapped.
         * @param {string} type the "parent" type.
         * @returns  {string}
         */
        this.mapSubtypeToColor = (subtype, type) => {
            const subtypeFilters = {
                'INSIGHT': this.mapInsightTypeToColor,
                'RECIPE': this.mapRecipeTypeToColor,
                'WORKSPACE_STORY': () => 'workspace-object-datasory-icon__color',
                'SAVED_MODEL': (subType) => this.mapSavedModelTypeToClassColor(subType, true),
                'MODEL_EVALUATION_STORE': (subType) => this.mapEvaluationStoreToClassColor(subType),
                'MODEL_COMPARISON': (subType) => this.mapComparisonToClassColor(subType),
              };
            if (subtypeFilters[type.toUpperCase()]) {
                return subtypeFilters[type.toUpperCase()](subtype);
            } else {
                return this.mapTypeToColor(type);
            }
        }

        /**
         *
         * @param {string} taskType
         * @param {string} predictionType
         * @returns {string}
         */
        this.mapMlTaskTypeToIcon = (taskType, predictionType) => {
            if (!taskType || !predictionType) {
                return;
            }
            if (taskType.toLowerCase() == 'clustering') {
                return 'icon-clustering';
            }
            // TODO @deepHub icons for new prediction types? Otherwise, fallback
            return 'icon-prediction-' +predictionType.toLowerCase();
        }

        /**
         *
         * @param {type} taskType
         * @returns {string}
         */
        // TODO @deepHub icons for new backend? Otherwise, fallback
        this.mapBackendTypeToIcon = (type, size) => 'icon-ml ' + this.iconProcessor(type ? type.toLowerCase() : '', TYPE_MAPPING.BACKEND_TYPES, undefined, defaultValue, size);

        /**
         *
         * @param {string} taskType
         * @param {string} backendType
         * @returns {string}
         */
        this.mapAnalysisTypeToIcon = (taskType, backendType, predictionType, size) => {
            const mlCategory = MLUtilsService.getMLCategory(taskType, backendType, predictionType);

            return this.iconProcessor(mlCategory, TYPE_MAPPING.VISUAL_ANALYSIS_TYPES, undefined, defaultValue, size);
        }

        /**
         *
         * @param {string} mlCategory
         * @param {number} size
         * @returns {string}
         */
        this.mapSavedModelMLCategoryToIcon = (mlCategory, size, suffix) => {
            return this.iconProcessor(mlCategory, TYPE_MAPPING.SAVED_MODEL_TYPES, undefined, defaultValue, size, suffix);
        }

        /**
         *
         * @param {string} taskType
         * @param {string} backendType
         * @param {string} predictionType
         * @param {string} savedModelType
         * @param {string} proxyModelProtocol
         * @param {number} size
         * @returns {string}
         */
        this.mapSavedModelSubtypeToIcon = (taskType, backendType, predictionType, savedModelType, proxyModelProtocol, size) => {
            const mlCategory = MLUtilsService.getMLCategory(taskType, backendType, predictionType, savedModelType, proxyModelProtocol);
            return this.iconProcessor(mlCategory, TYPE_MAPPING.SAVED_MODEL_TYPES, undefined, defaultValue, size);
        }

        this.mapSavedModelTypeToClassColor = (savedModelType, withUniverse) => {
            let classColor = withUniverse ? 'universe-color ' : ' ';
            // FINE_TUNING is technically an ML category and not a type (we already treat subtypes as ml categories in mapSubtypeToIcon)
            classColor += ['LLM_GENERIC', 'FINE_TUNING', 'PYTHON_AGENT', 'PLUGIN_AGENT', 'TOOLS_USING_AGENT', 'RETRIEVAL_AUGMENTED_LLM'].includes(savedModelType) ? 'llm-saved-model' : 'saved-model';
            return classColor;
        };

        this.mapSavedModelMLCategoryToColor = (mlCategory, withUniverse) => {
            let classColor = withUniverse ? 'universe-color ' : ' ';
            classColor += ['fine_tuning', 'python_agent', 'plugin_agent', 'tools_using_agent', 'retrieval_augmented_llm'].includes(mlCategory) ? 'llm-saved-model' : 'saved-model';
            return classColor;
        };

        this.mapAgentToolTypeToIcon = (agentToolType, size) => {
            const uiData = AgentToolService.getAgentToolTypesUiDataForType(agentToolType);
            if (uiData === undefined) {
                return "";
            }
            return uiData.icon.replace(/20$/, size);
        };

        this.mapLlmIdToIcon = (llmId, size) => {
            const llmType = LLMUtilsService.mapIdToType(llmId);
            return this.iconProcessor(llmType, TYPE_MAPPING.LLM_TYPES, undefined, defaultValueForLlm, size);
        };

        /**
         *
         * @param {string} type
         * @returns {string}
         */
        this.mapToFsProviderDisplayName = (type) => typeProcessor(type, TYPE_MAPPING.FS_PROVIDER_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, undefined, defaultValue);

        /**
         *
         * @param {string} type
         * @returns {string}
         */
        this.mapToNiceType = (type) => typeProcessor(type, TYPE_MAPPING.DATASET_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, undefined,
            this.mapConnectionTypeToNameForItem.bind(this))

        /**
         *
         * @param {string} type
         * @param {boolean} noBackground
         * @returns {string}
         */
        this.mapInsightTypeToColor = (type, noBackground = true) => {
            let color = '';
            switch (type) {
                case 'text':
                case 'iframe':
                case 'image':
                color = 'comments';
                break;
                default:
                color = (DashboardUtils.getInsightHandler(type) || {}).color;
                break;
            }

            return !noBackground ? 'universe-background insight-icon ' + color : color;
        }

        /**
         *
         * @param {string} type
         * @returns {string}
         */

        this.mapInsightTypeToDisplayableName = (type) => {
            var handler = DashboardUtils.getInsightHandler(type);
            if (!handler || !handler.name) return type;
            return handler.name;
        }

        this.mapWebappTypeToIcon = (type, size) => {
            return this.iconProcessor(type,  TYPE_MAPPING.WEBAPPS_TYPES, undefined, defaultValueForWebApp(type), size);
        }

        this.mapWebappTypeToColor = (type) => {
            if (WebAppsService.getBaseType(type) == type) {
                return 'notebook'; //native webapp => code color
            } else {
                return 'flow';
            }
        }

        this.mapWebappTypeToName = (type) => {
            return typeProcessor(type, TYPE_MAPPING.WEBAPPS_TYPES, TYPE_MAPPING.CONVERSION_FIELD_NAME, undefined,
                defaultValueForWebApp(type));
        }

        /**
         * @param {string} type
         * @returns {string}
         */
        this.mapRecipeTypeToColor = (type) => {
            if (!type) {
                return 'universe-color recipe-custom';
            }
            const colorClass = this.mapRecipeTypeToColorClass(type);
            if (!colorClass) {
                return '';
            }
            return 'universe-color ' + colorClass;
        }

        /**
         * @param {string} objectType
         * @returns {string}
         */
        this.mapChartTypeToColor = (objectType) => {
            if (objectType === 'dataset') {
                return 'dataset';
            } else if (objectType === 'analysis') {
                return 'analysis';
            }
            return 'chart';
        }

        /**
         *
         * @param {string} type
         * @returns {string}
         */
        const getStandardType = (type) => {
            if (!type) return '';
            if (type.endsWith('_NOTEBOOK')) {
                return 'notebook';
            }
            return type.toLowerCase();
        }

        /**
         *
         * @param {string} type
         * @param {cord<string, any>} types
         * @param {string} conversionField
         * @param {Logger} Logger
         * @param {Function} defaultValueFunction
         * @returns {string}
         */
        const typeProcessor = (type, types, conversionField, Logger, defaultValueFunction) => {
            if(!type) {
                return '';
            }
            const existingType = types[type.toLowerCase()];
            const result = existingType && existingType[conversionField];
            if (result !== undefined) {
                return result;
            }
            return defaultValueFunction(type, conversionField, Logger);
        }

        /**
         * Returns a function to be used by a mapXTypeToIcon filter.
         *
         * @param {string} type
         * @param {Record<string, any>} types
         * @param {Logger} Logger
         * @param {Function} defaultValueFunction
         * @param {number} size the size of the produced icon (only used for modern icons)
         * @returns {string}
         */
        this.iconProcessor = (type, types, Logger, defaultValueFunction, size, suffix = '') =>  {
            const newIconOnly = typeProcessor(type, types, 'newIconOnly', Logger, () => false);
            const icon = typeProcessor(type, types, TYPE_MAPPING.CONVERSION_FIELD_ICON, Logger, defaultValueFunction);

            if (newIconOnly) {
                if (!size) {
                    LoggerProvider.getLogger('icons-migration').warn(`New icon mapping used without size for type ${type}."`);
                    size = 16;
                }

                return icon + '-' + (suffix ? suffix + '-' : '') + size;
            }

            return icon;
        }

        /**
         *
         * @param {string} key
         * @param {string} conversionField
         * @param {Logger} Logger
         * @returns {string}
         */
        const defaultValue = (key, conversionField, Logger) => {
            const defaultPrefix = TYPE_MAPPING.CONVERSION_FIELD_ICON === conversionField ? 'icon-' : '';
            const result = defaultPrefix + key.toLowerCase();
            if (Logger !== undefined) {
                Logger.error('Unknown type: ' + key + '.Returning default value: ' + result);
            }
            return result;
        }


        const defaultValueForDataset = (originalKey, conversionField, Logger) => {
            const key = originalKey.toLowerCase();
            if (key.startsWith('custom') || key.startsWith('fsprovider_') || key.startsWith('sample_')) {
                if (TYPE_MAPPING.CONVERSION_FIELD_ICON === conversionField) {
                    return PluginsService.getDatasetIcon(originalKey);
                } else {
                    return PluginsService.getDatasetLabel(originalKey);
                }
            }
            return defaultValue(key, conversionField, Logger);
        }

        const defaultValueForRecipe = (originalKey, conversionField, Logger) => {
            const key = originalKey.toLowerCase();
            if (key.startsWith('custom')) {
                if (TYPE_MAPPING.CONVERSION_FIELD_ICON === conversionField) {
                    return PluginsService.getRecipeIcon(originalKey);
                }
            } else if (key.startsWith('app_')) {
                for (let ar of window.dkuAppConfig.appRecipes) {
                    if (ar.recipeType == originalKey) {
                        return ar.icon;
                    }
                }
            }
            return defaultValue(key, conversionField, Logger);
        }


        const defaultValueForWebApp = (type) => {
            return function(originalKey, conversionField, Logger) {
                if (conversionField === TYPE_MAPPING.CONVERSION_FIELD_ICON) {
                    return WebAppsService.getWebAppIcon(originalKey) || 'icon-code';
                } else if (conversionField === TYPE_MAPPING.CONVERSION_FIELD_NAME) {
                    return WebAppsService.getWebAppTypeName(originalKey) || type;
                } else {
                    return defaultValue(originalKey.toLowerCase(), conversionField, Logger);
                }
            };
        }

        const defaultValueForInsight = (type) => {
            return function() {
                return (DashboardUtils.getInsightHandler(type) || {}).icon;
            }
        }

        const defaultValueForLlm = () => {
            return function() {
                return 'dku-icon-question-circle-outline';
            }
        }
    });

    app.service('PrettyPrintDoubleService', function() {
        // regexp to find string values that are doubles in scientific notation
        const scientificNotationRegexp = /^[+-]?\d*(\.\d*)?[eE][+-]?\d+$/;

        // Keep this regex in sync with java class FrenchDoubleMeaning (note that we
        // removed the ? after the exponential term because we only want to detect
        // scientific notation numbers with an exponential)
        const frenchDoubleScientificNotationRegexp = /^[+-]?([0-9]{1,3} ([0-9]{3} )*)?[0-9]+(([eE][+-]?[0-9]+)$|,[0-9]+([eE][+-]?[0-9]+))$/

        // convert a double as non-scientific notation if exponent value is <= 15. Otherwise, rawValue is unchanged
        function avoidScientificNotation(rawValue) {
            let formatter = null;
            let doubleValue = null;

            if(rawValue && scientificNotationRegexp.test(rawValue)) {
                // Regular Decimal number
                // only apply on values the look like a scientific notation

                // we force en-US as a base in order to have consistent UI for all users
                formatter = new Intl.NumberFormat('en-US', {
                    numberingSystem: 'standard',
                    useGrouping: false,
                    minimumFractionDigits: 1,
                    maximumFractionDigits: 15,
                });

                doubleValue = parseFloat(rawValue); // might be nan for non-number cells
            } else if(rawValue && frenchDoubleScientificNotationRegexp.test(rawValue)) {
                // Decimal (comma), a.k.a FrenchDoubleMeaning
                // only apply on values the look like a scientific notation

                // we force fr-FR as a base in order to have consistent UI for all users
                formatter = new Intl.NumberFormat('fr-FR', {
                    numberingSystem: 'standard',
                    useGrouping: true,
                    style: "decimal",
                    minimumFractionDigits: 1,
                    maximumFractionDigits: 15,
                });

                doubleValue = parseFloat(rawValue.replace(/ /g, "").replace(/,/g,".")); // might be nan for non-number cells
            }

            if(formatter !== null && doubleValue !== null && !isNaN(doubleValue) && Math.abs(doubleValue)>=1e-15 && Math.abs(doubleValue)<1e16) {
                return formatter.format(doubleValue)
            }

            return rawValue;
        }

        // apply avoidScientificNotation to all columns that have a float or double type
        function patchFormulaPreview(previewTable, sampleType, columns) {
            previewTable.colNames.forEach((colName, colIdx) => {
                const colWithType = colIdx === 0 // first column is always output, but is not named
                    ? {type: sampleType}
                    : columns.find(col => col.name === colName);
                const colType = colWithType && colWithType.type;
                if(colType === 'double' || colType === 'float') {
                    previewTable.rows.forEach(row => {
                        row.data[colIdx] = avoidScientificNotation(row.data[colIdx]);
                    });
                }
            })
        }

        return {
            avoidScientificNotation,
            patchFormulaPreview,
        };
    })
    app.service('Deprecation', [function() {
        const deprecation = {};

        deprecation.isPythonDeprecated = (pythonInterpreter) => ["PYTHON27", "PYTHON34", "PYTHON35", "PYTHON36", "PYTHON37", "PYTHON38"].includes(pythonInterpreter);

        return deprecation;

    }]);
    app.service('PathUtils', [function() {
        const pathUtils = {};

        //set trailing slash
        pathUtils.makeT = (path) =>
            path && path.substr(-1) === '/' ? path : (path || '') + '/';

        //set no trailing slash
        pathUtils.makeNT = (path) =>
            (path || '').replace(/\/+$/, '');

        //set leading slash
        pathUtils.makeL = (path) =>
            path && path[0] === '/' ? path : '/' + (path || '');

        //set no leading slash
        pathUtils.makeNL = (path) =>
            (path || '').replace(/^\/+/, '');

        //set leading & trailing slash
        pathUtils.makeLT = (path) =>
            pathUtils.makeT(pathUtils.makeL(path));

        //set no leading & no trailing slash
        pathUtils.makeNLNT = (path) =>
            pathUtils.makeNT(pathUtils.makeNL(path));

        //concat path segments with no leading slash & no trailing slash
        pathUtils.concatNLNT =  (...paths) =>
           paths
               .filter(p => p)                        //remove undefined/null/empty entries
               .map(p => p.replace(/^\/+|\/+$/g, '')) //remove leading & ending slashes
               .filter(p => p.length>0)               //remove empty entries
               .join('/');

        //concat path segments with leading slash & no trailing slash
        pathUtils.concatLNT =  (...paths) =>
           '/' + pathUtils.concatNLNT(...paths);

        pathUtils.absPathFromUserPath = (usrPath, currentAbsPath) => {
            const path = (usrPath || '').trim();
            if (!path) return '/'; //discutable (legacy?) behavior, '' means '/'
            if (path[0] === '/') return pathUtils.makeLT(path);
            return pathUtils.concatLNT(currentAbsPath, path);
        };

        return pathUtils;
    }]);

app.service('ClipboardReadWriteService', function(ClipboardUtils, CreateModalFromTemplate) {
    const writeItemsToClipboard = function(items, separator = "\n") {
        let toCopy = "";
        if (items && items.length) {
            toCopy = items.join(separator);
        }
        ClipboardUtils.copyToClipboard(toCopy);
    }

    const readItemsFromNavigatorClipboard = async function() {
        if (!navigator.clipboard || !navigator.clipboard.readText) {
            throw new Error('Browser does not support the readText API');
        }

        try {
            const data = await navigator.clipboard.readText();
            if (!data) return;

            let items = data.trim().split(/[\r\n]+/);
            if (items && Array.isArray(items) && items.length > 0) {
                if (items.length === 1) {
                    items = items[0].split(",").map(e => e.trim());
                }
                return items;
            }
        } catch (error) {
            throw new Error('Failed to read from clipboard');
        }
    };

    const readItemsUsingPopup = async function(scope, popupHeader) {
        return new Promise((resolve, reject) => {
            let newScope = scope.$new();
            CreateModalFromTemplate("/templates/list-copy-paste-modal.html", newScope, 'PasteModalController', function(modalScope) {
                modalScope.header = popupHeader;

                // Override the onPasteText defined in the PasteModalController
                modalScope.onPasteText = function(event) {
                    let data = event.originalEvent.clipboardData.getData('text/plain');
                    if (!data) {
                        modalScope.uiState.hasError = true;
                        return;
                    }
                    try {
                        let stringArray = data.trim().split(/[\r\n]+/);
                        if (!stringArray || !Array.isArray(stringArray) || stringArray.length === 0) {
                            modalScope.uiState.hasError = true;
                            reject(new Error('Not a valid array'));
                            return;
                        }

                        if (stringArray.length === 1) {
                            stringArray = stringArray[0].split(",").map(e => e.trim());
                        }

                        let classes = stringArray.map((class_label, index) => ({ class_label, index }));
                        if (!modalScope.validateData(classes)) {
                            modalScope.uiState.hasError = true;
                            reject(new Error('Invalid data'));
                            return;
                        }

                        modalScope.uiState.editMode = false;
                        modalScope.uiState.hasError = false;
                        modalScope.uiState.items = classes;
                    } catch (error) {
                        modalScope.uiState.hasError = true;
                        reject(error);
                    }
                };

                modalScope.pasteItems = function() {
                    resolve(modalScope.uiState.items.map(item => item.class_label));
                };

                modalScope.validateData = (classes) => {
                    try {
                        return classes.every(aClass => aClass.class_label.length > 0);
                    } catch (e) {
                        return false;
                    }
                }
            });
        });
    }

    const readItemsFromClipboard = async function(scope, popupHeader) {
        try {
            return await readItemsFromNavigatorClipboard();
        } catch (_) {
            return await readItemsUsingPopup(scope, popupHeader)
        }
    }

    return {
        writeItemsToClipboard: writeItemsToClipboard,
        readItemsFromClipboard: readItemsFromClipboard
    }

});


app.service("FilePatternUtils", function(){

    //if there is a char like this in the glob string we need to replace - Does not include * or ? which mean something in GLOB
    const REGEXP_CHARS_PATTERN = /[/\-\\^$+.()|[\]{}]/g;

    //Replace with escaped match
    const REGEXP_CHARS_REPLACEMENT = '\\$&';

    // basically matches /**/ or **/ at the beginning but final slash not included in the capture and clever things so it is not inefficient
    const DEEP_WILDCARD_PATTERN = /(?<=^|\/)\*\*(?=$|\/)/g;

    const srv = {
        /**
         * Convert a glob string to a regexp to use on file paths.
         * It is not a global regexp, it is not stateful, and either matches the whole path string or not.
         * E.g. use `.test(path)` to see if this matches given path.
         *
         * Follows implemention in com.dataiku.dip.fs.FileSelectionRule.java
         *
         * @param {string} expr GLOB string
         * @returns {RegExp}
         * @throws exception if expr could not be compiled as a regexp
         */
        fullPathGlobRegExp: function(expr) {
            expr = expr.trim();
            DEEP_WILDCARD_PATTERN.lastIndex = 0;
            let specs = expr.replaceAll(/\/+/g, "/").split(DEEP_WILDCARD_PATTERN);
            let globRegExpString = "^";

            if (expr === "**" || expr === "**/*") {
                // Simplest to hard code these to match anything
                globRegExpString += ".*";
            } else {
                if (expr.startsWith("**/")) {
                    globRegExpString += "(?:^|.*?/)"; // can be any prefix path, including empty
                    specs = specs.slice(1);
                    if (specs.length > 0) {
                        specs[0] = specs[0].substring(1); // eat up the leading / from the next spec, / included above
                    }
                }

                for (let i = 0; i < specs.length; i++) {
                    let spec = specs[i];
                    if (i > 0) {
                        globRegExpString += ".*?(?<=\\/)";  // `**` is any char sequence denoting intermediate directories, including empty
                        spec = spec.substring(1); // eat up the leading / from the upcoming spec, / mandatory above
                    }

                    REGEXP_CHARS_PATTERN.lastIndex = 0;
                    globRegExpString += spec.replaceAll(REGEXP_CHARS_PATTERN, REGEXP_CHARS_REPLACEMENT)
                                        .replaceAll("?", "[^/]")     // `?` is any char but /
                                        .replaceAll("*", "[^/]*");  // `*` is any char sequence (including empty) without /   (NOTE: removed  ? from end of replacement that is in the backend)
                }

                if (expr.endsWith("/**")) {
                    // note this is .++ in backend, the extra + for possessive, but this is not possible in JS
                    globRegExpString += ".+";  // `**` is any char sequence
                }
            }

            globRegExpString += "$";

            // g flag not needed (and we'd would have to reset the lastIndex every time)
            return new RegExp(globRegExpString, "i");
        },

        /**
         * Convert a glob string to a regexp to use on file names (not whole paths just the name)
         * It is not a global regexp, it is not stateful, and either matches the whole name string or not.
         * E.g. use `.test(filename)` to see if this matches given name.
         *
         * Follows implemention in com.dataiku.dip.fs.FileSelectionRule.java - globChunkMatches()
         *
         * @param {string} expr GLOB string for a file name (not a path)
         * @returns {RegExp}
         * @throws exception if expr could not be compiled as a regexp
         */
        fileNameGlobRegExp: function(expr) {
            REGEXP_CHARS_PATTERN.lastIndex = 0;
            let globRegexString = "^"+ expr.replaceAll(REGEXP_CHARS_PATTERN, '\\$&')
                               .replaceAll("?", ".")    // `?` is any char
                               .replaceAll("*", ".*?")  // `*` is any char sequence (including empty)
                            + "$";

            // g flag not needed (and we'd would have to reset the lastIndex every time)
            return new RegExp(globRegexString, "i");
        },

        // Removes leading and trailing `/` slashes and collapses multiple to one where they are retained
        // Returns empty string if nothing left after this
        // Equivalent to PathUtils.slashes(path, false, false, true, "");
        normalizeSlashes: function (path) {
            const trimmedPath =  path.trim();
            if (!trimmedPath) return null;
            const collapsedPath  = path.trim().replaceAll(/\/+/g, "/");
            const slashAtStart = collapsedPath.startsWith("/");
            const slashAtEnd = collapsedPath.endsWith("/");

            if (slashAtStart) {
                return slashAtEnd ? collapsedPath.slice(1, -1) : collapsedPath.slice(1);
            } else {
                return slashAtEnd ?  collapsedPath.slice(0, -1) : collapsedPath;
            }
        },

        /**
         * @param {string} path - file path ending in file name -  must be normalised with normalizeSlashes
         * @returns file name at the end of the path with no slashes
         */
        extractFileNameFromPath: function (path) {
            if (!path) return null;
            return path.substring(path.lastIndexOf('/') + 1);
        }
    };
    return srv;
});

app.service('DataikuCloudService', function($q, DataikuAPI, $rootScope) {
    this.isDataikuCloud = function() {
        return $rootScope.appConfig.deploymentMode === 'CLOUD';
    };

    this.getLaunchpadUrl = function() {
        if (!this.isDataikuCloud()) {
            return;
        }

        const saasFrontendHook = window.dkuSaas || {}

        const host = saasFrontendHook.API_HOST || ''; // format is usually api-XXXXXXXX-YYYYYYY, though "api" can change
        const hostMatch = host.match(/[a-z-]*-.{8}-(.{8})/);
        const spaceId = hostMatch ?  (hostMatch.length > 1 ? hostMatch[1] : '') : '';

        return `${saasFrontendHook.URL_CONSOLE}/spaces/${spaceId}`;
    }

    this.getCloudInfo = function() {
        const info = {
            isDataikuCloud: this.isDataikuCloud(),
            isSpaceAdmin: $rootScope.appConfig.admin
        };
        const deferred = $q.defer();
        deferred.resolve(info);
        return deferred.promise;
    }
});

app.service('PuppeteerLoadedService', function (Debounce) {
    this.getDebouncedSetField = function($scope, $element, loadedStateField) {
        const setPuppeteerField = function() {
            const thisLoadedStateField = loadedStateField ? loadedStateField : "puppeteerHook_elementContentLoaded";

            $scope[thisLoadedStateField] = true;
            // in cases where we set a specific field, we set it also as an attribute on the element to be used by the puppeteer code css selector
            if (loadedStateField) {
                $element.attr(loadedStateField, true);
            }
        };

        return Debounce().withDelay(50,200).wrap(setPuppeteerField);
    };
});

//this is a singleton to share the rating feedback parameters for all AI features
app.factory("RatingFeedbackParams", () => {
    const state = {
        requestIdForFeedback: null,
        featureRated: null,
        showRatingFeedback: false,
    };

    return state;
});

app.factory("AvailableLanguages", ($rootScope) => {
    let languageList = [{id: 'en', label: 'English'}, {id: 'ja', label: '日本語'}, { id: 'fr', label: 'Français' }]
    languageList.sort((a, b) => a.label.localeCompare(b.label));
    return languageList;
});

app.service('ProfileService', function ($rootScope) {
    this.isTechnicalAccount = function () {
        return $rootScope?.appConfig?.userProfile?.profile === 'TECHNICAL_ACCOUNT';
    }
});

app.factory('AccessibleObjectsCacheService', function(DataikuAPI, $q) {
    const TYPE_WITHOUT_PROJECT_KEY = ['PROJECT', 'WORKSPACE', 'DATA_COLLECTION', 'APP'];
    const EMPTY_ARRAY = [];
    const EMPTY_CACHE_ENTRY = { value: EMPTY_ARRAY, promise: $q.when(EMPTY_ARRAY) }; // avoid having subsequent calls with the same param return a different ref

    const svc = {
        /**
         * Get a cached lazy accessible object getter. API query are done automatically on request, but shared between many calls with the same parameters.
         * Designed to share API call data between many object-picker instances or similar in order to avoid doing the same API queries multiple times
         *
         * Usage example to select many projects without doing one project list call per selector:
         * $scope.getAccessibleObjects = AccessibleObjectsCacheService.createCachedGetter('READ', setErrorInScope.bind($scope)).asValue;
         * (...)
         *  <div ng-for="item of manySelectors"
         *     type="PROJECT" object-picker="item.projectKey"
         *     input-available-objects="getAccessibleObjects('PROJECT')"
         *  ></div>
         *
         * @param {string} mode 'READ' or 'WRITE' to filter on readable or writeable objects (default 'READ')
         * @param {function} setErrorInScopeFunction If provided, used in API call catch statement to set API call error in the scope
         * @returns {object} Object containing two functions (keys `asValue` or `asPromise`)
         *
         * The asValue function:
         *      @param {string} type Type of the accessible objects to list (e.g. 'PROJECT', 'WORKSPACE', 'DATASET', 'WEB_APP', ...)
         *      @param {string} projectKey (optional for types that are not project objects)
         *      @returns {AccessibleObject[]} empty array while API query is pending, then the list of object once API call is finished (do not snapshot the returned value!)
         *
         * The asPromise function:
         *      @param {string} type Type of the accessible objects to list (e.g. 'PROJECT', 'WORKSPACE', 'DATASET', 'WEB_APP', ...)
         *      @param {string} projectKey (optional for types that are not project objects)
         *      @returns {Promise<AccessibleObject[]>} Promise of the list of objects (do not snapshot the returned value!)

         */
        createCachedGetter(mode = "READ", setErrorInScopeFunction = () => {}) {
            const cache = {};
            return {
                asValue: (type, projectKey) => getAccessibleObjects(type, projectKey, mode, cache, setErrorInScopeFunction).value,
                asPromise: (type, projectKey) => getAccessibleObjects(type, projectKey, mode, cache, setErrorInScopeFunction).promise,
            };
        },
    }

    return svc;

    //////////////

    function getAccessibleObjects(type, projectKey, mode, accessibleObjectsCache, setErrorInScopeFunction) {
        if (!type) {
            return EMPTY_CACHE_ENTRY;
        }

        if(TYPE_WITHOUT_PROJECT_KEY.includes(type)) {
            if (angular.isDefined(accessibleObjectsCache[type])) {
                return accessibleObjectsCache[type];
            }

            const listingPromise = DataikuAPI.taggableObjects.listAccessibleObjects(undefined, type, mode).then(function({data}) {
                accessibleObjectsCache[type].value = data;
                return data;
            }).catch(setErrorInScopeFunction);
            accessibleObjectsCache[type] = { value: EMPTY_ARRAY, promise: listingPromise };

            return accessibleObjectsCache[type];
        } else {
            if(!projectKey) {
                return EMPTY_CACHE_ENTRY;
            }
            if(!angular.isDefined(accessibleObjectsCache[type])) {
                accessibleObjectsCache[type] = {}
            }
            if(angular.isDefined(accessibleObjectsCache[type][projectKey])) {
                return accessibleObjectsCache[type][projectKey];
            }

            const listingPromise = DataikuAPI.taggableObjects.listAccessibleObjects(projectKey, type, mode).then(function({data}) {
                accessibleObjectsCache[type][projectKey].value = data;
                return data;
            }).catch(setErrorInScopeFunction);
            accessibleObjectsCache[type][projectKey] = { value: EMPTY_ARRAY, promise: listingPromise };

            return accessibleObjectsCache[type][projectKey]
        }
    }
});

app.service('SemanticVersionService', function ($rootScope) {
    this.compareVersions = function (version1, version2) {
            const splitVersion = (v) => v.split('.').map(Number);
            const [v1, v2] = [splitVersion(version1), splitVersion(version2)];

            for (let i = 0; i < Math.max(v1.length, v2.length); i++) {
                const num1 = v1[i] || 0;
                const num2 = v2[i] || 0;

                if (num1 > num2) return 1;
                if (num1 < num2) return -1;
            }
            return 0;
        }
});

app.service('ProjectStandardsService', function() {
    const STATUS = {
        RUN_SUCCESS: 'RUN_SUCCESS',
        RUN_ERROR: 'RUN_ERROR',
    };

    function mapSeverity(result) {
        switch (result.status) {
            case STATUS.RUN_SUCCESS:
                return result.severity;
            default:
                return 0;
        }
    }

    function getProjectStandardsSummaryFromReport(projectStandardsRunReport) {
        const bundleChecksRunInfo = Object.values(projectStandardsRunReport.bundleChecksRunInfo);
        const worstSeverityNumber = Math.max(
                ...bundleChecksRunInfo.map(runInfo =>
                    mapSeverity(runInfo.result)
                ),
                0
            );
        return {
            worstSeverityNumber: worstSeverityNumber,
            checksInError: bundleChecksRunInfo.filter(c => c.result.status === STATUS.RUN_ERROR).length,
            checkCount: bundleChecksRunInfo.length,
        }
    }

    function getProjectStandardsSummaryFromRunReportSummary(projectStandardsRunReportSummary) {
        const checkCount = Object.values(projectStandardsRunReportSummary.nbOfChecksBySeverity).reduce((acc, val) => acc + val, 0)
                             + projectStandardsRunReportSummary.nbOfChecksNotApplicable
                             + projectStandardsRunReportSummary.nbOfChecksInError;
        const severitiesWithAtLeastOneCheck = Object.entries(projectStandardsRunReportSummary.nbOfChecksBySeverity)
                                      .filter(([key, value]) => value > 0)
                                      .map(([key, value]) => key);
        return {
            worstSeverityNumber:  Math.max(...severitiesWithAtLeastOneCheck, 0),
            checksInError: projectStandardsRunReportSummary.nbOfChecksInError,
            checkCount: checkCount,
        }
    }

    return {
        getProjectStandardsSummaryFromReport: getProjectStandardsSummaryFromReport,
        getProjectStandardsSummaryFromRunReportSummary: getProjectStandardsSummaryFromRunReportSummary,
        STATUS: STATUS
    }
});

/**
 * Utility for retrieving permission information (e.g. list of users, groups) required for populating permission
 * form and caching it, such that multiple consumers can leverage it without having to fetch the information
 * multiple times from the backend.
 *
 * This can be useful when you need to build multiple components that need this information on the same page, e.g.
 * in plugin presets.
 *
 * Can retrieve:
 *  * `getUsersAndGroups` which returns {allUsers: ..., allUsersLogin: ..., allGroups: ...}
 *  * `canAdminPlugin` which returns a boolean
 **/
app.factory('PermissionsDataFetcher', function(DataikuAPI, $q) {

    function _cachedDataFetcher(promiseFunction, containingScope) {
        let cache = {};
        let runningPromise = {};

        return function(arg = null) {
            const cacheKey = arg ? arg : "__DKU_CACHE_KEY";
            if (cache[cacheKey]) {
                return $q.when(cache[cacheKey].value);
            }

            // In case another consumer already triggered the promise but result is not there yet, and then not cached,
            // reusing the same promise rather than re-triggering the promise.
            if (runningPromise[cacheKey]) {
                return runningPromise[cacheKey].promise;
            }
            const deferred = $q.defer();
            runningPromise[cacheKey] = deferred;

            const promiseCall = arg ? promiseFunction(arg) : promiseFunction();
            promiseCall
                .then(function(result) {
                    // Wrapping result in a simple object to make sure it resolves to true when checking it,
                    // and not bother, even when it happens that `result` is the `false` boolean.
                    cache[cacheKey] = {value: result};
                    deferred.resolve(result);
                })
                .catch(setErrorInScope.bind(containingScope))
                .finally(function() {
                    runningPromise[cacheKey] = null;
                });

            return deferred.promise;
        }
    }

    return function(containingScope) {
        const usersAndGroupsPromise = () => {
            return $q.all([DataikuAPI.security.listUsers(),
                           DataikuAPI.security.listGroups(false)])
                .then(function(results) {
                    let users = results[0].data;
                    let groups = results[1].data;
                    let sortedUsers = users.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    return {
                        allUsers: sortedUsers,
                        allUsersLogin: sortedUsers.map(user => '@' + user.login),
                        allGroups: groups
                    }
                });
        };
        const canAdminPluginPromise = (pluginId) => {
            return DataikuAPI.plugins.canAdminPlugin(pluginId).then(function(result) {
                return result.data;
            })
        };
        const usersAndGroupsFetcher = _cachedDataFetcher(usersAndGroupsPromise, containingScope);
        const canAdminPluginDataFetcher = _cachedDataFetcher(canAdminPluginPromise, containingScope);

        return {
            getUsersAndGroups: usersAndGroupsFetcher,
            canAdminPlugin: canAdminPluginDataFetcher
        }
    }
});

app.service('ConnectionsService', function() {
    this.getCredentialsError = function(errorMessage) {
        if (!errorMessage) {
            return null;
        }

        const regex = /User '(.*?)' does not have credentials for connection '(.*?)' to access (.*)/;
        const match = errorMessage.match(regex);
        if (match) {
            const username = match[1];
            const connectionName = match[2];
            const connectionType = match[3];
            if (connectionType === "RemoteMCP") {
                return {
                    payload: {
                        connectionName: connectionName,
                        user: username,
                        sameUser: true,
                    },
                    fixability: "USER_FILL_CREDENTIALS",
                    code: "ERR_CONNECTION_NO_CREDENTIALS",
                }
            }
        }
        return null;
    };
});

})();

;
function gentleTruncate(str, len) {
    /**
     * Truncate a string to make sure it takes at most
     * n characters.
     * Whenever possible truncates on special chars.
     *
     * If str is not a string, returns str unchanged.
     */
    if ((typeof str != "string") || (str.length <= len)) {
        return str;
    }

    // eslint-disable-next-line no-useless-escape
    var cutOn = /[ ,\.,;\-\\\"\n\?\!\|]/g
    var truncated = str.substring(0, len-1);
    var lastSeparatorIdx = regexLastIndexOf(cutOn, truncated);

    // we don't want to cut more too much.
    if (lastSeparatorIdx > len / 2) {
        truncated = str.substring(0, lastSeparatorIdx);
    }
    return truncated + '…';
}

var now = new Date().getTime();
var NOW_REFRESH_INTERVAL = 1000 * 5; // 5 Seconds

setInterval(function(){
    now = new Date().getTime();
}, NOW_REFRESH_INTERVAL);

function friendlyDuration(diffInSeconds, translate) {
    const sec = Math.floor((diffInSeconds >= 60 ? diffInSeconds % 60 : diffInSeconds));
    const min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    const hrs = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 24 ? diffInSeconds % 24 : diffInSeconds);
    const days = Math.floor((diffInSeconds = (diffInSeconds / 24)) >= 30 ? diffInSeconds % 30 : diffInSeconds);
    const months = Math.floor((diffInSeconds = (diffInSeconds / 30)) >= 12 ? diffInSeconds % 12 : diffInSeconds);
    const years = Math.floor(diffInSeconds / 12);
    const values = { years: years, months: months, days: days, hours: hrs, minutes: min, seconds: sec };
    if (years > 0) {
        if (years <= 6 && months > 0) {
            return translate("FILTER.FRIENDLY_DURATION.YEARS_MONTHS", "{{years}} {{years === 1 ? 'year': 'years'}} and {{months}} {{months === 1 ? 'one month': (months + ' months')}}", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.YEARS", "{{years}} {{years === 1 ? 'year': 'years'}}", values);
        }
    } else if (months > 0) {
        if (months <= 6 && days > 0) {
            return translate("FILTER.FRIENDLY_DURATION.MONTHS_DAYS", "{{months}} {{months === 1 ? 'month': 'months'}} and {{days === 1 ? 'one day' : (days + ' days')}}", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.MONTHS", "{{months}} {{months === 1 ? 'month': 'months'}}", values);
        }
    } else if (days > 0) {
        if (days <= 3 && hrs > 0) {
            return translate("FILTER.FRIENDLY_DURATION.DAYS_HOURS", "{{days}} {{days === 1 ? 'day': 'days'}} and {{hours === 1 ? 'one hour' : (hours + ' hours')}}", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.DAYS", "{{days}} {{days === 1 ? 'day': 'days'}}", values);
        }
    } else if (hrs > 0) {
        if (min > 1) {
            return translate("FILTER.FRIENDLY_DURATION.HOURS_MINUTES", "{{hours}} {{hours === 1 ? 'hour': 'hours'}} and {{minutes}} minutes", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.HOURS", "{{hours}} {{hours === 1 ? 'hour': 'hours'}}", values);
        }
    } else if (min > 0) {
        if (sec > 1) {
            return translate("FILTER.FRIENDLY_DURATION.MINUTES_SECONDS", "{{minutes}} {{minutes === 1 ? 'minute': 'minutes'}} and {{seconds}} seconds", values);
        } else {
            return translate("FILTER.FRIENDLY_DURATION.MINUTES", "{{minutes}} {{minutes === 1 ? 'minute': 'minutes'}}", values);
        }
    } else {
        return translate("FILTER.FRIENDLY_DURATION.ABOUT_SECONDS", "about {{seconds <= 1 ? 'a second' : (seconds + ' seconds')}}", values);
    }
}

/* eslint-disable-next-line no-redeclare */
function durationHHMMSS(diffInSeconds) {
    var sec = Math.floor((diffInSeconds >= 60 ? diffInSeconds % 60 : diffInSeconds));
    var min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    var hours = Math.floor( diffInSeconds / 60);
    var sb = "";
    if (hours > 0) {
        sb += (hours + "h ");
    }
    if (min > 0) {
        sb += (min + "m ");
    }
    sb += (sec + "s");
    return sb;
}

/* eslint-disable-next-line no-redeclare */
function durationHHMM(diffInSeconds) {
    var min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    var hours = Math.floor( diffInSeconds / 60);
    var sb = "";
    if (hours > 0) {
        sb += (hours + "h ");
    }
    if (min > 0) {
        sb += (min + "m ");
    }
    return sb;
}

function durationHHMMSSPadded(diffInSeconds) {
    if (diffInSeconds == 0) diffInSeconds = 1;
    var sec = Math.floor((diffInSeconds >= 60 ? diffInSeconds % 60 : diffInSeconds));
    var min = Math.floor((diffInSeconds = (diffInSeconds / 60)) >= 60 ? diffInSeconds % 60 : diffInSeconds);
    var hours = Math.floor( diffInSeconds / 60);

    function pad(number) {
        if (number < 10) return "0" + number;
        else return number;
    }
    return pad(hours) + "h" + pad(min) + "m" + pad(sec) + "s";
}


function friendlyDurationShort(seconds, ref, noSeconds, translate) {
    const sec = Math.floor(seconds >= 60 ? seconds % 60 : seconds);
    const min = Math.floor((seconds = (seconds / 60)) >= 60 ? seconds % 60 : seconds);
    const hours = Math.floor((seconds = (seconds / 60)) >= 24 ? seconds % 24 : seconds);
    const days = Math.floor((seconds = (seconds / 24)) >= 30 ? seconds % 30 : seconds);
    const months = Math.floor((seconds = (seconds / 30)) >= 12 ? seconds % 12 : seconds);
    const years = Math.floor((seconds = (seconds / 12)));
    const values = { years: years, months: months, days: days, hours: hours, minutes: min, seconds: sec };
    let sb = "";

    if (years > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.YEARS", "{{years}} {{years === 1 ? 'year': 'years'}}", values);
    } else if (months > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.MONTHS", "{{months}} {{months === 1 ? 'month': 'months'}}", values);
    } else if (days > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.DAYS", "{{days}} {{days === 1 ? 'day': 'days'}}", values);
    } else if (hours > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.HOURS", "{{hours}} {{hours === 1 ? 'hour': 'hours'}}", values);
    } else if (min > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.MINUTES", "{{minutes}} {{minutes === 1 ? 'minute': 'minutes'}}", values);
    } else if(!noSeconds && sec > 0) {
        sb = translate("FILTER.FRIENDLY_DURATION.SECONDS", "{{seconds}} {{seconds === 1 ? 'second' : 'seconds'}}", values);
    }

    switch (ref) {
        case 'ago':
            return sb
                ? translate("FILTER.FRIENDLY_DURATION.DURATION_AGO", "{{duration}} ago", { duration: sb })
                : translate("FILTER.FRIENDLY_DURATION.JUST_NOW", "just now");
        case 'in':
            return sb
                ? translate("FILTER.FRIENDLY_DURATION.IN_DURATION", "in {{duration}}", { duration: sb })
                : translate("FILTER.FRIENDLY_DURATION.IMMEDIATELY", "immediately");
        default:
            return sb ? sb : (noSeconds
                ? translate("FILTER.FRIENDLY_DURATION.LESS_THAN_ONE_MIN", "< 1 minute")
                : translate("FILTER.FRIENDLY_DURATION.LESS_THAN_ONE_SEC", "< 1 second"));
    }
}

function dateDayDiff(date1, date2) {
    var d1 = new Date(date1);
    var d2 = new Date(date2);
    d1.setHours(0);
    d1.setMinutes(0);
    d1.setSeconds(0);
    d1.setMilliseconds(0);
    d2.setHours(0);
    d2.setMinutes(0);
    d2.setSeconds(0);
    d2.setMilliseconds(0);
    var dayLength = 24*60*60*1000;
    return Math.floor(d1.getTime()/dayLength) - Math.floor(d2.getTime()/dayLength);
}
function dateMinuteDiff(date1, date2) {
    var d1 = new Date(date1);
    var d2 = new Date(date2);
    d1.setSeconds(0);
    d1.setMilliseconds(0);
    d2.setSeconds(0);
    d2.setMilliseconds(0);
    var msToMin = 60*1000;
    return Math.floor(d1.getTime()/msToMin) - Math.floor(d2.getTime()/msToMin);
}


(function() {

'use strict';

var app = angular.module('dataiku.filters', [ 'dataiku.constants' ]);

app.filter('slugify', function(SlugifyService){
    return SlugifyService.transform;
});


app.filter('join', function(){
    return function(input){
        return input.join(",");
    };
});


app.filter('capitalize', function(){
    return function(input){
        if(input && input.length>0) {
            return input.charAt(0).toUpperCase() + input.slice(1);
        } else {
            return '';
        }
    };
});

app.filter('capitalizeWord', function() {
    return function(input) {
        input = input.toLowerCase();
        return (input || '').split(' ').reduce((result, word) => {
            return result + ' ' + word[0].toUpperCase() + word.slice(1);
        }, '');
    };
});


app.filter('pluralize', function() {
    /**
     * Pluralize an item name.
     * @param {number}            num         - The quantity of the item
     * @param {string}            singular    - The singular form of the item name (only used when num is worth 1)
     * @param {string}            plural      - The plural form of the item name
     * @param {d3 formatter}      [format]    - Optional d3.js formatter for num
     * @param {boolean | string}  [no]        - Optional indicator of the filter behavior when num is worth 0:
     *                                        If false (default), use '0 ' + plural
     *                                        If true, use 'no ' + plural
     *                                        If a string, use it
     */
    return function(num, singular, plural, format, no) {
        if (no && num == 0) return no === true ? 'no ' + plural : no;
        return (format ? d3.format(format)(num) : num) + " " + (num === 1 ? singular : plural);
    }
});


app.filter('plurify', function($translate) {
    return function(singular, num, plural) {
        const usePlural = num > 1 || (num === 0 && ($translate && $translate.proposedLanguage() || $translate.use() || "en") === 'en'); // in english zero uses plural mode (crazy guys)
        return usePlural ? (typeof (plural) !== "undefined" ? plural : singular + 's') : singular;
    };
});

app.filter('breakify', function (LoggerProvider) {
    return function (text, breakOnRegex) {
        try {
            const re = new RegExp(breakOnRegex, 'g');
            const indices = [];
            let m;
            do {
                m = re.exec(text);
                if (m) {
                    indices.push(m.index + m[0].length);
                }
            } while (m);
            indices.reverse().forEach(pos => {
                if (text) {
                    text = [text.slice(0, pos), '<wbr>', text.slice(pos)].join('');
                }
            });
            return text;
        } catch(err) {
            LoggerProvider.getLogger('DisplayFilters').error("Error ", err);
            return text;
        }
    };
});

app.filter('uncamel', function (LoggerProvider) {
    return function (text) {
        try {
            return text.replace(/[A-Z]/g, function(l) { return ' ' + l.toLowerCase(); })
        } catch(err) {
            LoggerProvider.getLogger('DisplayFilters').error("Error ", err);
            return text;
        }
    };
});

app.filter('objectSize', function() {
    return function(object) {
        if (!object) return 0;
        return Object.keys(object).length;
    };
});


app.filter("objectKeys", function() {
    return function(obj) {
        if (!obj) {
            return [];
        }
        return Object.keys(obj);
    }
});


app.filter('listSlice', function(){
    return function(input, from, to) {
        if (!input || input.length <= from) {
            return [];
        }
        return input.slice(from, to);
    }
});


// `[n] | range`        => 0 .. (n-1)
// `[from, to] | range` => from .. to (inclusive)
app.filter('range', function() {
    return function(input) {
        switch (input.length) {
            case  1:
                if (input[0] <= 0) return [];
                input = [0, input[0] - 1]; break;
            case  2:
                if (input[1] < input[0]) return [];
                break;
            default: return input;
        }
        var len = input[1] - input[0] + 1,
            result = new Array(len);
        for (var i = 0; i < len; i++) {
            result[i] = i + input[0];
        }
        return result;
    };
});


app.filter('gentleTruncate', function () {
    return function (text, length, end) {

        if (isNaN(length))
            length = 10;

        return gentleTruncate(text, length);

    };
});


app.filter('onlyNumbers', function(){
    return function(text){
        if (!text) { return ''; }
        return text.replace(/[^0-9]/g, '');
    };
});


app.filter('ordinal', function(){
    return function(number){
        return number < 14 && number > 10 ? 'th' : ['th','st','nd','rd','th','th','th','th','th','th'][number % 10];
    };
});


app.filter('ordinalWithNumber', function(){
    return function(number){
        return number + (number < 14 && number > 10 ? 'th' : ['th','st','nd','rd','th','th','th','th','th','th'][number % 10]);
    };
});

// ratio to % w/ fixed precision & optional non-breaking space
// special cases: '<1%' and '>99%' when <.01 but >0 (for precision == 2, '<0.01%'...)
app.filter('smartPercentage', function() {
    return function(ratio, precision, spaces) {
        precision = Math.max(+(precision || 0), 0);
        var tens = Math.pow(10, precision),
            min = 1 / tens / 100, // e.g. precision = 2 =>  0.01
            max = 1 - min,        //                    => 99.99
            out = [];
        if (ratio < 1 && ratio > max) {
            ratio = max;
            out.push('>');
        } else if (ratio > 0 && ratio < min) {
            ratio = min;
            out.push('<');
        }
        out.push((Math.round(ratio * 100 * tens) / tens).toFixed(precision), '%');
        return out.join(spaces ? '\u00A0' : '');
    }
});


app.filter('processorByType', function(){
    return function(processors, type){
        if (processors) {
            for (var i = 0; i < processors.processors.length; i++) {
                if (processors.processors[i].type == type) {
                    return processors.processors[i];
                }
            }
        }
        throw Error("Unknown processor " +type);
    }
});


app.filter('cleanFacetValue', function(DKU_NO_VALUE, DKU_OTHERS_VALUE) {
    return function(input) {
        if (input === DKU_NO_VALUE) {
            return "<em>No value</em>";
        } else if (input === DKU_OTHERS_VALUE) {
            return "<em>Others</em>";
        } else {
            return input;
        }
    };
});

app.filter('chartLabelValue', function(ChartLabels) {
    /**
     * @param {AxisElt}                             input the label object from the backend.
     * @param {NumberFormattingOptions | undefined} formattingOptions the number formatting options.
     * @param {number | undefined}                  minValue the minimum value.
     * @param {number | undefined}                  maxValue the maximum value.
     * @param {number | undefined}                  numValues the number of values.
     */
    return function(input, formattingOptions, minValue, maxValue, numValues) {
        return ChartLabels.getFormattedLabel(input, formattingOptions, minValue, maxValue, numValues);
    };
});


app.filter("prettyjson", function(){
    return function(input) {
         return JSON.stringify(input,undefined,3);
    }
});


app.filter("jsonOrString", function(){
    return function(input) {
        if (typeof input == "string") {
            return input;
        }
        else {
            return JSON.stringify(input);
        }
    }
});


app.filter('friendlyTime', function(translate) {
    return function(input) {
        var diffInSeconds = parseInt(input, 10) / 1000;
        return friendlyDuration(diffInSeconds, translate);
    };
});


app.filter('friendlyTimeDeltaForward', function(translate) {
    const filter =  function(input) {
        var diffInSeconds = (parseInt(input, 10) - now) / 1000;
        return translate("FILTER.FRIENDLY_TIME_DELTA_FORWARD", "in {{period}}", {period: friendlyDuration(diffInSeconds, translate)});
    };
    filter.$stateful = true;
    return filter;
});


app.filter('friendlyTimeDelta', function(translate) {
    const filter = input => {
        var diffInSeconds = (now - parseInt(input, 10)) / 1000;
        return translate("FILTER.FRIENDLY_TIME_DELTA", "{{period}} ago", {period: friendlyDuration(diffInSeconds, translate)});
    };
    filter.$stateful = true;
    return filter;
});


app.filter('friendlyTimeDeltaShort', function(translate) {
    const filter = function(input) {
        var diffInSeconds = (now - parseInt(input, 10)) / 1000;
        return friendlyDurationShort(diffInSeconds, 'ago', true, translate);
    };
    filter.$stateful = true;
    return filter;
});

app.filter('friendlyTimeDeltaHHMMSS', function() {
    const filter = function(input) {
        var diffInSeconds = Math.max(0, (now - parseInt(input, 10)) / 1000);
        return durationHHMMSS(diffInSeconds);
    };
    filter.$stateful= true;
    return filter;
});

app.filter('friendlyDuration', function(translate){
    return function (input) { return friendlyDuration(parseInt(input, 10) / 1000, translate); };
});


app.filter('friendlyDurationShort', function(translate) {
    return function(input, ref, noSeconds) {
        return friendlyDurationShort(parseInt(input, 10) / 1000, ref, noSeconds, translate);
    };
});


app.filter('friendlyDurationSec', function(translate) {
    return function(input) {
        return friendlyDuration(parseInt(input, 10), translate);
    };
});


app.filter('durationHHMMSS', function() {
    return function(input) {
        return durationHHMMSS(parseInt(input, 10)/1000);
    };
});


app.filter('durationHHMMSSPadded', function() {
    return function(input) {
        return durationHHMMSSPadded(parseInt(input, 10)/1000);
    };
});


app.filter('yesNo', function(){
    return function(input) {
        if (input) return "Yes";
        else return "No";
    }
});

app.filter('friendlyDate', function($filter, translate) {
    return function(time, format) {
        var today = new Date();
        var date = new Date(time);
        const defaultFormat = date.getFullYear() === today.getFullYear()
            ? 'EEEE, d MMMM'
            : 'EEEE, d MMMM y';
        format = format || defaultFormat;
        if(dateDayDiff(date, today)===0){
            return translate('FILTER.FRIENDLY_DATE.TODAY', 'Today')
        } else if(dateDayDiff(date, today)===-1){
            return translate('FILTER.FRIENDLY_DATE.YESTERDAY', 'Yesterday')
        } else if(dateDayDiff(date, today)===1){
            return translate('FILTER.FRIENDLY_DATE.TOMORROW', 'Tomorrow')
        } else {
            return $filter('date')(date, format);
        }
    };
});

app.filter('dateDayDiff', function() {
    return function(time) {
        var today = new Date();
        var date = new Date(time);
        return dateDayDiff(date, today);
    };
});


app.filter('friendlyDateRange', function($filter) {
    const day = $filter('friendlyDate');
    const time = date => $filter('date')(date, 'HH:mm');
    function duration(date1, date2, options) {
        if (options && options.noDuration || !date2){
            return '';
        }
        let delta = (new Date(date2) - new Date(date1));
        return ' (duration: ' + $filter('durationHHMMSS')(delta) + ')';
    }

    function unavailableEndDate(date) {
        return day(date) + ', started at ' +  time(date);
    }
    function sameMinute(date1, date2, options) {
        return day(date1) + ' ' +  time(date1) + duration(date1, date2, options);
    }
    function sameDay(date1, date2, options) {
        return day(date1) + ', ' +  time(date1) + ' to ' + time(date2) + duration(date1, date2, options);
    }
    function differentDays(date1, date2, options) {
        return 'From ' + day(date1) + ' ' +  time(date1)+ ' to ' + day(date2) + ' ' +  time(date2) + duration(date1, date2, options);
    }

    return function(date1, date2, options) {
        options = options || {};
        if (!date2) {
            return unavailableEndDate(date1);
        } else if (dateMinuteDiff(date1, date2) == 0) {
            return sameMinute(date1, date2, options);
        } else if (dateDayDiff(date1, date2) == 0) {
            return sameDay(date1, date2, options);
        } else {
            return differentDays(date1, date2, options);
        }
    };
});


app.filter('friendlyDateTime', function($filter, $translate, translate) {
    var sameDay = function (date1, date2) {
        return date1.getFullYear() === date2.getFullYear() &&
               date1.getMonth() === date2.getMonth() &&
               date1.getDate() === date2.getDate();
    }
    return function(time, format, timeFormat) {
        var today = new Date(),
            yesterday = new Date(),
            tomorrow = new Date();
        yesterday.setDate(yesterday.getDate() - 1);
        tomorrow.setDate(tomorrow.getDate() + 1);

        const date = new Date(time);
        let datePart;
        if (sameDay(date, today)) {
            datePart = translate('FILTER.FRIENDLY_DATE.TODAY', 'Today')
        } else if (sameDay(date, yesterday)) {
            datePart = translate('FILTER.FRIENDLY_DATE.YESTERDAY', 'Yesterday')
        } else if (sameDay(date, tomorrow)) {
            datePart = translate('FILTER.FRIENDLY_DATE.TOMORROW', 'Tomorrow')
        } else {
            if (!format) {
                const lang = $translate && $translate.proposedLanguage() || $translate.use() || "en-US";
                datePart = date.toLocaleString(lang, { weekday: "long", year: "numeric", month: "long", day: "numeric", });
            } else {
                datePart = $filter('date')(date, format);
            }
        }

        const timePart = $filter('date')(date, timeFormat || 'HH:mm');
        return translate("FILTER.FRIENDLY_DATE.DATE_TIME_JOINER", "{{date}} at {{time}}", {date:datePart, time:timePart});
    };
});

app.filter('dateUTC', function($filter) {
    return function(time, format) {
        return $filter('date')(time, format, 'UTC');
    }
});

app.filter('utcDate', function() {
    return function(time, format) {
        format = format || 'EEEE, d MMMM';
        return moment.utc(time).format(format);
    };
});

app.filter('typeToName', function($filter) {
    return function(input) {
        if (!input) {
            return '';
        }
        return $filter('capitalizeWord')($filter('niceConst')(input));
    }
});

app.filter('groupToName', function() {
    return function(input) {
        return input === "$$ALL_USERS$$" ? "All Users" : input;
    }
});


app.filter('recipeTypeToName', function(RecipeDescService) {
    return RecipeDescService.getRecipeTypeName;
});


app.filter('datasetTypeToName', function(TypeMappingService) {
    return function(type) {
        return TypeMappingService.mapDatasetTypeToName(type)
    };
});

app.filter('columnTypeToName', function(TypeMappingService) {
    return function(type) {
        if (type === 'date') {
            return 'datetime\u00A0with\u00A0tz'; // yes, it's a nbsp. Leave it there, otherwise these types get line-wrapped in small spaces
        } else if (type === 'dateonly') {
            return 'date\u00A0only'; // yes, it's a nbsp. Leave it there, otherwise these types get line-wrapped in small spaces
        } else if (type === 'datetimenotz') {
            return 'datetime\u00A0no\u00A0tz'; // yes, it's a nbsp. Leave it there, otherwise these types get line-wrapped in small spaces
        } else {
            return type;
        }
    };
});


app.filter('pluginIdToName', function(LoggerProvider, PluginsService) {
    return function(pluginId) {
        if(!pluginId) {
            return '';
        }
        let desc = PluginsService.getPluginDesc(pluginId);
        if (desc) {
            return desc.label;
        }
        return pluginId;
    };
});


app.filter("niceProfileName", function(){
    var dict = {
        /* Generic */
        "READER" : "Reader",
        "AI_CONSUMER" : "AI Consumer",
        "AI_ACCESS_USER" : "AI Access User",
        "NONE": "None",

        /* Legacy (2019 and before) profiles */
        "DATA_SCIENTIST": "Data scientist",
        "DATA_ANALYST": "Data analyst",

        /* 2020-FY2024 */
        "DESIGNER" : "Designer",
        "VISUAL_DESIGNER": "Visual Designer",
        "PLATFORM_ADMIN": "Platform admin",
        "EXPLORER": "Explorer",

        /* FY2025 */
        "DATA_DESIGNER" : "Data Designer",
        "ADVANCED_ANALYTICS_DESIGNER": "Advanced Analytics Designer",
        "FULL_DESIGNER": "Full Designer",
        "GOVERNANCE_MANAGER": "Governance Manager",
        "TECHNICAL_ACCOUNT": "Technical Account"
    };
    return function(input) {
        return dict[input] || input;
    };
});

app.filter('itemToColor', function(typeToColorFilter) {
    return function(item) {
        return typeToColorFilter(item.type);
    };
});


app.filter('typeToColor', function($filter) {
    const supportedTypes = [
        "project",
        "dataset",
        "streaming_endpoint",
        "recipe",
        "analysis",
        "notebook",
        "scenario",
        "saved_model",
        "model_evaluation_store",
        "genai_evaluation_store",
        "model_comparison",
        "genai_comparison",
        "managed_folder",
        "web_app",
        "code_studio",
        "report",
        "dashboard",
        "insight",
        "article",
        "app",
        "labeling_task",
        "prompt_studio",
        "agent_tool",
        "retrievable_knowledge"
    ];

    function getStandardType(type) {
        if (!type) return;
        if (type.endsWith("_NOTEBOOK")) {
            return "notebook";
        }
        return type.toLowerCase();
    }
    return function(type, background) {
        if (!type) return "";
        
        // Special case for GEN_AI_MODEL and AI_AGENT
        if (type === 'GEN_AI_MODEL' || type === 'AI_AGENT') {
            return 'agent-tool';
        }
        
        const recipeColor = $filter('recipeTypeToColor')(type);
        if (recipeColor) {
            return recipeColor;
        }

        const stdType = getStandardType(type);
        if (supportedTypes.indexOf(stdType) >= 0) {
            return background ? "universe-background " + stdType : "universe-color " + stdType;
        }
        return "";
    };
});


app.filter('insightTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapInsightTypeToIcon;
});


app.filter('insightTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapInsightTypeToColor;
});

app.filter('chartTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapChartTypeToColor;
});

app.filter('insightTypeToDisplayableName', function(TypeMappingService) {
    return TypeMappingService.mapInsightTypeToDisplayableName;
});

app.filter('webappTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapWebappTypeToIcon;
});

app.filter('webappTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapWebappTypeToColor;
});

app.filter('webappTypeToName', function(TypeMappingService) {
   return TypeMappingService.mapWebappTypeToName;
});

app.filter('connectionTypeToNameForList', function(TypeMappingService) {
    return TypeMappingService.mapConnectionTypeToNameForList;
});
app.filter('connectionTypeToNameForItem', function(TypeMappingService) {
    return TypeMappingService.mapConnectionTypeToNameForItem;
});

app.filter("connectionTypeToIcon", function(TypeMappingService ){
     return TypeMappingService.mapConnectionTypeToIcon;
});

app.filter("credentialTypeToIcon", function(TypeMappingService ){
    return TypeMappingService.mapCredentialTypeToIcon;
});

app.filter('datasetTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapDatasetTypeToIcon;
});

app.filter('recipeTypeToColorClass', function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToColorClass;
});

app.filter('recipeTypeToColor', function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToColor;
});
app.filter('recipeTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToIcon;
});


app.filter('modelTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapModelTypeToIcon;
});

app.filter("recipeTypeToLanguage", function(TypeMappingService) {
    return TypeMappingService.mapRecipeTypeToLanguage;
});

app.filter('typeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapTypeToIcon;
});

/* used to translate old icons names provided by TypeMappingService to new icon collection name */
app.filter('toModernIcon', function(TypeMappingService) {
    return TypeMappingService.mapOldIconTypeToModernIcon;
})

app.filter('subTypeToIcon', function(TypeMappingService) {
  return TypeMappingService.mapSubtypeToIcon;
});

app.filter('subTypeToColor', function(TypeMappingService) {
   return TypeMappingService.mapSubtypeToColor;
});

app.filter('mlTaskTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapMlTaskTypeToIcon;
});


app.filter('backendTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapBackendTypeToIcon;
});

app.filter('analysisTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapAnalysisTypeToIcon;
});

app.filter('savedModelSubtypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapSavedModelSubtypeToIcon;
});

app.filter('savedModelTypeToClassColor', function(TypeMappingService) {
    return TypeMappingService.mapSavedModelTypeToClassColor;
});

app.filter('savedModelMLCategoryToClassColor', function(TypeMappingService) {
    return TypeMappingService.mapSavedModelMLCategoryToColor;
});

app.filter('agentToolTypeToIcon', function(TypeMappingService) {
    return TypeMappingService.mapAgentToolTypeToIcon;
});

app.filter('llmIdToIcon', function(TypeMappingService) {
    return TypeMappingService.mapLlmIdToIcon;
});

// Boldify a pattern in the input.
//
// Note:
// - Input is plain text (and not HTML, because it will be escaped)
// - Output is HTML (sanitized, ready for display)
app.filter('boldify', function(){
    function preg_quote( str ) {
        // eslint-disable-next-line no-useless-escape
        return (str+'').replace(/([\\\.\+\*\?\[\^\]\$\(\)\{\}\=\!\<\>\|\:])/g, "\\$1");
    }

    return function(input, replacements) {
        if (!replacements || replacements.length == 0 || !input) return sanitize(input);
        // Implementation notes:
        // - It is not possible to escape HTML entities and then boldify the pattern: this may produce invalid HTML if the pattern matches)
        // - It is not possible to boldify the pattern and then escape HTML entities: the boldification will be escaped
        // => Strategy is to split the string into tokens. Tokens are sanitized individually, and then enclosed in <b>...</b> if they match the pattern.

        var regex = new RegExp(replacements.map(function(e){
            if (e instanceof RegExp) {
                return "(?:"+e.source+")";
            } else {
                return "(?:"+preg_quote(e)+")";
            }
        }).join("|"), "gi");

        const highlightedSections = [];
        const rawTokenBoundaries = [0, input.length];

        input.replace(regex, function(val, pos) {
            highlightedSections.push([pos, pos + val.length]);
            rawTokenBoundaries.push(pos);
            rawTokenBoundaries.push(pos + val.length);
        });

        const tokenBoundaries = _.chain(rawTokenBoundaries).uniq().sortBy().value();

        let output = '';
        for(let i = 1; i < tokenBoundaries.length; i++) {
            const tokStartPos = tokenBoundaries[i-1];
            const tokEndPos = tokenBoundaries[i];
            let tokHighlighted = false;

            for(let j = 0; j < highlightedSections.length; j++) {
                const hlStartPos = highlightedSections[j][0];
                const hlEndPos = highlightedSections[j][1];

                if(hlStartPos < tokEndPos && tokStartPos < hlEndPos) {
                    tokHighlighted = true;
                    break;
                }
            }

            const token = input.substring(tokStartPos, tokEndPos);

            if(tokHighlighted) output += '<b>';
            output += sanitize(token);
            if(tokHighlighted) output += '</b>';
        }

        return output;
    }
});

app.filter('connectionNameFormatter', function () {
    const virtualConnectionRegex = "^@virtual\\((.+)\\):(connection:)?(.+)$";

    return function (input) {
        if (!input) return input;
        const match = input.match(virtualConnectionRegex);
        return match ? `Hive (${match[3]})` : input;
    }
});

app.filter('niceType', function(TypeMappingService) {
    return TypeMappingService.mapToNiceType;
});

app.filter('niceLogin', function() {
    return (login) => login === 'no:auth' ? 'Dataiku' : login;
});

app.filter('nicePrecision', function() {
    return function(val, p) {
        if (val == undefined || val != val || val == null || !val.toFixed)
            return undefined;
        if (Math.abs(val) < Math.pow(10, p)) {
            if (Math.round(val) == val) {
                /* Don't add stuff to integers */
                return val.toFixed(0);
            } else {
                return val.toPrecision(p);
            }
        } else {
            return val.toFixed(0);
        }
    };
});

app.filter('ifEmpty', function() {
    return (value, defaultValue) => value == null ? defaultValue : value;
});

app.filter('niceConst', function(Ng2NiceConstService) {
    return Ng2NiceConstService.transform.bind(Ng2NiceConstService);
});

app.filter('niceMLBackendType', function(LoggerProvider) {
    var niceNames = {
        'PY_MEMORY': 'Python (in memory)',
        'MLLIB': 'Spark MLLib',
        'H2O': 'Sparkling Water (H2O)',
        'VERTICA': 'Vertica Advanced Analytics',
        'DEEP_HUB': 'Deep Learning (PyTorch)'
    };

    var Logger = LoggerProvider.getLogger('DisplayFilters');

    return function(input) {
        if (!niceNames[input]) {
            Logger.warn("ML backend has no display name: "+input);
            return input;
        } else {
            return niceNames[input];
        }
    }
});


// input | replace:search_str:replacement
// input | replace:search_regexp:flags:replacement
app.filter('replace', function() {
    return function(input, regexp, flags, replace) {
        if (typeof replace === 'undefined') {
            replace = flags;
        } else {
            regexp = new RegExp(regexp, flags);
        }
        return regexp ? input.replace(regexp, replace) : input;
    }
});


app.filter('startAt', function() {
    return function(input, first) {
        first = parseInt(first, 10);
        var out = [];
        if (first >= input.length) {
            return out;
        }
        for (let i = first; i<input.length; i++) {
            out.push(input[i]);
        }
        return out;
    };
});


app.filter('shakerStepIcon', function(ShakerProcessorsUtils) {
    return function(step, size) {
        return ShakerProcessorsUtils.getStepIcon(step.type, step.params, size);
    };
});


app.filter('shakerStepDescription', function(ShakerProcessorsUtils, $filter) {
    return function(step, processors) {
        return ShakerProcessorsUtils.getHtmlStepDescriptionOrError(step, processors);
    };
});


app.filter('objToParamsList', function(){
    return function(obj) {
        var arr = [];
        $.each(obj, function(key, val){
            arr.push(key + ": " + val);
        });
        return arr.join(', ');
    }
});


app.filter('filesize', function(){
    return function(size){
        if (size >= 1024*1024*1024) {
            return Math.round(size / 1024 / 1024 / 1024 * 100)/100 + ' GB';
        } else if (size >= 1024*1024){
            return Math.round(size / 1024 / 1024*100)/100 + ' MB';
        } else {
            return Math.round(size / 1024 *100)/100 + ' KB';
        }
    };
});


app.filter('fileSizeOrNA', function(){
    return function(size){
        if (size < 0) {
            return "N/A";
        } else if (size >= 1024*1024*1024) {
            return Math.round(size / 1024 / 1024 / 1024 * 100)/100 + ' GB';
        } else if (size >= 1024*1024){
            return Math.round(size / 1024 / 1024*100)/100 + ' MB';
        } else {
            return Math.round(size / 1024 *100)/100 + ' KB';
        }
    };
});


app.filter('fileSizeInMB', function(){
    return function(size){
        if (size < 0) {
            return "N/A";
        } else {
            return Math.round(size / 1024 / 1024*100)/100 + ' MB';
        }
    };
});


app.filter("displayMeasure", function() {
    return function(measure) {
        if (measure.function == 'SUM') {
            return 'Sum of ' + measure.column.name;
        } else if (measure.function == 'AVG') {
            return 'Average of ' + measure.column.name;
        } else {
            return "I have no idea what this is about " + measure.column;
        }
    };
});


app.filter("percentage", function() {
    return function(val) {
        return Math.round(val * 100) + '%';
    };
});


app.filter('toKVArray', function(){
    return function(dict) {
        if (dict){
            return $.map(dict, function(v, k){
                return {k:k, v:v};
            });
        } else {
            return [];
        }
    };
});


function objectToArray(dict, saveKey) {
    if (dict) {
        return $.map(dict, function (v, k) {
            if (saveKey) {
                v['_key'] = k;
            }
            return v;
        });
    }
    return [];
}

app.filter('toArray', function(){
    return (dict) => objectToArray(dict);
});

app.filter('toArrayWithKey', function () {
    return (dict) => objectToArray(dict, true);
});


app.filter('datasetPartition', function() {
    return function(val) {
        if (val.partition && val.partition != 'NP') {
            return val.dataset + " (" + val.partition + ")";
        } else {
            return val.dataset;
        }
    };
});


app.filter('breakText', function(){
    return function(text, breakon){
        var bo = breakon || '_';
        return text.replace(new RegExp(bo, 'g'), bo + '&#8203;');
    };
});


app.filter('truncateText', function(){
   return function(text, val){
        if (val == null) val = 30;
       return text.substring(0, val);
   };
});

app.filter('subString', function(){
    return function(text, start, end){
        if (start > end) {
            let s = start;
            start = end;
            end = s;
        }
        return text.substring(start, end);
    };
 });

app.filter('sanitize', function() {
    return function(x) {
        return sanitize(x);
    };
});

app.filter('sanitizeHighlighted', function() {
    return function(x) {
        return sanitizeHighlighted(x);
    };
});

app.filter("stripHtml", function($sanitize) {
    return (htmlString) => $("<div>").html($sanitize(htmlString)).text();
})

app.filter('escape', function() {
   return function(x) {
       return escape(x);
   };
});


app.filter('escapeHtml', function() {
    var chars = /[<>&'"]/g,
        esc = (function(_) { return this[_]; }).bind({
            '<': '&lt;', '>': '&gt;', '&': '&amp;', '"': '&quot;', "'": '&apos;' });
    return function(s) { return s.replace(chars, esc); };
});


app.filter('unescapeHtml', function() {
    var chars = /&(lt|gt|amp|quot|apos|#(\d+)|#x([0-9a-fA-F]+));?/g,
        esc = (function(_, code, dec, hex) {
            if (code in this) return this[code];
            if (dec || hex) return String.fromCodePoint(parseInt(dec || hex, dec ? 10 : 16));
            return _;
        }).bind({ lt: '<', gt: '>', amp: '&', quot: '"', apos: "'" });
    return function(s) { return s.replace(chars, esc); };
});


app.filter('quotedString', function() {
    return function(x) {
        return `'${x.replaceAll('\\', '\\\\').replaceAll("'", "\\'")}'`;
    };
});

app.filter('meaningLabel', function($rootScope) {
    return function(input) {
        return $rootScope.appConfig.meanings.labelsMap[input] || input;
    };
});


app.filter('buildPartitionsDesc', function() {
    return function(input) {
        if (input.useExplicit) {
            return input.explicit;
        } else if (input.start) {
            if (input.start == input.end) {
                return input.start;
            } else {
                return input.start + " / " +input.end;
            }
        } else {
            return input;
        }
    };
});


// Similar to linky for doesn't detect emails
app.filter('parseUrlFilter', function() {
    // eslint-disable-next-line no-useless-escape
    var urlPattern = /(\b(https?|ftp|file):\/\/[-A-Z0-9+&@#\/%?=~_|!:,.;]*[-A-Z0-9+&@#\/%=~_|])/ig;

    return function(text, target) {
        if(!target) {
            target = '_blank';
        }
        if(text) {
            return text.replace(urlPattern,'<a href="$1" target="'+target+'">$1</a>');
        }
        else {
            return '';
        }
    };
});


app.filter('dynamicFormat', function($filter) {
      return function(value, filterName) {
        return $filter(filterName)(value);
      };
});


app.factory('$localeDurations', [function () {
    return {
        'one': {
            year: '{} year',
            month: '{} month',
            week: '{} week',
            day: '{} day',
            hour: '{} hour',
            minute: '{} minute',
            second: '{} second'
        },
        'other': {
            year: '{} years',
            month: '{} months',
            week: '{} weeks',
            day: '{} days',
            hour: '{} hours',
            minute: '{} minutes',
            second: '{} seconds'
        }
    };
}]);


app.filter('duration', ['$locale', '$localeDurations', function ($locale, $localeDurations) {
    return function duration(value, unit, precision) {
        var unit_names = ['year', 'month', 'week', 'day', 'hour', 'minute', 'second'],
            units = {
                year: 86400*365.25,
                month: 86400*31,
                week: 86400*7,
                day: 86400,
                hour: 3600,
                minute: 60,
                second: 1
            },
            words = [],
            max_units = unit_names.length;


        precision = parseInt(precision, 10) || units[precision || 'second'] || 1;
        value = (parseInt(value, 10) || 0) * (units[unit || 'second'] || 1);

        if (value >= precision) {
            value = Math.round(value / precision) * precision;
        } else {
            max_units = 1;
        }

        var i, n;
        for (i = 0, n = unit_names.length; i < n && value !== 0; i++) {
            var unit_name = unit_names[i],
                unit_value = Math.floor(value / units[unit_name]);

            if (unit_value !== 0) {
                words.push(($localeDurations[unit_value] || $localeDurations[$locale.pluralCat(unit_value)] || {unit_name: ('{} ' + unit_name)})[unit_name].replace('{}', unit_value));
                if (--max_units === 0) break;
            }

            value = value % units[unit_name];
        }

        if (words.length){
            return words.join(" ");
        }
        return '0s';
    };
}]);

app.filter('fsProviderDisplayName', function(TypeMappingService) {
    return TypeMappingService.mapToFsProviderDisplayName;
});

app.filter('cleanConnectionName', function() {
    return function(input) {
        if (input && input.startsWith("@virtual(impala-jdbc):")) {
            return "Impala builtin";
        } else {
            return input;
        }
    };
});

app.filter('uniqueStrings', function() {
    return function(x) {
        if (!x) return x;
        var uniqueValues = [];
        x.forEach(function(v) {
           if (uniqueValues.indexOf(v) < 0) {
               uniqueValues.push(v);
           }
        });
        return uniqueValues;
    };
 });

app.filter('encodeHTML', function() {
    return rawStr => String(rawStr).replace(/[\u00A0-\u9999<>&]/gim, i => '&#'+i.charCodeAt(0)+';');
});

app.filter('map2Object', function() {
    return function(input) {
      var output = {};
      input.forEach((value, key) => output[key] = value);
      return output;
    };
});

app.filter('filterConfigContent', function () {
    return function (items) {
        if (!items) return items;
        let filtered = {};
        angular.forEach(items, function (value, key) {
            if (value && key !== 'savedModels' && key !== 'llmSavedModels') { // These are handled separately with more details (ML Saved Models, Fine-tune models, ...)
                filtered[key] = value;
            }
        });
        return filtered;
    };
});

app.filter("bundleProjectContent", function() {
    const bundleContentConfigMap = {
        datasets: 'Datasets',
        recipes: 'Recipes',
        mlSavedModels: 'ML Saved models',
        modelEvaluationStores: 'Model Evaluation Stores',
        genAiEvaluationStores: 'GenAi Evaluation Stores',
        managedFolders: 'Managed folders',
        scenarios: 'Scenarios',
        analysis: 'Analyses',
        jupyterNotebooks: 'Jupyter notebooks',
        labelingTasks: 'Labeling tasks',
        sqlNotebooks: 'SQL notebooks',
        insights: 'Insights',
        dashboards: 'Dashboards',
        knowledgeBanks: 'Knowledge Banks',
        promptStudios: 'Prompt Studios',
        llmFineTunedModels: 'LLM Fine-tune models',
        llmRagModels: 'LLM RAG models',
        agents: 'Agents',
    }
    return input => bundleContentConfigMap[input];
});

app.filter('displayablePredictionType', function() {
    const map = {
        "BINARY_CLASSIFICATION": "binary classification",
        "REGRESSION": "regression",
        "MULTICLASS": "multiclass",
        "DEEP_HUB_IMAGE_OBJECT_DETECTION": "object detection",
        "DEEP_HUB_IMAGE_CLASSIFICATION": "image classification"
    }
    return input => map[input];
});

app.filter("targetRoleName", function() {
    return function(isCausal) {
        return isCausal ? "outcome" : "target";
    };
})

/**
 * Sample input: config_option-1 or config-option_1
 * Sample ouptut: Config option 1
 */
app.filter('prettyConfigOption', function(){
    return function(input){
        if(!input || !input.length) {
            return '';
        }
        let output = input.replace(/[-_]/g,' ');
        return output.charAt(0).toUpperCase() + output.slice(1);
    };
});

app.filter('formatCategoricalValue', function() {
    return function(input) {
        return input === '' ? '<empty>' : input;
    }
})

function andOrList(andor) {
    return function(input) {
        if (!input || !input.length) {
            return '';
        }

        if (input.length > 2) {
            input = [input.slice(0, -1).join(', '), input.slice(-1)[0]];
        }

        return input.join(' '+andor+' ');
    }
}

/*
    Input: array of strings
    Returns a comma-separated list with "and" before the last item (when list length is > 1)
*/
app.filter('andList', function() {
    return andOrList('and');
});

/*
    Input: array of strings
    Returns a comma-separated list with "or" before the last item (when list length is > 1)
*/
app.filter('orList', function(translate) {
    return andOrList(translate('GLOBAL.FILTERS.OR_LIST.OR', 'or'));
});

/*
    Returns a light or dark hex code given background color hex code string as input
*/
app.filter('colorContrast', function() {
    const DARK = '#000000';
    const LIGHT = '#FFFFFF';
    const YIQ_THRESHOLD = 128;

    return function(hexCode) {
        // https://24ways.org/2010/calculating-color-contrast/
        const [r, g, b] = extractColorValues(hexCode);
        const yiq = ((r * 299) + (g * 587) + (b * 114)) / 1000;

        return isNaN(yiq) ? DARK : (yiq >= YIQ_THRESHOLD) ? DARK : LIGHT;
    }

    function extractColorValues(color) {
        color = (color.charAt(0) === '#') ? color.substring(1, color.length) : color;
        return [
            parseInt(color.substring(0, 2), 16),
            parseInt(color.substring(2, 4), 16),
            parseInt(color.substring(4, 6), 16)
        ];
    }
});

 /**
 * Parses a string and converts it to an array
 *
 * If string is array-like ('["item1","item2"]') return as array object (["item1","item2"]). If regular string, return single element array
 * If not a string, return null
 * @param string str
 */
app.filter('stringToArray', function(LoggerProvider) {
    return function(str) {
        if (typeof str === 'string') {
            if (str.startsWith('[') && str.endsWith(']')) {
                try {
                    const arr = JSON.parse(str);

                    if (Array.isArray([arr])) {
                        return arr;
                    }
                } catch (err) {
                    // was not valid JSON string
                    LoggerProvider.getLogger('DisplayFilters').error("Error ", err);
                    return null;
                }
            }

            // if normal string, return as singleton array
            return [str];
        }
    }
});

app.filter('platformKey', function(DetectUtils) {
    return function(key) {
        switch (key) {
            case 'mod':
            case 'cmd':
            case 'command':
                return DetectUtils.isMac() ? "⌘" : "Ctrl";
            case 'ctrl':
                return DetectUtils.isMac() ? "⌃" : "Ctrl";
            case 'shift':
                return DetectUtils.isMac() ? "⇧" : "Shift";
            case 'alt':
                return DetectUtils.isMac() ? "⌥" : "Alt";
            default:
                return key;
        }
    }
});

 /**
 * Extracts the value of a specified key for each element in an array of objects
 * and returns an array of these values
 *
 * Example: [{id: 5, id: 6, id: 7}] -> [5, 6, 7]
 * @param Array array
 * @param string key
 */
app.filter('extractKey', function() {
    return function(array, key) {
        return (array || []).map(item => item[key]);
    };
});

app.filter('prettifyScenarioType', function(ScenarioUtils) {
    /**
     * Returns the display name for a given scenario step. Also handle plugin steps.
     * @param step: the step object containing the type attribute in technical format
     * @returns the display name of the step type, translated
     */
    return function(step) {
        return ScenarioUtils.getStepDisplayType(step);
    }
});

 /**
 * Displays a fallback string if the input string is null
 *
 * @param string input containing string
 * @param string fallback string if input is null/undefined
 * @returns the string or the default value
 */
app.filter('stringWithFallback', function() {
    return function(input, fallback = 'N/A') {
        return input == null ? fallback : input;
    }
});

app.filter('sortFeatureGenerationWindowOperations', function() {
    return function(items) {
        if (!items || !items.length) {
            return;
        }

        const sortOrder = ["MIN", "MAX", "MEAN", "MEDIAN", "SUM", "STD"];

        // Use .slice() to create a shallow copy to avoid modifying the original array
        return items.slice().sort((a, b) => {
            let indexA = sortOrder.indexOf(a.operation);
            let indexB = sortOrder.indexOf(b.operation);

            if (indexA === -1) {
                indexA = Infinity;
            }
            if (indexB === -1) {
                indexB = Infinity;
            }

            return indexA - indexB;
        });
    };
});

app.filter('pluginRecipesCategory', function() {
     return function(plugins, recipesCategory) {
         if (!plugins || !recipesCategory) {
             return plugins;
         }
         return plugins.filter(function(plugin) {
             return plugin.recipesCategory === recipesCategory;
         });
     }
});

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.directives.bootstrap', ['dataiku.filters']);


app.factory("getBootstrapTooltipPlacement", function() {
    return function(force_position) {
        if (!force_position || force_position.indexOf('tooltip-')==-1) {
            return function(tip, element) {
                var overflowParent = $(element).overflowParent()
                var overflowParentOffset = $(overflowParent).offset()

                var offset = $(element).offset();
                var top = offset.top + $(element).height() * 0.5 - overflowParentOffset.top;
                var left = offset.left + $(element).width() * 0.5 - overflowParentOffset.left;

                var height = $(overflowParent).outerHeight();
                var width = $(overflowParent).outerWidth();

                if (left < width * 0.33) {return 'right'}
                else if (left > width * 0.66) {return 'left'}
                else if (top < height * 0.5) {return 'bottom'}
                else if (top >= height * 0.5) {return 'top'}
                else {return 'bottom'}

            }
        } else {
            return force_position.replace("tooltip-","");
        }
    }
});


app.directive("toggle", function(getBootstrapTooltipPlacement) {
    return {
        restrict: 'A',
        link: function(scope, element, attrs) {
            switch (attrs.toggle) {
                case "tooltip-bottom":
                case "tooltip-top":
                case "tooltip-left":
                case "tooltip-right":
                case "tooltip":
                    $(element).on('hide.bs.modal', function(e) {
                        // Avoid wrong behavior with stacked modals: related to sc-86865, generalized to all tooltips
                        // since it does not look like there is a valid reason to let hide.bs.modal bubble up from
                        // here: hide.bs.modal should only be triggered from the modal container, not from a link.
                        e.stopPropagation();
                    });
                    var params = {
                        placement: attrs.placement ? attrs.placement : getBootstrapTooltipPlacement(attrs.toggle),
                        animation: attrs.animation ? scope.$eval(attrs.animation) : false,
                    };
                    if (attrs.container) params.container = attrs.container;
                    if (attrs.trigger) params.trigger = attrs.trigger;
                    element.tooltip(params);
                    // Remove other tooltips on hover out
                    element.hover(function() { $(this).tooltip('show'); }, function() {
                        $('.tooltip').not(element.next()).remove();
                    });
                    attrs.$observe('title', function() {
                        element.attr('data-original-title', attrs.title);
                        element.attr('title', '');
                    })
                    break;
                case "popover":
                    var content = scope.$eval(attrs.content)
                    var closePopover = function() {
                        element.popover('destroy');
                    }
                    var openPopover = function() {
                        var opts = {
                            placement: getBootstrapTooltipPlacement(),
                            container: attrs.container || 'body',
                            html: true,
                            trigger: 'manual',
                            content: content,
                            title: '<a class="close"><i class="icon-remove"></i></a> ' + attrs.title
                        }
                        element.popover(opts);
                        element.popover('show');
                        element.data('popover').$tip.find('.popover-title > .close').click(closePopover);
                        /* Don't close the popover when clicking on it */
                        element.data('popover').$tip.click(function(e) {
                            e.stopPropagation();
                        });
                    }
                    var justOpened = false;
                    $(element).click(function(e) {
                        if (element.data('popover')) {
                            closePopover()
                        } else {
                            openPopover()
                            justOpened = true;
                        }
                    });
                    $(document).on('click', function(e) {
                        if (!justOpened && element.data('popover')) {
                            closePopover();
                        }
                        justOpened = false;

                    });
                    break;
                case "dropdown":
                    $(element).click(function() {
                        if (attrs['dropdownPosition'] === 'fixed') return;
                        var overflowParent = element.overflowParent()
                        var overflowParentOffset = $(overflowParent).offset()
                        var height = $(overflowParent).outerHeight();
                        var width = $(overflowParent).outerWidth();

                        var offset = $(element).offset();

                        var vert = 0.5 * height - (offset.top - overflowParentOffset.top);
                        var horiz = 0.5 * width - (offset.left - overflowParentOffset.left);

                        if (vert > 0) {
                            element.closest('.dropup,.dropdown').removeClass('dropup').addClass('dropdown')
                        } else {
                            element.closest('.dropup,.dropdown').removeClass('dropdown').addClass('dropup')
                        }

                        if (horiz > 0) {
                            element.closest('.dropup,.dropdown').find('.dropdown-menu').removeClass('pull-right')
                        } else {
                            element.closest('.dropup,.dropdown').find('.dropdown-menu').addClass('pull-right')
                        }
                    })
                    break;
            }
        }
    }
});


/**
 * More manual but also more flexible alternative to showTooltipOnTextOverflow to show a tooltip only when the inner text overflows.
 * This directive only controls whether the tooltip is enabled or not. You can control the tooltip content, layout & position as you wish.
 * If no title is set, this directive uses the element text content as tooltip text (only updates the content on mouse-enter) (no title set means title attribute doesn't exist, not only that it's falsy)
 * If the content doesn't overflow, the default-tooltip @ property is used as a tooltip if it is present and truthy (again, value updated on mouse-enter only)
 * 
 * While showTooltipOnTextOverflow assumes that content is a single-line auto-ellipsed content, this directive allows custom inner layout. For example, you can do multi-line:
 * <div show-tooltip-on-content-overflow title="blah blah" toggle="tooltip" container="body"
 *     style="width:100px;word-wrap: normal;overflow: hidden;text-overflow: ellipsis;">
 *     My multi-line text with some possibly loooooooooooooooong words that won't fit on a line and will be ellipsed
 * </div>
 * 
 * This directive can be used with any content (not only text), though overflow is only considered at the first nesting level
 * (if text overflows from an inner element that itself doesn't overflow from the container, it does not count as overflow for the container)
 */
app.directive("showTooltipOnContentOverflow", function() { // show-tooltip-on-content-overflow
    return {
        restrict: 'A',
        scope: {},
        transclude: true,
        template: '<div style="width:100%; height: 0; margin:0"></div><ng-transclude></ng-transclude>',
        link(scope, $element, attr) {
            const autoContent = !('title' in attr);
            const div = $element.children()[0]; // we use an invisible inner div to read inner width of element (without padding)
            const range = document.createRange();
            $element.on('mouseenter focus', function() {
                const divWidth = div.getBoundingClientRect().width; // the inner size of the element
                range.selectNodeContents($element[0]);
                let contentWidth = range.getBoundingClientRect().width; // the size of what is in the element and may overflow
                if (attr.overflowWidthCalculationAdjustment) {
                    contentWidth = contentWidth + (Number(attr.overflowWidthCalculationAdjustment) || 0);
                }
                if(contentWidth > divWidth) {
                    $element.tooltip('enable');
                    if(autoContent) {
                        $element.attr('data-original-title', $element.text());
                    }
                    $element.tooltip('show');
                } else if(attr.defaultTooltip) {
                    $element.tooltip('enable');
                    $element.attr('data-original-title', attr.defaultTooltip);
                    $element.tooltip('show');
                } else {
                    $element.tooltip('disable');
                    $element.tooltip('hide');
                }
            });
        }
    }
});

/**
 * On text overflow detection, this directive displays a tooltip on hover containing the original text and formats the text with an ellipsis.
 *
 * Note: The element on which this directive is applied must already have the correct target width.
 * Overflow detection happens when the width of the text is larger than the width of the container element.
 *
 * @param {string}      textTooltip           - The text to display. Note that if you want to display a falsy value (e.g. 0), you should wrap with `String` to prevent ngShow from hiding the directive altogether
 * @param {string}      tooltipDirection      - Position of tooltip (default: tooltip-right; other possible values: tooltip, tooltip-top, tooltip-bottom, tooltip-left)
 * @param {string}      textOverflowClass     - CSS class to be applied to text (if overflow)
 * @param {boolean}     observeResize         - True (default) to re-detect overflow when the element is resized or text changes. Consider using false for static content (may improve performance)
 * @param {boolean}     allowHtml             - The text to display contains HTML
 */
app.directive("showTooltipOnTextOverflow", function ($compile, $timeout, $filter, $sanitize) { // show-tooltip-on-text-overflow
    const template = `
        <div class="{{ getTooltipClass() }}"
             ng-show="textTooltip"
             ng-bind-html="getHtmlContent()"
             title="{{ getTooltipTitle() }}"
             toggle="{{ getTooltipDirection() }}"
             container="body"
        >
        </div>`;

    const stripHtml = (unsafeHtml) => {
        // DOMParser evaluates in another HTMLDocument with scripting disabled.
        const doc = new DOMParser().parseFromString(unsafeHtml, 'text/html');
        return doc.body.innerText || "";
    };

    return {
        restrict: 'A',
        scope: {
            textTooltip: '<',
            allowHtml: '<',
            tooltipDirection: '<?',
            textOverflowClass: '@',
            observeResize: '<?',
            alwaysShowText: '<?',
            alwaysShowTooltip: '<?'
        },
        template: template,
        controller: ($scope) => {
            // At the very beginning, initialize the template with as much
            // width as possible. Then, if the content is cropped with an
            // ellipsis, show a tooltip with the whole content.
            $scope.withEllipse = false;

            $scope.getTooltipClass = () => {
                if ($scope.withEllipse && !$scope.alwaysShowText) {
                    return $scope.textOverflowClass || "ellipsed";
                }

                // We need this when the template is initialized so that it is
                // later possible to determine if the content is cropped (with
                // an ellipsis). See IntersectionObserver usages below.
                return "width-fit-content";
            };

            $scope.getTooltipTitle = () => {
                if (!$scope.withEllipse || $scope.alwaysShowText) {
                    // The content fits entirely, no need to show the tooltip.
                    return "";
                }

                if ($scope.allowHtml) {
                    return stripHtml($scope.textTooltip);
                }

                return $scope.textTooltip;
            };

            $scope.getTooltipDirection = () => {
                return $scope.tooltipDirection || 'tooltip-right';
            };

            // Set the tooltip content, depending on whether it supports
            // html or plain text.
            $scope.getHtmlContent = () => {
                if ($scope.allowHtml) {
                    return $scope.textTooltip;
                }

                return sanitize($scope.textTooltip);
            };
        },
        link: (scope, element) => {
            if (!('IntersectionObserver' in window)) {
                return;
            }

            // Use better names to refer to the html elements.
            const directiveTargetElement = element[0];
            if (!angular.isObject(directiveTargetElement)) {
                return;
            }

            const thisElement = directiveTargetElement.lastChild;
            directiveTargetElement.classList.add("ellipsed");

            // Object to detect when the tooltip overflows its container element.
            let intersectObserver = null;

            // Object to detect when the container element is resized.
            let resizeObserver = null;

            const onIntersectChange = (intersections) => {
                const anyIntersection = intersections
                    .some((it) => it.intersectionRatio > 0 && it.intersectionRatio < 1);

                if (anyIntersection || scope.alwaysShowTooltip) {
                    // We know that the content overflows its container, no need
                    // to monitor it anymore.
                    intersectObserver.disconnect();

                    // This handler can be called outside of AngularJS cycles,
                    // so we notify it that something has changed.
                    scope.$applyAsync(() => {
                        scope.withEllipse = true;
                    });
                }
            }

            intersectObserver = new IntersectionObserver(onIntersectChange, {
                root: directiveTargetElement,
                rootMargin: '2px', // Tolerance margin.
                threshold: 1,
            });

            const connectObservers = () => {
                intersectObserver.observe(thisElement);

                if (resizeObserver != null) {
                    resizeObserver.observe(directiveTargetElement);
                }
            };

            const disconnectObservers = () => {
                intersectObserver.disconnect();

                if (resizeObserver != null) {
                    resizeObserver.disconnect();
                }
            };

            // Reset to initial template and re-observe
            const resetToInitialRender = () => {
                disconnectObservers();

                // This handler can be called outside of AngularJS cycles,
                // so we notify it that something has changed.
                scope.$applyAsync(() => {
                    scope.withEllipse = false;
                });

                connectObservers();
            };

            // Detect when the container element is resized.
            const observeResize = angular.isDefined(scope.observeResize) ?
                scope.observeResize : true;

            if (observeResize) {
                // Computing the width of the element can be expensive, so we set elWidth to 0 for the
                // first resizing check. Then we will rely on the width provided by the resizeObserver.
                scope.elWidth = 0;

                resizeObserver = new ResizeObserver(entries => {
                    entries.forEach(entry => {
                        if (entry.contentRect.width > scope.elWidth) {
                            // We use setTimeout to change observed element out of the loop to prevent error
                            // in safari : "ResizeObserver loop completed with undelivered notifications"
                            setTimeout(() => resetToInitialRender());
                        }

                        scope.elWidth = entry.contentRect.width;
                    });
                });
            }

            scope.$watch('textTooltip', (newVal, oldVal) => {
                if (newVal !== oldVal) {
                    resetToInitialRender();
                }
            });

            scope.$watch('alwaysShowText', (newVal) => {
                if (newVal && directiveTargetElement) {
                    directiveTargetElement.classList.remove("ellipsed");
                }
            });

            scope.$on('$destroy', () => {
                disconnectObservers();
                $('.tooltip').remove();
            });

            // Set the tooltip content and start monitoring for resize
            // and / or intersection events.
            resetToInitialRender();
        }
    }
});

// Initialy taken from angular-strap but with some fixes (ng options parser accepts filters + fix the dropdown opening problem)
if (/\bdisable_dku_fancy=true\b/.test(document.cookie)) {
    // Selenium: disable this directive + don't hide the <select> inputs
    // eslint-disable-next-line no-console
    console.warn("Disabling dkuBsSelect"); /*@console*/ // NOSONAR: OK to use console.
    $(function() {
        document.body.classList.add('disable-dku-fancy');
    });
} else {
app.directive('dkuBsSelect', function($timeout, translate) {
    // eslint-disable-next-line no-useless-escape
    var NG_OPTIONS_REGEXP = /^\s*(.*?)(?:\s+as\s+(.*?))?(?:\s+group\s+by\s+(.*))?\s+for\s+(?:([\$\w][\$\w\d]*)|(?:\(\s*([\$\w][\$\w\d]*)\s*,\s*([\$\w][\$\w\d]*)\s*\)))\s+in\s+(.*)$/;

    return {
        restrict: 'A',
        require: '?ngModel',
        link: function(scope, element, attrs, controller) {
            element.css('visibility', 'hidden');
            scope.translate = translate;
            var magicContainer = $('<div></div>');
            element.magicContainer = magicContainer;
            $('body').append(magicContainer);
            var lastDisabled;
            var refresh = function(newValue, oldValue) {
                if (magicContainer) {
                    const dkuBsSelectOptions = scope.$eval(attrs.dkuBsSelect) || {}
                    if (dkuBsSelectOptions.tokens) {
                        // We add the data-tokens to the option elements if the options have tokens enable
                        refreshSearchTokens()
                    }
                    if (attrs.immediateRefresh) {
                        element.find('option[data-content]').each(function() {
                            var $this = $(this);
                            $this.data('content', $this.attr('data-content'));
                        });

                        element.selectpicker('refresh');
                    } else {
                        scope.$applyAsync(function() {
                            // data-content is cached in jQuery's .data('content') and .selectpicker('refresh') won't pick up the new content if we don't refresh that (https://github.com/silviomoreto/bootstrap-select/issues/298)
                            element.find('option[data-content]').each(function() {
                                var $this = $(this);
                                $this.data('content', $this.attr('data-content'));
                            });

                            element.selectpicker('refresh');
                        }, 0);
                    }
                }
            };
            var refreshIfChanged = function(newValue, oldValue) {
                if (!angular.equals(newValue, oldValue)) {
                    refresh(newValue, oldValue);
                }
            };

            function refreshSearchTokens() {
                const tokens = scope.$eval(attrs.tokens)
                $('option', element).each((i, e) => {
                    if ($(e).val() !== '?') {
                        $(e).attr('data-tokens', tokens[i - 1])
                    }
                })
            }

            const refreshStyle = function(newValue, oldValue) {
                if (!angular.equals(newValue, oldValue)) {
                    element.selectpicker('updateStyle', newValue);
                }
            };

            scope.$on('selectPickerRefresh', refresh);

            // Hide the empty element
            var fixupEmptyOption = function() {
                var disabled = $('option', element).map(function(idx, op) {
                    return $(op).attr('disabled') == "disabled"
                }).get();
                if (lastDisabled == null) {
                    lastDisabled = disabled;
                } else if (lastDisabled.join() != disabled.join()) {
                    refresh();
                    lastDisabled = disabled;
                }
                scope.$evalAsync(function() {
                    magicContainer.find('ul>li>a>span[class=text]:empty').parent().parent().css('display', 'none');
                });
            };

            function addWatches() {
                var options = $.extend({
                    searchPlaceholder: translate('GLOBAL.FILTER_PLACEHOLDER', 'Filter...'),
                    noneResultsText: translate('GLOBAL.NONE_RESULTS_TEXT', 'No results match'),
                    noneSelectedText: attrs.noneSelectedText || translate('GLOBAL.NOTHING_SELECTED', 'Nothing selected'),
                    container: magicContainer,
                    style: 'dku-select-button btn--secondary',
                    deepWatch: true
                }, scope.$eval(attrs.dkuBsSelect) || {});
                
                //  Allows to add a class to parent element for CSS isolation
                if (options.class) {
                    element.addClass(options.class);
                }

                element.selectpicker(options);
                element.selectpicker('updateStyle', scope.$eval(attrs.dkuStyle))
                if (options.titlePrefix) {
                    element.next().find('>button').attr('title-prefix', options.titlePrefix);
                }
                element.next().find('>button').click(fixupEmptyOption);
                if (options.pullRight) {
                    element.parent().find("div.dropdown-menu").addClass("pull-right");
                }
                if (options.wideDropdownOptions) {
                    const dropdownMenu = element.parent().find("div.dropdown-menu");
                    dropdownMenu.addClass("wide");
                    // The first child of the .dropdown-menu is also a .dropdown-menu...
                    dropdownMenu.find(".dropdown-menu").addClass("wide");
                }
                if (options.customDropdownAttrs) {
                    for (var attr in options.customDropdownAttrs) {
                        element.parent().find("div.dropdown-menu").attr(attr, options.customDropdownAttrs[attr]);
                    }
                }
                scope.$watch(attrs.dkuStyle, refreshStyle);

                // If we have a controller (i.e. ngModelController) then wire it up
                if (controller) {
                    // Watch for changes to the model value
                    scope.$watch(attrs.ngModel, refresh);

                    // Update the select menu when another model change
                    // It's supposed to be used whenever the ng-options is too complicated to parse
                    if (attrs.watchModel) {
                        scope.$watch(attrs.watchModel, refreshIfChanged, options.deepWatch);
                        scope.$watchCollection(attrs.watchModel, refreshIfChanged);
                    } else {
                        // Watch for changes to the options
                        // We try to detect the underlying model variable using regexps
                        // It is not robust
                        if (attrs.ngOptions) {
                            var match = attrs.ngOptions.match(NG_OPTIONS_REGEXP);

                            if (match && match[7]) {
                                var variable = match[7].split('|')[0];
                                if(variable) {
                                    scope.$watch(variable, refreshIfChanged, options.deepWatch);
                                }
                            }
                        }
                        if (attrs.ngDisabled) {
                            scope.$watch(attrs.ngDisabled, refreshIfChanged);
                        }
                    }
                }
            }

            // if the attributes immediateRefresh is set, watches are added without
            // waiting for the rendering to finish (on the next angularjs digest cycle start)
            if (attrs.immediateRefresh) {
                addWatches();
            } else {
                $timeout(addWatches);
            }

            scope.$on('$destroy', function() {
                element.selectpicker('destroy');
                if (magicContainer) {
                    magicContainer.remove();
                    magicContainer = null;
                    element.magicContainer = null;
                }
            });
        }
    };
});
}

// this select menu is actually a dkuBsSelect with a replacement for the dropdown menu element
app.directive('optionsDescriptions', function($timeout, Logger, $sanitize) {

    var delayed = function(f,x){return function(){$timeout(f,x)}};

    return {
        restrict: 'A',
        link: function(scope, element, attrs) {
            var lastOriginalDropDownHTML; // save dropdown menu state to manage updates
            var content;

            var clicked  = function(i) {
                $($('ul.dropdown-menu>li>a', element.magicContainer)[i]).trigger('click');
            };
            
            // Check if first item is a header
            const isHeader = firstItem => $('.text', firstItem).text() == "";

            // Get class from orginal dropdown lines and set it to current description lines
            const setOriginalLinesClass = () => { 
                const items = $('ul.dropdown-menu>li', element.magicContainer); // Lines of original dropdown
                const newLines = attrs.layout == "list"? $("li", content) : $('tr', content);
                let headerOffset = isHeader(items[0]) ? 1 : 0;

                for (let i = headerOffset; i < items.length; i++) {
                    $(newLines[i - headerOffset]).toggleClass('selected', $(items[i]).is('.selected'));
                    $(newLines[i - headerOffset]).toggleClass('hide', $(items[i]).is('.hide'));
                }
                // Show empty results line from original dropdown if it exists
                $('li.no-results', content).removeClass('hide'); 
            }

            var setPopup = function() {
                $("div.dropdown-menu.open", element.magicContainer).css("overflow-y", "scroll");
                var originalDropDownHTML = $('div.dropdown-menu', element.magicContainer).html();

                if (originalDropDownHTML != lastOriginalDropDownHTML) { // update the menu if any change was made in the original dropdown menu
                    var items = $('ul.dropdown-menu>li', element.magicContainer); // the labels of the original popup have class "text"
                    var disabled =  $('option', element).map(function(idx, op){return $(op).attr('disabled') == "disabled"}).get();
                    const hasHeader = isHeader(items[0]);

                    // Enable use of searchbox from original dropdown if it exists
                    const searchbox = $(element.magicContainer).find('input'); 
                    if (items.length == 1 && hasHeader) { // List is empty (there is only a placeholder header)
                        searchbox && searchbox.hide(); 
                    } else {
                        searchbox && searchbox.show() && searchbox.on('input propertychange', setOriginalLinesClass);
                    }
                    

                    if (content) {
                        content.remove();
                    }
                    if (items.length > 0) {
                        var descriptions = scope.$eval(attrs.optionsDescriptions)
                        if (!descriptions) {
                            Logger.error("No description provided for the select options");
                        }

                        if (attrs.layout == "list") {
                            content = $('<ul class="dku-bs-select-options-descriptions-listlayout inner selectpicker">');

                            // Items descriptions begin after header if present
                            let headerOffset = hasHeader ? 1 : 0;

                            for (let i = headerOffset; i < items.length; i++) {
                                let label = $('.text', items[i]).text();
                                if (label && label.length) {
                                    let line = $('<li>').click((function(n){ return function(evt){ evt.stopPropagation(); clicked(n) }; })(i));
                                    if (disabled[i]) {
                                        line.addClass('disabled');
                                    }
                                    line.html('<div class="main-label">'+sanitize(label)+'</div><div class="description">'+$sanitize(descriptions[i - headerOffset])+'</div>');
                                    content.append(line);
                                }
                            }
                        } else {
                            content = $('<table class="dku-bs-select-options-descriptions-tablelayout">');
                            for (let i = 0; i < items.length; i++) {
                                let label = $('.text', items[i]).text();
                                if (label && label.length) {
                                    let line = $('<tr>').click((function(n){ return function(evt){ evt.stopPropagation(); clicked(n) }; })(i));
                                    if (disabled[i]) {
                                        line.addClass('disabled');
                                    }
                                    line.html('<td class="main-label"><div>'+sanitize(label)+'</div></td><td class="description">'+$sanitize(descriptions[i])+'</td>');
                                    content.append(line);
                                }
                            }
                        }
                    } else {
                        content = $('<div>').text('No available options');
                    }
                    $('div.dropdown-menu', element.magicContainer).append(content);
                    setOriginalLinesClass();
                    lastOriginalDropDownHTML = originalDropDownHTML;
                }
            };

            setTimeout(function() {
                element.next().find('>button').click(delayed(setPopup, 0));
                $(element.magicContainer).addClass('select-with-descriptions');
                if (attrs.optionsDescriptionsClassname !== undefined) {
                    $(element.magicContainer).addClass(attrs.optionsDescriptionsClassname)
                }
            }, 500);

            scope.$on('$destroy',function() {
                content = null;
            });
        }
    }
});


app.directive('optionsAnnotations', function($timeout, Logger) {

    var delayed = function(f,x){return function(){$timeout(f,x)}};

    return {
        restrict: 'A',
        require: ['select', 'ngModel'],
        link: function(scope, element, attrs) {
            var annotations;
            var updateAnnotations = function() {
                annotations = scope.$eval(attrs.optionsAnnotations);
                setPopup();
            };

            var annotate = function(elt, annotation) {
                if ($('.annotation', elt).length == 0) {
                    elt.append('<span class="annotation">'+(annotation||'')+'<span>');
                } else {
                    $('.annotation', elt).text(annotation||'');
                }
            };

            // var annotateAfter = function(elt, annotation) {
            //   var parent = elt.parent();
            //   if ($('.annotation', parent).length == 0) {
            //     elt.after('<span class="annotation"><span>');
            //   }
            //   $('.annotation', parent).text(annotation);
            // }

            var initMainLableAnnotation = function() {
                var options = $('option', element);
                var selectedValue = $(element).find(":selected").text();
                var mainLabel = element.next().find('>button>span:first-child'); //element indicating selected item

                //TODO perform options annotation in one or two DOM changes

                // when the select is first loaded, we don't know, from the selected value, which option is selected
                // if only one option has a value corresponding to the selected value, we can determine which one is selected
                var guess;
                options.each(function(idx, item) {
                    if ($(item).text() == selectedValue) {
                guess = (guess === void 0 || guess == annotations[idx]) ? annotations[idx] : null; //null if several items have the same value as the selected one but different annotations
                }
                });
                // if (guess != null) {
                //   annotateAfter(mainLabel, guess);
                // }

                //update mainLabel annotation
                // scope.$watch('selectedColumn', function(nv,ov) {
                //   var selectedIndex = $('option', element).get().indexOf($(element).find(":selected").get(0));
                //   annotateAfter(mainLabel, annotations[selectedIndex]);
                // });
            };

            var setPopup = function() {
                var items = $('ul.dropdown-menu>li:visible', element.magicContainer); // the labels of the original popup have class "text", select picker creates invisible items, don't know why
                if (items.length > 0) {// TODO check if update is needed
                    if (!annotations) {
                        Logger.error("No description provided for the select options");
                    }
                    items.each(function(idx, item) {
                        annotate($('a', item), annotations[idx]);
                    });
                }
            };

            updateAnnotations();
            scope.$watch(attrs.optionsAnnotations, updateAnnotations, true);

            setTimeout(function() {
                element.next().find('>button').click(delayed(setPopup));
                $(element.magicContainer).addClass('select-with-annotations');
                $(element).next().addClass('select-with-annotations');
                initMainLableAnnotation();
            }, 500);
        }
    }
});


app.directive('sortOrderButton', function(LocalStorage, translate) {
    function get(orderName, def) {
        if (orderName === undefined || orderName === null) return def;
        return (LocalStorage.get("dssOrders") || {})[orderName];
    }
    function set(orderName, value) {
        if (orderName === undefined || orderName === null) return;
        //TODO cleanup mechanism
        var orders = LocalStorage.get("dssOrders") || {};
        orders[orderName] = value;
        LocalStorage.set("dssOrders", orders);
    }

    return {
        scope : {'val':'=value','rememberChoice':'=', 'disabled':'&'},
        restrict:'E',
        template: '<span style="display: inline-block;vertical-align:middle;font-size:0;">'
        +'<button class="{{buttonClass}}" onfocus="this.blur();" ng-click="change(!val)" ng-disabled="disabled()">'
        +'<i class="dku-icon-sort-ascending-16" ng-show="!val" title="' + translate("WINDOW_RECIPE.WINDOWS.ASCENDING_ORDER", "Ascending order") + '"/>'
        +'<i class="dku-icon-sort-descending-16" ng-show="val" title="' + translate("WINDOW_RECIPE.WINDOWS.DESCENDING_ORDER", "Descending order") + '"/>'
        +'</button>'
        +'</span>',

        link: function (scope, element, attrs) {
            var options = $.extend({
                buttonClass: 'btn btn--secondary btn--dku-icon btn--icon'
            }, scope.$eval(attrs.sobOpt) || {});
            scope.buttonClass = options.buttonClass;
            scope.val = !!get(scope.rememberChoice, scope.val);
            scope.change = function(v) {
                scope.val = v;
                set(scope.rememberChoice, scope.val);
            };

            if (options.hasOwnProperty('disabledValue')) {
                scope.$watch(scope.disabled, function() {
                    if (scope.disabled()) {
                        scope.change(options.disabledValue);
                    }
                })
            }
        }
    };
});


app.directive('sortDropdown', function () {
    return {
        require: "?ngModel",
        link: function (scope, element, attrs,ngModel) {

            var rememberChoice = function() { return scope.$eval(attrs.rememberChoice); };

            if(rememberChoice()) {
                var cookieKey = 'sort_choice_'+rememberChoice();
                var lastChoice = getCookie(cookieKey);
                if(lastChoice != null) {
                    ngModel.$setViewValue(lastChoice);
                }
            }

            // Watch for any changes from outside the directive and refresh
            scope.$watch(attrs.ngModel, function () {

                if(rememberChoice()) {
                    var expires = 60*24*365;
                    var cookieKey = 'sort_choice_'+rememberChoice();
                    // eslint-disable-next-line no-undef
                    setCookie(cookieKey,ngModel.$viewValue,expires);
                }
            });
        }
    };
});

app.directive("progressBarWithThreshold", function() {
    return {
        scope : {
            success : '@',
            warning: '@',
            error : '@',
            neutral : '@',
            allowEmpty : '=',
            title: '@',
            active : '=',
            properOrder: '=',
            threshold: '@'
        },
        restrict : "ECA",
        template: '<div style="position: relative;">' +
            '<progress-bar class="progress-validity padbot0" success="{{success || 0}}" ' +
            'warning="{{warning || 0}}" error="{{error || 0}}" title="{{title}}" ng-attr-neutral="{{neutral || 0}}"></progress-bar>' +
            '<div class="progress-bar-with-threshold__threshold" style="left: calc({{threshold}}% - 8px)/* need to subtract half of component width */"></div>' +
            '</div>'
    }
});

app.directive("progressBar", function() {
    return {
        scope : {
            success : '@',
            warning: '@',
            error : '@',
            neutral : '@',
            allowEmpty : '=',
            title: '@',
            active : '=',
            properOrder: '='
        },
        restrict : "ECA",
        template: '<div class="progress mbot4" rel="tooltip" '+
        ' title="{{title}}" ng-class="{active: active}">' +
        '<div class="bar bar-info" ng-show="neutral >0" style="width:{{ humanized_neutral }}%"></div>' +

        '<div class="bar bar-success" ng-show="success >0" style="width:{{ humanized_success }}%"></div>' +
        '<div class="bar bar-warning" ng-if="properOrder" ng-show="warning > 0" style="width:{{ humanized_warning }}%"></div>' +
        '<div class="bar bar-danger" ng-show="error > 0" style="width:{{ humanized_error }}%"></div>' +
        '<div class="bar bar-warning" ng-if="!properOrder" ng-show="warning > 0" style="width:{{ humanized_warning }}%"></div>' +
        '</div>',
        link : function(scope, element, attrs) {
            var HUMAN_THRESHOLD = 5;
            scope.$watch(function(){return scope.success+scope.error+scope.warning+scope.neutral}, function(newValue, oldValue) {

                var error = isNaN(scope.error) ? 0 : parseFloat(scope.error);
                var warning = isNaN(scope.warning) ? 0 : parseFloat(scope.warning);
                var success = isNaN(scope.success) ? 0 : parseFloat(scope.success);
                var neutral = isNaN(scope.neutral) ? 0 : parseFloat(scope.neutral);

                scope.humanized_error = error > 0 ? Math.max(HUMAN_THRESHOLD, error): error
                scope.humanized_warning = warning > 0 ? Math.max(HUMAN_THRESHOLD, warning): warning;
                scope.humanized_success = success > 0 ? Math.max(HUMAN_THRESHOLD, success) : success;
                scope.humanized_neutral = neutral > 0 ? Math.max(HUMAN_THRESHOLD, neutral) : neutral;

                if (!scope.allowEmpty) {
                    var norm = (scope.humanized_success + scope.humanized_error + scope.humanized_warning + scope.humanized_neutral) / 100.;
                    scope.humanized_warning /= norm;
                    scope.humanized_success /= norm;
                    scope.humanized_error /= norm;
                    scope.humanized_neutral /= norm;
                }
            });
        }
    }
});


app.directive("exactSimpleProgressBar", function(){
    return {
        scope : {
            info : '=',
            success : '=',
            warning : '=',
            error : '='
        },
        template : '<div style="margin-bottom:4px" class="progress">'
            +'<div class="bar bar-info" ng-show="info>0" style="width:{{info*100}}%"></div>'
            +'<div class="bar bar-success" ng-show="success>0" style="width:{{success*100}}%"></div>'
            +'<div class="bar bar-warning" ng-show="warning>0" style="width:{{warning*100}}%"></div>'
            +'<div class="bar bar-error" ng-show="error>0" style="width:{{error*100}}%"></div>'
            +'</div>'

    }
});

})();

;
(function() {
    'use strict';
    var app = angular.module('dataiku.directives.dip', ['dataiku.filters', 'dataiku.services', 'ui.keypress', 'dataiku.widgets.tageditfield']);

    app.directive("userPicture", function(Notification, $rootScope, UserImageUrl) {
        $rootScope.userPicturesHash= Math.random() * 200000 | 0;
        Notification.registerEvent("user-profile-picture-updated", function() {
            $rootScope.userPicturesHash = Math.random() * 200000 | 0;
        });
        return {
            template : "<div class=''></div>",
            replace : true,
            scope : {
                userPicture : '=',
                size : '<',
                fitParent : '<',
            },
            link : function(scope, element) {
                function update() {
                    const size = scope.size || 20;
                    const avatarClass = scope.fitParent ? "avatar-fit" : "avatar" + size;
                    element.addClass(avatarClass);
                    var profileImageBaseUrl = UserImageUrl(scope.userPicture, size);
                    element.css("background-image", "url("+profileImageBaseUrl + ")");
                }
                scope.$watch("userPicture", function(nv, ov) {
                    if (!nv) return;
                    update();
                });
                const userPicturesHashWatcherUnsubscribe = $rootScope.$watch("userPicturesHash", function() {
                    update();
                });
                scope.$on('$destroy', userPicturesHashWatcherUnsubscribe);
            }
        };
    });

    app.directive("avatar", function() {
        return {
            template : '<a toggle="tooltip" title="{{displayName}}" href="{{ $state.href(\'profile.user\',{userLogin:login}) }}">'+
                            '<span user-picture="login" size="20"></span>'+
                        '</a>',
            replace : true,
            scope : {
                displayName : "=", login: "="
            }
        }
    });

    app.directive("closeTooltipsOnExit", function() {
        return {
            scope: false,
            link: function(scope, element, attrs) {
                function closeTooltips() {
                    $('.nvtooltip').remove(); // nvd3 tooltips (scatterplot)
                    $('.tooltip').remove();
                };
                scope.$watch(attrs.ngHide, function(nv) {if (nv) closeTooltips()});
                scope.$watch(attrs.ngShow, function(nv) {if (!nv) closeTooltips()});
                scope.$on('$destroy', closeTooltips);
            },
        }
    });

    app.directive('dkuBindHtmlUnsafe', ['$compile', function($compile) {
        return function(scope, element, attr) {
            element.addClass('ng-binding').data('$binding', attr.dkuBindHtmlUnsafe);
            scope.$watch(attr.dkuBindHtmlUnsafe, function dkuBindHtmlUnsafeWatchAction(value) {
                element.html(value);
                $compile(element.children())(scope);
            });
        };
    }]);

    app.directive('childFocus', function() {
        return {
            link: function(scope, elem, attrs) {
                var $elem = $(elem);
                $elem.find('input, label, select, option, button').each(
                    function(idx, item) {
                        $(item).focus(function () {
                            $elem.addClass('child-focus');
                        });
                        $(item).blur(function () {
                            $elem.removeClass('child-focus');
                        });
                    }
                );
            }
        };
    });

    app.directive('checkDatasetNameUnique', function(DataikuAPI, $stateParams) { // check-dataset-name-unique
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                DataikuAPI.datasets.listNames($stateParams.projectKey).success(function(data) {
                    scope.unique_datasets_names = data;
                    /* Re-apply validation as soon as we get the list */
                    apply_validation(ngModel.$modelValue);
                });
                var initialValue = null, initialValueInitialized = false;
                function apply_validation(value) {
                    ngModel.$setValidity('datasetNameUnique', true);
                    // Implicitly trust the first value (== our own name)
                    if (initialValueInitialized == false && value != undefined && value != null && value.length > 0) {
                        initialValue = value;
                        initialValueInitialized = true;
                    }
                    // It is fake, but other check will get it.
                    if (value == null || value.length === 0) return value;
                    // We are back to our name, accept.
                    if (initialValueInitialized && value == initialValue) return value;

                    var valid = true;
                    if(scope.unique_datasets_names) {
                        for(var k in scope.unique_datasets_names) {
                            if(scope.unique_datasets_names[k].toLowerCase()==value.toLowerCase()) {
                                valid = false;
                            }
                        }
                    }

                    ngModel.$setValidity('datasetNameUnique', valid);
                    return value;
                }
                 //For DOM -> model validation
                ngModel.$parsers.unshift(apply_validation);

                //For model -> DOM validation
                ngModel.$formatters.unshift(function(value) {
                    apply_validation(value);
                    return value;
                });
            }
        };
    });

    /**
     * We need different variants of the uniqueness checking
     * checkMethod(datasetNames, currentValue, initialValue)
     * - value is the current input content (to be validated)
     * - refValue is the value of the refValue parameter given to the directive (typically current value to be used as whitelisted value if relevant - might be undefined!)
     * - datasetNames is the list of datasets to check uniqueness against (including current dataset in the context of a rename)
     * - streamingEndpointNames is the list of streaming endpoints to check uniqueness against
     * datasetNames ans seNames are guaranteed to be defined
     * 
     * It must return true if the value is valid (given existing datasetNames,...), false otherwise
     * 
     * Required context: this directive takes the projectKey from stateParams
     */
    const checkDatasetNameUniqueDirectiveBuilder = (checkMethod) => {
        return function(DataikuAPI, $stateParams, $q) {
            return {
                scope: {
                    refName: '<'
                },
                require: 'ngModel',
                link: function(scope, elem, attrs, ngModel) {
                    let datasetsNames, seNames;
                    $q.all([
                        DataikuAPI.datasets.listNames($stateParams.projectKey),
                        DataikuAPI.streamingEndpoints.listNames($stateParams.projectKey)
                    ]).then(([datasetRes, seRes]) => {
                        datasetsNames = datasetRes.data;
                        seNames = seRes.data;
                        /* Re-apply validation as soon as we get the list */
                        apply_validation(ngModel.$modelValue);
                    });

                    function apply_validation(value) {
                        const valid = value == null || value.length === 0 // It is fake, but other check will get it.
                            || !datasetsNames || !seNames // consider as valid as long as lists are not available
                            || checkMethod(value, scope.refName, datasetsNames, seNames); // the actual check

                        ngModel.$setValidity('datasetNameUnique', valid);
                        return value;
                    }

                    //For DOM -> model validation
                    ngModel.$parsers.unshift(apply_validation);
    
                    //For model -> DOM validation
                    ngModel.$formatters.unshift(function(value) {
                        apply_validation(value);
                        return value;
                    });
                }
            };
        }
    }

    /**
     * Make the form input invalid if its content is not a valid name for a new dataset.
     * It has to be different (case-insensitive) from any existing dataset of SE in this project.
     * 
     * Required context: this directive takes the projectKey from stateParams
     */
    app.directive('checkNewDatasetNameUnique', checkDatasetNameUniqueDirectiveBuilder((value, refValue, datasetNames, seNames) => { // check-new-dataset-name-unique
        const lcValue = value.toLowerCase();
        return datasetNames.every((n) => n.toLowerCase() !== lcValue)
            && seNames.every((n) => n.toLowerCase() !== lcValue);
    }));

    /**
     * Makes a form input invalid if its content is not a valid name for a dataset rename (it has to be different from the current name, but also not already used by an other dataset - case-insensitive)
     * This is almost the same thing as checkNewDatasetNameUnique, except we allow the new name to be the same as the previous one with a different case
     * @param current-name : the current name of the dataset
     * 
     * Required context: this directive takes the projectKey from stateParams
     */
    app.directive('checkRenameDatasetNameUnique', checkDatasetNameUniqueDirectiveBuilder((value, refValue, datasetNames, seNames) => { // check-rename-dataset-name-unique
        const lcValue = value.toLowerCase();
        return (refValue && refValue!==value && refValue.toLowerCase() === lcValue) // just a case change
            || (
                datasetNames.every((n) => n.toLowerCase() !== lcValue)
                && seNames.every((n) => n.toLowerCase() !== lcValue)
            );
    }));

    app.directive('checkHiveHandlesDatasetName', function(DataikuAPI, $stateParams, translate) {
        return {
            restrict: 'E',
            replace: true,
            template: "<span ng-hide='okForHive'><i class='icon-warning-sign'></i>&nbsp;" + translate("FLOW.CREATE_RECIPE.OUTPUT.WARNING_NAME_HIVE_IMPALA", "A dataset for Hive/Impala can only contain alphanumeric characters and underscore") + "</span>",
            scope: {
                datasetName : '=',
                datasetConnectionType : '=',
                connectionId: '=',          // when the connection list comes from a call to get-managed-dataset-options
                connectionConnection: '=',  // when the connection list comes from a call to list-managed-dataset-connections
                connectionList : '='        // when the type is not directly sent via datasetConnectionType
            },
            link: function(scope, elem, attrs, ngModel) {
                // check for hdfs datasets, to warn when they won't work in hive because of their names. Other databases don't
                // have this problem since the dataset is the table so you can't create one with an incorrect name
                scope.okForHive = true;
                var validate = function() {
                    scope.okForHive = true;
                    if (scope.datasetName == null || scope.datasetName.length == 0) return;
                    var selectedConnectionType = scope.datasetConnectionType;
                    if ( selectedConnectionType == null || selectedConnectionType.length == 0) {
                        if (scope.connectionList != null && scope.connectionList.length > 0) {
                            if (scope.connectionId != null && scope.connectionId.length > 0) {
                                scope.connectionList.forEach(function(c) {
                                    if ( c.id == scope.connectionId) {
                                        selectedConnectionType = c.connectionType;
                                    }
                                });
                            }
                            if (scope.connectionConnection != null && scope.connectionConnection.length > 0) {
                                scope.connectionList.forEach(function(c) {
                                    if ( c.connection == scope.connectionConnection) {
                                        selectedConnectionType = c.type;
                                    }
                                });
                            }
                        }
                    }
                    if (selectedConnectionType != 'HDFS') return; // here we're specifically focusing hive tables
                    scope.okForHive = /^[0-9a-zA-Z_]+$/.test(scope.datasetName);
                };
                scope.$watch('datasetName', function() {validate();});
                scope.$watch('connectionId', function() {validate();});
                scope.$watch('connectionList', function() {validate();});
                scope.$watch('datasetConnectionType', function() {validate();});
                scope.$watch('connectionConnection', function() {validate();});
            }
        };
    });


    app.directive('checkNewManagedFolderLabelUnique', function(DataikuAPI, $stateParams) {
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                DataikuAPI.managedfolder.list($stateParams.projectKey).success(function(data) {
                    scope.unique_boxes_names = $.map(data, function(box) {
                        return box.name;
                    });
                    // Re-apply validation as soon as we get the list
                    apply_validation(ngModel.$modelValue);
                });
                function apply_validation(value) {
                    ngModel.$setValidity('boxNameUnique', true);
                    // It is fake, but other check will get it.
                    if (value == null || value.length === 0) return value;
                    var valid = true;
                    if(scope.unique_boxes_names) {
                        for(var k in scope.unique_boxes_names) {
                            if(scope.unique_boxes_names[k].toLowerCase()==value.toLowerCase()) {
                                valid = false;
                            }
                        }
                    }
                    ngModel.$setValidity('boxNameUnique', valid);
                    return value;
                }
                //For DOM -> model validation
                ngModel.$parsers.unshift(apply_validation);

                //For model -> DOM validation
                ngModel.$formatters.unshift(function(value) {
                    apply_validation(value);
                    return value;
                });
            }
        };
    });

    app.service('clickRouter', function() {
        return {
            routeClickEvent: function (element, evt, clickableEmitterSelector, fullClickAttribute, mainClickAttributName, preventRecursionMarker, alsoPreventDefault, closestScope) {
                if(evt.originalEvent && evt.originalEvent.preventRecursionMarker === preventRecursionMarker) {
                    // Prevent event recursion : do not handle this event if we generated it!
                    return;
                }
                var sameElement = (evt.target == element[0]);
                var clickableEmitter = $(evt.target).closest(clickableEmitterSelector, closestScope).length > 0;
                var mainClickEl = fullClickAttribute ? element.find('[' + mainClickAttributName + '="' + fullClickAttribute + '"]') : element.find('['+ mainClickAttributName +']')
                const mainClickDisabled = evt.target.hasAttribute("disable-main-click");  // explicitely disable "main-click" behavior on element

                if(mainClickEl.is(evt.target) || mainClickDisabled) {
                    // The user clicked on the main click target already or the click has been explicitely disabled, so we don't need to do anything
                    // Especially useful for checkbox to avoid the double toggle effect
                    return;
                }

                if (( sameElement || !clickableEmitter ) && mainClickEl.length) {
                    // you cannot redispatch an existing event :(
                    var cloneEvent = document.createEvent('MouseEvents');
                    cloneEvent.preventRecursionMarker = preventRecursionMarker;
                    var e = evt.originalEvent;
                    cloneEvent.initMouseEvent(e.type, e.bubbles, e.cancelable, window, e.detail,
                        e.screenX, e.screenY, e.clientX, e.clientY, e.ctrlKey, e.altKey, e.shiftKey,
                        e.metaKey, e.button, e.relatedTarget);
                    mainClickEl[0].dispatchEvent(cloneEvent);
                    e.stopPropagation();
                    if (alsoPreventDefault) e.preventDefault(); // for right-click, because we don't want the browser contextual menu
                }
            }
        }
    });

    app.directive('stopClickEventPropagation', function () {
        return {
            restrict: 'A',
            link: function ($scope, $element) {
                $element.on('click', function(e) {
                    if (['BUTTON', 'A'].includes(e.target.tagName)) {
                        e.stopPropagation();
                    }
                });
            }
        };
     });

// attrs: 'ignoreElement' allows to prevent click event redirection for one specific element, if needed
    app.directive('fullClick', function(clickRouter) {
        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                var preventRecursionMarker = {};

                element.on("click.fullclick", function(evt) {
                    if (scope.isClickable && !scope.isClickable()) {return false;}
                    if (!attrs.ignoreElement || !evt.target.matches(attrs.ignoreElement)) {
                        clickRouter.routeClickEvent(element, evt, 'a,button,[ng-click]', attrs.fullClick, 'main-click', preventRecursionMarker, false, attrs.containClick ? element[0] : undefined);
                    }
                });
                scope.$on("$destroy", function() {
                    element.off("click.fullclick");
                });
            }
        };
    });

    app.directive('fullRightClick', function(clickRouter) {
        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                var preventRecursionMarker = {};
                element.bind("contextmenu", function(evt) {
                    clickRouter.routeClickEvent(element, evt, '[ng-right-click]', attrs.fullRighClick, 'main-right-click', preventRecursionMarker, true);
                });
                scope.$on("$destroy", function() {
                    element.off("contextmenu");
                });
            }
        };
    });

    app.directive('stopPropagation', function() {
        return function(scope, element, attrs) {
            $(element).click(function(event) {
                event.stopPropagation();
            });
        }
    });

    app.directive('preventDefault', function() {
        return function(scope, element, attrs) {
            $(element).click(function(event) {
                event.preventDefault();
            });
        }
    });

    app.directive('autoFocus', function($timeout) {
        return {
            restrict: 'A',
            link: function(scope, element, attr) {
                attr.$observe('autoFocus', function() {
                    if ((attr.autoFocus === "true") || (attr.autoFocus === true) || (attr.autoFocus===undefined)) {
                        $timeout(function() {element.focus();}, 0);
                    }
                });
            }
        };
    });

     app.directive('autoFocusChild', function($timeout) {
        return {
            restrict: 'A',
            link: function(scope, element, attr) {
                attr.$observe('autoFocusChild', function() {
                    if ((attr.autoFocus === "true") || (attr.autoFocus === true) || (attr.autoFocus===undefined)) {
                        $timeout(function() {
                            element.find("button, input, select").focus();
                        }, 0);
                    }
                });
            }
        };
    });


    app.directive('dkuFrame', function () {
        return {
            restrict: 'E',
            require: '?ngModel',
            replace: true,
            transclude: true,
            template: '<iframe height="100%" detectIframeClicks width="100%" frameborder="0"></iframe>',
            link: function (scope, element, attrs) {
                element.attr('src', attrs.iframeSrc);
            }
        };
    });


    app.factory("ProgressStackMessageBuilder", function($filter, translate) {
        // caution: you can have both hasResult == false and response.progress == null
        // => with a remote future (executed in the fek), there are some extra steps
        // at the end of the computation, where the backend will not have progress for
        
        function bytesToSize(bytes) {
            const sizes = ['B', 'KB', 'MB', 'GB', 'TB'];
            if (bytes === 0) return '0 B';
            const i = parseInt(Math.floor(Math.log(bytes) / Math.log(1024)), 10);
            return Math.round(bytes / Math.pow(1024, i), 2) + ' ' + sizes[i];
        }
        
        function roundAfterTwoDecimals(number) {
            return Math.round(parseFloat(number)*100)/100
        }

        function buildMessage(stackElt, isHTML) {
            let message = isHTML ? `<div class='waiting__message-text-content'>${stackElt.name}</div> ` : `${stackElt.name}`;
            switch(stackElt.unit) {
                case "SIZE":
                    if (stackElt.target > 0) {
                        message += isHTML 
                            ? `<div><small>(${bytesToSize(stackElt.cur)} / ${bytesToSize(stackElt.target)})</small></div>` 
                            : ` - (${bytesToSize(stackElt.cur)} / ${bytesToSize(stackElt.target)})`;
                    } else if (stackElt.cur > 0) {
                        message += isHTML 
                            ? `<div><small>(${bytesToSize(stackElt.cur)})</small></div>` 
                            : ` - (${bytesToSize(stackElt.cur)})`;
                    }
                    break;
                default:
                    if (stackElt.target > 0) {
                        message += isHTML 
                            ? `<div><small>(${roundAfterTwoDecimals(stackElt.cur)} / ${roundAfterTwoDecimals(stackElt.target)})</small></div>` 
                            : ` - (${roundAfterTwoDecimals(stackElt.cur)} / ${roundAfterTwoDecimals(stackElt.target)})`;
                    } else if (stackElt.cur > 0) {
                        message += isHTML 
                            ? `<div><small>(${roundAfterTwoDecimals(stackElt.cur)})</small></div>` 
                            : ` - (${roundAfterTwoDecimals(stackElt.cur)})`;
                    }
                    break;
            }
            return message;
        }

        return {
            getPercentage : function(progress) {
                var percentage = 0;
                var fractionOf = 100;
                if (progress && progress.states) {
                    angular.forEach(progress.states, function(state) {
                        if(state.target > -1) {
                            if (state.target > 0) {
                                fractionOf = fractionOf / (state.target);
                                percentage += fractionOf * state.cur;
                            } else {
                                percentage += fractionOf;
                            }
                        }
                    });
                }
                return percentage;
            },
            buildFull : function(stack) { // called on trainInfo
                if (stack && stack.length) {
                    const stackMessage = [];
                    for (var i = 0; i < stack.length; i++) {
                        stackMessage.push(stack[i].name)
                    }
                    return stackMessage.join(": ");
                } else {
                    return "<span>Please wait...</span>";
                }
            },
            build(progress, includeAll, isHtml = true) {
                if (progress && progress.states && progress.states.length) {
                    const messageStack = [];
                    let lastStackElt = null;
                    for (let i = 0; i < progress.states.length; i++) {
                        if (i == progress.states.length - 1 || progress.states[i].important || includeAll) {
                            const stackMessage = buildMessage(progress.states[i], isHtml);
                            lastStackElt = progress.states[i];
                            messageStack.push(stackMessage);
                        }
                    }
                    if (lastStackElt != null) {
                        const k = messageStack.length - 1;
                        let lastStackEltDuration = $filter('friendlyDurationShort')(lastStackElt.msSinceStart);
                        if (isHtml) {
                            lastStackEltDuration = lastStackEltDuration.replace('<', '&lt;');
                            messageStack[k] = `${messageStack[k]}<div><small>${translate("PROGRESS_STACK_MESSAGES.STARTED", `Started ${lastStackEltDuration} ago`, { time: lastStackEltDuration })}</small></div>`;
                        } else {
                            messageStack[k] = `${messageStack[k]} - ${translate("PROGRESS_STACK_MESSAGES.STARTED", `Started ${lastStackEltDuration} ago`, { time: lastStackEltDuration })}`;
                        }
                    }
                    return isHtml ? messageStack.join("<br />") : messageStack.join(" - ");
                } else {
                    const pleaseWaitText = translate("PROGRESS_STACK_MESSAGES.WAIT", "Please wait...")
                    return isHtml ? `<div class='waiting__message-text-content'>${pleaseWaitText}</div>` : pleaseWaitText;
                }
            }
        };
    });

    app.directive('futureWaiting', function(DataikuAPI, ProgressStackMessageBuilder, $rootScope) {
        return {
            templateUrl : '/templates/future-waiting.html',
            scope: {
                response : '=',
                embedded : '<?'
            },
            // IN scope : "response"
            link : function(scope, element, attrs) {
                scope.percentage = 0;
                scope.started = false;

                scope.$watch('response', function(nv, ov) {
                    if(nv && !nv.hasResult) {
                        scope.percentage = ProgressStackMessageBuilder.getPercentage(scope.response.progress);
                        scope.started = scope.response.progress && scope.response.progress.states && scope.response.progress.states.length;
                        scope.stackMessage = ProgressStackMessageBuilder.build(scope.response.progress);
                    }
                });

                scope.abort = function() {
                    if (scope.response && scope.response.jobId) {
                        DataikuAPI.futures.abort(scope.response.jobId).error(setErrorInScope.bind(scope));
                    }
                };
            }
        };
    });

    function formatDoclinkRef(page) {
        if (page.length > 0 && page[0] == '/') {
            page = page.substring(1);
        }

        const anchorPos = page.indexOf('#');
        if (anchorPos >=0) {
            // add '.html' extension before anchor
            return page.substring(0, anchorPos) + '.html' + page.substring(anchorPos);
        } else {
            return page + '.html';
        }
    }

    function makeElementOpenDoclink(urlRoot, OpalsService, element, page) {
        const href = urlRoot + formatDoclinkRef(page);
        element.click(function(e) {
            // If a modal is open we fallback to opening a new tab otherwise OPALS
            // would open on the background which is not useful
            OpalsService.isEnabled().then(function(isOpalsEnabled) {
                if (isOpalsEnabled && $('.modal-container, mat-dialog-container').length === 0) {
                    OpalsService.navigateToAndShowDrawer(OpalsService.PAGES.EMBEDDED_BROWSER, { href });
                } else {
                    window.open(href, '_blank');
                }
            });
            e.preventDefault();
        });
        element.attr({
            href: href,
            target: '_blank',
        });
    }

    app.directive('doclink', function($rootScope, OpalsService) {
        return {
            restrict: 'E',
            replace: 'true',
            template: '<a />',
            scope: {
                page: '@',
                title: '@',
                showIcon: '=?'
            },
            link: function($scope, element) {
                makeElementOpenDoclink($rootScope.versionDocRoot, OpalsService, element, $scope.page);
                element.html($scope.title);
                OpalsService.isEnabled().then(function(isOpalsEnabled) {
                    if (!isOpalsEnabled && !!$scope.showIcon) {
                        element.append(' <i class="dku-icon-arrow-external-link-12"/>');
                    }
                });
            }
        };
    });

    app.directive('kbaseLinkWrapper', function($rootScope, OpalsService) {
        return {
            restrict: 'EA',
            transclude: true,
            template: '<a><span ng-transclude /></a>',
            link: function($scope, element, attrs) {
                const elt = element.find("a");
                makeElementOpenDoclink($rootScope.kbaseRootUrl, OpalsService, elt, attrs.page);
            }
        };
    });

    app.directive('doclinkWrapper', function($rootScope, OpalsService) {
        return {
            restrict: 'EA',
            transclude: true,
            template: '<a><span ng-transclude /></a>',
            link: function($scope, element, attrs) {
                const elt = element.find("a");
                makeElementOpenDoclink($rootScope.versionDocRoot, OpalsService, elt, attrs.page);
            }
        };
    });

    app.directive('learnLink', function($rootScope) {
        return {
            restrict : 'A',
            transclude:true,
            template : '<a target="_blank"><span ng-transclude /></a>',
            link : function($scope, element, attrs) {
                var page = attrs.page;
                element.find("a")[0].href = $rootScope.learnRootUrl + page + ".html";
            }
        };
    });

    app.directive('aboutPartitioningBox', function(CreateModalFromTemplate) {
        return {
            restrict : 'A',
            replace : 'true',
            template : '<a class="about-trigger"><i class="icon-question-sign" /></a>',
            link: function(scope, element, attrs) {
                var out = scope.$parent,
                    $elt = $(element);
                if (out.appConfig.communityEdition && !out.appConfig.licensing.ceEntrepriseTrial
                        && !attrs.skipCommunityPopup) {
                    $elt.on('click', function(e) {
                        CreateModalFromTemplate("/templates/profile/community-vs-enterprise-modal.html",
                            out, null, function(newScope) { newScope.lockedFeature = 'Partitioning is'; });
                        e.preventDefault();
                    });
                } else {
                    $elt.on('click', out.showAboutPartitioning);
                }
            }
        };
    });

    app.directive('selectedIndex', ['keypressHelper', '$parse','Debounce', function(keypressHelper, $parse,Debounce) {
        return {
            require: 'ngModel',
            restrict: 'A',
            scope: true,
            link: function(scope, element, attrs, ngModel) {

                var parsedSelectedIndexAttr = $parse(attrs.selectedIndex);

                var deselect = $parse(attrs.deselect);
                var inModal = $parse(attrs.inModal);
                scope.$watch(() => ngModel.$modelValue, () => {
                    const initialIndex = parseInt(attrs.initialIndex === undefined ? -1 : attrs.initialIndex, 10);
                    scope.selected = {
                        index: initialIndex,
                        item: initialIndex < 0 ? null : ngModel.$modelValue[initialIndex],
                        itemDelayed:null
                    };
                });
                scope.selectNext = function(e) {
                    e.preventDefault();
                    e.stopPropagation();

                    for (let index = scope.selected.index + 1; index < ngModel.$viewValue.length; index++) {
                        const nextIndex = Math.min(ngModel.$viewValue.length - 1, index);
                        if (ngModel.$viewValue[nextIndex].selectable === undefined || ngModel.$viewValue[nextIndex].selectable === true) {
                            scope.selectIndex(nextIndex);
                            return;
                        }
                    }
                };
                scope.selectPrevious = function(e) {
                    e.preventDefault();
                    e.stopPropagation();

                    for (let index = scope.selected.index - 1; index >= -1; index--) {
                        const nextIndex = Math.max(-1, index);
                        if (nextIndex === -1 || ngModel.$viewValue[nextIndex].selectable === undefined || ngModel.$viewValue[nextIndex].selectable === true) {
                            scope.selectIndex(nextIndex);
                            return;
                        }
                    }
                };
                scope.go = function(e) {
                    if(scope.selected.index >= 0) {
                        parsedSelectedIndexAttr(scope, {item:ngModel.$viewValue[scope.selected.index]});
                    }
                };

                scope.handleEnter = function(evt) {
                    if (["INPUT", "SELECT", "TEXTAREA", "BUTTON"].indexOf(evt.target.tagName) == -1) {
                        scope.go();
                    }
                }

                keypressHelper('keydown', scope, $(document), {
                    Keydown: "{'down': 'selectNext($event)','up': 'selectPrevious($event)','enter': 'handleEnter($event)'}"
                }, '', false, !inModal());

                var sd = Debounce().withDelay(50,500).withScope(scope);

                scope.selectIndex = function(index) {
                    var newItem = null;
                    if(scope.selected.index >= 0 && ngModel.$viewValue[scope.selected.index]) {
                        ngModel.$viewValue[scope.selected.index].selectedIndex = false;
                    }
                    if(index>=0 && ngModel.$viewValue[index]) {
                        newItem = ngModel.$viewValue[index];
                    }

                    if (deselect() && scope.selected.item === newItem) {
                        newItem = null;
                    }

                    // selected.item is the currently selected item
                    scope.selected.item = newItem;

                    // selected.confirmedItem is the currently selected item, defined with a little delay
                    scope.selected.confirmedItem = null;
                    sd.exec(function() {
                        scope.selected.confirmedItem = newItem;
                    });

                    scope.selected.index = index;
                    scope.$emit('selectedIndex', index);
                };
            }
        };
    }]);

    app.directive('editableChartInsightSummary', function($stateParams, $rootScope) {
        return {
            scope : true,
            link : function($scope, element, attrs) {
                $scope.$stateParams = $stateParams;
                $scope.editSummaryState = {};
                $scope.startEdit = function() {
                    $scope.editSummaryState.editSummary = true;
                    $scope.editSummaryState.description = $scope.insight.description;
                    $scope.editSummaryState.tags = angular.copy($scope.insight.tags);
                    clearSelection();
                }
                $scope.cancelEdit = function() {
                    $scope.editSummaryState.editSummary = false;
                }
                $scope.validateEdit = function() {
                    $scope.editSummaryState.editSummary = false;
                    $scope.insight.description = $scope.editSummaryState.description;
                    $scope.insight.tags = angular.copy($scope.editSummaryState.tags);
                    $rootScope.$broadcast("objectSummaryEdited");
                }
            }
        };
    });

    app.directive('foldable', function() {
        return {
            scope : true,
            link : function($scope, element, attrs) {
                if (attrs.foldable == "true") {
                    $scope.unfolded = true;
                }
                $scope.toggleFold = function() {
                    $scope.unfolded = !$scope.unfolded;
                };
            }
        };
    });


   app.directive('abortConfirmation', function() {
       return {
           restrict: 'AE',
           scope : {
               aborting:'=',
               abortFn : '=',
               abortParamsArr : '='
           },
           templateUrl: "/templates/profile/abort-confirmation.html",
           link : function($scope, element, attrs) {

               $scope.showConfirmationForm = function() {
                   $scope.aborting = true;
               }

               $scope.hideConfirmationForm = function() {
                   $scope.aborting = false;
               }

               $scope.abort = function() {
                   $scope.abortFn.apply(this, $scope.abortParamsArr);
               }
           }
       }
   });

    /**
     * directive to hold a multi-selection in a list.
     *
     * it takes parameters from the following attributes:
     *  - ng-model : the list on which the directive operates is passed as a ng-model on the same html element
     *               it can also be an object (its keys will then be added as '_name' attribute of the elements)
     *  - auto-select-first : if true, will start with the first element of the list selected
     *  - auto-focus-first : if true, will start with the first element of the list focused
     *  - select-click-behaviour
     *     if none, will focus the line
     *     if 'select-one', the selected object will be checked and others unselected
     *     if 'select-add', it will select the element (like meta + click) (note, this does not work - see selectAddOrToggle in code below) 
     *  - double-select-unselects : if set, focusing a focused element will unset focus
     *  - keep-focus-on : if set, when clicking on a checkbox re-focus the element
     * - is-item-selectable-predicate: supply the name of a method on the scope taking an item to decide whether 
     *        selecting it is possible (optional, if not provided we assume all can be selected)
     * 
     * it adds a 'selection' object in the scope, with the following fields:
     *  - selectedObject : current selected object
     *  - keyboard : boolean controlling whether selection was made with keyboard
     *  - confirmedItem : equals selectedObject after a delay
     *  - selectedObjects : current list of selected objects
     *
     *  - all : everything is selected
     *  - some : something (not everything) is selected
     *  - none : nothing is selected
     *  - single : 1 object exactly is selected
     *  - multiple : several objects are selected
     *  - filtered : not all items in list are displayed due to filters
     *  - loaded : everything is fully loaded
     *
     *  - allObjects : list of all objects (selected or not)
     *  - filterQuery : text for the filter (used in a $filter('filter')(...) )
     *  - filterParams : Filtering parameters ({userQueryTargets:'name of the param for q search',
     *                   propertyRules: {dict for renaming special props}})
     *
     *  - customFilter : a custom filter specifing function(objects) {return filteredobjects}
     *  - customFilterWatch : name of a model or array of name of models to watch to trigger updateFilter
     *
     *  - orderQuery : string/list of strings for the order ( used in a $filter('orderBy')(...) )
     *  - orderReversed : boolean controlling the reversed state of the orderBy
     *  - filteredObjects : list of visible objects, matching the filter
     *  - filteredSelectedObjects : list of visible selected objects, matching the filter
     *
     * If you specify selection.partialProperty = 'Something' it will create (for each value available)
     *  - partial[values].all : every object matching object[partialProperty] = value is selected
     *  - partial[values].some : something (not everything) matching object[partialProperty] = value is selected
     * -> your need to regenselectionstate to register the param on first load
     *
     * each object in the list gets 2 hidden fields:
     *  - $idx : index in the original list
     *  - $selected : flag indicated selection or not
     *
     * additionally, the following methods for manipulating the selection are added in the scope:
     *  - checkBoxChanged(object, $event) : a mass selection checkbox is clicked on some object
     *  - objectClicked(object, $event) : an object in the list is clicked (not on the mass selection checkbox)
     *  - removeObject(object) : removes an object from the list
     *  - removeSelected()) : remove the selected objects
     *  - updateMassSelectionCheckbox() : to use in a ng-change on the checkbox linked to $scope.selection.all
     *  - updateOrderQuery(value) : sets orderQuery to value, toggling orderReversed if stays the same
     */
    app.directive("filteredMultiSelectRows", function($filter, $timeout, ListFilter, Debounce, Throttle, Fn, CollectionFiltering, ActiveProjectKey, ActivateOldRightPanel) {
        return {
            scope : false,
            link : function($scope, element, attrs) {
                $scope.ActivateOldRightPanel = ActivateOldRightPanel;

                const keyProperties = attrs.keyProperties ? Array.isArray(attrs.keyProperties) ? attrs.keyProperties : [attrs.keyProperties] : undefined;
                var filterWithPartialPropertyValue = function(objects, val) {
                    return objects.filter(function(o) {
                        return o[$scope.selection.partialProperty] === val;
                    });
                }

                var computePartialStates = function() {
                    var partialValues = $scope.selection.filteredObjects.map(function(m) {
                        return m[$scope.selection.partialProperty];
                    }).filter(function(value, index, self) {
                        return self.indexOf(value) === index;
                    });
                    var ssp = $scope.selection.partial;
                    partialValues.map(function(val) {
                        var partialFilteredSelectedObjects = filterWithPartialPropertyValue($scope.selection.filteredSelectedObjects, val);
                        var partialFilteredObjects = filterWithPartialPropertyValue($scope.selection.filteredObjects, val);
                        ssp[val] = {
                            all: partialFilteredSelectedObjects.length == partialFilteredObjects.length && partialFilteredObjects.length > 0,
                            some: partialFilteredSelectedObjects.length > 0 && partialFilteredSelectedObjects.length < partialFilteredObjects.length,
                            none: partialFilteredSelectedObjects.length == 0,
                        }
                    });

                }

                var sd = Debounce().withDelay(50,500).withScope($scope);
                var regenSelectionStateFromFlags = function() {
                    const ss = $scope.selection;

                    if (ss.rememberSelection) {
                        ss.allObjects.forEach(el => {
                            const comparator = getRememberSelectionComparator(el);
                            const selectedObjectIndex = ss.selectedObjects.findIndex(comparator);
                            el.$selected = selectedObjectIndex >= 0;
                            if (el.$selected) {
                                ss.selectedObjects[selectedObjectIndex] = el;
                            }
                        });
                    } else {
                        ss.selectedObjects = ss.allObjects.filter(function (o) {
                            return o.$selected;
                        });
                    }

                    if(!ss.filteredObjects) {
                        ss.filteredObjects = [];
                    }
                    ss.filteredSelectedObjects = ss.filteredObjects.filter(f => f.$selected);
                    // regen flags for the tri-state
                    ss.none = ss.filteredSelectedObjects.length === 0;
                    ss.single = ss.selectedObjects.length === 1 && ss.selectedObject!==null;
                    ss.multiple = ss.selectedObjects.filter($scope.isItemSelectablePredicate).length > 1;
                    ss.all = ss.filteredSelectedObjects.length === ss.filteredObjects.filter($scope.isItemSelectablePredicate).length && ss.filteredObjects.length > 0 && ss.filteredSelectedObjects.length > 0;
                    ss.some = ss.filteredSelectedObjects.length > 0 && ss.filteredSelectedObjects.length < ss.filteredObjects.filter($scope.isItemSelectablePredicate).length;
                    ss.filtered = ss.filteredObjects.length !== ss.allObjects.length;

                    if (ss.partialProperty) {computePartialStates();}
                    // enfore sanity of the selected selectedObject field
                    if (ss.selectedObject && ss.allObjects.indexOf(ss.selectedObject) < 0) {
                        //try and find matching object in the refreshed list corresponding to the currently selected object
                        // this crucial when using the Manage Tag modal via the Edit metadata screen, from a list of taggable objects.
                        const matchId = ss.selectedObject.id;
                        const matchingNewObj = ss.allObjects.find((obj) => {
                            return obj.id == matchId && obj.createdOn == ss.selectedObject.createdOn;
                        });
                        ss.selectedObject = matchingNewObj ? matchingNewObj : null;
                    }
                    sd.exec(function() {ss.confirmedItem = ss.selectedObject});
                };

                var updateSorted = function() {
                    if ($scope.selection.orderQuery && $scope.selection.orderQuery !== "" && !$.isEmptyObject($scope.selection.orderQuery)) {
                        $scope.selection.filteredObjects = $filter('orderBy')($scope.selection.filteredObjects, $scope.selection.orderQuery, $scope.selection.orderReversed);
                    }
                    regenSelectionStateFromFlags();
                };

                var updateFilter = function() {
                    $scope.selection.allObjects = $scope.$eval(attrs.ngModel);
                    if (!$scope.selection.allObjects) {
                        $scope.selection.allObjects = [];
                        $scope.selection.loaded = false;
                    } else {
                        $scope.selection.loaded = true;
                    }
                    if ($scope.selection.allObjects.constructor !== Array) {
                        $scope.selection.allObjects = $.map($scope.selection.allObjects, function(v,k) {v._name=k; return [v]});
                    }
                    $scope.selection.allObjects.forEach(function (c, i) {
                        c.$idx = i;
                    });
                    $scope.selection.filteredObjects = $scope.selection.allObjects;

                    if ($scope.selection.filterQuery) {
                        // For some filters, like on the home page we want to re-order the results so that item that matches on
                        // one of the key properties (typically the "name") are displayed in front of the other items (that also
                        // matches the query but on words in "description" or in some non-displayed fields)
                        if ($scope.selection.filterQuery.userQuery && keyProperties !== undefined) {
                            $scope.selection.filteredObjects = ListFilter.filter($scope.selection.filteredObjects, $scope.selection.filterQuery.userQuery, keyProperties);
                        } else {
                            $scope.selection.filteredObjects = CollectionFiltering.filter($scope.selection.filteredObjects, $scope.selection.filterQuery, $scope.selection.filterParams);
                        }
                    }
                    if ($scope.selection.customFilter) {
                        $scope.selection.filteredObjects = $scope.selection.customFilter($scope.selection.filteredObjects);
                    }
                    updateSorted();
                    if ($scope.selection.filterQuery.userQuery && keyProperties !== undefined) {
                        ListFilter.sortByMatchingQuality($scope.selection.filteredObjects);
                    }
                };

                var debouncedUpdateFilter = Throttle().withDelay(500).wrap(updateFilter);

                function setSelectedObject(newObject) {
                    if (newObject && newObject.$selected) {
                        $scope.selection.selectedObject = newObject;
                    } else {
                        $scope.selection.selectedObject = null;
                    }

                    // if this list is flagged as updating the ActiveProjectKey service,
                    // pass to the service the projectKey of the newly selected object, where it exists
                    if (attrs.updateActiveProjectKey) {
                        const projectKey = (!!newObject && newObject.projectKey) ? newObject.projectKey : undefined;
                        ActiveProjectKey.set(projectKey);
                    }
                }

                $scope.updateOrderQuery = function(value) {
                    var ss = $scope.selection;
                    if (ss.orderQuery === value) {
                        ss.orderReversed = !ss.orderReversed;
                    } else {
                        ss.orderReversed = false;
                        ss.orderQuery = value;
                    }
                }

                $scope.removeSelected = function() {
                    $scope.selection.selectedObjects.forEach($scope.removeObject);
                }
                $scope.removeObject = function(object) {
                    var idx = $scope.selection.allObjects.indexOf(object);
                    if (idx === -1) {
                        return;
                    }

                    $scope.selection.allObjects.splice(idx, 1);
                    updateFilter();
                };

                var clearNonEmptyKeys = function (obj) {
                    if ($.isPlainObject(obj)) {
                        for (var k in obj) {
                            obj[k] = clearNonEmptyKeys(obj[k]);
                        }
                        return obj;
                    } else if ($.isArray(obj)) {
                        return []
                    } else {
                        return "";
                    }
                };
                $scope.clearFilters = function () {
                    $scope.selection.filterQuery = clearNonEmptyKeys($scope.selection.filterQuery);
                    if ($scope.selection.inclusiveFilter) $scope.selection.inclusiveFilter = clearNonEmptyKeys($scope.selection.inclusiveFilter);
                };
                $scope.isEmptyFilter = function () {
                    return angular.equals($scope.selection.filterQuery, clearNonEmptyKeys(angular.copy($scope.selection.filterQuery))) &&
                        (!$scope.selection.inclusiveFilter || angular.equals($scope.selection.inclusiveFilter, clearNonEmptyKeys(angular.copy($scope.selection.inclusiveFilter))));
                };

                $scope.updateMassSelectionCheckbox = function (partialPropertyValue) {
                    var state, filteredObjects;
                    if (partialPropertyValue) {
                        filteredObjects = filterWithPartialPropertyValue($scope.selection.filteredObjects, partialPropertyValue)
                        state = $scope.selection.partial[partialPropertyValue].none;
                    } else {
                        filteredObjects = $scope.selection.filteredObjects;
                        state = $scope.selection.none;
                    }
                    filteredObjects.forEach(function (object) {
                        handleObjectSelection(object, state);
                    });
                    setSelectedObject(null);
                    regenSelectionStateFromFlags();
                    if (attrs.keepFocusOn) {
                        element.focus();
                    }
                };

                function handleObjectSelection(object, checked) {
                    if (!$scope.isItemSelectablePredicate(object)) {
                        return;
                    }

                    object.$selected = checked;
                    if ($scope.selection.rememberSelection) {
                        addOrRemoveRememberedObject(object);
                    }

                }

                function getRememberSelectionComparator(object) {
                    let comparator;
                    if (angular.isString($scope.selection.rememberSelectionComparator)) {
                        comparator = (el) => {
                            return el[$scope.selection.rememberSelectionComparator] === object[$scope.selection.rememberSelectionComparator];
                        };
                    } else if (angular.isFunction($scope.selection.rememberSelectionComparator)) {
                        comparator = (el) => {
                            return $scope.selection.rememberSelectionComparator(el, object);
                        };
                    } else {
                        comparator = (el) => {
                            return el === object;
                        };
                    }

                    return comparator;
                }

                function addOrRemoveRememberedObject(object) {
                    const comparator = getRememberSelectionComparator(object);
                    const rememberedElementIndex = $scope.selection.selectedObjects.findIndex(comparator);
                    if (object.$selected && rememberedElementIndex === -1) {
                        $scope.selection.selectedObjects.push(object);
                    } else if (!object.$selected && rememberedElementIndex !== -1) {
                        $scope.selection.selectedObjects.splice(rememberedElementIndex, 1);
                    }
                }

                var clickHappened = function(object, event, source, oldSelected) {
                    var idx = $scope.selection.filteredObjects.indexOf(object);
                    if ($scope.selection.rememberSelection) {
                        addOrRemoveRememberedObject(object);
                    }
                    if (idx === -1) {
                        return;
                    }

                    if (source == "checkbox") {
                        setSelectedObject(object);
                    } else if (event.ctrlKey || event.metaKey) {
                        handleObjectSelection(object, !oldSelected);
                    } else {
                        if (attrs.doubleSelectUnselects && $scope.selection.selectedObjects.length == 0 && $scope.selection.selectedObject === object) {
                            setSelectedObject(null);
                        }

                        if (source == "click") {
                            // select-add was not working as both the code in the else branch here AND that below when attrs.selectClickBehaviour is truthy ran
                            // (and the else here unselects all the other objects which screws it all up)
                            // So... adding this new attribute was considered less risky than fixing it
                            if (attrs.selectAddOrToggle) {
                                handleObjectSelection(object, !object.$selected);
                                setSelectedObject(object);
                            } else {
                                $scope.selection.filteredSelectedObjects.forEach(obj => handleObjectSelection(obj, false));
                                // If multiple objects are selected and we click on an object, unselect everything except the clicked one
                                const selected = $scope.selection.selectedObjects.length > 1 ? true : !oldSelected;
                                handleObjectSelection(object, selected);
                                setSelectedObject(object);
                            }
                        }
                    }

                    if (event.shiftKey && $scope.selection.selectedObjects.length > 0) {
                        // extend selection within the filtered objects
                        var firstSelected = $scope.selection.filteredObjects.indexOf($scope.selection.selectedObjects[0]);
                        var lastSelected = $scope.selection.filteredObjects.indexOf($scope.selection.selectedObjects[$scope.selection.selectedObjects.length -1]);
                        var newFirstSelected = Math.min(firstSelected, idx);
                        var newLastSelected = Math.max(lastSelected, idx);
                        $scope.selection.filteredObjects.forEach(function(o, i) {
                            handleObjectSelection(o, i >= newFirstSelected && i <= newLastSelected);
                        });
                    } else if (attrs.selectClickBehaviour && source === 'click') {
                        if (attrs.selectClickBehaviour === 'select-one') {
                            $scope.selection.selectedObjects.forEach(function(o) {
                                handleObjectSelection(o, false);
                            });
                            handleObjectSelection(object, true);
                        } else { //'select-add' presumably
                            handleObjectSelection(object, !object.$selected);
                        }
                    }
                    
                    regenSelectionStateFromFlags();

                    // If only one item remains selected in the list, select it for the right panel
                    if ($scope.selection.selectedObjects.length === 1 && !$scope.selection.selectedObject) {
                        $scope.selection.selectedObject = $scope.selection.selectedObjects[0];
                    }
                };

                $scope.checkBoxChanged = function(object,event) {
                    // in a timeout so that the checkbox doesn't get disconnected from its state
                    var oldSelected = object.$selected;
                    return $timeout(() => {
                        clickHappened(object, event, "checkbox", oldSelected);
                        if (attrs.keepFocusOn) {
                            element.focus();
                        }
                    });
                };
                $scope.objectClicked = function(object, event) {
                    /* Actually handles the click */
                    clickHappened(object, event, "click", object.$selected);
                    event.preventDefault();
                };
                $scope.regenSelectionStateFromFlags = regenSelectionStateFromFlags;
                $scope.updateSorted = updateSorted;

                var keyCodes = {
                    tab: 9,
                    pageup: 33,
                    pagedown: 34,
                    left: 37,
                    up: 38,
                    right: 39,
                    down: 40,
                    space: 32,
                };
                $scope.multiSelectKeydown = function(event, callFromFatTable) {
                    function setNewSelected(newObj) {
                        setSelectedObject(newObj);
                        sd.exec(function() {
                            $scope.selection.confirmedItem = $scope.selection.selectedObject;
                        });
                    }

                    function selectIfNotSelected(newObj) {
                        if (!newObj || newObj.$selected) {
                            return;
                        }
                        if (!event.shiftKey) {
                            $scope.selection.selectedObjects.forEach(o => handleObjectSelection(o, false));
                        }
                        newObj.$selected = true;
                        setNewSelected(newObj);
                    }

                    if ($(event.target).is('input:text')||$(event.target).is('textarea')) {
                        return;
                    }

                    var object = $scope.selection.selectedObject;
                    var idx = $scope.selection.filteredObjects.indexOf(object);
                    if (idx === -1) {return;}

                    if (event.keyCode === keyCodes.up) {
                        event.preventDefault();
                        if (idx > 0) {
                            for (let i = 1; i <= idx; i++) {
                                if ($scope.selection.filteredObjects[i].$selected) {
                                    selectIfNotSelected($scope.selection.filteredObjects[i-1]);
                                    break;
                                }
                            }
                        }
                    } else if (event.keyCode === keyCodes.down) {
                        event.preventDefault();
                        if (idx < $scope.selection.filteredObjects.length - 1) {
                            for (let i = $scope.selection.filteredObjects.length - 2; i >= idx; i--) {
                                if ($scope.selection.filteredObjects[i].$selected) {
                                    selectIfNotSelected($scope.selection.filteredObjects[i+1]);
                                    break;
                                }
                            }
                        }
                    }
                    if (callFromFatTable && (event.keyCode === keyCodes.up || event.keyCode === keyCodes.down)) {
                        $scope.$broadcast('scrollToLine', idx);
                    }
                    regenSelectionStateFromFlags();
                }

                function initSelection() {
                    if ($scope.selection === undefined) {$scope.selection={};}
                    $scope.selection.rememberSelection = attrs.rememberSelection !== undefined;


                    $scope.isItemSelectablePredicate = (el) => {
                        // Note - this stops working (along with other uses of attributes) if you nest the filtered-multi-select-rows directive 
                        // (e.g. when it was in both list-items-fattable.html and the html extending it in list-items-2.html )
                        const passedFunction = $scope.$eval(attrs.isItemSelectablePredicate);
                        return passedFunction === undefined || angular.isFunction(passedFunction) && passedFunction(el) === true
                    }

                    if ($scope.selection.rememberSelection) {
                        $scope.selection.rememberSelectionComparator = $scope.$eval(attrs.rememberSelectionComparator);
                        $scope.selection.selectedObjects = [];
                    }
                    if ($scope.selection.orderQuery===undefined) {$scope.selection.orderQuery=[];}
                    if ($scope.selection.filterQuery===undefined) {$scope.selection.filterQuery={};}
                    if ($scope.selection.orderReversed===undefined) {$scope.selection.orderReversed=false;}
                    // fill with empty & wait for scope to refresh to continue
                    if ($scope.selection.allObjects===undefined) {$scope.selection.allObjects=[];}
                    if ($scope.selection.filteredObjects===undefined) {$scope.selection.filteredObjects=[];}
                    if ($scope.selection.filterParams===undefined) {$scope.selection.filterParams={};}
                    if ($scope.selection.partial === undefined) {$scope.selection.partial = {};}

                    if (attrs.updateActiveProjectKey) setSelectedObject(null);

                    $timeout(function () {
                        updateFilter();
                        if (attrs.autoSelectFirst && $scope.selection.allObjects.length > 0) {
                            $scope.selection.filteredObjects.forEach(function (o) {
                                handleObjectSelection(o, false);
                            });
                            setSelectedObject($scope.selection.filteredObjects[0]);
                            handleObjectSelection($scope.selection.filteredObjects[0], true);
                            regenSelectionStateFromFlags();
                        }
                        if (attrs.autoFocusFirst) {
                            $scope.selection.filteredObjects.forEach(function (o) {
                                handleObjectSelection(o, false);
                            });
                            setSelectedObject($scope.selection.filteredObjects[0]);
                            regenSelectionStateFromFlags();
                        }
                    });
                }
                initSelection();
                $scope.$watch('selection.filterQuery', debouncedUpdateFilter, true);
                if ($scope.selection.customFilterWatch) {
                    if (Array.isArray($scope.selection.customFilterWatch)) {
                        // array of elements to watch
                        $scope.selection.customFilterWatch.forEach(customFilter => $scope.$watch(customFilter, debouncedUpdateFilter, true));
                    } else {
                        // single element to watch
                        $scope.$watch($scope.selection.customFilterWatch, debouncedUpdateFilter, true);
                    }
                }
                $scope.$watch('selection.orderQuery', updateSorted, true);
                $scope.$watch('selection.orderReversed', updateSorted);

                // We can't deep watch all the objects of the list (very slow), so this event can be used to refresh list's ordering when an item was mutated
                $scope.$on('refresh-list', updateFilter);

                //when we don't need to reapply the filter just update the sort order and update selection
                $scope.$on('refresh-sort-and-selection', updateSorted);

                //when we *just* want to get the selection upddated without reapplying all the filters or the sort order
                $scope.$on('refresh-selection', regenSelectionStateFromFlags);

                $scope.$watch(attrs.ngModel, updateFilter); // $watchCollection doesn't trigger on array reference changes
                $scope.$watchCollection(attrs.ngModel, updateFilter); // to catch changes caused by adding/removing
            }
        };
    });

    app.directive("dkuFiltered", function(Fn, CollectionFiltering, $compile) {
        // small directive wrapping CollectionFiltering (magic around angular filter)
        // use <div dku-filtered="collectionToFilter"></div>
        // Defines a new scope with :
        //  * objects.all : list of all elements of collectionToFilter
        //  * objects.filterQuery : filterQuery to be passed to CollectionFiltering (see its doc)
        //  * objects.filterParams : filterParams to be passed to CollectionFiltering (see its doc)
        //  * objects.filtered : filtered elements of collectionToFilter
        return {
            scope : true,
            link : function($scope, element, attrs) {
                $scope.objects = {};
                var updateFilter = function() {
                    $scope.objects.filtered = CollectionFiltering.filter($scope.objects.all, $scope.objects.filterQuery, $scope.objects.filterParams);
                }
                var updateObjects = function(nv) {
                    $scope.objects.all = nv || [];
                    if ($scope.objects.all.constructor !== Array) {
                        $scope.objects.all = $.map($scope.objects.all, function(v,k) {v._name=k; return [v]});
                    }
                    updateFilter();
                };
                $scope.updateObjects = updateObjects;
                $scope.$watch(attrs.dkuFiltered, updateObjects, true);
                $scope.$watch('objects.filterQuery', updateFilter, true);
            }
        };
    });

    app.directive('activityIndicator', function() {
        return {
            restrict: 'E',
            scope : {
                activityIndicator : '='
            },
            templateUrl: "/templates/activity-indicator.html"
        }
    });

    app.directive("onMessage", function() {
        return {
            restrict: "A",
            scope: {
                onMessage: "&",
            },
            link: function($scope, $element) {
                function listener(event) {
                    if (event.source === $element[0].contentWindow) {
                        $scope.onMessage({ event });
                    }
                }

                window.addEventListener("message", listener);

                $scope.$on("$destroy", function() {
                    window.removeEventListener("message", listener);
                });
            },
        };
    });

})();

;
(function(){
    'use strict';
    var app = angular.module('dataiku.directives.styling', ['dataiku.filters', 'dataiku.services', 'ui.keypress']);

    app.directive('autoFocus', function($timeout){
        return {
            restrict: 'A',
            link: function(scope, element, attr){
                attr.$observe('autoFocus', function(){
                    if ((attr.autoFocus === "true") || (attr.autoFocus === true) || (attr.autoFocus===undefined)) {
                        $timeout(function(){element.focus();}, 0);
                    }
                });
            }
        };
    });

    app.directive('remainingHeight', function($timeout, $rootScope, Logger) {
        return {
            scope: true,
            link: function(scope, element) {
                Logger.warn("Used deprecated remainingHeight on", element);
                var resize = function(){
                    scope.remainingHeight = $(window).height() - element.offset().top
                        - parseInt($(element).css('padding-top')) - parseInt($(element).css('padding-bottom'));
                    if(!$rootScope.$$phase) scope.$apply();
                };

                $(window).on('resize', resize);
                $timeout(resize, 0);
                //$timeout(resize, 3000);

                scope.$on('reflow',function() { resize(); }); // Force remainingHeight recomputation
                scope.$on('$destroy', function() {
                    $(window).off('resize', resize);
                });
            }
        };
    });

    app.directive('remainingHeightNoScope', function($timeout, $rootScope, Logger) {
        return {
            link: function(scope, element) {
                Logger.warn("Used deprecated remainingHeightNoScope")
                var resize = function(){
                    scope.remainingHeight = $(window).height() - element.offset().top;
                    if(!$rootScope.$$phase) scope.$apply();
                };
                $(window).on('resize', resize);
                $timeout(resize, 0);
                scope.$on('$destroy', function() {
                    $(window).off('resize', resize);
                });
            }
        };
    });

    app.directive('scrollableToBottom', function() {
        return {
            template : '<div remaining-height style="overflow: auto; max-height: {{remainingHeight}}px;" ng-transclude></div>',
            transclude : true
        };
    });

    app.directive('scrollToMe', function($timeout){
        return {
            scope: {
                onScrollTriggered: '&',
                scrollToMeDuration: '<?'
            },
            link: function(scope, element, attrs){

                attrs.$observe('scrollToMe', function() {
                    if(attrs.scrollToMe === 'true') {
                        if (['center', 'start', 'end', 'nearest'].includes(attrs.scrollAlign)) {
                            $timeout(function() {
                                element[0].scrollIntoView({
                                    behavior: 'smooth',
                                    block: attrs.scrollAlign,
                                    inline: 'center'
                                });
                            }, attrs.scrollDelay || 0);
                        } else {
                            if (attrs.scrollDelay !== undefined) {
                                $timeout(scrollToMe, attrs.scrollDelay);
                            } else {
                                scrollToMe();
                            }
                        }

                        scope.onScrollTriggered();
                    }
                });

                function scrollToMe() {
                    /**
                     *  Checking for vertical scroll and doing it if possible 
                     */
                    let $scrollParent = element.parents().filter(function() {
                        return (/(auto|scroll)/).test(($.css(this, 'overflow')) + ($.css(this, 'overflow-y')));
                      }).eq(0);
                      if (!isNaN($scrollParent.length) && $scrollParent.length > 0) {
                          // only if not already visible
                          const offsetWithinScroll = $(element[0]).offset().top - $($scrollParent[0]).offset().top;

                          if (offsetWithinScroll < 0){ // Element is above parent
                              $scrollParent.clearQueue();
                              $scrollParent.animate({
                                  scrollTop: $scrollParent.scrollTop() + offsetWithinScroll,
                              }, scope.scrollToMeDuration);
                          } else if (offsetWithinScroll + element.outerHeight() > $scrollParent.innerHeight()) { // element is under parent
                              $scrollParent.clearQueue();
                              $scrollParent.animate({
                                  scrollTop: $scrollParent.scrollTop() + Math.min(offsetWithinScroll, offsetWithinScroll + element.outerHeight() - $scrollParent.innerHeight()),
                              }, scope.scrollToMeDuration);
                          }
                      }

                    /**
                     * Then checking for horizontal scroll and doing it if possible
                     */
                    $scrollParent = element.parents().filter(function() {
                    return (/(auto|scroll)/).test($.css(this, 'overflow-x'));
                    }).eq(0);
                    if (!isNaN($scrollParent.length) && $scrollParent.length > 0) {
                        // only if not already visible
                        const offsetWithinScroll = element[0].offsetLeft - $scrollParent[0].offsetLeft;

                        if (offsetWithinScroll < $scrollParent.scrollLeft()){
                            // left
                            $scrollParent.clearQueue();
                            $scrollParent.animate({
                                scrollLeft:offsetWithinScroll,
                            }, scope.scrollToMeDuration);
                        }
                        if((offsetWithinScroll + element.outerWidth()) > ($scrollParent.scrollLeft() + $scrollParent.width())){
                            // right
                            $scrollParent.clearQueue();
                            $scrollParent.animate({
                                scrollLeft:offsetWithinScroll - $scrollParent.innerWidth() + element.outerWidth(),
                            }, scope.scrollToMeDuration);
                        }
                    }
                }
            }
        };
    });

})();

;
(function(){
    'use strict';

    var app = angular.module('dataiku.directives.widgets', ['dataiku.filters', 'dataiku.services', 'ui.keypress', 'dataiku.common.lists']);

    /* "Generic" widgets */

    app.directive("plusIcon", function(){
        return {
            restrict : 'A',
            replace:true,
            template : '<span class="dku-plus-icon-16">+</span>'
        }
    });
    app.directive("timesIcon", function(){
        return {
            restrict : 'A',
            replace:true,
            template : '<span style="font-size:1.3em; vertical-align: top">&times</span>'
        }
    });

    const addStarComponentBehaviour = ($ctrl, InterestWording) => {
        const toggle = (nextStatus) => {
            $ctrl.onToggle({ nextStatus });
        };

        $ctrl.isStarring = () => $ctrl.status;
        $ctrl.toggleStar = () => toggle(true);
        $ctrl.toggleUnstar = () => toggle(false);

        const { labels, tooltips } = InterestWording;
        $ctrl.labels = { ...labels };
        $ctrl.tooltips = { ...tooltips };
    };

    app.component('starInterest', {
        templateUrl: '/templates/widgets/star-interest.html',
        bindings: {
            status: '<',
            onToggle: '&',
            tooltipPosition: '@?',
        },
        controller: function(InterestWording) {
            const $ctrl = this;
            addStarComponentBehaviour($ctrl, InterestWording);
        },
    });

    app.component('starButton', {
        templateUrl: '/templates/widgets/star-button.html',
        bindings: {
            status: '<',
            onToggle: '&',
            nbStarred: '<',
            onShowUsersWithStar: '&',
            disabled: '<?',
        },
        controller: function(InterestWording) {
            const $ctrl = this;
            addStarComponentBehaviour($ctrl, InterestWording);

            $ctrl.isDisabled = () => !!($ctrl.disabled);
        },
    });

    const addWatchComponentBehaviour = ($ctrl, InterestWording, WatchInterestState) => {
        const toggle = (nextStatus) => {
            $ctrl.onToggle({ nextStatus });
        };

        const { values: { YES, ENO }, isWatching } = WatchInterestState;
        $ctrl.isWatching = () => isWatching($ctrl.status);
        $ctrl.toggleWatch = () => toggle(YES);
        $ctrl.toggleUnwatch = () => toggle(ENO);

        const { labels, tooltips } = InterestWording;
        $ctrl.labels = { ...labels };
        $ctrl.tooltips = { ...tooltips };
    };

    app.component('watchInterest', {
        templateUrl: '/templates/widgets/watch-interest.html',
        bindings: {
            status: '<',
            onToggle: '&',
            tooltipPosition: '@?',
        },
        controller: function(InterestWording, WatchInterestState) {
            const $ctrl = this;
            addWatchComponentBehaviour($ctrl, InterestWording, WatchInterestState);
        },
    });

    app.component('watchButton', {
        templateUrl: '/templates/widgets/watch-button.html',
        bindings: {
            status: '<',
            onToggle: '&',
            nbWatching: '<',
            onShowWatchingUsers: '&',
        },
        controller: function(InterestWording, WatchInterestState) {
            const $ctrl = this;
            addWatchComponentBehaviour($ctrl, InterestWording, WatchInterestState);
        },
    });

    app.component('multiRegionBar', {
        bindings : {
            regions: '<' // regions should be a list with objects like {size: [int] length of the region, color: [str] color code of the region}
        },
        templateUrl: "/templates/directives/multi-region-bar.html",
        controller: function multiRegionBarController() {
            const $ctrl = this;
            let updateTotal = function () {
                $ctrl.total = $ctrl.regions.reduce((acc, region) => acc + region.size, 0);
            }

            $ctrl.onInit = function() {
                updateTotal();
            }
            $ctrl.$onChanges = function () {
                updateTotal();
            }
        }
    })

    app.directive("apiErrorAlert", function ($rootScope, $injector, ActivityIndicator, CreateModalFromTemplate, OpalsService, ClipboardUtils) {
        // FM doesn't have DataikuAPI or RequestCenterService
        let DataikuAPI = $injector.has('DataikuAPI') ? $injector.get('DataikuAPI') : null;
        let RequestCenterService = $injector.has('RequestCenterService') ? $injector.get('RequestCenterService') : null;

        return {
            restrict : 'A',
            scope: {
                apiErrorAlert : '=',
                closable : '=',
                errorFoldable : '@',
                canBeUnexpected : '=?'
            },
            link : function($scope) {
                if ($scope.canBeUnexpected === undefined) {
                    $scope.canBeUnexpected = true;
                }
                $scope.options = {
                    canBeUnexpected : $scope.canBeUnexpected,
                    closable : $scope.closable,
                    errorFoldable : $scope.errorFoldable
                }
                $scope.open = true;
                $scope.reset = function() {
                    if ($scope.apiErrorAlert) {
                        $scope.apiErrorAlert.httpCode  = null;
                        $scope.apiErrorAlert.errorType = null;
                    }
                }

                $scope.openHelpCenterDoc = function(event, href) {
                    // if the help center is not activated, just let the event open a new tab by default
                    // also check if there is no open modal as it prevent the use of opals
                    OpalsService.isEnabled().then(function(isOpalsEnabled) {
                        if (isOpalsEnabled && $('.modal-container').length === 0) {
                            OpalsService.navigateToAndShowDrawer(OpalsService.PAGES.EMBEDDED_BROWSER, { href });
                        } else {
                            window.open(href, '_blank');
                        }
                    });
                    event.preventDefault();
                }
                $scope.isUnauthorizedProfileError = $scope.apiErrorAlert && $scope.apiErrorAlert.code && $scope.apiErrorAlert.code==="ERR_USER_ACTION_FORBIDDEN_BY_PROFILE";
                $scope.isCredentialError = $scope.apiErrorAlert && $scope.apiErrorAlert.code && ($scope.apiErrorAlert.code==="ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL" ||$scope.apiErrorAlert.code==="ERR_CONNECTION_NO_CREDENTIALS");

                $scope.canCopyErrorToClipboard = function() {
                    return $scope.apiErrorAlert
                        // any exception that uses smart-log-tail should show the copy error button
                        && ['com.dataiku.dip.io.CustomPythonKernelException',
                            'com.dataiku.dip.exceptions.ExternalProcessFailedException',
                            'com.dataiku.dip.io.SocketBlockLinkIOException',
                            'com.dataiku.dip.io.SocketBlockLinkKernelException'].includes($scope.apiErrorAlert.errorType)
                        && $scope.apiErrorAlert.logTail
                        && $scope.apiErrorAlert.logTail.lines;
                }
                $scope.copyErrorToClipboard = function() {
                    ClipboardUtils.copyToClipboard($scope.apiErrorAlert.logTail.lines.join('\n'));
                }
            },
            templateUrl: '/templates/api-error-alert.html',
            controller: function($scope) {
                this.$onInit = function() {
                    if ($scope.apiErrorAlert && $scope.apiErrorAlert.code && $scope.apiErrorAlert.code === "ERR_USER_ACTION_FORBIDDEN_BY_PROFILE") {
                        $rootScope.sendWT1RequestUpgradeProfileShow();
                    }
                }
            }
        }
    });

    app.directive("sidekickAlert", function() {
        return {
            restrict : "E",
            transclude: true,
            templateUrl : "/templates/sidekick-alert.html"
        }
    });

    app.filter("detailedMessageOrMessage", function(){
        return function(input) {
            if (!input) return "";
            return input.detailedMessage || input.message;
        }
    });

    /**
        /!\ Note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.
    **/
    app.factory("disableElement", function(getBootstrapTooltipPlacement) {
        return function(element, disabled, message, position) {
            if (disabled === true) {
                element.addClass("disabled");
                element.prop("disabled", "disabled");
                element.css("position", "relative");
                element.css("pointer-events", "auto");
                var div = $('<div>').addClass("fh disabled-if-overlay").attr("title", message).appendTo(element);
                div.on('click', function () { return false; });
                if (message && message.length) {
                    div.tooltip({container: "body", placement: getBootstrapTooltipPlacement(position)})
                        .on('hide.bs.modal', function(e) {
                            // Avoid wrong behavior with stacked modals: see sc-86865
                            e.stopPropagation();
                        });
                }
            } else if (disabled === false) {
                element.removeClass("disabled");
                element.css("pointer-events", null);
                element.prop("disabled", null);
                element.find('.disabled-if-overlay').tooltip('destroy').remove();
            }
        }
    });

    /*
        /!\ Do not use on a HTML element that also uses a `title` attribute, as two overlapping tooltips would appear.
        Instead you should handle the tooltip content in all cases (disabled + your other cases) in the `title` attribute. Do not forget to escape the ' in the
        disabled if RO message: "You don\'t have write permissions for this project".
        Also note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.
    */
    app.directive("disabledIfRo", function(disableElement, translate){
        return {
            restrict : 'A',
            link : function(scope, element) {
                scope.$watch("!canWriteProject()", function(nv) {
                    if (nv === undefined) return;
                    return disableElement(element, nv, translate("PROJECT.PERMISSIONS.WRITE_ERROR", "You don't have write permissions for this project"));
                });
            }
        }
    });

    app.directive("disabledIfProjectFolderRo", function($rootScope, disableElement, translate) {
        return {
            restrict : 'A',
            link : function(scope, element) {
                scope.$watch(function() {
                    return !$rootScope.isDSSAdmin() && !$rootScope.canWriteInProjectFolder();
                }, function(nv) {
                    if (nv === undefined) return;
                    return disableElement(element, nv, translate("PROJECT.FOLDER.PERMISSIONS.WRITE_ERROR", "You don't have write contents permissions on this folder"));
                });
            }
        }
    });

    /*
        /!\ If you use the disabledMessage attribute, do not use on a HTML element that also uses a `title` attribute, as two overlapping tooltips would appear.
        Instead you should handle the tooltip content in all cases (disabled + your other cases) in the `title` attribute.
        Also note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.
    */
    app.directive("disabledIf", function(disableElement){
        return {
            restrict : 'A',
            link: function(scope, element, attrs) {
                scope.$watch(attrs.disabledIf, function(nv) {
                    return disableElement(element, nv, attrs.disabledMessage, attrs.disabledPosition);
                });
            }
        }
    });

    /*
        /!\ Do not use on a HTML element that also uses a 'title' attribute, as two overlapping tooltips would appear.
        Instead you should handle the tooltip content in all cases (disabled + your other cases) in the `title` attribute.
        Also note that you should **not** use any directive leveraging `disableElement` factory on an element that supports the `disabled` attribute (e.g. button)
       if you also want to use `title`, as in Chrome it would prevent the tooltip from triggering; instead use `ng-class` to conditionally set `disabled`.

        Sets element as disabled if disabledIfMessage is a non-empty string
        and displays that string as the tooltip
    */
    app.directive("disabledIfMessage", function(disableElement){
        return {
            restrict : 'A',
            scope : {
                disabledIfMessage: '='
            },
            link: function(scope, element, attrs) {
                scope.$watch('disabledIfMessage', function(nv) {
                    return disableElement(element, !!nv, scope.disabledIfMessage, attrs.disabledPosition);
                });
            }
        }
    });

    app.directive("disabledBlockIfRo", function(){
        return {
            restrict : 'A',
            link : function(scope, element) {
                scope.$watch("canWriteProject()", function(nv) {
                    if (nv === false) {
                        element.addClass("disabled-block");
                    } else if( nv === true) {
                        element.removeClass("disabled-block");
                    }
                });
            }
        }
    });

    /* similar to ng-show but uses CSS visibility rather than display property (no movement in the page) */
    app.directive('visibleIf', function() {
        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                var toggle = function (show){
                    $(element).css('visibility', show ? 'visible': 'hidden');
                };
                toggle(scope.$eval(attrs.visibleIf));
                scope.$watch(attrs.visibleIf, toggle);
            }
        };
    });

    /* Directive that can be used to forward API errors to an blockApiError in scope */
    app.directive("apiErrorContext", function() {
        return {
            controller: function ($scope) {
                this.setError = setErrorInScope.bind($scope);
            }
        }
    });

    /* API error that displays as an alert block */
    app.directive('blockApiError', function() {
        return {
            templateUrl: '/templates/block-api-error.html',
            replace: false,
            restrict: 'ECA',
            link: function(scope) {
              // can be used by children to report their error.
              scope.setError = setErrorInScope.bind(scope);
            }
        };
    });

    /* API warning InfoMessages that displays as an alert block */
    app.component('blockApiWarning', {
        templateUrl: '/templates/block-api-warning.html',
        bindings: {
            warningMessages: '='
        },
        controller: function() {
            this.resetWarnings = () => {
                if (this.warningMessages)
                    this.warningMessages.messages = null;
            };
        }
    });

    app.directive("tlUser", function($rootScope){
        return {
            template : `
                <span class="who" ng-if="item.user" toggle="tooltip" title="{{item.details.userDisplayName || item.user}} (@{{item.user}})">
                    <span ng-if="item.user === 'no:auth'" ng-bind="item.user | niceLogin"></span>
                    <a ng-if="item.user !== 'no:auth'" href="/profile/{{item.user}}/">
                        {{item.user == rootScope.appConfig.user.login ? "You" : (item.details.userDisplayName ? item.details.userDisplayName : item.user)}}
                    </a>
                </span>`,
            scope : false,
            link : function($scope) {
                $scope.rootScope= $rootScope;
            }
        }
    });

    app.directive('metadataObjectModal', () => {
        return {
            scope : true,
            link : function($scope) {
                if ($scope.metadataObjectParent.customMeta && $scope.metadataObjectParent.customMeta.kv) {
                    $scope.localObject = angular.copy($scope.metadataObjectParent.customMeta.kv);
                } else {
                    $scope.localObject = { };
                }
                $scope.save = function() {
                    $scope.metadataObjectParent.customMeta = { 'kv' : angular.copy($scope.localObject) };
                    $scope.$emit("metadataObjectUpdated");
                    $scope.dismiss();
                };
            }
        }
    });

     app.directive('metadataObjectLink', function(CreateModalFromTemplate){
        return {
            restrict : 'AE',
            scope : {
                metadataObjectParent : '='
            },
            template: '<div class="metadata-object-link"><pre class="small-pre">{{metadataObjectParent.customMeta.kv | json}}</pre><button title="Object metadata" class="btn btn--secondary" ng-click="openModal()"><i class="icon-superscript" />&nbsp;Edit</button></div>',
            link : ($scope) => {
                $scope.openModal = function(){
                    CreateModalFromTemplate("/templates/widgets/metadata-object-modal.html", $scope,
                        null, null);
                }
            }
        }
    });

     app.directive('overrideTableModal', function($stateParams, DataikuAPI){
        return {
            scope : true,
            // compile/pre to execute before listForm's link (for transcope)
            compile : function(){ return { pre: ($scope) => {
                $scope.simplifiedObjectToOverride = {};
                $scope.$watch("objectToOverride", function(nv){
                    if (nv) {
                        $scope.simplifiedObjectToOverride = angular.copy($scope.objectToOverride);
                        $.each($scope.simplifiedObjectToOverride, function(k) {
                            if (k.indexOf("$") == 0) {
                                delete $scope.simplifiedObjectToOverride[k];
                            }
                        });
                        delete $scope.simplifiedObjectToOverride["overrideTable"];
                        delete $scope.simplifiedObjectToOverride["change"];
                        delete $scope.simplifiedObjectToOverride["versionTag"];
                    }
                }, true);
                if ($scope.overrideTableParent.overrideTable) {
                    $scope.localTable = angular.copy($scope.overrideTableParent.overrideTable);
                } else {
                    $scope.localTable = { "overrides" : []};
                }
                $scope.save = function() {
                    if ($scope.localTable.overrides.length > 0) {
                        $scope.overrideTableParent.overrideTable = angular.copy($scope.localTable)
                    } else {
                        $scope.overrideTableParent.overrideTable = null;
                    }
                    $scope.$emit("overrideTableUpdated");
                    $scope.dismiss();

                };
                $scope.getValue = (function(override) {
                    DataikuAPI.variables.expandExpr($stateParams.projectKey, override.expr).success(function(data){
                        override.$$computedValue = data.id;
                    }).error(setErrorInScope.bind(this));
                }).bind($scope); // bind on parent scope
            } }; }
        }
    });

    app.directive('overrideTableBtnLink', function(CreateModalFromTemplate){
        return {
            scope : {
                overrideTableParent : '=',
                objectToOverride : '='
            },
            template: '<div class="override-table-link"><pre class="small-pre">{{overrideDesc}}</pre><button title="Override variables" class="btn btn--secondary" ng-click="openModal()"><i class="icon-superscript" />&nbsp;Edit</button></div>',
            link : ($scope) => {
                $scope.overrideDesc = '';
                $scope.$watch('overrideTableParent.overrideTable', function(nv) {
                    if ( nv == null) return;
                    var desc = '';
                    if ( $scope.overrideTableParent.overrideTable.overrides != null ) {
                        $scope.overrideTableParent.overrideTable.overrides.forEach(function(override) {
                            desc = desc + override.path + " ";
                        });
                    }
                    $scope.overrideDesc = desc;
                }, true);
                $scope.openModal = function(){
                    CreateModalFromTemplate("/templates/widgets/override-table-modal.html", $scope,
                        null, null);
                }
            }
        }
    });
     app.directive('overrideTableLink', function(CreateModalFromTemplate){
        return {
            scope : {
                overrideTableParent : '=',
                objectToOverride : '='
            },
            template: '<a title="Override variables" ng-class="{\'override-table-link\': true, \'overriden\': overrideTableParent.overrideTable.overrides.length}" ng-click="openModal()"><i class="dku-icon-text-superscript-16" /></a>',
            link : ($scope) => {
                $scope.openModal = function(){
                    CreateModalFromTemplate("/templates/widgets/override-table-modal.html", $scope,
                        null, null);
                }
            }
        }
    });
    app.directive('dkuIndeterminate', function() {
        return {
            restrict: 'A',
            link: function(scope, element, attributes) {
                scope.$watch(attributes.dkuIndeterminate, function(value) {
                    element.prop('indeterminate', !!value);
                });
            }
        };
    });

    app.directive('validFile',function(){
        return {
            require:'ngModel',
            link:function(scope, el, attrs, ngModel) {
                el.bind('change', function() {
                    var val = 'multiple' in attrs ? this.files : this.files[0];
                    scope.$apply(function() {
                        ngModel.$setViewValue(val);
                        ngModel.$render();
                    });
                });
            }
        };
    });


    app.directive('sparkline', function() {
        return {
            scope: {
                sparkline: '='
            },
            link: function(scope, element) {
                const data = scope.sparkline;
                const rect = element[0].parentElement.getBoundingClientRect();
                const x = d3.scale.linear().domain([0, data.length-1]).range([0, rect.width]);
                const y = d3.scale.linear().domain([0, d3.max(data) || 0]).range([rect.height, 4]);
                const line = d3.svg.line().x((d, i) => x(i)).y(d => y(d || 0));

                d3.select(element[0]).html("")
                    .append("svg:svg")
                    .append("svg:path")
                    .attr("d", line(data))
                    .attr("stroke-width", "2px")
                    .attr("stroke", "#add8e6")
                    .attr("fill", "#add8e6")
                    .attr("fill-opacity", .3);
            }
        }
    });

    app.directive('weekDaysPicker', function(translate) {
        return {
            scope : {
                selection:'=ngModel',
                onChange:'&?'
            },
            template: `<div class="weekdays-picker">
                <span ng-repeat="day in days" ng-click="toggle(day.value)" ng-class="[{selected: hasSelected(day.value)}]" >{{day.label[0]}}</span>
            </div>`,
            link: function($scope) {
                //  global WEEKDAYS we use an object because the technical name should stays the same
                $scope.days = [...WEEKDAYS].map(weekday => {
                return {
                    "value": weekday,
                    "label": translate('GLOBAL.WEEKDAYS.' + weekday.toUpperCase(), weekday)
                }});

                $scope.hasSelected = (day) => {
                    return $scope.selection.includes(day);
                };

                $scope.toggle = (day) => {
                    if($scope.selection.includes(day)) {
                        $scope.selection = $scope.selection.filter(s => s !== day);
                    } else {
                        $scope.selection = [...$scope.selection, day];

                    }
                    if($scope.onChange) {
                        $scope.onChange($scope.selection);
                    }
                };

                $scope.$watch('selection', () => {
                    if(!Array.isArray($scope.selection)) {
                        $scope.selection = [];
                    }
                });
            }
        }
    });

    app.directive('inlineDateRangePicker', ['$timeout', () => { // inline-date-range-picker
        return {
            scope : {
                from:'=',
                to:'=',
                tz:'=',
                onChange:'&?' // () => { $event: { from: Date, to: Date, tz: string } }
            },
            template: `
                <div class="inline-date-range-picker" ng-class="{ 'inline-date-range-picker--big-inputs': hasBiggerInputs }">
                    <fieldset>
                        <div class="fieldLabel" translate="WIDGET.DATE_RANGE_PICKER.FROM">From</div>
                        <div class="inline-date-range-picker__row">
                            <input class="inline-date-range-picker__date" ng-model="from" ng-keypress="onFromKeypress($event)" ng-change="onFromChange()" ng-blur="onFromChange()" type="date" name="dateFrom" />
                            <input class="inline-date-range-picker__time" ng-model="from" ng-model-options="{ timeSecondsFormat: 'ss' }" ng-keypress="onFromKeypress($event)" ng-change="onFromChange()" ng-blur="onFromChange()" type="time" name="timeFrom" step="1" />
                        </div>
                    </fieldset>
                    <fieldset>
                        <div class="fieldLabel" translate="WIDGET.DATE_RANGE_PICKER.TO">To</div>
                        <div class="inline-date-range-picker__row">
                            <input class="inline-date-range-picker__date" ng-model="to" ng-keypress="onToKeypress($event)" ng-change="onToChange()" ng-blur="onToChange()" type="date" name="dateTo" />
                            <input class="inline-date-range-picker__time" ng-model="to" ng-model-options="{ timeSecondsFormat: 'ss' }" ng-keypress="onToKeypress($event)" ng-change="onToChange()" ng-blur="onToChange()" type="time" name="timeTo" step="1" />
                        </div>
                    </fieldset>
                    <fieldset>
                        <div class="fieldLabel" translate="WIDGET.DATE_RANGE_PICKER.TIMEZONE">Timezone</div>
                        <basic-select class="inline-date-range-picker__timezone" items="timezone_ids" ng-model="tz"></basic-select>
                    </fieldset>
                    <div class="ff-date-range-filter-hint" ng-show="to && from && to < from">
                        <i class="icon-warning-sign"></i>&nbsp;<span translate="WIDGET.DATE_RANGE_PICKER.WARNING.START_DATE_AFTER_END_DATE">Start date is later than end date. The result will always be empty.</span>
                    </div>
                </div>
                `,
            link : function($scope) {
                $scope.timezone_ids = [ "UTC",
                    "Africa/Abidjan", "Africa/Addis_Ababa", "Africa/Algiers", "Africa/Bamako", "Africa/Bangui", "Africa/Brazzaville", "Africa/Cairo",
                    "Africa/Casablanca", "Africa/Conakry", "Africa/Dakar", "Africa/Djibouti", "Africa/Harare", "Africa/Johannesburg", "Africa/Kigali",
                    "Africa/Kinshasa", "Africa/Lagos", "Africa/Libreville", "Africa/Mogadishu", "Africa/Nairobi", "Africa/Ndjamena", "Africa/Niamey",
                    "Africa/Nouakchott", "Africa/Ouagadougou", "Africa/Tripoli", "Africa/Tunis",
                    "America/Adak", "America/Anchorage", "America/Argentina/Buenos_Aires", "America/Aruba", "America/Bogota", "America/Cancun", "America/Caracas",
                    "America/Cayenne", "America/Cayman", "America/Chicago", "America/Costa_Rica", "America/Dawson_Creek", "America/Denver", "America/Detroit",
                    "America/El_Salvador", "America/Goose_Bay", "America/Grenada", "America/Guadeloupe", "America/Guatemala", "America/Guyana", "America/Halifax",
                    "America/Havana", "America/Indianapolis", "America/Jamaica", "America/Juneau", "America/La_Paz", "America/Lima", "America/Los_Angeles",
                    "America/Martinique", "America/Mexico_City", "America/Monterrey", "America/Montevideo", "America/Montserrat", "America/Nassau", "America/New_York",
                    "America/Noronha", "America/Panama", "America/Puerto_Rico", "America/Santiago", "America/Sao_Paulo", "America/St_Johns", "America/Tijuana",
                    "America/Toronto", "America/Vancouver", "America/Winnipeg",
                    "Arctic/Longyearbyen",
                    "Asia/Baghdad", "Asia/Bahrain", "Asia/Baku", "Asia/Bangkok", "Asia/Beirut", "Asia/Brunei", "Asia/Calcutta",
                    "Asia/Damascus", "Asia/Dhaka", "Asia/Dubai", "Asia/Hebron", "Asia/Ho_Chi_Minh", "Asia/Hong_Kong", "Asia/Irkutsk",
                    "Asia/Jakarta", "Asia/Jerusalem", "Asia/Kabul", "Asia/Karachi", "Asia/Kathmandu", "Asia/Kuala_Lumpur", "Asia/Kuwait",
                    "Asia/Macao", "Asia/Macau", "Asia/Manila", "Asia/Phnom_Penh", "Asia/Qatar", "Asia/Riyadh", "Asia/Saigon",
                    "Asia/Seoul", "Asia/Shanghai", "Asia/Singapore", "Asia/Taipei", "Asia/Tehran", "Asia/Tel_Aviv", "Asia/Tokyo",
                    "Atlantic/Azores", "Atlantic/Bermuda", "Atlantic/Canary", "Atlantic/Cape_Verde", "Atlantic/Madeira", "Atlantic/Reykjavik",
                    "Australia/ACT", "Australia/Adelaide", "Australia/Brisbane", "Australia/Canberra", "Australia/Darwin", "Australia/Eucla", "Australia/Lord_Howe",
                    "Australia/Melbourne", "Australia/NSW", "Australia/North", "Australia/Perth", "Australia/Queensland", "Australia/South", "Australia/Sydney",
                    "Australia/Tasmania", "Australia/Victoria", "Australia/West",
                    "Brazil/East", "Brazil/West",
                    "Canada/Atlantic", "Canada/Central", "Canada/East-Saskatchewan", "Canada/Eastern", "Canada/Mountain", "Canada/Newfoundland", "Canada/Pacific",
                    "Canada/Saskatchewan", "Canada/Yukon",
                    "Europe/Amsterdam", "Europe/Athens", "Europe/Belfast", "Europe/Berlin", "Europe/Bratislava", "Europe/Brussels", "Europe/Bucharest",
                    "Europe/Budapest", "Europe/Busingen", "Europe/Copenhagen", "Europe/Dublin", "Europe/Helsinki", "Europe/Istanbul", "Europe/Kiev",
                    "Europe/Lisbon", "Europe/Ljubljana", "Europe/London", "Europe/Luxembourg", "Europe/Madrid", "Europe/Malta", "Europe/Minsk",
                    "Europe/Monaco", "Europe/Moscow", "Europe/Nicosia", "Europe/Oslo", "Europe/Paris", "Europe/Prague", "Europe/Riga",
                    "Europe/Rome", "Europe/Sarajevo", "Europe/Sofia", "Europe/Stockholm", "Europe/Tallinn", "Europe/Uzhgorod", "Europe/Vienna",
                    "Europe/Vilnius", "Europe/Warsaw", "Europe/Zagreb", "Europe/Zurich",
                    "Indian/Cocos", "Indian/Maldives", "Indian/Mauritius", "Indian/Mayotte", "Indian/Reunion",
                    "Pacific/Apia", "Pacific/Auckland", "Pacific/Chatham", "Pacific/Enderbury", "Pacific/Gambier", "Pacific/Guam", "Pacific/Honolulu",
                    "Pacific/Kiritimati", "Pacific/Marquesas", "Pacific/Niue", "Pacific/Noumea", "Pacific/Pitcairn", "Pacific/Tahiti", "Pacific/Wallis"
                ];

                $scope.hasBiggerInputs = navigator.userAgent.includes("Gecko/");

                // $scope.updateOn = 'blur';
                $scope.$watch('updateOn', (value) => {
                    $scope.updateOn = value || 'blur';
                });

                let currentTo = null;
                $scope.$watch("to", (nv, ov) => {
                    if (nv) {
                        currentTo = new Date(nv.getTime());
                    } else {
                        currentTo = null;
                    }
                    if (ov && !nv) {
                        // Trigger an onChange event when user click on the "clear" button
                        // (side-effect: also trigger an event when user enters an invalid date)
                        $scope.onChange && $scope.onChange({
                            $event: {
                                from: $scope.from,
                                to: nv,
                                tz: $scope.tz
                            }
                        });
                    }
                });

                let currentFrom = null;
                $scope.$watch("from", (nv, ov) => {
                    if (nv) {
                        currentFrom = new Date(nv.getTime());
                    } else {
                        currentFrom = null;
                    }
                    if (ov && !nv) {
                        // Trigger an onChange event when user click on the "clear" button
                        // (side-effect: also trigger an event when user enters an invalid date)
                        $scope.onChange && $scope.onChange({
                            $event: {
                                from: nv,
                                to: $scope.to,
                                tz: $scope.tz
                            }
                        });
                    }
                });

                $scope.$watch("tz", (nv, ov) => {
                    if (nv !== ov) {
                        $scope.onChange && $scope.onChange({
                            $event: {
                                from: $scope.from,
                                to: $scope.to,
                                tz: nv
                            }
                        });
                    }
                });

                let isManualEdit = false;
                $scope.onFromKeypress = ({key}) => {
                    if (key === 'Enter') {
                        onFromEdited();
                    } else {
                        isManualEdit = true;
                    }
                };

                $scope.onFromChange = () => {
                    if (isManualEdit) {
                        isManualEdit = false;
                        return;
                    }
                    onFromEdited();
                };


                $scope.onFromBlur = () => {
                    if (!isManualEdit) {
                        return;
                    }
                    onFromEdited();
                };

                $scope.onToKeypress = ({key}) => {
                    if (key === 'Enter') {
                        onToEdited();
                    } else {
                        isManualEdit = true;
                    }
                };

                $scope.onToChange = () => {
                    if (isManualEdit) {
                        isManualEdit = false;
                        return;
                    }
                    onToEdited();
                };

                $scope.onToBlur = () => {
                    if (!isManualEdit) {
                        return;
                    }
                    onToEdited();
                };

                function onFromEdited() {
                    if (currentFrom === $scope.from) {
                        return;
                    }
                    if ($scope.from && $scope.to) {
                        if ($scope.from.getTime() > $scope.to.getTime()) {
                            $scope.to = new Date($scope.from.getTime());
                        }
                    }
                    $scope.onChange && $scope.onChange({
                        $event: {
                            from: $scope.from,
                            to: $scope.to,
                            tz: $scope.tz
                        }
                    });
                }

                function onToEdited() {
                    if (currentTo === $scope.to) {
                        return;
                    }
                    if ($scope.from && $scope.to) {
                        if ($scope.from.getTime() > $scope.to.getTime()) {
                            $scope.from = new Date($scope.to.getTime());
                        }
                    }
                    $scope.onChange && $scope.onChange({
                        $event: {
                            from: $scope.from,
                            to: $scope.to,
                            tz: $scope.tz
                        }
                    });
                }
            }
        }
    }]);

    app.directive('executionPlan', function() {
        return {
            restrict: "AE",
            scope: {
                executionPlan: '=ngModel'
            },
            templateUrl: '/templates/widgets/execution-plan.html',
            link: () => {
                //nothing to do for now...
            }
        };
    });

    // SO : http://stackoverflow.com/questions/18368485/angular-js-resizable-div-directive
    app.directive('resizer', function($document,Throttle,$rootScope) {
        return function($scope, $element, $attrs) {

            $element.addClass('content-resizer');

            $element.on('mousedown', function(event) {
                $element.parent().addClass("resizing");
                event.preventDefault();
                $document.on('mousemove', mousemove);
                $document.on('mouseup', mouseup);
            });

            function mousemove(event) {

                if ($attrs.resizer == 'vertical') {
                    // Handle vertical resizer
                    let x = event.pageX;

                    if ($attrs.resizerContainer) {
                        const containerWidth = $($attrs.resizerContainer).width();
                        const maxWidth = containerWidth - parseInt($attrs.resizerWidth);
                        x = Math.min(maxWidth, x);
                    }
                    x = Math.max(0, x);

                    $element.css({
                        left: x + 'px'
                    });

                    $($attrs.resizerLeft).css({
                        width: x + 'px'
                    });
                    $($attrs.resizerRight).css({
                        left: (x + parseInt($attrs.resizerWidth)) + 'px'
                    });

                } else {
                    // Handle horizontal resizer
                    let y = window.innerHeight - event.pageY;
                    if ($attrs.resizerContainer) {
                        const containerHeight = $($attrs.resizerContainer).height();
                        const maxHeight = containerHeight - parseInt($attrs.resizerHeight);
                        y = Math.min(maxHeight, y);
                    }
                    y = Math.max(0, y);

                    $element.css({
                        bottom: y + 'px'
                    });

                    $($attrs.resizerTop).css({
                        bottom: (y + parseInt($attrs.resizerHeight)) + 'px'
                    });
                    $($attrs.resizerBottom).css({
                        height: y + 'px'
                    });
                }
            }

            function mouseup() {
                $document.unbind('mousemove', mousemove);
                $document.unbind('mouseup', mouseup);
                $element.parent().removeClass("resizing");
                $rootScope.$broadcast('reflow');
            }
        };
    });

    app.directive('fatTable', function($compile,$rootScope,Debounce,$http,$templateCache) {

         return {
            restrict : 'A',
            scope : {
                rows:'=',
                as : '@',
                rowIndexAs:'@',
                headers:'=',
                columnWidths : '=',
                headerTemplate:'@',
                cellTemplate:'@',
                printNewLinesAsSymbols:'@',
                rowHeight:'=',
                headerHeight : '=',
                digestChildOnly:'=?'
            },
            link : function(scope, element, attrs) {
                $http.get(scope.cellTemplate, {cache: $templateCache}).then(function(resp) {
                    let cellTemplateHTML = resp.data;
                    $http.get(scope.headerTemplate, {cache: $templateCache}).then(function(resp) {
                        let headerTemplateHTML = resp.data;
                        // We don't use Debounce here because it always triggers a full digest cycle!
                        var digestTimeout = undefined;
                        function debouncedDigestCycle() {
                            if(digestTimeout === undefined) {
                                digestTimeout = setTimeout(function() {
                                    digestTimeout = undefined;
                                    if(scope.digestChildOnly) {
                                        var elmScope = element.scope();
                                        if(elmScope) {
                                            // Partial digestion
                                            elmScope.$digest();
                                        }
                                    } else {
                                        // Full digestion
                                        $rootScope.$digest();
                                    }
                                },10);
                            }
                        }

                        function cleanDOM(div) {
                            // Destroy the cell's __fat_scope__
                            var fs = div.__fat_scope__;
                            if(fs) {
                                fs.$destroy();
                            }
                            div.__fat_scope__ = undefined;
                            // Make sure there is no refs to the scope in JQuery's cache
                            $(div).data('$scope',null);
                        }

                        function buildModel() {
                           var tableData = new fattable.SyncTableModel();
                           tableData.getCellSync = function(i,j) {
                               var arr = scope.rows;
                               if(!arr || !arr.length || i<0 || i>=arr.length) {
                                   return {i:i,j:j,v:undefined,t:'c'};
                               } else {
                                   var row = arr[i];
                                   if(!row || !row.length || j < 0 || j >=row.length) {
                                        return {i:i,j:j,v:undefined,t:'c'};
                                   }
                                   return {i:i,j:j,v:row[j],t:'c'};
                               }
                           };
                           tableData.getHeaderSync = function(i) {
                               var arr = scope.headers;
                               if(!arr || !arr.length || i<0 || i>=arr.length) {
                                   return {i:i,v:undefined,t:'h'};
                               } else {
                                  return {i:i,v:arr[i],t:'h'};
                               }
                           };
                           return tableData;
                        }

                        var livingCells = [];

                        function buildPainter() {
                           var painter = new fattable.Painter();

                           var prepareElement = function(template) {
                              return function(cellDiv, data) {
                                  if(!cellDiv.__fat_scope__) {
                                      var elementScope = element.scope();
                                      if(elementScope) {
                                          cellDiv.__fat_scope__ = elementScope.$new();
                                          $(cellDiv).append($compile(template)(cellDiv.__fat_scope__));
                                      }
                                  }
                                  if(cellDiv.__fat_scope__) {
                                      let v = data.v;
                                      if (scope.printNewLinesAsSymbols && (typeof v === 'string' || v instanceof String)) {
                                          v = v.replace(/(\r\n|\n)/g, "¶");
                                      }
                                      cellDiv.__fat_scope__[attrs.as] = v;
                                      if(attrs.rowIndexAs && data.t == 'c') {
                                            cellDiv.__fat_scope__[attrs.rowIndexAs] = data.i;
                                      }
                                      debouncedDigestCycle();
                                  }
                               };
                           };

                           painter.fillCell = prepareElement(cellTemplateHTML);
                           painter.fillHeader = prepareElement(headerTemplateHTML);

                           painter.fillCellPending = (cellDiv) => {
                               cellDiv.textContent = "";
                               cellDiv.className = "pending";
                           };

                           painter.fillHeaderPending = (cellDiv) => {
                              cellDiv.textContent = "";
                              cellDiv.className = "pending";
                           };

                           painter.setupCell = function(div) {
                              livingCells.push(div);
                           };

                           painter.setupHeader = painter.setupCell;

                           painter.cleanUpCell = function(div) {
                               livingCells = livingCells.filter(function(x) {
                                   return x!=div;
                               });
                               cleanDOM(div);
                           };

                           painter.cleanUpHeader = painter.cleanUpCell;

                           return painter;
                        }
                        var oldTable;

                        function redraw() {
                            if(oldTable) {
                                oldTable.cleanUp();
                                // bug in fattable : cleanUp() at line 702 is not checking the variable holding the scroll proxy, so
                                // the scroll elements still try to call onScroll (until the next DOM rebuild where they're removed)
                                if (oldTable.scroll != null) {
                                    oldTable.scroll.onScroll = function() {}; // NOSONAR: noop
                                }
                            }
                            const table = fattable({
                                "container": element[0],
                                "model": buildModel(),
                                "nbRows": scope.rows? scope.rows.length:0,
                                "rowHeight": scope.rowHeight,
                                "headerHeight": scope.headerHeight,
                                "painter": buildPainter(),
                                "columnWidths": scope.columnWidths
                            });
                            if(oldTable && oldTable.scroll) {
                                var y = oldTable.scroll.scrollTop;
                                var x = oldTable.scroll.scrollLeft;
                                table.scroll.setScrollXY(x,y);
                            }
                            oldTable = table;
                        }

                       var debouncedRedraw = Debounce().withDelay(50,200).wrap(redraw);
                       scope.$watch('rows', debouncedRedraw, false);
                       scope.$watch('headers', debouncedRedraw, false);
                       $(window).on('resize', debouncedRedraw);

                       element.scope().$on("reflow", debouncedRedraw);

                       scope.$on("$destroy", function () {
                           if(oldTable) {
                                oldTable.cleanUp();
                                oldTable=null;
                           }
                           for(var i = 0 ; i < livingCells.length ; i++) {
                                cleanDOM(livingCells[i]);
                           }
                           livingCells = [];
                           $(window).off("resize", debouncedRedraw);
                       });
                    });
               });
            }
         };
    });

    app.directive('registerModelForForm', function () {
        return {
            scope: {form: '=registerModelForForm'},
            require: 'ngModel',
            controller: function ($element,$scope) {
                var ngModel = $element.controller('ngModel');
                $scope.form.$addControl(ngModel);
            }
        };
    });

    app.directive('fatRepeat', function($compile, $rootScope, $timeout, Debounce, FatTouchableService, FatDraggableService) {

        return {
            transclude:true,
            scope:{
                fatRepeat:'=', // Array
                fatDraggable:'=?', // If items should be draggable
                fatDraggableOnDrop:'=?', // Callback called when drag ends
                as:'@', // Name of each item
                rowHeight:'=', // Height of each row
                colWidth: '=?', // width of each column
                digestChildOnly:'=?', // If true, doesn't trigger a full digest cycle each time a cell updated, but call
                // $digest() on child scope only. It's generally MUCH faster, but you need to make sure
                // that your watches have no side effects on parent scopes.
                initScope: '=?', // item scope init function
                tableModel: '=?', // Custom fattable Model
                inForm:'=?',
                layoutMode: '@?', //one of row, mosaic (or potentially left blank for list mode)
                listPadding: '=?', // padding to be introduced before first and after last item
                fTrackTable: '&', // to allow containers to control the scroll or other aspect of the underlying table
                disableScrollTo: '@', // Disable scrollToLine event
                enableAsync: '=?',
                nbRows: '=?',
                chunkSize: '=?',
                getRowChunk: '=?',
                pageFromData: '=?',
                allowHorizontalScroll: '=?', // compute dynamically the row width to allow horizontal scrolling. Only valid for list mode.
                enableDragMove: '<?', // whether the underlying fattable allows middle-click dragging to scroll (default to true)
                dedicatedWidthForVScrollbar: "<?", // whether we always allocate a width for the verticall scrollbar (defaults to true)
            },
            restrict:'A',
            compile: function(_element,_attrs,transclude) {

                return function(scope, element, attrs) {
                    const HORIZ_SCROLL_H = 1;
                    const VERT_SCROLL_W = 8;


                    element.addClass(scope.layoutMode ? 'fat-row' : 'fat-repeat');
                    if (scope.layoutMode=="row") {
                        $(element).css('height', (parseInt(scope.rowHeight,10) + HORIZ_SCROLL_H).toString() + 'px');
                    }

                    // We don't use Debounce here because it always triggers a full digest cycle!
                    var digestTimeout = undefined;
                    function debouncedDigestCycle() {
                        if(digestTimeout === undefined) {
                            digestTimeout = setTimeout(function() {
                                digestTimeout = undefined;
                                if(scope.digestChildOnly) {
                                    var elmScope = element.scope();
                                    if(elmScope) {
                                        // Partial digestion
                                        elmScope.$digest();
                                    }
                                } else {
                                    // Full digestion
                                    $rootScope.$digest();
                                }
                            },10);
                        }
                    }

                    function cleanDOM(div) {
                        // Destroy the cell's __fat_scope__
                        var fs = div.__fat_scope__;
                        if(fs) {
                            fs.$destroy();
                        }
                        div.__fat_scope__ = undefined;
                        // Make sure there is no refs to the scope in JQuery's cache
                        $(div).data('$scope',null);
                    }

                    function buildModel() {
                        if (scope.tableModel) {
                            return scope.tableModel.call(scope);
                        }
                        if (scope.enableAsync) {
                            const asyncTableData = new fattable.PagedAsyncTableModel();
                            asyncTableData.fetchCellPage = (pageName, cb) => {
                                var promise = scope.getRowChunk(pageName);
                                promise.then(response => {
                                    cb(scope.pageFromData(pageName, response.data));
                                });
                            };
                            asyncTableData.cellPageName = (row, col) => {
                                return Math.trunc(row / Math.max(1, scope.chunkSize));
                            };
                            if (scope.fatRepeat && scope.fatRepeat.length > 0) {
                                var initialPageName = asyncTableData.cellPageName(0, 0);
                                const initialPage = scope.pageFromData(initialPageName, {items: [...scope.fatRepeat]});
                                asyncTableData.pageCache.set(initialPageName, initialPage);
                            }
                            return asyncTableData;
                        }
                        const tableData = new fattable.SyncTableModel();

                        tableData.getCellSync = function(i,j) {
                            const arr = scope.fatRepeat;
                            const idx = i*scope.numColumns + j;
                            if(!arr || idx<0 || idx>=arr.length) {
                                return undefined;
                            }
                            return arr[idx];
                        };
                        return tableData;
                    }

                    var livingCells = [];

                    // Scroll bar handling. Observes the changes inside the table and redraw with a forced width if needed
                    const vScrollBarWidth = 11;
                    const getContentBaseWidth = () => element.width() - (scope.dedicatedWidthForVScrollbar !== false ? vScrollBarWidth : 0); // the size available for the content when no scroll is possible or needed
                    let forcedMinimumColumWidth = 0;
                    let resetMinimumColumnWidth = true;
                    let mutationObserver;
                    if(scope.allowHorizontalScroll) {
                        element.addClass('fat-repeat--with-horizontal-scroll')
                        mutationObserver = new MutationObserver(() => {
                                mutationObserver.takeRecords();

                                const requiredWidth = livingCells.reduce(
                                    (acc, cell)=> Math.max(cell.scrollWidth, acc),
                                    0,
                                );

                                if(requiredWidth != forcedMinimumColumWidth && requiredWidth > getContentBaseWidth()) {
                                    forcedMinimumColumWidth = requiredWidth;
                                    resetMinimumColumnWidth = false;
                                    debouncedRedraw();
                                }
                            },
                        );
                        mutationObserver.observe(element[0], {childList: true, subtree: true});
                    }

                    function buildPainter() {
                        var painter = new fattable.Painter();
                        painter.fillCell = function(cellDiv, data) {
                            cellDiv.className = '';
                            if(!cellDiv.__fat_scope__) {
                                var elementScope = element.scope();
                                if(elementScope) {
                                    cellDiv.__fat_scope__ = elementScope.$new();
                                    transclude(cellDiv.__fat_scope__,function(clone) {
                                        $(cellDiv).append(clone);
                                    });
                                }
                            }
                            if(cellDiv.__fat_scope__) {
                                cellDiv.__fat_scope__[attrs.as] = data;
                                debouncedDigestCycle();
                                if (scope.initScope) {
                                    scope.initScope(cellDiv.__fat_scope__);
                                }
                            }
                        };
                        painter.fillCellPending = (cellDiv) => {
                            cellDiv.className = "fat-repeat-pending-row";
                        };
                        painter.setupCell = function(div) {
                            livingCells.push(div);
                        };
                        painter.cleanUpCell = function(div) {
                            livingCells = livingCells.filter(function(x) {
                                return x!=div;
                            });
                            cleanDOM(div);
                        };
                        return painter;
                    }

                    var oldTable;

                    function redraw() {
                        if (oldTable) {
                            oldTable.cleanUp();
                        }
                        let fatRepeatLength = scope.fatRepeat? scope.fatRepeat.length:0;
                        if (scope.layoutMode=="row") {
                            // row mode
                            scope.numColumns = fatRepeatLength;
                            scope.numRows = 1;
                        }
                        else if (scope.layoutMode=="mosaic") {
                            scope.numColumns = Math.floor((element.innerWidth() - VERT_SCROLL_W)/ scope.colWidth);
                            scope.numRows = Math.ceil(fatRepeatLength / scope.numColumns);
                        }
                        else if (scope.enableAsync) {
                            scope.numRows = scope.nbRows;
                            scope.numColumns = 1;
                        } else {
                            scope.numRows = fatRepeatLength;
                            scope.numColumns = 1;
                        }

                        if (scope.listPadding && scope.layoutMode != "row") { // pad the end via a whole row
                            scope.numRows++
                        }

                        let columnWidths = [getContentBaseWidth()];

                        if (['mosaic','row'].includes(scope.layoutMode)) {
                            columnWidths = Array.from({length: scope.numColumns}, () => scope.colWidth);
                            if (columnWidths.length>0  && scope.layoutMode=="row" && scope.listPadding) {
                                if (typeof scope.listPadding === 'string') scope.listPadding = parseInt(scope.listPadding, 10);
                                columnWidths[columnWidths.length-1] += 2 * scope.listPadding;
                            }
                        } else if(scope.allowHorizontalScroll) { // list mode with horizontal scroll bar
                            if(resetMinimumColumnWidth) {
                                forcedMinimumColumWidth = 0;
                            }
                            columnWidths = [Math.max(forcedMinimumColumWidth, columnWidths[0])];
                            resetMinimumColumnWidth = true;
                        }

                        var table = fattable({
                            "container": element[0],
                            "model": buildModel(),
                            "nbRows": scope.numRows,
                            "rowHeight": scope.rowHeight,
                            "headerHeight": 0,
                            "painter": buildPainter(),
                            "columnWidths": columnWidths,
                            "enableDragMove": scope.enableDragMove !== false // defaults to true if not defined
                        });

                        if (attrs.fatDraggable !== undefined && typeof scope.fatDraggableOnDrop === 'function' && !scope.$$destroyed) {
                            FatDraggableService.setDraggable({
                                element: table.container,
                                onDrop: scope.fatDraggableOnDrop,
                                axis: 'y',
                                scrollBar: table.scroll,
                                classNamesToIgnore: ['icon-sort-by-attributes', 'sort-indication', 'pull-right']
                            })

                            // We have to set the fat-draggable__item class to an inner child because fattable re-use
                            // the same cell div for different items
                            for (let cellKey in table.cells) {
                                if (!table.cells.hasOwnProperty(cellKey)) continue;
                                let cellDiv = table.cells[cellKey];
                                let cellDivColumnHeader = cellDiv && cellDiv.children && cellDiv.children[0];
                                cellDivColumnHeader && cellDivColumnHeader.classList.add('fat-draggable__item');
                            }
                        }

                        if (isTouchDevice()) {
                            if (oldTable && typeof(scope.unsetTouchable) === "function") {
                                scope.unsetTouchable();
                            }
                            scope.unsetTouchable = FatTouchableService.setTouchable(scope, element, table);
                        }

                        if (oldTable) {
                            var y = oldTable.scroll.scrollTop;
                            var x = oldTable.scroll.scrollLeft;
                            table.scroll.setScrollXY(x,y);
                        }

                        oldTable = table;

                        if (scope.fTrackTable) scope.fTrackTable({table:oldTable});

                        if (scope.layoutMode=="row" && scope.listPadding) {
                            $(element).find('.fattable-viewport').css('padding-left', scope.listPadding);
                        }

                        if (scope.inForm) {
                            _element.find('[ng-model]').each((idx, el) => {
                                scope.inForm.$addControl(angular.element(el).controller('ngModel'));
                            });
                        }
                    }

                    scope.$on('moveScroll', (event, x, y) => {
                        oldTable.scroll.setScrollXY(oldTable.scroll.scrollLeft + x, oldTable.scroll.scrollTop + y);
                    });

                    scope.$watchCollection('fatRepeat', redraw);

                    scope.$on('redrawFatTable', redraw);
                    scope.$on('repaintFatTable', function () { debouncedRedraw(); }); //works better wrapped in a fnc!

                    if (scope.disableScrollTo === undefined) {
                        scope.$on('scrollToLine', function(e, lineNum) {
                            if (oldTable) {
                                let nbRowsVisible = oldTable.h / oldTable.rowHeight; // we need the float value
                                let firstVisibleRow = oldTable.scroll.scrollTop / oldTable.rowHeight; // we need the float value
                                let x = oldTable.scroll.scrollLeft;
                                if (lineNum == -1) {
                                    let y = oldTable.nbRows * oldTable.rowHeight;
                                    oldTable.scroll.setScrollXY(x, y);
                                } else if (lineNum <= firstVisibleRow) {
                                    let y = Math.max(lineNum, 0) * oldTable.rowHeight;
                                    oldTable.scroll.setScrollXY(x,y);
                                } else if (lineNum >= firstVisibleRow + nbRowsVisible - 1) {
                                    let y = (Math.min(lineNum, oldTable.nbRows) + 1) * oldTable.rowHeight - oldTable.h;
                                    oldTable.scroll.setScrollXY(x,y);
                                }
                            }
                        });
                    }

                    var debouncedRedraw = Debounce().withDelay(50,200).wrap(redraw);
                    $(window).on('resize', debouncedRedraw);

                    element.scope().$on("reflow", debouncedRedraw);

                    scope.$on("$destroy", function () {
                        if(oldTable) {
                            oldTable.cleanUp();
                            oldTable=null;
                        }
                        for(var i = 0 ; i < livingCells.length ; i++) {
                            cleanDOM(livingCells[i]);
                        }
                        livingCells = [];
                        $(window).off("resize", debouncedRedraw);
                        if(mutationObserver) {
                            mutationObserver.disconnect();
                        }
                    });


                };
            }
        };
    });

    app.directive('spinner', function() {
        return {
            template: '<div id="qa_spinner" class="spinnerContainer"></div>',
            replace: true,
            restrict: 'E',
            link: function(_scope, element, attrs) {
                let opts = {
                  lines: 6, // The number of lines to draw
                  length: 0, // The length of each line
                  width: 10, // The line thickness
                  radius: 10, // The radius of the inner circle
                  corners: 1, // Corner roundness (0..1)
                  rotate: 0, // The rotation offset
                  color: '#fff', // #rgb or #rrggbb
                  speed: attrs.speed || 1, // Rounds per second
                  trail: 60, // Afterglow percentage
                  shadow: false, // Whether to render a shadow
                  hwaccel: false, // Whether to use hardware acceleration
                  className: 'spinner', // The CSS class to assign to the spinner
                  zIndex: 2e9, // The z-index (defaults to 2000000000)
                  top: 'auto', // Top position relative to parent in px
                  left: 'auto' // Left position relative to parent in px
                };
                new Spinner(opts).spin(element[0]);
          }
       };
    });

    app.directive('formTemplate', function(){
        return {
            templateUrl: '/templates/form-template.html',
            replace: true,
            restrict: 'E',
            link: function(scope, element, attrs){
                function initializeDefaultValues() {
                    if (scope.formDefinition) {
                        for (const formDefinitionElement of scope.formDefinition) {
                            if (!formDefinitionElement.params) {
                                continue;
                            }
                            if (!scope.model.hasOwnProperty(formDefinitionElement.name)) {
                                scope.model[formDefinitionElement.name] = {};
                            }
                            for (const param of formDefinitionElement.params) {
                                if (!scope.model[formDefinitionElement.name].hasOwnProperty(param.name) && param.defaultValue) {
                                    scope.model[formDefinitionElement.name][param.name] = param.defaultValue;
                                }
                            }
                        }
                    }
                }

                if (attrs.monitor) {
                    scope.model = {};
                    scope.$watch(attrs.monitor, () => {
                        const model = scope.$eval(attrs.model);
                        if (model && Object.keys(model).length > 0) {
                            scope.model = model;
                        }
                        scope.formDefinition = scope.$eval(attrs.formDefinition);

                        initializeDefaultValues();
                    });
                } else {
                    scope.model = scope.$eval(attrs.model);
                    scope.formDefinition = scope.$eval(attrs.formDefinition);
                }
            }
        };
    });

    app.directive('formTemplateElement', function(){
        return {
            templateUrl: '/templates/form-template-element.html',
            replace: true,
            restrict: 'EA',
            scope: {
                model: '=',
                field: '=',
                onCoreParamsChanged: '&',
                disabled: '=',
                label: '@',
                validator: '<?',
                errorMessage: '@?'
            },
            link: () => {
                // noop
            },
            controller: function($scope) {
                $scope.label = $scope.label || $scope.field.label || $scope.field.name;
                $scope.getInputType = function(value) {
                    if ($scope.field.maskIf && typeof $scope.field.maskIf === 'string') {
                        $scope.field.maskIf = new Function('value',
                            'return (' + $scope.field.maskIf + ')(value)');
                    }
                    return $scope.field.maskIf && $scope.field.maskIf(value) ? "password" : "text";
                };
                // If not supplied, the validator should always return true
                if (!$scope.validator) {
                    $scope.validator = () => true;
                }
            }
        };
    });

    app.directive('forceInteger', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, attrs, ngModel) {
                function fromUser(text) {
                    return text;
                }
                function toUser(text, ) {
                    if(attrs.forceInteger == 'false'){
                        return text;
                    }
                    return parseInt(text || 0, 10);
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);

            }
        };
    });

    app.directive('forceIntegerAllowEmpty', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, attrs, ngModel) {
                function fromUser(text) {
                    if (text === '') {
                        return null; // Allow empty input
                    }
                    return text;
                }

                function toUser(text) {
                    if (text === '' || text == null) {
                        return ''; // Keep the empty input
                    }
                    return parseInt(text, 10);
                }

                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });

    app.directive('convertSpecialChar', function() {
         return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, _attr, ngModel) {
                function fromUser(text) {
                    /* global convertSpecialChars */
                    const result = convertSpecialChars(text);
                    ngModel.$setValidity('singleChar', 1 === result.length);
                    return result;
                }
                function toUser(text) {
                    // For the moment we do not convert the unicode character into its ASCII representation as it can
                    // be displayed as-is in text inputs.
                    return text == null ? null : text.replace('\t', '\\t');
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });

    app.directive('convertPercentage', function() {
         return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, _attr, ngModel) {
                // calling round to avoid long decimal tail after floating point math operations e.g. 0.072*100=7.199999999999999
                function round(n){
                    return Math.round(n * 10 ** 12) / 10 ** 12;
                }

                function fromUser(text) {
                    return text == null ? null : round(parseFloat(text) / 100);
                }
                function toUser(text) {
                    return text == null ? null : round(parseFloat(text) * 100);
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });
    app.directive('forceDouble', function() {
        return {
            restrict: 'A',
            require: 'ngModel',
            link: function(_scope, _element, _attr, ngModel) {
                function fromUser(text) {
                    return text;
                }

                function toUser(text) {
                    // Don't silently replace empty by zero !
                    if(text!=null && text!=undefined && text!=='') {
                        return parseFloat(text);
                    } else return '';
                }
                ngModel.$parsers.push(fromUser);
                ngModel.$formatters.push(toUser);
            }
        };
    });

    // Droparea is a component that handle file drops on it and callbacks the method defined on its drop parameter
    app.directive("droparea", function($filter, $timeout){
        return {
            restrict: 'E',
            template: '<div class="droparea h100"  ng-class="{candrop: candrop}">'+
                '<form class="upload h100" ></form>'+
                '<div ng-transclude class="nested-template-container h100" ></div>' +
            '</div>',
            replace: true,
            transclude: true,
            scope: {
                drop: '&',
                validate: '&',
                //paramaters used by droparea directive to expose to its parent a candrop flag
                isDroppable: '=?',
                customId: '@?',
            },
            link: function(scope, element, attrs){
                scope.multiple = 'multiple' in attrs;
                scope.noUploadOnClick = 'noUploadOnClick' in attrs;
                scope.candrop = false;
                scope.$watch('candrop', () => {
                    scope.isDroppable = scope.candrop;
                })

                // input fallback
                function addFileManually() {
                    var evt = document.createEvent("MouseEvents");
                    evt.initEvent('click', true, true );
                    input[0].dispatchEvent(evt);
                }

                if (!scope.noUploadOnClick)  {
                    element.click(function(e){
                        e.stopPropagation();
                        addFileManually();
                    });
                }

                element.on('click', 'form.upload input', function(e){
                    e.stopPropagation();
                }).on('change', 'form.upload input', function() {
                    const files = this.files;
                    scope.$apply(function() {
                        scope.drop({'files': files});
                    });
                    createInput();
                });

                var input;
                const id = scope.customId || 'qa_upload_dataset_input-files';
                function createInput(){
                    element.find('form.upload').find('input').remove();
                    input = $(`<input type="file" name="file" id="${id}" multiple />`);
                    element.find('form.upload').append(input);
                }
                createInput();

                // drop file
                function applyDragEnterLeave(e) {
                    e.stopPropagation();
                    e.preventDefault();
                    scope.candrop = false;
                    scope.$apply();
                }
                function cancelEnterLeaveTimeout() {
                    if (scope.enterLeaveTimeout) {
                        $timeout.cancel(scope.enterLeaveTimeout);
                    }
                }
                function dragEnterLeave(e) {
                    cancelEnterLeaveTimeout();
                    //debouncing applyDragEnterLeave to prevent flickering when hovering element's children
                    scope.enterLeaveTimeout = $timeout(function() {
                        applyDragEnterLeave(e);
                    }, 100);
                }
                element.bind("dragenter", dragEnterLeave);
                element.bind("dragleave", dragEnterLeave);
                element.bind("dragover", function(e) {
                    cancelEnterLeaveTimeout();
                    e.stopPropagation();
                    e.preventDefault();
                    scope.$apply(function(){
                        var evt = e.originalEvent;
                        if (evt.dataTransfer &&
                            evt.dataTransfer.types &&
                            (
                                (evt.dataTransfer.types.indexOf && evt.dataTransfer.types.indexOf('Files') >= 0) ||
                                (evt.dataTransfer.types.contains && evt.dataTransfer.types.contains('Files'))
                            )
                        ) {
                            // feedback
                            scope.candrop = true;
                            var af = evt.dataTransfer.effectAllowed;
                            evt.dataTransfer.dropEffect = ('move' == af || 'linkMove' == af) ? 'move' : 'copy';
                        }
                    });
                });
                element.bind("drop", function(e) {
                    e.stopPropagation();
                    e.preventDefault();
                    scope.$apply(function(){
                        var evt = e.originalEvent;
                        if (evt.dataTransfer &&
                            evt.dataTransfer.types &&
                            (
                                (evt.dataTransfer.types.indexOf && evt.dataTransfer.types.indexOf('Files') >= 0) ||
                                (evt.dataTransfer.types.contains && evt.dataTransfer.types.contains('Files'))
                            ) &&
                            (scope.multiple || evt.dataTransfer.files.length == 1)
                        ){
                            scope.drop({'files': evt.dataTransfer.files});
                            scope.candrop = false;
                        }
                    });
                });
            }
        };
    });

    app.directive('commaSeparatedView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    if (value == null || value.length == 0) return [];
                    return value.split(",");
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return "";
                    return value.join(",");
                });
            }
        };
    });
    app.directive('jsonArrayView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    ngModel.$setValidity('json', true);
                    if (value == null || value.length == 0) return [];
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                         ngModel.$setValidity('json', false);
                         return null;
                    }
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return null;
                    return JSON.stringify(value);
                });
            }
        };
    });
    app.directive('jsonArrayPrettyView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    ngModel.$setValidity('json', true);
                    if (value == null) return [];
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                         ngModel.$setValidity('json', false);
                         return null;
                    }
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return null;
                    return JSON.stringify(value, undefined, 3);
                });
            }
        };
    });

    app.directive('jsonObjectPrettyView', function (Logger){
            return {
                require: 'ngModel',
                link: function(scope, elem, attrs, ngModel) {
                    var el = elem[0];
                    //For DOM -> model transformations
                    ngModel.$parsers.push(function(value) {
                        ngModel.dkuJSONError = null;
                        ngModel.$setValidity('json', true);
                        if (value == null || value.length == 0) return null;
                        try {
                            return JSON.parse(value);
                        } catch (e) {
                            ngModel.$setValidity('json', false);
                            Logger.info("Error while parsing JSON: ", value, e);
                            ngModel.dkuJSONError = e.toString();
                            if ('keepOldIfInvalid' in attrs) {
                                return ngModel.$modelValue;
                            } else {
                                return null;
                            }
                        }
                    });

                    //For model -> DOM transformation
                    ngModel.$formatters.push(function(value) {
                        if (value == undefined) return null;
                        var prevSelStart = el.selectionStart;
                        var prevSelEnd = el.selectionEnd;
                        var prevScroll = el.scrollTop;
                        if(el == document.activeElement) {
                            setTimeout(function() {
                                el.setSelectionRange(prevSelStart,prevSelEnd);
                                el.scrollTop = prevScroll;
                            },0);
                        }
                        return JSON.stringify(value,undefined,3);
                    });

                    if (attrs.deepUpdate) {
                        scope.$watch(attrs.ngModel, () => {
                            try {
                                var formatters = ngModel.$formatters, idx = formatters.length;
                                var viewValue = ngModel.$modelValue;
                                while (idx--) {
                                    viewValue = formatters[idx](viewValue);
                                }
                                if (viewValue != null) {
                                    ngModel.$viewValue = viewValue;
                                    ngModel.$render();
                                }
                            } catch (e) {
                                Logger.info("JSON is invalid, not rendering ...")
                            }
                        }, true);
                    }
                }
            };
    });

    app.directive('jsonObjectView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    ngModel.$setValidity('json', true);
                    if (value == null || value.length == 0) return null;
                    try {
                        return JSON.parse(value);
                    } catch (e) {
                         ngModel.$setValidity('json', false);
                         return null;
                    }
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return null;
                    return JSON.stringify(value);
                });
            }
        };
    });

    app.directive('commaSeparatedIntegerView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    if (value == null) return [];
                    var ret = value.split(",").map(function(x) { return parseInt(x, 10); }).filter(function(x) { return !isNaN(x);});
                    return ret;
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return "";
                    return value.join(",");
                });
            }
        };
    });

    app.directive('commaSeparatedFloatView', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                //For DOM -> model transformations
                ngModel.$parsers.push(function(value) {
                    if (value == null) return [];
                    var ret = value.split(",").map(function(x) { return parseFloat(x, 10); }).filter(function(x) { return !isNaN(x);});
                    return ret;
                });

                //For model -> DOM transformation
                ngModel.$formatters.push(function(value) {
                    if (value == undefined) return "";
                    return value.join(",");
                });
            }
        };
    });


    app.directive('customValidation', function (){
        return {
            require: 'ngModel',
            link: function(scope, elem, attrs, ngModel) {
                function apply_validation(value) {
                    ngModel.$setValidity('customValidation', true);
                    let cv = scope.$eval(attrs.customValidation);
                    var valid = cv && cv(value);
                    ngModel.$setValidity('customValidation', valid);
                    return value;
                }

                //For DOM -> model validation
                ngModel.$parsers.push(apply_validation);

                //For model -> DOM validation
                ngModel.$formatters.push(function(value) {
                    apply_validation();
                    return value;
                });
            }
        };
    });

    app.directive('fixedPanes', function($timeout,$rootScope){
        return {
            restrict: 'A',
            link: function(scope, element, attrs){
                scope.showLeftPane = scope.$eval(attrs.showLeftPane) || false;
                scope.showRightPane = scope.$eval(attrs.showRightPane)  || false;

                scope.setShowLeftPane = function(showLeftPane) {
                    if (scope.showLeftPane != showLeftPane) {
                        scope.showLeftPane = showLeftPane;
                        $timeout(function(){
                            scope.$broadcast('resizePane');
                            $rootScope.$broadcast('reflow');
                        }, 250);

                    }
                }
                scope.openLeftPane = function() {
                    scope.setShowLeftPane(true);
                }
                scope.closeLeftPane = function() {
                    scope.setShowLeftPane(false);
                }
                scope.toggleLeftPane = function(){
                    scope.setShowLeftPane(!scope.showLeftPane);
                };


                scope.setShowRightPane = function(showRightPane) {
                    if (scope.showRightPane != showRightPane) {
                        scope.showRightPane = showRightPane;
                        $timeout(function(){
                            scope.$broadcast('resizePane');
                            $rootScope.$broadcast('reflow');
                        }, 250);
                    }
                }
                scope.openRightPane = function() {
                    scope.setShowRightPane(true);
                }
                scope.closeRightPane = function() {
                    scope.setShowRightPane(false);
                }
                scope.toggleRightPane = function(){
                    scope.setShowRightPane(!scope.showRightPane);
                };
            }
        };
    });

    app.directive('watchScroll', function() {
       return {
           restrict : 'A',
            link: function(_scope, element){
                $(element).addClass("watch-scroll");
                function scrollStart() {
                    $(element).addClass("scrolling");
                }
                function scrollEnd() {
                    $(element).removeClass("scrolling");
                }
                var scrolling = false;
                $(element).scroll(function() {
                    if (!scrolling) scrollStart();
                    clearTimeout($.data(this, 'scrollTimer'));
                    $.data(this, 'scrollTimer', setTimeout(function() {
                        scrollEnd();
                    }, 200));
                });
            }
        };
    });

    app.directive('tabs', function($filter, $location, $compile, $timeout, Logger) {
        return {
            restrict: 'E',
            scope: true,
            // replace: true,
            myTemplateNonScrollable: '<div class="tabbable">' +
                '<ul class="nav nav-tabs">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }}" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab">' +
                        '<span class="title"><i ng-show="pane.icon" ng-class="getIconClass(pane.icon)"></i><span fw500-width> {{ pane.title }}</span></span>'+
                        '<br ng-if="pane.subtitle"/><span class="subtitle" ng-if="pane.subtitle" ng-bind-html="pane.subtitle"></span>'+
                        '</a>' +
                    '</li>' +
                '</ul>' +
                '<div class="tab-content" ></div>' +
            '</div>',
            myTemplateScrollable: '<div class="tabbable">' +
                '<div class="scroller scroller-left"><i class="icon-chevron-left"></i></div>'+
                '<div class="scroller scroller-right"><i class="icon-chevron-right"></i></div>'+
                '<div class="tabs-scroll-wrapper">'+
                '<ul class="nav nav-tabs tabs-scroll-zone">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }}" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab"><i ng-show="pane.icon" class="icon-{{ pane.icon }}"></i> {{ pane.title }}</a>' +
                    '</li>' +
                '</ul></div>' +
                '<div class="tab-content" ></div>' +
            '</div>',
            myTemplateNewStyleScrollable: '<div class="tabbable">' +
                '<div class="scroller scroller-left"><i class="icon-chevron-left"></i></div>'+
                '<div class="scroller scroller-right"><i class="icon-chevron-right"></i></div>'+
                '<div class="tabs-scroll-wrapper">'+
                '<ul class="column-header-tabs tabs-scroll-zone">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }} tab" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<span class="title" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab"><i ng-show="pane.icon" class="icon-{{ pane.icon }}"></i> {{ pane.title }}</span>' +
                    '</li>' +
                '</ul></div>' +
                '<div class="tab-content" ></div>' +
            '</div>',
             myTemplateNewStyle: '<div class="">' +
                '<ul class="column-header-tabs" style="margin: 0">' +
                    '<li ng-repeat="pane in panes|filter:{visible:true}" class="{{ pane.position }} tab" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                        '<span class="title" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab"><i ng-show="pane.icon" class="icon-{{ pane.icon }}"></i> {{ pane.title }}</span>' +
                    '</li>' +
                '</ul>' +
                '<div class="tab-content" ></div>' +
            '</div>',
            myTemplateNonScrollableWithNewFeature: '<div class="tabbable">' +
                '<ul class="nav nav-tabs">' +
                    '<li ng-repeat="pane in panes|filter:{visible: true}" class="{{ pane.position }}" ng-class="{active:pane.selected}" style="{{ paneHeaderStyle }}">' +
                    	'<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab" ng-if="pane.newInVersionFeature !== undefined">' +
                            '<ng2-new-in-version-popover ' +
                                'class="tabs-new-in-version-popover" ' +
                                'target-version="{{pane.newInVersionTargetVersion}}" ' +
                                'popover-id="{{pane.newInVersionPopoverId}}" ' +
                                'feature-name="{{pane.newInVersionFeature}}" ' +
                                'display-mode="{{pane.newInVersionDisplayMode}}">' +
                                '<div popoverContent>' +
                                    '<p>{{pane.newInVersionContent}}</p>' +
                                '</div>' +
                                '<div popoverTarget>' +
                                    '<span class="title"><i ng-show="pane.icon" ng-class="getIconClass(pane.icon)"></i><span fw500-width> {{ pane.title }}</span></span>'+
                                    '<br ng-if="pane.subtitle"/><span class="subtitle" ng-if="pane.subtitle" ng-bind-html="pane.subtitle"></span>'+
                                '</div>' +
                            '</ng2-new-in-version-popover>' +
                        '</a>' +

                        '<a href="" ng-click="select(pane, noHashUpdate)" class="qa_generic_widget-tab" ng-if="pane.newInVersionFeature === undefined">' +
                        '<span class="title"><i ng-show="pane.icon" ng-class="getIconClass(pane.icon)"></i><span fw500-width> {{ pane.title }}</span></span>'+
                        '<br ng-if="pane.subtitle"/><span class="subtitle" ng-if="pane.subtitle" ng-bind-html="pane.subtitle"></span>'+
                        '</a>' +
                    '</li>' +
                '</ul>' +
                '</ul>' +
                '<div class="tab-content" ></div>' +
            '</div>',
                
            compile: function(tElement, attrs){
                // get the panes of the tabs
                var originalContent = $('<div></div>').html(tElement.contents()).contents();
                var el;
                if (attrs.hasOwnProperty("newFeature")) {
                    el = $(this.myTemplateNonScrollableWithNewFeature);
                } else if (attrs.newStyle && attrs.scrollable) {
                    el = $(this.myTemplateNewStyleScrollable);
                } else if (attrs.newStyle) {
                    el = $(this.myTemplateNewStyle);
                } else if (attrs.scrollable) {
                    el = $(this.myTemplateScrollable);
                } else  {
                    el = $(this.myTemplateNonScrollable);
                }
                if (tElement.hasClass('vertical-flex')) {
                    el.addClass('vertical-flex h100');
                    el.children(':not(.tab-content)').addClass('noflex');
                    el.children('.tab-content').addClass('flex');
                }
                tElement.replaceWith(el);
                el.find('.tab-content').append(originalContent);
                return this.link;
            },
            controller: function($scope, $element, $rootScope) {
                var panes = $scope.panes = [];
                $scope.select = function(pane, force) {
                    $scope.$emit("opalsCurrentTab", pane.slug);
                    if (!pane.selected){
                        angular.forEach(panes, function(p) {
                            p.selected = false;
                        });
                        pane.selected = true;
                        $timeout(function() {
                            pane.displayed = true;
                        });
                        if ($scope.onSelect) {
                            $scope.onSelect(pane);
                        }
                        $scope.$emit('paneSelected', pane);
                        $scope.$broadcast('paneSelected', pane);
                        $rootScope.$broadcast("reflow");

                        if (!force){
                            // TEMPORARY TEMPORARY :
                            // Disable updating of the location hash, because
                            //  it breaks ui-router's interaction with browser history.
                            //   - If you are on state A with no hash, transitionTo(B), then back takes you back to A
                            //   - If you are on state A with no hash, then on state B with hash, transitionTo(C)
                            //      - will propagate the hash so you will actually be on C#hash, which we don't want
                            //      - back button will take you back to A, the B#hash state has disappeared from browser
                            //        history
                            $location.hash(pane.slug, true).replace(true);
                        }
                    }
                };

                this.select = $scope.select;
                this.addPane = function(pane) {
                    panes.push(pane);
                    if (panes.length == 1) {
                        $scope.select(pane, true);
                    }
                };

                $scope.$on('tabSelect', function(e, slug){
                    var pane = $filter('filter')(panes, {'slug':slug});
                    if(pane && pane.length){
                       $scope.select(pane[0], $scope.noHashUpdate);
                    } else {
                        Logger.warn("Failed to select a pane for slug ", slug, " amongst", panes, " filtered", pane);
                    }
                });
                this.verticalFlex = $element.hasClass('vertical-flex');

                $scope.getIconClass = function(icon) {
                    if (icon && icon.startsWith('dku-icon')) {
                        return icon;
                    } else {
                        return 'icon-' + icon;
                    }
                };
            },
            link : ($scope, element, attrs) => {
                $scope.noHashUpdate = "noHashUpdate" in attrs;
                if (attrs.scrollable) {

                    var totalWidth = function(){
                        var itemsWidth = 0;
                        $('.tabs-scroll-zone li', element).each(function(){
                            var itemWidth = $(this).outerWidth();
                            itemsWidth+=itemWidth;
                        });
                        return itemsWidth;
                    };
                    var scrollBarWidths = 46; /* 2x23 */

                    var hiddenWidth = function(){
                        return (
                            ($('.tabs-scroll-wrapper', element).outerWidth()) -
                            totalWidth()-getLeftPosi())
                            -
                            scrollBarWidths;
                    };
                    var getLeftPosi = function(){
                        return $('.tabs-scroll-zone').position().left;
                    };

                    $('.scroller-right', element).click(function() {
                        $('.scroller-left', element).show();
                        $('.scroller-right', element).hide();
                        $('.tabs-scroll-zone', element).animate({left:"+="+hiddenWidth()+"px"});
                    });

                    $(".scroller-left", element).click(function() {
                        $('.scroller-right', element).show();
                        $('.scroller-left', element).hide();
                        $('.tabs-scroll-zone', element).animate({left:"-="+getLeftPosi()+"px"});
                    });
                    $timeout(function(){
                        if (($('.tabs-scroll-wrapper', element).outerWidth()) < totalWidth()) {
                            $('.scroller-right').show();
                        }
                    }, 0);
                }
                if (attrs.paneHeaderStyle) {
                    $scope.paneHeaderStyle = attrs.paneHeaderStyle;
                } else {
                    $scope.paneHeaderStyle = '';
                }
            }
        };
    });

    app.directive('pane', function($filter, $location, $timeout, $compile, $rootScope, $stateParams) {
        var paneTemplate = $compile('<div class="tab-pane" ng-class="{active: struct.selected}"></div>');
        return {
            require: '^tabs',
            restrict: 'E',
            terminal: true,
            scope: true,
            compile: function(tElement){
                // get the content of the pane
                var transcludeFunction = $compile(tElement.contents());

                return function(scope, element, attrs, tabsCtrl){

                    // replace the pane by the paneTemplate
                    paneTemplate(scope, function(clone){
                        element.replaceWith(clone);
                        element = clone;
                    });

                    // append the content of the pane
                    transcludeFunction(scope, function(clone){
                        element.append(clone);
                    });

                    scope.struct = {
                        title: attrs.title,
                        tabClass: attrs.tabClass,
                        subtitle: scope.$eval(attrs.subtitle),
                        slug: attrs.slug || $filter('slugify')(attrs.title),
                        icon: attrs.icon,
                        visible: angular.isUndefined(attrs.visiblePane)?true:scope.$eval(attrs.visiblePane),
                        position: attrs.position,
                        newInVersionFeature: attrs.newInVersionFeature,
                        newInVersionTargetVersion: attrs.newInVersionTargetVersion,
                        newInVersionPopoverId: attrs.newInVersionPopoverId,
                        newInVersionDisplayMode: attrs.newInVersionDisplayMode,
                        newInVersionContent: attrs.newInVersionContent,
                    };

                    element.addClass("tab-" + scope.struct.slug);
                    if (scope.struct.tabClass) {
                        element.addClass(scope.struct.tabClass);
                    }
                    if (tabsCtrl.verticalFlex) { element.addClass('fh'); }

                    attrs.$observe('title', function(val){
                        // If the title attribute is modified
                        scope.struct.title = val;
                    });
                    scope.$watch(attrs.subtitle, (nv) => {
                        // If the title attribute is modified
                        scope.struct.subtitle = nv;
                    });

                    // Removing the title from the element itself, to prevent a tooltip when hovering anywhere over
                    // the content.
                    element.removeAttr('title');
                    // having a pb when combined with ng-repeat
                    $timeout(function(){element.removeAttr('title');}, 10);

                    // register itself
                    tabsCtrl.addPane(scope.struct);
                    if (($location.hash() === scope.struct.slug || $stateParams.tabSelect === scope.struct.slug) && scope.struct.visible){
                        tabsCtrl.select(scope.struct);
                    }

                    attrs.$observe('visiblePane', function(value){
                        scope.struct.visible = angular.isUndefined(value)?true:value=="true";
                    });

                    scope.$watch('struct.selected', (nv) => {
                        if (nv){
                            if (attrs.noResizeHack == null) {
                                // ugly hack, this will trigger a window resize event, thus refreshing flowchart layout
                                // and CodeMirror-alikes
                                window.dispatchEvent(new Event('resize'));
                            }
                            $rootScope.$broadcast("reflow");
                        }
                    });
                };
            },
        };
    });

    app.directive("detectIframeClicks", function() {
        return {
            scope: false,
            restrict: "A",
            link: function(_scope, element){
                var overIFrame = false;
                element.mouseenter(function() {
                    overIFrame = true;
                });
                element.mouseleave(function() {
                    overIFrame = false;
                });
                $(window).blur(function() {
                    if (overIFrame) {
                        $(document).trigger("click");
                    }
                });
            }
        }
    })

    app.directive('sortTable', function($rootScope){
        return {
            scope: true,
            controller: function($scope, $attrs) {
                this.setSort = function(col) {
                    if($scope.sortColumn){
                        $scope.cols[$scope.sortColumn].removeClass('sort-descending').removeClass('sort-ascending');
                    }
                    if ($scope.sortColumn === col) {
                        $scope.sortDescending = !$scope.sortDescending;
                    } else {
                        $scope.sortColumn = col;
                        $scope.sortDescending = false;
                    }
                    this.refresh();
                };

                this.refresh = function(){
                    if($scope.cols[$scope.sortColumn]){
                        if ($scope.sortDescending) {
                            $scope.cols[$scope.sortColumn].addClass("sort-descending");
                        } else {
                            $scope.cols[$scope.sortColumn].addClass("sort-ascending");
                        }
                        if(! $rootScope.$$phase) $scope.$apply();
                    }
                };

                $scope.sortColumn = $attrs.sortColumn;
                $scope.sortDescending = $scope.$eval($attrs.sortDescending) || false;
                if($attrs.sortTable){
                    if ($attrs.sortTable[0] == "-") {
                        $scope.sortDescending = true;
                        $scope.sortColumn = $attrs.sortTable.substring(1);
                    } else {
                        $scope.sortColumn = $attrs.sortTable;
                    }
                }

                $scope.cols = {};
                this.addCol = function(col, element){
                    $scope.cols[col] = element;
                    element.addClass("sortable");
                    if(angular.isUndefined($scope.sortColumn)){
                        this.setSort(col);
                    } else {
                        this.refresh();
                    }
                };
            }
        };
    });

    /* This is an alternative version of sortTable, featuring a two way binding of sortColumn and sortDescending using
        standard angular features rather than playing with attributes.

        Using it, one can easily pass initial sortColumn and sortDescending as variables, and get back all modifications
        performed by the user.

        Note: when this version is used, one should not use sortColumn and sortDescending as parameters of the orderBy angular pipe.
        Variables passed as parameters should be used.

        Example:

            <table sort-table-dyn
                sort-column="myScope.mySortColumn" sort-descending="myScope.mySortDescending">
                    ...
                <tbody>
                    <tr ng-repeat="item in selection.filteredObjects | orderBy:myScope.mySortColumn:myScope.mySortDescending">

        It is similar to sortTable. However, adding double binding to the former implementation seemed would lead to a code
        a lot less understandable, especially if we want to avoid a refactoring of all current sort-table usages.

        Note: when using a complex sort expression such as a getter function or an array of sort expression, ensure you add the `use-complex-sort-expression='true'` to your component.
    */
    app.directive('sortTableDyn', function($rootScope){
        return {
            scope:{
                sortColumn: '=',
                sortDescending: '=',
                useComplexSortExpression: '@'
            },
            controller: function($scope) {
                this.setSort = function(col) {
                    if ($scope.useComplexSortExpression) {
                        if ($scope.sortColumn === col) {
                            $scope.sortDescending = !$scope.sortDescending;
                        } else {
                            $scope.sortColumn = col;
                            $scope.sortDescending = false;
                        }
                        $scope.currentSortColumnName = $scope.sortColumn;
                        $scope.currentSortDescending = $scope.sortDescending;
                    } else {
                        // Initialize the local sort direction with the one from the component.
                        if ($scope.currentSortDescending === undefined) {
                            $scope.currentSortDescending = $scope.sortDescending;
                        }

                        // If we are trying to sort on the same column that we are already sorted with,
                        // inverse the order using `-column` notation instead of reversing the entire order
                        if ($scope.currentSortColumnName === col) {
                            $scope.currentSortDescending = !$scope.currentSortDescending;
                            if ($scope.currentSortDescending) {
                                $scope.sortColumn = `-${col}`;
                            } else {
                                $scope.sortColumn = col;
                            }
                        } else {
                            $scope.sortColumn = col;
                            $scope.currentSortDescending = false;
                        }

                        $scope.currentSortColumnName = col;
                        $scope.sortDescending = false;
                    }
                    this.refresh();
                };

                this.refresh = function(){
                    Object.values($scope.cols).forEach(e => e.removeClass('sort-descending').removeClass('sort-ascending'));
                    if($scope.cols[$scope.currentSortColumnName]){
                        if ($scope.currentSortDescending) {
                            $scope.cols[$scope.currentSortColumnName].addClass("sort-descending");
                        } else {
                            $scope.cols[$scope.currentSortColumnName].addClass("sort-ascending");
                        }
                        if(! $rootScope.$$phase) $scope.$apply();
                    }
                };


                $scope.cols = {};
                this.addCol = function(col, element){
                    $scope.cols[col] = element;
                    element.addClass("sortable");
                    this.refresh();
                };
            }
        };
    });

    app.directive('sortCol', function(){
        return {
            scope: true,
            require: '^sortTable',
            link: function(scope, element, attrs, sortTableCtrl){
                sortTableCtrl.addCol(attrs.sortCol, element);
                element.on('click', function(){
                    sortTableCtrl.setSort(attrs.sortCol);
                });
            }
        };
    });

    app.directive('sortColDyn', function(){
        return {
            scope: true,
            require: '^sortTableDyn',
            link: function(scope, element, attrs, sortTableCtrl){
                sortTableCtrl.addCol(attrs.sortColDyn, element);
                element.on('click', function(){
                    sortTableCtrl.setSort(attrs.sortColDyn);
                });
            }
        };
    });

    app.directive('daterangepicker', function($rootScope){
        return {
            restrict: 'A',
            template : ' <div class="input-append" style="margin-bottom:0px;"><input type="text" style="opacity:0;'
                 +'position:absolute;top:-3000px;left:-3000px"/>'
                 +'<input type="text" class="theInput" ng-disabled="!!disabled"/>',
            scope: {
                startDate: '=',
                endDate: '=',
                opensDirection:'@',
                opens: '@',
                format: '@',
                timePickerIncrement : '@',
                presetsToEndOfDay : '=',
                fieldWidth : '@',
                singleDatePicker: '=',
                onChange: "=?",
                disabled: '<?'
            },
            replace : true,
            link: function(scope, element, attrs){

                var input = element.find('.theInput');
                var picker = undefined;

                if (scope.fieldWidth) {
                    input.width(scope.fieldWidth);
                }

                element.find('input').keydown(function(e){
                    if(e.keyCode==13 && picker) {
                        picker.hide();
                    }
                });

                // Create the date picker if not already done, and only if the format is set
                function init() {
                    if(!picker && scope.format) {
                        var endOfDayDelta = scope.presetsToEndOfDay ? 1 : 0;
                        picker = input.daterangepicker({
                              format:scope.format,
                              timePickerIncrement: scope.timePickerIncrement ? parseInt(scope.timePickerIncrement) : 60,
                              timePicker: scope.format.indexOf('HH')!=-1,
                              opens: attrs.opens || 'right',
                              timePicker12Hour:false,
                              autoApply:true,
                              separator : ' / ',
                              singleDatePicker: !!scope.singleDatePicker,
                              ranges: {
                                 'Today': [moment(), moment().add('days', endOfDayDelta)],
                                 'Yesterday': [moment().subtract('days', 1), moment().subtract('days', 1 - endOfDayDelta)],
                                 'Last 7 Days': [moment().subtract('days', 6), moment().add('days', endOfDayDelta)],
                                 'Last 30 Days': [moment().subtract('days', 29), moment().add('days', endOfDayDelta)],
                                 'This Month': [moment().startOf('month'), moment().endOf('month')],
                                 'Last Month': [moment().subtract('month', 1).startOf('month'), moment().subtract('month', 1).endOf('month')]
                               },
                               locale: {firstDay: 1},
                               opensDirection: attrs.opensDirection || 'down'
                        },changeDate).data('daterangepicker');

                        picker.element.on('hide.daterangepicker', function (ev, picker) {
                            if (picker.element.val().length === 0) {
                                scope.startDate = null;
                                scope.endDate = null;
                                if(!$rootScope.$$phase) {
                                    $rootScope.$digest();
                                }
                            }
                        });

                        // open upwards if no room below and vice versa
                        if (attrs.opensDirection === 'auto') {
                            picker.element.on('show.daterangepicker', function(ev, picker) {
                                const currentDirection = picker.opensDirection;
                                let newDirection = 'down'

                                if (picker.element.offset().top + picker.element.outerHeight() + picker.container.outerHeight() > $(window).height()) {
                                    newDirection = 'up';
                                }

                                // need to close and reopen for change to take effect
                                if (currentDirection !== newDirection) {
                                    picker.opensDirection = newDirection;
                                    picker.hide();
                                    picker.show();
                                }
                            });
                        }
                    }
                    if(picker)
                        return true;

                    return false;
                }

                var insideWatch = false;
                var insideUserCallback = false;

                // Update the scope from the date picker state
                function changeDate() {

                    if(!init() || insideWatch) return;
                    try {
                        insideUserCallback = true;
                        picker.updateFromControl();
                        if(!scope.format) return;
                        scope.startDate = picker.startDate.format(scope.format);
                        scope.endDate = picker.endDate.format(scope.format);
                        if(!$rootScope.$$phase) {
                            $rootScope.$digest();
                        }
                        if (scope.onChange) {
                            scope.onChange();
                        }
                    } finally {
                        insideUserCallback = false;
                    }
                }

                // Update date picker state from scope
                scope.$watch('[startDate,endDate]', (nv) => {
                    if(!init() || insideUserCallback) return;
                    try {
                        insideWatch = true;
                        if (!nv[0] && !nv[1]){
                            picker.element.val("");
                            return;
                        }
                        if(scope.startDate) {
                            picker.setStartDate(moment(scope.startDate,scope.format));
                        }
                        if(scope.endDate) {
                            picker.setEndDate(moment(scope.endDate,scope.format));
                        }
                    } finally {
                        insideWatch = false;
                    }
                },true);

                scope.$watch('format',function(nv, ov) {
                    if(!picker) {
                        init();
                    } else if (nv != ov) {
                        picker.format = nv;
                        picker.timePicker = nv.indexOf('HH')!=-1;
                        picker.timePickerIncrement= scope.timePickerIncrement ? parseInt(scope.timePickerIncrement) : 60;
                        picker.updateInputText();
                        changeDate();
                        picker.updateCalendars();
                    }

                });

                scope.$on('$destroy',function() {
                    if(picker) {
                        picker.remove();
                        picker = undefined;
                    }
                })
            }
        };
    });


    /*
     *   Very similar to dkuHelpPopover, but the template can be inlined
     *   Usage :
     *
     *   <button class="btn btn-small" dku-inline-popover>
     *        <label>
     *            <span class="icon-question">&nbsp;</span>
     *                      Button text
     *        </label>
     *        <content title="Help me">
     *            <h2>Introduction</h2>
     *            <p>Blablabla</p>
     *        </content>
     *   </button>
     *
     */
    app.directive('dkuInlinePopover',function($timeout, $interpolate, $compile) {
        return {
            restrict : 'A',
            scope : true,
            transclude:true,
            template:'',

            compile:function(_element,_attrs,transclude) {
                return function(scope,element,attrs) {
                    let destroyPopover;
                    transclude(scope.$new(), function(clone) {

                        var contentFilter = clone.filter("content");
                        var buttonText = clone.filter('label').contents();
                        var popoverContent = contentFilter.contents();
                        var title = contentFilter.attr("title") ? $interpolate(contentFilter.attr("title"))(scope) : null;
                        let popoverRef = null;

                        // I'VE NO FUCKING IDEA OF WHY IT WORKS
                        // timeout is the universal fix :D
                        $timeout(function() {element.append(buttonText);});
                        var shown = false;
                        var options = {
                                html: true,
                                content: popoverContent,
                                placement: (attrs.placement in scope) ? scope[attrs.placement]  : (attrs.placement || 'right'),
                                container: attrs.container?attrs.container:undefined,
                                title: title
                        };

                        if (attrs.on === "hover") {
                            options.animation = false;
                            element.popover(options).on("mouseenter", function () {
                                $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                $(this).popover("show");
                                if (attrs.popoverClass) {
                                    $timeout(() => {
                                        const popover = element.data('popover').$tip;
                                        popoverRef = popover;
                                        popover.addClass(attrs.popoverClass);
                                    });
                                }
                            }).on("mouseleave", function () {
                                $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                popoverRef = null;
                                $(this).popover("hide");
                            });
                        } else {  // click
                            element.popover(options);
                            function show() {
                                shown = true;
                                //capturing jquery destroy popover function for the current element, otherwise the method is not available anymore on $destroy of the directive (certainly because the elements got already removed from the DOM at that point)
                                destroyPopover = (element.data('popover')['destroy']).bind(element.data('popover'));
                                window.setTimeout(function() {
                                    $("html").click(hide);
                                    const popover = element.data('popover').$tip;
                                    if (attrs.clickable) popover.click(stopPropagation);
                                    if (attrs.popoverClass) {
                                        popover.addClass(attrs.popoverClass);

                                        $compile(popoverContent)(scope);
                                        // re-show to get appropriate sizing
                                        element.popover('show');
                                    }
                                }, 0);
                            }

                            function stopPropagation($event) {
                                $event.stopPropagation();
                            }

                            function hide() {
                                shown = false;
                                element.popover('hide');
                                $("html").unbind("click", hide);
                                const popover = element.data('popover').$tip;
                                popover.unbind('click', stopPropagation);
                                popover.hide();
                            }

                            element.click(function() {
                                if(shown) {
                                    hide();
                                    $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                } else {
                                    show();
                                    $timeout(function() {scope.$broadcast("codeViewerUIRefreshToggle");});
                                }
                            });
                        }

                        $(document).on("keydown.dkuInlinePopover", function(e) {
                            if(e.key === "Escape"){
                                destroy();
                            }
                        });

                        scope.uiRefreshToggle = false;

                        scope.$on("$destroy", function() {
                            destroy();
                        });

                        function destroy(){
                            if (typeof destroyPopover === "function") {
                                destroyPopover();
                            }
                            element.popover('destroy');
                            if (popoverRef) {
                                popoverRef.remove();
                                popoverRef = null;
                            }
                            $(document).off("keydown.dkuInlinePopover");
                        }
                    });
                };
            }
        };
    });

    /**
     * Very similar to dkuHelpPopover, but the template is a Markdown string
     *   Usage :
     *
     *   <button class="btn btn-small" dku-md-popover="# Yeah\n* Markdown" title="popover title">
     *        <label>
     *            <span class="icon-question">&nbsp;</span>
     *                      Button text
     *        </label>
     *        <content title="Help me">
     *            <h2>Introduction</h2>
     *            <p>Blablabla</p>
     *        </content>
     *   </button>
     *
     */
    app.directive('dkuMdPopover',function(MarkedSettingService, translate) {
        return {
            restrict : 'A',
            link : function($scope, element, attrs) {
                var shown = false;
                let destroyPopover;

                const getTitle = () => attrs.dkuMdTitle || translate("GLOBAL.HELP", "Help")

                var hide = function() {
                    $("html").unbind("click", hide);
                    element.popover('hide');
                    shown=false;
                };

                var show = function() {
                    shown = true;
                    //capturing jquery destroy popover function for the current element, otherwise the method is not available anymore on $destroy of the directive (certainly because the elements got already removed from the DOM at that point)
                    destroyPopover = (element.data('popover')['destroy']).bind(element.data('popover'));

                    marked.setOptions(MarkedSettingService.get($scope, attrs));
                    var contentElt = marked(attrs.dkuMdPopover);
                    var ret = $("<div class=\"" + (attrs.popoverClazz||"") + "\"></div>");
                    ret.html(contentElt);
                    element.popover('show');
                    var popover = element.data('popover');
                    $(popover.$tip)
                        .find('.popover-content')
                        .empty().append(ret)
                        .off("click.dku-pop-over")
                        .on("click.dku-pop-over", function(e) {
                            e.stopPropagation();
                        });
                    element.popover('show');

                    // In case the title has changed since the popup was initially built, update it.
                    let currentTitle = getTitle();
                    if (currentTitle !== options.title) {
                        $(popover.$tip).find('.popover-title').html(currentTitle);
                    }

                    window.setTimeout(function() { $("html").click(hide); }, 0);
                };
                var placement = element.data("placement") || "right";
                var options = {
                    html: true,
                    content: "",
                    placement: placement,
                    title: getTitle()
                };
                var container = element.data("container") || "body";
                if (container) {
                    options.container = container;
                }

                element.popover(options);
                element.click(function() {
                    if(shown) {
                        hide();
                    } else {
                        show();
                    }
                });

                $scope.$on('$destroy', function() {
                    if (typeof destroyPopover === "function") {
                        destroyPopover();
                    }
                });
            }
        };
    });

    /*
     * Usage :
     * bl br tl tr
     *
     *   <button class="btn btn-small" dku-inline-popup position="bl">
     *        <label>
     *            <span class="icon-question">&nbsp;</span>
     *                      Button text
     *        </label>
     *        <content title="Help me">
     *            <h2>Introduction</h2>
     *            <p>Blablabla</p>
     *        </content>
     *   </button>
     *
     * The popup content is created lazily. It is then only hidden. It is removed when
     * parent is destroyed
     */
    app.directive('dkuInlinePopup',function($timeout) {
        return {
            restrict : 'A',
            scope : true,
            transclude:true,
            template:'',
            compile:function(_element,_attrs,transclude) {
                return function(scope, element) {
                    transclude(scope.$new(), function(clone) {
                        var state = { popupElement : null};
                        var shown = false;
                        var buttonText = clone.filter('label').contents();
                        var popupContent = clone.filter('content').contents();
                        var addClazz = clone.filter('content').attr('class').split(/\s+/);


                        $timeout(function() {element.append(buttonText);});

                        var hide = function() {
                            if (state.popupElement) {
                                state.popupElement.hide();
                            }
                            $("html").unbind("click", hide);
                            shown=false;
                        };
                        var show = function() {
                            shown = true;
                            if (state.popupElement ==null) {
                                state.popupElement = $("<div class='dku-inline-popup' />");
                                $.each(addClazz, function(idx, val) { state.popupElement.addClass(val)});
                                state.popupElement.append(popupContent);
                                $("body").append(state.popupElement);
                            }
                            state.popupElement.css("overflow", "scroll");
                            state.popupElement.css("position", "absolute");
                            state.popupElement.css("left", element.offset().left);
                            var windowh = window.innerHeight || document.documentElement.clientHeight || document.body.clientHeight;
                            var etop = element.offset().top + element.outerHeight();
                            state.popupElement.css("top", etop);
                            state.popupElement.css("max-height", windowh - etop);

                            state.popupElement.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                                e.stopPropagation();
                            });
                            window.setTimeout(function() { $("html").click(hide) }, 0);

                            state.popupElement.show()
                        };
                        scope.hide = function(){
                            hide();
                        }

                        element.click(function(){
                            if (shown) hide();
                            else show();
                        })
                        scope.$on("$destroy", function() {
                            hide();
                            if (state.popupElement) {
                                state.popupElement.remove();
                                state.popupElement = null;
                            }
                        });
                    });
                };
            }
        };
    });


    app.directive('editableSlider', function($parse, $compile) {
        return {
            restrict : 'A',
            link : function($scope, element, attrs) {
                var tpl = $compile('<div class="ngrs-value-runner">'
                    +'    <div class="ngrs-value ngrs-value-min" style="max-width: 70px;">'
                    +'      <input ng-show="sliderEditManual1" select-all auto-focus="{{sliderEditManual1}}" step="0.1" ng-blur="sliderEditManual1=false;'+attrs.onHandleUp+'()" ng-type="number" next-on-enter blur-model="'+attrs.modelMin+'" ng-disabled="disableEditable" id="qa_editable_slider_min_input" />'
                    +'      <div ng-hide="sliderEditManual1"><a ng-click="sliderEditManual1=true" style="color: inherit; overflow: hidden; text-overflow: ellipsis;" title="{{'+attrs.modelMin+'}}" id="qa_editable_slider_min_pointer">{{'+attrs.modelMin+'}}</a></div>'
                    +' </div>'
                    +' <div class="ngrs-value ngrs-value-max" style="max-width: 70px;">'
                    +'    <input ng-show="sliderEditManual2" select-all auto-focus="{{sliderEditManual2}}" step="0.1" ng-blur="sliderEditManual2=false;'+attrs.onHandleUp+'()" ng-type="number" next-on-enter blur-model="'+attrs.modelMax+'" ng-disabled="disableEditable" id="qa_editable_slider_max_input"/>'
                    +'    <div ng-hide="sliderEditManual2"><a ng-click="sliderEditManual2=true" style="color: inherit; overflow: hidden; text-overflow: ellipsis;" title="{{'+attrs.modelMax+'}}" id="qa_editable_slider_max_pointer">{{'+attrs.modelMax+'}}</a></div>'
                    +'   </div>'
                    +'</div>');

                $scope.sliderEditManual1 = false;
                $scope.sliderEditManual2 = false;

                $scope.disableEditable = false;
                $scope.disableEditableAsString = attrs.disableEditableAsString;
                if ($scope.disableEditableAsString) {
                    $scope.disableEditable = $scope.$eval($scope.disableEditableAsString);
                    $scope.$watch($scope.disableEditableAsString, (nv) => {
                        $scope.disableEditable = nv;
                    });
                }

                attrs.$set('showValues','false');
                element.append(tpl($scope));
            }
        }
    });

    app.directive('dkuHelpPopover', function($parse, $compile, $http, $timeout, $q, $templateCache) {
        return {
            restrict : 'A',
            scope:true,
            controller: function($scope){
                this.dismissPopover = function() {
                    $scope.dismissPopover();
                }
                this.togglePopover = function() {
                    $scope.togglePopover();
                }
            },
            link : function($scope, element, attrs) {

                // By default, a bootstrap popover is not displayed if it has no title and no content
                // We need to create such popover in some cases!
                function execInPatchedEnv(fn) {
                    var old = $.fn.popover.Constructor.prototype.hasContent;
                    $.fn.popover.Constructor.prototype.hasContent = function() {
                        return true;
                    };
                    try {
                        fn();
                    } finally {
                        $.fn.popover.Constructor.prototype.hasContent = old;
                    }
                }

                var shown = false;
                var getter = $parse(attrs.dkuHelpPopover);
                var templateUrl = getter($scope);

                var hide = function() {
                    $("html").unbind("click", blur);
                    element.popover('hide');
                    shown=false;
                };

                //selector for elements that do not create blur/hide event
                //(all children will block hide event)
                var noBlur = [];
                if (attrs.noBlur) {
                    noBlur = $scope.$eval(attrs.noBlur);
                }
                var blur = function(evt) {
                    var ignore = false;
                    $.each(noBlur, function(_, selector) {
                        if($(evt.target).is(selector) || $(evt.target).parents(selector).length) {
                            ignore = true;
                        }
                    });
                    ignore || hide();
                }

                $scope.dismissPopover = function(){
                    hide();
                }

                var theTip;

                var show = function() {

                    shown = true;
                    $http.get(templateUrl, {cache: $templateCache}).then(function(response) {

                        var tmplData = $('<div/>').html(response.data).contents();
                        var tmpl = $compile(tmplData);
                        var popover = element.data('popover');
                        var html = tmpl($scope);
                        execInPatchedEnv(function() {
                            element.popover('show');
                            $(popover.$tip)
                                .find('.popover-content')
                                .html(html)
                                .off("click.dku-pop-over")
                                .off("click.dku-pop-over")
                                .off("click",".dropdown-menu")
                                .on("click.dku-pop-over", function(e) {
                                    e.stopPropagation();
                            });
                            theTip = popover.$tip;
                            element.popover('show');

                            if (attrs.noArrow) {
                                $(popover.$tip).find(".arrow").remove();
                                $(popover.$tip).css("top", $(popover.$tip).offset().top-20);
                            }
                            if (attrs.contentClazz) {
                                $(popover.$tip).find('.popover-content').addClass(attrs.contentClazz);
                            }
                            if(attrs.titleClazz) {
                                $(popover.$tip).find('.popover-title').addClass(attrs.titleClazz);
                            }
                            if (attrs.forceTopPositive) {
                                var currentTop = $(popover.$tip).css('top');
                                if (currentTop.charAt(0) === '-') {
                                    $(popover.$tip).css('top', '0px');
                                    // shift arrow accordingly
                                    $(popover.$tip).find('.arrow').css('transform', 'translateY(' + currentTop + ')');
                                }
                            }
                        });
                        window.setTimeout(function() { $("html").click(blur); }, 0);

                    });
                };

                var toggle = function() {
                    if (shown) {
                        hide();
                    }
                    else {
                        show();
                    }
                };

                $scope.togglePopover = function(){
                    return toggle();
                }

                $scope.showPopover = function(){
                    return show();
                }

                $scope.hidePopover = function(){
                    return hide();
                }

                var placement = element.data("placement") || "right";

                var options = {
                    html: true,
                    content: "",
                    placement: placement,
                    title: element.data("title")
                };
                var container = element.data("container");
                if (container) {
                    options.container = container;
                }
                element.popover(options);
                element.click(toggle);
                element[0].showPopover = function() {
                    if (!shown) {
                        show();
                    }
                };

                $scope.$on('$destroy', function() {
                       if(theTip && theTip.remove) {
                            theTip.remove();
                       }
                });

                element[0].hidePopover = function() {
                    if (shown) {
                        hide();
                    }
                };


            }
        };
    });

    app.directive('bzGauge', function() {
        return {
            restrict: 'E',
            template: "<div class='bz-gauge'><div class='mercury' style='width: {{ gaugeWidth }}%; background-color: {{ color }};'></div>",
            replace: true,
            scope: {
                color: "=color",
                val: "=val",
                total: "=total"
            },
            link: function(scope) {
                scope.gaugeWidth = (100.* scope.val / scope.total) | 0;
            }
        }
    })

    var debouncer = function(f, delay) {
      var delayer = null;
      return function() {
        if (delayer === null) {
          f();
          delayer = setTimeout(function() {
            delayer = null;
          }, delay)
        }
      }
    }

    app.directive('editableLabel', function() {
      return {
            restrict: 'E',
            template: "<div class='editable-label'><label>{{ val }}</label><input></input></div>",
            replace: true,
            scope: {
                val: '=ngModel',
            },
            link: function(scope, element) {
                var $label = element.find("label");
                var $input = element.find("input");

                var isEdit = false;

                var enterEdition = function() {
                  isEdit = true;
                  element.addClass("edit");
                  $input.focus();
                  $input.val(scope.val);
                }

                scope.validate = function(val) {
                  var trimmed = val.trim();
                  return (trimmed.length > 0);
                }

                var quitEdition = function() {
                  if (!isEdit) {
                    return;
                  }
                  var candidate = $input.val();
                  if (scope.validate(candidate)) {
                    scope.val = candidate.trim();
                    scope.$apply();
                    element.removeClass("edit");
                  }
                  isEdit = false;
                }

                var toggleEdition = function() {
                  if (isEdit) {
                    quitEdition();
                  }
                  else {
                    enterEdition();
                  }
                }

                toggleEdition = debouncer(toggleEdition, 400);
                $input.blur(toggleEdition);
                $input.change(toggleEdition);
                $label.click(toggleEdition);

            }
        }
    });


    app.directive('editableText', function() {
      return {
            restrict: 'E',
            template: "<div class='editable-text'><div>{{ message() }}</div><textarea></textarea></div>",
            replace: true,
            scope: {
                emptyMessage: '@emptyMessage',
                val: '=ngModel',
            },
            link: function(scope, element) {
                var $input = element.find("textarea");

                var isEdit = false;

                if (scope.val === undefined) {
                  scope.val = "";
                }
                scope.message = function() {
                  if (scope.val && (scope.val.trim().length > 0)) {
                    return scope.val;
                  }
                  else {
                    return scope.emptyMessage;
                  }
                }

                var enterEdition = function() {
                  isEdit = true;
                  element.addClass("edit");
                  $input.focus();
                  $input.val(scope.val);
                }

                scope.validate = function(val) {
                  var trimmed = val.trim();
                  return (trimmed.length > 0);
                }

                var quitEdition = function() {
                  if (!isEdit) {
                    return;
                  }
                  var candidate = $input.val();
                  if (scope.validate(candidate)) {
                    scope.val = candidate.trim();
                    scope.$apply();
                    element.removeClass("edit");
                  }
                  isEdit = false;
                }

                var toggleEdition = function() {
                  if (isEdit) {
                    quitEdition();
                  }
                  else {
                    enterEdition();
                  }
                }

                toggleEdition = debouncer(toggleEdition, 100);

                $input.blur(toggleEdition);
                $input.change(toggleEdition);
                element.click(toggleEdition);

            }
        }
    });

    app.directive('multiSelect', function(){
        return {
            require: 'ngModel',
            scope: true,
            link: function(scope, _element, _attrs, ngModel){
                scope.selectedItems = [];
                scope.allToggled = false;
                scope.someToggled = false;

                scope.toggleItem = function(item){
                    if (scope.selectedItems.indexOf(item) < 0) {
                        scope.selectedItems.push(item);
                    } else {
                        scope.selectedItems.splice(scope.selectedItems.indexOf(item), 1);
                    }
                    scope.allToggled = scope.selectedItems.length === ngModel.$viewValue.length;
                    scope.someToggled = scope.selectedItems.length > 0 && !scope.allToggled;

                    item.selected = !item.selected;
                };
                scope.toggleAll = function(){
                    var selected;
                    if(!scope.allToggled){
                        scope.selectedItems = angular.copy(ngModel.$viewValue);
                        selected = true;
                    } else {
                        scope.selectedItems = [];
                        selected = false;
                    }
                    scope.allToggled = scope.selectedItems.length === ngModel.$viewValue.length;
                    scope.someToggled = scope.selectedItems.length > 0 && !scope.allToggled;

                    angular.forEach(ngModel.$viewValue, function(item){
                        item.selected = selected;
                    });
                };
                scope.$on('clearMultiSelect', function(){
                    scope.selectedItems = [];
                    angular.forEach(ngModel.$viewValue, function(item){
                        item.selected = false;
                    });
                    scope.allToggled = false;
                    scope.someToggled = false;
                });
            }
        };
    });


    app.directive('modal', function($window){
        // This directive ensure the proper height of the modals
        return {
            restrict: 'C',
            link: function(scope, element, attrs){
                if (attrs.autoSize == "false") return;

                var content = element.find('.modal-body');

                // body height
                let oldMinHeight = content.css('minHeight');
                content.css('height', 0); //to get the padding height
                content.css('minHeight', 0);
                var paddingHeight = content.innerHeight();
                content.css('height', '');
                content.css('minHeight', oldMinHeight);


                function findOverflown(node){
                    if (node === undefined)
                        return [];
                    var overflownNodes = [];
                    for (var i = node.childNodes.length - 1; i >= 0; i--) {
                        var child = node.childNodes[i];
                        if (child.offsetHeight > 0){
                            var overflowCss = $(child).css('overflow');
                            var overflowYCss = $(child).css('overflow-y');
                            var scrollValues = ['auto', 'scroll'];
                            /* global contains */
                            if (contains(scrollValues, overflowCss) || contains(scrollValues, overflowYCss)) {
                                // Special code mirror cases
                                if(!$(child).hasClass('CodeMirror-hscrollbar') && !$(child).hasClass('CodeMirror-vscrollbar')
                                    && !$(child).hasClass('CodeMirror-scroll')) {
                                    overflownNodes.push(child);
                                }
                            } else {
                                overflownNodes = overflownNodes.concat(findOverflown(child));
                            }
                        }
                    }
                    return overflownNodes;
                }

                var sizeModal = function(){
                    // sometimes the modal-body is not instantiated at load, get it (again) here. If the
                    // modal-body was not there at modal creation time, then padding might be crappy
                    var content = element.find('.modal-body');
                    if (content.hasClass('modal-no-sizing')) return;

                    var oldMinHeight = content.css('minHeight');
                    content.css('height', '');
                    content.css('minHeight', '');
                    // find overflown elements
                    // We maximize the height of overflown content so they'll be the ones with the scrollbar
                    // var overflown = content.find('*').filter(function () {
//                         // select only the overflown content visible (positive height)
//                         return ['auto', 'scroll'].indexOf($(this).css('overflow')) >= 0 && this.offsetHeight > 0;
//                     });
//
                    var overflown = $(findOverflown(content[0]));

                    // remember current scroll position
                    var scrolls = overflown.map(function(){return this.scrollTop;});

                    overflown.css({'maxHeight': 'none', height: 0});
                    // height of non overflown content
                    var nonOverflownHeight = content.innerHeight();

                    overflown.height('');
                    var newHeight;
                    if (element.innerHeight() > $($window).height()) {
                        newHeight = $($window).height() - element.find('.modal-header').innerHeight() - element.find('.modal-footer').innerHeight() - paddingHeight - 10*2;
                    } else {
                        newHeight = content.innerHeight() - paddingHeight;
                    }
                    // preventing borders to be blurry : since modals are going to be centered on the screen, if the
                    // window height ends up being odd, then everything is going to be 1/2 pixel misaligned, and
                    // borders become all blurry messes... So we fix the modal-body height to be even. The rest of
                    // the window height is header+footer, so you have to set their size(s) to obtain a final even height.
                    if ( element.innerHeight() % 2 == 1 ) {
                        var maxHeight = parseInt(content.css('maxHeight'));
                        if ( newHeight + 1 > maxHeight) {
                            content.css('height', newHeight - 1);
                        } else {
                            content.css('height', newHeight + 1);
                        }
                    }

                    if(overflown.length){
                        // dispatch the remaining height between overflown content
                        var heightPerOverflown = (content.innerHeight() - nonOverflownHeight) / overflown.length;
                        if (heightPerOverflown > 0) {
                            overflown.height(heightPerOverflown);
                        }
                        // is the focused element within the modal ?
                        if(! $(document.activeElement).parents(element).length){
                            // focus overflow to allow scroll
                            overflown.attr('tabindex', 0).focus();
                        }
                        overflown.each(function(i){
                            // preserve former scroll
                            $(this).scrollTop(scrolls[i]);
                        });
                    }
                    content.css('minHeight', oldMinHeight);
                };

                // resize when window change
                $($window).on('resize.modal', sizeModal);
                // resize when something change (tab change, form expand...)
                scope.$watch(sizeModal, null);
                // init resize
                sizeModal();


                scope.$on('$destroy', function(){
                    $($window).off('resize.modal');
                });

                // focus first input
                element.find('input').first().focus();
            }
        };
    });

    app.directive('bsTypeahead', () => {
        return {
            priority: 100,
            link: function(scope,element, attr){
                var typeahead = element.data('typeahead');

                // override show function
                typeahead.show = function () {
                    var pos = $.extend({}, this.$element.offset(), {
                        height: this.$element[0].offsetHeight
                    });

                    $(document.body).append(this.$menu);
                    this.$menu.addClass(attr.class);
                    this.$menu.css({
                        position: 'absolute',
                        top: pos.top + pos.height,
                        left: pos.left,
                        'z-index': 5000
                    }).show();

                    this.shown = true;
                    return this;
                };
            }
        };
    });

    /**
     * select displaying columns of a provided schema together with their types
     * If the `$$groupKey` is present on the columns items, then the columns will be displayed under groups given the key,
     * Otherwise they will be just displayed in the same order as they appear in the columns array
     *
     * @param {Array} columns - Array containing the columns with their `name`, `type` and optional `$$groupKey` fields
     * @param {Object} selectedColumn - NgModel representing the selected column
     * @param {boolean} disableTypes - If true, no types will be displayed by the names
     * @param {boolean} immediateRefresh - If true, the column list will be updated as soon as there is any change on the schema
     *
     */
    app.directive('columnSelect', ($filter) => {
        return {
            restrict:'E',
            scope: {
                selectedColumn: '=ngModel',
                columns: '=',
                disableTypes: '=',
                immediateRefresh: '='
            },
            templateUrl: '/templates/widgets/column-select.html',
            link: {
                pre: function(scope) {
                    //compute types in prelink function so that it is available to optionsAnnotations
                    function computeTypes() {
                        let typeToName = $filter("columnTypeToName");
                        scope.filteredTypes = scope.disableTypes ? [] : scope.columns.map(function(column){ return typeToName(column.type)});
                    }
                    computeTypes();
                    scope.uiState = {selectedColumn: scope.selectedColumn};
                    scope.$watch("columns", computeTypes, true);
                    scope.$watch("uiState.selectedColumn", function(){
                        scope.selectedColumn = scope.uiState.selectedColumn;
                    });
                    scope.$watch("selectedColumn", function(){
                        scope.uiState.selectedColumn = scope.selectedColumn;
                    });
                }
            }
        };
    });

    //select diplaying columns of a provided schema together with their types with provided filter
    app.directive('columnSelectWithFilter', () => {
        return {
            restrict:'E',
            scope: {
                selectedColumn: '=ngModel',
                columns: '=',
                disableTypes: '=',
                filterFn: '='
            },
            template: '<select '+
                    ' dku-bs-select="{\'liveSearch\' : true}" '+
                    ' ng-model="uiState.selectedColumn"'+
                    ' class="qa_recipe_split-select-column"'+
                    ' ng-options="column.name as column.name for column in columns | filter: filterFn"'+
                    ' options-annotations="types" '+
                    ' />',
            link: {
                pre: function(scope) {
                    //compute types in prelink function so that it is available to optionsAnnotations
                    function computeTypes() {
                        scope.types = scope.disableTypes ? [] : scope.columns.filter(scope.filterFn).map(function(column){ return column.type});
                    }
                    computeTypes();
                    scope.uiState = {selectedColumn: scope.selectedColumn};
                    scope.$watch("columns", computeTypes, true);
                    scope.$watch("uiState.selectedColumn", function(){
                        scope.selectedColumn = scope.uiState.selectedColumn;
                    });
                    scope.$watch("selectedColumn", function(){
                        scope.uiState.selectedColumn = scope.selectedColumn;
                    });
                }
            }
        };
    });

    app.factory('ColumnGeneratorService', function() {
        function getIconFromType(type) {
            switch (type) {
                case "NUMERIC":
                    return "#";
                case "CATEGORY":
                    return "<span class='icon icon-font'></span>"
                case "TEXT":
                    return "<span class='icon-italic'></span>"
                case "VECTOR":
                    return "<span style='font-size: 14px'>[ ]</span>"
                default:
                    return "";
            }
        }
        function getHTMLContent(name, perFeature) {
            const itemElement = $('<div class="ml-col-select__item">');
            const iconWrapper = $('<div class="ml-col-select__icon-wrapper">');
            iconWrapper.append(getIconFromType(perFeature[name].type));
            itemElement.prop('title', name);
            itemElement.append(iconWrapper);
            itemElement.append(document.createTextNode(' ' + name));
            return itemElement.prop('outerHTML');
        }
        return {
            getPossibleColumns: function(columns, authorizedTypes, authorizedRoles, perFeature) {
                return columns.filter(x => {
                    const isTypeAuthorized = !authorizedTypes || authorizedTypes.includes(perFeature[x].type);
                    const isRoleAuthorized = !authorizedRoles || authorizedRoles.includes(perFeature[x].role);
                    return isTypeAuthorized && isRoleAuthorized;
                }).sort().map(v => {
                    return {
                        name: v,
                        html: getHTMLContent(v, perFeature)
                    }
                });
            }
        }
    })

    app.directive('mlColumnSelectWithType', function(ColumnGeneratorService) {
        return {
            restrict: 'E',
            scope: {
                perFeature: "=",
                selectedColumn: "=ngModel",
                // authorizedTypes is optional. If not specified, all types are authorized. Otherwise,
                // must be an array of authorized types (e.g ["CATEGORY", "NUMERIC"])
                authorizedTypes: "=",
                // authorizedRoles is optional. If not specified, all roles are authorized. Otherwise,
                // must be an array of authorized roles (e.g ["INPUT", "REJECT"])
                authorizedRoles: "=",
                // alreadyComputedColumns is optional. If specified, must be a Set.
                alreadyComputedColumns: "=",
            },
            template: '<select ng-model="selectedColumn"' +
                      '        dku-bs-select="{\'liveSearch\':true}"' +
                      '        options-annotations="columnsAnnotations">' +
                      '    <option ng-repeat="c in columns"' +
                      '            value="{{c.name}}"' +
                      '            data-content="{{c.html}}">' +
                      '            {{c.name}}' +
                      '    </option>' +
                      '</select>',
            link: ($scope) => {
                function setUpColumns() {
                    $scope.columns = ColumnGeneratorService.getPossibleColumns(Object.keys($scope.perFeature), $scope.authorizedTypes, $scope.authorizedRoles, $scope.perFeature).map(col => {
                        col.isComputed = ($scope.alreadyComputedColumns && $scope.alreadyComputedColumns.has(col.name));
                        return col
                    })

                    $scope.columnsAnnotations = $scope.columns.map(v => v.isComputed ? 'already computed' : '');
                }

                $scope.$watch("perFeature", function(nv) {
                    if (nv !== undefined) {
                        setUpColumns();
                    }
                });

                $scope.$watch("alreadyComputedColumns", function(nv) {
                    if (nv !== undefined) {
                        setUpColumns();
                    }
                });

            }
        }
    });


    app.directive('mappingEditor',function(Debounce, $timeout) {
        return {
            restrict:'E',
            scope: {
                mapping: '=ngModel',
                onChange: '&',
                noChangeOnAdd: '<',
                addLabel: '@',
                validate: '=?',
                colors: '=?',
                withColor: '=?',
                keepInvalid: '=?',
                required: '<',
                requiredKey: '<?',
                requiredValue: '<?',
                uniqueKeys: '<?',
                typeAhead: '=',
                keyPlaceholder: '@',
                valuePlaceholder: '@'
            },
            templateUrl : '/templates/shaker/mappingeditor.html',
            compile: () => ({
                pre: function (scope, element, attrs) {
                    const textarea = element.find('textarea');
                    textarea.on('keydown', function (e) {
                        let keyCode = e.keyCode || e.which;
                        //tab key
                        if (keyCode === 9) {
                            e.preventDefault();
                            if (!scope.$$phase) scope.$apply(function () {
                                let tabPosition = textarea[0].selectionStart;
                                scope.bulkMapping = scope.bulkMapping.slice(0, tabPosition) + '\t' + scope.bulkMapping.slice(tabPosition);
                                $timeout(function () {
                                    textarea[0].selectionEnd = tabPosition + 1;
                                });
                            });
                        }
                    });
                    scope.changeMode = function () {
                        if (!scope.showBulk) {
                            scope.bulkMapping = scope.mapping.map(m => (m.from === undefined ? '' : m.from) + '\t' + (m.to === undefined ? '' : m.to)).join('\n');
                        }
                        scope.showBulk = !scope.showBulk;
                    };

                    scope.$watch('bulkMapping', Debounce().withDelay(400, 400).wrap((nv) => {
                            if (!angular.isUndefined(nv)) {
                                if (!nv.length) {
                                    scope.mapping = [];
                                } else {
                                    scope.mapping = nv.split('\n').map((l, index) => {
                                        const color = scope.mapping && scope.mapping[index] && scope.mapping[index].color ? scope.mapping[index].color : 'grey';
                                        //regexp to split into no more than 2 parts (everything to the right of a tab is one piece)
                                        const parts = l.split(/\t(.*)/);
                                        return {from: parts[0], to: parts[1], color: color};
                                    });
                                }
                            }
                        })
                    );
                    if (angular.isUndefined(scope.mapping)) {
                        scope.mapping = [];
                    }
                    if (!scope.addLabel) scope.addLabel = 'Add another';



                    if ('preAdd' in attrs) {
                        scope.preAdd = scope.$parent.$eval(attrs.preAdd);
                    } else {
                        scope.preAdd = Object.keys(scope.mapping).length === 0;
                    }
                    if (scope.onChange) {
                        scope.callback = scope.onChange.bind(scope, {mapping: scope.mapping});
                    }
                }
            })

        };
    });


    /**
     * Simple form form a sampling edition with no filters.
     * Supports changing dataset on the fly
     * Does not support auto refresh mechanism.
     */
    app.directive("samplingForm", function(DataikuAPI, $stateParams, DatasetInfoCache, DatasetUtils, SamplingData, translate) {
        return {
            scope : {
                selection : '=',
                datasetSmartName : '<',
                backendType : '=',
                label : '=',
                disabled: '=',
                helpMessage: '@',
                showPartitionsSelector: '=',
            },
            templateUrl : '/templates/widgets/sampling-form.html',
            link : function($scope) {
                $scope.translate = translate;
                $scope.streamSamplingMethods = SamplingData.streamSamplingMethods;
                $scope.streamSamplingMethodsDesc = SamplingData.streamSamplingMethodsDesc;

                $scope.getPartitionsList = function () {
                    return DataikuAPI.datasets.listPartitions($scope.dataset).error(setErrorInScope.bind($scope))
                        .then(function (ret) {
                            return ret.data;
                        });
                };

                $scope.$watch("datasetSmartName",(nv) => {
                    if (nv) {
                        var loc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.datasetSmartName);
                        var promise = DatasetInfoCache.getSimple(loc.projectKey, loc.name)
                        promise.then(function(data){
                            $scope.dataset = data;
                            if ($scope.dataset.partitioning.dimensions.length == 0){
                                $scope.selection.partitionSelectionMethod = "ALL";
                            }
                            $scope.$broadcast("datasetChange")
                        });
                    }
                });
            }
        }
});



    app.component("changePartitionedEnabledModal", {
        bindings: {
            modalControl: '<',
            settingsLost: '<'
        },
        templateUrl: "/templates/analysis/prediction/change-partitioned-enabled-modal.html",
        controller: function changePartitionedEnabledModalController() {
            const $ctrl = this;
            $ctrl.resolve = function(){
                $ctrl.modalControl.resolve();
            }
            $ctrl.dismiss = function() {
                $ctrl.modalControl.dismiss();
            }
        }
    });

    app.component("uncertaintyForm", {
        bindings: {
            uncertainty : '='
        },
        templateUrl: '/templates/widgets/uncertainty-form.html'
    })

    /**
     * Simple form for sampling edition with partitions and no filters.
     * Supports changing dataset on the fly
     * Does not support auto refresh mechanism.
     */
    app.directive("partitionedModelForm", function(DataikuAPI, $stateParams, DatasetInfoCache, DatasetUtils, CreateModalFromComponent, changePartitionedEnabledModalDirective) {
        return {
            scope : {
                splitPolicy: '=',
                datasetSmartName : '=',
                mlTaskDesign : '='
            },
            templateUrl : '/templates/widgets/partitioned-model-form.html',
            link : function($scope) {
                $scope.getPartitionsList = function () {
                    return DataikuAPI.datasets.listPartitions($scope.dataset)
                        .error(setErrorInScope.bind($scope))
                        .then(resp => resp.data);
                };

                $scope.partitionedModel = $scope.mlTaskDesign.partitionedModel
                $scope.uiState = {
                    partitionedModelEnabled: $scope.partitionedModel.enabled
                }
                $scope.settingsLost = function () {
                    let lost = []
                    if ($scope.mlTaskDesign && $scope.mlTaskDesign.overridesParams && $scope.mlTaskDesign.overridesParams.overrides.length !== 0) lost.push("Overrides");
                    if ($scope.mlTaskDesign && $scope.mlTaskDesign.uncertainty && $scope.mlTaskDesign.uncertainty.predictionIntervalsEnabled) lost.push("Uncertainty");
                    return lost
                }
                $scope.onChangePartitionEnabled = function(){
                    if ($scope.uiState.partitionedModelEnabled && $scope.settingsLost().length !== 0){
                        CreateModalFromComponent(changePartitionedEnabledModalDirective, {settingsLost: $scope.settingsLost()})
                            .then(
                                () => {
                                    $scope.mlTaskDesign.overridesParams.overrides = []
                                    if ($scope.mlTaskDesign.uncertainty) {
                                        $scope.mlTaskDesign.uncertainty.predictionIntervalsEnabled = false
                                    }
                                    $scope.partitionedModel.enabled = $scope.uiState.partitionedModelEnabled;
                                },
                                () => {
                                    // When modal is dismissed we reset the ui state to the previous value
                                    $scope.uiState.partitionedModelEnabled = $scope.partitionedModel.enabled;
                                });
                    }
                    else{
                        $scope.partitionedModel.enabled = $scope.uiState.partitionedModelEnabled;
                    }
                };

                $scope.partitioningDisabledReason = function () {
                    if (!$scope.dataset) {
                        return "Loading…";
                    } else if ($scope.dataset.partitioning.dimensions.length == 0) {
                        return "input dataset is not partitioned";
                    } else if ($scope.splitPolicy != 'SPLIT_MAIN_DATASET') {
                        return "train/test split policy is not compatible";
                    }
                    return; // not disabled
                };

                $scope.dimensionsList = function() {
                    return $scope.dataset.partitioning.dimensions
                        .map(dim => `<b>${sanitize(dim.name)}</b>`)
                        .join(', ');
                };

                $scope.$watch("datasetSmartName", function(nv) {
                    if (nv) {
                        const loc = DatasetUtils.getLocFromSmart($stateParams.projectKey, $scope.datasetSmartName);
                        DatasetInfoCache.getSimple(loc.projectKey, loc.name).then(function(data){
                            $scope.dataset = data;
                            if ($scope.dataset.partitioning.dimensions.length === 0){
                                $scope.partitionedModel.ssdSelection.partitionSelectionMethod = "ALL";
                            }
                            $scope.$broadcast("datasetChange")
                        });
                    }
                });
            }
        }
    });


    /**
     * Simple form for inserting ordering rules for export/sampling
     * which is a list of columns and order (asc or desc)
     */
    app.directive("orderingRulesForm", function() {
        return {
            scope: {
                rules: '='
            },
            templateUrl: '/templates/widgets/ordering-rules-form.html'
        };
    });

    app.directive("ngScopeElement", function () {
        var directiveDefinitionObject = {
            restrict: "A",
            compile: function compile() {
                return {
                    pre: function preLink(scope, iElement, iAttrs) {
                        scope[iAttrs.ngScopeElement] = iElement;
                    }
                };
            }
        };
        return directiveDefinitionObject;
    });


    app.directive('customElementPopup', function($timeout, $compile, $rootScope) {
        // Attrs:
        //   - cep-position = align-left-bottom, align-right-bottom, smart
        //   - cep-width = fit-main (adapt size of popover to size of mainzone)
    return {
        restrict: 'A',
        scope: true, // no isolated scope, we want our user to call us
        compile: function(element, attrs) {
            var closeOthers = attrs.closeOthers !== "false",  // opt-out
                closeOnClick = attrs.closeOnClick === "true", // opt-in
                allowModals = attrs.allowModals === "true", // don't close after modal is opened/closed
                popoverTemplate = element.find('.popover').detach(),
                dismissDeregister = null
            return function($scope, element, attrs) {
                var identifier = {}, popover = null,
                    position = attrs.cepPosition || "align-left-bottom",
                    hidePopoverButton = attrs.hidePopoverButton === "true",
                    startMainZone = $(".mainzone", element),
                    popoverShown = false,
                    onHideCallback = attrs.onHideCallback,
                    onShowCallback = attrs.onShowCallback;

                function isChildOfPopup(target) {
                    return $(target).closest(popover).length > 0;
                }

                function hideIfNoIdentifierOrNotMe(event, evtIdentifier) {
                    const isButtonClicked = $(event.target).closest(element).length > 0;
                    if (isButtonClicked || allowModals && $(event.target).closest('.modal-container, .modal-backdrop').length > 0) {
                        return;
                    }
                    const evtIdChange = (!evtIdentifier || identifier !== evtIdentifier);
                    if (evtIdChange && !isChildOfPopup(event.target) && !(evtIdentifier &&  isChildOfPopup(evtIdentifier))){
                        hide();
                    }
                }
                function hide() {
                    if (popover) {
                        popover.hide().detach();
                    }
                    $timeout(() => {popoverShown = false;});
                    $("html").unbind("click", hideIfNoIdentifierOrNotMe);
                    (startMainZone.length ? startMainZone : $(".mainzone", element)).removeClass('popover-shown');
                    if (onHideCallback && $scope.$eval(onHideCallback) instanceof Function) {
                        $scope.$eval(onHideCallback)();
                    }
                }
                function addPositionOffset(direction, value) {
                    try {
                        var res = value + parseInt(attrs['cepOffset' + direction]);
                        return isNaN(res) ? value : res;
                    } catch(e) {
                        return value;
                    }
                }
                function show() {
                    // clear other sub-popovers
                    $rootScope.$broadcast('dismissSubPopovers');
                    var mainZone = startMainZone.length ? startMainZone : $(".mainzone", element),
                        mzOff = mainZone.offset();
                    popoverShown = true;
                    if (popover === null) {
                        popover = $compile(popoverTemplate.get(0).cloneNode(true))($scope);
                        // here the template is compiled but not resolved
                        // => popover.innerWidth() etc. are incorrect until the next $digest
                    }
                    popover.css('visibility', 'hidden').appendTo("body");

                    function computeSmartPosition(offset) {
                        if (mzOff.left * 2 < window.innerWidth) {
                            offset.left = mzOff.left;
                        } else {
                            offset.right = window.innerWidth - mzOff.left - mainZone.innerWidth();
                        }
                        if (mzOff.top * 2 < window.innerHeight) {
                            offset.top = mzOff.top + mainZone.height();
                        } else {
                            offset.bottom = window.innerHeight - mzOff.top;
                        }
                        popover.css(offset);
                    }

                    window.setTimeout(function() {  // see above
                        let offset;
                        switch (position) {
                        case 'align-left-bottom':
                            offset = {
                                left: addPositionOffset('Left',mzOff.left),
                                top: addPositionOffset('Top',mzOff.top + mainZone.innerHeight())
                            };
                            if (mzOff.top + mainZone.innerHeight() + popover[0].scrollHeight > window.innerHeight) {
                                offset.height = Math.max(window.innerHeight - mzOff.top - mainZone.innerHeight() - 4, 50);
                                offset.overflow = "auto";
                            } else {
                                offset.height = '';
                                offset.overflow = '';
                            }
                            popover.css(offset);
                            break;
                        case 'align-right-bottom':
                            popover.css({
                                top: addPositionOffset('Top',mzOff.top + mainZone.innerHeight()),
                                left: addPositionOffset('Left', mzOff.left + mainZone.innerWidth() - popover.innerWidth())
                            });
                            break;
                        case 'smart':
                            offset = { left: 'auto', right: 'auto', top: 'auto', bottom: 'auto' };
                            computeSmartPosition(offset);
                            break;
                        case 'auto':
                            offset = { left: 'auto', right: 'auto', top: 'auto', bottom: 'auto' };
                            if (window.innerHeight - (mzOff.top+mainZone.innerHeight() + popover.innerHeight()) > 0) {
                                offset.top = mzOff.top + mainZone.innerHeight();
                                offset.left = mzOff.left + mainZone.innerWidth() - popover.innerWidth();
                                offset.right = mzOff.right + mainZone.innerWidth();
                                popover.css(offset);
                             }
                            else {
                                computeSmartPosition(offset);
                             }
                             break;
                        case 'align-left-top':
                            if (hidePopoverButton) {
                                popover.css({ left: mzOff.left, top: mzOff.top, bottom: 'auto' });
                            } else {
                                popover.css({ left: mzOff.left, bottom: window.innerHeight - mzOff.top, top: 'auto' });
                            }
                            break;
                        case 'align-right':
                            popover.css({ left: mzOff.left+mainZone.outerWidth(), top: mzOff.top, bottom: 'auto' });
                            break;
                        case 'align-right-top':
                            popover.css({ left: mzOff.left + mainZone.innerWidth() - popover.innerWidth(),
                                bottom: window.innerHeight - mzOff.top, top: 'auto' });
                            break;
                        }
                        if (attrs.cepWidth === 'fit-main') {
                            popover.css("width", mainZone.innerWidth());
                        }
                        popover.css('visibility', 'visible');
                    }, 100);

                    popover.show();
                    mainZone.addClass('popover-shown');
                    popover.add(".mainzone", element).off("click.dku-pop-over");
                    popover.on("click.dku-pop-over", function() {
                            if (closeOthers) {
                                $("html").triggerHandler('click.cePopup', identifier);
                            }
                        });
                    $(".mainzone", element).on("click.dku-pop-over", function() {
                        if (closeOthers) {
                            $("html").triggerHandler('click.cePopup', identifier);
                        }
                    });
                    if (closeOnClick) popover.on("click.dku-pop-over", hide);
                    window.setTimeout(function() {
                        $("html").on('click.cePopup', function (event, evtIdentifier) {
                            hideIfNoIdentifierOrNotMe(event, evtIdentifier);
                        });
                    }, 0);

                    if (dismissDeregister) {
                        dismissDeregister();
                    }
                    dismissDeregister = $rootScope.$on("dismissPopovers", function(){
                        if (!allowModals) {
                            hide();
                            if (dismissDeregister) {
                                dismissDeregister();
                                dismissDeregister = null;
                            }
                        }
                    });

                    if (onShowCallback && $scope.$eval(onShowCallback) instanceof Function) {
                        $scope.$eval(onShowCallback)();
                    }
                }

                $scope.showPopover = show;
                $scope.hidePopover = hide;
                $scope.popoverShown = function() { return popoverShown; };
                $scope.togglePopover = function(event) {
                    if (popoverShown) {
                        hide();
                    }
                    else {
                        if (event) $("html").triggerHandler('click.cePopup', identifier);
                        show();
                    }
                };
            };
        } };
    });

    app.directive("sidebarTabL1Link", ($compile) => {
        return {
            replace: true,
            priority: 100,
            scope : {
                tabName : "@",
                label : "@",
                disabled: "@",
                tooltipText: "@"
            },
            link : function(scope, element, attrs) {
                scope.$watch(() => attrs.disabled, () => {
                    let template;
                    if (scope.$eval(attrs.disabled)) {
                        template = '<li class="l1" style="color: #666666; opacity: 0.5;">' +
                            '       <a data-toggle="tooltip" data-placement="right" title="{{tooltipText}}" data-container="body">' +
                            '           {{ label }}' +
                            '       </a>'+
                            '       </li>';
                    } else {
                        template = '<li class="l1" tab-active="{{tabName}}" full-click><a main-click tab-set="{{tabName}}">{{label}}</a></li>';
                    }
                    element.html(template);
                    $compile(element.contents())(scope);
                });
            }
        }
    });

    app.directive("sidebarTabL1Href", () => {
        return {
            template: '<li class="l1" tab-active="{{tabName}}" full-click><a main-click ui-sref=".{{tabName}}" data-qa-analysis-page__left-tab="{{tabName}}">{{label}}</a></li>',
            replace: true,
            priority: 100,
            scope : {
                tabName : "@",
                label : "@"
            }
        }
    });

    app.directive("sidebarTabL2Link", function($compile){
        return {
            replace: true,

            priority: 100,
            scope : {
                tabName : "@",
                linkName : "@",
                label : "@",
                disableLink: "@",
                disableMessage: "@",
                useHref: "@",
                hrefParams: "=",
                sidekickPulsar: "=",
                alternativeTabs: "="
            },
            link : function(scope, element, attrs) {
                scope.$watch(() => attrs.disableLink, () => {
                    let template;
                    if (scope.$eval(attrs.disableLink)) {
                        template = `<li toggle="tooltip" container="body" title="{{disableMessage}}" style="opacity: 0.5">
                        <div class="l2">{{label}}</div>
                    </li>`;
                    } else if (attrs.useHref){
                        template = '<li class="l2" tab-active="{{tabName}}" alternative-tabs="{{alternativeTabs}}" full-click><div class="padded"><a main-click ui-sref=".{{linkName ? linkName : tabName}}(hrefParams)">{{label}}<span class="mleft8 sidekick-pulsar" ng-if="sidekickPulsar"></span></a></div></li>';
                    } else {
                        template = '<li class="l2" tab-active="{{tabName}}" alternative-tabs="{{alternativeTabs}}" full-click><div class="padded"><a main-click tab-set="{{tabName}}">{{label}}<span class="mleft8 sidekick-pulsar" ng-if="sidekickPulsar"></span></a></div></li>';
                    }
                    element.html(template);
                    $compile(element.contents())(scope);
                });
            }
        }
    });

    app.directive("sidebarRoutingTabL2Link", function($state, $timeout, $compile){
        return {
            replace: true,
            priority: 100,
            scope : {
                tabName : "@",
                tabModel: "=",
                label : "@",
                sidekickPulsar: "="
            },
            link: function(scope, element, attrs) {
                scope.disabledMessage = attrs.disabledMessage;
                scope.onDisableUpdate = function(nv) {
                    element.empty();
                    let tpl;
                    if (nv) {
                        tpl = $compile(`
                        <div>
                            <li toggle="tooltip" container="body" title="{{disabledMessage}}" style="opacity: 0.5">
                                <div class="l2">{{label}}</div>
                            </li>
                        </div>
                        `);
                    } else {
                        tpl = $compile(`
                        <div>
                            <li class="l2" ng-class="{'active': tabName == tabModel}">
                                <div class="padded"><a ng-click="goTab()">{{label}}<span class="mleft8 sidekick-pulsar" ng-if="sidekickPulsar"></a></span></a></div>
                            </li>
                        </div>
                        `);
                    }
                    element.append(tpl(scope));
                }
                attrs.$observe('disableLink', function(nv) {
                    scope.onDisableUpdate(nv);
                });
                scope.onDisableUpdate(attrs.disableLink);
                scope.goTab = function() {
                    scope.tabModel = scope.tabName;
                    const stateNameParts = $state.current.name.split(".");
                    stateNameParts[stateNameParts.length-1] = scope.tabName;
                    $timeout(function() {
                        $state.go(stateNameParts.join('.'));
                    });
                }
            }
        }
    });

    app.directive("topLevelTabState", function($rootScope){
        return {
            template: '<a class="tab" ng-class="{\'enabled\' : topNav.tab == tabName}" ui-sref="{{sref}}">{{label}}</a>',
            replace: true,
            scope : {
                tabName : "@",
                sref : "@",
                label: '@'
            },
            link : function($scope) {
                $scope.topNav = $rootScope.topNav;
            }
        }
    });


    app.service('ModalSeeThroughService', () => {
        const getHeaderHideClass = (seeThrough) => seeThrough ? 'see-through' : 'non-see-through';
        const svc = {
            isSeeThrough: false,
            toggleSeeThrough() {
                svc.setSeeThrough(!svc.isSeeThrough);
            },
            setSeeThrough(val) {
                svc.isSeeThrough = val;
                const divs = $('div.modal-container, div.modal-backdrop, div.popover');
                const toAdd = getHeaderHideClass(svc.isSeeThrough);
                const toRemove = getHeaderHideClass(!svc.isSeeThrough);
                divs.addClass(toAdd).removeClass(toRemove);
            }
        };
        return svc;
    })


    const dkuModalHeader = ($timeout, ModalSeeThroughService, translate) => {

        return {
            template : `
            <div class="vertical-flex ais gap-2x">
                <div class="modal-header {{modalClass}}">
                    <i ng-if="modalTotem || hasSvgTotem()"
                        class="modal-header__totem {{modalTotem}}"
                        ng-transclude="totemSvg"
                    ></i>
                    <h4 class="modal-header__title" ng-transclude="title">
                        <div class="mx-textellipsis" show-tooltip-on-content-overflow toggle="tooltip-bottom" title="{{modalTitle}}" container="body">
                            {{modalTitle}}
                        </div>
                    </h4>
                    <div class="modal-header__menu-wrapper" ng-if="hasMenu()" ng-transclude="menu"></div>
                    <div class="modal-header__button-wrapper">
                        <button id="modal-btn-see-through"
                            type="button"
                            class="btn btn--text btn--dku-icon see-through no-modal-autofocus"
                            ng-click="toggleSeeThrough()"
                            toggle="tooltip-bottom" title="{{ModalSeeThroughService.isSeeThrough ? '${translate('MODALS.BACK_TO_DIALOG', 'Back to dialog')}' : '${translate('MODALS.SEE_BEHIND', 'See behind')}'}}" container="body"
                        >
                            <i class="dku-modal-see-through__icon--not-hovered {{ModalSeeThroughService.isSeeThrough ? 'dku-icon-eye-off-20' : 'dku-icon-eye-20'}}"></i>
                            <i class="dku-modal-see-through__icon--hovered {{ModalSeeThroughService.isSeeThrough ? 'dku-icon-eye-20' : 'dku-icon-eye-off-20'}}"></i>
                        </button>
                        <button id="modal-btn-close" type="button"
                            class="btn btn--text btn--dku-icon no-modal-autofocus"
                            data-dismiss="{{modalClose && modalClose() ? '' : 'modal'}}" ng-click="close()"
                            toggle="tooltip-bottom" title="${ translate('MODALS.CLOSE_WINDOW', 'Close window') }"
                        >
                            <i class="dku-icon-dismiss-20"></i>
                        </button>
                    </div>
                </div>
                <ul ng-if="hasTabs()" class="modal-tabs" ng-transclude="tabs"></ul>
            </div>
            `,
            scope : { modalTitle : "@", modalTotem : "@?", modalClass: "@?", modalClose: "&?" },
            replace : true,
            transclude: {
                'title': '?dkuModalTitle',
                'totemSvg': '?dkuModalTotem',
                'tabs': '?dkuModalTabs',
                'menu': '?dkuModalMenu'
            },
            link: function (scope, $element, $attrs, $thisCtrl, $transclude) {
                // edge case - when a new modal opens, we make if visible regardless of the current see-through status
                ModalSeeThroughService.setSeeThrough(false);

                const $seeThroughButton = $element.find('#modal-btn-see-through');
                scope.ModalSeeThroughService = ModalSeeThroughService;
                scope.close = function () {
                    if (scope.modalClose && scope.modalClose() instanceof Function) return scope.modalClose()();
                    return false;
                }
                scope.hasTabs = function () {return $transclude.isSlotFilled('tabs');}
                scope.hasMenu = function () {return $transclude.isSlotFilled('menu');}
                scope.hasSvgTotem = function () {return $transclude.isSlotFilled('totemSvg');}

                scope.toggleSeeThrough = function () {
                    // force-reload the tooltip for changed text
                    $seeThroughButton.tooltip('hide');
                    $timeout(() => $seeThroughButton.tooltip('show'));
                    ModalSeeThroughService.toggleSeeThrough();
                }

                scope.$on('$destroy', () => {
                    // tooltip is attached to body (only way to prevent transparency to affect it), so we need to manually destroy it in case user close modal with mouse still on button
                    // since the modal DOM is removed before the scope is destroyed, using $element.tooltip('destroy') doesn't work.
                    // this method is a bit brutal but in practice any tooltip displayed at this moment should come from inside the modal, so it's very unlikely it has a negative side-effect
                    $('.tooltip').remove();
                    // always restore visibility on modal close.
                    ModalSeeThroughService.setSeeThrough(false);
                });
            }
        }
    }

    // deprectated. use dkuModalHeader with optional modalTotem attribute.
    app.directive("dkuModalHeaderWithTotem", dkuModalHeader); // dku-modal-header-with-totem

    app.directive("dkuModalHeader", dkuModalHeader); // dku-modal-header


    app.directive('dkuEnter', function() {
        return function(scope, element, attrs) {
            element.bind("keydown keypress", function(event) {
                if(event.which === 13) {
                        scope.$apply(function(){
                                scope.$eval(attrs.dkuEnter, {$event: event});
                        });
                        event.preventDefault();
                }
            });
        };
    });


    app.directive('cancelOnEscape', function() {
        return function(_scope, element) {
            var val = element.val();

            element.bind("focus", () => {
                val = element.val();
            });

            element.bind("keydown keypress", function(event) {
                if(event.which === 27) {
                    element.val(val);
                    element.blur();
                    event.preventDefault();
                }
            });
        };
    });

    app.directive('svgTitles', function($sanitize) {
        function go(tooltip, stack, evt) {
            if (stack.length) {
                if (evt) {
                    var pos = {};
                    if (evt.clientX * 2 > window.innerWidth ) {
                        pos.right = (window.innerWidth - evt.clientX) + 'px';
                        pos.left  = 'auto';
                    } else {
                        pos.left  = evt.clientX + 'px';
                        pos.right = 'auto';
                    }
                    if (evt.clientY * 2 > window.innerHeight) {
                        pos.bottom = (window.innerHeight - evt.clientY) + 'px';
                        pos.top    = 'auto';
                    } else {
                        pos.top    = evt.clientY + 'px';
                        pos.bottom = 'auto';
                    }
                    tooltip.style(pos);
                }
                tooltip.html($sanitize(stack[stack.length - 1].getAttribute('data-title')));
                tooltip.style({display: 'block'});
            } else {
                tooltip.style({display: 'none', left: '0', top: '0'});
            }
        }
        return { restrict: 'A', scope: false, controller: function($element) {
            var _elt = $element.get(0),
                elt = _elt.tagName.toLowerCase() === 'svg' ? _elt : _elt.querySelector('svg'),
                svg = d3.select(elt),
                stack = [],
                tooltip = d3.select(document.body.insertBefore(document.createElement('div'), null))
                            .attr('class', 'svg-title-tooltip');
            return {
                update: function() {
                    svg.selectAll('[data-title]')
                    .on('mouseover.svgTitle', function mouseover() {
                        var i = stack.indexOf(this);
                        if (stack.length === 0 || i + 1 !== stack.length) {
                            if (i !== -1) { stack.splice(i, 1); }
                            stack.push(this);
                        }
                        go(tooltip, stack, d3.event);
                    }).on('mouseout.svgTitle', function mouseout() {
                        var i = stack.indexOf(this);
                        if (i !== -1) { stack.splice(i, 1); }
                        go(tooltip, stack, d3.event);
                    });
                },
                delete: function() {
                    tooltip.remove();
                }
            };
        }, link: function(scope, element, attrs, ctrl) {
            ctrl.update();
            scope.$on('$destroy', ctrl.delete);
        } };
    });

    app.directive('dkuBetterTooltip', ($sanitize) => {
        // Attrs:
        //   - dbt-placement = "top" / "bottom"
        //   - dbt-title
        var ret = {
            restrict : 'A',
        };
        ret.link = function(_scope, element, attrs) {
            var tooltip = null;
            function show(){
                tooltip = $("<div />");
                tooltip.html($sanitize(attrs.dbtTitle));
                tooltip.addClass("dbt-tooltip");
                tooltip.css("pointer-events", "none");
                if (attrs.dbtClazz) {
                    tooltip.addClass(attrs.dbtClazz);
                }
                $("body").append(tooltip); //so we have access to dimensions

                var posLeft = 0;
                var posTop = 0;
                var left = $(element).offset().left;
                var top = $(element).offset().top;
                var placement = attrs.dbtPlacement;
                var rect = $(element).get(0).getBoundingClientRect();
                if (placement == "top") {
                    posLeft = left + rect.width / 2 - tooltip.width()/2;
                    posTop = top - tooltip.height() - 10;
                } else if(placement == "top-right"){
                    posLeft = left + rect.width;
                    posTop = top - tooltip.height() - 10;
                } else if(placement == "top-left"){
                    posLeft = left - tooltip.width() - 10;
                    posTop = top - tooltip.height() - 10;
                } else if(placement == "bottom-left"){
                    posLeft = left - tooltip.width() - 10;
                    posTop = top + rect.height;
                } else if(placement == "bottom-right"){
                    posLeft = left + rect.width;
                    posTop = top + rect.height;
                } else if(placement == "bottom"){
                    posLeft = left + rect.width / 2 - tooltip.width()/2;
                    posTop = top + rect.height;
                } else if(placement == "left"){
                    posLeft = left - tooltip.width() - 10;
                    posTop = top + rect.height / 2 - tooltip.height()/2;
                } else if(placement == "right"){
                    posLeft = left + rect.width;
                    posTop = top + rect.height / 2 - tooltip.height()/2;
                }
                tooltip.css("left", posLeft);
                tooltip.css("top", posTop);
                $("body").append(tooltip);
            }
            function hide(){
                tooltip.remove();
            }
            element.on("mouseover.dbt", show);
            element.on("mouseout.dbt", hide);
        };
        return ret;
    });

    app.directive('svgTooltip', ($sanitize) => {

        return {
            scope: false,
            restrict: 'A',
            link: function($scope, element, attrs) {

                var $container = $(attrs.container || 'body').filter(':visible');
                var $tooltip = $('<div class="svg-tooltip ' + (attrs.tooltipClass || '') + '"></div>').appendTo($container);

                $scope.setTooltipContent = function(content) {
                    $tooltip.html($sanitize(content));
                };

                $scope.hideTooltip = function() {
                    $tooltip.css("opacity", 0);
                };

                $scope.showTooltip = function(x, y) {
                    let containerOffset = $container.offset();
                    let elOffset = $(element).offset();
                    $tooltip.css("top", (y + elOffset.top - containerOffset.top + 5) + "px");
                    $tooltip.css("left", (x + elOffset.left - containerOffset.left +  5) + "px");
                    $tooltip.css("opacity", 1);
                };

                $scope.$on("$destroy", function() {
                   $tooltip.remove();
                });
            }
        };
    });


    /**
     * In the new dataset page, each dataset name is a link-looking button to create the dataset.
     * the exact look depends on the dataset type status (license, available connection...)
     */
    app.component('newDatasetButton', {
        bindings : {
            type : "<",
        },
        template: `<button
            ng-class="{'dataset-notlicensed-ce' : $ctrl.type.status == 'NOT_LICENSED_CE'}"
            ng-disabled="$ctrl.type.status == 'NOT_LICENSED_EE' || $ctrl.type.status == 'NO_CONNECTION'"
            title="{{$ctrl.type.tooltip}}"
        >
            {{$ctrl.type.label}}
        </button>`,
    });

    /**
     * This directive watches the width of the new dataset page, and automatically updates the width of a centered container so that the tiles are always centered and aligned.
     * I used this method because there is no way through css alone to guarantee that the tiles will flow correctly, that the plugin block will be aligned with the main block, and that the top-right links are aligned with the tiles.
     */
    app.directive('newDatasetPageAlignmentManager', function() { // new-dataset-page-alignment-manager
        return {
            restrict: 'A',
            link: function($scope, $element) {
                const tileWidth = 301; // the tile is 300px, but when zoom level is < 100%, border grows slightly

                const resizeObserver = new ResizeObserver((el) => {
                    const elementWidth = el[0].contentRect.width;
                    $element.find('.new-dataset-page__centered-container').css('width', tileWidth * Math.floor(elementWidth / tileWidth));
                });

                resizeObserver.observe($element[0]);
                $scope.$on('$destroy', () => {
                    resizeObserver.disconnect();
                })
            }
        }
    })

    app.filter("singleChecklistState", function(translate){
        return function(input) {
            var total = 0, done = 0;
            input.items.forEach(function(x) {
                total++;
                if (x.done) done++;
            });
            if (total == 0) return "";
            return (
                '<span>'
                + translate(
                    (
                        done <= 1
                        ? 'SINGLE_CHECK_LIST_STATE.DONE'
                        : 'SINGLE_CHECK_LIST_STATE.DONE_PLURAL'
                    ),
                    "({{ done }}/{{ total }} done)",
                    { done, total }
                )
                + "</span>"
            );
        }
    });
    /** Emits "checklistEdited" on any change */
    app.directive('objectChecklist', ($rootScope) => {
        return {
            restrict : 'A',
            scope : {
                "checklist" : "=",
                "itemsOnly" : "=",
                "readOnly" : "="
            },
            templateUrl: "/templates/widgets/checklist.html",
            link : ($scope, element) => {
                $scope.state = {
                    addingItem : false,
                    editingItem : null, // The item being edited
                    editingItemText : null // The new text of the item being edited
                };

                $scope.onItemStateChange = function(){
                    $scope.$emit("checklistEdited", "item-state-change");
                };

                $scope.enterEditItem = function(item, $event){
                    if ($event.target.tagName.toLowerCase() == "a") {
                            return;
                    }
                    // Cancel the other one first
                    if ($scope.state.editingItem) {
                        $scope.cancelEditItem();
                    }

                    item.editingText = true;
                    $scope.state.editingItem = item;
                    $scope.state.editingItemText = item.text;

                    window.setTimeout(function() {
                        $(".checklist-items .edit-zone", element).on("click.checklistEditItem", function(e) {
                            e.stopPropagation();
                        });
                        $("html").on("click.checklistEditItem", function(event){
                            if ($(event.target).hasClass('modal-backdrop') || $(event.target.parentNode).hasClass('modal-header') || $(event.target.parentNode).hasClass('modal-footer')) {
                                return;
                            }
                            $scope.$apply(function(){$scope.cancelEditItem()});
                        })
                    }, 0);
                };

                $scope.validateEditItem = function(){
                    if ($scope.state.editingItemText.trim().length == 0) return;

                    $scope.state.editingItem.text = $scope.state.editingItemText;
                    $scope.cancelEditItem();
                    $scope.$emit("checklistEdited", "validate-edit");
                };

                $scope.cancelEditItem = function(){
                    if ($('.codemirror-editor-modal').is(':visible')) {
                        return;
                    }
                    $scope.state.editingItem.editingText = false;
                    $scope.state.editingItem = null;
                    $scope.state.editingItemText = null;
                    $(".checklist-items .edit-zone", element).off("click.checklistEditItem");
                    $("html").off("click.checklistEditItem");
                };

                $scope.deleteItem = function(item) {
                    $scope.checklist.items.splice($scope.checklist.items.indexOf(item), 1);
                    $scope.$emit("checklistEdited", "delete");
                };

                $scope.enterAddItem = function() {
                    $scope.state.addingItem = true;
                    $scope.state.newItemText = "";
                    window.setTimeout(function() {
                        $(".new-item-zone", element).on("click.checklistAddNewItem", function(e) {
                                e.stopPropagation();
                        });

                        $("html").on("click.checklistAddNewItem", function(event){
                            if ($(event.target).hasClass('modal-backdrop') || $(event.target.parentNode).hasClass('modal-header') || $(event.target.parentNode).hasClass('modal-footer')) {
                                return;
                            }
                            $scope.$apply(function(){$scope.leaveAddItem()});
                        })
                    }, 0);
                };

                $scope.leaveAddItem = function(){
                    if ($('.codemirror-editor-modal').is(':visible')) {
                        return;
                    }
                    $scope.state.addingItem = false;
                    $(".new-item-zone", element).off("click.checklistAddNewItem");
                    $("html").off("click.checklistAddNewItem");
                };

                $scope.addNewItem = function() {
                    if ($scope.state.newItemText.length == 0) return;

                    $scope.checklist.items.push({
                        text : $scope.state.newItemText,
                        createdOn : new Date().getTime(),
                        createdBy : $rootScope.appConfig.login
                    });
                    $scope.$emit("checklistEdited", "add");
                    $scope.state.newItemText = "";
                };

                $scope.$watch("checklist", function(nv){
                    if (nv && nv.$newlyCreated) {
                        $scope.enterAddItem();
                        nv.$newlyCreated = false;
                    }
                });

                $scope.$on("$destroy", function(){
                    // noop
                })
            }
        }
    });


app.directive("sparkOverrideConfig", function($rootScope){
    return { // task = holder of sparkPreparedDFStorageLevel, MLTask or MLLib recipe desc
        scope : { config: '=', task: '=', taskType: '@' },
        templateUrl : '/templates/widgets/spark-override-config.html',
        link : function($scope) {
            $scope.rootAppConfig = $rootScope.appConfig;
            /* Initialize with first Spark conf */
            $scope.$watch("config", function(nv) {
                if (nv && nv.inheritConf == null) {
                    if ($rootScope.appConfig.sparkExecutionConfigs.length) {
                        nv.inheritConf = $rootScope.appConfig.sparkExecutionConfigs[0];
                    }
                }
            });
        }
    }
})

// (!) DEPRECATED: prefer using dss-slider (an Angular component) or its downgraded version.
app.directive("dkuSlider", function($window, $timeout){
    return {
        scope : {
            min : '=',
            max : '=',
            value : '=',
            nbDecimalPlaces : '=?',
            onChange: '&?'
        },
        link: function($scope, elem) {
            $scope.sliding = false;

            $scope.startSliding = function($event) {
                $scope.computeNewCursorPosition($event);
                $scope.sliding = true;
                $($window).on('mouseup', $scope.stopSliding);
                $($window).on('mousemove', $scope.slideCursor);
            }

            $scope.slideCursor = function($event) {
                $event.preventDefault(); //useful to avoid selecting content on vertical mouse move while sliding
                if ($scope.sliding) {
                    $scope.computeNewCursorPosition($event);
                }
            }

            $scope.stopSliding = function() {
                $scope.sliding = false;
                $($window).off('mouseup', $scope.stopSliding);
                $($window).off('mousemove', $scope.slideCursor);
                $scope.value = $scope.cursorValue;
                if ($scope.onChange) {
                    $scope.onChange();
                }
                $timeout(function(){$scope.value = $scope.cursorValue;});
            }


            $scope.computeNewCursorPosition = function($event) {
                var sliderWidth = $(elem).width() - $(elem).find('.cursor').width() - 1;
                var sliderXPosition = $(elem).offset().left;
                var xPosition = $event.pageX - sliderXPosition;
                if (xPosition < 0) {
                    xPosition = 0;
                }
                if (xPosition > sliderWidth) {
                    xPosition = sliderWidth;
                }
                $(elem).find('.cursor').css('left',xPosition + 'px');
                $scope.computeNewScopeValue(xPosition, sliderWidth);
                $scope.fixExtremityVisibility();
            }

            $scope.roundAccordingToNbDecimalPlaces = function(num) {
                if (typeof($scope.nbDecimalPlaces)==='undefined') {
                    $scope.nbDecimalPlaces = 0;
                }
                return Number(Math.round(num + "e+"+$scope.nbDecimalPlaces) + "e-"+$scope.nbDecimalPlaces);
            }

            $scope.computeNewScopeValue = function(xPosition, sliderWidth) {
                var range = $scope.max - $scope.min;
                $scope.cursorValue = $scope.roundAccordingToNbDecimalPlaces(range*xPosition/sliderWidth + $scope.min);
                $timeout(() => $scope.$apply());
            }

            $scope.initCursorPosition = function() {
                if (_.isNil($scope.value)) {
                    $scope.value = $scope.roundAccordingToNbDecimalPlaces(($scope.max - $scope.min)/2 + $scope.min);
                }
                $scope.cursorValue = $scope.value;
                var range = $scope.max - $scope.min;
                var xPosition = ($scope.value - $scope.min)*100/range;
                $(elem).find(".cursor").css("left", "auto");
                $(elem).find(".cursor").css("right", "auto");
                if (xPosition<50) {
                    $(elem).find('.cursor').css('left',xPosition + '%');
                } else {
                    $(elem).find('.cursor').css('right',100-xPosition + '%');
                }
                const maxLabelOpacity = xPosition > 90 ? 0 : 1;
                const minLabelOpacity = xPosition < 10 ? 0 : 1;
                $(elem).find('.range-max').css('opacity', maxLabelOpacity);
                $(elem).find('.range-min').css('opacity', minLabelOpacity);

                //  Mandatory to update cursor value when component is visible at two different places
                $timeout(() => $scope.$apply());
            }

            $timeout(function() {
                $scope.initCursorPosition();
            });

            $scope.$watch("value", function(nv) {
                if (_.isNil(nv)) return;
                $scope.initCursorPosition();
            });

            $scope.fixExtremityVisibility = function() {
                var cursorValueLeft = $(elem).find('.cursor-value').offset().left;
                var cursorValueRight = $(elem).find('.cursor-value').offset().left + $(elem).find('.cursor-value').width();
                var rangeMinRight = $(elem).find('.range-min').offset().left + $(elem).find('.range-min').width();
                var rangeMaxLeft = $(elem).find('.range-max').offset().left;
                var confortableGap = 5;
                if (rangeMinRight + confortableGap > cursorValueLeft) {
                    $(elem).find('.range-min').stop(true, false).fadeTo(40, 0);
                } else {
                    $(elem).find('.range-min').stop(true, false).fadeTo(40, 1);
                }
                if (rangeMaxLeft - confortableGap < cursorValueRight) {
                    $(elem).find('.range-max').stop(true, false).fadeTo(40, 0);
                } else {
                    $(elem).find('.range-max').stop(true, false).fadeTo(40, 1);
                }
            }

        },
        templateUrl : '/templates/widgets/dku-slider.html'
    }
});


app.directive("dkuArrowSlider",['$timeout', function($timeout) {
    return {
        restrict: "A",

        link: function(scope, elem, attrs) {

          /*
           * Inner variables
           */

            let frameSelector = attrs.frameSelector;
            let sliderSelector = attrs.sliderSelector;
            let slidingElementsSelector = sliderSelector + ' > *:visible';

            let minOffsetLeft = 0;
            let maxOffsetRight = 0;

            let modelListName = attrs.modelListName;

            scope.canSlideRightFlag = false;
            scope.canSlideLeftFlag = false;

          /*
           * Watchers / init
           */

            scope.$watch(modelListName, function(nv, ov) {
                let nvLength = nv ? nv.length : 0;
                let ovLength = ov ? ov.length : 0;
                //$timeout is needed to make sure slideToEnd is called after the currently processing $digest is done,
                //ie once the ng-repeat refresh is done and the new chip has been added
                $timeout(function() {
                    scope.computeNeedSlider();
                    if (scope.needSlider()) {
                        scope.initArrowSlider();
                    }
                    //$timeout to make sure arrow slider initialization is done (otherwise positioning computations may be off)
                    $timeout(function() {
                        if (nvLength < ovLength) {
                            if (!scope.needSlider()) {
                                slideToBegining();
                                removeArrowSliderStyle();
                            } else if (!isLastChipBeyondSliderBottom()) {
                                slideToEnd();
                            }
                        } else if (nvLength > ovLength) {
                            if (scope.needSlider()) {
                                slideToEnd();
                            } else {
                                scope.$broadcast('DKU_ARROW_SLIDER:animation_over');
                            }
                        }
                    }, 0);
                }, 0, false);
            }, true);

            scope.onResize = function() {
                scope.computeNeedSlider();
                if (!scope.needSlider()) {
                    slideToBegining();
                } else {
                    initOffsetsExtremas();
                    if (!isLastChipBeyondSliderBottom()) {
                        slideToEnd();
                    }
                    setCanSlideTags(scope.canSlideLeft(), scope.canSlideRight());
                }
                $timeout(function() {
                    scope.$apply();
                });
            }

            let loop;
            function resizeHandler() {
                clearTimeout(loop);
                loop = setTimeout(scope.onResize, 30);   //so that resize callback is called only once resize is done for good
            }

            $(window).on("resize.dkuArrowSlider", resizeHandler);
            scope.$on("$destroy", function(){
                $(window).off("resize.dkuArrowSlider", resizeHandler);
            });

            scope.$on("slideToId", function(event, frameSelectorAttr, sliderSelectorAttr, targetId) {
                if (frameSelector == frameSelectorAttr && sliderSelector == sliderSelectorAttr) {
                    slideToId(targetId);
                }
            });

            scope.initArrowSlider = function () {
                if (scope.needSlider()) {
                    $(frameSelector).css('position', 'relative');
                    $(sliderSelector).css('position', 'absolute');
                    $(sliderSelector).css('will-change', 'left');
                    $(sliderSelector).css('transition', 'left 150ms ease-out, right 150ms ease-out');
                    if (isNaN($(sliderSelector).css('left').replace('px', ''))) {
                        $(sliderSelector).css('left', '0px');
                        setCanSlideTags(false, true);
                    }
                    initOffsetsExtremas();
                    $timeout(function() {
                        scope.$broadcast("DKU_ARROW_SLIDER:arrow_slider_initialized");
                    });
                }
            };

           function removeArrowSliderStyle() {
                $(frameSelector).removeAttr("style");
                $(sliderSelector).removeAttr("style");
            }

            function initOffsetsExtremas() {
                minOffsetLeft = $(frameSelector).offset().left;
                maxOffsetRight = minOffsetLeft + $(frameSelector).width();
            }

            /*
            * Animation functions
            */

            scope.slideLeft = function() {
                let lastHiddenElement;
                let slidingElement = $(slidingElementsSelector);
                if (!slidingElement) return;
                for (let i = 0; i < slidingElement.length; i++) {
                    let elem = slidingElement[i];
                    if (!isElementVisible(elem)) {
                        lastHiddenElement = elem;
                    } else {
                        //if the element we wanna display is not the last one, make sure user can see there is more to come
                        if (i-1 > 0) {
                            let newSlidingElementLeft = getLeftAsNumber(sliderSelector) + getHiddenWidth(lastHiddenElement) + 20;
                            animatedSlide(newSlidingElementLeft);
                            setCanSlideTags(true, true);
                        } else {
                            slideToBegining();
                        }
                        break;
                    }
                }
            };

            scope.slideRight = function() {
                let lastHiddenElement;
                let slidingElement = $(slidingElementsSelector);
                if (!slidingElement) return;
                for (let i = slidingElement.length - 1; i >= 0; i--) {
                    let elem = slidingElement[i];
                    if (!isElementVisible(elem)) {
                        lastHiddenElement = elem;
                    } else {
                        //if the element we wanna display is not the last one, make sure user can see there is more to come
                        if (i + 1 < slidingElement.length - 1) {
                            let newSlidingElementLeft = getLeftAsNumber(sliderSelector) - getHiddenWidth(lastHiddenElement) - 20;
                            animatedSlide(newSlidingElementLeft);
                            setCanSlideTags(true, true);
                        } else {
                            slideToEnd();
                        }
                        break;
                    }
                }
            };

            function slideToBegining() {
                animatedSlide(0);
                setCanSlideTags(false, true);
            }

            function slideToEnd() {
                let newSlidingElementLeft = - (getTrueSliderWidth() - $(frameSelector).width()) -1 ;
                animatedSlide(newSlidingElementLeft);
                setCanSlideTags(true, false);
            }

            function slideToId(id) {
                let targetElementSelector = sliderSelector + ' > [id="'+ id +'"]:visible';
                if ($(targetElementSelector).length > 0 && !isElementVisible(targetElementSelector)) {
                    let targetElementOffsetLeft = $(targetElementSelector).offset().left;
                    let sliderOffsetLeft = $(sliderSelector).offset().left;

                    let sliderWidth = getTrueSliderWidth();
                    let frameWidth = $(frameSelector).width();

                    let targetElementPosition = targetElementOffsetLeft - sliderOffsetLeft - 20;
                    let widthAfterTargetElement =  sliderWidth - targetElementPosition;

                    if ($(targetElementSelector).is(':first-child')) {
                        slideToBegining();
                    } else if (widthAfterTargetElement > frameWidth) {
                        animatedSlide(-targetElementPosition, true);
                    } else {
                        slideToEnd();
                    }
                } else {
                    scope.$broadcast('DKU_ARROW_SLIDER:animation_over');
                }
            }

            function animatedSlide(newPosition, checkCanSlide) {
                $(sliderSelector).on('transitionend', function() {
                    scope.$broadcast('DKU_ARROW_SLIDER:animation_over');
                    if (checkCanSlide) {
                        setCanSlideTags(scope.canSlideLeft(), scope.canSlideRight());
                    }
                    $(sliderSelector).off('transitionend');
                });
                $(sliderSelector).css('left', newPosition + 'px');
            }

          /*
           * Double Click Handler
           */

            function dblClickHandler(counter, timer, clickFunc, dblClickFunc) {
                return function() {
                    if (counter <= 2) {
                        counter++;
                    }
                    if (counter == 1) {
                        clickFunc();
                        timer = $timeout(function(){
                            counter = 0;
                        }, 150);
                    }
                    if (counter == 2) {
                        dblClickFunc();
                        $timeout.cancel(timer);
                        counter = 0;
                    }
                };
            }

            let leftClickCounter = 0;
            let leftClickTimer;
            scope.slideLeftClickHandler = dblClickHandler(leftClickCounter, leftClickTimer, scope.slideLeft, slideToBegining);

            let rightClickCounter = 0;
            let rightClickTimer;
            scope.slideRightClickHandler = dblClickHandler(rightClickCounter, rightClickTimer, scope.slideRight, slideToEnd);

            /*
             * Checking if sliding is needed / possible functions
             */

            function neededDomElementsExist () {
                return $(sliderSelector).length > 0 && $(slidingElementsSelector).length > 0;
            }

            let isNeedSlider = false;
            scope.computeNeedSlider = function() {
                isNeedSlider =  neededDomElementsExist() && getTrueSliderWidth() > $(frameSelector).width();
            };

            scope.needSlider = function() {
                return isNeedSlider;
            };

            scope.canSlideRight = function() {
                return scope.needSlider() && !isElementVisible($(slidingElementsSelector)[$(slidingElementsSelector).length - 1]);
            };

            scope.canSlideLeft = function() {
                return scope.needSlider() && !isElementVisible($(slidingElementsSelector)[0]);
            };

            function setCanSlideTags(canSlideLeftFlag, canSlideRightFlag) {
                scope.canSlideLeftFlag = canSlideLeftFlag;
                scope.canSlideRightFlag = canSlideRightFlag;
            }

            /*
             * Private visual computing helpers
             */

            function isElementVisible(elem) {
                elem = $(elem);
                if (elem.length > 0) {
                    var elemOffsetLeft = elem.offset().left;
                    var elemOffsetRight = elemOffsetLeft + elem.innerWidth();
                    return !scope.needSlider() || (minOffsetLeft <= elemOffsetLeft && elemOffsetRight <= maxOffsetRight);
                }
            }

            function isLastChipBeyondSliderBottom() {
                var lastEl = $(slidingElementsSelector)[$(slidingElementsSelector).length - 1];
                var lastElRightOffset = $(lastEl).offset().left + $(lastEl).outerWidth();
                return lastElRightOffset >= maxOffsetRight;
            }

            function getTrueSliderWidth() {
                var maxIndex = $(slidingElementsSelector).length - 1;
                return $($(slidingElementsSelector)[maxIndex]).offset().left + $($(slidingElementsSelector)[maxIndex]).outerWidth() - $($(slidingElementsSelector)[0]).offset().left;
            }

            function getHiddenWidth(elem) {
                var elemOffsetLeft = $(elem).offset().left;
                var elemOffsetRight = elemOffsetLeft + $(elem).outerWidth();

                if (elemOffsetLeft < minOffsetLeft) {
                   return  minOffsetLeft - elemOffsetLeft;
                } else if (maxOffsetRight < elemOffsetRight) {
                    return elemOffsetRight - maxOffsetRight;
                }
                return 0;
            }

            function getLeftAsNumber(elem) {
                var left = $(elem).css('left');
                left = left.replace('px', '');
                if (!isNaN(left)) {
                    return parseInt(left);
                }
                return 0;
            }

      /*
           * Initialisation
           */
            $timeout(function() {
                scope.computeNeedSlider();
                if (scope.needSlider()) {
                    scope.initArrowSlider();
                }
            }, 0);

        }
    };
}]);


app.directive("uiCheckbox", function() {
    return {
        scope: {},
        require: "ngModel",
        restrict: "A",
        replace: "true",
        template: "<button type=\"button\"  ng-class=\"{'chkbox-btn-normal' : true, 'btn' : true, 'checked': checked===true}\">" +
            "<i ng-class=\"{'dku-icon-checkmark-12': checked===true}\"></span>" +
            "</button>",
        link: function(scope, elem, attrs, modelCtrl) {
            scope.size = "default";
            // Default Button Styling
            scope.stylebtn = {};
            // Default Checkmark Styling
            scope.styleicon = {"width": "10px", "left": "-1px"};
            // If size is undefined, Checkbox has normal size (Bootstrap 'xs')
            if(attrs.large !== undefined) {
                scope.size = "large";
                scope.stylebtn = {"padding-top": "2px", "padding-bottom": "2px", "height": "30px"};
                scope.styleicon = {"width": "8px", "left": "-5px", "font-size": "17px"};
            }
            if(attrs.larger !== undefined) {
                scope.size = "larger";
                scope.stylebtn = {"padding-top": "2px", "padding-bottom": "2px", "height": "34px"};
                scope.styleicon = {"width": "8px", "left": "-8px", "font-size": "22px"};
            }
            if(attrs.largest !== undefined) {
                scope.size = "largest";
                scope.stylebtn = {"padding-top": "2px", "padding-bottom": "2px", "height": "45px"};
                scope.styleicon = {"width": "11px", "left": "-11px", "font-size": "30px"};
            }

            var trueValue = true;
            var falseValue = false;

            // If defined set true value
            if(attrs.ngTrueValue !== undefined) {
                trueValue = attrs.ngTrueValue;
            }
            // If defined set false value
            if(attrs.ngFalseValue !== undefined) {
                falseValue = attrs.ngFalseValue;
            }

            // Check if name attribute is set and if so add it to the DOM element
            if(scope.name !== undefined) {
                elem.name = scope.name;
            }

            // Update element when model changes
            scope.$watch(function() {
                if(modelCtrl.$modelValue === trueValue || modelCtrl.$modelValue === true) {
                    modelCtrl.$setViewValue(trueValue);
                } else {
                    modelCtrl.$setViewValue(falseValue);
                }
                return modelCtrl.$modelValue;
            }, () => {
                scope.checked = modelCtrl.$modelValue === trueValue;
            }, true);

            // On click swap value and trigger onChange function
            elem.bind("click", function() {
                scope.$apply(function() {
                    if(modelCtrl.$modelValue === falseValue) {
                        modelCtrl.$setViewValue(trueValue);
                    } else {
                        modelCtrl.$setViewValue(falseValue);
                    }
                });
            });
        }
    };
});

const EDITABLE_LIST_ITEM_PREFIX = 'it'; // this is the property name used for items in the editableList* directives

/**
 * This directive is a fork of list-form that implements the new editable lists specifications.
 *
 * @param {Array}       ngModel                         - The list to bind to display.
 * @param {string}      [label]                         - Text to display when referring to this component.
 * @param {boolean}     [sortable=false]                - True to make the list sortable. Allows to rearrange list order by drag-and-dropping.
 * @param {Function}    [onAdd]                         - The function called when adding an item.
 * @param {Function}    [addPosition]                   - An optional function to provide the position for new item insertion.
 * @param {Function}    [onRemove]                      - The function called when removing an item.
 * @param {Function}    [onChange]                      - Callback called on change.
 * @param {boolean}     [noChangeOnAdd=false]           - True to prevent the callback onChange to be called when an item is added.
 * @param {boolean}     [required=false]                - Can the items of the list be empty. Used with the 'editable-list-input' component.
 * @param {Object}      [template]                      - Template of the items in the list. Used when the list items are objects.
 * @param {Function}    [prepare]                       - The function called on list update, to set items default value.
 * @param {Function}    [transcope]                     - Functions/objects to pass to the editableList scope.
 * @param {Array}       [suggests]                      - List of possible values of an item of the list. Can be displayed in a dropdown under a text input for instance.
 * @param {boolean}     [hasDivider=true]               - False to hide the divider line between items.
 * @param {string}      addLabel                        - Text to display in the add button; Optional if disableAdd is true.
 * @param {boolean}     [disableAdd=false]              - True to hide the Add button.
 * @param {boolean}     [disableRemove=false]           - True to hide the Remove buttons.
 * @param {boolean}     [enableEdit=false]              - True to show the Edit buttons.
 * @param {(item: list-item) => Promise<falsy | list-item>} [editItem]
 *                                                      - The callback called when editing an item. Parameter is the edited item, must return a promise, if resolved value is falsy, edit is canceled, otherwise must return the modified item.
 * @param {boolean}     [disableCreateOnEnter=false]    - True to prevent creating a new item when pressing Enter in the last focused item of the list. Focus the first item of the list instead.
 * @param {boolean}     [skipToNextFocusable=false]     - True to focus the next focusable item on Enter if the immediate next item can't be focused (e.g. deleted item).
 * @param {boolean}     [fullWidthList=false]           - True to make the list fill the full width available in the container.

 */
app.directive('editableList', function($timeout) { return {
    restrict: 'E',
    transclude: true, replace: true,
    templateUrl: '/templates/widgets/editable-list.html',
    require: '?ngModel',
    scope: {
        label: '@',
        ngModel: '<',
        sortable: '=',
        onAdd: '<',
        addPosition: '<?',
        onRemove: '<',
        onChange: '=',
        noChangeOnAdd: '<',
        required: '<',
        template: '=',
        prepare: '=',
        transcope: '=',
        suggests: '=',
        hasDivider: '<',
        addLabel: '@',
        disableAdd: '<',
        disableRemove: '<',
        enableEdit: '<',
        editItem: '<',
        disableCreateOnEnter: '<',
        skipToNextFocusable: '<',
        fullWidthList: '<',
        keyPlaceholder: '@',
        valuePlaceholder: '@'
    },
    compile: function(elt, attrs, transclude) {
        const ITEMS_CLASSNAME = 'editable-list__items';
        const ITEM_CLASSNAME = 'editable-list__item';
        const DIVIDER_CLASSNAME = 'editable-list__item--divider';
        const ICON_CONTAINER_CLASSNAME = 'editable-list__icon';
        const DRAG_ICON_CLASSNAME = 'editable-list__drag-icon';
        const DRAG_ICON_QA_SELECTOR = 'data-qa-editable-list-drag';
        const DELETE_BUTTON_CLASSNAME = 'editable-list__delete';
        const EDIT_BUTTON_CLASSNAME = 'editable-list__edit';
        const DELETE_BUTTON_QA_SELECTOR = 'data-qa-editable-list-delete';
        const EDIT_BUTTON_QA_SELECTOR = 'data-qa-editable-list-edit';
        const ITEM_TEMPLATE_CLASSNAME = 'editable-list__template';
        const EDITING_CLASSNAME = 'editable-list__template--editing';

        var itemsExpr = attrs.ngModel,
            klass = attrs['class'],
            focusableInputs = ['input:not([type=checkbox])', 'textarea', 'select'];

        return function(scope, elt){
            var lis = []; // the LIs

            if (klass) { // report CSS classes
                elt.className += ' ' + klass;
            }

            var insertTranscope = function(into) {
                if (typeof scope.transcope === 'object') {
                    for (var k in scope.transcope) {
                        into[k] = scope.transcope[k];
                    }
                }
            };

            insertTranscope(scope);
            scope.ngModel = [];

            scope.$parent.$watch(itemsExpr, function(v) {
                scope.ngModel = v || [];
            });

            // default hasDivider to true
            if (!angular.isDefined(scope.hasDivider)) {
                scope.hasDivider = true;
            }

            // Utilities
            function parentOf(child, className) {
                while (child && !child.classList.contains(className)) {
                    child = child.parentElement;
                }
                return angular.element(child);
            }

            function templateOf(child) {
                return parentOf(child, ITEM_TEMPLATE_CLASSNAME);
            }

            function liOf(child) {
                return parentOf(child, ITEM_CLASSNAME);
            }

            function indexOf(li) {
                for (var i = 0; i < lis.length; i++) {
                    if (lis[i].element[0] === li) return i;
                }
                // cond always true, prevent error w/ CodePen loop
                if (i || !lis.length) return -1;
            }

            function prepare(it) {
                if (scope.prepare) {
                    scope.prepare(it);
                }
            }

            function template() {
                switch(typeof scope.template) {
                    case 'function': return scope.template();
                    case 'object': return angular.extend({}, scope.template);
                    case 'string' : return scope.template;
                    default: return {};
                }
            }

            function regularEnter(evt) {  // press button, return in textarea...
                return evt.target.tagName.toLowerCase() !== 'input'
                        || evt.target.type === 'button';
            }

            // Edit entry and update list
            scope.edit = function(i) {
                if (scope.enableEdit && angular.isFunction(scope.editItem)) {
                    const itemToEdit = scope.ngModel[i];
                    scope.editItem(itemToEdit).then((newItem) => {
                        if (!newItem) return;
                        scope.ngModel[i] = newItem;
                        update(scope.ngModel, false);
                        scope.$parent.$apply();
                    });
                }
            };

            // Remove & update DOM
            scope.remove = function(i) {
                const removedElt = scope.ngModel.splice(i, 1)[0];
                update(scope.ngModel, false);
                if (!scope.disableRemove) {
                    scope.$parent.$apply();
                }
                scope.onRemove && scope.onRemove(removedElt);
            };

            var changing = false;

            function updateSuggests() {
                for (var i = lis.length; i-- > 0;) {
                    lis[i].scope.suggests = scope.suggests;
                }
            }

            function update(items, preventOnChange) {
                var change = !changing && scope.onChange && !preventOnChange;
                
                changing = true;

                for (var i = lis.length; i-- > 0;) {
                    lis[i].element.remove();
                    lis[i].scope.$destroy();
                    lis.splice(i, 1);
                }

                for (i = items.length - 1; i >= 0; i--) {
                    var childScope = scope.$new(),
                        childLi = angular.element('<li class="' + ITEM_CLASSNAME + (scope.hasDivider ? ' ' + DIVIDER_CLASSNAME : '') + '"></li>'),
                        childDrag = angular.element('<div class="' + ICON_CONTAINER_CLASSNAME + '"><i ' + DRAG_ICON_QA_SELECTOR + ' class="' + DRAG_ICON_CLASSNAME +' dku-icon-dots-multiple-16"></i></div>'),
                        childEdit = angular.element('<button type="button" ' + EDIT_BUTTON_QA_SELECTOR + ' class="btn btn--secondary btn--text btn--dku-icon btn--icon ' + EDIT_BUTTON_CLASSNAME + ' " tabindex="-1"> <i class="dku-icon-edit-16"></i></button>'),
                        childDelete = angular.element('<button type="button" ' + DELETE_BUTTON_QA_SELECTOR + ' class="btn btn--text btn--danger btn--dku-icon btn--icon ' + DELETE_BUTTON_CLASSNAME + ' " tabindex="-1"> <i class="dku-icon-trash-16"></i></button>'),
                        childTemplate = angular.element('<div class="' + ITEM_TEMPLATE_CLASSNAME + '"></div>');

                    childScope[EDITABLE_LIST_ITEM_PREFIX] = items[i];
                    childScope.suggests = scope.suggests;
                    prepare(childScope[EDITABLE_LIST_ITEM_PREFIX]);
                    childScope.$index = i;
                    childDelete.click(scope.remove.bind(this, i));
                    childEdit.click(scope.edit.bind(this, i));

                    scope.sortable && childLi.append(childDrag);
                    childLi.append(childTemplate);
                    scope.enableEdit && childLi.append(childEdit);
                    !scope.disableRemove && childLi.append(childDelete);

                    transclude(childScope, function(clone) {
                        childTemplate.prepend(clone);
                    });

                    lis.unshift({ element: childLi, scope: childScope });
                    itemsContainerEl.prepend(childLi);
                }

                const children = itemsContainerEl.children();
                const lastChild = children[children.length - 1];
                itemsContainerEl[0].scrollTop = lastChild && lastChild.offsetTop || 0;

                if (change) {
                    scope.onChange(scope.ngModel);
                }

                changing = false;
            }

            if (scope.onChange) {
                // Use a jQuery event handler, not a DOM one, because we want
                // the .trigger("change") in the bs-typeahead to trigger this
                $(elt[0]).on('change', function(evt) {
                    function doIt(){
                        changing = true;
                        scope.onChange(scope.ngModel);
                        changing = false;
                    }

                    if (!changing) {
                        /* This is the same hack that we did to fix #1222.
                         * When you have a bs-typeahead, you have an non-empty field, then
                         * you chnage data to get a suggestion. Clicking on the suggestion
                         * will exit the input field, triggering a change event.
                         * However, the change event triggers before the click has its own actions:
                         * which is, changing the value of the input and triggering another
                         * "change" and "input" event.
                         * By delaying the taking into account of this, we leave time to the browser
                         * to process the click and to have it repercuted to the Angular model
                         */
                        var uglyBSHack = $(evt.target).attr("bs-typeahead") != null;
                        if (uglyBSHack) {
                            window.setTimeout(doIt, 150);
                        } else {
                            doIt();
                        }
                    }
                });
            }

            scope.$watchGroup(['ngModel', 'disableRemove'], function ([newNgModel]) {
                update(newNgModel, false);
            });

            scope.$watch('transcope', function () {
                insertTranscope(scope);
            });

            if (scope.suggests) { scope.$watch('suggests', updateSuggests, true); }

            // Editing row, focus & blur
            var editing = null;

            function edit(li) {
                if (editing === li) return;
                if (editing) editing.removeClass(EDITING_CLASSNAME);
                editing = li;
                if (editing) {
                    editing.addClass(EDITING_CLASSNAME);
                }
            }

            elt[0].addEventListener('focus', function(evt) {
                if (focusableInputs.indexOf(evt.target.tagName.toLowerCase()) >= 0) {
                    edit(templateOf(evt.target));
                    evt.target.select();
                }
            }, true);

            elt[0].addEventListener('blur', function(evt) {
                if (focusableInputs.indexOf(evt.target.tagName.toLowerCase()) >= 0) {
                    edit(null);
                    window.getSelection().removeAllRanges();
                }
            }, true);

            function skipToNextFocusable(next) {
                let nextElement = lis[next].element[0];
                let focusable = nextElement.querySelector(focusableInputs.join(', '));
                while (next > -1 && !focusable) {
                    next = indexOf(nextElement.nextSibling);
                    if (next < 0) {
                        break;
                    }
                    nextElement = lis[next].element[0];
                    focusable = nextElement.querySelector(focusableInputs.join(', '));
                }
                return next;
            }

            elt.on('keydown', function(evt) {
                var next = null;
                switch (evt.keyCode) {
                    case 27:
                        evt.target.blur();
                        return true;
                    case 13:
                         if (regularEnter(evt)) return true;
                         evt.target.blur();
                         next = indexOf(templateOf(evt.target)[0].parentElement.nextSibling);
                         break;
                    default:
                        return true;
                }
                if (scope.skipToNextFocusable && next > -1) {
                    next = skipToNextFocusable(next);
                }
                next = scope.disableCreateOnEnter && next != null && next < 0 ? 0 : next;
                if (next > -1) {
                    const nextElement = lis[next].element[0];
                    const focusable = nextElement.querySelector(focusableInputs.join(', '));
                    if (focusable) focusable.focus();
                } else {
                    scope.add();
                }

                evt.preventDefault();
                evt.stopPropagation();
                return false;
            });

            var itemToAdd = template();

            prepare(itemToAdd);

            var deregWatchPrepare = scope.$watch('prepare', function() {
                if (scope.prepare) {
                    prepare(itemToAdd);
                    deregWatchPrepare();
                }
            });

            const itemsContainerEl = elt.find('.' + ITEMS_CLASSNAME);

            scope.add = function() {
                itemToAdd = template();
                prepare(itemToAdd);

                let addAtIndex = scope.addPosition ? scope.addPosition() : lis.length;
                if (addAtIndex < 0 || addAtIndex > lis.length) {
                    //to recover from errors in addPosition() function
                    addAtIndex = lis.length;
                }

                scope.ngModel.splice(addAtIndex, 0, itemToAdd);
                scope.hasAddedItem = true;
                update(scope.ngModel, scope.noChangeOnAdd);
                let addedElement = lis[addAtIndex].element[0];

                $timeout(function() {
                    const focusable = addedElement.querySelector(focusableInputs.join(', '));
                    if (focusable) focusable.focus();
                });
                scope.onAdd && scope.onAdd();
            }

            // Drag / drop
            if (!scope.sortable) return;

            elt.addClass('editablie-list--sortable');

            var dragging = null, draggingIndex = null, draggingOpacityTimeout = null;

            // Only allow dragging on handles
            elt.on('mouseover', function(evt) {
                if (evt.target.classList.contains(DRAG_ICON_CLASSNAME)) {
                    liOf(evt.target).prop('draggable', true);
                }
            });

            elt.on('mouseout', function(evt) {
                if (evt.target.classList.contains(DRAG_ICON_CLASSNAME) && !dragging) {
                    liOf(evt.target).prop('draggable', false);
                }
            });

            // Actual drag/drop code
            elt.on('dragstart', function(evt) {
                (evt.originalEvent || evt).dataTransfer.setData('text/plain', null);
                (evt.originalEvent || evt).dataTransfer.effectAllowed = 'move';
                dragging = liOf(evt.target)[0];
                draggingIndex = indexOf(dragging);
                itemsContainerEl.addClass('dragging');
                evt.target.classList.add('dragging');
                draggingOpacityTimeout = window.setTimeout(function() {
                    dragging.style.opacity = 0;
                }, 200); // later to let time for snapshot
            });

            elt.on('dragenter', function(evt) {
                if (!dragging || evt.target === elt[0]) return;
                var li = liOf(evt.target)[0];
                if (!li || li === dragging) return;
                li.classList.add(draggingIndex < indexOf(li) ? 'drag-below' : 'drag-above');
            });

            elt.on('dragleave', function(evt) {
                evt.target.classList.remove('drag-above', 'drag-below');
            });

            elt.on('dragover', function(evt){
                if (!dragging || evt.target === elt[0]) return;
                var li = liOf(evt.target)[0];
                if (!li || li === dragging) return;
                evt.preventDefault();
                (evt.originalEvent || evt).dataTransfer.dropEffect = 'move';
            });

            elt.on('drop', function(evt) {
                if (!dragging) return;
                evt.preventDefault();
                const dropIndex = indexOf(evt.target), dragIndex = draggingIndex;
                const itemsContainer = elt.find('.' + ITEMS_CLASSNAME)[0];
                if (dropIndex > draggingIndex) { // insert after
                    itemsContainer.insertBefore(dragging, evt.target.nextSibling);
                } else { // insert before
                    itemsContainer.insertBefore(dragging, evt.target);
                }
                dragEnd();
                scope.$apply(function() {
                    scope.ngModel.splice(dropIndex, 0, scope.ngModel.splice(dragIndex, 1)[0]);
                    update(scope.ngModel, false);
                });
            });

            elt.on('dragend', dragEnd);

            function dragEnd() {
                dragging.style.opacity = 1;
                itemsContainerEl.removeClass('dragging');
                dragging.classList.remove('dragging');
                if (draggingOpacityTimeout != null) {
                    window.clearTimeout(draggingOpacityTimeout);
                }
                dragging = null;
                draggingIndex = null;
                draggingOpacityTimeout = null;
                elt.find('.drag-above').removeClass('drag-above');
                elt.find('.drag-below').removeClass('drag-below');
            }
        };
    }
}; });


app.directive('checkCategoryNameUnique', function() {
    return {
        require: 'ngModel',
        scope: false,
        compile: function() {
            return function(scope, _elt, _attrs, ngModel) {
                const index = scope.$index;

                function checkUnique(nv) {
                    const isUnique = !scope.generalSettings.globalTagsCategories.find((it, idx) => it.name == nv[index].name && idx != index);
                    ngModel.$setValidity('unique', isUnique);
                    return nv;
                }

                scope.$watch('generalSettings.globalTagsCategories', checkUnique, true);
            }
        }
    }

});

app.directive('editableListInput', function ($parse, $timeout) {
    return {
        replace: true,
        require: '?ngModel',
        restrict: 'E',
        scope: {
            ngModel: '=',
            type: '@',
            onChange: '&',
            onKeyUpCallback: '&',
            placeholder: '@',
            required: '<',
            bsTypeahead: '=',
            classes: '@',
            unique: '<',
            trimParam: '<',
            checkWarning: '=', // needs to be a function that evaluates fast, it's not debounced
            disableFormValidation: '<', // if true, this item  being invalid doesn't make the parent form invalid
            pattern: '@',
            fullWidthInput: '<'
        },
        templateUrl: '/templates/widgets/editable-list-input.html',
        compile: function() {

            return function(scope, elt, attrs, ngModel) {
                const propertyToCompare = attrs.ngModel.split(EDITABLE_LIST_ITEM_PREFIX + '.')[1];
                var setItemValidity;
                var $elt = $(elt);

                function updateModel(evt) {
                    const localScope = angular.element(evt.target).scope();
                    if (localScope) {
                        $parse(attrs.ngModel).assign(scope.$parent, localScope.ngModel);
                    }
                    if (setItemValidity) {
                        setItemValidity();
                    }
                }

                scope.onKeyUp = function(evt) {
                    updateModel(evt);
                    setTimeout(() => $elt.trigger("keyup"));
                    if (scope.onKeyUpCallback) {
                        scope.onKeyUpCallback();
                    }
                };

                var editableListScope = scope.$parent.$parent;
                scope.parentListItems = editableListScope ? editableListScope.ngModel : [];
                const index = scope.$parent.$index;
                if (editableListScope.editableListForm) {
                    if (scope.disableFormValidation) {
                        editableListScope.editableListForm.$removeControl(ngModel.$$parentForm);
                    } else {
                        editableListScope.editableListForm.$addControl(ngModel.$$parentForm);
                    }
                }

                function checkUnique() {
                    if (!scope.ngModel) {
                        return;
                    }
                    const isUnique = !scope.parentListItems.find((it, idx) => resolveValue(it, propertyToCompare) === scope.ngModel && idx != index);
                    ngModel.$setValidity('unique', isUnique);
                }

                if (scope.unique || scope.required) {
                    setItemValidity = () => {
                        $timeout(() => scope.$parent[EDITABLE_LIST_ITEM_PREFIX].$invalid = ngModel.$$parentForm.$invalid);
                    }
                    setItemValidity();
                }

                //Since we delete and recreate the editableList items on update (add/remove) we loose information on which one have been touched/modified.
                //For validation we need this information to keep displaying the error message when recreated so we store it in the item
                //if parentScope contains different editableListInput we need to differentiate $touched
                scope.touchedId = `$touched.${attrs.ngModel}`;

                scope.onBlur = function() {
                    scope.$parent[EDITABLE_LIST_ITEM_PREFIX][scope.touchedId] = true;
                }

                // if an already saved input is empty/invalid we want to display the error message even if not touched
                if (!editableListScope.hasAddedItem) {
                    scope.onBlur();
                }

                if (scope.unique) {
                    scope.$watch('parentListItems', checkUnique, true);
                }

                let changing = false;
                $(elt[0]).on('change', function(evt) {
                    function doIt(){
                        changing = true;
                        updateModel(evt);
                        $elt.trigger("change");
                        changing = false;
                    }

                    if (!changing) {
                        /* This is the same hack that we did to fix #1222.
                         * When you have a bs-typeahead, you have an non-empty field, then
                         * you chnage data to get a suggestion. Clicking on the suggestion
                         * will exit the input field, triggering a change event.
                         * However, the change event triggers before the click has its own actions:
                         * which is, changing the value of the input and triggering another
                         * "change" and "input" event.
                         * By delaying the taking into account of this, we leave time to the browser
                         * to process the click and to have it repercuted to the Angular model
                         */
                        var uglyBSHack = $(evt.target).attr("bs-typeahead") != null;
                        if (uglyBSHack) {
                            window.setTimeout(doIt, 200);
                        } else {
                            doIt();
                        }
                    }
                });
            }
        }
    }
});

app.directive("timeZoneList", function(CachedAPICalls) {
    return {
        restrict : 'A',
        link : function(scope) {
            CachedAPICalls.timezonesList.then(function(timezonesList) {
                scope.timezone_ids = timezonesList.ids;
            }).catch(setErrorInScope.bind(this));
        }
    }
});

// Performance-oriented one-way binding
// Raw (unescaped) HTML only, no expression, must be updated explicily
// Must be bound in a map, e.g. {a: "Label", b: "<strong>error</strong>"}
app.directive('fastBind', function() {
    return {
        scope: false,
        priority: -1,
        link: function(scope, element, attrs) {
            var elts = [], keys = [], root = element[0];
            element.find('[fast-bound]').each(function(i){
                elts[i] = this;
                keys[i] = this.getAttribute('fast-bound');
            });
            scope[attrs.fastBind] = function(map) {
                if (!map) {
                    element[0].style.visibility = 'hidden';
                } else {
                    for (let i = elts.length - 1; i>=0; i--) {
                        elts[i].innerHTML = map[keys[i]];
                    }
                    root.style.visibility = 'visible';
                }
            };
        }
    };
});

app.directive("smartLogTail", function(){
    return {
        restrict : 'A',
        replace: true,
        scope : {
            smartLogTail : '=',
            emptyPlaceholder: '@',
        },
        template : '<pre class="smart-log-tail-content">'+
                '<span ng-if="smartLogTail.lines.length == 0" style="font-style: italic">{{ emptyPlaceholder }}</span>'+
                '<span ng-repeat="line in smartLogTail.lines track by $index" '+
                    'ng-class="{\'text-error\':  smartLogTail.status[$index] == TAIL_STATUS.ERROR,'+
                           '\'text-warning\': smartLogTail.status[$index] == TAIL_STATUS.WARNING,'+
                           '\'text-success\': smartLogTail.status[$index] == TAIL_STATUS.SUCCESS, }">'+
                    '{{line}}'+
                '</span>'+
                '</pre>',
        link : function(scope){
            scope.TAIL_STATUS = {
                DEBUG: 0,
                INFO: 1,
                WARNING: 2,
                ERROR: 3,
                SUCCESS: 4
            };
        }
    }
});

app.directive("automationEditOverlay", function(){
    return {
        replace:true,
        template : '<div ng-cloak ng-if="appConfig.isAutomation" class="automation-edit-overlay"><div class="text"><div class="line1">Automation node</div><div class="line2">Edits will be lost at next bundle import</div></div></div>',
    }
});

app.directive("infoMessagesList", function(translate){
    return {
        restrict : 'A',
        scope : {
            infoMessagesList : '='
        },
        template : '<ul class="info-messages-list"><li ng-repeat="message in infoMessagesList">'+
                    '<div ng-class="\'message-\' + (message.severity.toLowerCase())">'+
                        '<div ng-if="message.title && message.details">'+
                            '<h4 >{{message.title}}</h4>'+
                            '<span>{{message.details}}</span>'+
                            '<span ng-show="message.line">' + translate('RECIPE.STEP.INFO.AT_LINE', ' (at line {{line}})', { line: '{{message.line}}' }) +
                        '</div>'+
                        '<div ng-if="message.title && !message.details">'+
                            '<span>{{message.title}}</span>'+
                            '<span ng-show="message.line">' + translate('RECIPE.STEP.INFO.AT_LINE', ' (at line {{line}})', { line: '{{message.line}}' }) +
                        '</div>'+

                    '</div>'+
                '</li></ul>',
    }
});

app.directive('masterBreadcrumb', function($rootScope) {
    return {
        templateUrl: '/templates/master-breadcrumb.html',
        scope: true,
        link: function($scope) {
            $scope.breadcrumbData = $rootScope.masterBreadcrumbData;
        }
    }
});

app.service("InfoMessagesModal", function($q, CreateModalFromTemplate){
    var InfoMessagesModal = {
        /* Shows only if there is a message */
        showIfNeeded : function(parentScope, messages, modalTitle, subHeader = null) {

            if (messages.messages.length > 0) {
                return CreateModalFromTemplate("/templates/widgets/info-messages-display.html", parentScope, null, function(newScope){
                    newScope.modalTitle = modalTitle;
                    newScope.messages = messages;
                    newScope.subHeader = subHeader;
                });
            }
            return $q.resolve();
        }
    }
    return InfoMessagesModal;
});

app.directive("refreshCodemirrorOn", function($timeout){
    return {
        link : function($scope, element, attrs) {
            $scope.$watch(attrs.refreshCodemirrorOn, () => {
                $timeout(function(){
                    element.find(".CodeMirror").each(function(_, e) {
                        if (e.CodeMirror) e.CodeMirror.refresh();
                    });
                }, 0);
            });
        }
    }
})

app.filter("infoMessageAlertClass", function(){
    return function(input){
        var dict = {
            'ERROR': 'alert-danger',
            'WARNING': 'alert-warning',
            'INFO': 'alert-info',
            'SUCCESS': 'alert-success',
        };
        return dict[input.severity];
    }
})

app.filter("severityAlertClass", function(){
    return function(input){
        var dict = {
            'ERROR': 'alert-danger',
            'WARNING': 'alert-warning',
            'INFO': 'alert-info'
        };
        return input != null ? dict[input] : 'alert-info';
    }
})

app.directive("infoMessagesRawListWithAlert", function(){
    return {
        templateUrl : '/templates/widgets/info-messages-raw-list-with-alert.html',
        scope : {
            data : '=infoMessagesRawListWithAlert',
            showReduceAction: '='
        }
    }
})

app.directive("deployerInfoMessagesRawListWithAlert", function(){
    return {
        templateUrl : '/templates/widgets/deployer-info-messages-raw-list-with-alert.html',
        scope : {
            data : '=deployerInfoMessagesRawListWithAlert',
            showReduceAction: '=',
            reducedAtStartup: '='
        },
        link: function($scope) {
            $scope.$watchCollection("data.messages", () => {
                const messages = $scope.data.messages.slice().sort((m1, m2) => {
                    if (m1.severity === 'ERROR') return -1;
                    if (m2.severity === 'ERROR') return 1;
                    if (m1.severity === 'WARNING') return -1;
                    if (m2.severity === 'WARNING') return 1;
                    if (m1.severity === 'SUCCESS') return -1;
                    if (m2.severity === 'SUCCESS') return 1;
                    return 0;
                });

                $scope.data.$reduced = $scope.reducedAtStartup;

                $scope.getAlertSeverityClass = function() {
                    switch($scope.data.maxSeverity) {
                        case 'ERROR':
                            return 'alert-danger';
                        case 'WARNING':
                            return 'alert-warning';
                        case 'SUCCESS':
                            return 'alert-success';
                        case 'INFO':
                            return 'alert-info';
                    }
                }

                const nbError = messages.filter(m => m.severity === 'ERROR').length;
                const nbWarning = messages.filter(m => m.severity === 'WARNING').length;

                if (nbError && nbWarning) {
                    $scope.header = `${nbError} error${nbError > 1 ? 's' : ''} and ${nbWarning} warning${nbWarning > 1 ? 's' : ''} were encountered`;
                }
                else if (nbError) {
                    $scope.header = `${nbError} error${nbError > 1 ? 's were' : ' was'} encountered`;
                }
                else if (nbWarning) {
                    $scope.header = `${nbWarning} warning${nbWarning > 1 ? 's were' : ' was'} encountered`;
                }
                else if ($scope.data.messages.filter(m => m.severity === 'SUCCESS').length) {
                    $scope.header = "Success";
                }
                else {
                    $scope.header = "Information";
                }
            });
        }
    }
})

app.directive("infoMessagesRawList", function(){
    return {
        templateUrl : '/templates/widgets/info-messages-raw-list.html',
        scope : {
            data : '=infoMessagesRawList'
        }
    }
})

app.directive("featureLocked", function(){
    return {
        templateUrl : '/templates/widgets/feature-locked.html',
        restrict : 'EA',
        scope : {
            featureName : '='
        }
    }
});

app.directive("qrCode", function(){
    return {
        scope : {
            qrCode : '='
        },
        template : "<div class='qr'></div>",
        link : function($scope, element) {
            $scope.$watch("qrCode", function(nv) {
                if (nv) {
                    /* global QRCode */
                    new QRCode(element.find(".qr")[0], {
                        text : $scope.qrCode,
                        width: 128,
                        height: 128,
                        colorDark : "#000000",
                        colorLight : "#ffffff",
                        correctLevel : QRCode.CorrectLevel.H
                    });
                }
            });
        }
    }
});

app.directive("dkuFoldable", function(){
    return {
        scope : true,
        controller : ['$scope', '$attrs', function($scope, $attrs) {
            $scope.foldableOpen = $scope.$eval($attrs.open);
        }],
        link : function($scope, _element, attrs){
            $scope.foldableToggle = function(){
                $scope.foldableOpen = !$scope.foldableOpen;
            };
            function setChevronClazz(){
                $scope.foldableChevronClazz = $scope.foldableOpen ? "dku-icon-chevron-up-16" : "dku-icon-chevron-down-16";
            }
            $scope.$watch("foldableOpen", setChevronClazz);
            setChevronClazz();
            $scope.$watch(attrs.open, function(nv, ov){
                if (nv != ov) $scope.foldableToggle();
            });
        }
    }
});

app.directive("dkuFoldableRightPanel", function(LocalStorage, STANDARDIZED_SIDE_PANEL_KEY){
    return {
        scope : true,
        require : "dkuFoldable",
        link : function($scope, _element, attrs) {
            let objectType = $scope.objectType !== undefined ? $scope.objectType : "defaultObjectType";
            // Strip the LOCAL_ and FOREIGN_ prefix to have the same expanded/collapsed section in the Flow and in the actual object item view (ex: Dataset).
            if (objectType.startsWith("LOCAL_")) {
                objectType = objectType.substring(6);
            } else if (objectType.startsWith("FOREIGN_")) {
                objectType = objectType.substring(8);
            }
            const key = STANDARDIZED_SIDE_PANEL_KEY + '.' + objectType + '.' + attrs.name;
            let localValue = LocalStorage.get(key);
            if(localValue !== undefined){
                $scope.foldableOpen =  localValue;
            }
            $scope.$watch("foldableOpen", function(nv, ov){
                if (nv != ov) {
                    LocalStorage.set(key, nv);
                }
            });
        }
    }
});

/*
 * Add on for dkuFoldable
 */
app.directive("openOnDragEnter", function($timeout) {
    return {
        restrict: 'A',
        link : function($scope, element) {
            var previousState = null;
            var nesting = 0;

            element.on('dragenter', () => {
                if (nesting++ === 0) {
                    previousState = $scope.foldableOpen;
                    $scope.$apply(function() {
                        $scope.foldableOpen = true;
                    });
                }
            });
            element.on('dragleave', () => {
                if (--nesting === 0) {
                    $scope.$apply(function() {
                        $scope.foldableOpen = previousState;
                        previousState = null;
                    });
                }
            });
            $(document).on('dragend', () => {
                $timeout(function() { nesting = 0; });
            });
        }
    }
});

app.directive("rightColumnDescriptionTags", function(){
    return {
        scope : {
            object : '='
        },
        templateUrl : '/templates/widgets/right-column-description-tags.html'
    }
});

/*
 * Verry usefull to repeat template passed through transclude in a ng-repeat directive.
 * In the HTML template 'inject' directive call should be at the same level as 'ng-repeat' directive call
 */

app.directive('dkuInject', function(){
  return {
    link: function($scope, $element, $attrs, controller, $transclude) {
      if (!$transclude) {
        /* global minErr, startingTag */
        throw minErr('ngTransclude')('orphan',
         'Illegal use of ngTransclude directive in the template! ' +
         'No parent directive that requires a transclusion found. ' +
         'Element: {0}',
         startingTag($element));
      }
      var innerScope = $scope.$new();
      $transclude(innerScope, function(clone) {
        $element.empty();
        $element.append(clone);
        $element.on('$destroy', function() {
          innerScope.$destroy();
        });
      });
    }
  };
});

/*
 * Custom carousel : can take any html template and use it to populate the slides (makes use the dkuInject directive to combine transclude and ngRepeat)
 */

app.directive('dkuCarousel', function($timeout){
    return {
        transclude: true,
        templateUrl: '/templates/projects/dku-carousel.html',
        restrict: 'A',
        scope: {
            entries : '=dkuCarousel',
            initialIndex : '=?'
        },
        link: function($scope, element){

            $scope.element = element;

            $scope.index = 0;
            if ($scope.initialIndex) {
                $scope.index = $scope.initialIndex;
            }

            $scope.slideLeft = function() {
                if (!$scope.entries) return;
                var maxIndex = $scope.entries.length - 1;
                var newIndex = $scope.index - 1;
                if (newIndex < 0) {
                    newIndex = maxIndex;
                }
                slide(newIndex, -1);
            };

            $scope.slideRight = function() {
                if (!$scope.entries) return;
                var maxIndex = $scope.entries.length - 1;
                var newIndex = $scope.index + 1;
                if (newIndex > maxIndex) {
                    newIndex = 0;
                }
                slide(newIndex, 1);
            };

            var slide = function (newIndex, direction) {
                var slider = $(element).find('.slider');

                // In order to give the illusion the carousel is wrapping
                var firstSlideClone = $(element).find('.slide:first-child').clone().addClass('clone');
                var lastSlideClone = $(element).find('.slide:last-child').clone().addClass('clone');
                $(slider).prepend(lastSlideClone);
                $(slider).append(firstSlideClone);

                var slides = $(element).find('.slide');
                var domNbSlides = $scope.entries.length + 2;        //since we've juste added to new slides
                var domIndex = $scope.index + 1;    //since we've just preprended a new slide
                var newDomIndex = domIndex + 1 * direction;

                var leftPosition = -1 * domIndex * 100 / domNbSlides;
                var newLeftPosition = -1 * newDomIndex * 100 / domNbSlides;
                var sliderWidth = domNbSlides * 100;
                $(slider).addClass('animating');
                $(slider).css('width', sliderWidth + '%');
                $(slider).css('transform', 'translate(' + leftPosition + '%, 0)');
                $(slides).css('width', 100/domNbSlides + '%');

                $timeout(function() {
                    $(slider).addClass('transition');
                    $(slider).css('transform', 'translate(' + newLeftPosition + '%, 0)');
                }, 0);

                $timeout(function() {
                    $(slider).removeClass('transition');
                    $scope.index = newIndex;
                    $(slider).removeAttr('style');
                    $(slider).removeClass('animating');
                    $(slides).removeAttr('style');
                    firstSlideClone.remove();
                    lastSlideClone.remove();
                }, 200);
            }
        }
    };
});

app.directive('displayAmount', function(){
    return {
        template: '<span>{{amount}} {{unit}}<span ng-if="amount > 1">s</span></span>',
        restrict: 'AE',
        scope: {
            unit: '=',
            amount: '='
        },
        link: function(){
            // noop
        }
    };
});


app.directive('editableText', function($timeout){
    return {
        template: '<div class="dku-editable-text">' +
        '<div ng-show="!editing" ng-click="edit()" class="horizontal-flex">' +
        '<div class="flex mx-textellipsis">{{model || placeholder}}</div><div class="noflex"><i class="icon-pencil" /></div>' +
        '</div>' +
        '<input type="text" ng-model="model" placeholder="{{placeholder}}" ng-blur="editing = false" ng-show="editing" blur-on-enter />' +
        '</div>',
        restrict: 'A',
        scope: {
            model: '=editableText',
            placeholder: '='
        },

        link: function($scope, element){
            var input = $(element.find('input'));
            $scope.edit = function() {
                $scope.editing = true;
                $timeout(function() {
                    input.focus();
                });
            };
        }
    };
});

app.directive('ngRightClick', function($parse) {
    return function (scope, element, attrs) {
        var fn = $parse(attrs.ngRightClick);
        element.bind('contextmenu', function (event) {
            scope.$apply(function () {
                if (attrs.ngRightClickPreventDefault !== 'false') {
                    event.preventDefault();
                }
                fn(scope, {$event: event});
            });
        });
    };
});

// intercept clicks to allow preventing it based on a condition
// works for any mouse button
app.directive('checkClickable', function() {
    return {
        scope: { allowClick: '<checkClickable' },
        link: function(scope, element) {
            const maybePreventClick = (event) => {
                const allow = typeof scope.allowClick === 'function' ? scope.allowClick() : scope.allowClick;
                if(!allow) {
                    event.preventDefault();
                }
            };
            element.on('click', maybePreventClick);
            element.on('auxclick', maybePreventClick);
        }
    }
});

//TODO: use this factory in folder_edit.js instead of openMenu() when the two branchs will be merged together
app.factory("openDkuPopin", function($timeout, $compile) {
    //Args in options: template, isElsewhere, callback, popinPosition = "SMART", onDismiss, doNotCompile, arrow
    return function($scope, $event, options) {
        var opts = angular.extend({popinPosition: 'SMART'}, options);
        var newDOMElt = $(opts.template);

        var newScope = $scope.$new();
        if (!opts.doNotCompile) {
            $compile(newDOMElt)(newScope);
        }

        // By default an arrow is displayed if the popin position strategy is SMART
        if (!angular.isDefined(opts.arrow)) {
            opts.arrow = opts.popinPosition == 'SMART';
        }

        newScope.dismiss = function(skipOnDismiss){
            newScope.$destroy();
            if (typeof(opts.onDismiss) === "function" && !skipOnDismiss) {
                opts.onDismiss(newScope);
            }
        };

        newScope.$on("$destroy", () => $timeout(() => {
            newDOMElt.remove();
            $('body').off('click', hideOnClickElsewhere);
            $('body').off('contextmenu', hideOnClickElsewhere);
        }))

        var hideOnClickElsewhere = function(e) {
            if(typeof(opts.isElsewhere)==="function" && opts.isElsewhere(newDOMElt, e)) {
                newScope.dismiss();
            }
        };

        $timeout(function(){
            newScope.$apply(function() {
                $("body").append(newDOMElt);
                switch (opts.popinPosition) {
                    case "SMART":
                        smartPositionning();
                        break;
                    case "CLICK":
                        clickPositionning();
                        break;
                    default:
                        break;
                }
                newDOMElt.show();
                $('body').on('click', hideOnClickElsewhere);
                $('body').on('contextmenu', hideOnClickElsewhere);
                if (typeof(opts.callback) === "function") {
                    opts.callback(newScope);
                }
            });
        });


        // Positions the popin so that it is always aligned with one side of the element
        function smartPositionning() {
            var element_X = $($event.target).offset().left;
            var element_Y = $($event.target).offset().top;
            var element_W = $($event.target).outerWidth(true);
            var element_H = $($event.target).outerHeight(true);
            var popin_W = $(newDOMElt).outerWidth(true);
            var popin_H = $(newDOMElt).outerHeight(true);
            var window_W = window.innerWidth;
            var window_H = window.innerHeight;

            var popin_on_bottom = (element_Y + element_H + popin_H < window_H);
            var popin_aligned_on_right = (element_X + popin_W > window_W);

            var popin_X = (popin_aligned_on_right) ? (element_X + element_W - popin_W) : (element_X);
            var popin_Y = (popin_on_bottom) ? (element_Y + element_H) : (element_Y - popin_H);

            newDOMElt.css("top", popin_Y);
            newDOMElt.css("left", popin_X);

            // Add an arrow linking the popin to the element which triggered it
            if (opts.arrow) {
                var cssClass = "";
                cssClass += (popin_on_bottom) ? 'bottom-' : 'top-';
                cssClass += (popin_aligned_on_right) ? 'left' : 'right';
                newDOMElt.addClass(cssClass);
                newDOMElt.addClass(popin_on_bottom ? 'bottom' : 'top');
            }
        }

        // Positions the popin so that its content is displayed at the location of the mouse click
        function clickPositionning() {
            var mouse_X = $event.clientX;
            var mouse_Y = $event.clientY;
            var popin_W = $(newDOMElt).outerWidth(true);
            var popin_H = $(newDOMElt).outerHeight(true);
            var window_W = window.innerWidth;
            var window_H = window.innerHeight;

            var popin_on_bottom = (mouse_Y + popin_H < window_H);
            var popin_on_right = (mouse_X + popin_W < window_W);

            var popin_X = (popin_on_right) ? (mouse_X) : (mouse_X - popin_W);
            var popin_Y = (popin_on_bottom) ? (mouse_Y) : (mouse_Y - popin_H);

            newDOMElt.css("top", popin_Y);
            newDOMElt.css("left", popin_X);

            // Add an arrow linking the popin to the element which triggered it
            if (opts.arrow) {
                var cssClass = "";
                cssClass += (popin_on_bottom) ? 'bottom-' : 'top-';
                cssClass += (popin_on_right) ? 'right' : 'left';
                newDOMElt.addClass(cssClass);
            }
        }

        //returning function to remove popin
        return newScope.dismiss;
    }
});

app.factory('TreeViewSortableService', function() {
    let currentNode;
    return {
        setCurrent: (node) => {currentNode = node;},
        getCurrent: () => currentNode
    };
});

app.directive('treeView', function($timeout, openDkuPopin, TreeViewSortableService) {
    return {
        templateUrl: '/templates/widgets/tree-view-node.html',
        restrict: 'AE',
        scope: {
            nodes: '=treeView',
            rootNodes: '=?',
            depth: '<',
            nodeName: '<',
            onClick: '<',
            onArrowClick: '<',
            iconClass: '<',
            iconTitle: '<',
            rightIconClass: '<',
            rightIconTitle: '<',
            nodeClass: '<',
            showDragHandles: '<',
            scrollToNodeFn: '=?',
            getTaxonomyMassExpandCollapseStateFn: '=?',
            expandAllFn: '=?',
            collapseAllFn: '=?',
            setReduceFn: '=?',
            setUnfoldedNodeIdsFn: '<?',
            getUnfoldedNodeIdsFn: '<?',
            getNodeIdsHavingChildrenFn: '<?',
            getRightClickMenuTemplate: '<?',
            contextMenuFns: '=?',
            showContextMenu: '=?'
        },
        link: function($scope, $el) {
            if (!$scope.nodes) {
                throw new Error("No nodes provided to tree view");
            }


            $scope.depth = $scope.depth || 0;
            $scope.MARGIN_PER_DEPTH = 15;
            $scope.uiState = {};

            $scope.activateSortable = function() {
                const parentNode = ($scope.$parent && $scope.$parent.node || $scope.uiState);
                (TreeViewSortableService.getCurrent() || {}).$sortableEnabled = false;
                parentNode.$sortableEnabled = true;
                $scope.nodes.forEach(n => n.$reduced = true);
                TreeViewSortableService.setCurrent(parentNode);
            };

            // disabled dragstart event in order to disable native browser drag'n'drop and allow mouse up event when drag'n'dropping
            $timeout(() => $el.find('> ul > li > div.tree-view-node > div > span.handle-row').each((idx, el) => { el.ondragstart = () => false }));
            $scope.onSortableStop = function() {
                for (let i = 0; i < $scope.nodes.length; i++) {
                    delete $scope.nodes[i].$tempReduced;
                }
            };
            $scope.onNodeMouseDown = function(node) {
                node.$tempReduced = true;
            };
            $scope.treeViewSortableOptions = {
                axis:'y',
                cursor: 'move',
                cancel: '',
                handle: '.tree-view-drag-handle',
                update: $scope.onSortableStop
            };

            $scope.onNodeClick = function(node, evt) {
                if (!$(evt.originalEvent.explicitOriginalTarget).is('i.icon-reorder')) {
                    $scope.onClick(node);
                }
            };

            $scope.openContextMenu = function (node, $event) {
                if (!$scope.showContextMenu) {
                    return;
                }

                node.$rightClicked = true;

                let template = `<ul class="dropdown-menu" ng-click="popupDismiss()">`;
                switch ($scope.getNodeMassExpandCollapseStateFn(node.id)) { //NOSONAR
                    case "EXPAND_ALL":
                        template += `
                            <li class="dku-border-bottom">
                                <a href="#" ng-click="expandChildren('`+node.id+`')">
                                    <i class="dku-icon-chevron-double-down-16 icon-fixed-width" /> Expand all
                                </a>
                            </li>`;
                        break;
                    case "COLLAPSE_ALL":
                        template += `
                            <li class="dku-border-bottom">
                                <a href="#" ng-click="collapseChildren('`+node.id+`');">
                                    <i class="dku-icon-chevron-double-right-16 icon-fixed-width" /> Collapse children
                                </a>
                            </li>`;
                        break;
                    default:
                        break;
                }
                template += $scope.getRightClickMenuTemplate(node);

                template += `</ul>`;

                let isElsewhere = function (elt, e) {
                    let result = $(e.target).parents('.dropdown-menu').length == 0;
                    if (result) {
                        delete node.$rightClicked;
                    }
                    return result;
                };

                $scope.popupDismiss = openDkuPopin($scope, $event, {template:template, isElsewhere:isElsewhere, popinPosition:'CLICK'});
            };

            // Return an array containing all the parent nodes and the searched node itself
            function getAncestorsOfNodeId(nodeId, nodes = $scope.rootNodes, ancestors=[]) {
                let n = nodes.find(n => n.id == nodeId);

                if (angular.isDefined(n)) {
                    return ancestors.concat(n);
                } else {
                    for (let i=0; i<nodes.length; i++) {
                        let node = nodes[i];
                        if (node.children && node.children.length > 0) {
                            let r = getAncestorsOfNodeId(nodeId, node.children, ancestors.concat(node));
                            if (r) {
                                return r;
                            }
                        }
                    }
                    return null;
                }
            }

            // Return an array containing all the children nodes and the node itself
            function getDescendantsOfNode(node, descendants=[node]) {
                if (node.children.length > 0) {
                    for (let i=0; i<node.children.length; i++) {
                        let child = node.children[i];
                        let r = getDescendantsOfNode(child, [child]);
                        descendants = descendants.concat(r);
                    }
                }
                return descendants;
            }

            function getNodeFromNodeId(nodeId, nodes = $scope.rootNodes) {
                let n = nodes.find(n => n.id == nodeId);

                if (angular.isDefined(n)) {
                    return n;
                } else {
                    for (let i=0; i<nodes.length; i++) {
                        let node = nodes[i];
                        if (node.children && node.children.length > 0) {
                            let r = getNodeFromNodeId(nodeId, node.children);
                            if (r) {
                                return r;
                            }
                        }
                    }
                    return null;
                }
            }


            /*** Folding & Unfolding functions ***/
            $scope.getTaxonomyMassExpandCollapseStateFn = function() {
                let IDsUnfolded = $scope.getUnfoldedNodeIdsFn();
                let IDsHavingChildren = $scope.getNodeIdsHavingChildrenFn();

                if (IDsHavingChildren.length == 0) {
                    return "";
                } else if (IDsHavingChildren.length > IDsUnfolded.length) {
                    return "EXPAND_ALL";
                } else {
                    return "COLLAPSE_ALL";
                }
            };

            $scope.getNodeMassExpandCollapseStateFn = function(nodeId) {
                let node = getNodeFromNodeId(nodeId);
                if (angular.isDefined(node)) {
                    let descendants = getDescendantsOfNode(node).filter(n => n.id != nodeId);
                    let hasDescendants = descendants.length > 0;
                    let hasCollapsedChildren = !!descendants.find(n => n.$reduced == true && angular.isDefined(n.children) && n.children.length > 0);
                    let hasGrandChildren = !!descendants.find(n => angular.isDefined(n.children) && n.children.length > 0);
                    if (!hasDescendants) {
                        return "";
                    } else if (node.$reduced || hasCollapsedChildren) {
                        return "EXPAND_ALL";
                    } else if (hasGrandChildren) {
                        return "COLLAPSE_ALL";
                    } else {
                        return "";
                    }
                }
            }

            $scope.expandAllFn = function() {
                setReduceMultiple($scope.nodes, false);
            };

            $scope.collapseAllFn = function() {
                setReduceMultiple($scope.nodes, true);
            };

            $scope.expandChildren = function(nodeId) {
                let node = getNodeFromNodeId(nodeId);
                setReduceMultiple([node], false);
            };

            $scope.collapseChildren = function(nodeId) {
                let node = getNodeFromNodeId(nodeId);
                if (angular.isDefined(node.children)) {
                    setReduceMultiple(node.children, true);
                }
            };

            function expandNodes(nodes) {
                setReduceMultiple(nodes, false, false);
            }

            function setReduceMultiple(nodes, fold, recursive = true) {
                if (!angular.isDefined(nodes) || !nodes.length || nodes.length == 0) {
                    return;
                }

                angular.forEach(nodes, function(node) {
                    $scope.setReduceFn(node, fold);

                    if (recursive) {
                        setReduceMultiple(node.children, fold, recursive);
                    }
                });
            }

            $scope.setReduceFn = function(node, reduce) {
                // No need to set the value if it hasn't changed
                if (!angular.isDefined(node) || node.$reduced == reduce) {
                    return;
                }

                if (reduce) {
                    node.$reduced = true;
                } else {
                    delete node.$reduced;
                }

                // Keeping parent aware of this change if he gave us some way to.
                if (typeof($scope.getUnfoldedNodeIdsFn)==="function" && typeof($scope.setUnfoldedNodeIdsFn)==="function") {
                    let unfoldedNodeIds = $scope.getUnfoldedNodeIdsFn();
                    let index = unfoldedNodeIds.indexOf(node.id);

                    if (reduce) {
                        if (index > -1) {
                            unfoldedNodeIds.splice(index, 1);
                        }
                    } else {
                        if (index == -1 && node.children.length > 0) {
                            unfoldedNodeIds.push(node.id);
                        }
                    }
                    $scope.setUnfoldedNodeIdsFn(unfoldedNodeIds);
                }
            }


            /*** Scrolling functions ***/
            $scope.scrollToNodeFn = function(nodeId, duration) {
                let ancestors = getAncestorsOfNodeId(nodeId);

                if (ancestors) {
                    let node = ancestors.pop();
                    expandNodes(ancestors);
                    $timeout(() => $scope.triggerScroll(node, duration));
                }
            };

            $scope.shouldScrollToNode = function(node) {
                return node.$scrollToMe;
            };

            $scope.scrollDuration = function(node) {
                return node.$scrollDuration;
            };

            $scope.triggerScroll = function(node, duration) {
                node.$scrollToMe = true;
                node.$scrollDuration = duration;
            };

            $scope.onScrollTriggered = function(node) {
                node.$scrollToMe = false;
                delete node.$scrollDuration;
            };
        }
    };
});


app.directive("dkuHtmlTooltip", function($timeout, openDkuPopin) {
    return {
        template : '<div ng-mouseenter="displayTooltip($event)" ng-mouseleave="removeTooltip()" class="dku-html-tooltip-activation-zone {{triggerClass}}"><div ng-transclude="trigger"></div><div style="display: none;" ng-transclude="content"></div></div>',
        restrict: 'A',
        scope: {
            fromModal: '=?',
            tooltipClass: '@',
            triggerClass: '@',
            position: '@?', // if not present, uses openDkuPopin, otherwises sets the position manually
        },
        transclude: {
            'trigger': 'tooltipTrigger',
            'content': 'tooltipContent'
        },
        link: function($scope, elmnt, attrs) {

            var tooltipDisplayed = false;

            /*
             * _removeTooltip will be returned by openDkuPopin service when popin will actually be displayed.
             * This is b/c openDkuPopin create a new $scope and a DOM element when displaying a popin,
             * and returns a method to destroy this specific scope and remove new DOM element from DOM.
             */
            var _removeTooltip = function() {
                // noop
            };

            $scope.removeTooltip = function() {
                _removeTooltip();
                _removeTooltip = function() {
                    // noop
                };
                tooltipDisplayed = false;
            };

            // Remove the tooltip when the current directive is destroy while it was opened
            $scope.$on('$destroy', ()=> _removeTooltip());

            $scope.displayTooltip = function($event) {
                if (!tooltipDisplayed && (!attrs.dkuHtmlTooltipShow || $scope.$parent.$eval(attrs.dkuHtmlTooltipShow))) {
                    var cssClass = $scope.fromModal ? "dku-html-tooltip from-modal" : "dku-html-tooltip";
                    var content = $(elmnt).find('tooltip-content')[0].innerHTML;
                    var template = `<div class="${cssClass} ${$scope.tooltipClass || ''}">${content}</div>`;
                    var isElsewhere = function (tooltipElement, event) {
                        return $(event.target).parents('.dku-html-tooltip').length == 0 || $(event.target).parents('.dku-html-tooltip')[0] != tooltipElement;
                    };
                    if (
                        ['top', 'bottom', 'left', 'right', 'top-right', 'top-left', 'bottom-right', 'bottom-left'].includes(
                            $scope.position
                        )
                    ) {
                        _removeTooltip = openManuallyPositionedTooltip(template, elmnt, $scope.position);
                    } else {
                        var dkuPopinOptions = {
                            template: template,
                            isElsewhere: isElsewhere,
                            doNotCompile: true
                        };
                        _removeTooltip = openDkuPopin($scope, $event, dkuPopinOptions);
                    }
                    tooltipDisplayed = true;
                }
            };

            /*
             * Opens a tooltip for the element, using the template, at a manually set position
             * (top, bottom, right, left, top-right, top-left, bottom-right, bottom-left)
             * Returns a function that removes the tooltip
             */
            function openManuallyPositionedTooltip(template, element, position) {
                let tooltip = $(template);
                $('body').append(tooltip); //so we have access to dimensions

                let posLeft = 0;
                let posTop = 0;
                // left/top offset of the element
                const left = $(element).offset().left;
                const top = $(element).offset().top;
                // (Outer) width/height of the element
                const outerWidth = $(element).outerWidth();
                const outerHeight = $(element).outerHeight();
                // (Outer) width/heigth of the tooltip
                const tooltipOuterWidth = tooltip.outerWidth();
                const tooltipOuterHeight = tooltip.outerHeight();
                // Margin (= 2 * size of the arrow)
                const margin = 10;

                // We set the tooltip left/top offset according to desired position
                if (position == 'top') {
                    posLeft = left + outerWidth / 2 - tooltipOuterWidth / 2;
                    posTop = top - tooltipOuterHeight - 3 * margin / 2;
                } else if (position == 'top-right') {
                    posLeft = left + outerWidth;
                    posTop = top - tooltipOuterHeight - 3 * margin / 2;
                } else if (position == 'top-left') {
                    posLeft = left - tooltipOuterWidth + margin;
                    posTop = top - tooltipOuterHeight - 3 * margin / 2;
                } else if (position == 'bottom-left') {
                    posLeft = left - tooltipOuterWidth + margin;
                    posTop = top + outerHeight + margin / 2;
                } else if (position == 'bottom-right') {
                    posLeft = left + outerWidth;
                    posTop = top + outerHeight + margin / 2;
                } else if (position == 'bottom') {
                    posLeft = left + outerWidth / 2 - tooltipOuterWidth / 2;
                    posTop = top + outerHeight + margin / 2;
                } else if (position == 'left') {
                    posLeft = left - tooltipOuterWidth - margin / 2;
                    posTop = top - tooltipOuterHeight / 2;
                } else if (position == 'right') {
                    posLeft = left + outerWidth + 3 * margin / 2;
                    posTop = top - tooltipOuterHeight / 2;
                }
                tooltip.css('left', posLeft);
                tooltip.css('top', posTop);
                tooltip.addClass(position); // Adds the arrow at the right position

                $('body').append(tooltip);

                function dismiss() {
                    tooltip.remove();
                }

                return dismiss;
            }
        }
    }
});


app.directive("summaryOfError", function(Dialogs){
    return {
        // No isolated scope because we need to attach modals to the parent
        restrict: 'A',
        scope: true,
        templateUrl: "/templates/errors/summary-of-error.html",
        link: function($scope, _element, attrs) {


            $scope.$watch(attrs.summaryOfError, (nv) => {
                $scope.error = nv;
                $scope.isCredentialError = $scope.error && $scope.error.code && ($scope.error.code==="ERR_CONNECTION_OAUTH2_REFRESH_TOKEN_FLOW_FAIL" || $scope.error.code==="ERR_CONNECTION_NO_CREDENTIALS");
            });


            $scope.openMoreInfo = function(){
                 Dialogs.displaySerializedError($scope, $scope.error);
            }
        }
    }
});

    app.directive("errorFixability", function($rootScope, $state){
    return {
        restrict: 'A',
        templateUrl: "/templates/errors/error-fixability.html",
        scope: {
            error : "="
        },
        link: function($scope) {
            $scope.$state = $state;
            $scope.wl = $rootScope.wl;
        }
    }
});

app.directive('barMetrics', function () {
    return {
        scope: {data: '=barMetrics', height:'='},
        template: '<svg class="gpu-bar-metrics"></svg>',
        link: function ($scope, el) {
            const HEIGHT = $scope.height || 10;

            let svg = d3.select(el[0]).select('svg');
            let defs = svg.append('defs');

            let mainGradient = defs.append('linearGradient')
                .attr('id', 'mainGradient');

            mainGradient.append('stop')
                .attr('class', 'gpu-bar-metrics__stop-left')
                .attr('offset', '0');
            mainGradient.append('stop')
                .attr('class', 'gpu-bar-metrics__stop-right')
                .attr('offset', '1');

            let container = d3.select(el[0]).select('svg').append('g').attr('class', 'container').attr('transform', `translate(0, ${Math.max(HEIGHT/2,10)})`);
            $scope.$watchCollection('data', function (data) {
                if (data) {
                    const PREFIX_WIDTH = el[0].clientWidth / 4;
                    const POSTFIX_WIDTH = el[0].clientWidth / 5;
                    const MAX_BAR_WIDTH = el[0].clientWidth-PREFIX_WIDTH-POSTFIX_WIDTH;
                    const scales = data.map(d => d3.scale.linear().domain([d.min, d.max]).range([0, MAX_BAR_WIDTH]));
                    let g = container.selectAll('g.metric').data($scope.data);
                    let SPACE_BETWEEN = 8;
                    let gEnter = g.enter().append('g').attr('class', 'metric').attr('transform', (d, i) => `translate(0, ${i * (HEIGHT + SPACE_BETWEEN )})`);
                    gEnter.append('rect').attr('x',PREFIX_WIDTH).attr('y',-HEIGHT/2).attr('width', MAX_BAR_WIDTH).attr('height', HEIGHT).classed('gpu-bar-metrics__filled', true);
                    gEnter.append('rect').attr('x',PREFIX_WIDTH).attr('y',-HEIGHT/2).attr('class', 'val-inverted').attr('width', MAX_BAR_WIDTH).attr('height', HEIGHT).attr('fill', '#ececec');
                    gEnter.append('text').attr("alignment-baseline","middle").attr('text-anchor','start').attr('y', 0).attr('height', 50).text(d => d.label).attr('x', 0).attr("font-size", "10px").attr("fill", "grey");

                    gEnter.append('text').attr("alignment-baseline","middle").attr('text-anchor', 'end').attr('class', 'percentage').attr('y', 0).attr('x', el[0].clientWidth).attr("font-size", "10px").attr("fill", "grey");
                    g.select('text.percentage').text(d => `${Math.round(d.value * 100 / d.max)} %`);
                    g.select('rect.val-inverted').transition().attr('width', (d, i) => MAX_BAR_WIDTH - scales[i](d.value)).attr('x', (d, i) => PREFIX_WIDTH + scales[i](d.value));
                }
            });
        }
    }
});

app.directive('gitCheckoutSelect', function (SpinnerService, DataikuAPI) {
    return {
        scope: {
            gitCheckout: '=ngModel',
            gitRepository: '=repository',
            gitLogin: '=login',
            gitPassword: '=password'
        },
        templateUrl: '/templates/widgets/git-checkout-select.html',
        link: function ($scope) {
            $scope.gitCustomCheckout = true;
            $scope.gitLoadedRefsRepo = ''; // The repository where the currently loaded references are from

            $scope.$watch('gitRepository', function(nv) {
                if (nv && nv !== '') {
                    // When you change the input repository, you shouldn't have the old references list
                    $scope.gitCustomCheckout = $scope.gitRepository !== $scope.gitLoadedRefsRepo;
                }
            });

            $scope.listRemoteRefs = function () {
                // If the parent scope can show the error (`block-api-error` directive), it'll be better presented that way
                // Otherwise, we'll display it in that directive
                const errorScope = angular.isFunction($scope.$parent.setError) ? $scope.$parent : $scope;

                if ($scope.gitRepository && $scope.gitRepository !== '') {
                    resetErrorInScope(errorScope);

                    SpinnerService.lockOnPromise(DataikuAPI.git.listRemoteRefs($scope.gitRepository, $scope.gitLogin, $scope.gitPassword)
                        .then(function(response) {
                            $scope.gitCheckoutRefs = response.data;
                            $scope.gitCustomCheckout = false;
                            $scope.gitLoadedRefsRepo = $scope.gitRepository
                        }, function(err) {
                            $scope.gitCheckoutRefs = [];
                            $scope.gitCustomCheckout = true;
                            $scope.gitLoadedRefsRepo = '';
                            errorScope.setError(err);
                        })
                    )
                }
            };
        }
    }
});

app.directive('gitRemoteUrl', function (SpinnerService, DataikuAPI) {
    return {
        scope: {
            gitRef: '=ngModel'
        },
        templateUrl: '/templates/widgets/git-remote-url.html',
        link: function ($scope) {
            $scope.remoteUrlPaste = (event) => {
                if (!$scope.gitRef.remote || $scope.gitRef.remote.trim() === '') {
                    const pastedUrl = (event.originalEvent.clipboardData.getData("text") || '').trim();
                    if (pastedUrl) {
                        const matches = /^(https?:\/\/)([^:@]+):?([^@]+)?@(.+)$/i.exec(pastedUrl);
                        if (matches && matches.length === 5) {
                            const [,scheme,login,password,rest] = matches;
                            $scope.gitRef.remote = scheme + rest;
                            $scope.gitRef.remoteLogin = login;
                            $scope.gitRef.remotePassword = password;
                            event.preventDefault();
                            event.stopPropagation();
                            return false;
                        }
                    }
                }
                return true;
            };

            $scope.isHTTPSRemote = () => $scope.gitRef && /^\s*https?:/i.test($scope.gitRef.remote || '');
        }
    }
});

/**
 * Monitoring directive that displays information about currently available gpus, and can allow selection of gpus
 *
 * @param {Array<string>>} metrics - the metrics to display from nvidia-smi e.g. ['GPU','Memory']
 * @param {Array} selected - the selected gpus (works a bit like ng-model=listOfSelectedGPUS)
 * @param {boolean} selectable - whether the template will be selectable via mouseclick
 * @param {boolean} singleGpuOnly - whether selection will be forced to 1 gpu
*/
app.directive('gpuOnlineStats', function ($interval, Notification,Logger) {
    return {
        scope: {metrics:"=", selected:"=", onSelectedChange:"&?", selectable:"=", singleGpuOnly: "=?"},
        transclude: true,
        restrict: 'A',
        template: `<div ng-if="gpuCount" class="gpu-online-stats-wrapper">
                        <div ng-show="gpuStats && gpuResponse.status === 'OK'" ng-repeat="i in range(gpuCount)" class="gpu-online-stats__gpu-block" ng-click="selectable && selected && clickOnGpu(i)" ng-class="{selected:isGPUselected(i),'gpu-online-stats__gpu-block--selectable':selectable }">
                            <i class="icon icon-dku-gpu-card"></i>
                            <div>
                                <div class="gpu-online-stats__title">
                                    <span show-tooltip-on-text-overflow text-tooltip="gpuResponse.stats[i].name +' ['+ gpuResponse.stats[i].index + ']'" tooltip-direction="'tooltip'" ></span>
                                    <span class="gpu-online-stats__secondary-title">{{gpuResponse.stats[i].memoryTotal}} MB</span>
                                </div>
                                <div bar-metrics="gpuStats[i]" class="gpu-online-stats__gpu-graph" height="6"></div>
                            </div>
                        </div>
                    </div>
                    <div class="alert alert-error" ng-show="gpuResponse.status === 'ERROR'" style="text-align: center">
                        <span>{{gpuResponse.error}}</span>
                    </div>`,
        link: function (scope) {
            if (scope.selected && !(scope.selected instanceof Array)) {
                Logger.error(`gpuOnlineStats directive accepts a Array as a <selected> parameter, but ${scope.selected.constructor.name} was given`)
                scope.selected = null;
            }
            Notification.publishToBackend('gpu-monitoring-start');

            let KEEP_ALIVE_INTERVAL_MS = 2*1000;
            let cancelKeepAlive = $interval(function () {
                Notification.publishToBackend('timeoutable-task-keepalive', {
                    taskId: "gpu-monitoring"
                });
            }, KEEP_ALIVE_INTERVAL_MS);
            scope.range = function (n) {
                return Array.range(n);
            };

            scope.clickOnGpu = i => {
                if (!scope.selected.includes(i)){
                    if (scope.singleGpuOnly) { // first need to remove all the others
                        scope.selected.length = 0;
                    }
                    scope.selected.push(i);

                } else {
                    scope.selected.splice(scope.selected.indexOf(i),1);
                }

                if (angular.isFunction(scope.onSelectedChange)) {
                    scope.onSelectedChange();
                }
            };
            scope.isGPUselected = i => scope.selected && scope.selected.includes(i);

            scope.$on('$destroy', function () {
                $interval.cancel(cancelKeepAlive);
            });
            const gpuStatsResponseUnsubscribe = Notification.registerEvent("gpu-stats-response", function (evt, message) {
                scope.gpuResponse = message.response;
                if (scope.gpuResponse.status === 'OK') {

                    scope.gpuStats = message.response.stats.map(g => [{
                        label: 'Memory',
                        value: g.memoryUsed,
                        min: 0,
                        max: g.memoryTotal
                    }, {
                        label: 'GPU',
                        value: g.utilizationGpu,
                        min: 0,
                        max: 100
                    }].filter(d => d.value !== undefined && (!scope.metrics || scope.metrics.includes(d.label))));
                    scope.gpuCount = scope.gpuStats.length;
                }
            });
            scope.$on('$destroy', gpuStatsResponseUnsubscribe);
        }
    }
});
app.component('gpuSelector', {
    templateUrl: '/templates/widgets/gpu-selector.html',
    bindings: {
        gpuConfig: '<',
        gpuCapabilities: '<',
        onSettingsChange: '&?',
        selectedEnv: '<',
        name: '<',
        inContainer: '<',
        forceSingleGpu: '<',
        envMode: '<?',
        defaultEnvName: '<?',
        isScoreEval: '<?',
        llmMeshInUse: '<?'
    },
    controller: function($rootScope, $stateParams, $attrs, Notification, DataikuAPI, $filter, GpuUsageService, GPU_SUPPORTING_CAPABILITY, Debounce, $scope) {
        const $ctrl = this;
        $ctrl.numGpus = undefined;
        $ctrl.gpuStatus = undefined;
        $ctrl.possibleUsageWarning = "";
        $ctrl.currentEnvName = "";
        $ctrl.changeSettings = () => {}; // no-op unless configured in onInit
        $ctrl.onNumGpusChange = onNumGpusChange;
        $ctrl.GPU_SUPPORTING_CAPABILITY = GPU_SUPPORTING_CAPABILITY;
        $ctrl.allowModeChange = true;

        let lastGpuStats = [];
        let gpuCount;

        $ctrl.$onInit = () => {
            $ctrl.allowModeChange = GpuUsageService.allowChangingMode(!!$ctrl.isScoreEval, $ctrl.gpuCapabilities)

            getGpuUsability();
            $ctrl.currentEnvName = GpuUsageService.getCurrentEnvName($ctrl.selectedEnv, $ctrl.envMode, $ctrl.defaultEnvName);

            const isKeras = $ctrl.gpuCapabilities.includes(GPU_SUPPORTING_CAPABILITY.KERAS);
            if (isKeras && !$ctrl.gpuConfig.params.perGPUMemoryFraction) {
                $ctrl.gpuConfig.params.perGPUMemoryFraction = 0.7;
            }
            registerGpuStatsResponseEvent();

            if (angular.isFunction($ctrl.onSettingsChange)) {
                $ctrl.changeSettings = Debounce().withDelay(200, 500).wrap(() => $ctrl.onSettingsChange());
            }

            // needed to ensure all gpu params are in sync
            disableGpuBasedOnEnvSupport();
            updateForContainerizedExecution();
        };

        $ctrl.$onChanges = (changes) => {
            if ($ctrl.gpuConfig) {
                if (changes.inContainer) {
                    updateForContainerizedExecution();
                }

                if (changes.selectedEnv || changes.envMode || changes.defaultEnvName) {
                    $ctrl.currentEnvName = GpuUsageService.getCurrentEnvName($ctrl.selectedEnv, $ctrl.envMode, $ctrl.defaultEnvName);
                    disableGpuBasedOnEnvSupport();
                }

                if (changes.gpuCapabilities) {
                    getGpuUsability();
                    disableGpuBasedOnEnvSupport();
                }
            }
        };

        /**
         * Checks how many gpus can be used given the current enabled capabilities
         */
        function getGpuUsability() {
            const gpuUsability = GpuUsageService.checkGpuUsability($ctrl.gpuCapabilities, $ctrl.forceSingleGpu);
            $ctrl.singleGpuOnly = gpuUsability.singleGpuOnly;
            $ctrl.possibleUsageWarning = gpuUsability.possibleUsageWarning;
        }

        /**
         * Checks the selected code env, and disables the gpu exec option if it is known to not be supported by the env
         */
        function disableGpuBasedOnEnvSupport() {
            if ($ctrl.isScoreEval) {
                // We don't verify whether the environment supports GPU when in either the scoring or eval recipes.
                // Given that the model has already been created, we don't need to bother automatically disabling settings, based on what the
                // environment 'might' support
                $ctrl.envSupportsGpu = true;
            }
            else {
                GpuUsageService.getEnvSupportsGpu($rootScope.appConfig.isAutomation, $ctrl.gpuCapabilities, $ctrl.currentEnvName).then(result => {
                    $ctrl.envSupportsGpu = result;
                    if (!$ctrl.envSupportsGpu) {
                        $ctrl.gpuConfig.params.useGpu = false;
                    }
                });
            }
        }

        /**
         * Checks whether task is set to run in container, and sets the number of gpus based on the gpulist
         */
        function updateForContainerizedExecution() {
            if ($ctrl.inContainer) {
                if ($ctrl.gpuConfig.params.gpuList && $ctrl.gpuConfig.params.gpuList.length > 0) {
                    $ctrl.numGpus = $ctrl.gpuConfig.params.gpuList.length;
                } else {
                    // normally don't hit this, as new tasks set gpu0 in gpuList
                    // here in case someone changes from non-container to container
                    $ctrl.numGpus = 1;
                    onNumGpusChange(); // does paramUpdate call
                }
            }
        }

        /**
         * Event listener for the 'number of desired gpus' input when running in container
         * Creates a gpulist based on the selected number of gpus
         */
        function onNumGpusChange() {
            if ($ctrl.numGpus && $ctrl.numGpus > 0) {
                $ctrl.gpuConfig.params.gpuList = Array.range($ctrl.numGpus);
                $ctrl.onSettingsChange()
            }
        }

        function registerGpuStatsResponseEvent() {
            const gpuStatsResponseUnsubscribe = Notification.registerEvent("gpu-stats-response", function (evt, message) {
                $ctrl.gpuStatus = message.response.status;
                if (!gpuCount) {
                    gpuCount = message.response.stats.length;
                }

                if (gpuCount > 0 && gpuCount <  $ctrl.gpuConfig.params.gpuList.length) {
                    // available gpus appear to changed, resetting selection
                    $ctrl.gpuConfig.params.gpuList = Array.range(gpuCount);
                }
                lastGpuStats = message.response.stats;
            });
            $scope.$on('$destroy', gpuStatsResponseUnsubscribe);
        }
    }
});

    /*
        See NPSSurveyState for NPS survey timing logic
    */
    app.component('npsSurvey', {
        templateUrl: '/templates/widgets/nps-survey.html',
        bindings: {
            appConfig: '='
        },
        controller: ['$rootScope', '$scope', '$timeout', '$filter', 'DataikuAPI', 'WT1', 'Logger',
            function npsSurveyCtrl($rootScope, $scope, $timeout, $filter, DataikuAPI, WT1, Logger) {
                const ctrl = this;
                ctrl.Actions = {
                    NEW: 1,
                    SUBMIT: 2,
                    POSTPONE: 3,
                    OPTOUT: 4
                };
                ctrl.showSurvey = false;
                ctrl.active = false;
                ctrl.finished = false;
                ctrl.scores = Array.from({length: 11}, (_, i) => (i));
                ctrl.response = '';
                ctrl.futureSurveyParticipation = false;

                ctrl.display = function(){
                    ctrl.email = this.appConfig.user && this.appConfig.user.email ? this.appConfig.user.email : '';
                    ctrl.showSurvey = true;
                    $timeout(function() { ctrl.active = true; });
                }

                // we do not have a service where to trigger the display of the service
                // instead we expose the display method onto the rootScope
                $rootScope.showNpsSurvey = ctrl.display;

                ctrl.hide = function hide(){
                    $timeout(() => {
                        ctrl.active = false;
                        // to avoid a flicker showing the form again when done submitting
                        $timeout(() => {
                            ctrl.showSurvey = false;
                            ctrl.finished = false;
                            ctrl.response = '';
                            ctrl.futureSurveyParticipation = false;
                            ctrl.email = this.appConfig.user && this.appConfig.user.email ? this.appConfig.user.email : '';
                            ctrl.surveyScore = undefined;
                        }, 0);
                    }, 1000);
                }

                ctrl.$onInit = function() {
                    if (ctrl.appConfig) {
                        Logger.debug("appConfig loaded, initializing.");
                        if (ctrl.appConfig.npsSurveyActive) {
                            ctrl.display();
                        }
                    } else {
                        Logger.info("appConfig not loaded yet, watching it to wait for its initialization");
                        const deregisterAppConfigListener = $scope.$watch("$ctrl.appConfig", () => {
                            if (ctrl.appConfig) {
                                Logger.debug("appConfig loaded, initializing.");
                                if (ctrl.appConfig.npsSurveyActive) {
                                    ctrl.display();
                                }
                                deregisterAppConfigListener();
                            }
                        });
                    }
                };

                ctrl.finish = function(action) {
                    // if user clicks on something after submitting
                    if (ctrl.finished) {
                        return;
                    }

                    DataikuAPI.profile.setNPSSettings(action).success((data) => {
                        let eventParams = {
                            action: action,
                            npsState: data && data.state ? data.state : ''
                        };

                        if (action === 'SUBMIT') {
                            WT1.event('nps-survey', angular.extend(eventParams, {
                                score: ctrl.surveyScore,
                                response: $filter('escapeHtml')(ctrl.response || ''),
                                email: $filter('escapeHtml')(ctrl.email || ''),
                                futureSurveyParticipation: ctrl.futureSurveyParticipation || false,
                            }));
                            ctrl.finished = true;
                            ctrl.hide();
                        } else {
                            ctrl.hide();

                            WT1.event('nps-survey-decline', eventParams);
                        }
                    }).error(setErrorInScope.bind($scope));
                };

                ctrl.selectScore = function(score) {
                    ctrl.surveyScore = score;
                };
            }
        ]
    });


/*
    Tiny directive to handle the display of a sort icon in a table.

    Fields:
      * isSortCol: whether the current col is used for sorting (and the icon should be displayed)
      * ascending: whether the sort is ascending
      * iconOnRight: whether the icon is put on the right of the column name

    Besides, if you want to display a grayer version of the icon when hovering on the column name,
    your must add the .contains-sort-by-column-icon in the container of the column

    Example:
        <div class="my-container contains-sort-by-column-icon">
            <span>My title column</span>
            <sort-by-column-icon isSortCol="isSortCol" ascending="selection.orderReverse" icon-on-right="true"></sort-by-column-icon>
        </div>
*/
app.directive("sortByColumnIcon", function() {
    return {
        scope: {
            isSortCol: "=",
            ascending: "=",
            iconOnRight: "@"
        },
        template: `<div class="sort-by-column-icon__wrapper" ng-class="iconOnright ? 'icon-on-right' : 'icon-on-left'">
                      <i ng-if="!isSortCol" class="icon-sort-by-attributes-alt sort-by-column-icon--display-on-hover"></i>
                      <i ng-if="isSortCol && ascending" class="icon-sort-by-attributes"></i>
                      <i ng-if="isSortCol && !ascending" class="icon-sort-by-attributes-alt"></i>
                   </div>`
    }
});

app.filter('formatConfusionMatrixValue', function() {
    return function(value, isWeighted) {
        if (typeof value !== 'number') {
            return value; // for when it's percentage
        }
        return isWeighted ? value.toFixed(2) : value.toFixed(0);
    };
});

/*
    Binary Classification Confusion Matrix widget
*/
app.directive("bcConfusionMatrix", function() {
    return {
        templateUrl:"/templates/widgets/bc_confusion_matrix.html",
        scope: {
            modelClasses: "=",
            data: "=",
            displayMode: "=",
            metricsWeighted: "="
        },
        controller: function($scope) {
            // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
            $scope.puppeteerHook_elementContentLoaded = true;
        }
    }
});

/*
    Multi Classification Confusion Matrix widget
*/
app.directive("mcConfusionMatrix", function() {
    return {
        templateUrl:"/templates/ml/prediction-model/mc_confusion.html",
        scope: {
            modelData: "=",
            displayMode: "=",
            metricsWeighted: "="
        },
        controller: 'MultiClassConfusionMatrixController'
    }
});

/**
 * Generic component that displays a d3 brush.
 */
app.directive('rangeBrush', function(Fn) {
    return {
        restrict : 'A',
        templateUrl : '/templates/widgets/range-brush.html',
        scope : {
            range : '=',
            selectedRange : '=',
            onChange : '&',
            onDrillDown : '&',
            snapRanges : '=',
            onInit: '&',
            brushWidth: '@',
            enablePadding: '=?'
        },
        replace : true,
        link : function($scope, element) {
            $scope.enablePadding = angular.isDefined($scope.enablePadding) ? $scope.enablePadding : true;

            const padding = $scope.enablePadding ? 10 : 4; // 4 to be able to display the handles
            const brushHeight = 60;
            const dateLineHeight = 25;
            const triggerHeight = 18;
            const triggerWidth = 0.8 * triggerHeight;
            const handleHeight = 20;
            const handleWidth = 8;
            const separatorHeight = 3;
            const separatorOffset = -2;

            // the svg needs to update when the width changes (different # of ticks, for ex)
            var eventName = 'resize.brush.' + $scope.$id;
            $(window).on(eventName, function() { if ( $scope.range != null ) $scope.refreshRange();});
            $scope.$on('$destroy', function(){$(window).off(eventName)});
            // also add a watch on the width for the cases where the size changes as a result of
            // stuff being shown/hidden
            $scope.$watch(
                function () {return element.width();},
                function () { if ( $scope.range != null ) $scope.refreshRange(); }
            );

            // get the brush : the root of the template
            var brushSvg = d3.select(element[0]);
            // resize
            brushSvg.attr("height", brushHeight);

            // add stuff in the svg (layer in this order: to get the display and click-sensitivity right)
            var xAxisG = brushSvg.append("g").attr("class", "x axis").attr("transform", "translate(0, " + dateLineHeight + ")");

            var brushG = brushSvg.append("g").attr("class", "x brush"); // brush (catches mouse events to drag handles and brush extend)
            var triggersG = brushSvg.append("g").attr("class", "x triggers").attr("transform", "translate(0, " + dateLineHeight + ")"); // layer with the triggers, clickable
            var brushInversionG = brushSvg.append("g").attr("class", "x brush-inversion"); // the inverse of the brush (click-through)
            var brushHandlesG = brushSvg.append("g").attr("class", "x brush-handles"); // the brush handles (click-through)
            var brushContentG = brushSvg.append("g").attr("class", "x brush-content"); // where to append stuff (like chart preview)

            const brushContentWidth = $scope.brushWidth - (2 * padding);

            if ($scope.brushWidth) {
                brushSvg.style("width", $scope.brushWidth + 'px');
                brushContentG.attr("transform", "translate(" + padding + ", 0)");
                brushContentG.attr("width", brushContentWidth);
            }

            brushSvg.on('dblclick', () => {
                var insideExtent = false;
                if ( xScale != null ) {
                    var pos = d3.mouse(element[0]);
                    var xPos = xScale.invert(pos[0]).getTime();
                    insideExtent = $scope.selectedRange.from < xPos && $scope.selectedRange.to > xPos;
                }
                if (insideExtent && $scope.onDrillDown() != null) {
                    // why can't I pass the first () in the html? dunno...
                    $scope.onDrillDown()($scope.selectedRange.from, $scope.selectedRange.to);
                } else {
                    $scope.selectedRange.from = $scope.range.from;
                    $scope.selectedRange.to = $scope.range.to;
                    $scope.refreshRange();
                    $scope.onChange();
                }
            });

            var xScale = null;

            // update the total range, and then the graph
            $scope.refreshRange = function() {
                if ( $scope.range == null || $scope.selectedRange == null ) {
                    return;
                }
                var width = $(element).innerWidth();
                // the full range we are selecting in
                var axisRange = [new Date($scope.range.from), new Date($scope.range.to)];
                // the selected range
                var extentRange = [new Date($scope.selectedRange.from), new Date($scope.selectedRange.to)];
                // make the scale
                xScale = d3.time.scale().domain(axisRange).range([padding, width - padding]);
                // prepare the brush callback
                var brushed = function() {
                    var extent = brush.extent();
                    // If we are simply clicking on the brush (one point interval), go back to previous range.
                    if (extent[1] - extent[0] === 0) {
                        extent[0] = new Date($scope.selectedRange.from);
                        extent[1] = new Date($scope.selectedRange.to);
                    }
                    if (d3.event.mode === "move") {
                        if ( $scope.rounding == 'day') {
                            var startDay = d3.time.day.round(extent[0]);
                            var daySpan = Math.round((extent[1] - extent[0]) / (24 * 3600 * 1000));
                            var endDay = d3.time.day.offset(startDay, daySpan);
                            extent = [startDay, endDay];
                        }
                    } else {
                        if ( $scope.rounding == 'day') {
                            extent = extent.map(d3.time.day.round);
                            if (extent[0] >= extent[1] ) {
                                extent[0] = d3.time.day.floor(extent[0]);
                                extent[1] = d3.time.day.ceil(extent[1]);
                            }
                        }
                    }
                    d3.select(this).call(brush.extent(extent));
                    var xS = xScale(extent[0]);
                    var xE = xScale(extent[1]);
                    brushInversionG.selectAll('.s').attr("x", 0).attr("width", xS);
                    brushInversionG.selectAll('.e').attr("x", xE).attr("width", width - xE);
                    brushHandlesG.selectAll(".s").attr("transform", "translate(" + xS + ", 0)");
                    brushHandlesG.selectAll(".e").attr("transform", "translate(" + xE + ", 0)");

                    $scope.selectedRange.from = extent[0].getTime();
                    $scope.selectedRange.to = extent[1].getTime();
                    $scope.onChange();
                };

                // make the brush
                var brush = d3.svg.brush().x(xScale).on("brush", brushed).extent(extentRange);
                // make the axis from the scale
                var xAxis = d3.svg.axis().scale(xScale).tickFormat(Fn.getCustomTimeFormat()).orient("top").tickSize(-(brushHeight - dateLineHeight));
                // and create the svg objects
                var a = xAxisG.call(xAxis);
                var b = brushG.call(brush);
                triggersG.selectAll("*").remove();
                var t = triggersG.selectAll(".trigger").data($scope.snapRanges);

                var xS = xScale(extentRange[0]);
                var xE = xScale(extentRange[1]);

                // draw the triggers
                var triggerPadding = (brushHeight - dateLineHeight - triggerHeight) / 2;
                t.enter().append("path")
                    .classed("trigger", true)
                    .classed("success", function(d) {return d.outcome == 'SUCCESS';})
                    .classed("failed", function(d) {return d.outcome == 'FAILED';})
                    .classed("aborted", function(d) {return d.outcome == 'ABORTED';})
                    .classed("warning", function(d) {return d.outcome == 'WARNING';})
                    .attr("d", function(d) { return "M" + xScale(new Date(d.start)) + "," + triggerPadding + " l"+triggerWidth+","+(triggerHeight/2)+" l-"+triggerWidth+","+(triggerHeight/2)+" z"; })
                    .on("click", function(d){
                        $scope.selectedRange.from = d.start; $scope.selectedRange.to = d.end; $scope.refreshRange(); $scope.onChange();
                    });

                // remove the axis line
                a.selectAll(".domain").remove();

                // style the brush
                b.selectAll("rect").attr("y", 0).attr("height", brushHeight);
                // create the handles the handles
                brushHandlesG.selectAll(".resize").remove();
                brushHandlesG.append("g").classed("resize", true).classed("s", true).attr("transform", "translate(" + xS + ", 0)");
                brushHandlesG.append("g").classed("resize", true).classed("e", true).attr("transform", "translate(" + xE + ", 0)");
                var bh = brushHandlesG.selectAll(".resize");
                bh.append("rect").classed("separator", true).attr("y", 0).attr("height", brushHeight).attr("x", separatorOffset).attr("width", separatorHeight);
                bh.append("rect").classed("handle", true).attr("y", (brushHeight - handleHeight) / 2).attr("height", handleHeight).attr("x", -(handleWidth/2)).attr("width", handleWidth);
                // add the invert of the brush for the overlay outside of the brush
                brushInversionG.selectAll("rect").remove();
                brushInversionG.append("rect").attr("x", 0).attr("width", xS).attr("y", 0).attr("height", brushHeight).classed("s", true);
                brushInversionG.append("rect").attr("x", xE).attr("width", width - xE).attr("y", 0).attr("height", brushHeight).classed("e", true);
            };

            // add event handler to adjust the brush when the selection changes
            $scope.$watch('range', (nv) => {
                if ( nv == null ) return;
                $scope.refreshRange();
            }, true);
            $scope.$watch('snapRanges', (nv) => {
                if ( nv == null ) return;
                $scope.refreshRange();
            }, true);
            $scope.$watch('selectedRange', (nv) => {
                if ( nv == null ) return;
                $scope.refreshRange();
            }, true);
            $scope.$watch('brushWidth', () => {
                brushSvg.style("width", $scope.brushWidth + 'px');
            }, true);

            $scope.onInit && typeof $scope.onInit === 'function' && $scope.onInit({ brushContentG: brushContentG, brushContentHeight: brushHeight, brushContentWidth: brushContentWidth });
        }
    };
});
app.directive('datasetCreatorSelector', function ($parse) {
    return {
        templateUrl: '/templates/widgets/dataset-creator-selector.html',
        require:'^ngModel',
        scope: {
            ngModel: '=',
            managedDatasetOptions: '=',
            newDataset: '=',
            canCreate: '=',
            canSelectForeign: '=',
            markCreatedAsBuilt: '=',
            qa: '@'
        },
        controller: ['$scope', 'DataikuAPI', '$stateParams', 'DatasetUtils', 'SqlConnectionNamespaceService', function ($scope, DataikuAPI, $stateParams, DatasetUtils, SqlConnectionNamespaceService) {
            addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);
            $scope.partitioningOptions = [{id: "NP", label: "Not partitioned"}];
            $scope.io = {"newOutputTypeRadio": "create"};
            $scope.uiState = {mode:'select'};

            $scope.isCreationAllowed = angular.isDefined($scope.canCreate) ? $scope.canCreate : false;

            $scope.getDatasetCreationSettings = function () {
                let datasetCreationSetting = {
                    connectionId: ($scope.newDataset.connectionOption || {}).id,
                    specificSettings: {
                        overrideSQLCatalog: $scope.newDataset.overrideSQLCatalog,
                        overrideSQLSchema: $scope.newDataset.overrideSQLSchema,
                        formatOptionId: $scope.newDataset.formatOptionId,
                    },
                    partitioningOptionId: $scope.newDataset.partitioningOption,
                    inlineDataset: $scope.inlineDataset,
                    zone : $scope.zone,
                    markCreatedAsBuilt: $scope.markCreatedAsBuilt,
                };
                if ($scope.newDataset &&
                    $scope.newDataset.connectionOption &&
                    $scope.newDataset.connectionOption.fsProviderTypes &&
                    $scope.newDataset.connectionOption.fsProviderTypes.length > 1) {
                    datasetCreationSetting['typeOptionId'] = $scope.newDataset.typeOption;
                }
                return datasetCreationSetting;
            };

            $scope.createAndUseNewOutputDataset = function (force) {
                const projectKey = $stateParams.projectKey,
                    datasetName = $scope.newDataset.name,
                    settings = $scope.getDatasetCreationSettings();

                if (force) {
                    doCreateAndUseNewOutputDataset(projectKey, datasetName, settings);
                } else {
                    DataikuAPI.datasets.checkNameSafety(projectKey, datasetName, settings).success(data => {
                        $scope.uiState.backendWarnings = data.messages;
                        if (!data.messages || !data.messages.length) {
                            doCreateAndUseNewOutputDataset(projectKey, datasetName, settings);
                        }
                    }).error(setErrorInScope.bind($scope));
                }
            };

            function doCreateAndUseNewOutputDataset(projectKey, datasetName, settings) {
                DataikuAPI.datasets.newManagedDataset(projectKey, datasetName, settings)
                    .success(dataset => {
                        DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success((data) => {
                            $scope.uiState.mode = 'select';
                            if (!$scope.canSelectForeign) {
                                data = data.filter(fs => fs.localProject);
                            }
                            $scope.availableDatasets = data;
                            $scope.uiState.model = dataset.name;
                        });
                    }).error(setErrorInScope.bind($scope));
            }

            DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success((data) => {
                if (!$scope.canSelectForeign) {
                    data = data.filter(fs => fs.localProject);
                }
                $scope.availableDatasets = data;
            });

            DataikuAPI.datasets.getManagedDatasetOptionsNoContext($stateParams.projectKey).success(function (data) {
                $scope.managedDatasetOptions = data;
                if (!$scope.newDataset.connectionOption && data.connections.length) {
                    const fsConnection = data.connections.find(e => {
                        return 'Filesystem' === e.connectionType;
                    });
                    if (fsConnection) {
                        $scope.newDataset.connectionOption = fsConnection;
                    } else {
                        $scope.newDataset.connectionOption = data.connections[0];
                    }
                }
                if (!$scope.newDataset.formatOptionId && $scope.newDataset.connectionOption.formats.length) {
                    $scope.newDataset.formatOptionId = $scope.newDataset.connectionOption.formats[0].id;
                }
                $scope.partitioningOptions = [
                    {"id": "NP", "label": "Not partitioned"},
                ].concat(data.projectPartitionings);

            });

            $scope.fetchCatalogs = function(connectionName, connectionType) {
                SqlConnectionNamespaceService.listSqlCatalogs(connectionName, $scope, 'dataset-creator-selector', connectionType);
            };

            $scope.fetchSchemas = function(connectionName, connectionType) {
                const catalog = $scope.newDataset.overrideSQLCatalog ||
                    ($scope.newDataset.connectionOption ? $scope.newDataset.connectionOption.unoverridenSQLCatalog : '');
                SqlConnectionNamespaceService.listSqlSchemas(connectionName, $scope, catalog, 'dataset-creator-selector', connectionType);
            };

            $scope.$watch("newDataset.connectionOption.connectionName", function() {
                if ($scope.newDataset.connectionOption) {
                    SqlConnectionNamespaceService.setTooltips($scope, $scope.newDataset.connectionOption.connectionType);
                }

                SqlConnectionNamespaceService.resetState($scope, $scope.newDataset);
            });

        }],
        link: function (scope, el, attrs, ngModel) {
            scope.uiState = scope.uiState || {};
            scope.uiState.model = $parse(attrs.ngModel)(scope.$parent);
            scope.$watch("uiState.model", (nv, ov) => {
                if (ov === nv) return;
                if (nv === ngModel.$viewValue) return;
                // set the new value in the field and render
                ngModel.$setViewValue(nv);
                ngModel.$render();
            });
            scope.$watch("ngModel", (nv, ov) => {
                if (ov === nv) return;
                if (nv === scope.uiState.model) return;
                scope.uiState.model = ngModel.$viewValue;
            });
        }
    }
})

    /**
     * Display tags list if it fits else only one + the rest inside a popover.
     *
     * @param {Array}       items                   - List of the tags to display
     * @param {Array}       highlighted             - List of the tags to highlight (put in bold)
     * @param {Object}      tagsMap                 - Map of all the available tags. Used mainly to get the tag colors
     * @param {Array}       globalTagsCategories    - List of existing global tag categories applying to this object type
     * @param {string}      objectType              - Taggable type of the object on which the tags are applied
     * @param {boolean}     editable                - If editable, displays the add tag/category buttons
     * @param {Function}    onTagClick              - The method to call when clicking on a tag
     *                                                  - called with $event and tag when clicking on a tag -> onTagClick($event, tag)
     *                                                  - called with $event and category when clicking on a tag category -> onTagClick($event, category)
     *                                                  - called with just $event when clicking on the add tag button -> onTagClick($event)
     * @param {boolean}     displayPopoverOnHover   - Whether the popover should be displayed on hover on the ellipsis (not equivalent to !editable because you might still want to click on the tag if you can't edit, i.e. objects list page)
     *
     * <responsive-tags-list items="item.tags" tags-map="projectTagsMap" object-type="'OBJECT_TYPE'" on-tag-click="selectTag($event, tag)"></responsive-tags-list>
     */
    app.directive('responsiveTagsList', function ($timeout, Assert, translate) {
        return {
            templateUrl: '/templates/analysis/responsive-tags-list.html',
            scope: {
                items: '<',
                highlighted: '<',
                tagsMap: '<',
                globalTagsCategories: '<',
                objectType: '<',
                editable: '=?',
                onTagClick: '&?',
                displayPopoverOnHover: '<'
            },
            link: function ($scope, $element, attrs) {
                const ELLIPSIS_BUTTON_WIDTH = 45;
                let tagsListObserver;
                let tagsListContainer;
                $scope.canTagsFit = true;
                $scope.highlighted = $scope.highlighted || []; // default value
                $scope.translate = translate;

                // Bootstrap Popover's generated content sometimes looses AngularJS scope.
                // So we cannot rely on the later and therefore have to handle tag click through window object.
                // See https://github.com/angular/angular.js/issues/1323
                if (!window.onTagClick) {
                    window.onTagClick = (event) => {
                        if (!$scope.onTagClick) return;
                        const category = $(event.target).attr('data-category');
                        const tag = $(event.target).closest('.responsive-tags-list__tag').attr('data-tag');
                        $scope.onTagClick({$event: event, tag: tag, category: category});
                        $scope.$apply();
                    }
                }

                function computeFittingTags(tagsList) {
                    const fittingTags = { tags: [], categories: [] };
                    const containerHeight = tagsListContainer.offsetHeight;
                    const containerWidth = tagsListContainer.offsetWidth;

                    for (let index = 0; index < tagsList.length; index++) {
                        const tag = $(tagsList[index])[0];
                        const heightFits = tag.offsetTop + tag.offsetHeight < containerHeight;
                        const isLastLine = tag.offsetTop + 2 * tag.offsetHeight > containerHeight;
                        const widthFits = !isLastLine || tag.offsetLeft + tag.offsetWidth + ELLIPSIS_BUTTON_WIDTH < containerWidth;
                        if (heightFits && widthFits) {
                            if (index < $scope.items.length) {
                                fittingTags.tags.push($scope.items[index]);
                            } else if ($scope.globalTagsCategories && index < $scope.items.length + $scope.globalTagsCategories.length) {
                                fittingTags.categories.push($scope.globalTagsCategories[index - $scope.items.length]);
                            }
                        } else if (!heightFits) {
                            break;
                        }
                    }
                    return fittingTags;
                }

                if (!('IntersectionObserver' in window)) {
                    // Graceful degradation: will simply drop the tags in a scrollable area.
                    $scope.hasFitBeenChecked = true;
                } else {
                    const options = {
                        threshold: [0, 0.5, 1],
                        root: $element[0].parentElement
                    };
                    const intersectionCallback = (entries) => {
                        const entry = entries[0];
                        $scope.canTagsFit = entry.intersectionRatio === 1;

                        tagsListContainer = $element[0].parentElement;
                        if (!$scope.canTagsFit && !$scope.fittingTags) {
                            $scope.fittingTags = computeFittingTags(entry.target.children);
                        }
                        $scope.hasFitBeenChecked = true;
                        $scope.$apply();
                    }
                    tagsListObserver = new IntersectionObserver(intersectionCallback, options);
                    // Wait for template to be injected before checking
                    $timeout(() => {
                        tagsListObserver.observe($element[0].querySelector('#tag-list'));
                    }, 0);
                }

                // Detect element resize
                const resizeObserver = new ResizeObserver(() => {
                    // display #tag-list to trigger intersectionObserver
                    // and see which tags fit if it overflows
                    delete $scope.fittingTags;
                    $scope.canTagsFit = true;
                    $timeout(() => $scope.$apply());
                });
                resizeObserver.observe($element[0].parentElement);

                $scope.$watch("items", () => {
                    // display #tag-list to trigger intersectionObserver
                    // and see which tags fit if it overflows
                    delete $scope.fittingTags;
                    $scope.canTagsFit = true;
                    $timeout(() => $scope.$apply());
                }, true);

                $scope.$on('$destroy', () => {
                    tagsListObserver && tagsListObserver.disconnect();
                    resizeObserver && resizeObserver.disconnect();
                    delete window.onTagClick;
                });
            }
        }
    });

    app.controller("ResponsiveTagsListPopoverController", function($scope, translate) {
        $scope.translate = translate;
        $scope.getPlacement = function(tip, el) {
            let offset = $(el).offset();
            let top = offset.top;
            let height = $(document).outerHeight();
            return 0.5 * height - top > 0 ? 'bottom' : 'top';
        }
    })

    app.directive('codeViewer', function() {
        return {
            restrict: 'AE',
            replace: true,
            templateUrl : '/templates/widgets/code-viewer.html',
            scope : {
                code : '<',
                mimeType : '@'
            },
            controller : function($scope, CodeMirrorSettingService) {
                $scope.codeMirrorSettings = CodeMirrorSettingService.get(
                    $scope.mimeType || 'text/plain', {onLoad: function(cm){
                        cm.setOption("readOnly", true);
                        $scope.codeMirror = cm;
                    }}
                );

                $scope.uiRefreshToggle = false;

                function toggle() {
                    $scope.uiRefreshToggle = !$scope.uiRefreshToggle;
                }
                $scope.$on("codeViewerUIRefreshToggle", toggle);
            }
        }
    });

    /**
     * Directive for managing a list of steps
     *
     * Example usage:
     * <div stepper="stepperState.stepInfo" current-step="stepperState.currentStep" disable-steps="!disableSteps" />
     *
     * stepper (steps): An array of step objects
     * - step object:
     *   - label:       step header
     *   - description: step subtitle
     *   - getError(): function which returns an error message if there is one
     *   - getWarning(): function with returns a warning message if there is one (errors take precedence)
     *   - postAction(): function called right after leaving a step
     * currentStep: variable containing the current step (integer)
     * disableSteps: boolean for disabling step interaction (use a function to disable individual steps)
     */
    app.directive('stepper', function() {
        return {
            restrict: 'A',
            templateUrl: "/templates/widgets/stepper.html",
            scope: {
                steps: '<stepper',
                currentStep: '=',
                disableSteps: '<'
            },
            link : function(scope) {
                let previousStep = scope.currentStep;

                function goToStep() {
                    if (previousStep === scope.currentStep) return;

                    const stepInfo = scope.steps[previousStep];

                    // perform actions before leaving previous step
                    if (stepInfo && stepInfo.postAction) {
                        stepInfo.postAction();
                    }

                    previousStep = scope.currentStep;
                }

                scope.stepClicked = function(step) {
                    if (!scope.disableSteps && scope.currentStep !== step) {
                        scope.currentStep = step;
                    }
                }

                scope.$watch('currentStep', goToStep);
            }
        };
    });
    app.directive('stepperStep', function() {
        return {
            restrict: 'A',
            templateUrl: "/templates/widgets/stepper-step.html",
            scope: {
                stepNumber: '<',
                step: '<stepperStep',
                isLastStep: '<',
                isCurrentStep: '<',
                isCompletedStep: '<',
                disableStep: '<'
            },
            link: function(scope) {
                scope.step.getError = scope.step.getError || (() => '');
                scope.step.getWarning = scope.step.getWarning || (() => '');
            }
        };
    });

    /** Discrete progress ring with different sections (can be styled in css).
     * values           Array of values (each value is the number of sections for a given class)
     * maximum          Maximum number of sections
     * centerValue      Value to be shown in the center of the progress ring
     * classes          Array of classes for styling
     * classNotFilled   Class of the remaining sections if maximum > sum of values
     * radius           Radius of the progress ring (in px)
     * strokeWidth      Width of the stroke (in px)
     * maxAngle         Maximum angle of the progress ring (180 = half circle / 360 = full circle) (in deg)
     **/
    app.component('progressRing', {
        templateUrl: '/templates/widgets/progress-ring.html',
        bindings: {
            values: '<',
            maximum: '<',
            centerValue: '<',
            classes: '<',
            classNotFilled: '@',
            radius: '<',
            strokeWidth: '<',
            maxAngle: '<',
        },
        controller: [
            function () {
                const ctrl = this;

                ctrl.$onChanges = function () {
                    reDraw();
                };

                function reDraw() {
                    // Compute width and height of the svg depending on the radius and maxAngle
                    ctrl.svgWidth = ctrl.maxAngle > 180 ? 2 * ctrl.radius : (ctrl.maxAngle / 180) * 2 * ctrl.radius;

                    if (ctrl.maxAngle < 90) {
                        ctrl.svgHeight = (ctrl.maxAngle / 90) * ctrl.radius;
                    } else if (ctrl.maxAngle < 180) {
                        ctrl.svgHeight = ctrl.radius;
                    } else if (ctrl.maxAngle < 270) {
                        ctrl.svgHeight = (ctrl.maxAngle / 90 - 1) * ctrl.radius;
                    } else {
                        ctrl.svgHeight = 2 * ctrl.radius;
                    }
                    ctrl.svgHeight += 2; // To make room for the center value

                    const numberOfSections = ctrl.values.reduce((a, b) => a + b);
                    if (numberOfSections > ctrl.maximum) {
                        // If maximum is smaller than sum of values, use sum of values as maximum
                        ctrl.maximum = numberOfSections;
                    }

                    // 0.5px spacing between "sections"
                    let spacingInDegrees = 0.5 * ctrl.maxAngle / ctrl.radius;
                    // Single "section" size in degrees, taking into account the spacing between each one of them
                    let sectionInDegrees =
                        (ctrl.maxAngle - (ctrl.maximum - 1) * spacingInDegrees) / ctrl.maximum;
                    // If resulting section is smaller than 1px, reduce the spacing between them to half the section size
                    if (sectionInDegrees / 180 * ctrl.radius < 1) {
                        spacingInDegrees = 0.5 * sectionInDegrees;
                        sectionInDegrees = (ctrl.maxAngle - (ctrl.maximum - 1) * spacingInDegrees) / ctrl.maximum;
                    }

                    // Fill sections path data (describe and class)
                    ctrl.paths = [];
                    let sectionNumber = 0;
                    for (const [classValue, nSections] of ctrl.classes.map((c, i) => [c, ctrl.values[i]])) {
                        for (let i = 0; i < nSections; i++) {
                            ctrl.paths.push({
                                describe: describeArc(
                                    ctrl.radius,
                                    ctrl.radius,
                                    ctrl.radius - 2 * ctrl.strokeWidth,
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees),
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees) - sectionInDegrees
                                ),
                                class: classValue,
                            });
                            sectionNumber++;
                        }
                    }

                    // Fill not filled sections if there are some remaining
                    if (numberOfSections < ctrl.maximum) {
                        for (let i = 0; i < ctrl.maximum - numberOfSections; i++) {
                            ctrl.paths.push({
                                describe: describeArc(
                                    ctrl.radius,
                                    ctrl.radius,
                                    ctrl.radius - 2 * ctrl.strokeWidth,
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees),
                                    180 - sectionNumber * (spacingInDegrees + sectionInDegrees) - sectionInDegrees
                                ),
                                class: ctrl.classNotFilled,
                            });
                            sectionNumber++;
                        }
                    }
                }

                function polarToCartesian(centerX, centerY, radius, angleInDegrees) {
                    const angleInRadians = (-angleInDegrees * Math.PI) / 180.0;

                    return {
                        x: centerX + radius * Math.cos(angleInRadians),
                        y: centerY + radius * Math.sin(angleInRadians),
                    };
                }

                function describeArc(x, y, radius, startAngle, endAngle) {
                    const start = polarToCartesian(x, y, radius, endAngle);
                    const end = polarToCartesian(x, y, radius, startAngle);

                    const largeArcFlag = Math.abs(endAngle - startAngle) <= 180 ? '0' : '1';

                    const d = ['M', start.x, start.y, 'A', radius, radius, 0, largeArcFlag, 0, end.x, end.y].join(' ');

                    return d;
                }
            },
        ],
    });

    /** Text that can be edited once clicked on
     * textValue        Current text value (two-way bound). Gets set to defaultValue on load if empty.
     * defaultValue     (optional) Default text shown when directive first loads (before editing) if textValue is empty
     *                  If defaultValue is passed in, title color will be grayed out if textValue === defaultValue
     * placeholder      (optional) Placeholder text shown in input field while editing
     * fontSize         (optional) Font size to be used in in both view and edit mode; default value matches style guide
     **/
    app.directive('editableTextField', function() {
        return {
            restrict: 'E',
            templateUrl: '/templates/widgets/editable-text-field.html',
            scope: {
                textValue: '=ngModel',
                defaultValue: '<?',
                placeholder: '<?',
                fontSize: '<?'
            },
            link: function(scope) {
                scope.item = {};
                scope.placeholder = scope.placeholder || 'New name';

                // Comparator item naming
                scope.startEdit = function(item) {
                    item.$editing = true;
                    item.$textValue = scope.textValue === scope.defaultValue ? '' : scope.textValue;
                };

                scope.cancelEdit = function(item) {
                    item.$editing = false;
                };

                scope.validateEdit = function(item) {
                    scope.textValue = item.$textValue || scope.textValue;
                    item.$editing = false;
                };

                scope.$watch('defaultValue', (nv, ov) => {
                    if (!scope.textValue || scope.textValue === ov) {
                        scope.textValue = nv;
                    }
                });
            }
        }
    });


    /** Text block that will be displayed on one line. If the text block does not fit on one line,
     * a show more/less button will be displayed.
     **/
    app.directive("truncateTextVertically", function () {
        const template = `
                <div ng-transclude ng-class="{ 'truncated': showMoreLess && !isOpen }"></div>
                <div ng-show="showMoreLess" class="mtop8">
                    <a ng-show="!isOpen" ng-click="isOpen = !isOpen">Show more&hellip;</a>
                    <a ng-show="isOpen" ng-click="isOpen = !isOpen">Show less&hellip;</a>
                </div>`;
        return {
            restrict: 'A',
            template: template,
            transclude: true,
            link: ($scope, element) => {
                const textElement = element.find('[ng-transclude]')[0];
                $scope.isOpen = false;

                const updateShouldShowMoreLess = function() {
                    const truncatedClass = 'truncated';

                    // ensure element has truncated class to calculate heights for showMoreLess correctly
                    const alreadyTruncated = textElement.classList.contains(truncatedClass);
                    if (!alreadyTruncated) {
                        textElement.classList.add(truncatedClass);
                    }
                    $scope.showMoreLess = textElement.offsetHeight < textElement.scrollHeight || textElement.offsetWidth < textElement.scrollWidth;
                    if (!alreadyTruncated) {
                        textElement.classList.remove(truncatedClass);
                    }
                };
                updateShouldShowMoreLess();

                const resizeObserver = new ResizeObserver(() => {
                    const oldValue = $scope.showMoreLess;
                    updateShouldShowMoreLess();
                    if (oldValue !== $scope.showMoreLess) {
                        // make sure Angular runs a digest to take into account the new value of showMoreLess
                        $scope.$applyAsync();
                    }
                });
                resizeObserver.observe(textElement);

                $scope.$on('$destroy', function() {
                    resizeObserver.disconnect();
                });
            }
        }
    });

    /** Basic chart (using only divs) to display percentages
     * data                 Format: [[categoryName, value (in decimal)], ...]
     * colors               Array of colors to use for each bar
     * showFilter           (optional) Add input for filtering if true
     * onCategoryClicked       (optional) Fires when category is clicked
     **/
    app.component('basicPercentChart', {
        templateUrl: '/templates/widgets/basic-percent-chart.html',
        bindings: {
            data: '<',
            colors: '<',
            showFilter: '<?',
            onCategoryClicked: '&?'
        },
        controller: function(Debounce) {
            const $ctrl = this;
            const DEFAULT_COLOR = '#999';
            const MAX_CHART_HEIGHT = 350;

            $ctrl.query = '';
            $ctrl.rowHeight = 20;

            $ctrl.$onChanges = () => {
                const maxValue = Math.max.apply(null, $ctrl.data.map(row => row[1]));
                $ctrl.chartData = $ctrl.data.map((row, index) => ({
                    category: row[0],
                    value: Math.round(row[1] * 100) + '%',
                    width: Math.round(row[1] / maxValue * 100) + '%',
                    color: $ctrl.colors[index] || DEFAULT_COLOR,
                }));
                filter();

                $ctrl.chartHeight = Math.min($ctrl.data.length * $ctrl.rowHeight + 1, MAX_CHART_HEIGHT);
            };

            const filter = () => {
                $ctrl.filteredChartData = $ctrl.chartData.filter(row => row.category.toLowerCase().indexOf($ctrl.query
                .toLowerCase()) >= 0);
            };
            $ctrl.filterDebounced = Debounce().withDelay(150, 150).wrap(filter);

            $ctrl.onCategoryClick = (category) => {
                if ($ctrl.onCategoryClicked) {
                    $ctrl.onCategoryClicked({ category });
                }
            };
        }
    });

    /**
     * A component to help displaying items that need to be filtered and paginated, but can also be reordered, added or removed.
     * Because of the limitations of angular transclusion, the contained items are not directly repeated, but you have to use a ng-repeat on the paginatedItems 'output' parameter
     * This component can be used with filteredPaginatedListItem that contains the buttons to manipulate items. Alternatively, you could design your own variant if the UI needs to be different.
     *
     * Example:
     * <filtered-paginated-list items="myArrayOfItems" page-size="20" new-item-template="defaultItemValue" paginated-items="tempvalues.paginatedItems">
            <li ng-repeat="item in tempvalues.paginatedItems"
                ng-class="{'blokc--highlighted': rule.highlighted">
                <filtered-paginated-list-controls item="item" />
                <my-custom-element item="item.model" />
            </li>
        </filtered-paginated-list>
     *
     * @param items = an array conaining the items to display
     * @param pageSize the number of items to display per page (default 50)
     * @param newItemTemplate the default value of a new item (will be cloned on item creation)
     * @param paginatedItems "Output" parameter, the list of items to display for the current page. Should be used to display content
     *      paginatedItems is an array of 'enriched' input items:
     *      - model is the original item (and can be modified in the transcluded elements)
     *      - rank is the position in the full list (starts at 1)
     *      - highlighted is an indicator that and the element has recently been moved or added, and should be highlighted in order for the user to keep track of it.
     */
    app.component('filteredPaginatedList', { // filtered-paginated-list
        templateUrl: '/templates/widgets/filtered-paginated-list.html',
        transclude: true,
        bindings: {
            configItems: "=items",
            paginatedItems: '=',
            pageSize: "<?",
            newItemTemplate: "<?",
        },
        controller: function($scope, ListFilter, CollectionFiltering, $timeout, Dialogs) {
            this.$onInit = function () {
                let filteredItems = [];
                let items = this.configItems.map((model, i) => ({
                    rank: i + 1,
                    model
                }));
                this.scrollToRank = -1;
                this.pageSize = this.pageSize || 50;

                this.pagination = new ListFilter.Pagination(items, this.pageSize);
                this.search = '';
                // this function update the ranks after an add, delete, move...
                // ranks start at 1 because we want the seach to find the right item.
                const updateRanks = (start, end = items.length + 1) => {
                    for(let i = start-1 ; i < end-1 ; i ++) {
                        items[i].rank = i + 1;
                    }
                };

                // refresh filter & pagination after any query change or list change (add, move, remove...)
                // not triggered for item content change (an item is not suddenly hidden because it's modified and doesn't match the filter anymore)
                const updateView = () => {
                    filteredItems = this.search
                        ? CollectionFiltering.filter(items, {
                             userQuery: this.search,
                        }, {
                            exactMatch: ['rank'],
                        })
                        : items;

                    this.pagination.updateAndGetSlice(filteredItems);
                    this.paginatedItems = this.pagination.slice;
                };


                const scrollTo = (item) => {
                    const filteredIndex = filteredItems.indexOf(item);
                    if(filteredIndex === -1) return; // when the item we move is filtered out, cancel the scroll out instead of going to page 0

                    const targetPage = Math.floor(filteredIndex / this.pageSize) + 1;
                    this.pagination.goToPage(targetPage);
                    this.scrollToRank = item.rank;

                    // the counter instead of simple boolean handles the fact that a same item could be clicked twice in less the 800ms
                    item.highlighted = (item.highlighted || 0) + 1;
                    $timeout(() => item.highlighted -= 1, 800);

                    // we need to reset scrollToRank so that the same element can trigger a scroll again
                    $timeout(() => this.scrollToRank = -1);
                };


                $scope.$watch(() => this.search, updateView);

                $scope.$watch(() => this.pagination.page, () => {
                    this.pagination.update();
                    this.paginatedItems = this.pagination.slice;
                });
                const moveToRank = (item, destRank) => {
                    const rank = item.rank;

                    items.splice(rank - 1, 1);
                    items.splice(destRank - 1, 0, item);

                    this.configItems.splice(rank - 1, 1);
                    this.configItems.splice(destRank - 1, 0, item.model);

                    updateRanks(
                        Math.min(rank, destRank),
                        Math.max(rank, destRank, items.length - 2) + 1
                    );
                    updateView();
                    scrollTo(item);
                };

                const moveTo = (item) => {
                    return Dialogs.prompt($scope, "Move to", `Move before the rule having rank (max ${items.length+1}):`, item.rank, {
                        minValue: 1,
                        maxValue: items.length+1,
                        type: 'number'
                    })
                    .then((newRank) => {
                        // if item.rank < newRank, we have to take into account the fact that removing item from the list will change target rule rank
                        if (newRank > item.rank) {
                            moveToRank(item, Math.max(0, Math.min(items.length, newRank - 1)));
                        } else {
                            moveToRank(item, Math.max(0, Math.min(items.length, newRank)));
                        }
                    })
                    .catch(() => {}) // reject = move canceled, nothing to do
                };
                const canMoveTo = () => items.length > 1;

                const moveUp = (item) => {
                    const filteredIndex = filteredItems.indexOf(item);
                    const previousVisibleRule = filteredItems[filteredIndex - 1];
                    moveToRank(item, previousVisibleRule.rank);
                };
                const canUp = (item) => filteredItems[0] && item.rank > filteredItems[0].rank;

                const moveDown = (item) => {
                    const filteredIndex = filteredItems.indexOf(item);
                    const nextVisibleRule = filteredItems[filteredIndex + 1];
                    moveToRank(item, nextVisibleRule.rank);
                };
                const canDown = (item) => filteredItems[filteredItems.length - 1] && item.rank < filteredItems[filteredItems.length - 1].rank;

                const moveTop = (item) => moveToRank(item, 1);
                const canTop = (item) => item.rank > 1;

                const moveBottom = (item) => moveToRank(item, items.length);
                const canBottom = (item) => item.rank < items.length;


                const insertNewItemAtRank = (rank) => {
                    const newValue = angular.copy(this.newItemTemplate || {});
                    const newItem = { model: newValue };
                    items.splice(rank - 1, 0, newItem);
                    this.configItems.splice(rank - 1, 0, newValue);

                    updateRanks(rank);
                    updateView();
                    scrollTo(newItem);
                };

                const insertNewItemBefore = (item) => insertNewItemAtRank(item.rank);
                const insertNewItemAfter = (item) => insertNewItemAtRank(item.rank + 1);
                const insertNewItemFirst = () => insertNewItemAtRank(1);
                const insertNewItemLast = () => insertNewItemAtRank(items.length + 1);

                const remove = (item) => {
                    items.splice(item.rank - 1, 1);
                    this.configItems.splice(item.rank - 1, 1);
                    updateRanks(item.rank);
                    updateView();
                };

                this.api = {
                    moveTop, canTop,
                    moveUp, canUp,
                    moveDown, canDown,
                    moveBottom, canBottom,
                    moveTo, canMoveTo,

                    insertNewItemAfter, insertNewItemBefore,
                    insertNewItemFirst, insertNewItemLast,
                    remove,
                };
            }
        }
    });

    /**
     * Optional child component for filteredPaginatedList
     * Displays buttons to move / add / delete items and the item rank
     * @param item current item (used for the actions)
     * @param rankLabel text used to label the rank display (default = 'Item rank:')
     */
    app.component('filteredPaginatedListControls', {
        transclude: true,
        templateUrl: '/templates/widgets/filtered-paginated-list-item.html',
        require: {
            list: '^^filteredPaginatedList',
        },
        bindings: {
            item: "=",
            rankLabel: "@?",
        },
    });

    app.component('messageAlert', {
         templateUrl : '/templates/widgets/message-alert.html',
         bindings: {
             text: '@',
             severity: '@',
         },
    });

    app.component('limitedVisibilityLock', { // limited-visibility-lock
        template: `
        <span class="limited-visibility-lock"
            toggle="tooltip" container="{{$ctrl.tooltipContainer || 'body'}}" title="You don't have full access to view this {{$ctrl.objectNiceName}}"
        >
            <i class="icon-lock"></i>
        </span>
        `,
        bindings: {
            objectType: '<',
            tooltipContainer: '<?'
        },
        controller: function($filter) {
            this.$onChanges = (changes) => {
                if(changes.objectType) {
                    this.objectNiceName = this.objectType === 'APP' ? 'application' : $filter('niceTaggableType')(this.objectType)
                }
            }
        }
    });

    /** Simple list for numerical values.
     * values: Array of numerical values
     **/
    app.component('numericalList', {
        template: `
        <div class="numerical-list__container">
            <div class="numerical-list__item" ng-repeat="value in $ctrl.values">{{value | smartNumber}}</div>
        </div>
        `,
        bindings: {
            values: '<',
        },
    });

    /**
     * Simple warning block when the currently selected connection for a managed folder doesn't allow managed folders to be defined on itself. Defined as a component simply to avoid code duplication.
     */
    app.component('fsProviderSettingsNoManagedFolderWarning', { // fs-provider-settings-no-managed-folder-warning
        template: `
        <div class="alert alert-warning" ng-if="$ctrl.selectedConnection && $ctrl.connections && !$ctrl.connections.includes($ctrl.selectedConnection)">
            <i class="icon-warning-sign"></i>
            The settings associated with {{$ctrl.selectedConnection}} prevent you to use this connection for managed folders.
        </div>`,
        bindings: {
            connections: '<', // the list of available connections, may be undef while loading from backend
            selectedConnection: '<' // the currently selected connection
        }
    });

    app.component("recipeAllAvailablePartitioningWarning", {
        templateUrl: '/templates/widgets/recipe-all-available-partitioning-warning.html',
    });

    app.component("variablesExpansionLoopConfig", {
        bindings: {
            config: '<',
            actionThatIsRepeated: '@',
            variablesUsageContext: '@',
            excludeDatasetFullName: "<?",
        },
        templateUrl: "/templates/widgets/variables-expansion-loop-config.html",
        controller: function($scope, $stateParams, DatasetsService, SmartId) {
            const ctrl = this;

            ctrl.formId = generateUniqueId();

            DatasetsService.listWithAccessible($stateParams.projectKey).success(function(data) {
                let datasets = data;
                if (ctrl.excludeDatasetFullName) {
                    datasets = datasets.filter(function(dataset) {
                        return dataset.projectKey + "." + dataset.name !== ctrl.excludeDatasetFullName;
                    });
                }
                datasets.forEach(function(dataset) {
                    dataset.smartName = SmartId.create(dataset.name, dataset.projectKey);
                    dataset.localProject = dataset.projectKey === $stateParams.projectKey;
                    dataset.usable = true;
                });
                ctrl.datasets = datasets;
            }).error(setErrorInScope.bind($scope));
        },
    });

    app.component('basicSearchBox', { // basic-search-box
        template: `
        <span class="basic-search-box__addon-icon">
            <i class="dku-icon-search-16"></i>
        </span>
        <input type="search"
            id="{{$ctrl.inputId || $ctrl.fallbackId}}"
            class="basic-search-box__input"
            ng-model="$ctrl.ngModel"
            ng-change="$ctrl.ngModelChange()"
            ng-disabled="$ctrl.disabled"
            placeholder="{{$ctrl.placeholder}}"
        />
        <i ng-if="$ctrl.ngModel" class="dku-icon-dismiss-16" ng-click="$ctrl.clear()"></i>
        `,
        require: { ngModelCtrl: 'ngModel' },
        bindings: {
            placeholder: '@',
            ngModel: '<',
            disabled: '<?',
            autofocus: '<?',
            inputId: '@',
            clickClear: '&?'
        },
        controller: function($element, $timeout) {
            const ctrl = this;

            ctrl.$onInit = function () {
                ctrl.fallbackId = 'basic-search-box-'+Math.round(Math.random()*100000);
                ctrl.placeholder = ctrl.placeholder === undefined ? "Search\u2026" : ctrl.placeholder;
            }

            ctrl.$onChanges = (changes) => {
                // forwarding autofocus directly as an html prop on the input doesn't work well with angularjs compilation
                if(changes.autofocus && ctrl.autofocus) {
                    $timeout(() => $element.find('input').focus());
                }
            }

            ctrl.clear = () => {
                ctrl.ngModelCtrl.$setViewValue('');
                ctrl.clickClear && ctrl.clickClear();
            }

            ctrl.ngModelChange = () => ctrl.ngModelCtrl.$setViewValue(ctrl.ngModel);
        }
    });

    app.component('externalLink', {
        bindings: {
            href: '@'
        },
        transclude: true,
        template: `<a target="_blank" ng-href="{{$ctrl.href}}" ng-click="$ctrl.removeFocus()">
                        <ng-transclude></ng-transclude>
                        <i class="dku-icon-arrow-external-link-12"></i>
                   </a>`,
        controller: function($element) {
            const ctrl = this;
            ctrl.removeFocus = function() {
                document.activeElement.blur();
            };
        }
    });

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.directives.forms', ['dataiku.directives.forms']);

app.directive('checkUnique', function() {
    return {
        require: 'ngModel',
        scope: {
            exclude: '=',
            caseSensitive: '='
        },
        link: function(scope, elem, attrs, ngModel) {

            function format(v) {
                const input = v || '';
                return scope.caseSensitive ? input : input.toLowerCase();
            }

            function apply_validation(value) {
                ngModel.$setValidity('unique', true);
                if (scope.exclude && value) {
                    const valid = !scope.exclude.find(x => format(x) == format(value));
                    ngModel.$setValidity('unique', valid);
                }
                return value;
            }

            //For DOM -> model validation
            ngModel.$parsers.unshift(apply_validation);

            //For model -> DOM validation
            ngModel.$formatters.unshift(function(value) {
                apply_validation(value);
                return value;
            });
        }
    };
});


app.directive("objectTypePicker", function(TAGGABLE_TYPES) {
   return {
       templateUrl: '/templates/widgets/object-type-picker.html',
       restrict: 'A',
       scope: {
            objectTypePicker: '=',     // @param (optional) forwarded to dkuBsSelect
            objectType: '=',           // @model the selected taggable type
            exclude: '<?',             // @param (optional) array of types to exclude (from taggable-types)
            include: '<?',             // @param (optional) array of types to include (ignored if exclude if non-null) - may contain some non-taggable types. Currently supported: APP
            ngDisabled: '=?',
            allOption: '=?'
       },
       link: function($scope) {
            $scope.pickableTypes = TAGGABLE_TYPES;
            $scope.$watchGroup(['exclude', 'include'], ([exclude, include]) => {
                if (exclude) {
                    $scope.pickableTypes = TAGGABLE_TYPES.filter(type => !exclude.includes(type));
                } else if (include) {
                    $scope.pickableTypes = $scope.include;
                } else {
                    $scope.pickableTypes = TAGGABLE_TYPES;
                }
            });
       }
   }
});


app.directive("projectKeyPicker", function(DataikuAPI) {
    return {
        templateUrl: '/templates/widgets/project-key-picker.html',
        restrict: 'AE',
        scope: {
            projectKeyPicker: '=',  // @param (optional) forwarded to dkuBsSelect
            projectKey: '=',        // @model the selected projectKey
            project: '=?',          // @model bound to the selected project
            filter: '=',            // @param (optional) filter project function
        },
        link: function($scope) {
            function findProject() {
                $scope.project = $scope.projects.find(prj => prj.projectKey == $scope.projectKey);
            }
            DataikuAPI.projects.list()
                .success(function (projects) {
                    $scope.projects = $scope.filter ? projects.filter($scope.filter) : projects;
                    findProject();
                });//TODO @errorHandling
            $scope.$watch("projectKey", function() {
                if ($scope.projects) {
                    findProject();
                }
            });
        }
    }
});


app.directive('computablePicker', function(DataikuAPI, $stateParams) {
    return {
        template: '<div dataset-selector="computable" available-datasets="availableComputables"></div>',
        restrict: 'A',
        scope: {
            computable: '=computablePicker',
            type: '@'
        },
        link: function($scope, element) {
            DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {type: $scope.type}).success(function(data) {
                $scope.availableComputables = data;
            }).error(setErrorInScope.bind($scope.$parent));
        }
    };
});


app.directive('objectPicker', function(DataikuAPI, $stateParams, $rootScope, ActiveProjectKey, Logger, translate, $filter) {
    return {
        templateUrl : '/templates/widgets/object-picker.html',
        restrict: 'A',
        scope: {
            objectSmartId: '=objectPicker',                  // @model bound to the smartId of the selected object
            object: '=?',                                    // @model bound to the selected object
            type: '@',                                       // @param AccessibleObject type (taggable or APP)
            unusable: '=?',                                  // @param smartIds of unusable objects (as array, or map: id -> unusable (boolean))
            emptyMessage: '@?',
            selectClass: '@?',
            errorScope: '=?',
            inputAvailableObjects: '<?',                     // @param (array<AvailableObject>, optional) if provided, the list of options & you are responsible for updating it as required (notably when type & projectKey change). If not provided, the component will do the API queries to fetch available objects based on projectKey, type and permission mode. If you are using many instances if this component, you should probably use this input & share the lists of available objects in order to avoid triggering many identical API calls (see AccessibleObjectsCacheService).
            permissionMode: '@?',                            // @param the desired ReaderAuthorization.Mode (defaults to READ)
            hideForeign: '=?',                               // @param (boolean) to hide foreign objects,
            projectKey: '=?',
            disabled: '<?',
            multiSelection: '<?',
            toCustomIcon: '<?',
            toCustomColor: '<?',
            compareWith: '<?',
            filterObjectsFn: '<?'
        },
        link: function($scope) {
            $scope.object = null;
            $scope.defaultPlaceholder = createDefaultPlaceholder($scope.type);
            const projectKey = $scope.projectKey || ActiveProjectKey.get();


            $scope.onChanges = (args) => {
                if (args && Array.isArray(args) && args.length) {
                    $scope.objectSmartId = args.map(arg => arg.smartId);
                    return;
                } else if (args && args.smartId) {
                    $scope.objectSmartId = args.smartId;
                    return;
                }
                $scope.objectSmartId = undefined;
            }

            function findObjects() {
                if ($scope.objectSmartId && Array.isArray($scope.objectSmartId)) {
                    $scope.object = $scope.availableObjects.filter(d => $scope.objectSmartId.includes(d.smartId));
                    return;
                } else if ($scope.objectSmartId) {
                    $scope.object = $scope.availableObjects.filter(d => d.smartId == $scope.objectSmartId)[0];
                    return;
                }
                $scope.object = null;
            }

            function refreshAvailableObjects(availableObjects) {
                if ($scope.filterObjectsFn) {
                    $scope.availableObjects = availableObjects.filter($scope.filterObjectsFn);
                } else {
                    $scope.availableObjects = availableObjects;
                }
                updateUsability();
                findObjects();
            }

            function refreshObjects() {
                if (angular.isDefined($scope.inputAvailableObjects)) {
                    refreshAvailableObjects($scope.inputAvailableObjects);
                    return;
                }

                if (!projectKey && !['PROJECT', 'WORKSPACE', 'DATA_COLLECTION', 'APP'].includes($scope.type)) {
                    Logger.info('No project key specified, not listing accessible objects');
                    return;
                }

                DataikuAPI.taggableObjects.listAccessibleObjects(projectKey, $scope.type, $scope.permissionMode).success(function(data) {
                    refreshAvailableObjects(data)
                }).error(function(data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope.errorScope || $scope.$parent)(data, status, headers, config, statusText);
                });
            }

            function createDefaultPlaceholder(type) {
                const translatedType = translate('TAGGABLE_TYPE.'+type,type);
                return translate('SCENARIO.INTEGRATIONS.INTEGRATION_PARAMS.ATTACHMENTS.OBJECT_PICKER.NO_OBJECT_SELECTED','No {{type}} selected', {type: $filter('niceConst')(translatedType, ' ', true) });
            }

            $scope.$watch("type", function(nv, ov) {
                $scope.defaultPlaceholder = createDefaultPlaceholder($scope.type)

                if ($scope.type === "SAVED_MODEL") {
                    $scope.toCustomIcon = (item) => {
                        const object = item.object;
                        const taskType = (object.miniTask || {}).taskType;
                        const backendType = (object.miniTask || {}).backendType;
                        const predictionType = (object.miniTask || {}).predictionType;
                        const savedModelType = object.savedModelType;
                        const proxyModelProtocol = (object.proxyModelConfiguration || {}).protocol;
                        return $filter("savedModelSubtypeToIcon")(taskType, backendType, predictionType, savedModelType, proxyModelProtocol, 24);
                    }

                    $scope.toCustomColor = (item) => {
                        return $filter("savedModelTypeToClassColor")(item.object.savedModelType, true);
                    }
                }

                if (!nv || (nv == ov && $scope.availableObjects)) return;

                if (ov && nv != ov) {
                    $scope.object = null;
                    $scope.objectSmartId = null;
                }

                refreshObjects();
            });

            $scope.$watch("permissionMode", function(nv, ov) {
                if (!nv || !ov || nv == ov) return;
                refreshObjects();
            });

            $scope.$watch('objectSmartId', function() {
                if ($scope.availableObjects) {
                    findObjects()
                }
            }, true);

            $scope.$watch('object', function() {
                if ($scope.availableObjects && Array.isArray($scope.object)) {
                    $scope.objectSmartId = $scope.object.map(o => o.smartId);
                } else if ($scope.availableObjects && $scope.object && $scope.object.smartId) {
                    $scope.objectSmartId = $scope.object.smartId;
                }
            });

            $scope.$watch('inputAvailableObjects', function() {
                refreshObjects();
            });

            $scope.$watch(function(scope) {
                if (angular.isObject(scope.unusable)) {
                    var simplified = {};
                    for (var key in scope.unusable) {
                        simplified[key] = !!scope.unusable[key];
                    }
                    return simplified;
                } else {
                    return scope.unusable;
                }
            }, updateUsability, true);

            function updateUsability(nv, ov) {
                if (!$scope.availableObjects) return;

                if (!$scope.unusable) {
                    $scope.availableObjects = $scope.availableObjects.map(function (item) {
                        return {
                            ...item,
                            usable: true,
                        }
                    });
                } else if (angular.isArray($scope.unusable)) {
                    $scope.availableObjects = $scope.availableObjects.map(function (item) {
                        return {
                            ...item,
                            usable: ($scope.unusable || []).indexOf(item.smartId) == -1,
                        }
                    });
                } else if (angular.isObject($scope.unusable)) {
                    $scope.availableObjects = $scope.availableObjects.map(function (item) {
                        return {
                            ...item,
                            usable: !$scope.unusable[item.smartId],
                        }
                    });
                }
            }
        }
    };
});

app.directive('dashboardTilePicker', function(DataikuAPI, ActiveProjectKey, Logger) {
    return {
        templateUrl : '/templates/widgets/dashboard-tile-picker.html',
        restrict: 'A',
        scope: {
            objectSmartId: '=dashboardTilePicker',      // @model bound to the smartId of the selected object
            object: '=?',                               // @model bound to the selected object
            sourceObject: '=?',                         // @model bound to the dataset source of selected tile object
            objectChanged: '&',                                    
            emptyMessage: '@?',
            availableTiles: '=?',
        },
        link: function($scope) {
            $scope.availableTiles && $scope.availableTiles.forEach((item) => {
                item.usable = true;
            });

            const projectKey = $scope.projectKey || ActiveProjectKey.get();

            function findObjectSource() {
                if (!$scope.availableObjects) return;
                $scope.sourceObject = $scope.availableObjects.find(function (d) {
                    return d.smartId == $scope.object.datasetSmartName;
                });
            }

            function fetchSources() {
                if (!projectKey && $scope.type != 'PROJECT') {
                    Logger.info('No project key specified, not listing accessible objects');
                    return;
                }
                DataikuAPI.taggableObjects.listAccessibleObjects(projectKey, "DATASET", $scope.permissionMode).success(function (data) {
                    $scope.availableObjects = data;
                }).error(function (data, status, headers, config, statusText) {
                    setErrorInScope.bind($scope.errorScope || $scope.$parent)(data, status, headers, config, statusText);
                });
            }

            fetchSources();

            $scope.$watch('objectSmartId', (nv) => {
                if (!nv) { return }
                for (let insight of $scope.availableTiles) {
                    if (insight.id === nv) {
                        $scope.object = insight;
                        findObjectSource();
                        break;
                    }
                }
            });
        }
    };
});

app.service('PopoverPosition', function() {

    this.computePosition = function($scope, element, attrs, popover) {
        var mainZone = $(".mainzone", element);
        popover.css("width", attrs.popoverWidth ? attrs.popoverWidth : Math.max(500, mainZone.width() + 20));
        if ($scope.popoverPlacement == 'right') {
            let ccsLeft = mainZone.offset().left - popover.width() + mainZone.width() +4;
            popover.css("left", ccsLeft > 0 ? ccsLeft: 1);
        } else {
            popover.css("left", mainZone.offset().left);
        }

        let cssTop = mainZone.offset().top + mainZone.height() + ($scope.marginTop ? parseInt($scope.marginTop) : 8);
        if ($scope.popoverPlacement == 'auto') {
            if (mainZone.offset().top + mainZone.height() + 280 > $(window).height()) {
                cssTop = mainZone.offset().top - 310;
            }
        }
        if (cssTop + mainZone.height() + popover.height() > $(window).height()) {
            // + 12 for (pad 10 + border 1 and li pad 1)
            cssTop = $(window).height() - popover.height() -  mainZone.height() + 13;
        }
        popover.css("top", cssTop);
    };

});

app.directive('datasetSelector', function(ListFilter, $compile, PopoverPosition, translate) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableDatasets : '=',
            datasetSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?',
            hideForeign: '=?',
            emptyMessage: '@?',
            multi: '@',
            hideGroupTitles: '@?',
            onDatasetSelectorChange: '&?',
            discriminationProperties: '@?',
            useNewIcons: '<?', // this argument is used to migrate gradually to the new icons, todo: it should be removed when all the icons are migrated
        },
        templateUrl : '/templates/dataset-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            $scope.translate = translate;

            var popover = null;

            if ($scope.multi) {
                $scope.noLiveUpdate = true;
                $scope.datasetSelector = $scope.datasetSelector || [];
            } 

            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
                $scope.$apply();
            });

            /* List management */
            function update() {
                $scope.displayedDatasets = [];
                if (!$scope.availableDatasets) return;

                const headerObjects = $scope.availableDatasets.filter(dataset => dataset.header);
                const otherObjects = $scope.availableDatasets.filter(dataset => !dataset.header);
                $scope.filtered = otherObjects;
                
                $scope.filtered = ListFilter.filter(otherObjects, $scope.filter.query, $scope.discriminationProperties, !!$scope.discriminationProperties);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        if ($scope.hideForeign) {
                            continue;
                        }
                        group = translate("FLOW.CREATE_RECIPE.PROJECT", "Project: ") +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, datasets : [], sort:sort}
                    }
                    groups[group].datasets.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].datasets.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.displayedGroups.unshift({
                    datasets: headerObjects,
                    header: true
                });

                $scope.currentlySelected = null;
                for (let i in $scope.availableDatasets) {
                    // datasets generally have a smartName/smartId; other objects may only have an id
                    if (($scope.availableDatasets[i].smartName || $scope.availableDatasets[i].smartId || $scope.availableDatasets[i].id) == $scope.datasetSelector) {
                        $scope.currentlySelected = $scope.availableDatasets[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", update, true);
            $scope.$watchCollection("availableDatasets", update);
            update();

            /* Model management */

            $scope.select = function(details) {
                const itemId = details.smartName || details.smartId || details.id;
                if (!$scope.multi) {
                    $scope.datasetSelector = itemId;
                    hide();
                } else {
                    $scope.datasetSelector.push(itemId);
                }
            };

            $scope.itemClicked = function(details) {
                const itemId = details.smartName || details.smartId || details.id;

                if (!$scope.multi) {
                    if (itemId === $scope.datasetSelector) {
                        $scope.datasetSelector = null;
                    } else {
                        $scope.datasetSelector = itemId;
                    }
                    hide();
                } else {
                    const detailsIndex = $scope.datasetSelector.indexOf(itemId);
                    if (detailsIndex >= 0) {
                        $scope.datasetSelector.splice(detailsIndex, 1);
                    } else {
                        $scope.datasetSelector.push(itemId);
                    }
                }
            };

            $scope.isItemSelected = function(item) {
                const itemId = item.smartName || item.smartId || item.id;

                if (!$scope.multi) {
                    return $scope.datasetSelector === itemId;
                } else {
                    return $scope.datasetSelector.indexOf(itemId) >= 0;
                }
            }

            $scope.$watch("datasetSelector", function(newValue, oldValue) {
                update();
                if (angular.isFunction($scope.onDatasetSelectorChange)) {
                    $scope.onDatasetSelectorChange();
                }
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();
            
            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };

            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");

                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });

                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    // Up and down in list
                    if(event.keyCode==38 || event.keyCode==40) {
                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.datasets.length ; j++) {
                                    var ds = group.datasets[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }
                             $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.datasetSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } 
                    // Enter
                    else if (event.keyCode === 13) {
                        if ($scope.currentlySelected) {
                            $scope.itemClicked($scope.currentlySelected);
                            $scope.$apply();
                        }
                    } else {
                        $scope.currentlySelected = null;
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


//TODO @dssObjects factorize
app.directive('savedModelSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableSavedModels : '=',
            savedModelSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/model-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedSavedModels = [];
                if (!$scope.availableSavedModels) return;

                $scope.filtered = $scope.availableSavedModels;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableSavedModels, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, savedModels : [], sort:sort}
                    }
                    groups[group].savedModels.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].savedModels.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableSavedModels) {
                    if (($scope.availableSavedModels[i].smartName || $scope.availableSavedModels[i].smartId) == $scope.savedModelSelector) {
                        $scope.currentlySelected = $scope.availableSavedModels[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableSavedModels", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.savedModelSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.savedModelSelector) {
                    $scope.savedModelSelector = null;
                } else {
                    $scope.savedModelSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("savedModelSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.savedModels.length ; j++) {
                                    var ds = group.savedModels[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.savedModelSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


//TODO @dssObjects factorize
app.directive('modelEvaluationStoreSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableModelEvaluationStores : '=',
            modelEvaluationStoreSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/evaluation-store-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedModelEvaluationStores = [];
                if (!$scope.availableModelEvaluationStores) return;

                $scope.filtered = $scope.availableModelEvaluationStores;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableModelEvaluationStores, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, modelEvaluationStores : [], sort:sort}
                    }
                    groups[group].modelEvaluationStores.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (let g in groups) {
                    groups[g].modelEvaluationStores.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableModelEvaluationStores) {
                    if (($scope.availableModelEvaluationStores[i].smartName || $scope.availableModelEvaluationStores[i].smartId) == $scope.modelEvaluationStoreSelector) {
                        $scope.currentlySelected = $scope.availableModelEvaluationStores[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableModelEvaluationStores", function() {
                update();
            }, true);
            update();

            /* Store management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.modelEvaluationStoreSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.modelEvaluationStoreSelector) {
                    $scope.modelEvaluationStoreSelector = null;
                } else {
                    $scope.modelEvaluationStoreSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("modelEvaluationStoreSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.modelEvaluationStores.length ; j++) {
                                    var ds = group.modelEvaluationStores[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.modelEvaluationStoreSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});

//TODO @dssObjects factorize
app.directive('apiEndpointSelector', function($timeout, ListFilter, $compile, PopoverPosition, WT1) {
    var ret = {
        restrict : 'A',
        scope : {
            availableApiEndpoints : '=',
            apiEndpointSelector : '=',
            entryEndpointId: '=',
            popoverPlacement: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?',
            wt1SelectEventName: '<?'
        },
        templateUrl : '/templates/api-endpoint-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;

            /* List management */
            function update() {
                $scope.displayedApiEndpoints = [];
                if (!$scope.availableApiEndpoints) return;

                $scope.filtered = $scope.availableApiEndpoints;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableApiEndpoints, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    group =  $scope.filtered[i].id[0].toUpperCase();
                    sort = "AAAAAAAA" + group;
                    if (! groups[group]) {
                         groups[group] = {title : group, apiEndpoints : [], sort:sort}
                    }
                    groups[group].apiEndpoints.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (let g in groups) {
                    groups[g].apiEndpoints.sort(function(a,b) { return a.id.localeCompare(b.id)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                if ($scope.apiEndpointSelector) {
                    for (let i in $scope.availableApiEndpoints) {
                        if (($scope.availableApiEndpoints[i].id) === $scope.apiEndpointSelector.id) {
                            $scope.currentlySelected = $scope.availableApiEndpoints[i];
                        }
                    }
                }
            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableApiEndpoints", function() {
                update();
            }, true);
            update();

            /* Store management */

            $scope.select = function(details) {
                $scope.apiEndpointSelector = details;
                hide();
            };

            $scope.itemClicked = function(details) {
                if ($scope.wt1SelectEventName) {
                    WT1.event($scope.wt1SelectEventName);
                }
                $scope.apiEndpointSelector = details;
                hide();
            };

            $scope.$watch("apiEndpointSelector", function(newValue, oldValue) {
                update();
            });

            $scope.typeToIcon = function(type) {
                switch (type) {
                    case "STD_PREDICTION":
                        return "dku-icon-modelize-24";
                    case "STD_FORECAST":
                        return "dku-icon-time-series-24";
                    case "STD_CAUSAL_PREDICTION":
                        return "dku-icon-lever-24";
                    case "R_FUNCTION":
                        return "dku-icon-r-24";
                    case "PY_FUNCTION":
                        return "dku-icon-python-24";
                    case "DATASETS_LOOKUP":
                        return "dku-icon-dataset-24";
                    case "SQL_QUERY":
                        return "dku-icon-sql-24";
                    case "STD_CLUSTERING":
                        return "dku-icon-clustering-24";
                    case "CUSTOM_PREDICTION":
                        return "dku-icon-custom-prediction-python-24";
                    case "CUSTOM_R_PREDICTION":
                        return "dku-icon-custom-prediction-r-24";
                    default:
                        return "dku-icon-gear-24";
                }
            }

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.apiEndpoints.length ; j++) {
                                    var ds = group.apiEndpoints[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.apiEndpointSelector = $scope.currentlySelected;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});

//TODO @dssObjects factorize
app.directive('folderSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableFolders : '=',
            folderSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/folder-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedFolders = [];
                if (!$scope.availableFolders) return;

                $scope.filtered = $scope.availableFolders;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableFolders, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, folders : [], sort:sort}
                    }
                    groups[group].folders.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].folders.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableFolders) {
                    if (($scope.availableFolders[i].smartName || $scope.availableFolders[i].smartId) == $scope.folderSelector) {
                        $scope.currentlySelected = $scope.availableFolders[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableFolders", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.folderSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.folderSelector) {
                    $scope.folderSelector = null;
                } else {
                    $scope.folderSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("folderSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                }
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.folders.length ; j++) {
                                    var ds = group.folders[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.folderSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
}
    return ret;
});


//TODO @dssObjects factorize
app.directive('labelingTaskSelector', function($timeout, ListFilter, $compile, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableLabelingTasks : '=',
            labelingTaskSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/labeling-task-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedLabelingTasks = [];
                if (!$scope.availableLabelingTasks) return;

                $scope.filtered = $scope.availableLabelingTasks;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableLabelingTasks, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, labelingTasks : [], sort:sort}
                    }
                    groups[group].labelingTasks.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (let g in groups) {
                    groups[g].labelingTasks.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableLabelingTasks) {
                    if (($scope.availableLabelingTasks[i].smartName || $scope.availableLabelingTasks[i].smartId) == $scope.labelingTaskSelector) {
                        $scope.currentlySelected = $scope.availableLabelingTasks[i];
                    }
                }

            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableLabelingTasks", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.labelingTaskSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.labelingTaskSelector) {
                    $scope.labelingTaskSelector = null;
                } else {
                    $scope.labelingTaskSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("labelingTaskSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.labelingTasks.length ; j++) {
                                    var ds = group.labelingTasks[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.labelingTaskSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
}
    return ret;
});


//TODO @dssObjects factorize
app.directive('streamingEndpointSelector', function($timeout, ListFilter, $compile, StateUtils, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableStreamingEndpoints : '=',
            streamingEndpointSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/streaming-endpoint-selector.html',
    };

    ret.compile = function(element, attrs) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {
        
            $scope.StateUtils = StateUtils; 
            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedStreamingEndpoints = [];
                if (!$scope.availableStreamingEndpoints) return;

                $scope.filtered = $scope.availableStreamingEndpoints;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableStreamingEndpoints, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, streamingEndpoints : [], sort:sort}
                    }
                    groups[group].streamingEndpoints.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].streamingEndpoints.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableStreamingEndpoints) {
                    if (($scope.availableStreamingEndpoints[i].smartName || $scope.availableStreamingEndpoints[i].smartId) == $scope.streamingEndpointSelector) {
                        $scope.currentlySelected = $scope.availableStreamingEndpoints[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableStreamingEndpoints", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.streamingEndpointSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.streamingEndpointSelector) {
                    $scope.streamingEndpointSelector = null;
                } else {
                    $scope.streamingEndpointSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("streamingEndpointSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                } 
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.streamingEndpoints.length ; j++) {
                                    var ds = group.streamingEndpoints[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.streamingEndpointSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});


//TODO @dssObjects factorize
app.directive('knowledgeBankSelector', function($timeout, ListFilter, $compile, StateUtils, PopoverPosition) {
    var ret = {
        restrict : 'A',
        transclude: true,
        scope : {
            type:'@',
            availableKnowledgeBanks : '=',
            knowledgeBankSelector : '=',
            transclude : '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate : '@',
            ngDisabled: '=?'
        },
        templateUrl : '/templates/knowledge-bank-selector.html',
    };

    ret.compile = function(element) {
        var popoverTemplate = element.find('.popover').detach();
        return function($scope, element, attrs) {

            $scope.StateUtils = StateUtils;
            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            /* List management */
            function update() {
                $scope.displayedKnowledgeBanks = [];
                if (!$scope.availableKnowledgeBanks) return;

                $scope.filtered = $scope.availableKnowledgeBanks;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableKnowledgeBanks, $scope.filter.query);

                var groups = {}
                for (var i in  $scope.filtered) {
                    var group = "";
                    var sort = "";
                    if ( $scope.filtered[i].localProject) {
                        group =  ($scope.filtered[i].label || $scope.filtered[i].name) [0].toUpperCase();
                        sort = "AAAAAAAA" + group;
                    } else {
                        group = "Project: " +  $scope.filtered[i].projectKey;
                        sort = group;
                    }
                    if (! groups[group]) {
                         groups[group] = {title : group, knowledgeBanks : [], sort:sort}
                    }
                    groups[group].knowledgeBanks.push( $scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].knowledgeBanks.sort(function(a,b) { return (a.label || a.name).localeCompare(b.label || b.name)})
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a,b) { return a.sort.localeCompare(b.sort)})

                $scope.currentlySelected = null;
                for (let i in $scope.availableKnowledgeBanks) {
                    if (($scope.availableKnowledgeBanks[i].smartName || $scope.availableKnowledgeBanks[i].smartId) == $scope.knowledgeBankSelector) {
                        $scope.currentlySelected = $scope.availableKnowledgeBanks[i];
                    }
                }


            }
            $scope.filter = {
                allProjects : true,
                query : ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableKnowledgeBanks", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                //ngModel.$setViewValue(details.smartName);
                $scope.knowledgeBankSelector = details.smartName || details.smartId;
                hide();
            };

            $scope.itemClicked = function(details) {
                //ngModel.$setViewValue(details.smartName);
                if ((details.smartName || details.smartId) == $scope.knowledgeBankSelector) {
                    $scope.knowledgeBankSelector = null;
                } else {
                    $scope.knowledgeBankSelector = details.smartName || details.smartId;
                }
                hide();
            };

            $scope.$watch("knowledgeBankSelector", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                }
            };

            var hide = function() {
                popover.hide().detach();
                $("html").unbind("click", globalClickListenener);
                popoverShown=false;
            };
            var show = function() {
                popoverShown = true;
                if (popover == null) {
                    popover = $compile(popoverTemplate.clone())($scope);
                }
                popover.appendTo("body");
                PopoverPosition.computePosition($scope, element, attrs, popover);
                popover.show();

                popover.find("input").off('blur.dsSelector').on('blur.dsSelector',function() {
                    popover.find("input").focus();
                });
                popover.find("input").focus();

                popover.off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                $(".mainzone", element).off("click.dku-pop-over").on("click.dku-pop-over", function(e) {
                    //e.stopPropagation();
                });
                window.setTimeout(function() { $("html").click(globalClickListenener)}, 0);

                popover.find("input").off('keydown.dsSelector').on('keydown.dsSelector',function(event) {

                    if(event.keyCode==38 || event.keyCode==40 || event.keyCode==13) {
                        event.stopPropagation();
                    } else {
                        return;
                    }

                    if(event.keyCode==38 || event.keyCode==40) {

                        if($scope.displayedGroups &&  $scope.displayedGroups.length>0) {

                            var previous = null;
                            var next = null;
                            var current = null;
                            var first = null;
                            var foundCurrent = false;
                            var last = null;
                            var updateNext = false;

                            for(var k = 0 ; k < $scope.displayedGroups.length ; k++) {
                                var group = $scope.displayedGroups[k];
                                for(var j = 0 ; j < group.knowledgeBanks.length ; j++) {
                                    var ds = group.knowledgeBanks[j];
                                    if(!ds.usable) {
                                        continue;
                                    }
                                    if(!first) {
                                        first = ds;
                                    }
                                    last = ds;
                                    if(foundCurrent) {
                                        if(updateNext) {
                                            next = ds;
                                            updateNext=false;
                                        }
                                    } else {
                                        previous = current;
                                        current = ds;

                                        if($scope.currentlySelected == ds) {
                                            foundCurrent = true;
                                            updateNext = true;
                                        }
                                    }
                                }
                            }

                            $scope.$apply(function() {
                                if(foundCurrent) {
                                    if(event.keyCode == 40) {
                                        if(next) {
                                            $scope.currentlySelected = next;
                                        } else {
                                            $scope.currentlySelected = first;
                                        }
                                    }
                                    if(event.keyCode == 38) {
                                        if(previous) {
                                            $scope.currentlySelected = previous;
                                        } else {
                                            $scope.currentlySelected = last;
                                        }
                                    }
                                } else {
                                    if(first) {
                                        $scope.currentlySelected = first;
                                    }
                                }

                                if($scope.currentlySelected && !$scope.noLiveUpdate) {
                                    $scope.knowledgeBankSelector = $scope.currentlySelected.smartName || $scope.currentlySelected.smartId;
                                }
                            });
                        }

                    } else if(event.keyCode==13) {
                        if($scope.currentlySelected) {
                            $scope.select($scope.currentlySelected);
                            $scope.$apply();
                        }
                    }

                });
            };

            $scope.togglePopover =function() {
                if (popoverShown) hide();
                else show();
            }

            $scope.$on('$destroy', function() {
                if (popoverShown) hide();
            });
        }
    }
    return ret;
});

//TODO @dssObjects factorize
app.directive('agentToolSelector', function($timeout, $filter, ListFilter, $compile, StateUtils, SmartId, PopoverPosition) {
    var ret = {
        restrict: 'A', transclude: true, scope: {
            type: '@',
            availableAgentTools: '<',
            contextProject: '<',
            agentToolSelector: '=',
            transclude: '@',
            popoverPlacement: '@',
            marginTop: '@',
            noLiveUpdate: '@',
            ngDisabled: '=?'
        }, templateUrl: '/templates/agent-tool-selector.html',
    };

    ret.compile = function(element) {
        var popoverTemplate = element.find('.popover').detach();

        return function($scope, element, attrs) {
            $scope.StateUtils = StateUtils;
            var popover = null;
            if ($scope.transclude) $(element).on('click', function(event) {
                if (event && event.target && event.target.hasAttribute('href') || $scope.ngDisabled) return;
                $scope.togglePopover();
            });

            $scope.isToolAvailable = function() {
                if (!$scope.currentlySelected) {
                    return false;
                }
                if (!$scope.availableAgentTools) {
                    return false;
                }
                return $scope.availableAgentTools.find(tool => tool.toolRef === $scope.currentlySelected.toolRef) !== undefined;
            };

            /* List management */
            function update() {
                $scope.displayedAgentTools = [];
                if (!$scope.availableAgentTools) return;

                $scope.availableAgentTools = $scope.availableAgentTools.map(tool => ({
                    ...tool,
                    toolRef: SmartId.fromTor(tool, $scope.contextProject),
                    localProject: tool.projectKey === $scope.contextProject,
                    usable: true,
                }));

                $scope.filtered = $scope.availableAgentTools;

                // Filter on terms
                $scope.filtered = ListFilter.filter($scope.availableAgentTools, $scope.filter.query);

                var groups = {}
                for (var i in $scope.filtered) {
                    var group = $filter('niceAgentToolType')($scope.filtered[i].type);
                    if (!groups[group]) {
                        groups[group] = { title: group, agentTools: [], sort: group }
                    }
                    groups[group].agentTools.push($scope.filtered[i]);
                }
                $scope.displayedGroups = [];
                for (var g in groups) {
                    groups[g].agentTools.sort(function(a, b) {
                        if (a.localProject && !b.localProject) {
                            return -1;
                        }
                        if (b.localProject && !a.localProject) {
                            return 1;
                        }
                        return (a.label || a.name).localeCompare(b.label || b.name)
                    })
                    $scope.displayedGroups.push(groups[g]);
                }
                $scope.displayedGroups.sort(function(a, b) {
                    return a.sort.localeCompare(b.sort)
                })

                $scope.currentlySelected = null;
                for (let i in $scope.availableAgentTools) {
                    if ($scope.agentToolSelector.toolRef == $scope.availableAgentTools[i].toolRef) {
                        $scope.currentlySelected = $scope.availableAgentTools[i];
                    }
                }

                // ghost item
                if ($scope.agentToolSelector && $scope.agentToolSelector.toolRef && !$scope.currentlySelected) {
                    const smartId = SmartId.resolve($scope.agentToolSelector.toolRef, $scope.contextProject);
                    $scope.currentlySelected = {
                        toolRef: $scope.agentToolSelector.toolRef,
                        projectKey: smartId.projectKey,
                        id: smartId.id,
                        usable: false,
                    };
                }
            }

            $scope.filter = {
                allProjects: true, query: ""
            }
            $scope.$watch("filter", function() {
                update();
            }, true);
            $scope.$watch("availableAgentTools", function() {
                update();
            }, true);
            update();

            /* Model management */

            $scope.select = function(details) {
                $scope.agentToolSelector.toolRef = details.toolRef;
                hide();
            };

            $scope.itemClicked = function(details) {
                if (!$scope.agentToolSelector) {
                    $scope.agentToolSelector = {};
                }
                if ($scope.agentToolSelector.toolRef == details.toolRef) {
                    $scope.agentToolSelector.toolRef = "";
                } else {
                    $scope.agentToolSelector.toolRef = details.toolRef;
                }
                hide();
            };

            $scope.$watch("agentToolSelector", function(newValue, oldValue) {
                update();
            });
            $scope.$watch("agentToolSelector.toolRef", function(newValue, oldValue) {
                update();
            });

            /* Popover management */
            var popoverShown = false;
            $(popover).hide();

            var globalClickListenener = function(event) {
                // Do not close the popover when clicking on it (outside of a clickable item)
                if (!event.target.closest('.dss-object-selector-popover')) {
                    hide();
                }
            };

            var hide = function() {
                popover.hide().det