(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("AfgRecipeCreationController", function($scope, $controller, $stateParams, RecipeComputablesService) {
        $scope.recipeType = "generate_features";

        $scope.recipe = {
            type: 'generate_features',
            projectKey: $stateParams.projectKey,
            inputs: {
                main: {
                    items: []
                }
            },
            outputs: {
                main: {
                    items: []
                }
            }
        };

        RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map) {
            // Retrieve all the datasets of the project and store them in $scope.computablesMap.
            // The object computablesMap maps the name of a dataset with its configuration.
            $scope.setComputablesMap(map);
        });

        $scope.inputDatasetsOnly = true;
        $controller("SingleOutputDatasetRecipeCreationController", { $scope: $scope });

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                const niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./, "");
                $scope.maybeSetNewDatasetName(niceInputName + "_feats");
            }
        };

        $scope.getCreationSettings = function() {
            let inputs = [];
            if ($scope.io.inputDataset !== undefined) {
                inputs.push($scope.io.inputDataset);
            }
            return { virtualInputs: inputs };
        };

        $scope.showOutputPane = function() {
            return !!($scope.io.inputDataset);
        };

        $scope.$watchCollection('recipe.inputs.main.items', function(nv) {
            if (nv && nv.length) {
                $scope.io.inputDataset = nv[0].ref;
            }
        })
    });

    app.controller("AfgRecipeController", function($scope, $controller, $q, $stateParams, Logger, $timeout, CreateModalFromTemplate, RecipesUtils, AutoFeatureGenerationRecipeService, Dialogs, DatasetUtils, DataikuAPI) {
        $scope.hooks.onRecipeLoaded = function() {
            Logger.info("On Recipe Loaded");
            $scope.$watch("script.data", onScriptChanged);
            // the onScriptChanged will be called because adding a $watch on the scope triggers an 'initialization' run
        };
        var visualCtrl = $controller('VisualRecipeEditorController', { $scope: $scope }); //Controller inheritance
        $scope.specificControllerLoadedDeferred.resolve();

        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

        $scope.hooks.getPayloadData = function() {
            if (!$scope.params || Object.keys($scope.params).length === 0) return;
            return angular.toJson($scope.params);
        };

        $scope.fixUpParams = function(params) {
            /**
             * Fixes recipe params in case input columns have been deleted or renamed by Removing selected columns that dont exist
             */
            const allColumns = {};
            for (let i = 0; i < params.virtualInputs.length; i++) {
                allColumns[i] = $scope.getAfgColumns(AutoFeatureGenerationRecipeService.getDatasetName(i, params.virtualInputs, $scope.recipe.inputs));
            }
            AutoFeatureGenerationRecipeService.filterSelectedColumns(params, allColumns);
        }

        function onScriptChanged(nv, ov) {
            if (nv) {
                $scope.params = JSON.parse($scope.script.data);
                visualCtrl.saveServerParams(); //keep for dirtyness detection
                $scope.fixUpParams($scope.params);
                $scope.hooks.updateRecipeStatus();
            }
        }

        $scope.uiState = {
            currentStep: 'dataRelationships'
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) {
                return $q.reject("payload not ready");
            }
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, { reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan }).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) {
                    return deferred.reject();
                }
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.$watchCollection("recipe.outputs.main.items", function() {
            const outputs = RecipesUtils.getOutputsForRole($scope.recipe, "main");
            if (outputs.length === 1) {
                $scope.outputDatasetName = outputs[0].ref;
            }
            $scope.updateRecipeStatusLater();
        });

        $scope.getDateColumnsFromDataset = function(datasetName) {
            const allColumns = $scope.getColumns(datasetName);
            return AutoFeatureGenerationRecipeService.getDateColumns(allColumns);
        }

        $scope.showNewDatasetModal = function(virtualIndex) {
            $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
            $scope.editTimeSettings = false;
            $scope.tableOneIndex = virtualIndex;
            $scope.tableTwoIndex = $scope.params.virtualInputs.length;
            CreateModalFromTemplate("/static/dataiku/auto-feature-generation/dataset-modal/dataset-modal.component.html", $scope);
        };

        $scope.showEditCutoffTimeModal = function() {
            $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
            CreateModalFromTemplate("/static/dataiku/auto-feature-generation/cutoff-time-modal/cutoff-time-modal.component.html", $scope);
        }

        $scope.showEditTimeSettingsModal = function(index) {
            $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
            $scope.editTimeSettings = true;
            $scope.tableOneIndex = null;
            $scope.tableTwoIndex = index;
            $scope.timeIndexMode = $scope.timeIndexColumn ? AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DATE_COLUMN.name : AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DEFAULT.name;
            CreateModalFromTemplate("/static/dataiku/auto-feature-generation/dataset-modal/dataset-modal.component.html", $scope);
        }

        $scope.disableEditTimeSettingsModal = function(virtualInputIndex) {
            const datasetName = $scope.getDatasetName(virtualInputIndex);
            const hasDateColumns = AutoFeatureGenerationRecipeService.hasDateColumns(datasetName, $scope.computablesMap);
            return AutoFeatureGenerationRecipeService.disableTimeIndexEdition($scope.params.cutoffTime.mode, hasDateColumns, $scope.params.virtualInputs[virtualInputIndex].timeIndexColumn);
        }

        $scope.disableEditCutoffTimeModal = function(){
            const primaryDatasetName = $scope.getDatasetName(0);
            return !AutoFeatureGenerationRecipeService.hasDateColumns(primaryDatasetName, $scope.computablesMap) && $scope.params.cutoffTime.mode !== AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name;
        }

        $scope.setCutoffTimeTooltip = function(){
            return $scope.disableEditCutoffTimeModal() ? "No date columns found." : null;
        }

        $scope.setTimeIndexTooltip = function(virtualInputIndex) {
            const datasetName = $scope.getDatasetName(virtualInputIndex);
            const hasDateColumns = AutoFeatureGenerationRecipeService.hasDateColumns(datasetName, $scope.computablesMap);
            if(!hasDateColumns) {
                return "No date columns found.";
            } else if (!AutoFeatureGenerationRecipeService.isCutoffTimeDate($scope.params.cutoffTime.mode)) {
                return "Cutoff time is required to configure enrichment dataset time settings.";
            } return null;
        }

        $scope.isCutoffTimeShown = function() {
            return AutoFeatureGenerationRecipeService.isCutoffTimeShown($scope.params.virtualInputs.length, $scope.params.cutoffTime.mode);
        }

        $scope.isTimeIndexDefined = function(index) {
            if (index < $scope.params.virtualInputs.length && $scope.params.virtualInputs[index].timeIndexColumn) {
                return true;
            }
            return false;
        }

        $scope.getTimeIndexValue = function(index) {
            if (index < $scope.params.virtualInputs.length && $scope.params.virtualInputs[index].timeIndexColumn) {
                return $scope.params.virtualInputs[index].timeIndexColumn;
            }
            return AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DEFAULT.label;
        }

        $scope.isTimeWindowDefined = function(index){
            return $scope.params.virtualInputs[index].timeWindows.length;
        }

        $scope.getTimeWindowValue = function(index){
            const timeWindow = $scope.params.virtualInputs[index].timeWindows[0];
            return timeWindow.from + " to " + timeWindow.to + " " + AutoFeatureGenerationRecipeService.TIME_UNITS[timeWindow.windowUnit].label + " before cutoff time";
        }

        $scope.isCutoffTimeDefined = function() {
            if ($scope.params.virtualInputs.length && $scope.params.virtualInputs[0].timeIndexColumn) {
                return true;
            }
            return false;
        }

        $scope.getCutoffTimeValue = function() {
            if ($scope.params.virtualInputs.length && $scope.params.virtualInputs[0].timeIndexColumn) {
                return $scope.params.virtualInputs[0].timeIndexColumn;
            }
            return AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DEFAULT.label;
        }

        $scope.relationshipDesc = AutoFeatureGenerationRecipeService.relationshipDesc;

        $scope.getAfgColumns = function(datasetName) {
            return angular.copy($scope.getColumns(datasetName));
        }

        $scope.getDatasetsList = function() {
            return AutoFeatureGenerationRecipeService.getUsedDatasets($scope.params.virtualInputs, $scope.recipe.inputs);
        }

        $scope.addEmptyCondition = function(relationship, current) {
            const newCondition = {
                column1: {
                    table: relationship.table1,
                    name: $scope.getAfgColumns($scope.getDatasetName(relationship.table1))[0].name
                },
                column2: {
                    table: relationship.table2,
                    name: $scope.getAfgColumns($scope.getDatasetName(relationship.table2))[0].name
                },
                type: 'EQ'
            };
            relationship.on = relationship.on || [];
            relationship.on.push(newCondition);
            if (current) {
                current.condition = newCondition;
            }
            //  Update recipe status to validate the conditions of the relationship
            if (relationship.on.length === 1) {
                $scope.hooks.updateRecipeStatus();
            }
        };

        $scope.removeCondition = function(scope, relationship, condition) {
            if (scope.current && scope.current.condition === condition) {
                scope.current.condition = null;
            }
            const index = relationship.on.indexOf(condition);
            relationship.on.splice(index, 1);
            //  Update recipe status to validate the new conditions
            $scope.hooks.updateRecipeStatus();
        };


        $scope.removeAllConditions = function(scope, relationship) {
            if ( scope.current != null ) {
                scope.current.condition = null;
            }
            relationship.on = [];
            $scope.hooks.updateRecipeStatus();
        };

        const removeDatasets = function (virtualIndexes, deletedDatasetVirtualIndex) {
            AutoFeatureGenerationRecipeService.removeDatasets(virtualIndexes, deletedDatasetVirtualIndex, $scope.params, $scope.recipe, $scope.columnsTabData);
        }

        $scope.removeDataset = function(virtualIndex) {
            let datasetsToBeRemoved = [];
            if ($scope.params.virtualInputs.length > 2) {
                datasetsToBeRemoved = AutoFeatureGenerationRecipeService.getDependantDatasets(virtualIndex, $scope.params.relationships);
            }
            datasetsToBeRemoved.push(virtualIndex);
            if (datasetsToBeRemoved.length === 1) {
                removeDatasets(datasetsToBeRemoved, virtualIndex);
                if ($scope.params.virtualInputs.length === 0) {
                    $scope.showNewDatasetModal(0);
                }
            } else {
                const datasetList = datasetsToBeRemoved.map(function(index) {
                    return $scope.getDatasetName(index);
                })
                Dialogs.confirm($scope,
                    'Remove datasets',
                    'The following datasets will be removed from the recipe:' +
                    '<ul><li>' + datasetList.join('</li><li>') + '</li></ul>'
                )
                    .then(function() {
                        removeDatasets(datasetsToBeRemoved, virtualIndex);
                        if ($scope.params.virtualInputs.length === 0) {
                            $scope.showNewDatasetModal(0);
                        }
                    });
            }
            // Update recipe status to retrieve any schema change
            $scope.hooks.updateRecipeStatus();
        };

        $scope.range = AutoFeatureGenerationRecipeService.range;

        $scope.getDatasetName = function(virtualIndex) {
            return AutoFeatureGenerationRecipeService.getDatasetName(virtualIndex, $scope.params.virtualInputs, $scope.recipe.inputs);
        }

        $scope.showRelationshipEditModal = function(relationship, tab) {
            //check if the modal is already shown
            const newScope = $scope.$new();
            newScope.relationship = relationship;
            newScope.current = {};
            newScope.current.tab = tab || 'conditions'
            newScope.current.condition = null; //no selected condition when the modal is created


            CreateModalFromTemplate("/templates/recipes/visual-recipes-fragments/relationship-edit-modal.html", newScope, "RelationshipEditController", (scope, el) => {
                $timeout(() => {
                        scope.AfgBlockBodyEl = el[0].getElementsByClassName('modal-body')[0];
                    }
                );
            });
        };

        $scope.getDatasetColorClass = AutoFeatureGenerationRecipeService.getDatasetColorClass;

        $scope.getLeftSymbol = AutoFeatureGenerationRecipeService.getLeftSymbolFromRelationship;

        $scope.getRightSymbol = AutoFeatureGenerationRecipeService.getRightSymbolFromRelationship;

        $scope.getRelationshipStyle = AutoFeatureGenerationRecipeService.getRelationshipClass;

        $scope.params = $scope.params || {};

        $scope.columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData($scope.recipe);

        $scope.toggleColumnsForComputation = function(datasetId) {
            $scope.columnsTabData[datasetId].isSectionOpen = !$scope.columnsTabData[datasetId].isSectionOpen;
        }

        $scope.updateSelectedColumns = function(datasetId, selectedColumns) {
            $scope.columnsTabData[datasetId].selectedColumns = angular.copy(selectedColumns);
            // Update recipe status to get new schema and any column limit warnings
            $scope.updateRecipeStatusLater(700);
        }

        $scope.updateSelectedFeatures = function() {
            // Update recipe status to get new schema and any column limit warnings
            $scope.updateRecipeStatusLater(700);
        }

        $scope.getAvailableReplacementDatasets = function(datasets) {
            const primaryDatasetName = AutoFeatureGenerationRecipeService.getDatasetName(0, $scope.params.virtualInputs, $scope.recipe.inputs);
            const primaryDataset = $scope.computablesMap[primaryDatasetName];
            return AutoFeatureGenerationRecipeService.checkDatasetsUsability(datasets, primaryDataset, $scope.outputDatasetName, $scope.recipe, ($scope.appConfig && $scope.appConfig.sparkEnabled) ? $scope.appConfig.sparkEnabled : false);
        }

        $scope.onInputReplaced = function (replacement, virtualIndex) {
            const recipeInputs = $scope.recipe.inputs;
            const oldInput = angular.copy($scope.params.virtualInputs[virtualIndex]);

            const recipeSerialized = angular.copy($scope.recipe);
            const payload = $scope.hooks.getPayloadData();
            let columnsForComputation = [];
            //Get default columns before replacing the input, otherwise the backend returns an empty list of columns
            DataikuAPI.flow.recipes.autofeaturegeneration.getDefaultColumns($stateParams.projectKey, replacement.name, recipeSerialized, payload)
                .success(function (defaultColumns) {
                    columnsForComputation = defaultColumns;
                })
                .error(setErrorInScope.bind($scope))
                .finally(function () {
                    AutoFeatureGenerationRecipeService.replaceVirtualInput($scope.params, virtualIndex, replacement.name, $scope.computablesMap, recipeInputs);
                    $scope.params.virtualInputs[virtualIndex].selectedColumns = columnsForComputation;
                    //Open all the sections of the columns for computation tab
                    $scope.columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData($scope.recipe);
                    DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                        .then(function () {
                            AutoFeatureGenerationRecipeService.resyncSchemas($scope.params, virtualIndex, $scope.computablesMap, recipeInputs, oldInput);
                            $scope.hooks.updateRecipeStatus();
                        })
                });
        }

        $scope.$watch("topNav.tab", function(nv, ov) {
            // Necessary to fix UI bugs on the fatRepeat directive that occur when changing recipe tabs.
            if (nv==="settings") {
                $scope.$broadcast("repaintFatTable");
            }
        });

    });

    /*
    Controller for relationship edit modal
    */
    app.controller("RelationshipEditController", function($scope) {
        $scope.uiState = $scope.uiState || {};
        if ($scope.relationship.on.length === 0) {
            $scope.addEmptyCondition($scope.relationship);
            $scope.current.condition = $scope.relationship.on[0];
        }

        $scope.getColumn = function(condition, columnIdx) {
            const col = !columnIdx || columnIdx === 1 ? 'column1' : 'column2';
            return $scope.getColumnFromName($scope.getDatasetName(condition[col].table), condition[col].name);
        };

        $scope.getColumnFromName = function(datasetName, name) {
            return $scope.getAfgColumns(datasetName).filter(function(col) {
                return col.name === name
            })[0];
        };

        $scope.ok = function(){
            $scope.dismiss();
            //  Update recipe status to validate the conditions of the relationship when a user clicks on the OK button
            $scope.hooks.updateRecipeStatus();
        }

        $scope.$on('$destroy', function() {
            //  Update recipe status to validate the conditions of the relationship when a user clicks outside of the modal
            $scope.hooks.updateRecipeStatus();
        });
    });

    app.directive('afgBlockEmpty', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-block-empty.html',
        };
    });

    app.directive('afgBlockDropdownRelationship', function(AutoFeatureGenerationRecipeService) {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-block-dropdown-relationship.html',
            link: function(scope, element, attrs) {
                scope.relationshipTypes = AutoFeatureGenerationRecipeService.SUPPORTED_RELATIONSHIP_TYPES;
                scope.relationshipIndex = attrs.relationshipIndex;
                scope.setRelationshipType = function(relationship, type) {
                    relationship.type = type;
                    // Update recipe status to get any new schema changes and validate the new relationship type
                    scope.hooks.updateRecipeStatus();
                };

                scope.getRelationshipTypeLabel = function(relationship) {
                    const relationshipDescription = AutoFeatureGenerationRecipeService.getRelationshipDescription(relationship);
                    return relationshipDescription.type;
                }

                scope.getRelationshipTypeDescription = function(relationship) {
                    const relationshipDescription = AutoFeatureGenerationRecipeService.getRelationshipDescription(relationship);
                    return relationshipDescription.description;
                }

                scope.getClass = function(type) {
                    return `{selected: relationship.type === '${type}'}`;
                };
            }
        }
    });

    app.directive('afgConditionsEditor', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-conditions-editor.html'
        };
    });

    app.directive('afgCondition', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-condition.html',
            link: function(scope, element, attrs) {
                scope.relationshipIndex = attrs.relationshipIndex;
                scope.displayRightActions = attrs.displayRightActions;
                scope.actOnConditionClicked = attrs.actOnConditionClicked;
                scope.toggleConditionFocus = function(condition) {
                    if (scope.current.condition === condition) {
                        scope.current.condition = null;
                    } else {
                        scope.current.condition = condition;
                    }
                };
            }
        };
    });

    app.directive('afgBlockWithRelationship', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/afg-block-with-relationship.html',
            link: function(scope, element, attrs) {
                scope.relationshipIndex = attrs.relationshipIndex;
            }
        };
    });
})();