(function(){
    'use strict';
    
    const app = angular.module('dataiku.shaker');

    app.component('imageView', {
        bindings: {
            requestedSampleId: '<',
            inputDatasetProjectKey: "<",
            inputDatasetName: "<",
            imageViewSettings: '<',
            nbColumnsInDataset: "<",
            shakerHooks: "=",
            filterRequestBuilderFn: "<"
        },
        controller: function(){
            const $ctrl = this;
            
            function setScriptAndFilters() {
                $ctrl.resolvedScript = $ctrl.shakerHooks.shakerForQuery();
                $ctrl.filters = {'elements': $ctrl.filterRequestBuilderFn($ctrl.resolvedScript.explorationFilters)};
            }

            this.$onInit = () => {
                setScriptAndFilters();
                // We make sure to update script & table when table has been refreshed on the table view, 
                // for instance if the search was changed
                $ctrl.shakerHooks.onTableRefresh = function() {
                    setScriptAndFilters();
                }
            }
        },
        templateUrl: "/templates/shaker/image-view-component.html"
    });

    app.component('imageViewSettings', {
        bindings: {
            settings: "=",
            columns: "<"
        },
        controller: function($scope, DataikuAPI, FutureProgressModal, $stateParams) {
            const $ctrl = this;

            $ctrl.autoDetectCategoriesForImageView = function() {
                DataikuAPI.shakers.autoDetectCategoriesForImageView($stateParams.projectKey, $stateParams.datasetName, $ctrl.settings).success(function(data) {
                    const newScope = $scope.$new();
                    FutureProgressModal.show(newScope, data, "Auto detecting categories for image view").then(function(newCategories) {
                        if (newCategories) { // will be undefined if computation was aborted
                            $ctrl.settings.annotationParams.annotationCategories = newCategories;
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $ctrl.autoDetectButtonDisabledMessage = function() {
                if (!$ctrl.settings.annotationParams || !$ctrl.settings.annotationParams.enabled) { // Should not happen
                    return "Annotation params must be enabled";
                }
                if (!$ctrl.settings.annotationParams.annotationColumn) {
                    return "Annotation column must be defined";
                }
                if (!$ctrl.settings.annotationParams.annotationType) {
                    return "Annotation type must be defined";
                }
                return null;
            }
        },
        templateUrl: "/templates/shaker/image-view-settings-component.html"
    });

    app.component('filterAdder', {
        bindings: {
            explorationFilters: "<",
            tableHeaders: "<",
            addColumnFilterCallback: "<"
        },
        templateUrl: "/templates/shaker/filter-adder-component.html",
        controller: function() {
            const $ctrl = this;

            // This component "hacks" the `dku-bs-select` directive by just making it a selector and not really an input. As soon as a
            // column is selected, we put back the placeholder value to be able to filter other columns.
            $ctrl.addColumnFilter = function() {
                const column = $ctrl.tableHeaders.find(h => h.name === $ctrl.tmpFilterToAdd);
                $ctrl.addColumnFilterCallback(column.name, {}, 'full_string', column.selectedType.name, column.isDouble);
                $ctrl.tmpFilterToAdd = null; // Resetting the tmp filter to have the widget displaying placeholder again
            };

            $ctrl.allColumnsFiltered = function() {
                return $ctrl.filteredColumnNames && $ctrl.allColumnNames && $ctrl.filteredColumnNames.length === $ctrl.allColumnNames.length;
            };

            $ctrl.removeFilteredColumns = function (columnName) {
                return $ctrl.filteredColumnNames.indexOf(columnName) < 0;
            };

            function updateColumnNames(filteredColumnNames) {
                $ctrl.filteredColumnNames = filteredColumnNames;
                $ctrl.allColumnNames = $ctrl.tableHeaders.map(h => h.name);
            }
            
            let currentFilteredColumnNames = [];
            this.$onInit = () => {
                updateColumnNames(currentFilteredColumnNames);
                $ctrl.tmpFilterToAdd = null;
            }

            this.$doCheck = () => {
                if (!angular.equals(currentFilteredColumnNames, $ctrl.explorationFilters.map(f => f.column))) {
                    currentFilteredColumnNames = $ctrl.explorationFilters.map(f => f.column);
                    updateColumnNames(currentFilteredColumnNames);
                }
            }
        }
    });

    app.component('imagePreviewComponent', {
        bindings: {
            "imagePath": "<",
            "imageViewSettings": "<",
            "popupContainer": "<"
        },
        template: `
            <ng2-dataset-view-image-preview [image-path]="$ctrl.imagePath" [image-view-settings]="$ctrl.imageViewSettings"></ng2-dataset-view-image-preview>`,

        controller: function(DataikuAPI, $scope, $stateParams) {
            const $ctrl = this;

            // bindings are not properly set now so we CAN'T just do: $ctrl.$postLink = $ctrl.popupContainer.onPopupCreated
            $ctrl.$postLink = () => $ctrl.popupContainer.onPopupCreated();
        }
    })
})();