(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesZoomHelper', LinesZoomHelper);

    const CLIP_PATH_ID = 'lines-chart-clip-path';

    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesZoomHelper(ChartDataUtils, D3ChartAxes, ChartAxesUtils, LinesUtils, ReferenceLines, ChartYAxisPosition, ChartUsableColumns, ChartCustomMeasures, ChartZoomLoader,
        ChartSetErrorInScope, Logger, ChartDimension, ChartActivityIndicator, ChartsStaticData) {

        const svc = {

            /*
             * The greyed out areas that represent missing data in the current aggregation level.
             * They appear when zooming out and panning.
             */
            createMissingDataArea(g) {
                const clipPathUniqueId = CLIP_PATH_ID + generateUniqueId();
                const area = g.append('rect')
                    .attr('opacity', '0.6')
                    .attr('class', 'missing-data-area')
                    .attr('x', '0')
                    .attr('width', '0')
                    .attr('y', '0')
                    .attr('height', '0')
                    .attr('clip-path', 'url(#' + clipPathUniqueId + ')')
                    .style('-webkit-clip-path', 'url(#' + clipPathUniqueId + ')')
                    .style('pointer-events', 'none');

                return area;
            },

            /*
             * Updates the greyed out areas upon zooming out and panning.
             */
            updateMissingDataAreas(chartBase, zoomUtils) {
                if (zoomUtils.displayIntervalBeforeInteraction && zoomUtils.displayIntervalBeforeInteraction !== zoomUtils.dataInterval) {
                    const dataIntervalMin = zoomUtils.dataInterval[0];
                    const dataIntervalMax = zoomUtils.dataInterval[1];
                    const displayIntervalBeforeInteractionMin = zoomUtils.displayIntervalBeforeInteraction[0];
                    const displayIntervalBeforeInteractionMax = zoomUtils.displayIntervalBeforeInteraction[1];

                    if (displayIntervalBeforeInteractionMin > dataIntervalMin) {
                        const scaledDataIntervalMin = chartBase.xAxis.scale()(dataIntervalMin);
                        const scaledIntervalBeforeInteractionMin = chartBase.xAxis.scale()(displayIntervalBeforeInteractionMin);

                        zoomUtils.leftMissingDataAreas.forEach(area => {
                            area.attr('x', scaledDataIntervalMin)
                                .attr('width', scaledIntervalBeforeInteractionMin - scaledDataIntervalMin)
                                .attr('height', chartBase.vizHeight);
                        });
                    }

                    if (displayIntervalBeforeInteractionMax < dataIntervalMax) {
                        const scaledDataIntervalMax = chartBase.xAxis.scale()(dataIntervalMax);
                        const scaledIntervalBeforeInteractionMax = chartBase.xAxis.scale()(displayIntervalBeforeInteractionMax);

                        zoomUtils.rightMissingDataAreas.forEach(area => {
                            area.attr('x', scaledIntervalBeforeInteractionMax)
                                .attr('width', scaledDataIntervalMax - scaledIntervalBeforeInteractionMax)
                                .attr('height', chartBase.vizHeight);
                        });
                    }
                }
            },

            /*
             * Resets the greyed out areas upon new pivot request.
             */
            cleanMissingDataAreas(zoomUtils) {
                zoomUtils.leftMissingDataAreas.forEach(area => {
                    area.attr('x', '0')
                        .attr('y', '0')
                        .attr('width', '0')
                        .attr('height', '0');
                });

                zoomUtils.rightMissingDataAreas.forEach(area => {
                    area.attr('x', '0')
                        .attr('y', '0')
                        .attr('width', '0')
                        .attr('height', '0');
                });
            },

            cleanZoomListeners(zoom) {
                zoom && zoom.on('zoomstart', null);
                zoom && zoom.on('zoom', null);
                zoom && zoom.on('zoomend', null);
                zoom = null;
            },

            updateTicksFormat(chartDef, container, xAxis) {
                const xExtent = D3ChartAxes.getCurrentAxisExtent(xAxis);
                const xMin = xExtent[0];
                const xMax = xExtent[1];

                if (!isFinite(xMin) || !isFinite(xMax)) {
                    return;
                }

                const computedDateDisplayUnit = ChartDataUtils.computeDateDisplayUnit(xMin, xMax);

                xAxis.tickFormat(date => {
                    return computedDateDisplayUnit.formatDateFn(date, computedDateDisplayUnit.dateFormat);
                });
            },

            /**
             * Wrapper for ChartDataUtils.getMeasureExtents().
             *
             * @param {ChartDef.java}   chartDef    - The chart definition.
             * @param {Object}          chartBase   - Everything that the chart might need.
             * @param {Array}           interval    - The x min and max values to use as filter when computing the extents.
             */
            getYExtentsForInterval(chartBase, chartDef, chartHandler, interval) {
                const xMin = interval[0];
                const xMax = interval[1];
                const chartData = chartBase.chartData;
                const results = {};
                const includeEmptyBins = LinesUtils.hasEmptyBinsToIncludeAsZero(ChartDimension.getGenericDimension(chartDef), chartData);
                const yExtents = ChartDataUtils.getMeasureExtents(chartDef, chartData, 'x', [xMin, xMax], includeEmptyBins);
                results.recordsCount = yExtents.recordsCount;
                results.pointsCount = yExtents.pointsCount;
                const leftAxis = chartBase.yAxes.filter(axis => axis.position === ChartYAxisPosition.LEFT),
                    rightAxis = chartBase.yAxes.filter(axis => axis.position === ChartYAxisPosition.RIGHT),
                    leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT),
                    rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT);

                const dataSpec = chartHandler.getDataSpec();
                const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
                const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, dataSpec.context).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartBase.xAxis, undefined),
                    referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                    referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: yExtents[leftYAxisID].onlyPercent }, [rightYAxisID]: { isPercentScale: yExtents[rightYAxisID].onlyPercent } });

                if (leftAxis && leftAxis.length) {
                    results[leftYAxisID] = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]);
                }

                if (rightAxis && rightAxis.length) {
                    results[rightYAxisID] = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);
                }

                return results;
            },

            /**
             * Check if extents of left and right axis are valid ie have:
             *  * Finite numbers
             *  * Different mix and max values
             *
             * @param {Array} leftYExtent        - Min and max interval for left Y axis.
             * @param {Array} rightYExtent       - Min and max interval for right Y axis.
             *
             * @returns {Boolean} True if both y extents are valid.
             */
            hasValidYExtents(leftYExtent, rightYExtent) {
                const isLeftYExtentFinite = leftYExtent && isFinite(leftYExtent[0]) && isFinite(leftYExtent[1]);
                const isRightYExtentFinite = rightYExtent && isFinite(rightYExtent[0]) && isFinite(rightYExtent[1]);
                const isLeftYExtentValid = !leftYExtent || (isLeftYExtentFinite && leftYExtent[0] !== leftYExtent[1]);
                const isRightY2ExtentValid = !rightYExtent || (isRightYExtentFinite && rightYExtent[0] !== rightYExtent[1]);

                return isLeftYExtentValid && isRightY2ExtentValid;
            },

            // Simply updates any useful zoom info for incoming zoom actions.
            updateZoomUtils(zoomUtils, inspectedZoom) {

                zoomUtils.displayInterval = inspectedZoom.displayInterval;

                const isDisplayIntervalValid = inspectedZoom.yExtents.pointsCount > 1
                    && svc.hasValidYExtents(inspectedZoom.yExtents[ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT)], inspectedZoom.yExtents[ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT)])
                    && inspectedZoom.displayInterval[0] !== inspectedZoom.displayInterval[1];

                if (isDisplayIntervalValid) {
                    zoomUtils.lastValidDisplayInterval = inspectedZoom.displayInterval;
                }

                zoomUtils = { ...zoomUtils, ...inspectedZoom.yExtents };
                zoomUtils.disableZoomFiltering = inspectedZoom.disableZoomFiltering;
                zoomUtils.previousZoomEvent = d3.event;

                return zoomUtils;
            },

            redrawChart(chartBase, chartDef, zoomUtils, containerSelection, xAxis, drawFrame) {
                const leftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
                const rightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
                svc.updateAxes(containerSelection, chartDef, xAxis, leftYAxis, rightYAxis, zoomUtils, chartBase.vizHeight, chartBase.vizWidth);

                drawFrame(zoomUtils.frameIndex, chartBase, true);
                svc.updateMissingDataAreas(chartBase, zoomUtils);
            },

            buildChartInteractionErrorMessage(data, status, headers) {
                const knownError = ChartSetErrorInScope.buildValidityForKnownError(data, status, headers);
                if (knownError !== undefined) {
                    return knownError.message;
                } else if (data.message) {
                    return data.message;
                }
                return 'An unknown error occurred while interacting with the chart.';
            },

            /**
             * Asks for new data inside the given display interval and create a new one accordingly.
             *
             * @param {Function}                pivotRequestCallback    - The function that executes the pivot request
             * @param {Function}                linesChartCallback      - The function that recomputes and redraws the line chart
             * @param {ChartDef.java}           chartDef                - The chart definition.
             * @param {Object}                  chartBase               - Everything that the chart might need.
             * @param {LinesZoomUtils}          zoomUtils               - All the data related to zoom
             * @param {Function}                cleanFrame              - The callback that will remove the chart from DOM.
             * @param {Object}                  uiDisplayState          - Everything the UI might need.
             * @param {Object}                  chartActivityIndicator  - Activity indicator displayed in chart
             * @param {Object}                  request                 - Request to be executed
             * @param {Object}                  chartContainer
             */
            computePivotRequest(pivotRequestCallback, linesChartCallback, chartDef, chartBase, zoomUtils, chartHandler, cleanFrame, uiDisplayState, chartActivityIndicator, request, chartContainer) {

                pivotRequestCallback(request, false, true).success(function(data) {
                    if (data.result.pivotResponse.axisLabels[0] && data.result.pivotResponse.axisLabels[0].length === 1) {
                        Logger.info('Not enough data in the result: chart won\'t be refreshed.');
                        svc.cleanOfflineFeedbacks(zoomUtils);
                        return;
                    }

                    const responseSequenceId = data.result.pivotResponse.sequenceId;

                    if (responseSequenceId === zoomUtils.sequenceId) {
                        Logger.info('Sequence ids match (' + responseSequenceId + '). Deactivate offline zoom and refresh the chart.');
                        svc.cleanAll(chartBase, cleanFrame, zoomUtils);
                        zoomUtils.preventThumbnailUpdate = true;
                        zoomUtils.offlineZoomDisabled = true;
                        linesChartCallback(chartContainer, chartDef, chartHandler, zoomUtils.axesDef, data.result.pivotResponse, pivotRequestCallback, zoomUtils.displayInterval, uiDisplayState, chartActivityIndicator);
                        uiDisplayState.chartTopRightLabel = ChartDataUtils.computeRecordsStatusLabel(
                            data.result.pivotResponse.beforeFilterRecords,
                            data.result.pivotResponse.afterFilterRecords,
                            ChartDimension.getComputedMainAutomaticBinningModeLabel(data.result.pivotResponse, chartDef),
                            chartDef.type
                        );
                        uiDisplayState.samplingSummaryMessage = ChartDataUtils.getSamplingSummaryMessage(data.result.pivotResponse, chartDef.type);
                        uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(
                            chartDef.type,
                            data.result.pivotResponse.afterFilterRecords,
                            [],
                            undefined
                        );
                    } else {
                        Logger.info('Sequence ids do not match (' + responseSequenceId + ', ' + zoomUtils.sequenceId + '): chart won\'t be refreshed.');
                    }
                }).error(function(data, status, headers) {
                    Logger.info('An error occurred during zoom pivot request');
                    ChartActivityIndicator.displayBackendError(
                        chartActivityIndicator,
                        svc.buildChartInteractionErrorMessage(data, status, headers)
                    );
                    uiDisplayState.chartTopRightLabel = ChartDataUtils.computeNoRecordsTopRightLabel();
                    uiDisplayState.chartRecordsFinalCountTooltip = ChartDataUtils.getRecordsFinalCountTooltip(chartDef.type, 0);
                    uiDisplayState.samplingSummaryMessage = data.result && ChartDataUtils.getSamplingSummaryMessage(data.result.pivotResponse, chartDef.type);
                    svc.cleanOfflineFeedbacks(zoomUtils);
                });
            },

            // Get an updated y axis domain upon zooming
            getUpdatedYDomain(chartDef, yAxis, yExtent) {
                if (!yExtent) {
                    return;
                }
                let yMin = yExtent[0];
                let yMax = yExtent[1];

                const { customExtent, includeZero } = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, yAxis.id);
                const shouldIncludeZero = customExtent.editMode === ChartsStaticData.AUTO_EXTENT_MODE && includeZero;
                if (shouldIncludeZero) {
                    if (yMin > 0) {
                        yMin = 0;
                    } else if (yMax < 0) {
                        yMax = 0;
                    }
                } else if (customExtent.editMode === ChartsStaticData.MANUAL_EXTENT_MODE) {
                // Temp fix to enable custom extent => then custom extent should only be applied when no zoom is on (only at initial state)
                    const customAxisExtent = ChartAxesUtils.getManualExtent(customExtent);
                    yMin = customAxisExtent[0];
                    yMax = customAxisExtent[1];
                }

                return [yMin, yMax];
            },

            /**
             * Clean offline feedbacks, the chart, and every listeners previously attached for interactivity.
             *
             * @param {Object}      chartBase       - Everything that the chart might need.
             * @param {Function}    cleanFrame      - Function that removes the chart.
             */
            cleanAll(chartBase, cleanFrame, zoomUtils) {
                svc.cleanOfflineFeedbacks(zoomUtils);
                svc.cleanZoomListeners(zoomUtils.zoom);
                cleanFrame(chartBase);
            },

            /**
             * Clean every visual feedbacks the user may have when interacting:
             *  - Grey areas for missing data
             *  - Grey text color for top-right aggregations info.
             *
             * @param {Object}  zoomUtils       - Data related to zoom
             */
            cleanOfflineFeedbacks(zoomUtils) {
                ChartZoomLoader.displayLoader(false, zoomUtils.instanceId);
                svc.cleanMissingDataAreas(zoomUtils);
            },

            computeBrushDimensions(chartBase) {
                return {
                    paddingLeft: chartBase.margins.left,
                    width: chartBase.vizWidth
                };
            },

            onBrushInit(chartBase, chartDef, drawBrush, zoomUtils) {
                return function(brushContentG, brushContentHeight, brushContentWidth) {

                    const xAxisLogScale = (chartBase.xSpec && chartBase.xSpec.type == 'MEASURE' && chartDef.xAxisFormatting.isLogScale);

                    const xAxis = D3ChartAxes.createAxis(chartBase.chartData, chartBase.xSpec, chartBase.isPercentChart, xAxisLogScale, undefined, chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting, chartDef);
                    const yAxes = [];
                    for (const key in chartBase.ySpecs) {
                        const yAxisLogScale = ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, key);
                        const yAxisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, key);
                        const axis = D3ChartAxes.createAxis(chartBase.chartData, chartBase.ySpecs[key], chartBase.isPercentChart, yAxisLogScale, yAxisFormatting.includeZero, yAxisFormatting.axisValuesFormatting.numberFormatting, chartDef);
                        axis && yAxes.push(axis);
                    };

                    xAxis.setScaleRange([0, brushContentWidth]);

                    const leftAxis = ChartAxesUtils.getLeftYAxis(yAxes);
                    const rightAxis = ChartAxesUtils.getRightYAxis(yAxes);

                    if (leftAxis) {
                        if (chartBase.ySpecs[ChartsStaticData.LEFT_AXIS_ID].ascendingDown) {
                            leftAxis.setScaleRange([0, brushContentHeight]);
                        } else {
                            leftAxis.setScaleRange([brushContentHeight, 0]);
                        }
                    }

                    if (rightAxis) {
                        rightAxis.setScaleRange([brushContentHeight, 0]);
                    }

                    const brushAxes = {
                        x: xAxis,
                        [ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT)]: leftAxis,
                        [ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT)]: rightAxis
                    };

                    drawBrush(chartBase, brushContentG, brushAxes);
                    zoomUtils.hasBrushBeenDrawn = true;
                };
            },

            updateAxes(containerSelection, chartDef, xAxis, leftYAxis, rightYAxis, zoomUtils, vizHeight, vizWidth) {
                svc.updateXAxis(containerSelection, chartDef, xAxis, vizHeight, [leftYAxis, rightYAxis]);
                svc.updateYAxes(containerSelection, chartDef, leftYAxis, rightYAxis, zoomUtils, vizWidth);
            },

            updateXAxis(containerSelection, chartDef, xAxis, vizHeight, yAxes) {
                if (!chartDef.xAxisFormatting.displayAxis) {
                    return;
                }

                const allNegative = !yAxes.filter(yAxis => yAxis).some(yAxis => {
                    const currentYExtent = D3ChartAxes.getCurrentAxisExtent(yAxis);
                    return currentYExtent && currentYExtent[1] >= 0;
                });
                svc.updateTicksFormat(chartDef, containerSelection.node(), xAxis);
                const xAxisSelection = [ ...containerSelection.selectAll('.chart-svg .x.axis')][0];
                xAxisSelection.forEach(xG => {
                    xG = d3.select(xG);
                    xG.call(xAxis);
                    xG.selectAll('.tick text')
                        .attr('transform', function() {
                            const labelAngle = 0.5;
                            const translateValue = '-33, 15';
                            const rotateValue = labelAngle * -180 / Math.PI;

                            return `translate(${translateValue}), rotate(${rotateValue}, 0, 0)`;
                        });
                    const gridlinesPosition = { x1: 0, x2: 0, y1: 0, y2: allNegative ? vizHeight : -vizHeight };
                    if (chartDef.gridlinesOptions.vertical.show) {
                        D3ChartAxes.drawGridlines(xG, 'x', chartDef.gridlinesOptions.vertical.lineFormatting, gridlinesPosition, containerSelection);
                    }
                });
            },

            updateYAxes(containerSelection, chartDef, leftAxis, rightAxis, zoomUtils, vizWidth) {
                if (leftAxis) {
                    svc.updateYAxis(containerSelection, chartDef, leftAxis, zoomUtils, '.y1.axis', vizWidth);
                }
                if (rightAxis) {
                    svc.updateYAxis(containerSelection, chartDef, rightAxis, zoomUtils, '.y2.axis', vizWidth);
                }
            },

            updateYAxis(containerSelection, chartDef, yAxis, zoomUtils, selector, vizWidth) {
                const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, yAxis);
                if (axisFormatting && !axisFormatting.displayAxis) {
                    return;
                }

                const [yMin, yMax] = svc.getUpdatedYDomain(chartDef, yAxis, zoomUtils[yAxis.id]);
                yAxis.scale().domain([yMin, yMax]);

                [...containerSelection.selectAll(selector)[0]].forEach(yG => {
                    d3.select(yG).call(yAxis);
                    if (chartDef.gridlinesOptions.horizontal.show && D3ChartAxes.shouldDrawGridlinesForAxis(chartDef.type, yAxis, chartDef.gridlinesOptions.horizontal.displayAxis)) {
                        const gridlinesPosition = { x1: 0, x2: ChartAxesUtils.isRightYAxis(yAxis) ? -vizWidth : vizWidth, y1: 0, y2: 0 };
                        D3ChartAxes.drawGridlines(d3.select(yG), 'y', chartDef.gridlinesOptions.horizontal.lineFormatting, gridlinesPosition, containerSelection);
                    }
                });
            }
        };
        return svc;
    }

})();
