(function() {
    'use strict';

    angular.module('dataiku.shared').component('pivotFilterList', {
        bindings: {
            // inputs
            datasetLoadingError: '<',       // boolean
            filters: '<',                   // ChartFilter[]
            filterFacets: '<',              // FilterFacet[]
            filterFacetsLoadingError: '<',  // ErrorData
            nameOnly: '<',                  // boolean
            getFiltersRequestOptions: '<',  // () => FiltersRequestOptions
            readOnly: '<',                  // boolean,
            samplingParams: '<',            // { engineType: string, refreshableSelection: any }
            listClasses: '<',               // string
            isInChartContext: '<',          // boolean
            reorderingContainer: '<',       // string
            direction: '<',                 // FiltersPanelDirection
            theme: '<',                     // DSSVisualizationTheme
            // outputs
            canAllBeClearedChange: '&',     // ($canAllBeCleared: boolean) => void
            filtersInit: '&',               // ($api: { clearAllFilters: () => void }) => void
            filtersChange: '&',             // ({ $currentValue: ChartFilter[], $previousValue: ChartFilter[] }) => void
            reorderingStart: '&?',          // () => void
            reorderingStop: '&?'            // () => void
        },
        templateUrl: '/static/dataiku/shared/components/pivot-filter-list/pivot-filter-list.component.html',
        controller: function($timeout, Debounce, FiltersPanelDirection, translate) {
            const BASE_FILTER_SORT_OPTIONS = {
                cursor: 'move',
                handle: '.box-header',
                items: '> li',
                update() {
                    $timeout(() =>
                         ctrl.filtersChange({ $currentValue: ctrl.filters, $previousValue: previousFilters })
                    );
                },
                start() {
                    $timeout(() => {
                        ctrl.reordering = true;
                    });
    
                    if (ctrl.reorderingStart) {
                        ctrl.reorderingStart();
                    }
                },
                stop() {
                    $timeout(() => {
                        ctrl.reordering = false;
                    });
    
                    if (ctrl.reorderingStop) {
                        ctrl.reorderingStop();
                    }
                }
            };

            const ctrl = this;

            ctrl.FiltersPanelDirection = FiltersPanelDirection;
            ctrl.translate = translate;

            /** @type {Map<string, boolean>} a map mapping whether a filter can be cleared by its id */
            const canFiltersBeClearedById = new Map()
            /** @type {boolean}  is true if at least a filter can be cleared, false otherwise */
            let canAllFiltersBeCleared;
            // Because we debounce the emission of the new filters to the parent, we need a place to store the intermediary filters state.
            let tmpFilters = null;
            /** @type {Map<string, Record<string, (...args: any[]) => void>>} a map mapping filter apis by their ids. */
            const filterApisById = new Map()
            let previousFilters;

            ctrl.filterSortOptions = null;
            ctrl.reordering = false;

            ctrl.$onInit = function() {
                ctrl.filtersInit({ $api: { clearAllFilters } });
            };

            ctrl.$onChanges = function(changes) {
                if (!ctrl.direction) {
                    ctrl.direction = FiltersPanelDirection.Vertical;
                }

                if (changes.filters && changes.filters.currentValue) {
                    const currentFilterIds = new Set(changes.filters.currentValue.map(({ id }) => id));
                    let shouldRecomputeCanAllBeCleared = false;
                    for (const previousFilterId of canFiltersBeClearedById.keys()) {
                        // previous id not in current ids means filter has been removed
                        if (!currentFilterIds.has(previousFilterId)) {
                            shouldRecomputeCanAllBeCleared = true;
                            canFiltersBeClearedById.delete(previousFilterId);
                            filterApisById.delete(previousFilterId);
                        }
                    }
                    if (shouldRecomputeCanAllBeCleared) {
                        computeCanAllBeClearedState();
                    }

                    previousFilters = angular.copy(ctrl.filters);
                }
                if (changes.readOnly || changes.isInChartContext || changes.filters || changes.reorderingContainer || changes.direction) {
                    const disabled = ctrl.readOnly || ctrl.filters == null || ctrl.filters.length <= 1;
                    const containment = ctrl.reorderingContainer || undefined;
                    const axis = ctrl.direction === FiltersPanelDirection.Horizontal ? 'x' : 'y';
                    ctrl.filterSortOptions = {
                        ...BASE_FILTER_SORT_OPTIONS,
                        containment,
                        disabled,
                        axis
                    };
                }
            };

            /**
             * Handles a filter change by computing the new filters list and emitting the change to the parent.
             * @param {FrontendChartFilter | null} filter   - if the filter is null it means it has been removed
             * @param {bool} canPropagageBeforeFacetUpdate  - whether the filter can be propagated before the facet update or not
             * @param {string} filterId                     - the filter id
             */
            ctrl.handleFilterChange = function(filter, canPropagageBeforeFacetUpdate, filterId, needAggregationRequest = true) {
                const oldFilters = tmpFilters || ctrl.filters;
                const filterIndex = oldFilters.findIndex(({ id }) => id === filterId);
                const newFilters = [
                    ...oldFilters.slice(0, filterIndex),
                    ...(filter ? [filter] : []),
                    ...oldFilters.slice(filterIndex + 1)
                ];
                tmpFilters = newFilters;
                emitFiltersChange(newFilters, ctrl.filters, canPropagageBeforeFacetUpdate, needAggregationRequest);
            };

            ctrl.handleCanBeClearedChange = function(filterId, canBeCleared) {
                canFiltersBeClearedById.set(filterId, canBeCleared);
                if (canFiltersBeClearedById.size !== ctrl.filters.length) {
                    return;
                }
                computeCanAllBeClearedState();
            };

            ctrl.handleFilterInit = function(filterId, filterApi) {
                filterApisById.set(filterId, filterApi);
            };

            // Actions like clearAllFilters can trigger a change on multiple filters at the same time.
            // To handle these changes as one big change, we debounce the emission of the new filters to the parent.
            const emitFiltersChange = Debounce().withDelay(100, 100).wrap((currentValue, previousValue, canPropagageBeforeFacetUpdate, needAggregationRequest) => {
                tmpFilters = null;
                ctrl.filtersChange({ $currentValue: currentValue, $previousValue: previousValue, $canPropagageBeforeFacetUpdate: canPropagageBeforeFacetUpdate, needAggregationRequest })
            });

            function clearAllFilters() {
                for (const filterApi of filterApisById.values()) {
                    filterApi.clearFilter();
                }
            }

            function computeCanAllBeClearedState () {
                const newCanAllBeCleared = Array.from(canFiltersBeClearedById.values()).some(canBeCleared => canBeCleared === true);
                if (newCanAllBeCleared !== canAllFiltersBeCleared) {
                    canAllFiltersBeCleared = newCanAllBeCleared;
                    ctrl.canAllBeClearedChange({ $canAllBeCleared: newCanAllBeCleared });
                }
            }
        }
    });
})();
