import dataiku


class ProjectStandardsCheckRunStatus:
    RUN_SUCCESS = "RUN_SUCCESS"
    RUN_ERROR = "RUN_ERROR"
    NOT_APPLICABLE = "NOT_APPLICABLE"


class ProjectStandardsCheckRunResult(object):

    def __init__(self, status, severity, message, details=None):
        """
        :param status: Status of the run.
        :type status: :class:`dataiku.project_standards.project_standards_check.ProjectStandardsCheckRunStatus`
        :param severity: Severity of a potential issue, between 0 and 5. 0 means no issue, 5 means critical issue. Ignored if the run is not a success.
        :type severity: int, optional
        :param message: A message to include in the report
        :type message: str
        :param details: Additional metadata about the run
        :type details: dict, optional
        """
        self.status = status
        self.severity = severity
        self.message = message
        self.details = details if details is not None else {}

    def get_raw(self):
        return {"status": self.status, "severity": self.severity, "message": self.message, "details": self.details}

    @staticmethod
    def success(message="", details=None):
        """
        Create a "success" run result. The check ran without issue.

        :param message: A message to include in the report
        :type message: str, optional
        :param details: Additional metadata about the run
        :type details: dict, optional
        :rtype: dict
        """
        return ProjectStandardsCheckRunResult(ProjectStandardsCheckRunStatus.RUN_SUCCESS, 0, message, details).get_raw()

    @staticmethod
    def failure(severity, message="", details=None):
        """
        Create a "failure" run result. The check found some issues.

        :param severity: Severity of the issues, between 0 and 5. 0 means no issue, 5 means critical issue.
        :type severity: int
        :param message: A message to include in the report
        :type message: str, optional
        :param details: Additional metadata about the run
        :type details: dict, optional
        :rtype: dict
        """
        return ProjectStandardsCheckRunResult(
            ProjectStandardsCheckRunStatus.RUN_SUCCESS, severity, message, details
        ).get_raw()

    @staticmethod
    def error(message="", details=None):
        """
        Create an "error" run result. The check could not run until the end because of an error.

        :param message: A message to include in the report
        :type message: str, optional
        :param details: Additional metadata about the run
        :type details: dict, optional
        :rtype: dict
        """
        return ProjectStandardsCheckRunResult(
            ProjectStandardsCheckRunStatus.RUN_ERROR, None, message, details
        ).get_raw()

    @staticmethod
    def not_applicable(message="", details=None):
        """
        Create a "not applicable" run result. The check was not executed.

        :param message: A message to include in the report
        :type message: str, optional
        :param details: Additional metadata about the run
        :type details: dict, optional
        :rtype: dict
        """
        return ProjectStandardsCheckRunResult(
            ProjectStandardsCheckRunStatus.NOT_APPLICABLE, None, message, details
        ).get_raw()


class ProjectStandardsCheckSpec(object):
    """The base interface for a Python Project Standards Check"""

    def __init__(self, project_key, original_project_key, config, plugin_config):
        """
        :param project_key: the project key
        :type project_key: str
        :param original_project_key: **deprecated**, use ``project_key``
        :type original_project_key: str
        :param config: the dict of the configuration of the object
        :type config: dict
        :param plugin_config: contains the plugin settings
        :type plugin_config: dict
        """
        self.config = config
        self.plugin_config = plugin_config
        self.project = dataiku.api_client().get_project(project_key)
        self.original_project_key = original_project_key

    def run(self):  # NOSONAR - not static, will be overridden
        """
        Run the check

        :returns: the run result.
            Use `ProjectStandardsCheckRunResult.success(message)` or `ProjectStandardsCheckRunResult.failure(severity, message)` depending on the result.
            Use `ProjectStandardsCheckRunResult.not_applicable(message)` if the check is not applicable to the project.
            Use `ProjectStandardsCheckRunResult.error(message)` if you want to mark the check as an error. You can also raise an Exception.
        """
        raise Exception("unimplemented")
