#!/bin/bash

# Check if output directory argument is provided
if [[ -z "$1" ]]; then
  echo "Usage: $0 <output_directory>"
  exit 1
fi

# Set the output directory from the first argument
OUTPUT_DIR="$1"

# Configuration 
DB_NAME="dataiku"
DB_USER="dataiku"
PGPASSWORD="dataiku"
DB_HOST="localhost"
DB_PORT="5432"

# Tables to exclude (add tables here to exclude them)
EXCLUDED_TABLES=("fmusersession" "tenant")

export PGPASSWORD

# Create output directory if it doesn't exist
mkdir -p "$OUTPUT_DIR"

# Convert EXCLUDED_TABLES array to a format suitable for SQL query
EXCLUDE_TABLES_SQL=$(printf ",'%s'" "${EXCLUDED_TABLES[@]}")
EXCLUDE_TABLES_SQL="${EXCLUDE_TABLES_SQL:1}"  # Remove leading comma

# List all tables in the database
tables=$(psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT table_name FROM information_schema.tables WHERE table_schema='public' AND table_name NOT IN ($EXCLUDE_TABLES_SQL);")

# Check if tables were found
if [[ -z "$tables" ]]; then
  echo "No tables found in database $DB_NAME."
  exit 1
fi

# Dump each table to a separate SQL file
for table in $tables; do
  echo "Dumping table: $table"
  pg_dump -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -t "$table" -F plain -f "$OUTPUT_DIR/${table}.sql"
  if [[ $? -ne 0 ]]; then
    echo "Error dumping table $table"
  else
    echo "Table $table dumped successfully."
  fi
done

# Unset password
unset PGPASSWORD

echo "All tables dumped to $OUTPUT_DIR"