#!/bin/bash

# Configuration
DB_NAME="dataiku"
DB_USER="dataiku"
DB_HOST="localhost"
DB_PORT="5432"
BACKUP_DIR="$DIP_HOME/db_backups"
DUMP_DIR="$1"
PGPASSWORD="dataiku"

# Tables to exclude (add tables here to exclude them)
EXCLUDED_TABLES=("fmusersession" "tenant")

# Check if dump directory argument is provided
if [[ -z "$DUMP_DIR" ]]; then
  echo "Usage: $0 <db_directory>"
  exit 1
fi

# Export password for PostgreSQL commands
export PGPASSWORD


# Convert EXCLUDED_TABLES array to a format suitable for SQL and pg_dump
EXCLUDE_TABLES_SQL=$(printf ",'public.%s'" "${EXCLUDED_TABLES[@]}")
EXCLUDE_TABLES_SQL="${EXCLUDE_TABLES_SQL:1}"  # Remove leading comma for SQL query

# Step 1: Create a backup of the current database, excluding specified tables
echo "Backing up the current database, excluding tables: ${EXCLUDED_TABLES[*]}..."
mkdir -p "$BACKUP_DIR"
BACKUP_FOLDER="$BACKUP_DIR/${DB_NAME}_backup_$(date +%Y%m%d_%H%M%S)/"
$(dirname "$0")/_dump_postgres_tables.sh "$BACKUP_FOLDER"
if [[ $? -ne 0 ]]; then
  echo "Error: Failed to back up the current database."
  unset PGPASSWORD
  exit 1
fi

# Step 2: Drop all tables in the database except excluded tables
echo "Deleting all tables in the database except: ${EXCLUDED_TABLES[*]}..."
TABLES=$(psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT tablename FROM pg_tables WHERE schemaname='public' AND tablename NOT IN ($EXCLUDE_TABLES_SQL);")

if [[ -z "$TABLES" ]]; then
  echo "No tables found to delete."
else
  for table in $TABLES; do
    if [[ " ${EXCLUDED_TABLES[@]} " =~ " $table " ]]; then
      echo "Skipping whitelisted table: $table"
      continue  # Skip this table, do not drop it
    fi

    echo "Dropping table: $table"
    psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -c "DROP TABLE IF EXISTS \"$table\" CASCADE;"
    if [[ $? -ne 0 ]]; then
      echo "Error: Failed to drop table $table."
      unset PGPASSWORD
      exit 1
    fi
  done
fi

# Step 3: Reload the dump files into the now-empty database, excluding specified tables
echo "Restoring database from dump files in $DUMP_DIR, excluding tables: ${EXCLUDED_TABLES[*]}..."
for dump_file in "$DUMP_DIR"/*.sql; do
  # Skip files that match any excluded table name
  for excluded_table in "${EXCLUDED_TABLES[@]}"; do
    if [[ "$dump_file" == *"$excluded_table.sql" ]]; then
      echo "Skipping '$excluded_table' table during import."
      continue 2  # Skip to the next dump file
    fi
  done
  
  echo "Restoring from $dump_file"
  psql -h "$DB_HOST" -p "$DB_PORT" -U "$DB_USER" -d "$DB_NAME" -f "$dump_file"
  if [[ $? -ne 0 ]]; then
    echo "Error: Failed to restore from $dump_file."
    unset PGPASSWORD
    exit 1
  fi
done

# Clean up password
unset PGPASSWORD

echo "Database restored successfully from dump files, excluding tables: ${EXCLUDED_TABLES[*]}."