# Setting Java processes options
from __future__ import print_function
import re, subprocess, logging
import sys
from base import is_os_windows

def get_major(version):
    """Extracts the major version from the version string for Java 9+"""
    match = re.compile(b'(java|openjdk) version "([1-9][0-9]*)[0-9.]*"').match(version.splitlines()[0])
    if not match:
        raise Exception(b"Could not parse Java version: " + version)
    return int(match.group(2))

def get_jvm(javabin):
    """Gets JVM type and version, as a tuple (jvm_kind, major_version)"""
    process = subprocess.Popen([javabin, "-version"], stderr=subprocess.PIPE)
    out, err = process.communicate()
    retcode = process.poll()
    if retcode:
        print(err, file=sys.stderr) # NOSONAR
        raise subprocess.CalledProcessError(retcode, javabin)
    # Lookup known versions
    if b"\nJava HotSpot" in err:
        return ("hotspot", get_major(err))

    elif b"\nOpenJDK " in err:
        return ("openjdk", get_major(err))

    elif b"\nIBM J9 VM " in err or b"\nEclipse OpenJ9 VM " in err:
        return ("j9", get_major(err))

    else:
        return ("unknown", get_major(err))

def get_java_home(javabin):
    """Gets JAVA_HOME value for a java binary. Returns null if failure to extract"""
    process = subprocess.Popen([javabin, "-XshowSettings:properties", "-version"], stderr=subprocess.PIPE)
    out, err = process.communicate()
    retcode = process.poll()
    if retcode:
        print(err, file=sys.stderr) # NOSONAR
        return None

    err_str = err.decode("utf-8")
    r = re.compile("\\s*java\\.home\\s*=\\s*(.*)")
    for line in err_str.splitlines():
        m = r.match(line)
        if m is not None:
            java_home = m.group(1).strip()
            return java_home
    print("JAVA_HOME not found in XshowSettings output", file=sys.stderr) # NOSONAR
    return None

def get_common_java_opts(jvm):
    """Gets Java options that are common to all processes
    (ie added to the common DKU_JAVA_OPTS variable rather than to the per-process DKU_XXX_JAVA_OPTS)"""

    ret = "-ea -Dfile.encoding=utf8 -Djava.awt.headless=true"
    if is_os_windows():
        ret += " -Djava.io.tmpdir=$env:DKU_JAVA_TMP_DIR"
    else:
        ret += " -Djava.io.tmpdir=$DKU_JAVA_TMP_DIR"
        ret += " -Djava.security.egd=file:/dev/./urandom"
    # Required 2 options to enable proxy auth
    # https://www.oracle.com/java/technologies/javase/8u111-relnotes.html -> Disable Basic authentication for HTTPS tunneling
    ret += " -Djdk.http.auth.tunneling.disabledSchemes= -Djdk.http.auth.proxying.disabledSchemes="
    # snowflake keypair support : https://app.shortcut.com/dataiku/story/219526/
    ret += " -Dnet.snowflake.jdbc.enableBouncyCastle=true"

    # Java 17 compatibility fixes
    for module in ("java.base/java.lang", "java.base/java.lang.reflect", "java.base/java.io", "java.base/java.nio", "java.base/java.net", "java.base/java.util"):
        ret += " --add-opens {}=ALL-UNNAMED".format(module)

    # For CDP 7.1.9 and Java 17: https://docs.cloudera.com/cdp-private-cloud-base/7.1.9/installation/topics/cdpdc-java-requirements.html
    ret += " --add-opens=java.base/java.lang=ALL-UNNAMED --add-opens=java.management/com.sun.jmx.mbeanserver=ALL-UNNAMED --add-exports=java.management/com.sun.jmx.mbeanserver=ALL-UNNAMED --add-exports=java.base/sun.net.dns=ALL-UNNAMED --add-exports=java.base/sun.net.util=ALL-UNNAMED"

    return ret

def get_perprocess_java_opts(installConfig, jvm, prefix, permgenDefault="256m", xmxDefault="2g", gcDefault="auto_lowpause", addGCLogging=True):
    ret = ""

    # Memory
    xmx = installConfig.getPrefixedOption("javaopts", prefix, "xmx", xmxDefault)
    ret += "-Xmx%s" % (xmx)

    # GC
    if gcDefault == "auto_lowpause":
        # For "latency-first processes" (the backend), use G1 preferably
        if jvm[0] in [ "hotspot", "openjdk" ]:
            gcDefault = "g1"
        else:
            gcDefault = "parallel"

    gc = installConfig.getPrefixedOption("javaopts", prefix, "gc", gcDefault)

    if gc == "custom":
        # Don't add any GC option
        pass
    else:
        if gc == "parallel":
            ret += " -XX:+UseParallelGC"
            # since parallel is not used for "low latency" processes, it's used for
            # processes less sensitive to lags, so we can also limit the number of
            # threads spawned by the GC (with is around 5/8*numCores, so can be
            # quite a lot)
            try:
                import multiprocessing
                num_cores = int(multiprocessing.cpu_count())
            except:
                logging.warning("Cannot retrieve number of cores, assume 8")
                num_cores = 8 # wild guess
            # make a not-so-stupid default for the number of threads. At least
            # 2 because otherwise parallel GC has no advantage over serial, at
            # most 8 because as said above, we don't need super-duper fast GC
            # with small pauses in all the DSS subprocesses (and there can be
            # quite a few...)
            num_threads = max(2, min(8, num_cores // 4))
            ret += " -XX:ParallelGCThreads=%s" % num_threads
        elif gc == "concmarksweep":
            ret += " -XX:+UseConcMarkSweepGC"
        elif gc == "g1":
            ret += " -XX:+UseG1GC"
        else:
            logging.warn("Unknown GC: " + gc)

        if addGCLogging and jvm[0] in [ "hotspot", "openjdk" ]:
            ret += " -Xlog:gc,gc+cpu=info:stderr:t,um,l,ti,tg"

    # Other
    additionalOpts = installConfig.getPrefixedOption("javaopts", prefix, "additional.opts", "")
    ret += " %s" % additionalOpts

    return ret

def get_perprocess_java_library_path(installConfig, jvm, prefix):
    return installConfig.getPrefixedOption("javaopts", prefix, "library.path", '')

if __name__ == "__main__":
    if len(sys.argv) < 3:
        raise Exception("Usage: java.py jvm|java_home java_bin")
    if sys.argv[1] == "jvm":
        print(get_jvm(sys.argv[2]))
    elif sys.argv[1] == "java_home":
        print(get_java_home(sys.argv[2]))
    else:
        raise Exception("first argument must be jvm or java_home")