(function() {
'use strict';

const app = angular.module('dataiku.admin.code-studios', []);

app.controller("CodeStudioTemplateCoreController", function($scope, CreateModalFromTemplate, DataikuAPI, FutureProgressModal, Dialogs, $q, WT1) {
    $scope.openDeleteCodeStudioTemplateModal = function(codeStudioTemplateId, actionAfterDeletion) {
        var newScope = $scope.$new();
        newScope.codeStudioTemplateId = codeStudioTemplateId;
        newScope.actionAfterDeletion = actionAfterDeletion || function(){/*This is intentional empty*/};
        CreateModalFromTemplate("/templates/admin/code-studios/delete-code-studio-template-modal.html", newScope, "CodeStudioTemplateDeleteController");
    }
    $scope.openMassDeleteCodeStudioTemplateModal = function(codeStudioTemplateIds, actionAfterDeletion) {
        var newScope = $scope.$new();
        newScope.codeStudioTemplateIds = codeStudioTemplateIds;
        newScope.actionAfterDeletion = actionAfterDeletion || function(){/*This is intentional empty*/};
        CreateModalFromTemplate("/templates/admin/code-studios/mass-delete-code-studio-template-modal.html", newScope, "CodeStudioTemplateMassDeleteController");
    }

    $scope.build = function(codeStudioTemplateId, withNoCache) {
        return DataikuAPI.codeStudioTemplates.build(codeStudioTemplateId, withNoCache)
            .then(response => FutureProgressModal.show($scope, response.data, "Build template", undefined, 'static', 'false', true)
                .then(result => {
                    if (result) { // undefined in case of abort
                        Dialogs.infoMessagesDisplayOnly($scope, "Build result", result.messages, result.futureLog);
                    }
                })
            ).catch(setErrorInScope.bind($scope));
    };
    $scope.massBuild = function(codeStudioTemplateIds) {
        var deferred = $q.defer();
        DataikuAPI.codeStudioTemplates.massBuild(codeStudioTemplateIds)
            .then(response => FutureProgressModal.show($scope, response.data, "Build templates", undefined, 'static', 'false', true)
                .then(result => {
                    deferred.resolve(result);
                    if (result) { // undefined in case of abort
                        Dialogs.infoMessagesDisplayOnly($scope, "Build result", result.messages, result.futureLog);
                    }
                })
            ).catch(function (a,b,c) {
                setErrorInScope.bind($scope)(a,b,c);
                deferred.reject("Not build");
            });
        return deferred.promise;
    };

    $scope.removeTooltip = function(){
        const tooltip = document.querySelector('.tooltip.in');
        if (tooltip) tooltip.remove();
    }

    $scope.makeContainerSelectionFn = function(scope) {
        return function(newMode) {
            if (newMode) {
                scope.uiState._mode = newMode;
                switch (newMode) {
                    case "ALLOWED":
                        scope.codeStudioTemplate.allContainerConfs = false;
                        break;
                    case "ALL":
                        scope.codeStudioTemplate.allContainerConfs = true;
                        break;
                }
            }
            return scope.uiState._mode;
        }
    };
});

app.controller("CodeStudioTemplateDeleteController", function($scope, Assert, DataikuAPI, Dialogs) {
    $scope.uiState = {};
    $scope.deleteRuntimes = false;
    $scope.deleteImages = true;

    $scope.refreshUsages = function() {
        DataikuAPI.codeStudioTemplates.getUsage($scope.codeStudioTemplateId).success(function(data) {
            $scope.uiState.usages = {
                totalCount: data.totalCount,
                totalRunningCount: data.totalRunningCount,
                notAccessibleCount: data.notAccessibleCount,
                projects: data.projects.map(p => ({
                    projectKey: p.projectKey,
                    count: p.codeStudios.length,
                    runningCount: p.codeStudios.filter(k => k.running).length}))};
        }).error(setErrorInScope.bind($scope));
    };

    $scope.delete = function() {
        DataikuAPI.codeStudioTemplates.delete($scope.codeStudioTemplateId, $scope.deleteRuntimes, $scope.deleteImages).success(function(data){
            if (data.messages.length > 0) {
                Dialogs.infoMessagesDisplayOnly($scope, "Deletion partially successful.", data, null).then(x =>
                {
                    $scope.dismiss();
                    $scope.actionAfterDeletion()
                });
            } else {
                $scope.dismiss();
                $scope.actionAfterDeletion();
            }
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller("CodeStudioTemplateMassDeleteController", function($scope, Assert, DataikuAPI, FutureProgressModal, Dialogs) {
    $scope.uiState = {};
    $scope.deleteRuntimes = false;
    $scope.deleteImages = true;

    $scope.refreshUsages = function() {
        DataikuAPI.codeStudioTemplates.massGetUsage($scope.codeStudioTemplateIds).success(function(data) {
            let byProject = new Map();
            data.forEach(u => u.projects.forEach(p => {
                if(! byProject.has(p.projectKey)) {
                    byProject.set(p.projectKey, {projectKey:p.projectKey, count: p.codeStudios.length, runningCount: p.codeStudios.filter(k => k.running).length});
                } else {
                    byProject.get(p.projectKey).count += p.codeStudios.length;
                    byProject.get(p.projectKey).runningCount += p.codeStudios.filter(k => k.running).length;
                }
            }));
            $scope.uiState.usages = {
                totalCount: data.reduce((t, u) => t + u.totalCount, 0),
                totalRunningCount: data.reduce((t, u) => t + u.totalRunningCount, 0),
                notAccessibleCount: data.reduce((t, u) => t + u.notAccessibleCount, 0),
                projects: Array.from(byProject.values())
            };
        }).error(setErrorInScope.bind($scope));
    };

    $scope.delete = function() {
        DataikuAPI.codeStudioTemplates.massDelete($scope.codeStudioTemplateIds, $scope.deleteRuntimes, $scope.deleteImages).success(function(data){
            if (data.messages.length > 0) {
                Dialogs.infoMessagesDisplayOnly($scope, "Deletion partially successful.", data, null).then(x =>
                {
                    $scope.dismiss();
                    $scope.actionAfterDeletion()
                });
            } else {
                $scope.dismiss();
                $scope.actionAfterDeletion();
            }
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller("CodeStudioTemplatesListController", function ($scope, $controller, DataikuAPI, CreateModalFromTemplate, TopNav, WT1) {
    $controller("CodeStudioTemplateCoreController", {$scope:$scope});

    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {};
    $scope.selection = { orderQuery: "label", orderReversed: false };

    let makeContainerConfListDisplay = function(item) {
        if (!item.defaultConf) {
            return item.buildFor.map(c => {return {name:c, isDefault:false}});
        } else if (item.buildFor.indexOf(item.defaultConf) >= 0) {
            return item.buildFor.map(c => {return {name:c, isDefault:c==item.defaultConf}});
        } else if (item.buildFor.indexOf("*") >= 0) {
            return [{name:item.defaultConf, isDefault:true}, {name:"*", isDefault:false}];
        } else {
            return [{name:item.defaultConf, isDefault:true, notBuilt:true}].concat(item.buildFor.map(c => {return {name:c, isDefault:false}}));
        }
    };

    $scope.codeStudioTemplates = [];
    $scope.fetchedList = false;
    $scope.refreshList = function() {
        DataikuAPI.codeStudioTemplates.listHeads().success(function(data) {
            $scope.codeStudioTemplates = data.items;
            $scope.codeStudioTemplates.forEach(item => {item.$buildConfForDisplay = makeContainerConfListDisplay(item);});
            $scope.fetchedList = true;
        }).error(setErrorInScope.bind($scope));
    };
    $scope.refreshList();

    $scope.createCodeStudioTemplate = function($event) {
        $scope.withAllTplType = $event.ctrlKey || $event.metaKey;
        CreateModalFromTemplate("/templates/admin/code-studios/new-code-studio-template-modal.html", $scope, "NewCodeStudioTemplateController")
    };

    $scope.uploadCodeStudioTemplate = function() {
        CreateModalFromTemplate("/templates/admin/code-studios/upload-code-studio-template-modal.html", $scope, "UploadCodeStudioTemplateController")
    };

    $scope.deleteCodeStudioTemplate = function(codeStudioTemplate) {
        WT1.event("codestudiotemplate_action_delete", {type:codeStudioTemplate.type});
        $scope.openDeleteCodeStudioTemplateModal(codeStudioTemplate.id, $scope.refreshList);
    };

    $scope.deleteSelectedTemplates = function() {
        WT1.event("codestudiotemplate_action_deleteall", {types:$scope.selection.selectedObjects.map(i => i.type)});
        $scope.openMassDeleteCodeStudioTemplateModal($scope.selection.selectedObjects.map(i => i.id), $scope.refreshList);
    };
    $scope.buildSelectedTemplates = function() {
        WT1.event("codestudiotemplate_action_buildall", {types:$scope.selection.selectedObjects.map(i => i.type)});
        $scope.massBuild($scope.selection.selectedObjects.map(i => i.id)).then($scope.refreshList);
    };
    $scope.massSetContainerConfs = function() {
        var newScope = $scope.$new();
        newScope.codeStudioTemplateIds = $scope.selection.selectedObjects.map(i => i.id);
        newScope.actionAfterDeletion = $scope.refreshList;
        CreateModalFromTemplate("/templates/admin/code-studios/mass-set-code-studio-template-container-config-modal.html", newScope, "CodeStudioTemplateMassSetContainerConfigController");
    };

    $scope.getUsages = function() {
        WT1.event("codestudiotemplate_action_usagesall");
        $scope.uiState.fetchingUsages = true;
        DataikuAPI.codeStudioTemplates.massGetUsage().success(function(data) {
            let byId = data.reduce((m, u) => {m[u.codeStudioTemplateId]=u; return m;}, {});
            $scope.codeStudioTemplates.forEach(item => {item.$usages = byId[item.id];});
        }).error(setErrorInScope.bind($scope)).finally(function() {$scope.uiState.fetchingUsages = false;});
    };
});

app.controller("CodeStudioTemplateMassSetContainerConfigController", function($scope, Assert, DataikuAPI, FutureProgressModal) {
    $scope.uiState = {};

    $scope.codeStudioTemplate = {type:"manual", params:{}, id:"fake"}; // fake used to hold the actual container conf settings

    // default values:
    $scope.codeStudioTemplate.allowContainerConfOverride = true;
    $scope.codeStudioTemplate.allContainerConfs = true;
    $scope.codeStudioTemplate.containerConfs = [];
    $scope.uiState._mode = "ALL";

    $scope.containerSelection = $scope.makeContainerSelectionFn($scope);

    DataikuAPI.containers.listNames("KUBERNETES", "USER_CODE")
        .success(data => {
            $scope.containerNames = data;
            $scope.containerNamesForRun = [''].concat(data);
        })
        .error(setErrorInScope.bind($scope));

    $scope.setConfigs = function() {
        DataikuAPI.codeStudioTemplates.massSetContainerConf($scope.codeStudioTemplateIds, $scope.codeStudioTemplate).success(function(data){
            $scope.dismiss();
            $scope.actionAfterDeletion();
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller("NewCodeStudioTemplateController", function($scope, $rootScope, $state, DataikuAPI, WT1) {

    $scope.newCodeStudioTemplate = {type:'block_based', params: {}};

    WT1.event("codestudiotemplate_action_new");

    $scope.templateTypes = [];
    $scope.templateTypeDescriptions = [];
    DataikuAPI.codeStudioTemplates.listTypes($scope.withAllTplType).success(function(data){
        $scope.templateTypes.push(...data);
        $scope.templateTypeDescriptions.push(...(data.map(t => t.description)));
    }).error(setErrorInScope.bind($scope));

    $scope.create = function(){
        WT1.event("codestudiotemplate_action_create", {type:$scope.newCodeStudioTemplate.type});
        var parentScope = $scope.$parent.$parent;
        DataikuAPI.codeStudioTemplates.create($scope.newCodeStudioTemplate.label, $scope.newCodeStudioTemplate.type).success(function(data){
            $scope.dismiss();
            parentScope.refreshList();
            $state.go("admin.code-studios.code-studio", {codeStudioTemplateId:data.id});
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("UploadCodeStudioTemplateController", function($scope, $state, $stateParams, Assert, TopNav, DataikuAPI, FutureProgressModal, Dialogs, Logs, $q, Fn, WT1) {
    $scope.newCodeStudioTemplate = {};
    $scope.importSpec = {codeEnvs:[]};
    $scope.availableCodeEnvs = [];

    $scope.findCodeEnv = function(codeEnvs, codeEnvName) {
        return Array.dkuFindFn(codeEnvs, function(c) { return c.envName == codeEnvName });
    };

    $scope.codeEnvComparator = function(sourceCodeEnv) {
        var
            source = $scope.findCodeEnv(
            $scope.codeEnvs, sourceCodeEnv);
        var sourceEnvLang = source && source.envLang;
        return function(codeEnv) {
            if (codeEnv.envLang == sourceEnvLang) {
                return "AAAAA" + codeEnv.envLang + "." + codeEnv.name;
            } else {
                return "ZZZZZ" + codeEnv.envLang + "." + codeEnv.name;
            }
        };
    };

    $scope.refreshCodeEnvs = function() {
        $scope.availableCodeEnvs = [{envLang:'PYTHON', envName:'Builtin', builtin:true}, {envLang:'R', envName:'Builtin', builtin:true}];
        DataikuAPI.codeenvs.list('PYTHON').success(function (data) {
            $scope.availableCodeEnvs = $scope.availableCodeEnvs.concat(data);
        }).error(setErrorInScope.bind($scope.errorScope));
        DataikuAPI.codeenvs.list('R').success(function (data) {
            $scope.availableCodeEnvs = $scope.availableCodeEnvs.concat(data);
        }).error(setErrorInScope.bind($scope.errorScope));
    };

    $scope.import = function() {
        Assert.trueish($scope.newCodeStudioTemplate.file, "No template file");

        WT1.event("codestudiotemplate_action_import");

        DataikuAPI.codeStudioTemplates.import($scope.newCodeStudioTemplate.file, $scope.newCodeStudioTemplate.label || '', $scope.importSpec).then(function(data) {
            let result = JSON.parse(data);
            if (result.templateId) {
                $scope.dismiss();
                $state.go("admin.code-studios.code-studio", {codeStudioTemplateId:result.templateId});
            } else {
                $scope.messages = result.messages.messages;
                $scope.codeEnvs = result.codeEnvs;
                $scope.usedCodeEnvs = result.codeEnvs.map(Fn.prop('envName'));
                $scope.refreshCodeEnvs();
            }
        }, function(payload) {
            setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h) {return payload.getResponseHeader(h)});
        });
    }
});

app.controller("CodeStudioTemplateController", function($scope, $rootScope, $controller, $stateParams, Assert, DataikuAPI, $state, TopNav, FutureProgressModal,
                                                     ActivityIndicator, $q, Logs, CreateModalFromTemplate, WT1, Dialogs) {
    $controller("CodeStudioTemplateCoreController", {$scope:$scope});

    $scope.codeStudioTemplateResourcesEmptyCta = {
        title: "No file in this Code Studio template resources.",
        text: "Create your own files to use them in your blocks definition such as 'Append to Dockerfile' or 'Add Starter Files'.\nThese resources are versioned, so you should avoid big files.",
        btnAction: "create",
        btnLabel: "Create your first Code Studio template resource file"
    };

    $scope.appConfig = $rootScope.appConfig;
    $scope.addLicInfo = $rootScope.addLicInfo;

    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {active:'info', _mode:null, blockTypes:{}, canUpdate:false, logs: [], selectedLog: null};

    $scope.codeStudioTemplate = {};
    $scope.origCodeStudioTemplate = {};

    const getBlockTypeAttribute = (block, attr) => $scope.uiState.blockTypes[block.type] ? $scope.uiState.blockTypes[block.type][attr] : undefined;
    $scope.niceBlockTypeLabel = (block) => getBlockTypeAttribute(block, 'label') || 'Unknown block (definition missing)';
    $scope.niceBlockTypeDesc = (block) => getBlockTypeAttribute(block, 'description') || '';
    $scope.isLockedBlock = (block) => getBlockTypeAttribute(block, 'isLocked');

    $scope.refreshItem = function() {
        DataikuAPI.codeStudioTemplates.getFullInfo($stateParams.codeStudioTemplateId).success(function(data) {
            $scope.codeStudioTemplate = data.codeStudioTemplate;
            $scope.codeStudioDesc = data.desc;
            $scope.uiState.canUpdate = data.canUpdate;
            if (data.canUpdate) {
            	// pass the $stateParams.codeStudioTemplateId so that permissions to edit CodeStudios can be checked
			    DataikuAPI.codeStudioBlocks.listTypes($stateParams.codeStudioTemplateId).success(function(data) {
			        $scope.uiState.blockTypes = data.reduce((acc, blockType) => {
			            acc[blockType.type] = blockType;
			            return acc;
			        }, {});
			    }).error(setErrorInScope.bind($scope));
            }

            $scope.origCodeStudioTemplate = angular.copy($scope.codeStudioTemplate);

            $scope.uiState._mode = !$scope.codeStudioTemplate.allContainerConfs && $scope.codeStudioTemplate.containerConfs.length !== 0
                ? 'ALLOWED'
                : 'ALL';

            if (data.canUpdate) {
            	$scope.refreshLogs();
			}

        }).error(setErrorInScope.bind($scope));
    };
    $scope.refreshItem();

    $scope.$watch('uiState.active', function() {
        if (!$scope.uiState.active || !$scope.codeStudioTemplate) return;
        WT1.event("codestudiotemplate_"+$scope.uiState.active+"_viewed", {type:$scope.codeStudioTemplate.type});
    });

    $scope.codeStudioTemplateIsDirty = function() {
        if (!$scope.codeStudioTemplate || !$scope.origCodeStudioTemplate) return false;
        return !angular.equals($scope.codeStudioTemplate, $scope.origCodeStudioTemplate);
    };

    $scope.foldBlock = function(block, all){
        block.$show = !block.$show;
        if (all) {
            $scope.codeStudioTemplate.params.blocks.forEach(b => b.$show = block.$show);
        }
    };

    checkChangesBeforeLeaving($scope, $scope.codeStudioTemplateIsDirty);

    $scope.save = function() {
        WT1.event("codestudiotemplate_action_save", {type:$scope.codeStudioTemplate.type});
        $scope.saveCodeStudioTemplate()
              .catch(setErrorInScope.bind($scope));
    };

    $scope.saveCodeStudioTemplate = function(){
        resetErrorInScope($scope);

        if (!$scope.codeStudioTemplateIsDirty()) { // for when it's called with a keystroke or from start button
            const deferred = $q.defer();
            deferred.resolve("Saved");
            return deferred.promise;
        }
        // safekeep stuff from the UI
        const blocksShowFlags = $scope.codeStudioTemplate.params.blocks ? $scope.codeStudioTemplate.params.blocks.map(b => b.$show || false) : null;
        return DataikuAPI.codeStudioTemplates.save(angular.copy($scope.codeStudioTemplate), null).then(function({data}) {
            $scope.codeStudioTemplate = data;
            $scope.origCodeStudioTemplate = angular.copy($scope.codeStudioTemplate);
            if (blocksShowFlags && $scope.codeStudioTemplate.params.blocks) {
                $scope.codeStudioTemplate.params.blocks.forEach((b, i) => b.$show = blocksShowFlags[i]);
            }
            ActivityIndicator.success("Template saved");
        });
    };

    $scope.deleteCodeStudioTemplate = function() {
        WT1.event("codestudiotemplate_action_delete", {type:$scope.codeStudioTemplate.type});
        $scope.openDeleteCodeStudioTemplateModal($scope.codeStudioTemplate.id, function() {
            $state.go("admin.code-studios.list");
        });
    };

    $scope.gotEntryPoint = function() {
        if ($scope.codeStudioTemplate.type === "block_based") {
            //we look for entrypoint block with a non-empty entrypoint defined
            //and for plugin's blocks (pycdstdioblk_code-studio) which all have an entrypoint (for now)
            return !!$scope.codeStudioTemplate.params.blocks.find(block =>
                block.type.startsWith('pycdstdioblk_code-studio')
                || (block.type === 'entrypoint' && block.params && (block.params.entrypoint || '').trim().length > 0)
            );
        }
        return true;
    };

    $scope.saveAndBuild = function(withNoCache) {
        WT1.event("codestudiotemplate_action_build", {type:$scope.codeStudioTemplate.type, withNoCache:withNoCache});
        $scope.saveCodeStudioTemplate()
            .then(() => $scope.build($stateParams.codeStudioTemplateId, withNoCache))
            .then($scope.refreshItem)
            .catch(setErrorInScope.bind($scope));
    };

    $scope.checkSaveAndBuild = function(withNoCache) {
        if ($scope.gotEntryPoint()) {
            $scope.saveAndBuild(withNoCache);
        } else {
            Dialogs.confirmInfoMessages($scope, "There is no EntryPoint defined", null, 'Are you sure you want to build this template ?', false).then(function() {
                $scope.saveAndBuild(withNoCache);
            });
        }
    }

    $scope.containerSelection = $scope.makeContainerSelectionFn($scope);

    $scope.removeOutdatedContainerConfs = function() {
        $scope.codeStudioTemplate.containerConfs = $scope.codeStudioTemplate.containerConfs.filter(o => $scope.outdatedContainerConfs.indexOf(o) === -1);
    };

    DataikuAPI.containers.listNames("KUBERNETES", "USER_CODE")
        .success(data => {
            $scope.containerNames = data;
            $scope.containerNamesForRun = [''].concat(data);
            if ($scope.codeStudioTemplate && $scope.codeStudioTemplate.params) {
                $scope.outdatedContainerConfs = $scope.codeStudioTemplate.containerConfs.filter(o => $scope.containerNames.indexOf(o) === -1)
            }
        })
        .error(setErrorInScope.bind($scope));

    $scope.$watch("containerNames && codeStudioTemplate.containerConfs", function(nv, ov) {
       if ($scope.codeStudioTemplate && $scope.codeStudioTemplate.params) {
            $scope.outdatedContainerConfs = $scope.codeStudioTemplate.containerConfs.filter(o => $scope.containerNames.indexOf(o) === -1)
        }
    });

    $scope.blockTypeCanBeAdded = function(blockType){
        return !blockType.isUnique || !$scope.codeStudioTemplate.params.blocks || !$scope.codeStudioTemplate.params.blocks.find(b => b.type === blockType.type);
    };

    $scope.addBlock = function(){
        CreateModalFromTemplate("/templates/admin/code-studios/add-block-modal.html", $scope, "CodeStudioTemplateAddBlockController").then(function(blockType){
            WT1.event("codestudiotemplate_action_addblock", {type:blockType.type});
            if( $scope.blockTypeCanBeAdded(blockType) ) {
                $scope.codeStudioTemplate.params.blocks = $scope.codeStudioTemplate.params.blocks || [];
                let newBlock = {type:blockType.type, params:{}, '$show': true};
                $scope.codeStudioTemplate.params.blocks.push(newBlock);
                // some type-specific param init
                if (newBlock.type == 'custom_action') {
                    newBlock.params.actionBehavior = 'WAIT_FOR_RETURN_VALUE';
                    newBlock.params.actionIcon = 'icon-bolt';
                }
                if (newBlock.type == 'exposedPort') {
                    newBlock.params.exposedPort = {label: "", exposeHtml: true, port: 8080, proxiedUrlSuffix: "/"};
                }
            }
        });
    };

    $scope.removeBlock = function(index) {
        let deleted = $scope.codeStudioTemplate.params.blocks.splice(index, 1);
        $scope.removeTooltip();
        WT1.event("codestudiotemplate_action_removeblock", {type:(deleted[0] || {}).type});
    };

    $scope.addSimpleDeploymentExposedPort = function(block) {
        block.params.exposedPorts = block.params.exposedPorts || [];
        block.params.exposedPorts.push({label : "", exposeHtml : true, port : 8080, proxiedUrlSuffix : "/"});
    };

    $scope.removeSimpleDeploymentExposedPort = function(block, idx){
        block.params.exposedPorts.splice(idx, 1);
        $scope.removeTooltip();
    };

    $scope.updateEntrypointExposedPort = function(showExposedPort, block) {
        block.params.exposedPort = showExposedPort ? {label : "", exposeHtml : true, port : 8080, proxiedUrlSuffix : "/"} : null;
    };

    const allZones = ['code_studio_versioned', 'code_studio_resources', 'project_lib_versioned', 'project_lib_resources', 'recipes', 'notebooks', 'user_versioned', 'user_resources'];
    $scope.buildNewSyncZone = function() {
        let existing = ((this.params||{}).syncedZones || []).map(z => z.zone);
        let available = allZones.filter(n => existing.indexOf(n) < 0);
        let added = available.length ? available[0] : 'code_studio_versioned';
        const folderName = added.replaceAll('_', '-').replaceAll('code-studio', 'code_studio');
        return {zone:added, pathInZone:'', pathInContainer:'/home/dataiku/workspace/' + folderName};
    };

    let syncedZonesConflicts = []; // can be at the template level, since there is only one such block or element defining the zones
    $scope.listSyncedZoneConflicts = function(syncedZones) {
        // done like in backend
        let path2zones = {};
        (syncedZones||[]).forEach(z => {
            if (!z.pathInContainer) return;
            path2zones[z.pathInContainer] = (path2zones[z.pathInContainer] || []);
            let zoneDesc = z.zone + (z.pathInZone ? '(' + z.pathInZone + ')' : '');
            if (path2zones[z.pathInContainer].indexOf(zoneDesc) < 0) {
                path2zones[z.pathInContainer].push(zoneDesc);
            }
        });
        let newConflicts = {};
        angular.forEach(path2zones, (zones, pathInContainer) => {
            if (zones.length > 1) {
                newConflicts[pathInContainer] = zones.sort().join(', ');
            }
        });
        // smart update to avoid infinite digest cycles
        let removed = syncedZonesConflicts.filter(c => newConflicts[c.pathInContainer] == null);
        removed.forEach(c => syncedZonesConflicts.splice(syncedZonesConflicts.indexOf(c), 1));
        angular.forEach(newConflicts, (zones, pathInContainer) => {
            let e = syncedZonesConflicts.filter(c => c.pathInContainer == pathInContainer)[0];
            if (e) {
                e.zones = zones;
            } else {
                syncedZonesConflicts.push({pathInContainer:pathInContainer, zones:zones});
            }
        });
        return syncedZonesConflicts;
    };

    $scope.authorisedModes = ['DESIGN_MANAGED', 'AUTOMATION_SINGLE', 'AUTOMATION_VERSIONED', 'PLUGIN_MANAGED']
    $scope.availableCodeEnvs = [];
    DataikuAPI.codeenvs.list('PYTHON').success(function(data) {
        data.filter(ce => $scope.authorisedModes.indexOf(ce.deploymentMode) != -1).forEach(function(ce) {
            $scope.availableCodeEnvs.push({envLang:'PYTHON', envName:ce.envName})
        });
    }).error(setErrorInScope.bind($scope));
    DataikuAPI.codeenvs.list('R').success(function(data) {
        data.filter(ce => $scope.authorisedModes.indexOf(ce.deploymentMode) != -1).forEach(function(ce) {
            $scope.availableCodeEnvs.push({envLang:'R', envName:ce.envName})
        });
    }).error(setErrorInScope.bind($scope));


    $scope.refreshLogs = function() {
        DataikuAPI.codeStudioTemplates.listLogs($stateParams.codeStudioTemplateId).success(function(data) {
            $scope.uiState.logs = data;
            //we want, on init and after a build select the most recent
            $scope.uiState.selectedLog = $scope.uiState.logs.length > 0 ? $scope.uiState.logs[0] : undefined;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.$watch("uiState.selectedLog.buildId", function(nv, ov) {
        if (ov == nv) return;
        if (!$scope.uiState.selectedLog.buildId) {
            $scope.uiState.currentLog = null;
        } else {
            DataikuAPI.codeStudioTemplates.getLogInfo($stateParams.codeStudioTemplateId, $scope.uiState.selectedLog.buildId).success(function(data) {
                $scope.uiState.currentLog = data;
            }).error(setErrorInScope.bind($scope));
        }
    });
    $scope.downloadBuildDiagnosis = function(log) {
        ActivityIndicator.success("Preparing build diagnosis ...");
        downloadURL(DataikuAPI.codeStudioTemplates.getBuildDiagnosisURL($stateParams.codeStudioTemplateId, log.buildId));
    };
    $scope.getBuildLogURL = function(log) {
        return log ? DataikuAPI.codeStudioTemplates.getBuildLogURL($stateParams.codeStudioTemplateId, log.buildId) : '';
    };
    $scope.exportTemplate = function() {
        WT1.event("codestudiotemplate_action_export", {type:$scope.codeStudioTemplate.type});
        $scope.saveCodeStudioTemplate().then(function() {
            ActivityIndicator.success("Preparing template export ...");
            downloadURL(DataikuAPI.codeStudioTemplates.getExportURL($stateParams.codeStudioTemplateId));
        }).catch(setErrorInScope.bind($scope));
    };

    $scope.exportToPlugin = function() {
        CreateModalFromTemplate("/templates/admin/code-studios/export-code-studio-template-to-plugin-modal.html", $scope, "ExportCodeStudioTemplateToPluginController", function(modalScope) {
            modalScope.codeStudioTemplate = $scope.codeStudioTemplate;
            modalScope.saveCodeStudioTemplate = $scope.saveCodeStudioTemplate; // pass the save-if-dirty method
        });
    };

    $scope.flatUsageList = [];
    $scope.computeFlatUsageList = function() {
        $scope.flatUsageList = $scope.uiState.usage.projects.reduce((r, project) => {
            if( project.$expanded || project.codeStudios.length === 1 ) {
                project.codeStudios.forEach((kub, idx) =>
                    r.push({projectKey: project.projectKey, $expanded: true, $idx: idx, $count: project.codeStudios.length, ...kub})
                );
            } else {
                r.push({$idx: 0, $count: project.codeStudios.length, ...project});
            }
            return r;
        }, []);
    };

    $scope.toggleUsage = function(projectKey){
        const project = $scope.uiState.usage.projects.find(p => p.projectKey === projectKey);
        project.$expanded = !project.$expanded;
        $scope.computeFlatUsageList();
    };

    $scope.refreshUsages = function() {
        WT1.event("codestudiotemplate_action_usages", {type:$scope.codeStudioTemplate.type});
        DataikuAPI.codeStudioTemplates.getUsage($stateParams.codeStudioTemplateId, true).success(function(data) {
            if( $scope.uiState.usage ) {
                //we want to keeps current expanded ui state
                if ($scope.uiState.usage && $scope.uiState.usage.projects) {
                    const oldExpanded = $scope.uiState.usage.projects.reduce((oldExpanded, p) => {
                        oldExpanded[p.projectKey] = p.$expanded;
                        return oldExpanded;
                    }, {});
                    if (data.projects) {
                        data.projects = data.projects.map(p => {
                            p.$expanded = oldExpanded[p.projectKey];
                            return p;
                        });
                    }
                }
            } else {
                //first pull, we want the first project expanded
                if (data.projects && data.projects.length > 0)
                    data.projects[0].$expanded = true;
            }
            $scope.uiState.usage = data;
            $scope.computeFlatUsageList();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.getFirstItems = function(items, k, nb){
        const rest = items.length - nb;
        return items.slice(0, nb).map(u => u[k]).join(', ') + (rest > 0 ? ', and ' + rest + ' more...' : '' );
    };

    $scope.isOutdated = function(item){
        //we check if the CodeStudio runtime has been started before the last build.
        return item.runningSince && $scope.uiState.logs.length > 0 && item.runningSince < $scope.uiState.logs[0].builtOn;
    };

    // make sure we only edit one field at a time (otherwise it's messy)
    let currentEditingField = {description: null, shortDesc: null};
    let stopCurrentEdit = function() {
        if (currentEditingField.description) {
            $scope.cancelEditLabel(currentEditingField.description);
        }
        if (currentEditingField.shortDesc) {
            $scope.cancelEditShortDesc(currentEditingField.shortDesc);
        }
    };

    $scope.startEditDescription = function(codeStudioTemplate) {
        stopCurrentEdit();
        $scope.uiState.$editingDescription = true;
        $scope.uiState.edited = codeStudioTemplate.description;
        currentEditingField.description = codeStudioTemplate;
    };
    $scope.stopEditDescription = function(codeStudioTemplate) {
        $scope.uiState.$editingDescription = false;
        codeStudioTemplate.description = $scope.uiState.edited;
        currentEditingField.description = null;
    };
    $scope.cancelEditDescription = function(codeStudioTemplate) {
        $scope.uiState.$editingDescription = false;
        currentEditingField.description = null;
    };
    $scope.startEditShortDesc = function(codeStudioTemplate) {
        stopCurrentEdit();
        $scope.uiState.$editingShortDesc = true;
        $scope.uiState.edited = codeStudioTemplate.shortDesc;
        currentEditingField.shortDesc = codeStudioTemplate;
    };
    $scope.stopEditShortDesc = function(codeStudioTemplate) {
        $scope.uiState.$editingShortDesc = false;
        codeStudioTemplate.shortDesc = $scope.uiState.edited;
        currentEditingField.shortDesc = null;
    };
    $scope.cancelEditShortDesc = function(codeStudioTemplate) {
        $scope.uiState.$editingShortDesc = false;
        currentEditingField.shortDesc = null;
    };


});


app.controller("CodeStudioTemplateAddBlockController", function($scope, $rootScope, $state) {

    $scope.filteredBlocks = [];
    $scope.blocksCount = Object.keys($scope.uiState.blockTypes).length;

    function refreshFilteredBlocks() {
        const query = ($scope.uiState.blocQuery || '').toLowerCase();
        function filterQuery(b) {
            if (b.label && b.label.toLowerCase().indexOf(query) >= 0) return true;
            if (b.description && b.description.toLowerCase().indexOf(query) >= 0) return true;
            return false;
        }
        const blocks = Object.values($scope.uiState.blockTypes).filter(filterQuery);
        blocks.sort((a, b) => a.uiDisplayPriority - b.uiDisplayPriority);
        $scope.filteredBlocks = blocks;
    }
    refreshFilteredBlocks();

    $scope.$watch('uiState.blocQuery', refreshFilteredBlocks);

    $scope.selectBlock = function(blockType){
        $scope.resolveModal(blockType);
    };
});


app.controller("ExportCodeStudioTemplateToPluginController", function($scope, $rootScope, $state, DataikuAPI, PluginsService, WT1, StateUtils, FutureProgressModal) {
    DataikuAPI.plugindev.list().success(function(data) {
        $scope.devPlugins = data;
    }).error(setErrorInScope.bind($scope));

    $scope.convert = {
        mode: 'NEW'
    };

    $scope.isIdValid = function() {
        if ($scope.convert.mode === 'EXISTING') {
            if (!$scope.convert.targetPluginId) return false;
            return PluginsService.isValidComponentId($scope.convert.targetFolder,
                                                    $scope.convert.targetPluginId,
                                                    // eslint-disable-next-line no-undef
                                                    []); // accept that a component exists with the same name, to update it
        } else {
            if (!$scope.convert.newPluginId) return false;
            return PluginsService.isValidComponentId($scope.convert.targetFolder,
                                                    $scope.convert.newPluginId,
                                                    []);
        }
    };

    $scope.doExport = function() {
        WT1.event("codestudiotemplate_action_pluginify", {type:$scope.codeStudioTemplate.type});
        const pluginId = $scope.convert.mode == 'NEW' ? $scope.convert.newPluginId : $scope.convert.targetPluginId;
        const templateLabel = $scope.convert.targetFolder;
        $scope.saveCodeStudioTemplate()
            .then(() => DataikuAPI.codeStudioTemplates.exportToPlugin($scope.codeStudioTemplate.id, pluginId, templateLabel, $scope.convert.overwrite))
            .then(({data}) => {
                $scope.dismiss();
                $scope.reloadPluginConfiguration();
                StateUtils.go.pluginEditor(data.pluginId, data.pathToFiles);
            })
            .catch(setErrorInScope.bind($scope));
    };
});


app.directive('codeStudioTemplateAddCodeEnvBlock', function($rootScope, DataikuAPI) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-studios/add-code-env-block.html',
        scope : {
            block  : '=',
            availableCodeEnvs: '='
        },
        link : function($scope, element, attrs) {
            $scope.ui = { codeEnv: null, versionIds: [] };
            $scope.appConfig = $rootScope.appConfig;

            // setup $scope.ui.codeEnv that the <select /> will be bound to
            let init = function() {
                if ($scope.block && $scope.availableCodeEnvs) {
                    $scope.ui.codeEnv = {envLang:$scope.block.params.envLang, envName:$scope.block.params.envName};
                    // get the one in the code env list if it exists (that's what angular wants to see)
                    $scope.ui.codeEnv = $scope.availableCodeEnvs.filter(c => c.envLang == $scope.ui.codeEnv.envLang && c.envName == $scope.ui.codeEnv.envName)[0] || $scope.ui.codeEnv;
                }
            };
            $scope.$watch('block', init); // shallow watch, for when the block itself changes
            $scope.$watch('availableCodeEnvs', init); // shallow watch, for when the block itself changes

            // version stuff
            let fetchVersions = function() {
                if (!$rootScope.appConfig.isAutomation) return;
                if (!$scope.ui.codeEnv || !$scope.ui.codeEnv.envLang || !$scope.ui.codeEnv.envName) return;
                DataikuAPI.admin.codeenvs.automation.get($scope.ui.codeEnv.envLang, $scope.ui.codeEnv.envName).success(function(data) {
                    if (data.noVersion) {
                        $scope.ui.versionIds = [];
                    } else if (data.currentVersion) {
                        $scope.ui.versionIds = [];
                    } else {
                        $scope.ui.versionIds = ['Latest'].concat(data.versions.map(v => v.versionId));
                    }
                }).error(setErrorInScope.bind($scope));
            };

            // propagate changes to the block params
            let update = function() {
                if ($scope.block && $scope.ui.codeEnv) {
                    $scope.block.params.envLang = $scope.ui.codeEnv.envLang;
                    $scope.block.params.envName = $scope.ui.codeEnv.envName;
                }
                if ($scope.ui.codeEnv) {
                    // refresh version list if needed
                    fetchVersions();
                }
            };
            $scope.$watch('ui.codeEnv', update); // shallow watch, because the codeEnv object is what's bound directly

            $scope.getFullEnvDir = function() {
                if ($scope.block.params.envLang === 'PYTHON') {
                    return '/opt/dataiku/python-code-envs/' + $scope.getEnvDir();
                } else if ($scope.block.params.envLang === 'R') {
                    return '/opt/dataiku/r-code-envs/' + $scope.getEnvDir();
                }
            }

            $scope.getEnvDir = function() {
                if ($scope.block.params.envDir) {
                    return $scope.block.params.envDir;
                } else if ($scope.block.params.envName) {
                    return $scope.block.params.envName.replaceAll(/[^a-zA-Z0-9-_]/ig, '_');
                } else {
                    return '';
                }
            }

        }
    };
});

app.directive('codeStudioTemplateSecurityPermissions', function(PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/code-studios/security-permissions.html',
        scope : {
            codeStudioTemplate  : '='
        },
        link : function($scope, element, attrs) {
            $scope.ui = {};

            $scope.securityPermissionsHooks = {};
            $scope.securityPermissionsHooks.makeNewPerm = function() {
                return {
                    use: false,
                    update: false
                };
            };
            $scope.securityPermissionsHooks.fixupPermissionItem = function(p) {
                p.$useDisabled = p.update;
                p.$updateDisabled = false;
            };
            $scope.securityPermissionsHooks.fixupWithDefaultPermissionItem = function(p, d) {
                if (d.use || d.$useDisabled) {
                    p.$useDisabled = true;
                }
                if (d.update || d.$updateDisabled) {
                    p.$updateDisabled = true;
                }
            };

            $scope.$watch("codeStudioTemplate.owner", function() {
                $scope.ui.ownerLogin = $scope.codeStudioTemplate.owner;
            });

            // Ownership mgmt
            $scope.$watch("ui.ownerLogin", function() {
                PermissionsService.transferOwnership($scope, $scope.codeStudioTemplate, "Code Studio template");
            });
        }
    };
});

app.controller("PluginCodeStudioController", function($scope, PluginConfigUtils) {
    let currentLoadedType = null;
    function reloadDesc() {
        if (!$scope.codeStudioTemplate || !$scope.codeStudioTemplate.type || currentLoadedType == $scope.codeStudioTemplate.type) return; // nothing to do
        currentLoadedType = $scope.codeStudioTemplate.type;

        var loadedDescs = $scope.appConfig.customPythonPluginCodeStudioTemplates.filter(function(x){
            return x.elementType == currentLoadedType;
        });
        if (loadedDescs.length > 0) {
            $scope.loadedDesc = loadedDescs[0];
        } else {
            $scope.loadedDesc = {};
        }

        $scope.pluginDesc = $scope.appConfig.loadedPlugins.filter(function(x){
            return x.id == $scope.loadedDesc.ownerPluginId;
        })[0];

        if (!$scope.codeStudioTemplate.params.config) {
            $scope.codeStudioTemplate.params.config = {}
        }
        if ($scope.loadedDesc && $scope.loadedDesc.desc) {
            PluginConfigUtils.setDefaultValues($scope.loadedDesc.desc.params, $scope.codeStudioTemplate.params.config);
        }
    }

    $scope.$watch('codeStudioTemplate.type', reloadDesc);
    reloadDesc();
});

app.controller("PluginCodeStudioBlockController", function($scope, PluginConfigUtils) {
    let currentLoadedType = null;
    function reloadDesc() {
        if (!$scope.block || !$scope.block.type || currentLoadedType == $scope.block.type) return; // nothing to do
        currentLoadedType = $scope.block.type;

        var loadedDescs = $scope.appConfig.customPythonPluginCodeStudioBlocks.filter(function(x){
            return x.elementType == currentLoadedType;
        });
        if (loadedDescs.length > 0) {
            $scope.loadedDesc = loadedDescs[0];
        } else {
            $scope.loadedDesc = {};
        }

        $scope.pluginDesc = $scope.appConfig.loadedPlugins.filter(function(x){
            return x.id == $scope.loadedDesc.ownerPluginId;
        })[0];

        if (!$scope.block.params.config) {
            $scope.block.params.config = {}
        }
        if ($scope.loadedDesc && $scope.loadedDesc.desc) {
            PluginConfigUtils.setDefaultValues($scope.loadedDesc.desc.params, $scope.block.params.config);
        }
    }

    $scope.$watch('block.type', reloadDesc);
    reloadDesc();
});

})();