(function() {
    'use strict';

    const app = angular.module('dataiku.analysis.mlcore');

    app.component("tsAgGrid", {
        bindings: {
            filters: "<",
            headers: "<",
            rows: "<",
            onFilterChange: "&",
        },
        template: `
            <style>
            .ag-theme-alpine.ag-theme-alpine.ag-theme-alpine .monospace-column .ag-header-cell-text {
                font-family: Monaco, Menlo, Consolas, "Courier New", monospace;
                padding: 2px 4px;
                color: #d14;
                background-color: #f7f7f9;
                border: 1px solid #e1e1e8;
                white-space: nowrap;
            }
            </style>
            <div id="ts-ag-grid-container" class="ag-theme-alpine" style="width:100%; height: 100%; padding-bottom: 20px;"></div>
        `,
        controller: function(AgGrid, $element) {
            const $ctrl = this;
            let rows;
            let headers;
            let gridApi;

            $ctrl.$onInit = function() {
                const $container = $element.find('#ts-ag-grid-container')[0];
                rows = $ctrl.rows;
                headers = $ctrl.headers;

                initGrid($container, initGridOptions(headers, rows));
            }

            $ctrl.$onChanges = (changes) => {
                if (!gridApi) return;

                if (changes && changes.headers) {
                    headers = $ctrl.headers;
                    gridApi.setGridOption('columnDefs', headers);
                }

                if (changes && changes.rows) {
                    rows = $ctrl.rows;
                    gridApi.setGridOption('rowData', rows);
                }

                if (changes && changes.filters) {
                    gridApi.onFilterChanged();
                }
            };

            function doesExternalFilterPass(node) {
                if (node.data) {
                    for (const column of Object.keys($ctrl.filters)) {
                        if($ctrl.filters[column].length > 0) {
                            const filteredValues = $ctrl.filters[column];

                            if (!filteredValues.includes(node.data[column])) {
                                return false;
                            }
                        }
                    }
                }
                return true;
            }

            function initGrid($container, gridOptions) {
                gridApi = AgGrid.createGrid($container, gridOptions);
            }

            function initGridOptions(headers, rows) {
                return {
                    rowData: rows,
                    columnDefs: headers,
                    defaultColDef: {
                        sortable: true,
                        resizable: true,
                        filter: true,
                        menuTabs: ["filterMenuTab", "generalMenuTab"], // hiding the column filter tab because there is an issue when clicking items
                        filterParams: {
                            maxNumConditions: 10,
                        }
                    },
                    columnMenu: 'legacy',
                    suppressPropertyNamesCheck: true,
                    alwaysMultiSort: true,
                    suppressColumnVirtualisation: true, // required so that sizeColumnsToFit() actually does all columns, might negatively affect performance 🤷
                    suppressCsvExport: true,
                    suppressExcelExport: true,
                    pinnedBottomRowData: [],
                    getRowStyle: params => {
                        if (params.node.rowPinned) { // Specific style for bottom average row
                            return {
                                fontWeight:'bold',
                            };
                        }
                    },
                    onFilterChanged: () => {
                        if (gridApi && $ctrl.onFilterChange) {
                            $ctrl.onFilterChange({ gridApi });
                        }
                    },
                    isExternalFilterPresent: isExternalFilterPresent,
                    doesExternalFilterPass: doesExternalFilterPass,
                    tooltipShowDelay: 0,
                    tooltipHideDelay: 1000,
                    onGridReady: () => gridApi.getAllDisplayedColumns().length < 10 ? gridApi.sizeColumnsToFit() : gridApi.autoSizeAllColumns() // this only actually fires the first time we construct the grid
                }
            }

            function isExternalFilterPresent() {
                return Object.keys($ctrl.filters).length > 0;
            }
        }
    });
})();
