(function() {
'use strict';

const app = angular.module('ui.keypress',[]);
// forked from https://github.com/angular-ui/ui-utils


app.factory("KeyHandlingUtils", function($parse){
    var keysByCode = {
        8: 'backspace',
        9: 'tab',
        13: 'enter',
        27: 'esc',
        32: 'space',
        33: 'pageup',
        34: 'pagedown',
        35: 'end',
        36: 'home',
        37: 'left',
        38: 'up',
        39: 'right',
        40: 'down',
        45: 'insert',
        46: 'delete'
    };
    var alpha = 'abcdefghijklmnopqrstuvwxyz';
    for(var i=alpha.length-1; i >=0; i--){
        keysByCode[i+65] = alpha[i];
    }

    var svc = {
        keysByCode : keysByCode,

        prepareCombinations : function(params){
            var combinations = [];
            angular.forEach(params, function (v, k) {
                var combination, expression;
                expression = $parse(v);
                angular.forEach(k.split(' '), function(variation) {
                    combination = {
                        expression: expression,
                        keys: {}
                    };
                    angular.forEach(variation.split('-'), function (value) {
                        combination.keys[value] = true;
                    });
                    combinations.push(combination);
                });
            });
            return combinations
        },

        isTriggered : function(event, combination) {
            var metaPressed = !!(event.metaKey && !event.ctrlKey);
            var altPressed = !!event.altKey;
            var ctrlPressed = !!event.ctrlKey;
            var shiftPressed = !!event.shiftKey;
            var keyCode = event.keyCode;

            // normalize keycodes
            if (!shiftPressed && keyCode >= 97 && keyCode <= 122) {
                keyCode = keyCode - 32;
            }
            var mainKeyPressed = combination.keys[keysByCode[keyCode]] || (keyCode && combination.keys[keyCode.toString()]);


            var metaRequired = !!combination.keys.meta;
            var altRequired = !!combination.keys.alt;
            var ctrlRequired = !!combination.keys.ctrl;
            var shiftRequired = !!combination.keys.shift;
            var ret =
                    mainKeyPressed &&
                    ( metaRequired === metaPressed ) &&
                    ( altRequired === altPressed ) &&
                    ( ctrlRequired === ctrlPressed ) &&
                    ( shiftRequired === shiftPressed )
            return ret;
        }

    }
    return svc;
});


app.directive("dkuLocalKeypress", function(KeyHandlingUtils, Logger){
 return {
        link: function (scope, elm, attrs) {
            var params = scope.$eval(attrs.dkuLocalKeypress);
            var combinations = KeyHandlingUtils.prepareCombinations(params);

            var cb = function(event) {
                combinations.forEach(function(comb) {
                    if (KeyHandlingUtils.isTriggered(event, comb)) {
                        Logger.info("Trigger", comb);
                        scope.$apply(function () {
                            comb.expression(scope, { '$event': event });
                        });
                        event.stopPropagation();
                        event.preventDefault();
                    };
                });
            }
            elm.on("keydown.dkuKeyPressDirective", cb);
            scope.$on('$destroy', function() {
                elm.off('keydown.dkuKeyPressDirective"', cb);
            });
        }
    };
});


app.factory('keypressHelper', ['$parse', 'KeyHandlingUtils', function keypress($parse, KeyHandlingUtils){
    var capitaliseFirstLetter = function (string) {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };

    return function(mode, scope, elm, attrs, prefix, ignoreUpDown, inactiveWhenModal) {
        var params, combinations = [];
        params = scope.$eval(attrs[prefix+capitaliseFirstLetter(mode)]);

        // Prepare combinations for simple checking
        angular.forEach(params, function (v, k) {
            var combination, expression;
            expression = $parse(v);
            angular.forEach(k.split(' '), function(variation) {
                combination = {
                    expression: expression,
                    keys: {}
                };
                angular.forEach(variation.split('-'), function (value) {
                    combination.keys[value] = true;
                });
                combinations.push(combination);
            });
        });

        var callbackOnKeypress = function (event) {
        	if ( inactiveWhenModal ) {
            	if ($(document).find(".modal, .in").length > 0) {
            		// there is an open modal, let it handle the keypress
            		return;
            	}
        	}
            if (ignoreUpDown){
                if(['input','select'].indexOf(event.target.tagName.toLowerCase())!=-1) {
                    if(event.keyCode == 38 || event.keyCode == 40) {
                        return;
                    }
                }
            }
            // No need to do that inside the cycle
            var metaPressed = !!(event.metaKey && !event.ctrlKey);
            var altPressed = !!event.altKey;
            var ctrlPressed = !!event.ctrlKey;
            var shiftPressed = !!event.shiftKey;
            var keyCode = event.keyCode;
            if (keyCode === undefined) return; // seems to be common

            // normalize keycodes
            if (mode === 'keypress' && !shiftPressed && keyCode >= 97 && keyCode <= 122) {
                keyCode = keyCode - 32;
            }

            // Iterate over prepared combinations
            angular.forEach(combinations, function (combination) {
                var mainKeyPressed = combination.keys[KeyHandlingUtils.keysByCode[keyCode]] || combination.keys[keyCode.toString()];

                var metaRequired = !!combination.keys.meta;
                var altRequired = !!combination.keys.alt;
                var ctrlRequired = !!combination.keys.ctrl;
                var shiftRequired = !!combination.keys.shift;

                if (
                    mainKeyPressed &&
                    ( metaRequired === metaPressed ) &&
                    ( altRequired === altPressed ) &&
                    ( ctrlRequired === ctrlPressed ) &&
                    ( shiftRequired === shiftPressed )
                ) {
                    if (prefix == 'global'){
                        event.preventDefault();
                    }

                    // Run the function
                    scope.$apply(function () {
                        combination.expression(scope, { '$event': event });
                    });
                }
            });
        };

        // Check only matching of pressed keys one of the conditions
        elm.on(mode+'.keypressDirective', callbackOnKeypress);

        // This will unbind itself once the scope is destroyed
        scope.$on('$destroy', function() {
            elm.off('.keypressDirective', callbackOnKeypress);
        });
    };
}]);

/**
 * Bind one or more handlers to particular keys or their combination
 * @param hash {mixed} keyBindings Can be an object or string where keybinding expression of keys or keys combinations and AngularJS Exspressions are set. Object syntax: "{ keys1: expression1 [, keys2: expression2 [ , ... ]]}". String syntax: ""expression1 on keys1 [ and expression2 on keys2 [ and ... ]]"". Expression is an AngularJS Expression, and key(s) are dash-separated combinations of keys and modifiers (one or many, if any. Order does not matter). Supported modifiers are 'ctrl', 'shift', 'alt' and key can be used either via its keyCode (13 for Return) or name. Named keys are 'backspace', 'tab', 'enter', 'esc', 'space', 'pageup', 'pagedown', 'end', 'home', 'left', 'up', 'right', 'down', 'insert', 'delete'.
 * @example <input ui-keypress="{enter:'x = 1', 'ctrl-shift-space':'foo()', 'shift-13':'bar()'}" /> <input ui-keypress="foo = 2 on ctrl-13 and bar('hello') on shift-esc" />
 **/
app.directive('uiKeydown', ['keypressHelper', function(keypressHelper){
    return {
        link: function (scope, elm, attrs) {
            keypressHelper('keydown', scope, elm, attrs, 'ui');
        }
    };
}]);


app.directive('ui2Keydown', ['keypressHelper', function(keypressHelper){
    return {
        link: function (scope, elm, attrs) {
            keypressHelper('keydown', scope, elm, attrs, 'ui2', true);
        }
    };
}]);


app.directive('uiKeypress', ['keypressHelper', function(keypressHelper){
    return {
        link: function (scope, elm, attrs) {
            keypressHelper('keypress', scope, elm, attrs, 'ui');
        }
    };
}]);


app.directive('uiKeyup', ['keypressHelper', function(keypressHelper){
    return {
        link: function (scope, elm, attrs) {
            keypressHelper('keyup', scope, elm, attrs, 'ui');
        }
    };
}]);


app.directive('globalKeydown', ['keypressHelper', function(keypressHelper){
    return {
        link: function (scope, elm, attrs) {
            keypressHelper('keydown', scope, $(document), attrs, 'global');
        }
    };
}]);


app.directive('listenKeydown', ['keypressHelper', function(keypressHelper){
    return {
        link: function (scope, elm, attrs) {
            keypressHelper('keydown', scope, $(document), attrs, 'listen');
        }
    };
}]);

})();