(function() {
'use strict';

const app = angular.module('dataiku.recipes');


app.controller("FineTuningRecipeCreationController", function($scope, $stateParams, DataikuAPI, DatasetUtils, $controller, Logger, SmartId) {
    $controller("_RecipeCreationControllerBase", {$scope:$scope});
    $controller("_RecipeOutputNewManagedBehavior", {$scope:$scope});

    $scope.recipeType = "nlp_llm_finetuning";

    // // for safety, to use the _RecipeOutputNewManagedBehavior fully (maybe one day)
    // $scope.setErrorInTopScope = function(scope) {
    //     return setErrorInScope.bind($scope);
    // };
    
    // $scope.singleOutputRole = {name:"finetuned_model", arity:"UNARY", acceptsSavedModel:true};

    function makeMainRole(refs) {
        return {
            main: {
                items: refs.filter(function(ref) {return !!ref;}).map(function(ref) {return {ref: ref}; })
            }
        }
    }

    $scope.creationSettings = {
    };

    // // Override to gather recipe type specific settings
    // $scope.getCreationSettings = function () {
    //     return {};
    // }

    // Creates the recipe object and sends it to the backend
    $scope.doCreateRecipe = function() {
        const inputs = $scope.recipe && $scope.recipe.inputs ? $scope.recipe.inputs : makeMainRole([$scope.io.inputDataset]);

        if ($scope.io.validationDataset) {
            inputs["validation_dataset"] = {
                items: [{ref: $scope.io.validationDataset }]
            }
        }

        const outputModelName = $scope.creationSettings && $scope.creationSettings.outputModelName;
        const recipe = {
            type: $scope.recipeType,
            projectKey: $stateParams.projectKey,
            name: "finetune_" + outputModelName,
            inputs: inputs,
            //outputs: makeMainRole([outputName]),
        };

        const settings = angular.copy($scope.creationSettings);
        if ($scope.zone) {
            settings.zone = $scope.zone;
        }
        return DataikuAPI.flow.recipes.generic.create(recipe, settings);
    };

    $scope.showOutputPane = function() {
        return !!$scope.io.inputDataset || !!$scope.io.inputFolder;
    };

    $scope.formIsValid = function() {
        if (!$scope.inputFolderOnly && !($scope.io.inputDataset && $scope.activeSchema && $scope.activeSchema.columns && $scope.activeSchema.columns.length)) {
            return false;
        }
        return !!($scope.creationSettings && $scope.creationSettings.outputModelName);
        // TODO @nlp recipe fine tuning: add check on unicity of output LLM name?
    };

    let updateInputDatasetSchema = function() {
        if ($scope.availableInputDatasets == null) return;
        if (!$scope.io.inputDataset) return;
        let resolvedSmartId = SmartId.resolve($scope.io.inputDataset, contextProjectKey);
        // get the object to first assert that we need to grab the schema
        let availableInput = $scope.availableInputDatasets.filter(o => o.name == resolvedSmartId.id && o.projectKey == resolvedSmartId.projectKey)[0];
        if (availableInput == null || availableInput.type == 'DATASET') {
            DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, contextProjectKey).success(function(data){
                $scope.activeSchema = data.schema;
            }).error(setErrorInScope.bind($scope));
        } else if (availableInput.type == 'STREAMING_ENDPOINT') {
            DataikuAPI.streamingEndpoints.get(resolvedSmartId.projectKey, resolvedSmartId.id).success(function(data){
                $scope.activeSchema = data.schema;
            }).error(setErrorInScope.bind($scope));
        } else {
            // other objects don't have a schema
            $scope.activeSchema = {columns:[]};
        }
    };

    const inputsIndex = {};
    DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, $scope.recipeType, $scope.datasetsOnly).then(function(data){
        $scope.availableInputDatasets = data[0];
        if ($scope.filterUsableInputsOn) {
            $scope.availableInputDatasets.forEach(function(c) {
                let usability = c.usableAsInput[$scope.filterUsableInputsOn] || {};
                c.usable = usability.usable;
                c.usableReason = usability.reason;
            });
        } else if ($scope.inputDatasetsOnly) {
            $scope.availableInputDatasets = data[0].filter(function(computable){
                return computable.usableAsInput['main'] && computable.usableAsInput['main'].usable;
            });
        }
        $scope.availableOutputDatasets = data[1].filter(function(computable){
            return computable.usableAsOutput['main'] && computable.usableAsOutput['main'].usable && !computable.alreadyUsedAsOutputOf;
        });
        $scope.availableInputDatasets.forEach(function(it) {
            inputsIndex[it.id] = it;
        });
        updateInputDatasetSchema(); // if the inputDataset arrived before the availableInputDatasets
    });

    let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.io.inputDataset = preselectedInputDataset;
        $scope.preselectedInputDataset = preselectedInputDataset;
    });

    $scope.$watch("io.inputDataset", function(nv) {
        if (!nv) return;

        if ($scope.preselectedInputDataset && $scope.io.inputDataset != $scope.preselectedInputDataset){
            $scope.zone = null;
        }

        updateInputDatasetSchema();
    });

});

app.controller("FineTuningRecipeEditionController", function($scope, $controller, $timeout, DataikuAPI, FinetuningUtilsService) {

    $controller("_NLPLLMRecipeControllerBase", {$scope: $scope});

    $scope.loadLLMs("FINE_TUNING");

    $scope.onFineTuningLLMChange = () => {
        $scope.onLLMChange();
        if (!$scope.isLlmCompatibleWithSystemMessage() && !$scope.desc.systemMessageMode) {
            // Set default value if not set before.
            $scope.desc.systemMessageMode = 'NONE';
        }
    };

    $scope.isLlmCompatibleWithSystemMessage = (llmId) => {
        const llmRef = $scope.getLLMRef(llmId);
        if (!llmRef) return false;
        if (llmRef.type === "HUGGINGFACE_TRANSFORMER_LOCAL") {
            // We assume that HF models support and let the user select the correct mode.
            return true;
        } else {
            return llmRef.handlesSystemMessage;
        }
    };

    $scope.refreshModelDeployments = function() {
        DataikuAPI.savedmodels.llmGeneric.deployments.list($scope.recipe.projectKey,$scope.recipe.outputs.finetuned_model.items[0].ref)
            .then(function(res){
                $scope.deployments = res.data;
            })
            .catch(setErrorInScope.bind($scope));
    }

    $timeout(function() {
        if (!$scope.desc.llmId) {
            angular.element('#fine-tuning__llm-selector + .bootstrap-select').click().addClass('open');
        }
        $scope.refreshModelDeployments();

        DataikuAPI.savedmodels.get($scope.recipe.projectKey,$scope.recipe.outputs.finetuned_model.items[0].ref)
            .then(function(res){
                $scope.savedModel = res.data;
            })
            .catch(setErrorInScope.bind($scope));
    }, 200);

    $scope.systemMessageModeOptions = [
        {
            name: 'No system message',
            value: 'NONE'
        },
        {
            name: 'Dynamic',
            value: 'DYNAMIC'
        },
        {
            name: 'Static',
            value: 'STATIC'
        }
    ];

    $scope.quantizationOptions = [
        {
            name: '4 bits',
            value: 'Q_4BIT'
        },
        {
            name: '8 bits',
            value: 'Q_8BIT',
        },
        {
            name: 'No quantization',
            value: 'NONE'
        }
    ];

    $scope.refreshCheckpointOptions = function() {
        // Depending on the presence of validation dataset, we may miss an eval_strategy, so the options may change
        // https://huggingface.co/docs/transformers/v4.50.0/en/main_classes/trainer#transformers.TrainingArguments
        if ($scope.recipe.inputs.validation_dataset && $scope.recipe.inputs.validation_dataset.items.length > 0) {
            $scope.checkpointOptions = [
                {
                    name: 'Keep all evaluation checkpoints',
                    value: 'KEEP_ALL_EVAL'
                },
                {
                    name: 'Keep only the best checkpoint',
                    value: 'KEEP_BEST_ONLY'
                },
                {
                    name: 'Keep only the best and last checkpoints',
                    value: 'KEEP_BEST_AND_LAST_ONLY'
                }
            ];
        }
        else {
            $scope.checkpointOptions = [
                {
                    name: 'Keep all evaluation checkpoints',
                    value: 'KEEP_ALL_EVAL'
                },
                {
                    name: 'Keep only the last checkpoint',
                    value: 'KEEP_BEST_ONLY'
                }
            ];
        }
    }

    $scope.getLLMRef = function(llmId) {
        if (!$scope.availableLLMs || !llmId) return;
        return $scope.availableLLMs.find((llm) => llm.id == llmId);
    }

    $scope.hasHyperparameters = function(llmId) {
        const llmRef = $scope.getLLMRef(llmId);
        if (!llmRef) return false;
        return ["HUGGINGFACE_TRANSFORMER_LOCAL", "SAVED_MODEL_FINETUNED_HUGGINGFACE_TRANSFORMER", "BEDROCK", "SAVED_MODEL_FINETUNED_BEDROCK", "AZURE_OPENAI_MODEL", "SAVED_MODEL_FINETUNED_AZURE_OPENAI", "OPENAI", "SAVED_MODEL_FINETUNED_OPENAI"].includes(llmRef.type);
    }

    $scope.supportsModelDeployment = function() {
        const llmRef = $scope.getLLMRef($scope.desc.llmId);
        if (!llmRef) return;
        return FinetuningUtilsService.supportsModelDeployment(llmRef.type);
    }

    $scope.getDeletedDeploymentsCountMessage = FinetuningUtilsService.getDeletedDeploymentsCountMessage;
    $scope.getNewDeploymentMessage = FinetuningUtilsService.getNewDeploymentMessage

    $scope.$watch('recipe.outputs.finetuned_model', $scope.refreshModelDeployments);
    $scope.$watch('recipe.inputs.validation_dataset.items.length', $scope.refreshCheckpointOptions);
});


app.component("finetuningHyperparametersEditor", {
    bindings: {
        hyperparameters: '=',
        quantizationOptions: '<',
        checkpointOptions: '<',
        llmType: '<'
    },
    templateUrl: 'templates/recipes/fine-tuning/hyperparameters.html',
    controller: function() {

        const $ctrl = this;
        $ctrl.isRemoteFineTuning = function() {
            return ["BEDROCK", "SAVED_MODEL_FINETUNED_BEDROCK", "AZURE_OPENAI_MODEL", "SAVED_MODEL_FINETUNED_AZURE_OPENAI", "OPENAI", "SAVED_MODEL_FINETUNED_OPENAI"].includes($ctrl.llmType);
        }
        $ctrl.$doCheck = () => {
            const isCheckpointModeInvalid = !$ctrl.checkpointOptions.map(e => e.value).includes($ctrl.hyperparameters.localHuggingFace.checkpointMode);
            if (isCheckpointModeInvalid) {
                $ctrl.hyperparameters.localHuggingFace.checkpointMode = "KEEP_BEST_ONLY";
            }
        }
    }
});

}());
