(function(){
'use strict';

const app = angular.module('dataiku.savedmodels', ['dataiku.ml.report', 'dataiku.lambda']);

app.service("SavedModelsService", function($q, DataikuAPI, $stateParams, $state, CreateModalFromTemplate, SmartId, $rootScope, MLModelsUIRouterStates,
    ActiveProjectKey, FullModelLikeIdUtils, RecipeComputablesService, translate, Dialogs, WT1, PluginConfigUtils) {
    const listModels = function(projectKey, filter) {
        const deferred = $q.defer();
        DataikuAPI.savedmodels.listWithAccessible(projectKey).success(function(data){
            const savedModels = data.filter(filter);
            savedModels.forEach(function(sm) {
                if (sm.projectKey !== projectKey) {
                    sm.name = sm.projectKey + "." + sm.name;
                    sm.id = sm.projectKey + "." + sm.id;
                }
            });
            deferred.resolve(savedModels);
        });
        return deferred.promise;
    };

    const svc = {
        nonAPIServiceableMsg: function(sm) {
            if (!sm) {
                return null;
            }
            if (sm.savedModelType === "PROXY_MODEL") {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_EXTERNAL_MODELS", "An API can not be created from External Models");
            } else if (this.isAgent(sm)) {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_AGENTS", "An API can not be created from an Agent");
            } else if (this.isRetrievalAugmentedLLM(sm)) {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_RA_LLM", "An API can not be created from a Retrieval-Augmented LLM");
            } else if (this.isLLMGeneric(sm)) {
                return translate("PROJECT.SAVED_MODEL.CANNOT_CREATE_API_FROM_LLM_MODELS", "An API can not be created from LLM models");
            }
            if (sm.miniTask?.backendType === 'VERTICA') {
                return translate("PROJECT.PERMISSIONS.VERTICA_NOT_SUPPORTED", "Vertica ML backend is no longer supported");
            }
            return null;
        },
        listAPIServiceableModels: function(projectKey) {
            return listModels(projectKey, sm => !svc.nonAPIServiceableMsg(sm));
        },
        listPredictionModels: function(projectKey) {
            return listModels(projectKey, sm => sm.miniTask && sm.miniTask.taskType === 'PREDICTION');
        },
        listEvaluablePredictionModels: function(projectKey) {
            return listModels(projectKey, sm => sm.miniTask && sm.miniTask.taskType === 'PREDICTION' && !['VERTICA', 'DEEP_HUB'].includes(sm.miniTask.backendType));
        },
        listProxyModels: function(projectKey) {
            return listModels(projectKey, sm => sm.savedModelType === 'PROXY_MODEL');
        },
        listClusteringModels: function(projectKey) {
            return listModels(projectKey, sm => sm.miniTask && sm.miniTask.taskType === 'CLUSTERING');
        },
        isActiveVersion: function(fullModelId, savedModel) {
            if (!fullModelId || !savedModel) return;
            return FullModelLikeIdUtils.parse(fullModelId).versionId === savedModel.activeVersion;
        },
        isPartition: function(fullModelId) {
            if (!fullModelId) return;
            return !!FullModelLikeIdUtils.parse(fullModelId).partitionName;
        },
        isExternalMLflowModel: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["MLFLOW_PYFUNC", "PROXY_MODEL"].includes(model.savedModelType);
            }
            return model && model.modeling && model.modeling.algorithm && (
                ["VIRTUAL_MLFLOW_PYFUNC", "VIRTUAL_PROXY_MODEL"].includes(model.modeling.algorithm))
        },
        isMLflowModel: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return "MLFLOW_PYFUNC" === model.savedModelType;
            }
            return model && model.modeling && model.modeling.algorithm && (
                "VIRTUAL_MLFLOW_PYFUNC" === model.modeling.algorithm)
        },
        isProxyModel: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["PROXY_MODEL"].includes(model.savedModelType);
            }
            return model && model.modeling && model.modeling.algorithm && (
                ["VIRTUAL_PROXY_MODEL"].includes(model.modeling.algorithm))
        },
        isPartitionedModel: function(model) {
           return (model && model.partitioning && model.partitioning.dimensions && model.partitioning.dimensions.length > 0);
        },
        isAgent: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["PYTHON_AGENT", "PLUGIN_AGENT", "TOOLS_USING_AGENT"].includes(model.savedModelType);
            }
        },
        isRetrievalAugmentedLLM: function(model) {
            // To keep in sync with SavedModel.savedModelType.savedModelHandlingType
            if (model && model.savedModelType){
                return ["RETRIEVAL_AUGMENTED_LLM"].includes(model.savedModelType);
            }
        },
        isLLMGeneric: function(model) {
            if (model && model.savedModelType) {
                return model.savedModelType.startsWith("LLM_GENERIC");
            }
        },
        isLLM: function(model) {
            return this.isLLMGeneric(model) || this.isAgent(model) || this.isRetrievalAugmentedLLM(model);
        },
        isVisualMLModel: function(model) {
            if (model && model.savedModelType){
                return "DSS_MANAGED" === model.savedModelType;
            }
            return model && model.modeling && model.modeling.algorithm && (
                !["VIRTUAL_MLFLOW_PYFUNC", "VIRTUAL_PROXY_MODEL"].includes(model.modeling.algorithm));
        },
        hasModelData: function(model) {
            return !this.isAgent(model) && !this.isRetrievalAugmentedLLM(model);
        },
        createAndPinInsight: function(model, settingsPane) {
            const insight = {
                projectKey: ActiveProjectKey.get(),
                type: 'saved-model_report',
                params: {savedModelSmartId: SmartId.create(model.id, model.projectKey)},
                name: "Full report of model " + model.name
            };
            let params;
            if (settingsPane) {
                params = MLModelsUIRouterStates.savedModelPaneToDashboardTile(settingsPane, $stateParams);
            }

            CreateModalFromTemplate("/templates/dashboards/insights/create-and-pin-insight-modal.html", $rootScope, "CreateAndPinInsightModalController", function (newScope) {
                newScope.init(insight, params);
            });
        },
        // keep in sync with com.dataiku.dip.server.services.AccessibleObjectsService.AccessibleObject#fromTaggableObject
        asAccessibleObjects: function(models, contextProjectKey) {
            if (!models || !models.length) {
                return [];
            }
            return models.map(m => {
                const isLocal = m.projectKey == contextProjectKey;
                const obj = {
                    type: "SAVED_MODEL",
                    projectKey: m.projectKey,
                    id: m.id,
                    localProject: isLocal,
                    smartId: isLocal?m.id:m.projectKey+"."+m.id,
                    label: m.name,
                    isReaderAccessible: true,
                    subType: m.miniTask.taskType
                };
                obj.object = obj;
                return obj;
            });
        },
        newCodeAgent: function(agentName, zoneId, from) {
            return DataikuAPI.savedmodels.agents.createPython($stateParams.projectKey, agentName, zoneId).success(function (data) {
                WT1.event(
                    'agent-create', {
                        savedModelType: 'PYTHON_AGENT',
                        from: from
                    });
                $state.go('projects.project.savedmodels.savedmodel.versions', { projectKey: $stateParams.projectKey, smId: data.id });
            });
        },
        newCodeAgentPrompt: function(zoneId, from) {
            Dialogs.prompt($rootScope, 'New Code Agent', 'Agent name').then(function (agentName) {
                svc.newCodeAgent(agentName, zoneId, from).error(setErrorInScope.bind($rootScope));
            });
        },
        newVisualAgent: function(agentName, zoneId, from) {
            return DataikuAPI.savedmodels.agents.createToolsUsing($stateParams.projectKey, agentName, zoneId).success(function (data) {
                WT1.event(
                    'agent-create', {
                        savedModelType: 'TOOLS_USING_AGENT',
                        from: from
                    });
                $state.go('projects.project.savedmodels.savedmodel.agent.design', { smId: data.id, fullModelId: `S-${data.projectKey}-${data.id}-${data.activeVersion}` });
            });
        },
        newVisualAgentPrompt: function(zoneId, from) {
            Dialogs.prompt($rootScope, 'New Visual Agent', 'Agent name').then(function (agentName) {
                svc.newVisualAgent(agentName, zoneId, from).error(setErrorInScope.bind($rootScope));
            });
        },
        newPluginAgent: function(agentId, agentName, zoneId, from) {
            return DataikuAPI.savedmodels.agents.createPlugin($stateParams.projectKey, agentId, agentName, zoneId).success(function (data) {
                WT1.event(
                    'agent-create', {
                        savedModelType: 'PLUGIN_AGENT',
                        from: from
                    });
                $state.go('projects.project.savedmodels.savedmodel.versions', { smId: data.id });
            });
        },
        newPluginAgentPrompt: function(agentLabel, agentId, zoneId, from) {
            Dialogs.prompt($rootScope, 'New ' + agentLabel, 'Agent name').then(function (agentName) {
                svc.newPluginAgent(agentId, agentName, zoneId, from).error(setErrorInScope.bind($rootScope));
            });
        },
        getAllPluginAgents: function() {
            const visibilityFilter = PluginConfigUtils.shouldComponentBeVisible($rootScope.appConfig.loadedPlugins);
            return $rootScope.appConfig.customAgents
                .filter(visibilityFilter) 
                .map(agent => ({
                        label: agent.desc.meta.label,
                        agentId: agent.desc.id,
                        description: agent.desc.meta.description
                    })
                );
        },
        getLlmEndpoint: function(savedModelType) {
            return savedModelType === 'RETRIEVAL_AUGMENTED_LLM' ? 'retrievalAugmentedLLMs' : 'agents';
        },
    };

    return svc;
});

/* ************************************ List / Right column  *************************** */

app.controller("SavedModelPageRightColumnActions", function($controller, $scope, $rootScope, $state, DataikuAPI, $stateParams, ActiveProjectKey, ActivityIndicator, SavedModelRenameService) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success((data) => {
        data.description = data.shortDesc;
        data.nodeType = 'LOCAL_SAVEDMODEL';
        data.realName = data.name;
        data.name = data.id;
        data.interest = {};

        $scope.selection = {
            selectedObject : data,
            confirmedItem : data,
        };
    }).error(setErrorInScope.bind($scope));

    $scope.renameSavedModel = function () {
        const savedModel = $scope.savedModel;

        SavedModelRenameService.renameSavedModel({
            scope: $scope,
            state: $state,
            projectKey: savedModel.projectKey,
            savedModelId: savedModel.id,
            savedModelName: savedModel.name,
            onSave: () => { ActivityIndicator.success("Saved"); }
        });
    };
});


app.directive('savedModelRightColumnSummary', function($controller, $state, $stateParams, SavedModelCustomFieldsService, $rootScope, FlowGraphSelection, SmartId,
    DataikuAPI, CreateModalFromTemplate, QuickView, TaggableObjectsUtils, SavedModelsService, LambdaServicesService, ActiveProjectKey, ActivityIndicator,
    FlowBuildService, AnyLoc, SelectablePluginsService, Logger, translate, TypeMappingService, SavedModelRenameService, PluginCategoryService) {

    return {
        templateUrl :'/templates/savedmodels/right-column-summary.html',

        link : function(scope, element, attrs) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;
            scope.LambdaServicesService = LambdaServicesService;

            scope.createAndPinInsight = SavedModelsService.createAndPinInsight;

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.refreshData = function() {
                var projectKey = scope.selection.selectedObject.projectKey;
                var name = scope.selection.selectedObject.name;
                scope.canAccessObject = false;

                DataikuAPI.savedmodels.getFullInfo(ActiveProjectKey.get(), SmartId.create(name, projectKey)).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey || scope.selection.selectedObject.name != name) {
                        return; // too late!
                    }
                    data.realName = data.name;
                    scope.savedModelData = data;
                    scope.savedModel = data.model;
                    scope.savedModel.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;
                    scope.selection.selectedObject.interest = data.interest;
                    scope.isLocalSavedModel = projectKey == ActiveProjectKey.get();
                    scope.isMLSavedModel = SavedModelsService.isVisualMLModel(data.model) || SavedModelsService.isLLMGeneric(data.model);
                    scope.objectAuthorizations = data.objectAuthorizations;
                    scope.isRetrainableSavedModel = scope.isLocalSavedModel && !SavedModelsService.isExternalMLflowModel(scope.savedModel) && !SavedModelsService.isAgent(scope.savedModel) && !SavedModelsService.isRetrievalAugmentedLLM(scope.savedModel);
                    scope.canAccessObject = true;
                }).catch(setErrorInScope.bind(scope));
            };

            scope.publishEnabled = function() {
                if (!$state.is('projects.project.savedmodels.savedmodel.prediction.report')
                    && !$state.is('projects.project.savedmodels.savedmodel.clustering.report')) {
                    return true;
                }
                if (SavedModelsService.isPartition($stateParams.fullModelId)) {
                    scope.publishDisabledReason = "Only the overall model can be published";
                    return false;
                }
                if (!SavedModelsService.isActiveVersion($stateParams.fullModelId, scope.smContext.savedModel)) {
                    scope.publishDisabledReason = "Only the active version can be published";
                    return false;
                }
                return true;
            }

            scope.$on("objectSummaryEdited", function() {
                DataikuAPI.savedmodels.save(scope.savedModel, {summaryOnly: true})
                .success(function(data) {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            });

            scope.$watch("selection.selectedObject",function() {
                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.savedModel = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.zoomToOtherZoneNode = function(zoneId) {
                const otherNodeId = scope.selection.selectedObject.id.replace(/zone__.+?__saved/, "zone__" + zoneId + "__saved");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId, id: graphVizUnescape(otherNodeId) }))
                }
                else {
                    scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[otherNodeId]);
                }
            }

            scope.renameSavedModel = function() {
                const savedModel = scope.savedModel;

                SavedModelRenameService.renameSavedModel({
                    scope: scope,
                    state: $state,
                    projectKey: savedModel.projectKey,
                    savedModelId: savedModel.id,
                    savedModelName: savedModel.name
                });
            }

            scope.trainModel = function() {
                const modalOptions = {
                    upstreamBuildable: scope.savedModelData.upstreamBuildable,
                    downstreamBuildable: scope.savedModelData.downstreamBuildable,
                };
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "SAVED_MODEL",
                            AnyLoc.makeLoc(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name), modalOptions);
            };

            scope.editCustomFields = function() {
                if (!scope.selection.selectedObject) {
                    return;
                }
                DataikuAPI.savedmodels.getSummary(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.name).success(function(data) {
                    let savedModel = data.object;
                    let modalScope = angular.extend(scope, {objectType: 'SAVED_MODEL', objectName: savedModel.name, objectCustomFields: savedModel.customFields});
                    CreateModalFromTemplate("/templates/taggable-objects/custom-fields-edit-modal.html", modalScope).then(function(customFields) {
                        SavedModelCustomFieldsService.saveCustomFields(savedModel, customFields);
                    });
                }).error(setErrorInScope.bind(scope));
            };

            scope.selectablePlugins = SelectablePluginsService.listSelectablePlugins({'SAVED_MODEL' : 1});
            scope.noRecipesCategoryPlugins = PluginCategoryService.standardCategoryPlugins(scope.selectablePlugins, ['code'])

            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);

            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "SAVED_MODEL", ActiveProjectKey.get(), scope.selection.selectedObject.name).success(function(data) {

                    scope.selection.selectedObject.interest = data;
                    scope.savedModelData.interest = data;

                }).error(setErrorInScope.bind(scope));
            }

            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            scope.$on("$destroy", interestsListener);

            scope.cannotCreateAPIServiceMsg = function() {
                if (!scope.canWriteProject()) {
                    return translate("PROJECT.PERMISSIONS.WRITE_ERROR", "You don't have write permissions for this project");
                }
                const msg = SavedModelsService.nonAPIServiceableMsg(scope.savedModel);
                if (msg) {
                    return msg;
                }
                return null;
            }

            scope.getIcon = function(selection) {
                return TypeMappingService.mapSavedModelSubtypeToIcon(
                    selection.selectedObject.type || selection.selectedObject.taskType || (selection.selectedObject.miniTask || {}).taskType,
                    selection.selectedObject.backendType,
                    selection.selectedObject.predictionType || (selection.selectedObject.miniTask || {}).predictionType,
                    selection.selectedObject.savedModelType,
                    selection.selectedObject.externalSavedModelType || (selection.selectedObject.proxyModelConfiguration && selection.selectedObject.proxyModelConfiguration.protocol),
                    24
                );
            }
        }
    }
});

app.service("SavedModelCustomFieldsService", function($rootScope, TopNav, DataikuAPI, ActivityIndicator, WT1){
    let svc = {};

    svc.saveCustomFields = function(savedModel, newCustomFields) {
        WT1.event('custom-fields-save', {objectType: 'SAVED_MODEL'});
        let oldCustomFields = angular.copy(savedModel.customFields);
        savedModel.customFields = newCustomFields;
        return DataikuAPI.savedmodels.save(savedModel, {summaryOnly: true})
            .success(function(data) {
                ActivityIndicator.success("Saved");
                $rootScope.$broadcast('customFieldsSaved', TopNav.getItem(), savedModel.customFields);
                $rootScope.$broadcast('reloadGraph');
            })
            .error(function(a, b, c) {
                savedModel.customFields = oldCustomFields;
                setErrorInScope.bind($rootScope)(a, b, c);
            });
    };

    return svc;
});

app.controller('_ModelListController', function($scope, $controller, $stateParams, CreateModalFromComponent, createExternalSavedModelModalDirective, createExternalSavedModelSelectorModalDirective, $state, TopNav, TypeMappingService) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'realName', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["realName", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.maxItems = 20;

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.savedmodels.savedmodel.versions", {projectKey : $stateParams.projectKey, smId : data.id});
    };

    $scope.prepareListItems = function(listItems) {
        // dirty things to handle the discrepancy between the types of selected objects
        // which can have info displayed in the right panel
        listItems.forEach(sm => {
            sm.realName = sm.name;
            sm.name = sm.id;
            let proxyModelProtocol = null;
            if (sm.proxyModelConfiguration) {
                proxyModelProtocol = sm.proxyModelConfiguration.protocol;
            }
            sm.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(sm.type, sm.backendType, sm.predictionType, sm.savedModelType, proxyModelProtocol, 24);
            sm.colorClassName = TypeMappingService.mapSavedModelTypeToClassColor(sm.savedModelType);
        });

        return listItems;
    };

    $scope.newExternalSavedModel = function(externalModelTypeName) {
        if (externalModelTypeName) {
            CreateModalFromComponent(createExternalSavedModelModalDirective, { externalModelTypeName }, ['modal-wide']);
        }
        else {
            CreateModalFromComponent(createExternalSavedModelSelectorModalDirective, {}, ['modal-wide']);
        }
    };
});

app.controller('SavedModelListController', function($scope, $controller, $stateParams, DataikuAPI, TopNav) {
    $controller('_ModelListController', {$scope: $scope});

    $scope.list = function() {
        DataikuAPI.savedmodels.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = $scope.prepareListItems(data.filter(sm => sm?.type !== 'LLM_GENERIC_RAW'));
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();
});

app.controller('GenAIListController', function($scope, $controller, $rootScope, $stateParams, DataikuAPI, TopNav, CreateModalFromTemplate, SavedModelsService) {
    $controller('_ModelListController', {$scope: $scope});

    $scope.list = function() {
        DataikuAPI.savedmodels.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = $scope.prepareListItems(data.filter(sm => sm?.type === 'LLM_GENERIC_RAW'));
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    $scope.pluginAgentsTypes = $rootScope.appConfig.customAgents.map(ca => ca.desc);

    $scope.newAgent = function() {
        CreateModalFromTemplate('/templates/savedmodels/agents/new-agent-selector-modal.html', $scope, 'AgentSelectorModalController');
    };
});

app.filter("niceGenAIModelType", function($rootScope) {
    return function(sm) {
        if (sm.savedModelType === 'PYTHON_AGENT') {
            return "Code Agent";
        } else if (sm.savedModelType === 'TOOLS_USING_AGENT') {
            return "Visual Agent";
        } else if (sm.savedModelType === 'PLUGIN_AGENT') {
            const pluginAgent = $rootScope.appConfig.customAgents.find(ca => ca.agentType === sm.pluginAgentType);
            return pluginAgent ? pluginAgent.desc.meta.label : "Custom Agent";
        } else if (sm.savedModelType === 'RETRIEVAL_AUGMENTED_LLM') {
            return "Retrieval-augmented LLM";
        } else if (sm.savedModelType === 'LLM_GENERIC') {
            return "Fine-tuned LLM";
        } else {
            return sm.savedModelType;
        }
    };
});


app.controller("SavedModelController", function($scope, $rootScope, Assert, DataikuAPI, CreateModalFromTemplate, $state,
    $stateParams, SavedModelsService, MLExportService, ActiveProjectKey, WebAppsService, FullModelLikeIdUtils,
    createOrAppendMELikeToModelComparisonModalDirective, CreateModalFromComponent, MLModelsUIRouterStates, ExportModelDatasetService,
    AnyLoc, FlowBuildService, SmartId, TopNav) {
    $scope.versionsContext = {}
    $scope.smContext = {};
    $scope.uiState = {};
    $scope.clearVersionsContext = function(){
        // eslint-disable-next-line no-undef
        clear($scope.versionsContext);
    };
    $scope.isAgent = function(snippetData) {
        return SavedModelsService.isAgent(snippetData);
    };
    $scope.agentModelIsDirty = function() {
        return $rootScope.agentModelIsDirty();
    };
    $scope.saveAgentModel = function() {
        return $rootScope.saveAgentModel();
    };

    $scope.isRetrievalAugmentedLLM = function(snippetData) {
        return SavedModelsService.isRetrievalAugmentedLLM(snippetData);
    };
    $scope.retrievalAugmentedModelIsDirty = function() {
        return $rootScope.retrievalAugmentedModelIsDirty();
    };
    $scope.saveRetrievalAugmentedModel = function() {
        return $rootScope.saveRetrievalAugmentedModel();
    };

    $scope.isLLM = function(snippetData) {
        return SavedModelsService.isLLM(snippetData);
    };

    $scope.showMainPanel = function(tabsType){
        return ['SAVED_MODEL', 'PLUGIN_AGENT-SAVED_MODEL-VERSION', 'RETRIEVAL-AUGMENTED_LLM-SAVED_MODEL-VERSION'].includes(tabsType);
    };

    $scope.trainModel = function() {
        FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "SAVED_MODEL",
                            AnyLoc.makeLoc($stateParams.projectKey, $stateParams.smId));
    };

    $scope.goToAnalysisModelFromVersion = function(){
        Assert.trueish($scope.smContext.model, 'no model data');
        Assert.trueish($scope.smContext.model.smOrigin, 'no origin analysis');

        const id = $scope.smContext.model.smOrigin.fullModelId;
        const elements = FullModelLikeIdUtils.parse(id);

        var params =  {
            projectKey: elements.projectKey, // ProjectKey from SavedModels is updated when reading it
            analysisId: elements.analysisId,
            mlTaskId: elements.mlTaskId,
            fullModelId: id
        }

        if ($scope.smContext.model.smOrigin.origin == "EXPORTED_FROM_ANALYSIS") {
           $scope.goToAnalysis(params);
         } else {
            CreateModalFromTemplate("/templates/savedmodels/go-to-analysis-model-modal.html", $scope, null, function(newScope){
                newScope.go = function(){
                    newScope.dismiss();
                    $scope.goToAnalysis(params);
                }
            })
        }
    }

    $scope.goToAnalysis = function(params) {
        let state = "projects.project.analyses.analysis.ml.";
        let api;
        if ($state.includes("projects.project.savedmodels.savedmodel.prediction")) {
            state += "predmltask";
            api = DataikuAPI.analysis.pml;
        } else {
            state += "clustmltask";
            api = DataikuAPI.analysis.cml;
        }
        api.getTaskStatus(params.projectKey, params.analysisId, params.mlTaskId)
            .success(function (data) {
                if (data.fullModelIds.some(model => model.fullModelId.fullId === params.fullModelId)) {
                    state += ".model.report"
                }
                else {
                    state += ".list.results"
                }
                $state.go(state, params);
            })
            .error(function () {
                setErrorInScope.bind($scope);
                $state.go(`${state}.list.results`, params);
            });
    }

    $scope.goToExperimentRunFromVersion = function() {
        Assert.trueish($scope.smContext.model, 'no model data');
        Assert.trueish($scope.smContext.model.mlflowOrigin, 'no MLflow origin');
        Assert.trueish($scope.smContext.model.mlflowOrigin.type === 'EXPERIMENT_TRACKING_RUN', 'MLflow origin not from experiment/run');
        Assert.trueish($scope.smContext.model.mlflowOrigin.runId, 'MLflow origin malformed (no runId)');

        const origin = $scope.smContext.model.mlflowOrigin;
        const params = {
            projectKey: $stateParams.projectKey,
            runId: origin.runId
        };
        $state.go("projects.project.experiment-tracking.run-details", params);
    }

    $scope.showPredictedData = function() {
        let model = $scope.smContext.model;
        if (!model) {
            return false;
        }

        if ($scope.isExternalMLflowModel(model)) {
            // MLFlow and external models do not support predicted data straighforwardly
            return false;
        }

        // For partitioned models only show predicted data tab for individual partitions
        const isPartitionedModel = model.coreParams.partitionedModel && model.coreParams.partitionedModel.enabled;
        const isIndividualPartitionedModel = SavedModelsService.isPartition(model.fullModelId);
        if (isPartitionedModel && !isIndividualPartitionedModel) {
            return false;
        }

        const algoName = model.actualParams.resolved.algorithm;
        return !/^(MLLIB|SPARKLING|VERTICA|PYTHON_ENSEMBLE|SPARK_ENSEMBLE|KERAS)/i.test(algoName);
    }

    $scope.showExportModel = function() {
        return $scope.smContext.model && MLExportService.showExportModel($scope.appConfig);
    };
    $scope.mayExportModel = function(type) {
        return MLExportService.mayExportModel($scope.smContext.model, type);
    };
    $scope.downloadDocForbiddenReason = function() {
        return MLExportService.downloadDocForbiddenReason($scope.appConfig, $scope.smContext.model);
    }
    $scope.downloadDoc = function() {
        return MLExportService.downloadDoc($scope);
    }
    $scope.exportModelModal = function() {
        MLExportService.exportModelModal($scope, $scope.smContext.model)
    }
    $scope.disableExportModelModalReason = function() {
        return MLExportService.disableExportModelModalReason($scope.smContext.model);
    }

    $scope.exportTrainTestSets = function() {
        ExportModelDatasetService.exportTrainTestSets($scope, $scope.smContext.model.fullModelId);
    };
    $scope.exportTrainTestSetsForbiddenReason = function() {
        return ExportModelDatasetService.exportTrainTestSetsForbiddenReason($scope.smContext.model, $scope.canWriteProject());
    };

    $scope.exportPredictedDataForbiddenReason = function() {
        return ExportModelDatasetService.exportPredictedDataForbiddenReason($scope.smContext.model);
    };

    $scope.comparisonForbiddenReason = function() {
        if (!$scope.smContext.model) {
            return null;
        }
        if($scope.smContext.model.coreParams.backendType === "DEEP_HUB") {
            return "Computer vision model comparison is not supported";
        }
        if($scope.smContext.model.coreParams.taskType !== "PREDICTION") {
            return "Only prediction models can be compared";
        }
        if($scope.smContext.model.trainInfo.state !== 'DONE') {
            return "Cannot compare a model being trained or that failed";
        }
        if($scope.smContext.model.coreParams.partitionedModel && $scope.smContext.model.coreParams.partitionedModel.enabled) {
            return "Cannot compare a partitioned model";
        }
        if($scope.smContext.model.modeling.ensemble_params) {
            return "Cannot compare an ensembled model";
        }
        if(($scope.smContext.model.modeling.algorithm === "VIRTUAL_MLFLOW_PYFUNC") &&
            ($scope.smContext.model.coreParams.prediction_type === undefined)) {
            return "Non tabular MLflow models cannot be compared";
        }
        if(($scope.smContext.model.modeling.algorithm === "VIRTUAL_PROXY_MODEL") &&
            ($scope.smContext.model.coreParams.prediction_type === undefined)) {
            return "Non tabular External Models cannot be compared";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.compareModel = function(type) {
        CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
            fullIds: [$scope.smContext.model.fullModelId],
            modelTaskType: $scope.smContext.model.coreParams.prediction_type, // ModelTaskType values encompass PredictionType possible values. So this "cast" will work.
            allowImportOfRelatedEvaluations: true,
            suggestedMCName: `Compare 1 model version from ${$scope.smContext.model.userMeta.name}`,
            projectKey: $stateParams.projectKey,
            trackFrom: 'saved-model-page'
        });
    };

    $scope.createAndPinInsight = SavedModelsService.createAndPinInsight;
    $scope.getPublishDisabledReason = () => {
        for (const pane of ['exploration-constraints', 'exploration-results']) {
            if ($scope.uiState && $scope.uiState.settingsPane && $scope.uiState.settingsPane.includes(pane)) {
                return 'Publishing this page on dashboards is not supported';
            }
        }
        return null;
    }

    $scope.isActiveVersion = SavedModelsService.isActiveVersion;
    $scope.isPartition = SavedModelsService.isPartition;
    $scope.isExternalMLflowModel = SavedModelsService.isExternalMLflowModel;
    $scope.isMLflowModel = SavedModelsService.isMLflowModel;
    $scope.isVisualMLModel = SavedModelsService.isVisualMLModel;

    $scope.$on("$destroy", $scope.clearVersionsContext);

    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(sm) {
        $scope.savedModel = sm;
        if ($scope.savedModel.savedModelType === "LLM_GENERIC") {
             DataikuAPI.savedmodels.getFullInfo(ActiveProjectKey.get(), SmartId.create(sm.id, sm.projectKey)).then(function({data}) {
                $scope.savedModelFullInfo = data;  // to have the back to Fine tune recipe button
             });
        }
        TopNav.setPageTitle($scope.savedModel.name);
    }).error(setErrorInScope.bind($scope));
});

app.controller("_SavedModelGovernanceStatusController", function($scope, $rootScope, FullModelLikeIdUtils, DataikuAPI) {
    $scope.getGovernanceStatus = function(fullModelId, partitionedModel) {
        $scope.modelGovernanceStatus = undefined;
        if (!$rootScope.appConfig.governEnabled) return;
        if (!fullModelId) return;
        // ignore non-saved model version
        if (!FullModelLikeIdUtils.isSavedModel(fullModelId)) return;
        // ignore partition model versions
        if (FullModelLikeIdUtils.isPartition(fullModelId)) return;
        // ignore partitioned model versions
        if (partitionedModel && partitionedModel.enabled) return;

        const fmi = FullModelLikeIdUtils.parse(fullModelId);
        $scope.modelGovernanceStatus = { loading: true };
        DataikuAPI.savedmodels.getModelVersionGovernanceStatus(fmi.projectKey, fmi.savedModelId, fullModelId).success(function(data) {
            $scope.modelGovernanceStatus = { loading: false, data: data };
        }).error(function(a,b,c,d) {
            const fatalAPIError = getErrorDetails(a,b,c,d);
            fatalAPIError.html = getErrorHTMLFromDetails(fatalAPIError);
            $scope.modelGovernanceStatus = { loading: false, error: fatalAPIError };
        });
    };
});

app.filter('savedModelMLTaskHref', function($state, $stateParams, ActiveProjectKey, FullModelLikeIdUtils) {
    return function(sm) {
        if (!sm || !sm.lastExportedFrom) return;

        const elements = FullModelLikeIdUtils.parse(sm.lastExportedFrom);

        const params =  {
            projectKey: elements.projectKey,  // ProjectKey from SavedModels is updated when reading it
            analysisId: elements.analysisId,
            mlTaskId: elements.mlTaskId
        };

        const type = sm.type || sm.miniTask.taskType;
        let state = "projects.project.analyses.analysis.ml.";
        if (type == "PREDICTION") {
            state += "predmltask.list.results";
        } else {
            state += "clustmltask.list.results";
        }
        return $state.href(state, params);
    };
});

/* ************************************ Versions listing *************************** */



app.controller("SavedModelVersionsController", function($scope, Assert, DataikuAPI, $state, $stateParams, TopNav, $controller, Logger,
                                                        GraphZoomTrackerService, ActiveProjectKey, MLDiagnosticsService,
                                                        createOrAppendMELikeToModelComparisonModalDirective, createProxySavedModelVersionModalDirective,
                                                        CreateModalFromComponent, SavedModelsService, RecipeComputablesService, WT1, MetricsUtils, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS,
                                                        CreateModalFromTemplate, MlflowImportModelsService){
    TopNav.setItem(TopNav.ITEM_SAVED_MODEL, $stateParams.smId);
    angular.extend($scope, MLDiagnosticsService);

    if (!$stateParams.fromFlow) {
        // Do not change the focus item zoom if coming from flow
        GraphZoomTrackerService.setFocusItemByName("savedmodel", $stateParams.smId);
    }
    $scope.initialized = false;
    $scope.snippetSource = 'SAVED';

    $scope.isModelDone = function() {
        return true;
    };
    $scope.isModelRunning = function() {
        return false;
    };
    $scope.isSessionRunning = function() {
        return false;
    };

    $scope.isMetricFailed = MetricsUtils.isMetricFailed;
    $scope.getSpecificCustomMetricResult = MetricsUtils.getSpecificCustomMetricResult;

    $scope.isExternalMLflowModel = function(snippetData) {
        return SavedModelsService.isExternalMLflowModel(snippetData);
    }
    $scope.getExternalModelDetails = function(model) {
        return AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === model.proxyModelConfiguration.protocol);
    }
    $scope.isProxyModel = function(snippetData) {
        return SavedModelsService.isProxyModel(snippetData);
    }
     $scope.isMLflowModel = function(snippetData) {
        return SavedModelsService.isMLflowModel(snippetData);
    }
    $scope.isAgent = function(snippetData) {
        return SavedModelsService.isAgent(snippetData);
    };
    $scope.isRetrievalAugmentedLLM = function(snippetData) {
        return SavedModelsService.isRetrievalAugmentedLLM(snippetData);
    };
    $scope.isLLMGeneric = function(snippetData) {
        return SavedModelsService.isLLMGeneric(snippetData);
    };

    $scope.isARecipeOutput = false;

    $scope.newProxySavedModelVersion = function() {
        CreateModalFromComponent(createProxySavedModelVersionModalDirective, { savedModel: $scope.savedModel, smStatus: $scope.smStatus }, ['modal-wide']);
    }

    $scope.openImportMlflowModelVersionModal = function () {
        MlflowImportModelsService.openImportMlflowModelVersionModal({ savedModel: $scope.savedModel, predictionType: $scope.savedModel.miniTask.predictionType });
    }

    $scope.isTimeseriesPrediction = function () {
        return $scope.savedModel &&
               $scope.savedModel.miniTask &&
               $scope.savedModel.miniTask.predictionType == "TIMESERIES_FORECAST";
    }


    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(savedModel) {
        WT1.event(
            'saved-model-open', {
                savedModelType: savedModel.savedModelType,
                predictionType: (savedModel.miniTask || {}).predictionType,
                taskType: (savedModel.miniTask || {}).taskType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            });

        $scope.savedModel = savedModel;
        $scope.smContext.savedModel = savedModel;
        if (SavedModelsService.isLLM(savedModel)) {
            TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_SAVED_MODEL, "versions");
        } else {
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, TopNav.TABS_SAVED_MODEL, "versions");
        }
        TopNav.setItem(
            TopNav.ITEM_SAVED_MODEL,
            $stateParams.smId,
            {
                name: savedModel.name,
                taskType: (savedModel.miniTask || {}).taskType,
                backendType: (savedModel.miniTask || {}).backendType,
                savedModelType: savedModel.savedModelType,
                predictionType: (savedModel.miniTask || {}).predictionType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            }
        );

        if ($scope.savedModel.miniTask) {
            const taskType = $scope.savedModel.miniTask.taskType;
            Assert.trueish(['PREDICTION', 'CLUSTERING'].includes(taskType), 'Unknown task type');
            if (taskType === 'PREDICTION') {
                if ($stateParams.redirectToActiveVersion) {
                    $state.go('projects.project.savedmodels.savedmodel.prediction.report', {
                        fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                    }, {location: 'replace'});
                } else {
                    $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.prediction';
                    $controller("PredictionSavedModelVersionsController", { $scope });
                }
            } else if (taskType === "CLUSTERING") {
                if ($stateParams.redirectToActiveVersion) {
                    $state.go('projects.project.savedmodels.savedmodel.clustering.report', {
                        fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                    }, {location: 'replace'});
                } else {
                    $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.clustering';
                    $controller("ClusteringSavedModelVersionsController" , {$scope});
                }
            }
        } else if ($scope.savedModel.savedModelType == "LLM_GENERIC") {
            if ($stateParams.redirectToActiveVersion) {
                $state.go('projects.project.savedmodels.savedmodel.llmGeneric.report', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.llmGeneric';
                $controller("LLMGenericModelVersionsController", {$scope, $stateParams});
            }
        } else if ($scope.savedModel.savedModelType == "PYTHON_AGENT") {
            if ($stateParams.redirectToActiveVersion) {
                $state.go('projects.project.savedmodels.savedmodel.agent.design', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.agent';
                $controller("PythonAgentModelVersionsController", { $scope });
            }
        } else if ($scope.savedModel.savedModelType == "PLUGIN_AGENT") {
            $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.agent';
            $controller("PluginAgentModelVersionsController", {$scope});
        } else if ($scope.savedModel.savedModelType == "TOOLS_USING_AGENT") {
            if ($stateParams.redirectToActiveVersion) {
                $state.go('projects.project.savedmodels.savedmodel.agent.design', {
                    fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
                }, {location: 'replace'});
            } else {
                $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.agent';
                $controller("ToolsUsingAgentModelVersionsController", { $scope });
            }
        } else if ($scope.savedModel.savedModelType == "RETRIEVAL_AUGMENTED_LLM") {
            $scope.sRefPrefix = 'projects.project.savedmodels.savedmodel.retrievalaugmentedllm';
            $controller("RetrievalAugmentedLLMModelVersionsController", {$scope});

        } else {
            Logger.error("Unknown saved model type: " + $scope.savedModel.savedModelType);
        }
        RecipeComputablesService.getComputablesMap({type:"python"}).then((map) => {
            $scope.isARecipeOutput = map[savedModel.id].alreadyUsedAsOutputOf || null;
        });
    }).error(setErrorInScope.bind($scope));

    $scope.canDeleteSelectedModels = function() {
        if (!$scope.selection || !$scope.selection.selectedObjects) {return false}
        return (!$scope.selection.selectedObjects.every(function(o){return o.active}));
    };

    $scope.comparisonForbiddenReason = function() {
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length < 1) {
            return "At least one model version must be selected";
        }
        if ($scope.savedModel.savedModelType === 'LLM_GENERIC' 
            || $scope.savedModel.savedModelType === 'PYTHON_AGENT' 
            || $scope.savedModel.savedModelType === 'PLUGIN_AGENT'
            || $scope.savedModel.savedModelType === 'TOOLS_USING_AGENT'
            || $scope.savedModel.savedModelType === 'RETRIEVAL_AUGMENTED_LLM') {
            return 'Large Language Models cannot be compared';
        }
        if($scope.savedModel.miniTask.backendType === "DEEP_HUB") {
            return "Computer vision model comparison is not supported";
        }
        if($scope.savedModel.miniTask.taskType !== "PREDICTION") {
            return "Only prediction models can be compared";
        }
        if(!$scope.selection.selectedObjects.every(model => model.snippet.predictionType === $scope.selection.selectedObjects[0].snippet.predictionType)) {
            return "Model versions must have the same prediction type";
        }
        if($scope.selection.selectedObjects.some(model => model.snippet.partitionedModelEnabled)) {
            return "Partitioned models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => model.snippet.isEnsembled)) {
            return "Ensembled models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => (model.snippet.algorithm === "VIRTUAL_MLFLOW_PYFUNC") && (model.snippet.predictionType === undefined))) {
            return "Non tabular MLflow models cannot be compared";
        }
        if($scope.selection.selectedObjects.some(model => (model.snippet.algorithm === "VIRTUAL_PROXY_MODEL") && (model.snippet.predictionType === undefined))) {
            return "Non tabular External models cannot be compared";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.compareSelectedModels = function() {
        const nbModels = $scope.selection.selectedObjects.length;
        const smName = $scope.savedModel.name;

        CreateModalFromComponent(createOrAppendMELikeToModelComparisonModalDirective, {
            fullIds: $scope.selection.selectedObjects.map(me => me.snippet.fullModelId),
            modelTaskType: $scope.selection.selectedObjects[0].snippet.predictionType, // ModelTaskType values encompass PredictionType possible values. So this "cast" will work.
            allowImportOfRelatedEvaluations: true,
            suggestedMCName: `Compare ${nbModels} model versions from ${smName}`,
            projectKey: $stateParams.projectKey,
            trackFrom: 'saved-model-versions-list'
        });
    }

    $scope.duplicationForbiddenReason = function() {
        if(!$scope.selection || !$scope.selection.selectedObjects || $scope.selection.selectedObjects.length > 1) {
            return "Only one model version must be selected";
        }
        if (!$scope.canWriteProject()){
            return "You don't have write permissions for this project";
        }
    }

    $scope.duplicateSelectedModel = function() {
        CreateModalFromTemplate("/templates/savedmodels/agents/duplicate-agent-version-modal.html", $scope);
    }
});


app.controller("DuplicateAgentVersionController", function($scope, $state, $stateParams, SavedModelHelperService, ActiveProjectKey) {
    $scope.newVersion = {
        versionId : SavedModelHelperService.suggestedVersionId($scope.smStatus),
    }

    $scope.create = function() {
        const selectedModel = $scope.selection.selectedObjects[0];

        $scope.baseAPI.duplicateVersion(ActiveProjectKey.get(), $stateParams.smId, selectedModel.versionId, $scope.newVersion.versionId)
            .success(function(data) {
                $scope.dismiss();
                $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                    fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId
                });
            })
            .error(setErrorInScope.bind($scope));
    }
});

app.controller("PredictionSavedModelVersionsController", function($scope, DataikuAPI, $q, Fn, CreateModalFromTemplate, $state, $stateParams, TopNav,
                                                                  MLTasksNavService, PMLFilteringService, $controller, Dialogs, ActivityIndicator,
                                                                  ActiveProjectKey, PartitionedModelsService, CustomMetricIDService){
    angular.extend($scope, PartitionedModelsService);
    $scope.initialized = false;
    $scope.uiState.currentMetricIsCustom = false;
    function filterIntermediatePartitionedModels(status) {
        if (status.task && status.task.partitionedModel && status.task.partitionedModel.enabled) {
            /* Keeping all models exported from analysis (they don't have intermediate versions) */
            const analysisModels = status.versions
                .filter(model => model.smOrigin && model.smOrigin.origin === 'EXPORTED_FROM_ANALYSIS');

            /* Grouping models trained from recipe by their JobId */
            const recipeModelsByJobId = status.versions
                .filter(model => model.smOrigin && model.smOrigin.origin === 'TRAINED_FROM_RECIPE')
                .reduce((map, model) => {
                    map[model.smOrigin.jobId] = (map[model.smOrigin.jobId] || []).concat(model);
                    return map;
                }, {});

            /* Keeping most recent or active models in those groups */
            const recipeMostRecentModels = Object.entries(recipeModelsByJobId)
                .map((jobEntries) =>
                    jobEntries[1].reduce((mostRecentModel, currentModel) => {
                        if (!mostRecentModel || currentModel.active) {
                            return currentModel;
                        }

                        return (currentModel.snippet.trainDate > mostRecentModel.snippet.trainDate) ? currentModel : mostRecentModel;
                    }, null));

            status.versions = analysisModels.concat(recipeMostRecentModels);
        }
    }

    $scope.refreshStatus = function(){
        DataikuAPI.savedmodels.prediction.getStatus(ActiveProjectKey.get(), $stateParams.smId)
            .then(({data}) => {
                data.versions.map(function(v) { v.snippet.versionRank = +v.versionId || 0; });
                $scope.smStatus = data;
                $scope.setMainMetric();
                $scope.possibleMetrics = PMLFilteringService.getPossibleMetrics($scope.smStatus.task);
                $scope.allSnippets = data.versions.map(item => item.snippet);

                $scope.allMetrics = $scope.possibleMetrics;

                PMLFilteringService.getPossibleCustomMetrics($scope.allSnippets).map(item => {
                    $scope.allMetrics.push([item.id, item.name]);
                });

                $scope.allMetricsHooks = $scope.allMetrics.map((m) => m[0]);

                if ($scope.smStatus.task.modeling && !$scope.uiState.currentMetric) {
                    const modelingMetrics = $scope.smStatus.task.modeling.metrics;
                    $scope.uiState.currentMetric = modelingMetrics.evaluationMetric === "CUSTOM" ? CustomMetricIDService.getCustomMetricId(modelingMetrics.customEvaluationMetricName) : modelingMetrics.evaluationMetric;
                    $scope.uiState.currentMetricIsCustom = false;
                }
            })
            .catch(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.initialized = true;
            });
    }

    $scope.refreshStatus();

    $scope.setMainMetric = function() {
        if(!$scope.smStatus || !$scope.smStatus.versions || !$scope.uiState.currentMetric) { return; }

        $scope.uiState.currentMetricIsCustom = CustomMetricIDService.checkMetricIsCustom($scope.uiState.currentMetric);

        PMLFilteringService.setMainMetric($scope.smStatus.versions,
            ["snippet"],
            $scope.uiState.currentMetricIsCustom ? CustomMetricIDService.getCustomMetricName($scope.uiState.currentMetric) : $scope.uiState.currentMetric,
            $scope.smContext.savedModel.miniTask.modeling.metrics.customMetrics,
            $scope.uiState.currentMetricIsCustom
        );
    };
    $scope.$watch('uiState.currentMetric', $scope.setMainMetric);

    $scope.makeActive = function(data) {
        Dialogs.confirmPositive($scope, "Set model as active", "Do you want to set this model version as the active scoring version ?").then(function(){
            DataikuAPI.savedmodels.prediction.setActive(ActiveProjectKey.get(), $stateParams.smId, data.versionId)
                .success(function(data) {
                    $scope.refreshStatus();
                    if (data.schemaChanged) {
                        let warningMessage;
                        if ($scope.isTimeseriesPrediction()) {
                            warningMessage = "The newly selected model has a different preparation script schema, custom metric configuration or time series quantile configuration compared to the previous version.\n"
                        } else {
                            warningMessage = "The newly selected model has a different preparation script schema or custom metric configuration compared to the previous version.\n"
                        }
                        Dialogs.ackMarkdown($scope, "Schema changed", warningMessage +
                            "This change may affect the output schema of any downstream scoring and evaluation recipes."
                        );
                    }
                })
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteSelectedModels = function() {
        const deletableModels = $scope.selection.selectedObjects.filter(model => !model.active);
        const plural = deletableModels.length > 1 ? 's' : '';
        Dialogs.confirmAlert($scope, "Delete model" + plural, "Are you sure you want to delete " + deletableModels.length + " version" + plural + "? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.prediction.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, deletableModels.map(Fn.prop('versionId')))
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteModel = function(model) {
        Dialogs.confirmAlert($scope, "Delete model", "Are you sure you want to delete \"" + model.snippet.userMeta.name + "\"? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.prediction.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, [model.versionId])
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };
});


app.controller("ClusteringSavedModelVersionsController", function($scope, DataikuAPI, $q, Fn, CreateModalFromTemplate, $state, $stateParams, CMLFilteringService, $controller, Dialogs, ActivityIndicator, ActiveProjectKey){
    $scope.initialized = false;
    $scope.refreshStatus = function(){
        return DataikuAPI.savedmodels.clustering.getStatus(ActiveProjectKey.get(), $stateParams.smId)
            .then(({data}) => {
                $scope.smStatus = data;
                $scope.setMainMetric();
                $scope.possibleMetrics = CMLFilteringService.getPossibleMetrics($scope.smStatus.task);
                if (!$scope.uiState.currentMetric) {
                    $scope.uiState.currentMetric = "SILHOUETTE"; // Dirty tmp
                }
            })
            .catch(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.initialized = true;
            });
    }

    $scope.setMainMetric = function() {
        if(!$scope.smStatus || !$scope.smStatus.versions) { return; }
        CMLFilteringService.setMainMetric($scope.smStatus.versions,
            ["snippet"],
            $scope.uiState.currentMetric,
            $scope.smContext.savedModel.miniTask.modeling.metrics.customMetrics);
    };

    $scope.makeActive = function(data) {
        Dialogs.confirmPositive($scope, "Set model as active", "Do you want to set this model version as the active scoring version ?").then(function(){
            DataikuAPI.savedmodels.clustering.setActive(ActiveProjectKey.get(), $stateParams.smId, data.versionId)
                .success(function(data) {
                    $scope.refreshStatus();
                    if (data.schemaChanged) {
                        Dialogs.ack($scope, "Schema changed", "The preparation script schema of the selected version is different than " +
                            "the previously selected version, this may affect the ouput schema of downstream scoring recipes.");
                    }
                })
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteSelectedModels = function() {
        const deletableModels = $scope.selection.selectedObjects.filter(model => !model.active);
        const plural = deletableModels.length > 1 ? 's' : '';
        Dialogs.confirmAlert($scope, "Delete model" + plural, "Are you sure you want to delete " + deletableModels.length + " version" + plural + "? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.clustering.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, deletableModels.map(Fn.prop('versionId')))
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteModel = function(model) {
        Dialogs.confirmAlert($scope, "Delete model", "Are you sure you want to delete \"" + model.snippet.userMeta.name + "\"? This action is irreversible.").then(function(){
            DataikuAPI.savedmodels.clustering.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, [model.versionId])
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    // Watchers & init

    $scope.$watch('uiState.currentMetric', $scope.setMainMetric);
    $scope.refreshStatus();
});

app.controller("_CommonLLMModelVersionsController", function($scope, DataikuAPI, Fn, $stateParams, $controller, Dialogs, ActiveProjectKey, CreateModalFromComponent, makeActiveLLMGenericModalDirective) {
    $scope.refreshDeployments = function() {
        DataikuAPI.savedmodels.llmGeneric.deployments.list(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
            $scope.deployments = data;
        }).error(setErrorInScope.bind($scope));
    }
    $scope.refreshStatus = function(){
        DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data) {
            $scope.smStatus = data;
            $scope.allSnippets = data.versions.map(item => item.snippet);
            $scope.initialized = true;
        }).error(setErrorInScope.bind($scope));
        $scope.refreshDeployments();
    };
    $scope.refreshStatus();

    $scope.makeActive = function(data) {
        if (data.snippet.savedModelType === "LLM_GENERIC") {
            CreateModalFromComponent(makeActiveLLMGenericModalDirective, { deployments: $scope.deployments, savedModel: $scope.savedModel, newVersionId: data.versionId, llmType: data.snippet.llmSMInfo.llmType })
                .then($scope.refreshStatus);
        } else if (["PYTHON_AGENT", "TOOLS_USING_AGENT"].includes(data.snippet.savedModelType)) {
            Dialogs.confirmPositive($scope, "Set agent version as active", "Do you want to set this agent version as the active version when running this model?")
                .then(function() {
                    DataikuAPI.savedmodels.agents.setActive($scope.savedModel.projectKey, $scope.savedModel.id, data.versionId)
                        .success($scope.refreshStatus)
                        .error(setErrorInScope.bind($scope))
                }, function() {
                    // Dialog closed
                });
        }
    };

    $scope.deleteSelectedModels = function() {
        const deletableModels = $scope.selection.selectedObjects.filter(model => !model.active);
        const plural = deletableModels.length > 1 ? 's' : '';
        Dialogs.confirmAlert($scope, "Delete model" + plural, "Are you sure you want to delete " + deletableModels.length + " version" + plural + "? This action is irreversible. Any remote resource will be deleted as well.").then(function(){
            $scope.baseAPI.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, deletableModels.map(Fn.prop('versionId')))
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };

    $scope.deleteModel = function(model) {
        Dialogs.confirmAlert($scope, "Delete model", "Are you sure you want to delete \"" + model.snippet.userMeta.name + "\"? This action is irreversible. Any remote resource will be deleted as well.").then(function(){
            $scope.baseAPI.deleteVersions(ActiveProjectKey.get(), $stateParams.smId, [model.versionId])
                .success($scope.refreshStatus)
                .error(setErrorInScope.bind($scope));
        }, function() {
            // Dialog closed
        });
    };
});


app.controller("NewPluginAgentModalController", function($scope, DataikuAPI, $stateParams, $state) {
    $scope.newAgent = {

    }

    $scope.chooseType = function(id) {
        $scope.newAgent.type = id;
    }

    $scope.create = function() {
        DataikuAPI.savedmodels.agents.createPlugin($stateParams.projectKey, $scope.newAgent.type, $scope.newAgent.name)
            .success(function(data) {
                $scope.dismiss();
                $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                    fullModelId: "S-" + $stateParams.projectKey + "-" + $stateParams.smId + "-v1" // TODO
                });
            })
            .error(setErrorInScope.bind($scope))
    }

});


app.controller("NewPythonAgentVersionController", function($scope, DataikuAPI, $state, $stateParams, Dialogs, AgentCodeTemplates, AgentDefaultQuickTestQuery, SavedModelHelperService) {
    $scope.newVersion = {
        versionId: SavedModelHelperService.suggestedVersionId($scope.smStatus),
        templateId: 'simple-llm-mesh-proxy-streamed',
    };

    $scope.agentCodeTemplates = AgentCodeTemplates;

    $scope.create = function() {
        const smiv = {
            code: AgentCodeTemplates[$scope.newVersion.templateId].codeSample,
            pythonAgentSettings: {
                supportsImageInputs: false,
            }
        }

        if (AgentCodeTemplates[$scope.newVersion.templateId].quickTestQuery) {
            smiv.quickTestQuery = AgentCodeTemplates[$scope.newVersion.templateId].quickTestQuery;
        }
        else {
            smiv.quickTestQuery = AgentDefaultQuickTestQuery;
        }

        if (AgentCodeTemplates[$scope.newVersion.templateId].supportsImageInputs) {
            smiv.pythonAgentSettings.supportsImageInputs = true;
        }

        DataikuAPI.savedmodels.agents.createAgentVersion($stateParams.projectKey, $stateParams.smId, $scope.newVersion.versionId, smiv)
            .success(function(data) {
                $scope.dismiss();
                $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                    fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId
                });
            })
            .error(setErrorInScope.bind($scope))
    }
});

app.controller("PythonAgentModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, Dialogs,  CreateModalFromTemplate){
    $controller("_CommonLLMModelVersionsController", {$scope});

    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.agents;
    $scope.newAgentVersion = function() {
        CreateModalFromTemplate("/templates/savedmodels/agents/new-python-agent-version-modal.html", $scope);
    }

    const unwatchSmStatus = $scope.$watch('smStatus', function (newSmStatus) {
        if (newSmStatus) {
            if (newSmStatus.versions.length === 0) {
                $scope.newAgentVersion();
            }
            unwatchSmStatus();
        }
    });
});


app.controller("PluginAgentModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, Dialogs,  CreateModalFromTemplate, $state){
    $controller("_CommonLLMModelVersionsController", {$scope});

    $state.go('projects.project.savedmodels.savedmodel.agent.design', {
        smId: $scope.savedModel.id,
        fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`
    }, {
        location: 'replace'
    });

    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.agents;
    $scope.newAgentVersion = function() {
        Dialogs.ack($scope, "Not available for plugin agents", "Creating a new version is not available for plugin agents");
    };
});

app.controller("NewVisualAgentVersionController", function($scope, DataikuAPI, $state, $stateParams, Dialogs, SavedModelHelperService) {
    $scope.newVersion = {
        versionId : SavedModelHelperService.suggestedVersionId($scope.smStatus),
    }

    $scope.create = function() {
        const smiv = {
            toolsUsingAgentSettings: {
            }
        }

        DataikuAPI.savedmodels.agents.createAgentVersion($stateParams.projectKey, $stateParams.smId, $scope.newVersion.versionId, smiv)
            .success(function(data) {
                $scope.dismiss();
                $state.go("projects.project.savedmodels.savedmodel.agent.design", {
                    fullModelId: "S-" + data.projectKey + "-" + data.smId + "-" + data.smVersionId
                });
            })
            .error(setErrorInScope.bind($scope))
    }
});

app.controller("ToolsUsingAgentModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, Dialogs,  CreateModalFromTemplate){
    $controller("_CommonLLMModelVersionsController", {$scope});
    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.agents;
    $scope.newAgentVersion = function() {
        CreateModalFromTemplate("/templates/savedmodels/agents/new-visual-agent-version-modal.html", $scope);
    }

    if ($scope.savedModel.inlineVersions.length === 0) {
        $scope.newAgentVersion();
    }
});

app.controller("LLMGenericModelVersionsController", function($scope, $controller, DataikuAPI) {
    $controller("_CommonLLMModelVersionsController", {$scope});

    $scope.baseAPI = DataikuAPI.savedmodels.llmGeneric;
});

app.controller("RetrievalAugmentedLLMModelVersionsController", function($scope, DataikuAPI, $stateParams, $controller, $state){
    $controller("_CommonLLMModelVersionsController", {$scope});
    $scope.initialized = true;
    $scope.baseAPI = DataikuAPI.savedmodels.augmentedLLM;

    $state.go('projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design', {
        fullModelId: `S-${$stateParams.projectKey}-${$scope.savedModel.id}-${$scope.savedModel.activeVersion}`,
    }, {
        location: 'replace'
    });
});

app.controller("AgentSavedModelHistoryController", function($scope, $stateParams, DataikuAPI, ActiveProjectKey, TopNav) {
    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(savedModel) {
        $scope.savedModel = savedModel;
        TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, TopNav.TABS_SAVED_MODEL, "history");
        TopNav.setItem(
            TopNav.ITEM_SAVED_MODEL,
            $stateParams.smId,
            {
                name: savedModel.name,
                taskType: (savedModel.miniTask || {}).taskType,
                backendType: (savedModel.miniTask || {}).backendType,
                savedModelType: savedModel.savedModelType,
                predictionType: (savedModel.miniTask || {}).predictionType,
                proxyModelProtocol: (savedModel.proxyModelConfiguration || {}).protocol
            }
        );
    });
});

app.component("llmQuicktest", {
    bindings: {
        projectKey: "<",
        savedModelId: "<",
        savedModelType: "<",
        currentVersionId: "<",
        quickTestQuery: "=",
        saveModelFunction: "&",
        onTest: "&",
    },
    templateUrl : '/templates/savedmodels/llm-quicktest.html',
    controller: function($rootScope, $scope, $timeout, $q, DataikuAPI, ClipboardUtils, CodeMirrorSettingService, SavedModelHelperService, WT1) {
        const $ctrl = this;
        $ctrl.codeMirrorSettingService = CodeMirrorSettingService;
        $ctrl.quickTestResponse = null;
        $ctrl.quickTestEnabled = false;
        $ctrl.quickTestError = null;
        $ctrl.quickTestRequestCancellable = null;

        $scope.activeTab = 'response';
        $scope.uiState = {
            response: 'TEXT', // or FULL
            trace: 'LLM', // or FULL or GRAPHICAL
        };

        $ctrl.$onChanges = function(changes) {
            if (changes.savedModelType && changes.savedModelType.currentValue) {
                const currentModelType = changes.savedModelType.currentValue;

                if (currentModelType === "RETRIEVAL_AUGMENTED_LLM") {
                    $ctrl.api = DataikuAPI.savedmodels.retrievalAugmentedLLMs;
                } else {
                    $ctrl.api = DataikuAPI.savedmodels.agents;
                }
                $ctrl.quickTestEnabled = true;
            }
        };

        $ctrl.validateQuicktestQuery = function() {
            if (!$ctrl.quickTestQuery || typeof $ctrl.quickTestQuery !== 'object' || Array.isArray($ctrl.quickTestQuery)) {
                $ctrl.quickTestError = "Invalid test query, it should be a JSON object";
                return;
            }
            $ctrl.quickTestError = null;
        }

        $ctrl.quickTest = function() {
            if (!$ctrl.quickTestEnabled) {
                return;
            }
            $ctrl.quickTestEnabled = false;

            $ctrl.saveModelFunction().then(() => {
                $ctrl.onTest();
                $ctrl.quickTestRequestCancellable = $q.defer();

                $ctrl.api.test($ctrl.projectKey, $ctrl.savedModelId, $ctrl.currentVersionId, $ctrl.quickTestQuery, $ctrl.quickTestRequestCancellable.promise).then((response) => {
                    if ($ctrl.savedModelType == "RETRIEVAL_AUGMENTED_LLM") {
                        WT1.event(
                            'ra-llm-test-run', {
                                savedModelType: $ctrl.savedModelType,
                            });
                    }
                    else {
                        WT1.event(
                            'agent-test-run', {
                                savedModelType: $ctrl.savedModelType,
                            });
                    }
                    $ctrl.quickTestResponse = response.data;
                    $ctrl.graphicalTrace = SavedModelHelperService.generateAsciiTreeFromTrace(response.data.fullTrace);
                    $scope.activeTab = 'response';
                }).catch((error) => {
                    if (error && error.status === -1) {
                        // request was canceled
                    } else {
                        setErrorInScope.bind($scope)(error);
                    }
                    $ctrl.quickTestResponse = null;
                }).finally(() => {
                    $ctrl.quickTestEnabled = true;
                    $ctrl.quickTestRequestCancellable = null;
                });
            });
        };

        $ctrl.copyTraceToClipboard = function() {
            $scope.traceCopied = true;
            ClipboardUtils.copyToClipboard(JSON.stringify($scope.showFullTrace ? $ctrl.quickTestResponse.fullTrace : $ctrl.quickTestResponse.traceOfPython));
            $timeout(() => { $scope.traceCopied = false}, 5000);
        };

        $rootScope.$on('llmStopDevKernel', () => {
            if ($ctrl.quickTestRequestCancellable) {
                $ctrl.quickTestRequestCancellable.resolve("Request canceled by user");
            }
        });
    },
});

app.component('llmQuickchat', {
    bindings: {
        projectKey: '<',
        savedModel: '<',
        currentVersionId: '<',
        hasActiveLlm: '<',
        onSave: '&',
        onChat: '&',
        isDirty: '&'
    },
    templateUrl : '/templates/savedmodels/llm-quickchat.html',
    controller: function($scope, $q, DataikuAPI, PromptUtils, PromptChatService, SavedModelsService, Dialogs) {
        const $ctrl = this;
        $ctrl.activeChatTab = 'chat';
        let chatResetHandled = false; //chatResetHandled is used to avoid double reset of the chat in case of save of a new configuration

        $ctrl.$onChanges = function (changes) {
            if (changes.savedModel && !angular.equals(changes.savedModel.currentValue, changes.savedModel.previousValue)) {
                if (!chatResetHandled) {
                    $ctrl.resetChat();
                } else {
                    chatResetHandled = false;
                }
            }
        };

        $ctrl.resetChatWithDialog = () => {
            Dialogs.confirm($scope, 'Reset chat', 'Are you sure you want to reset this chat session? All current messages will be deleted.')
                .then(() => {
                    $ctrl.resetChat();
                })
                .catch(setErrorInScope.bind($scope));
        };

        $ctrl.resetChat = () => {
            $scope.$broadcast('prompt-chat--resetChat');
        };

        $ctrl.getLog = () => {
            return PromptChatService.getLog($ctrl.savedModel.id);
        };

        $ctrl.beforeSendChatMessage = (isEditOrRerun) => {
            if ($ctrl.isDirty()) {
                chatResetHandled = true;
                if(!isEditOrRerun){
                    $ctrl.resetChat();
                }
            }
        };

        $ctrl.sendChatMessage = (newMessage, abortController, chatMessages, callback) => {
            const lastUserMessage = newMessage;

            $ctrl.activeLLMIcon = {
                name: PromptUtils.getLLMIcon($ctrl.savedModel.savedModelType),
                style: PromptUtils.getLLMColorStyle('agent')
            };

            return $ctrl.save().then(() => {
                $ctrl.onChat();
                return DataikuAPI.savedmodels[SavedModelsService.getLlmEndpoint($ctrl.savedModel.savedModelType)].chat($ctrl.projectKey, $ctrl.savedModel.id, $ctrl.currentVersionId, { chatMessages, lastUserMessage }, callback, abortController).catch(setErrorInScope.bind($scope));
            }).catch(setErrorInScope.bind($scope));
        };

        $ctrl.onChatResponse = (lastMessageId) => {
            $ctrl.$lastMessageId = lastMessageId;
            $ctrl.chatLog = $ctrl.getLog();
        };

        $ctrl.onStopStreaming = (sessionId, activeSession, messages) => {
            const stoppedMessage = {
                id: generateRandomId(7),
                parentId: activeSession.runData?.parentId,
                version: activeSession.runData?.version,
                message: {
                    role: 'assistant',
                    content: activeSession.rawMessage
                },
                llmStructuredRef: {
                    id: buildSavedModelLlmId(),
                    type: $ctrl.savedModel.savedModelType,
                    savedModelSmartId: $ctrl.savedModel.id, 
                    savedModelVersionId: $ctrl.currentVersionId
                },
                completionSettings: {}
            };
            // if no raw message, throw an error
            if (activeSession.rawMessage === '') {
                stoppedMessage.error = true;
                stoppedMessage.llmError = 'Response generation was interrupted.';
            }

            const chatMessages = angular.copy(messages);
            chatMessages[stoppedMessage.id] = stoppedMessage;
            const response = {
                lastMessageId: stoppedMessage.id,
                chatMessages
            };
            return $q.when(response);
        };

        $ctrl.save = () => {
            return $ctrl.onSave();
        };

        function buildSavedModelLlmId() {
            let llmIdRoot = $ctrl.savedModel.savedModelType === 'RETRIEVAL_AUGMENTED_LLM' ? `retrieval-augmented-llm:${$ctrl.projectKey}.` : 'agent:';

            return llmIdRoot + $ctrl.savedModel.id + ':' + $ctrl.currentVersionId;
        }
    }
});

app.component('llmTestPanel', {
    bindings: {
        projectKey: '<',
        savedModel: '<',
        currentVersion: '<',
        hasActiveLlm: '<',
        onSave: "&",
        onChat: "&",
        isDirty: '&'
    },
    templateUrl : '/templates/savedmodels/llm-test-panel.html',
    controller: function($rootScope, $scope, DataikuAPI, SavedModelsService) {
        const $ctrl = this;
        $ctrl.displayMode = 'test';
        $ctrl.stopDevKernelEnabled = true;

        $ctrl.save = () => {
            return $ctrl.onSave();
        };

        $ctrl.stopDevKernel = function() {
            $rootScope.$emit('llmStopDevKernel', $ctrl.savedModel.id);
            DataikuAPI.savedmodels[SavedModelsService.getLlmEndpoint($ctrl.savedModel.savedModelType)].stopDevKernel($ctrl.projectKey, $ctrl.savedModel.id, $ctrl.currentVersion.versionId).then(() => {
                $ctrl.stopDevKernelEnabled = false;
            })
            .catch((error) => {
                if (error && error.status === 404) {
                    $ctrl.stopDevKernelEnabled = false;
                } else {
                    $ctrl.stopDevKernelEnabled = true;
                    setErrorInScope.bind($scope)(error);
                }
            });
        };
    }
});

/* ************************************ Settings *************************** */

app.controller("SavedModelSettingsController", function($scope, DataikuAPI, $q, CreateModalFromTemplate, $stateParams, TopNav, ComputableSchemaRecipeSave,
        SavedModelsService, WT1, ActiveProjectKey, Logger, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS){
    TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, TopNav.TABS_SAVED_MODEL, "settings");
    TopNav.setItem(TopNav.ITEM_SAVED_MODEL, $stateParams.smId);

    let savedSettings;
    DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(data) {
        $scope.savedModel = data;
        $scope.canHaveConditionalOutput = data.miniTask && data.miniTask.taskType === 'PREDICTION' && data.miniTask.predictionType === 'BINARY_CLASSIFICATION' && !SavedModelsService.isExternalMLflowModel(data);
        savedSettings = angular.copy(data);
        TopNav.setItem(
            TopNav.ITEM_SAVED_MODEL,
            $stateParams.smId,
            {
                name: data.name,
                taskType: (data.miniTask || {}).taskType,
                backendType: (data.miniTask || {}).backendType,
                predictionType: (data.miniTask || {}).predictionType,
                savedModelType: data.savedModelType,
                proxyModelProtocol: (data.proxyModelConfiguration || {}).protocol
            }
            );

        if (!$scope.canHaveConditionalOutput) return;
        $scope.targetRemapping = ['0', '1'];
        DataikuAPI.ml.prediction.getModelDetails(['S', data.projectKey, data.id, data.activeVersion].join('-')).success(function(data){
            $scope.targetRemapping = [];
            data.preprocessing.target_remapping.forEach(function(r){ $scope.targetRemapping[r.mappedValue] = r.sourceValue; });
        }).error(setErrorInScope.bind($scope));
    }).error(setErrorInScope.bind($scope));

    $scope.isProxyModel = function() {
        return SavedModelsService.isProxyModel($scope.savedModel);
    }

    $scope.isPartitionedModel = function() {
        return SavedModelsService.isPartitionedModel($scope.savedModel);
    }

    $scope.isMLflowModel = function() {
        return SavedModelsService.isMLflowModel($scope.savedModel);
    }

    $scope.$watch("savedModel", function() {
        if (!$scope.savedModel) {
            return;
        }

        if ($scope.savedModel.proxyModelConfiguration) {
            $scope.uiState.proxyModelConnection = $scope.savedModel.proxyModelConfiguration.connection === undefined ? null : $scope.savedModel.proxyModelConfiguration.connection;
            const fullProtocol = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === $scope.savedModel.proxyModelConfiguration.protocol);
            if (fullProtocol && fullProtocol.connectionType) {
                DataikuAPI.connections.getNames(fullProtocol.connectionType).success(function (data) {
                    const baseList = fullProtocol.canAuthenticateFromEnvironment?[{ name: null, label: "Environment" }]:[];
                    $scope.uiState.availableCompatibleConnections = baseList.concat(data.map(n => { return { name: n, label: n } }));
                }).error(setErrorInScope.bind($scope));
            }
        }
    });

    $scope.$watch("uiState.proxyModelConnection", function() {
        if (!$scope.savedModel || !$scope.savedModel.proxyModelConfiguration) {
            return;
        }
        $scope.savedModel.proxyModelConfiguration.connection = $scope.uiState.proxyModelConnection;
    });

    let oldNumberChecksOnAssertionsMetrics;
    let oldNumberChecks;
    $scope.save = function() {
        try {
            let numberChecksOnAssertionsMetrics = 0;
            let numberChecks = 0;
            if ($scope.savedModel && $scope.savedModel.metricsChecks && $scope.savedModel.metricsChecks.checks) {
                numberChecksOnAssertionsMetrics = $scope.savedModel.metricsChecks.checks.filter(m => m.metricId).filter(
                    m => m.metricId.startsWith("model_perf:ASSERTION_") ||
                        m.metricId === "model_perf:PASSING_ASSERTIONS_RATIO"
                ).length;
                numberChecks = $scope.savedModel.metricsChecks.checks.length || 0;
            }
            if (numberChecksOnAssertionsMetrics !== oldNumberChecksOnAssertionsMetrics ||
                numberChecks !== oldNumberChecks) {

                WT1.event("checks-save", {
                    numberChecksOnAssertionsMetrics: numberChecksOnAssertionsMetrics,
                    numberChecks: numberChecks
                });
            }
            oldNumberChecksOnAssertionsMetrics = numberChecksOnAssertionsMetrics;
            oldNumberChecks = numberChecks;
        }  catch (e) {
            Logger.error('Failed to report checks info', e);
        }
        DataikuAPI.savedmodels.save($scope.savedModel).success(function(data) {
            savedSettings = angular.copy($scope.savedModel);
            if ($scope.canHaveConditionalOutput && data && 'recipes' in data) {
                if (data.recipes.length) {
                    DataikuAPI.flow.recipes.getComputableSaveImpacts($scope.savedModel.projectKey, data.recipes, data.payloads).success(function(data){
                        if (!data.totalIncompatibilities) return;
                        CreateModalFromTemplate("/templates/recipes/fragments/recipe-incompatible-schema-multi.html", $scope, null,
                            function(newScope) {
                                ComputableSchemaRecipeSave.decorateChangedDatasets(data.computables, false);

                                newScope.schemaChanges = data;
                                newScope.customMessage = "The output datasets of scoring recipes using this model have incompatible schemas.";
                                newScope.noCancel = true;
                                function done(){ newScope.dismiss(); };
                                newScope.ignoreSchemaChangeSuggestion = done;
                                newScope.updateSchemaFromSuggestion = function() {
                                    $q.all(ComputableSchemaRecipeSave.getUpdatePromises(data.computables))
                                        .then(done).catch(setErrorInScope.bind($scope));
                                }
                            }
                        );
                    });
                } else if (data.hiddenRecipes) {    // TODO warn?
                }
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.dirtySettings = function() {
        return !angular.equals(savedSettings, $scope.savedModel);
    }
    checkChangesBeforeLeaving($scope, $scope.dirtySettings);
});


/* ************************************ Report *************************** */

app.controller("_SavedModelReportController", function($scope, TopNav, $stateParams, DataikuAPI, ActiveProjectKey, WebAppsService){
    if (!$scope.noSetLoc) {
        TopNav.setItem(TopNav.ITEM_SAVED_MODEL, $stateParams.smId);
    }

    const p = DataikuAPI.savedmodels.get(ActiveProjectKey.get(), $stateParams.smId).success(function(data) {
        $scope.savedModel = data;
        if ($scope.smContext) $scope.smContext.savedModel = data;
        if (!$scope.noSetLoc) {
            TopNav.setItem(TopNav.ITEM_SAVED_MODEL, $stateParams.smId, {
                name: data.name,
                taskType: (data.miniTask || {}).taskType,
                backendType: (data.miniTask || {}).backendType,
                predictionType: (data.miniTask || {}).predictionType,
                savedModelType: data.savedModelType,
                proxyModelProtocol: (data.proxyModelConfiguration || {}).protocol
            });
        }
    });
    if ($scope.noSpinner) {
        p.noSpinner();
    }

    $scope.fillVersionSelectorStuff = function(statusData, needsMetric){
        if (!$scope.versionsContext.activeMetric && needsMetric) {
            $scope.versionsContext.activeMetric = statusData.task.modeling.metrics.evaluationMetric;
        }
        $scope.versionsContext.versions = statusData.versions.filter(function(m){
            return m.snippet.trainInfo.state == "DONE" && m.snippet.fullModelId != $stateParams.fullModelId;
        });
        $scope.versionsContext.currentVersion = statusData.versions.filter(function(m){
            return m.snippet.fullModelId === $stateParams.fullModelId;
        })[0] || {}; // (partitioned models) ensure watch on versionsContext.currentVersion is fired (see ch45900)
        $scope.versionsContext.versions.sort(function(a, b) {
            var stardiff = (0+b.snippet.userMeta.starred) - (0+a.snippet.userMeta.starred)
            if (stardiff !=0) return stardiff;
            return b.snippet.sessionDate - a.snippet.sessionDate;
        });

        statusData.versions.forEach(function(version) {
            if (version.active) {
                $scope.versionsContext.activeVersion = version;
            }
        });

        if ($scope.versionsContext.currentVersion.snippet && $scope.savedModel) {
            let contentType = $scope.savedModel.contentType;
            if (!contentType) {
                if ($scope.savedModel.miniTask) {
                    contentType = `${$scope.savedModel.miniTask.taskType}/${$scope.savedModel.miniTask.backendType}`.toLowerCase();
                } else if ($scope.savedModel.savedModelType === "PYTHON_AGENT") {
                    contentType = "python";
                } else if ($scope.savedModel.savedModelType === "PLUGIN_AGENT") {
                    contentType = "plugin";
                } else if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
                    contentType = "tools-using";

                } else if ($scope.savedModel.savedModelType === "LLM_GENERIC") {
                    contentType = "fine-tuned";
                } else if ($scope.savedModel.savedModelType === "RETRIEVAL_AUGMENTED_LLM") {
                    contentType = "retrieval-augmented-llm";
                }
            }
            if (!contentType.endsWith("/")) {
                contentType += '/';
            }
            if ($scope.versionsContext.currentVersion.snippet.algorithm) {
                contentType += $scope.versionsContext.currentVersion.snippet.algorithm.toLowerCase();
            } else if ($scope.savedModel.miniTask && $scope.savedModel.miniTask.backendType === "DEEP_HUB") {
                contentType += $scope.savedModel.miniTask.predictionType.toLowerCase();
            }
            if ($scope.savedModel.miniTask) {
                $scope.modelSkins = WebAppsService.getSkins(
                    'SAVED_MODEL', $scope.versionsContext.currentVersion.versionId,
                    { predictionType: $scope.savedModel.miniTask.predictionType, backendType: $scope.savedModel.miniTask.backendType, contentType },
                    $scope.staticModelSkins
                );
            }
        }
    }
})

app.controller("PredictionSavedModelReportController", function($scope, DataikuAPI, $stateParams, TopNav, $controller, PMLFilteringService, ActiveProjectKey, GoToStateNameSuffixIfBase, MLModelsUIRouterStates, $state){
    $scope.noMlReportTourHere = true; // the tabs needed for the tour are not present

    $controller("_PredictionModelReportController",{$scope:$scope});
    $controller("_SavedModelReportController", {$scope:$scope});
    $controller("_SavedModelGovernanceStatusController", {$scope:$scope});

    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "PREDICTION-SAVED_MODEL-VERSION", "report");
    }

    // Fill the version selector
    const getStatusP = DataikuAPI.savedmodels.prediction.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.getGovernanceStatus($stateParams.fullModelId, data.task.partitionedModel);
        $scope.fillVersionSelectorStuff(data);
        $scope.versionsContext.versions.forEach(function(m){
            m.snippet.mainMetric = PMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
            m.snippet.mainMetricStd = PMLFilteringService.getMetricStdFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
        });
    });
    if ($scope.noSpinner) {
        getStatusP.noSpinner();
    }

    $scope.getPredictionDesignTabPrefix = () => MLModelsUIRouterStates.getPredictionDesignTabPrefix($scope);

    $scope.isClassicalPrediction = function() {
        if (!$scope.modelData) return;
        return ["BINARY_CLASSIFICATION", "REGRESSION", "MULTICLASS"].includes($scope.modelData.coreParams.prediction_type);
    };

    $scope.isTimeseriesPrediction = function() {
        if (!$scope.modelData) return;
        return $scope.modelData.coreParams.prediction_type === 'TIMESERIES_FORECAST';
    };

    $scope.isMLBackendType = function(mlBackendType) {
        if ($scope.modelData && $scope.modelData.coreParams) {
            return $scope.modelData.coreParams.backendType === mlBackendType;
        }
    };

    $scope.isDeepHubPrediction = function() {
        return $scope.isMLBackendType("DEEP_HUB");
    };

    $scope.isCausalPrediction = function() {
        if (!$scope.modelData) return;
        return ["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes($scope.modelData.coreParams.prediction_type);
    };

    const baseStateName = "projects.project.savedmodels.savedmodel.prediction.report";
    if ($state.current.name === baseStateName) {
        const deregister = $scope.$watch("modelData", (nv, ov) => {
            if (!nv) {
                return;
            }
            // redirect to summary page when arriving on "report"
            const route = MLModelsUIRouterStates.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false);
            $state.go(`.${route}`, null, {location: 'replace'});
            deregister();
        });
    }

    $scope.$on('$stateChangeSuccess', function (event, toState, toParams, fromState, fromParams) {
        if (!$scope.modelData) {
            return;
        }
        const suffix = MLModelsUIRouterStates.getPredictionReportSummaryTab($scope.isDeepHubPrediction(), false);
        GoToStateNameSuffixIfBase($state, toState, toParams, fromState, fromParams, baseStateName, suffix, event);
    });
});


app.controller("ClusteringSavedModelReportController", function($scope, $controller, $state, $stateParams, $q, DataikuAPI, CreateModalFromTemplate, TopNav, CMLFilteringService, ActiveProjectKey){
    $controller("_ClusteringModelReportController",{$scope:$scope});
    $controller("_SavedModelReportController", {$scope:$scope});
    $scope.clusteringResultsInitDone = false;

    const smId = $stateParams.smId // by the time getModelStatus is called in async logic, smId is no longer present in $stateParams, so needs to be cached
    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "CLUSTERING-SAVED_MODEL-VERSION", "report");
    }

    // Fills the version selector
    const getModelStatus = function() {
        return DataikuAPI.savedmodels.clustering.getStatus(ActiveProjectKey.get(), smId)
            .then(({data}) => {
                $scope.fillVersionSelectorStuff(data);
                $scope.versionsContext.versions.forEach(function(m){
                    m.snippet.mainMetric = CMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                });
        });
    }

    $scope.deferredAfterInitCModelReportDataFetch
        .then(() => {
            if (!$scope.noSetLoc) {
                TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.ITEM_SAVED_MODEL, "CLUSTERING-SAVED_MODEL-VERSION", "report");
            }
        })
        .then(getModelStatus)
        .then(() => {
            $scope.clusteringResultsInitDone = true;
        })
        .catch(setErrorInScope.bind($scope));

});



app.controller("LLMGenericSavedModelReportController", function($scope, $controller, $stateParams, DataikuAPI, ActiveProjectKey, TopNav, FinetuningUtilsService) {
    $controller("_SavedModelReportController", {$scope});
    $controller("_MLReportSummaryController", {$scope});

    // TODO @llm: adapt for dashboard (cf. PML/CML controller structure)
    DataikuAPI.ml.llm.getModelDetails($stateParams.fullModelId).success(function(data) {
        $scope.modelData = data;
        if (data.llmSMInfo.connection) {
            DataikuAPI.admin.connections.get(data.llmSMInfo.connection).success(function (data) {
                $scope.connection = data;
            });
        }
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.fillVersionSelectorStuff(data);
    });

    const smId = $stateParams.smId // by the time getModelStatus is called in async logic, smId is no longer present in $stateParams, so needs to be cached
    if (!$scope.noSetLoc) {
        TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "LLM-SAVED_MODEL-VERSION", "report");
    }

    $scope.supportsModelDeployment = FinetuningUtilsService.supportsModelDeployment;
});

app.controller("LLMGenericReportSummaryController", function($scope, $controller, $stateParams, DataikuAPI, Debounce, ActivityIndicator) {
    $controller("_MLReportSummaryController", {$scope});

    // Maybe can be factorized: see _ClusteringModelReportController / _PredictionModelReportController
    function saveMeta() {
        if ($scope.readOnly) return;
        DataikuAPI.ml.saveModelUserMeta($stateParams.fullModelId || $scope.fullModelId, $scope.modelData.userMeta).success(function(){
            ActivityIndicator.success("Saved")
        }).error(setErrorInScope.bind($scope));
    }

    const debouncedSaveMeta = Debounce().withDelay(400,1000).wrap(saveMeta);

    $scope.$watch("modelData.userMeta", function(nv, ov) {
        // Equality check here is needed here as something (not sure what) is updating the modelData scope object with an identical copy
        if (!nv || !ov || _.isEqual(nv, ov)) return;
        debouncedSaveMeta();
    }, true);
});

app.controller("LLMGenericReportTrainingInformationController", function($scope, $controller) {
    $controller("_MLReportSummaryController", {$scope});

    $scope.buildChartOptions = function(llmStepwiseTrainingMetrics) {
        const trainingLossData = [];
        const validationLossData = [];
        const fullValidationLossData = [];
        Object.entries(llmStepwiseTrainingMetrics.metrics).forEach(([step, metric]) => {
            if (metric.trainingMetric) {
                trainingLossData.push([step, metric.trainingMetric.loss])
            }
            if (metric.validationMetric) {
                validationLossData.push([step, metric.validationMetric.loss])
            }
            if (metric.fullValidationMetric) {
                fullValidationLossData.push([step, metric.fullValidationMetric.loss])
            }
        });
        const series = [];
        if (trainingLossData.length > 0) {
            series.push({
                data: trainingLossData, type: 'line', name: 'Training loss'
            })
        }
        if (validationLossData.length > 0) {
            series.push({
                data: validationLossData, type: 'line', name: 'Validation loss'
            })
        }
        if (fullValidationLossData.length > 0) {
            series.push({
                data: fullValidationLossData, type: 'line', name: 'Full validation loss'
            })
        }
        $scope.chartOptions = {
            xAxis: { name: "Step"},
            yAxis: { name: "Loss", scale: true },
            series,
            tooltip: {
                trigger: 'axis'
            },
            legend: {
                data: ['Training loss', 'Validation loss', 'Full validation loss']
            },
        }
    }
});

// use createOrAttachDeploymentModalDirective when calling CreateModalFromComponent
app.component("createOrAttachDeploymentModal", {
    bindings: {
        attachOnly: '<',
        llmType: '<',
        returnData: '=',
        modalControl: '<',
    },
    templateUrl: '/templates/ml/llm-generic/create-or-attach-deployment-modal.html',
    controller: function($scope, DataikuAPI, $stateParams, ActiveProjectKey) {
        const $ctrl = this;

        $ctrl.createOrAttach = function() {
            const apiCreateFn = $ctrl.attachOnly ? DataikuAPI.savedmodels.llmGeneric.deployments.attach : DataikuAPI.savedmodels.llmGeneric.deployments.create;
            apiCreateFn(ActiveProjectKey.get(), $stateParams.fullModelId, $ctrl.newDeploymentId)
                .success(function(data) {
                    $ctrl.returnData.deploymentWithStatus = data;
                    $ctrl.modalControl.resolve();
                })
                .error(setErrorInScope.bind($scope));
        }
    }
});

// use makeActiveLLMGenericModalDirective when calling CreateModalFromComponent
app.component("makeActiveLLMGenericModal", {
    bindings: {
        deployments: '<',
        savedModel: '<',
        newVersionId: '<',
        llmType: '<',
        modalControl: '<',
    },
    templateUrl: '/templates/ml/llm-generic/make-active-modal.html',
    controller: function($scope, FinetuningUtilsService, DataikuAPI) {
        const $ctrl = this;

        $ctrl.makeActive = function() {
            DataikuAPI.savedmodels.llmGeneric.setActive($ctrl.savedModel.projectKey, $ctrl.savedModel.id, $ctrl.newVersionId, $ctrl.deployNewActiveModel, $ctrl.deleteInactiveDeployments)
                .then($ctrl.modalControl.resolve)
                .catch(setErrorInScope.bind($scope).bind($scope))
        }

        $ctrl.showDeployNewModelCheckbox = function() {
            if (!['SAVED_MODEL_FINETUNED_AZURE_OPENAI', 'SAVED_MODEL_FINETUNED_BEDROCK'].includes($ctrl.llmType)) {
                return false; // Deployments are only relevant for Azure Open AI / Bedrock models
            }
            // Ignore the checkbox if a model is already deployed
            return !$ctrl.deployments.find(d => d.versionId === $ctrl.newVersionId)
        }

        $ctrl.initDeployNewModel = function() {
            $ctrl.deployNewActiveModel = $ctrl.showDeployNewModelCheckbox();
        }

        $ctrl.showDeleteInactiveModelsCheckbox = function() {
            if ($ctrl.deployments.length === 0) return false; // No deployments to delete
            if ($ctrl.deployments.length === 1) {
                // Ignore the checkbox as the deployed model is attached to the new active model version
                return !$ctrl.deployments.find(d => d.versionId === $ctrl.newVersionId);
            }
            return true;
        }

        $ctrl.getDeletedDeploymentsCountMessage = function() {
            return FinetuningUtilsService.getDeletedDeploymentsCountMessage($ctrl.deployments, $ctrl.savedModel, $ctrl.newVersionId, true, $ctrl.deleteInactiveDeployments);
        }
    }
});

app.controller("LLMSavedModelDeploymentController", function($scope, $controller, $stateParams, ActiveProjectKey, DataikuAPI, CreateModalFromTemplate, CreateModalFromComponent, createOrAttachDeploymentModalDirective, Dialogs) {
    $controller("_MLReportSummaryController", {$scope});

    $scope.loadDeployment = function() {
        DataikuAPI.savedmodels.llmGeneric.deployments.get(ActiveProjectKey.get(), $stateParams.fullModelId)
            .success(function(data){
                $scope.deploymentWithStatus = data;
            })
            .error((data, status, headers) => {
                $scope.disabledMessage = $scope.modelData.llmSMInfo.llmType === 'SAVED_MODEL_FINETUNED_AZURE_OPENAI'
                    ? 'Make sure you have a valid Azure ML connection in your Azure Open AI connection'
                    : 'Make sure your credentials in your Bedrock connection are properly set up.';
                setErrorInScope.call(this, data, status, headers)
            });
    }

    $scope.getConnectionName = function () {
        return ($scope.modelData.llmSMInfo.llmType === 'SAVED_MODEL_FINETUNED_AZURE_OPENAI' ? 'Azure' : 'Bedrock');
    }

    $scope.detachDeployment = function() {
        if (!$scope.deploymentWithStatus) return;
        Dialogs.confirmAlert(
            $scope,
            'Confirm model detachment',
            'Are you sure you want to detach your deployment ' + $scope.deploymentWithStatus.deploymentId + '?',
            "Your model will remain active on " + $scope.getConnectionName() + " and will continue to incur costs.",
            "WARNING"
        ).then(function() {
            DataikuAPI.savedmodels.llmGeneric.deployments.detach(ActiveProjectKey.get(), $stateParams.fullModelId)
                .success(function() {
                    $scope.deploymentWithStatus = undefined;
                    $scope.modelData.deployment = undefined;
                })
                .error(setErrorInScope.bind($scope));
        });

    }

    $scope.newDeploymentModal = function(attachOnly) {
        const dataHolder = {}
        CreateModalFromComponent(createOrAttachDeploymentModalDirective, { attachOnly: attachOnly, returnData: dataHolder, llmType: $scope.modelData.llmSMInfo.llmType }).then(() => {
            $scope.deploymentWithStatus = dataHolder.deploymentWithStatus;
            $scope.modelData.deployment = dataHolder.deploymentWithStatus;
        });
    }

    $scope.deleteDeployment = function() {
        if (!$scope.deploymentWithStatus) return;
        // $scope.loading = true;
        Dialogs.confirmAlert(
            $scope,
            'Confirm model deletion',
            'Are you sure you want to delete your deployment ' + $scope.deploymentWithStatus.deploymentId + '?',
            "This action is irreversible.",
            "WARNING"
        ).then(function() {
            DataikuAPI.savedmodels.llmGeneric.deployments.delete(ActiveProjectKey.get(), $stateParams.fullModelId)
                .success(function() {
                    $scope.deploymentWithStatus = undefined;
                    $scope.modelData.deployment = undefined;
                })
                .error(setErrorInScope.bind($scope));
        });
    }

});

/* ***************************** Scoring recipe creation ************************** */
app.service("RecipeModalCheckSavedModelBackendService", function() {
    return {
        getCustomInputRequirements: function(smId, inputDataset, needsInputDataFolder, managedFolderId) {
            const ret = [];

            // Recipe must always have an input dataset
            if (!inputDataset) {
                ret.push("an input dataset");
            }

            // Modal does not have a model set
            if (!smId) {
                ret.push("a model");
            }

            // Modal has a model selected that lacks a managedFolder input.
            if (needsInputDataFolder && !managedFolderId) {
                    ret.push("a managed folder");
            }

            if (ret.length === 0) {
                return null;
            }
            return ret.join(", ");
        }
    };
});

app.controller("NewPredictionScoringRecipeModalController", function($scope, $stateParams, $controller, DataikuAPI, Fn,
                                                                     SavedModelsService, ActiveProjectKey,
                                                                     RecipeModalCheckSavedModelBackendService) {
    $scope.recipe = { // needed for updateRecipeDesc()
        projectKey : ActiveProjectKey.get(),
        type: "prediction_scoring",
        inputs : {},
        outputs : {},
        params: {}
    };
    $scope.recipeType = "prediction_scoring";
    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

    $scope.scoringRecipe = {};

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_scored");
        }
    };

    $scope.managedFolder = {id: null};
    let selectedSavedModel = null;

    $scope.getCustomInputRequirements = function() {
        return RecipeModalCheckSavedModelBackendService.getCustomInputRequirements($scope.smId, $scope.io.inputDataset,
                                                                                   $scope.isManagedFolderRequired(), $scope.managedFolder.id);
    }

    $scope.isManagedFolderRequired = function(){
        const dataRole = $scope.recipeDesc.inputRoles.find(role => role.name == 'data');
        return dataRole && dataRole.required && $scope.isInputRoleAvailableForPayload(dataRole);
    }

    $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
        switch(role.name){
            case "data":
                // Data role is only available for deephub & classical task with image preprocessing on scoring recipes
                return selectedSavedModel && selectedSavedModel.needsInputDataFolder;
            default:
                throw new Error(`Rules for availability of input role "${role.name}" not implemented`);
        }
    };

    $scope.doCreateRecipe = function() {
        var createOutput = $scope.io.newOutputTypeRadio == 'create';

        var finalRecipe = angular.copy($scope.scoringRecipe);
        finalRecipe.inputDatasetSmartName = $scope.io.inputDataset;
        finalRecipe.managedFolderSmartId = $scope.managedFolder.id;
        finalRecipe.savedModelSmartName = $scope.smId;
        finalRecipe.createOutput = createOutput;
        finalRecipe.outputDatasetSmartName = createOutput ? $scope.newOutputDataset.name : $scope.io.existingOutputDataset;
        finalRecipe.outputDatasetCreationSettings = $scope.getDatasetCreationSettings();
        finalRecipe.zone = $scope.zone;

        return DataikuAPI.savedmodels.prediction.deployScoring(ActiveProjectKey.get(), finalRecipe);
    };

    $scope.subFormIsValid = function() {
        return !$scope.getCustomInputRequirements();
    };

    $scope.$watch("smId", function(nv) {
        if (!nv) return;
        selectedSavedModel = $scope.savedModels.find(sm => sm.id === $scope.smId);

        // Remove image folder from inputs of recipe if currently selected SM do not need the managedFolder input
        if (!$scope.isInputRoleAvailableForPayload({name: 'data'})) {
            $scope.managedFolder.id = null;
        }
    });
});


app.controller("NewClusteringScoringRecipeModalController", function($scope, Fn, $stateParams, $controller, DataikuAPI, SavedModelsService, ActiveProjectKey) {
    $scope.recipeType = "prediction_scoring";
    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

    $scope.scoringRecipe = {};

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_scored");
        }
    };

    $scope.doCreateRecipe = function() {
        var createOutput = $scope.io.newOutputTypeRadio == 'create';
        return DataikuAPI.savedmodels.clustering.deployScoring(
            ActiveProjectKey.get(),
            $scope.smId,
            $scope.io.inputDataset,
            createOutput,
            createOutput ? $scope.newOutputDataset.name : $scope.io.existingOutputDataset,
            $scope.getDatasetCreationSettings());
    };

    $scope.subFormIsValid = function() { return !!$scope.smId; };
});

/* ***************************** Evaluation recipe creation ************************** */

app.controller('NewEvaluationRecipeModalController', function($scope, $controller, $stateParams, $state, DataikuAPI,
                                                              DatasetUtils, RecipeComputablesService,
                                                              PartitionDeps, ActiveProjectKey, $rootScope,
                                                              RecipeModalCheckSavedModelBackendService) {
    $scope.recipeType = "evaluation";
    $scope.forceMainLabel = true;
    $controller("_RecipeCreationControllerBase", {$scope:$scope});

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_evaluated");
        }
    };

    addDatasetUniquenessCheck($scope, DataikuAPI, ActiveProjectKey.get());
    fetchManagedDatasetConnections($scope, DataikuAPI);

    let selectedSavedModel = null;

    $scope.isManagedFolderRequired = function(){
        const dataRole = $scope.recipeDesc.inputRoles.find(role => role.name == 'data');
        return dataRole && dataRole.required && $scope.isInputRoleAvailableForPayload(dataRole);
    }

    $scope.getCustomInputRequirements = function() {
        return RecipeModalCheckSavedModelBackendService.getCustomInputRequirements($scope.recipeParams.smId, $scope.recipeParams.inputDs,
            $scope.isManagedFolderRequired(), $scope.recipeParams.managedFolderSmartId);
    }

    $scope.recipe = {
        projectKey : ActiveProjectKey.get(),
        type: "evaluation",
        inputs : {},
        outputs : {},
        params: {}
    };
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.recipeParams.inputDs = preselectedInputDataset;
    });

    $scope.isInputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!
        if (role.name == 'data'){
            return selectedSavedModel && selectedSavedModel.needsInputDataFolder;
        }
        return true;
    };

    $scope.isOutputRoleAvailableForPayload = function(role) { // /!\ keep in sync with JAVA counterpart: PredictionRecipesMeta.java!

        if (!selectedSavedModel || !selectedSavedModel.miniTask) return false;
        const minitask = selectedSavedModel.miniTask;

        switch(role.name){
            case 'evaluationStore':
                // Only not partitionned classical models (binary classification, multiclass, regression) can have an evaluation recipe with a MES as output
                // TODO @causal @deephub handle MES as evaluation output
                return ["BINARY_CLASSIFICATION", "MULTICLASS", "REGRESSION", "TIMESERIES_FORECAST"].includes(minitask.predictionType)
                    && (!minitask.partitionedModel || !minitask.partitionedModel.enabled);
            default:
                throw new Error(`Rules for availability of output role "${role.name}" not implemented`);
        }
    };

    $scope.$watch("recipeParams.inputDs", function(nv, ov) {
        if (nv) {
            $scope.recipe.name = "evaluate_" + nv;
        }
        if ($scope.recipeParams.inputDs) {
            $scope.recipe.inputs.main = {items:[{ref:$scope.recipeParams.inputDs}]}; // for the managed dataset creation options
        } else {
            $scope.recipe.inputs.main = {items:[]}; // for the managed dataset creation options
        }
    }, true);

    $scope.$watch("recipeParams.smId", function(nv) {
        if (nv) {
            selectedSavedModel = $scope.savedModels.find(sm => sm.id === nv);

            $scope.recipe.inputs.model = {items:[{ref:nv}]}; // for the managed dataset creation options

            // Remove MES from outputs of recipe if currently selected SM does not support MES as eval output
            if (!$scope.isOutputRoleAvailableForPayload({name: "evaluationStore"})) {
                delete $scope.recipe.outputs["evaluationStore"];
            }

            // Remove image folder from inputs of recipe if currently selected SM do not need the managedFolder input
            if (!$scope.isInputRoleAvailableForPayload({name: 'data'})) {
                $scope.recipeParams.managedFolderSmartId = null;
            }
        } else {
            $scope.recipe.inputs.model = {items:[]}; // for the managed dataset creation options
        }
    }, true);

    DatasetUtils.listDatasetsUsabilityInAndOut(ActiveProjectKey.get(), "evaluation").then(function(data){
        $scope.availableInputDatasets = data[0];
    });

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
        $scope.setComputablesMap(map);
    });

    $scope.hasMain = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.main && outputs.main.items && outputs.main.items.length > 0 && outputs.main.items[0].ref
    }
    $scope.hasMetrics = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.metrics && outputs.metrics.items && outputs.metrics.items.length > 0 && outputs.metrics.items[0].ref
    }
    $scope.hasEvaluationStore = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.evaluationStore && outputs.evaluationStore.items && outputs.evaluationStore.items.length > 0 && outputs.evaluationStore.items[0].ref
    }

    $scope.canCreate = function(){
        return $scope.recipe.name
            && $scope.recipe.name.length > 0
            && $scope.recipe.outputs
            && !$scope.shouldDisplayOutputExplanation()
            && !($scope.newRecipeForm.$invalid)
            && !$scope.getCustomInputRequirements();
    }

    $scope.shouldDisplayOutputExplanation = function () {
        return !$scope.hasMain() && !$scope.hasMetrics() && !$scope.hasEvaluationStore();
    };

    $scope.generateOutputExplanation = function () {
        const requiredOutputRoles = [];
        $scope.recipeDesc.outputRoles.forEach((role, outputRoleidx) => {
            if (role.availabilityDependsOnPayload && !$scope.isOutputRoleAvailableForPayload(role)) return;
            requiredOutputRoles.push(role.name === "main" ? "main output" : '"' + (role.label || role.name) + '"');
        });
        const message = "This recipe requires at least one output in: "
            + requiredOutputRoles.slice(0, -1).join(', ')
            + (requiredOutputRoles.length === 2 ? ' or ' : ', or ')
            + requiredOutputRoles.slice(-1) + ".";
        return message;
    };

    $scope.createRecipe = function() {
        $scope.creatingRecipe = true;
        var finalRecipe = {};

        finalRecipe.inputDatasetSmartName = $scope.recipeParams.inputDs;
        finalRecipe.savedModelSmartName = $scope.recipeParams.smId;
        finalRecipe.managedFolderSmartId = $scope.recipeParams.managedFolderSmartId;
        finalRecipe.scoredDatasetSmartName = $scope.recipe.outputs.main && $scope.recipe.outputs.main.items && $scope.recipe.outputs.main.items.length>0 ? $scope.recipe.outputs.main.items[0].ref : null;
        finalRecipe.metricsDatasetSmartName = $scope.recipe.outputs.metrics && $scope.recipe.outputs.metrics.items && $scope.recipe.outputs.metrics.items.length>0 ? $scope.recipe.outputs.metrics.items[0].ref : null;
        finalRecipe.evaluationStoreSmartName = $scope.recipe.outputs.evaluationStore && $scope.recipe.outputs.evaluationStore.items && $scope.recipe.outputs.evaluationStore.items.length>0 ? $scope.recipe.outputs.evaluationStore.items[0].ref : null;
        finalRecipe.zone = $scope.zone;

        DataikuAPI.savedmodels.prediction.deployEvaluation(ActiveProjectKey.get(), finalRecipe)
            .success(function(data) {
                $scope.creatingRecipe = false;
                $scope.dismiss();
                $scope.$state.go('projects.project.recipes.recipe', {
                    recipeName: data.id
                });
            }).error(function(a, b, c) {
                $scope.creatingRecipe = false;
                setErrorInScope.bind($scope)(a,b,c);
            });

    };
});

app.controller('NewStandaloneEvaluationRecipeModalController', function($scope, $controller, $stateParams, $state, DataikuAPI,
DatasetUtils, RecipeComputablesService, PartitionDeps, ActiveProjectKey){
    $scope.recipeType = "standalone_evaluation";
    $controller("_RecipeCreationControllerBase", {$scope:$scope});

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_scored");
        }
    };

    addDatasetUniquenessCheck($scope, DataikuAPI, ActiveProjectKey.get());

    $scope.recipeParams = {
        inputDs: "",
        referenceDs : ""
    };

    $scope.recipe = {
        projectKey : ActiveProjectKey.get(),
        type: "standalone_evaluation",
        inputs : {},
        outputs : {},
        params: {}
    };
    $scope.$on("preselectInputDataset", function(scope, preselectedInputDataset) {
        $scope.recipeParams.inputDs = preselectedInputDataset;
    });
    $scope.$on("preselectReferenceDataset", function(scope, preselectReferenceDataset) {
        $scope.recipeParams.referenceDs = preselectReferenceDataset;
    });

    $scope.$watch("recipeParams.inputDs", function(nv, ov) {
        if (nv) {
            $scope.recipe.name = "standalone_evaluate_" + nv;
        }
        if ($scope.recipeParams.inputDs) {
            $scope.recipe.inputs.main = {items:[{ref:$scope.recipeParams.inputDs}]}; // for the managed dataset creation options
        } else {
            $scope.recipe.inputs.main = {items:[]}; // for the managed dataset creation options
        }
    }, true);

    DatasetUtils.listDatasetsUsabilityInAndOut(ActiveProjectKey.get(), "standalone_evaluation").then(function(data){
        $scope.availableInputDatasets = data[0];
    });

    RecipeComputablesService.getComputablesMap($scope.recipe, $scope).then(function(map){
        $scope.setComputablesMap(map);
    });

    $scope.hasMain = function() {
        const outputs = $scope.recipe.outputs;
        return outputs.main && outputs.main.items && outputs.main.items.length > 0 && outputs.main.items[0].ref
    }

    $scope.canCreate = function(){
        return $scope.recipe.name
            && $scope.recipe.name.length > 0
            && $scope.recipe.outputs
            && !$scope.shouldDisplayOutputExplanation()
            && !($scope.newRecipeForm.$invalid)
    }

    $scope.shouldDisplayOutputExplanation = function () { return !$scope.hasMain(); };

    $scope.createRecipe = function() {
        $scope.creatingRecipe = true;
        var finalRecipe = {};

        finalRecipe.inputDatasetSmartName = $scope.recipeParams.inputDs;
        finalRecipe.referenceDatasetSmartName = $scope.recipeParams.referenceDs;
        finalRecipe.evaluationStoreSmartName = $scope.recipe.outputs.main.items[0].ref;
        finalRecipe.zone = $scope.zone;

        DataikuAPI.savedmodels.prediction.deployStandaloneEvaluation(ActiveProjectKey.get(), finalRecipe)
            .success(function(data) {
                $scope.creatingRecipe = false;
                $scope.dismiss();
                $scope.$state.go('projects.project.recipes.recipe', {
                    recipeName: data.id
                });
            }).error(function(a, b, c) {
                $scope.creatingRecipe = false;
                setErrorInScope.bind($scope)(a,b,c);
            });;

    };
});

app.controller("SavedModelVersionSkinsController", function($scope, $state, VirtualWebApp, $rootScope, $timeout, Logger) {
    function setSkinFromTile() {
        // if insight or dashboard tile, we select the skin defined in the params
        if ($scope.tile.tileParams && $scope.tile.tileParams.advancedOptions && $scope.tile.tileParams.advancedOptions.customViews) {
            const viewId = $scope.tile.tileParams.advancedOptions.customViews.viewId;
            $scope.uiState.skin = $scope.modelSkins.find(s => s.id === viewId);
        }
    }

    let modelId = '';
    let version = '';
    if ($scope.savedModel && $scope.savedModel.id) {
        // when called from sm
        modelId = $scope.savedModel.id;
    } else if ($scope.insight &&  $scope.insight.$savedModel && $scope.insight.$savedModel.id) {
        // when called from insight
        modelId = $scope.insight.$savedModel.id;
    } else {
        Logger.error("Skin missing model's modelId");
    }
    if ($scope.versionsContext && $scope.versionsContext.currentVersion && $scope.versionsContext.currentVersion.versionId) {
        // when called from sm
        version = $scope.versionsContext.currentVersion.versionId
    } else if ($scope.insight && $scope.insight.$savedModel && $scope.insight.$savedModel.activeVersion) {
        // when called from insight
        version = $scope.insight.$savedModel.activeVersion;
    } else {
        Logger.error("Skin missing model's version");
    }
    if ($scope.tile && $scope.tile.insightId) {
        $scope.skinHolderClass = "skin-holder-insight-" + $scope.tile.insightId;
        const deregister = $scope.$watch('modelSkins', function (nv, ov) {
            if (!nv) {return}
            // make sure modelSkins is defined before calling setSkinFromTile
            $scope.$watch('tile.tileParams.advancedOptions.customViews.viewId', function () {
                setSkinFromTile(); // changes uiState.skin accordingly
            });
            deregister();
        });
    } else {
        $scope.skinHolderClass = "skin-holder"
    }

    $scope.$watch('uiState.skin', function() {
        if (!$scope.uiState.skin) {return;}
        if ($scope.tile && $scope.tile.tileParams && $scope.tile.tileParams.displayMode === 'skins'
            && $scope.tile.tileParams.advancedOptions && $scope.tile.tileParams.advancedOptions.customViews) {
            // we are in a dashboard tile and the tile has a custom config
            const tileView = $scope.tile.tileParams.advancedOptions.customViews;
            $scope.webAppCustomConfig = {
                ...tileView.viewParams
            }
        }

        // ng-class="skinHolderClass"  needs to be evaluated before changing skin
        $timeout(() =>
            VirtualWebApp.changeSkin($scope, 'SAVED_MODEL', $scope.uiState.skin, $scope.uiState, $scope.skinHolderClass, modelId,
                version, false)
        );
    }, true);
});

app.service("CreateSavedModelVersionService", function(PMLSettings, ClipboardReadWriteService, DatasetUtils, AnyLoc, DataikuAPI) {
    return {
        onInit: function($scope, predictionType) {
            $scope.predictionTypes = PMLSettings.task.predictionTypes.filter(type => type.classical);

            $scope.binaryIncorrectNumClasses = false;
            $scope.multiclassIncorrectNumClasses = false;

            $scope.hasInvalidClasses = function () {
                return $scope.binaryIncorrectNumClasses || $scope.multiclassIncorrectNumClasses;
            }

            $scope.deployingSmv = false;

            $scope.newSavedModelVersion = {
                versionId: null,
                predictionType: predictionType,
                datasetSmartName: null,
                targetColumn: null,
                classes: [],
                smvFuture: null,
                samplingParam: {"samplingMethod": "HEAD_SEQUENTIAL", "maxRecords": 10000, "ascending": true},
                binaryClassificationThreshold: 0.5,
                useOptimalThreshold: true,
                skipExpensiveReports: true,
                activate: true
            };

            $scope.uiState = {
                evaluateModel: $scope.newSavedModelVersion.predictionType !== "OTHER",
                activeModelVersion: null,
                progress: null,
            };
        },
        validateClasses: function($scope) {
            $scope.binaryIncorrectNumClasses = false;
            $scope.multiclassIncorrectNumClasses = false;
            if ($scope.isClassification()) {
                const classes = $scope.newSavedModelVersion.classes.filter(aClass => aClass && aClass.length > 0);
                if ($scope.newSavedModelVersion.predictionType === 'BINARY_CLASSIFICATION') {
                    if (classes.length != 2) {
                        $scope.binaryIncorrectNumClasses = true;
                    }
                } else if ($scope.newSavedModelVersion.predictionType === 'MULTICLASS') {
                    if (classes.length < 2) {
                        $scope.multiclassIncorrectNumClasses = true;
                    }
                }
            }
        },
        validateTarget: function($scope) {
            if (!$scope.uiState.evaluateModel) {
                $scope.invalidTarget = false;
            } else {
                $scope.invalidTarget = !$scope.targetColumnNames.includes($scope.newSavedModelVersion.targetColumnName);
                if ($scope.invalidTarget) {
                    $scope.currentInvalidTargetColumnName = $scope.newSavedModelVersion.targetColumnName;
                }
            }
        },
        resetErrorDisplay: function($scope) {
            $scope.invalidTarget = false;
            $scope.binaryIncorrectNumClasses = false;
            $scope.multiclassIncorrectNumClasses = false;
        },
        copyClasses: function($scope) {
            ClipboardReadWriteService.writeItemsToClipboard($scope.newSavedModelVersion.classes);
            $scope.$applyAsync()
        },
        pasteClasses: async function($scope) {
            try {
                let classes = await ClipboardReadWriteService.readItemsFromClipboard($scope, "The following classes will be used")
                $scope.newSavedModelVersion.classes = classes
                $scope.$applyAsync()
            } catch (error) {
                // do nothing
            }
        },
        isClassification: function($scope) {
            if (!$scope.newSavedModelVersion || !$scope.newSavedModelVersion.predictionType) {
                return false;
            }
            return $scope.newSavedModelVersion.predictionType === 'BINARY_CLASSIFICATION' || $scope.newSavedModelVersion.predictionType === 'MULTICLASS';
        },
        afterInit: function($scope, $stateParams, versions) {
            if (versions && versions.length) {
                const activeVersions = versions.filter(v => v.active);
                if (activeVersions && activeVersions.length) {
                    const activeVersion = activeVersions[0];
                    const activeVersionSnippet = activeVersion.snippet;
                    $scope.uiState.activeModelVersion = activeVersion.versionId;
                    if (activeVersionSnippet.proxyModelConfiguration) {
                        switch (activeVersionSnippet.proxyModelConfiguration.protocol) {
                            case "sagemaker":
                                $scope.newSavedModelVersion.sageMakerEndpointName = activeVersionSnippet.proxyModelConfiguration.endpoint_name;
                                break;
                            case "vertex-ai":
                                $scope.newSavedModelVersion.vertexEndpointId = activeVersionSnippet.proxyModelConfiguration.endpoint_id;
                                $scope.newSavedModelVersion.vertexProjectId = activeVersionSnippet.proxyModelConfiguration.project_id;
                                break;
                            case "azure-ml":
                                $scope.newSavedModelVersion.azureEndpointName = activeVersionSnippet.proxyModelConfiguration.endpoint_name;
                                $scope.newSavedModelVersion.azureResourceGroup = activeVersionSnippet.proxyModelConfiguration.resource_group;
                                $scope.newSavedModelVersion.azureSubscriptionId = activeVersionSnippet.proxyModelConfiguration.subscription_id;
                                $scope.newSavedModelVersion.azureWorkspace = activeVersionSnippet.proxyModelConfiguration.workspace;
                                break;
                            case "databricks":
                                $scope.newSavedModelVersion.endpointName = activeVersionSnippet.proxyModelConfiguration.endpointName;
                                break;
                        }
                    }
                    if (activeVersionSnippet.mlflowClassLabels && activeVersionSnippet.mlflowClassLabels.length) {
                        $scope.newSavedModelVersion.classes = activeVersionSnippet.mlflowClassLabels.map(l => l.label);
                    }
                    $scope.newSavedModelVersion.datasetSmartName = activeVersionSnippet.mlflowEvaluationDatasetSmartName;
                    $scope.uiState.signatureAndFormatsGuessingDatasetSmartName = activeVersionSnippet.mlflowSignatureAndFormatsGuessingDatasetSmartName;
                    $scope.newSavedModelVersion.samplingParam = activeVersionSnippet.mlflowEvaluationSamplingParam;
                    $scope.uiState.evaluateModel = !!$scope.newSavedModelVersion.datasetSmartName;
                    if ($scope.uiState.evaluateModel) {
                        $scope.newSavedModelVersion.targetColumnName = activeVersionSnippet.mlflowEvaluationTargetColumnName;
                    } else {
                        $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName = activeVersionSnippet.mlflowEvaluationTargetColumnName;
                    }
                }
            }

            DatasetUtils.listDatasetsUsabilityForAny($stateParams.projectKey).success(function(data){
                $scope.availableDatasets = data;
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("newSavedModelVersion.datasetSmartName", function(newDatasetSmartName, oldDatasetSmartName) {
                if (newDatasetSmartName) {
                    $scope.uiState.signatureAndFormatsGuessingDatasetSmartName = newDatasetSmartName;
                    const datasetLoc = AnyLoc.getLocFromSmart($stateParams.projectKey, newDatasetSmartName);
                    DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.localId, $stateParams.projectKey).success(function(dataset){
                        const columns = dataset && dataset.schema && dataset.schema.columns ? dataset.schema.columns : [];
                        $scope.targetColumnNames = columns.map(function(col) { return col.name });
                        if (!$scope.targetColumnNames.includes($scope.newSavedModelVersion.targetColumnName)) {
                            $scope.newSavedModelVersion.targetColumnName = null;
                        }
                    }).error(setErrorInScope.bind($scope));
                }
            });

            $scope.$watch("uiState.evaluateModel", function(newEvaluateModel, oldEvaluateModel) {
                if (newEvaluateModel === oldEvaluateModel || !$scope.uiState.guessFormat) {
                    return;
                }

                if (newEvaluateModel) {
                    $scope.newSavedModelVersion.datasetSmartName = $scope.uiState.signatureAndFormatsGuessingDatasetSmartName ? $scope.uiState.signatureAndFormatsGuessingDatasetSmartName : $scope.newSavedModelVersion.datasetSmartName;
                    $scope.targetColumnNames = $scope.signatureAndFormatsGuessingDatasetColumnNames ? $scope.signatureAndFormatsGuessingDatasetColumnNames : $scope.targetColumnNames;
                    $scope.newSavedModelVersion.targetColumnName = $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName ? $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName : $scope.newSavedModelVersion.targetColumnName;
                } else {
                    $scope.uiState.signatureAndFormatsGuessingDatasetSmartName = $scope.newSavedModelVersion.datasetSmartName ? $scope.newSavedModelVersion.datasetSmartName : $scope.uiState.signatureAndFormatsGuessingDatasetSmartName;
                    $scope.signatureAndFormatsGuessingDatasetColumnNames = $scope.targetColumnNames ? $scope.targetColumnNames : $scope.signatureAndFormatsGuessingDatasetColumnNames;
                    $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName = $scope.newSavedModelVersion.targetColumnName ? $scope.newSavedModelVersion.targetColumnName : $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName;
                }
            });
        }
    };
});

// use createProxySavedModelVersionModalDirective when calling CreateModalFromComponent
app.component("createProxySavedModelVersionModal", {
    bindings: {
        savedModel: '<',
        smStatus: '<',
        modalControl: '<'
    },
    templateUrl: '/templates/savedmodels/new-proxy-saved-model-version-modal.html',

    controller: function($scope, $state, Assert, DataikuAPI, WT1, $stateParams, SavedModelsService, AnyLoc, PMLSettings, DatasetUtils, MonoFuture, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS, ClipboardReadWriteService, FutureWatcher, CreateSavedModelVersionService) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            Assert.trueish(SavedModelsService.isProxyModel($ctrl.savedModel), 'Cannot create a saved model version with the GUI for a non-external model.');
            const details = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === $ctrl.savedModel.proxyModelConfiguration.protocol);
            if (!details) {
                return;
            }
            $scope.externalModelDetails = details;

            CreateSavedModelVersionService.onInit($scope, $ctrl.savedModel.miniTask.predictionType);

            $scope.isSageMaker = $ctrl.savedModel.proxyModelConfiguration.protocol == 'sagemaker';
            $scope.isVertex = $ctrl.savedModel.proxyModelConfiguration.protocol == 'vertex-ai';
            $scope.hasRegion = $scope.isSageMaker || $scope.isVertex;
            $scope.deployingSmv = false;

            $scope.newSavedModelVersion = {
                ...$scope.newSavedModelVersion,
                sageMakerEndpointName: null,
                vertexProjectId: $ctrl.savedModel.proxyModelConfiguration.project_id,
                vertexEndpointId: null,
                azureSubscriptionId: $ctrl.savedModel.proxyModelConfiguration.subscription_id,
                azureResourceGroup: $ctrl.savedModel.proxyModelConfiguration.resource_group,
                azureWorkspace: $ctrl.savedModel.proxyModelConfiguration.workspace,
                azureEndpointName: null,
            };

            $scope.uiState = {
                ...$scope.uiState,
                loadingSageMakerEndpointList: false,
                vertexProjectDisplayName: null,
                vertexInformationRetrievalError: null,
                guessFormat: true,
                signatureAndFormatsGuessingDatasetSmartName: null,
                signatureAndFormatsGuessingDatasetTargetColumnName: null,
                inputFormat: null,
                outputFormat: null
            };

            CreateSavedModelVersionService.afterInit($scope, $stateParams, $ctrl.smStatus.versions);

            if ($ctrl.savedModel.proxyModelConfiguration.protocol === 'vertex-ai') {
                retrieveVertexProjectInformation($scope.newSavedModelVersion.vertexProjectId, $ctrl.savedModel.proxyModelConfiguration.connection);
            }
        }

        $scope.$watch("uiState.signatureAndFormatsGuessingDatasetSmartName", function(newSignatureAndFormatsGuessingDatasetSmartName, oldSignatureAndFormatsGuessingDatasetSmartName) {
            if (newSignatureAndFormatsGuessingDatasetSmartName && $scope.uiState.guessFormat) {
                const signatureAndFormatsGuessingDatasetLoc = AnyLoc.getLocFromSmart($stateParams.projectKey, newSignatureAndFormatsGuessingDatasetSmartName);
                DataikuAPI.datasets.get(signatureAndFormatsGuessingDatasetLoc.projectKey, signatureAndFormatsGuessingDatasetLoc.localId, $stateParams.projectKey).success(function(dataset){
                    const columns = dataset && dataset.schema && dataset.schema.columns ? dataset.schema.columns : [];
                    $scope.signatureAndFormatsGuessingDatasetColumnNames = columns.map(function(col) { return col.name });
                    if (!$scope.signatureAndFormatsGuessingDatasetColumnNames.includes($scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName)) {
                        $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName = null;
                    }
                }).error(setErrorInScope.bind($scope));
            }
        });

        async function retrieveVertexProjectInformation(vertexProjectId, connectionName) {
            const project = await getVertexAIProject(vertexProjectId, connectionName);
            if (!project) {
                return;
            }
            $scope.uiState.vertexProjectDisplayName = project.displayName;
        }

        async function getVertexAIProject(vertexProjectId, connectionName) {
            try {
                const resp = await DataikuAPI.externalinfras.infos.getVertexAIProject(vertexProjectId, connectionName);
                const resp2 = await FutureWatcher.watchJobId(resp.data.jobId).update(data => {
                    if (data.progress && data.progress.states) {
                        $scope.uiState.progress = "Retrieving project information...";
                    } else {
                        $scope.uiState.progress = null;
                    }
                });
                return resp2.data.result;
            } catch (e) {
                $scope.uiState.vertexInformationRetrievalError = "Failed to retrieve project information.";
            }
        }

        $scope.$watch("newSavedModelVersion.targetColumnName", function(nv, ov) {
            CreateSavedModelVersionService.resetErrorDisplay($scope);
        });

        $scope.$watch("newSavedModelVersion.classes", function(nv, ov) {
            CreateSavedModelVersionService.resetErrorDisplay($scope);
        }, true);

        function getProxyModelVersionConfiguration() {
            const proxyModelConfiguration = {
                protocol: $ctrl.savedModel.proxyModelConfiguration.protocol,
                connection: $ctrl.savedModel.proxyModelConfiguration.connection
            };
            const proxyModelVersionConfiguration = {
                protocol: $ctrl.savedModel.proxyModelConfiguration.protocol
            };

            if (proxyModelConfiguration.protocol === "sagemaker") {
                proxyModelConfiguration.region = $ctrl.savedModel.proxyModelConfiguration.region;

                proxyModelVersionConfiguration.endpoint_name = $scope.newSavedModelVersion.sageMakerEndpointName;
            } else if (proxyModelConfiguration.protocol === "vertex-ai") {
                proxyModelConfiguration.project_id = $scope.newSavedModelVersion.vertexProjectId;
                proxyModelConfiguration.region = $ctrl.savedModel.proxyModelConfiguration.region;

                proxyModelVersionConfiguration.endpoint_id = $scope.newSavedModelVersion.vertexEndpointId;
            } else if (proxyModelConfiguration.protocol === "azure-ml") {
                proxyModelConfiguration.subscription_id = $scope.newSavedModelVersion.azureSubscriptionId;
                proxyModelConfiguration.resource_group = $scope.newSavedModelVersion.azureResourceGroup;
                proxyModelConfiguration.workspace = $scope.newSavedModelVersion.azureWorkspace;
                proxyModelVersionConfiguration.endpoint_name = $scope.newSavedModelVersion.azureEndpointName;
            } else if (proxyModelConfiguration.protocol === "databricks") {
                proxyModelVersionConfiguration.endpointName = $scope.newSavedModelVersion.endpointName;
            }
            proxyModelVersionConfiguration.proxyModelConfiguration = proxyModelConfiguration;
            return proxyModelVersionConfiguration;
        }

        function getModelVersionInfo() {
            let signatureAndFormatsGuessingDatasetSmartName, inputFormat, outputFormat;
            if ($ctrl.savedModel.proxyModelConfiguration.protocol === 'vertex-ai') {
                signatureAndFormatsGuessingDatasetSmartName = null;
                inputFormat = 'INPUT_VERTEX_DEFAULT';
                outputFormat = 'OUTPUT_VERTEX_DEFAULT';
            } else if ($scope.uiState.guessFormat) {
                signatureAndFormatsGuessingDatasetSmartName = $scope.uiState.signatureAndFormatsGuessingDatasetSmartName;
                inputFormat = "GUESS";
                outputFormat = "GUESS";
            } else {
                signatureAndFormatsGuessingDatasetSmartName = null;
                inputFormat = $scope.uiState.inputFormat;
                outputFormat = $scope.uiState.outputFormat;
            }
            const targetColumnName = $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.targetColumnName : $scope.uiState.signatureAndFormatsGuessingDatasetTargetColumnName;
            const gatherFeaturesFromDataset = $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.datasetSmartName : $scope.uiState.signatureAndFormatsGuessingDatasetSmartName;
            const evaluationDatasetSmartName =  $scope.uiState.evaluateModel ? $scope.newSavedModelVersion.datasetSmartName : null;
            const modelVersionInfo = {
                ...(evaluationDatasetSmartName && {evaluationDatasetSmartName}),
                ...(gatherFeaturesFromDataset && {gatherFeaturesFromDataset}),
                predictionType: $scope.newSavedModelVersion.predictionType,
                classLabels: [],
                targetColumnName: targetColumnName,
                signatureAndFormatsGuessingDataset: signatureAndFormatsGuessingDatasetSmartName,
                inputFormat,
                outputFormat,
            };
            if ($scope.isClassification()) {
                modelVersionInfo.classLabels = $scope.newSavedModelVersion.classes.map(classLabel => { return { label: classLabel }; });
            }
            modelVersionInfo.proxyModelVersionConfiguration = getProxyModelVersionConfiguration();
            return modelVersionInfo;
        }

        $scope.create = function() {
            CreateSavedModelVersionService.validateClasses($scope);
            CreateSavedModelVersionService.validateTarget($scope);
            if ($scope.hasInvalidClasses() || $scope.invalidTarget) {
                return;
            }
            $scope.deployingSmv = true;

            const modelVersionInfo = getModelVersionInfo();

            resetErrorInScope($scope);
            WT1.event("saved-model-create-proxy", { from: 'sm-list', predictionType: $scope.newSavedModelVersion.predictionType });
            MonoFuture($scope).wrap(DataikuAPI.savedmodels.prediction.createProxySavedModelVersion)(
                $stateParams.projectKey,
                $ctrl.savedModel.id,
                $scope.newSavedModelVersion.versionId,
                modelVersionInfo,
                $scope.newSavedModelVersion.samplingParam,
                $scope.newSavedModelVersion.binaryClassificationThreshold,
                $scope.newSavedModelVersion.useOptimalThreshold,
                $scope.newSavedModelVersion.activate,
                $scope.newSavedModelVersion.skipExpensiveReports,
                {"containerMode": "INHERIT"}
            ).success(function (futureResponse) {
                const smvData = futureResponse.result;
                $scope.smvFuture = null;
                $scope.deployingSmv = false;
                $ctrl.modalControl.dismiss();
                $state.go('projects.project.savedmodels.savedmodel.prediction.report', Object.assign({}, $stateParams, { fullModelId: smvData.fullModelId }));
            }).update(function (data) {
                $scope.smvFuture = data;
            }).error($scope.onSVMCreationError);
        }

        $scope.onSVMCreationError = function (data, status, headers) {
            $scope.smvFuture = null;
            $scope.deployingSmv = false;
            const elem = document.getElementById('new-smv-modal-body');
            if (elem) {
                elem.scrollTop = 0;
            }
            setErrorInScope.bind($scope)(data, status, headers);
        }

        $scope.isClassification = CreateSavedModelVersionService.isClassification.bind(this, $scope);
        $scope.copyClasses = CreateSavedModelVersionService.copyClasses.bind(this, $scope);
        $scope.pasteClasses = CreateSavedModelVersionService.pasteClasses.bind(this, $scope);
    }
});

// use createExternalSavedModelSelectorModalDirective when calling CreateModalFromComponent
app.component('createExternalSavedModelSelectorModal', {
    templateUrl: '/templates/savedmodels/new-external-saved-model-selector-modal.html',
    controller: function($scope, CreateModalFromComponent, createExternalSavedModelModalDirective) {
        const $ctrl = this;
        $scope.newExternalSavedModel = function(externalModelTypeName) {
            CreateModalFromComponent(createExternalSavedModelModalDirective, { externalModelTypeName }, ['modal-wide']);
        }
    }
});

app.controller('AgentSelectorModalController', function ($scope, SavedModelsService, StringUtils) {
    // listItems in coming from SavedModelListController
    const smNames = $scope.listItems.map((val) => val.realName);
    $scope.pluginAgents = SavedModelsService.getAllPluginAgents();
    $scope.newAgent = {
        agentType: 'TOOLS_USING_AGENT',
        pluginAgent: null,
        name: null,
        defaultName: StringUtils.transmogrify('Visual Agent', smNames, null, 1),
    };

    $scope.isNameUnique = function (value) {
        for (let k in smNames) {
            let name = smNames[k];
            if ((name || '').toLowerCase() === (value || '').toLowerCase()) {
                return false;
            }
        }
        return true;
    };

    $scope.selectAgentType = function (agentType) {
        $scope.newAgent.agentType = agentType;
        let defaultNameStart = 'Agent';
        switch (agentType) {
            case 'TOOLS_USING_AGENT':
                defaultNameStart = 'Visual Agent';
                break;
            case 'PYTHON_AGENT':
                defaultNameStart = 'Code Agent';
                break;
            case 'PLUGIN_AGENT':
                defaultNameStart = $scope.newAgent.pluginAgent.label;
                break;

        }
        $scope.newAgent.defaultName = StringUtils.transmogrify(defaultNameStart, smNames, null, 1);
    };

    $scope.selectPluginAgent = function (pluginAgent) {
        $scope.newAgent.pluginAgent = pluginAgent;
        $scope.selectAgentType('PLUGIN_AGENT');
    };

    $scope.createAgent = function () {
        let agentName = $scope.newAgent.defaultName;
        if ($scope.newAgent.name) {
            agentName = $scope.newAgent.name;
        }

        if ($scope.newAgent.agentType === 'TOOLS_USING_AGENT') {
            SavedModelsService.newVisualAgent(agentName, null, 'agent-list').error(setErrorInScope.bind($scope));
        } else if ($scope.newAgent.agentType === 'PYTHON_AGENT') {
            SavedModelsService.newCodeAgent(agentName, null, 'agent-list').error(setErrorInScope.bind($scope));
        } else if ($scope.newAgent.agentType === 'PLUGIN_AGENT' && $scope.newAgent.pluginAgent) {
            SavedModelsService.newPluginAgent($scope.newAgent.pluginAgent.agentId, agentName, null, 'agent-list').error(setErrorInScope.bind($scope));
        }
    };
});

// use createExternalSavedModelModalDirective when calling CreateModalFromComponent
app.component('createExternalSavedModelModal', {
    bindings: {
        externalModelTypeName: '<',
        modalControl: '<'
    },
    templateUrl: '/templates/savedmodels/new-external-saved-model-modal.html',
    controller: function($scope, $state, $stateParams, DataikuAPI, WT1, AnyLoc, PMLSettings, Assert, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS, FutureWatcher) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $scope.creatingSavedModel = false;

            $scope.uiState = {
                availableCompatibleConnections: null,
                advancedMode: false,
            };

            const details = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === $ctrl.externalModelTypeName);

            Assert.trueish(details, 'Invalid or empty externalModelTypeName: ' + $ctrl.externalModelTypeName);

            const isMLflow = details.name == 'mlflow';
            const isFinetuned = details.name == 'finetuned';
            const isProxyModel = !isMLflow && !isFinetuned;

            $scope.predictionTypes = PMLSettings.task.predictionTypes.filter(type => (type.classical || isMLflow && type.other) && !isFinetuned);

            $scope.isSageMaker = details.name == 'sagemaker';
            $scope.isVertex = details.name == 'vertex-ai';

            $scope.externalModelDetails = details;

            $scope.newExternalSavedModel = {
                name : null,
                predictionType : isFinetuned ? null : $scope.predictionTypes[0].type,
                protocol: isProxyModel ? details.name : null,
                connection: "",  // null is used to represent the Environment connection
                region: null,
                savedModelType: details.savedModelType,
                isMLflow: isMLflow,
                isFinetuned: isFinetuned,
                azureSubscriptionId: null,
                azureResourceGroup: null,
                azureWorkspace: null,
                vertexProjectId: null
            };

            $scope.$watch("newExternalSavedModel.protocol", function(protocol) {
                $scope.uiState.availableCompatibleConnections = null;
                if (protocol) {
                    const fullProtocol = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === protocol);
                    if (fullProtocol && fullProtocol.connectionType) {
                        DataikuAPI.connections.getNames(fullProtocol.connectionType).success(function (data) {
                            $scope.uiState.availableCompatibleConnections =
                                (($scope.newExternalSavedModel.protocol == 'databricks')?[]:[{ name: null, label: "Environment" }]).concat(data.map(n => { return { name: n, label: n } }));
                            if ($scope.uiState.availableCompatibleConnections.length == 1) {
                                // the only available connection is environment => select it
                                $scope.newExternalSavedModel.connection = null;
                            }
                        }).error(setErrorInScope.bind($scope));
                    }
                }
            });

            function getProxyModelConfiguration() {
                if (!$scope.newExternalSavedModel.protocol) {
                    return null;
                }

                const proxyModelConfiguration = {
                    protocol: $scope.newExternalSavedModel.protocol,
                    connection: $scope.newExternalSavedModel.connection,
                    region: $scope.newExternalSavedModel.region
                };
                if ($scope.newExternalSavedModel.protocol === "azure-ml") {
                    proxyModelConfiguration.subscription_id = $scope.newExternalSavedModel.azureSubscriptionId;
                    proxyModelConfiguration.resource_group = $scope.newExternalSavedModel.azureResourceGroup;
                    proxyModelConfiguration.workspace = $scope.newExternalSavedModel.azureWorkspace;
                } else if ($scope.newExternalSavedModel.protocol === "vertex-ai") {
                    proxyModelConfiguration.project_id = $scope.newExternalSavedModel.vertexProjectId;
                }
                // No additional fields necessary for SageMaker
                return proxyModelConfiguration;
            }

            $scope.create = function() {
                $scope.creatingSavedModel = true;
                resetErrorInScope($scope);
                WT1.event("saved-model-create-external", { from: 'sm-list', externalModelTypeName: $ctrl.externalModelTypeName, predictionType: $scope.newExternalSavedModel.predictionType });

                const proxyModelConfiguration = getProxyModelConfiguration();
                DataikuAPI.savedmodels.prediction.createExternal(
                    $stateParams.projectKey,
                    $scope.newExternalSavedModel.savedModelType,
                    $scope.newExternalSavedModel.predictionType,
                    $scope.newExternalSavedModel.name,
                    proxyModelConfiguration
                ).success(function(smData) {
                    $scope.creatingSavedModel = false;
                    $state.go('projects.project.savedmodels.savedmodel.versions', Object.assign({}, $stateParams, { smId: smData.id }));
                }).error(function(data, status, headers) {
                    $scope.creatingSavedModel = false;
                    setErrorInScope.bind($scope)(data, status, headers);
                });
            }
        }
    }
});

app.component("sagemakerVertexRegionCodeSelector", {
    bindings: {
        protocol: "<",          // 'vertex-ai' or 'sagemaker'
        value: '=',             // 2-way binding value of the selector (ex: 'eu-west1')
        required: '<',          // Adds a "*" to the title if set
        disabledMessage: "<",   // if set, disable input and tooltip this message
        additionalHelp: "<?"
    },
    template: `
    <div class="region-selector">
        <div block-api-error />
        <label class="control-label">Region{{$ctrl.required ? '*' : ''}}</label>
        <div class="controls"
             toggle="tooltip-left"
             title="{{$ctrl.disabledMessage ? $ctrl.disabledMessage : ''}}">
            <div ng-if="uiState.availableRegions != null"
                toggle="tooltip-left"
                container="body"
                title="{{uiState.selectedRegion.descWithDefault}}">
                <select
                    data-qa-new-model-form-region
                    dku-bs-select
                    ng-model="$ctrl.value"
                    ng-options="r.name as r.nameWithDefault for r in uiState.availableRegions"
                    data-live-search="true"
                    ng-disabled="$ctrl.disabledMessage"
                    ng-required="$ctrl.required"
                />
                <button class="btn btn--secondary" ng-click="uiState.availableRegions = null" ng-disabled="$ctrl.disabledMessage">
                    Enter custom
                </button>
                <span class="help-inline">Region name (eg: "{{ $ctrl.protocol === 'sagemaker' ? 'eu-west-3' : 'europe-west9' }}")</span>
            </div>
            <div ng-if="uiState.availableRegions == null">
                <input
                    type="text"
                    ng-model="$ctrl.value"
                    placeholder="Region"
                    ng-pattern="/^[a-zA-Z][a-zA-Z0-9-]{0,30}\\d$/"
                    ng-disabled="$ctrl.disabledMessage"
                    ng-required="$ctrl.required"
                    data-qa-new-model-form-region
                />
                <button type="button" class="btn btn--secondary" ng-click="fetchAvailableRegions()" ng-disabled="$ctrl.disabledMessage">Get regions</button>
                <span class="help-inline">Region name (eg: "{{ $ctrl.protocol === 'sagemaker' ? 'eu-west-3' : 'europe-west9' }}"){{$ctrl.additionalHelp?('. ' + $ctrl.additionalHelp):''}}</span>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, DataikuAPI) {
        const $ctrl = this;

        $scope.fetchAvailableRegions = function() {
            let listRegions;
            if ($ctrl.protocol === "sagemaker") {
                listRegions = DataikuAPI.externalinfras.infos.listSagemakerRegions;
            } else if ($ctrl.protocol === "vertex-ai") {
                listRegions = DataikuAPI.externalinfras.infos.listVertexAIRegions;
            }
            else {
                throw new Error('Fetch available regions is not supported for ' + $ctrl.protocol);
            }

            listRegions().then(function(resp) {
                $scope.uiState.availableRegions = resp.data;
                $scope.uiState.availableRegions.forEach(r => {
                    r.descWithDefault = r.description + (!r.isDefault?"":" (default from environment)");
                })
                $scope.uiState.availableRegions.forEach(r => r.nameWithDefault = r.name + (r.isDefault?" (default)":""));
                if ($ctrl.value) {
                    const matches = $scope.uiState.availableRegions.filter(r => r.name === $ctrl.value);
                    if (!matches || !matches.length) {
                        $ctrl.value = null;
                    }
                }
                if (!$ctrl.value) {
                    const defaultRegions = $scope.uiState.availableRegions.filter(r => r.isDefault);
                    if (defaultRegions && defaultRegions.length) {
                        // should really only be one at most...
                        $ctrl.value = defaultRegions[0].name;
                    }
                }
            }).catch(setErrorInScope.bind($scope));
        }

        $ctrl.$onInit = function() {
            $scope.uiState = {
                selectedRegion: null
            };

            $scope.$watch("$ctrl.value", function(nv) {
                if (!nv || !$scope.uiState.availableRegions || !$scope.uiState.availableRegions.length) {
                    $scope.uiState.selectedRegion = null;
                } else {
                    const selectedRegions = $scope.uiState.availableRegions.filter(r => r.name === nv);
                    if (selectedRegions && selectedRegions.length) {
                        $scope.uiState.selectedRegion = selectedRegions[0];
                    } else {
                        $scope.uiState.selectedRegion = null;
                    }
                }
            });
        }
    }
});

app.component("vertexProjectSelector", {
    bindings: {
        value: '=',                 // 2-way binding value of the selector (ex: 'team-miel-pops')
        disabledMessage: '<',       // if set, disable input and tooltip this message
        connectionName: '<',        // if set, use the credentials from the connection to log into GCP, else use environment
        required: '<',              // Adds a "*" to the title if set
        disableAutoFetch: '<',      // If projects list should be fetched on component initialization or on connection change (false by default)
    },
    template: `
    <div class="vertex-project-selector">
        <label class="control-label">{{uiState.existingVertexAIProjects ? "Project Name" : "Project ID"}}{{$ctrl.required ? '*' : ''}}</label>
        <div class="controls horizontal-flex"  style="line-height: 24px;"
             toggle="tooltip-left"
             title="{{$ctrl.disabledMessage ? $ctrl.disabledMessage : ''}}">
            <div ng-if="!uiState.loadingVertexAIProjectList && uiState.existingVertexAIProjects != null">
                <select
                        dku-bs-select
                        ng-model="$ctrl.value"
                        ng-options="ep.id as ep.displayName for ep in uiState.existingVertexAIProjects"
                        data-live-search="true"
                        ng-required="$ctrl.required"
                />
                <button class="btn btn--secondary" ng-click="uiState.existingVertexAIProjects = null;">
                    Enter custom
                </button>
            </div>
            <div ng-if="uiState.loadingVertexAIProjectList || uiState.existingVertexAIProjects == null">
                <input type="text"
                       ng-model="$ctrl.value"
                       placeholder="GCP Project ID"
                       ng-disabled="$ctrl.disabledMessage"
                       ng-pattern="/^[a-z][a-z0-9-]{4,28}[a-z0-9]$/"
                       ng-required="$ctrl.required"
                       data-qa-new-version-form-project-name/>
                <div class="dib" toggle="tooltip-right" title="{{$ctrl.isValidConnection($ctrl.connectionName) ? '' : 'Select a connection to fetch the project list'}}">
                    <button ng-if="!uiState.loadingVertexAIProjectList"
                            type="button"
                            class="btn btn--secondary"
                            ng-disabled="$ctrl.disabledMessage || !$ctrl.isValidConnection($ctrl.connectionName)"
                            ng-click="fetchAvailableProjects()">Get projects list</button>
                </div>
                <span ng-if="uiState.loadingVertexAIProjectList"><i class="icon-spin icon-spinner"></i> {{uiState.progress || "Listing projects..."}}</span>
                <div ng-if="uiState.fetchProjectsFailed !== null" class="alert alert-error mtop8">
                    <div><i class="icon-warning-sign alert-error"></i>&nbsp;An error occurred when fetching available projects.</div>
                    <div ng-if="uiState.fetchEndpointsFailed.length > 0">{{uiState.fetchEndpointsFailed}} <a ng-click="$ctrl.copyErrorToClipboard()"><i class="dku-icon-copy-step-16"/></a></div>
                </div>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, DataikuAPI, FutureWatcher, ClipboardUtils) {
        const $ctrl = this;

        $ctrl.$onInit = function () {
            $scope.uiState = {
                selectedProject: null,
                fetchProjectsFailed: null
            };

            if (!$ctrl.value && !$ctrl.disabledMessage && !$ctrl.disableAutoFetch) {
                $scope.fetchAvailableProjects();
            }
        }

        $scope.fetchAvailableProjects = function () {
            $scope.uiState.fetchProjectsFailed = null;
            $ctrl.value = null;
            DataikuAPI.externalinfras.infos.listVertexAIProjects($ctrl.connectionName)
                .then(function (resp) {
                    $scope.$applyAsync(() => {
                        $scope.uiState.loadingVertexAIProjectList = true;
                        $scope.uiState.progress = null;
                    });
                    FutureWatcher.watchJobId(resp.data.jobId)
                        .update(function (data) {
                            if (data.progress && data.progress.states) {
                                $scope.uiState.progress = "Fetching projects list...";
                            } else {
                                $scope.uiState.progress = null;
                            }
                        }).then(function (resp2) {
                        $scope.uiState.existingVertexAIProjects = resp2.data.result;
                        const defaultProject = $scope.uiState.existingVertexAIProjects.filter(r => r.isDefault);
                        if (defaultProject && defaultProject.length) {
                            // should really only be one at most...
                            $ctrl.value = defaultProject[0].id;
                        }

                    }).catch((err) => {
                        $scope.uiState.fetchProjectsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                        setErrorInScope.bind($scope);
                    }).finally(() => {
                        $scope.uiState.loadingVertexAIProjectList = false;
                    });
                })
                .catch((err) => {
                    $scope.uiState.fetchProjectsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                    setErrorInScope.bind($scope);
                }).finally(() => {
                $scope.uiState.loadingVertexAIProjectList = false;
            });
        }

        $scope.$watch("$ctrl.value", function (nv) {
            $scope.uiState.fetchProjectsFailed = null;
            if (!nv || !$scope.uiState.existingVertexAIProjects || !$scope.uiState.existingVertexAIProjects.length) {
                $scope.uiState.selectedProject = null;
            } else {
                const selectedProject = $scope.uiState.existingVertexAIProjects.filter(r => r.id === nv);
                if (selectedProject && selectedProject.length) {
                    $scope.uiState.selectedProject = selectedProject[0];
                } else {
                    $scope.uiState.selectedProject = null;
                }
            }
        });

        $scope.$watch("$ctrl.connectionName", function (nv) {
            if (!$ctrl.value && !$ctrl.disabledMessage && !["", undefined].includes(nv) && !$ctrl.disableAutoFetch) {
                $scope.fetchAvailableProjects();
            } else {
                $scope.uiState.existingVertexAIProjects = null;
            }
        });

        $ctrl.copyErrorToClipboard = function() {
            ClipboardUtils.copyToClipboard($scope.uiState.fetchEndpointsFailed);
        };

        $ctrl.isValidConnection = function(connectionName) {
            return ![undefined, ""].includes(connectionName);
        };
    }
});

app.component("externalPlatformEndpointSelector", {
    bindings: {
        protocol: "<",  // 'vertex-ai', 'sagemaker', 'azure-ml' or 'databricks'
        proxyModelConnection: "<", //
        projectId: "<", // project id input (ex "team-miel-pops")
        region: "<",    // region input. (ex: "europe-west1")
        canFetch: "<",  // enables the "Get Endpoint List" button
        value: "=",      // 2-way binding value of the selector (an endpoint id)
        azureWorkspace: "<", // $scope.newSavedModelVersion.azureWorkspace,
        azureResourceGroup: "<", // $scope.newSavedModelVersion.azureResourceGroup,
        azureSubscriptionId: "<"
    },
    template: `
    <div class="external-platform-endpoint-selector">
        <label class="control-label">{{uiState.existingEndpoints ? "Endpoint Name*" : "Endpoint ID*"}}</label>
        <div class="controls horizontal-flex"  style="line-height: 24px;">
            <div ng-if="!uiState.loadingEndpointList && uiState.existingEndpoints != null">
                <select
                        ng-if="$ctrl.protocol === 'databricks'"
                        dku-bs-select
                        ng-model="$ctrl.value"
                        ng-options="ep.name as ep.name for ep in uiState.existingEndpoints"
                        data-live-search="true"
                        required
                />
                <select
                        ng-if="$ctrl.protocol !== 'databricks'"
                        dku-bs-select
                        ng-model="$ctrl.value"
                        ng-options="ep.id || ep.name as ep.name for ep in uiState.existingEndpoints"
                        data-live-search="true"
                        required
                />
                <button class="btn btn--secondary" ng-click="uiState.existingEndpoints = null;">
                    Enter custom
                </button>
            </div>
            <div ng-if="uiState.loadingEndpointList || uiState.existingEndpoints == null">
                <input
                    type="text"
                    ng-model="$ctrl.value"
                    ng-pattern="endpointValidationPattern"
                    placeholder="Endpoint ID"
                    required
                    data-qa-new-version-form-endpoint-name
                />
                <button ng-if="!uiState.loadingEndpointList" ng-disabled="!$ctrl.canFetch" type="button" class="btn btn--secondary" ng-click="fetchAvailableEndpoints()">Get endpoints list</button>
                <span ng-if="uiState.loadingEndpointList"><i class="icon-spin icon-spinner"></i> {{uiState.progress || "Listing endpoints..."}}</span>
                <div ng-if="uiState.fetchEndpointsFailed !== null" class="alert alert-error mtop8">
                    <i class="icon-warning-sign alert-error"></i>&nbsp;An error occurred when fetching available Endpoints.
                    <div ng-if="uiState.fetchEndpointsFailed.length > 0">{{uiState.fetchEndpointsFailed}} <a ng-click="$ctrl.copyErrorToClipboard()"><i class="dku-icon-copy-step-16"/></a></div>
                </div>
            </div>
        </div>
        <div ng-if="uiState.selectedEndpoint.fullId">
            <label class="control-label"></label>
            <div class="controls" style="line-height: 24px;">
                <span class="help-inline" style="width: 100%">Endpoint id: {{uiState.selectedEndpoint.fullId}}</span>
            </div>
        </div>
    </div>
    `,

    controller: function($scope, DataikuAPI, FutureWatcher, ActiveProjectKey, ClipboardUtils) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $scope.uiState = {
                fetchEndpointsFailed: null
            };

            $scope.endpointValidationPattern = getEndpointValidationPattern();
        };

        $scope.fetchAvailableEndpoints = function() {
            $scope.uiState.fetchEndpointsFailed = null;

            let listEndpoints;

            if ($ctrl.protocol === "vertex-ai") {
                listEndpoints = DataikuAPI.externalinfras.infos.listVertexAIEndpoints.bind(this, $ctrl.projectId, $ctrl.region, $ctrl.proxyModelConnection);
            }
            else if ($ctrl.protocol === "sagemaker") {
                listEndpoints = DataikuAPI.externalinfras.infos.listSagemakerEndpointSummaries.bind(this, ActiveProjectKey.get(), $ctrl.region, $ctrl.proxyModelConnection);
            }
            else if ($ctrl.protocol === "azure-ml") {
                listEndpoints = DataikuAPI.externalinfras.infos.listAzureMLEndpoints.bind(this,
                    ActiveProjectKey.get(), $ctrl.azureWorkspace, $ctrl.azureResourceGroup, $ctrl.azureSubscriptionId, $ctrl.proxyModelConnection);
            }
            else if ($ctrl.protocol === "databricks") {
                listEndpoints = DataikuAPI.externalinfras.infos.listDatabricksEndpoints.bind(this,
                    ActiveProjectKey.get(), $ctrl.proxyModelConnection);
            }
            else {
                throw new Error('Fetch available endpoints is not supported for ' + $ctrl.protocol);
            }

            listEndpoints().then(function(resp) {
                $scope.$applyAsync(() => {
                    $scope.uiState.loadingEndpointList = true;
                    $scope.uiState.progress = null;
                });
                FutureWatcher.watchJobId(resp.data.jobId)
                    .update(function(data) {
                        if (data.progress && data.progress.states) {
                            $scope.uiState.progress = "Fetching endpoint list...";
                        } else {
                            $scope.uiState.progress = null;
                        }
                    }).then(function(resp2) {
                    if (resp2.data.aborted) {
                        $scope.uiState.fetchEndpointsFailed = "Fetch endpoint list aborted";
                    }
                    else {
                        $scope.uiState.existingEndpoints = resp2.data.result;
                        if (!$scope.uiState.existingEndpoints || !$scope.uiState.existingEndpoints.length) {
                            $ctrl.value = null;
                        } else {
                            $ctrl.value = $scope.uiState.existingEndpoints.map(e => e.name).find(n => n===$ctrl.value);
                        }
                    }
                }).catch((err) => {
                    $scope.uiState.fetchEndpointsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                    setErrorInScope.bind($scope);
                }).finally(() => { $scope.uiState.loadingEndpointList = false;});
            }).catch((err) => {
                    $scope.uiState.fetchEndpointsFailed = err.data ? err.data.errorType + ': ' + err.data.message : "";
                    setErrorInScope.bind($scope);
            }).finally(() => { $scope.uiState.loadingEndpointList = false;});
        };

        function getEndpointValidationPattern () {
            if ($ctrl.protocol === "sagemaker" || $ctrl.protocol === "vertex-ai") {
                return /^(?!-)(?!.*-$)[a-zA-Z0-9-]{1,63}$/
            }
            if ($ctrl.protocol === "azure-ml") {
                return /^[a-z][a-z0-9-]{1,30}[a-z0-9]$/;
            }
            if ($ctrl.protocol === "databricks") {
                return /^[a-zA-Z0-9]([\w-]{0,61}[a-zA-Z0-9])?$/;
            }
            throw new Error('Invalid protocol: ' + $ctrl.protocol);
        }

        $scope.$watch("$ctrl.projectId", function() {
            $scope.uiState.selectedEndpoint = null;
        });

        $scope.$watch("$ctrl.name", function() {
            $scope.uiState.selectedEndpoint = null;
        });

        $scope.$watch("$ctrl.value", function(nv) {
            $scope.uiState.fetchEndpointsFailed = null;
            if (!nv || !$scope.uiState.existingEndpoints || !$scope.uiState.existingEndpoints.length) {
                $scope.uiState.selectedEndpoint = null;
            } else {
                let selectedEndpoint;
                if ($ctrl.protocol === "databricks") {
                    selectedEndpoint = $scope.uiState.existingEndpoints.filter(e => e.name === nv);
                } else {
                    selectedEndpoint = $scope.uiState.existingEndpoints.filter(e => e.id ? e.id === nv : e.name === nv);
                }
                if (selectedEndpoint && selectedEndpoint.length) {
                    $scope.uiState.selectedEndpoint = selectedEndpoint[0];
                } else {
                    $scope.uiState.selectedEndpoint = null;
                }
            }
        });

        $ctrl.copyErrorToClipboard = function() {
            ClipboardUtils.copyToClipboard($scope.uiState.fetchEndpointsFailed);
        };
    }

});

app.controller('ExternalModelSetupMonitoringModalController', function($scope, $stateParams, DataikuAPI, ActivityIndicator, ActiveProjectKey, $state, WT1, StringUtils){

    DataikuAPI.datasets.listNames($stateParams.projectKey)
        .success(function(datasetNames) {
            $scope.datasetName = StringUtils.transmogrify($scope.smName + "_logs", datasetNames)
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.modelevaluationstores.listHeads($stateParams.projectKey)
        .success(function(heads) {
            $scope.mesName = StringUtils.transmogrify($scope.smName + "_evaluation_store", heads.map(head => head.name))
        })
        .error(setErrorInScope.bind($scope));

    const cloudStorageConnectionType = 'EC2';

    DataikuAPI.connections.listCloudConnectionsHDFSRoot(cloudStorageConnectionType)
        .success(function (data) {
            const cloudStorageConnections = data;
            cloudStorageConnections.sort(function (a,b) {
                const comparison = isLikelyConnection(b, $scope.predictionLogsUri) - isLikelyConnection(a, $scope.predictionLogsUri);
                if (comparison === 0) {
                    return a.name.localeCompare(b.name); // sort alphabetically otherwise
                } else {
                    return comparison;
                }
            });
            $scope.cloudStorageConnections = cloudStorageConnections.map(connection => connection.name);

            if ($scope.cloudStorageConnections.length > 0){
                $scope.connection = $scope.cloudStorageConnections[0];
            }
        })
        .error(setErrorInScope.bind($scope));

    const smId = $stateParams.smId;
    const projectKey = ActiveProjectKey.get();

    /***
     * This methods, used in a sort, will put first in the list the connections that match the bucket and root of the logs URI,
     * second the conenctions that have no root/bucket, and last the connections that don't match it and won't work
     */
    function isLikelyConnection(connection, predictionLogsUri) {
        if (connection.predictionLogsRoot && predictionLogsUri.startsWith(connection.predictionLogsRoot)) {
            return connection.predictionLogsRoot.length;
        } else {
            return -1;
        }
    }

    $scope.ok = function() {
        const params = {
            createMes: $scope.createMes,
            createOutputDataset: $scope.createOutputDataset
        }
        DataikuAPI.savedmodels.prediction.setupProxyModelMonitoring(projectKey, smId, $scope.versionId, $scope.connection, params)
            .success(function(data) {
                ActivityIndicator.success(`External model monitoring created ! <a href="${$state.href('projects.project.flow', {id : 'dataset_' + $stateParams.projectKey + "." + data.inputDatasetRef.objectId})}">
                    View in flow
                </a>`, 5000);
                $scope.resolveModal();
                WT1.event('external-model-monitoring-created', {type: $scope.connection != null ? $scope.connection.type : null});
            })
            .catch((err) => {
                WT1.event('external-model-monitoring-failure', {type: $scope.connection != null ? $scope.connection.type : null});
                setErrorInScope.bind($scope)(err);
            });
    }
    })


app.controller("_AbstractSavedModelPredictedTableController", function($scope, $stateParams, $state, $controller, $q, Assert, DataikuAPI, MonoFuture, Logger, ExportModelDatasetService){
    $scope.exportPredictedData = function() {
        ExportModelDatasetService.exportPredictedData($scope, $scope.smContext.model.fullModelId);
    };

    $scope.shakerWithSteps = false;

    /* ********************* Callbacks for shakerExploreBase ******************* */

    // Nothing to save
    $scope.shakerHooks.saveForAuto = function() {
        return Promise.resolve({});
    }

    var monoFuturizedRefresh = MonoFuture($scope).wrap(DataikuAPI.savedmodels.predicted.predictedRefreshTable);

    $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
        return monoFuturizedRefresh($stateParams.fullModelId, $scope.shaker, filtersOnly, filterRequest);
    }

    $scope.shakerHooks.shakerForQuery = function(){
        var queryObj = angular.copy($scope.shaker);
        if ($scope.isRecipe) {
            queryObj.recipeSchema = $scope.recipeOutputSchema;
        }
        queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
        return queryObj;
    }

    $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
        // withFullSampleStatistics, fullSamplePartitionId are not relevant in this context
        DataikuAPI.savedmodels.predicted.detailedColumnAnalysis($stateParams.fullModelId, $scope.shakerHooks.shakerForQuery(), columnName, alphanumMaxResults).success(function(data){
                    setAnalysis(data);
        }).error(function(a, b, c) {
            if (handleError) {
                handleError(a, b, c);
            }
            setErrorInScope.bind($scope)(a, b, c);
        });
    };

    $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
        return DataikuAPI.savedmodels.predicted.predictedGetTableChunk($stateParams.fullModelId, $scope.shaker,
            firstRow, nbRows, firstCol, nbCols, filterRequest)
    }

    // Load shaker
    DataikuAPI.savedmodels.predicted.getPredictionDisplayScript($stateParams.fullModelId).success(function(data){
        $scope.baseInit();
        $scope.shaker = data;
        $scope.originalShaker = angular.copy($scope.shaker);
        $scope.fixupShaker();
        $scope.refreshTable(false);
    }).error(setErrorInScope.bind($scope));

});


app.directive("predictionSavedModelPredictedTable", function() {
    return {
        scope: true,
        controller: function($scope, $stateParams, $controller, DataikuAPI, ActiveProjectKey, PMLFilteringService, TopNav, WT1) {
            WT1.event("savedmodel-pml-predicted-table-open");
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "PREDICTION-SAVED_MODEL-VERSION", "predictedtable");
            $controller("_PredictionModelReportController",{$scope:$scope});
            $controller("_SavedModelReportController", {$scope:$scope});
            $controller("_SavedModelGovernanceStatusController", {$scope:$scope});

            // Fill the version selector
            const getStatusP = DataikuAPI.savedmodels.prediction.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
                $scope.getGovernanceStatus($stateParams.fullModelId, data.task.partitionedModel);
                $scope.fillVersionSelectorStuff(data);
                $scope.versionsContext.versions.forEach(function(m){
                    m.snippet.mainMetric = PMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                    m.snippet.mainMetricStd = PMLFilteringService.getMetricStdFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                });
            });

            $controller("_AbstractSavedModelPredictedTableController", {$scope:$scope});

        }
    }
});

app.directive("clusteringSavedModelPredictedTable", function() {
    return {
        scope: true,
        controller: function($scope, $stateParams, $controller, DataikuAPI, ActiveProjectKey, CMLFilteringService, TopNav, WT1) {
            WT1.event("savedmodel-cml-predicted-table-open");
            TopNav.setLocation(TopNav.TOP_SAVED_MODELS, TopNav.LEFT_SAVED_MODELS, "CLUSTERING-SAVED_MODEL-VERSION", "predictedtable");
            $controller("_ClusteringModelReportController",{$scope:$scope});
            $controller("_SavedModelReportController", {$scope:$scope});
            $scope.clusteringResultsInitDone = false;

            // Fills the version selector
            const getStatusC = DataikuAPI.savedmodels.clustering.getStatus(ActiveProjectKey.get(), $stateParams.smId)
                .then(({data}) => {
                    $scope.fillVersionSelectorStuff(data);
                    $scope.versionsContext.versions.forEach(function(m){
                        m.snippet.mainMetric = CMLFilteringService.getMetricFromSnippet(m.snippet, $scope.versionsContext.activeMetric);
                    });
            });

            $controller("_AbstractSavedModelPredictedTableController", {$scope:$scope});
        }
    }
});

/* ***************************** Retrieval Augmented LLM creation ************************** */

app.controller('CreateRetrievalAugmentedLLMModalController', function ($scope, DataikuAPI, $state, $stateParams, SmartId, StringUtils) {
    $scope.input = {
        preselectedInput: null,
    };
    $scope.params = {
        name: null,
        knowledgeBank: null,
        knowledgeBankRef: null,
        llmId: null,
    };

    DataikuAPI.savedmodels.retrievalAugmentedLLMs
        .list($stateParams.projectKey)
        .success(function (data) {
            $scope.retrievalAugmentedLLMNames = data.map((sm) => sm.name);
            $scope.$watch('input.preselectedInput', function (preselectedInput) {
                if (Array.isArray(preselectedInput) && preselectedInput.length === 1 && preselectedInput[0].type === 'RETRIEVABLE_KNOWLEDGE') {
                    $scope.params.knowledgeBankRef = SmartId.create(preselectedInput[0].id, preselectedInput[0].projectKey ?? $stateParams.projectKey);
                }
            });
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.pretrainedModels
        .listAvailableLLMs($stateParams.projectKey, 'GENERIC_COMPLETION')
        .then(function ({ data }) {
            $scope.availableAugmentableLLMs = data.identifiers.filter((id) => id.type !== 'RETRIEVAL_AUGMENTED' && id.type !== 'SAVED_MODEL_AGENT');
        })
        .catch(setErrorInScope.bind($scope));

    const defaultNameStart = 'Retrieval of ';


    $scope.$watch('params.knowledgeBankTO', function () {
        if ((!$scope.params.name || $scope.params.name.startsWith(defaultNameStart)) && $scope.params.knowledgeBankTO) {
            $scope.params.name = StringUtils.transmogrify(defaultNameStart + $scope.params.knowledgeBankTO.label, $scope.retrievalAugmentedLLMNames, null, 1);
        }
    });

    $scope.isNameUnique = function (value) {
        for (let k in $scope.retrievalAugmentedLLMNames) {
            let name = $scope.retrievalAugmentedLLMNames[k];
            if ((name || '').toLowerCase() === (value || '').toLowerCase()) {
                return false;
            }
        }
        return true;
    };

    $scope.createRetrievalAugmentedLLM = function () {
        DataikuAPI.savedmodels.retrievalAugmentedLLMs
            .create($stateParams.projectKey, $scope.params.name, $scope.params.knowledgeBankRef, $scope.params.llmId)
            .success(function (data) {
                $state.go('projects.project.savedmodels.savedmodel.versions', {
                    smId: data.id,
                    projectKey: $stateParams.projectKey,
                });
            })
            .catch(setErrorInScope.bind($scope));
    };
});

})();
