// @ts-check
(function() {
    'use strict';
    /** @typedef {import("../../../types").ExpandCollapseToolPanelParams} ExpandCollapseToolPanelParams */
    /** @typedef {import("../../../types").ExpandCollapseToolPanelSection} ExpandCollapseToolPanelSection */
    /** @typedef {import("../../../../../../../../../../server/src/frontend/node_modules/ag-grid-community").IToolPanelComp} IToolPanelComp */

    angular.module('dataiku.charts')
        .factory('ExpandCollapseToolPanel', function(LinkedList, ChartStoreFactory, $rootScope) {

            const EXPAND_COLLAPSE_TOOL_PANEL_LABEL = 'Expand/Collapse | Resize';

            /**
             * @implements IToolPanelComp
             */
            class ExpandCollapseToolPanel {
                constructor() {
                    /**
                     * @type ExpandCollapseToolPanelSection[]
                     */
                    this._sections = [
                        { label: 'Expand', id: 'expand-section', actions: [
                            { label: 'Expand rows', id: 'expand-rows', callback: (params) => this.expandRows(params) },
                            { label: 'Expand columns', id: 'expand-columns', callback: (params) => this.expandColumns(params) },
                            { label: 'Expand all', id: 'expand-all', callback: (params) => this.expandAll(params) }
                        ] },
                        { label: 'Collapse', id: 'collapse-section', actions: [
                            { label: 'Collapse rows', id: 'collapse-rows', callback: (params) => this.collapseRows(params) },
                            { label: 'Collapse columns', id: 'collapse-columns', callback: (params) => this.collapseColumns(params) },
                            { label: 'Collapse all', id: 'collapse-all', callback: (params) => this.collapseAll(params) }
                        ] },
                        { label: 'Column size', id: 'column-size-section', actions: [
                            { label: 'Auto-size all columns', id: 'auto-size-all', callback: () => this.autoSizeAllColumns() }
                        ] }
                    ];
                    this.$rootElement;
                    this.chartDef;
                }

                refresh() { }

                destroy() {
                    this.clearClickListeners();
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                expandRows(params) {
                    this.setRowsOpened(params, true);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                collapseRows(params) {
                    this.setRowsOpened(params, false);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                collapseAll(params) {
                    this.collapseRows(params);
                    this.collapseColumns(params);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                expandAll(params) {
                    this.expandRows(params);
                    this.expandColumns(params);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                expandColumns(params) {
                    this.setColumnsOpened(params, true);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                collapseColumns(params) {
                    this.setColumnsOpened(params, false);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 * @param {boolean} newValue
                 */
                setColumnsOpened(params, newValue) {
                    this.chartDef.pivotTableOptions.areColumnExpandedByDefault = newValue;
                    this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus = {};
                    const columnGroups = params.api.getAllDisplayedColumnGroups();
                    const queue = new LinkedList();
                    columnGroups.forEach(group => queue.push(group.getDefinition()));

                    while (queue.length) {
                        const columnDef = queue.shift();
                        const groupId = columnDef && columnDef.groupId;
                        if (groupId) {
                            if (columnDef.children){
                                columnDef.children.forEach(child => queue.push(child));
                            }
                            params.api.setColumnGroupOpened(groupId, newValue);
                        }
                    }
                    // Force header cells to refresh. If not the header cell renderer is not called.
                    params.api.refreshHeader();
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 * @param {boolean} newValue
                 */
                setRowsOpened(params, newValue) {
                    this.chartDef.pivotTableOptions.areRowsExpandedByDefault = newValue;
                    this.chartDef.pivotTableOptions.rowIdByCustomExpandedStatus = {};
                    const queue = new LinkedList();
                    params.api.forEachNode(node => queue.push(node));;

                    while (queue.length) {
                        const row = queue.shift();
                        if (row.hasChildren()) {
                            if (row.childrenAfterGroup){
                                row.childrenAfterGroup.forEach(child => queue.push(child));
                            }
                            row.setExpanded(newValue);
                        }
                    }
                }

                autoSizeAllColumns() {
                    this.chartDef.pivotTableOptions.columnIdByCustomWidth = {};
                    // Forcing redraw so that width gets recomputed
                    const store = ChartStoreFactory.get(this.chartDef.$chartStoreId);
                    store.setGridApi(null);
                    $rootScope.$broadcast('redraw');
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                init(params) {
                    this.setParams(params);
                    this.eGui = document.createElement('div');
                    this.eGui.classList.add('pivot-table-expand-collapse-tool-panel');

                    const renderPanel = () => {
                        this.eGui.innerHTML = this.createTemplate();
                        this.initClickListeners(params);
                        this.setToolPanelButtonIcon();
                    };
                    params.api.addEventListener('gridReady', renderPanel);
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                setParams(params){
                    this.$rootElement = params.$rootElement;
                    this.chartDef = params.chartDef;
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                customRefresh(params){
                    this.setParams(params);
                    this.clearClickListeners();
                    this.eGui.innerHTML = this.createTemplate();
                    this.initClickListeners(params);
                }

                getGui() {
                    return this.eGui;
                }

                /**
                 * @param {ExpandCollapseToolPanelParams} params
                 */
                initClickListeners(params) {
                    for (const { actions } of this._sections) {
                        actions.forEach(({ id, callback }) => this.$rootElement.find(`.${this.getActionClassFromId(id)}`).on('click', () => callback(params)));
                    }
                }

                setToolPanelButtonIcon() {
                    /** @type HTMLElement[] */
                    const toolPanelButtons = this.$rootElement.find('.ag-side-button').toArray();
                    const expandCollapseButton = toolPanelButtons.filter(element => element.textContent.includes(EXPAND_COLLAPSE_TOOL_PANEL_LABEL))[0];
                    const iconElement = expandCollapseButton.querySelector('.ag-icon');
                    iconElement.classList.remove('ag-icon', 'ag-icon-');
                    iconElement.classList.add('dku-icon-chevron-double-down-16');
                }

                clearClickListeners() {
                    for (const { actions } of this._sections) {
                        actions.forEach(({ id }) => this.$rootElement.find(`.${this.getActionClassFromId(id)}`).off('click'));
                    }
                }

                /**
                 * @param {string} id
                 * @returns
                 */
                getActionClassFromId(id) {
                    return `pivot-table-expand-collapse-tool-panel__${id}`;
                }

                createTemplate() {
                    return this._sections.map((section) => {
                        return `
                            <div class="pivot-table-expand-collapse-tool-panel__section-title">${section.label}</div>
                            <ul>
                                ${section.actions.map(({ label, id }) => `<li><button class="${this.getActionClassFromId(id)}">${label}</button></li>`).join('')}
                            </ul>`;
                    }).join('');
                }
            }
            ExpandCollapseToolPanel.label = EXPAND_COLLAPSE_TOOL_PANEL_LABEL;
            ExpandCollapseToolPanel.id = 'expandCollapse';
            return ExpandCollapseToolPanel;
        });
})();
